/*
 * etPan! -- a mail user agent
 *
 * Copyright (C) 2001, 2002 - DINH Viet Hoa
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the libEtPan! project nor the names of its
 *    contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

/*
 * $Id: etpan-config.c,v 1.21 2004/12/16 12:57:18 hoa Exp $
 */

#include "etpan-config.h"

#include <sys/stat.h>
#include <sys/types.h>
#include <limits.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <paths.h>

#include "etpan-app.h"
#include "etpan-errors.h"
#include "etpan-tools.h"
#include <libetpan/libetpan.h>
#include "etpan-folder-discover.h"
#include "etpan-cfg-discover.h"
#include "etpan-cfg-sender.h"

static int etpan_cache_directory_init(struct etpan_storage_config *
    storage_config)
{
  char path[PATH_MAX];
  int r;
  unsigned int i;

  snprintf(path, PATH_MAX, "%s/%s", etpan_get_home_dir(), ETPAN_APP_DATA_PATH);
  r = etpan_mkdir(path);
  if (r < 0) {
    return ERROR_CACHE;
  }

  snprintf(path, PATH_MAX, "%s/%s", etpan_get_home_dir(), ETPAN_LOG_PATH);
  r = etpan_mkdir(path);
  if (r < 0) {
    return ERROR_CACHE;
  }

  snprintf(path, PATH_MAX, "%s/%s", etpan_get_home_dir(),
      ETPAN_DEFAULT_CONFIG_PATH);
  r = etpan_mkdir(path);
  if (r < 0) {
    return ERROR_CACHE;
  }
  
  snprintf(path, PATH_MAX, "%s/%s", etpan_get_home_dir(),
      ETPAN_DEFAULT_SMIME_PATH);
  r = etpan_mkdir(path);
  if (r < 0) {
    return ERROR_CACHE;
  }

  snprintf(path, PATH_MAX, "%s/%s", etpan_get_home_dir(),
      ETPAN_DEFAULT_SMIME_CERT_PATH);
  r = etpan_mkdir(path);
  if (r < 0) {
    return ERROR_CACHE;
  }

  snprintf(path, PATH_MAX, "%s/%s", etpan_get_home_dir(), ETPAN_TMP_PATH);
  r = etpan_mkdir(path);
  if (r < 0) {
    return ERROR_CACHE;
  }

  snprintf(path, PATH_MAX, "%s/%s", etpan_get_home_dir(), ETPAN_TMP_MIME_PATH);
  r = etpan_mkdir(path);
  if (r < 0) {
    return ERROR_CACHE;
  }

  snprintf(path, PATH_MAX, "%s/%s", etpan_get_home_dir(), ETPAN_CACHE_PATH);
  r = etpan_mkdir(path);
  if (r < 0) {
    return ERROR_CACHE;
  }

  snprintf(path, PATH_MAX, "%s/%s",
      etpan_get_home_dir(), ETPAN_FLAGS_PATH);
  r = etpan_mkdir(path);
  if (r < 0) {
    return ERROR_CACHE;
  }

  for(i = 0 ; i < carray_count(storage_config->storage_tab) ; i ++) {
    struct mailstorage * storage;
    
    storage = carray_get(storage_config->storage_tab, i);
    
    if (storage->sto_id == NULL)
      continue;
    
    snprintf(path, PATH_MAX, "%s/%s/%s",
        etpan_get_home_dir(), ETPAN_FLAGS_PATH, storage->sto_id);
    r = etpan_mkdir(path);
    if (r < 0) {
      return ERROR_CACHE;
    }
    
    snprintf(path, PATH_MAX, "%s/%s/%s",
        etpan_get_home_dir(), ETPAN_CACHE_PATH, storage->sto_id);
    r = etpan_mkdir(path);
    if (r < 0) {
      return ERROR_CACHE;
    }
  }
  
  return NO_ERROR;
}

static int etpan_account_config_init(char * config_path,
    struct etpan_account_config ** paccount_config)
{
  struct etpan_account_config * account_config;
  char path[PATH_MAX];
  int r;
  struct stat buf;
  clist * list;

  snprintf(path, PATH_MAX, "%s/account", config_path);

  r = etpan_account_config_read(path, &account_config);
  if (r == NO_ERROR) {
    * paccount_config = account_config;
    return NO_ERROR;
  }

  r = stat(path, &buf);
  if (r == 0) {
    printf("parse error - %s\n", path);
    getchar();
  }

  list = clist_new();
  if (list == NULL)
    goto err;

  account_config = etpan_account_config_new(list, NULL);
  if (account_config == NULL)
    goto free_list;

  * paccount_config = account_config;

  return NO_ERROR;
  
 free_list:
  clist_free(list);
 err:
  return ERROR_MEMORY;
}

static int etpan_storage_config_init(char * config_path,
    struct etpan_storage_config ** pstorage_config)
{
  struct etpan_storage_config * storage_config;
  char path[PATH_MAX];
  int r;
  struct mailstorage * storage;
  char cache_directory[PATH_MAX];
  char flags_directory[PATH_MAX];
  char location[PATH_MAX];
  char * user;
  struct stat buf;
  
  snprintf(path, PATH_MAX, "%s/storage", config_path);

  r = etpan_storage_config_read(path, &storage_config);
  if (r == NO_ERROR) {
    * pstorage_config = storage_config;
    return NO_ERROR;
  }

  r = stat(path, &buf);
  if (r == 0) {
    printf("parse error - %s\n", path);
    getchar();
  }

  /*
    initializes empty storage config
  */

  user = getenv("USER");
  if (user == NULL) {
    goto err;
  }

  storage =  mailstorage_new("default-mbox");
  if (storage == NULL) {
    goto err;
  }
  
  snprintf(cache_directory,
      PATH_MAX, "%s/%s/default-mbox", etpan_get_home_dir(), ETPAN_CACHE_PATH);

  snprintf(flags_directory,
      PATH_MAX, "%s/%s/default-mbox", etpan_get_home_dir(), ETPAN_FLAGS_PATH);

  snprintf(location, PATH_MAX, "%s/%s", ETPAN_SYSTEM_MAIL_PATH, user);

  r = mbox_mailstorage_init(storage, location, 1,
      cache_directory, flags_directory);
  if (r != NO_ERROR) {
    goto free_storage;
  }
  
  storage_config = etpan_storage_config_new();
  if (storage_config == NULL) {
    goto free_storage;
  }

  r = etpan_storage_config_add(storage_config, storage, 0);
  if (r != NO_ERROR) {
    goto free_config;
  }
  
  * pstorage_config = storage_config;
  
  return NO_ERROR;
  
 free_config:
  etpan_storage_config_free(storage_config);
 free_storage:
  mailstorage_free(storage);
 err:
  return ERROR_MEMORY;
}

static int config_one_folder(struct etpan_storage_config * storage_config,
    struct mailfolder * root, struct mailfolder ** folder_result)
{
  struct mailfolder * folder;
  struct mailstorage * storage;
  int r;
  
  if (carray_count(storage_config->storage_tab) == 0) {
    return ERROR_CONFIG;
  }
  
  storage = carray_get(storage_config->storage_tab, 0);
  if (storage->sto_driver == NULL)
    return ERROR_CONFIG;
  
  if (strcasecmp(storage->sto_driver->sto_name, "mbox") != 0)
    return ERROR_CONFIG;
  
  folder = mailfolder_new(storage, "mailbox", "mailbox");
  if (folder != NULL) {
      r = mailfolder_add_child(root, folder);
      if (r != NO_ERROR) {
          mailfolder_free(folder);
          return ERROR_MEMORY;
      }
  }
  
  * folder_result = folder;
  
  return NO_ERROR;
}

static int etpan_vfolder_config_init(char * config_path,
    struct etpan_account_config * account_config,
    struct etpan_storage_config * storage_config,
    struct etpan_vfolder_config ** pvfolder_config)
{
  struct etpan_vfolder_config * vfolder_config;
  char path[PATH_MAX];
  int r;
  struct mailfolder * root;
  struct stat buf;
  chash * folder_prop;
  struct mailfolder * folder;
  
  snprintf(path, PATH_MAX, "%s/vfolder", config_path);

  r = etpan_vfolder_config_read(path, storage_config, account_config,
      &vfolder_config);
  if (r == NO_ERROR) {
    * pvfolder_config = vfolder_config;
    return NO_ERROR;
  }

  r = stat(path, &buf);
  if (r == 0) {
    fprintf(stderr, "parse error - %s\n", path);
    getchar();
  }

  folder_prop = chash_new(CHASH_DEFAULTSIZE, CHASH_COPYKEY);
  if (folder_prop == NULL)
    goto err;
  
  root = mailfolder_new(NULL, NULL, NULL);
  if (root == NULL)
    goto free_prop;
  
  folder = NULL;
  config_one_folder(storage_config, root, &folder);
  
  vfolder_config = etpan_vfolder_config_new(root, folder_prop,
        folder, NULL, NULL);
  if (vfolder_config == NULL)
    goto free_root;
  
  * pvfolder_config = vfolder_config;

  return NO_ERROR;

 free_root:
  mailfolder_free(root);
 free_prop:
  chash_free(folder_prop);
 err:
  return ERROR_MEMORY;
}


#if 0
int etpan_offline_config_init(struct etpan_storage_config * storage_config,
    struct etpan_offline_config ** poffline_config)
{
  struct etpan_offline_config * offline_config;
  char path[PATH_MAX];
  int r;
  chash * folder_hash;
  clist * list;
  struct stat buf;


  r = etpan_offline_config_read(path, storage_config, &offline_config);
  if (r == NO_ERROR) {
    * poffline_config = offline_config;
    return NO_ERROR;
  }

  r = stat(path, &buf);
  if (r == 0) {
    printf("parse error - %s\n", path);
    getchar();
  }
  
  folder_hash = chash_new(CHASH_DEFAULTSIZE, CHASH_COPYKEY);
  if (folder_hash == NULL)
    goto err;

  list = clist_new();
  if (list == NULL)
    goto free_folder;

  offline_config = etpan_offline_config_new(folder_hash, list);
  if (offline_config == NULL)
    goto free_list;

  * poffline_config = offline_config;

  return NO_ERROR;

 free_list:
  clist_free(list);
 free_folder:
  chash_free(folder_hash);
 err:  
  return ERROR_MEMORY;
}
#endif


static int etpan_mime_config_init(char * config_path,
    struct etpan_mime_config ** pmime_config)
{
  struct etpan_mime_config * mime_config;
  char path[PATH_MAX];
  int r;
  clist * list;
  struct stat buf;

  snprintf(path, PATH_MAX, "%s/mime-config", config_path);

  r = etpan_mime_config_read(path, &mime_config);
  if (r == NO_ERROR) {
    * pmime_config = mime_config;
    return NO_ERROR;
  }

  r = stat(path, &buf);
  if (r == 0) {
    printf("parse error - %s\n", path);
    getchar();
  }

  list = clist_new();
  if (list == NULL)
    goto err;

  mime_config = etpan_mime_config_new(list);
  if (mime_config == NULL) {
    goto free_list;
  }

  * pmime_config = mime_config;

  return NO_ERROR;

 free_list:
  clist_free(list);
 err:
  return ERROR_MEMORY;
}

#if 0
int etpan_processing_config_init(struct etpan_vfolder_config * vfolder_config,
    struct etpan_processing_config **
    pprocessing_config)
{
  struct etpan_processing_config * processing_config;
  char path[PATH_MAX];
  int r;
  clist * preglobal;
  clist * postglobal;
  int expunge;
  chash * folder_rules;
  chash * folder_expunge;
  clist * rules;
  struct stat buf;

  snprintf(path, PATH_MAX, "%s/.libetpan/processing.xml", etpan_get_home_dir());

  r = etpan_processing_config_read(vfolder_config, path, &processing_config);
  if (r == NO_ERROR) {
    * pprocessing_config = processing_config;
    return NO_ERROR;
  }

  r = stat(path, &buf);
  if (r == 0) {
    printf("parse error - %s\n", path);
    getchar();
  }
  
  /*
    initializes empty processing config
   */
  
  preglobal = clist_new();
  if (preglobal == NULL)
    goto err;

  postglobal = clist_new();
  if (postglobal == NULL)
    goto free_pre;

  expunge = 0;

  folder_rules = chash_new(CHASH_DEFAULTSIZE, CHASH_COPYKEY);
  if (folder_rules == NULL)
    goto free_post;
  
  folder_expunge = chash_new(CHASH_DEFAULTSIZE, CHASH_COPYALL);
  if (folder_expunge == NULL)
    goto free_folder_rules;

  rules = clist_new();
  if (folder_expunge == NULL)
    goto free_folder_expunge;

  processing_config = etpan_processing_config_new(preglobal, postglobal,
    expunge, folder_rules, folder_expunge, rules);
  if (processing_config == NULL)
    goto free_rules;

    * pprocessing_config = processing_config;

  return NO_ERROR;

 free_rules:
  clist_free(rules);
 free_folder_expunge:
  chash_free(folder_expunge);
 free_folder_rules:
  chash_free(folder_rules);
 free_post:
  clist_free(postglobal);
 free_pre:
  clist_free(preglobal);
 err:
  return ERROR_MEMORY;
}
#endif

#if 0
int etpan_filtering_config_init(struct etpan_vfolder_config * vfolder_config,
    struct etpan_offline_config * offline_config,
    struct etpan_processing_config ** pfiltering_config)
{
  struct etpan_processing_config * filtering_config;
  char path[PATH_MAX];
  int r;
  clist * preglobal;
  clist * postglobal;
  int expunge;
  chash * folder_rules;
  chash * folder_expunge;
  clist * rules;
  struct stat buf;

  snprintf(path, PATH_MAX, "%s/.libetpan/config/filtering.xml", etpan_get_home_dir());

  r = etpan_filtering_config_read(vfolder_config, offline_config,
      path, &filtering_config);
  if (r == NO_ERROR) {
    * pfiltering_config = filtering_config;
    return NO_ERROR;
  }

  r = stat(path, &buf);
  if (r == 0) {
    printf("parse error - %s\n", path);
    getchar();
  }

  /*
    initializes empty filtering config
   */

  preglobal = clist_new();
  if (preglobal == NULL)
    goto err;

  postglobal = clist_new();
  if (postglobal == NULL)
    goto free_pre;

  expunge = 0;

  folder_rules = chash_new(CHASH_DEFAULTSIZE, CHASH_COPYKEY);
  if (folder_rules == NULL)
    goto free_post;
  
  folder_expunge = chash_new(CHASH_DEFAULTSIZE, CHASH_COPYALL);
  if (folder_expunge == NULL)
    goto free_folder_rules;

  rules = clist_new();
  if (folder_expunge == NULL)
    goto free_folder_expunge;

  filtering_config = etpan_processing_config_new(preglobal, postglobal,
    expunge, folder_rules, folder_expunge, rules);
  if (filtering_config == NULL)
    goto free_rules;

  * pfiltering_config = filtering_config;

  return NO_ERROR;

 free_rules:
  clist_free(rules);
 free_folder_expunge:
  chash_free(folder_expunge);
 free_folder_rules:
  chash_free(folder_rules);
 free_post:
  clist_free(postglobal);
 free_pre:
  clist_free(preglobal);
 err:
  return ERROR_MEMORY;
}
#endif

static int etpan_abook_config_init(char * config_path,
    struct etpan_global_config * global_config,
    struct etpan_abook_config ** pabook_config)
{
  struct etpan_abook_config * abook_config;
  char path[PATH_MAX];
  int r;
#if 0
  clist * entry_list;
  clist * list_list;
#endif
  carray * tab;
  struct stat buf;

  snprintf(path, PATH_MAX, "%s/abook-config", config_path);

  r = etpan_abook_config_read(global_config, path, &abook_config);
  if (r == NO_ERROR) {
    * pabook_config = abook_config;
    return NO_ERROR;
  }

  r = stat(path, &buf);
  if (r == 0) {
    printf("parse error - %s\n", path);
    getchar();
  }

  /*
    initializes empty abook config
   */

#if 0
  entry_list = clist_new();
  if (entry_list == NULL)
    goto err;

  list_list = clist_new();
  if (list_list == NULL)
    goto free_entry;
#endif
  tab = carray_new(16);
  if (tab == NULL)
    goto err;
  
  abook_config = etpan_abook_config_new(tab);
  if (abook_config == NULL) {
    goto free_tab;
  }

  * pabook_config = abook_config;

  return NO_ERROR;

 free_tab:
  carray_free(tab);
 err:
  return ERROR_MEMORY;
}


static int etpan_smime_config_init(char * config_path,
    struct etpan_smime_config ** psmime_config)
{
  struct etpan_smime_config * smime_config;
  char path[PATH_MAX];
  int r;
  struct stat buf;
  char * cert_dir;
  char * CA_dir;
  char * private_keys_dir;
  char dir_buf[PATH_MAX];
  
  snprintf(path, PATH_MAX, "%s/smime-config", config_path);

  r = etpan_smime_config_read(path, &smime_config);
  if (r == NO_ERROR) {
    * psmime_config = smime_config;
    return NO_ERROR;
  }

  r = stat(path, &buf);
  if (r == 0) {
    printf("parse error - %s\n", path);
    getchar();
  }

  /*
    initializes empty smime config
   */
  
  snprintf(dir_buf, sizeof(dir_buf), "%s/%s",
      etpan_get_home_dir(), ETPAN_DEFAULT_SMIME_CERT_PATH);
  cert_dir = strdup(dir_buf);
  if (cert_dir == NULL)
    goto err;

  snprintf(dir_buf, sizeof(dir_buf), "%s/%s",
      etpan_get_home_dir(), ETPAN_DEFAULT_SMIME_CA_PATH);
  CA_dir = strdup(dir_buf);
  if (CA_dir == NULL)
    goto free_cert_dir;

  snprintf(dir_buf, sizeof(dir_buf), "%s/%s",
      etpan_get_home_dir(), ETPAN_DEFAULT_SMIME_PRIVATE_KEYS_PATH);
  private_keys_dir = strdup(dir_buf);
  if (private_keys_dir == NULL)
    goto err;
  
  smime_config = etpan_smime_config_new(cert_dir, CA_dir,
      private_keys_dir, 1, 1);
  if (smime_config == NULL) {
    goto free_CA_dir;
  }
  
  * psmime_config = smime_config;
  
  return NO_ERROR;
  
 free_cert_dir:
  free(cert_dir);
 free_CA_dir:
  free(CA_dir);
 err:
  return ERROR_MEMORY;
}


static int etpan_global_config_init(char * config_path,
    struct etpan_global_config ** pglobal_config)
{
  struct etpan_global_config * global_config;
  char path[PATH_MAX];
  int r;
  char * display_charset;
  char * editor_charset;
  char * message_charset;
  char * editor;
  int res;
  chash * color_config;
  struct stat buf;
  char * sendmail_path;
  
  snprintf(path, PATH_MAX, "%s/global", config_path);

  r = etpan_global_config_read(path, &global_config);
  if (r == NO_ERROR) {
    * pglobal_config = global_config;
    return NO_ERROR;
  }

  r = stat(path, &buf);
  if (r == 0) {
    printf("parse error - %s\n", path);
    getchar();
  }

  /*
    initializes global config
   */

  color_config = chash_new(CHASH_DEFAULTSIZE, CHASH_COPYALL);
  if (color_config == NULL) {
    res = ERROR_MEMORY;
    goto err;
  }

  display_charset = strdup(ETPAN_DEFAULT_DISPLAY_CHARSET);
  if (display_charset == NULL) {
    res = ERROR_MEMORY;
    goto free_color;
  }

  editor_charset = strdup(ETPAN_DEFAULT_EDITOR_CHARSET);
  if (editor_charset == NULL) {
    res = ERROR_MEMORY;
    goto free_display_charset;
  }

  message_charset = strdup(ETPAN_DEFAULT_MESSAGE_CHARSET);
  if (message_charset == NULL) {
    res = ERROR_MEMORY;
    goto free_editor_charset;
  }

  editor = getenv("EDITOR");
  if (editor == NULL)
    editor = ETPAN_DEFAULT_EDITOR;
  
  editor = strdup(editor);
  if (editor == NULL)
    goto free_message_charset;
  
  sendmail_path = ETPAN_DEFAULT_SENDMAIL_PATH;
  sendmail_path = strdup(sendmail_path);
  if (sendmail_path == NULL)
    goto free_editor;
  
  global_config = etpan_global_config_new(display_charset,
      editor_charset, message_charset, editor,
      ETPAN_DEFAULT_REPLY_QUOTE_LIMIT,
      ETPAN_DEFAULT_NETWORK_TIMEOUT,
      ETPAN_DEFAULT_POLL_DELAY,
      sendmail_path, 1, 0);
  if (global_config == NULL)
    goto free_sendmail_path;
  
  * pglobal_config = global_config;

  return NO_ERROR;
  
 free_sendmail_path:
  free(sendmail_path);
 free_editor:
  free(editor);
 free_message_charset:
  free(message_charset);
 free_editor_charset:
  free(editor_charset);
 free_display_charset:
  free(display_charset);
 free_color:
  chash_free(color_config);
 err:
  return ERROR_MEMORY;
}



static int etpan_color_config_init(char * config_path,
    struct etpan_color_config ** pcolor_config)
{
  struct etpan_color_config * color_config;
  char path[PATH_MAX];
  int r;
  struct stat buf;
  chash * color_hash;
  int res;

  snprintf(path, PATH_MAX, "%s/color", config_path);
  
  r = etpan_color_config_read(path, &color_config);
  if (r == NO_ERROR) {
    * pcolor_config = color_config;
    return NO_ERROR;
  }
  
  r = stat(path, &buf);
  if (r == 0) {
    printf("parse error - %s\n", path);
    getchar();
  }
  
  color_hash = chash_new(CHASH_DEFAULTSIZE, CHASH_COPYALL);
  if (color_hash == NULL) {
    res = ERROR_MEMORY;
    goto err;
  }
  
  color_config = etpan_color_config_new(color_hash);
  if (color_config == NULL) {
    res = ERROR_MEMORY;
    goto free_hash;
  }
  
  * pcolor_config = color_config;
  
  return NO_ERROR;
  
 free_hash:
  chash_free(color_hash);
 err:
  return res;
}

#if 0
int etpan_msg_list_key_config_init(struct etpan_color_config ** pcolor_config)
{
  struct etpan_color_config * color_config;
  char path[PATH_MAX];
  int r;
  struct stat buf;
  chash * color_hash;
  int res;

  snprintf(path, PATH_MAX, "%s/.libetpan/config/color", etpan_get_home_dir());
  
  r = etpan_color_config_read(path, &color_config);
  if (r == NO_ERROR) {
    * pcolor_config = color_config;
    return NO_ERROR;
  }
  
  r = stat(path, &buf);
  if (r == 0) {
    printf("parse error - %s\n", path);
    getchar();
  }
  
  color_hash = chash_new(CHASH_DEFAULTSIZE, CHASH_COPYALL);
  if (color_hash == NULL) {
    res = ERROR_MEMORY;
    goto err;
  }
  
  color_config = etpan_color_config_new(color_hash);
  if (color_config == NULL) {
    res = ERROR_MEMORY;
    goto free_hash;
  }
  
  * pcolor_config = color_config;
  
  return NO_ERROR;
  
 free_hash:
  chash_free(color_hash);
 err:
  return res;
}
#endif

static int etpan_discover_config_init(char * config_path,
    struct etpan_app * app,
    struct etpan_account_config * account_config,
    struct etpan_storage_config * storage_config,
    struct etpan_vfolder_config * vfolder_config,
    struct etpan_discovery_manager ** result)
{
  struct etpan_discovery_manager * discovery_manager;
  char path[PATH_MAX];
  int r;
  struct stat buf;
  
  snprintf(path, PATH_MAX, "%s/discovery", config_path);
  
  r = etpan_discover_config_read(path, app,
      account_config,
      storage_config,
      &discovery_manager);
  if (r == NO_ERROR) {
    * result = discovery_manager;
    return NO_ERROR;
  }
  
  r = stat(path, &buf);
  if (r == 0) {
    printf("parse error - %s\n", path);
    getchar();
  }
  
  discovery_manager = etpan_discovery_manager_new(app);
  if (discovery_manager == NULL)
    return ERROR_MEMORY;
  
  * result = discovery_manager;
  
  return NO_ERROR;
}

static int etpan_sender_config_init(char * config_path,
    struct etpan_sender_config ** result)
{
  struct etpan_sender_config * config;
  char path[PATH_MAX];
  int r;
  struct etpan_sender_item * item;
  
  snprintf(path, PATH_MAX, "%s/sender", config_path);
  
  r = etpan_sender_config_read(path, &config);
  if (r == NO_ERROR) {
    
    if (config->default_sender[0] == '\0') {
      chashiter * iter;
      chashdatum value;
      
      if (chash_count(config->sender_hash) == 0) {
        item = etpan_sender_item_new("sendmail", SENDER_TYPE_COMMAND,
            ETPAN_DEFAULT_SENDMAIL_PATH,
            NULL, 0, NULL, NULL);
        if (item == NULL) {
          etpan_sender_config_free(config);
          return ERROR_MEMORY;
        }
        
        r = etpan_sender_config_add_item(config, item);
        if (r != MAIL_NO_ERROR) {
          etpan_sender_item_free(item);
          etpan_sender_config_free(config);
          return ERROR_MEMORY;
        }
      }
      
      iter = chash_begin(config->sender_hash);
      chash_value(iter, &value);
      item = value.data;
      strncpy(config->default_sender, item->name,
          sizeof(config->default_sender));
      config->default_sender[sizeof(config->default_sender)] = '\0';
    }
    
    * result = config;
    return NO_ERROR;
  }
  
  config = etpan_sender_config_new();
  if (config == NULL)
    return ERROR_MEMORY;
  
  strncpy(config->default_sender, "sendmail", sizeof(config->default_sender));
  
  item = etpan_sender_item_new("sendmail", SENDER_TYPE_COMMAND,
      ETPAN_DEFAULT_SENDMAIL_PATH,
      NULL, 0, NULL, NULL);
  if (item == NULL) {
    etpan_sender_config_free(config);
    return ERROR_MEMORY;
  }
  
  r = etpan_sender_config_add_item(config, item);
  if (r != MAIL_NO_ERROR) {
    etpan_sender_item_free(item);
    etpan_sender_config_free(config);
    return ERROR_MEMORY;
  }
  
  * result = config;
  
  return NO_ERROR;
}





/* write config */

static void etpan_storage_config_write(char * config_path,
    struct etpan_app * app,
    struct etpan_storage_config * config)
{
  char tmp_filename[PATH_MAX];
  char filename[PATH_MAX];
  int r;

  snprintf(tmp_filename, PATH_MAX,
      "%s/storage.tmp", config_path);
  snprintf(filename, PATH_MAX,
      "%s/storage", config_path);
  
  r = etpan_cfg_storage_write(tmp_filename, config);
  if (r != NO_ERROR) {
    ETPAN_APP_LOG((app, "error writing storage config file - %s",
                      tmp_filename));
  }
  r = rename(tmp_filename, filename);
  if (r < 0) {
    ETPAN_APP_LOG((app, "error writing storage config file - %s",
                      filename));
    unlink(tmp_filename);
  }
  ETPAN_APP_LOG((app, "storage config file written - %s", filename));
}

static void etpan_vfolder_config_write(char * config_path,
    struct etpan_app * app,
    struct etpan_vfolder_config * vfolder_config,
    struct etpan_account_config * account_config)
{
  char tmp_filename[PATH_MAX];
  char filename[PATH_MAX];
  int r;

  snprintf(tmp_filename, PATH_MAX,
      "%s/vfolder.tmp", config_path);
  snprintf(filename, PATH_MAX,
      "%s/vfolder", config_path);
  
  r = etpan_cfg_vfolder_write(tmp_filename,
      vfolder_config, account_config);
  if (r != NO_ERROR) {
    ETPAN_APP_LOG((app, "error writing folder config file - %s",
                      tmp_filename));
  }
  r = rename(tmp_filename, filename);
  if (r < 0) {
    ETPAN_APP_LOG((app, "error writing folder config file - %s",
                      filename));
    unlink(tmp_filename);
  }
  ETPAN_APP_LOG((app, "folder config file written - %s", filename));
}

static void etpan_account_config_write(char * config_path,
    struct etpan_app * app,
    struct etpan_account_config * config)
{
  char tmp_filename[PATH_MAX];
  char filename[PATH_MAX];
  int r;

  snprintf(tmp_filename, PATH_MAX,
      "%s/account.tmp", config_path);
  snprintf(filename, PATH_MAX,
      "%s/account", config_path);
  
  r = etpan_cfg_account_write(tmp_filename, config);
  if (r != NO_ERROR) {
    ETPAN_APP_LOG((app, "error writing account config file - %s",
                      tmp_filename));
  }
  r = rename(tmp_filename, filename);
  if (r < 0) {
    ETPAN_APP_LOG((app, "error writing account config file - %s",
                      filename));
    unlink(tmp_filename);
  }
  ETPAN_APP_LOG((app, "account config file written - %s", filename));
}

static void etpan_global_config_write(char * config_path,
    struct etpan_app * app,
    struct etpan_global_config * config)
{
  char tmp_filename[PATH_MAX];
  char filename[PATH_MAX];
  int r;

  snprintf(tmp_filename, PATH_MAX,
      "%s/global.tmp", config_path);
  snprintf(filename, PATH_MAX,
      "%s/global", config_path);
  
  r = etpan_cfg_global_write(tmp_filename, config);
  if (r != NO_ERROR) {
    ETPAN_APP_LOG((app, "error writing global config file - %s",
                      tmp_filename));
  }
  r = rename(tmp_filename, filename);
  if (r < 0) {
    ETPAN_APP_LOG((app, "error writing global config file - %s",
                      filename));
    unlink(tmp_filename);
  }
  ETPAN_APP_LOG((app, "global config file written - %s", filename));
}

void etpan_config_write(struct etpan_app * app,
    struct etpan_app_config * config)
{
  etpan_global_config_write(config->path, app, config->global_config);
  etpan_storage_config_write(config->path, app, config->storage_config);
  etpan_vfolder_config_write(config->path, app, config->vfolder_config,
      config->account_config);
  etpan_account_config_write(config->path, app, config->account_config);
}

static int recursive_mark_storage(chash * storage_hash,
    struct mailfolder * folder)
{
  unsigned int i;
  chashdatum key;
  chashdatum value;
  int r;
  
  key.data = &folder->fld_storage;
  key.len = sizeof(folder->fld_storage);
  value.data = folder->fld_storage;
  value.len = 0;
  
  r = chash_set(storage_hash, &key, &value, NULL);
  if (r < 0)
    return ERROR_MEMORY;
  
  for(i = 0 ; i < carray_count(folder->fld_children) ; i ++) {
    struct mailfolder * child;
    
    child = carray_get(folder->fld_children, i);
    r = recursive_mark_storage(storage_hash, child);
    if (r != NO_ERROR)
      return r;
  }
  
  return NO_ERROR;
}

static int cleanup_storage(struct etpan_app_config * config)
{
  chash * storage_hash;
  int res;
  unsigned int i;
  int r;
  carray * storage_to_delete;
  chashiter * iter;
  
  storage_hash = chash_new(CHASH_DEFAULTSIZE, CHASH_COPYKEY);
  if (storage_hash == NULL) {
    res = ERROR_MEMORY;
    goto err;
  }
  
  r = recursive_mark_storage(storage_hash, config->vfolder_config->root);
  if (r != NO_ERROR) {
    res = r;
    goto free_hash;
  }
  
  for(iter = chash_begin(config->discovery_config->vpath_hash) ;
      iter != NULL ;
      iter = chash_next(config->discovery_config->vpath_hash, iter)) {
    chashdatum key;
    chashdatum value;
    struct etpan_discovery_info * info;
    struct mailstorage * storage;
    
    chash_value(iter, &value);
    info = value.data;
    storage = info->storage;
    
    key.data = &storage;
    key.len = sizeof(storage);
    value.data = storage;
    value.len = 0;
    r = chash_set(storage_hash, &key, &value, NULL);
    if (r < 0) {
      res = ERROR_MEMORY;
      goto free_hash;
    }
  }
  
  storage_to_delete = carray_new(16);
  if (storage_to_delete == NULL) {
    res = ERROR_MEMORY;
    goto free_hash;
  }
  
  for(i = 0 ; i < carray_count(config->storage_config->storage_tab) ; i ++) {
    struct mailstorage * storage;
    chashdatum key;
    chashdatum value;
    
    storage = carray_get(config->storage_config->storage_tab, i);
    
    key.data = &storage;
    key.len = sizeof(storage);
    r = chash_get(storage_hash, &key, &value);
    if (r < 0) {
      carray_add(storage_to_delete, storage, NULL);
    }
  }
  
  for(i = 0 ; i < carray_count(storage_to_delete) ; i ++) {
    struct mailstorage * storage;
    
    storage = carray_get(storage_to_delete, i);
    etpan_storage_config_delete(config->storage_config, storage);
  }
  
  carray_free(storage_to_delete);
  
  chash_free(storage_hash);
  
  return NO_ERROR;
  
 free_hash:
  chash_free(storage_hash);
 err:
  return res;
}

int etpan_app_config_init(char * path, struct etpan_app * app,
    struct etpan_app_config * config)
{
  int r;
  int res;
  
  strncpy(config->path, path, sizeof(config->path));
  config->path[sizeof(config->path) - 1] = '\0';
  
  r = etpan_global_config_init(path, &config->global_config);
  if (r != NO_ERROR) {
    ETPAN_APP_LOG((app, "error initializing global config"));
    res = r;
    goto err;
  }
  
  r = etpan_account_config_init(path, &config->account_config);
  if (r != NO_ERROR) {
    ETPAN_APP_LOG((app, "error initializing account config"));
    res = r;
    goto free_global;
  }
  
  r = etpan_storage_config_init(path, &config->storage_config);
  if (r != NO_ERROR) {
    ETPAN_APP_LOG((app, "error initializing storage config"));
    res = r;
    goto free_account;
  }
  
  /*
    initialize cache directory after storage config
    to create cache directories for storage
  */
  r = etpan_cache_directory_init(config->storage_config);
  if (r != NO_ERROR) {
    ETPAN_APP_LOG((app, "error creating cache directories"));
    res = r;
    goto free_storage;
  }
  
  r = etpan_vfolder_config_init(path, config->account_config,
      config->storage_config,
      &config->vfolder_config);
  if (r != NO_ERROR) {
    ETPAN_APP_LOG((app, "error initializing vfolder config"));
    res = r;
    goto free_storage;
  }

  r = etpan_mime_config_init(path, &config->mime_config);
  if (r != NO_ERROR) {
    ETPAN_APP_LOG((app, "error initializing mime config"));
    res = r;
    goto free_vfolder;
  }
  
  r = etpan_abook_config_init(path, config->global_config,
      &config->abook_config);
  if (r != NO_ERROR) {
    ETPAN_APP_LOG((app, "error initializing abook config"));
    res = r;
    goto free_mime;
  }

  r = etpan_smime_config_init(path, &config->smime_config);
  if (r != NO_ERROR) {
    ETPAN_APP_LOG((app, "error initializing abook config"));
    res = r;
    goto free_abook;
  }

  r = etpan_color_config_init(path, &config->color_config);
  if (r != NO_ERROR) {
    ETPAN_APP_LOG((app, "error initializing color"));
    res = r;
    goto free_smime;
  }
  
  r = etpan_discover_config_init(path, app,
      config->account_config,
      config->storage_config,
      config->vfolder_config,
      &config->discovery_config);
  if (r != NO_ERROR) {
    ETPAN_APP_LOG((app, "error initializing discover"));
    res = r;
    goto free_color;
  }
  
  r = etpan_sender_config_init(path,
      &config->sender_config);
  if (r != NO_ERROR) {
    ETPAN_APP_LOG((app, "error initializing sender"));
    res = r;
    goto free_discover;
  }
  
  cleanup_storage(config);
  
  /* set post config things */
  
  /* S/MIME config */
  
  mailprivacy_smime_set_cert_dir(app->privacy,
      config->smime_config->cert_dir);
  
  mailprivacy_smime_set_CA_dir(app->privacy,
      config->smime_config->CA_dir);
  
  mailprivacy_smime_set_private_keys_dir(app->privacy,
      config->smime_config->private_keys_dir);
  
  mailprivacy_smime_set_CA_check(app->privacy,
      config->smime_config->CA_check);
  
  mailprivacy_smime_set_store_cert(app->privacy,
      config->smime_config->auto_extract_cert);
  
  return NO_ERROR;

 free_sender:
  etpan_sender_config_free(config->sender_config);
 free_discover:
  etpan_discovery_manager_free(config->discovery_config);
 free_color:
  etpan_color_config_free(config->color_config);
 free_smime:
  etpan_smime_config_free(config->smime_config);
 free_abook:
  etpan_abook_config_free(config->abook_config);
 free_mime:
  etpan_mime_config_free(config->mime_config);
 free_vfolder:
  etpan_vfolder_config_free(config->vfolder_config);
 free_storage:
  etpan_storage_config_free(config->storage_config);
 free_account:
  etpan_account_config_free(config->account_config);
 free_global:
  etpan_global_config_free(config->global_config);
 err:
  return res;
}

void etpan_app_config_done(struct etpan_app_config * config)
{
  if (config->sender_config != NULL)
    etpan_sender_config_free(config->sender_config);
  if (config->discovery_config != NULL)
    etpan_discovery_manager_free(config->discovery_config);
  if (config->color_config != NULL)
    etpan_color_config_free(config->color_config);
  if (config->smime_config != NULL)
    etpan_smime_config_free(config->smime_config);
  if (config->abook_config != NULL)
    etpan_abook_config_free(config->abook_config);
  if (config->mime_config != NULL)
    etpan_mime_config_free(config->mime_config);
  if (config->vfolder_config != NULL)
    etpan_vfolder_config_free(config->vfolder_config);
  if (config->storage_config != NULL)
    etpan_storage_config_free(config->storage_config);
  if (config->account_config != NULL)
    etpan_account_config_free(config->account_config);
  if (config->global_config != NULL)
    etpan_global_config_free(config->global_config);
}
