/*
 * etPan! -- a mail user agent
 *
 * Copyright (C) 2001, 2002 - DINH Viet Hoa
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the libEtPan! project nor the names of its
 *    contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

/*
 * $Id: etpan-folder-common.c,v 1.11 2004/12/27 15:27:38 hoa Exp $
 */

#include "etpan-folder-common.h"
#include "etpan-subapp.h"
#include "etpan-subapp-thread.h"
#include "etpan-errors.h"
#include "etpan-folder-params.h"
#include "etpan-app-subapp.h"
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <ncurses.h>
#include <libetpan/libetpan.h>
#include <libetpan/charconv.h>
#include "etpan-msg-list-app.h"
#include "etpan-cfg-vfolder.h"
#include "etpan-app.h"
#include "etpan-msg-new.h"
#include <regex.h>
#include "etpan-search-input.h"

static int etpan_folder_forward_search(struct
    etpan_folder_common_app_state * state,
    unsigned int first);

static int etpan_folder_backward_search(struct
    etpan_folder_common_app_state * state,
    unsigned int last);

static void etpan_set_folder_state(struct
    etpan_folder_common_app_state * state);

static int etpan_folder_expand_node(struct
    etpan_folder_common_app_state * state,
    struct mailfolder * folder);

static int next_unread_folder(struct etpan_folder_common_app_state * state);

static int search_folder(struct etpan_subapp * app,
    struct etpan_folder_common_app_state * state);

void etpan_folder_common_handle_key(struct etpan_subapp * app,
    struct etpan_folder_common_app_state * state, int key)
{
  struct mailfolder * folder;
  
  switch (key) {
  case '0':
  case '1':
  case '2':
  case '3':
  case '4':
  case '5':
  case '6':
  case '7':
  case '8':
  case '9':
  case 'G':
    break;
  default:
    state->chosen = 0;
    break;
  }

  switch (key) {
  case '0':
  case '1':
  case '2':
  case '3':
  case '4':
  case '5':
  case '6':
  case '7':
  case '8':
  case '9':
    state->chosen *= 10;
    state->chosen += key - '0';
    break;
  case 'G':
    if (state->chosen == 0) {
      if (state->folder_tab->len > 0) {
        state->selected = etpan_folder_backward_search(state,
            state->folder_tab->len - 1);
      }
    }
    else
      state->selected = etpan_folder_forward_search(state, state->chosen - 1);
    
    state->chosen = 0;
    break;
    
  case 'j':
  case KEY_DOWN:
    if (state->selected_window_forward_count > 1)
      state->selected = state->selected_window_forward[1];
    break;
          
  case 'k':
  case KEY_UP:
    if (state->selected_window_backward_count > 1)
      state->selected = state->selected_window_backward[1];
    break;
    
  case KEY_NPAGE:
    if (state->selected_window_forward_count > 0)
      state->selected = state->selected_window_forward[state->selected_window_forward_count - 1];
    break;
    
  case KEY_PPAGE:
    if (state->selected_window_backward_count > 0)
      state->selected = state->selected_window_backward[state->selected_window_backward_count - 1];
    break;
    
  case KEY_HOME:
    state->selected = etpan_folder_forward_search(state, 0);
    break;
    
  case KEY_END:
    if (state->folder_tab->len > 0) {
      state->selected = etpan_folder_backward_search(state,
          state->folder_tab->len - 1);
    }
    break;
    
  case '=':
    etpan_set_folder_state(state);
    break;
    
  case '+':
     /* open thread */
    folder = carray_get(state->folder_tab, state->selected);
    etpan_folder_change_opened(state->params, folder);
    break;
    
  case '*':
    etpan_folder_expand_node(state, state->root);
    break;
    
  case '/':
    search_folder(app, state);
    break;

  case 'n':
    next_unread_folder(state);
    break;
    
  case '\n':
    break;
  }
}

static int etpan_folder_common_display_precalc(struct
    etpan_folder_common_app_state * state);
static void etpan_folder_common_display_precalc_free(struct
    etpan_folder_common_app_state * state);

void etpan_folder_common_set_color(struct etpan_subapp * app,
    struct etpan_folder_common_app_state * state)
{
  etpan_app_set_color(app->app, "main",
      &state->main_attr, A_NORMAL);
  etpan_app_set_color(app->app, "selection",
      &state->selection_attr, A_REVERSE);
  etpan_app_set_color(app->app, "status",
      &state->status_attr, A_REVERSE);
}

int etpan_folder_common_init(struct etpan_folder_common_app_state * state)
{
  state->root = NULL;

  /* colors */
  state->main_attr = A_NORMAL;
  state->selection_attr = A_REVERSE;
  state->status_attr = A_REVERSE;
  
  /* cursor & selection */
  state->chosen = 0;
  state->first = 0;
  state->selected = 0;
  
  /* display optimization */
  state->window_size = 0;
  state->selected_window_backward = NULL;
  state->selected_window_backward_count = 0;
  state->selected_window_forward = NULL;
  state->selected_window_forward_count = 0;
  
  state->folder_tab = carray_new(128);
  if (state->folder_tab == NULL)
    goto err;

  state->prefix_hash = chash_new(CHASH_DEFAULTSIZE, CHASH_COPYALL);
  if (state->prefix_hash == NULL)
    goto free_folder_tab;
  
  state->params = etpan_folder_params_new();
  if (state->params == NULL)
    goto free_prefix;
  
  state->flags = 0;
  state->app = NULL;
  state->search_str = NULL;
  
  return NO_ERROR;
  
 free_prefix:
  chash_free(state->prefix_hash);
 free_folder_tab:
  carray_free(state->folder_tab);
 err:
  return ERROR_MEMORY;
}

void etpan_folder_common_done(struct etpan_folder_common_app_state * state)
{
  if (state->selected_window_backward != NULL)
    free(state->selected_window_backward);
  if (state->selected_window_forward != NULL)
    free(state->selected_window_forward);

  etpan_folder_params_free(state->params);
  chash_free(state->prefix_hash);
  carray_free(state->folder_tab);
}


int
etpan_folder_common_update(struct etpan_folder_common_app_state * state)
{
  int r;
  
  etpan_folder_common_display_precalc_free(state);
  
  r = etpan_folder_params_add_recursive(state->params, state->root);
  if (r != NO_ERROR)
    goto err;
  
  r = etpan_folder_common_display_precalc(state);
  if (r != NO_ERROR)
    goto clear;
  
  return NO_ERROR;
  
 clear:
  etpan_folder_params_clear(state->params);
 err:
  return r;
}

int
etpan_folder_common_set_root_folder(struct etpan_folder_common_app_state *
    state, struct mailfolder * root,
    int flags)
{
  int r;
  
  etpan_folder_common_display_precalc_free(state);
  etpan_folder_params_clear(state->params);
  
  state->root = root;
  state->first = 0;
  state->selected = 0;
  state->flags = flags;
  state->app = NULL;
  state->search_str = NULL;
  
  if (root != NULL) {
    r = etpan_folder_params_add_recursive(state->params, root);
    if (r != NO_ERROR)
      goto err;
    
    r = etpan_folder_common_display_precalc(state);
    if (r != NO_ERROR)
      goto clear;
    
    etpan_set_folder_state(state);
  }
  
  return NO_ERROR;
  
 clear:
  etpan_folder_params_clear(state->params);
 err:
  return r;
}


void etpan_folder_common_flush(struct etpan_subapp * app,
    struct etpan_folder_common_app_state * state)
{
  etpan_folder_common_set_root_folder(state, NULL, 0);
}

/* ***************************************** */
/* implementation */

static int
etpan_folder_forward_search(struct etpan_folder_common_app_state * state,
    unsigned int first)
{
  unsigned int i;

  for(i = first ; i < state->folder_tab->len ; i++) {
    struct mailfolder * folder;

    folder = carray_get(state->folder_tab, i);
    
    if (etpan_folder_is_visible(state->params, folder)) {
      return i;
    }
  }

  return first;
}

static int
etpan_folder_backward_search(struct etpan_folder_common_app_state * state,
    unsigned int last)
{
  unsigned int i;

  i = last;
  do {
    struct mailfolder * folder;

    folder = carray_get(state->folder_tab, i);
    
    if (etpan_folder_is_visible(state->params, folder))
      return i;
    

    if (i == 0)
      break;

    i --;

  } while (1);

  return last;
}

#define FOLDER_TYPE_STR_SENT "Sent"
#define FOLDER_TYPE_STR_DRAFT "Draft"
#define FOLDER_TYPE_STR_DEFAULT "Default"

static void snprint_folder_info(struct etpan_app * app,
    struct etpan_folder_common_app_state * state,
    char * output, char * buffer, char * fill, int width,
    struct mailfolder * folder)
{
  chashdatum key;
  chashdatum value;
  char * prefix;
  char * thread_prefix;
  int has_child;
  int r;
  char * folder_name;
  char * decoded_folder_name;
  char * dsp_folder_name;
  uint32_t num;
  uint32_t unseen;
  uint32_t recent;

  key.data = &folder;
  key.len = sizeof(folder);
  r = chash_get(state->prefix_hash, &key, &value);
  if (r < 0)
    prefix = "";
  else
    prefix = value.data;

  has_child = (carray_count(folder->fld_children) > 0);

  if (has_child) {
    if (etpan_folder_is_opened(state->params, folder))
      thread_prefix = "(-)";
    else
      thread_prefix = "(+)";
  }
  else
    thread_prefix = "   ";

  if (folder->fld_virtual_name != NULL)
    folder_name = folder->fld_virtual_name;
  else
    folder_name = "/";

  decoded_folder_name = NULL;
#if 0
  r = charconv(app->config.global_config->display_charset,
      app->config.global_config->editor_charset,
      folder_name, strlen(folder_name),
      &decoded_folder_name);
  if (r != MAIL_CHARCONV_NO_ERROR)
    dsp_folder_name = folder_name;
  else
    dsp_folder_name = decoded_folder_name;
#endif
  dsp_folder_name = folder_name;
  
  r = etpan_folder_get_stat(state->params, folder,
      &num, &recent, &unseen);
  
  if (width <= 20)
    * buffer = 0;
  else {
    if ((state->flags & ETPAN_FOLDER_COMMON_SHOW_SPECIAL_FOLDERS) != 0) {
      int type;
      char type_str[80];
      size_t type_len;
      char * cur;
      
      type = etpan_folder_get_folder_type(state->params, folder);
      
      type_len = sizeof(type_str);
      * type_str = '\0';
      
      if ((type & ETPAN_FOLDER_TYPE_DEFAULT) != 0) {
        snprintf(type_str, type_len, " %s", FOLDER_TYPE_STR_DEFAULT);
        type_len -= sizeof(FOLDER_TYPE_STR_DEFAULT);
        cur += sizeof(FOLDER_TYPE_STR_DEFAULT);
      }

      if ((type & ETPAN_FOLDER_TYPE_SENT) != 0) {
        snprintf(type_str, type_len, " %s", FOLDER_TYPE_STR_SENT);
        type_len -= sizeof(FOLDER_TYPE_STR_SENT);
        cur += sizeof(FOLDER_TYPE_STR_SENT);
      }

      if ((type & ETPAN_FOLDER_TYPE_DRAFT) != 0) {
        snprintf(type_str, type_len, " %s", FOLDER_TYPE_STR_DRAFT);
        type_len -= sizeof(FOLDER_TYPE_STR_DRAFT);
        cur += sizeof(FOLDER_TYPE_STR_DRAFT);
      }
      
      if (* type_str != '\0')
        snprintf(buffer, width - 20, "%s %s%s (%s)%s",
            thread_prefix, prefix, dsp_folder_name, type_str + 1, fill);
      else
        snprintf(buffer, width - 20, "%s %s%s%s",
            thread_prefix, prefix, dsp_folder_name, fill);
    }
    else {
      snprintf(buffer, width - 20, "%s %s%s%s",
          thread_prefix, prefix, dsp_folder_name, fill);
    }
  }
  
  if ((folder->fld_storage != NULL) &&
      ((state->flags & ETPAN_FOLDER_COMMON_SHOW_STAT) != 0)) {
    if (r) {
      snprintf(output, width, "%s %4i %4i %4i%s",
          buffer, num, recent, unseen, fill);
    }
    else
      snprintf(output, width, "%s (no information)%s",
          buffer, fill);
  }
  else {
    snprintf(output, width, "%s%s", buffer, fill);
  }

  if (decoded_folder_name != NULL)
    free(decoded_folder_name);
}

/* this function will display the list of messages */

static void draw_line(struct etpan_subapp * subapp,
    struct etpan_folder_common_app_state * state,
    WINDOW * w, char * output, char * buffer, char * fill,
    char * help_str)
{
  unsigned int i;
  unsigned int count;
  unsigned int folders_count;
  unsigned int percent;
  unsigned int list_lines;
  
  list_lines = subapp->display_height - 1;
  
  /* list */
  
  wattron(w, state->main_attr);
  count = 0;
  if (state->root != NULL) {
    for(i = state->first ; count < list_lines ; i++) {
      struct mailfolder * folder;
      
      if (i >= carray_count(state->folder_tab))
        break;
      
      folder = carray_get(state->folder_tab, i);
      
      if (etpan_folder_is_visible(state->params, folder)) {
        snprint_folder_info(subapp->app, state,
            output, buffer, fill, subapp->display_width + 1, folder);
        
        if (i == state->selected) {
          wattroff(w, state->main_attr);
          wattron(w, state->selection_attr);
        }
        mvwaddstr(w, count, 0, output);
        if (i == state->selected) {
          wattroff(w, state->selection_attr);
          wattron(w, state->main_attr);
        }
        count ++;
      }
    }
  }
  
  while (count < list_lines) {
    mvwaddstr(w, count, 0, fill);
    count ++;
  }
  
  wattroff(w, state->main_attr);
  
  /* status bar */
  
  folders_count = carray_count(state->folder_tab);
  if (folders_count == 0)
    percent = 0;
  else if (folders_count == 1)
	percent = 100;
  else
    percent = state->selected * 100 / (folders_count-1);
  
  wattron(w, state->status_attr);
  if (help_str == NULL)
    snprintf(output, subapp->display_width + 1, " %3i %% | %i folders%s",
        percent, folders_count, fill);
  else
    snprintf(output, subapp->display_width + 1,
        " %3i %% | %i folders | %s%s",
        percent, folders_count, help_str, fill);
  mvwaddstr(w, subapp->display_height - 1, 0, output);
  wattroff(w, state->status_attr);
}

static int etpan_folder_common_update_view(struct etpan_subapp * subapp,
    struct etpan_folder_common_app_state * state)
{
  unsigned int count;
  unsigned int i;
  struct mailfolder * folder;
  unsigned int list_lines;
  
  list_lines = subapp->display_height - 1;
  
  if (state->folder_tab->len == 0)
    return NO_ERROR;
  
  if (state->selected >= state->folder_tab->len)
    state->selected = state->folder_tab->len - 1;
  
  folder = carray_get(state->folder_tab, state->selected);
  if (!etpan_folder_is_visible(state->params, folder)) {
    do {
      folder = folder->fld_parent;
      if (folder != NULL)
        etpan_folder_set_opened(state->params, folder, TRUE);
    } while (folder != NULL);
  }
    
  if (state->window_size != list_lines) {
    uint32_t * buf;
    
    buf = realloc(state->selected_window_backward,
        list_lines * sizeof(* state->selected_window_backward));
    if (buf == NULL)
      return ERROR_MEMORY;
    state->selected_window_backward = buf;
    
    buf = realloc(state->selected_window_forward,
        list_lines * sizeof(* state->selected_window_backward));
    if (buf == NULL)
      return ERROR_MEMORY;
    state->selected_window_forward = buf;
    
    state->window_size = list_lines;
  }
  
  count = 0;
  for(i = state->selected ; count < list_lines ; i++) {
    if (i >= state->folder_tab->len)
      break;

    folder = carray_get(state->folder_tab, i);

    if (etpan_folder_is_visible(state->params, folder)) {
      state->selected_window_forward[count] = i;
      count ++;
    }
  }
  state->selected_window_forward_count = count;

  count = 0;
  i = state->selected;
  while (count < list_lines) {
    if (i >= state->folder_tab->len)
      break;
    
    folder = carray_get(state->folder_tab, i);

    if (etpan_folder_is_visible(state->params, folder)) {
      state->selected_window_backward[count] = i;
      count ++;
    }

    if (i == 0)
      break;

    i --;
  }
  state->selected_window_backward_count = count;

  if (state->first <
      state->selected_window_backward[state->selected_window_backward_count - 1])
    state->first =
      state->selected_window_backward[state->selected_window_backward_count - 1];
  
  if (state->selected < state->first)
    state->first = state->selected;
  
  return NO_ERROR;
}

int etpan_folder_common_display(struct etpan_subapp * app,
    struct etpan_folder_common_app_state * state, WINDOW * w,
    char * help_str)
{
  etpan_folder_common_update_view(app, state);
  
  draw_line(app, state, w,
      app->app->output, app->app->buffer, app->app->fill,
      help_str);
  
  return NO_ERROR;
}


static int auto_state_folder(struct etpan_folder_common_app_state * state,
    struct mailfolder * folder)
{
  unsigned int i;
  int r;
  uint32_t num;
  uint32_t recent;
  uint32_t unseen;
  int opened;

  opened = 0;

  for(i = 0 ; i < carray_count(folder->fld_children) ; i ++) {
    struct mailfolder *  child;

    child = carray_get(folder->fld_children, i);

    if (!opened) {
      r = etpan_folder_get_stat(state->params, child, &num, &recent, &unseen);
      if (r && (unseen != 0)) {
        etpan_folder_set_opened(state->params, folder, TRUE);
        opened = 1;
      }
    }

    if (auto_state_folder(state, child)) {
      if (!opened) {
        etpan_folder_set_opened(state->params, folder, TRUE);
        opened = 1;
      }
    }
  }

  if (!opened)
    etpan_folder_set_opened(state->params, folder, FALSE);

  return opened;
}

static void
etpan_set_folder_state(struct etpan_folder_common_app_state * state)
{
  auto_state_folder(state, state->root);
  etpan_folder_set_opened(state->params, state->root, TRUE);
}

static int
etpan_folder_expand_node(struct etpan_folder_common_app_state * state,
    struct mailfolder * folder)
{
  unsigned int i;

  etpan_folder_set_opened(state->params, folder, TRUE);

  for(i = 0 ; i < carray_count(folder->fld_children) ; i ++) {
    struct mailfolder * child;

    child = carray_get(folder->fld_children, i);
    
    etpan_folder_expand_node(state, child);
  }
  
  return NO_ERROR;
}

static int
etpan_folder_common_display_precalc_sub(struct
    etpan_folder_common_app_state * state,
    MMAPString * prefix, struct mailfolder * folder,
    int level, int has_next)
{
  int r;
  
  chashdatum key;
  chashdatum value;

  r = carray_add(state->folder_tab, folder, NULL);
  if (r < 0)
    return ERROR_MEMORY;

  key.data = &folder;
  key.len = sizeof(folder);

  value.data = prefix->str;
  value.len = prefix->len + 1;

  r = chash_set(state->prefix_hash, &key, &value, NULL);
  if (r < 0)
    return ERROR_MEMORY;

  if (carray_count(folder->fld_children) > 0) {
    char old_prefix[2];
    unsigned int i;

    if (level >= 1) {
      memcpy(old_prefix, prefix->str + prefix->len - 2, 2);
      if (has_next)
        memcpy(prefix->str + prefix->len - 2, "| ", 2);
      else
        memcpy(prefix->str + prefix->len - 2, "  ", 2);
    }

    for(i = 0 ; i < carray_count(folder->fld_children) ; i ++) {
      struct mailfolder * child;
      int sub_has_next;

      child = carray_get(folder->fld_children, i);

      if (i != carray_count(folder->fld_children) - 1) {
	if (level >= 0) {
	  if (mmap_string_append(prefix, "+-") == NULL)
            return ERROR_MEMORY;
        }
	sub_has_next = TRUE;
      }
      else {
	if (level >= 0) {
	  if (mmap_string_append(prefix, "\\-") == NULL)
            return ERROR_MEMORY;
        }
	sub_has_next = FALSE;
      }

      r = etpan_folder_common_display_precalc_sub(state,
          prefix, child, level + 1, sub_has_next);
      if (r != NO_ERROR)
        return r;

      if (mmap_string_truncate(prefix, prefix->len - 2) == NULL) {
        return ERROR_MEMORY;
      }
    }
    if (level >= 1) {
      memcpy(prefix->str + prefix->len - 2, old_prefix, 2);
    }
  }

  return NO_ERROR;
}

static int
etpan_folder_common_display_precalc(struct
    etpan_folder_common_app_state * state)
{
  MMAPString * prefix;

  prefix = mmap_string_new("");
  if (prefix == NULL)
    return ERROR_MEMORY;

  if (state->root != NULL)
    etpan_folder_common_display_precalc_sub(state, prefix,
        state->root, 0, FALSE);
  
  mmap_string_free(prefix);

  return NO_ERROR;
}

static void
etpan_folder_common_display_precalc_free(struct
    etpan_folder_common_app_state * state)
{
  chash_clear(state->prefix_hash);
  
  carray_set_size(state->folder_tab, 0);
}


struct mailfolder *
etpan_folder_common_get_selected_folder(struct
    etpan_folder_common_app_state * state)
{
  if (state->root == NULL)
    return NULL;
  
  if (state->selected >= carray_count(state->folder_tab))
    return NULL;
  
  return carray_get(state->folder_tab, state->selected);
}


static int next_unread_folder(struct etpan_folder_common_app_state * state)
{
  unsigned int i;
  int r;
  
  for (i = state->selected + 1 ; i < carray_count(state->folder_tab) ; i ++) {
    struct mailfolder * folder;
    uint32_t num;
    uint32_t recent;
    uint32_t unseen;
    
    folder = carray_get(state->folder_tab, i);
    
    if (folder->fld_storage != NULL) {
      r = etpan_folder_get_stat(state->params, folder,
          &num, &recent, &unseen);
      if (r && (unseen != 0)) {
        state->selected = i;
        return NO_ERROR;
      }
    }
  }
  
  return ERROR_INVAL;
}

int
etpan_folder_common_next_unread_folder(struct
    etpan_folder_common_app_state * state)
{
  return next_unread_folder(state);
}

void etpan_folder_common_set_node_visible(struct
    etpan_folder_common_app_state * state,
    struct mailfolder * folder)
{
  do {
    if (etpan_folder_is_visible(state->params, folder))
      break;
    
    folder = folder->fld_parent;
    if (folder != NULL)
      etpan_folder_set_opened(state->params, folder, TRUE);
    
  } while (folder != NULL);
}

void etpan_folder_common_set_stat(struct
    etpan_folder_common_app_state * state,
    struct mailfolder * folder,
    int enabled, uint32_t number, uint32_t recent, uint32_t unseen)
{
  etpan_folder_set_stat(state->params, folder, enabled,
      number, recent, unseen);
}

int etpan_folder_common_get_stat(struct
    etpan_folder_common_app_state * state,
    struct mailfolder * folder,
    uint32_t * number, uint32_t * recent, uint32_t * unseen)
{
  return etpan_folder_get_stat(state->params, folder, 
      number, recent, unseen);
}

static void search_upcall(struct etpan_subapp * search_app, int valid,
    void * data);

static int search_folder(struct etpan_subapp * app,
    struct etpan_folder_common_app_state * state)
{
  struct etpan_subapp * search_app;
  int r;
  
  state->app = app;
  
  search_app = etpan_app_find_subapp(app->app, "search-input",
      0, NULL, NULL);
  if (search_app == NULL) {
    search_app = etpan_search_input_new(app->app);
    if (search_app == NULL)
      return ERROR_MEMORY;
  }
  
  etpan_subapp_set_parent(search_app, app);
  
  r = etpan_search_input_set(search_app, "search: ", 256, state->search_str,
      0, search_upcall, state);
  if (r != NO_ERROR)
    return r;
  
  etpan_app_switch_subapp(search_app, 0);

  return NO_ERROR;
}

static void search_upcall(struct etpan_subapp * search_app, int valid,
    void * data)
{
  char * val;
  struct etpan_subapp * app;
  regex_t reg;
  unsigned int i;
  int found;
  struct etpan_folder_common_app_state * state;
  int r;
  
  state = data;
  app = state->app;
  
  if (valid == ETPAN_INPUT_COMMON_CANCEL) {
    ETPAN_APP_LOG((app->app, "cancel search"));
    goto err;
  }
  
  val = etpan_search_input_get_value(search_app);
  
  /* keep search string */
  if (state->search_str != NULL)
    free(state->search_str);
  state->search_str = strdup(val);
  
  if (regcomp(&reg, val, REG_EXTENDED | REG_ICASE) != 0) {
    ETPAN_APP_LOG((app->app, "error search - error in regular expression"));
    goto err;
  }
  
  found = 0;
  for(i = state->selected + 1 ; i < carray_count(state->folder_tab) ; i ++) {
    struct mailfolder * folder;
    
    folder = carray_get(state->folder_tab, i);
    
    if (folder->fld_virtual_name == NULL)
      continue;
    
    r = regexec(&reg, folder->fld_virtual_name, 0, NULL, 0);
    if (r == 0) {
      found = 1;
      state->selected = i;
      break;
    }
  }
  
  regfree(&reg);
  
  if (!found) {
    ETPAN_APP_LOG((app->app, "error search - could not find"));
  }
  
  etpan_app_quit_subapp(search_app);
  
  return;
  
 err:
  etpan_app_quit_subapp(search_app);
  return;
}
