/*
 * etPan! -- a mail user agent
 *
 * Copyright (C) 2001, 2002 - DINH Viet Hoa
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the libEtPan! project nor the names of its
 *    contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

/*
 * $Id: etpan-header-editor-app.c,v 1.14 2003/11/24 20:04:18 hoa Exp $
 */

#include "etpan-header-editor-app.h"

#include "etpan-app-subapp.h"
#include "etpan-subapp.h"
#include "etpan-errors.h"
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <ncurses.h>
#include "etpan-imf-helper.h"
#include "etpan-address-input.h"
#include "etpan-search-input.h"
#include "etpan-header-input.h"
#include "etpan-app.h"
#include "etpan-msg-new.h"
#include "etpan-subapp-thread.h"
#include "etpan-mime-tools.h"
#include "etpan-header-common.h"
#include "etpan-help-viewer.h"

static void handle_key(struct etpan_subapp * app, int key);
static void display(struct etpan_subapp * app, WINDOW * w);
static void set_color(struct etpan_subapp * app);
static int init(struct etpan_subapp * subapp);
static void done(struct etpan_subapp * subapp);
static void leave(struct etpan_subapp * app, struct etpan_subapp * new_app);
static int display_init(struct etpan_subapp * app);

static struct etpan_subapp_driver etpan_header_editor_app_driver = {
  .name = "header-editor",
  .always_handle_key = 0,
  .always_on_top = 0,
  .get_idle_delay = NULL,
  .idle = NULL,
  .set_fd = NULL,
  .handle_fd = NULL,
  .handle_key = handle_key,
  .handle_resize = NULL,
  .display = display,
  .set_color = set_color,
  .init = init,
  .done = done,
  .enter = NULL,
  .leave = leave,
  .display_init = display_init,
  .display_done = NULL,
};


struct app_state {
  struct etpan_header_common_state common_state;
};

static void set_color(struct etpan_subapp * app)
{
  struct app_state * state;
  
  state = app->data;
  
  etpan_header_common_set_color(app, &state->common_state);
}

static int show_help(struct etpan_subapp * app);

static void handle_key(struct etpan_subapp * app, int key)
{
  struct app_state * state;
  
  state = app->data;

  etpan_header_common_handle_key(app, &state->common_state, key);
  
  switch (key) {
  case 'q':
    etpan_header_common_fields_adjust(app, &state->common_state);
    
    etpan_app_quit_subapp(app);
    break;

  case KEY_F(1):
  case '?':
    show_help(app);
    break;
  }
}


static int display_init(struct etpan_subapp * app)
{
  etpan_subapp_set_title(app, "etPan! - header fields editor");
  return etpan_app_subapp_display_init(app);
}

static void display(struct etpan_subapp * app, WINDOW * w)
{
  struct app_state * state;
  
  state = app->data;
  
  etpan_header_common_display(app, &state->common_state, w,
      "q: finished");
}

static int init(struct etpan_subapp * subapp)
{
  struct app_state * state;
  int r;

  state = malloc(sizeof(* state));
  if (state == NULL)
    goto err;
  
  subapp->data = state;
  
  r = etpan_common_header_init(subapp, &state->common_state);
  if (r != NO_ERROR)
    goto free;
  
  return NO_ERROR;
  
 free:
  free(state);
 err:
  return ERROR_MEMORY;
}

static void done(struct etpan_subapp * subapp)
{
  struct app_state * state;
  
  state = subapp->data;
  
  etpan_header_common_done(subapp, &state->common_state);
  free(state);
}


void etpan_header_editor_app_flush(struct etpan_subapp * app)
{
  struct app_state * state;
  
  state = app->data;
  
  etpan_header_common_flush(app, &state->common_state);
}

static void leave(struct etpan_subapp * app, struct etpan_subapp * new_app)
{
  etpan_header_editor_app_flush(app);
}

void etpan_header_editor_app_set(struct etpan_subapp * app,
    struct mailimf_fields * fields)
{
  struct app_state * state;
  
  state = app->data;
  
  etpan_header_editor_app_flush(app);
  
  etpan_header_common_set_fields(app, &state->common_state,
      fields);
}

struct mailimf_fields *
etpan_header_editor_app_get_fields(struct etpan_subapp * app)
{
  struct app_state * state;
  
  state = app->data;
  
  return etpan_header_common_get_fields(app, &state->common_state);
}


struct etpan_subapp * etpan_header_editor_app_new(struct etpan_app * app)
{
  return etpan_subapp_new(app, &etpan_header_editor_app_driver);
}


#define HELP_TEXT \
"\
Help for header fields editor\n\
-----------------------------\n\
\n\
The application to headers will let you edit the selected\n\
header of the message.\n\
\n\
- Ctrl-W,\n\
  Ctrl-X     : switch between applications\n\
\n\
- arrow keys : move cursor\n\
\n\
- f          : adds From field   \n\
- t          : adds To field\n\
- c          : adds Cc field\n\
- b          : adds Bcc field\n\
- r          : adds Reply-To field\n\
- F          : adds Resent-From field\n\
- T          : adds Resent-To field\n\
- C          : adds Resent-Cc field\n\
- B          : adds Resent-Bcc field\n\
- s          : adds subject field\n\
- o          : adds custom field\n\
\n\
- [Enter]    : edit current field\n\
- d          : delete current field\n\
\n\
- q          : finished\n\
\n\
- ?          : help\n\
- Ctrl-L     : Console log\n\
\n\
(? or q to exit help)\n\
"

static int show_help(struct etpan_subapp * app)
{
  return etpan_show_help(app, HELP_TEXT, sizeof(HELP_TEXT) - 1);
}
