/*
 * etPan! -- a mail user agent
 *
 * Copyright (C) 2001, 2002 - DINH Viet Hoa
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the libEtPan! project nor the names of its
 *    contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

/*
 * $Id: etpan-help-viewer.c,v 1.3 2003/11/24 20:04:19 hoa Exp $
 */

#include "etpan-help-viewer.h"

#include "etpan-app.h"
#include "etpan-app-subapp.h"
#include "etpan-subapp.h"
#include "etpan-errors.h"
#include <stdlib.h>
#include <ncurses.h>
#include "etpan-viewer-common.h"

static void handle_key(struct etpan_subapp * app, int key);
static void display(struct etpan_subapp * app, WINDOW * w);
static void set_color(struct etpan_subapp * app);
static int init(struct etpan_subapp * subapp);
static void done(struct etpan_subapp * subapp);
static void leave(struct etpan_subapp * app, struct etpan_subapp * new_app);
static int display_init(struct etpan_subapp * app);

static struct etpan_subapp_driver etpan_help_viewer_app_driver = {
  .name = "help-viewer",
  .always_handle_key = 0,
  .always_on_top = 1,
  .get_idle_delay = NULL,
  .idle = NULL,
  .set_fd = NULL,
  .handle_fd = NULL,
  .handle_key = handle_key,
  .handle_resize = NULL,
  .display = display,
  .set_color = set_color,
  .init = init,
  .done = done,
  .enter = NULL,
  .leave = leave,
  .display_init = display_init,
  .display_done = NULL,
};

struct app_state {
  struct etpan_viewer_common_app_state common_state;
};

static void handle_key(struct etpan_subapp * app, int key)
{
  struct app_state * state;
  
  state = app->data;
  
  etpan_viewer_common_handle_key(app, &state->common_state, key);
  
  switch (key) {
  case KEY_F(1):
  case '?':
  case 'q':
    etpan_app_quit_subapp(app);
    break;
  }
}

static int display_init(struct etpan_subapp * app)
{
  etpan_subapp_set_title(app, "etPan! - help");
  return etpan_app_subapp_display_init(app);
}

static void set_color(struct etpan_subapp * app)
{
  struct app_state * state;
  
  state = app->data;
  
  etpan_viewer_common_set_color(app, &state->common_state);
}

static void display(struct etpan_subapp * app, WINDOW * w)
{
  struct app_state * state;
  
  state = app->data;
  
  etpan_viewer_common_display(app, &state->common_state, w, "q: exit help");
}

static int init(struct etpan_subapp * subapp)
{
  struct app_state * state;
  int r;
  
  state = malloc(sizeof(* state));
  if (state == NULL)
    goto err;
  
  r = etpan_viewer_common_init(subapp, &state->common_state);
  if (r != NO_ERROR)
    goto free;
  
  subapp->data = state;
  
  return NO_ERROR;
  
 free:
  free(state);
 err:
  return ERROR_MEMORY;
}

static void done(struct etpan_subapp * subapp)
{
  struct app_state * state;
  
  state = subapp->data;
  /* do nothing */
  etpan_viewer_common_done(subapp, &state->common_state);
  free(state);
}

struct etpan_subapp * etpan_help_viewer_app_new(struct etpan_app * app)
{
  return etpan_subapp_new(app, &etpan_help_viewer_app_driver);
}

static void leave(struct etpan_subapp * app, struct etpan_subapp * new_app)
{
  struct app_state * state;
  
  state = app->data;

  etpan_help_viewer_app_flush(app);
}

int etpan_help_viewer_app_set(struct etpan_subapp * app,
    char * text, size_t size, carray * attr)
{
  struct app_state * state;
  int r;
  int res;
  
  state = app->data;
  
  etpan_help_viewer_app_flush(app);
  
  r = etpan_viewer_common_set_message(&state->common_state,
      NULL, NULL, text, size,
      attr, 0);
  if (r != NO_ERROR) {
    res = r;
    goto err;
  }

  return NO_ERROR;

 err:
  return res;
}

void etpan_help_viewer_app_flush(struct etpan_subapp * app)
{
  struct app_state * state;
  
  state = app->data;
  
  etpan_viewer_common_flush(&state->common_state);
}


int etpan_show_help(struct etpan_subapp * app, char * text, size_t size)
{
  struct etpan_subapp * help_app;
  
  help_app = etpan_app_find_subapp(app->app, "help-viewer",
      0, NULL, NULL);
  if (help_app == NULL) {
    help_app = etpan_help_viewer_app_new(app->app);
    if (help_app == NULL) {
      ETPAN_APP_LOG((app->app, "coult not show help - not enough memory"));
      return ERROR_MEMORY;
    }
  }
  
  etpan_help_viewer_app_set(help_app, text, size, NULL);
  
  etpan_subapp_set_parent(help_app, app);
  etpan_app_switch_subapp(help_app, 0);
  
  return NO_ERROR;
}
