/*
 * etPan! -- a mail user agent
 *
 * Copyright (C) 2001, 2002 - DINH Viet Hoa
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the libEtPan! project nor the names of its
 *    contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

/*
 * $Id: etpan-msg-reply.c,v 1.14 2004/09/23 23:54:08 hoa Exp $
 */

#include "etpan-msg-reply.h"

#include <libetpan/libetpan.h>
#include "etpan-errors.h"
#include <string.h>
#include <stdlib.h>
#include "etpan-imf-helper.h"
#include "etpan-cfg-vfolder.h"
#include "etpan-msg-new.h"
#include "etpan-header-editor-app.h"
#if 0
#include "etpan-mime-message-driver.h"
#endif
#include "etpan-subapp-thread.h"
#include <unistd.h>
#include "etpan-thread-manager.h"
#include "etpan-app.h"
#include "etpan-app-subapp.h"
#include "etpan-mime-tools.h"

/* work with headers */

static int get_in_reply_to(struct mailimf_single_fields * fields,
    clist ** result)
{
  clist * in_reply_to;
  char * msgid;
  int r;

  if (fields->fld_message_id == NULL) {
    * result = NULL;

    return NO_ERROR;
  }
    
  in_reply_to = clist_new();
  if (in_reply_to == NULL)
    goto err;
  
  msgid = fields->fld_message_id->mid_value;
  
  msgid = strdup(msgid);
  if (msgid == NULL)
    goto free_list;
  
  r = clist_append(in_reply_to, msgid);
  if (r < 0) {
    free(msgid);
    goto free_list;
  }
  
  * result = in_reply_to;

  return NO_ERROR;

 free_list:
  clist_foreach(in_reply_to, (clist_func) free, NULL);
  clist_free(in_reply_to);
 err:
  return ERROR_MEMORY;
}

static int get_references(struct mailimf_single_fields * fields,
    clist ** result)
{
  char * msgid;
  clist * new_references;
  int r;

  msgid = NULL;

  if ((fields->fld_message_id == NULL) && (fields->fld_references == NULL)) {
    * result = NULL;

    return NO_ERROR;
  }

  new_references = clist_new();
  if (new_references == NULL)
    goto err;

  if (fields->fld_message_id != NULL)
    msgid = fields->fld_message_id->mid_value;
  
  if (fields->fld_references != NULL) {
    clist * references;
    clistiter * ref_cur;
    
    references = fields->fld_references->mid_list;
    
    for(ref_cur = clist_begin(references) ;
        ref_cur != NULL ; ref_cur = clist_next(ref_cur)) {
      char * ref_id;
      
      ref_id = clist_content(ref_cur);
      ref_id = strdup(ref_id);
      if (ref_id == NULL)
        goto free_list;

      r = clist_append(new_references, ref_id);
      if (r < 0) {
        free(ref_id);
        goto free_list;
      }
    }
  }

  if (msgid != NULL) {
    msgid = strdup(msgid);
    if (msgid == NULL)
      goto free_list;

    r = clist_append(new_references, msgid);
    if (r < 0) {
      free(msgid);
      goto free_list;
    }
  }

  * result = new_references;

  return NO_ERROR;

 free_list:
  clist_foreach(new_references, (clist_func) free, NULL);
  clist_free(new_references);
 err:
  return ERROR_MEMORY;
}

static char * get_reply_subject(struct mailimf_single_fields * fields)
{
  char * str;
  char * newstr;

  if (fields->fld_subject != NULL)
    str = fields->fld_subject->sbj_value;
  else
    str = "";
  
  while (* str == ' ')
    str ++;
  
  if (strncasecmp(str, "Re: ", 4) != 0) {
    newstr = malloc(strlen(str) + 5);
    if (newstr == NULL)
      return NULL;
    
    strcpy(newstr, "Re: ");
    strcat(newstr, str);
  }
  else {
    newstr = strdup(str);
    if (newstr == NULL)
      return NULL;
  }
  
  return newstr;
}


static int
get_reply_address(struct mailimf_single_fields * single_fields,
    struct mailimf_address_list ** result)
{
  struct mailimf_address_list * addr_list;

  if (single_fields->fld_from == NULL) {
    * result = NULL;
    return NO_ERROR;
  }

  addr_list = etpan_mailbox_to_address_list(single_fields->fld_from->frm_mb_list);
  if (addr_list == NULL)
    return ERROR_MEMORY;

  * result = addr_list;

  return NO_ERROR;
}

static int
get_followup_address(struct mailimf_single_fields * single_fields,
    struct mailimf_fields * fields,
    struct mailimf_address_list ** result)
{
  struct mailimf_address_list * addr_list;
  clistiter * cur;
  int r;

  if (single_fields->fld_reply_to != NULL) {
    addr_list = etpan_dup_address_list(single_fields->fld_reply_to->rt_addr_list);
    if (addr_list == NULL)
      goto err;

    * result = addr_list;

    return NO_ERROR;
  }
  
  for(cur = clist_begin(fields->fld_list) ; cur != NULL ;
      cur = clist_next(cur)) {
    struct mailimf_field * field;

    field = clist_content(cur);

    if (field->fld_type == MAILIMF_FIELD_OPTIONAL_FIELD) {
      if (strcasecmp(field->fld_data.fld_optional_field->fld_name,
              "List-Post") == 0) {
        if (strncasecmp(field->fld_data.fld_optional_field->fld_value,
                "<mailto:", 8) == 0) {
          size_t cur_token;

          cur_token = 8;
          r = mailimf_address_list_parse(field->fld_data.fld_optional_field->fld_value,
              strlen(field->fld_data.fld_optional_field->fld_value),
              &cur_token, &addr_list);
          if (r == MAILIMF_NO_ERROR) {
            * result = addr_list;
            return NO_ERROR;
          }
        }
      }
    }
  }

  r = get_reply_address(single_fields, &addr_list);
  if (r != NO_ERROR)
    goto err;

  * result = addr_list;
  
  return NO_ERROR;

 err:
  return ERROR_MEMORY;
}

static char * get_followup_newsgroups(struct mailimf_fields * fields)
{
  clistiter * cur;
  
  for(cur = clist_begin(fields->fld_list) ; cur != NULL ;
      cur = clist_next(cur)) {
    struct mailimf_field * field;

    field = clist_content(cur);

    if (field->fld_type == MAILIMF_FIELD_OPTIONAL_FIELD) {
      if (strcasecmp(field->fld_data.fld_optional_field->fld_name,
              "Followup-to") == 0)
	return field->fld_data.fld_optional_field->fld_value;
    }
  }

  for(cur = clist_begin(fields->fld_list) ; cur != NULL ;
      cur = clist_next(cur)) {
    struct mailimf_field * field;

    field = clist_content(cur);

    if (field->fld_type == MAILIMF_FIELD_OPTIONAL_FIELD) {
      if (strcasecmp(field->fld_data.fld_optional_field->fld_name,
              "Newsgroups") == 0)
	return field->fld_data.fld_optional_field->fld_value;
    }
  }

  return NULL;
}


static int set_newsgroups_field(struct mailimf_fields * new_fields,
    struct mailimf_fields * fields)
{
  struct mailimf_optional_field * opt_field;
  struct mailimf_field * field;
  char * addr;
  int r;
  int res;
  char * name;
  char * value;

  addr = get_followup_newsgroups(fields);
  
  if (addr == NULL) {
    res = MAIL_ERROR_INVAL;
    goto err;
  }
  
  name = strdup("Newsgroups");
  if (name == NULL) {
    res = MAIL_ERROR_MEMORY;
    goto err;
  }

  value = strdup(addr);
  if (value == NULL) {
    res = MAIL_ERROR_MEMORY;
    goto free_name;
  }
  
  opt_field = mailimf_optional_field_new(name, value);
  if (opt_field == NULL) {
    res = MAIL_ERROR_MEMORY;
    goto free_value;
  }

  field = mailimf_field_new(MAILIMF_FIELD_OPTIONAL_FIELD,
      NULL, NULL, NULL, NULL, NULL, NULL, NULL,
      NULL, NULL, NULL, NULL,
      NULL, NULL, NULL, NULL,
      NULL, NULL, NULL, NULL,
      NULL, NULL, opt_field);
  if (field == NULL) {
    res = MAIL_ERROR_MEMORY;
    goto free_opt_field;
  }

  r = clist_append(new_fields->fld_list, field);
  if (r < 0) {
    res = MAIL_ERROR_MEMORY;
    goto free_field;
  }

  return MAIL_NO_ERROR;

 free_field:
  mailimf_field_free(field);
  goto err;
 free_opt_field:
  mailimf_optional_field_free(opt_field);
  goto err;
 free_value:
  free(value);
 free_name:
  free(name);
 err:
  return res;
}




int etpan_followup_to_get_fields(struct etpan_app * app,
    struct mailfolder * folder,
    struct mailimf_fields * fields,
    struct mailimf_fields ** result)
{
  struct mailimf_single_fields single_fields;
  struct mailimf_fields * new_fields;
  struct mailimf_mailbox_list * from;
  struct mailimf_address_list * to;
  clist * in_reply_to;
  clist * references;
  char * subject;
  int nntp_protocol;
  struct etpan_account_info * account;
  int r;
  
  nntp_protocol = etpan_is_nntp_protocol(folder);
  
  account = etpan_vfolder_get_account(app->config.vfolder_config,
      app->config.account_config, folder);
  
  mailimf_single_fields_init(&single_fields, fields);
  
  /* build headers */

  if (nntp_protocol)
    to = NULL;
  else {
    r = get_followup_address(&single_fields, fields, &to);
    if (r != NO_ERROR)
      goto err;
  }

  if (account == NULL) {
    from = NULL;
  }
  else {
    from = etpan_get_from_field(app, account);
    if (from == NULL)
      goto free_to;
  }

  r = get_in_reply_to(&single_fields, &in_reply_to);
  if (r != NO_ERROR) {
    goto free_from;
  }

  r = get_references(&single_fields, &references);
  if (r != NO_ERROR) {
    goto free_in_reply_to;
  }

  subject = get_reply_subject(&single_fields);
  if (subject == NULL)
    goto free_references;

  new_fields = mailimf_fields_new_with_data(from, /* from */
      NULL /* sender */, NULL /* reply-to */, to,
      NULL, NULL, in_reply_to, references, subject);
  if (new_fields == NULL)
    goto free_subject;

  if (nntp_protocol) {
    r = set_newsgroups_field(new_fields, fields);
    if (r != NO_ERROR) {
      mailimf_fields_free(fields);
      goto free_new_fields;
    }
  }

  r = etpan_set_xmailer(new_fields, nntp_protocol);
  if (r != NO_ERROR) {
    goto free_new_fields;
  }

  * result = new_fields;

  return NO_ERROR;

 free_new_fields:
  mailimf_fields_free(new_fields);
  goto err;
 free_subject:
  free(subject);
 free_references:
  if (references != NULL) {
    clist_foreach(references, (clist_func) free, NULL);
    clist_free(references);
  }
 free_in_reply_to:
  if (in_reply_to != NULL) {
    clist_foreach(in_reply_to, (clist_func) free, NULL);
    clist_free(in_reply_to);
  }
 free_from:
  if (from != NULL)
    mailimf_mailbox_list_free(from);
 free_to:
  if (to != NULL)
    mailimf_address_list_free(to);
 err:
  return ERROR_MEMORY;;
}


int etpan_reply_all_get_fields(struct etpan_app * app,
    struct mailfolder * folder,
    struct mailimf_fields * fields,
    struct mailimf_fields ** result)
{
  struct mailimf_single_fields single_fields;
  struct mailimf_fields * new_fields;
  struct mailimf_mailbox_list * from;
  struct mailimf_address_list * to;
  struct mailimf_address_list * cc;
  clist * in_reply_to;
  clist * references;
  char * subject;
  int nntp_protocol;
  struct etpan_account_info * account;
  int r;

  nntp_protocol = etpan_is_nntp_protocol(folder);

  account = etpan_vfolder_get_account(app->config.vfolder_config,
      app->config.account_config, folder);
  
  mailimf_single_fields_init(&single_fields, fields);

  /* build headers */
  
  if (nntp_protocol)
    to = NULL;
  else {
    r = get_followup_address(&single_fields, fields, &to);
    if (r != NO_ERROR)
      goto err;
  }

  r = etpan_get_to_cc_address(&single_fields, &cc);
  if (r != NO_ERROR) {
    goto free_to;
  }
  
  if (account == NULL)
    from = NULL;
  else {
    from = etpan_get_from_field(app, account);
    if (from == NULL)
      goto free_cc;
  }
  
  r = get_in_reply_to(&single_fields, &in_reply_to);
  if (r != NO_ERROR)
    goto free_from;
  
  r = get_references(&single_fields, &references);
  if (r != NO_ERROR)
    goto free_in_reply_to;

  subject = get_reply_subject(&single_fields);
  if (subject == NULL)
    goto free_references;

  new_fields = mailimf_fields_new_with_data(from, /* from */
					    NULL, /* sender */
					    NULL, /* reply-to */
					    to,
					    cc,
					    NULL,
					    in_reply_to,
					    references,
					    subject);
  if (new_fields == NULL)
    goto free_subject;

  if (nntp_protocol) {
    r = set_newsgroups_field(new_fields, fields);
    if (r != NO_ERROR) {
      goto free_new_fields;
    }
  }

  r = etpan_set_xmailer(new_fields, nntp_protocol);
  if (r != NO_ERROR)
    goto free_new_fields;

  * result = new_fields;

  return NO_ERROR;

 free_new_fields:
  mailimf_fields_free(new_fields);
  goto err;
 free_subject:
  free(subject);
 free_references:
  if (references != NULL) {
    clist_foreach(references, (clist_func) free, NULL);
    clist_free(references);
  }
 free_in_reply_to:
  if (in_reply_to != NULL) {
    clist_foreach(in_reply_to, (clist_func) free, NULL);
    clist_free(in_reply_to);
  }
 free_from:
  if (from != NULL)
    mailimf_mailbox_list_free(from);
 free_cc:
  if (cc != NULL)
    mailimf_address_list_free(cc);
 free_to:
  if (to != NULL)
    mailimf_address_list_free(to);
 err:
  return ERROR_MEMORY;
}


int etpan_reply_to_get_fields(struct etpan_app * app,
    struct mailfolder * folder,
    struct mailimf_fields * fields,
    struct mailimf_fields ** result)
{
  struct mailimf_single_fields single_fields;
  struct mailimf_fields * new_fields;
  struct mailimf_mailbox_list * from;
  struct mailimf_address_list * to;
  clist * in_reply_to;
  clist * references;
  char * subject;
  int nntp_protocol;
  struct etpan_account_info * account;
  int r;

  nntp_protocol = FALSE;

  account = etpan_vfolder_get_account(app->config.vfolder_config,
      app->config.account_config, folder);

  mailimf_single_fields_init(&single_fields, fields);
  
  /* build headers */

  r = get_reply_address(&single_fields, &to);
  if (r != NO_ERROR)
    goto err;

  if (account == NULL)
    from = NULL;
  else {
    from = etpan_get_from_field(app, account);
    if (from == NULL)
      goto free_to;
  }

  r = get_in_reply_to(&single_fields, &in_reply_to);
  if (r != NO_ERROR)
    goto free_from;
  
  r = get_references(&single_fields, &references);
  if (r != NO_ERROR)
    goto free_in_reply_to;

  subject = get_reply_subject(&single_fields);
  if (subject == NULL)
    goto free_references;

  new_fields = mailimf_fields_new_with_data(from, /* from */
					    NULL, /* sender */
					    NULL, /* reply-to */
					    to,
					    NULL,
					    NULL,
					    in_reply_to,
					    references,
					    subject);
  if (new_fields == NULL)
    goto free_subject;

  if (nntp_protocol) {
    r = set_newsgroups_field(new_fields, fields);
    if (r != NO_ERROR) {
      goto free_new_fields;
    }
  }

  r = etpan_set_xmailer(new_fields, nntp_protocol);
  if (r != NO_ERROR)
    goto free_new_fields;

  * result = new_fields;

  return MAIL_NO_ERROR;

 free_new_fields:
  mailimf_fields_free(new_fields);
  goto err;
 free_subject:
  free(subject);
 free_references:
  if (references != NULL) {
    clist_foreach(references, (clist_func) free, NULL);
    clist_free(references);
  }
 free_in_reply_to:
  if (in_reply_to != NULL) {
    clist_foreach(in_reply_to, (clist_func) free, NULL);
    clist_free(in_reply_to);
  }
 free_from:
  if (from != NULL)
    mailimf_mailbox_list_free(from);
 free_to:
  if (to != NULL)
    mailimf_address_list_free(to);
 err:
  return ERROR_MEMORY;
}




static char * get_forward_subject(struct mailimf_single_fields * fields)
{
  char * str;
  char * newstr;

  if (fields->fld_subject != NULL)
    str = fields->fld_subject->sbj_value;
  else
    str = "";
  
  while (* str == ' ')
    str ++;
  
  newstr = malloc(strlen(str) + 5);
  if (newstr == NULL)
    return NULL;

  strcpy(newstr, "Fw: ");
  strcat(newstr, str);
  
  return newstr;
}

/*
  etpan_forward_get_fields()

  dest_addr is duplicated by this function.
*/


int etpan_forward_get_fields(struct etpan_app * app,
    struct mailfolder * folder,
    struct mailimf_fields * fields,
    struct etpan_account_info * account,
    struct mailimf_address_list * dest_addr,
    struct mailfolder * post_folder,
    struct mailimf_fields ** result)
{
  struct mailimf_fields * new_fields;
  struct mailimf_mailbox_list * from;
  struct mailimf_address_list * to;
  char * subject;
  clist * list;
  int r;
  struct mailimf_single_fields single_fields;
  int nntp_protocol;
  
  /* build headers */
  
  nntp_protocol = FALSE;
  
  if (post_folder) {
    nntp_protocol = etpan_is_nntp_protocol(post_folder);
  }

  to = NULL;
  if (dest_addr == NULL) {
    if (!nntp_protocol) {
      list = clist_new();
      to = mailimf_address_list_new(list);
      
      if (to == NULL)
        goto err;
    }
  }
  else {
    to = etpan_dup_address_list(dest_addr);
    if (to == NULL)
      goto err;
  }
  
  mailimf_single_fields_init(&single_fields, fields);
  
  if (account == NULL)
    account = etpan_vfolder_get_account(app->config.vfolder_config,
        app->config.account_config, folder);
  
  if (account == NULL) {
    from = NULL;
  }
  else {
    from = etpan_get_from_field(app, account);
    if (from == NULL)
      goto free_to;
  }
  
  subject = get_forward_subject(&single_fields);
  if (subject == NULL)
    goto free_from;
  
  new_fields = mailimf_fields_new_with_data(from, /* from */
					    NULL, /* sender */
					    NULL, /* reply-to */
					    to,
					    NULL,
					    NULL,
					    NULL,
					    NULL,
					    subject);
  if (new_fields == NULL)
    goto free_subject;
  
  if (nntp_protocol) {
    r = etpan_set_current_newsgroups_header(new_fields, post_folder);
    if (r != NO_ERROR)
      goto free_fields;
  }
  
  r = etpan_set_xmailer(new_fields, nntp_protocol);
  if (r != NO_ERROR)
    goto free_fields;

  * result = new_fields;

  return NO_ERROR;

 free_fields:
  mailimf_fields_free(new_fields);
  goto err;
 free_subject:
  free(subject);
 free_from:
  if (from != NULL)
    mailimf_mailbox_list_free(from);
 free_to:
  if (to != NULL)
    mailimf_address_list_free(to);
 err:
  return ERROR_MEMORY;
}


static void thread_render_reply_handle_cancel(struct etpan_subapp * app,
    struct etpan_thread_op * op, int app_op_type, void * data)
{
  etpan_queue_unref_msg(app, op->data.mailaccess.msg);
}


struct compose_reply_action_arg {
  struct etpan_subapp * app;
  struct mailfolder * folder;
  mailmessage * msg;
  int reply_type;
};

/* extracted/modified from etpan-msg-list-app.c - begin */

static int check_message(struct etpan_subapp * app,
    struct mailfolder * folder, mailmessage * msg)
{
  struct mailstorage * storage;
  
  storage = NULL;
  if (folder != NULL)
    storage = folder->fld_storage;
  
  return etpan_subapp_thread_op_add(app,
      THREAD_ID_COMPOSE_CHECK_MSG, ETPAN_THREAD_MESSAGE_CHECK,
      storage, folder,
      msg, NULL,
      NULL,
      NULL, NULL, NULL);
}

/* extracted/modified from etpan-msg-list-app.c - end */

static void compose_reply_action(void * data)
{
  struct compose_reply_action_arg * arg;
  struct etpan_subapp * app;
  struct mailfolder * folder;
  mailmessage * msg;
  int reply_type;
  int r;

  arg = data;
  
  app = arg->app;
  folder = arg->folder;
  msg = arg->msg;
  reply_type = arg->reply_type;
  free(arg);
  
  if (msg->msg_flags != NULL) {
    switch (reply_type) {
    case ETPAN_THREAD_REPLY:
    case ETPAN_THREAD_REPLY_ALL:
    case ETPAN_THREAD_REPLY_FOLLOWUP_TO:
      msg->msg_flags->fl_flags |= MAIL_FLAG_ANSWERED;
      break;
    case ETPAN_THREAD_REPLY_FORWARD:
    case ETPAN_THREAD_REPLY_FORWARD_AS_ATTACHMENT:
      msg->msg_flags->fl_flags |= MAIL_FLAG_FORWARDED;
      break;
    }
    r = check_message(app, folder, msg);
    if (r != NO_ERROR) {
      ETPAN_APP_LOG((app->app, "flag message - not enough memory"));
    }
  }
  
  etpan_queue_unref_msg(app, msg);
}

static void compose_reply_action_cancel(void * data)
{
  struct compose_reply_action_arg * arg;
  struct etpan_subapp * app;
  struct mailfolder * folder;
  mailmessage * msg;
  int reply_type;
  
  arg = data;
  
  app = arg->app;
  folder = arg->folder;
  msg = arg->msg;
  reply_type = arg->reply_type;
  free(arg);
  
  etpan_queue_unref_msg(app, msg);
}


static void thread_render_reply_callback(struct etpan_subapp * app,
    struct etpan_thread_op * op, int app_op_type, void * data)
{
  struct etpan_message_render_reply_result * result;
  struct etpan_message_render_reply_arg * arg;
  int r;
  struct mailfolder * post_folder;
  struct compose_reply_action_arg * reply_action_arg;
  int reply_type;
  
  /* free arg */
  arg = op->arg;
  
  reply_type = arg->reply_type;
  post_folder = arg->post_folder;
  /* free given address list */
  if (arg->dest_addr != NULL)
    mailimf_address_list_free(arg->dest_addr);
  free(arg);
  op->arg = NULL;
  
  /* handle end of operation */
  
  if (op->err != NO_ERROR) {
    mailmessage * msg;
        
    msg = op->data.mailaccess.msg;
    
    if (msg->msg_uid != NULL)
      ETPAN_APP_LOG((app->app,
                        "reply message - error while retrieving message %i, %s",
                        msg->msg_index, msg->msg_uid));
    else
      ETPAN_APP_LOG((app->app,
                        "reply message - error while retrieving message %i",
                        msg->msg_index));
    
    goto unref;
  }
  
  result = op->result;
  
  reply_action_arg = malloc(sizeof(* reply_action_arg));
  if (reply_action_arg == NULL) {
    ETPAN_APP_LOG((app->app, "reply message - not enough memory"));
    goto free_result;
  }
  
  reply_action_arg->app = app;
  reply_action_arg->folder = op->data.mailaccess.folder;
  reply_action_arg->msg = op->data.mailaccess.msg;
  reply_action_arg->reply_type = reply_type;
  
  r = etpan_compose(app, post_folder, result->mime, result->edit_part,
      result->edit_part_line, compose_reply_action, reply_action_arg,
      compose_reply_action_cancel);
  if (r != NO_ERROR) {
    ETPAN_APP_LOG((app->app, "reply message - not enough memory"));
    free(reply_action_arg);
    goto free_result;
  }
  
  free(result);
  op->result = NULL;
  
  return;
  
 free_result:
  etpan_message_mime_clear(result->mime);
  mailmime_free(result->mime);
  free(result);
  op->result = NULL;
 unref:
  etpan_queue_unref_msg(app, op->data.mailaccess.msg);
}



int etpan_reply_message(struct etpan_subapp * app, int reply_type,
    mailmessage * msg,
    struct mailmime * mime,
    struct etpan_account_info * account,
    struct mailimf_address_list * dest_addr,
    struct mailfolder * post_folder)
{
  struct etpan_message_render_reply_arg * arg;
  int r;
  int res;
  
  if (etpan_subapp_thread_has_match_op(app, THREAD_ID_COMPOSE_RENDER_REPLY,
          NULL, NULL, msg, mime)) {
    res = ERROR_BUSY;
    goto err;
  }
  
  if (msg->msg_uid != NULL)
    ETPAN_APP_LOG((app->app, "retrieving message %i, %s",
                      msg->msg_index, msg->msg_uid));
  else
    ETPAN_APP_LOG((app->app, "retrieving message %i", msg->msg_index));

  r = etpan_queue_ref_msg(app, msg);
  if (r != NO_ERROR) {
    ETPAN_APP_LOG((app->app, "reply message - not enough memory"));
    res = ERROR_MEMORY;
    goto err;
  }
  
  arg = malloc(sizeof(* arg));
  if (arg == NULL) {
    ETPAN_APP_LOG((app->app, "reply message - not enough memory"));
    res = ERROR_MEMORY;
    goto unref;
  }
  arg->reply_type = reply_type;
  arg->account = account;
  arg->dest_addr = dest_addr;
  arg->post_folder = post_folder;
  
  r = etpan_subapp_thread_msg_op_add(app, THREAD_ID_COMPOSE_RENDER_REPLY,
      ETPAN_THREAD_MESSAGE_RENDER_REPLY_MIME,
      msg, mime,
      arg,
      thread_render_reply_callback, NULL,
      thread_render_reply_handle_cancel);
  if (r != NO_ERROR) {
    free(arg);
    ETPAN_APP_LOG((app->app, "reply message - not enough memory"));
    res = ERROR_MEMORY;
    goto unref;
  }
  
  return NO_ERROR;

 unref:
  etpan_queue_unref_msg(app, msg);
 err:
  return ERROR_MEMORY;
}

