#include "etpan-cfg-sender.h"

#include <unistd.h>
#include <sys/types.h>
#include <sys/time.h>
#include <sys/resource.h>
#include <sys/wait.h>
#include <string.h>
#include <sys/stat.h>
#include <stdlib.h>
#include <sys/select.h>
#include <libetpan/libetpan.h>
#include <sys/mman.h>
#include <fcntl.h>

#include "etpan-errors.h"
#include "etpan-imf-helper.h"

static int error_from_smtp_error(int error)
{
  switch (error) {
  case MAILSMTP_NO_ERROR:
    return NO_ERROR;
    
  case MAILSMTP_ERROR_UNEXPECTED_CODE:
  case MAILSMTP_ERROR_SERVICE_NOT_AVAILABLE:
  case MAILSMTP_ERROR_STREAM:
    return ERROR_STREAM;
    
  case MAILSMTP_ERROR_HOSTNAME:
  case MAILSMTP_ERROR_NOT_IMPLEMENTED:
  case MAILSMTP_ERROR_ACTION_NOT_TAKEN:
  case MAILSMTP_ERROR_EXCEED_STORAGE_ALLOCATION:
  case MAILSMTP_ERROR_IN_PROCESSING:
  case MAILSMTP_ERROR_INSUFFICIENT_SYSTEM_STORAGE:
    return ERROR_INVAL;
    
  case MAILSMTP_ERROR_MAILBOX_UNAVAILABLE:
  case MAILSMTP_ERROR_MAILBOX_NAME_NOT_ALLOWED:
    return ERROR_INVAL;
    
  case MAILSMTP_ERROR_BAD_SEQUENCE_OF_COMMAND:
  case MAILSMTP_ERROR_USER_NOT_LOCAL:
  case MAILSMTP_ERROR_TRANSACTION_FAILED:
    return ERROR_INVAL;
    
  case MAILSMTP_ERROR_MEMORY:
    return ERROR_MEMORY;
    
  case MAILSMTP_ERROR_AUTH_NOT_SUPPORTED:
  case MAILSMTP_ERROR_AUTH_LOGIN:
  case MAILSMTP_ERROR_AUTH_REQUIRED:
  case MAILSMTP_ERROR_AUTH_TOO_WEAK:
  case MAILSMTP_ERROR_AUTH_TRANSITION_NEEDED:
  case MAILSMTP_ERROR_AUTH_TEMPORARY_FAILTURE:
  case MAILSMTP_ERROR_AUTH_ENCRYPTION_REQUIRED:
  case MAILSMTP_ERROR_STARTTLS_TEMPORARY_FAILURE:
  case MAILSMTP_ERROR_STARTTLS_NOT_SUPPORTED:
    return ERROR_AUTH;
    
  case MAILSMTP_ERROR_CONNECTION_REFUSED:
    return ERROR_CONNECT;
  default:
    return ERROR_INVAL;
  }
}

static void
mailimf_single_resent_fields_init(struct mailimf_single_fields * single_fields,
    struct mailimf_fields * fields)
{
  clistiter * cur;

  memset(single_fields, 0, sizeof(struct mailimf_single_fields));

  cur = clist_begin(fields->fld_list);
  while (cur != NULL) {
    struct mailimf_field * field;

    field = clist_content(cur);

    switch (field->fld_type) {
    case MAILIMF_FIELD_RESENT_DATE:
      if (single_fields->fld_orig_date == NULL)
        single_fields->fld_orig_date = field->fld_data.fld_resent_date;
      cur = clist_next(cur);
      break;
    case MAILIMF_FIELD_RESENT_FROM:
      if (single_fields->fld_from == NULL) {
        single_fields->fld_from = field->fld_data.fld_resent_from;
        cur = clist_next(cur);
      }
      else {
        clist_concat(single_fields->fld_from->frm_mb_list->mb_list,
                     field->fld_data.fld_resent_from->frm_mb_list->mb_list);
        mailimf_field_free(field);
        cur = clist_delete(fields->fld_list, cur);
      }
      break;
    case MAILIMF_FIELD_RESENT_SENDER:
      if (single_fields->fld_sender == NULL)
        single_fields->fld_sender = field->fld_data.fld_resent_sender;
      cur = clist_next(cur);
      break;
    case MAILIMF_FIELD_RESENT_TO:
      if (single_fields->fld_to == NULL) {
        single_fields->fld_to = field->fld_data.fld_resent_to;
        cur = clist_next(cur);
      }
      else {
        clist_concat(single_fields->fld_to->to_addr_list->ad_list,
                     field->fld_data.fld_resent_to->to_addr_list->ad_list);
        mailimf_field_free(field);
        cur = clist_delete(fields->fld_list, cur);
      }
      break;
    case MAILIMF_FIELD_RESENT_CC:
      if (single_fields->fld_cc == NULL) {
        single_fields->fld_cc = field->fld_data.fld_resent_cc;
        cur = clist_next(cur);
      }
      else {
        clist_concat(single_fields->fld_cc->cc_addr_list->ad_list, 
                     field->fld_data.fld_resent_cc->cc_addr_list->ad_list);
        mailimf_field_free(field);
        cur = clist_delete(fields->fld_list, cur);
      }
      break;
    case MAILIMF_FIELD_RESENT_BCC:
      if (single_fields->fld_bcc == NULL) {
        single_fields->fld_bcc = field->fld_data.fld_resent_bcc;
        cur = clist_next(cur);
      }
      else {
        clist_concat(single_fields->fld_bcc->bcc_addr_list->ad_list,
                     field->fld_data.fld_resent_bcc->bcc_addr_list->ad_list);
        mailimf_field_free(field);
        cur = clist_delete(fields->fld_list, cur);
      }
      break;
    case MAILIMF_FIELD_RESENT_MSG_ID:
      if (single_fields->fld_message_id == NULL)
        single_fields->fld_message_id = field->fld_data.fld_resent_msg_id;
      cur = clist_next(cur);
      break;
    default:
      cur = clist_next(cur);
      break;
    }
  }
}

#define MAX_FROM 1024

static int smtp_send_message_with_fields(struct etpan_sender_item * sender,
    struct mailimf_fields * fields, char * message, size_t length)
{
  mailsmtp * smtp;
  int res;
  int r;
  struct mailimf_single_fields single_fields;
  struct mailimf_mailbox * mb_from;
  char * from;
  char from_value[MAX_FROM];
  struct mailimf_mailbox_list * to_list;
  struct mailimf_mailbox_list * cc_list;
  struct mailimf_mailbox_list * bcc_list;
  struct mailimf_mailbox_list * recipient_list;
  clist * smtp_recipient_list;
  clistiter * cur;
  struct mailimf_mailbox * mb;
  
  mailimf_single_resent_fields_init(&single_fields, fields);
  if ((single_fields.fld_to != NULL) || (single_fields.fld_cc != NULL) ||
      (single_fields.fld_bcc != NULL) || (single_fields.fld_from != NULL)) {
    /* resent fields */
    
    /* do nothing */
  }
  else {
    mailimf_single_fields_init(&single_fields, fields);
  }
  
  mb_from = NULL;
  if (single_fields.fld_sender != NULL) {
    mb_from = single_fields.fld_sender->snd_mb;
  }
  else if (single_fields.fld_from != NULL) {
    clistiter * iter;
    
    iter = clist_begin(single_fields.fld_from->frm_mb_list->mb_list);
    if (iter != NULL)
      mb_from = clist_content(iter);
  }
  
  if (mb_from == NULL) {
    char hostname[MAX_FROM];
    int r;
    char * user;
    
    /* must build dummy sender */
    user = getenv("USER");
    if (user == NULL) {
      res = ERROR_NO_FROM;
      goto err;
    }
    
    r = gethostname(hostname, sizeof(hostname));
    if (r != NO_ERROR) {
      res = ERROR_NO_FROM;
      goto err;
    }
    
    snprintf(from_value, sizeof(from_value), "%s@%s", user, hostname);
    from = from_value;
  }
  else {
    from = mb_from->mb_addr_spec;
  }

  /* build recipient list */
  
  to_list = NULL;
  if (single_fields.fld_to != NULL) {
    to_list = etpan_address_to_mailbox_list(single_fields.fld_to->to_addr_list);
    if (to_list == NULL) {
      res = ERROR_MEMORY;
      goto err;
    }
  }
  
  cc_list = NULL;
  if (single_fields.fld_cc != NULL) {
    cc_list = etpan_address_to_mailbox_list(single_fields.fld_cc->cc_addr_list);
    if (cc_list == NULL) {
      if (to_list != NULL)
        mailimf_mailbox_list_free(to_list);
      
      res = ERROR_MEMORY;
      goto err;
    }
  }
  
  bcc_list = NULL;
  if (single_fields.fld_bcc != NULL) {
    bcc_list = etpan_address_to_mailbox_list(single_fields.fld_bcc->bcc_addr_list);
    if (bcc_list == NULL) {
      if (cc_list != NULL)
        mailimf_mailbox_list_free(cc_list);
      if (to_list != NULL)
        mailimf_mailbox_list_free(to_list);
      
      res = ERROR_MEMORY;
      goto err;
    }
  }
  
  recipient_list = mailimf_mailbox_list_new_empty();
  if (recipient_list == NULL) {
    if (bcc_list != NULL)
      mailimf_mailbox_list_free(bcc_list);
    if (cc_list != NULL)
      mailimf_mailbox_list_free(cc_list);
    if (to_list != NULL)
      mailimf_mailbox_list_free(to_list);
    
    res = ERROR_MEMORY;
    goto err;
  }
  
  if (to_list != NULL) {
    r = etpan_append_mailbox_list(recipient_list, to_list);
    if (r != MAILIMF_NO_ERROR) {
      if (bcc_list != NULL)
        mailimf_mailbox_list_free(bcc_list);
      if (cc_list != NULL)
        mailimf_mailbox_list_free(cc_list);
      if (to_list != NULL)
        mailimf_mailbox_list_free(to_list);
      
      res = ERROR_MEMORY;
      goto err;
    }
  }

  if (cc_list != NULL) {
    r = etpan_append_mailbox_list(recipient_list, cc_list);
    if (r != MAILIMF_NO_ERROR) {
      if (bcc_list != NULL)
        mailimf_mailbox_list_free(bcc_list);
      if (cc_list != NULL)
        mailimf_mailbox_list_free(cc_list);
      if (to_list != NULL)
        mailimf_mailbox_list_free(to_list);
      
      res = ERROR_MEMORY;
      goto err;
    }
  }

  if (bcc_list != NULL) {
    r = etpan_append_mailbox_list(recipient_list, bcc_list);
    if (r != MAILIMF_NO_ERROR) {
      if (bcc_list != NULL)
        mailimf_mailbox_list_free(bcc_list);
      if (cc_list != NULL)
        mailimf_mailbox_list_free(cc_list);
      if (to_list != NULL)
        mailimf_mailbox_list_free(to_list);
      
      res = ERROR_MEMORY;
      goto err;
    }
  }

  if (bcc_list != NULL)
    mailimf_mailbox_list_free(bcc_list);
  if (cc_list != NULL)
    mailimf_mailbox_list_free(cc_list);
  if (to_list != NULL)
    mailimf_mailbox_list_free(to_list);
  
  smtp_recipient_list = esmtp_address_list_new();
  if (smtp_recipient_list == NULL) {
    mailimf_mailbox_list_free(recipient_list);
    
    res = ERROR_MEMORY;
    goto free_mblist;
  }

  for(cur = clist_begin(recipient_list->mb_list) ; cur != NULL ;
      cur = clist_next(cur)) {
    mb = clist_content(cur);
    
    r = esmtp_address_list_add(smtp_recipient_list, mb->mb_addr_spec, 0, NULL);
    if (r != MAILSMTP_NO_ERROR) {
      mailimf_mailbox_list_free(recipient_list);
      
      res = ERROR_MEMORY;
      goto free_smtplist;
    }
  }

  smtp = mailsmtp_new(0, NULL);
  if (smtp == NULL) {
    res = ERROR_MEMORY;
    goto free_smtplist;
  }
  
  r = mailsmtp_socket_connect(smtp, sender->host, sender->port);
  if (r != MAILSMTP_NO_ERROR) {
    mailsmtp_free(smtp);
    
    res = error_from_smtp_error(r);
    goto free_smtp;
  }
  
  r = mailsmtp_init(smtp);
  if (r != MAILSMTP_NO_ERROR) {
    mailsmtp_free(smtp);
    
    res = error_from_smtp_error(r);
    goto free_smtp;
  }
  
  r = mailesmtp_send(smtp,
      from, 0, NULL,
      smtp_recipient_list,
      message, length);
  if (r != MAILSMTP_NO_ERROR) {
    mailsmtp_free(smtp);
    
    res = error_from_smtp_error(r);
    goto free_smtp;
  }
  
  mailsmtp_free(smtp);
  
  esmtp_address_list_free(smtp_recipient_list);
  mailimf_mailbox_list_free(recipient_list);
  
  return NO_ERROR;
  
 free_smtp:
  mailsmtp_free(smtp);
 free_smtplist:
  esmtp_address_list_free(smtp_recipient_list);
 free_mblist:
  mailimf_mailbox_list_free(recipient_list);
 err:
  return res;
}

static int smtp_send_message(struct etpan_sender_item * sender,
    char * message, size_t length)
{
  size_t cur_token;
  struct mailimf_fields * fields;
  int r;
  int res;
  
  cur_token = 0;
  
  r =  mailimf_fields_parse(message, length, &cur_token,
      &fields);
  if (r != MAILIMF_NO_ERROR) {
    res = ERROR_PARSE;
    goto err;
  }
  
  r = smtp_send_message_with_fields(sender, fields, message, length);
  if (r != NO_ERROR) {
    res = r;
    goto free_fields;
  }
  
  mailimf_fields_free(fields);
  
  return NO_ERROR;
  
 free_fields:
  mailimf_fields_free(fields);
 err:
  return res;
}

static int smtp_send_mime(struct etpan_sender_item * sender,
    struct mailmime * mime)
{
  int r;
  int res;
  MMAPString * str;
  int col;
  
  if (mime->mm_type != MAILMIME_MESSAGE) {
    res = ERROR_INVAL;
    goto err;
  }
  
  str = mmap_string_new("");
  if (str == NULL) {
    res = ERROR_MEMORY;
    goto err;
  }
  
  col = 0;
  r = mailmime_write_mem(str, &col, mime);
  if (r != MAILIMF_NO_ERROR) {
    res = ERROR_MEMORY;
    goto free_str;
  }
  
  r = smtp_send_message_with_fields(sender,
      mime->mm_data.mm_message.mm_fields,
      str->str, str->len);
  if (r != NO_ERROR) {
    res = r;
    goto free_str;
  }
  
  mmap_string_free(str);
  
  return NO_ERROR;
  
 free_str:
  mmap_string_free(str);
 err:
  return res;
}


static int command_send_message(struct etpan_sender_item * sender,
    char * message, size_t length)
{
  int res;
  int r;
  int status;
  pid_t pid;
  int fd_input[2];
  int fd_output[2];
  MMAPString * output_str;
  char buf[4096];
  int max_fd;
  char * p;
  int source_open;
  int half_close;

  half_close = 0;
  
  r = pipe(fd_input);
  if (r < 0) {
    res = ERROR_COMMAND;
    goto err;
  }
  
  r = pipe(fd_output);
  if (r < 0) {
    res = ERROR_COMMAND;
    goto close_input;
  }
  
  pid = fork();
  switch (pid) {
  case 0:
    /* child, redirect file descriptors */
    close(fd_input[1]);
    dup2(fd_input[0], 0);
    dup2(fd_output[1], 1);
    dup2(fd_output[1], 2);
    
    execl(sender->command, sender->command, "-t", NULL);
    exit(EXIT_FAILURE);
    
  case -1:
    res = ERROR_COMMAND;
    goto close_output;
    
  default:
    /* parent */
    half_close = 1;
    
    close(fd_output[1]);
    close(fd_input[0]);
    
    max_fd = fd_input[1];
    if (fd_output[0] > max_fd)
      max_fd = fd_output[0];
    
    output_str = mmap_string_new("");
    if (output_str == NULL) {
      res = ERROR_COMMAND;
      goto close_output;
    }
    
    source_open = 1;
    p = message;
    while (1) {
      fd_set read_fds;
      fd_set write_fds;
      
      /* 
         no more use of fd_input[0] and fd_output[1]);
      */
      
      FD_ZERO(&read_fds);
      FD_ZERO(&write_fds);
      FD_SET(fd_output[0], &read_fds);
      if (source_open)
        FD_SET(fd_input[1], &write_fds);
      
      r = select(max_fd + 1, &read_fds, &write_fds, NULL, NULL);
      if (FD_ISSET(fd_output[0], &read_fds)) {
        ssize_t read_bytes;
        
        read_bytes = read(fd_output[0], buf, sizeof(buf));
        if (read_bytes < 0) {
          /* error */
          break;
        }
        else if (read_bytes == 0) {
          /* closed */
          break;
        }
        
        if (mmap_string_append_len(output_str, buf, read_bytes) == NULL) {
          res = ERROR_MEMORY;
          goto free_str;
        }
      }
      else if (length > 0) {
        if (FD_ISSET(fd_input[1], &write_fds)) {
          ssize_t written_bytes;
          
          written_bytes = write(fd_input[1], p, length);
          if (written_bytes < 0) {
            break;
          }
          else {
            p += written_bytes;
            length -= written_bytes;
            
            if (length == 0) {
              close(fd_input[1]);
              source_open = 0;
            }
          }
        }
      }
    }
    
    r = wait4(pid, &status, 0, NULL);
    if (r < 0) {
      res = ERROR_COMMAND;
      goto free_str;
    }
    
    if (WEXITSTATUS(status) != 0) {
      res = ERROR_COMMAND;
      goto free_str;
    }
    
    if (WIFSIGNALED(status) != 0) {
      res = ERROR_COMMAND;
      goto free_str;
    }
    
    /* should output to console */
    
    mmap_string_free(output_str);
    
    close(fd_output[0]);
    if (source_open)
      close(fd_input[1]);
    
    break;
  }
  
  return NO_ERROR;
  
 free_str:
  mmap_string_free(output_str);
 close_output:
  if (!half_close)
    close(fd_output[1]);
  close(fd_output[0]);
 close_input:
  close(fd_input[1]);
  if (!half_close)
    close(fd_input[0]);
 err:
  return res;
}

static int command_send_mime(struct etpan_sender_item * sender,
    struct mailmime * mime)
{
  int r;
  int res;
  MMAPString * str;
  int col;
  
  str = mmap_string_new("");
  if (str == NULL) {
    res = ERROR_MEMORY;
    goto err;
  }
  
  col = 0;
  r = mailmime_write_mem(str, &col, mime);
  if (r != MAILIMF_NO_ERROR) {
    res = ERROR_MEMORY;
    goto free_str;
  }
  
  r = command_send_message(sender,
      str->str, str->len);
  if (r != NO_ERROR) {
    res = r;
    goto free_str;
  }
  
  mmap_string_free(str);
  
  return NO_ERROR;
  
 free_str:
  mmap_string_free(str);
 err:
  return res;
}

int etpan_send_message(struct etpan_sender_item * sender,
    char * message, size_t length)
{
  switch (sender->type) {
  case SENDER_TYPE_SMTP:
    return smtp_send_message(sender, message, length);
    
  case SENDER_TYPE_COMMAND:
    return command_send_message(sender, message, length);
    
  default:
    return ERROR_INVAL;
  }
}

int etpan_send_mime(struct etpan_sender_item * sender,
    struct mailmime * mime)
{
  switch (sender->type) {
  case SENDER_TYPE_SMTP:
    return smtp_send_mime(sender, mime);
    
  case SENDER_TYPE_COMMAND:
    return command_send_mime(sender, mime);
    
  default:
    return ERROR_INVAL;
  }
}

int etpan_send_message_file(struct etpan_sender_item * sender,
    char * filename)
{
  int fd;
  int r;
  char * mem;
  struct stat stat_buf;
  int res;
  
  fd = open(filename, O_RDONLY);
  if (fd < 0) {
    res = ERROR_FILE;
    goto err;
  }
  
  r = fstat(fd, &stat_buf);
  if (r < 0) {
    res = ERROR_FILE;
    goto close;
  }
  
  mem = mmap(NULL, stat_buf.st_size, PROT_READ, MAP_PRIVATE, fd, 0);
  if (mem == MAP_FAILED) {
    res = ERROR_FILE;
    goto close;
  }
  
  r = etpan_send_message(sender, mem, stat_buf.st_size);
  if (r != NO_ERROR) {
    res = r;
    goto unmap;
  }
  
  munmap(mem, stat_buf.st_size);
  close(fd);
  
  return NO_ERROR;
  
 unmap:
  munmap(mem, stat_buf.st_size);
 close:
  close(fd);
 err:
  return res;
}
