#include "etpan-abook-vcard.h"
#include "etpan-abook-driver.h"
#include "etpan-errors.h"
#include <fcntl.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <sys/mman.h>
#include <string.h>
#include "vcc.h"
#include <ctype.h>
#include <libetpan/libetpan.h>

struct vcard_data {
  struct etpan_global_config * global_config;
  VObject * vcard_list;
};


static int lookup(struct etpan_abook * abook,
    const char * key, carray ** result);

static int connect(struct etpan_abook * abook);

static void uninitialize(struct etpan_abook * abook);

static struct etpan_abook_driver local_abook_vcard = {
  .name = "vcard",
  .connect = connect,
  .lookup = lookup,
  .uninitialize = uninitialize,
};

static int connect(struct etpan_abook * abook)
{
  return NO_ERROR;
}

static void uninitialize(struct etpan_abook * abook)
{
  VObject * t;
  VObject * v;
  struct vcard_data * data;
  
  data = abook->data;
  v = data->vcard_list;
  
  while (v) {
    t = v;
    v = nextVObjectInList(v);
    cleanVObject(t);
  }
  
  free(data);
}

static int parse_vcards_file(char * filename, VObject ** result);
static int parse_vcards_mem(char * data, size_t length,
    VObject ** result);

struct etpan_abook *
etpan_abook_vcard_new(struct etpan_global_config * global_config,
    char * filename)
{
  struct etpan_abook * abook;
  int r;
  VObject * v;
  struct vcard_data * data;
  
  abook = malloc(sizeof(* abook));
  if (abook == NULL)
    goto err;
  
  data = malloc(sizeof(* data));
  if (data == NULL)
    goto free;
  
  r = parse_vcards_file(filename, &v);
  if (r != NO_ERROR)
    goto free_data;
  
  data->vcard_list = v;
  data->global_config = global_config;
  
  abook->data = data;
  abook->driver = &local_abook_vcard;
  abook->connected = 0;
  
  return abook;
  
 free_data:
  free(data);
 free:
  free(abook);
 err:
  return NULL;
}

static int parse_vcards_file(char * filename, VObject ** result)
{
  int fd;
  struct stat stat_buf;
  int r;
  char * data;
  int res;
  
  fd = open(filename, O_RDONLY);
  if (fd < 0) {
    res = ERROR_FILE;
    goto err;
  }
  
  r = fstat(fd, &stat_buf);
  if (r < 0) {
    res = ERROR_FILE;
    goto close_fd;
  }
  
  data = mmap(NULL, stat_buf.st_size, PROT_READ, MAP_PRIVATE, fd, 0);
  if (data == MAP_FAILED) {
    res = ERROR_FILE;
    goto close_fd;
  }
  
  r = parse_vcards_mem(data, stat_buf.st_size, result);
  
  munmap(data, stat_buf.st_size);
  
  close(fd);
  
  return r;
  
 close_fd:
  close(fd);
 err:
  return res;
}

static int parse_vcards_mem(char * data, size_t length,
    VObject ** result)
{
  VObject * v;
  
  v = Parse_MIME(data, length);
  if (v == NULL)
    return ERROR_PARSE;
  
  * result = v;
  
  return NO_ERROR;
}



static int match_abook(char * pattern, int pattern_len, char * value)
{
  char * dup_value;
  int r;
  char * p;

  r = 0;

  if (value == NULL)
    goto err;

  dup_value = strdup(value);
  if (dup_value == NULL)
    goto err;

  for(p = dup_value ; * p != 0 ; p ++)
    * p = (char) toupper((unsigned char) * p);

  if (strncasecmp(pattern, dup_value, pattern_len) == 0)
    r = 1;
  
  free(dup_value);
  
  return r;
  
 err:
  return r;
}

static struct etpan_abook_entry *
get_entry_from_vcard(char * vcard_name, char * vcard_addr)
{
  struct etpan_abook_entry * entry;
  char * name;
  char * addr;
  char * nick;
  
  if (vcard_name != NULL) {
    name = strdup(vcard_name);
    if (name == NULL)
      goto err;
  }
  else {
    name = NULL;
  }
  
  if (vcard_addr != NULL) {
    addr = strdup(vcard_addr);
    if (addr == NULL)
      goto free_name;
  }
  else {
    addr = NULL;
  }
  
  nick = NULL;
  
  entry = etpan_abook_entry_new(name, addr, nick);
  if (entry == NULL)
    goto free_addr;
  
  return entry;
  
 free_addr:
  free(addr);
 free_name:
  free(name);
 err:
  return NULL;
}

static void match(carray * entry_list, char * dup_key, size_t key_len,
    char * name, char * addr)
{
  int r;
  
  if (match_abook(dup_key, key_len, name) || 
      match_abook(dup_key, key_len, addr)) {
    struct etpan_abook_entry * entry;
    
    entry = get_entry_from_vcard(name, addr);
    if (entry == NULL)
      return;
    
    r = carray_add(entry_list, entry, NULL);
    if (r != NO_ERROR) {
      etpan_abook_entry_free(entry);
      return;
    }
  }
}

static char * get_value(struct etpan_global_config * global_config,
    VObject * eachProp)
{
  int type;
  char * addr;
  char * converted;
  int r;
  
  addr = NULL;
  type = vObjectValueType(eachProp);
  switch (type) {
  case VCVT_STRINGZ:
    addr = strdup(vObjectStringZValue(eachProp));
    break;
    
  case VCVT_USTRINGZ:
    addr = fakeCString(vObjectUStringZValue(eachProp));
    break;
  }
  
  if (addr == NULL)
    return NULL;
  
  r = charconv(global_config->display_charset,
      "utf-8", addr, strlen(addr), &converted);
  if (r != MAIL_CHARCONV_NO_ERROR) {
    free(addr);
    return NULL;
  }
  
  free(addr);
  
  return converted;
}

static int lookup(struct etpan_abook * abook,
    const char * key, carray ** result)
{
  char * dup_key;
  size_t key_len;
  carray * entry_list;
  char * p;
  unsigned int i;
  VObject * v;
  struct vcard_data * data;
  int r;
  
  dup_key = strdup(key);
  if (dup_key == NULL)
    goto err;
  
  key_len = strlen(dup_key);
  
  for(p = dup_key ; * p != 0 ; p ++)
    * p = (char) toupper((unsigned char) * p);
  
  entry_list = carray_new(16);
  if (entry_list == NULL)
    goto free_key;
  
  data = abook->data;
  v = data->vcard_list;
  
  while (v) {
    carray * addr_tab;
    VObjectIterator t;
    
    initPropIterator(&t, v);
    
    addr_tab = carray_new(16);
    if (addr_tab == NULL)
      goto free_entry_list;
    
    while (moreIteration(&t)) {
      VObject * eachProp;
      const char * n;
      char * addr;
      
      eachProp = nextVObject(&t);
      
      n = vObjectName(eachProp);
      if (strcmp(n, VCEmailAddressProp) == 0) {
        addr = get_value(data->global_config, eachProp);
        r = carray_add(addr_tab, addr, NULL);
        if (r < 0)
          free(addr);
      }
    }
    
    for(i = 0 ; i < carray_count(addr_tab) ; i ++) {
      char * addr;
      int once;
      char * fullName;
      
      addr = carray_get(addr_tab, i);
      
      once = 0;
      
      fullName = NULL;
      /* full name */
      initPropIterator(&t, v);
      while (moreIteration(&t)) {
        VObject * eachProp;
        const char * n;
        
        eachProp = nextVObject(&t);
        
        n = vObjectName(eachProp);
        
        if (strcmp(n, VCFullNameProp) == 0) {
          fullName = get_value(data->global_config, eachProp);
          match(entry_list, dup_key, key_len, fullName, addr);
          once = 1;
        }
      }
      
      /* name */
      initPropIterator(&t, v);
      while (moreIteration(&t)) {
        VObject * eachProp;
        const char * n;
        
        eachProp = nextVObject(&t);
        
        n = vObjectName(eachProp);
        
        if (strcmp(n, VCNameProp) == 0) {
          char * firstName;
          char * lastName;
          char * tmpFullName;
          size_t tmpFullNameSize;
          VObjectIterator nameIter;
          
          firstName = NULL;
          lastName = NULL;
          
          initPropIterator(&nameIter, eachProp);
          while (moreIteration(&nameIter)) {
            VObject * eachProp;
            const char * n;
            char * name;
            
            eachProp = nextVObject(&nameIter);
            n = vObjectName(eachProp);
            name = get_value(data->global_config, eachProp);
            if (name == NULL)
              continue;
            
            if ((lastName == NULL) && (strcmp(n, VCFamilyNameProp) == 0))
              lastName = strdup(name);
            
            if ((firstName == NULL) && (strcmp(n, VCGivenNameProp) == 0))
              firstName = strdup(name);
            
            free(name);
            once = 1;
          }
          
          if ((firstName != NULL) && (lastName != NULL)) {
            tmpFullNameSize = strlen(firstName) + strlen(lastName) + 2;
            tmpFullName = malloc(tmpFullNameSize);
            if (tmpFullName != NULL) {
              snprintf(tmpFullName, tmpFullNameSize, "%s %s",
                  firstName, lastName);
              if (strcmp(fullName, tmpFullName) != 0)
                match(entry_list, dup_key, key_len, tmpFullName, addr);
            
              snprintf(tmpFullName, tmpFullNameSize, "%s %s",
                  lastName, firstName);
              if (strcmp(fullName, tmpFullName) != 0)
                match(entry_list, dup_key, key_len, tmpFullName, addr);
              
              free(tmpFullName);
            }
          }
          
          free(lastName);
          free(firstName);
        }
      }
      free(fullName);
      
      if (!once) {
        match(entry_list, dup_key, key_len, NULL, addr);
      }
    }
    
    for(i = 0 ; i < carray_count(addr_tab) ; i ++) {
      char * addr;
      
      addr = carray_get(addr_tab, i);
      
      free(addr);
    }
    carray_free(addr_tab);
    
    v = nextVObjectInList(v);
  }
  
  * result = entry_list;

  return NO_ERROR;
  
 free_entry_list:
  for(i = 0 ; i < carray_count(entry_list) ; i ++) {
    struct etpan_abook_entry * entry;
    
    entry = carray_get(entry_list, i);
    etpan_abook_entry_free(entry);
  }
 free_key:
  free(dup_key);
 err:
  return ERROR_MEMORY;
}
