/****************************************************************\
*                                                                *
*  The exonerate server                                          *
*                                                                *
*  Guy St.C. Slater..   mailto:guy@ebi.ac.uk                     *
*  Copyright (C) 2000-2006.  All Rights Reserved.                *
*                                                                *
*  This source code is distributed under the terms of the        *
*  GNU Lesser General Public License. See the file COPYING       *
*  or http://www.fsf.org/copyleft/lesser.html for details        *
*                                                                *
*  If you use this code, please keep this notice intact.         *
*                                                                *
\****************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <glib.h>
#include <ctype.h> /* For isspace() */

#include "argument.h"
#include "socket.h"
#include "dataset.h"

typedef struct {
    Dataset *dataset;
} Exonerate_Server;

static Exonerate_Server *Exonerate_Server_create(gchar *dataset_path){
    register Exonerate_Server *exonerate_server
     = g_new(Exonerate_Server, 1);
    g_message("making server ...");
    exonerate_server->dataset = Dataset_read(dataset_path);
    g_message("... finished making server");
    return exonerate_server;
    }

static void Exonerate_Server_destroy(Exonerate_Server *exonerate_server){
    Dataset_destroy(exonerate_server->dataset);
    g_free(exonerate_server);
    return;
    }

/**/

typedef struct {
} Exonerate_Server_Connection;

static Exonerate_Server_Connection *Exonerate_Server_Connection_create(void){
    register Exonerate_Server_Connection *connection
     = g_new(Exonerate_Server_Connection, 1);
    return connection;
    }

static void Exonerate_Server_Connection_destroy(
            Exonerate_Server_Connection *connection){
    g_free(connection);
    return;
    }

/**/

static gpointer Exonerate_Server_Connection_open(gpointer user_data){
    return Exonerate_Server_Connection_create();
    }

static void Exonerate_Server_Connection_close(gpointer connection_data,
                                              gpointer user_data){
    register Exonerate_Server_Connection *server_connection
        = connection_data;
    Exonerate_Server_Connection_destroy(server_connection);
    return;
    }

static GPtrArray *Exonerate_Server_get_word_list(gchar *msg){
    register GPtrArray *word_list = g_ptr_array_new();
    register gchar *prev, *ptr;
    for(ptr = msg; isspace(*ptr); ptr++); /* skip start */
    prev = ptr;
    while(*ptr){
        if(isspace(*ptr)){
            *ptr = '\0';
            do ptr++; while(isspace(*ptr));
            if(!*ptr)
                break;
            g_ptr_array_add(word_list, prev); /* add a word */
            prev = ptr;
            }
        ptr++;
        }
    if(prev != ptr)
        g_ptr_array_add(word_list, prev); /* add final word */
    return word_list;
    }

static gboolean Exonerate_Server_process(gchar *msg, gchar **reply,
                                         gpointer connection_data,
                                         gpointer user_data){
    register gint msg_len = strlen(msg);
    register gboolean keep_connection = TRUE;
    register Exonerate_Server *server = user_data;
    register GPtrArray *word_list;
    register gchar *word, *id, *item, *str;
    register gint start, len, num;
    register Dataset_Sequence *ds;
    register Sequence *seq, *subseq;
    g_assert(msg);
    if(!msg[msg_len-1] == '\n')
        msg[--msg_len] = '\0';
    word_list = Exonerate_Server_get_word_list(msg);
    switch(word_list->len){
        case 1:
            break;
        default:
            break;
        }
    if(word_list->len){
        word = word_list->pdata[0];
        if(!strncmp(word, "help", 4)){
            (*reply) = g_strdup_printf(
                                "exonerate-server commands:\n"
                                " commands: [ help | version | exit ]\n");
        } else if(!strncmp(word, "version", 7)){
            (*reply) = g_strdup_printf(
                                "exonerate-server version: %s\n", VERSION);
        } else if(!strncmp(word, "exit", 4)){
            (*reply) = g_strdup_printf("bye ...\n");
            keep_connection = FALSE;
        } else if(!strncmp(word, "dbinfo", 6)){
            (*reply) = g_strdup_printf("dbinfo: %s %s %d %d %d\n",
                  (server->dataset->header->type & 1)?"dna":"protein",
                  (server->dataset->header->type & (1<<1))
                      ?"softmasked":"unmasked",
                  (gint)server->dataset->header->number_of_seqs,
                  (gint)server->dataset->header->max_seq_len,
                  (gint)server->dataset->header->total_seq_len);
        } else if(!strncmp(word, "lookup", 6)){
            if(word_list->len == 2){
                id = word_list->pdata[1];
                (*reply) = g_strdup_printf("lookup %s %d\n", id,
                            Dataset_lookup_id(server->dataset, id));
                }
        } else if(!strncmp(word, "get", 3)){
            if(word_list->len >= 3){
                item = word_list->pdata[1];
                if(!strncmp(item, "info", 4)){
                    num = atoi(word_list->pdata[2]);
                    if((num >= 0) && (num < server->dataset->seq_list->len)){
                        ds = server->dataset->seq_list->pdata[num];
                        seq = Dataset_get_sequence(server->dataset, num);
                        (*reply) = g_strdup_printf("seqinfo: %d %d %s%s%s\n",
                                            (gint)ds->key->length,
                                            (gint)ds->gcg_checksum,
                                            ds->id,
                                            seq->def?" ":"",
                                            seq->def?seq->def:"");
                        Sequence_destroy(seq);
                        }
                } else if(!strncmp(item, "seq", 3)){
                    if(word_list->len == 3){
                        num = atoi(word_list->pdata[2]);
                        g_message("getting seq [%d]", num);
                        if((num >= 0)
                        && (num < server->dataset->seq_list->len)){
                            ds = server->dataset->seq_list->pdata[num];
                            seq = Dataset_get_sequence(server->dataset, num);
                            str = Sequence_get_str(seq);
                            (*reply) = g_strdup_printf("seq: [%s]\n", str);
                            g_free(str);
                            Sequence_destroy(seq);
                            }
                        }
                } else if(!strncmp(item, "subseq", 3)){
                    if(word_list->len == 5){
                        num = atoi(word_list->pdata[2]);
                        start = atoi(word_list->pdata[3]);
                        len = atoi(word_list->pdata[4]);
                        g_message("getting subseq [%d] [%d] [%d] ",
                                num, start,len);
                        if((num >= 0)
                        && (num < server->dataset->seq_list->len)
                        && (start >= 0)
                        && (len > 0)){
                            ds = server->dataset->seq_list->pdata[num];
                            seq = Dataset_get_sequence(server->dataset, num);
                            if((start+len) <= seq->len){
                                subseq = Sequence_subseq(seq, start, len);
                                str = Sequence_get_str(subseq);
                                Sequence_destroy(subseq);
                                (*reply) = g_strdup_printf("subseq: [%s]\n",
                                                           str);
                                g_free(str);
                                }
                            Sequence_destroy(seq);
                            }
                        }
                } else {
                    (*reply) = g_strdup_printf("bad command\n");
                    }
                }
        } else {
            (*reply) = g_strdup_printf("Unknown command: %s\n", msg);
            }
        }
    g_ptr_array_free(word_list, TRUE);
    if(!(*reply))
        (*reply) = g_strdup("Unknown command\n");
    return keep_connection;
    }

static void run_server(gint port, gchar *dataset_path){
    register Exonerate_Server *exonerate_server
           = Exonerate_Server_create(dataset_path);
    register SocketServer *ss = SocketServer_create(port, 2,
                       Exonerate_Server_process,
                       Exonerate_Server_Connection_open,
                       Exonerate_Server_Connection_close,
                       exonerate_server);
    g_message("listening ...");
    /* FIXME: make max_connections a command line option */
    while(SocketServer_listen(ss));
    SocketServer_destroy(ss);
    Exonerate_Server_destroy(exonerate_server);
    return;
    }

int Argument_main(Argument *arg){
    gint port;
    gchar *dataset_path;
    register ArgumentSet *as = ArgumentSet_create("Exonerate Server options");
    ArgumentSet_add_option(as, 'p', "port", "port",
            "Port number to run server on", "12345",
            Argument_parse_int, &port);
    ArgumentSet_add_option(as, '\0', "dataset", "path",
            "Path to dataset file", NULL,
            Argument_parse_string, &dataset_path);
    Argument_absorb_ArgumentSet(arg, as);
    Argument_process(arg, "exonerate-server", "Exonerate Server.\n",
                     "Guy St.C. Slater.  guy@ebi.ac.uk June 2006\n");
    g_message("port [%d]", port);
    run_server(port, dataset_path);
    g_message("server exiting");
    return 0;
    }

