/***************************************************************************
                          FLFieldDB.cpp  -  description
                             -------------------
    begin                : Wed Jul 4 2001
    copyright            : (C) 2001,2002 by Federico Albujer Zornoza
    email                : mail@infosial.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <math.h>

#include "FLFieldDB.h"
#include "../../flbase/FLSqlCursor.h"
#include "../../flbase/FLFormSearchDB.h"
#include "../../flbase/FLTableMetaData.h"
#include "../../flbase/FLFieldMetaData.h"
#include "../../flbase/FLRelationMetaData.h"
#include "../../flbase/FLManager.h"
#include "../../flbase/FLAction.h"
#include "../../flbase/FLReceiver.h"
#include "../../process/FLInterface.h"
#include "../fltabledb/FLTableDB.h"

FLLineEdit::FLLineEdit (QWidget * parent, const char *name):
QLineEdit (parent, name)
{
}

void
FLLineEdit::keyPressEvent (QKeyEvent * e)
{
  if (e->key () == Key_F2)
	{
	  emit key_F2_Pressed ();

	  e->ignore ();
	}
  else
	QLineEdit::keyPressEvent (e);
}

PixmapView::PixmapView (QWidget * parent):QScrollView (parent)
{
  viewport ()->setBackgroundMode (PaletteBase);
}

void
PixmapView::setPixmap (const QPixmap & pix)
{
  pixmap = pix;
  resizeContents (pixmap.size ().width (), pixmap.size ().height ());
  viewport ()->repaint (FALSE);
}

void
PixmapView::drawContents (QPainter * p, int cx, int cy, int cw, int ch)
{
  p->fillRect (cx, cy, cw, ch, colorGroup ().brush (QColorGroup::Base));
  p->drawPixmap (0, 0, pixmap);
}

void
PixmapView::previewUrl (const QUrl & u)
{
  if (u.isLocalFile ())
	{
	  QString path = u.path ();
	  QPixmap pix (path);

	  if (!pix.isNull ())
		setPixmap (pix);
	}
}

FLFieldDB::FLFieldDB (QWidget * parent, const char *name):
FLWidgetFieldDB (parent, name),
editor_ (0),
fieldName_ (QString::null),
tableName_ (QString::null),
actionName_(QString::null),
foreignField_ (QString::null),
fieldRelation_ (QString::null),
cursor_ (0),
cursorAux (0),
refreshOne (false),
cursorInit(false),
cursorAuxInit(false)
{
  QObject *obj = this;

  do
	{
	  obj = obj->parent ();
	  if (!obj)
		break;
	}
  while (!((QWidget *) obj)->inherits ("FLFormDB"));

  QWidget *topWidget = (QWidget *) obj;

  if (!topWidget)
	{
	  qWarning (tr ("FLFieldDB : EL widget de nivel superior deber ser de la clase FLFormDB o heredar de ella"));
	}
  else
	cursor_ = ((FLFormDB *) topWidget)->cursor ();

  if (!name)
	setName ("FLFieldDB");
}

FLFieldDB::~FLFieldDB ()
{
  if (cursor_)
	{
	  if (tableName ().isEmpty () && !foreignField ().isEmpty () && !fieldRelation ().isEmpty ())
		disconnect (cursor_, SIGNAL (bufferChanged (QString)), this, SLOT (refresh (QString)));
	  disconnect (cursor_, SIGNAL (newBuffer ()), this, SLOT (refresh ()));
	  if (cursorAux)
		delete cursor_;
	}

  if (editor_)
    disconnect ( editor_, 0, 0, 0 );
}

void
FLFieldDB::updateValue (const QString & t)
{
  if (!cursor_)
	return;

  FLTableMetaData *tMD = cursor_->metadata ();

  if (!tMD)
	return;

  FLFieldMetaData * field = tMD->field(fieldName_);

  if (!field)
    return;

  int type = field->type ();
  
  if (type == QVariant::Double)
	{
	  QString s;
	  s.setNum (t.toDouble (), 'f', field->partDecimal());
	}

  cursor_->setValueBuffer (fieldName_, t);
}

void
FLFieldDB::updateValue (const QDate & d)
{
  if (!cursor_)
	return;

  cursor_->setValueBuffer (fieldName_, d);
}

void
FLFieldDB::updateValue (const QTime & t)
{
  if (!cursor_)
	return;

  cursor_->setValueBuffer (fieldName_, t);
}

void
FLFieldDB::updateValue (bool b)
{
  if (!cursor_)
	return;

  cursor_->setValueBuffer (fieldName_, b);
}

void
FLFieldDB::updateValue ()
{
  if (!cursor_)
	return;

  if (!editor_->isA ("QTextEdit"))
	return;

  cursor_->setValueBuffer (fieldName_, ((QTextEdit *) editor_)->text ());
}

void
FLFieldDB::initCursor ()
{
  if (!cursor_)
	return;

  if (tableName ().isEmpty () || foreignField ().isEmpty () || fieldRelation ().isEmpty ())
	{
         if (!foreignField ().isEmpty () && !fieldRelation ().isEmpty ())
		connect (cursor_, SIGNAL (bufferChanged (QString)), this, SLOT (refresh (QString)));

         if (cursorInit)
            return;
         connect (cursor_, SIGNAL (newBuffer ()), this, SLOT (refresh ()));
         cursorInit = true;
	  return;
	}

  if (!cursorAux)
	{
         if (cursorAuxInit)
            return;

	  FLTableMetaData *tMD = FLManager::metadata (tableName ());

	  if (!tMD)
		return;

	  disconnect (cursor_, SIGNAL (newBuffer ()), this, SLOT (refresh ()));
	  cursorAux = cursor_;
	  cursor_ = new FLSqlCursor (tableName_, true, 0, cursorAux, tMD->relation (fieldRelation_, foreignField_));
	  if (!cursor_)
		{
		  cursor_ = cursorAux;
		  connect (cursor_, SIGNAL (newBuffer ()), this, SLOT (refresh ()));
		  cursorAux = 0;
		  return;
		}

	  if (pushButtonDB)
		pushButtonDB->close ();
	  if (textLabelDB)
		textLabelDB->close ();

	  cursor_->setModeAccess (FLSqlCursor::BROWSE);
	  connect (cursor_, SIGNAL (newBuffer ()), this, SLOT (refresh ()));

          cursorAuxInit = true;
	}
}

void
FLFieldDB::initEditor ()
{
  if (!cursor_)
	return;

  if (editor_)
	return;

  FLTableMetaData *tMD = cursor_->metadata ();

  if (!tMD)
	return;

  FLFieldMetaData * field = tMD->field(fieldName_);

  if (!field)
    return;

  int type = field->type();
  int len = field->length();
  int partInteger = field->partInteger();
  int partDecimal = field->partDecimal();

  QString rt;

  if (field->relationM1())
    rt = field->relationM1()->foreignTable();
  bool hasPushButtonDB = false;

  if (rt.isEmpty () && type != QVariant::Pixmap)
	{
	  pushButtonDB->close ();
	  if (type != QVariant::Bool)
		textLabelDB->setText (field->alias());
	  textLabelDB->setFont (qApp->font ());
	}
  else
	{
	  textLabelDB->close ();
	  pushButtonDB->setText (field->alias());
	  pushButtonDB->setFont (qApp->font ());
	  hasPushButtonDB = true;
	}

  switch (type)
	{
	case QVariant::Double:
	case QVariant::String:
	  componentDB->close ();
	  editor_ = new FLLineEdit (this, "editor");

	  ((FLLineEdit *) editor_)->setFrameShape (QLineEdit::Box);
	  ((FLLineEdit *) editor_)->setFrameShadow (QLineEdit::Plain);
	  ((FLLineEdit *) editor_)->setFont (qApp->font ());
	  ((FLLineEdit *) editor_)->setLineWidth (1);

	  if (type == QVariant::Double)
		{
		  ((FLLineEdit *) editor_)->setValidator (new FLDoubleValidator (0, pow (10, partInteger) - 1, partDecimal, editor_));
		  ((FLLineEdit *) editor_)->setAlignment (Qt::AlignRight);
		}
	  else
		((FLLineEdit *) editor_)->setMaxLength (len);

	  editor_->setSizePolicy (QSizePolicy ((QSizePolicy::SizeType) 7, (QSizePolicy::SizeType) 0, editor_->sizePolicy ().hasHeightForWidth ()));
	  FLWidgetFieldDBLayout->addWidget (editor_);

	  connect (editor_, SIGNAL (textChanged (const QString &)), this, SLOT (updateValue (const QString &)));
	  if (hasPushButtonDB)
		connect ((FLLineEdit *) editor_, SIGNAL (key_F2_Pressed ()), pushButtonDB, SLOT (animateClick ()));
	  break;

	case QVariant::UInt:
	case QVariant::Int:
	case FLFieldMetaData::Serial:
	  componentDB->close ();
	  editor_ = new QSpinBox (this, "editor");

	  ((QSpinBox *) editor_)->setFont (qApp->font ());

	  editor_->setSizePolicy (QSizePolicy ((QSizePolicy::SizeType) 7, (QSizePolicy::SizeType) 0, editor_->sizePolicy ().hasHeightForWidth ()));
	  FLWidgetFieldDBLayout->addWidget (editor_);

	  connect (editor_, SIGNAL (valueChanged (const QString &)), this, SLOT (updateValue (const QString &)));
	  break;

	case QVariant::Pixmap:
	  break;

	case QVariant::Date:
	  componentDB->close ();
	  editor_ = new QDateEdit (this, "editor");

	  ((QDateEdit *) editor_)->setFont (qApp->font ());

	  editor_->setSizePolicy (QSizePolicy ((QSizePolicy::SizeType) 7, (QSizePolicy::SizeType) 0, editor_->sizePolicy ().hasHeightForWidth ()));
	  FLWidgetFieldDBLayout->addWidget (editor_);

	  ((QDateEdit *) editor_)->setOrder (QDateEdit::DMY);

	  connect (editor_, SIGNAL (valueChanged (const QDate &)), this, SLOT (updateValue (const QDate &)));
	  {
		QDate d;

		((QDateEdit *) editor_)->setDate (d.currentDate ());
	  }
     break;

	case QVariant::Time:
    {
	  componentDB->close ();
     QTime timeNow = QTime::currentTime();
        
	  editor_ = new QTimeEdit (this, "editor");

	  ((QTimeEdit *) editor_)->setFont (qApp->font ());
          
	  editor_->setSizePolicy (QSizePolicy ((QSizePolicy::SizeType) 7, (QSizePolicy::SizeType) 0, editor_->sizePolicy ().hasHeightForWidth ()));
	  FLWidgetFieldDBLayout->addWidget (editor_);

	  connect (editor_, SIGNAL (valueChanged (const QTime &)), this, SLOT (updateValue (const QTime &)));

      ((QTimeEdit *) editor_)->setTime( timeNow );
     ((QTimeEdit *) editor_)->setRange( timeNow, timeNow.addSecs( 60 * 60 ) );
    }
	  break;

	case QVariant::StringList:
	  componentDB->close ();
	  editor_ = new QTextEdit (this, "editor");

	  ((QTextEdit *) editor_)->setFrameShape (QLineEdit::Box);
	  ((QTextEdit *) editor_)->setFrameShadow (QLineEdit::Plain);
	  ((QTextEdit *) editor_)->setFont (qApp->font ());
	  ((QTextEdit *) editor_)->setLineWidth (1);

	  editor_->setSizePolicy (QSizePolicy ((QSizePolicy::SizeType) 7, (QSizePolicy::SizeType) 0, editor_->sizePolicy ().hasHeightForWidth ()));
	  FLWidgetFieldDBLayout->addWidget (editor_);

	  connect (editor_, SIGNAL (textChanged ()), this, SLOT (updateValue ()));
	  break;

	case QVariant::Bool:
	  componentDB->close ();
	  textLabelDB->close ();
	  editor_ = new QCheckBox (this, "editor");

	  ((QCheckBox *) editor_)->setText (tMD->fieldNameToAlias (fieldName_));
	  ((QCheckBox *) editor_)->setFont (qApp->font ());

	  editor_->setSizePolicy (QSizePolicy ((QSizePolicy::SizeType) 7, (QSizePolicy::SizeType) 0, editor_->sizePolicy ().hasHeightForWidth ()));
	  FLWidgetFieldDBLayout->addWidget (editor_);

	  connect (editor_, SIGNAL (toggled (bool)), this, SLOT (updateValue (bool)));
	  break;
	}

  QPalette pal;
  QColorGroup cg;

  cg.setColor (QColorGroup::Foreground, black);
  cg.setColor (QColorGroup::Button, QColor (220, 220, 220));
  cg.setColor (QColorGroup::Light, white);
  cg.setColor (QColorGroup::Midlight, QColor (242, 242, 242));
  cg.setColor (QColorGroup::Dark, QColor (78, 78, 78));
  cg.setColor (QColorGroup::Mid, QColor (183, 183, 183));
  cg.setColor (QColorGroup::Text, black);
  cg.setColor (QColorGroup::BrightText, white);
  cg.setColor (QColorGroup::ButtonText, black);
  cg.setColor (QColorGroup::Base, white);
  cg.setColor (QColorGroup::Background, QColor (220, 220, 220));
  cg.setColor (QColorGroup::Shadow, black);
  cg.setColor (QColorGroup::Highlight, QColor (84, 112, 152));
  cg.setColor (QColorGroup::HighlightedText, white);
  cg.setColor (QColorGroup::Link, QColor (0, 0, 255));
  cg.setColor (QColorGroup::LinkVisited, QColor (255, 0, 255));
  pal.setActive (cg);
  cg.setColor (QColorGroup::Foreground, black);
  cg.setColor (QColorGroup::Button, QColor (220, 220, 220));
  cg.setColor (QColorGroup::Light, white);
  cg.setColor (QColorGroup::Midlight, QColor (242, 242, 242));
  cg.setColor (QColorGroup::Dark, QColor (78, 78, 78));
  cg.setColor (QColorGroup::Mid, QColor (183, 183, 183));
  cg.setColor (QColorGroup::Text, black);
  cg.setColor (QColorGroup::BrightText, white);
  cg.setColor (QColorGroup::ButtonText, black);
  cg.setColor (QColorGroup::Base, white);
  cg.setColor (QColorGroup::Background, QColor (220, 220, 220));
  cg.setColor (QColorGroup::Shadow, black);
  cg.setColor (QColorGroup::Highlight, QColor (84, 112, 152));
  cg.setColor (QColorGroup::HighlightedText, white);
  cg.setColor (QColorGroup::Link, QColor (0, 0, 255));
  cg.setColor (QColorGroup::LinkVisited, QColor (255, 0, 255));
  pal.setInactive (cg);
  cg.setColor (QColorGroup::Foreground, QColor (128, 128, 128));
  cg.setColor (QColorGroup::Button, QColor (220, 220, 220));
  cg.setColor (QColorGroup::Light, white);
  cg.setColor (QColorGroup::Midlight, QColor (253, 253, 253));
  cg.setColor (QColorGroup::Dark, QColor (110, 110, 110));
  cg.setColor (QColorGroup::Mid, QColor (146, 146, 146));
  cg.setColor (QColorGroup::Text, black);
  cg.setColor (QColorGroup::BrightText, white);
  cg.setColor (QColorGroup::ButtonText, black);
  cg.setColor (QColorGroup::Base, white);
  cg.setColor (QColorGroup::Background, QColor (220, 220, 220));
  cg.setColor (QColorGroup::Shadow, black);
  cg.setColor (QColorGroup::Highlight, QColor (0, 0, 128));
  cg.setColor (QColorGroup::HighlightedText, white);
  cg.setColor (QColorGroup::Link, QColor (0, 0, 255));
  cg.setColor (QColorGroup::LinkVisited, QColor (255, 0, 255));
  pal.setDisabled (cg);

  if (editor_)
	{
	  editor_->setPalette (pal);
	  editor_->setFocusPolicy (QWidget::StrongFocus);
	  setFocusProxy (editor_);
	  if (hasPushButtonDB)
		pushButtonDB->setFocusPolicy (QWidget::NoFocus);
	}
  else
	{
	  componentDB->setPalette (pal);
	  componentDB->setFocusPolicy (QWidget::NoFocus);
	  if (hasPushButtonDB)
		setFocusProxy (pushButtonDB);
	}

  if (hasPushButtonDB)
	pushButtonDB->setPalette (pal);
  else
	textLabelDB->setPalette (pal);
}

void
FLFieldDB::searchValue ()
{
  if (!cursor_)
	return;

  if (fieldName_.isEmpty ())
	return;

  FLTableMetaData *tMD = cursor_->metadata ();

  if (!tMD)
	return;

  FLFieldMetaData * field = tMD->field(fieldName_);

  if (!field)
    return;

  int type = field->type();

  if (type == QVariant::Pixmap)
	{

	  PixmapView *p = new PixmapView;
	  QFileDialog *fd = new QFileDialog (this);

	  fd->setContentsPreviewEnabled (TRUE);
	  fd->setContentsPreview (p, p);
	  fd->setPreviewMode (QFileDialog::Contents);
	  fd->setCaption (tr ("Elegir archivo"));
	  fd->setFilter ("*.xpm");

	  QString filename;

	  if (fd->exec () == QDialog::Accepted)
		filename = fd->selectedFile ();

	  if (filename.isEmpty ())
		{
		  componentDB->clear ();
		  updateValue ("");
		  return;
		}

         QFile f(filename);
         QByteArray ba;
         if ( f.open(IO_ReadOnly) )
            ba = f.readAll();
         f.close();

         QPixmap pix;
         QString s(ba);

         if (s.isEmpty())
            {
		  componentDB->clear ();
		  updateValue ("");
		  return;
		}

	  if (!QPixmapCache::find (s.left(100), pix))
		{
		  pix.loadFromData (ba);
		  QPixmapCache::insert (s.left(100), pix);
		}

	  if (!pix.isNull ())
		{
		  componentDB->resize (pix.size ());
		  componentDB->setPixmap (pix);
		  componentDB->repaint ();
		}
	  else
		componentDB->clear ();

	  updateValue (s);
	  return;
	}

  if (!field->relationM1())
    {
        qWarning("FLFieldDB : " + tr ("El campo de busqueda debe tener una relacin M1"));
        return;
    }

  FLFormSearchDB *f = 0;
  FLSqlCursor *c = 0;
  FLFieldMetaData *fMD = field->associatedField ();
  FLAction *a;
  FLReceiver *r=0;
  
  if (fMD)
	{
      if (!fMD->relationM1())
        {
            qWarning("FLFieldDB : " + tr ("El campo asociado debe tener una relacin M1"));
            return;
        }

	  c = new FLSqlCursor (fMD->relationM1()->foreignTable());
	  QString s = cursor_->valueBuffer (fMD->name ()).toString ();

	  if (s.isEmpty ())
		{
		  QMessageBox::warning (qApp->mainWidget (), tr ("Aviso"), tr ("Debe indicar un valor para ") + fMD->alias (), QMessageBox::Ok, 0, 0);
		  return;
		}
	  c->select ("upper(" + fMD->relationM1 ()->foreignField () + ")=" + FLManager::formatValue (fMD, QVariant (s.upper ())));
	  c->next ();

         if (actionName().isEmpty())
            a = FLManager::action (c->metadata ()->name ());
         else
            a = FLManager::action (actionName());

         r = FLInterface::getReceiver (a->masterProcess ());
	  	  f = new FLFormSearchDB (c, qApp->mainWidget(),r);
	}
  else
       {
         if (actionName().isEmpty())
            a = FLManager::action (field->relationM1()->foreignTable());
         else
            a = FLManager::action (actionName_);

         r = FLInterface::getReceiver (a->masterProcess ());
         f = new FLFormSearchDB (field->relationM1()->foreignTable(), qApp->mainWidget(),r);
       }
  
  f->cursor()->setAction(a);

  QWidget *w = 0;

  if (r)
    w = QWidgetFactory::create (a->form (), r, f);
  else
    w = QWidgetFactory::create (a->form (), f, f);     

  if (fMD && w)
	{
	  QObject *obj = w->child ("tableDBRecords");

	  if (obj)
		{
		  ((FLTableDB *) obj)->setTableName ((const char *)field->relationM1()->foreignTable());
		  ((FLTableDB *) obj)->setFieldRelation ((const char *)field->associatedFieldFilterTo ());
		  ((FLTableDB *) obj)->setForeignField ((const char *)fMD->relationM1 ()->foreignField ());
		}
	}

  if (!w)
	{
	  qWarning (tr ("FLFieldDB : No se ha podido cargar ") + a->form () + tr ("'. Ejecute 'make install' despues de compilar la aplicacin."));
	  return;
	}

  f->setMainWidget (w);
  f->setFocus();
  f->exec ();

  if (((FLFormSearchDB *) f)->accepted ())
	{
	  cursor_->setValueBuffer (fieldName_, f->cursor ()->valueBuffer (field->relationM1()->foreignField()));
	  refresh ();
	}
    
  if (f)
	f->close ();

  if (c)
	delete c;
}

QCString
FLFieldDB::fieldName () const
{
  return (const char *)fieldName_;
}

QCString
FLFieldDB::actionName () const
{
  return (const char *)actionName_;
}

QCString
FLFieldDB::tableName () const
{
  return (const char *)tableName_;
}

QCString
FLFieldDB::foreignField () const
{
  return (const char *)foreignField_;
}

QCString
FLFieldDB::fieldRelation () const
{
  return (const char *)fieldRelation_;
}

void
FLFieldDB::refresh (QString fN)
{
  if (!cursor_)
	return;

  FLTableMetaData * tMD = cursor_->metadata();

  if (!tMD)
    return;

  QVariant v;

  if (fN.isEmpty ())
	v = cursor_->valueBuffer (fieldName_);
  else
	{
	  if (!cursorAux && fN.lower () == fieldRelation_.lower () && refreshOne)
		{
		  FLFieldMetaData * field = tMD->field(fieldRelation_);

          if (!field)
            return;

          if (!field->relationM1())
            {
                qWarning ("FLFieldDB : " + tr ("El campo de la relacin debe estar relacionado en M1"));
                return;
            }
          FLSqlCursor c (field->relationM1()->foreignTable());
		  QString s = cursor_->valueBuffer (fieldRelation_).toString ();

		  c.select ("upper(" + field->relationM1()->foreignField() + ")=" +
					FLManager::formatValue (field, QVariant (s.upper ())));
		  if (c.next ())
			v = c.valueBuffer (foreignField_);
		  else
			return;
		}
	  else
		{
		  if (!cursorAux && fN.lower () == fieldRelation_.lower ())
			refreshOne = true;
		  return;
		}
	}

  int modeAccess = cursor_->modeAccess ();

  FLFieldMetaData * field = tMD->field(fieldName_);

  if (!field)
    return;

  int type = field->type();
  int partDecimal = field->partDecimal();

  if (!field->visible())
        close();

  setDisabled (cursor_->fieldDisabled (fieldName_)
			  || (modeAccess == FLSqlCursor::EDIT && field->isPrimaryKey() ));

  switch (type)
	{
	case QVariant::Double:
	  if (modeAccess == FLSqlCursor::BROWSE)
		setDisabled (true);

	  if (!editor_)
		return;

	  if (v.isValid ())
		{
		  QString s;

		  s.setNum (v.toDouble (), 'f', partDecimal);
		  ((FLLineEdit *) editor_)->setText (s);
		}
	  else
		((FLLineEdit *) editor_)->setText (QString::null);
	  break;

	case QVariant::String:
	  if (modeAccess == FLSqlCursor::BROWSE)
		setDisabled (true);

	  if (!editor_)
		return;

	  if (v.isValid ())
		((FLLineEdit *) editor_)->setText (v.toString ());
	  else
		((FLLineEdit *) editor_)->setText (QString::null);
	  break;

	case QVariant::UInt:
	case QVariant::Int:
	case FLFieldMetaData::Serial:
	  if (modeAccess == FLSqlCursor::BROWSE)
		setDisabled (true);

	  if (!editor_)
		return;

	 if (v.isValid ())
		((QSpinBox *) editor_)->setValue (v.toInt ());
	 else
		((QSpinBox *) editor_)->setValue (0);
	  break;

	case QVariant::Pixmap:
	  {
		QCString cs;

		if (v.isValid ())
		  cs = v.toCString ();

		if (cs.isEmpty ())
		  {
			componentDB->clear ();
			return;
		  }

		QPixmap pix;

		if (!QPixmapCache::find (cs.left(100), pix))
		  {
			pix.loadFromData (cs);
			QPixmapCache::insert (cs.left(100), pix);
		  }
		if (!pix.isNull ())
		  {
			componentDB->resize (pix.size ());
			componentDB->setPixmap (pix);
			componentDB->repaint ();
		  }
		else
		  componentDB->clear ();
	  }

	  if (modeAccess == FLSqlCursor::BROWSE)
		pushButtonDB->setDisabled (true);
	  break;

	case QVariant::Date:
	  if (!editor_)
		{
		  qWarning ("FLFieldDB : No se ha podido crear el editor para el campo " + fieldName_);
		  return;
		}
	  if (modeAccess == FLSqlCursor::BROWSE)
		setDisabled (true);

	  {
		QDate d = v.toDate ();

		if (!d.isValid ())
		  d = QDate::currentDate ();
		((QDateEdit *) editor_)->setDate (d);
	  }
	  break;

	case QVariant::Time:
	  if (!editor_)
		return;

	  if (modeAccess == FLSqlCursor::BROWSE)
		setDisabled (true);

	  {
		QTime t = v.toTime ();

		if (!t.isValid ())
		  t = QTime::currentTime ();
		((QTimeEdit *) editor_)->setTime (t);
	  }
	  break;

	case QVariant::StringList:
	  if (modeAccess == FLSqlCursor::BROWSE)
		setDisabled (true);

	  if (!editor_)
		return;

	  ((QTextEdit *) editor_)->setText (v.toString ());
	  break;

	case QVariant::Bool:
	  if (!editor_)
		return;

	  if (modeAccess == FLSqlCursor::BROWSE)
		setDisabled (true);

	  ((QCheckBox *) editor_)->setChecked (v.toBool ());
	  break;
	}
}

void
FLFieldDB::setTableName (const QCString & fT)
{
  tableName_ = fT;
  initCursor ();
  initEditor ();
}

void
FLFieldDB::setForeignField (const QCString & fN)
{
  foreignField_ = fN;
  initCursor ();
  initEditor ();
}

void
FLFieldDB::setFieldRelation (const QCString & fN)
{
  fieldRelation_ = fN;
  initCursor ();
  initEditor ();
}

void
FLFieldDB::setFieldName (const QCString & fN)
{ 
  fieldName_ = fN;
  initCursor ();
  initEditor ();
}

void
FLFieldDB::setActionName (const QCString & aN)
{
  actionName_ = aN;
}

FLDoubleValidator::FLDoubleValidator (QObject * parent, const char *name):
QDoubleValidator (parent, name)
{
}

FLDoubleValidator::FLDoubleValidator (double bottom, double top, int decimals, QObject * parent, const char *name):
QDoubleValidator (bottom, top, decimals, parent, name)
{
}

QValidator::State FLDoubleValidator::validate (QString & input, int &i) const
{
  if (input.isEmpty ())
	return QValidator::Acceptable;
  QValidator::State state = QDoubleValidator::validate (input, i);
  if (state == QValidator::Invalid || state == QValidator::Intermediate)
	{
	  QString
		s =
		input.
		right (input.length () - 1);

	  if (input.left (1) == "-" && (QDoubleValidator::validate (s, i) == QValidator::Acceptable || s.isEmpty ()))
		return QValidator::Acceptable;
	  return QValidator::Invalid;
	}
  else
	return QValidator::Acceptable;
}
