# This file is part of the Falcon repository manager
# Copyright (C) 2005-2008 Dennis Kaarsemaker
# See the file named COPYING in the root of the source tree for license details
#
# mail-maintainer.py - Mail maintainer and uploader when a package is accepted

import falcon
from gettext import gettext as _
from email import FeedParser
from email.MIMEText import MIMEText
from email.MIMEMultipart import MIMEMultipart
import smtplib

class MailerPlugin(falcon.plugin.FalconPlugin):
    """Mailer plugin"""
    name = _("Mail the maintainer")
    desc = _("Mail the package maintainer when packages are installed")
    api  = (1,1)

    def __init__(self):
        self.conf.register('frommail', '', falcon.questions.Entry(_("Sender address"), [falcon.validators.mailaddress]))
        self.conf.register('extramail', '', falcon.questions.Entry(_("Extra mailaddresses to send to"),
                           [falcon.validators.blank_ok(falcon.validators.commaseparated(falcon.validators.mailaddress))]))
        self.conf.register('whitelist', '', falcon.questions.Entry(_("File with whitelist of recipients")))
        self.conf.register('server', 'localhost', falcon.questions.Entry(_("Hostname of the smtp server to use")))

    def post_install(self, result, component, package):
        if result == package and self.conf.frommail:
            if self.conf.whitelist:
                whitelist = falcon.util.readfile(self.conf.whitelist)
            else:
                falcon.util.warning(_("No whitelist found for mailaddresses, not sending mail"))
                return
            addresses = []
            maintainer = package.controlfields['Maintainer']
            if '<' in maintainer:
                maintainer = maintainer[maintainer.find('<')+1:maintainer.find('>')]
            if maintainer in whitelist:
                addresses.append(maintainer)
            if package.changelog:
                p = FeedParser.FeedParser()
                p.feed(package.changelog)
                p = p.close()
                uploader = p['Changed-By']
                if uploader and ('<' in uploader):
                    uploader = uploader[uploader.find('<')+1:uploader.find('>')]
                if uploader in whitelist:
                    addresses.append(uploader)

            for m in self.conf.extramail.split(','):
                if m.strip():
                    addresses.append(m.strip())
            addresses = set(addresses)
            if not addresses:
                return
            falcon.util.output(_("Sending mail to %s") % ', '.join(addresses))
            
            msg = MIMEMultipart()
            msg['Subject'] = _("Installed package %s %s") % (package.packagename, package.version)
            msg['From'] = '"%s" <%s>' % (falcon.conf.description.replace('"',''), self.conf.frommail)
            msg['To'] = ', '.join(addresses)
            msg.preamble = _("Package %s (%s) has been installed into %s") % (package.packagename, package.version, str(component))
            f = MIMEText(package.control)
            msg.attach(f)
            if package.changelog:
                f = MIMEText(package.changelog)
                msg.attach(f)
            try:
                s = smtplib.SMTP()
                if falcon.conf.debug:
                    s.set_debuglevel(1)
                s.connect(self.conf.server)
                s.sendmail(self.conf.frommail, addresses, msg.as_string())
                s.quit()
            except:
                falcon.util.warning(_("Sending e-mail failed"))
                raise # This will cause the plugin to be disabled and a traceback to be displayed if run with -v

