#
# The Falcon Programming Language
#
# Global variables and configurations needed for the
# Falcon build environment
#
# The initenv.sh will copy this script on the top of
# the devel tree. Theoretically, this scirpt won't change
# between releases, so there shouldn't be the need to
# copy it very often by hand.
#
# However, the copied script in the devel tree will not
# be overwritten by initenv.sh.
#
# Input variables:
# TARGET_DIR - defaults to environ FALCON_ACTIVE_TREE
# BUILD_DIR - defaults to envifon FALCON_BUILD_TREE
# BUILD_TYPE - Sets CMAKE_BUILD_TYPE and default is read from settings.cmake
#
##############################################################################
#  Version informations
##############################################################################

# This is the "product version"
SET(FALCON_VERSION_MAJOR   0)
SET(FALCON_VERSION_MINOR   8)
SET(FALCON_VERSION_REVISION   8)

SET(FALCON_VERSION_NAME "bluejay")

#SONAME and soversion (unix so library informations for engine)
# Remember that SONAME never follows project versioning, but
# uses a VERSION, REVISION, AGE format, where
# VERSION: generational version of the project
# REVISION: times this version has been touched
# AGE: Number of version for which binary compatibility is granted
# In eample, 1.12.5 means that this lib may be dynlinked against
# every program using this lib versioned from 1.8 to 1.12.

SET(FALCON_SONAME_VERSION 1)
SET(FALCON_SONAME_REVISION 9)
SET(FALCON_SONAME_AGE 0)

# A couple of useful shortcuts
SET(FALCON_SONAME "${FALCON_SONAME_VERSION}.${FALCON_SONAME_REVISION}.${FALCON_SONAME_AGE}")
SET(FALCON_SONAME_REV "${FALCON_SONAME_VERSION}.${FALCON_SONAME_REVISION}")

#Automatically generated version info for RC scripts and sources
#CMAKE is good at this, let's use this feature
SET(FALCON_VERSION_RC   "${FALCON_VERSION_MAJOR}, ${FALCON_VERSION_MINOR}, ${FALCON_VERSION_REVISION}, 0")
SET(FALCON_VERSION_ID   "${FALCON_VERSION_MAJOR}.${FALCON_VERSION_MINOR}.${FALCON_VERSION_REVISION}")

#Import environment variables
SET(FALCON_BUILD_NAME   $ENV{FALCON_BUILD_NAME})
SET(FALCON_ACTIVE_TREE   $ENV{FALCON_ACTIVE_TREE})
SET(FALCON_SRC_TREE   $ENV{FALCON_SRC_TREE})
SET(FALCON_DEVEL_TREE   $ENV{FALCON_DEVEL_TREE})
SET(FALCON_BUILD_TREE   $ENV{FALCON_BUILD_TREE})

##############################################################################
# Including specific build informations
# This is a user-configurable script containing vars
# CMAKE_BUILD_TYPE: default/release  -->defines a set of options and things to do.
##############################################################################

IF("${BUILD_TYPE}" STREQUAL "" )
   INCLUDE("$ENV{FALCON_DEVEL_TREE}/settings.cmake" OPTIONAL)
   SET(BUILD_TYPE ${CMAKE_BUILD_TYPE})
   MESSAGE( "Setting CMAKE_BUILD_TYPE=${CMAKE_BUILD_TYPE}")
ELSE("${BUILD_TYPE}" STREQUAL "" )
   SET( CMAKE_BUILD_TYPE ${BUILD_TYPE} )
ENDIF("${BUILD_TYPE}" STREQUAL "" )

##############################################################################
#  Other defaults
##############################################################################
INCLUDE(TestBigEndian)

TEST_BIG_ENDIAN(falcon_big_endian)
IF(falcon_big_endian)
   SET(FALCON_LITTLE_ENDIAN 0)
ELSE(falcon_big_endian)
   SET(FALCON_LITTLE_ENDIAN 1)
ENDIF(falcon_big_endian)

#Set the target directory
IF("${TARGET_DIR}" STREQUAL "" )
   SET(TARGET_DIR $ENV{FALCON_ACTIVE_TREE})
ENDIF("${TARGET_DIR}" STREQUAL "" )

IF("${BUILD_DIR}" STREQUAL "" )
   SET(BUILD_DIR $ENV{FALCON_BUILD_TREE})
ENDIF("${BUILD_DIR}" STREQUAL "" )

IF("${FINAL_DESTINATION}" STREQUAL "" )
   #In windows, we normally install in c:\falcon
   IF(WIN32)
      IF($ENV{PRGORAMS})
         SET(FINAL_DESTINATION  "C:\\\\$ENV{PROGRAMS}\\\\falcon" )
      ELSE($ENV{PRGORAMS})
         SET(FINAL_DESTINATION  "C:\\\\Program Files\\\\falcon" )
      ENDIF($ENV{PRGORAMS})
   ELSE(WIN32)
      SET(FINAL_DESTINATION  "/usr" )
   ENDIF(WIN32)
ENDIF("${FINAL_DESTINATION}" STREQUAL "" )

IF("${TARGET_LIB_DIR}" STREQUAL "" )
   SET(TARGET_LIB_DIR  "lib" )
ENDIF("${TARGET_LIB_DIR}" STREQUAL "" )

#In windows, we normally install in c:\falcon
IF(WIN32)
   SET( FALCON_HOST_SYSTEM "WINDOWS" )
   SET( FALCON_SYSTEM_WIN 1 )
ELSE(WIN32)
   IF(APPLE)
      SET( FALCON_HOST_SYSTEM "MAC" )
      SET( FALCON_SYSTEM_MAC 1 )
   ELSEIF(UNIX)
      SET( FALCON_HOST_SYSTEM "UNIX" )
      SET( FALCON_SYSTEM_UNIX 1 )
   ELSE(APPLE)
      MESSAGE( "Sorry, can't determine system type" )
   ENDIF(APPLE)
ENDIF(WIN32)


##############################################################################
#  Used macro
##############################################################################

MACRO(FALCON_CONFIG itemlist)
   FOREACH(item ${${itemlist}} )
      MESSAGE( "Configuring ${item}.in into ${item}" )
      CONFIGURE_FILE(
         ${item}.in
         ${item}
         ESCAPE_QUOTES
         @ONLY )
   ENDFOREACH(item)
ENDMACRO(FALCON_CONFIG)

MACRO(FALCON_ADD_SUBDIRECTORY item)
   IF(DEFINED FALCON_PROJECT_BASE)
      ADD_SUBDIRECTORY("${item}" "${BUILD_DIR}/${FALCON_PROJECT_BASE}/${item}")
   ELSE(DEFINED FALCON_PROJECT_BASE)
      MESSAGE( FATAL_ERROR "Sorry, the variable FALCON_PROJECT_BASE must be defined in the main CMakeLists.txt" )
   ENDIF(DEFINED FALCON_PROJECT_BASE)
ENDMACRO(FALCON_ADD_SUBDIRECTORY)

MACRO(FALCON_MODULE_INSTALL tgt)
   IF(WIN32)
      INSTALL( TARGETS ${tgt}
         DESTINATION ${TARGET_DIR}/bin )
   ELSE(WIN32)
      INSTALL( TARGETS ${tgt}
         DESTINATION ${TARGET_DIR}/${TARGET_LIB_DIR}/falcon )
      IF("${BUILD_TYPE}" STREQUAL "release" )
         IF("${FALCON_STRIP_BIN}" STREQUAL "STRIP" )
            INSTALL( CODE "EXECUTE_PROCESS( COMMAND strp ${TARGET_DIR}/${TARGET_LIB_DIR}/falcon/${tgt}.so )" )
         ENDIF("${FALCON_STRIP_BIN}" STREQUAL "STRIP" )
      ENDIF("${BUILD_TYPE}" STREQUAL "release" )
   ENDIF(WIN32)
ENDMACRO(FALCON_MODULE_INSTALL)

MACRO(FALCON_INSTALL tgt path)
   INSTALL( TARGETS ${tgt}
      DESTINATION ${TARGET_DIR}/${path} )
   IF(UNIX)
      IF("${BUILD_TYPE}" STREQUAL "release" )
         IF("${FALCON_STRIP_BIN}" STREQUAL "STRIP" )
            INSTALL( CODE "EXECUTE_PROCESS( COMMAND strip ${TARGET_DIR}/${path}/${tgt} )" )
         ENDIF("${FALCON_STRIP_BIN}" STREQUAL "STRIP" )
      ENDIF("${BUILD_TYPE}" STREQUAL "release" )
   ENDIF(UNIX)
ENDMACRO(FALCON_INSTALL)

MACRO(FALCON_LIB_INSTALL tgt )
   INSTALL(TARGETS ${tgt}
      RUNTIME DESTINATION ${TARGET_DIR}/bin
      LIBRARY DESTINATION ${TARGET_DIR}/${TARGET_LIB_DIR}
      ARCHIVE DESTINATION ${TARGET_DIR}/${TARGET_LIB_DIR}
   )
   IF(UNIX)
      IF("${BUILD_TYPE}" STREQUAL "release" )
         IF("${FALCON_STRIP_BIN}" STREQUAL "STRIP" )
            INSTALL( CODE "EXECUTE_PROCESS( COMMAND strip --strip-unneeded ${TARGET_DIR}/${TARGET_LIB_DIR}/lib${tgt}.so )" )
         ENDIF("${FALCON_STRIP_BIN}" STREQUAL "STRIP" )
      ENDIF("${BUILD_TYPE}" STREQUAL "release" )
   ENDIF(UNIX)
ENDMACRO(FALCON_LIB_INSTALL)

MACRO(FALCON_INCLUDE_SETTINGS)
   #include also from-source builds to allow compilation from source
   INCLUDE_DIRECTORIES(BEFORE ${FALCON_SRC_TREE}/core/include)
   LINK_DIRECTORIES(BEFORE ${FALCON_SRC_TREE}/core/engine )
   # Common for all feathers
   INCLUDE_DIRECTORIES( ${FALCON_BUILD_TREE}/include)
   LINK_DIRECTORIES( ${FALCON_BUILD_TREE}/core/engine )
   #Also, include files that may be configured
   INCLUDE_DIRECTORIES( ${TARGET_DIR}/include)
   LINK_DIRECTORIES( ${TARGET_DIR}/${TARGET_LIB_DIR} )
ENDMACRO(FALCON_INCLUDE_SETTINGS)

