#!/bin/sh

func_complete_build()
{
cat << EOT

      Performing build

######################################################
EOT

cd $FALCON_BUILD_TREE
make || func_errors
cat << EOT

######################################################
#     BUILD IS SUCCESFULL                            #
#                                                    #
# You can install The Falcon Programming Language    #
# on this system issuing the                         #
#                                                    #
#     $ ./build.sh -i                                #
#                                                    #
# command                                            #
######################################################
EOT
exit 0
}

func_complete_inst()
{
cat << EOT

      Performing installation

######################################################
EOT

cd $FALCON_BUILD_TREE
make DESTDIR=$FINAL_TARGET install || func_errors
cat << EOT

######################################################
#     INSTALL IS SUCCESFULL                          #
#                                                    #
# Falcon is installed in the target path.            #
# You should run                                     #
#                                                    #
#   $ ldconfig                                       #
#                                                    #
# to update system library cache                     #
######################################################
EOT
exit 0
}

func_errors()
{
cat << EOT
######################################################
#     BUILD PROCESS FAILED!                          #
#                                                    #
# We are sorry, something went wrong. Please, verify #
# the dependencies and other pre-requisite listed    #
# in the README file are correctly set-up.           #
#                                                    #
# In that case, please report the error conditions   #
# to                                                 #
#                                                    #
#       http://www.falconpl.org                      #
#       (Contacts area)                              #
#                                                    #
# Thanks for your cooperation                        #
######################################################
EOT
exit 1
}

func_usage()
{
cat << EOT

Falcon source package build and install tool
usage:
         $0 -p <perefix> [-f <final> ] [options]
  -or-   $0 -i [options]

        -p Define standard installation prefix (absolute, i.e. /usr/local)
        -f Defines a final destination different from prefix (absolute, i.e $PWD/pkgdir)
        -i Perform installation step

   Other options
        -j   Number of processors to use in make
        -d   Compile a debug enabled version.
        -s   Do NOT strip binaries even if in reselase mode.
        -int Do NOT use internal versions of ZLIB and PCRE
        -l   set a different library path under prefix (default is lib)
        -r   Change RPATH (defaults to {prefix|final}/lib).
        -b   Recompile grammar files through bison.
        -el  Use editline

   Environment variables
      CFLAGS      - extra C flags to pass to the compiler
      CXXFLAGS    - extra C++ flags to pass to the compiler
      LDFLAGS     - extra LD flags to pass to the linker
      EXTRA_CMAKE - Extra cmake flags to be passed to the build process
EOT
}


cat << EOT
######################################################
#     Falcon source distribution build facility      #
######################################################
EOT

SPEC_RPATH=""
SPEC_LDIR="lib"
DO_INSTALL="no"
DEBUG="no"
STRIP="yes"
SKIP_BISON="yes"
USE_INTERNAL_LIBS="yes"
USE_READLINE="no"
USE_EDITLINE="no"

until [ -z "$1" ]; do
   case "$1" in
      "-p") shift; TARGET_DEST=$1;;
      "-f") shift; FINAL_DEST=$1;;
      "-i") DO_INSTALL="yes";;
      "-j") shift; export MAKEFLAGS="$MAKEFLAGS -j$1";;
      "-l") shift; SPEC_LDIR="$1";;
      "-r") shift; SPEC_RPATH="-DFALCON_RPATH:STRING=\"$1\"";;
      "-int") USE_INTERNAL_LIBS="no";;
      "-b") SKIP_BISON="";;
      "-d") DEBUG="yes";;
      "-s") STRIP="no";;
      "-el") USE_EDITLINE="yes";;
      *) func_usage ; exit 1 ;;
   esac
   shift
done

if [ -z "$TARGET_DEST" ]; then
   if [ "$DO_INSTALL" = "no" ]; then
      echo "$0: the parameter -p <prefix> must be given"
      func_usage
      exit 1 
   fi
fi

if [ "$USE_EDITLINE" = "yes" -a "$USE_READLINE" = "yes" ]; then
   echo "Only one of -rl and -el options can be speciried"
   exit 1
fi


FALCON_SRC_TREE=`pwd`
FALCON_DEVEL_TREE="$FALCON_SRC_TREE/devel"
if [ "x$DEBUG" = "xyes" ]; then
   FALCON_ACTIVE_TREE=$FALCON_SRC_TREE/devel/debug
else
   FALCON_ACTIVE_TREE=$FALCON_SRC_TREE/devel/release
fi
FALCON_BUILD_TREE=$FALCON_ACTIVE_TREE/build


# if already configured.
if [ -z "$TARGET_DEST" ]; then
  if [ -e "$FALCON_BUILD_TREE/Makefile" ]; then 
     func_complete_inst
  else
      func_usage
      exit 0
  fi

else

if [ "x$DEBUG" = "xyes" ]; then
   FALCON_BUILD_TYPE="Debug"
else
   if [ "x$STRIP" = "xno" ]; then
      FALCON_BUILD_TYPE="RelWithDebugInfo"
   else
      FALCON_BUILD_TYPE="Release"
   fi
fi


if [ -z "$FINAL_DEST" ]; then
   FINAL_DEST="$TARGET_DEST"
fi


cat << EOT

      Launching CMake configuration
######################################################
   - Target directory : $TARGET_DEST
   - Final Destination: $FINAL_DEST

EOT

mkdir -p $FALCON_BUILD_TREE
cd $FALCON_BUILD_TREE || func_error

cmake -Wno-dev \
      -DFALCON_WITH_FEATHERS="../modules/feathers" \
      -DCMAKE_BUILD_TYPE:STRING="$FALCON_BUILD_TYPE" \
      -DCMAKE_INSTALL_PREFIX:STRING="$TARGET_DEST" \
      -DFALCON_INSTALL_TREE:STRING="$TARGET_DEST" \
      -DFALCON_FINAL_DEST:STRING="$FINAL_DEST" \
      -DFALCON_SKIP_BISON:BOOL=$SKIP_BISON \
      -DFALCON_WITH_READLINE:BOOL=$USE_READLINE \
      -DFALCON_WITH_INTERNAL_EDITLINE:BOOL=$USE_EDITLINE \
      -DFALCON_LIB_DIR:STRING="$SPEC_LDIR" \
      -DFALCON_INSTALL_TREE_LIB:STRING="$SPEC_LDIR" \
      -DFALCON_WITH_INTERNAL_PCRE:BOOL=$USE_INTERNAL_LIBS \
      -DFALCON_WITH_INTERNAL_ZLIB:BOOL=$USE_INTERNAL_LIBS \
      -DFALCON_WITH_MANPAGES:BOOL=ON \
      $SPEC_RPATH \
      $EXTRA_CMAKE \
    $FALCON_SRC_TREE/core || func_errors

if [ "$DO_INSTALL" = "yes" ]; then
   func_complete_inst
else
   func_complete_build
fi


fi

