/*
	FATSort, utility for sorting FAT directory structures
	Copyright (C) 2004 Boris Leidner <fatsort@formenos.de>

	This program is free software; you can redistribute it and/or
	modify it under the terms of the GNU General Public License
	as published by the Free Software Foundation; either version 2
	of the License, or (at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the Free Software
	Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/

#include <stdlib.h>
#include <stdio.h>
#include <malloc.h>
#include <sys/types.h>
#include <string.h>
#include <unistd.h>
#include <stdarg.h>

// file attributes
#define ATTR_READ_ONLY 0x01
#define ATTR_HIDDEN 0x02
#define ATTR_SYSTEM 0x04
#define ATTR_VOLUME_ID 0x08
#define ATTR_DIRECTORY 0x10
#define ATTR_ARCHIVE 0x20
#define ATTR_LONG_NAME (ATTR_READ_ONLY | ATTR_HIDDEN | ATTR_SYSTEM | ATTR_VOLUME_ID)
#define ATTR_LONG_NAME_MASK (ATTR_READ_ONLY | ATTR_HIDDEN | ATTR_SYSTEM | ATTR_VOLUME_ID | ATTR_DIRECTORY | ATTR_ARCHIVE)

// constants for the LDIR structure
#define DE_FREE (char) 0xe5
#define DE_FOLLOWING_FREE 0x00
#define LAST_LONG_ENTRY 0x40

#define DIR_ENTRY_SIZE 32

// program information
#define INFO_PROGRAM "FATSort Utility"
#define INFO_VERSION "0.9.4"
#define INFO_AUTHOR "Boris Leidner <fatsort@formenos.de>"
#define INFO_HEADER INFO_PROGRAM " " INFO_VERSION " by " INFO_AUTHOR

// Directory entry structures
// Structure for long directory names
struct sLongDirEntry {
	u_char LDIR_Ord;		// Order of entry in sequence
	char LDIR_Name1[10];		// Chars 1-5 of long name
	u_char LDIR_Attr;		// Attributes (ATTR_LONG_NAME must be set)
	u_char LDIR_Type;		// Type
	u_char LDIR_Checksum;		// Short name checksum
	char LDIR_Name2[12];		// Chars 6-11 of long name
	u_int16_t LDIR_FstClusLO;		// Zero
	char LDIR_Name3[4];		// Chars 12-13 of long name
} __attribute__((packed));

// Structure for old short directory names
struct sShortDirEntry {
	char DIR_Name[11];		// Short name
	u_char DIR_Atrr;		// File attributes
	u_char DIR_NTRes;		// Reserved for NT
	u_char DIR_CrtTimeTenth;	// Time of creation in ms
	u_int16_t DIR_CrtTime;		// Time of creation
	u_int16_t DIR_CrtDate;		// Date of creation
	u_int16_t DIR_LstAccDate;		// Last access date
	u_int16_t DIR_FstClusHI;		// Hiword of first cluster
	u_int16_t DIR_WrtTime;		// Time of last write
	u_int16_t DIR_WrtDate;		// Date of last write
	u_int16_t DIR_FstClusLO;		// Loword of first cluster
	u_int32_t DIR_FileSize;		// file size in bytes
} __attribute__((packed));

union sDirEntry {
	struct sShortDirEntry ShortDirEntry;
	struct sLongDirEntry LongDirEntry;
} __attribute__((packed));

// Bootsector structures
// FAT12 and FAT16
struct sFAT12_16 {
	u_char BS_DrvNum;		// Physical drive number
	u_char BS_Reserved;		// Current head
	u_char BS_BootSig;		// Signature
	u_int32_t BS_VolID;		// Volume ID
	char BS_VolLab[11];		// Volume Label
	char BS_FilSysType[8];		// FAT file system type (e.g. FAT, FAT12, FAT16, FAT32)
} __attribute__((packed));

// FAT32
struct sFAT32 {
	u_int32_t BPB_FATSz32;		// Sectors per FAT
	u_int16_t BPB_ExtFlags;		// Flags
	u_int16_t BPB_FSVer;		// Version
	u_int32_t BPB_RootClus;		// Root Directory Cluster
	u_int16_t BPB_FSInfo;		// Sector of FSInfo structure
	u_int16_t BPB_BkBootSec;		// Sector number of the boot sector copy in reserved sectors
	char BPB_Reserved[12];		// for future expansion
	char BS_DrvNum;			// see fat12/16
	char BS_Reserved1;		// see fat12/16
	char BS_BootSig;		// ...
	u_int32_t BS_VolID;
	char BS_VolLab[11];
	char BS_FilSysType[8];
} __attribute__((packed));

union sFATxx {
	struct sFAT12_16 FAT12_16;
	struct sFAT32 FAT32;
} __attribute__((packed));

// First sector = boot sector
struct bootsector {
	char BS_JmpBoot[3];		// Jump instruction (to skip over header on boot)
	char BS_OEMName[8];		// OEM Name (padded with spaces)
	u_int16_t BPB_BytesPerSec;	// Bytes per sector
	u_char BPB_SecPerClus;		// Sectors per cluster
	u_int16_t BPB_RsvdSecCnt;		// Reserved sector count (including boot sector)
	u_char BPB_NumFATs;		// Number of file allocation tables
	u_int16_t BPB_RootEntCnt;		// Number of root directory entries
	u_int16_t BPB_TotSec16;		// Total sectors (bits 0-15)
	u_char BPB_Media;		// Media descriptor
	u_int16_t BPB_FATSz16;		// Sectors per file allocation table
	u_int16_t BPB_SecPerTrk;		// Sectors per track
	u_int16_t BPB_NumHeads;		// Number of heads
	u_int32_t BPB_HiddSec;		// Hidden sectors
	u_int32_t BPB_TotSec32;		// Total sectors (bits 16-47)
	union sFATxx FATxx;
} __attribute__((packed));

struct sLongDirEntryList {
/*
	list structures for directory entries
	list structure for a long name entry
*/
	struct sLongDirEntry *lde;
	struct sLongDirEntryList *next;
};

struct sDirEntryList {
/*
	list structure for every file with short
	name entries and long name entries
*/
	char *sname, *lname;		// short and long name strings
	struct sShortDirEntry *sde;	// short dir entry
	struct sLongDirEntryList *ldel;	// long name entries in a list
	int32_t entries;			// number of entries
	struct sDirEntryList *next;	// next dir entry
};

const struct sDirEntryList __INITDIRLIST__ = {0};

// --------- option flags

u_int32_t OPT_VERSION, OPT_HELP, OPT_INFO, OPT_QUIET, OPT_IGNORE_CASE, OPT_ORDER, OPT_LIST;

// ----------


void infomsg(char *str, ...) {
/*
	info messages that can be muted with an command line option
*/
	va_list argzeiger;

	if (!OPT_QUIET) {
		va_start(argzeiger,str);
		vprintf(str,argzeiger);
		va_end(argzeiger);
	}

}

void myerror(const char *prefix, const char *msg) {
/*
	Prints error message to stderr
*/

	fprintf(stderr, "%s: %s\n", prefix, msg);

}

// ----------
// List functions

struct sDirEntryList *
	newDirEntry(char *sname, char *lname, struct sShortDirEntry *sde, struct sLongDirEntryList *ldel, u_int32_t entries) {
/*
	create a new directory entry holder
*/

	struct sDirEntryList *tmp;

	if ((tmp=malloc(sizeof(struct sDirEntryList)))==NULL) {
		perror(__func__);
		exit(1);
	}
	if ((tmp->sname=malloc(strlen(sname)+1))==NULL) {
		perror(__func__);
		exit(1);
	}
	strcpy(tmp->sname, sname);
	if ((tmp->lname=malloc(strlen(lname)+1))==NULL) {
		perror(__func__);
		exit(1);
	}
	strcpy(tmp->lname, lname);

	if ((tmp->sde=malloc(sizeof(struct sShortDirEntry)))==NULL) {
		perror(__func__);
		exit(1);
	}
	memcpy(tmp->sde, sde, DIR_ENTRY_SIZE);
	tmp->ldel=ldel;
	tmp->entries=entries;
	tmp->next = NULL;
	return tmp;
}

struct sLongDirEntryList *
	newLongDirEntry(struct sLongDirEntry *lde) {
/*
	create new long dir name list entry
*/

	struct sLongDirEntryList *tmp;

	if ((tmp=malloc(sizeof(struct sLongDirEntryList)))==NULL) {
		perror(__func__);
		exit(1);
	}
	if ((tmp->lde=malloc(sizeof(struct sLongDirEntry)))==NULL) {
		perror(__func__);
		exit(1);
	}
	memcpy(tmp->lde, lde, DIR_ENTRY_SIZE);
	tmp->next = NULL;
	return tmp;
}

struct sLongDirEntryList *
	insertLongDirEntryList(struct sLongDirEntryList *new, struct sLongDirEntryList *list) {
/*
	inserts a long directory entry to list
*/

	struct sLongDirEntryList *tmp;

	if (list != NULL) {
		tmp=list;
		while(tmp->next != NULL) {
			tmp=tmp->next;
		}
		tmp->next=new;
		return list;
	} else {
		return new;
	}

}

int32_t cmpEntries(struct sDirEntryList *de1, struct sDirEntryList *de2) {
/*
	compare two directory entries

*/

	char *s1,*s2;

	if ((de1->lname != NULL) && (de1->lname[0] != '\0')) {
		s1=de1->lname;
	} else {
		s1=de1->sname;
	}
	if ((de2->lname != NULL) && (de2->lname[0] != '\0')) {
		s2=de2->lname;
	} else {
		s2=de2->sname;
	}

	// directories will be put above normal files
	if (OPT_ORDER == 0) {
		if ((de1->sde->DIR_Atrr & ATTR_DIRECTORY) && !(de2->sde->DIR_Atrr & ATTR_DIRECTORY)) {
			return -1;
		} else if (!(de1->sde->DIR_Atrr & ATTR_DIRECTORY) && (de2->sde->DIR_Atrr & ATTR_DIRECTORY)) {
			return 1;
		}
	} else if (OPT_ORDER == 1) {
		if ((de1->sde->DIR_Atrr & ATTR_DIRECTORY) && !(de2->sde->DIR_Atrr & ATTR_DIRECTORY)) {
			return 1;
		} else if (!(de1->sde->DIR_Atrr & ATTR_DIRECTORY) && (de2->sde->DIR_Atrr & ATTR_DIRECTORY)) {
			return -1;
		}
	}

	if (OPT_IGNORE_CASE) {
		return strcasecmp(s1, s2);
	} else {
		return strcmp(s1, s2);
	}
}

void insertDirEntryList(struct sDirEntryList *new, struct sDirEntryList *list) {
/*
	inserts a directory entry in list
*/
	struct sDirEntryList *tmp, *dummy;

	tmp=list;
	while((tmp->next != NULL) && (cmpEntries(new, tmp->next) >= 0)) {
		tmp=tmp->next;
	}
	dummy=tmp->next;
	tmp->next=new;
	new->next=dummy;
}

// ----------

int32_t read_bootsector(FILE *fd, struct bootsector *bs) {
/*
	reads bootsector
*/

	if (fread(bs, sizeof(struct bootsector), 1, fd) < 1) {
		if (feof(fd)) {
			myerror(__func__, "Boot sector is too short!");
		} else {
			myerror(__func__, "Unable to read from file!");
		}
		return -1;
	}
	return 0;
}

u_int32_t getCountOfClusters(struct bootsector *bs) {
/*
	calculates count of clusters
*/
	u_int32_t RootDirSectors, FATSz, TotSec, DataSec, FATType;

	RootDirSectors = ((bs->BPB_RootEntCnt * 32) + (bs->BPB_BytesPerSec - 1));
	RootDirSectors = RootDirSectors / (bs->BPB_BytesPerSec);

	if (bs->BPB_FATSz16 != 0) {
		FATSz = bs->BPB_FATSz16;
	} else {
		FATSz = bs->FATxx.FAT32.BPB_FATSz32;
	}
	if (bs->BPB_TotSec16 != 0) {
		TotSec = bs->BPB_TotSec16;
	} else {
		TotSec = bs->BPB_TotSec32;
	}
	DataSec = TotSec - (bs->BPB_RsvdSecCnt + (bs->BPB_NumFATs * FATSz) + RootDirSectors);

	return DataSec / bs->BPB_SecPerClus;
}

u_int32_t getFATType(struct bootsector *bs) {
/*
	retrieves FAT type
*/
	u_int32_t CountOfClusters;

	CountOfClusters=getCountOfClusters(bs);
	if (CountOfClusters < 4096) { // FAT12!
		return 12;
	} else if (CountOfClusters < 65525) { // FAT16!
		return 16;
	} else { // FAT23!
		return 32;
	}
}

int32_t putFATEntry(FILE *fd, struct bootsector *bs, u_int32_t cluster, u_int32_t data) {
/*
	write a FAT entry
*/
	u_int32_t FATOffset, FATType, FATSz, BSOffset;
	u_int32_t value, i;

	FATType = getFATType(bs);

	if (bs->BPB_FATSz16 != 0) {
		FATSz = bs->BPB_FATSz16;
	} else {
		FATSz = bs->FATxx.FAT32.BPB_FATSz32;
	}

	if (FATType == 16) {
		FATOffset = cluster * 2;
		BSOffset = bs->BPB_RsvdSecCnt * bs->BPB_BytesPerSec + FATOffset;
		fseek(fd, BSOffset, SEEK_SET);
		if (fwrite(&data, 2, 1, fd)<1) {
			myerror(__func__, "Unable to write to file!");
			return -1;
		}
	} else if (FATType == 32) {
		FATOffset = cluster * 4;
		BSOffset = bs->BPB_RsvdSecCnt * bs->BPB_BytesPerSec + FATOffset;
		if (getFATEntry(fd, bs, cluster, &value)==-1) {
			myerror(__func__, "Unable to read FAT entry!");
			return -1;
		}
		value = (value & 0xf0000000) | (data & 0x0fffffff);
		for(i=0; i<bs->BPB_NumFATs; i++) {
			fseek(fd, BSOffset+i*FATSz*bs->BPB_BytesPerSec, SEEK_SET);
			if (fwrite(&value, 4, 1, fd) < 1) {
				myerror(__func__, "Unable to write to file!");
				return -1;
			}
		}
	} else {
		myerror(__func__, "FAT12 is not supported!");
		return -1;
	}

	return 0;

}

int32_t getFATEntry(FILE *fd, struct bootsector *bs, u_int32_t cluster, u_int32_t *data) {
/*
	retrieves FAT entry for a cluster number
*/
	u_int32_t FATOffset, FATType, FATSz, BSOffset;

	*data=0;
	FATType = getFATType(bs);

	if (bs->BPB_FATSz16 != 0) {
		FATSz = bs->BPB_FATSz16;
	} else {
		FATSz = bs->FATxx.FAT32.BPB_FATSz32;
	}

	if (FATType == 16) {
		FATOffset = cluster * 2;
		BSOffset = bs->BPB_RsvdSecCnt * bs->BPB_BytesPerSec + FATOffset;
		if (fseek(fd, BSOffset, SEEK_SET) == -1) {
			myerror(__func__, "Seek error!");
			return -1;
		}
		if (fread(data, 2, 1, fd)<1) {
			myerror(__func__, "Unable to read from file!");
			return -1;
		}
	} else if (FATType == 32) {
		FATOffset = cluster * 4;
		BSOffset = bs->BPB_RsvdSecCnt * bs->BPB_BytesPerSec + FATOffset;
		if (fseek(fd, BSOffset, SEEK_SET) == -1) {
			myerror(__func__, "Seek error!");
			return -1;
		}
		if (fread(data, 4, 1, fd) < 1) {
			myerror(__func__, "Unable to read from file!");
			return -1;
		}
		*data = *data & 0x0fffffff;
	} else {
		myerror(__func__, "FAT12 is not supported!");
		return -1;
	}

	return 0;

}

void parseLongFilenamePart(struct sLongDirEntry *lde, char *str) {
/*
	retrieves a part of a long fil ename from a
	directory entry
	(thanks to M$ for this ugly hack...)
*/

	u_int32_t len=0;//strlen(str);
	str[len]=(char) (*(&lde->LDIR_Ord+1));
	str[len+1]=(char) (*(&lde->LDIR_Ord+3));
	str[len+2]=(char) (*(&lde->LDIR_Ord+5));
	str[len+3]=(char) (*(&lde->LDIR_Ord+7));
	str[len+4]=(char) (*(&lde->LDIR_Ord+9));
	str[len+5]=(char) (*(&lde->LDIR_Ord+14));
	str[len+6]=(char) (*(&lde->LDIR_Ord+16));
	str[len+7]=(char) (*(&lde->LDIR_Ord+18));
	str[len+8]=(char) (*(&lde->LDIR_Ord+20));
	str[len+9]=(char) (*(&lde->LDIR_Ord+22));
	str[len+10]=(char) (*(&lde->LDIR_Ord+24));
	str[len+11]=(char) (*(&lde->LDIR_Ord+28));
	str[len+12]=(char) (*(&lde->LDIR_Ord+30));
	str[len+13]=0;
}

void parseShortFilename(struct sShortDirEntry *sde, char *str) {
/*
	parses the short name of a file
*/
	char *s;
	strncpy(str, sde->DIR_Name, 8);
	str[8]='\0';
	s=strchr(str, ' ');
	if (s!=NULL) s[0]='\0';
	if ((char)(*(sde->DIR_Name+8)) != ' ') {
		strcat(str, ".");
		strncat(str, sde->DIR_Name+8, 3);
		str[12]='\0';
	}
}

int32_t parseEntry(FILE *fd, union sDirEntry *de) {
/*
	parses one directory entry
*/
	if ((fread(de, DIR_ENTRY_SIZE, 1, fd)<1)) {
		myerror(__func__, "Unable to read from file!");
		return -1;
	}
	if (de->LongDirEntry.LDIR_Attr == 0) return 0; // no more entries
	// long dir entry
	if ((de->LongDirEntry.LDIR_Attr & ATTR_LONG_NAME_MASK) == ATTR_LONG_NAME) return 2;

	return 1; // short dir entry
}

u_int32_t getClusterOffset(struct bootsector *bs, u_int32_t cluster) {
/*
	returns the offset of a specific cluster in the
	data region of the file system
*/
	u_int32_t FATSz, RootDirSectors, FirstDataSector;

	if (bs->BPB_FATSz16 != 0) {
		FATSz = bs->BPB_FATSz16;
	} else {
		FATSz = bs->FATxx.FAT32.BPB_FATSz32;
	}

	RootDirSectors = ((bs->BPB_RootEntCnt * DIR_ENTRY_SIZE) + (bs->BPB_BytesPerSec - 1)) / bs->BPB_BytesPerSec;
	FirstDataSector = (bs->BPB_RsvdSecCnt + (bs->BPB_NumFATs * FATSz) + RootDirSectors);

	return (((cluster - 2) * bs->BPB_SecPerClus) + FirstDataSector) * bs->BPB_BytesPerSec;

}

int32_t parseClusterChain(FILE *fd, struct bootsector *bs, int32_t chain[], struct sDirEntryList *list) {
/*
	parses a cluster chain and puts directory entries to list
*/
	int32_t i=0, j, maxEntries, entries=0, ret;
	union sDirEntry de;
	struct sLongDirEntryList *llist;
	char tmp[512], dummy[512], sname[512], lname[512];

	maxEntries = bs->BPB_SecPerClus * bs->BPB_BytesPerSec / DIR_ENTRY_SIZE;

	llist = NULL;
	lname[0]='\0';
	while (chain[i] != -1) {
		fseek(fd, getClusterOffset(bs, chain[i]), SEEK_SET);
		for (j=1;j<=maxEntries;j++) {
			entries++;
			ret=parseEntry(fd, &de);
			if (ret == -1) {
				myerror(__func__, "Could not read directory entry!");
				return -1;
			} else if (ret == 0) {
				break;
			} else if (ret == 2) {
				parseLongFilenamePart(&de.LongDirEntry, tmp);
				// insert long dir entry in list
				llist=insertLongDirEntryList(newLongDirEntry(&de.LongDirEntry), llist);
				strcpy(dummy, tmp);
				strcat(dummy, lname);
				strcpy(lname, dummy);
			} else {
				parseShortFilename(&de.ShortDirEntry, sname);
				if (OPT_LIST && strcmp(sname, ".") && strcmp (sname, "..") && (sname[0] != DE_FREE)) {
					printf("%s\n", (lname[0] != '\0') ? lname : sname);
				}
				insertDirEntryList(newDirEntry(sname, lname, &de.ShortDirEntry, llist, entries), list);
				entries=0;
				llist = NULL;
				lname[0]='\0';
			}
		}
		i++;
	}

	return 0;
}

int32_t parseDirEntry(FILE *fd, struct sDirEntryList *list, u_int32_t *entries) {
/*
	parses an entry of a directory structure
*/

	char sname[512], lname[512], tmp[512], dummy[512];
	union sDirEntry de;
	u_int32_t count=0;

	struct sLongDirEntryList *llist = NULL;

	if ((fread(&de, DIR_ENTRY_SIZE, 1, fd)<1)) {
		myerror(__func__, "Unable to read from file!");
		return -1;
	}
	if (de.LongDirEntry.LDIR_Ord == 0) {
		return 1;
	}

	lname[0]='\0';
	*entries=0;
	while (((de.LongDirEntry.LDIR_Attr & ATTR_LONG_NAME_MASK) == ATTR_LONG_NAME) ){//&&
		//(de.LongDirEntry.LDIR_Ord != DE_FREE))  {

		parseLongFilenamePart(&de.LongDirEntry, tmp);

		// insert long dir entry in list
		llist=insertLongDirEntryList(newLongDirEntry(&de.LongDirEntry), llist);

		count++;
		(*entries)++;
		strcpy(dummy, tmp);
		strcat(dummy, lname);
		strcpy(lname, dummy);
		if (fread(&de, DIR_ENTRY_SIZE, 1, fd)<1) {
			myerror(__func__, "Unable to read from file!");
			return -1;
		}
	}
	//if (de.LongDirEntry.LDIR_Ord != DE_FREE) {
		parseShortFilename(&de.ShortDirEntry, sname);
		(*entries)++;

		insertDirEntryList(newDirEntry(sname, lname, &de.ShortDirEntry, llist, *entries), list);

		if (OPT_LIST && strcmp(sname, ".") && strcmp (sname, "..") && (sname[0] != DE_FREE)) {
			printf("%s\n", (lname[0] != '\0') ? lname : sname);
		}

		return 2;
	//}

	return 0;

}

int32_t writeList(FILE *fd, struct sDirEntryList *list) {
/*
	writes directory entries to file
*/
	struct sLongDirEntryList *tmp;

	while(list->next!=NULL) {
		tmp=list->next->ldel;
		while(tmp != NULL) {
			if (fwrite(tmp->lde, DIR_ENTRY_SIZE, 1, fd)<1) {
				perror(__func__);
				return -1;
			}
			tmp=tmp->next;
		}
		if (fwrite(list->next->sde, DIR_ENTRY_SIZE, 1, fd)<1) {
			perror(__func__);
			return -1;
		}
		list=list->next;
	}

	return 0;
}

int32_t sort_FAT16_rootdir(FILE *fd, struct bootsector *bs) {
/*
	sorts the root directory of a FAT16 file system
*/

	u_int32_t FATSz, FATType, BSOffset, i, ret;
	u_int32_t entries,count=0, c, value;
	union sDirEntry data;
	char str[512], lname[512], newpath[1024];
	char empty[32]={0};

	struct sDirEntryList list = __INITDIRLIST__;
	struct sDirEntryList *p;

	if (bs->BPB_FATSz16 != 0) {
		FATSz = bs->BPB_FATSz16;
	} else {
		FATSz = bs->FATxx.FAT32.BPB_FATSz32;
	}

	BSOffset = (bs->BPB_RsvdSecCnt + bs->BPB_NumFATs * FATSz)* bs->BPB_BytesPerSec;

	fseek(fd, BSOffset, SEEK_SET);

	if (OPT_LIST) {
		printf("/\n");
	}
	// read all entries and parse it to the list
	for (i=1;i<=bs->BPB_RootEntCnt;i++) {
		ret=parseDirEntry(fd, &list, &entries);
		count+=entries;
		if (ret == 1) {
			break;
		} else if (ret == -1) {
			return -1;
		}
		// well, one entry read, but not matched
		if (entries == 0) count++;
	}

	if (!OPT_LIST) {
		infomsg("Sorting directory /\n");

		fseek(fd, BSOffset, SEEK_SET);

		// write the sorted entries back to the fs
		if (writeList(fd, &list) == -1) {
			myerror(__func__, "Could not write FAT16 root directory!");
			return -1;
		}
		// wipe all follwing entries
		if (bs->BPB_RootEntCnt - count > 0) {
			if (fwrite(&empty, DIR_ENTRY_SIZE, 1, fd)<1) {
				perror(__func__);
				return -1;
			}
		}
	} else {
		printf("\n");
	}

	// sort sub directories
	p=list.next;
	while (p != NULL) {
		if ((p->sde->DIR_Atrr & ATTR_DIRECTORY) &&
			(p->sde->DIR_Name[0] != DE_FREE) &&
			!(p->sde->DIR_Atrr & ATTR_VOLUME_ID) &&
			(strcmp(p->sname, ".")) && strcmp(p->sname, "..")) {
			c= (p->sde->DIR_FstClusHI * 65536 + p->sde->DIR_FstClusLO);
			if (getFATEntry(fd, bs, c, &value) == -1) {
				myerror(__func__, "Could not read FAT entry!");
				return -1;
			}
			//printf("Dir: %s, Cluster: 0x%x\n", p->sname, c);
			strcpy(newpath, "/");
			if ((p->lname != NULL) && (p->lname[0] != '\0')) {
				strcat(newpath, p->lname);
				strcat(newpath, "/");
			} else {
				strcat(newpath, p->sname);
				strcat(newpath, "/");
			}

			if (sortClusterChain(fd, bs, c, newpath) == -1) {
				myerror(__func__, "Unable to sort directory!");
				return -1;
			}
		}
		p=p->next;
	}


	return 0;
}

int32_t getClusterChain(FILE *fd, struct bootsector *bs, u_int32_t startCluster, int32_t chain[]) {
/*
	retrieves an array of all clusters in a cluster chain
	starting with startCluster
*/
	u_int32_t cluster, FATType;
	u_int32_t data,i=0;

	if (getFATEntry(fd, bs, startCluster, &data)) {
		myerror(__func__, "Could not read FAT entry for startCluster!");
		return -1;
	}
	if (data==0) {
		myerror(__func__, "startCluster should not be free!");
		return -1;
	}

	cluster=startCluster;

	FATType=getFATType(bs);

	if (FATType == 12) {
		myerror(__func__, "FAT12 is not supported!");
		return -1;
	} else if (FATType == 16) {
		do {
			chain[i++]=cluster;
			if (getFATEntry(fd, bs, cluster, &data)) {
				myerror(__func__, "Could not read FAT entry for cluster!");
				return -1;
			}
			cluster=data;
		} while (cluster < 0xfff8);
		chain[i]=-1;
	} else {
		do {
			chain[i++]=cluster;
			if (getFATEntry(fd, bs, cluster, &data)) {
				myerror(__func__, "Could not read FAT entry");
				return -1;
			}
			cluster=data;
		} while (((cluster & 0x0fffffff) != 0x0ff8fff8) &&
			 ((cluster & 0x0fffffff) < 0x0ffffff8));
		chain[i]=-1;
	}

	return i;
}

int32_t writeClusterChain(FILE *fd, struct bootsector *bs, struct sDirEntryList *list, int32_t chain[]) {
/*
	writes all entries from list to the cluster chain
*/
	int32_t i=0, entries=0, cluster=0;
	u_int32_t MaxEntries;
	struct sLongDirEntryList *tmp;
	struct sDirEntryList *p=list->next;
	char empty[DIR_ENTRY_SIZE]={0};

	MaxEntries = bs->BPB_SecPerClus * bs->BPB_BytesPerSec / DIR_ENTRY_SIZE;
	if (fseek(fd, getClusterOffset(bs, chain[0]), SEEK_SET)==-1) {
		myerror(__func__, "Seek error!");
		return -1;
	}

	while(p != NULL) {
		if (entries+p->entries <= MaxEntries) {
			tmp=p->ldel;
			for (i=1;i<p->entries;i++) {
				if (fwrite(tmp->lde, DIR_ENTRY_SIZE, 1, fd)<1) {
					perror(__func__);
					return -1;
				}
				tmp=tmp->next;
			}
			if (fwrite(p->sde, DIR_ENTRY_SIZE, 1, fd)<1) {
				perror(__func__);
				return -1;
			}
			entries+=p->entries;
		} else {
			tmp=p->ldel;
			for (i=1;i<=MaxEntries-entries;i++) {
				if (fwrite(tmp->lde, DIR_ENTRY_SIZE, 1, fd)<1) {
					perror(__func__);
					return -1;
				}
				tmp=tmp->next;
			}
			cluster++; entries=p->entries - (MaxEntries - entries);	// next cluster
			if (fseek(fd, getClusterOffset(bs, chain[cluster]), SEEK_SET)==-1) {
				myerror(__func__, "Seek error!");
				return -1;
			}
			while(tmp!=NULL) {
				if (fwrite(tmp->lde, DIR_ENTRY_SIZE, 1, fd)<1) {
					perror(__func__);
					return -1;
				}
				tmp=tmp->next;
			}
			if (fwrite(p->sde, DIR_ENTRY_SIZE, 1, fd)<1) {
				perror(__func__);
				return -1;
			}
		}
		p=p->next;

	}
	if (entries < MaxEntries) {
		if (fwrite(empty, DIR_ENTRY_SIZE, 1, fd)<1) {
			perror(__func__);
			return -1;
		}
	}

	return 0;

}

int32_t sortClusterChain(FILE *fd, struct bootsector *bs, u_int32_t cluster, char *path) {
/*
	sorts the directory entries in a cluster
*/
	u_int32_t FATSz, FATType, BSOffset, i, ret, cnr, clen, nclen, value, c;
	u_int32_t RootDirSectors, FirstDataSector, ClusterOffset, ClusterSize;
	int32_t ClusterChain[512];
	u_int32_t  entries, count, ccount;
	union sDirEntry data;
	char str[512], lname[512], newpath[1024];

	char empty[32]={0};

	struct sDirEntryList list = __INITDIRLIST__;
	struct sDirEntryList *p;

	if ((clen=getClusterChain(fd, bs, cluster, ClusterChain)) == -1 ) {
		myerror(__func__, "Could not read cluster chain!");
		return -1;
	}

/*	i=0;
	while(ClusterChain[i]!=-1) {
		printf("->%x", getClusterOffset(bs, ClusterChain[i++]));
	}
	printf("\n");
*/
	if (!OPT_LIST) {
		parseClusterChain(fd, bs, ClusterChain, &list);
		infomsg("Sorting directory %s\n", path);
		if (writeClusterChain(fd, bs, &list, ClusterChain) == -1) {
			myerror(__func__, "Could not write cluster chain!");
			return -1;
		}

	} else {
		printf("%s\n", path);
		parseClusterChain(fd, bs, ClusterChain, &list);
		printf("\n");
	}

	// sort sub directories
	p=list.next;
	while (p != NULL) {
		if ((p->sde->DIR_Atrr & ATTR_DIRECTORY) &&
			(p->sde->DIR_Name[0] != DE_FREE) &&
			!(p->sde->DIR_Atrr & ATTR_VOLUME_ID) &&
			(strcmp(p->sname, ".")) && strcmp(p->sname, "..")) {

			c= (p->sde->DIR_FstClusHI * 65536 + p->sde->DIR_FstClusLO);
			if (getFATEntry(fd, bs, c, &value) == -1) {
				myerror(__func__, "Could not read FAT entry!");
				return -1;
			}

			strcpy(newpath, path);
			if ((p->lname != NULL) && (p->lname[0] != '\0')) {
				strcat(newpath, p->lname);
				strcat(newpath, "/");
			} else {
				strcat(newpath, p->sname);
				strcat(newpath, "/");
			}

			if (sortClusterChain(fd, bs, c, newpath) == -1) {
				myerror(__func__, "Unable to sort directory!");
				return -1;
			}

		}
		p=p->next;
	}

	return 0;
}

int32_t printFSInfo(char *filename) {
/*
	print file system information
*/
	u_int32_t FATSz, value;
	FILE *fd;
	struct bootsector bs;

	printf("\t- File system information -\n");

	if ((fd=fopen(filename, "r+")) == NULL) {
		perror(__func__);
		return -1;
	}

	// read boot sector
	if (read_bootsector(fd, &bs)) {
		myerror(__func__, "Could not read boot sector!");
		return -1;
	}

	if (bs.BPB_FATSz16 != 0) {
		FATSz = bs.BPB_FATSz16;
	} else {
		FATSz = bs.FATxx.FAT32.BPB_FATSz32;
	}

	printf("Device:\t\t\t\t%s\n", filename);
	fflush(stdout);
	printf("Type:\t\t\t\tFAT%u\n", getFATType(&bs));
	fflush(stdout);
	printf("Sector size:\t\t\t%u bytes\n", bs.BPB_BytesPerSec);
	fflush(stdout);
	printf("FAT size:\t\t\t%u sectors (%u bytes)\n", FATSz, FATSz * bs.BPB_BytesPerSec);
	printf("Cluster size:\t\t\t%u bytes\n", bs.BPB_SecPerClus * bs.BPB_BytesPerSec);
	printf("Cluster count:\t\t\t%u\n", getCountOfClusters(&bs));
	if (getFATType(&bs) == 32) {
		getFATEntry(fd, &bs, bs.FATxx.FAT32.BPB_RootClus, &value);
		printf("FAT32 root directory first cluster: 0x%x, Data offset: 0x%x, FAT entry: 0x%x\n",
			bs.FATxx.FAT32.BPB_RootClus,
			getClusterOffset(&bs, bs.FATxx.FAT32.BPB_RootClus), value);
	}

	fclose(fd);

}

int32_t sort_fs(char *filename) {
/*
	sort FAT file system
*/
	FILE *fd;
	struct bootsector bs;
	char data[512];

	int32_t cluster;
	int32_t FATType;
	int32_t chain[512];
	int32_t i;

	if ((fd=fopen(filename, "r+")) == NULL) {
		perror(__func__);
		return -1;
	}

	// read boot sector
	if (read_bootsector(fd, &bs)) {
		myerror(__func__, "Could not read boot sector!");
		return -1;
	}

	FATType = getFATType(&bs);

/*	printf("----- FAT%u -----\n", FATType);
	printf("Cluster size: %u bytes\n", bs.BPB_SecPerClus * bs.BPB_BytesPerSec);*/
	infomsg("File system: FAT%u.\n\n", FATType);

	if (FATType == 12) {
		// FAT12
		// sorry, too complicated ;)
		myerror(__func__, "FAT12 is not supported!");
	} else if (FATType == 16) {
		// FAT16
		// root directory has fixed size and position
		sort_FAT16_rootdir(fd, &bs);
	} else {
		// FAT32
		// root directory lies in cluster chain,
		// so sort it like all other directories
		sortClusterChain(fd, &bs, bs.FATxx.FAT32.BPB_RootClus, "/");
	}

	fclose(fd);
	return 0;
}

int32_t main(u_int32_t argc, char *argv[]) {
/*
	parse arguments and options and start sorting
*/
	FILE *fd;
	struct bootsector bs;
	char c;

	opterr=0;
	while ((c=getopt(argc, argv, "ivhqco:l")) != -1) {
		switch(c) {
			case 'i' : OPT_INFO = 1; break;
			case 'v' : OPT_VERSION = 1; break;
			case 'h' : OPT_HELP = 1; break;
			case 'q' : OPT_QUIET = 1; break;
			case 'c' : OPT_IGNORE_CASE = 1; break;
			case 'l' : OPT_LIST = 1; break;
			case 'o' :
				switch(optarg[0]) {
					case 'd': OPT_ORDER=0; break;
					case 'f': OPT_ORDER=1; break;
					case 'a': OPT_ORDER=2; break;
					default:
						printf("Unknown flag '%c' for option 'o'.\n", optarg[0]);
						printf("Use -h for more help.\n");
						return -1;
				}
				break;
			default :
				printf("Unknown option '%c'.\n", optopt);
				printf("Use -h for more help.\n");
				return -1;
		}
	}
	if (OPT_HELP) {
		printf(	INFO_HEADER "\n"
			"Usage: fatsort [options] device\n"
			"\n"
			"Options:\n"
			"\t-c\t Ignore case of file names\n"
			"\t-h\t Print some help\n"
			"\t-i\t Print file system information only\n"
			"\t-l\t Print current order of files only\n"
			"\t-o flag\t Sort order of files where flag is\n"
			"\t\t\td : directories first (default)\n"
			"\t\t\tf : files first\n"
			"\t\t\ta : files and directories are not distinguished\n"
			"\t-q\t Be quiet\n"
			"\t-v\t Print version information\n"
			"\n"
			"Device must be a FAT16 or FAT32 file system. FAT12 is not supported yet.\n"
			"\n"
			"Example: fatsort /dev/sda\n"
			"\n"
			"NOTE: THE FILE SYSTEM MUST BE CONSISTENT, OTHERWISE YOU CAN DAMAGE IT!!!\n"
			"USE THIS PROGRAM AT YOUR OWN RISK!\n"
			);
		return 0;
	} else if (OPT_VERSION) {
		printf(INFO_VERSION "\n" );
		return 0;
	} else if (optind < argc -1) {
		printf("Too many arguments!\n");
		printf("Use -h for more help.\n");
		return -1;
	} else if (optind == argc) {
		printf("Device must be given!\n");
		printf("Use -h for more help.\n");
		return -1;
	}

	if (OPT_INFO) {
		infomsg(INFO_HEADER "\n\n");
		printFSInfo(argv[optind]);
	} else {
		infomsg(INFO_HEADER "\n\n");
		return sort_fs(argv[optind]);
	}
	return 0;
}
