/* $Id: signals.c 4349 2009-02-05 09:50:47Z sand $
 *
 * Signal/Driver handling related functions.
 *
 * Copyright (C) 2008-2009 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#include "signals.h"
#include <stdlib.h>
#include <assert.h>
#include "glue-main.h" /* for time_virt */

/** transaction element of a driver */
struct drv_trans {
	/** simulation time at which the transaction is to happen */
	universal_integer sim_time;
	/** value of the driver at given time. */
	union fauhdli_value val;
};

/** ordering function for transactions of a driver in ascending simulation 
 *  time order.
 */
static int
__attribute__((__pure__))
drv_trans_compare(const void *_t1, const void *_t2)
{
	const struct drv_trans *t1 = (const struct drv_trans *)_t1;
	const struct drv_trans *t2 = (const struct drv_trans *)_t2;

	if (t1->sim_time < t2->sim_time) {
		return -1;
	}

	if (t1->sim_time == t2->sim_time) {
		return 0;
	}

	return 1;
}

static void
driver_connect_non_foreign(
	struct driver *driver,
	struct signal *_signal
)
{
	assert(driver != NULL);
	assert(_signal != NULL);

	slset_add(_signal->connected_drivers, driver);
	driver->connected_signal = _signal;
}

bool
driver_connect(
	struct driver *driver, 
	struct signal *_signal,
	struct glue_vhdl *glue_vhdl
)
{
	driver_connect_non_foreign(driver, _signal);

	if (_signal->is_foreign) {
		glue_vhdl_connect_out(glue_vhdl, 
					_signal->foreign_id,
					_signal->value,
					driver);
		driver->foreign_out = true;
		return true;
	}
	return false;
}

static void
driver_disconnect(struct driver *driver)
{
	if (driver->connected_signal == NULL) {
		/* not connected */
		return;
	}

	assert(driver->connected_signal->connected_drivers != NULL);
	slset_remove(driver->connected_signal->connected_drivers, driver);
	driver->connected_signal = NULL;
}

struct driver *
driver_create(union fauhdli_value init)
{
	struct driver *ret;
	ret = malloc(sizeof(struct driver));
	assert(ret != NULL);
	ret->driving_value = init;
	ret->connected_signal = NULL;
	ret->active = false;
	ret->transactions = slset_create(drv_trans_compare);
	ret->foreign_out = false;
	return ret;
}

void
driver_destroy(struct driver *driver)
{
	driver_disconnect(driver);
	slset_destroy_data(driver->transactions);
	free(driver);
}

void
driver_update(
	struct driver *driver,
	union fauhdli_value val,
	universal_integer sim_time
)
{
	struct drv_trans *t;
	/* FIXME */
	driver->driving_value = val;
	driver->active = true;

	t = malloc(sizeof(struct drv_trans));
	assert(t != NULL);

	t->sim_time = sim_time;
	t->val = val;

	slset_truncate_at(driver->transactions, t, true);
	slset_add(driver->transactions, t);
}

void
driver_propagate(
	struct driver *driver, 
	universal_integer sim_time,
	struct glue_vhdl *glue_vhdl
)
{
	struct drv_trans *t;

	assert(driver->connected_signal != NULL);

	if (driver->transactions->first == NULL) {
		/* no transactions */
		return;
	}

	t = (struct drv_trans *)driver->transactions->first->data;
	assert(sim_time <= t->sim_time);

	if (sim_time != t->sim_time) {
		/* entry in the future, will propagate later */
		return;
	}

	/* sim_tim == t->sim_time */
	slset_remove(driver->transactions, t);

	/* FIXME resolution function */
	/* FIXME type safety! */
	if (driver->connected_signal->value.univ_int != t->val.univ_int) {
		driver->connected_signal->event = true;
		driver->connected_signal->value = t->val;
	}

	if (driver->foreign_out) {
		assert(driver->connected_signal->is_foreign == true);
		glue_vhdl_set(glue_vhdl, 
				driver->connected_signal->foreign_id, 
				t->val, 
				driver);
	}
	
	free(t);
}

universal_integer
driver_get_next_event(const struct driver *drv)
{
	const struct slset_entry *i;
	const struct drv_trans *t;

	if (slset_empty(drv->transactions)) {
		return INT64_MAX;
	}

	i = drv->transactions->first;
	t = (const struct drv_trans *)i->data;

	return t->sim_time;
}

/** callback to update a hidden driver of a foreign signal.
 *  @param _drv pointer to hidden driver instance.
 *  @param value new dirving value.
 */
static void
foreign_in_driver_update(void *_drv, union fauhdli_value value)
{
	struct driver *drv = (struct driver *)_drv;
	driver_update(drv, value, time_virt());
}

struct signal *
signal_create(
	union fauhdli_value init,
	const char *foreign,
	struct glue_vhdl *glue_vhdl,
	const char *name
)
{
	struct signal *ret;

	ret = malloc(sizeof(struct signal));
	assert(ret != NULL);
	ret->value = init;
	ret->connected_drivers = slset_create(NULL);
	ret->event = false;

	if (foreign == NULL) {
		ret->is_foreign = false;
		ret->foreign_id = 0;
		return ret;
	}

	/* foreign signal */
	ret->is_foreign = true;
	ret->foreign_id = 
		glue_vhdl_create_signal(glue_vhdl, foreign, name);
	return ret;
}

void
signal_destroy(struct signal *_signal)
{
	if (_signal->connected_drivers != NULL) {
		slset_destroy(_signal->connected_drivers);
	}
	free(_signal);
}

void
signal_connect_foreign_in_driver(
	struct signal *sig,
	struct glue_vhdl *glue_vhdl,
	struct slset *process_drivers
)
{
	struct driver *drv;
	union fauhdli_value init;

	/* FIXME obtain foreign value */
	init.univ_int = 0;

	drv = driver_create(init);
	driver_connect_non_foreign(drv, sig);
	sig->value = init;
	
	glue_vhdl_connect_in(glue_vhdl,
				sig->foreign_id, 
				foreign_in_driver_update,
				drv);
	slset_add(process_drivers, drv);
}

void
signal_read(
	const struct signal *sig,
	union fauhdli_value *val
)
{
	*val = sig->value;
}

unsigned int
fauhdli_get_sig_id(const void *_sig)
{
	const struct signal *sig = (const struct signal*)_sig;

	assert(sig != NULL);
	assert(sig->is_foreign);

	return sig->foreign_id;
}

unsigned int
fauhdli_get_sig_id_driver(const void *_drv)
{
	const struct driver *drv = (const struct driver *)_drv;

	assert(drv != NULL);
	assert(drv->foreign_out == true);
	assert(drv->connected_signal != NULL);
	assert(drv->connected_signal->is_foreign == true);

	return drv->connected_signal->foreign_id;
}
