/* $Id: SubtypeIndication.hpp 4323 2009-01-27 13:48:12Z potyra $ 
 *
 * Copyright (C) 2007-2009 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#ifndef __SUBTYPE_INDICATION_HPP_INCLUDED
#define __SUBTYPE_INDICATION_HPP_INCLUDED

#include "frontend/ast/Name.hpp"
#include "frontend/ast/DiscreteRange.hpp"
#include "frontend/ast/TypeDeclaration.hpp"
#include "frontend/misc/Symbol.hpp"

namespace ast {

//! any Name referring to a type or subtype.
/** This class contains the superset of Names referring to a type or
 *  subtype, eventually with a unique constraint and/or a
 *  Resolution function.
 */
class SubtypeIndication : public TypeDeclaration {
public:
	//! c'tor
	/** @param typeMark Name referring to the type or subtype
	 *  @param loc location of the type name.
	 */
	SubtypeIndication(Name *typeMark, Location loc);

	//! alternate c'tor
	/** Construct a SubtypeIndication directly by a given type.
	 *  @param byType type that the SubtypeIndication refers to.
	 *  @param loc Location of the SubtypeIndication.
	 */
	SubtypeIndication(const TypeDeclaration *byType, Location loc);

	//! Accept a Visitor.
 	/** All leaf AST nodes need to implement this method.
         *
         *  @param visitor the Visitor that can visit this node.
         */
	virtual void accept(Visitor& visitor) {
		visitor.visit(*this);
	}

	/** Put a textual representation of the AstNode on the stream.
	 *  @param stream stream to put the textual representation to.
	 */
	virtual void put(std::ostream &stream) const {
		stream << "Subtype ";
		if (this->typeName != NULL) {
			stream << this->typeName;
		} else {
			stream << "(unnammed)";
		}
		
		assert(this->declaration != NULL);
		stream << " : ";
		if (this->declaration->name != NULL) {
			stream << *this->declaration->name;
		} else {
			stream << "(unnamed)";
		}

		if (this->constraint) {
			stream << ' ' << this->constraint;
		}

		if (this->indexConstraint) {
			stream << " (";
			util::MiscUtil::listPut(this->indexConstraint, stream,
						", ");
			stream << ')';
		}


		if (this->resolutionFunction) {
			stream << " resolved by " << this->resolutionFunction;
		}
	}


	/** Name referring to the type or subtype */
	Name *typeName;

	/** optional range constraint */
	DiscreteRange *constraint;

	/** optional index constraint */
	std::list<DiscreteRange*> *indexConstraint;

	/** optional additional resolution function (currently not
	 *  yet supported.
	 */
	Name *resolutionFunction;

	/** the referred to type declaration, not reference counted! */
	const TypeDeclaration *declaration;

protected:
	/** Destructor */
	virtual ~SubtypeIndication() {
		util::MiscUtil::terminate(typeName);
		util::MiscUtil::terminate(constraint);
		util::MiscUtil::terminate(resolutionFunction);
	}
};

}; /* namespace ast */

#endif /*__SUBTYPE_INDICATION_HPP_INCLUDED */
