/* $Id: ElementAssociation.hpp 4323 2009-01-27 13:48:12Z potyra $ 
 *
 * Copyright (C) 2007-2009 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#ifndef __ELEMENT_ASSOCIATION_HPP_INCLUDED
#define __ELEMENT_ASSOCIATION_HPP_INCLUDED

#include <list>
#include "frontend/ast/AstNode.hpp"
#include "frontend/misc/RangeSet.hpp"

namespace ast {

//! one particular element association.
/** This class represents VHDL element associations of an Aggregate.
 */
class ElementAssociation : public AstNode {
public:
	//! c'tor 
	/** @param formalChoices the formal (target) part containing a list 
	 *         of Expressions.
	 *  @param actualExpression the actual (source) part containing the 
	 *         actual value to be used.
	 */
	ElementAssociation(
		std::list<Expression*> *formalChoices,
		Expression *actualExpression,
		Location loc
		) :	AstNode(loc),
			choices(formalChoices),
			actual(actualExpression),
			range(NULL) {}

	//! Accept a Visitor.
 	/** All leaf AST nodes need to implement this method.
         *
         *  @param visitor the Visitor that can visit this node.
         */
	virtual void accept(Visitor& visitor) {
		visitor.visit(*this);
	}

	/** Put a textual representation of the AstNode on the stream.
	 *  @param stream stream to put the textual representation to.
	 */
	virtual void put(std::ostream &stream) const {
		util::MiscUtil::listPut(this->choices, stream, " | ");
		stream << " => " << this->actual;
	}

	/** formal part */
	std::list<Expression*> *choices;
	/** actual part */
	Expression *actual;

	/** RangeSet containing the locally static values of the choices.
	 *  May not be set in one corner case.
	 */
	RangeSet *range;
protected:
	/** Destructor */
	virtual ~ElementAssociation(void) {
		util::MiscUtil::lterminate(this->choices);
		util::MiscUtil::terminate(this->actual);

		if (this->range != NULL) {
			delete this->range;
		}
	}
};

}; /* namespace ast */

#endif /* __ELEMENT_ASSOCIATION_HPP_INCLUDED */
