/*
 * $Id: system.c,v 1.263 2009-02-06 15:26:29 vrsieh Exp $
 *
 * Copyright (C) 2003-2009 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#include "config.h"

#include <assert.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "system.h"
#include "glue-main.h"
#include "gui.h"

#include "sig_boolean.h"
#include "sig_cardbus.h"
#include "sig_dio24.h"
#include "sig_dio48.h"
#include "sig_eth.h"
#include "sig_opt_rgb.h"
#include "sig_host_bus.h"
#include "sig_ide_bus.h"
#include "sig_integer.h"
#include "sig_isa_bus.h"
#include "sig_manage.h"
#include "sig_match.h"
#include "sig_mem_bus.h"
#include "sig_parallel.h"
#include "sig_pci_bus.h"
#include "sig_power.h"
#include "sig_ps2.h"
#include "sig_scsi_bus.h"
#include "sig_serial.h"
#include "sig_shugart_bus.h"
#include "sig_sound.h"
#include "sig_std_logic.h"
#include "sig_telephone.h"
#include "sig_usb_bus.h"
#include "sig_vga.h"
#include "sig_video.h"
#include "sig_fault.h"

#if 0
#include "cim_audio.h"
#include "cim_boolean.h"
#include "cim_cardbus.h"
#include "cim_dio24.h"
#include "cim_dio48.h"
#include "cim_eth.h"
#include "cim_fault.h"
#include "cim_opt_rgb.h"
#include "cim_integer.h"
#include "cim_parallel.h"
#include "cim_serial.h"
#include "cim_string.h"
#include "cim_telephone.h"
#include "cim_usb.h"
#include "cim_video.h"
#endif

#include "simulator/chassis.h"
#include "simulator/chassis_aui.h"
#include "simulator/chassis_gui.h"
#include "simulator/chassis_at.h"
#include "simulator/chassis_at_aui.h"
#include "simulator/chassis_at_gui.h"
#include "simulator/power_supply.h"
#include "simulator/power_supply_gui.h"
#include "simulator/power_supply_at.h"
#include "simulator/power_supply_at_gui.h"
#include "simulator/mb_ibm_at.h"
#include "simulator/mb_ga_686dlx.h"
#if 0
#include "simulator/cpu_generic_host.h"
#endif
#include "simulator/cpu_intel_80386.h"
#include "simulator/cpu_intel_80386_interpreter.h"
#include "simulator/cpu_intel_80686_klamath.h"
#include "simulator/cpu_intel_80686_coppermine.h"
#include "simulator/cpu_intel_x86_64.h"
#include "simulator/memory.h"
#include "simulator/memory_gui.h"
#include "simulator/isa_gen_ne2000.h"
#include "simulator/isa_gen_ne2000_gui.h"
#include "simulator/isa_gen_idectrl.h"
#include "simulator/isa_gen_soundblaster16.h"
#include "simulator/isa_gen_parallel.h"
#include "simulator/isa_gen_serial.h"
#include "simulator/isa_gen_vga.h"
#include "simulator/pci_cirrus_gd5446.h"
#include "simulator/pci_cirrus_gd5446_2mb.h"
#include "simulator/pci_cirrus_gd5446_3mb.h"
#include "simulator/pci_intel_eepro100.h"
#include "simulator/pci_intel_eepro100_gui.h"
#include "simulator/pci_ti_1520.h"
#include "simulator/pci_lsi_20810.h"
#include "simulator/pci_gen_dio24.h"
#include "simulator/pci_gen_dio48.h"
#include "simulator/scsi_gen_cdrom.h"
#include "simulator/scsi_gen_cdrom_gui.h"
#include "simulator/scsi_gen_disk.h"
#include "simulator/ide_gen_cdrom.h"
#include "simulator/ide_gen_cdrom_gui.h"
#include "simulator/ide_gen_disk.h"
#include "simulator/ide_gen_disk_gui.h"
#include "simulator/floppy_gen_floppydrive.h"
#include "simulator/floppy_gen_floppydrive_gui.h"
#include "simulator/keyboard.h"
#include "simulator/keyboard_gui.h"
#include "simulator/ps2_gen_mouse_4.h"
#include "simulator/ps2_gen_mouse_4_gui.h"
#include "simulator/vt102_monitor.h"
#include "simulator/vt102_monitor_gui.h"
#include "simulator/monitor.h"
#include "simulator/monitor_gui.h"
#include "pattern-matcher/patternm.h"
#include "pattern-matcher/mouse_matcher.h"
#include "simulator/loudspeaker.h"
#include "simulator/loudspeaker_aui.h"
#include "simulator/usb2serial.h"
#include "simulator/usbstorage.h"
#include "simulator/vt102_keyboard.h"
#include "simulator/vt102_keyboard_gui.h"
#include "simulator/vt102_core.h"
#include "simulator/vt102_core_gui.h"
#include "simulator/cb_intel_eepro100.h"
#include "simulator/cb_intel_eepro100_gui.h"
#include "simulator/dhcp_server.h"
#include "simulator/serial_modem.h"
#include "simulator/serial_modem_gui.h"
#include "simulator/printer.h"
#include "simulator/telephone_switch.h"
#include "simulator/telephone_switch_gui.h"
#include "simulator/telephone_isp.h"
#include "simulator/telephone_isp_gui.h"
#include "simulator/elevator.h"
#include "simulator/elevator_gui.h"
#include "simulator/elevator_rc_gui.h"
#include "simulator/network_bridge.h"
#include "simulator/network_router.h"
#include "simulator/dio24_loopback.h"
#include "simulator/dio48_loopback.h"

static void *cpssp = &cpssp;

#if 0
/* CIM Ports */
static struct cim_audio port_audio[100];
static struct cim_boolean port_boolean[1000];
static struct cim_cardbus port_cardbus[100];
static struct cim_dio24 port_dio24[100];
static struct cim_dio48 port_dio48[100];
static struct cim_eth port_eth[100];
static struct cim_fault port_fault[100];
static struct cim_integer port_integer[100];
static struct cim_parallel port_parallel[100];
static struct cim_serial port_serial[100];
static struct cim_string port_string[100];
static struct cim_telephone port_telephone[100];
static struct cim_usb port_usb_bus[100];
static struct cim_video port_video[100];
#endif

/* Generic nodes. */
static struct {
	char *name;
} node[64];
static unsigned int node_count = 0;

/* Generic GUI pages. */
static struct {
	char *name;
} page[64];
static unsigned int page_count = 0;

/* Generic signals. */
static struct {
	char *type;
	char *name;

	void *ptr;
} sig[10000];

/* Generic components. */
static struct {
	char *type;
	char *name;
	unsigned int node;
	unsigned int page;
} comp[200];

/* Generic generics. */
static struct {
	unsigned int comp_id;
	char *name;
	char *value;
} generic[2000];
static unsigned int generic_count = 0;

/* Generic connections. */
static struct {
	unsigned int comp_id;
	char *port;
	void *sig_ptr;
} conn[2000];

/* FIXME */
unsigned char floppydrive[4];


int
system_node_create(const char *name)
{
	unsigned int i;

	for (i = 0; i < node_count; i++) {
		if (strcmp(node[i].name, name) == 0) {
			/* Found */
			return -1;
		}
	}

	node[node_count].name = strdup(name);
	assert(node[node_count].name);

	return node_count++;
}

int
system_node_destroy(unsigned int node_id)
{
	/* FIXME */
	return 0;
}

int
system_page_create(const char *name)
{
	unsigned int i;

	for (i = 0; i < page_count; i++) {
		if (strcmp(page[i].name, name) == 0) {
			/* Found */
			return -1;
		}
	}

	page[page_count].name = strdup(name);
	assert(page[page_count].name);

	gui_page_add(page[page_count].name);

	return page_count++;
}

int
system_page_destroy(unsigned int page_id)
{
	/* FIXME */
	return 0;
}

int
system_sig_create(const char *type, const char *name)
{
	unsigned int id;

	if (strcmp(type, "character") == 0) {
		/* FIXME */
		type = "integer";
	}

	for (id = 0; id < sizeof(sig) / sizeof(sig[0]); id++) {
		if (sig[id].name
		 && strcmp(sig[id].name, name) == 0) {
			/* Found */
			return -1;
		}
	}
	for (id = 0; ; id++) {
		if (id == sizeof(sig) / sizeof(sig[0])) {
			/* Table Full */
			return -1;
		}
		if (! sig[id].name) {
			break;
		}
	}

	sig[id].type = strdup(type);
	assert(sig[id].type);

	sig[id].name = strdup(name);
	assert(sig[id].name);

	sig[id].ptr = NULL;

	return id;
}

void *
system_sig_get(unsigned int id)
{
	assert(/* 0 <= id && */ id < sizeof(sig) / sizeof(sig[0]));
	assert(sig[id].type);
	assert(sig[id].name);

	if (sig[id].ptr) {
		return sig[id].ptr;
	}

	assert(! sig[id].ptr);

	if (strcmp(sig[id].type, "agp_bus") == 0) {
		sig_agp_bus_create("agp_bus", id);
		sig[id].ptr = sig_agp_bus_init("agp_bus", id);
	} else if (strcmp(sig[id].type, "boolean") == 0) {
		sig_boolean_create("boolean", id);
		sig[id].ptr = sig_boolean_init("boolean", id);
	} else if (strcmp(sig[id].type, "cardbus") == 0) {
		sig_cardbus_create("cardbus", id);
		sig[id].ptr = sig_cardbus_init("cardbus", id);
	} else if (strcmp(sig[id].type, "dio24") == 0) {
		sig_dio24_create("dio24", id);
		sig[id].ptr = sig_dio24_init("dio24", id);
	} else if (strcmp(sig[id].type, "dio48") == 0) {
		sig_dio48_create("dio48", id);
		sig[id].ptr = sig_dio48_init("dio48", id);
	} else if (strcmp(sig[id].type, "eth") == 0) {
		sig_eth_create("eth", id);
		sig[id].ptr = sig_eth_init("eth", id);
	} else if (strcmp(sig[id].type, "fault") == 0) {
		sig_fault_create("fault", id);
		sig[id].ptr = sig_fault_init("fault", id);
	} else if (strcmp(sig[id].type, "opt_rgb") == 0) {
		sig_opt_rgb_create("opt_rgb", id);
		sig[id].ptr = sig_opt_rgb_init("opt_rgb", id);
	} else if (strcmp(sig[id].type, "host_bus") == 0) {
		sig_host_bus_create("host_bus", id);
		sig[id].ptr = sig_host_bus_init("host_bus", id);
	} else if (strcmp(sig[id].type, "ide_bus") == 0) {
		sig_ide_bus_create("ide_bus", id);
		sig[id].ptr = sig_ide_bus_init("ide_bus", id);
	} else if (strcmp(sig[id].type, "integer") == 0) {
		sig_integer_create("integer", id);
		sig[id].ptr = sig_integer_init("integer", id);
	} else if (strcmp(sig[id].type, "isa_bus") == 0) {
		sig_isa_bus_create("isa_bus", id);
		sig[id].ptr = sig_isa_bus_init("isa_bus", id);
	} else if (strcmp(sig[id].type, "manage") == 0) {
		sig_manage_create("manage", id);
		sig[id].ptr = sig_manage_init("manage", id);
	} else if (strcmp(sig[id].type, "match") == 0) {
		sig_match_create("match", id);
		sig[id].ptr = sig_match_init("match", id);
	} else if (strcmp(sig[id].type, "mem_bus") == 0) {
		sig_mem_bus_create("mem_bus", id);
		sig[id].ptr = sig_mem_bus_init("mem_bus", id);
	} else if (strcmp(sig[id].type, "parallel") == 0) {
		sig_parallel_create("parallel", id);
		sig[id].ptr = sig_parallel_init("parallel", id);
	} else if (strcmp(sig[id].type, "pci_bus") == 0) {
		sig_pci_bus_create("pci_bus", id);
		sig[id].ptr = sig_pci_bus_init("pci_bus", id);
	} else if (strcmp(sig[id].type, "power_board") == 0) {
		sig_power_board_create("power_board", id);
		sig[id].ptr = sig_power_board_init("power_board", id);
	} else if (strcmp(sig[id].type, "power_board_at") == 0) {
		sig_power_board_at_create("power_board_at", id);
		sig[id].ptr = sig_power_board_at_init("power_board_at", id);
	} else if (strcmp(sig[id].type, "power_device") == 0) {
		sig_power_device_create("power_device", id);
		sig[id].ptr = sig_power_device_init("power_device", id);
	} else if (strcmp(sig[id].type, "ps2") == 0) {
		sig_ps2_create("ps2", id);
		sig[id].ptr = sig_ps2_init("ps2", id);
	} else if (strcmp(sig[id].type, "scsi_bus") == 0) {
		sig_scsi_bus_create("scsi_bus", id);
		sig[id].ptr = sig_scsi_bus_init("scsi_bus", id);
	} else if (strcmp(sig[id].type, "serial") == 0) {
		sig_serial_create("serial", id);
		sig[id].ptr = sig_serial_init("serial", id);
	} else if (strcmp(sig[id].type, "shugart_bus") == 0) {
		sig_shugart_bus_create("shugart_bus", id);
		sig[id].ptr = sig_shugart_bus_init("shugart_bus", id);
	} else if (strcmp(sig[id].type, "sound") == 0) {
		sig_sound_create("sound", id);
		sig[id].ptr = sig_sound_init("sound", id);
	} else if (strcmp(sig[id].type, "std_logic") == 0) {
		sig_std_logic_create("std_logic", id);
		sig[id].ptr = sig_std_logic_init("std_logic", id);
	} else if (strcmp(sig[id].type, "string") == 0) {
		sig_string_create("string", id);
		sig[id].ptr = sig_string_init("string", id);
	} else if (strcmp(sig[id].type, "telephone") == 0) {
		sig_telephone_create("telephone", id);
		sig[id].ptr = sig_telephone_init("telephone", id);
	} else if (strcmp(sig[id].type, "usb_bus") == 0) {
		sig_usb_bus_create("usb_bus", id);
		sig[id].ptr = sig_usb_bus_init("usb_bus", id);
	} else if (strcmp(sig[id].type, "vga") == 0) {
		sig_vga_create("vga", id);
		sig[id].ptr = sig_vga_init("vga", id);
	} else if (strcmp(sig[id].type, "video") == 0) {
		sig_video_create("video", id);
		sig[id].ptr = sig_video_init("video", id);
	} else {
		assert(0);
	}

	return sig[id].ptr;
}

int
system_sig_unget(void *ptr)
{
	unsigned int id;

	for (id = 0; ; id++) {
		if (id == sizeof(sig) / sizeof(sig[0])) {
			/* Signal not found. */
			return -1;
		}
		if (sig[id].ptr == ptr) {
			break;
		}
	}

	if (strcmp(sig[id].type, "boolean") == 0) {
		sig_boolean_destroy("boolean", id);
	} else if (strcmp(sig[id].type, "cardbus") == 0) {
		sig_cardbus_destroy("cardbus", id);
	} else if (strcmp(sig[id].type, "dio24") == 0) {
		sig_dio24_destroy("dio24", id);
	} else if (strcmp(sig[id].type, "dio48") == 0) {
		sig_dio48_destroy("dio48", id);
	} else if (strcmp(sig[id].type, "eth") == 0) {
		sig_eth_destroy("eth", id);
	} else if (strcmp(sig[id].type, "fault") == 0) {
		sig_fault_destroy("fault", id);
	} else if (strcmp(sig[id].type, "opt_rgb") == 0) {
		sig_opt_rgb_destroy("opt_rgb", id);
	} else if (strcmp(sig[id].type, "host_bus") == 0) {
		sig_host_bus_destroy("host_bus", id);
	} else if (strcmp(sig[id].type, "ide_bus") == 0) {
		sig_ide_bus_destroy("ide_bus", id);
	} else if (strcmp(sig[id].type, "integer") == 0) {
		sig_integer_destroy("integer", id);
	} else if (strcmp(sig[id].type, "isa_bus") == 0) {
		sig_isa_bus_destroy("isa_bus", id);
	} else if (strcmp(sig[id].type, "manage") == 0) {
		sig_manage_destroy("manage", id);
	} else if (strcmp(sig[id].type, "match") == 0) {
		sig_match_destroy("match", id);
	} else if (strcmp(sig[id].type, "mem_bus") == 0) {
		sig_mem_bus_destroy("mem_bus", id);
	} else if (strcmp(sig[id].type, "parallel") == 0) {
		sig_parallel_destroy("parallel", id);
	} else if (strcmp(sig[id].type, "pci_bus") == 0) {
		sig_pci_bus_destroy("pci_bus", id);
	} else if (strcmp(sig[id].type, "agp_bus") == 0) {
		sig_agp_bus_destroy("agp_bus", id);
	} else if (strcmp(sig[id].type, "scsi_bus") == 0) {
		sig_scsi_bus_destroy("scsi_bus", id);
	} else if (strcmp(sig[id].type, "power_board") == 0) {
		sig_power_board_destroy("power_board", id);
	} else if (strcmp(sig[id].type, "power_board_at") == 0) {
		sig_power_board_at_destroy("power_board_at", id);
	} else if (strcmp(sig[id].type, "power_device") == 0) {
		sig_power_device_destroy("power_device", id);
	} else if (strcmp(sig[id].type, "ps2") == 0) {
		sig_ps2_destroy("ps2", id);
	} else if (strcmp(sig[id].type, "serial") == 0) {
		sig_serial_destroy("serial", id);
	} else if (strcmp(sig[id].type, "sound") == 0) {
		sig_sound_destroy("sound", id);
	} else if (strcmp(sig[id].type, "std_logic") == 0) {
		sig_std_logic_destroy("std_logic", id);
	} else if (strcmp(sig[id].type, "shugart_bus") == 0) {
		sig_shugart_bus_destroy("shugart_bus", id);
	} else if (strcmp(sig[id].type, "string") == 0) {
		sig_string_destroy("string", id);
	} else if (strcmp(sig[id].type, "telephone") == 0) {
		sig_telephone_destroy("telephone", id);
	} else if (strcmp(sig[id].type, "usb_bus") == 0) {
		sig_usb_bus_destroy("usb_bus", id);
	} else if (strcmp(sig[id].type, "vga") == 0) {
		sig_vga_destroy("vga", id);
	} else if (strcmp(sig[id].type, "video") == 0) {
		sig_video_destroy("video", id);
	} else {
		fprintf(stderr, "%s\n", sig[id].type);
		assert(0);
	}

	sig[id].ptr = NULL;

	return 0;
}

int
system_sig_destroy(unsigned int id)
{
	if (sizeof(sig) / sizeof(sig[0]) <= id
	 || ! sig[id].type
	 || ! sig[id].name) {
		/* No such signal. */
		return -1;
	}

	if (sig[id].ptr) {
		/* Hack - FIXME */
		system_sig_unget(sig[id].ptr);
	}

	if (sig[id].ptr) {
		/* Signal busy. */
		return -1;
	}

	free(sig[id].type);
	sig[id].type = NULL;

	free(sig[id].name);
	sig[id].name = NULL;

	return 0;
}

int
system_comp_create(
	const char *type,
	const char *name,
	unsigned int node,
	unsigned int page
)
{
	unsigned int id;

	for (id = 0; id < sizeof(comp) / sizeof(comp[0]); id++) {
		if (comp[id].name
		 && strcmp(comp[id].name, name) == 0) {
			/* Found */
			return -1;
		}
	}
	for (id = 0; ; id++) {
		if (id == sizeof(comp) / sizeof(comp[0])) {
			/* Table Full */
			return -1;
		}
		if (! comp[id].name) {
			break;
		}
	}

	comp[id].type = strdup(type);
	assert(comp[id].type);

	comp[id].name = strdup(name);
	assert(comp[id].name);

	comp[id].node = node;
	comp[id].page = page;

	return id;
}

int
system_comp_generic_set(unsigned int id, const char *n, const char *v)
{
	char *name;
	char *value;

	generic[generic_count].comp_id = id;

	assert(n);
	name = strdup(n);
	assert(name);
	generic[generic_count].name = name;

	if (v) {
		value = strdup(v);
		assert(value);
	} else {
		value = NULL;
	}
	generic[generic_count].value = value;

	generic_count++;
	
	return 0;
}

int
system_comp_port_connect(unsigned int id, const char *port, unsigned int sig_id)
{
	unsigned int conn_id;

	if (sizeof(comp) / sizeof(comp[0]) <= id
	 || ! comp[id].type
	 || ! comp[id].name) {
		/* No such component. */
		return -1;
	}

	if (sizeof(sig) / sizeof(sig[0]) <= sig_id
	 || ! sig[sig_id].type
	 || ! sig[sig_id].name) {
		/* No such signal. */
		return -1;
	}

	for (conn_id = 0; conn_id < sizeof(conn) / sizeof(conn[0]); conn_id++) {
		if (! conn[conn_id].port) {
			continue;
		}

		if (conn[conn_id].comp_id == id
		 && strcmp(conn[conn_id].port, port) == 0) {
			/* Port already defined. */
			return -1;
		}
	}

	/* Lookup free connection entry. */
	for (conn_id = 0; ; conn_id++) {
		if (conn_id == sizeof(conn) / sizeof(conn[0])) {
			/* Table full. */
			return -1;
		}
		if (! conn[conn_id].port) {
			break;
		}
	}

	conn[conn_id].comp_id = id;
	conn[conn_id].port = strdup(port);
	assert(conn[conn_id].port);
	conn[conn_id].sig_ptr = system_sig_get(sig_id);
	
	return 0;
}

static const char *
generic_get(unsigned int id, const char *name)
{
	unsigned int i;

	for (i = 0; ; i++) {
		if (i == generic_count) {
			fprintf(stderr, "WARNING: Generic %s not set for component!\n",
					name);
			return NULL;
		}
		if (generic[i].comp_id == id
		 && strcmp(generic[i].name, name) == 0) {
			return generic[i].value;
		}
	}
}

static void *
sig_lookup(unsigned int id, const char *port)
{
	unsigned int conn_id;

	for (conn_id = 0; ; conn_id++) {
		assert(conn_id < sizeof(conn) / sizeof(conn[0]));

		if (! conn[conn_id].port) {
			continue;
		}

		if (conn[conn_id].comp_id == id
		 && strcmp(conn[conn_id].port, port) == 0) {
			return conn[conn_id].sig_ptr;
		}
	}
}

int
system_comp_init(unsigned int id)
{
	assert(id < sizeof(comp) / sizeof(comp[0]));
	assert(comp[id].type);
	assert(comp[id].name);

	if (strcmp(comp[id].type, "chassis") == 0) {
		chassis_create(id,
				comp[id].name);
		chassis_aui_create(id,
				comp[id].name);
		chassis_gui_create(comp[id].page,
				id,
				comp[id].name);
		chassis_init(id,
				sig_lookup(id, "mech_power_button"),
				sig_lookup(id, "n_power_button"),
				sig_lookup(id, "mech_reset_button"),
				sig_lookup(id, "n_reset_button"),
				sig_lookup(id, "power_led"),
				sig_lookup(id, "opt_power_led"),
				sig_lookup(id, "ide_led"),
				sig_lookup(id, "opt_ide_led"),
				sig_lookup(id, "speaker"),
				sig_lookup(id, "audio_speaker"));
		chassis_aui_init(id,
				sig_lookup(id, "mech_power_button"),
				sig_lookup(id, "n_power_button"),
				sig_lookup(id, "mech_reset_button"),
				sig_lookup(id, "n_reset_button"),
				sig_lookup(id, "power_led"),
				sig_lookup(id, "opt_power_led"),
				sig_lookup(id, "ide_led"),
				sig_lookup(id, "opt_ide_led"),
				sig_lookup(id, "speaker"),
				sig_lookup(id, "audio_speaker"));
		chassis_gui_init(id,
				sig_lookup(id, "mech_power_button"),
				sig_lookup(id, "n_power_button"),
				sig_lookup(id, "mech_reset_button"),
				sig_lookup(id, "n_reset_button"),
				sig_lookup(id, "power_led"),
				sig_lookup(id, "opt_power_led"),
				sig_lookup(id, "ide_led"),
				sig_lookup(id, "opt_ide_led"),
				sig_lookup(id, "speaker"),
				sig_lookup(id, "audio_speaker"));

	} else if (strcmp(comp[id].type, "chassis_at") == 0) {
		chassis_at_create(id,
				comp[id].name);
		chassis_at_aui_create(id,
				comp[id].name);
		chassis_at_gui_create(comp[id].page,
				id,
				comp[id].name);
		chassis_at_init(id,
				sig_lookup(id, "mech_reset_button"),
				sig_lookup(id, "n_reset_button"),
				sig_lookup(id, "power_led"),
				sig_lookup(id, "opt_power_led"),
				sig_lookup(id, "speaker"),
				sig_lookup(id, "audio_speaker"));
		chassis_at_aui_init(id,
				sig_lookup(id, "mech_reset_button"),
				sig_lookup(id, "n_reset_button"),
				sig_lookup(id, "power_led"),
				sig_lookup(id, "opt_power_led"),
				sig_lookup(id, "speaker"),
				sig_lookup(id, "audio_speaker"));
		chassis_at_gui_init(id,
				sig_lookup(id, "mech_reset_button"),
				sig_lookup(id, "n_reset_button"),
				sig_lookup(id, "power_led"),
				sig_lookup(id, "opt_power_led"),
				sig_lookup(id, "speaker"),
				sig_lookup(id, "audio_speaker"));

	} else if (strcmp(comp[id].type, "power_supply") == 0) {
		power_supply_create(id,
				comp[id].name);
		power_supply_gui_create(comp[id].page,
				id,
				comp[id].name);
		power_supply_init(id,
				sig_lookup(id, "power_230v"),
				sig_lookup(id, "mech_power_switch"),
				sig_lookup(id, "power_board"),
				sig_lookup(id, "power_dev0"),
				sig_lookup(id, "power_dev1"),
				sig_lookup(id, "power_dev2"),
				sig_lookup(id, "power_dev3"),
				sig_lookup(id, "power_dev4"),
				sig_lookup(id, "power_dev5"),
				sig_lookup(id, "power_dev6"),
				sig_lookup(id, "power_dev7"));
		power_supply_gui_init(id,
				sig_lookup(id, "power_230v"),
				sig_lookup(id, "mech_power_switch"),
				sig_lookup(id, "power_board"),
				sig_lookup(id, "power_dev0"),
				sig_lookup(id, "power_dev1"),
				sig_lookup(id, "power_dev2"),
				sig_lookup(id, "power_dev3"),
				sig_lookup(id, "power_dev4"),
				sig_lookup(id, "power_dev5"),
				sig_lookup(id, "power_dev6"),
				sig_lookup(id, "power_dev7"));

		sig_boolean_connect_out(sig_lookup(id, "power_230v"),
				cpssp, 0);
		sig_boolean_connect_out(sig_lookup(id, "mech_power_switch"),
				cpssp, 0);
		sig_boolean_set(sig_lookup(id, "power_230v"),
				cpssp, 1);
		sig_boolean_set(sig_lookup(id, "mech_power_switch"),
				cpssp, 1);

	} else if (strcmp(comp[id].type, "power_supply_at") == 0) {
		power_supply_at_create(id,
				comp[id].name);
		power_supply_at_gui_create(comp[id].page,
				id,
				comp[id].name);
		power_supply_at_init(id,
				sig_lookup(id, "power_230v"),
				sig_lookup(id, "mech_power_switch"),
				sig_lookup(id, "power_board"),
				sig_lookup(id, "power_dev0"),
				sig_lookup(id, "power_dev1"),
				sig_lookup(id, "power_dev2"),
				sig_lookup(id, "power_dev3"),
				sig_lookup(id, "power_dev4"),
				sig_lookup(id, "power_dev5"),
				sig_lookup(id, "power_dev6"),
				sig_lookup(id, "power_dev7"));
		power_supply_at_gui_init(id,
				sig_lookup(id, "power_230v"),
				sig_lookup(id, "mech_power_switch"),
				sig_lookup(id, "power_board"),
				sig_lookup(id, "power_dev0"),
				sig_lookup(id, "power_dev1"),
				sig_lookup(id, "power_dev2"),
				sig_lookup(id, "power_dev3"),
				sig_lookup(id, "power_dev4"),
				sig_lookup(id, "power_dev5"),
				sig_lookup(id, "power_dev6"),
				sig_lookup(id, "power_dev7"));

		sig_boolean_connect_out(sig_lookup(id, "power_230v"),
				cpssp, 0);
		sig_boolean_connect_out(sig_lookup(id, "mech_power_switch"),
				cpssp, 0);
		sig_boolean_set(sig_lookup(id, "power_230v"),
				cpssp, 1);

	} else if (strcmp(comp[id].type, "mb_ibm_at") == 0) {
		mb_ibm_at_create(id,
				comp[id].name,
				generic_get(id, "rtc_start"));
		mb_ibm_at_init(id,
				sig_lookup(id, "power"),
				sig_lookup(id, "isa0"),
				sig_lookup(id, "isa1"),
				sig_lookup(id, "isa2"),
				sig_lookup(id, "isa3"),
				sig_lookup(id, "isa4"),
				sig_lookup(id, "isa5"),
				sig_lookup(id, "isa6"),
				sig_lookup(id, "isa7"),
				sig_lookup(id, "keyboard"),
				sig_lookup(id, "speaker"),
				sig_lookup(id, "power_led"),
				sig_lookup(id, "n_reset_button"));

	} else if (strcmp(comp[id].type, "mb_ga_686dlx") == 0) {
		mb_ga_686dlx_create(id,
				comp[id].name,
				generic_get(id, "rtc_start"));
		mb_ga_686dlx_init(id,
				sig_lookup(id, "power"),
				sig_lookup(id, "cpu0"),
				sig_lookup(id, "cpu1"),
				sig_lookup(id, "mem0"),
				sig_lookup(id, "mem1"),
				sig_lookup(id, "mem2"),
				sig_lookup(id, "mem3"),
				sig_lookup(id, "agp"),
				sig_lookup(id, "pci0"),
				sig_lookup(id, "pci1"),
				sig_lookup(id, "pci2"),
				sig_lookup(id, "pci3"),
				sig_lookup(id, "isa0"),
				sig_lookup(id, "isa1"),
				sig_lookup(id, "isa2"),
				sig_lookup(id, "ide0"),
				sig_lookup(id, "ide1"),
				sig_lookup(id, "scsi"),
				sig_lookup(id, "usb0"),
				sig_lookup(id, "usb1"),
				sig_lookup(id, "ser0"),
				sig_lookup(id, "ser1"),
				sig_lookup(id, "par"),
				sig_lookup(id, "keyboard"),
				sig_lookup(id, "mouse"),
				sig_lookup(id, "speaker"),
				sig_lookup(id, "shugart"),
				sig_lookup(id, "power_led"),
				sig_lookup(id, "ide_led"),
				sig_lookup(id, "n_power_button"),
				sig_lookup(id, "n_reset_button"));

	} else if (strcmp(comp[id].type, "cpu_intel_80386") == 0) {
		cpu_intel_80386_create(id,
				comp[id].name);
		cpu_intel_80386_init(id,
				sig_lookup(id, "conn"));
	} else if (strcmp(comp[id].type, "cpu_intel_80386_interpreter") == 0) {
		cpu_intel_80386_interpreter_create(id,
				comp[id].name);
		cpu_intel_80386_interpreter_init(id,
				sig_lookup(id, "conn"));

	} else if (strcmp(comp[id].type, "cpu_intel_80686_klamath") == 0) {
		cpu_intel_80686_klamath_create(id,
				comp[id].name);
		cpu_intel_80686_klamath_init(id,
				sig_lookup(id, "conn"));

	} else if (strcmp(comp[id].type, "cpu_intel_80686_coppermine") == 0) {
		cpu_intel_80686_coppermine_create(id,
				comp[id].name);
		cpu_intel_80686_coppermine_init(id,
				sig_lookup(id, "conn"));

	} else if (strcmp(comp[id].type, "cpu_intel_x86_64") == 0) {
		cpu_intel_x86_64_create(id,
				comp[id].name);
		cpu_intel_x86_64_init(id,
				sig_lookup(id, "conn"));

	} else if (strcmp(comp[id].type, "memory") == 0) {
		memory_create(id,
				comp[id].name,
				generic_get(id, "size"));
		memory_gui_create(comp[id].page,
				id,
				comp[id].name,
				generic_get(id, "size"));
		memory_init(id,
				sig_lookup(id, "manage"),
				sig_lookup(id, "conn"),
				sig_lookup(id, "bitflip"),
				sig_lookup(id, "stuck_at_0"),
				sig_lookup(id, "stuck_at_1"),
				sig_lookup(id, "coupling"));
		memory_gui_init(id,
				sig_lookup(id, "manage"),
				sig_lookup(id, "conn"),
				sig_lookup(id, "bitflip"),
				sig_lookup(id, "stuck_at_0"),
				sig_lookup(id, "stuck_at_1"),
				sig_lookup(id, "coupling"));

	} else if (strcmp(comp[id].type, "isa_gen_ne2000") == 0) {
		isa_gen_ne2000_create(id,
				comp[id].name,
				generic_get(id, "irq"),
				generic_get(id, "ioaddr"),
				generic_get(id, "mac"));
		isa_gen_ne2000_gui_create(comp[id].page,
				id,
				comp[id].name,
				generic_get(id, "irq"),
				generic_get(id, "ioaddr"),
				generic_get(id, "mac"));
		isa_gen_ne2000_init(id,
				sig_lookup(id, "isa"),
				sig_lookup(id, "eth"),
				sig_lookup(id, "opt_busy_led"),
				sig_lookup(id, "recv_loss"),
				sig_lookup(id, "send_loss"));
		isa_gen_ne2000_gui_init(id,
				sig_lookup(id, "isa"),
				sig_lookup(id, "eth"),
				sig_lookup(id, "opt_busy_led"),
				sig_lookup(id, "recv_loss"),
				sig_lookup(id, "send_loss"));

	} else if (strcmp(comp[id].type, "isa_gen_idectrl") == 0) {
		isa_gen_idectrl_create(id,
				comp[id].name,
				generic_get(id, "irq"),
				generic_get(id, "ioaddr_8"),
				generic_get(id, "ioaddr_1"));
		isa_gen_idectrl_init(id,
				sig_lookup(id, "isa"),
				sig_lookup(id, "ide"));

	} else if (strcmp(comp[id].type, "isa_gen_soundblaster16") == 0) {
		isa_gen_soundblaster16_create(id,
				comp[id].name,
				generic_get(id, "irq"),
				generic_get(id, "ioaddr"),
				generic_get(id, "dma8"),
				generic_get(id, "dma16"));
		isa_gen_soundblaster16_init(id,
				sig_lookup(id, "isa"),
				sig_lookup(id, "speaker"));

	} else if (strcmp(comp[id].type, "isa_gen_parallel") == 0) {
		isa_gen_parallel_create(id,
				comp[id].name);
		isa_gen_parallel_init(id,
				sig_lookup(id, "isa"),
				sig_lookup(id, "par"));

	} else if (strcmp(comp[id].type, "isa_gen_serial") == 0) {
		isa_gen_serial_create(id,
				comp[id].name);
		isa_gen_serial_init(id,
				sig_lookup(id, "isa"),
				sig_lookup(id, "ser"));

	} else if (strcmp(comp[id].type, "isa_gen_vga") == 0) {
		isa_gen_vga_create(id,
				comp[id].name);
		isa_gen_vga_init(id,
				sig_lookup(id, "isa"),
				sig_lookup(id, "vga"));

	} else if (strcmp(comp[id].type, "pci_cirrus_gd5446") == 0) {
		pci_cirrus_gd5446_create(id,
				comp[id].name);
		pci_cirrus_gd5446_init(id,
				sig_lookup(id, "pci"),
				sig_lookup(id, "vga"));

	} else if (strcmp(comp[id].type, "pci_cirrus_gd5446_2mb") == 0) {
		pci_cirrus_gd5446_2mb_create(id,
				comp[id].name);
		pci_cirrus_gd5446_2mb_init(id,
				sig_lookup(id, "pci"),
				sig_lookup(id, "vga"));

	} else if (strcmp(comp[id].type, "pci_cirrus_gd5446_3mb") == 0) {
		pci_cirrus_gd5446_3mb_create(id,
				comp[id].name);
		pci_cirrus_gd5446_3mb_init(id,
				sig_lookup(id, "pci"),
				sig_lookup(id, "vga"));

	} else if (strcmp(comp[id].type, "pci_intel_eepro100") == 0) {
		pci_intel_eepro100_create(id,
				comp[id].name,
				generic_get(id, "mac"));
		pci_intel_eepro100_gui_create(comp[id].page,
				id,
				comp[id].name,
				generic_get(id, "mac"));
		pci_intel_eepro100_init(id,
				sig_lookup(id, "manage"),
				sig_lookup(id, "pci"),
				sig_lookup(id, "eth"),
				sig_lookup(id, "opt_busy_led"),
				sig_lookup(id, "recv_loss"),
				sig_lookup(id, "send_loss"));
		pci_intel_eepro100_gui_init(id,
				sig_lookup(id, "manage"),
				sig_lookup(id, "pci"),
				sig_lookup(id, "eth"),
				sig_lookup(id, "opt_busy_led"),
				sig_lookup(id, "recv_loss"),
				sig_lookup(id, "send_loss"));

	} else if (strcmp(comp[id].type, "pci_ti_1520") == 0) {
		pci_ti_1520_create(id,
				comp[id].name);
		pci_ti_1520_init(id,
				sig_lookup(id, "pci"),
				sig_lookup(id, "card0"),
				sig_lookup(id, "card1"));

	} else if (strcmp(comp[id].type, "pci_lsi_20810") == 0) {
		pci_lsi_20810_create(id,
				comp[id].name);
		pci_lsi_20810_init(id,
				sig_lookup(id, "pci"),
				sig_lookup(id, "scsi"));

	} else if (strcmp(comp[id].type, "pci_gen_dio24") == 0) {
		pci_gen_dio24_create(id,
				comp[id].name);
		pci_gen_dio24_init(id,
				sig_lookup(id, "pci"),
				sig_lookup(id, "dio24"));

	} else if (strcmp(comp[id].type, "pci_gen_dio48") == 0) {
                pci_gen_dio48_create(id,
				comp[id].name);
                pci_gen_dio48_init(id,
                        	sig_lookup(id, "pci"),
                        	sig_lookup(id, "dio48"));

	} else if (strcmp(comp[id].type, "scsi_gen_cdrom") == 0) {
		scsi_gen_cdrom_create(id,
				comp[id].name,
				generic_get(id, "scsi_id"));
		scsi_gen_cdrom_gui_create(comp[id].page,
				id,
				comp[id].name,
				generic_get(id, "scsi_id"));
		scsi_gen_cdrom_init(id,
				sig_lookup(id, "power"),
				sig_lookup(id, "scsi"),
				sig_lookup(id, "change"));
		scsi_gen_cdrom_gui_init(id,
				sig_lookup(id, "power"),
				sig_lookup(id, "scsi"),
				sig_lookup(id, "change"));

	} else if (strcmp(comp[id].type, "scsi_gen_disk") == 0) {
		scsi_gen_disk_create(id,
				comp[id].name,
				generic_get(id, "scsi_id"),
				generic_get(id, "size"));
		scsi_gen_disk_init(id,
				sig_lookup(id, "power"),
				sig_lookup(id, "scsi"));

	} else if (strcmp(comp[id].type, "ide_gen_cdrom") == 0) {
		ide_gen_cdrom_create(id,
				comp[id].name,
				generic_get(id, "unit"));
		ide_gen_cdrom_gui_create(comp[id].page,
				id,
				comp[id].name,
				generic_get(id, "unit"));
		ide_gen_cdrom_init(id,
				sig_lookup(id, "power"),
				sig_lookup(id, "ide"),
				sig_lookup(id, "change"),
				sig_lookup(id, "fault"));
		ide_gen_cdrom_gui_init(id,
				sig_lookup(id, "power"),
				sig_lookup(id, "ide"),
				sig_lookup(id, "change"),
				sig_lookup(id, "fault"));

	} else if (strcmp(comp[id].type, "ide_gen_disk") == 0) {
		ide_gen_disk_create(id,
				comp[id].name,
				generic_get(id, "cow"),
				generic_get(id, "create"),
				generic_get(id, "sparse"),
				generic_get(id, "sync"),
				generic_get(id, "image"),
				generic_get(id, "cylinders"),
				generic_get(id, "heads"),
				generic_get(id, "sectors"),
				generic_get(id, "unit"),
				generic_get(id, "size"));
		ide_gen_disk_gui_create(comp[id].page,
				id,
				comp[id].name,
				generic_get(id, "cow"),
				generic_get(id, "create"),
				generic_get(id, "sparse"),
				generic_get(id, "sync"),
				generic_get(id, "image"),
				generic_get(id, "cylinders"),
				generic_get(id, "heads"),
				generic_get(id, "sectors"),
				generic_get(id, "unit"),
				generic_get(id, "size"));
		ide_gen_disk_init(id,
				sig_lookup(id, "power"),
				sig_lookup(id, "ide"),
				sig_lookup(id, "disk_fault"),
				sig_lookup(id, "block_fault"));
		ide_gen_disk_gui_init(id,
				sig_lookup(id, "power"),
				sig_lookup(id, "ide"),
				sig_lookup(id, "disk_fault"),
				sig_lookup(id, "block_fault"));

	} else if (strcmp(comp[id].type, "floppy_gen_floppydrive") == 0) {
		floppy_gen_floppydrive_create(id,
				comp[id].name,
				generic_get(id, "model"),
				generic_get(id, "unit"));
		floppy_gen_floppydrive_gui_create(comp[id].page,
				id,
				comp[id].name,
				generic_get(id, "model"),
				generic_get(id, "unit"));
		floppy_gen_floppydrive_init(id,
				sig_lookup(id, "power"),
				sig_lookup(id, "shugart"),
				sig_lookup(id, "opt_busy_led"),
				sig_lookup(id, "change"));
		floppy_gen_floppydrive_gui_init(id,
				sig_lookup(id, "power"),
				sig_lookup(id, "shugart"),
				sig_lookup(id, "opt_busy_led"),
				sig_lookup(id, "change"));

	} else if (strcmp(comp[id].type, "keyboard") == 0) {
		keyboard_create(id,
				comp[id].name);
		keyboard_gui_create(comp[id].page,
				id,
				comp[id].name);
		keyboard_init(id,
				sig_lookup(id, "ps2"),
				sig_lookup(id, "key0"),
				sig_lookup(id, "key1"),
				sig_lookup(id, "key2"),
				sig_lookup(id, "key3"),
				sig_lookup(id, "key4"),
				sig_lookup(id, "key5"),
				sig_lookup(id, "key6"),
				sig_lookup(id, "key7"),
				sig_lookup(id, "key8"),
				sig_lookup(id, "key9"),
				sig_lookup(id, "key10"),
				sig_lookup(id, "key11"),
				sig_lookup(id, "key12"),
				sig_lookup(id, "key13"),
				sig_lookup(id, "key14"),
				sig_lookup(id, "key15"),
				sig_lookup(id, "key16"),
				sig_lookup(id, "key17"),
				sig_lookup(id, "key18"),
				sig_lookup(id, "key19"),
				sig_lookup(id, "key20"),
				sig_lookup(id, "key21"),
				sig_lookup(id, "key22"),
				sig_lookup(id, "key23"),
				sig_lookup(id, "key24"),
				sig_lookup(id, "key25"),
				sig_lookup(id, "key26"),
				sig_lookup(id, "key27"),
				sig_lookup(id, "key28"),
				sig_lookup(id, "key29"),
				sig_lookup(id, "key30"),
				sig_lookup(id, "key31"),
				sig_lookup(id, "key32"),
				sig_lookup(id, "key33"),
				sig_lookup(id, "key34"),
				sig_lookup(id, "key35"),
				sig_lookup(id, "key36"),
				sig_lookup(id, "key37"),
				sig_lookup(id, "key38"),
				sig_lookup(id, "key39"),
				sig_lookup(id, "key40"),
				sig_lookup(id, "key41"),
				sig_lookup(id, "key42"),
				sig_lookup(id, "key43"),
				sig_lookup(id, "key44"),
				sig_lookup(id, "key45"),
				sig_lookup(id, "key46"),
				sig_lookup(id, "key47"),
				sig_lookup(id, "key48"),
				sig_lookup(id, "key49"),
				sig_lookup(id, "key50"),
				sig_lookup(id, "key51"),
				sig_lookup(id, "key52"),
				sig_lookup(id, "key53"),
				sig_lookup(id, "key54"),
				sig_lookup(id, "key55"),
				sig_lookup(id, "key56"),
				sig_lookup(id, "key57"),
				sig_lookup(id, "key58"),
				sig_lookup(id, "key59"),
				sig_lookup(id, "key60"),
				sig_lookup(id, "key61"),
				sig_lookup(id, "key62"),
				sig_lookup(id, "key63"),
				sig_lookup(id, "key64"),
				sig_lookup(id, "key65"),
				sig_lookup(id, "key66"),
				sig_lookup(id, "key67"),
				sig_lookup(id, "key68"),
				sig_lookup(id, "key69"),
				sig_lookup(id, "key70"),
				sig_lookup(id, "key71"),
				sig_lookup(id, "key72"),
				sig_lookup(id, "key73"),
				sig_lookup(id, "key74"),
				sig_lookup(id, "key75"),
				sig_lookup(id, "key76"),
				sig_lookup(id, "key77"),
				sig_lookup(id, "key78"),
				sig_lookup(id, "key79"),
				sig_lookup(id, "key80"),
				sig_lookup(id, "key81"),
				sig_lookup(id, "key82"),
				sig_lookup(id, "key83"),
				sig_lookup(id, "key84"),
				sig_lookup(id, "key85"),
				sig_lookup(id, "key86"),
				sig_lookup(id, "key87"),
				sig_lookup(id, "key88"),
				sig_lookup(id, "key89"),
				sig_lookup(id, "key90"),
				sig_lookup(id, "key91"),
				sig_lookup(id, "key92"),
				sig_lookup(id, "key93"),
				sig_lookup(id, "key94"),
				sig_lookup(id, "key95"),
				sig_lookup(id, "key96"),
				sig_lookup(id, "key97"),
				sig_lookup(id, "key98"),
				sig_lookup(id, "key99"),
				sig_lookup(id, "key100"),
				sig_lookup(id, "key101"),
				sig_lookup(id, "key102"),
				sig_lookup(id, "key103"),
				sig_lookup(id, "key104"),
				sig_lookup(id, "key105"),
				sig_lookup(id, "key106"),
				sig_lookup(id, "key107"),
				sig_lookup(id, "key108"),
				sig_lookup(id, "key109"),
				sig_lookup(id, "key110"),
				sig_lookup(id, "key111"),
				sig_lookup(id, "key112"),
				sig_lookup(id, "key113"),
				sig_lookup(id, "key114"),
				sig_lookup(id, "key115"),
				sig_lookup(id, "key116"),
				sig_lookup(id, "key117"),
				sig_lookup(id, "key118"),
				sig_lookup(id, "key119"),
				sig_lookup(id, "key120"),
				sig_lookup(id, "key121"),
				sig_lookup(id, "key122"),
				sig_lookup(id, "key123"),
				sig_lookup(id, "key124"),
				sig_lookup(id, "key125"),
				sig_lookup(id, "key126"),
				sig_lookup(id, "key127"),
				sig_lookup(id, "opt_num_led"),
				sig_lookup(id, "opt_caps_led"),
				sig_lookup(id, "opt_scroll_led"));
		keyboard_gui_init(id,
				sig_lookup(id, "ps2"),
				sig_lookup(id, "key0"),
				sig_lookup(id, "key1"),
				sig_lookup(id, "key2"),
				sig_lookup(id, "key3"),
				sig_lookup(id, "key4"),
				sig_lookup(id, "key5"),
				sig_lookup(id, "key6"),
				sig_lookup(id, "key7"),
				sig_lookup(id, "key8"),
				sig_lookup(id, "key9"),
				sig_lookup(id, "key10"),
				sig_lookup(id, "key11"),
				sig_lookup(id, "key12"),
				sig_lookup(id, "key13"),
				sig_lookup(id, "key14"),
				sig_lookup(id, "key15"),
				sig_lookup(id, "key16"),
				sig_lookup(id, "key17"),
				sig_lookup(id, "key18"),
				sig_lookup(id, "key19"),
				sig_lookup(id, "key20"),
				sig_lookup(id, "key21"),
				sig_lookup(id, "key22"),
				sig_lookup(id, "key23"),
				sig_lookup(id, "key24"),
				sig_lookup(id, "key25"),
				sig_lookup(id, "key26"),
				sig_lookup(id, "key27"),
				sig_lookup(id, "key28"),
				sig_lookup(id, "key29"),
				sig_lookup(id, "key30"),
				sig_lookup(id, "key31"),
				sig_lookup(id, "key32"),
				sig_lookup(id, "key33"),
				sig_lookup(id, "key34"),
				sig_lookup(id, "key35"),
				sig_lookup(id, "key36"),
				sig_lookup(id, "key37"),
				sig_lookup(id, "key38"),
				sig_lookup(id, "key39"),
				sig_lookup(id, "key40"),
				sig_lookup(id, "key41"),
				sig_lookup(id, "key42"),
				sig_lookup(id, "key43"),
				sig_lookup(id, "key44"),
				sig_lookup(id, "key45"),
				sig_lookup(id, "key46"),
				sig_lookup(id, "key47"),
				sig_lookup(id, "key48"),
				sig_lookup(id, "key49"),
				sig_lookup(id, "key50"),
				sig_lookup(id, "key51"),
				sig_lookup(id, "key52"),
				sig_lookup(id, "key53"),
				sig_lookup(id, "key54"),
				sig_lookup(id, "key55"),
				sig_lookup(id, "key56"),
				sig_lookup(id, "key57"),
				sig_lookup(id, "key58"),
				sig_lookup(id, "key59"),
				sig_lookup(id, "key60"),
				sig_lookup(id, "key61"),
				sig_lookup(id, "key62"),
				sig_lookup(id, "key63"),
				sig_lookup(id, "key64"),
				sig_lookup(id, "key65"),
				sig_lookup(id, "key66"),
				sig_lookup(id, "key67"),
				sig_lookup(id, "key68"),
				sig_lookup(id, "key69"),
				sig_lookup(id, "key70"),
				sig_lookup(id, "key71"),
				sig_lookup(id, "key72"),
				sig_lookup(id, "key73"),
				sig_lookup(id, "key74"),
				sig_lookup(id, "key75"),
				sig_lookup(id, "key76"),
				sig_lookup(id, "key77"),
				sig_lookup(id, "key78"),
				sig_lookup(id, "key79"),
				sig_lookup(id, "key80"),
				sig_lookup(id, "key81"),
				sig_lookup(id, "key82"),
				sig_lookup(id, "key83"),
				sig_lookup(id, "key84"),
				sig_lookup(id, "key85"),
				sig_lookup(id, "key86"),
				sig_lookup(id, "key87"),
				sig_lookup(id, "key88"),
				sig_lookup(id, "key89"),
				sig_lookup(id, "key90"),
				sig_lookup(id, "key91"),
				sig_lookup(id, "key92"),
				sig_lookup(id, "key93"),
				sig_lookup(id, "key94"),
				sig_lookup(id, "key95"),
				sig_lookup(id, "key96"),
				sig_lookup(id, "key97"),
				sig_lookup(id, "key98"),
				sig_lookup(id, "key99"),
				sig_lookup(id, "key100"),
				sig_lookup(id, "key101"),
				sig_lookup(id, "key102"),
				sig_lookup(id, "key103"),
				sig_lookup(id, "key104"),
				sig_lookup(id, "key105"),
				sig_lookup(id, "key106"),
				sig_lookup(id, "key107"),
				sig_lookup(id, "key108"),
				sig_lookup(id, "key109"),
				sig_lookup(id, "key110"),
				sig_lookup(id, "key111"),
				sig_lookup(id, "key112"),
				sig_lookup(id, "key113"),
				sig_lookup(id, "key114"),
				sig_lookup(id, "key115"),
				sig_lookup(id, "key116"),
				sig_lookup(id, "key117"),
				sig_lookup(id, "key118"),
				sig_lookup(id, "key119"),
				sig_lookup(id, "key120"),
				sig_lookup(id, "key121"),
				sig_lookup(id, "key122"),
				sig_lookup(id, "key123"),
				sig_lookup(id, "key124"),
				sig_lookup(id, "key125"),
				sig_lookup(id, "key126"),
				sig_lookup(id, "key127"),
				sig_lookup(id, "opt_num_led"),
				sig_lookup(id, "opt_caps_led"),
				sig_lookup(id, "opt_scroll_led"));

	} else if (strcmp(comp[id].type, "ps2_gen_mouse_4") == 0) {
		ps2_gen_mouse_4_create(id,
				comp[id].name);
		ps2_gen_mouse_4_gui_create(comp[id].page,
				id,
				comp[id].name);
		ps2_gen_mouse_4_init(id,
				sig_lookup(id, "ps2"),
				sig_lookup(id, "deltax"),
				sig_lookup(id, "deltay"),
				sig_lookup(id, "deltaz"),
				sig_lookup(id, "button1"),
				sig_lookup(id, "button2"),
				sig_lookup(id, "button3"),
				sig_lookup(id, "button4"),
				sig_lookup(id, "button5"));
		ps2_gen_mouse_4_gui_init(id,
				sig_lookup(id, "ps2"),
				sig_lookup(id, "deltax"),
				sig_lookup(id, "deltay"),
				sig_lookup(id, "deltaz"),
				sig_lookup(id, "button1"),
				sig_lookup(id, "button2"),
				sig_lookup(id, "button3"),
				sig_lookup(id, "button4"),
				sig_lookup(id, "button5"));

	} else if (strcmp(comp[id].type, "vt102_monitor") == 0) {
		vt102_monitor_create(id,
				comp[id].name);
		vt102_monitor_gui_create(comp[id].page,
				id,
				comp[id].name);
		vt102_monitor_init(id,
				sig_lookup(id, "power_230v"),
				sig_lookup(id, "in_video"),
				sig_lookup(id, "opt_video"),
				sig_lookup(id, "screen_shot"));
		vt102_monitor_gui_init(id,
				sig_lookup(id, "power_230v"),
				sig_lookup(id, "in_video"),
				sig_lookup(id, "opt_video"),
				sig_lookup(id, "screen_shot"));

	} else if (strcmp(comp[id].type, "monitor") == 0) {
		monitor_create(id,
				comp[id].name);
		monitor_gui_create(comp[id].page,
				id,
				comp[id].name);
		monitor_init(id,
				sig_lookup(id, "power_230v"),
				sig_lookup(id, "vga"),
				sig_lookup(id, "opt_video"),
				sig_lookup(id, "screen_shot"));
		monitor_gui_init(id,
				sig_lookup(id, "power_230v"),
				sig_lookup(id, "vga"),
				sig_lookup(id, "opt_video"),
				sig_lookup(id, "screen_shot"));

		sig_boolean_connect_out(sig_lookup(id, "power_230v"),
				cpssp, 1);
		sig_boolean_set(sig_lookup(id, "power_230v"),
				cpssp, 1);

	} else if (strcmp(comp[id].type, "patternm") == 0) {
		patternm_create(id,
				comp[id].name);
		patternm_init(id,
				sig_lookup(id, "video"),
				sig_lookup(id, "slot0"),
				sig_lookup(id, "slot1"),
				sig_lookup(id, "slot2"),
				sig_lookup(id, "slot3"),
				sig_lookup(id, "slot4"),
				sig_lookup(id, "slot5"),
				sig_lookup(id, "slot6"),
				sig_lookup(id, "slot7"),
				sig_lookup(id, "slot8"),
				sig_lookup(id, "slot9"),
				sig_lookup(id, "slot10"),
				sig_lookup(id, "slot11"),
				sig_lookup(id, "slot12"),
				sig_lookup(id, "slot13"),
				sig_lookup(id, "slot14"),
				sig_lookup(id, "slot15"),
				sig_lookup(id, "text0"),
				sig_lookup(id, "text0_state"),
				sig_lookup(id, "text1"),
				sig_lookup(id, "text1_state"),
				sig_lookup(id, "text2"),
				sig_lookup(id, "text2_state"),
				sig_lookup(id, "text3"),
				sig_lookup(id, "text3_state"),
				sig_lookup(id, "asc_text0"),
				sig_lookup(id, "asc_text0_state"),
				sig_lookup(id, "asc_text1"),
				sig_lookup(id, "asc_text1_state"),
				sig_lookup(id, "asc_text2"),
				sig_lookup(id, "asc_text2_state"),
				sig_lookup(id, "asc_text3"),
				sig_lookup(id, "asc_text3_state"),
				sig_lookup(id, "pattern_rectangle0"),
				sig_lookup(id, "match_rectangle0_x"),
				sig_lookup(id, "match_rectangle0_y"),
				sig_lookup(id, "match_rectangle0_w"),
				sig_lookup(id, "match_rectangle0_h"),
				sig_lookup(id, "pattern_rectangle1"),
				sig_lookup(id, "match_rectangle1_x"),
				sig_lookup(id, "match_rectangle1_y"),
				sig_lookup(id, "match_rectangle1_w"),
				sig_lookup(id, "match_rectangle1_h"),
				sig_lookup(id, "pattern_rectangle2"),
				sig_lookup(id, "match_rectangle2_x"),
				sig_lookup(id, "match_rectangle2_y"),
				sig_lookup(id, "match_rectangle2_w"),
				sig_lookup(id, "match_rectangle2_h"),
				sig_lookup(id, "pattern_rectangle3"),
				sig_lookup(id, "match_rectangle3_x"),
				sig_lookup(id, "match_rectangle3_y"),
				sig_lookup(id, "match_rectangle3_w"),
				sig_lookup(id, "match_rectangle3_h"));

	} else if (strcmp(comp[id].type, "mouse_matcher") == 0) {
		mouse_matcher_create(id,
				comp[id].name);
		mouse_matcher_init(id,
				sig_lookup(id, "pointerdir"),
				sig_lookup(id, "match_state"),
				sig_lookup(id, "event"),
				sig_lookup(id, "x"),
				sig_lookup(id, "y"),
				sig_lookup(id, "slot0"),
				sig_lookup(id, "slot1"),
				sig_lookup(id, "slot2"),
				sig_lookup(id, "slot3"),
				sig_lookup(id, "slot4"),
				sig_lookup(id, "slot5"),
				sig_lookup(id, "slot6"),
				sig_lookup(id, "slot7"),
				sig_lookup(id, "slot8"),
				sig_lookup(id, "slot9"),
				sig_lookup(id, "slot10"),
				sig_lookup(id, "slot11"));

	} else if (strcmp(comp[id].type, "loudspeaker") == 0) {
		loudspeaker_create(id,
				comp[id].name);
		loudspeaker_aui_create(id,
				comp[id].name);
		loudspeaker_init(id,
				sig_lookup(id, "sound_in"),
				sig_lookup(id, "audio_sound"));
		loudspeaker_aui_init(id,
				sig_lookup(id, "sound_in"),
				sig_lookup(id, "audio_sound"));

	} else if (strcmp(comp[id].type, "usb2serial") == 0) {
		usb2serial_create(id,
				comp[id].name);
		usb2serial_init(id,
				sig_lookup(id, "serial"),
				sig_lookup(id, "usb"));

	} else if (strcmp(comp[id].type, "usbstorage") == 0) {
		usbstorage_create(id,
				comp[id].name);
		usbstorage_init(id,
				sig_lookup(id, "usb"));

	} else if (strcmp(comp[id].type, "vt102_keyboard") == 0) {
		vt102_keyboard_create(id,
				comp[id].name);
		vt102_keyboard_gui_create(comp[id].page,
				id,
				comp[id].name);
		vt102_keyboard_init(id,
				sig_lookup(id, "keyboard"),
				sig_lookup(id, "key0"),
				sig_lookup(id, "key1"),
				sig_lookup(id, "key2"),
				sig_lookup(id, "key3"),
				sig_lookup(id, "key4"),
				sig_lookup(id, "key5"),
				sig_lookup(id, "key6"),
				sig_lookup(id, "key7"),
				sig_lookup(id, "key8"),
				sig_lookup(id, "key9"),
				sig_lookup(id, "key10"),
				sig_lookup(id, "key11"),
				sig_lookup(id, "key12"),
				sig_lookup(id, "key13"),
				sig_lookup(id, "key14"),
				sig_lookup(id, "key15"),
				sig_lookup(id, "key16"),
				sig_lookup(id, "key17"),
				sig_lookup(id, "key18"),
				sig_lookup(id, "key19"),
				sig_lookup(id, "key20"),
				sig_lookup(id, "key21"),
				sig_lookup(id, "key22"),
				sig_lookup(id, "key23"),
				sig_lookup(id, "key24"),
				sig_lookup(id, "key25"),
				sig_lookup(id, "key26"),
				sig_lookup(id, "key27"),
				sig_lookup(id, "key28"),
				sig_lookup(id, "key29"),
				sig_lookup(id, "key30"),
				sig_lookup(id, "key31"),
				sig_lookup(id, "key32"),
				sig_lookup(id, "key33"),
				sig_lookup(id, "key34"),
				sig_lookup(id, "key35"),
				sig_lookup(id, "key36"),
				sig_lookup(id, "key37"),
				sig_lookup(id, "key38"),
				sig_lookup(id, "key39"),
				sig_lookup(id, "key40"),
				sig_lookup(id, "key41"),
				sig_lookup(id, "key42"),
				sig_lookup(id, "key43"),
				sig_lookup(id, "key44"),
				sig_lookup(id, "key45"),
				sig_lookup(id, "key46"),
				sig_lookup(id, "key47"),
				sig_lookup(id, "key48"),
				sig_lookup(id, "key49"),
				sig_lookup(id, "key50"),
				sig_lookup(id, "key51"),
				sig_lookup(id, "key52"),
				sig_lookup(id, "key53"),
				sig_lookup(id, "key54"),
				sig_lookup(id, "key55"),
				sig_lookup(id, "key56"),
				sig_lookup(id, "key57"),
				sig_lookup(id, "key58"),
				sig_lookup(id, "key59"),
				sig_lookup(id, "key60"),
				sig_lookup(id, "key61"),
				sig_lookup(id, "key62"),
				sig_lookup(id, "key63"),
				sig_lookup(id, "key64"),
				sig_lookup(id, "key65"),
				sig_lookup(id, "key66"),
				sig_lookup(id, "key67"),
				sig_lookup(id, "key68"),
				sig_lookup(id, "key69"),
				sig_lookup(id, "key70"),
				sig_lookup(id, "key71"),
				sig_lookup(id, "key72"),
				sig_lookup(id, "key73"),
				sig_lookup(id, "key74"),
				sig_lookup(id, "key75"),
				sig_lookup(id, "key76"),
				sig_lookup(id, "key77"),
				sig_lookup(id, "key78"),
				sig_lookup(id, "key79"),
				sig_lookup(id, "key80"),
				sig_lookup(id, "key81"),
				sig_lookup(id, "key82"),
				sig_lookup(id, "key83"),
				sig_lookup(id, "key84"),
				sig_lookup(id, "key85"),
				sig_lookup(id, "key86"),
				sig_lookup(id, "key87"),
				sig_lookup(id, "key88"),
				sig_lookup(id, "key89"),
				sig_lookup(id, "key90"),
				sig_lookup(id, "key91"),
				sig_lookup(id, "key92"),
				sig_lookup(id, "key93"),
				sig_lookup(id, "key94"),
				sig_lookup(id, "key95"),
				sig_lookup(id, "key96"),
				sig_lookup(id, "key97"),
				sig_lookup(id, "key98"),
				sig_lookup(id, "key99"),
				sig_lookup(id, "key100"),
				sig_lookup(id, "key101"),
				sig_lookup(id, "key102"),
				sig_lookup(id, "key103"),
				sig_lookup(id, "key104"),
				sig_lookup(id, "key105"),
				sig_lookup(id, "key106"),
				sig_lookup(id, "key107"),
				sig_lookup(id, "key108"),
				sig_lookup(id, "key109"),
				sig_lookup(id, "key110"),
				sig_lookup(id, "key111"),
				sig_lookup(id, "key112"),
				sig_lookup(id, "key113"),
				sig_lookup(id, "key114"),
				sig_lookup(id, "key115"),
				sig_lookup(id, "key116"),
				sig_lookup(id, "key117"),
				sig_lookup(id, "key118"),
				sig_lookup(id, "key119"),
				sig_lookup(id, "key120"),
				sig_lookup(id, "key121"),
				sig_lookup(id, "key122"),
				sig_lookup(id, "key123"),
				sig_lookup(id, "key124"),
				sig_lookup(id, "key125"),
				sig_lookup(id, "key126"),
				sig_lookup(id, "key127"));
		vt102_keyboard_gui_init(id,
				sig_lookup(id, "keyboard"),
				sig_lookup(id, "key0"),
				sig_lookup(id, "key1"),
				sig_lookup(id, "key2"),
				sig_lookup(id, "key3"),
				sig_lookup(id, "key4"),
				sig_lookup(id, "key5"),
				sig_lookup(id, "key6"),
				sig_lookup(id, "key7"),
				sig_lookup(id, "key8"),
				sig_lookup(id, "key9"),
				sig_lookup(id, "key10"),
				sig_lookup(id, "key11"),
				sig_lookup(id, "key12"),
				sig_lookup(id, "key13"),
				sig_lookup(id, "key14"),
				sig_lookup(id, "key15"),
				sig_lookup(id, "key16"),
				sig_lookup(id, "key17"),
				sig_lookup(id, "key18"),
				sig_lookup(id, "key19"),
				sig_lookup(id, "key20"),
				sig_lookup(id, "key21"),
				sig_lookup(id, "key22"),
				sig_lookup(id, "key23"),
				sig_lookup(id, "key24"),
				sig_lookup(id, "key25"),
				sig_lookup(id, "key26"),
				sig_lookup(id, "key27"),
				sig_lookup(id, "key28"),
				sig_lookup(id, "key29"),
				sig_lookup(id, "key30"),
				sig_lookup(id, "key31"),
				sig_lookup(id, "key32"),
				sig_lookup(id, "key33"),
				sig_lookup(id, "key34"),
				sig_lookup(id, "key35"),
				sig_lookup(id, "key36"),
				sig_lookup(id, "key37"),
				sig_lookup(id, "key38"),
				sig_lookup(id, "key39"),
				sig_lookup(id, "key40"),
				sig_lookup(id, "key41"),
				sig_lookup(id, "key42"),
				sig_lookup(id, "key43"),
				sig_lookup(id, "key44"),
				sig_lookup(id, "key45"),
				sig_lookup(id, "key46"),
				sig_lookup(id, "key47"),
				sig_lookup(id, "key48"),
				sig_lookup(id, "key49"),
				sig_lookup(id, "key50"),
				sig_lookup(id, "key51"),
				sig_lookup(id, "key52"),
				sig_lookup(id, "key53"),
				sig_lookup(id, "key54"),
				sig_lookup(id, "key55"),
				sig_lookup(id, "key56"),
				sig_lookup(id, "key57"),
				sig_lookup(id, "key58"),
				sig_lookup(id, "key59"),
				sig_lookup(id, "key60"),
				sig_lookup(id, "key61"),
				sig_lookup(id, "key62"),
				sig_lookup(id, "key63"),
				sig_lookup(id, "key64"),
				sig_lookup(id, "key65"),
				sig_lookup(id, "key66"),
				sig_lookup(id, "key67"),
				sig_lookup(id, "key68"),
				sig_lookup(id, "key69"),
				sig_lookup(id, "key70"),
				sig_lookup(id, "key71"),
				sig_lookup(id, "key72"),
				sig_lookup(id, "key73"),
				sig_lookup(id, "key74"),
				sig_lookup(id, "key75"),
				sig_lookup(id, "key76"),
				sig_lookup(id, "key77"),
				sig_lookup(id, "key78"),
				sig_lookup(id, "key79"),
				sig_lookup(id, "key80"),
				sig_lookup(id, "key81"),
				sig_lookup(id, "key82"),
				sig_lookup(id, "key83"),
				sig_lookup(id, "key84"),
				sig_lookup(id, "key85"),
				sig_lookup(id, "key86"),
				sig_lookup(id, "key87"),
				sig_lookup(id, "key88"),
				sig_lookup(id, "key89"),
				sig_lookup(id, "key90"),
				sig_lookup(id, "key91"),
				sig_lookup(id, "key92"),
				sig_lookup(id, "key93"),
				sig_lookup(id, "key94"),
				sig_lookup(id, "key95"),
				sig_lookup(id, "key96"),
				sig_lookup(id, "key97"),
				sig_lookup(id, "key98"),
				sig_lookup(id, "key99"),
				sig_lookup(id, "key100"),
				sig_lookup(id, "key101"),
				sig_lookup(id, "key102"),
				sig_lookup(id, "key103"),
				sig_lookup(id, "key104"),
				sig_lookup(id, "key105"),
				sig_lookup(id, "key106"),
				sig_lookup(id, "key107"),
				sig_lookup(id, "key108"),
				sig_lookup(id, "key109"),
				sig_lookup(id, "key110"),
				sig_lookup(id, "key111"),
				sig_lookup(id, "key112"),
				sig_lookup(id, "key113"),
				sig_lookup(id, "key114"),
				sig_lookup(id, "key115"),
				sig_lookup(id, "key116"),
				sig_lookup(id, "key117"),
				sig_lookup(id, "key118"),
				sig_lookup(id, "key119"),
				sig_lookup(id, "key120"),
				sig_lookup(id, "key121"),
				sig_lookup(id, "key122"),
				sig_lookup(id, "key123"),
				sig_lookup(id, "key124"),
				sig_lookup(id, "key125"),
				sig_lookup(id, "key126"),
				sig_lookup(id, "key127"));

	} else if (strcmp(comp[id].type, "vt102_core") == 0) {
		vt102_core_create(id,
				comp[id].name);
		vt102_core_gui_create(comp[id].page,
				id,
				comp[id].name);
		vt102_core_init(id,
				sig_lookup(id, "mech_power_switch"),
				sig_lookup(id, "serial"),
				sig_lookup(id, "keyboard"),
				sig_lookup(id, "video"));
		vt102_core_gui_init(id,
				sig_lookup(id, "mech_power_switch"),
				sig_lookup(id, "serial"),
				sig_lookup(id, "keyboard"),
				sig_lookup(id, "video"));

	} else if (strcmp(comp[id].type, "cb_intel_eepro100") == 0) {
		cb_intel_eepro100_create(id,
				comp[id].name,
				generic_get(id, "mac"));
		cb_intel_eepro100_gui_create(comp[id].page,
				id,
				comp[id].name,
				generic_get(id, "mac"));
		cb_intel_eepro100_init(id,
				sig_lookup(id, "card"),
				sig_lookup(id, "eth"),
				sig_lookup(id, "opt_power_led"),
				sig_lookup(id, "opt_busy_led"),
				sig_lookup(id, "recv_loss"),
				sig_lookup(id, "send_loss"));
		cb_intel_eepro100_gui_init(id,
				sig_lookup(id, "card"),
				sig_lookup(id, "eth"),
				sig_lookup(id, "opt_power_led"),
				sig_lookup(id, "opt_busy_led"),
				sig_lookup(id, "recv_loss"),
				sig_lookup(id, "send_loss"));

	} else if (strcmp(comp[id].type, "dhcp_server") == 0) {
		dhcp_server_create(id,
				comp[id].name,
				generic_get(id, "mac"),
				generic_get(id, "ip"),
				generic_get(id, "first_ip"),
				generic_get(id, "netmask"),
				generic_get(id, "broadcast"),
				generic_get(id, "default_router"),
				generic_get(id, "dns_server"));
		dhcp_server_init(id,
				sig_lookup(id, "eth"));

	} else if (strcmp(comp[id].type, "serial_modem") == 0) {
		serial_modem_create(id,
				comp[id].name);
		serial_modem_gui_create(comp[id].page,
				id,
				comp[id].name);
		serial_modem_init(id,
				sig_lookup(id, "switch"),
				sig_lookup(id, "serial"),
				sig_lookup(id, "phone"),
				sig_lookup(id, "opt_online_led"),
				sig_lookup(id, "opt_rxd_led"),
				sig_lookup(id, "opt_txd_led"));
		serial_modem_gui_init(id,
				sig_lookup(id, "switch"),
				sig_lookup(id, "serial"),
				sig_lookup(id, "phone"),
				sig_lookup(id, "opt_online_led"),
				sig_lookup(id, "opt_rxd_led"),
				sig_lookup(id, "opt_txd_led"));

	} else if (strcmp(comp[id].type, "printer") == 0) {
		printer_create(id,
				comp[id].name);
		printer_init(id,
				sig_lookup(id, "par"));

	} else if (strcmp(comp[id].type, "telephone_switch") == 0) {
		telephone_switch_create(id,
				comp[id].name,
				generic_get(id, "num1"),
				generic_get(id, "num2"),
				generic_get(id, "num3"));
		telephone_switch_gui_create(comp[id].page,
				id,
				comp[id].name,
				generic_get(id, "num1"),
				generic_get(id, "num2"),
				generic_get(id, "num3"));
		telephone_switch_init(id,
				sig_lookup(id, "switch"),
				sig_lookup(id, "phone0"),
				sig_lookup(id, "phone1"),
				sig_lookup(id, "phone2"),
				sig_lookup(id, "carrier0"),
				sig_lookup(id, "carrier1"),
				sig_lookup(id, "carrier2"),
				sig_lookup(id, "connected0"),
				sig_lookup(id, "connected1"),
				sig_lookup(id, "connected2"));
		telephone_switch_gui_init(id,
				sig_lookup(id, "switch"),
				sig_lookup(id, "phone0"),
				sig_lookup(id, "phone1"),
				sig_lookup(id, "phone2"),
				sig_lookup(id, "carrier0"),
				sig_lookup(id, "carrier1"),
				sig_lookup(id, "carrier2"),
				sig_lookup(id, "connected0"),
				sig_lookup(id, "connected1"),
				sig_lookup(id, "connected2"));

		sig_boolean_connect_out(sig_lookup(id, "switch"), cpssp, 0);
		sig_boolean_set(sig_lookup(id, "switch"), cpssp, 1);

	} else if (strcmp(comp[id].type, "telephone_isp") == 0) {
		telephone_isp_create(id,
				comp[id].name);
		telephone_isp_gui_create(comp[id].page,
				id,
				comp[id].name);
		telephone_isp_init(id,
				sig_lookup(id, "phone"),
				sig_lookup(id, "switch"));
		telephone_isp_gui_init(id,
				sig_lookup(id, "phone"),
				sig_lookup(id, "switch"));

		sig_boolean_connect_out(sig_lookup(id, "switch"), cpssp, 0);
		sig_boolean_set(sig_lookup(id, "switch"), cpssp, 1);

	} else if (strcmp(comp[id].type, "elevator") == 0) {
		elevator_create(id,
				comp[id].name);
		elevator_gui_create(comp[id].page,
				id,
				comp[id].name);
		elevator_init(id,
				sig_lookup(id, "mech_power_switch"),
				sig_lookup(id, "dio48"),
				sig_lookup(id, "opt_led_a"),
				sig_lookup(id, "opt_led_b"),
				sig_lookup(id, "opt_led_c"),
				sig_lookup(id, "opt_led_d"),
				sig_lookup(id, "opt_led_e"),
				sig_lookup(id, "opt_led_f"),
				sig_lookup(id, "opt_led_g"),
				sig_lookup(id, "opt_led_arrow_down"),
				sig_lookup(id, "opt_led_arrow_up"),
				sig_lookup(id, "opt_led_panel_0"),
				sig_lookup(id, "opt_led_panel_1"),
				sig_lookup(id, "opt_led_panel_2"),
				sig_lookup(id, "opt_led_0_up"),
				sig_lookup(id, "opt_led_1_down"),
				sig_lookup(id, "opt_led_1_up"),
				sig_lookup(id, "opt_led_2_down"),
				sig_lookup(id, "mech_button_panel_0"),
				sig_lookup(id, "mech_button_panel_1"),
				sig_lookup(id, "mech_button_panel_2"),
				sig_lookup(id, "mech_button_0_up"),
				sig_lookup(id, "mech_button_1_down"),
				sig_lookup(id, "mech_button_1_up"),
				sig_lookup(id, "mech_button_2_down"),
				sig_lookup(id, "mech_door_0"),
				sig_lookup(id, "mech_door_1"),
				sig_lookup(id, "mech_door_2"),
				sig_lookup(id, "mech_cabin")
				);
		elevator_gui_init(id,
				sig_lookup(id, "mech_power_switch"),
				sig_lookup(id, "dio48"),
				sig_lookup(id, "opt_led_a"),
				sig_lookup(id, "opt_led_b"),
				sig_lookup(id, "opt_led_c"),
				sig_lookup(id, "opt_led_d"),
				sig_lookup(id, "opt_led_e"),
				sig_lookup(id, "opt_led_f"),
				sig_lookup(id, "opt_led_g"),
				sig_lookup(id, "opt_led_arrow_down"),
				sig_lookup(id, "opt_led_arrow_up"),
				sig_lookup(id, "opt_led_panel_0"),
				sig_lookup(id, "opt_led_panel_1"),
				sig_lookup(id, "opt_led_panel_2"),
				sig_lookup(id, "opt_led_0_up"),
				sig_lookup(id, "opt_led_1_down"),
				sig_lookup(id, "opt_led_1_up"),
				sig_lookup(id, "opt_led_2_down"),
				sig_lookup(id, "mech_button_panel_0"),
				sig_lookup(id, "mech_button_panel_1"),
				sig_lookup(id, "mech_button_panel_2"),
				sig_lookup(id, "mech_button_0_up"),
				sig_lookup(id, "mech_button_1_down"),
				sig_lookup(id, "mech_button_1_up"),
				sig_lookup(id, "mech_button_2_down"),
				sig_lookup(id, "mech_door_0"),
				sig_lookup(id, "mech_door_1"),
				sig_lookup(id, "mech_door_2"),
				sig_lookup(id, "mech_cabin")
				);

	} else if (strcmp(comp[id].type, "elevator_rc") == 0) {
		elevator_rc_gui_create(comp[id].page,
				id,
				comp[id].name);
		elevator_rc_gui_init(id,
				sig_lookup(id, "dio48"));

	} else if (strcmp(comp[id].type, "network_bridge") == 0) {
		network_bridge_create(id,
				comp[id].name,
				generic_get(id, "interface"));
		network_bridge_init(id,
				sig_lookup(id, "eth"));

	} else if (strcmp(comp[id].type, "network_router") == 0) {
		network_router_create(id,
				comp[id].name,
				generic_get(id, "mac"),
				generic_get(id, "ip"),
				generic_get(id, "first_ip"),
				generic_get(id, "netmask"),
				generic_get(id, "broadcast"),
				generic_get(id, "default_router"),
				generic_get(id, "dns_server"));
		network_router_init(id,
				sig_lookup(id, "console"),
				sig_lookup(id, "eth"));

	} else if (strcmp(comp[id].type, "dio24_loopback") == 0) {
		dio24_loopback_create(id,
				comp[id].name);
		dio24_loopback_init(id,
				sig_lookup(id, "dio24"));

	} else if (strcmp(comp[id].type, "dio48_loopback") == 0) {
                dio48_loopback_create(id,
				comp[id].name);
                dio48_loopback_init(id,
                        	sig_lookup(id, "dio48"));
				
	} else {
		fprintf(stderr, "%s\n", comp[id].type);
		assert(0);
	}
	return 0;
}

int
system_comp_exit(unsigned int id)
{
	if (sizeof(comp) / sizeof(comp[0]) <= id
	 || ! comp[id].type
	 || ! comp[id].name) {
		/* No such component. */
		return -1;
	}

	/* Nothing to do, jet. */

	return 0;
}

int
system_comp_destroy(unsigned int id)
{
	if (sizeof(comp) / sizeof(comp[0]) <= id
	 || ! comp[id].type
	 || ! comp[id].name) {
		/* No such component. */
		return -1;
	}

	if (strcmp(comp[id].type, "chassis") == 0) {
		chassis_destroy(id);
		chassis_aui_destroy(id);
		chassis_gui_destroy(id);
	} else if (strcmp(comp[id].type, "chassis_at") == 0) {
		chassis_at_destroy(id);
		chassis_at_aui_destroy(id);
		chassis_at_gui_destroy(id);
	} else if (strcmp(comp[id].type, "power_supply") == 0) {
		power_supply_destroy(id);
		power_supply_gui_destroy(id);
	} else if (strcmp(comp[id].type, "power_supply_at") == 0) {
		power_supply_at_destroy(id);
		power_supply_at_gui_destroy(id);
	} else if (strcmp(comp[id].type, "mb_ibm_at") == 0) {
		mb_ibm_at_destroy(id);
	} else if (strcmp(comp[id].type, "mb_ga_686dlx") == 0) {
		mb_ga_686dlx_destroy(id);
	} else if (strcmp(comp[id].type, "cpu_intel_80386") == 0) {
		cpu_intel_80386_destroy(id);
	} else if (strcmp(comp[id].type, "cpu_intel_80386_interpreter") == 0) {
		cpu_intel_80386_interpreter_destroy(id);
	} else if (strcmp(comp[id].type, "cpu_intel_80686_klamath") == 0) {
		cpu_intel_80686_klamath_destroy(id);
	} else if (strcmp(comp[id].type, "cpu_intel_80686_coppermine") == 0) {
		cpu_intel_80686_coppermine_destroy(id);
	} else if (strcmp(comp[id].type, "cpu_intel_x86_64") == 0) {
		cpu_intel_x86_64_destroy(id);
	} else if (strcmp(comp[id].type, "memory") == 0) {
		memory_destroy(id);
		memory_gui_destroy(id);
	} else if (strcmp(comp[id].type, "isa_gen_ne2000") == 0) {
		isa_gen_ne2000_destroy(id);
		isa_gen_ne2000_gui_destroy(id);
	} else if (strcmp(comp[id].type, "isa_gen_idectrl") == 0) {
		isa_gen_idectrl_destroy(id);
	} else if (strcmp(comp[id].type, "isa_gen_soundblaster16") == 0) {
		isa_gen_soundblaster16_destroy(id);
	} else if (strcmp(comp[id].type, "isa_gen_parallel") == 0) {
		isa_gen_parallel_destroy(id);
	} else if (strcmp(comp[id].type, "isa_gen_serial") == 0) {
		isa_gen_serial_destroy(id);
	} else if (strcmp(comp[id].type, "isa_gen_vga") == 0) {
		isa_gen_vga_destroy(id);
	} else if (strcmp(comp[id].type, "pci_cirrus_gd5446") == 0) {
		pci_cirrus_gd5446_destroy(id);
	} else if (strcmp(comp[id].type, "pci_cirrus_gd5446_2mb") == 0) {
		pci_cirrus_gd5446_2mb_destroy(id);
	} else if (strcmp(comp[id].type, "pci_cirrus_gd5446_3mb") == 0) {
		pci_cirrus_gd5446_3mb_destroy(id);
	} else if (strcmp(comp[id].type, "pci_intel_eepro100") == 0) {
		pci_intel_eepro100_destroy(id);
		pci_intel_eepro100_gui_destroy(id);
	} else if (strcmp(comp[id].type, "pci_ti_1520") == 0) {
		pci_ti_1520_destroy(id);
	} else if (strcmp(comp[id].type, "pci_lsi_20810") == 0) {
		pci_lsi_20810_destroy(id);
	} else if (strcmp(comp[id].type, "pci_gen_dio24") == 0) {
		pci_gen_dio24_destroy(id);
	} else if (strcmp(comp[id].type, "pci_gen_dio48") == 0) {
                pci_gen_dio48_destroy(id);
	} else if (strcmp(comp[id].type, "scsi_gen_cdrom") == 0) {
		scsi_gen_cdrom_destroy(id);
		scsi_gen_cdrom_gui_destroy(id);
	} else if (strcmp(comp[id].type, "scsi_gen_disk") == 0) {
		scsi_gen_disk_destroy(id);
	} else if (strcmp(comp[id].type, "ide_gen_cdrom") == 0) {
		ide_gen_cdrom_destroy(id);
		ide_gen_cdrom_gui_destroy(id);
	} else if (strcmp(comp[id].type, "ide_gen_disk") == 0) {
		ide_gen_disk_destroy(id);
	} else if (strcmp(comp[id].type, "floppy_gen_floppydrive") == 0) {
		floppy_gen_floppydrive_destroy(id);
		floppy_gen_floppydrive_gui_destroy(id);
	} else if (strcmp(comp[id].type, "keyboard") == 0) {
		keyboard_destroy(id);
		keyboard_gui_destroy(id);
	} else if (strcmp(comp[id].type, "ps2_gen_mouse_4") == 0) {
		ps2_gen_mouse_4_destroy(id);
		ps2_gen_mouse_4_gui_destroy(id);
	} else if (strcmp(comp[id].type, "vt102_monitor") == 0) {
		vt102_monitor_destroy(id);
		vt102_monitor_gui_destroy(id);
	} else if (strcmp(comp[id].type, "monitor") == 0) {
		monitor_destroy(id);
		monitor_gui_destroy(id);
	} else if (strcmp(comp[id].type, "patternm") == 0) {
		patternm_destroy(id);
	} else if (strcmp(comp[id].type, "mouse_matcher") == 0) {
		mouse_matcher_destroy(id);
	} else if (strcmp(comp[id].type, "loudspeaker") == 0) {
		loudspeaker_destroy(id);
		loudspeaker_aui_destroy(id);
	} else if (strcmp(comp[id].type, "usb2serial") == 0) {
		usb2serial_destroy(id);
	} else if (strcmp(comp[id].type, "usbstorage") == 0) {
		usbstorage_destroy(id);
	} else if (strcmp(comp[id].type, "vt102_keyboard") == 0) {
		vt102_keyboard_destroy(id);
		vt102_keyboard_gui_destroy(id);
	} else if (strcmp(comp[id].type, "vt102_core") == 0) {
		vt102_core_destroy(id);
		vt102_core_gui_destroy(id);
	} else if (strcmp(comp[id].type, "cb_intel_eepro100") == 0) {
		cb_intel_eepro100_destroy(id);
		cb_intel_eepro100_gui_destroy(id);
	} else if (strcmp(comp[id].type, "dhcp_server") == 0) {
		dhcp_server_destroy(id);
	} else if (strcmp(comp[id].type, "serial_modem") == 0) {
		serial_modem_destroy(id);
		serial_modem_gui_destroy(id);
	} else if (strcmp(comp[id].type, "printer") == 0) {
		printer_destroy(id);
	} else if (strcmp(comp[id].type, "telephone_switch") == 0) {
		telephone_switch_destroy(id);
		telephone_switch_gui_destroy(id);
	} else if (strcmp(comp[id].type, "telephone_isp") == 0) {
		telephone_isp_destroy(id);
		telephone_isp_gui_destroy(id);
	} else if (strcmp(comp[id].type, "elevator") == 0) {
		elevator_destroy(id);
		elevator_gui_destroy(id);
	} else if (strcmp(comp[id].type, "elevator_rc") == 0) {
		elevator_rc_gui_destroy(id);
	} else if (strcmp(comp[id].type, "network_bridge") == 0) {
		network_bridge_destroy(id);
	} else if (strcmp(comp[id].type, "network_router") == 0) {
		network_router_destroy(id);
	} else if (strcmp(comp[id].type, "dio24_loopback") == 0) {
		dio24_loopback_destroy(id);
	} else if (strcmp(comp[id].type, "dio48_loopback") == 0) {
                dio48_loopback_destroy(id);
	} else {
		fprintf(stderr, "%s\n", comp[id].type);
		assert(0);
	}

	free(comp[id].type);
	comp[id].type = NULL;

	free(comp[id].name);
	comp[id].name = NULL;

	return 0;
}

void
system_comp_connect(unsigned int comp_id, const char *port, unsigned int sig_id)
{
	struct sig_manage *sig_manage;
	void *sig_ptr;

	sig_manage = sig_lookup(comp_id, "manage");
	assert(sig_manage);

	sig_ptr = system_sig_get(sig_id);
	assert(sig_ptr);

	sig_manage_connect_port(sig_manage, port, sig_ptr);

	/* FIXME */
	// ret = system_sig_unget(sig_ptr);
	// assert(0 <= ret);
}

void
system_comp_disconnect(unsigned int comp_id, const char *port)
{
	struct sig_manage *sig_manage;

	sig_manage = sig_lookup(comp_id, "manage");
	assert(sig_manage);

	sig_manage_disconnect_port(sig_manage, port);
}

int
system_comp_lookup(const char *name)
{
	unsigned int id;

	for (id = 0; id < sizeof(comp) / sizeof(comp[0]); id++) {
		if (comp[id].name == NULL) {
			/* past last entry */
			return -1;
		}

		if (strcmp(comp[id].name, name) == 0) {
			return id;
		}
	}

	return -1;
}
