/* $Id: glue-vhdl.c,v 1.67 2009-05-07 09:49:20 vrsieh Exp $
 *
 *  Glue layer implementing the foreign interface of fauhdli for FAUmachine,
 *  so that fauhdli can access FAUmachine signals and components.
 *  Must not be called by FAUmachine itself.
 *
 * Copyright (C) 2008-2009 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#include "config.h"
#include <assert.h>
#include <string.h>
#include <stdio.h>
#include <glue-vhdl.h>
#include "glue-log.h"
#include "glue-main.h"
#include "glue-shm.h"
#include "../node-pc/system.h"
#include "sig_boolean.h"
#include "sig_integer.h"
#include "sig_gen.h"
#include "sig_string.h"
#include "simsetup.h"
#include "structure.h"

/* maximum possible signals. Must match system.c! */
#define GLUE_VHDL_MAX_SIGS 10000

/** mapping of one vhdl signal to FAUmachine signal */
struct glue_vhdl_signal_t {
	/** type of signal */
	enum sig_gen_type type;
	/** link back to glue_vhdl instance */
	struct glue_vhdl *self;
	/** driver pointer (for callback) */
	void *_drv;
	/** callback to hand update through to VHDL interpreter */
	void (*drv_update)(void *_drv, union fauhdli_value value);
};

/** parameter passing type */
struct glue_vhdl_param_t {
	/** name of the parameter */
	const char *name;
	/** passed argument */
	union fauhdli_value arg;
};

/** glue vhdl instance state */
struct glue_vhdl {
	/** all signals */
	struct glue_vhdl_signal_t signals[GLUE_VHDL_MAX_SIGS];
	/** all parameters for a foreign procedure call */
	struct glue_vhdl_param_t params[100];
	/** interpreter instance (as from glue_vhdl_enable_event_wakeups) */
	void *interpreter;
	/** callback to wakeup interpreter */
	void (*wakeup)(void *_interpreter);
	/** number of instantiated components */
	unsigned int inst_comps;
	/** number of instantiated signals */
	unsigned int inst_sigs;
	/** number of parameters present */
	unsigned int num_params;

	/** cache of page_name -> page_id mapping */
	struct {
		/** name of page */
		const char *page_name;
		/** corresponding page id */
		unsigned int page_id;
	} page_cache[10];
	/** number of first free slot in page_cache */
	unsigned int page_cache_count;

	/** cache of host -> node_id mapping */
	struct {
		/** host entry in simulation.setup */
		const char *host;
		/** corresponding node id */
		unsigned int node_id;
	} node_cache[10];
	/** number of first free slot in node_cache */
	unsigned int node_cache_count;

};

static void
glue_vhdl_wakeup_interpreter(struct glue_vhdl *s)
{
	/* in delta cycle, or call already scheduled. do nothing */
	if (s->interpreter == NULL) {
		return;
	}

	/* there should be one callback with the old t_next set...
	 * remove it */
	assert(s->wakeup != NULL);
	time_call_delete(s->wakeup, s->interpreter);

	/* wake up interpreter now */
	time_call_at(time_virt(), s->wakeup, s->interpreter);
	s->wakeup = NULL;
	s->interpreter = NULL;
}

/** callback to proxy to boolean changes to fauhdli 
 *  @param _s glue_vhdl_signal_t entry.
 *  @param val boolean value.
 */
static void
glue_vhdl_data_get_boolean(void *_s, unsigned int val)
{
	struct glue_vhdl_signal_t *s = (struct glue_vhdl_signal_t *)_s;
	union fauhdli_value f_val;

	f_val.univ_int = val;

	assert(s->_drv != NULL);
	assert(s->drv_update != NULL);
	s->drv_update(s->_drv, f_val);
	
	glue_vhdl_wakeup_interpreter(s->self);
}

static void
glue_vhdl_data_get_integer(void *_s, int val)
{
	struct glue_vhdl_signal_t *s = (struct glue_vhdl_signal_t *)_s;
	union fauhdli_value f_val;

	f_val.univ_int = val;

	assert(s->_drv != NULL);
	assert(s->drv_update != NULL);
	s->drv_update(s->_drv, f_val);

	glue_vhdl_wakeup_interpreter(s->self);
}

/** determine signal kind by name
 *  TODO move to sig_gen.c?
 *  @param name of signal type.
 *  @return signal kind
 */
static enum sig_gen_type
glue_vhdl_determine_type(const char *name)
{
	if (strcmp(name, "boolean") == 0) {
		return SIG_GEN_BOOLEAN;
	} else if (strcmp(name, "integer") == 0) {
		return SIG_GEN_INTEGER;
	} else if (strcmp(name, "cstring") == 0) {
		return SIG_GEN_STRING;
	} else if (strcmp(name, "serial") == 0) {
		return SIG_GEN_SERIAL;
	} else if (strcmp(name, "match") == 0) {
		return SIG_GEN_MATCH;
	} else if (strcmp(name, "video") == 0) {
		return SIG_GEN_VIDEO;
	} else if (strcmp(name, "opt_rgb") == 0) {
		return SIG_GEN_OPT_RGB;
	} else if (strcmp(name, "usb_bus") == 0) {
		return SIG_GEN_USB_BUS;
	} else if (strcmp(name, "eth") == 0) {
		return SIG_GEN_ETH;
	} else if (strcmp(name, "mem_bus") == 0) {
		return SIG_GEN_MEM_BUS;
	} else if (strcmp(name, "pci_bus") == 0) {
		return SIG_GEN_PCI_BUS;
	} else if (strcmp(name, "parallel") == 0) {
		return SIG_GEN_PARALLEL;
	} else if (strcmp(name, "isa_bus") == 0) {
		return SIG_GEN_ISA_BUS;
	} else if (strcmp(name, "ide_bus") == 0) {
		return SIG_GEN_IDE_BUS;
	} else if (strcmp(name, "host_bus") == 0) {
		return SIG_GEN_HOST_BUS;
	} else if (strcmp(name, "agp_bus") == 0) {
		return SIG_GEN_AGP_BUS;
	} else if (strcmp(name, "ps2") == 0) {
		return SIG_GEN_PS2;
	} else if (strcmp(name, "ps2_main") == 0) {
		return SIG_GEN_PS2_MAIN;
	} else if (strcmp(name, "scsi_bus") == 0) {
		return SIG_GEN_SCSI_BUS;
	} else if (strcmp(name, "shugart_bus") == 0) {
		return SIG_GEN_SHUGART_BUS;
	} else if (strcmp(name, "sound") == 0) {
		return SIG_GEN_SOUND;
	} else if (strcmp(name, "std_logic") == 0) {
		return SIG_GEN_STD_LOGIC;
	} else if (strcmp(name, "vga") == 0) {
		return SIG_GEN_VGA;
	} else if (strcmp(name, "power_board") == 0) {
		return SIG_GEN_POWER_BOARD;
	} else if (strcmp(name, "power_board_at") == 0) {
		return SIG_GEN_POWER_BOARD_AT;
	} else if (strcmp(name, "power_device") == 0) {
		return SIG_GEN_POWER_DEVICE;
	} else if (strcmp(name, "dio48") == 0) {
		return SIG_GEN_DIO48;
	} else if (strcmp(name, "dio24") == 0) {
		return SIG_GEN_DIO24;
	} else if (strcmp(name, "magneto_optical") == 0) {
		return SIG_GEN_MAGNETO_OPTICAL;
	}


	faum_log(FAUM_LOG_WARNING, "glue-vhdl", __func__, 
		"Could not determine signal type '%s'.\n", name);
	return SIG_GEN_MANAGE;
}

static void
glue_vhdl_get_arg(
	const struct glue_vhdl *s,
	const char *arg,
	union fauhdli_value *param
)
{
	unsigned int i;
	for (i = 0; i < s->num_params; i++) {
		if (strcmp(arg, s->params[i].name) == 0) {
			/* found */
			*param = s->params[i].arg;
			return;
		}
	}

	faum_log(FAUM_LOG_FATAL, "glue-vhdl", __func__,
		"Argument %s not found, aborting.\n", arg);
	assert(0);
}

/** flatten a VHDL constant unconstraint string array to a c-string
 *  @param s_ptr base pointer to temporary.
 *  @param s_len length of the string.
 *  @return flattened c-string.
 */
static char *
glue_vhdl_flatten_string(union fauhdli_value s_ptr, size_t s_len)
{
	static char buf[16384];
	size_t i;
	union fauhdli_value *walk = (union fauhdli_value *)s_ptr.pointer;

	assert(s_len < sizeof(buf));
	
	for (i = 0; i < s_len; i++, walk++) {
		buf[i] = (char)(walk->univ_int);
	}
	buf[i] = '\0';
	return buf;
}

static void
glue_vhdl_send_string(
	union fauhdli_value dest_driver,
	union fauhdli_value s_array,
	union fauhdli_value s_lb,
	union fauhdli_value s_ub
)
{
	size_t sz; 
	const char *msg;
	unsigned int sig_id;
	void *_sig;
	struct sig_string *sig;
	
	sz = (size_t)(s_ub.univ_int - s_lb.univ_int + 1);
	msg = glue_vhdl_flatten_string(s_array, sz);
	sig_id = fauhdli_get_sig_id_driver(dest_driver.pointer);
	_sig = system_sig_get(sig_id);

	assert(_sig != NULL);
	sig = (struct sig_string *)_sig;


#if 0 /* debug code */
	faum_log(FAUM_LOG_DEBUG, "glue-vhdl", __func__,
		"sending %s to signal %d\n", msg, sig_id);
#endif /* debug code */
	sig_string_set(sig, dest_driver.pointer, msg);
}

static void
glue_vhdl_terminate(void)
{
	frontend_quit();
}

static void
glue_vhdl_toggle_debug(void)
{
	loglevel ^= 1;
}

static void
glue_vhdl_shorcut_drv_out(
	union fauhdli_value s_driver,
	union fauhdli_value comp_array,
	union fauhdli_value comp_lb,
	union fauhdli_value comp_ub,
	union fauhdli_value comp_port_array,
	union fauhdli_value comp_port_lb,
	union fauhdli_value comp_port_ub
)
{
	size_t sz;
	int c_id;
	unsigned int sig_id;
	char *s;

	sz = (size_t)(comp_ub.univ_int - comp_lb.univ_int + 1);
	s = glue_vhdl_flatten_string(comp_array, sz);

	c_id = system_comp_lookup(s);
	assert(c_id >= 0);

	sz = (size_t)(comp_port_ub.univ_int - comp_port_lb.univ_int + 1);
	s = glue_vhdl_flatten_string(comp_port_array, sz);

	sig_id = fauhdli_get_sig_id_driver(s_driver.pointer);
	system_comp_connect(c_id, s, sig_id);
}

/* FIXME move to simsetup.c? */
const char *
glue_vhdl_simsetup_get_closest(char *path, const char *key)
{
	const char *ret;
	char *delim;

	while(1) {
		ret = simsetup_get(path, key);
		if (ret != NULL) {
			return ret;
		}
		
		delim = strrchr(path, '/');
		if (delim == NULL) {
			return delim;
		}
		*delim = '\0';
	}

	return ret;
}


static unsigned int
glue_vhdl_page_cache_get(struct glue_vhdl *s, const char *page_name)
{
	unsigned int i;
	unsigned int page_id;

	for (i = 0; i < s->page_cache_count; i++) {
		if (strcmp(s->page_cache[i].page_name, page_name) == 0) {
			return s->page_cache[i].page_id;
		}
	}

	/* create new page, add new entry */
	page_id = system_page_create(page_name);

	assert(s->page_cache_count 
		< sizeof(s->page_cache) / sizeof(s->page_cache[0]));
	s->page_cache[s->page_cache_count].page_name = page_name;
	s->page_cache[s->page_cache_count].page_id = page_id;
	s->page_cache_count++;

	return page_id;
}

static unsigned int
glue_vhdl_node_cache_get(struct glue_vhdl *s, const char *host)
{
	unsigned int i;
	unsigned int node_id;

	for (i = 0; i < s->node_cache_count; i++) {
		if (strcmp(s->node_cache[i].host, host) == 0) {
			return s->node_cache[i].node_id;
		}
	}

	/* create new page, add new entry */
	node_id = system_node_create(host);

	assert(s->node_cache_count 
		< sizeof(s->node_cache) / sizeof(s->node_cache[0]));
	s->node_cache[s->node_cache_count].host = host;
	s->node_cache[s->node_cache_count].node_id = node_id;
	s->node_cache_count++;

	return node_id;
}



static unsigned int
glue_vhdl_get_page(struct glue_vhdl *s, const char *name)
{
	static char buf[16348];
	size_t i;
	const char *page_name;
	unsigned int ret;

	/* FIXME convert ':' to '/' in name, cf. simsetup.l 
	 */
	assert(strlen(name) < sizeof(buf) - 1);
	for (i = 0; i < strlen(name); i++) {
		if (name[i] == ':') {
			buf[i] = '/';
		} else {
			buf[i] = name[i];
		}
	}
	buf[i] = '\0';
	/* end FIXME */

	page_name = glue_vhdl_simsetup_get_closest(buf, "page");
	if (page_name == NULL) {
		/* no page specified in simsetup, use plain "/" with
		   a page name of "default"
		 */
		ret = glue_vhdl_page_cache_get(s, "default");
		return ret;
	}

	ret = glue_vhdl_page_cache_get(s, page_name);
	return ret;
}

static unsigned int
glue_vhdl_get_node(struct glue_vhdl *s, const char *name)
{
	static char buf[16348];
	size_t i;
	const char *page_name;
	unsigned int ret;

	/* FIXME convert ':' to '/' in name, cf. simsetup.l 
	 */
	assert(strlen(name) < sizeof(buf) - 1);
	for (i = 0; i < strlen(name); i++) {
		if (name[i] == ':') {
			buf[i] = '/';
		} else {
			buf[i] = name[i];
		}
	}
	buf[i] = '\0';
	/* end FIXME */

	page_name = glue_vhdl_simsetup_get_closest(buf, "host");
	if (page_name == NULL) {
		/* no host specified in simsetup, use plain "/" with
		   a host entry of "default"
		 */
		ret = glue_vhdl_node_cache_get(s, "default");
		return ret;
	}

	ret = glue_vhdl_node_cache_get(s, page_name);
	return ret;
}

static void
glue_vhdl_simsetup_set(unsigned int comp_id, const char *comp_name)
{
	static char buf[16348];
	size_t i;
	struct structure_component *sc;
	struct structure_keyval *key_val;

	/* FIXME convert ':' to '/' in name, cf. simsetup.l (see above)
	 */
	assert(strlen(comp_name) < sizeof(buf) - 1);
	for (i = 0; i < strlen(comp_name); i++) {
		if (comp_name[i] == ':') {
			buf[i] = '/';
		} else {
			buf[i] = comp_name[i];
		}
	}
	buf[i] = '\0';
	/* end FIXME */

	sc = simsetup_for_component(buf);
	if (sc == NULL) {
		return;
	}

	
	list_Foreach(&sc->simsetups, key_val) {
		/* FIXME hm... should page be handled specially? */
		if (strcmp(key_val->name, "page") == 0) {
			continue;
		}
		/* FIXME type string? */
		system_comp_generic_set(comp_id, 
					"string",
					key_val->name, 
					key_val->value);
	}
}

void
glue_vhdl_proc_set(
	struct glue_vhdl *s,
	const char *proc,
	const char *param, 
	union fauhdli_value value
)
{
	assert(s->num_params < (sizeof(s->params) / sizeof(s->params[0])));
	s->params[s->num_params].name = param;
	s->params[s->num_params].arg = value;
	s->num_params++;
}


void
glue_vhdl_proc_call(struct glue_vhdl *s, const char *proc)
{
	if (strcmp(proc, "send_string") == 0) {
		union fauhdli_value dest_driver;
		union fauhdli_value s_array;
		union fauhdli_value s_lb;
		union fauhdli_value s_ub;

		glue_vhdl_get_arg(s, "dest__driver__", &dest_driver);
		glue_vhdl_get_arg(s, "s", &s_array);
		glue_vhdl_get_arg(s, "s_lb_0", &s_lb);
		glue_vhdl_get_arg(s, "s_ub_0", &s_ub);

		glue_vhdl_send_string(dest_driver, s_array, s_lb, s_ub);

	} else if (strcmp(proc, "terminate") == 0) {
		glue_vhdl_terminate();

	} else if (strcmp(proc, "debug") == 0) {
		glue_vhdl_toggle_debug();

	} else if ((strcmp(proc, "shortcut_bool_out") == 0)
		|| (strcmp(proc, "shortcut_int_out")  == 0)) {
		union fauhdli_value s_driver;
		union fauhdli_value comp_array;
		union fauhdli_value comp_lb;
		union fauhdli_value comp_ub;
		union fauhdli_value comp_port_array;
		union fauhdli_value comp_port_lb;
		union fauhdli_value comp_port_ub;

		glue_vhdl_get_arg(s, "s__driver__", &s_driver);
		glue_vhdl_get_arg(s, "comp", &comp_array);
		glue_vhdl_get_arg(s, "comp_lb_0", &comp_lb);
		glue_vhdl_get_arg(s, "comp_ub_0", &comp_ub);
		glue_vhdl_get_arg(s, "comp_port", &comp_port_array);
		glue_vhdl_get_arg(s, "comp_port_lb_0", &comp_port_lb);
		glue_vhdl_get_arg(s, "comp_port_ub_0", &comp_port_ub);

		glue_vhdl_shorcut_drv_out(
				s_driver,
				comp_array,
				comp_lb,
				comp_ub,
				comp_port_array,
				comp_port_lb,
				comp_port_ub);
	} else {
		/* not implemented */
		faum_log(FAUM_LOG_FATAL, "glue-vhdl", __func__,
			"Trying to call unimplemented foreign procedure "
			"%s.\n", proc);
		assert(0);
	}

	s->num_params = 0;
}


void
glue_vhdl_set(
	struct glue_vhdl *s, 
	unsigned int sig_id, 
	union fauhdli_value data,
	void *drv
)
{
	void *_sig;

	assert(sig_id < GLUE_VHDL_MAX_SIGS);

	_sig = system_sig_get(sig_id);
	assert(_sig != NULL);
	
	switch (s->signals[sig_id].type) {
	case SIG_GEN_BOOLEAN:
		sig_boolean_set(
			(struct sig_boolean *)_sig, 
			drv, 
			(unsigned int)data.univ_int);
		break;

	case SIG_GEN_INTEGER:
		sig_integer_set(
			(struct sig_integer *)_sig,
			drv,
			(int)data.univ_int);
		break;

	case SIG_GEN_BOOLEAN_OR:
	case SIG_GEN_MANAGE:
	case SIG_GEN_MATCH:
	case SIG_GEN_STRING:
	case SIG_GEN_SERIAL:
	case SIG_GEN_VIDEO:
	case SIG_GEN_OPT_RGB:
	case SIG_GEN_USB_BUS:
	case SIG_GEN_ETH:
	case SIG_GEN_MEM_BUS:
	case SIG_GEN_PCI_BUS:
	case SIG_GEN_PARALLEL:
	case SIG_GEN_ISA_BUS:
	case SIG_GEN_IDE_BUS:
	case SIG_GEN_HOST_BUS:
	case SIG_GEN_AGP_BUS:
	case SIG_GEN_PS2:
	case SIG_GEN_PS2_MAIN:
	case SIG_GEN_SCSI_BUS:
	case SIG_GEN_SHUGART_BUS:
	case SIG_GEN_SOUND:
	case SIG_GEN_STD_LOGIC:
	case SIG_GEN_VGA:
	case SIG_GEN_POWER_BOARD:
	case SIG_GEN_POWER_BOARD_AT:
	case SIG_GEN_POWER_DEVICE:
	case SIG_GEN_DIO48:
	case SIG_GEN_DIO24:
	case SIG_GEN_MAGNETO_OPTICAL:
		/* TODO */
		assert(0);
		break;
	}
}

void
glue_vhdl_connect_out(
	struct glue_vhdl *s, 
	unsigned int sig_id,
	union fauhdli_value init,
	void *drv
)
{
	void *_sig;
	assert(sig_id < GLUE_VHDL_MAX_SIGS);
	_sig = system_sig_get(sig_id);

	switch (s->signals[sig_id].type) {
	case SIG_GEN_BOOLEAN:
		sig_boolean_connect_out((struct sig_boolean *)_sig,
					drv,
					(unsigned int)init.univ_int);
		break;

	case SIG_GEN_INTEGER:
		sig_integer_connect_out((struct sig_integer *)_sig,
					drv,
					(int)init.univ_int);
		break;

	case SIG_GEN_STRING:
		/* don't need to connect_out to string */
		break;

	case SIG_GEN_BOOLEAN_OR:
	case SIG_GEN_MANAGE:
	case SIG_GEN_MATCH:
	case SIG_GEN_SERIAL:
	case SIG_GEN_VIDEO:
	case SIG_GEN_OPT_RGB:
	case SIG_GEN_USB_BUS:
	case SIG_GEN_ETH:
	case SIG_GEN_MEM_BUS:
	case SIG_GEN_PCI_BUS:
	case SIG_GEN_PARALLEL:
	case SIG_GEN_ISA_BUS:
	case SIG_GEN_IDE_BUS:
	case SIG_GEN_HOST_BUS:
	case SIG_GEN_AGP_BUS:
	case SIG_GEN_PS2:
	case SIG_GEN_PS2_MAIN:
	case SIG_GEN_SCSI_BUS:
	case SIG_GEN_SHUGART_BUS:
	case SIG_GEN_SOUND:
	case SIG_GEN_STD_LOGIC:
	case SIG_GEN_VGA:
	case SIG_GEN_POWER_BOARD:
	case SIG_GEN_POWER_BOARD_AT:
	case SIG_GEN_POWER_DEVICE:
	case SIG_GEN_DIO48:
	case SIG_GEN_DIO24:
	case SIG_GEN_MAGNETO_OPTICAL:

		assert(0);
		break;
	}
}

void
glue_vhdl_connect_in(
	struct glue_vhdl *s,
	unsigned int sig_id,
	void (*drv_update)(void *_drv, union fauhdli_value value),
	void *drv
)
{
	void *_sig;

	assert(sig_id < GLUE_VHDL_MAX_SIGS);

	_sig = system_sig_get(sig_id);
	assert(_sig != NULL);

	s->signals[sig_id]._drv = drv;
	s->signals[sig_id].drv_update = drv_update;
	
	switch (s->signals[sig_id].type) {
	case SIG_GEN_BOOLEAN: {
		static const struct sig_boolean_funcs f = {
			.set = glue_vhdl_data_get_boolean
		};

		sig_boolean_connect_in(
			(struct sig_boolean *)_sig,
			&s->signals[sig_id],
			&f);
		break;
	    }
	case SIG_GEN_INTEGER: {
		static const struct sig_integer_funcs f = {
			.set = glue_vhdl_data_get_integer
		};

		sig_integer_connect_in(
			(struct sig_integer *)_sig,
			&s->signals[sig_id],
			&f);
		break;
	    }
	case SIG_GEN_BOOLEAN_OR:
	case SIG_GEN_MANAGE:
	case SIG_GEN_MATCH:
	case SIG_GEN_STRING:
	case SIG_GEN_SERIAL:
	case SIG_GEN_VIDEO:
	case SIG_GEN_OPT_RGB:
	case SIG_GEN_USB_BUS:
	case SIG_GEN_ETH:
	case SIG_GEN_MEM_BUS:
	case SIG_GEN_PCI_BUS:
	case SIG_GEN_PARALLEL:
	case SIG_GEN_ISA_BUS:
	case SIG_GEN_IDE_BUS:
	case SIG_GEN_HOST_BUS:
	case SIG_GEN_AGP_BUS:
	case SIG_GEN_PS2:
	case SIG_GEN_PS2_MAIN:
	case SIG_GEN_SCSI_BUS:
	case SIG_GEN_SHUGART_BUS:
	case SIG_GEN_SOUND:
	case SIG_GEN_STD_LOGIC:
	case SIG_GEN_VGA:
	case SIG_GEN_POWER_BOARD:
	case SIG_GEN_POWER_BOARD_AT:
	case SIG_GEN_POWER_DEVICE:
	case SIG_GEN_DIO48:
	case SIG_GEN_DIO24:
	case SIG_GEN_MAGNETO_OPTICAL:
		/* TODO */
		assert(0);
		break;
	}
}

unsigned int
glue_vhdl_create_signal(
	struct glue_vhdl *s, 
	const char *type,
	const char *name
)
{
	unsigned int sig_id;
	enum sig_gen_type t = glue_vhdl_determine_type(type);
	/* FIXME need to special case cstring, due to conflicting type
	 *       string */
	switch (t) {
	case SIG_GEN_STRING:
		sig_id = system_sig_create("string", name);
		break;

	default:
		sig_id = system_sig_create(type, name);
	}
	assert(sig_id < GLUE_VHDL_MAX_SIGS);

	s->signals[sig_id].type = t;
	s->signals[sig_id].self = s;
	s->signals[sig_id]._drv = NULL;
	s->signals[sig_id].drv_update = NULL;

	/* signal id's must be sequential */
	assert(s->inst_sigs == sig_id);
	s->inst_sigs++;

	return sig_id;
}

void
glue_vhdl_enable_event_wakeups(
	struct glue_vhdl *s, 
	void *_instance, 
	void (*cb)(void *_inst)
)
{
	s->interpreter = _instance;
	s->wakeup = cb;
}

unsigned int
glue_vhdl_comp_create(
	struct glue_vhdl *s, 
	const char *type, 
	const char *name,
	unsigned int stack_depth
)
{
	/* need secrect "manage" signal */
	unsigned int sig_id = system_sig_create("manage", name);
	unsigned int comp_id;
	unsigned int node_id;
	unsigned int page_id;

	page_id = glue_vhdl_get_page(s, name);
	node_id = glue_vhdl_get_node(s, name);

#if 0
	faum_log(FAUM_LOG_DEBUG, "glue-vhdl", __func__,
		"creating component %s of type %s\n", name, type);
#endif
	comp_id = system_comp_create(
				type, 
				name, 
				node_id,
				page_id);

	/* connect hidden manage signal */
	system_comp_port_connect(comp_id, "manage", sig_id);

	/* manage signal is a glue-vhdl created signal, too! */
	assert(s->inst_sigs == sig_id);
	s->inst_sigs++;

	/* component ids must be sequential */
	assert(s->inst_comps == comp_id);
	s->inst_comps++;

	/* handle simulation.setup specifiers for component. */
	glue_vhdl_simsetup_set(comp_id, name);

	return comp_id;
}

void
glue_vhdl_comp_init(struct glue_vhdl *s, unsigned int comp)
{
	system_comp_init(comp);
}

void
glue_vhdl_comp_port_connect(
	struct glue_vhdl *s,
	unsigned int comp,
	const char *port,
	unsigned int sig
)
{
	system_comp_port_connect(comp, port, sig);
}

void
glue_vhdl_comp_generic_nonarray_set(
	struct glue_vhdl *s,
	unsigned int comp,
	const char *generic,
	union fauhdli_value val,
	const char *type
)
{
	char buf[30];
	int ret;

	/* TODO float generics, composites */
	ret = snprintf(buf, sizeof(buf), "%" PRIi64, val.univ_int);
	assert(ret < sizeof(buf));

	system_comp_generic_set(comp, type, generic, buf);
}

void
glue_vhdl_comp_generic_array_set(
	struct glue_vhdl *s,
	unsigned int comp,
	const char *generic,
	const char *element_type,
	union fauhdli_value base,
	universal_integer array_size
)
{
	if (strcmp(element_type, "character") == 0) {
		const char *value;

		assert(0 <= array_size);
		
		value = glue_vhdl_flatten_string(base, array_size);

#if 0 /* debug output ... */
		faum_log(FAUM_LOG_DEBUG, "glue-vhdl", __func__,
			"setting generic %s of comp %d to %s\n",
			generic, comp, value);
#endif
		system_comp_generic_set(comp, "string", generic, value);
	} else {
		/* TODO ... */
		assert(0);
	}
}


struct glue_vhdl *
glue_vhdl_create(void)
{
	struct glue_vhdl *s;

	shm_create("glue_vhdl", 1, sizeof(struct glue_vhdl));
	s = shm_map("glue_vhdl", 1, sizeof(struct glue_vhdl), 0);

	s->interpreter = NULL;
	s->wakeup = NULL;
	s->inst_comps = 0;
	s->inst_sigs = 0;
	s->num_params = 0;
	s->page_cache_count = 0;
	s->node_cache_count = 0;

	return s;
}

void
glue_vhdl_destroy(struct glue_vhdl *s)
{
	unsigned int i;

	/* tear down components */
	for (i = 0; i < s->inst_comps; i++) {
		system_comp_exit(i);
	}
	for (i = 0; i < s->inst_comps; i++) {
		system_comp_destroy(i);
	}
	/* tear down signals */
	for (i = 0; i < s->inst_sigs; i++) {
		system_sig_destroy(i);
	}
	/* tear down pages */
	for (i = 0; i < s->page_cache_count; i++) {
		system_page_destroy(s->page_cache[i].page_id);
	}
	/* tear down nodes */
	for (i = 0; i < s->node_cache_count; i++) {
		system_node_destroy(s->node_cache[i].node_id);
	}


	shm_unmap(s, sizeof(struct glue_vhdl));
	shm_destroy("glue_vhdl", 1);
}
