/*
 * $Id: arch_ide_disk.c,v 1.24 2009-04-25 15:29:44 potyra Exp $ 
 *
 * Copyright (C) 2007-2009 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

/* Debugging Options */
#define DEBUG_CONTROL_FLOW	0
#define DEBUG_INTERRUPT		0
#define DEBUG_OPCODES		0

#include "config.h"
#include "compiler.h"

#include <assert.h>
#include "fixme.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "glue-main.h"
#include "glue-shm.h"
#include "glue-storage.h"
#include "sig_ide.h"

struct cpssp {
	/*
	 * Config
	 */
	unsigned int unit;
	unsigned int serial_number;
	char media_name[128];
	char media_image[128];
	int media_cow;
	int media_create;
	int media_sync;

	unsigned int phys_linear;
	unsigned int phys_cyls;
	unsigned int phys_heads;
	unsigned int phys_secs;

	/*
	 * Ports
	 */
	struct sig_boolean *sig_5V;
	struct sig_boolean *sig_12V;
	struct sig_ide_bus *cable;
	int fi_fd;

	/*
	 * State
	 */
	struct storage media;

	unsigned long cyls;
	unsigned long heads;
	unsigned long secs;

	/* Note: We can read/write up to 256 blocks at most. */
	/*volatile*/ unsigned char buf[256 * 512];

	volatile unsigned int head;
	volatile unsigned int tail;
	volatile unsigned int count;

	unsigned int reset;
	unsigned int newcmd;
	unsigned int blocked;

	unsigned int irq;

	volatile unsigned char error;	/* error reg   (read only)  */
	volatile unsigned char feature;	/* feature reg (write only) */
	volatile unsigned char nsector;	/* sector count reg */
	volatile unsigned char sector;	/* sector reg */
	volatile unsigned char lcyl;	/* low cylinder reg */
	volatile unsigned char hcyl;	/* high cylinder reg */
	volatile unsigned char select;	/* select reg */
	volatile unsigned char status;	/* status reg */
	volatile unsigned char command;	/* command reg */
	unsigned char control;		/* device control reg */

#if DISK_MULTIPLE_SUPPORT
	unsigned char multsect;		/* current multiple sector */
#endif

	enum {
		PIO,
#if CONFIG_SW_DMA_MODE_0_SUPPORTED \
 || CONFIG_SW_DMA_MODE_1_SUPPORTED \
 || CONFIG_SW_DMA_MODE_2_SUPPORTED
		SW_DMA,
#endif
#if CONFIG_MW_DMA_MODE_0_SUPPORTED \
 || CONFIG_MW_DMA_MODE_1_SUPPORTED \
 || CONFIG_MW_DMA_MODE_2_SUPPORTED
		MW_DMA,
#endif
#if CONFIG_ULTRA_DMA_MODE_0_SUPPORTED \
 || CONFIG_ULTRA_DMA_MODE_1_SUPPORTED \
 || CONFIG_ULTRA_DMA_MODE_2_SUPPORTED
		ULTRA_DMA,
#endif
	} transfer_mode;

	unsigned char pio_mode;

#if CONFIG_SW_DMA_MODE_0_SUPPORTED \
 || CONFIG_SW_DMA_MODE_1_SUPPORTED \
 || CONFIG_SW_DMA_MODE_2_SUPPORTED
	unsigned char sw_dma_mode;
#endif
#if CONFIG_MW_DMA_MODE_0_SUPPORTED \
 || CONFIG_MW_DMA_MODE_1_SUPPORTED \
 || CONFIG_MW_DMA_MODE_2_SUPPORTED
	unsigned char mw_dma_mode;
#endif
#if CONFIG_ULTRA_DMA_MODE_0_SUPPORTED \
 || CONFIG_ULTRA_DMA_MODE_1_SUPPORTED \
 || CONFIG_ULTRA_DMA_MODE_2_SUPPORTED
	unsigned char ultra_dma_mode;
#endif

#if DISK_SMART_SUPPORT
	int smart_enabled;
#endif
#if DISK_LOOK_AHEAD_SUPPORT
	int look_ahead_enabled;
#endif
#if DISK_WRITE_CACHE_SUPPORT
	int write_cache_enabled;
#endif

	/*
	 * Faults
	 */
	int faulty_disk;
	unsigned long long faulty_blk[10];

	/*
	 * Processes
	 */
	struct process process;
};


#define UNIT   ((cpssp->select >> 4) & 1)

static void
NAME_(irq_update)(struct cpssp *cpssp)
{
	unsigned int val;

	if (cpssp->irq
	 && ((cpssp->control >> 1) & 1) == 0
	 && ((cpssp->select >> 4) & 1) == cpssp->unit) {
		val = 1;
	} else {
		val = 0;
	}
	if (DEBUG_INTERRUPT
	 && loglevel) {
		fprintf(stderr, "%s: irq=%d\n", __FUNCTION__, val);
	}
	sig_ide_bus_irq(cpssp->cable, cpssp, val);
}

static void
NAME_(gen_irq)(struct cpssp *cpssp)
{
	cpssp->irq = 1;
	NAME_(irq_update)(cpssp);
}

static void
NAME_(ugen_irq)(struct cpssp *cpssp)
{
	cpssp->irq = 0;
	NAME_(irq_update)(cpssp);
}

static void
NAME_(reset)(struct cpssp *cpssp)
{
	cpssp->cyls = cpssp->phys_cyls;
	cpssp->heads = cpssp->phys_heads;
	cpssp->secs = cpssp->phys_secs;

	cpssp->head      = 0;
	cpssp->tail      = 0;
	cpssp->count     = 0;

	NAME_(ugen_irq)(cpssp);

	/*
	 * Diagnostic code
	 * (dev 0 passed, dev 1 passed or not present)
	 * (see Execute Device Diagnostic, page 106)
	 */
	cpssp->error     = 1;

	cpssp->feature   = 0;

	/*
	 * Reset Signature
	 */
	cpssp->nsector   = 1;
	cpssp->sector    = 1;
	cpssp->lcyl      = 0;
	cpssp->hcyl      = 0;
	cpssp->select    = 0;

	cpssp->status    = READY_STAT | SEEK_STAT;
	cpssp->command   = 0;
	cpssp->control   = 0;

#if DISK_MULTIPLE_SUPPORT
	cpssp->multsect  = 0x10;
#endif
#if DISK_SMART_SUPPORT
	cpssp->smart_enabled = 1;
#endif
#if DISK_LOOK_AHEAD_SUPPORT
	cpssp->look_ahead_enabled = 1;
#endif
#if DISK_WRITE_CACHE_SUPPORT
	cpssp->write_cache_enabled = 0;
#endif

	cpssp->transfer_mode = PIO;
	cpssp->pio_mode  = 0;
#if CONFIG_SW_DMA_MODE_0_SUPPORTED \
 || CONFIG_SW_DMA_MODE_1_SUPPORTED \
 || CONFIG_SW_DMA_MODE_2_SUPPORTED
	cpssp->sw_dma_mode = 0;
#endif
#if CONFIG_MW_DMA_MODE_0_SUPPORTED \
 || CONFIG_MW_DMA_MODE_1_SUPPORTED \
 || CONFIG_MW_DMA_MODE_2_SUPPORTED
	cpssp->mw_dma_mode = 0;
#endif
#if CONFIG_ULTRA_DMA_MODE_0_SUPPORTED \
 || CONFIG_ULTRA_DMA_MODE_1_SUPPORTED \
 || CONFIG_ULTRA_DMA_MODE_2_SUPPORTED
	cpssp->ultra_dma_mode = 0;
#endif
}

static void
NAME_(send)(struct cpssp *cpssp, unsigned int size, int dma, int block)
{
	cpssp->head = size;
	cpssp->tail = 0;
	cpssp->count = size;

	cpssp->status |= READY_STAT | DRQ_STAT;
	cpssp->status &= ~BUSY_STAT;
	if (dma) {
		sig_ide_bus_dmarq_set(cpssp->cable, 1);
	}
	while (block
	    && ! cpssp->reset
	    && ! cpssp->newcmd
	    && 0 < cpssp->count) {
		cpssp->blocked = 1;
		sched_sleep();
		cpssp->blocked = 0;
	}
}

static void
NAME_(recv)(struct cpssp *cpssp, unsigned int size, int dma)
{
	cpssp->head = 0;
	cpssp->tail = 0;
	cpssp->count = size;

	cpssp->status |= READY_STAT | DRQ_STAT;
	cpssp->status &= ~BUSY_STAT;
	if (dma) {
		sig_ide_bus_dmarq_set(cpssp->cable, 1);
	}
	while (! cpssp->reset
	    && ! cpssp->newcmd
	    && 0 < cpssp->count) {
		cpssp->blocked = 1;
		sched_sleep();
		cpssp->blocked = 0;
	}
}

static int
NAME_(check_lba)(struct cpssp *cpssp, unsigned long block)
{
	if (cpssp->phys_linear <= block) {
		fprintf(stderr, "Bad LBA block: %lu\n", block);
		fprintf(stderr, "Max LBA block: %lu\n", cpssp->phys_linear);
		return -1;
	}

	return 0;
}

static int
NAME_(read_lba)(
	struct cpssp *cpssp,
	unsigned char *buffer,
	unsigned long blkno,
	unsigned int blocks
)
{
	unsigned int i;

	if (NAME_(check_lba)(cpssp, blkno + blocks - 1)) {
		return -1;
	}

	for (i = 0; ; i++) {
		if (i == sizeof(cpssp->faulty_blk) / sizeof(cpssp->faulty_blk[0])) {
			/* Not a bad sector. */
			break;
		}
		if (blkno == cpssp->faulty_blk[i]) {
			return -1;
		}
	}

	return storage_read_write(IO_READ, &cpssp->media,
			buffer, blkno * 512ULL, blocks * 512);
}


static int
NAME_(read_chs)(
	struct cpssp *cpssp,
	unsigned char *buffer,
	unsigned short cylp,
	unsigned char headp,
	unsigned char secp,
	unsigned int blocks
)
{
	unsigned long blkno;

	blkno = (cylp * cpssp->heads + headp) * cpssp->secs + secp - 1;

	return NAME_(read_lba)(cpssp, buffer, blkno, blocks);
}

static int
NAME_(write_lba)(
	struct cpssp *cpssp,
	unsigned char *buffer,
	unsigned long blkno,
	unsigned int blocks
)
{
	unsigned int i;

	if (NAME_(check_lba)(cpssp, blkno + blocks - 1)) {
		return -1;
	}

	for (i = 0; ; i++) {
		if (i == sizeof(cpssp->faulty_blk) / sizeof(cpssp->faulty_blk[0])) {
			/* Not a bad sector. */
			break;
		}
		if (blkno == cpssp->faulty_blk[i]) {
			/* It's a bad sector. */
			return -1;
		}
	}

	return storage_read_write(IO_WRITE, &cpssp->media,
			buffer, blkno * 512ULL, blocks * 512);
}

static int
NAME_(write_chs)(
	struct cpssp *cpssp,
	unsigned char *buffer,
	unsigned short cylp,
	unsigned char headp,
	unsigned char secp,
	unsigned int blocks
)
{
	unsigned long blkno;

	blkno = (cylp * cpssp->heads + headp) * cpssp->secs + secp - 1;

	return NAME_(write_lba)(cpssp, buffer, blkno, blocks);
}

static void
umide_not_implemented(struct cpssp *cpssp)
{
	cpssp->error = 1 << 2; /* Abort */
	cpssp->status |= ERR_STAT;
}

#if DISK_NOP_SUPPORT
/*
 * Opcode: 0x00
 *
 * ATA1: Optional
 * ATA2: Optional
 * ATA3: Optional
 * ATA4: Optional
 */
static void
NAME_(nop)(struct cpssp *cpssp)
{
	cpssp->error = 1 << 2; /* Abort */
	cpssp->status |= ERR_STAT;

	NAME_(gen_irq)(cpssp);
}
#endif /* DISK_NOP_SUPPORT */

#if DISK_RECALIBRATE_SUPPORT
/*
 * Opcode: 0x10 ... 0x1f
 *
 * ATA1: Mandatory
 * ATA2: Optional
 * ATA3: Optional
 */
static void
NAME_(recalibrate)(struct cpssp *cpssp)
{
	NAME_(gen_irq)(cpssp);
}
#endif /* DISK_RECALIBRATE_SUPPORT */

/*
 * Opcode: 0x20
 *
 * ATA1: Mandatory
 * ATA2: Mandatory
 * ATA3: Mandatory
 * ATA4: Mandatory
 *
 * Opcode: 0x21
 *
 * ATA1: Mandatory
 * ATA2: Mandatory
 * ATA3: Mandatory
 * ATA4: Mandatory
 */
static void
NAME_(read)(struct cpssp *cpssp)
{
	int lba;
	uint16_t cyl;
	uint8_t head;
	uint8_t sector;
	unsigned int blocks;

	lba = (cpssp->select >> 6) & 1;
	cyl = (uint16_t) cpssp->lcyl + (uint16_t) cpssp->hcyl * 256;
	head = (uint8_t) cpssp->select & 0x0f;
	sector = (uint8_t) cpssp->sector;

	do {
		/*
		 * How many blocks should we read at once?
		 */
		if (cpssp->nsector == 0) {
			blocks = 256;
		} else {
			blocks = cpssp->nsector;
		}
		switch (cpssp->command) {
		case WIN_READ:
		case WIN_READ + 1:
			blocks = 1;
			break;
		case WIN_READDMA:
		case WIN_READDMA + 1:
			break;
#if DISK_MULTIPLE_SUPPORT
		case WIN_MULTREAD:
			if (cpssp->multsect == 0) {
				blocks = 1;
			} else if (cpssp->multsect < blocks) {
				blocks = cpssp->multsect;
			}
			break;
#endif
		default:
			assert(0); /* Mustn't happen. */
		}

		/*
		 * Read blocks.
		 */
		if (lba) {
			/* LBA mode */
			uint32_t pos;

			pos = (head << 24) | (cyl << 8) | (sector << 0);

			NAME_(read_lba)(cpssp,
					cpssp->buf, pos, blocks);

			pos += blocks;
			head = (pos >> 24) & 0x0f;
			cyl = (pos >> 8) & 0xffff;
			sector = (pos >> 0) & 0xff;

		} else {
			/* C/H/S mode */
			unsigned int i;

			NAME_(read_chs)(cpssp,
					cpssp->buf, cyl, head, sector,
					blocks);

			for (i = 0; i < blocks; i++) {
				sector++;
				if (cpssp->secs < sector) {
					sector = 1;
					head += 1;
					if (cpssp->heads <= head) {
						head = 0;
						cyl += 1;
					}
				}
			}
		}

#if 0
		if (ret != 512) {
			/* UNC = Uncorrectable data error - Schmidt page 59/60 */
			cpssp->error = 1 << 6;
			cpssp->status |= ERR_STAT;
		}
#endif

		/*
		 * Update C/H/S info.
		 */
		cpssp->nsector -= blocks;
		cpssp->hcyl = (cyl >> 8) & 0xff;
		cpssp->lcyl = (cyl >> 0) & 0xff;
		cpssp->select = (cpssp->select & 0xf0) | head;
		cpssp->sector = sector;

		/*
		 * Send blocks to host.
		 */
		if (cpssp->command != WIN_READDMA
		 && cpssp->command != WIN_READDMA + 1) {
			NAME_(gen_irq)(cpssp);
		}
		NAME_(send)(cpssp, blocks * 512,
				cpssp->command == WIN_READDMA
				|| cpssp->command == WIN_READDMA + 1,
				cpssp->nsector != 0
				|| cpssp->command == WIN_READDMA
				|| cpssp->command == WIN_READDMA + 1);
	} while (! cpssp->reset
	      && ! cpssp->newcmd
	      && cpssp->nsector);

	if (cpssp->command == WIN_READDMA
	 || cpssp->command == WIN_READDMA + 1) {
		NAME_(gen_irq)(cpssp);
	}
}

#if DISK_READWRITE_LONG_SUPPORT
/*
 * Opcode: 0x22
 *
 * ATA1: Mandatory
 * ATA2: Optional
 * ATA3: Optional
 *
 * Opcode: 0x23
 *
 * ATA1: Mandatory
 * ATA2: Optional
 * ATA3: Optional
 */
static void
NAME_(read_long)(struct cpssp *cpssp)
{
	fixme();
}
#endif /* DISK_READWRITE_LONG_SUPPORT */

/*
 * Opcode: 0x30
 *
 * ATA1: Mandatory
 * ATA2: Mandatory
 * ATA3: Mandatory
 * ATA4: Mandatory
 *
 * Opcode: 0x31
 *
 * ATA1: Mandatory
 * ATA2: Mandatory
 * ATA3: Mandatory
 * ATA4: Mandatory
 */
static void
NAME_(write)(struct cpssp *cpssp)
{
	int lba;
	uint16_t cyl;
	uint8_t head;
	uint8_t sector;
	unsigned int blocks;

	lba = (cpssp->select >> 6) & 1;
	cyl = (uint16_t) cpssp->lcyl + (uint16_t) cpssp->hcyl * 256;
	head = (uint8_t) cpssp->select & 0x0f;
	sector = (uint8_t) cpssp->sector;

	do {
		/*
		 * How many blocks should we write at once?
		 */
		if (cpssp->nsector == 0) {
			blocks = 256;
		} else {
			blocks = cpssp->nsector;
		}
		switch (cpssp->command) {
		case WIN_WRITE:
		case WIN_WRITE + 1:
			blocks = 1;
			break;
		case WIN_WRITEDMA:
		case WIN_WRITEDMA + 1:
			break;
#if DISK_MULTIPLE_SUPPORT
		case WIN_MULTWRITE:
			if (cpssp->multsect == 0) {
				blocks = 1;
			} else if (cpssp->multsect < blocks) {
				blocks = cpssp->multsect;
			}
			break;
#endif
		default:
			assert(0); /* Mustn't happen. */
		}

		/*
		 * Receive blocks from host.
		 */
		NAME_(recv)(cpssp, blocks * 512,
				cpssp->command == WIN_WRITEDMA
				|| cpssp->command == WIN_WRITEDMA + 1);
		if (cpssp->reset
		 || cpssp->newcmd) {
			break;
		}

		/*
		 * Write blocks.
		 */
		if (lba) {
			/* LBA */
			uint32_t pos;

			pos = (head << 24) | (cyl << 8) | (sector << 0);

			NAME_(write_lba)(cpssp,
					cpssp->buf, pos, blocks);

			pos += blocks;
			head = (pos >> 24) & 0x0f;
			cyl = (pos >> 8) & 0xffff;
			sector = (pos >> 0) & 0xff;

		} else {
			/* C/H/S */
			unsigned int i;

			NAME_(write_chs)(cpssp,
					cpssp->buf, cyl, head, sector,
					blocks);

			for (i = 0; i < blocks; i++) {
				sector += 1;
				if (cpssp->secs < sector) {
					sector = 1;
					head += 1;
					if (cpssp->heads <= head) {
						head = 0;
						cyl += 1;
					}
				}
			}
		}

#if 0
		if (ret != 512) {
			/* UNC = Uncorrectable data error - Schmidt page 59/60 */
			cpssp->error = 1 << 6;
			cpssp->status |= ERR_STAT;
		}
#endif

		/*
		 * Update C/H/S info.
		 */
		cpssp->nsector -= blocks;
		cpssp->hcyl = (cyl >> 8) & 0xff;
		cpssp->lcyl = (cyl >> 0) & 0xff;
		cpssp->select = (cpssp->select & 0xf0) | head;
		cpssp->sector = sector;

		if (cpssp->command != WIN_WRITEDMA
		 && cpssp->command != WIN_WRITEDMA + 1) {
			NAME_(gen_irq)(cpssp);
		}
	} while (cpssp->nsector);

	if (cpssp->command == WIN_WRITEDMA
	 || cpssp->command == WIN_WRITEDMA + 1) {
		NAME_(gen_irq)(cpssp);
	}
}

#if DISK_READWRITE_LONG_SUPPORT
/*
 * Opcode: 0x32
 *
 * ATA1: Mandatory
 * ATA2: Optional
 * ATA3: Optional
 *
 * Opcode: 0x33
 *
 * ATA1: Mandatory
 * ATA2: Optional
 * ATA3: Optional
 */
static void
NAME_(write_long)(struct cpssp *cpssp)
{
	fixme();
}
#endif /* DISK_READWRITE_LONG_SUPPORT */

/*
 * Opcode: 0x40
 *
 * ATA1: Mandatory
 * ATA2: Mandatory
 * ATA3: Mandatory
 * ATA4: Mandatory
 *
 * Opcode: 0x41
 *
 * ATA1: Mandatory
 * ATA2: Mandatory
 * ATA3: Mandatory
 * ATA4: Mandatory
 */
static void
NAME_(read_verify_sectors)(struct cpssp *cpssp)
{
	/* FIXME */
	NAME_(gen_irq)(cpssp);
}

/*
 * Opcode: 0x70
 *
 * ATA1: Mandatory
 * ATA2: Mandatory
 * ATA3: Mandatory
 * ATA4: Mandatory
 */
static void
NAME_(seek)(struct cpssp *cpssp)
{
	/* FIXME */
	NAME_(gen_irq)(cpssp);
}

/*
 * Opcode: 0x90
 *
 * ATA1: Mandatory
 * ATA2: Mandatory
 * ATA3: Mandatory
 * ATA4: Mandatory
 */
static void
NAME_(execute_device_diagnostic)(struct cpssp *cpssp)
{
	cpssp->lcyl = 0x00;
	cpssp->hcyl = 0x00;
	cpssp->nsector = 0x01;
	cpssp->select = 0x00;
	cpssp->sector = 0x01;
	cpssp->error = 0x01; /* Passed */

	NAME_(gen_irq)(cpssp);
}

/*
 * Opcode: 0x91
 *
 * ATA1: Mandatory
 * ATA2: Mandatory
 * ATA3: Mandatory
 * ATA4: Mandatory if C/H/S translation supported
 */
static void
NAME_(initialize_device_parameters)(struct cpssp *cpssp)
{
	unsigned int nsectors;
	unsigned long blocks;

	blocks = cpssp->phys_linear;

	nsectors = (cpssp->nsector == 0) ? 256 : cpssp->nsector;

	cpssp->secs = nsectors;
	blocks /= nsectors;
	cpssp->heads = (cpssp->select & 0x0f) + 1;
	blocks /= (cpssp->select & 0x0f) + 1;
	cpssp->cyls = blocks;

	NAME_(gen_irq)(cpssp);
}

#if DISK_DOWNLOAD_MICROCODE_SUPPORT
/*
 * Opcode: 0x92
 *
 * ATA1: Optional
 * ATA2: Optional
 * ATA3: Optional
 * ATA4: Optional
 */
static void
NAME_(download_microcode)(struct cpssp *cpssp)
{
	fixme();
}
#endif

#if DISK_POWER_MANAGEMENT_SUPPORT
/*
 * Opcode: 0x94 or 0xe0
 *
 * ATA1: Optional
 * ATA2: Optional -- Power Management Feature Set
 * ATA3: Optional -- Power Management Feature Set
 * ATA4: Mandatory
 */
static void
NAME_(standby_immediate)(struct cpssp *cpssp)
{
	/* FIXME */
	NAME_(gen_irq)(cpssp);
}

/*
 * Opcode: 0x95
 *
 * ATA1: Optional
 * ATA2: Optional -- Power Management Feature Set
 * ATA3: Optional -- Power Management Feature Set
 *
 * Opcode: 0xe1
 *
 * ATA1: Optional
 * ATA2: Optional -- Power Management Feature Set
 * ATA3: Optional -- Power Management Feature Set
 * ATA4: Mandatory
 */
static void
NAME_(idle_immediate)(struct cpssp *cpssp)
{
	/* FIXME */
	NAME_(gen_irq)(cpssp);
}

/*
 * Opcode: 0x96 or 0xe2
 *
 * ATA1: Optional
 * ATA2: Optional -- Power Management Feature Set
 * ATA3: Optional -- Power Management Feature Set
 * ATA4: Mandatory
 */
static void
NAME_(standby)(struct cpssp *cpssp)
{
	/* FIXME */
	NAME_(gen_irq)(cpssp);
}

/*
 * Opcode: 0x97
 *
 * ATA1: Optional
 * ATA2: Optional -- Power Management Feature Set
 * ATA3: Optional -- Power Management Feature Set
 *
 * Opcode: 0xe3
 *
 * ATA1: Optional
 * ATA2: Optional -- Power Management Feature Set
 * ATA3: Optional -- Power Management Feature Set
 * ATA4: Mandatory
 */
static void
NAME_(idle)(struct cpssp *cpssp)
{
	/* FIXME */
	NAME_(gen_irq)(cpssp);
}

/*
 * Opcode: 0x98
 *
 * ATA1: Optional
 * ATA2: Optional -- Power Management Feature Set
 * ATA3: Optional -- Power Management Feature Set
 *
 * Opcode: 0xe5
 *
 * ATA1: Optional
 * ATA2: Optional -- Power Management Feature Set
 * ATA3: Optional -- Power Management Feature Set
 * ATA4: Mandatory
 */
static void
NAME_(check_power_mode)(struct cpssp *cpssp)
{
	cpssp->nsector = 0xff; /* Active Mode */

	NAME_(gen_irq)(cpssp);
}

/*
 * Opcode: 0x99 or 0xe6
 *
 * ATA1: Optional
 * ATA2: Optional -- Power Management Feature Set
 * ATA3: Optional -- Power Management Feature Set
 * ATA4: Mandatory
 */
static void
NAME_(sleep)(struct cpssp *cpssp)
{
	/* FIXME */
	NAME_(gen_irq)(cpssp);
}
#endif /* DISK_POWER_MANAGEMENT_SUPPORT */

#if DISK_SMART_SUPPORT

#define PUT8(val)	{ *(uint8_t *) p = val; p += 1; }
#define PUT16(val)	{ *(uint16_t *) p = val; p += 2; }

static void
NAME_(smart_read_attribute_values)(struct cpssp *cpssp)
{
	uint8_t *p;
	unsigned int attr;
	unsigned int sum;
	unsigned int i;

	p = (uint8_t *) cpssp->buf;
	PUT16(0x0004); /* Revision Number */
	attr = 0;

	/* Attribute 0x01: Raw Read Error Rate */
	PUT8(0x01); /* ID */
	PUT16((1 << 3)		/* Error Rate */
		| (1 << 0));	/* Life-critical */
	PUT8(95); /* Value */
	PUT8(94); /* Worst */
	PUT8(0x08); /* ? */
	PUT8(0x00); /* ? */
	PUT8(0x0a); /* ? */
	PUT8(0x00); /* ? */
	PUT8(0x00); /* ? */
	PUT8(0x00); /* ? */
	PUT8(0x00); /* ? */
	attr++;

	/* Attribute 0x02: Throughput Performance */
	PUT8(0x02); /* ID */
	PUT16((1 << 2)		/* Performance-related */
		| (1 << 0));	/* Life-critical */
	PUT8(100); /* Value */
	PUT8(100); /* Worst */
	PUT8(0x00); /* ? */
	PUT8(0x00); /* ? */
	PUT8(0x00); /* ? */
	PUT8(0x00); /* ? */
	PUT8(0x00); /* ? */
	PUT8(0x00); /* ? */
	PUT8(0x00); /* ? */
	attr++;

	/* Attribute 0x03: Spin Up Time */
	PUT8(0x03); /* ID */
	PUT16((1 << 2)		/* Performance-related */
		| (1 << 0));	/* Life-critical */
	PUT8(131); /* Value */
	PUT8(100); /* Worst */
	PUT8(0xfc); /* ? */
	PUT8(0x01); /* ? */
	PUT8(0xe8); /* ? */
	PUT8(0x01); /* ? */
	PUT8(0x0a); /* ? */
	PUT8(0x00); /* ? */
	PUT8(0x00); /* ? */
	attr++;

	/* Attribute 0x04: Start/Stop Count */
	PUT8(0x04); /* ID */
	PUT16((1 << 4));	/* Events count */
	PUT8(100); /* Value */
	PUT8(100); /* Worst */
	PUT8(0xfd); /* ? */
	PUT8(0x01); /* ? */
	PUT8(0x00); /* ? */
	PUT8(0x00); /* ? */
	PUT8(0x00); /* ? */
	PUT8(0x00); /* ? */
	PUT8(0x00); /* ? */
	attr++;

	/* Attribute 0x05: Reallocated Sector Count */
	PUT8(0x05); /* ID */
	PUT16((1 << 5)		/* Self-preserve */
		| (1 << 4)	/* Events count */
		| (1 << 0));	/* Life-critical */
	PUT8(100); /* Value */
	PUT8(100); /* Worst */
	PUT8(0x00); /* ? */
	PUT8(0x00); /* ? */
	PUT8(0x00); /* ? */
	PUT8(0x00); /* ? */
	PUT8(0x00); /* ? */
	PUT8(0x00); /* ? */
	PUT8(0x00); /* ? */
	attr++;

	/* Attribute 0x07: Seek Error Rate */
	PUT8(0x07); /* ID */
	PUT16((1 << 3)		/* Error rate */
		| (1 << 0));	/* Life-critical */
	PUT8(100); /* Value */
	PUT8(100); /* Worst */
	PUT8(0x00); /* ? */
	PUT8(0x00); /* ? */
	PUT8(0x00); /* ? */
	PUT8(0x00); /* ? */
	PUT8(0x00); /* ? */
	PUT8(0x00); /* ? */
	PUT8(0x00); /* ? */
	attr++;

	/* Attribute 0x08: Seek Time Performance */
	PUT8(0x08); /* ID */
	PUT16((1 << 2)		/* Performance-related */
		| (1 << 0));	/* Life-critical */
	PUT8(100); /* Value */
	PUT8(100); /* Worst */
	PUT8(0x00); /* ? */
	PUT8(0x00); /* ? */
	PUT8(0x00); /* ? */
	PUT8(0x00); /* ? */
	PUT8(0x00); /* ? */
	PUT8(0x00); /* ? */
	PUT8(0x00); /* ? */
	attr++;

	/* Attribute 0x09: Power On Hours Count */
	PUT8(0x09); /* ID */
	PUT16((1 << 4));	/* Events count */
	PUT8(97); /* Value */
	PUT8(97); /* Worst */
	PUT8(0xf5); /* ? */
	PUT8(0x53); /* ? */
	PUT8(0x00); /* ? */
	PUT8(0x00); /* ? */
	PUT8(0x00); /* ? */
	PUT8(0x00); /* ? */
	PUT8(0x00); /* ? */
	attr++;

	/* Attribute 10: Spin Retry Count */
	PUT8(0x0a); /* ID */
	PUT16((1 << 4)		/* Events count */
		| (1 << 0));	/* Life-critical */
	PUT8(100); /* Value */
	PUT8(100); /* Worst */
	PUT8(0x00); /* ? */
	PUT8(0x00); /* ? */
	PUT8(0x00); /* ? */
	PUT8(0x00); /* ? */
	PUT8(0x00); /* ? */
	PUT8(0x00); /* ? */
	PUT8(0x00); /* ? */
	attr++;

	/* Attribute 12: Drive Power Cycle Count */
	PUT8(0x0c); /* ID */
	PUT16((1 << 5)		/* Self-preserve */
		| (1 << 4));	/* Events count */
	PUT8(100); /* Value */
	PUT8(100); /* Worst */
	PUT8(0x89); /* ? */
	PUT8(0x00); /* ? */
	PUT8(0x00); /* ? */
	PUT8(0x00); /* ? */
	PUT8(0x00); /* ? */
	PUT8(0x00); /* ? */
	PUT8(0x00); /* ? */
	attr++;

	for ( ; attr < 30; attr++) {
		PUT16(0);
		PUT16(0);
		PUT16(0);
		PUT16(0);
		PUT16(0);
		PUT16(0);
	}

#if ! ATA4_SUPPORT
	/* 362-367: Reserved */
	PUT16(0);
	PUT16(0);
	PUT16(0);
#else
	/* 362: Off-line Data Collection Status */
	PUT8(2); /* Data collection completed without error. */

	/* 363: Vendor Specific */
	PUT8(0);

	/* 364-365: Total Time in Seconds to Complete Off-line Test */
	PUT16(10);

	/* 366: Vendor Specific */
	PUT8(0);

	/* 367: Off-line Data Collection Capabilities */
	PUT8((1 << 2) /* Off-line Test aborted by other command. */
		| (1 << 0)); /* Off-line Test instruction implemented.*/
#endif

	/* 368-369: SMART Capabilities */
	PUT16((1 << 1) /* AUTOSAVE supported */
		| (1 << 0)); /* Pre-Power saving supported */

	/* 370-385: Reserved */
	for (i = 0; i < 16 / 2; i++) {
		PUT16(0);
	}

	/* 386-511: Vendor Specific */
	for (i = 0; i < 126 / 2; i++) {
		PUT16(0);
	}
	assert(p == (uint8_t *) &cpssp->buf[512]);

	sum = 0;
	p = (uint8_t *) cpssp->buf;
	for (i = 0; i < 511; i++) {
		sum += *p++;
	}
	*p++ = 0 - sum;
	assert(p == (uint8_t *) &cpssp->buf[512]);

	NAME_(gen_irq)(cpssp);
	NAME_(send)(cpssp, 512, 0, 0);
}

static void
NAME_(smart_read_attribute_thresholds)(struct cpssp *cpssp)
{
	uint8_t *p;
	unsigned int attr;
	unsigned int sum;
	unsigned int i;

	p = (uint8_t *) cpssp->buf;
	PUT16(0x0004); /* Revision Number */
	attr = 0;

	/* Attribute 0x01: Raw Read Error Rate */
	PUT8(0x01); /* ID */
	PUT8(60); /* Threshold */
	PUT16(0);
	PUT16(0);
	PUT16(0);
	PUT16(0);
	PUT16(0);
	attr++;

	/* Attribute 0x02: Throughput */
	PUT8(0x02); /* ID */
	PUT8(50); /* Threshold */
	PUT16(0);
	PUT16(0);
	PUT16(0);
	PUT16(0);
	PUT16(0);
	attr++;

	/* Attribute 0x03: Spin Up Time */
	PUT8(0x03); /* ID */
	PUT8(24); /* Threshold */
	PUT16(0);
	PUT16(0);
	PUT16(0);
	PUT16(0);
	PUT16(0);
	attr++;

	/* Attribute 0x04: Start/Stop Count */
	PUT8(0x04); /* ID */
	PUT8(0); /* Threshold */
	PUT16(0);
	PUT16(0);
	PUT16(0);
	PUT16(0);
	PUT16(0);
	attr++;

	/* Attribute 0x05: Reallocated Sector Count */
	PUT8(0x05); /* ID */
	PUT8(5); /* Threshold */
	PUT16(0);
	PUT16(0);
	PUT16(0);
	PUT16(0);
	PUT16(0);
	attr++;

	/* Attribute 0x07: Seek Error Rate */
	PUT8(0x07); /* ID */
	PUT8(67); /* Threshold */
	PUT16(0);
	PUT16(0);
	PUT16(0);
	PUT16(0);
	PUT16(0);
	attr++;

	/* Attribute 0x08: Seek Time Performance */
	PUT8(0x08); /* ID */
	PUT8(20); /* Threshold */
	PUT16(0);
	PUT16(0);
	PUT16(0);
	PUT16(0);
	PUT16(0);
	attr++;

	/* Attribute 0x09: Power On Hours Count */
	PUT8(0x09); /* ID */
	PUT8(0); /* Threshold */
	PUT16(0);
	PUT16(0);
	PUT16(0);
	PUT16(0);
	PUT16(0);
	attr++;

	/* Attribute 10: Spin Retry Count */
	PUT8(0x0a); /* ID */
	PUT8(60); /* Threshold */
	PUT16(0);
	PUT16(0);
	PUT16(0);
	PUT16(0);
	PUT16(0);
	attr++;

	/* Attribute 12: Drive Power Cycle Count */
	PUT8(0x0c); /* ID */
	PUT8(0); /* Threshold */
	PUT16(0);
	PUT16(0);
	PUT16(0);
	PUT16(0);
	PUT16(0);
	attr++;

	for ( ; attr < 30; attr++) {
		PUT16(0);
		PUT16(0);
		PUT16(0);
		PUT16(0);
		PUT16(0);
		PUT16(0);
	}
	for (i = 0; i < 18 / 2; i++) {
		PUT16(0); /* Reserved */
	}
	for (i = 0; i < 132 / 2; i++) {
		PUT16(0); /* Vendor specific */
	}
	assert(p == (uint8_t *) &cpssp->buf[512]);

	sum = 0;
	p = (uint8_t *) cpssp->buf;
	for (i = 0; i < 511; i++) {
		sum += *p++;
	}
	*p++ = 0 - sum;
	assert(p == (uint8_t *) &cpssp->buf[512]);

	NAME_(gen_irq)(cpssp);
	NAME_(send)(cpssp, 512, 0, 0);
}

/*
 * Opcode: 0xb0
 *
 * ATA1: Reserved
 * ATA2: Optional
 * ATA3: Optional -- SMART Feature Set
 * ATA4: Optional -- SMART Feature Set
 */
static void
NAME_(smart)(struct cpssp *cpssp)
{
	if (cpssp->lcyl != 0x4f
	 || cpssp->hcyl != 0xc2) {
		/* Unknown SMART command. */
		goto unknown;

	} else switch (cpssp->feature) {
	case 0xd0:
		/*
		 * Read attribute values.
		 *
		 * ATA3: Optional and not recommended.
		 * ATA4: Optional
		 */
		if (! cpssp->smart_enabled) goto disabled;

		NAME_(smart_read_attribute_values)(cpssp);
		break;

	case 0xd1:
		/*
		 * Read attribute thresholds.
		 * Optional and not recommended.
		 */
		if (! cpssp->smart_enabled) goto disabled;

		NAME_(smart_read_attribute_thresholds)(cpssp);
		break;

	case 0xd2:
		/*
		 * Enable/disable attribute autosave.
		 *
		 * ATA4: Mandatory
		 */
		if (! cpssp->smart_enabled) goto disabled;

		switch (cpssp->nsector) {
		case 0x00:
			/* Disable attribute autosave. */
			/* FIXME */
			break;
		case 0xf1:
			/* Enable attribute autosave. */
			/* FIXME */
			break;
		default:
			goto unknown;
		}

		NAME_(gen_irq)(cpssp);
		break;

	case 0xd3:
		/*
		 * Save attribute values.
		 *
		 * ATA3: Optional and not recommended.
		 * ATA4: Optional and not recommended.
		 */
		if (! cpssp->smart_enabled) goto disabled;

		/* FIXME */
		break;

	case 0xd4:
		/*
		 * SMART execute off-line immediately.
		 *
		 * ATA4: Optional
		 */
		if (! cpssp->smart_enabled) goto disabled;

		/* FIXME */
		NAME_(gen_irq)(cpssp);
		break;

	case 0xd8:
		/*
		 * SMART enable operations.
		 *
		 * ATA4: Mandatory
		 */
		cpssp->smart_enabled = 1;

		NAME_(gen_irq)(cpssp);
		break;

	case 0xd9:
		/*
		 * SMART disable operations.
		 *
		 * ATA4: Mandatory
		 */
		if (! cpssp->smart_enabled) goto disabled;

		cpssp->smart_enabled = 0;
		NAME_(gen_irq)(cpssp);
		break;

	case 0xda:
		/*
		 * Return status.
		 *
		 * ATA4: Mandatory
		 */
		if (! cpssp->smart_enabled) goto disabled;

		/* FIXME */
		NAME_(gen_irq)(cpssp);
		break;

	default:
	unknown:;
	disabled:;
		cpssp->error = 1 << 2; /* Abort */
		cpssp->status |= ERR_STAT;

		NAME_(gen_irq)(cpssp);
		break;
	}
}
#endif /* DISK_SMART_SUPPORT */

#if DISK_MULTIPLE_SUPPORT
/*
 * Opcode: 0xc4
 *
 * ATA1: Optional
 * ATA2: Optional
 * ATA3: Mandatory
 * ATA4: Mandatory
 */
static void
NAME_(read_multiple)(struct cpssp *cpssp)
{
	NAME_(read)(cpssp);
}

/*
 * Opcode: 0xc5
 *
 * ATA1: Optional
 * ATA2: Optional
 * ATA3: Mandatory
 * ATA4: Mandatory
 */
static void
NAME_(write_multiple)(struct cpssp *cpssp)
{
	NAME_(write)(cpssp);
}

/*
 * Opcode: 0xc6
 *
 * ATA1: Optional
 * ATA2: Optional
 * ATA3: Mandatory
 * ATA4: Mandatory
 */
static void
NAME_(set_multiple_mode)(struct cpssp *cpssp)
{
	if (cpssp->nsector <= MAX_IDE_MULTIPLE_SECTOR) {
		cpssp->multsect = cpssp->nsector;

	} else {
		/* not supported blocksize => disable */
		cpssp->multsect = 0;
		cpssp->error = 1 << 6;
		cpssp->status |= ERR_STAT;
	}

	NAME_(gen_irq)(cpssp);
}
#endif /* DISK_MULTIPLE_SUPPORT */

#if DISK_DMA_SUPPORT
/*
 * Opcode: 0xc8
 *
 * ATA1: Optional
 * ATA2: Optional
 * ATA3: Mandatory
 * ATA4: Mandatory
 *
 * Opcode: 0xc9
 *
 * ATA1: Optional
 * ATA2: Optional
 * ATA3: Mandatory
 * ATA4: Mandatory
 */
static void
NAME_(read_dma)(struct cpssp *cpssp)
{
	NAME_(read)(cpssp);
}
#endif /* DISK_DMA_SUPPORT */

#if DISK_DMA_SUPPORT
/*
 * Opcode: 0xca
 *
 * ATA1: Optional
 * ATA2: Optional
 * ATA3: Mandatory
 * ATA4: Mandatory
 *
 * Opcode: 0xcb
 *
 * ATA1: Optional
 * ATA2: Optional
 * ATA3: Mandatory
 * ATA4: Mandatory
 */
static void
NAME_(write_dma)(struct cpssp *cpssp)
{
	NAME_(write)(cpssp);
}
#endif /* DISK_DMA_SUPPORT */

#if DISK_READ_BUFFER_SUPPORT
/*
 * Opcode: 0xe4
 *
 * ATA1: Optional
 * ATA2: Optional
 * ATA3: Optional
 * ATA4: Optional
 */
static void
NAME_(read_buffer)(struct cpssp *cpssp)
{
	/* Just send bytes from buffer. */
	NAME_(gen_irq)(cpssp);
	NAME_(send)(cpssp, 512, 0, 0);
}
#endif /* DISK_READ_BUFFER_SUPPORT */

#if DISK_WRITE_BUFFER_SUPPORT
/*
 * Opcode: 0xe8
 *
 * ATA1: Optional
 * ATA2: Optional
 * ATA3: Optional
 * ATA4: Optional
 */
static void
NAME_(write_buffer)(struct cpssp *cpssp)
{
	/* Just receive bytes to buffer. */
	NAME_(recv)(cpssp, 512, 0);
	NAME_(gen_irq)(cpssp);
}
#endif /* DISK_WRITE_BUFFER_SUPPORT */

/*
 * Opcode: 0xec
 *
 * ATA1: Optional
 * ATA2: Mandatory
 * ATA3: Mandatory
 * ATA4: Mandatory
 */
static void
NAME_(identify_device)(struct cpssp *cpssp)
{
	uint16_t *p = (uint16_t *) cpssp->buf;

	memset(p, 0, 512);

	/* 0: General configuration: */
	p[0] |= 0 << 15; /* ATA Device */
#if ATA1_SUPPORT
	p[0] |= 0 << 14; /* 1=Format speed tolerance gap required */
	p[0] |= 0 << 13; /* 1=Track offset option available */
	p[0] |= 0 << 12; /* 1=Data strobe offset option available */
	p[0] |= 1 << 11; /* 1=Rotational speed tolerance is > 0.5% */
	p[0] |= 1 << 10; /* 1=Disk transfer rate > 10Mbs */
	p[0] |= 0 << 9; /* 1=Disk transfer rate > 5Mbs but <= 10Mbs */
	p[0] |= 0 << 8; /* 1=Disk transfer rate <= 5Mbs */
#else
	p[0] |= 0 << 14; /* Obsolete */
	/* ... */
	p[0] |= 0 << 8; /* Obsolete */
#endif
	p[0] |= 0 << 7; /* Not removable */
	p[0] |= 1 << 6; /* Not removable controller and/or device */
#if ATA1_SUPPORT
	p[0] |= 0 << 5; /* 1=Spindle motor control option implemented */
	p[0] |= 1 << 4; /* 1=Head switch time > 15usec */
	p[0] |= 1 << 3; /* 1=Not MFM encoded */
	p[0] |= 0 << 2; /* 1=Soft sectored */
	p[0] |= 1 << 1; /* 1=Hard sectored */
#else
	p[0] |= 0 << 5; /* Obsolete */
	/* ... */
	p[0] |= 0 << 1; /* Obsolete */
#endif
	p[0] |= 0 << 0; /* Reserved */

	/* 1: Number of logical cylinders */
	p[1] = cpssp->phys_cyls;

	/* 2: Reserved */

	/* 3: Number of logical heads */
	p[3] = cpssp->phys_heads;

#if ATA1_SUPPORT
	/* 4: Number of unformatted bytes per track */
	p[4] = 0x8d90; /* reported by Seagate ST3660A - FIXME */

	/* 5: Number of unformatted bytes per sector */
	p[5] = 0x0248; /* reported by Seagate ST3660A - FIXME */
#else
	/* 4-5: Obsolete */
#endif

	/* 6: Number of logical sectors per logical track */
	p[6] = cpssp->phys_secs;

	/* 7-9: Vendor specific */

	/* 10-19: Serial number */
	{
		char buf[20 + 1];
		unsigned int i;

		sprintf(buf, "%-20s", SERIAL_NUMBER(cpssp->serial_number));
		for (i = 0; i < 20 - 10; i++) {
			p[10 + i] = (buf[i * 2 + 0] << 8)
			          | (buf[i * 2 + 1] << 0);
		}
	}

#if ATA1_SUPPORT
	/* 20: Buffer type */
	p[20] = 3; /* Dual ported multi-sector buffer with read caching */

	/* 21: Buffer size in 512 byte increments */
	p[21] = sizeof(cpssp->buf) / 512;
#else
	/* 20-21: Obsolete */
#endif

#if DISK_READWRITE_LONG_SUPPORT
	/* 22: Number of vendor specific bytes available on READ/WRITE LONG */
	p[22] = READWRITE_LONG_BYTES;
#else
	/* 22: ? - FIXME */
#endif

	/* 23-26: Firmware revision ("1.0")*/
	{
		char buf[8 + 1];
		unsigned int i;

		sprintf(buf, "%-8s", FIRMWARE_REVISION);
		for (i = 0; i < 27 - 23; i++) {
			p[23 + i] = (buf[i * 2 + 0] << 8)
			          | (buf[i * 2 + 1] << 0);
		}
	}

	/* 27-46: Model number */
	{
		char buf[40 + 1];
		unsigned int i;

		sprintf(buf, "%-40s", MODEL_NUMBER);
		for (i = 0; i < 47 - 27; i++) {
			p[27 + i] = (buf[i * 2 + 0] << 8)
			          | (buf[i * 2 + 1] << 0);
		}
	}

	/* 47: */
#if ! ATA4_SUPPORT
	p[47] |= 0x00 << 8; /* Vendor specific */
#else
	p[47] |= 0x80 << 8; /* Vendor specific */
#endif
#if ! DISK_MULTIPLE_SUPPORT
	p[47] |= 0x00 << 0; /* Read/write multiple not supported */
#else
	p[47] |= MAX_IDE_MULTIPLE_SECTOR << 0; /* Maximum number of sectors */
#endif

#if ATA1_SUPPORT
	/* 48: */
	/* FIXME */
#else
	/* 48: Reserved */
#endif

	/* 49: Capabilities */
	p[49] |= 0 << 15; /* Reserved */
	p[49] |= 0 << 14; /* Reserved */
	p[49] |= DISK_POWER_MANAGEMENT_SUPPORT << 13; /* Standby timer */
	p[49] |= 0 << 12; /* Reserved */
	p[49] |= CONFIG_PIO_MODE_3_SUPPORTED << 11; /* IORDY supported */
	p[49] |= 1 << 10; /* IORDY can be disabled - FIXME */
#if ATA1_SUPPORT || ATA2_SUPPORT || ATA4_SUPPORT
	p[49] |= 1 << 9; /* LBA supported */
	p[49] |= DISK_DMA_SUPPORT << 8; /* DMA supported */
#else
	p[49] |= 0 << 9; /* Obsolete */
	p[49] |= 0 << 8; /* Obsolete */
#endif
	p[49] |= 0 << 7; /* Vendor specific */
	/* ... */
	p[49] |= 0 << 0; /* Vendor specific */

#if ! ATA4_SUPPORT
	/* 50: Reserved */
#else
	/* 50: Capabilities */
	p[50] |= 0 << 15; /* Should be cleared */
	p[50] |= 1 << 14; /* Should be set */
	p[50] |= 0 << 13; /* Reserved */
	/* ... */
	p[50] |= 0 << 1; /* Reserved */
	p[50] |= 0 << 0; /* Device specific standby timer value minimum */
#endif

	/* 51: */
	/* PIO data transfer cycle timing mode */
	if (CONFIG_PIO_MODE_2_SUPPORTED) {
		p[51] |= 0x02 << 8;
	} else if (CONFIG_PIO_MODE_1_SUPPORTED) {
		p[51] |= 0x01 << 8;
	} else {
		p[51] |= 0x00 << 8;
	}
	p[51] |= 0x00 << 0; /* Vendor specific */

#if ATA1_SUPPORT || ATA2_SUPPORT
	/* 52: */
	p[52] |= 0x02 << 8; /* DMA data transfer cycle timing mode */
	p[52] |= 0x00 << 0; /* Vendor specific */
#else
	/* 52: Obsolete/Vendor specific */
#endif

	/* 53: */
	p[53] |= 0 << 15; /* Reserved */
	/* ... */
#if ! ATA4_SUPPORT
	p[53] |= 0 << 2; /* Reserved */
#else
	p[53] |= 1 << 2; /* The fields reported in word 88 are valid */
#endif
#if ! (ATA2_SUPPORT || ATA3_SUPPORT)
	p[53] |= 0 << 1; /* Reserved */
#else
	p[53] |= 1 << 1; /* The fields reported in word 64-70 are valid */
#endif
	p[53] |= 1 << 0; /* The fields reported in word 54-58 are valid */

	/* 54: Number of current logical cylinders */
	p[54] = cpssp->cyls;

	/* 55: Number of current logical heads */
	p[55] = cpssp->heads;

	/* 56: Number of current logical sectors per track */
	p[56] = cpssp->secs;

	/* 57-58: Current capacity in sectors */
	p[57] = (cpssp->secs * cpssp->heads * cpssp->cyls) & 0xFFFF;
	p[58] = (cpssp->secs * cpssp->heads * cpssp->cyls) >> 16;

	/* 59: */
	p[59] |= 0 << 15; /* Reserved */
	/* ... */
	p[59] |= 0 << 9; /* Reserved */
#if ! DISK_MULTIPLE_SUPPORT
	p[59] |= 0 << 8; /* Multiple sector setting is invalid. */
	p[59] |= 0 << 0;
#else
	p[59] |= (cpssp->multsect != 0) << 8; /* Multiple sector setting is valid. */
	p[59] |= cpssp->multsect << 0;
#endif

	/* 60-61: Total number of user addressable sectors (LBA mode only) */
	p[60] = cpssp->phys_linear & 0xFFFF;
	p[61] = cpssp->phys_linear >> 16;

#if ATA1_SUPPORT || ATA2_SUPPORT
	/* 62: */
#if CONFIG_SW_DMA_MODE_0_SUPPORTED \
 || CONFIG_SW_DMA_MODE_1_SUPPORTED \
 || CONFIG_SW_DMA_MODE_2_SUPPORTED
	/* Selected single word DMA mode */
	if (cpssp->transfer_mode == SW_DMA) {
		p[62] |= 1 << (cpssp->sw_dma_mode + 8);
	}
#endif
	p[62] |= 0 << 7; /* Reserved */
	/* ... */
	p[62] |= 0 << 3; /* Reserved */
	p[62] |= CONFIG_SW_DMA_MODE_2_SUPPORTED << 2;
	p[62] |= CONFIG_SW_DMA_MODE_1_SUPPORTED << 1;
	p[62] |= CONFIG_SW_DMA_MODE_0_SUPPORTED << 0;
#else
	/* 62: Obsolete */
#endif

	/* 63: */
#if CONFIG_SW_DMA_MODE_0_SUPPORTED \
 || CONFIG_SW_DMA_MODE_1_SUPPORTED \
 || CONFIG_SW_DMA_MODE_2_SUPPORTED
	/* Selected Multiword DMA mode */
	if (cpssp->transfer_mode == MW_DMA) {
		p[63] |= 1 << (cpssp->mw_dma_mode + 8);
	}
#endif
	p[63] |= 0 << 7; /* Reserved */
	/* ... */
	p[63] |= 0 << 3; /* Reserved */
	p[63] |= CONFIG_MW_DMA_MODE_2_SUPPORTED << 2;
	p[63] |= CONFIG_MW_DMA_MODE_1_SUPPORTED << 1;
	p[63] |= CONFIG_MW_DMA_MODE_0_SUPPORTED << 0;

#if ! (ATA2_SUPPORT || ATA3_SUPPORT || ATA4_SUPPORT)
	/* 64-68: Reserved */

#else /* ATA2_SUPPORT || ATA3_SUPPORT || ATA4_SUPPORT */
	/* 64: PIO modes supported. */
	p[64] |= 0 << 15; /* Reserved */
	/* ... */
	p[64] |= 0 << 8; /* Reserved */
	p[64] |= 0 << 7; /* Reserved */
	/* ... */
	p[64] |= 0 << 2; /* Reserved */
	p[64] |= CONFIG_PIO_MODE_4_SUPPORTED << 1;
	p[64] |= CONFIG_PIO_MODE_3_SUPPORTED << 0;

	/* 65: Minimum Multiword DMA transfer cycle time per word (ns) */
	p[65] = 0x0078;

	/* 66: Manufacturer's recommended Multiword DMA transfer cycle time */
	p[66] = 0x0078;

	/* 67: Minimum PIO transfer cycle time without flow control */
	p[67] = 0x0172;

	/* 68: Minimum PIO transfer cycle time IORDY flow control */
	p[68] = 0x0078;
#endif /* ATA2_SUPPORT || ATA3_SUPPORT || ATA4_SUPPORT */

	/* 69-70: Reserved */
	/* 71-74: Reserved */
#if ! ATA4_SUPPORT
	/* 75: Reserved */
#else
	/* 75: Queue depth */
	p[75] |= 0 << 15; /* Reserved */
	/* ... */
	p[75] |= 0 << 5; /* Reserved */
	p[75] |= 0 << 0; /* Maximum queue depth */
#endif

	/* 76-79: Reserved */

#if ! (ATA3_SUPPORT || ATA4_SUPPORT)
	/* 80-81: Reserved */

#else /* ATA3_SUPPORT || ATA4_SUPPORT */
	/* 80: Major version number */
	p[80] |= 0 << 15; /* Reserved */
	p[80] |= 0 << 14; /* Reserved for ATA/ATAPI-14 */
	/* ... */
	p[80] |= 0 << 8; /* Reserved for ATA/ATAPI-8 */
	p[80] |= ATA7_SUPPORT << 7; /* ATA/ATAPI-7 support */
	p[80] |= ATA6_SUPPORT << 6; /* ATA/ATAPI-6 support */
	p[80] |= ATA5_SUPPORT << 5; /* ATA/ATAPI-5 support */
	p[80] |= ATA4_SUPPORT << 4; /* ATA/ATAPI-4 support */
	p[80] |= ATA3_SUPPORT << 3; /* ATA/ATAPI-3 support */
	p[80] |= ATA2_SUPPORT << 2; /* ATA/ATAPI-2 support */
	p[80] |= ATA1_SUPPORT << 1; /* ATA/ATAPI-1 support */
	p[80] |= 0 << 0; /* Reserved */
#endif /* ATA3_SUPPORT || ATA4_SUPPORT */

#if ATA4_SUPPORT
	/* 81: Minor version number */
	p[81] = 0x0017; /* ATA/ATAPI-4 T13 1153D revision 17 */

#elif ATA3_SUPPORT
	/* 81: Minor version number */
	p[81] = 0x000c; /* ATA-3 X3T13 2008D revision 7 */

#else
	/* 81: Reserved */
#endif

#if ! (ATA3_SUPPORT || ATA4_SUPPORT)
	/* 82-83: Reserved */

#else /* ATA3_SUPPORT || ATA4_SUPPORT */
	/* 82: Command set supported */
#if ! ATA4_SUPPORT
	p[82] |= 0 << 15; /* Reserved */
	/* ... */
	p[82] |= 0 << 4; /* Reserved */
#else /* ATA4_SUPPORT */
	p[82] |= 0 << 15; /* Reserved */
	p[82] |= DISK_NOP_SUPPORT << 14; /* NOP supported */
	p[82] |= DISK_READ_BUFFER_SUPPORT << 13; /* READ BUFFER supported */
	p[82] |= DISK_WRITE_BUFFER_SUPPORT << 12; /* WRITE BUFFER supported */
	p[82] |= 0 << 11; /* Obsolete */
	p[82] |= 0 << 10; /* Host protected area supported */
	p[82] |= 0 << 9; /* DEVICE RESET command supported */
	p[82] |= 0 << 8; /* Service interrupt supported */
	p[82] |= 0 << 7; /* Release interrupt supported */
	p[82] |= DISK_LOOK_AHEAD_SUPPORT << 6; /* Look-ahead supported */
	p[82] |= DISK_WRITE_CACHE_SUPPORT << 5; /* Write cache supported */
	p[82] |= 0 << 4; /* PACKET command supported */
#endif /* ATA4_SUPPORT */
	p[82] |= DISK_POWER_MANAGEMENT_SUPPORT << 3; /* Power Management */
	p[82] |= 0 << 2; /* Removable Media */
	p[82] |= 0 << 1; /* Security Mode */
	p[82] |= DISK_SMART_SUPPORT << 0; /* SMART feature */

	/* 83: Command sets supported */
	p[83] |= 0 << 15; /* Shall be cleared to zero */
	p[83] |= 1 << 14; /* Shall be set to one */
#if ! ATA4_SUPPORT
	p[83] |= 0 << 13; /* Reserved */
	/* ... */
	p[83] |= 0 << 0; /* Reserved */
#else /* ATA4_SUPPORT */
	p[83] |= 0 << 13; /* Reserved */
	/* ... */
	p[83] |= 0 << 5; /* Reserved */
	p[83] |= 0 << 4; /* Removable media status notification supported */
	p[83] |= 0 << 3; /* Advanced power management supported */
	p[83] |= 0 << 2; /* CFA feature set supported */
	p[83] |= 0 << 1; /* READ/WRITE DMA QUEUED supported */
	p[83] |= DISK_DOWNLOAD_MICROCODE_SUPPORT << 0;
#endif /* ATA4_SUPPORT */
#endif /* ATA3_SUPPORT || ATA4_SUPPORT */

#if ! ATA4_SUPPORT
	/* 84-88: Reserved */

#else /* ATA4_SUPPORT */
	/* 84: */
	p[84] |= 0 << 15; /* Should be cleared */
	p[84] |= 1 << 14; /* Should be set */
	p[84] |= 0 << 13; /* Reserved */
	/* ... */
	p[84] |= 0 << 0; /* Reserved */

	/* 85: */
	p[85] |= 0 << 15; /* Obsolete */
	p[85] |= DISK_NOP_SUPPORT << 14; /* NOP command supported */
	p[85] |= DISK_READ_BUFFER_SUPPORT << 13; /* READ BUFFER supported */
	p[85] |= DISK_WRITE_BUFFER_SUPPORT << 12; /* WRITE BUFFER supported */
	p[85] |= 0 << 11; /* Obsolete */
	p[85] |= 0 << 10; /* Host protected area supported */
	p[85] |= 0 << 9; /* DEVICE RESET command supported */
	p[85] |= 0 << 8; /* Service interrupt enabled */
	p[85] |= 0 << 7; /* Release interrupt enabled */
#if DISK_LOOK_AHEAD_SUPPORT
	p[85] |= cpssp->look_ahead_enabled << 6; /* Look-ahead enabled */
#endif
#if DISK_WRITE_CACHE_SUPPORT
	p[85] |= cpssp->write_cache_enabled << 5; /* Write cache enabled */
#endif
	p[85] |= 0 << 4; /* PACKET command supported */
	p[85] |= DISK_POWER_MANAGEMENT_SUPPORT << 3; /* Power Management */
	p[85] |= 0 << 2; /* Removable Media */
	p[85] |= 0 << 1; /* Security Mode enabled */
#if DISK_SMART_SUPPORT
	p[85] |= cpssp->smart_enabled << 0; /* SMART feature set enabled */
#endif

	/* 86: */
	p[86] |= 0 << 15; /* Reserved */
	/* ... */
	p[86] |= 0 << 5; /* Reserved */
	p[86] |= 0 << 4; /* Removable media status notification enabled */
	p[86] |= 0 << 3; /* Advanced power management enabled */
	p[86] |= 0 << 2; /* CFA feature set enabled */
	p[86] |= 0 << 1; /* READ/WRITE DMA QUEUED supported */
	p[86] |= DISK_DOWNLOAD_MICROCODE_SUPPORT << 0;

	/* 87: */
	p[87] |= 0 << 15; /* Should be cleared */
	p[87] |= 1 << 14; /* Should be set */
	p[87] |= 0 << 13; /* Reserved */
	/* ... */
	p[87] |= 0 << 0; /* Reserved */

	/* 88: */
	p[88] |= 0 << 15; /* Reserved */
	/* ... */
	p[88] |= 0 << 11; /* Reserved */
#if CONFIG_ULTRA_DMA_MODE_2_SUPPORTED \
 || CONFIG_ULTRA_DMA_MODE_1_SUPPORTED \
 || CONFIG_ULTRA_DMA_MODE_0_SUPPORTED
	/* Selected Ultra DMA mode */
	if (cpssp->transfer_mode == ULTRA_DMA) {
		p[88] |= 1 << (cpssp->ultra_dma_mode + 8);
	}
#endif
	p[88] |= 0 << 7; /* Reserved */
	/* ... */
	p[88] |= 0 << 3; /* Reserved */
	p[88] |= CONFIG_ULTRA_DMA_MODE_2_SUPPORTED << 2;
	p[88] |= CONFIG_ULTRA_DMA_MODE_1_SUPPORTED << 1;
	p[88] |= CONFIG_ULTRA_DMA_MODE_0_SUPPORTED << 0;
#endif /* ATA4_SUPPORT */

	/* 89-127: Reserved */

#if ! (ATA3_SUPPORT || ATA4_SUPPORT)
	/* 128: Reserved */

#else /* ATA3_SUPPORT || ATA4_SUPPORT */
	/* 128: Security status */
	p[128] |= 0 << 15; /* Reserved */
	/* ... */
	p[128] |= 0 << 9; /* Reserved */
	p[128] |= 0 << 8; /* Security level 0=High, 1=Maximum */
	p[128] |= 0 << 7; /* Reserved */
	/* ... */
	p[128] |= 0 << 5; /* Reserved */
	p[128] |= 0 << 4; /* 1=Security count expired */
	p[128] |= 0 << 3; /* 1=Security frozen */
	p[128] |= 0 << 2; /* 1=Security locked*/
	p[128] |= 0 << 1; /* 1=Security enabled */
	p[128] |= 0 << 0; /* 1=Security supported */
#endif /* ATA3_SUPPORT || ATA4_SUPPORT */

	/* 129-159: Vendor specific */

	/* 160-255: Reserved */

	NAME_(gen_irq)(cpssp);
	NAME_(send)(cpssp, 512, 0, 0);
}

/*
 * Opcode: 0xef
 *
 * ATA1: Optional
 * ATA2: Optional
 * ATA3: Mandatory
 * ATA4: Mandatory
 */
static void
NAME_(set_features)(struct cpssp *cpssp)
{
	switch (cpssp->feature) {
#if DISK_WRITE_CACHE_SUPPORT
	case 0x02: /* Enable write cache */
		cpssp->write_cache_enabled = 1;
		break;

	case 0x82: /* Disable write cache */
		cpssp->write_cache_enabled = 0;
		break;
#endif

	case 0x03: /* Set transfer mode */
		switch ((cpssp->nsector >> 3) & 0x1f) {
		case 0x00: /* PIO default mode */
			cpssp->transfer_mode = PIO;
			cpssp->pio_mode = 2; /* FIXME */
			break;

#if CONFIG_PIO_MODE_3_SUPPORTED \
 || CONFIG_PIO_MODE_4_SUPPORTED
		case 0x01: /* PIO flow control mode */
			cpssp->transfer_mode = PIO;
			cpssp->pio_mode = cpssp->nsector & 0x7;
			break;
#endif

#if CONFIG_SW_DMA_MODE_0_SUPPORTED \
 || CONFIG_SW_DMA_MODE_1_SUPPORTED \
 || CONFIG_SW_DMA_MODE_2_SUPPORTED
		case 0x02: /* Single-word DMA mode */
			cpssp->transfer_mode = SW_DMA;
			cpssp->sw_dma_mode = cpssp->nsector & 0x7;
			break;
#endif

#if CONFIG_MW_DMA_MODE_0_SUPPORTED \
 || CONFIG_MW_DMA_MODE_1_SUPPORTED \
 || CONFIG_MW_DMA_MODE_2_SUPPORTED
		case 0x04: /* Multi-word DMA mode */
			cpssp->transfer_mode = MW_DMA;
			cpssp->mw_dma_mode = cpssp->nsector & 0x7;
			break;
#endif

#if CONFIG_ULTRA_DMA_MODE_0_SUPPORTED \
 || CONFIG_ULTRA_DMA_MODE_1_SUPPORTED \
 || CONFIG_ULTRA_DMA_MODE_2_SUPPORTED
		case 0x08: /* Ultra DMA mode */
			cpssp->transfer_mode = ULTRA_DMA;
			cpssp->ultra_dma_mode = cpssp->nsector & 0x7;
			break;
#endif
		default:
			umide_not_implemented(cpssp);
			break;
		}
		break;

#if DISK_LOOK_AHEAD_SUPPORT
	case 0x55: /* Disable read look-ahead. */
		cpssp->look_ahead_enabled = 0;
		break;

	case 0xaa: /* Enable read look-ahead. */
		cpssp->look_ahead_enabled = 1;
		break;
#endif

	default:
		fprintf(stderr, "%s: Warning: tried to set feature: 0x%02x: not implemented\n",
				__FUNCTION__, cpssp->feature);
		umide_not_implemented(cpssp);
	}

	NAME_(gen_irq)(cpssp);
}

static void
ide_report_command(struct cpssp *cpssp)
{
	const char *p;

	switch (cpssp->command) {
#if DISK_NOP_SUPPORT
	case 0x00:                  p = "Nop"; break;
#endif
#if DISK_RECALIBRATE_SUPPORT
	case 0x10:                  p = "Recalibrate"; break;
#endif
	case WIN_READ:              p = "Read (with retries)"; break;
	case WIN_READ + 1:          p = "Read (without retries)"; break;
	case WIN_WRITE:             p = "Write (with retries)"; break;
	case WIN_WRITE + 1:         p = "Write (without retries)"; break;
	case 0x40:                  p = "Read Verify Sector(s) (with retries)";
	                            break;
	case 0x40 + 1:              p = "Read Verify Sector(s) (without retries)";
	                            break;
	case 0x70:                  p = "Seek"; break;
	case 0x90:                  p = "Execute Device Diagnostic"; break;
	case 0x91:                  p = "Initialize Device Parameters"; break;
#if DISK_POWER_MANAGEMENT_SUPPORT
	case 0x94:
	case 0xe0:                  p = "Standby Immediate"; break;
	case 0x95:
	case 0xe1:                  p = "Idle Immediate"; break;
	case 0x96:
	case 0xe2:                  p = "Standby"; break;
	case 0x97:
	case 0xe3:                  p = "Idle"; break;
	case 0x98:
	case 0xe5:                  p = "Check Power Mode"; break;
	case 0x99:
	case 0xe6:                  p = "Sleep"; break;
#endif
#if DISK_SMART_SUPPORT
	case 0xb0:                  p = "Smart"; break;
#endif
	case WIN_MULTREAD:          p = "Multiple Read"; break;
	case WIN_MULTWRITE:         p = "Multiple Write"; break;
	case 0xc6:                  p = "Set Multiple Mode"; break;
	case WIN_READDMA:           p = "Read DMA (with retries)"; break;
	case WIN_READDMA + 1:       p = "Read DMA (without retries)"; break;
	case WIN_WRITEDMA:          p = "Write DMA (with retries)"; break;
	case WIN_WRITEDMA + 1:      p = "Write DMA (without retries)"; break;
#if DISK_READ_BUFFER_SUPPORT
	case 0xe4:                  p = "Read Buffer"; break;
#endif
#if DISK_WRITE_BUFFER_SUPPORT
	case 0xe8:                  p = "Write Buffer"; break;
#endif
	case 0xec:                  p = "Identify"; break;
	case 0xef:                  p = "Set Features"; break;
	default:                    p = "Unkown"; break;
	}

	fprintf(stderr, "%30s(0x%02x) (%u/%u/%u)", p, cpssp->command,
			cpssp->cyls, cpssp->heads, cpssp->secs);

	if (cpssp->command == WIN_READ
	 || cpssp->command == WIN_READ + 1
	 || cpssp->command == WIN_WRITE
	 || cpssp->command == WIN_WRITE + 1
	 || cpssp->command == 0x70 /* Seek */
	 || cpssp->command == WIN_MULTREAD
	 || cpssp->command == WIN_MULTWRITE
	 || cpssp->command == WIN_READDMA
	 || cpssp->command == WIN_READDMA + 1
	 || cpssp->command == WIN_WRITEDMA
	 || cpssp->command == WIN_WRITEDMA + 1) {
		unsigned int blocks;
		unsigned int cyl;
		unsigned int head;
		unsigned int sector;

		blocks = (cpssp->nsector == 0) ? 256 : cpssp->nsector;
		cyl = (unsigned int) cpssp->lcyl + (unsigned int) cpssp->hcyl * 256;
		head = (unsigned char) cpssp->select & 0x0f;
		sector = (unsigned char) cpssp->sector;

		if ((cpssp->select >> 6) & 1) {
			/* LBA */
			unsigned long pos;

			pos = (head << 24) | (cyl << 8) | (sector << 0);
			fprintf(stderr, " lba = %lu", pos);

		} else {
			/* C/H/S */
			fprintf(stderr, " chs = %u/%u/%u", cyl, head, sector);
		}

		fprintf(stderr, " sectors = %u", blocks);
	}

	fprintf(stderr, "\n");
}

static void __attribute__((__noreturn__))
NAME_(process)(void *_cpssp)
{
	struct cpssp *cpssp = _cpssp;

	cpssp->status = SEEK_STAT;

	for (;;) {
		/*
		 * Wait for command.
		 */
		while (! cpssp->reset
		    && ! cpssp->newcmd) {
			cpssp->status &= ~BUSY_STAT;
			cpssp->status |= READY_STAT;
			sched_sleep();
		}
		if (cpssp->reset) {
			/*
			 * Execute Reset
			 */
			cpssp->reset = 0;
			NAME_(reset)(cpssp);

		} else { assert(cpssp->newcmd);
			/*
			 * Execute Command
			 */
			cpssp->newcmd = 0;

			if (DEBUG_OPCODES
			 && loglevel) {
				ide_report_command(cpssp);
			}

			switch (cpssp->command) {
#if DISK_NOP_SUPPORT
			case 0x00:
				NAME_(nop)(cpssp);
				break;
#endif /* DISK_NOP_SUPPORT */
#if DISK_RECALIBRATE_SUPPORT
			case 0x10:
				NAME_(recalibrate)(cpssp);
				break;
#endif /* DISK_RECALIBRATE_SUPPORT */
			case WIN_READ:		/* 0x20 */
			case WIN_READ + 1:	/* 0x21 */
				NAME_(read)(cpssp);
				break;
#if DISK_READWRITE_LONG_SUPPORT
			case 0x22:
			case 0x23 + 1:
				NAME_(read_long)(cpssp);
				break;
#endif /* DISK_READWRITE_LONG_SUPPORT */
			case WIN_WRITE:		/* 0x30 */
			case WIN_WRITE + 1:	/* 0x31 */
				NAME_(write)(cpssp);
				break;
#if DISK_READWRITE_LONG_SUPPORT
			case 0x32:
			case 0x33:
				NAME_(write_long)(cpssp);
				break;
#endif /* DISK_READWRITE_LONG_SUPPORT */
			case 0x40:
			case 0x41:
				NAME_(read_verify_sectors)(cpssp);
				break;
			case 0x70:
				NAME_(seek)(cpssp);
				break;
			case 0x90:
				NAME_(execute_device_diagnostic)(cpssp);
				break;
			case 0x91:
				NAME_(initialize_device_parameters)(cpssp);
				break;
#if DISK_DOWNLOAD_MICROCODE_SUPPORT
			case 0x92:
				NAME_(download_microcode)(cpssp);
				break;
#endif
#if DISK_POWER_MANAGEMENT_SUPPORT
			case 0x94:
			case 0xe0:
				NAME_(standby_immediate)(cpssp);
				break;
			case 0x95:
			case 0xe1:
				NAME_(idle_immediate)(cpssp);
				break;
			case 0x96:
			case 0xe2:
				NAME_(standby)(cpssp);
				break;
			case 0x97:
			case 0xe3:
				NAME_(idle)(cpssp);
				break;
			case 0x98:
			case 0xe5:
				NAME_(check_power_mode)(cpssp);
				break;
			case 0x99:
			case 0xe6:
				NAME_(sleep)(cpssp);
				break;
#endif /* DISK_POWER_MANAGEMENT_SUPPORT */
#if DISK_SMART_SUPPORT
			case 0xb0:
				NAME_(smart)(cpssp);
				break;
#endif /* DISK_SMART_SUPPORT */
#if DISK_MULTIPLE_SUPPORT
			case WIN_MULTREAD:	/* 0xc4 */
				NAME_(read_multiple)(cpssp);
				break;
			case WIN_MULTWRITE:	/* 0xc5 */
				NAME_(write_multiple)(cpssp);
				break;
			case 0xc6:
				NAME_(set_multiple_mode)(cpssp);
				break;
#endif /* DISK_MULTIPLE_SUPPORT */
#if DISK_DMA_SUPPORT
			case WIN_READDMA:	/* 0xc8 */
			case WIN_READDMA + 1:	/* 0xc9 */
				NAME_(read_dma)(cpssp);
				break;
			case WIN_WRITEDMA:	/* 0xca */
			case WIN_WRITEDMA + 1:	/* 0xcb */
				NAME_(write_dma)(cpssp);
				break;
#endif /* DISK_DMA_SUPPORT */
#if DISK_READ_BUFFER_SUPPORT
			case 0xe4:
				NAME_(read_buffer)(cpssp);
				break;
#endif /* DISK_READ_BUFFER_SUPPORT */
#if DISK_WRITE_BUFFER_SUPPORT
			case 0xe8:
				NAME_(write_buffer)(cpssp);
				break;
#endif /* DISK_WRITE_BUFFER_SUPPORT */
			case 0xec:
				NAME_(identify_device)(cpssp);
				break;
			case 0xef:
				NAME_(set_features)(cpssp);
				break;
			default:
				cpssp->error = 1 << 2; /* Abort */
				cpssp->status |= ERR_STAT;
				NAME_(gen_irq)(cpssp);
				break;
			}
		}
	}
}

static int
NAME_(inw)(void *_cpssp, unsigned short port, uint16_t *valp)
{
	struct cpssp *cpssp = (struct cpssp *) _cpssp;

	if (DEBUG_CONTROL_FLOW
	 && loglevel) {
		fprintf(stderr, "%s called(port=0x%04x)\n", __FUNCTION__, port);
	}

	if (cpssp->faulty_disk) {
		return 1;
	}

	if (UNIT != cpssp->unit) {
		return 1;
	}

	switch (port) {
	case 0:		/* data port */
		if (! (cpssp->status & BUSY_STAT)
		 && cpssp->status & DRQ_STAT) {
			assert(sizeof(*valp) <= cpssp->count);
			assert(cpssp->tail + sizeof(*valp) <= cpssp->head);

			memcpy(valp, &cpssp->buf[cpssp->tail], sizeof(*valp));
			cpssp->tail += sizeof(*valp);
			cpssp->count -= sizeof(*valp);

			if (cpssp->count == 0) {
				sig_ide_bus_dmarq_set(cpssp->cable, 0);
				cpssp->status &= ~DRQ_STAT;
				if (cpssp->blocked) {
					cpssp->status |= BUSY_STAT;
					cpssp->status &= ~READY_STAT;
					sched_wakeup(&cpssp->process);
				}
			}
		} else {
			fprintf(stderr, "WARNING: %s: inw: status=0x%02x, count=%u.\n",
					SNAME, cpssp->status, cpssp->count);
			*valp = 0;
		}
		break;
	case 1:		/* error register */
		*valp = cpssp->error;
		break;
	case 2:		/* sector count register */
		*valp = cpssp->nsector;
		break;
	case 3:		/* sector number register */
		*valp = cpssp->sector;
		break;
	case 4:		/* cylinder low register */
		*valp = cpssp->lcyl;
		break;
	case 5:		/* cylinder high register */
		*valp = cpssp->hcyl;
		break;
	case 6:		/* drive/head register */
		*valp = cpssp->select | 0xa0;
		break;

	case 7:		/* status register */
		*valp = cpssp->status;
		NAME_(ugen_irq)(cpssp);
		break;
	case 8 + 6:	/* altstatus register */
		*valp = cpssp->status;
#if 0
		cpssp->status &= ~ERR_STAT;
#endif
		break;
	case 8 + 7:	/* drive address register */
		fprintf(stderr, "WARNING: drive address register read!\n");
		break;
	default:
		assert(0);
		*valp = 0;	/* Just to make gcc happy... */
		/*NOTREACHED*/
	}

	if (DEBUG_CONTROL_FLOW
	 && loglevel) {
		fprintf(stderr, "%s return(*valp=0x%04x)\n",
				__FUNCTION__, *valp);
	}
	return 0;
}

static void
NAME_(outw)(void *_cpssp, unsigned short port, uint16_t value)
{
	struct cpssp *cpssp = (struct cpssp *) _cpssp;

	if (DEBUG_CONTROL_FLOW
	 && loglevel) {
		fprintf(stderr, "%s called(value=0x%04x, port=0x%04x)\n",
				__FUNCTION__, value, port);
	}

	if (cpssp->faulty_disk) {
		return;
	}

	switch (port) {
	case 1:		/* feature register (write only) */
		cpssp->feature = value;
		return;
	case 2:		/* sector count register */
		cpssp->nsector = value;
		return;
	case 3:		/* sector number register */
		cpssp->sector = value;
		return;
	case 4:		/* cylinder low register */
		cpssp->lcyl = value;
		return;
	case 5:		/* cylinder high register */
		cpssp->hcyl = value;
		return;
	case 6:		/* drive/head register */
		cpssp->select = value & ~0xa0;
		NAME_(irq_update)(cpssp);
		return;
	case 8 + 6:	/* device control register */
		cpssp->control = value;
		if (value & 0x04) {
			/* if SFRS bit set -> reset */
			cpssp->status |= BUSY_STAT;
			cpssp->reset = 1;
			sched_wakeup(&cpssp->process);
		}
		return;
	}
	
	if (UNIT != cpssp->unit) {
		return;
	}

	switch (port) {
	case 0:		/* data port */
		if (! (cpssp->status & BUSY_STAT)
		 && cpssp->status & DRQ_STAT) {
			assert(sizeof(value) <= cpssp->count);
			assert(cpssp->head + sizeof(value) <= sizeof(cpssp->buf));

			memcpy(&cpssp->buf[cpssp->head], &value, sizeof(value));
			cpssp->head += sizeof(value);
			cpssp->count -= sizeof(value);

			if (cpssp->count == 0) {
				sig_ide_bus_dmarq_set(cpssp->cable, 0);
				cpssp->status &= ~DRQ_STAT;
				if (cpssp->blocked) {
					cpssp->status |= BUSY_STAT;
					cpssp->status &= ~READY_STAT;
					sched_wakeup(&cpssp->process);
				}
			}
		} else {
			fprintf(stderr, "WARNING: %s: outw: status=0x%02x, count=%u.\n",
					SNAME, cpssp->status, cpssp->count);
		}
		break;
	case 7:		/* command register */
		if (! (cpssp->status & BUSY_STAT)
		 /* && cpssp->status & READY_STAT */) {
			cpssp->command = value;
			cpssp->status |= BUSY_STAT;
			cpssp->status &= ~(READY_STAT | DRQ_STAT | ERR_STAT);
			cpssp->newcmd = 1;
			NAME_(ugen_irq)(cpssp);
			sched_wakeup(&cpssp->process);
		} else {
			fprintf(stderr, "WARNING: %s: outw: status=0x%02x.\n",
					SNAME, cpssp->status);
		}
		break;
	default:
		assert(0);
		/* NOTREACHED --tg 21:26 2002-05-20 */
	}

	if (DEBUG_CONTROL_FLOW
	 && loglevel) {
		fprintf(stderr, "%s return()\n", __FUNCTION__);
	}
}

static void
NAME_(power_set)(void *_css, unsigned int val)
{
	struct cpssp *css = (struct cpssp *) _css;

	if (val) {
		/* Power On Event */
		NAME_(reset)(css);

	} else {
		/* Power Off Event */
		/* Nothing to do (yet). */
	}
}

static void
NAME_(disk_fault_set)(
	void *_cpssp,
	unsigned long long loc0,
	unsigned long long loc1,
	unsigned int val
)
{
	struct cpssp *cpssp = (struct cpssp *) _cpssp;

	if (val) {
		/* Add new fault. */
		assert(! cpssp->faulty_disk);
		cpssp->faulty_disk = 1;

	} else {
		/* Remove old fault. */
		assert(cpssp->faulty_disk);
		cpssp->faulty_disk = 0;
	}
}

static void
NAME_(block_fault_set)(
	void *_cpssp,
	unsigned long long loc0,
	unsigned long long loc1,
	unsigned int val
)
{
	struct cpssp *cpssp = (struct cpssp *) _cpssp;
	unsigned int i;

	if (val) {
		/* Add new fault. */
		for (i = 0; ; i++) {
			if (i == sizeof(cpssp->faulty_blk) / sizeof(cpssp->faulty_blk[0])) {
				/* Too many faults. */
				fixme(); /* FIXME */
				break;
			}
			if (cpssp->faulty_blk[i] == -1) {
				cpssp->faulty_blk[i] = loc0;
				break;
			}
		}
	} else {
		/* Remove old fault. */
		for (i = 0; ; i++) {
			if (i == sizeof(cpssp->faulty_blk) / sizeof(cpssp->faulty_blk[0])) {
				/* Fault not found. */
				fixme(); /* FIXME */
				break;
			}
			if (cpssp->faulty_blk[i] == loc0) {
				cpssp->faulty_blk[i] = -1;
				break;
			}
		}
	}
}

void
NAME_(init)(
	unsigned int nr,
	struct sig_power_device *port_power,
	struct sig_ide_bus *port_ide,
	struct sig_fault *fault_disk_fault,
	struct sig_fault *fault_block_fault
)
{
	static const struct sig_boolean_funcs power_funcs = {
		.set = NAME_(power_set),
	};
	static const struct sig_ide_bus_device_funcs funcs = {
		.inw = NAME_(inw),
		.outw = NAME_(outw),
	};
	static const struct sig_fault_funcs disk_fault_funcs = {
		.set = NAME_(disk_fault_set),
	};
	static const struct sig_fault_funcs block_fault_funcs = {
		.set = NAME_(block_fault_set),
	};
	struct cpssp *cpssp;

	cpssp = shm_map(SNAME, nr, sizeof(*cpssp), 0);

	cpssp->sig_5V = port_power->power_5V;
	sig_boolean_connect_in(port_power->power_5V, cpssp, &power_funcs);
	cpssp->sig_12V = port_power->power_12V;

	cpssp->cable = port_ide;
	sig_ide_bus_connect_device(port_ide, cpssp, &funcs);

	sig_fault_connect(fault_disk_fault, cpssp, &disk_fault_funcs);
	sig_fault_connect(fault_block_fault, cpssp, &block_fault_funcs);

	storage_init(&cpssp->media);
	storage_open(&cpssp->media, 1);

	sched_process_init(&cpssp->process, NAME_(process), cpssp);
}

void
NAME_(create)(
	unsigned int nr,
	const char *name,
	const char *cow,
	const char *create,
	const char *_sync,
	const char *image,
#ifndef CYLINDERS
	const char *cylinders,
	const char *heads,
	const char *sectors,
#endif
	const char *unit
#ifndef CYLINDERS
	, const char *size
#endif
)
{
	struct cpssp *cpssp;
	char path[1024];	/* 4 would be enough (.cow/.map) */
	unsigned int i;

	shm_create(SNAME, nr, sizeof(*cpssp));
	cpssp = shm_map(SNAME, nr, sizeof(*cpssp), 0);

	if (! cow) cow = "no";
	if (! create) create = "no";
	if (! _sync) _sync = "no";
	if (! image) image = "";
	if (! unit) unit = "0";

#ifdef CYLINDERS
	cpssp->phys_secs = SECTORS;
	cpssp->phys_heads = HEADS;
	cpssp->phys_cyls = CYLINDERS;
	cpssp->phys_linear = LINEAR;

#else
	if (cylinders && heads && sectors && ! size) {
		/* Set geometry via C/H/S setting. */
		cpssp->phys_secs = strtoul(sectors, NULL, 0);
		cpssp->phys_heads = strtoul(heads, NULL, 0);
		cpssp->phys_cyls = strtoul(cylinders, NULL, 0);

		cpssp->phys_linear = cpssp->phys_secs
				* cpssp->phys_heads * cpssp->phys_cyls;

	} else if (! cylinders && ! heads && ! sectors && size) {
		/* Set geometry via capacity setting. */
		unsigned long long blocks;

		cpssp->phys_linear = strtoul(size, NULL, 0) * 1024ULL * 1024 / 512;

		blocks = cpssp->phys_linear;

		cpssp->phys_secs = 63;
		blocks /= 63;
		cpssp->phys_heads = 16;
		blocks /= 16;
		cpssp->phys_cyls = (0x4000 <= blocks) ? 0x3fff : blocks;

	} else {
		fprintf(stderr, "%s: Bad C/H/S or size setting.\n", SNAME);
		exit(1);
	}
#endif

	cpssp->media_cow = (*cow == 'Y' || *cow == 'y');
	cpssp->media_create = (*create == 'Y' || *create == 'y');
	cpssp->media_sync = (*_sync == 'Y' || *_sync == 'y');

	strcpy(cpssp->media_image, image);

	cpssp->unit = strtoul(unit, NULL, 0);
	cpssp->serial_number = nr;

	assert(strlen(SNAME "-XX") + 1 < sizeof(cpssp->media_name));
	sprintf(cpssp->media_name, SNAME "-%d", nr);

	assert(strlen(cpssp->media_name) + strlen(".media") < sizeof(path));
	sprintf(path, "%s.media", cpssp->media_name);
	storage_create(&cpssp->media,
			path,
			1,	/* writable */
			cpssp->media_image,
			(unsigned long) ((cpssp->phys_linear
				* 512ULL + 1024*1024-1) / (1024 * 1024)),
			512,	/* blocksize */
			cpssp->media_create,
			cpssp->media_cow,
			cpssp->media_sync);

#if DISK_SMART_SUPPORT
	cpssp->smart_enabled = 1;
#endif
#if DISK_LOOK_AHEAD_SUPPORT
	cpssp->look_ahead_enabled = 1;
#endif
#if DISK_WRITE_CACHE_SUPPORT
	cpssp->write_cache_enabled = 0;
#endif

	cpssp->faulty_disk = 0;
	for (i = 0; i < sizeof(cpssp->faulty_blk) / sizeof(cpssp->faulty_blk[0]); i++) {
		cpssp->faulty_blk[i] = -1;
	}

	shm_unmap(cpssp, sizeof(*cpssp));
}

void
NAME_(destroy)(unsigned int nr)
{
	struct cpssp *cpssp;

	cpssp = shm_map(SNAME, nr, sizeof(*cpssp), 0);

	storage_destroy(&cpssp->media);

	shm_unmap(cpssp, sizeof(*cpssp));
	shm_destroy(SNAME, nr);
}

#undef DEBUG_CONTROL_FLOW
#undef DEBUG_INTERRUPT
#undef DEBUG_OPCODES
