/*
 * $Id: setup.c,v 1.2 2009-03-02 20:22:23 vrsieh Exp $ 
 *
 * Copyright (C) 2004-2009 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#include "config.h"
#include "build_config.h"

/* ==================== REAL-MODE INIT ==================== */
#ifdef INIT_RM

#include <stddef.h>
#include "compiler.h"
CODE16;

#include "setup.h"
#include "cmos.h"
#include "video.h"
#include "video.h"
#include "stdio.h"
#include "kbd.h"
#include "debug.h"
#include "pci.h"
#include "segment.h"
#include "const.h"

#define SCR_BAK_SEG1 0x100
#define SCR_BAK_SEG2 0x200

static CONST char * CONST bootorder_entries[] = {
	"Floppy  ",
	"CD-ROM  ",
	"Harddisk",
	"Int 19h "
};

static CONST char * CONST main_entries[MAIN_MENU_ENTRIES] = {
	"Date/Time          ",
	"Boot Order         ",
	"Console            ",
	"BIOS defaults      ",
	"Advanced Power Mgm.",
	"PCI Configuration  ",
	"Quit without saving",
	"Save and Exit      "
};

static CONST char main_pos[MAIN_MENU_ENTRIES][2] = {
	{6, 7},
	{6, 9},
	{6, 11},
	{6, 13},
	{6, 15},
	{6, 17},
	{6, 19},
	{6, 21}
};

static CONST char * CONST pci_irq_route_entries[] = {
	"NONE",
	" 3",
	" 4",
	" 5",
	" 6",
	" 7",
	" 9",
	"10",
	"11",
	"12",
	"14",
	"15"
};
#define NUMPCIROUTEENTRIES (sizeof(pci_irq_route_entries)/sizeof(char *))

static CONST char * CONST yesnoselection[] = {
	"No",
	"Yes"
};



static void
screen_copy(unsigned short srcseg, unsigned short dstseg)
{
	unsigned short i;

	for (i = 0; i< 80 * 25; i++) {
		put_word(dstseg, i * 2, get_word(srcseg, i * 2));
	}
}

static void
print_entry(unsigned short index)
{
	unsigned short x;
	unsigned short y;

	x = const_get(main_pos[index][0]);
	y = const_get(main_pos[index][1]);
	gotoxy(x, y, 0);
	putcstr((const char *) const_get(main_entries[index]));
}

static void
cmos_ext_to_mem(unsigned short seg)
{
	unsigned short i;

	/* copy cmos to memory */
	for (i = 0; i < 128; i++) {
		put_byte(seg, i, cmos_ext_read(i));
	}
}

static void
mem_to_cmos_ext(unsigned short seg)
{
	unsigned short i;

	/* copy memory back to cmos */
	for (i = 0; i < 128; i++) {
		cmos_ext_write(i, get_byte(seg, i));
	}
}

static void
color_entry_xyl(unsigned short x, unsigned short y,
	unsigned short length, unsigned char col)
{
	unsigned short i;

	for (i = 0; i < length; i++) {
		put_byte(0xb800, (y * 80 + x + i) * 2 + 1, col);
	}
}

static void
color_entry(unsigned short index, unsigned char col)
{
	unsigned short x;
	unsigned short y;

	x = const_get(main_pos[index][0]);
	y = const_get(main_pos[index][1]);
	color_entry_xyl(x - 1, y, 21, col);
}
	
static void
print_setup_screen(void)
{
	unsigned short i;

	/* all white/blue and clean */
	for (i = 0; i < 25 * 80; i++) {
		put_word(0xb800, i * 2, 0x1f20);
	}
	/* all yellow/blue and clean */
	for (i = 161; i < 239; i++) {
		put_word(0xb800, i * 2, 0x1e20);
		put_word(0xb800, (i + 80) * 2, 0x1e20);
	}
	/* horizontal double bars */
	for (i = 1; i < 79; i++) {
		gotoxy(i, 0, 0); putchar(205);
		gotoxy(i, 5, 0); putchar(205);
		gotoxy(i, 23, 0); putchar(205);
	}
	/* vertical double bars */
	for (i = 1; i < 23; i++) {
		gotoxy(0, i, 0); putchar(186);
		gotoxy(79, i, 0); putchar(186);
	}
	/* vertical single bar */
	for (i = 6; i < 23; i++) {
		gotoxy(39, i, 0); putchar(179);
	}

	/* top */
	/* left */
	gotoxy(0, 0, 0); putchar(201);
	/* right */
	gotoxy(79, 0, 0); putchar(187);
	/* bottom */
	/* left */
	gotoxy(0, 23, 0); putchar(200);
	/* right */
	gotoxy(79, 23, 0); putchar(188);
	/* middle */
	/* left */
	gotoxy(0, 5, 0); putchar(204);
	/* right */
	gotoxy(79, 5, 0); putchar(185);
	/* vertical middle */
	/* top */
	gotoxy(39, 5, 0); putchar(209);
	/* bottom */
	gotoxy(39, 23, 0); putchar(207);

	/* title text */
	gotoxy(24, 2, 0); putcstr(PACKAGE_NAME " BIOS v" PACKAGE_VERSION);
}

static void 
setup_drawselection(int x, int y, int width, CONST char * CONST values[],
			 unsigned char numentries, unsigned char selval,
			 unsigned char selected)
{
	int i;
	
	gotoxy(x+1, y, 0);
	for (i = 0; i < width; i++) putchar(32);
	gotoxy(x+1, y, 0);
	putcstr((const char *) const_get(values[selval % numentries]));
	color_entry_xyl(x, y, width+2, (selected) ? 0x4f : 0x1f);
}

/* return value:
 * high byte = code to exit (0 = escape, 1 = up, 2 = down)
 * low byte = selection */
static unsigned short
setup_selection(int x, int y, int width, CONST char * CONST values[],
		unsigned char numentries, unsigned char startval)
{
	unsigned short key;
	unsigned char selval;
	
	selval = (startval % numentries);
	while (1) {
		setup_drawselection(x, y, width, values, numentries, selval, 1);
		while (! kbd_keyavailable()) {
		}
		key = kbd_getkey(1);
		/*bprintf(" %04x \n", key);*/
		if (key == 0x011B) {	/* escape */
			return (0 << 8) | selval;
		
		} else if (key == 0x4800) {
			return (1 << 8) | selval;

		} else if (key == 0x5000) {
			return (2 << 8) | selval;

		} else if ((key == 0x4900)
			|| (key == 0x4b00)) {	/* page up / arrow left */
			selval = (selval + numentries - 1) % numentries;

		} else  if ((key == 0x5100)
			 || (key == 0x4d00)) {	/* page down / arrow right */
			selval = (selval + 1) % numentries;
		}
	}
}

static unsigned char
setup_boolean(CONST char *topic, CONST char *attr, unsigned char boolean)
{
	print_setup_screen();

	gotoxy(4, 7, 0);
	putcstr(topic);

	gotoxy(5, 9, 0);
	putcstr(attr);

	return (setup_selection(29, 9, 3, yesnoselection, 2, boolean) & 0x00FF);

}

#ifdef CONFIG_PCI_SUPPORT
static void
setup_pci(void)
{
	unsigned short i;
	unsigned short selected;
	unsigned char pciirqs[PCIIRQ_ENTRIES];

	print_setup_screen();
	
	gotoxy(39, 5, 0); putchar(205);
	gotoxy(39, 23, 0); putchar(205);
	for (i=6; i<23; i++) {
		gotoxy(39, i, 0); putchar(' ');
	}
	
	for (i = 0; i < PCIIRQ_ENTRIES; i++) {
		gotoxy(4 + ((i / 2) * 40), 7 + (i % 2), 0);
		bprintf("PCI IRQ Line %d: ", i+1);
	}
	/* print the device list. */
	gotoxy(4, 10, 0);
	putcstr("PCI Device List:");
	gotoxy(0, 11, 0);
	pci_listdevices("\xBA""   ");
	for (i = 0; i < PCIIRQ_ENTRIES; i++) {
		pciirqs[i] = get_byte(NV_SEG,
				offsetof(struct cmos_ext, pciirqs[i]));
		if ((pciirqs[i] < 10)
		 || ((9 + NUMPCIROUTEENTRIES) < pciirqs[i])) {
			/* Invalid setting, use default */
			if (i != (PCIIRQ_ENTRIES - 1)) {
				pciirqs[i] = i + 6;
			} else {
				pciirqs[i] = i;
			}
		} else {
			pciirqs[i] -= 10;
		}
	}
	selected = 0;
	while (1) {
		for (i = 0; i < PCIIRQ_ENTRIES; i++) {
			setup_drawselection(24+(i/2)*40, 7+(i%2), 4,
				   pci_irq_route_entries, NUMPCIROUTEENTRIES,
				   pciirqs[i], (selected == i));
		}
		i = setup_selection(24+(selected/2)*40, 7+(selected%2), 4,
				   pci_irq_route_entries, NUMPCIROUTEENTRIES,
				   pciirqs[selected]);
		pciirqs[selected] = i & 0x00ff;
		i >>= 8;
		if        (i == 0) {
			break;
		} else if (i == 1) {
			selected = (selected + PCIIRQ_ENTRIES - 1)
				% PCIIRQ_ENTRIES;
		} else if (i == 2) {
			selected = (selected + 1) % PCIIRQ_ENTRIES;
		}
	}
	for (i = 0; i < PCIIRQ_ENTRIES; i++) {
		pciirqs[i] += 10;
		put_byte(NV_SEG, offsetof(struct cmos_ext, pciirqs[i]),
				 pciirqs[i]);
	}
}
#endif	

static void
setup_boot_order(void)
{
	unsigned char bootorder[BOOTORDER_ENTRIES];
	unsigned short key;
	unsigned short i;
	unsigned short sel;


	print_setup_screen();

	gotoxy(4, 7, 0);
	putcstr("Boot order selection");

	/* FIXME */
	for (i = 0; i < BOOTORDER_ENTRIES; i++)
		bootorder[i] = i;

	sel = 0;
	while (1) {
		for (i=0; i<BOOTORDER_ENTRIES; i++) {
			gotoxy(6, 9 + i, 0);
			putcstr((const char *) const_get(bootorder_entries[bootorder[i]]));
		}

		color_entry_xyl(6, 9+sel, 10, 0x4f);

		while (! kbd_keyavailable()) {
		}

		key = kbd_getkey(1);

		if (key == 0x4800) {		/* cursor up */
			if (sel > 0) {
				color_entry_xyl(6, 9 + sel, 10, 0x1f);
				sel --;
			}

		} else  if (key == 0x5000) {	/* cursor down */
			if (sel < BOOTORDER_ENTRIES - 1) {
				color_entry_xyl(6, 9 + sel, 10, 0x1f);
				sel ++;
			}
		} else if (key == 0x4900) {	/* page up */
			if (sel > 0) {
				color_entry_xyl(6, 9 + sel, 10, 0x1f);
				i = bootorder[sel];
				bootorder[sel] = bootorder[sel - 1];
				bootorder[sel - 1] = i;
				sel --;
			}

		} else  if (key == 0x5100) {	/* page down */
			if (sel < BOOTORDER_ENTRIES - 1) {
				color_entry_xyl(6, 9 + sel, 10, 0x1f);
				i = bootorder[sel];
				bootorder[sel] = bootorder[sel + 1];
				bootorder[sel + 1] = i;
				sel ++;
			}

		} else if (key == 0x011B) {	/* escape */
			/* save values to cmos_ext */
			/* FIXME */
			/* get back */
			return;
		}
	}
}

int
setup_defaults(void)
{
	unsigned short i;
	
	if (cmos_ext_get(initialized)) {
		return 0;
	}

	/* clear old values */
	for (i = 0; i < 128; i++) {
		cmos_ext_write(i, 0);
	}

	cmos_put(x3c, 3); /* Boot Sequence CDROM,C,A */
	cmos_ext_put(initialized, 1);

	return 1;
}

void
setup_init(void)
{


	unsigned short i;
	unsigned short sel;
	unsigned short key;
	unsigned char x;
	unsigned char y;

	/* save old screen */
	savexy(&x, &y, 0);

	screen_copy(0xb800, SCR_BAK_SEG1);

	print_setup_screen();

	for (i=0; i < MAIN_MENU_ENTRIES; i++) {
		print_entry(i);
	}

	/* copy cmos_ext to memory */
	cmos_ext_to_mem(NV_SEG);

	sel = 0;
	color_entry(sel, 0x4f);

	while (1) {
		while (! kbd_keyavailable()) {
		}

		key = kbd_getkey(1);

		if (key == 0x4800) {		/* cursor up */
			if (sel > 0) {
				color_entry(sel, 0x1f);
				sel--;
				color_entry(sel, 0x4f);
			}

		} else  if (key == 0x5000) {	/* cursor down */
			if (sel < MAIN_MENU_ENTRIES - 1) {
				color_entry(sel, 0x1f);
				sel++;
				color_entry(sel, 0x4f);
			}

		} else  if (key == 0x4B00) {	/* cursor left */
			if (0 <= sel - 8) {
				color_entry(sel, 0x1f);
				sel -= 8;
				color_entry(sel, 0x4f);
			}

		} else  if (key == 0x4D00) {	/* cursor right */
			if (sel + 8 < MAIN_MENU_ENTRIES) {
				color_entry(sel, 0x1f);
				sel += 8;
				color_entry(sel, 0x4f);
			}

		} else if (key == 0x1C0D) {	/* return */
			if (sel == 1) {
				/* Boot order */
				screen_copy(0xb800, SCR_BAK_SEG2);
				setup_boot_order();
				screen_copy(SCR_BAK_SEG2, 0xb800);

			} else if (sel == 2) {
				/* Was: Console */

			} else if (sel == 3) {
				/* restore BIOS defaults */
				setup_defaults();
				/* copy cmos_ext to memory */
				cmos_ext_to_mem(NV_SEG);

			} else if (sel == 4) {
#ifdef CONFIG_APM_SUPPORT
				/* enable/disable apm */
				screen_copy(0xb800, SCR_BAK_SEG2);
				setup_apm();
				screen_copy(SCR_BAK_SEG2, 0xb800);
#endif

			} else if (sel == 5) {
#ifdef CONFIG_PCI_SUPPORT
				/* PCI Configuration */
				screen_copy(0xb800, SCR_BAK_SEG2);
				setup_pci();
				screen_copy(SCR_BAK_SEG2, 0xb800);
#endif

			} else if (sel == 6) {
				/* quit without saving */
				/* restore old screen */
				screen_copy(SCR_BAK_SEG1, 0xb800);
				gotoxy(x, y, 0);
				return;

			} else if (sel == 7) {
				/* Save and exit */
				/* restore old screen */
				screen_copy(SCR_BAK_SEG1, 0xb800);
				gotoxy(x, y, 0);
				/* copy memory to cmos_ext */
				mem_to_cmos_ext(NV_SEG);
				return;
			}

		} else if (key == 0x011B) {	/* escape */
			/* restore old screen */
			screen_copy(SCR_BAK_SEG1, 0xb800);
			gotoxy(x, y, 0);
			return;
		}
	}
}

void
setup_exit(void)
{
	uint16_t sum;
	unsigned int reg;

	sum = 0;
	for (reg = 0x10; reg < 0x2e; reg++) {
	sum += cmos_readb(reg);
	}
	cmos_writeb(0x2e, (sum >> 8) & 0xff);
	cmos_writeb(0x2f, (sum >> 0) & 0xff);

	sum = 0;
	for (reg = 0x40; reg < 0x7d; reg++) {
	sum += cmos_readb(reg);
	}
	cmos_writeb(0x7d, (sum >> 8) & 0xff);
	cmos_writeb(0x7e, (sum >> 0) & 0xff);
}

#endif /* INIT_RM */
