/*
 * Copyright (C) 2007-2017 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#define DEBUG_MEM	0
#define DEBUG_MEM_START	0
#define DEBUG_MEM_END	(DEBUG_MEM_START + 0xffffffff)

#define LOG_COUNT	0

#define COUNT		256

#ifdef CONFIG_CPU_CACHE_LINE_SIZE
#define LINE		CONFIG_CPU_CACHE_LINE_SIZE
#else
#define LINE		4096
#endif

#define OMASK		(LINE - 1)
#define AMASK		~OMASK

#ifdef STATE

struct {
#if LOG_COUNT
#if CONFIG_NEED_WRITE
	uint64_t umwb_count;
	uint64_t umww_count;
	uint64_t umwd_count;
	uint64_t umwq_count;
#endif
	uint64_t umrb_count;
	uint64_t umrw_count;
	uint64_t umrd_count;
	uint64_t umrq_count;
#endif /* LOG_COUNT */

#if CONFIG_NEED_WRITE
	struct {
		vaddr_t vaddr;
		paddr_t paddr;
		int (*cf)(void *, paddr_t, unsigned int, udata_t);
		void *cs;
		char *haddr;
	} write[2][COUNT];
#endif /* CONFIG_NEED_WRITE */
	struct {
		vaddr_t vaddr;
		paddr_t paddr;
		int (*cf)(void *, paddr_t, unsigned int, udata_t *);
		void *cs;
		char *haddr;
	} read[2][COUNT];
} NAME;

#endif /* STATE */
#ifdef EXPORT

#if CONFIG_NEED_WRITE
/*forward*/ static int
NAME_(umw8)(struct cpssp *cpssp, vaddr_t addr, bool user,
		uint8_t val, uint16_t *merrp);
/*forward*/ static int
NAME_(umw16)(struct cpssp *cpssp, vaddr_t addr, bool user,
		uint16_t val, uint16_t *merrp);
/*forward*/ static int
NAME_(umw32)(struct cpssp *cpssp, vaddr_t addr, bool user,
		uint32_t val, uint16_t *merrp);
#if defined CONFIG_CPU_LM_SUPPORT && CONFIG_CPU_LM_SUPPORT
/*forward*/ static int
NAME_(umw64)(struct cpssp *cpssp, vaddr_t addr, bool user,
		uint64_t val, uint16_t *merrp);
#endif
#endif /* CONFIG_NEED_WRITE */
/*forward*/ static int
NAME_(umr8)(struct cpssp *cpssp, vaddr_t addr, bool user,
		uint8_t *valp, uint16_t *merrp, int w_test);
/*forward*/ static int
NAME_(umr16)(struct cpssp *cpssp, vaddr_t addr, bool user,
		uint16_t *valp, uint16_t *merrp, int w_test);
/*forward*/ static int
NAME_(umr32)(struct cpssp *cpssp, vaddr_t addr, bool user,
		uint32_t *valp, uint16_t *merrp, int w_test);
#if CONFIG_NEED_WRITE \
	|| (defined CONFIG_CPU_LM_SUPPORT && CONFIG_CPU_LM_SUPPORT)
/*forward*/ static int
NAME_(umr64)(struct cpssp *cpssp, vaddr_t addr, bool user,
		uint64_t *valp, uint16_t *merrp, int w_test);
#endif
#if 80486 <= CONFIG_CPU
/*forward*/ static void
NAME_(invlpg)(struct cpssp *cpssp, vaddr_t vaddr);
#endif
/*forward*/ static void
NAME_(tlb_flush)(struct cpssp *cpssp, int all);
/*forward*/ static void
NAME_(unmap)(struct cpssp *cpssp, paddr_t paddr, paddr_t len);

/*forward*/ static void
NAME_(reset)(struct cpssp *cpssp);
/*forward*/ static void
NAME_(create)(struct cpssp *cpssp);
/*forward*/ static void
NAME_(destroy)(struct cpssp *cpssp);

#endif /* EXPORT */
#ifdef BEHAVIOR

#if CONFIG_NEED_WRITE
static int
NAME_(umw_fill)(struct cpssp *cpssp, vaddr_t vaddr, bool user, uint16_t *merrp)
{
	unsigned int n;

	vaddr &= AMASK;

	n = (vaddr / LINE) % COUNT;

	if ((cpssp->NAME.write[user][n].vaddr & ~0x10) != vaddr) {
		int (*cf)(void *, paddr_t, unsigned int, udata_t);
		void *cs;
		paddr_t paddr;
		char *haddr;
		int ret;

		ret = NAME_(cache1_map_w)(cpssp, vaddr, user,
				&cf, &cs, &paddr, &haddr, merrp);
		if (ret) {
			return 1;
		}

		cpssp->NAME.write[user][n].paddr = paddr;
		cpssp->NAME.write[user][n].cf = cf;
		cpssp->NAME.write[user][n].cs = cs;
		cpssp->NAME.write[user][n].haddr = haddr;

		if (cpssp->NAME.write[user][n].haddr) {
			cpssp->NAME.write[user][n].vaddr = vaddr;
		} else {
			cpssp->NAME.write[user][n].vaddr = vaddr | 0x10;
		}
	}

	return 0;
}

static int
NAME_(umw_slow)(
	struct cpssp *cpssp,
	vaddr_t vaddr,
	bool user,
	unsigned int bs,
	udata_t val,
	uint16_t *merrp
)
{
	vaddr_t voff;
	unsigned int n;

	voff = vaddr & OMASK;
	vaddr &= AMASK;

	if (NAME_(umw_fill)(cpssp, vaddr, user, merrp)) {
		return 1;
	}

	n = (vaddr / LINE) % COUNT;

	if (likely(cpssp->NAME.write[user][n].vaddr == vaddr)) {
		char *haddr;

		haddr = cpssp->NAME.write[user][n].haddr + voff;
		if (8 <= sizeof(val)) {
			if ((bs >> 7) & 1) {
				haddr[7] = ((uint64_t) val >> 56) & 0xff;
			}
			if ((bs >> 6) & 1) {
				haddr[6] = ((uint64_t) val >> 48) & 0xff;
			}
			if ((bs >> 5) & 1) {
				haddr[5] = ((uint64_t) val >> 40) & 0xff;
			}
			if ((bs >> 4) & 1) {
				haddr[4] = ((uint64_t) val >> 32) & 0xff;
			}
		}
		if (4 <= sizeof(val)) {
			if ((bs >> 3) & 1) {
				haddr[3] = ((uint32_t) val >> 24) & 0xff;
			}
			if ((bs >> 2) & 1) {
				haddr[2] = ((uint32_t) val >> 16) & 0xff;
			}
		}
		if (2 <= sizeof(val)) {
			if ((bs >> 1) & 1) {
				haddr[1] = ((uint16_t) val >> 8) & 0xff;
			}
		}
		if (1 <= sizeof(val)) {
			if ((bs >> 0) & 1) {
				haddr[0] = ((uint8_t) val >> 0) & 0xff;
			}
		}

	} else {
		(*cpssp->NAME.write[user][n].cf)(
			cpssp->NAME.write[user][n].cs,
			cpssp->NAME.write[user][n].paddr | voff,
			bs, val);
	}

	return 0;
}

static int
NAME_(umw)(struct cpssp *cpssp, paddr_t vaddr, int user, int size, udata_t val, uint16_t *merrp)
{
	const int B = sizeof(udata_t);
	unsigned int bs;
	unsigned int offset;
	unsigned int count;
	udata_t tmp;

#if CONFIG_NEED_AC
	if (user
	 && NAME_(core_ac_get)(cpssp)
	 && vaddr % size != 0) {
		/* Alignment check error. */
		return 17;
	}
#endif

	count = 0;

	offset = vaddr & (B - 1);
	bs = ((1 << size) - 1) << (vaddr & (B - 1));
	vaddr &= ~(B - 1);

	while (bs) {
		tmp = val >> (8 * count);
		tmp <<= (8 * offset);
		if (NAME_(umw_slow)(cpssp, vaddr, user, bs & ((1 << B) - 1),
				tmp, merrp)) {
			/* Page fault. */
			NAME_(core_cr2_set)(cpssp, vaddr + offset);
			return 14;
		}
		count += B - offset;

		offset = 0;
		bs >>= B;
		vaddr += B;
	}

	return 0;
}

static int
NAME_(umw8)(
	struct cpssp *cpssp,
	vaddr_t vaddr,
	bool user,
	uint8_t val,
	uint16_t *merrp
)
{
	unsigned int n;
	int ret;

#if LOG_COUNT
	cpssp->NAME.umwb_count++;
#endif

#if DEBUG_MEM
	if (2 <= DEBUG_MEM + loglevel
	 && DEBUG_MEM_START <= vaddr && vaddr < DEBUG_MEM_END) {
		fprintf(stderr, "%s: %llu: 0x%08lx <- 0x%02x\n", __FUNCTION__,
				cpssp->process.inst_cnt, vaddr, val);
	}
#endif

	n = (vaddr / LINE) % COUNT;

	if (likely(cpssp->NAME.write[user][n].vaddr == (vaddr & AMASK))) {
		char *haddr;

		haddr = cpssp->NAME.write[user][n].haddr + (vaddr & OMASK);
		*(uint8_t *) haddr = val;

		ret = 0;

	} else {
		ret = NAME_(umw)(cpssp, vaddr, user, sizeof(val), val, merrp);
	}

	return ret;
}

static int
NAME_(umw16)(
	struct cpssp *cpssp,
	vaddr_t vaddr,
	bool user,
	uint16_t val,
	uint16_t *merrp
)
{
	unsigned int n;
	int ret;

#if LOG_COUNT
	cpssp->NAME.umww_count++;
#endif

#if DEBUG_MEM
	if (2 <= DEBUG_MEM + loglevel
	 && DEBUG_MEM_START <= vaddr && vaddr < DEBUG_MEM_END) {
		fprintf(stderr, "%s: %llu: 0x%08lx <- 0x%04x\n", __FUNCTION__,
				cpssp->process.inst_cnt, vaddr, val);
	}
#endif

	n = (vaddr / LINE) % COUNT;

	if (likely(cpssp->NAME.write[user][n].vaddr == (vaddr & (AMASK | 0x1)))) {
		char *haddr;

		haddr = cpssp->NAME.write[user][n].haddr + (vaddr & OMASK);
		*(uint16_t *) haddr = val;

		ret = 0;

	} else {
		ret = NAME_(umw)(cpssp, vaddr, user, sizeof(val), val, merrp);
	}

	return ret;
}

static int
NAME_(umw32)(
	struct cpssp *cpssp,
	vaddr_t vaddr,
	bool user,
	uint32_t val,
	uint16_t *merrp
)
{
	unsigned int n;
	int ret;

#if LOG_COUNT
	cpssp->NAME.umwd_count++;
#endif

#if DEBUG_MEM
	if (2 <= DEBUG_MEM + loglevel
	 && DEBUG_MEM_START <= vaddr && vaddr < DEBUG_MEM_END) {
		fprintf(stderr, "%s: %llu: 0x%08lx <- 0x%08x\n", __FUNCTION__,
				cpssp->process.inst_cnt, vaddr, val);
	}
#endif

	n = (vaddr / LINE) % COUNT;

	if (likely(cpssp->NAME.write[user][n].vaddr == (vaddr & (AMASK | 0x3)))) {
		char *haddr;

		haddr = cpssp->NAME.write[user][n].haddr + (vaddr & OMASK);
		*(uint32_t *) haddr = val;

		ret = 0;

	} else {
		ret = NAME_(umw)(cpssp, vaddr, user, sizeof(val), val, merrp);
	}

	return ret;
}

#if defined CONFIG_CPU_LM_SUPPORT && CONFIG_CPU_LM_SUPPORT
static int
NAME_(umw64)(
	struct cpssp *cpssp,
	vaddr_t vaddr,
	bool user,
	uint64_t val,
	uint16_t *merrp
)
{
	unsigned int n;
	int ret;

#if LOG_COUNT
	cpssp->NAME.umwq_count++;
#endif

#if DEBUG_MEM
	if (2 <= DEBUG_MEM + loglevel
	 && DEBUG_MEM_START <= vaddr && vaddr < DEBUG_MEM_END) {
		fprintf(stderr, "%s: %llu: 0x%08lx <- 0x%08x\n", __FUNCTION__,
				cpssp->process.inst_cnt, vaddr, val);
	}
#endif

	n = (vaddr / LINE) % COUNT;

	if (likely(cpssp->NAME.write[user][n].vaddr == (vaddr & (AMASK | 0x7)))) {
		char *haddr;

		haddr = cpssp->NAME.write[user][n].haddr + (vaddr & OMASK);
		*(uint64_t *) haddr = val;

		ret = 0;

	} else {
		ret = NAME_(umw)(cpssp, vaddr, user, sizeof(val), val, merrp);
	}

	return ret;
}
#endif
#endif /* CONFIG_NEED_WRITE */

static int
NAME_(umr_fill)(struct cpssp *cpssp, vaddr_t vaddr, bool user, uint16_t *merrp)
{
	unsigned int n;

	vaddr &= AMASK;

	n = (vaddr / LINE) % COUNT;

	if ((cpssp->NAME.read[user][n].vaddr & ~0x10) != vaddr) {
		int (*cf)(void *, paddr_t, unsigned int, udata_t *);
		void *cs;
		paddr_t paddr;
		char *haddr;
		int ret;

		ret = NAME_(cache1_map_r)(cpssp, vaddr, user,
				&cf, &cs, &paddr, &haddr, merrp);
		if (ret) {
			return 1;
		}

		cpssp->NAME.read[user][n].paddr = paddr;
		cpssp->NAME.read[user][n].cf = cf;
		cpssp->NAME.read[user][n].cs = cs;
		cpssp->NAME.read[user][n].haddr = haddr;

		if (cpssp->NAME.read[user][n].haddr) {
			cpssp->NAME.read[user][n].vaddr = vaddr;
		} else {
			cpssp->NAME.read[user][n].vaddr = vaddr | 0x10;
		}
	}

	return 0;
}

static int
NAME_(umr_slow)(
	struct cpssp *cpssp,
	vaddr_t vaddr,
	bool user,
	unsigned int bs,
	udata_t *valp,
	uint16_t *merrp,
	int w_test
)
{
	vaddr_t voff;
	unsigned int n;

	voff = vaddr & OMASK;
	vaddr &= AMASK;

#if CONFIG_NEED_WRITE
	if ((w_test & 2)
	 && NAME_(umw_fill)(cpssp, vaddr, user, merrp)) {
		return 1;
	}
#endif
	if (NAME_(umr_fill)(cpssp, vaddr, user, merrp)) {
		return 1;
	}
#if CONFIG_NEED_WRITE
	if ((w_test & 1)
	 && NAME_(umw_fill)(cpssp, vaddr, user, merrp)) {
		return 1;
	}
#endif

	n = (vaddr / LINE) % COUNT;

	if (likely(cpssp->NAME.read[user][n].vaddr == vaddr)) {
		char *haddr;

		haddr = cpssp->NAME.read[user][n].haddr + voff;
		*valp = *(udata_t *) haddr;

	} else {
		(*cpssp->NAME.read[user][n].cf)(
			cpssp->NAME.read[user][n].cs,
			cpssp->NAME.read[user][n].paddr | voff,
			bs, valp);
	}

	return 0;
}

static int
NAME_(umr)(
	struct cpssp *cpssp,
	paddr_t vaddr,
	int user,
	int size,
	udata_t *valp,
	uint16_t *merrp,
	int w_test
)
{
	const int B = sizeof(udata_t);
	udata_t val;
	unsigned int bs;
	unsigned int offset;
	unsigned int count;
	udata_t tmp;

#if CONFIG_NEED_AC
	if (user
	 && NAME_(core_ac_get)(cpssp)
	 && vaddr % size != 0) {
		/* Alignment check error. */
		return 17;
	}
#endif

	val = 0;
	count = 0;

	offset = vaddr & (B - 1);
	bs = ((1 << size) - 1) << (vaddr & (B - 1));
	vaddr &= ~(B - 1);

	while (bs) {
		if (NAME_(umr_slow)(cpssp, vaddr, user, bs & ((1 << B) - 1),
				&tmp, merrp, w_test)) {
			/* Page fault. */
			NAME_(core_cr2_set)(cpssp, vaddr + offset);
			return 14;
		}
		tmp >>= (8 * offset);
		val |= tmp << (8 * count);
		count += B - offset;

		offset = 0;
		bs >>= B;
		vaddr += B;
	}

	*valp = val;

	return 0;
}

static int
NAME_(umr8)(
	struct cpssp *cpssp,
	vaddr_t vaddr,
	bool user,
	uint8_t *valp,
	uint16_t *merrp,
	int w_test
)
{
	unsigned int n;
	int ret;

#if LOG_COUNT
	cpssp->NAME.umrb_count++;
#endif

	n = (vaddr / LINE) % COUNT;

	if (
#if CONFIG_NEED_WRITE
	 (! w_test || (cpssp->NAME.write[user][n].vaddr & AMASK) == (vaddr & AMASK))
	 &&
#endif
	 cpssp->NAME.read[user][n].vaddr == (vaddr & AMASK)) {
		char *haddr;

		haddr = cpssp->NAME.read[user][n].haddr + (vaddr & OMASK);
		*valp = *(uint8_t *) haddr;
		ret = 0;

	} else {
		udata_t val;

		ret = NAME_(umr)(cpssp, vaddr, user, sizeof(*valp), &val, merrp, w_test);

		*valp = (uint8_t) val;
	}

#if DEBUG_MEM
	if (2 <= DEBUG_MEM + loglevel
	 && DEBUG_MEM_START <= vaddr && vaddr < DEBUG_MEM_END) {
		fprintf(stderr, "%s: %llu: 0x%08lx -> 0x%02x\n", __FUNCTION__,
				cpssp->process.inst_cnt, vaddr, *valp);
	}
#endif

	return ret;
}

static int
NAME_(umr16)(
	struct cpssp *cpssp,
	vaddr_t vaddr,
	bool user,
	uint16_t *valp,
	uint16_t *merrp,
	int w_test
)
{
	unsigned int n;
	int ret;

#if LOG_COUNT
	cpssp->NAME.umrw_count++;
#endif

	n = (vaddr / LINE) % COUNT;

	if (
#if CONFIG_NEED_WRITE
	 (! w_test || (cpssp->NAME.write[user][n].vaddr & AMASK) == (vaddr & (AMASK | 0x1)))
	 &&
#endif
	 cpssp->NAME.read[user][n].vaddr == (vaddr & (AMASK | 0x1))) {
		char *haddr;

		haddr = cpssp->NAME.read[user][n].haddr + (vaddr & OMASK);
		*valp = *(uint16_t *) haddr;

		ret = 0;

	} else {
		udata_t val;

		ret = NAME_(umr)(cpssp, vaddr, user, sizeof(*valp), &val, merrp, w_test);

		*valp = (uint16_t) val;
	}

#if DEBUG_MEM
	if (2 <= DEBUG_MEM + loglevel
	 && DEBUG_MEM_START <= vaddr && vaddr < DEBUG_MEM_END) {
		fprintf(stderr, "%s: %llu: 0x%08lx -> 0x%04x\n", __FUNCTION__,
				cpssp->process.inst_cnt, vaddr, *valp);
	}
#endif

	return ret;
}

static int
NAME_(umr32)(
	struct cpssp *cpssp,
	vaddr_t vaddr,
	bool user,
	uint32_t *valp,
	uint16_t *merrp,
	int w_test
)
{
	unsigned int n;
	int ret;

#if LOG_COUNT
	cpssp->NAME.umrd_count++;
#endif

	n = (vaddr / LINE) % COUNT;

	if (
#if CONFIG_NEED_WRITE
	 (! w_test || (cpssp->NAME.write[user][n].vaddr & AMASK) == (vaddr & (AMASK | 0x3)))
	 &&
#endif
	 cpssp->NAME.read[user][n].vaddr == (vaddr & (AMASK | 0x3))) {
		char *haddr;

		haddr = cpssp->NAME.read[user][n].haddr + (vaddr & OMASK);
		*valp = *(uint32_t *) haddr;

		ret = 0;

	} else {
		udata_t val;

		ret = NAME_(umr)(cpssp, vaddr, user, sizeof(*valp), &val, merrp, w_test);

		*valp = (uint32_t) val;
	}

#if DEBUG_MEM
	if (2 <= DEBUG_MEM + loglevel
	 && DEBUG_MEM_START <= vaddr && vaddr < DEBUG_MEM_END) {
		fprintf(stderr, "%s: %llu: 0x%08lx -> 0x%08x\n", __FUNCTION__,
				cpssp->process.inst_cnt, vaddr, *valp);
	}
#endif

	return ret;
}

#if CONFIG_NEED_WRITE \
	|| (defined CONFIG_CPU_LM_SUPPORT && CONFIG_CPU_LM_SUPPORT)
static int
NAME_(umr64)(
	struct cpssp *cpssp,
	vaddr_t vaddr,
	bool user,
	uint64_t *valp,
	uint16_t *merrp,
	int w_test
)
{
	unsigned int n;
	int ret;

#if LOG_COUNT
	cpssp->NAME.umrq_count++;
#endif

	n = (vaddr / LINE) % COUNT;

	if (
#if CONFIG_NEED_WRITE
	 (! w_test || (cpssp->NAME.write[user][n].vaddr & AMASK) == (vaddr & (AMASK | 0x7)))
	 &&
#endif
	 cpssp->NAME.read[user][n].vaddr == (vaddr & (AMASK | 0x7))) {
		char *haddr;

		haddr = cpssp->NAME.read[user][n].haddr + (vaddr & OMASK);
		*valp = *(uint64_t *) haddr;

		ret = 0;

	} else if (sizeof(udata_t) == 4) {
		udata_t val1;
		udata_t val0;

		ret = NAME_(umr)(cpssp, vaddr + 0, user,
				sizeof(val0), &val0, merrp, w_test);
		if (! ret) {
			ret = NAME_(umr)(cpssp, vaddr + 4, user,
					sizeof(val1), &val1, merrp, w_test);
		}
		*valp = ((uint64_t) val1 << 32) | ((uint64_t) val0 << 0);

	} else if (sizeof(udata_t) == 8) {
		udata_t val0;

		ret = NAME_(umr)(cpssp, vaddr, user,
				sizeof(*valp), &val0, merrp, w_test);

		*valp = (uint64_t) val0;

	} else {
		assert(0); /* Mustn't happen. */
	}

#if DEBUG_MEM
	if (2 <= DEBUG_MEM + loglevel
	 && DEBUG_MEM_START <= vaddr && vaddr < DEBUG_MEM_END) {
		fprintf(stderr, "%s: %llu: 0x%08lx -> 0x%016x\n", __FUNCTION__,
				cpssp->process.inst_cnt, vaddr, *valp);
	}
#endif

	return ret;
}
#endif /* CONFIG_NEED_WRITE - FIXME */

#if 80486 <= CONFIG_CPU
static void
NAME_(invlpg)(struct cpssp *cpssp, vaddr_t vaddr)
{
	unsigned int n;
	unsigned int user;

	vaddr &= ~0xfff;

	for (n = 0; n < COUNT; n++) {
		for (user = 0; user < 2; user++) {
#if CONFIG_NEED_WRITE
			if (vaddr <= cpssp->NAME.write[user][n].vaddr
			 && cpssp->NAME.write[user][n].vaddr < vaddr + 0xfff) {
				cpssp->NAME.write[user][n].vaddr = -1;
			}
#endif
			if (vaddr <= cpssp->NAME.read[user][n].vaddr
			 && cpssp->NAME.read[user][n].vaddr < vaddr + 0xfff) {
				cpssp->NAME.read[user][n].vaddr = -1;
			}
		}
	}
}
#endif /* 80486 <= CONFIG_CPU */

static void
NAME_(tlb_flush)(struct cpssp *cpssp, int all)
{
	NAME_(reset)(cpssp); /* Obey "all"! FIXME */
}

static void
NAME_(unmap)(struct cpssp *cpssp, paddr_t paddr, paddr_t len)
{
	NAME_(reset)(cpssp); /* Obey "paddr"/"len"! FIXME */
}

static void
NAME_(reset)(struct cpssp *cpssp)
{
	unsigned int n;
	unsigned int user;

	for (n = 0; n < COUNT; n++) {
		for (user = 0; user < 2; user++) {
#if CONFIG_NEED_WRITE
			cpssp->NAME.write[user][n].vaddr = -1;
#endif
			cpssp->NAME.read[user][n].vaddr = -1;
		}
	}
}

static void
NAME_(create)(struct cpssp *cpssp)
{
#if LOG_COUNT
#if CONFIG_NEED_WRITE
	cpssp->NAME.umwb_count = 0;
	cpssp->NAME.umww_count = 0;
	cpssp->NAME.umwd_count = 0;
	cpssp->NAME.umwq_count = 0;
#endif
	cpssp->NAME.umrb_count = 0;
	cpssp->NAME.umrw_count = 0;
	cpssp->NAME.umrd_count = 0;
	cpssp->NAME.umrq_count = 0;
#endif /* LOG_COUNT */
}

static void
NAME_(destroy)(struct cpssp *cpssp)
{
#if LOG_COUNT
#if CONFIG_NEED_WRITE
	fprintf(stderr, "umwb: %llu\n", cpssp->NAME.umwb_count);
	fprintf(stderr, "umww: %llu\n", cpssp->NAME.umww_count);
	fprintf(stderr, "umwd: %llu\n", cpssp->NAME.umwd_count);
	fprintf(stderr, "umwq: %llu\n", cpssp->NAME.umwq_count);
#endif
	fprintf(stderr, "umrb: %llu\n", cpssp->NAME.umrb_count);
	fprintf(stderr, "umrw: %llu\n", cpssp->NAME.umrw_count);
	fprintf(stderr, "umrd: %llu\n", cpssp->NAME.umrd_count);
	fprintf(stderr, "umrq: %llu\n", cpssp->NAME.umrq_count);
#endif /* LOG_COUNT */
}

#endif /* BEHAVIOR */

#undef AMASK
#undef OMASK
#undef LINE

#undef COUNT

#undef LOG_COUNT
