/*
 * Copyright (C) 2004-2009 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#include <assert.h>
#include <stdio.h>

#include "glue.h"

#include "sig_isa_dma.h"

int
sig_isa_dma_req(
	struct sig_isa_dma *b,
	void *s
)
{
	unsigned int nr;
	int (*func)(void *);

	for (nr = 0; ; nr++) {
		if (nr == b->member_count) {
			return -1;
		}
		if (b->member[nr].s == s) {
			continue;
		}
		func = b->member[nr].f->req;
		if (func
		 && func(b->member[nr].s) == 0) {
			return 0;
		}
	}
}

int
sig_isa_dma_ack_in(
	struct sig_isa_dma *b,
	void *s,
	uint32_t addr,
	unsigned int tc
)
{
	unsigned int nr;

	for (nr = 0; ; nr++) {
		int (*func)(void *, uint32_t, unsigned int);

		if (nr == b->member_count) {
			return -1;
		}
		if (b->member[nr].s == s) {
			continue;
		}
		func = b->member[nr].f->ack_in;
		if (func
		 && func(b->member[nr].s, addr, tc) == 0) {
			return 0;
		}
	}
}

int
sig_isa_dma_ack_out(
	struct sig_isa_dma *b,
	void *s,
	uint32_t addr,
	unsigned int tc
)
{
	unsigned int nr;

	for (nr = 0; ; nr++) {
		int (*func)(void *, uint32_t, unsigned int);

		if (nr == b->member_count) {
			return -1;
		}
		if (b->member[nr].s == s) {
			continue;
		}
		func = b->member[nr].f->ack_out;
		if (func
		 && func(b->member[nr].s, addr, tc) == 0) {
			return 0;
		}
	}
}

int
sig_isa_dma_ack_verify(
	struct sig_isa_dma *b,
	void *s,
	unsigned int tc
)
{
	unsigned int nr;

	for (nr = 0; ; nr++) {
		int (*func)(void *, unsigned int);

		if (nr == b->member_count) {
			return -1;
		}
		if (b->member[nr].s == s) {
			continue;
		}
		func = b->member[nr].f->ack_verify;
		if (func
		 && func(b->member[nr].s, tc) == 0) {
			return 0;
		}
	}
}

int
sig_isa_dma_ack_inb(
	struct sig_isa_dma *b,
	void *s,
	unsigned int tc,
	uint8_t *valp
)
{
	unsigned int nr;

	for (nr = 0; ; nr++) {
		int (*func)(void *, unsigned int, uint8_t *);

		if (nr == b->member_count) {
			*valp = -1;
			return -1;
		}
		if (b->member[nr].s == s) {
			continue;
		}
		func = b->member[nr].f->ack_inb;
		if (func && func(b->member[nr].s, tc, valp) == 0) {
			return 0;
		}
	}
}

int
sig_isa_dma_ack_inw(
	struct sig_isa_dma *b,
	void *s,
	unsigned int tc,
	uint16_t *valp
)
{
	unsigned int nr;

	for (nr = 0; ; nr++) {
		int (*func)(void *, unsigned int, uint16_t *);

		if (nr == b->member_count) {
			*valp = -1;
			return -1;
		}
		if (b->member[nr].s == s) {
			continue;
		}
		func = b->member[nr].f->ack_inw;
		if (func && func(b->member[nr].s, tc, valp) == 0) {
			return 0;
		}
	}
}

int
sig_isa_dma_ack_outb(
	struct sig_isa_dma *b,
	void *s,
	unsigned int tc,
	uint8_t val
)
{
	unsigned int nr;

	for (nr = 0; ; nr++) {
		int (*func)(void *, unsigned int, uint8_t);

		if (nr == b->member_count) {
			return -1;
		}
		if (b->member[nr].s == s) {
			continue;
		}
		func = b->member[nr].f->ack_outb;
		if (func && func(b->member[nr].s, tc, val) == 0) {
			return 0;
		}
	}
}

int
sig_isa_dma_ack_outw(
	struct sig_isa_dma *b,
	void *s,
	unsigned int tc,
	uint16_t val
)
{
	unsigned int nr;

	for (nr = 0; ; nr++) {
		int (*func)(void *, unsigned int, uint16_t);

		if (nr == b->member_count) {
			return -1;
		}
		if (b->member[nr].s == s) {
			continue;
		}
		func = b->member[nr].f->ack_outw;
		if (func && func(b->member[nr].s, tc, val) == 0) {
			return 0;
		}
	}
}

int
sig_isa_dma_ack_verifyb(
	struct sig_isa_dma *b,
	void *s,
	unsigned int tc
)
{
	unsigned int nr;

	for (nr = 0; ; nr++) {
		int (*func)(void *, unsigned int);

		if (nr == b->member_count) {
			return -1;
		}
		if (b->member[nr].s == s) {
			continue;
		}
		func = b->member[nr].f->ack_verifyb;
		if (func
		 && func(b->member[nr].s, tc) == 0) {
			return 0;
		}
	}
}

int
sig_isa_dma_ack_verifyw(
	struct sig_isa_dma *b,
	void *s,
	unsigned int tc
)
{
	unsigned int nr;

	for (nr = 0; ; nr++) {
		int (*func)(void *, unsigned int);

		if (nr == b->member_count) {
			return -1;
		}
		if (b->member[nr].s == s) {
			continue;
		}
		func = b->member[nr].f->ack_verifyw;
		if (func
		 && func(b->member[nr].s, tc) == 0) {
			return 0;
		}
	}
}

void
sig_isa_dma_connect(
	struct sig_isa_dma *b,
	void *s,
	const struct sig_isa_dma_funcs *f
)
{
	assert(b);
	assert(b->type == SIG_GEN_ISA_DMA);
	assert(b->member_count < sizeof(b->member) / sizeof(b->member[0]));

	b->member[b->member_count].s = s;
	b->member[b->member_count].f = f;
	b->member_count++;
}

static int
sig_isa_dma_s0_req(void *_f)
{
	struct sig_isa_dma_merge *f
			= (struct sig_isa_dma_merge *) _f;

	return sig_isa_dma_req(f->s1, f);
}

static int
sig_isa_dma_s0_ack_inb(void *_f, unsigned int tc, uint8_t *valp)
{
	struct sig_isa_dma_merge *f
			= (struct sig_isa_dma_merge *) _f;

	return sig_isa_dma_ack_inb(f->s1, f, tc, valp);
}

static int
sig_isa_dma_s0_ack_outb(void *_f, unsigned int tc, uint8_t val)
{
	struct sig_isa_dma_merge *f
			= (struct sig_isa_dma_merge *) _f;

	return sig_isa_dma_ack_outb(f->s1, f, tc, val);
}

static int
sig_isa_dma_s0_ack_inw(void *_f, unsigned int tc, uint16_t *valp)
{
	struct sig_isa_dma_merge *f
			= (struct sig_isa_dma_merge *) _f;

	return sig_isa_dma_ack_inw(f->s1, f, tc, valp);
}

static int
sig_isa_dma_s0_ack_outw(void *_f, unsigned int tc, uint16_t val)
{
	struct sig_isa_dma_merge *f
			= (struct sig_isa_dma_merge *) _f;

	return sig_isa_dma_ack_outw(f->s1, f, tc, val);
}

static int
sig_isa_dma_s1_req(void *_f)
{
	struct sig_isa_dma_merge *f
			= (struct sig_isa_dma_merge *) _f;

	return sig_isa_dma_req(f->s0, f);
}

static int
sig_isa_dma_s1_ack_inb(void *_f, unsigned int tc, uint8_t *valp)
{
	struct sig_isa_dma_merge *f
			= (struct sig_isa_dma_merge *) _f;

	return sig_isa_dma_ack_inb(f->s0, f, tc, valp);
}

static int
sig_isa_dma_s1_ack_outb(void *_f, unsigned int tc, uint8_t val)
{
	struct sig_isa_dma_merge *f
			= (struct sig_isa_dma_merge *) _f;

	return sig_isa_dma_ack_outb(f->s0, f, tc, val);
}

static int
sig_isa_dma_s1_ack_inw(void *_f, unsigned int tc, uint16_t *valp)
{
	struct sig_isa_dma_merge *f
			= (struct sig_isa_dma_merge *) _f;

	return sig_isa_dma_ack_inw(f->s0, f, tc, valp);
}

static int
sig_isa_dma_s1_ack_outw(void *_f, unsigned int tc, uint16_t val)
{
	struct sig_isa_dma_merge *f
			= (struct sig_isa_dma_merge *) _f;

	return sig_isa_dma_ack_outw(f->s0, f, tc, val);
}

struct sig_isa_dma_merge *
sig_isa_dma_merge(
	struct sig_isa_dma *s0,
	struct sig_isa_dma *s1
)
{
	static const struct sig_isa_dma_funcs s0_funcs = {
		.req = sig_isa_dma_s0_req,
		.ack_inb = sig_isa_dma_s0_ack_inb,
		.ack_outb = sig_isa_dma_s0_ack_outb,
		.ack_inw = sig_isa_dma_s0_ack_inw,
		.ack_outw = sig_isa_dma_s0_ack_outw,
	};
	static const struct sig_isa_dma_funcs s1_funcs = {
		.req = sig_isa_dma_s1_req,
		.ack_inb = sig_isa_dma_s1_ack_inb,
		.ack_outb = sig_isa_dma_s1_ack_outb,
		.ack_inw = sig_isa_dma_s1_ack_inw,
		.ack_outw = sig_isa_dma_s1_ack_outw,
	};
	struct sig_isa_dma_merge *m;

	m = shm_alloc(sizeof(*m));
	assert(m);

	m->s0 = s0;
	sig_isa_dma_connect(s0, m, &s0_funcs);
	m->s1 = s1;
	sig_isa_dma_connect(s1, m, &s1_funcs);

	return m;
}

void
sig_isa_dma_split(struct sig_isa_dma_merge *m)
{
	fixme();
}

struct sig_isa_dma *
sig_isa_dma_create(const char *name)
{
	struct sig_isa_dma *sig;

	sig = shm_alloc(sizeof(*sig));
	assert(sig);

	sig->type = SIG_GEN_ISA_DMA;
	sig->member_count = 0;

	return sig;
}

void
sig_isa_dma_destroy(struct sig_isa_dma *sig)
{
	assert(sig);
	assert(sig->type == SIG_GEN_ISA_DMA);

	shm_free(sig);
}

void
sig_isa_dma_suspend(struct sig_isa_dma *b, FILE *fSig)
{
	size_t size = sizeof(*b);
	
	generic_suspend(b, size, fSig);
}

void
sig_isa_dma_resume(struct sig_isa_dma *b, FILE *fSig)
{
	size_t size = sizeof(*b);
	
	generic_resume(b, size, fSig);
}
