/************************************************************************
 ************************************************************************
    FAUST compiler
	Copyright (C) 2003-2004 GRAME, Centre National de Creation Musicale
    ---------------------------------------------------------------------
    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 ************************************************************************
 ************************************************************************/
 
 
 
#include "propagate.hh"
#include "prim2.hh"
#include <assert.h>

////////////////////////////////////////////////////////////////////////
/**
 * propagate : box listOfSignal-> listOfSignal'
 *
 * Propage une liste de signaux de l'entre vers la sortie d'une boite
 * La boite  t annote aec son type 
 */
///////////////////////////////////////////////////////////////////////

//! mix une liste de signaux sur n bus				
siglist mix(const siglist& lsig, int nbus)
{
	int nlines	= lsig.size();
	
	siglist dst(nbus);
	
	for (int b=0; b<nbus; b++) {
		Tree t = (b<nlines) ? lsig[b] : sigInt(0);
		for (int i=b+nbus; i<nlines; i+=nbus) {
			t = sigAdd(t, lsig[i]);
		}
		dst[b] = t;
	}
	return dst;
}			

//! split une liste de signaux sur n bus				
siglist split(const siglist& inputs, int nbus)
{
	int nlines	= inputs.size();
	
	siglist outputs(nbus);
	
	for (int b=0; b<nbus; b++) {
		outputs[b] = inputs[b % nlines];
	}
	return outputs;
}			

//! Fabrique une liste de n projections d'un groupe rcursif
siglist makeSigProjList (Tree t, int n)
{
	siglist l(n);
	for (int i = 0; i < n; i++) l[i] = sigProj(i, t);
	return l;
}


//! Fabrique une liste de n entres
siglist makeSigInputList (int n)
{
	siglist l(n);
	for (int i = 0; i < n; i++) l[i] = sigInput(i);
	return l;
}

inline siglist makeList(Tree t)
{
	siglist l(1);
	l[0] = t;
	return l;
}

siglist listRange(const siglist& l, int i, int j)
{
	siglist r(j-i);
	for (int x = i; x < j; x++) r[x-i] = l[x];
	return r;
}

siglist listConcat(const siglist& a, const siglist& b)
{
	int n1 = a.size();
	int n2 = b.size();
	siglist r(n1+n2);
	
	for (int x=0; x<n1; x++) r[x] = a[x];
	for (int x=0; x<n2; x++) r[x+n1] = b[x];
	return r;
}

Tree listConvert(const siglist& a)
{
	int 	n = a.size();
	Tree 	t=nil;
	while (n--) t = cons(a[n],t);
	return t;
}

// siglist listConvertBack(Tree l)
// {
// 	siglist r;
// 	while (!isNil(l)) { r.push_back(hd(l)); l = tl(l); }
// 	return r;
// }

siglist listLift(const siglist& l)
{
	int 		n = l.size();
	siglist		r(n);
	
	for(int i = 0; i<n; i++) r[i] = lift(l[i]);
	return r;
}



/**
 * Propagate computes the outputs signals of a block-diagram according to a list of input signals.
 *
 *\param slotenv input signals associated with symbolic slots
 *\param path stack of user interface groups : (type,label)*
 *\param box block-diagram where we propagate the signals
 *\param lsig list of signals to be propagated into box
 *\return list of resulting signals
 */

siglist propagate (Tree slotenv, Tree path, Tree box, const siglist&  lsig)
{
	int		i;
	float	r;
	prim0	p0;
	prim1	p1;
	prim2	p2;
	prim3	p3;
	prim4	p4;
	prim5	p5;
	
	Tree	t1, t2, ff, label, cur, min, max, step, type, name, file, slot, body;
	
	// numbers and constants
	if (isBoxInt(box, &i)) 	{ 
		assert(lsig.size()==0); 
		return makeList(sigInt(i)); 
	}
	else if (isBoxReal(box, &r)) 	{ 
		assert(lsig.size()==0); 
		return makeList(sigReal(r)); 
	}
	
	else if (isBoxFConst(box, type, name, file)) 	{ 
		assert(lsig.size()==0); 
		return makeList(sigFConst(type, name, file)); 
	}
	
	// wire and cut
	else if (isBoxCut(box)) 				{ 
		assert(lsig.size()==1); 
		return siglist(); 
	}
	
	else if (isBoxWire(box)) 				{ 
		assert(lsig.size()==1); 
		return lsig;  
	}
	
	// slots and symbolic boxes
	else if (isBoxSlot(box)) 				{ 
		Tree sig;
		assert(lsig.size()==0); 
		if (!searchEnv(box,sig,slotenv)) {
			fprintf(stderr, "propagate : internal error (slot undefined)\n");
			exit(1);
		}
		return makeList(sig);
	}
	
	// slots and symbolic boxes
	else if (isBoxSymbolic(box, slot, body)) 				{ 
		assert(lsig.size()>0); 
		return propagate(pushEnv(slot,lsig[0],slotenv), path, body, listRange(lsig, 1, lsig.size()));
	}
	
	// primitives
	else if (isBoxPrim0(box, &p0)) 			{ 
		assert(lsig.size()==0); 
		return makeList( p0() );  
	}
	
	else if (isBoxPrim1(box, &p1)) 				{ 
		assert(lsig.size()==1); 
		return makeList( p1(lsig[0]) );  
	}
	
	else if (isBoxPrim2(box, &p2)) 				{ 
//		printf("prim2 recoit : "); print(lsig); printf("\n");
		assert(lsig.size()==2); 
		return makeList( p2(lsig[0],lsig[1]) );  
	}
	
	else if (isBoxPrim3(box, &p3)) 				{ 
		assert(lsig.size()==3); 
		return makeList( p3(lsig[0],lsig[1],lsig[2]) );  
	}
	
	else if (isBoxPrim4(box, &p4)) 				{ 
		assert(lsig.size()==4); 
		return makeList( p4(lsig[0],lsig[1],lsig[2],lsig[3]) );  
	}
	
	else if (isBoxPrim5(box, &p5)) 				{ 
		assert(lsig.size()==5); 
		return makeList( p5(lsig[0],lsig[1],lsig[2],lsig[3],lsig[4]) );  
	}
	
	else if (isBoxFFun(box, ff)) 				{ 
		//printf("propagate en boxFFun of arity %d\n", ffarity(ff));
		assert(int(lsig.size())==ffarity(ff)); 
		return makeList(sigFFun(ff, listConvert(lsig)));  
	}
	
	// user interface
	else if (isBoxButton(box, label)) 	{ 
		assert(lsig.size()==0); 
		return makeList(sigButton(cons(label, path))); 
	}
	
	else if (isBoxCheckbox(box, label)) 	{ 
		assert(lsig.size()==0); 
		return makeList(sigCheckbox(cons(label, path))); 
	}
	
	else if (isBoxVSlider(box, label, cur, min, max, step)) 	{ 
		assert(lsig.size()==0); 
		return makeList(sigVSlider(cons(label, path), cur, min, max, step)); 
	}
	
	else if (isBoxHSlider(box, label, cur, min, max, step)) 	{ 
		assert(lsig.size()==0); 
		return makeList(sigHSlider(cons(label, path), cur, min, max, step)); 
	}
	
	else if (isBoxNumEntry(box, label, cur, min, max, step)) 	{ 
		assert(lsig.size()==0); 
		return makeList(sigNumEntry(cons(label, path), cur, min, max, step)); 
	}
	
	else if (isBoxVGroup(box, label, t1)) 	{ 
		return propagate(slotenv,cons(cons(tree(0),label), path), t1, lsig); 
	}
	
	else if (isBoxHGroup(box, label, t1)) 	{ 
		return propagate(slotenv, cons(cons(tree(1),label), path), t1, lsig); 
	}
	
	else if (isBoxTGroup(box, label, t1)) 	{ 
		return propagate(slotenv, cons(cons(tree(2),label), path), t1, lsig); 
	}
	
	else if (isBoxVBargraph(box, label, min, max)) 	{ 
		assert(lsig.size()==1); 
		return makeList(sigVBargraph(cons(label, path), min, max, lsig[0])); 
	}
	
	else if (isBoxHBargraph(box, label, min, max)) 	{ 
		assert(lsig.size()==1); 
		return makeList(sigHBargraph(cons(label, path), min, max, lsig[0])); 
	}
	
	// propagation dans les constructeurs
	else if (isBoxSeq(box, t1, t2)) 	{ 
		int in1, out1, in2, out2;
		getBoxType(t1, &in1, &out1);
		getBoxType(t2, &in2, &out2);
			
		if (out1 == in2) {
			return propagate(slotenv, path, t2, propagate(slotenv, path,t1,lsig));
		} else if (out1 > in2) {
			siglist lr = propagate(slotenv, path, t1,lsig);
			return listConcat(propagate(slotenv, path, t2, listRange(lr, 0, in2)), listRange(lr, in2, out1));
		} else {
			return propagate(slotenv, path, t2, listConcat( propagate(slotenv, path, t1, listRange(lsig,0,in1)), listRange(lsig,in1,in1+in2-out1) ) );
		}
	}
	
	else if (isBoxPar(box, t1, t2)) 	{ 
		int in1, out1, in2, out2;
		getBoxType(t1, &in1, &out1);
		getBoxType(t2, &in2, &out2);
			
		return listConcat(	propagate(slotenv, path, t1, listRange(lsig, 0,  in1)), 
							propagate(slotenv, path, t2, listRange(lsig, in1, in1+in2)) );
	}
	
	else if (isBoxSplit(box, t1, t2)) 	{ 
		int in1, out1, in2, out2;
		getBoxType(t1, &in1, &out1);
		getBoxType(t2, &in2, &out2);
		
		siglist l1 = propagate(slotenv, path, t1, lsig);
		siglist l2 = split(l1, in2);
		return propagate(slotenv, path, t2, l2);
	}
	
	else if (isBoxMerge(box, t1, t2)) 	{ 
		int in1, out1, in2, out2;
		getBoxType(t1, &in1, &out1);
		getBoxType(t2, &in2, &out2);
		
		siglist l1 = propagate(slotenv, path, t1, lsig);
		siglist l2 = mix(l1, in2);
		return propagate(slotenv, path, t2, l2);
	}
	
	else if (isBoxRec(box, t1, t2)) 	{ 
		int in1, out1, in2, out2;
		getBoxType(t1, &in1, &out1);
		getBoxType(t2, &in2, &out2);
		
		siglist l0 = makeSigProjList(ref(1), in2);
		siglist l1 = propagate(slotenv, path, t2, l0);
		siglist l2 = propagate(slotenv, path, t1, listConcat(l1,listLift(lsig)));
		Tree g = rec(listConvert(l2));
		return makeSigProjList(g, out1);
	}

	fprintf(stderr, "propagation error\n");
	exit(1);
	return siglist();
}

	
Tree boxPropagateSig (Tree path, Tree box, const siglist& lsig)
{
	return listConvert(propagate(nil, path, box, lsig));
}

