/*
 * Copyright (C) 1999-2002 Chris Ross
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * o Redistributions of source code must retain the above copyright notice, this
 *   list of conditions and the following disclaimer.
 * o Redistributions in binary form must reproduce the above copyright notice,
 *   this list of conditions and the following disclaimer in the documentation
 *   and/or other materials provided with the distribution.
 * o Neither the name of the ferite software nor the names of its contributors may
 *   be used to endorse or promote products derived from this software without
 *   specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#ifndef __APHEX_H__
# define __APHEX_H__

# include <stdio.h>
# include <stdarg.h>
/*#include "../../../config.h"*/
#include <limits.h>
# ifndef WIN32
#  define DIR_DELIM '/'
# else
#  define DIR_DELIM '\\'
# endif

# ifndef PATH_MAX
#  define PATH_MAX 4096
# endif

# ifdef USE_PTHREAD
#  include <pthread.h>
# endif

typedef struct __aphex_file
{
    FILE *handle;
    char *filename;
    long  size;
}
AphexFile;

typedef struct __aphex_directoy
{
    char **contents;
    long   size;
}
AphexDirectory;

typedef struct __aphex_search_list
{
    char **path;
    long   size;
    long   next_free;
}
AphexSearchList;

typedef struct __aphex_thread
{
# ifdef USE_PTHREAD
    pthread_t       thread;
    pthread_attr_t  attrs;
    int             running;
# endif
}
AphexThread;

typedef struct __aphex_mutex
{
# ifdef USE_PTHREAD
    pthread_mutex_t     mutex;
    pthread_mutexattr_t attr;
#  if defined(USING_DARWIN) || defined(USING_CYGWIN) || defined(USING_SOLARIS)
    int             count;
    pthread_t       owner;
    pthread_cond_t  cond;
#  endif
# endif
    int recursive;
}
AphexMutex;

/* Disk Stuff */

char            *aphex_file_exists_path( char *pf, ... );
int              aphex_file_exists( char *pf, ... );
char            *aphex_file_to_string( char *file );

AphexFile       *aphex_open_file( char *filename, char *mode, AphexSearchList *paths );
void             aphex_close_file( AphexFile *file );
size_t           aphex_read_file( AphexFile *file, char *buffer, size_t length );

AphexSearchList *aphex_create_search_list();
void             aphex_add_to_list( AphexSearchList *list, char *path );
void             aphex_delete_search_list( AphexSearchList *list );

char            *aphex_get_home_directory();
char            *aphex_get_temp_directory();

AphexDirectory  *aphex_directory_read( char *path );
AphexDirectory  *aphex_directory_read_with_filter( char *path, char *filter );
AphexDirectory  *aphex_directory_read_unique_prefix( char *path );
AphexDirectory  *aphex_directory_read_subdirs( char *path );
void             aphex_directory_delete( AphexDirectory *dir );
char            *aphex_relative_to_absolute( char *path );

/* Configuration Reading Stuff */

void             aphex_configuration_load();
void             aphex_configuration_store();
char            *aphex_get_string( char *path );
void             aphex_set_string( char *path, char *value );
long             aphex_get_long( char *path );
void             aphex_set_long( char *path, long value );

/* thread stuff */

AphexThread     *aphex_thread_create();
void             aphex_thread_destroy( AphexThread *thread );
int              aphex_thread_start( AphexThread *thread, void *(*start_routine)(void *), void *arg, int detach );
void             aphex_thread_stop( AphexThread *thread );
void             aphex_thread_sleep( AphexThread *thread, int msecs );
void             aphex_thread_join( AphexThread *thread );
void             aphex_thread_exit( AphexThread *thread, void *rval );

AphexMutex      *aphex_mutex_create();
AphexMutex      *aphex_mutex_recursive_create();
void             aphex_mutex_destroy( AphexMutex *mutex );
int              aphex_mutex_lock( AphexMutex *mutex );
int              aphex_mutex_unlock( AphexMutex *mutex );

/* notify/poll stuff */

# define DEFAULT_TIMEOUT 100

int aphex_notify_can_read_with_timeout(int fd, int wait, int ce);
int aphex_notify_can_read(int fd, int ce);

#endif /* __APHEX_H__ */
