/*
 * Copyright (C) 2000-2002 Chris Ross and various contributors
 * Copyright (C) 1999-2000 Chris Ross
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * o Redistributions of source code must retain the above copyright notice, this
 *   list of conditions and the following disclaimer.
 * o Redistributions in binary form must reproduce the above copyright notice,
 *   this list of conditions and the following disclaimer in the documentation
 *   and/or other materials provided with the distribution.
 * o Neither the name of the ferite software nor the names of its contributors may
 *   be used to endorse or promote products derived from this software without
 *   specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

/**
 * !group Compiler
 * !description Not all the compiler functions are documented, this is because they are internal
 *              and dont concern anyone but the compiler hackers. If you are a compiler hacker
 *              you should not need function descriptions :)
 */

#include "ferite.h"
#include <setjmp.h>
#include "aphex.h"

/* these are used to keep track of verious bit's and pieces, and makes
 * it possible to compile ferite in one pass rather than several passes
 * like other langyages :) */
FeriteStack *ferite_fwd_look_stack = NULL;
FeriteStack *ferite_bck_look_stack = NULL;
FeriteStack *ferite_break_look_stack = NULL;
FeriteStack *ferite_compiled_arrays_stack = NULL;
FeriteStack *ferite_argcount_stack = NULL;

/* these are for tracking the index of the local variables. and also allowing us to block-nested-variables! */
FeriteHash *ferite_local_variable_hash = NULL;

/* mwhahhaa, we can keep track of these things :) */
FeriteStack *ferite_compile_stack = NULL;
FeriteCompileRecord *ferite_current_compile = NULL;

/* make life easier :) */
#define CURRENT_NAMESPACE ferite_current_compile->ns
#define CURRENT_FUNCTION  ferite_current_compile->function
#define CURRENT_CLASS     ferite_current_compile->cclass
#define CURRENT_VARS      ferite_current_compile->variables
#define CURRENT_SCRIPT    ferite_current_compile->script
#define CURRENT_VARS_HASH ferite_local_variable_hash

/* stolen from ferite_scanner.l -> so we can report errors on files */
extern char   *ferite_scanner_file;
extern int     ferite_scanner_lineno;
int            ferite_compile_error = 0;
jmp_buf        ferite_compiler_jmpback;
int            ferite_keep_native_function_data = 0;
int            ferite_compiler_current_block_depth = 0;
AphexMutex    *ferite_compiler_lock = NULL;

void ferite_init_compiler()
{
    FE_ENTER_FUNCTION;
    if( ferite_compiler_lock == NULL )
      ferite_compiler_lock = aphex_mutex_recursive_create();
    FE_LEAVE_FUNCTION( NOWT );
}

void ferite_deinit_compiler()
{
    FE_ENTER_FUNCTION;
    if( ferite_compiler_lock != NULL )
    {
        aphex_mutex_destroy( ferite_compiler_lock );
        ferite_compiler_lock = NULL;
    }
    FE_LEAVE_FUNCTION( NOWT );
}

FeriteBkRequest *ferite_create_request( FeriteOp *op, int type )
{
    FeriteBkRequest *ptr = NULL;

    FE_ENTER_FUNCTION;
    ptr = fmalloc( sizeof( FeriteBkRequest ) );
    ptr->reqop = op;
    ptr->type = type;
    FE_LEAVE_FUNCTION( ptr );
}

void ferite_destroy_request( FeriteBkRequest *ptr )
{
    FE_ENTER_FUNCTION;
    ffree( ptr );
    FE_LEAVE_FUNCTION( NOWT );
}

void ferite_delete_request_stack( FeriteStack *stack )
{
    int i;
    FE_ENTER_FUNCTION;
    for( i = 0; i <= stack->stack_ptr; i++ )
    {
        if( stack->stack[i] != NULL )
        {
            ffree( stack->stack[i] );
        }
    }
    ffree( stack->stack );
    ffree( stack );
    FE_LEAVE_FUNCTION( NOWT );
}

void ferite_start_compiler( FeriteScript *new_script )
{
    FE_ENTER_FUNCTION;
    aphex_mutex_lock( ferite_compiler_lock );
    ferite_current_compile = fmalloc( sizeof( FeriteCompileRecord ) );
    CURRENT_SCRIPT = new_script;
    CURRENT_CLASS = NULL;
    CURRENT_VARS = NULL;
    CURRENT_FUNCTION = NULL;
    CURRENT_NAMESPACE = new_script->mainns;

    ferite_fwd_look_stack = ferite_create_stack( new_script, FE_COMPILER_INTERNAL_STACK_SIZE );
    ferite_bck_look_stack = ferite_create_stack( new_script, FE_COMPILER_INTERNAL_STACK_SIZE );
    ferite_break_look_stack = ferite_create_stack( new_script, FE_COMPILER_INTERNAL_STACK_SIZE );
    ferite_compile_stack = ferite_create_stack( new_script, FE_COMPILER_INTERNAL_STACK_SIZE );
    ferite_compiled_arrays_stack = ferite_create_stack( new_script, FE_COMPILER_INTERNAL_STACK_SIZE );
    ferite_argcount_stack = ferite_create_stack( new_script, FE_COMPILER_INTERNAL_STACK_SIZE );
    FE_LEAVE_FUNCTION(NOWT);
}

void ferite_clean_compiler()
{
    FE_ENTER_FUNCTION;
    ferite_clean_parser();
    ferite_delete_request_stack( ferite_fwd_look_stack );
    ferite_delete_request_stack( ferite_bck_look_stack );
    ferite_delete_request_stack( ferite_break_look_stack );
    ferite_delete_request_stack( ferite_compile_stack );
    ferite_delete_request_stack( ferite_argcount_stack );
    ferite_delete_stack( NULL, ferite_compiled_arrays_stack );
    ffree( ferite_current_compile );
    ferite_fwd_look_stack = NULL;
    ferite_bck_look_stack = NULL;
    ferite_break_look_stack = NULL;
    ferite_compile_stack = NULL;
    ferite_compiled_arrays_stack = NULL;
    ferite_argcount_stack = NULL;
    aphex_mutex_unlock( ferite_compiler_lock );
    FE_LEAVE_FUNCTION( NOWT );
}

/**
 * !function ferite_compiler_keep_native_code
 * !declaration void ferite_compiler_keep_native_code()
 * !brief Tell ferite to keep the native code that is found within the compiled scripts
 */
void ferite_compiler_keep_native_code()
{
    FE_ENTER_FUNCTION;
    ferite_keep_native_function_data = 1;
    FE_LEAVE_FUNCTION(NOWT);
}

char *ferite_compiler_build_current_path()
{
    int i;
    static char nameBuffer[1024];
    char *sig = NULL;
    FeriteCompileRecord *c = NULL;

    FE_ENTER_FUNCTION;
    nameBuffer[0] = '\0';
    for( i = 1; i <= ferite_compile_stack->stack_ptr; i++ )
    {
        c = ferite_compile_stack->stack[i];
        if( c != NULL )
        {
            if( c->function != NULL )
            {
                FUD(("strcat'ing function\n" ));
                strcat( nameBuffer, c->function->name );
            }
            else if( c->ns != NULL )
            {
                if( c->ns->name == NULL ) continue;
                FUD(("strcat'ing namespace name\n" ));
                strcat( nameBuffer, c->ns->name );
            }
            else if( c->cclass != NULL )
            {
                FUD(("strcat'ing class name\n" ));
                strcat( nameBuffer, c->cclass->name );
            }
            strcat( nameBuffer, "." );
        }
    }
    if( CURRENT_FUNCTION != NULL )
    {
        strcat( nameBuffer, CURRENT_FUNCTION->name );
        strcat( nameBuffer, "_" );
        
        sig = ferite_function_generate_sig_string( CURRENT_SCRIPT, CURRENT_FUNCTION );
        strcat( nameBuffer, sig );
        ffree( sig );        
    }
    FE_LEAVE_FUNCTION( nameBuffer );
}

/**
 * !function ferite_script_include
 * !declaration FeriteVariable *ferite_script_include( FeriteScript *script, char *filename )
 * !brief This is used to include another script at runtime
 * !description This is a runtime equivelent to:<nl/>
 * <nl/>
 * uses "foo";<nl/>
 * <nl/>
 * !param FeriteScript *script The script to include into
 * !param char *filename The name of the script to include
 * !return The value that is returned after executing the 'main' body of the included script
 */
FeriteVariable *ferite_script_include( FeriteScript *script, char *filename )
{
    FeriteScript *ptr = NULL;
    FeriteFunction *function = NULL;
    FeriteNamespaceBucket *nsb = NULL;
    FeriteVariable *var = NULL;
    char *str = NULL;
    int retval = 0;

    FE_ENTER_FUNCTION;

    /* load the script... */
    ptr = ferite_new_script();
    ferite_script_load( ptr, filename );
    if( ptr->scripttext != NULL )
      str = fstrdup( ptr->scripttext );
    else
    {
        FE_LEAVE_FUNCTION(NULL);
    }
    ferite_script_delete( ptr );
    ferite_set_filename( filename );

    ferite_start_compiler( script );

    FUD(("Setting up parser\n"));
    ferite_prepare_parser( str );
    FUD(("Running parser\n"));

    retval = setjmp( ferite_compiler_jmpback );
    if( retval == 0 )
    {  /* we compiled correctly */
        ferite_parse();

        FUD(("Cleaning Up Parser\n"));
        ferite_clean_compiler();

        ffree( str );

        /* here we call the start function */
        nsb = ferite_namespace_element_exists( script, script->mainns, "!__include__" );

        /* get a pointer to the function */
        function = nsb->data;

        nsb->data = NULL; /* this is so we can delete the function from the namespace and not delete the
                           * funcion just yet */
        
        /* remove the function */
        ferite_delete_namespace_element_from_namespace( script, script->mainns, "!__include__" );

        /* execute it */
        var = ferite_script_function_execute( script, function, NULL );

        /* delete the function */
        ferite_delete_function_list( script, function );
        
        FE_LEAVE_FUNCTION(var);
    }
    else
    {
        /* uh, *cough* we didn't */
        ferite_error( CURRENT_SCRIPT, 0, "Error including script `%s'\n", filename );
        ferite_clean_compiler();
        ffree( str );
        FE_LEAVE_FUNCTION(NULL);
    }
    FE_LEAVE_FUNCTION(NULL);
}

/**
 * !function ferite_script_eval
 * !declaration FeriteVariable *ferite_script_eval( FeriteScript *script, char *str )
 * !brief This is used to evalutate a string as a script at runtime
 * !param FeriteScript *script The script to include into
 * !param char *str The string to evaulate
 * !return The value that is returned after executing the 'main' body of the evaluated script
 */
FeriteVariable *ferite_script_eval( FeriteScript *script, char *str )
{
    FeriteFunction *function = NULL;
    FeriteNamespaceBucket *nsb = NULL;
    FeriteVariable *var = NULL;
    int retval = 0;

    FE_ENTER_FUNCTION;
    ferite_start_compiler( script );
    FUD(("Setting up parser\n"));
    ferite_prepare_parser( str );
    FUD(("Running parser\n"));
    retval = setjmp( ferite_compiler_jmpback );
    if( retval == 0 )
    {  /* we compiled correctly */
        ferite_parse();
        FUD(("Cleaning Up Parser\n"));
        ferite_clean_compiler();
        /* here we call the start function */
        nsb = ferite_namespace_element_exists( script, script->mainns, "!__include__" );
        /* get a pointer to the function */
        function = nsb->data;
        nsb->data = NULL; /* this is so we can delete the function from the namespace and not delete the  funcion just yet */
        /* remove the function */
        ferite_delete_namespace_element_from_namespace( script, script->mainns, "!__include__" );
        /* execute it */
        var = ferite_script_function_execute( script, function, NULL );
        /* delete the function */
        ferite_delete_function_list( script, function );
        FE_LEAVE_FUNCTION(var);
    }
    else
    {
      /* uh, *cough* we didn't */
        ferite_error( CURRENT_SCRIPT, 0, "Error evaluating script `%s'\n", str );
        ferite_clean_compiler();
        FE_LEAVE_FUNCTION(NULL);
    }
    FE_LEAVE_FUNCTION(NULL);
}

/**
 * !function ferite_script_compile
 * !declaration FeriteScript *ferite_script_compile( char *filename )
 * !brief Compile a script with the default search paths
 * !param char *filename The name of the script to compile
 * !return A script which can either be executed or contains error information
 */
FeriteScript *ferite_script_compile( char *filename )
{
    FE_ENTER_FUNCTION;
    FE_LEAVE_FUNCTION( ferite_script_compile_with_path( filename, NULL ) );
}

/**
 * !function ferite_script_compile_with_path
 * !declaration FeriteScript *ferite_script_compile_with_path( char *filename, char **paths )
 * !brief Compile a script with the default search paths aswell as an extra set
 * !param char *filename The name of the script to compile
 * !param char **paths A NULL terminated array of paths to add during compilation
 * !return A script which can either be executed or contains error information
 */
FeriteScript *ferite_script_compile_with_path( char *filename, char **paths )
{
    FeriteScript *ptr = NULL;
    FeriteScript *compiled_script = NULL;

    FE_ENTER_FUNCTION;
    ptr = ferite_new_script();
    ferite_script_load( ptr, filename );
    if( ptr->scripttext != NULL )
    {
        ferite_set_filename( filename );
        compiled_script = ferite_compile_string_with_path( ptr->scripttext, paths );
        ferite_script_delete( ptr );
        if( compiled_script->mainns != NULL )
        {
            compiled_script->filename = fstrdup(filename);
            FE_LEAVE_FUNCTION(compiled_script);
        }
        else
        {
            ferite_error( compiled_script, 0, "Fatal error compiling script \"%s\"\n", filename );
            FE_LEAVE_FUNCTION( compiled_script );
        }
    }
    else
    {
        ferite_error( ptr, 0, "Can't open script %s\n", filename );
        FE_LEAVE_FUNCTION( ptr );
    }
    FE_LEAVE_FUNCTION( ptr );
}

/**
 * !function ferite_compile_string
 * !declaration FeriteScript *ferite_compile_string( char *str )
 * !brief Compile a string with the default search paths
 * !param char *str The string to compile
 * !return A script which can either be executed or contains error information
 */
FeriteScript *ferite_compile_string( char *str )
{
    FE_ENTER_FUNCTION;
    FE_LEAVE_FUNCTION( ferite_compile_string_with_path( str, NULL ) );
}

/**
 * !function ferite_compile_string_with_path
 * !declaration FeriteScript *ferite_compile_string_with_path( char *str, char **paths )
 * !brief Compile a str with the default search paths aswell as an extra set
 * !param char *str The script to compile
 * !param char **paths A NULL terminated array of paths to add during compilation
 * !return A script which can either be executed or contains error information
 */
FeriteScript *ferite_compile_string_with_path( char *str, char **paths )
{
    FeriteScript *script = NULL;
    int path_count = 0;

    FE_ENTER_FUNCTION;
    ferite_compile_error = 0;
    script = ferite_new_script();

    ferite_init_gc( script );

    if( ferite_scanner_file != NULL )
      ferite_stack_push( script->include_list, fstrdup(ferite_scanner_file) );

    if( !ferite_keep_native_function_data )
    {
        ferite_add_object_class( script );

        ferite_register_ns_variable( script, script->mainns, ferite_create_object_variable( NULL, "err", FE_STATIC ) );
        ferite_register_ns_variable( script, script->mainns, ferite_create_object_variable( NULL, "null", FE_STATIC ) );
        ferite_init_error_system( script, script->mainns );

        ferite_register_ns_variable( script, script->mainns, ferite_duplicate_variable( script, ferite_ARGV, NULL ) );
    }

    ferite_start_compiler( script );

    /* we now have exclusive lock on the compiler, we now add our paths the list */
    if( paths != NULL )
    {
        while( paths[path_count] != NULL )
          ferite_add_library_search_path( paths[path_count++] );
    }

    FUD(("Setting up parser\n"));
    ferite_prepare_parser( str );

    if( ferite_module_do_preload( script ) == 0 )
    {
        ferite_clean_compiler();
        ferite_script_clean( script );
        FE_LEAVE_FUNCTION( script );
    }

    FUD(("Running parser\n"));
    if( setjmp( ferite_compiler_jmpback ) == 0 )
    {  /* we compiled correctly */
        ferite_parse();
        FUD(("Cleaning Up Parser\n"));

        /* remove the paths from the system before we relinquish compiler control */
        if( paths != NULL )
        {
            while( path_count > 0 )
            {
                ferite_pop_library_search_path();
                path_count--;
            }
        }

        ferite_clean_compiler();
        FE_LEAVE_FUNCTION( script );
    }
    else
    {  /* uh, *cough* we didn't */
        if( ferite_scanner_file == NULL || strcmp( ferite_scanner_file, "-e" ) == 0 )
          ferite_error( CURRENT_SCRIPT, 0, "Fatal error compiling script\n" );

        /* remove the paths from the system before we relinquish compiler control */
        if( paths != NULL )
        {
            while( path_count > 0 )
            {
                ferite_pop_library_search_path();
                path_count--;
            }
        }

        ferite_clean_compiler();
        ferite_script_clean( script );
#ifdef DEBUG
        ferite_call_level = ferite_depth + 1; /* EVIL EVIL EVIL EVIL EVIL EVIL EVIL */
#endif
        FE_LEAVE_FUNCTION( script );
    }
    FE_LEAVE_FUNCTION( script );
}

int ferite_compiler_include_in_list( FeriteScript *script, char *name )
{
    int i = 0;

    FE_ENTER_FUNCTION;
    for( i = 0; i <= script->include_list->stack_ptr; i++ )
    {
        FUD(( "ferite_compiler_include_in_list: checking %s matching %s\n", name, (char *)(script->include_list->stack[i]) ));
        if( script->include_list->stack[i] != NULL && strcmp( name, script->include_list->stack[i] ) == 0 )
        {
            FE_LEAVE_FUNCTION(1);
        }
    }
    FE_LEAVE_FUNCTION(0);
}

void ferite_do_uses( char *name )
{
    FE_ENTER_FUNCTION;
    if( ferite_load_module( CURRENT_SCRIPT, CURRENT_NAMESPACE, name ) == 0 )
    {
        if( !ferite_keep_native_function_data )
        {
            longjmp( ferite_compiler_jmpback, 1 );
        }
    }
    FE_LEAVE_FUNCTION(NOWT);
}

int ferite_do_load_script( char *name )
{
    FeriteNamespaceBucket *nsb = NULL;
    FeriteVariable *var = NULL;
    char *scripttext, *realname;
    jmp_buf old_ferite_compiler_jmpback;
    char *start_function = "!__start__";
    
    FE_ENTER_FUNCTION;
    FUD(( "COMPILER: Request to load script %s\n", name ));
    realname = aphex_relative_to_absolute( name );

    if( ferite_compiler_include_in_list( CURRENT_SCRIPT, realname ) == 0 )
    {
        scripttext = aphex_file_to_string( realname );
        if( scripttext == NULL )
        {
            free( realname );
            FE_LEAVE_FUNCTION(-1);
        }
        ferite_stack_push( CURRENT_SCRIPT->include_list, fstrdup(realname) );
        free( realname ); /* aphex uses malloc - so we have to use real free */

        /* save the older enviroment */
        memcpy( &old_ferite_compiler_jmpback, &ferite_compiler_jmpback, sizeof(jmp_buf) );

        ferite_save_lexer();
        ferite_set_filename( name );
        FUD(( "COMPILER: Setting up parser (ferite_do_load_script)\n"));

        if( scripttext[0] == '#' )
        {
            int i = 0;
            for( i = 0; scripttext[i] != '\n'; i++ )
              scripttext[i] = ' ';
        }
    
        ferite_prepare_parser( scripttext );

        if( setjmp( ferite_compiler_jmpback ) == 0 )
        {
            FUD(( "COMPILER: Running parser (ferite_do_load_script)\n" ));
            ferite_parse();
            FUD(( "COMPILER: Cleaning Up Parser (ferite_do_load_script)\n" ));
            ferite_clean_parser();
            ferite_restore_lexer();
            free( scripttext );

            if( ferite_is_executing( CURRENT_SCRIPT ) )
              start_function = "!__include__";
            
            /* here we call the start function */
            nsb = ferite_namespace_element_exists( CURRENT_SCRIPT, CURRENT_NAMESPACE, start_function );
            var = ferite_script_function_execute( CURRENT_SCRIPT, nsb->data, NULL );
            ferite_variable_destroy( CURRENT_SCRIPT, var );

            /* we need to do this as we is not wantin to av a start function */
            ferite_delete_namespace_element_from_namespace( CURRENT_SCRIPT, CURRENT_NAMESPACE, start_function );
            /* restore the enviroment */
            memcpy( &ferite_compiler_jmpback, &old_ferite_compiler_jmpback, sizeof(jmp_buf) );
            FE_LEAVE_FUNCTION(1);
        }
        else
        {
            ferite_error( CURRENT_SCRIPT, 0, "Can't compile included file \"%s\", error on line %d\n", name, ferite_scanner_lineno );
            ferite_clean_parser();
            ferite_restore_lexer();
            ferite_compile_error = 1;
            free( scripttext );
            memcpy( &ferite_compiler_jmpback, &old_ferite_compiler_jmpback, sizeof(jmp_buf) );
            FE_LEAVE_FUNCTION(0);
        }
    }
    FUD(( "script %s has already been placed in the sciprt", name ));
    free( realname ); /* aphex uses malloc - so we have to use real free */
    FE_LEAVE_FUNCTION(1);
}

void ferite_do_function_header( char *name, int is_static, int is_native, int is_atomic )
{
    FeriteFunction *new_function = NULL, *existing_function = NULL;
    FeriteNamespaceBucket *nsb = NULL;
    FeriteCompileRecord *rec = NULL;
    int isclass = 0;

    FE_ENTER_FUNCTION;

    new_function = ferite_create_internal_function( CURRENT_SCRIPT, name );
    new_function->ccode->filename = fstrdup( (ferite_scanner_file == NULL ? "" : ferite_scanner_file) );
    new_function->is_static = is_static;

    if( CURRENT_CLASS == NULL )
    { /* add to a script */
        if( (nsb = ferite_namespace_element_exists( CURRENT_SCRIPT, CURRENT_NAMESPACE, name )) != NULL )
        {
            if( strcmp( name, "!__start__" ) != 0 )
            {
                /* This should hook up the function in the list */
                existing_function = nsb->data;
                new_function->next = existing_function->next;
                existing_function->next = new_function;
            }
            else
            {
                /* this is a special case */
                ffree( new_function->name );
                new_function->name = fstrdup( "!__include__" );
                ferite_register_ns_function( CURRENT_SCRIPT, CURRENT_NAMESPACE, new_function );
            }
        }
        else
            ferite_register_ns_function( CURRENT_SCRIPT, CURRENT_NAMESPACE, new_function );
    }
    else
    {
        /* add to a class */
        ferite_register_class_function( CURRENT_SCRIPT, CURRENT_CLASS, new_function, is_static );
        
        if( !is_static )
        {
            /* this is so that we have space for self && super in the system */
            ferite_stack_push( new_function->localvars, NULL );
            ferite_stack_push( new_function->localvars, NULL );
        }
        isclass = 1;
    }

    rec = ferite_current_compile;
    ferite_stack_push( ferite_compile_stack, ferite_current_compile );
    ferite_current_compile = fmalloc( sizeof( FeriteCompileRecord ) );
    CURRENT_SCRIPT = rec->script;
    CURRENT_FUNCTION = new_function;
    CURRENT_VARS = new_function->localvars;
    CURRENT_NAMESPACE = rec->ns;
    CURRENT_CLASS = rec->cclass;
    CURRENT_VARS_HASH = ferite_create_hash( CURRENT_SCRIPT, FE_FUNCTION_VARIABLE_SIZE );
    ferite_compiler_current_block_depth = 0;
    
    /* setup the mutex */
    if( is_atomic == 1 )
#ifdef THREAD_SAFE
      new_function->lock = (void*)aphex_mutex_recursive_create();
#else
    ferite_warning( CURRENT_SCRIPT, "'atomic' keyword can not be used for function '%s' - please compile ferite with threading!\n", name );
#endif

    FE_LEAVE_FUNCTION( NOWT );
}

void ferite_do_function_start()
{
    FE_ENTER_FUNCTION;
    if( CURRENT_CLASS != NULL && CURRENT_FUNCTION != NULL )
    {
        if( CURRENT_FUNCTION->is_static == 0 )
        {
            ferite_do_add_variable_to_paramlist( "super", "object" );
            ferite_do_add_variable_to_paramlist( "self", "object" );
        }
    }
    FE_LEAVE_FUNCTION( NOWT );
}

void ferite_do_function_native_block( char *code, char *file, int line )
{
    FeriteFunctionNative *ptr = NULL;

    FE_ENTER_FUNCTION;

    ferite_delete_hash( CURRENT_SCRIPT, CURRENT_VARS_HASH, ferite_delete_pointer );
    CURRENT_VARS_HASH = NULL;
    
    ferite_function_to_external( CURRENT_SCRIPT, CURRENT_FUNCTION );

    /* store the file information */
    ptr = fmalloc( sizeof(FeriteFunctionNative) );
    ptr->code = NULL;
    ptr->file = fstrdup( file );
    ptr->line = line;
    CURRENT_FUNCTION->native_information = ptr;

    if( ferite_keep_native_function_data == 1 )
        ptr->code = code;
    else
    {
        ffree( code );
        FUD(( "found native function %s\n", ferite_compiler_build_current_path() ));
        /* find the correct function */
        CURRENT_FUNCTION->fncPtr = ferite_module_find_function( ferite_compiler_build_current_path() );
        if( CURRENT_FUNCTION->fncPtr == NULL )
        {
            ferite_error( CURRENT_SCRIPT, 0, "Unable to locate native method '%s' - please check that it gets loaded in use the 'uses' statement.\n",
                          ferite_compiler_build_current_path() );
            longjmp( ferite_compiler_jmpback, 1 );
        }
    }
    ffree( ferite_current_compile );
    ferite_current_compile = ferite_stack_pop( ferite_compile_stack );
    FE_LEAVE_FUNCTION(NOWT);
}

void ferite_do_function_footer()
{
    FE_ENTER_FUNCTION;
    ferite_do_exit();
    
    ferite_delete_hash( CURRENT_SCRIPT, CURRENT_VARS_HASH, ferite_delete_pointer );
    CURRENT_VARS_HASH = NULL;
    
    /* ferite_opcode_dump( CURRENT_FUNCTION->ccode ); */
    if( ferite_show_debug )
    {
        FUD(("COMPILER: Dumping opcode list for %s\n", CURRENT_FUNCTION->name ));
        ferite_opcode_dump( CURRENT_FUNCTION->ccode );
        /* FIXME:
         FUD(("COMPILER: Dumping variable hash for function %s\n", CURRENT_FUNCTION->name ));
         ferite_hash_print( CURRENT_SCRIPT, CURRENT_FUNCTION->localvars );*/
    }
    ffree( ferite_current_compile );
    ferite_current_compile = ferite_stack_pop( ferite_compile_stack );
    FE_LEAVE_FUNCTION( NOWT );
}

void ferite_do_class_header( char *name, char *extends )
{
    FeriteClass *classtmpl = NULL;
    FeriteNamespaceBucket *nsb = NULL;
    FeriteScript *scr = NULL;

    FE_ENTER_FUNCTION;
    if( name != NULL )
    {
        classtmpl = ferite_register_inherited_class( CURRENT_SCRIPT, CURRENT_NAMESPACE, name, extends );
    }
    else
    {
        nsb = ferite_find_namespace( CURRENT_SCRIPT, CURRENT_NAMESPACE, extends, FENS_CLS );
        if( nsb == NULL )
        {
            ferite_warning( CURRENT_SCRIPT, "Class '%s' doesn't exist, creating a new class\n", extends );
            classtmpl = ferite_register_inherited_class( CURRENT_SCRIPT, CURRENT_NAMESPACE, name, extends );
        }
        else
        {
            classtmpl = nsb->data;
        }
    }
    ferite_stack_push( ferite_compile_stack, ferite_current_compile );
    scr = CURRENT_SCRIPT;

    ferite_current_compile = fmalloc( sizeof( FeriteCompileRecord ) );
    CURRENT_VARS = NULL;
    CURRENT_CLASS = classtmpl;
    CURRENT_SCRIPT = scr;
    CURRENT_FUNCTION = NULL;
    CURRENT_NAMESPACE = NULL;
    FE_LEAVE_FUNCTION( NOWT );
}

void ferite_do_class_footer()
{
    FE_ENTER_FUNCTION;
    if( ferite_show_debug )
    {
        FUD(("COMPILER: Dumping variable hash for class %s\n", CURRENT_CLASS->name ));
        ferite_hash_print( CURRENT_SCRIPT, CURRENT_CLASS->variables );
        FUD(("COMPILER: Dumping function hash for class %s\n", CURRENT_CLASS->name ));
        ferite_hash_print( CURRENT_SCRIPT, CURRENT_CLASS->functions );
    }
    ffree( ferite_current_compile );
    ferite_current_compile = ferite_stack_pop( ferite_compile_stack );
    FE_LEAVE_FUNCTION( NOWT );
}

void ferite_do_class_item_rename( char *from, char *to )
{
    FeriteVariable *var = NULL;
    FeriteFunction *func = NULL;

    FE_ENTER_FUNCTION;
    func = ferite_hash_get( CURRENT_SCRIPT, CURRENT_CLASS->functions, from );
    if( func != NULL )
    {
        FUD(( "COMPILE: renaming function %s to %s\n", from, to ));
        ferite_hash_delete( CURRENT_SCRIPT, CURRENT_CLASS->functions, from );
        ferite_hash_add( CURRENT_SCRIPT, CURRENT_CLASS->functions, to, func );
        FE_LEAVE_FUNCTION(NOWT);
    }

    var = ferite_hash_get( CURRENT_SCRIPT, CURRENT_CLASS->variables, from );
    if( var != NULL )
    {
        FUD(( "COMPILE: renaming variable %s to %s\n", from, to ));
        ferite_hash_delete( CURRENT_SCRIPT, CURRENT_CLASS->variables, from );
        ferite_hash_add( CURRENT_SCRIPT, CURRENT_CLASS->variables, to, func );
        FE_LEAVE_FUNCTION(NOWT);
    }
    ferite_warning( CURRENT_SCRIPT, "Unable to find class attribute '%s' to rename in class '%s'!\n", from, CURRENT_CLASS->name );
    FE_LEAVE_FUNCTION(NOWT);
}

void ferite_do_namespace_item_rename( char *from, char *to )
{

    FE_ENTER_FUNCTION;
    if( !ferite_rename_namespace_element( CURRENT_SCRIPT, CURRENT_NAMESPACE, from, to ) )
    {
        ferite_warning( CURRENT_SCRIPT, "Unable to find attribute '%s' to rename in namespace '%s'!\n", from, CURRENT_NAMESPACE->name );
    }
    FE_LEAVE_FUNCTION(NOWT);
}

void ferite_do_namespace_header( char *name )
{
    FeriteNamespace *ns = NULL;
    FeriteScript *scr = NULL;

    FE_ENTER_FUNCTION;
    if( ferite_namespace_element_exists( CURRENT_SCRIPT, CURRENT_NAMESPACE, name ) == NULL )
    {
        FUD(( "registering namespace %s\n", name ));
        ns = ferite_register_namespace( CURRENT_SCRIPT, name, CURRENT_NAMESPACE );
        ferite_stack_push( ferite_compile_stack, ferite_current_compile );
        scr = CURRENT_SCRIPT;

        ferite_current_compile = fmalloc( sizeof( FeriteCompileRecord ) );
        CURRENT_VARS = NULL;
        CURRENT_CLASS = NULL;
        CURRENT_SCRIPT = scr;
        CURRENT_FUNCTION = NULL;
        CURRENT_NAMESPACE = ns;
    }
    else
    {
        ferite_error( CURRENT_SCRIPT, 0, "A namespace element called '%s' already exists.\n", name );
        longjmp( ferite_compiler_jmpback, 1 );
    }
    FE_LEAVE_FUNCTION( NOWT );
}

void ferite_do_namespace_extends( char *name )
{
    FeriteNamespace *ns = NULL;
    FeriteNamespaceBucket *nsb = NULL;
    FeriteScript *scr = NULL;

    FE_ENTER_FUNCTION;
    if( (nsb=ferite_namespace_element_exists( CURRENT_SCRIPT, CURRENT_NAMESPACE, name )) != NULL )
    {
        FUD(( "extending namespace %s\n", name ));
        ns = nsb->data;
        ferite_stack_push( ferite_compile_stack, ferite_current_compile );
        scr = CURRENT_SCRIPT;

        ferite_current_compile = fmalloc( sizeof( FeriteCompileRecord ) );
        CURRENT_VARS = NULL;
        CURRENT_CLASS = NULL;
        CURRENT_SCRIPT = scr;
        CURRENT_FUNCTION = NULL;
        CURRENT_NAMESPACE = ns;
    }
    else
    {
        ferite_do_namespace_header( name );
    }
    FE_LEAVE_FUNCTION( NOWT );
}

void ferite_do_namespace_footer()
{
    FE_ENTER_FUNCTION;
    ffree( ferite_current_compile );
    ferite_current_compile = ferite_stack_pop( ferite_compile_stack );
    FE_LEAVE_FUNCTION( NOWT );
}

void ferite_do_add_variable( char *name, char *type, int is_global, int is_final, int is_static, int is_atomic )
{
    FeriteVariable *new_variable = NULL;
    FeriteVariableHash *current_hash = NULL;

    FE_ENTER_FUNCTION;

    if( strcmp( name, "err" ) == 0 || strcmp( name, "null" ) == 0 )
    {
        ferite_error( CURRENT_SCRIPT, 0, "Compile Error: Variable name \"%s\" is a reserved - it can not be used.\n", name );
        ferite_compile_error = 1;
        longjmp( ferite_compiler_jmpback, 1 );
    }

    if( is_global )
      current_hash = CURRENT_NAMESPACE->space;
    else
    {
        if( CURRENT_FUNCTION != NULL )
            current_hash = CURRENT_VARS_HASH;
        else if( CURRENT_CLASS != NULL )
            current_hash = CURRENT_CLASS->variables;
        else
            current_hash = CURRENT_NAMESPACE->space;
    }
    if( (new_variable = ferite_get_variable_from_hash( CURRENT_SCRIPT, current_hash, name )) == NULL )
    {
        if( strcmp( type, "number" ) == 0 )
          new_variable = ferite_create_number_long_variable( NULL, name, 0, FE_ALLOC );
        if( strcmp( type, "string" ) == 0 )
          new_variable = ferite_create_string_variable( NULL, name, NULL, FE_ALLOC );
        if( strcmp( type, "object" ) == 0 )
          new_variable = ferite_create_object_variable( NULL, name, FE_ALLOC );
        if( strcmp( type, "array" ) == 0 )
          new_variable = ferite_create_uarray_variable( NULL, name, 0, FE_ALLOC );
        if( strcmp( type, "void" ) == 0 )
          new_variable = ferite_create_void_variable( NULL, name, FE_ALLOC );

        if( new_variable != NULL )
        {
            if( is_atomic )
#ifdef THREAD_SAFE
              new_variable->lock = (void*)aphex_mutex_recursive_create();
#else
            ferite_warning( CURRENT_SCRIPT, "'atomic' keyword can not be used for variable '%s' - please compile ferite with threading!\n", name );
#endif
            if( is_final )
            {
                FUD(( "Marking variable %s as final.\n", name ));
                MARK_VARIABLE_AS_FINAL( new_variable );
            }

            if( is_global )
              ferite_register_ns_variable( CURRENT_SCRIPT, CURRENT_NAMESPACE, new_variable );
            else
            {
                if( CURRENT_FUNCTION != NULL )                  
                {
                    if( strcmp( new_variable->name, "self" ) == 0 && CURRENT_FUNCTION->klass != NULL )
                    {
                        CURRENT_VARS->stack[1] = new_variable;
                        ferite_hash_add( CURRENT_SCRIPT, CURRENT_VARS_HASH, new_variable->name, ferite_int_to_ptr( 1 ) );
                    }
                    else if( strcmp( new_variable->name, "super" ) == 0 && CURRENT_FUNCTION->klass != NULL )
                    {
                        CURRENT_VARS->stack[2] = new_variable;
                        ferite_hash_add( CURRENT_SCRIPT, CURRENT_VARS_HASH, new_variable->name, ferite_int_to_ptr( 2 ) );
                    }
                    else
                    {
                        ferite_stack_push( CURRENT_VARS, new_variable );                        
                        /* FIXME: must add checking for block nested stuff */
                        ferite_hash_add( CURRENT_SCRIPT, CURRENT_VARS_HASH, new_variable->name, ferite_int_to_ptr( CURRENT_VARS->stack_ptr ) );
                    }   
                }
                else if( CURRENT_CLASS != NULL )
                  ferite_register_class_variable( CURRENT_SCRIPT, CURRENT_CLASS, new_variable, is_static );
                else
                  ferite_register_ns_variable( CURRENT_SCRIPT, CURRENT_NAMESPACE, new_variable );
            }
            FE_LEAVE_FUNCTION( NOWT );
        }
        ferite_error( CURRENT_SCRIPT, 0, "Compile Error: Trying to declare variable of unknown type \"%s\"", type );
        ffree( name );
        ffree( type );
        ferite_compile_error = 1;
        longjmp( ferite_compiler_jmpback, 1 );
        FE_LEAVE_FUNCTION( NOWT );
    }
    else
    {
        ferite_error( CURRENT_SCRIPT, 0, "Compile Error: Variable \"%s\" already exists\n", name );
        ferite_compile_error = 1;
        longjmp( ferite_compiler_jmpback, 1 );
    }
}

void ferite_do_add_variable_with_value( char *name, FeriteVariable *new_variable, int is_global, int is_static, int is_atomic )
{
    FeriteVariableHash *current_hash = NULL;

    FE_ENTER_FUNCTION;

    if( strcmp( name, "err" ) == 0 || strcmp( name, "null" ) == 0 )
    {
        ferite_error( CURRENT_SCRIPT, 0, "Compile Error: Variable name \"%s\" is a reserved - it can not be used.\n", name );
        ferite_compile_error = 1;
        longjmp( ferite_compiler_jmpback, 1 );
    }

    if( is_global )
      current_hash = CURRENT_NAMESPACE->space;
    else
    {
        if( CURRENT_FUNCTION != NULL )
          current_hash = CURRENT_VARS_HASH;
        else if( CURRENT_CLASS != NULL )
          current_hash = CURRENT_CLASS->variables;
        else
          current_hash = CURRENT_NAMESPACE->space;
    }
    
    if( ferite_get_variable_from_hash( CURRENT_SCRIPT, current_hash, name ) == NULL )
    {
        if( new_variable != NULL )
        {
            if( is_atomic )
#ifdef THREAD_SAFE
              new_variable->lock = (void*)aphex_mutex_recursive_create();
#else
            ferite_warning( CURRENT_SCRIPT, "'atomic' keyword can not be used for variable '%s' - please compile ferite with threading!\n", name );
#endif
            if( is_global )
              ferite_register_ns_variable( CURRENT_SCRIPT, CURRENT_NAMESPACE, new_variable );
            else
            {
                if( CURRENT_FUNCTION != NULL )                  
                {
                    if( strcmp( new_variable->name, "self" ) == 0 && CURRENT_FUNCTION->klass != NULL )
                    {
                        CURRENT_VARS->stack[1] = new_variable;
                        ferite_hash_add( CURRENT_SCRIPT, CURRENT_VARS_HASH, new_variable->name, ferite_int_to_ptr( 1 ) );
                    }
                    else if( strcmp( new_variable->name, "super" ) == 0 && CURRENT_FUNCTION->klass != NULL )
                    {
                        CURRENT_VARS->stack[2] = new_variable;
                        ferite_hash_add( CURRENT_SCRIPT, CURRENT_VARS_HASH, new_variable->name, ferite_int_to_ptr( 2 ) );
                    }
                    else
                    {
                        ferite_stack_push( CURRENT_VARS, new_variable );                        
                        /* FIXME: must add checking for block nested stuff */
                        ferite_hash_add( CURRENT_SCRIPT, CURRENT_VARS_HASH, new_variable->name, ferite_int_to_ptr( CURRENT_VARS->stack_ptr ) );
                    }   
                }
                else if( CURRENT_CLASS != NULL )
                  ferite_register_class_variable( CURRENT_SCRIPT, CURRENT_CLASS, new_variable, is_static );
                else
                  ferite_register_ns_variable( CURRENT_SCRIPT, CURRENT_NAMESPACE, new_variable );
            }
            FE_LEAVE_FUNCTION( NOWT );
        }
    }
    else
    {
        ferite_error( CURRENT_SCRIPT, 0, "Compile Error: A variable called \"%s\" already exists\n", name );
        if( new_variable != NULL )
          ferite_variable_destroy( CURRENT_SCRIPT, new_variable );
        ferite_compile_error = 1;
        longjmp( ferite_compiler_jmpback, 1 );
    }
}

#ifdef DEBUG
# define REDUCE_DEPTH ferite_call_level -= 2
#else
# define REDUCE_DEPTH
#endif

#define CHECK_FUNCTION_CODE( blah ) \
   if( CURRENT_FUNCTION == NULL ) { \
      ferite_error( CURRENT_SCRIPT, 0, "Compile Error: on line %d, in %s\n", ferite_scanner_lineno, ferite_scanner_file ); \
      ferite_compile_error = 1; \
      blah; \
    REDUCE_DEPTH; \
      longjmp( ferite_compiler_jmpback, 1 ); \
   }

void ferite_do_add_variable_to_paramlist( char *name, char *type )
{
    FeriteVariable *new_variable = NULL;
    int max_argcount = 0;
    
    FE_ENTER_FUNCTION;
    CHECK_FUNCTION_CODE({ ffree(name); ffree(type); });

    max_argcount = FE_FUNCTION_PARAMETER_MAX_SIZE - 1;
    if( CURRENT_FUNCTION->klass != NULL )
      max_argcount -= 2;
    
    if( CURRENT_FUNCTION->arg_count > FE_FUNCTION_PARAMETER_MAX_SIZE )
    {
        ferite_error( CURRENT_SCRIPT, 0, "Compile Error: on line %d, in %s\n", ferite_scanner_lineno, ferite_scanner_file );
        ferite_error( CURRENT_SCRIPT, 0, "  Maximum number of parameters for function '%s' exceeded, must be less than %d\n",
                      CURRENT_FUNCTION->name, max_argcount );
        ferite_compile_error = 1;
        REDUCE_DEPTH;
        longjmp( ferite_compiler_jmpback, 1 );
    }
    
    ferite_do_add_variable( name, type, 0, 0, 0, 0 ); /* add the variable to the function */
    /* now lets add the paramter */
    if( strcmp( type, "number" ) == 0 )
      new_variable = ferite_create_number_long_variable( NULL, name, 0, FE_ALLOC );
    if( strcmp( type, "string" ) == 0 )
      new_variable = ferite_create_string_variable( NULL, name, NULL, FE_ALLOC );
    if( strcmp( type, "object" ) == 0 )
      new_variable = ferite_create_object_variable( NULL, name, FE_ALLOC );
    if( strcmp( type, "array" ) == 0 )
      new_variable = ferite_create_uarray_variable( NULL, name, 0, FE_ALLOC );
    if( strcmp( type, "void" ) == 0 )
      new_variable = ferite_create_void_variable( NULL, name, FE_ALLOC );
    CURRENT_FUNCTION->signature[CURRENT_FUNCTION->arg_count] = fmalloc( sizeof( FeriteParameterRecord ) );
    CURRENT_FUNCTION->signature[CURRENT_FUNCTION->arg_count]->variable = new_variable;
    CURRENT_FUNCTION->signature[CURRENT_FUNCTION->arg_count]->has_default_value = 0;
    CURRENT_FUNCTION->arg_count++;
    FE_LEAVE_FUNCTION( NOWT );
}

void ferite_do_exit()
{
    FeriteOp *op = NULL;

    FE_ENTER_FUNCTION;
    CHECK_FUNCTION_CODE( NOWT );
    op = ferite_get_next_op( CURRENT_FUNCTION->ccode );
    op->block_depth = ferite_compiler_current_block_depth;
    op->OP_TYPE = F_OP_EXIT;
    op->line = ferite_scanner_lineno;
    FE_LEAVE_FUNCTION( NOWT );
}

void ferite_do_push( FeriteVariable *var )
{
    FeriteOp *op = NULL;

    FE_ENTER_FUNCTION;
    CHECK_FUNCTION_CODE({ ferite_variable_destroy( CURRENT_SCRIPT, var ); });
    op = ferite_get_next_op( CURRENT_FUNCTION->ccode );
    op->block_depth = ferite_compiler_current_block_depth;
    OP_PUSH( op, var );
    op->line = ferite_scanner_lineno;
    FE_LEAVE_FUNCTION( NOWT );
}

void ferite_do_pop()
{
    FeriteOp *op = NULL;

    FE_ENTER_FUNCTION;
    CHECK_FUNCTION_CODE( NOWT );
    op = ferite_get_next_op( CURRENT_FUNCTION->ccode );
    op->block_depth = ferite_compiler_current_block_depth;
    op->OP_TYPE = F_OP_POP;
    op->line = ferite_scanner_lineno;
    FE_LEAVE_FUNCTION( NOWT );
}

void ferite_do_getargs()
{
    FeriteOp *op = NULL;

    FE_ENTER_FUNCTION;
    CHECK_FUNCTION_CODE( NOWT );
    op = ferite_get_next_op( CURRENT_FUNCTION->ccode );
    op->block_depth = ferite_compiler_current_block_depth;
    op->OP_TYPE = F_OP_ARGS;
    op->line = ferite_scanner_lineno;
    FE_LEAVE_FUNCTION( NOWT );
}

void ferite_do_variable_push( char *name )
{
    FeriteOp *op = NULL;
    int *index = NULL;
    
    FE_ENTER_FUNCTION;

    FUD(("DO_PUSHVAR: WANTING TO PUSH VARIABLE %s\n", name ));
    CHECK_FUNCTION_CODE({ ffree( name ); });
    op = ferite_get_next_op( CURRENT_FUNCTION->ccode );
    op->block_depth = ferite_compiler_current_block_depth;
    
    /* If we can find the variable locally, lets push it */
    if( (index = ferite_hash_get( CURRENT_SCRIPT, CURRENT_VARS_HASH, name)) != NULL )
    {
        op->OP_TYPE = F_OP_PUSHINDEX;
        op->addr = *index;
    }
    else
    {
        op->OP_TYPE = F_OP_PUSHVAR;
        op->opdata = fstrdup( name );
    }
    op->line = ferite_scanner_lineno;
    FE_LEAVE_FUNCTION( NOWT );
}

void ferite_do_variable_pushattr( char *name )
{
    FeriteOp *op = NULL;

    FE_ENTER_FUNCTION;

    FUD(("DO_PUSHATTR: WANTING TO PUSH VARIABLE %s\n", name ));
    CHECK_FUNCTION_CODE({ ffree( name ); });
    op = ferite_get_next_op( CURRENT_FUNCTION->ccode );
    op->OP_TYPE = F_OP_PUSHATTR;
    op->block_depth = ferite_compiler_current_block_depth;
    op->opdata = fstrdup( name );
    op->line = ferite_scanner_lineno;
    FE_LEAVE_FUNCTION( NOWT );
}

void ferite_do_many_op( int opptr , int n)
{
    FeriteOp *op = NULL;
    int *c = NULL;

    FE_ENTER_FUNCTION;
    op = ferite_get_next_op( CURRENT_FUNCTION->ccode );
    op->block_depth = ferite_compiler_current_block_depth;
    c = fmalloc(sizeof(int));
    *c = n;
    OP_MANY(op, opptr, c);
    op->line = ferite_scanner_lineno;
    FE_LEAVE_FUNCTION( NOWT );
}

void ferite_do_binary_op( int opptr )
{
    FeriteOp *op = NULL;

    FE_ENTER_FUNCTION;
    op = ferite_get_next_op( CURRENT_FUNCTION->ccode );
    op->block_depth = ferite_compiler_current_block_depth;
    OP_BINARY( op, opptr, NULL );
    op->line = ferite_scanner_lineno;
    FE_LEAVE_FUNCTION( NOWT );
}

void ferite_do_unary_op( int opptr )
{
    FeriteOp *op = NULL;

    FE_ENTER_FUNCTION;
    op = ferite_get_next_op( CURRENT_FUNCTION->ccode );
    op->block_depth = ferite_compiler_current_block_depth;
    OP_UNARY( op, opptr, NULL );
    op->line = ferite_scanner_lineno;
    FE_LEAVE_FUNCTION( NOWT );
}

#define CHECK_PARAM_COUNT                                              \
    max_argcount = FE_FUNCTION_PARAMETER_MAX_SIZE - 1;                 \
    if( CURRENT_FUNCTION->klass != NULL )                              \
        max_argcount -= 2;                                             \
    if( CURRENT_FUNCTION->arg_count > FE_FUNCTION_PARAMETER_MAX_SIZE ) \
      {                                                                \
          ferite_error( CURRENT_SCRIPT, 0, "Compile Error: on line %d, in %s\n", ferite_scanner_lineno, ferite_scanner_file );   \
          ferite_error( CURRENT_SCRIPT, 0, "  Maximum number of arguments for %s call '%s' exceeded, must be less than %d\n",    \
                            ( CURRENT_FUNCTION->klass == NULL ? "function" : "method" ), CURRENT_FUNCTION->name, max_argcount ); \
          ferite_compile_error = 1;                                    \
          REDUCE_DEPTH;                                                \
          longjmp( ferite_compiler_jmpback, 1 );                       \
      }

void ferite_do_function_call( char *name, int arg_count )
{
    FeriteOp *op = NULL;
    int max_argcount = 0;
    
    FE_ENTER_FUNCTION;
    
    CHECK_FUNCTION_CODE({ ffree( name ); });    
    CHECK_PARAM_COUNT;
      
    op = ferite_get_next_op( CURRENT_FUNCTION->ccode );
    op->block_depth = ferite_compiler_current_block_depth;
    op->OP_TYPE = F_OP_FUNCTION;
    op->opdata = fstrdup( name );

    /* the function call data */
    op->opdataf = fmalloc( sizeof( FeriteOpFncData ) );
    op->opdataf->argument_count = arg_count;
    op->opdataf->is_autoload = 0;
    op->opdataf->function = NULL;
    op->line = ferite_scanner_lineno;
    
    FE_LEAVE_FUNCTION( NOWT );
}

void ferite_do_object_function_call( char *name, int arg_count )
{
    FeriteOp *op = NULL;
    int max_argcount = 0;
   
    FE_ENTER_FUNCTION;
    
    CHECK_FUNCTION_CODE({ ffree( name ); });
    
    CHECK_PARAM_COUNT;
    
    op = ferite_get_next_op( CURRENT_FUNCTION->ccode );
    op->block_depth = ferite_compiler_current_block_depth;
    op->OP_TYPE = F_OP_METHOD;
    op->opdata = fstrdup( name );

    /* the function call data */
    op->opdataf = fmalloc( sizeof( FeriteOpFncData ) );
    op->opdataf->argument_count = arg_count;
    op->opdataf->is_autoload = 0;
    op->opdataf->function = NULL;
    
    op->line = ferite_scanner_lineno;
    FE_LEAVE_FUNCTION( NOWT );
}

void ferite_do_if_statement()
{
    FeriteBkRequest *req = NULL;
    FeriteOp *op = NULL;

    FE_ENTER_FUNCTION;
    CHECK_FUNCTION_CODE( NOWT );
    op = ferite_get_next_op( CURRENT_FUNCTION->ccode );
    op->block_depth = ferite_compiler_current_block_depth;
    op->OP_TYPE = F_OP_BNE;
    op->line = ferite_scanner_lineno;
    req = ferite_create_request( op, IF_JUMP_THEN );
    ferite_stack_push( ferite_fwd_look_stack, req );
    FE_LEAVE_FUNCTION( NOWT );
}

/*
 * This is the statement is purely here for && and ||, it's purpose is the
 * opposite of the above function.
 */
void ferite_do_not_if_statement()
{
    FeriteBkRequest *req = NULL;
    FeriteOp *op = NULL;

    FE_ENTER_FUNCTION;
    CHECK_FUNCTION_CODE( NOWT );
    op = ferite_get_next_op( CURRENT_FUNCTION->ccode );
    op->block_depth = ferite_compiler_current_block_depth;
    op->OP_TYPE = F_OP_BIE;
    op->line = ferite_scanner_lineno;
    req = ferite_create_request( op, IF_JUMP_THEN );
    ferite_stack_push( ferite_fwd_look_stack, req );
    FE_LEAVE_FUNCTION( NOWT );
}

void ferite_do_after_then_statement()
{
    FeriteBkRequest *req = NULL;
    FeriteOp *op = NULL;
    int addr;

    FE_ENTER_FUNCTION;
    addr = ferite_get_next_op_loc( CURRENT_FUNCTION->ccode );
    op = ferite_get_next_op_address( CURRENT_FUNCTION->ccode );
    op->OP_TYPE = F_OP_NOP;
    req = (FeriteBkRequest *)ferite_stack_pop( ferite_fwd_look_stack );
    FUD(("Popped %d off request stack\n", req->type));
    req->reqop->addr = addr;
    MARK_VARIABLE_AS_COMPILED( PTR2VAR(req->reqop->opdata) );
    ferite_destroy_request( req );
    FE_LEAVE_FUNCTION( NOWT );
}

/* blum */
void ferite_do_after_then_before_else_statement()
{
    FeriteBkRequest *req = NULL;
    FeriteOp *jump_else_op, *do_else_op;
    int do_else_op_offset;

    FE_ENTER_FUNCTION;
    /* we jump over the else block */
    jump_else_op = ferite_get_next_op( CURRENT_FUNCTION->ccode );
    jump_else_op->block_depth = ferite_compiler_current_block_depth;
    jump_else_op->OP_TYPE = F_OP_JMP;
    jump_else_op->line = ferite_scanner_lineno;

    /* we jump to this op when we do else {} */
    do_else_op_offset = ferite_get_next_op_loc( CURRENT_FUNCTION->ccode );
    do_else_op = ferite_get_next_op_address( CURRENT_FUNCTION->ccode ); /* so we know where to jump, but dont put an NOP there */
    do_else_op->OP_TYPE = F_OP_NOP;

    req = (FeriteBkRequest *)ferite_stack_pop( ferite_fwd_look_stack );
    FUD(("Popped %d off request stack\n", req->type));
    req->reqop->addr = do_else_op_offset;
    MARK_VARIABLE_AS_COMPILED( PTR2VAR(req->reqop->opdata) );
    ferite_destroy_request( req );

    req = ferite_create_request( jump_else_op, IF_JUMP_ELSE );
    ferite_stack_push( ferite_fwd_look_stack, req );
    FE_LEAVE_FUNCTION( NOWT );
}

void ferite_do_after_else_statement()
{
    FeriteBkRequest *req = NULL;
    FeriteOp *op = NULL;
    int next_op;

    FE_ENTER_FUNCTION;
    next_op = ferite_get_next_op_loc( CURRENT_FUNCTION->ccode );
    op = ferite_get_next_op_address( CURRENT_FUNCTION->ccode );
    op->OP_TYPE = F_OP_NOP;
    req = (FeriteBkRequest *)ferite_stack_pop( ferite_fwd_look_stack );
    FUD(("Popped %d off request stack\n", req->type));
    req->reqop->addr = next_op;
    MARK_VARIABLE_AS_COMPILED( PTR2VAR(req->reqop->opdata) );
    ferite_destroy_request( req );
    FE_LEAVE_FUNCTION( NOWT );
}

/* while stuff
 *
 * a:  expr
 *     BNE   b
 *     block
 *     JMP   a
 * b:
 */

void ferite_do_while_begin()
{
    FeriteBkRequest *req = NULL;
    FeriteOp *op = NULL;
    int next_op;

    FE_ENTER_FUNCTION;
    next_op = ferite_get_next_op_loc( CURRENT_FUNCTION->ccode );
    op = ferite_get_next_op_address( CURRENT_FUNCTION->ccode );
    op->OP_TYPE = F_OP_NOP;
    req = ferite_create_request( op, WHILE_JUMP_TO );
    req->addr = next_op;
    ferite_stack_push( ferite_bck_look_stack, req );

    /* We need this to make nested loop structures work */
    req = ferite_create_request( NULL, BREAK_SEPARATOR );
    ferite_stack_push( ferite_break_look_stack, req );
    FE_LEAVE_FUNCTION( NOWT );
}

void ferite_do_while_after_expr()
{
    FeriteBkRequest *req = NULL;
    FeriteOp *op = NULL;

    FE_ENTER_FUNCTION;
    op = ferite_get_next_op( CURRENT_FUNCTION->ccode );
    op->OP_TYPE = F_OP_BNE;
    op->block_depth = ferite_compiler_current_block_depth;
    op->line = ferite_scanner_lineno;
    req = ferite_create_request( op, WHILE_JUMP_BLOCK );
    ferite_stack_push( ferite_fwd_look_stack, req );
    FE_LEAVE_FUNCTION( NOWT );
}

void ferite_do_while_end()
{
    FeriteBkRequest *req = NULL;
    FeriteOp *op = NULL;
    int next_addr;

    FE_ENTER_FUNCTION;
    /* hook up jump back */
    op = ferite_get_next_op( CURRENT_FUNCTION->ccode );
    op->OP_TYPE = F_OP_JMP;
    op->block_depth = ferite_compiler_current_block_depth;
    op->line = ferite_scanner_lineno;

    next_addr = ferite_get_next_op_loc( CURRENT_FUNCTION->ccode );
    ferite_process_breaks( WHILE_JUMP_TO, next_addr );

    req = (FeriteBkRequest *)ferite_stack_pop( ferite_bck_look_stack );
    FUD(("Popped %d off request stack\n", req->type));
    /*   fprintf( stderr, "while end type %d\n", req->type );*/
    op->addr = req->addr;
    MARK_VARIABLE_AS_COMPILED( PTR2VAR(op->opdata) );
    ferite_destroy_request( req );

    /* hook up jump over */
    op = ferite_get_next_op_address( CURRENT_FUNCTION->ccode );
    op->OP_TYPE = F_OP_NOP;
    req = (FeriteBkRequest *)ferite_stack_pop( ferite_fwd_look_stack );
    FUD(("Popped %d off request stack\n", req->type));
    req->reqop->addr = next_addr;
    MARK_VARIABLE_AS_COMPILED( PTR2VAR(req->reqop->opdata) );
    ferite_destroy_request( req );
    FE_LEAVE_FUNCTION( NOWT );
}

/************ FOR LOOP STUFF */

/* for( init , */

void ferite_do_for_loop_start()
{
    FeriteBkRequest *req = NULL;
    FeriteOp *op = NULL;
    int next_op;

    FE_ENTER_FUNCTION;
    next_op = ferite_get_next_op_loc( CURRENT_FUNCTION->ccode );
    op = ferite_get_next_op_address( CURRENT_FUNCTION->ccode );
    op->OP_TYPE = F_OP_NOP;
    req = ferite_create_request( op, FOR_TEST_START );
    req->addr = next_op;
    ferite_stack_push( ferite_bck_look_stack, req );

    /* We need this to make nested loop structures work */
    req = ferite_create_request( NULL, BREAK_SEPARATOR );
    ferite_stack_push( ferite_break_look_stack, req );
    
    FE_LEAVE_FUNCTION( NOWT );
}

/* test , */

void ferite_do_for_loop_itterate()
{

    FeriteBkRequest *req = NULL;
    FeriteOp *op = NULL;
    int next_op;

    FE_ENTER_FUNCTION;
   /* jump out of the for loop */
    op = ferite_get_next_op( CURRENT_FUNCTION->ccode );
    op->OP_TYPE = F_OP_BNE;
    op->line = ferite_scanner_lineno;
    op->block_depth = ferite_compiler_current_block_depth;
    req = ferite_create_request( op, FOR_JUMP_BLOCK );
    ferite_stack_push( ferite_fwd_look_stack, req );

   /* jump over the increment block (3rd expr) of the while loop */
    op = ferite_get_next_op( CURRENT_FUNCTION->ccode );
    op->OP_TYPE = F_OP_JMP;
    op->line = ferite_scanner_lineno;
    op->block_depth = ferite_compiler_current_block_depth;
    req = ferite_create_request( op, FOR_JUMP_INCR );
    ferite_stack_push( ferite_fwd_look_stack, req );

   /* push on the addrs of the increment block start */
    next_op = ferite_get_next_op_loc( CURRENT_FUNCTION->ccode );
    op = ferite_get_next_op_address( CURRENT_FUNCTION->ccode );
    op->OP_TYPE = F_OP_NOP;
    req = ferite_create_request( op, FOR_INCR_START );
    req->addr = next_op;
    ferite_stack_push( ferite_bck_look_stack, req );
    FE_LEAVE_FUNCTION( NOWT );
}

/*    INCR ) */

void ferite_do_for_loop_block()
{
    FeriteBkRequest *req, *req_incr_start, *req_test_start;
    FeriteOp *op = NULL;
    int next_op;

    FE_ENTER_FUNCTION;
   /* so we dont lose this request */
    req_incr_start = ferite_stack_pop( ferite_bck_look_stack );
    FUD(("Popped %d off stack\n", req_incr_start->type ));

   /* setupa jmp op to go back to the test */
    req_test_start = ferite_stack_pop( ferite_bck_look_stack );
    op = ferite_get_next_op( CURRENT_FUNCTION->ccode );
    op->OP_TYPE = F_OP_JMP;
    op->addr = req_test_start->addr;
    op->line = ferite_scanner_lineno;
    op->block_depth = ferite_compiler_current_block_depth;
    MARK_VARIABLE_AS_COMPILED( PTR2VAR(op->opdata) );
    FUD(("Popped %d off stack\n", req_test_start->type ));
    ferite_destroy_request( req_test_start );

   /* push first pop back onto the stack */
    ferite_stack_push( ferite_bck_look_stack, req_incr_start );
    FUD(("Popped %d off stack\n", req_incr_start->type ));

   /* now lets sort out the jump increment block code */
    next_op = ferite_get_next_op_loc( CURRENT_FUNCTION->ccode );
    op = ferite_get_next_op_address( CURRENT_FUNCTION->ccode );
    op->OP_TYPE = F_OP_NOP;
    req = ferite_stack_pop( ferite_fwd_look_stack );
    FUD(("Popped %d off stack\n", req->type ));
    req->reqop->addr = next_op;
    MARK_VARIABLE_AS_COMPILED( PTR2VAR(req->reqop->opdata) );
    ferite_destroy_request( req );

    FE_LEAVE_FUNCTION( NOWT );
}

/* block */

void ferite_do_for_loop_end()
{
    FeriteBkRequest *req = NULL;
    FeriteOp *op = NULL;
    int next_op;

    FE_ENTER_FUNCTION;
    /* jump back to incr block */
    op = ferite_get_next_op( CURRENT_FUNCTION->ccode );
    op->OP_TYPE = F_OP_JMP;
    op->line = ferite_scanner_lineno;
    op->block_depth = ferite_compiler_current_block_depth;
    next_op = ferite_get_next_op_loc( CURRENT_FUNCTION->ccode );

    /* process dem breaks's */
    ferite_process_breaks( FOR_INCR_START, next_op );

    req = ferite_stack_pop( ferite_bck_look_stack );
    FUD(("Popped %d off stack\n", req->type ));
    op->addr = req->addr;
    MARK_VARIABLE_AS_COMPILED( PTR2VAR(op->opdata) );
    ferite_destroy_request( req );

    /* jump over block */
    op = ferite_get_next_op_address( CURRENT_FUNCTION->ccode );
    op->OP_TYPE = F_OP_NOP;
    req = ferite_stack_pop( ferite_fwd_look_stack );
    FUD(("Popped %d off stack\n", req->type ));
    req->reqop->addr = next_op;
    MARK_VARIABLE_AS_COMPILED( PTR2VAR(req->reqop->opdata) );
    ferite_destroy_request( req );
    FE_LEAVE_FUNCTION( NOWT );
}

/************** do stuff */

void ferite_do_do_start()
{
    FeriteBkRequest *req = NULL;
    FeriteOp *op = NULL;
    int next_op;

    FE_ENTER_FUNCTION;
    next_op = ferite_get_next_op_loc( CURRENT_FUNCTION->ccode );
    op = ferite_get_next_op_address( CURRENT_FUNCTION->ccode );
    op->OP_TYPE = F_OP_NOP;
    req = ferite_create_request( op, DO_START );
    req->addr = next_op;
    ferite_stack_push( ferite_bck_look_stack, req );

    /* We need this to make nested loop structures work */
    req = ferite_create_request( NULL, BREAK_SEPARATOR );
    ferite_stack_push( ferite_break_look_stack, req );
    
    FE_LEAVE_FUNCTION( NOWT );
}

void ferite_do_do_end()
{
    FeriteBkRequest *req = NULL;
    FeriteOp *op = NULL;
    int addr;

    FE_ENTER_FUNCTION;
    op = ferite_get_next_op( CURRENT_FUNCTION->ccode );
    op->line = ferite_scanner_lineno;
    op->block_depth = ferite_compiler_current_block_depth;
    op->OP_TYPE = F_OP_BIE;

   /* process dem breaks's */
    addr = ferite_get_next_op_loc( CURRENT_FUNCTION->ccode );
    ferite_process_breaks( DO_START, addr );

    req = ferite_stack_pop( ferite_bck_look_stack );
    FUD(("Popped %d off stack\n", req->type ));
    op->addr = req->addr;
    MARK_VARIABLE_AS_COMPILED( PTR2VAR(op->opdata) );
    ferite_destroy_request( req );
    FE_LEAVE_FUNCTION( NOWT );
}

void ferite_do_new_object( int arg_count )
{
    FeriteOp *op = NULL;
    int max_argcount = 0;
    
    FE_ENTER_FUNCTION;
    CHECK_FUNCTION_CODE(NOWT);
    CHECK_PARAM_COUNT;
    
    op = ferite_get_next_op( CURRENT_FUNCTION->ccode );
    op->OP_TYPE = F_OP_NEWOBJ;
    op->line = ferite_scanner_lineno;
    op->block_depth = ferite_compiler_current_block_depth;

    /* the function call data */
    op->opdataf = fmalloc( sizeof( FeriteOpFncData ) );
    op->opdataf->argument_count = arg_count;
    op->opdataf->is_autoload = 0;
    op->opdataf->function = NULL;
    
    FE_LEAVE_FUNCTION( NOWT );
}

void ferite_do_regex( char *pattern, char *swap, char type, char *flags )
{
    FeriteOp *op = NULL;

    FE_ENTER_FUNCTION;
    CHECK_FUNCTION_CODE({ ffree( pattern ); });
    op = ferite_get_next_op( CURRENT_FUNCTION->ccode );
    op->OP_TYPE = F_OP_RGX;
    op->opdata = ferite_generate_regex( CURRENT_SCRIPT, pattern, swap, type, flags );
    op->line = ferite_scanner_lineno;
    op->block_depth = ferite_compiler_current_block_depth;
    FE_LEAVE_FUNCTION( NOWT );
}

/****************************************************************************************/
/********* ERROR STUFF ******************************************************************/
void ferite_do_iferr_block()
{
    FeriteBkRequest *req = NULL;
    FeriteOp *op = NULL;

    FE_ENTER_FUNCTION;
    op = ferite_get_next_op( CURRENT_FUNCTION->ccode );
    op->OP_TYPE = F_OP_ERR;
    op->line = ferite_scanner_lineno;
    op->block_depth = ferite_compiler_current_block_depth;
    req = ferite_create_request( op, ERR_START );
    ferite_stack_push( ferite_fwd_look_stack, req );
    FE_LEAVE_FUNCTION( NOWT );
}

void ferite_do_before_fix_block()
{
    FeriteBkRequest *req, *req2;
    FeriteOp *op, *op_two;
    int addr;

    FE_ENTER_FUNCTION;

    op_two = ferite_get_next_op( CURRENT_FUNCTION->ccode );
    op_two->OP_TYPE = F_OP_JMP;
    op_two->line = ferite_scanner_lineno;
    op_two->block_depth = ferite_compiler_current_block_depth;
    req2 = ferite_create_request( op_two, FIX_BLOCK_JMP ); /* jump instruction to jump fix block */

    /* this sets the code to jump to the fix block if there is an error
     * goes back and sets the address to jump to */
    addr = ferite_get_next_op_loc( CURRENT_FUNCTION->ccode );
    op = ferite_get_next_op( CURRENT_FUNCTION->ccode );
    op->OP_TYPE = F_OP_ERR;
    op->addr = -1;
    op->line = ferite_scanner_lineno;
    op->block_depth = ferite_compiler_current_block_depth;
    MARK_VARIABLE_AS_COMPILED( PTR2VAR(op->opdata) );

    req = (FeriteBkRequest *)ferite_stack_pop( ferite_fwd_look_stack );
    FUD(("Popped %d off request stack\n", req->type));
    req->reqop->addr = addr;
    MARK_VARIABLE_AS_COMPILED( PTR2VAR(req->reqop->opdata) );
    ferite_destroy_request( req );

    ferite_stack_push( ferite_fwd_look_stack, req2 ); /* jump the fix block */

    FE_LEAVE_FUNCTION( NOWT );
}

void ferite_do_after_fix_block()
{
    FeriteBkRequest *req = NULL;
    FeriteOp *op = NULL;
    int addr;

   /* we dont have an else block */
    FE_ENTER_FUNCTION;
   /* the address to jump over the fix block */
    addr = ferite_get_next_op_loc( CURRENT_FUNCTION->ccode );
    op = ferite_get_next_op_address( CURRENT_FUNCTION->ccode );
    op->OP_TYPE = F_OP_NOP;
    req = (FeriteBkRequest *)ferite_stack_pop( ferite_fwd_look_stack );
    req->reqop->addr = addr;
    MARK_VARIABLE_AS_COMPILED( PTR2VAR(req->reqop->opdata) );
    ferite_destroy_request( req );
    FE_LEAVE_FUNCTION( NOWT );
}

void ferite_do_after_fix_before_else_block()
{
   /* we have an else block */
    FeriteBkRequest *req, *req2;
    FeriteOp *op = NULL;
    int addr;

    FE_ENTER_FUNCTION;
    op = ferite_get_next_op( CURRENT_FUNCTION->ccode );
    op->OP_TYPE = F_OP_JMP;
    op->line = ferite_scanner_lineno;
    op->block_depth = ferite_compiler_current_block_depth;
    req2 = ferite_create_request( op, JMP_ERR_ELSE ); /* jump instruction to jump else block */

   /* this sets the code to jump to the else block if there sn't an error */
    addr = ferite_get_next_op_loc( CURRENT_FUNCTION->ccode );
    op = ferite_get_next_op( CURRENT_FUNCTION->ccode );
    op->OP_TYPE = F_OP_ERR;
    op->addr = -1;
    op->line = ferite_scanner_lineno;
    op->block_depth = ferite_compiler_current_block_depth;
    MARK_VARIABLE_AS_COMPILED( PTR2VAR(op->opdata) );

    req = (FeriteBkRequest *)ferite_stack_pop( ferite_fwd_look_stack );
    FUD(("Popped %d off request stack\n", req->type));
    req->reqop->addr = addr;
    MARK_VARIABLE_AS_COMPILED( PTR2VAR(req->reqop->opdata) );
    ferite_destroy_request( req );

    ferite_stack_push( ferite_fwd_look_stack, req2 ); /* jump the fix block */

    FE_LEAVE_FUNCTION( NOWT );
}

void ferite_do_after_fix_else_statement()
{
    FeriteBkRequest *req = NULL;
    FeriteOp *op = NULL;
    int addr;

   /* we need this to jump over the else block */
    FE_ENTER_FUNCTION;
   /* the address to jump over the else block */
    addr = ferite_get_next_op_loc( CURRENT_FUNCTION->ccode );
    op = ferite_get_next_op_address( CURRENT_FUNCTION->ccode );
    op->OP_TYPE = F_OP_NOP;
    req = (FeriteBkRequest *)ferite_stack_pop( ferite_fwd_look_stack );
    req->reqop->addr = addr;
    MARK_VARIABLE_AS_COMPILED( PTR2VAR(req->reqop->opdata) );
    ferite_destroy_request( req );
    FE_LEAVE_FUNCTION( NOWT );
}

/* the dreaded switch statment */

void ferite_do_pre_switch()
{
    FeriteBkRequest *req = NULL;
    FeriteOp *jmp_op, *pop_op;
    int addr;

    FE_ENTER_FUNCTION;

   /* JMP */
    jmp_op = ferite_get_next_op( CURRENT_FUNCTION->ccode );
    jmp_op->OP_TYPE = F_OP_JMP;
    jmp_op->line = ferite_scanner_lineno;
    jmp_op->block_depth = ferite_compiler_current_block_depth;

   /* POP */
    pop_op = ferite_get_next_op( CURRENT_FUNCTION->ccode );
    pop_op->OP_TYPE = F_OP_POP;
    pop_op->line = ferite_scanner_lineno;
    pop_op->block_depth = ferite_compiler_current_block_depth;

    addr = ferite_get_next_op_loc( CURRENT_FUNCTION->ccode );
    jmp_op->addr = addr;
    MARK_VARIABLE_AS_COMPILED( PTR2VAR(jmp_op->opdata) );

   /* this is where the continue jumps to in a switch statement */
    req = ferite_create_request( jmp_op, SWITCH_CONTINUE_JUMP_TO );
    req->addr = addr;
    ferite_stack_push( ferite_bck_look_stack, req );

    /* We need this to make nested loop structures work */
    req = ferite_create_request( NULL, BREAK_SEPARATOR );
    ferite_stack_push( ferite_break_look_stack, req );
    
    FE_LEAVE_FUNCTION( NOWT );
}

void ferite_do_case_block_start()
{
    FeriteOp *op = NULL;
    FeriteBkRequest *req = NULL;
    int addr = 0;

    FE_ENTER_FUNCTION;

    /* case expr */
    op = ferite_get_next_op( CURRENT_FUNCTION->ccode );
    op->OP_TYPE = F_OP_CASE;
    op->addr = FERITE_OPCODE_case;
    op->opdata = NULL;
    op->line = ferite_scanner_lineno;
    op->block_depth = ferite_compiler_current_block_depth;

    /* BNE */
    op = ferite_get_next_op( CURRENT_FUNCTION->ccode );
    op->OP_TYPE = F_OP_BNE;
    op->line = ferite_scanner_lineno;
    op->block_depth = ferite_compiler_current_block_depth;

    /* here we hook up the addresses such that we come here if we hit this case block */
    addr = ferite_get_next_op_loc( CURRENT_FUNCTION->ccode );
    req = ferite_stack_pop( ferite_fwd_look_stack );
    if( req == NULL || req->type != SWITCH_NEXT_CASE_BLOCK )
    {
        if( req != NULL )
        {
            ferite_stack_push( ferite_fwd_look_stack, req );
        }
    }
    else
    {
        req->reqop->addr = addr;
        MARK_VARIABLE_AS_COMPILED( PTR2VAR(req->reqop->opdata) );
        ferite_destroy_request( req );
    }

   /* this sets up the branch to the next case block if the case statement doesn't match */
    req = ferite_create_request( op, SWITCH_NEXT_CASE );
    ferite_stack_push( ferite_fwd_look_stack, req );

    FE_LEAVE_FUNCTION( NOWT );
}

void ferite_do_default_block_start()
{
    FeriteBkRequest *req = NULL;
    int addr = 0;

    FE_ENTER_FUNCTION;

   /* here we hook up the addresses such that we come here if we hit this case block */
    addr = ferite_get_next_op_loc( CURRENT_FUNCTION->ccode );
    req = ferite_stack_pop( ferite_fwd_look_stack );
    if( req == NULL || req->type != SWITCH_NEXT_CASE_BLOCK )
    {
        if( req != NULL )
        {
            ferite_stack_push( ferite_fwd_look_stack, req );
        }
    }
    else
    {
        req->reqop->addr = addr;
        MARK_VARIABLE_AS_COMPILED( PTR2VAR(req->reqop->opdata) );
        ferite_destroy_request( req );
    }
    FE_LEAVE_FUNCTION( NOWT );
}

void ferite_do_case_block_end()
{
    FeriteOp *op = NULL;
    FeriteBkRequest *req = NULL;
    int addr= 0;

    FE_ENTER_FUNCTION;

   /* jump over the next case block */
    op = ferite_get_next_op( CURRENT_FUNCTION->ccode );
    op->OP_TYPE = F_OP_JMP;
    op->line = ferite_scanner_lineno;
    op->block_depth = ferite_compiler_current_block_depth;

   /* link up the jump to the next case statement */
    addr = ferite_get_next_op_loc( CURRENT_FUNCTION->ccode );
    req = ferite_stack_pop( ferite_fwd_look_stack );
    if( req->type != SWITCH_NEXT_CASE )
    {
        ferite_stack_push( ferite_fwd_look_stack, req );
    }
    else
    {
        req->reqop->addr = addr;
        MARK_VARIABLE_AS_COMPILED( PTR2VAR(req->reqop->opdata) );
        ferite_destroy_request( req );
    }

   /* this is for the afore mentioned jump */
    req = ferite_create_request( op, SWITCH_NEXT_CASE_BLOCK );
    ferite_stack_push( ferite_fwd_look_stack, req );

    FE_LEAVE_FUNCTION( NOWT );
}

void ferite_do_post_switch()
{
    FeriteOp *op = NULL;
    FeriteBkRequest *req = NULL;
    int addr;

    FE_ENTER_FUNCTION;

    /* here we hook up the addresses such that we come here if we hit this case block */
    addr = ferite_get_next_op_loc( CURRENT_FUNCTION->ccode );
    req = ferite_stack_pop( ferite_fwd_look_stack );
    if( req != NULL )
    {
        if( req->type != SWITCH_NEXT_CASE_BLOCK )
        {
            /*    fprintf( stderr, "foobar %d:%d\n", __LINE__, req->type );*/
            ferite_stack_push( ferite_fwd_look_stack, req );
        }
        else
        {
            req->reqop->addr = addr;
            MARK_VARIABLE_AS_COMPILED( PTR2VAR(req->reqop->opdata) );
            ferite_destroy_request( req );
        }
    }

    /* hook up breaks to jump to the pop */
    addr = ferite_get_next_op_loc( CURRENT_FUNCTION->ccode );
    ferite_process_breaks( SWITCH_CONTINUE_JUMP_TO, addr );
    
    req = ferite_stack_pop( ferite_bck_look_stack );
    ferite_destroy_request( req );
    
    /* POP */
    op = ferite_get_next_op( CURRENT_FUNCTION->ccode );
    op->OP_TYPE = F_OP_POP;
    op->block_depth = ferite_compiler_current_block_depth;
    op->line = ferite_scanner_lineno;

    FE_LEAVE_FUNCTION( NOWT );
}

/**
 * foreach 
 */
void ferite_do_foreach_start()
{
    FeriteBkRequest *req = NULL;
    FeriteOp *op = NULL;
    int next_op;

    FE_ENTER_FUNCTION;
    next_op = ferite_get_next_op_loc( CURRENT_FUNCTION->ccode );
    op = ferite_get_next_op_address( CURRENT_FUNCTION->ccode );
    op->OP_TYPE = F_OP_NOP;
    req = ferite_create_request( op, WHILE_JUMP_TO );
    req->addr = next_op;
    ferite_stack_push( ferite_bck_look_stack, req );

    /* We need this to make nested loop structures work */
    req = ferite_create_request( NULL, BREAK_SEPARATOR );
    ferite_stack_push( ferite_break_look_stack, req );
    
    FE_LEAVE_FUNCTION( NOWT );
}

void ferite_do_foreach_next( int count )
{
    FeriteBkRequest *req = NULL;
    FeriteOp *op = NULL;

    FE_ENTER_FUNCTION;
    /* do the for each */
    ferite_do_many_op( FERITE_OPCODE_foreach, count );
    
    /* if the foreach ahs finished */
    op = ferite_get_next_op( CURRENT_FUNCTION->ccode );
    op->OP_TYPE = F_OP_BNE;
    op->line = ferite_scanner_lineno;
    op->block_depth = ferite_compiler_current_block_depth;
    req = ferite_create_request( op, WHILE_JUMP_BLOCK );
    ferite_stack_push( ferite_fwd_look_stack, req );
    FE_LEAVE_FUNCTION( NOWT );
}

void ferite_do_foreach_end()
{
    FeriteBkRequest *req = NULL;
    FeriteOp *op = NULL;
    int next_addr;

    FE_ENTER_FUNCTION;
    /* hook up jump back */
    op = ferite_get_next_op( CURRENT_FUNCTION->ccode );
    op->OP_TYPE = F_OP_JMP;
    op->line = ferite_scanner_lineno;
    op->block_depth = ferite_compiler_current_block_depth;

    next_addr = ferite_get_next_op_loc( CURRENT_FUNCTION->ccode );
    ferite_process_breaks( WHILE_JUMP_TO, next_addr );

    req = (FeriteBkRequest *)ferite_stack_pop( ferite_bck_look_stack );
    FUD(("Popped %d off request stack\n", req->type));
    /*   fprintf( stderr, "while end type %d\n", req->type );*/
    op->addr = req->addr;
    MARK_VARIABLE_AS_COMPILED( PTR2VAR(op->opdata) );
    ferite_destroy_request( req );

    /* hook up jump over */
    op = ferite_get_next_op_address( CURRENT_FUNCTION->ccode );
    op->OP_TYPE = F_OP_NOP;
    req = (FeriteBkRequest *)ferite_stack_pop( ferite_fwd_look_stack );
    FUD(("Popped %d off request stack\n", req->type));
    req->reqop->addr = next_addr;
    MARK_VARIABLE_AS_COMPILED( PTR2VAR(req->reqop->opdata) );
    ferite_destroy_request( req );
    FE_LEAVE_FUNCTION( NOWT );
}

/* loop control */
void ferite_do_break()
{
    FeriteOp *op = NULL;
    FeriteBkRequest *req, *breq;
    int i;

    FE_ENTER_FUNCTION;
    for( i = ferite_bck_look_stack->stack_ptr; i > 0; i-- )
    {
        req = ferite_bck_look_stack->stack[i];
        if( req->type == FOR_INCR_START || req->type == WHILE_JUMP_TO || req->type == DO_START || req->type == SWITCH_CONTINUE_JUMP_TO )
        {
            op = ferite_get_next_op( CURRENT_FUNCTION->ccode );
            op->OP_TYPE = F_OP_JMP;
            op->line = ferite_scanner_lineno;
            breq = ferite_create_request( op, req->type );
            ferite_stack_push( ferite_break_look_stack, breq );
            break;
        }
    }
    if( op == NULL )
    {
        ferite_warning( CURRENT_SCRIPT, "Trying to use break in non-looping block. (ignoring)\n" );
    }
    FE_LEAVE_FUNCTION(NOWT);
}

/* for this we simply go up the ferite_bck_look_stack and get the address of the first
 * structure :) */
void ferite_do_continue()
{
    FeriteOp *op = NULL;
    FeriteBkRequest *req = NULL;
    int i;

    FE_ENTER_FUNCTION;
    for( i = ferite_bck_look_stack->stack_ptr; i > 0; i-- )
    {
        req = ferite_bck_look_stack->stack[i];
        if( req->type == FOR_INCR_START || req->type == WHILE_JUMP_TO || req->type == DO_START || req->type == SWITCH_CONTINUE_JUMP_TO )
        {
            op = ferite_get_next_op( CURRENT_FUNCTION->ccode );
            op->OP_TYPE = F_OP_JMP;
            op->addr = req->addr;
            op->line = ferite_scanner_lineno;
            MARK_VARIABLE_AS_COMPILED( PTR2VAR(op->opdata) );
            break;
        }
    }
    if( op == NULL )
    {
        ferite_warning( CURRENT_SCRIPT, "Trying to use continue in non-looping block. (ignoring)\n" );
    }
    FE_LEAVE_FUNCTION( NOWT );
}

void ferite_process_breaks( int starttag, int address )
{
    FeriteBkRequest *req = NULL;

    FE_ENTER_FUNCTION;
    req = ferite_stack_top( ferite_break_look_stack );
    while( req != NULL && req->type == starttag )
    {
        ferite_stack_pop( ferite_break_look_stack );

        req->reqop->addr = address;
        MARK_VARIABLE_AS_COMPILED( PTR2VAR(req->reqop->opdata ) );

        ferite_destroy_request( req );
        req = ferite_stack_top( ferite_break_look_stack );
    }

    if( req->type == BREAK_SEPARATOR )
    {
        ferite_stack_pop( ferite_break_look_stack );
        ferite_destroy_request( req );
    }
    
    FE_LEAVE_FUNCTION( NOWT );
}

/* this is builder stuff */
void ferite_do_create_builder_variable( char *name, char *data )
{
    FeriteVariable *var = NULL;
    FeriteNamespaceBucket *nsb = NULL;
    FeriteScript *script = NULL;
    
    FE_ENTER_FUNCTION;
    if( ferite_keep_native_function_data != 0 )
    {
        nsb = ferite_namespace_element_exists( CURRENT_SCRIPT, CURRENT_SCRIPT->mainns, name );
        if( nsb != NULL )
        {
            /* the variable already exists, so we simply extend it :) */
            var = nsb->data;
            ferite_str_data_cat( VAS(var), data, strlen(data) );
            FE_LEAVE_FUNCTION(NOWT);
        }
        ferite_register_ns_variable( CURRENT_SCRIPT, CURRENT_SCRIPT->mainns,
                                     fe_new_str( name, data, strlen(data), FE_CHARSET_DEFAULT ) );
    }
    FE_LEAVE_FUNCTION(NOWT);
}

/**
 * !end
 */
