/*
 * Copyright (C) 2000-2002 Chris Ross and various contributors
 * Copyright (C) 1999-2000 Chris Ross
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * o Redistributions of source code must retain the above copyright notice, this
 *   list of conditions and the following disclaimer.
 * o Redistributions in binary form must reproduce the above copyright notice,
 *   this list of conditions and the following disclaimer in the documentation
 *   and/or other materials provided with the distribution.
 * o Neither the name of the ferite software nor the names of its contributors may
 *   be used to endorse or promote products derived from this software without
 *   specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

/* This is left undocumented as this is controlled by the abstract interface */

#include "ferite.h"
#include <sys/timeb.h>
#include "aphex.h"

#ifdef THREAD_SAFE
# define LOCK_GC     aphex_mutex_lock( gc->lock )
# define UNLOCK_GC   aphex_mutex_unlock( gc->lock )
#else
# define LOCK_GC
# define UNLOCK_GC
#endif

#define FE_GENERATION_GC_DEFAULT_SIZE 40

FeriteGCGeneration *ferite_generation_create()
{
    FeriteGCGeneration *ptr = NULL;

    FE_ENTER_FUNCTION;
    ptr = fmalloc(sizeof(FeriteGCGeneration));
    ptr->size = FE_GENERATION_GC_DEFAULT_SIZE;
    ptr->next_free = 0;
    ptr->contents = fcalloc(sizeof(FeriteObject*)*ptr->size,1);
    ptr->younger = NULL;
    ptr->older = NULL;
    FE_LEAVE_FUNCTION(ptr);
}

void ferite_generation_destroy( FeriteScript *script, FeriteGCGeneration *g )
{
    int i = 0;

    FE_ENTER_FUNCTION;
    if( g != NULL )
    {
        if( g->older != NULL )
          ferite_generation_destroy( script, g->older );
        for( i = 0; i < g->next_free; i++ )
          if( g->contents[i] != NULL )
            ferite_delete_class_object( script, g->contents[i], FE_TRUE );
        ffree( g->contents );
        ffree( g );
    }
    FE_LEAVE_FUNCTION(NOWT);
}

/*!
 * \fn void ferite_init_gc()
 * \brief Initialise the garbage collector
 */
void ferite_init_generation_gc( FeriteScript *script )
{
    FE_ENTER_FUNCTION;
    if( script->gc == NULL )
        script->gc = ferite_generation_create();
    FE_LEAVE_FUNCTION( NOWT );
}

/*!
 * \fn void ferite_deinit_generation_gc()
 * \brief Clean the garbage collector up and shut it down
 */
void ferite_deinit_generation_gc( FeriteScript *script )
{
    FE_ENTER_FUNCTION;
    if( script->gc != NULL )
    {
        FUD(("GC: +-----------------------------+\n"));
        FUD(("GC: | CLEANING UP Generational GC |\n"));
        FUD(("GC: +-----------------------------+\n"));
        ferite_generation_destroy( script, script->gc );
	script->gc = NULL;
    }
    FE_LEAVE_FUNCTION( NOWT );
}

/*!
 * \fn void ferite_add_to_generation_gc( FeriteObject *obj )
 * \brief Add an object to the GC
 * \param obj The FeriteObject to add
 *
 * This function will automaticaly grow the gc if it runs out of space
 */
void ferite_add_to_generation_gc( FeriteScript *script, FeriteObject *obj )
{
    FeriteGCGeneration *g = script->gc;
    FE_ENTER_FUNCTION;
    if( g->next_free >= g->size )
    { /* we have no more space in this generation */
        ferite_check_gc_generation( script, g );
    }
   /* add the object */
    g->contents[g->next_free++] = obj;
    FE_LEAVE_FUNCTION( NOWT );
}

void ferite_check_generation_gc( FeriteScript *script )
{
    FE_ENTER_FUNCTION;
    ferite_check_gc_generation( script, script->gc );
    FE_LEAVE_FUNCTION(NOWT);
}

/*!
 * \fn void ferite_gc_check_generation()
 * \brief Check the GC and free up any objects that have yet to be cleaned up
 */
void ferite_check_gc_generation( FeriteScript *script, FeriteGCGeneration *g )
{
    int i = 0;
    FeriteGCGeneration *o = NULL;

    FE_ENTER_FUNCTION;
    FE_ASSERT( script != NULL && script->gc != NULL );
    for( i = 0; i < g->next_free; i++ )
    {
        if( g->contents[i] != NULL && g->contents[i]->refcount <= 0 )
          ferite_delete_class_object( script, g->contents[i], FE_TRUE );
        else /* the object isnt' ready to free, lets add it to the next level */
        {
            o = g->older;
            if( o != NULL )
            {
                if( o->next_free >= o->size )
                {
                    /*
                     * not enough space in the older generation so we check that
                     * generation for objects that can be deleted
                     */
                    /*printf( "not enough space in older, checking it\n" );*/
                    ferite_check_gc_generation( script, o );
                }
            }
            else /* we dont have an older generation so create a new one */
            {
                /*printf( "older == NULL\n" );*/
                o = ferite_generation_create();
                /*printf( "o=%p, o->contents=%p\n", o, o->contents );*/
                g->older = o;
                /*printf( "g->older=%p\n", g->older );*/
                o->younger = g;
                /*printf( "o->younger=%p\n", o->younger );*/
            }

            /* put the object in the next generation */
            o->contents[o->next_free++] = g->contents[i];
        }
        g->contents[i] = NULL;
    }
   /* by the time we have got here we should have an empty generation, so
    * we just set the next_free to 0 */
    g->next_free = 0;

    /* If the older generation is empty and is the top generation */
    if( g->older != NULL && g->older->older == NULL && g->older->next_free == 0 )
    {
        /*printf( "destroying older generation\n" );*/
        ferite_generation_destroy( script, g->older );
        g->older = NULL;
    }
    FE_LEAVE_FUNCTION( NOWT );
}

/*!
 * \fn void ferite_gc_generation_merge()
 * \brief Merge dead threads GC to parents GC, clear up all we can first
 * */
void ferite_merge_generation_gc( FeriteScript *script, void *g )
{
    int i;
    FeriteGCGeneration *gc = g;
    for( i = 0; i < gc->next_free; i++ )
    {
	if( gc->contents[i] != NULL && gc->contents[i]->refcount <= 0 )
            ferite_delete_class_object( script, gc->contents[i], FE_TRUE );
	else if( gc->contents[i] != NULL )
	{
	    ferite_add_to_generation_gc( script, gc->contents[i] );
	}
    }
    if( gc->older )
	ferite_merge_generation_gc( script, gc->older );
    ffree( gc->contents );
    ffree( gc );
}
