/*
 * Copyright (C) 2000-2002 Chris Ross and various contributors
 * Copyright (C) 1999-2000 Chris Ross
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * o Redistributions of source code must retain the above copyright notice, this
 *   list of conditions and the following disclaimer.
 * o Redistributions in binary form must reproduce the above copyright notice,
 *   this list of conditions and the following disclaimer in the documentation
 *   and/or other materials provided with the distribution.
 * o Neither the name of the ferite software nor the names of its contributors may
 *   be used to endorse or promote products derived from this software without
 *   specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include "ferite.h"
#include "triton.h"
#include "aphex.h"

/**
 * !group Modules
 * !description This set of functions are useful for module writers.
 */

FeriteModule    *ferite_root_module = NULL;
FeriteModule    *ferite_current_module = NULL;
FeriteHash      *ferite_native_function_hash = NULL;
FeriteStack     *ferite_module_search_list = NULL;
FeriteStack     *ferite_module_preload_list = NULL;
char            *ferite_native_search_path = NULL;

extern int       ferite_compile_error;

char *ferite_script_extensions[] = { ".fec", ".fe", NULL };

FeriteModule *ferite_create_module( char *name, char *filename )
{
    FeriteModule *ptr = NULL;
    
    FE_ENTER_FUNCTION;
    ptr = fmalloc( sizeof(FeriteModule) );
    ptr->name = fstrdup( name );
    ptr->filename = fstrdup( filename );
    ptr->handle = NULL;
    ptr->module_init = NULL;
    ptr->module_deinit = NULL;
    ptr->module_unregister = NULL;
    ptr->module_register = NULL;
    ptr->next = NULL;
    FE_LEAVE_FUNCTION( ptr );
}

int ferite_init_module_list()
{
    FE_ENTER_FUNCTION;
    /* initialise */
    FUD(( "MODULE LOADER: initialising module loader... " ));
    if( triton_init() != 0 )
    {
        FE_LEAVE_FUNCTION( 0 );
    }
    else
    {
        ferite_native_search_path = NULL;
        ferite_module_search_list = ferite_create_stack( NULL, 5 );
        ferite_module_preload_list = ferite_create_stack( NULL, 5 );
        ferite_native_function_hash = ferite_create_hash( NULL, 64 );
        ferite_root_module = ferite_create_module( "ferite_root_module", "" );
        ferite_current_module = ferite_root_module;
    }
    FE_LEAVE_FUNCTION( 1 );
}

void ferite_destroy_module_list( FeriteModule *mod )
{
    FE_ENTER_FUNCTION;
    if( mod != NULL )
    {
        if( mod->next != NULL )
          ferite_destroy_module_list( mod->next );

        FUD(( "MODULE LOADER: deleting module \"%s\"\n", mod->name ));

        mod->module_init = NULL;
        mod->module_deinit = NULL;

        if( mod->module_unregister != NULL )
        {
            (mod->module_unregister)();
        }

        mod->module_unregister = NULL;

        ffree( mod->filename );
        ffree( mod->name );
        ffree( mod );
        mod = NULL;
    }
    FE_LEAVE_FUNCTION( NOWT );
}

void ferite_deinit_module_list()
{
    int i = 0;

    FE_ENTER_FUNCTION;

   /* tidy up the search paths */
    for( i = 0; i <= ferite_module_search_list->stack_ptr; i++ )
    {
        if( ferite_module_search_list->stack[i] != NULL )
        {
            ffree( ferite_module_search_list->stack[i] );
        }
    }
    ferite_delete_stack( NULL, ferite_module_search_list );
    ferite_module_search_list = NULL;

   /* tidy up the preload lists */
    for( i = 0; i <= ferite_module_preload_list->stack_ptr; i++ )
    {
        if( ferite_module_preload_list->stack[i] != NULL )
        {
            ffree( ferite_module_preload_list->stack[i] );
        }
    }
    ferite_delete_stack( NULL, ferite_module_preload_list );
    ferite_module_preload_list = NULL;

    if( ferite_native_search_path != NULL )
      ffree( ferite_native_search_path );

    FUD(( "MODULE LOADER: closing all opened modules\n" ));
    ferite_module_delete_native_function_list();

    ferite_destroy_module_list( ferite_root_module );
    ferite_root_module = NULL;

    FUD(( "MODULE LOADER: closing down module loader\n" ));

    FE_LEAVE_FUNCTION( NOWT );
}

/**
 * !function ferite_load_module
 * !declaration int ferite_load_module( FeriteScript *script, FeriteNamespace *ns, char *name )
 * !brief Load a module (script or native)
 * !param FeriteScript *script The script to load the module into
 * !param FeriteNamespace *ns The namespace in which the module will be loaded
 * !param char *name The name of the module
 * !return 0 on fail, otherwise greater than 0 for sucess
 */
int ferite_load_module( FeriteScript *script, FeriteNamespace *ns, char *name )
{
    char *extension = NULL;
    char *filename = NULL;
    int result = 0;

    FE_ENTER_FUNCTION;

    extension = strrchr( name, '.' );         /* get the start of the extension */
    filename = strrchr( name, DIR_DELIM );    /* get the start of the filename */

    FUD(( "-------------------->" ));
    FUD(( "we have been asked to include %s\n", name ));

    if( extension == NULL ) /* no extension, we first try to load a native module
                             * then we try and load a script with that name */
    {
        result = ferite_load_script_module( script, name, 1 );
        if( result == 0 )
          ferite_error( script, 0, "Unable to find module '%s'\n", name );
        FE_LEAVE_FUNCTION(result);
    }
    else
    {
        if( strcmp( extension, ".lib" ) == 0 ) /* if it's a library */
        {
            FE_LEAVE_FUNCTION( ferite_load_native_module( name, script ) );
        }
        FE_LEAVE_FUNCTION( ferite_load_script_module( script, name, 0 ) );
    }
    FUD(( "<-------------------" ));

    FE_LEAVE_FUNCTION(0);
}

int ferite_load_script_module( FeriteScript *script, char *name, int do_extension )
{
    char filename[PATH_MAX];
    int i = 0, j = 0, result = 0;

    FE_ENTER_FUNCTION;
    /* so now we go though file extensions that it could be - in all all the search paths
     * .fec .feh .fe */
    for( i = 0; i <= ferite_module_search_list->stack_ptr; i++ )
    {
        if( ferite_module_search_list->stack[i] != NULL )
        {
            /* we check this path */
            if( do_extension == 1 )
            {
                for( j = 0; ferite_script_extensions[j] != NULL; j++ )
                {
                    snprintf( filename, PATH_MAX, "%s%c%s%s", (char *)ferite_module_search_list->stack[i], DIR_DELIM, name, ferite_script_extensions[j] );
                    result = ferite_do_load_script( filename );
                    if( result > -1 ){
                        FE_LEAVE_FUNCTION( result );
                    }
                }
            }
            else
            {
                snprintf( filename, PATH_MAX, "%s%c%s", (char *)ferite_module_search_list->stack[i], DIR_DELIM, name );
                result = ferite_do_load_script(filename);
                if( result > -1 ){
                    FE_LEAVE_FUNCTION( result );
                }
            }
        }
    }

    if( do_extension == 1 )
    {
        for( j = 0; ferite_script_extensions[j] != NULL; j++ )          
        {
            snprintf( filename, PATH_MAX, "%s%s", name, ferite_script_extensions[j] );
            result = ferite_do_load_script(filename);
            if( result > -1 ){
                FE_LEAVE_FUNCTION( result );
            }
        }
    }
    else
    {
        result = ferite_do_load_script(name);
        if( result > -1 ){
            FE_LEAVE_FUNCTION( result );
        }
    }
    ferite_error( script, 0, "Unable to find script module '%s'\n", name );

    FE_LEAVE_FUNCTION(0);
}

FeriteModule *ferite_module_find( char *name )
{
    FeriteModule *ptr = NULL;

    FE_ENTER_FUNCTION;
    FUD(( "MODULE LOADER: Trying to locate module \"%s\"\n", name ));
   /* check it'#s not already loaded */
    for( ptr = ferite_root_module; ptr != NULL; ptr = ptr->next )
    {
        FUD(("MODULE LOADER: Checking \"%s\" and \"%s\"\n", name, ptr->name ));
        if( strcmp( name, ptr->name ) == 0 )
        {
            FUD(( "MODULE LOADER: Module \"%s\" is allready loaded\n", name ));
            FE_LEAVE_FUNCTION(ptr); /* we return 1 because we have it :) */
        }
    }
    FE_LEAVE_FUNCTION(NULL);
}

int ferite_unload_native_module( char *name, FeriteScript *script )
{
    FeriteModule *ptr = NULL;
    
    FE_ENTER_FUNCTION;
    ptr = ferite_module_find( name );
    if( ptr != NULL && ptr->module_deinit != NULL )
    {
        (ptr->module_deinit)( script );
        FE_LEAVE_FUNCTION(1);
    }
    FE_LEAVE_FUNCTION(0);
}

int ferite_load_native_module( char *module_name, FeriteScript *script )
{
    char buf[PATH_MAX], real_buf[PATH_MAX], *extension = NULL, *name = NULL;
    void *handle = NULL;
    FeriteModule *ptr = NULL;

    FE_ENTER_FUNCTION;

    name = fstrdup( module_name );
    extension = strrchr( name, '.' );
    if( extension != NULL ) /* get rid of the extension */
      *extension = '\0';

    if( ferite_compiler_include_in_list( script, name ) == 0 )
    {
        ptr = ferite_module_find( name );
        if( ptr != NULL )
        {
            (ptr->module_init)( script );
            ffree( name );
            FE_LEAVE_FUNCTION(1); /* we return 1 because we have it :) */
        }

        /* we dont already have it, lets try to open it */
        if( ferite_native_search_path != NULL )
        {
            sprintf( buf, "%s%c%s", ferite_native_search_path, DIR_DELIM, name );
            
            /* now we try to load this module */
            FUD(( "MODULE LOADER: trying to load native module %s\n", buf ));
            handle = (void *)triton_openext( buf );
            if( handle == NULL )
            {
                FUD(( "MODULE LOADER: checking for file's existance: %s\n", real_buf ));
                if( aphex_file_exists( "%s%s", buf, triton_library_ext() ) == 1 )
                {
                    ferite_error( script, 0, "Library Loader: %s\n", triton_error() );
                    ffree( name );
                    FE_LEAVE_FUNCTION(0);
                }
            }
        }

        if( handle == NULL ) /* we failed to open, lets just try opening here and now */
        {
            sprintf( buf, "%s", name );
            
            FUD(( "MODULE LOADER: trying to load native module %s\n", buf ));
            handle = (void *)triton_openext( buf );
            if( handle == NULL )
            {
                sprintf( real_buf, "%s%s", buf, triton_library_ext() );                
                FUD(( "MODULE LOADER: checking for file's existance: %s\n", real_buf ));
                if( aphex_file_exists( "%s%s", buf, triton_library_ext() ) == 1 )
                {
                    ferite_error( script, 0, "Library Loader: %s\n", triton_error() );
                    ffree( name );
                    FE_LEAVE_FUNCTION(0);
                }
                else
                {
                    ferite_error( script, 0, "Library Loader: Can't find module '%s'\n", module_name );
                    ffree( name );
                    FE_LEAVE_FUNCTION(0);
                }
            }
        }

        /* if we have a handle go for it! */
        if( handle != NULL )
        {
            FUD(( "MODULE LOADER: loaded!\n" ));

            ferite_current_module->next = ferite_create_module( name, buf );
            FUD(( "MODULE LOADER: Loading module \"%s\" from file \"%s\"\n", name, buf ));
            ferite_current_module->next->handle = ptr;

            /* register function */
            sprintf( buf, "%s_register", name );
            ferite_current_module->next->module_register = (void (*)())triton_getsym( handle, buf );
            if( ferite_current_module->next->module_register == NULL )
            { /* we dont have a register function. we are in the sh1t */
                triton_close( handle );  /* close the lib */
                ferite_destroy_module_list( ferite_current_module->next );
                ferite_current_module->next = NULL;
                ferite_error( script, 0, "Library Loader: can't find '%s' in module '%s', ferite needs this - stopping.", buf, name );
                ffree( name );
                FE_LEAVE_FUNCTION(0);
            }
            ferite_current_module = ferite_current_module->next;

            /* init function */
            sprintf( buf, "%s_init", name );
            ferite_current_module->module_init = (void (*)(FeriteScript*))triton_getsym( handle, buf );

            /* deinit function -> we dont use this just yet */
            sprintf( buf, "%s_deinit", name );
            ferite_current_module->module_deinit = (void (*)(FeriteScript*))triton_getsym( handle, buf );

            /* module_unregister */
            sprintf( buf, "%s_unregister", name );
            ferite_current_module->module_unregister = (void (*)())triton_getsym( handle, buf );

            (ferite_current_module->module_register)();
            FUD(( "MODULE LOADER: Calling %s::module_init()\n", name ));
            (ferite_current_module->module_init)( script );

            ferite_stack_push( script->include_list, fstrdup(name) );
        }
        else
        {
            ferite_error( script, 0, "Library Loader: can't load module '%s', %s.", name, triton_error() );
            ffree( name );
            FE_LEAVE_FUNCTION(0);
        }
    }
    ffree( name );
    FE_LEAVE_FUNCTION(1);
}

/**
 * !function ferite_add_library_search_path
 * !declaration void ferite_add_library_search_path( char *path )
 * !brief Add a path to the list of paths that ferite searches for scripts
 * !param char *path The path to add
 */
void ferite_add_library_search_path( char *path )
{
    FE_ENTER_FUNCTION;

    /* We dont want to have the dirdelim at the end */
    if( path[strlen(path)-1] == DIR_DELIM )
        path[strlen(path)-1] = '\0';
    
    ferite_stack_push( ferite_module_search_list, fstrdup(path) );
    FE_LEAVE_FUNCTION(NOWT);
}

void ferite_pop_library_search_path()
{
    char *path = NULL;

    FE_ENTER_FUNCTION;
    path = ferite_stack_pop( ferite_module_search_list );
    if( path != NULL )
      ffree( path );
    FE_LEAVE_FUNCTION(NOWT);
}

/**
 * !function ferite_set_library_native_path
 * !declaration void ferite_set_library_native_path( char *path )
 * !brief Set the path where ferite looks for the native libraries to load
 * !param char *path The path where they libraries are located
 */
void ferite_set_library_native_path( char *path )
{
    FE_ENTER_FUNCTION;
    /* We dont want to have the dirdelim at the end */
    if( path[strlen(path)-1] == DIR_DELIM )
        path[strlen(path)-1] = '\0';
    
    ferite_native_search_path = fstrdup(path);
    FE_LEAVE_FUNCTION(NOWT);
}

void ferite_module_register_native_function( char *lookupname, FeriteVariable *(*ptr)( FeriteScript*,FeriteFunction*,FeriteVariable** ) )
{
    FeriteNativeFunctionRecord *record = NULL;

    FE_ENTER_FUNCTION;
    if( ferite_native_function_hash != NULL )
    {
        record = ferite_hash_get( NULL, ferite_native_function_hash, lookupname );
        if( record == NULL )
        {
            FUD(( "MODULE: Registering native function '%s'\n", lookupname ));
            record = fmalloc( sizeof( FeriteNativeFunctionRecord ) );
            record->function = ptr;
            ferite_hash_add( NULL, ferite_native_function_hash, lookupname, record );
        }
        else
          fprintf( stderr, "The native function '%s' has already exists, will not re-register.\n", lookupname );
    }
    FE_LEAVE_FUNCTION(NOWT);
}

/** 
 * !function ferite_module_register_fake_module
 * !declaration void ferite_module_register_fake_module( char *name, void *_register, void *_unregister, void *_init, void *_deinit )
 * !brief Register a fake native module.
 * !param char *name The name of the module. Even though this is a fake module it must be post fixed with the ".lib" extension.
 * !param void *_register A pointer to the register function - can be NULL
 * !param void *_unregister A pointer to the unregister function - can be NULL
 * !param void *_init A pointer to the init function - must not be NULL
 * !param void *_deinit A pointer to the deinit function - must not be NULL
 * !description This function is used to register fake modules from an application. This allows you to
 *              to setup a module to be loaded from an application. Combined with ferite_module_do_preload
 *              this will provide a method of exporting api to scripts using standard module methods.
 */
void ferite_module_register_fake_module( char *name, void *_register, void *_unregister, void *_init, void *_deinit )
{
    FE_ENTER_FUNCTION;
    ferite_current_module->next = ferite_create_module( name, NULL );    
    ferite_current_module->next->handle = NULL;
    
    ferite_current_module->next->module_register = _register;
    ferite_current_module->next->module_unregister = _unregister;
    ferite_current_module->next->module_init = _init;
    ferite_current_module->next->module_deinit = _deinit;
    (ferite_current_module->next->module_register)();
    
    ferite_current_module = ferite_current_module->next;
    FE_LEAVE_FUNCTION(NOWT);    
}

void ferite_delete_native_function_record( FeriteScript *script, void *data )
{
    FE_ENTER_FUNCTION;
    if( data != NULL )
      ffree( data );
    FE_LEAVE_FUNCTION(NOWT);
}

void ferite_module_delete_native_function_list()
{
    FE_ENTER_FUNCTION;
    if( ferite_native_function_hash != NULL )
    {
        ferite_delete_hash( NULL, ferite_native_function_hash, ferite_delete_native_function_record );
        ferite_native_function_hash = NULL;
    }
    FE_LEAVE_FUNCTION(NOWT);
}

void *ferite_module_find_function( char *name )
{
    void *ptr = NULL;
    FeriteNativeFunctionRecord *record = NULL;

    FE_ENTER_FUNCTION;
    record = ferite_hash_get( NULL, ferite_native_function_hash, name );
    if( record != NULL )
      ptr = record->function;
    FE_LEAVE_FUNCTION(ptr);
}

/**
 * !function ferite_module_add_preload
 * !declaration void ferite_module_add_preload( char *name )
 * !brief Add a module to the list of libraries that are loaded when a script is compiled
 * !param char *name The name of the module
 */
void ferite_module_add_preload( char *name )
{
    FE_ENTER_FUNCTION;
    ferite_stack_push( ferite_module_preload_list, fstrdup(name) );
    FE_LEAVE_FUNCTION(NOWT);
}

/**
 * !function ferite_module_do_preload
 * !declaration int ferite_module_do_preload( FeriteScript *script )
 * !brief Preload the modules into the script
 * !param FeriteScript *script The script to load into
 */
int ferite_module_do_preload( FeriteScript *script )
{
    int i = 0;

    FE_ENTER_FUNCTION;
    for( i = 0; i <= ferite_module_preload_list->stack_ptr; i++ )
    {
        if( ferite_module_preload_list->stack[i] != NULL )
        {
            if( ferite_load_module( script, script->mainns, ferite_module_preload_list->stack[i] ) <= 0 )
            {
                ferite_error( script, 0, "Attempt to pre-load module '%s' failed.\n", (char *)ferite_module_preload_list->stack[i] );
                FE_LEAVE_FUNCTION(0);
            }
        }
    }
    FE_LEAVE_FUNCTION(1);
}

/**
 * !end
 */
