/*
 * Copyright (C) 2000-2002 Chris Ross
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * o Redistributions of source code must retain the above copyright notice, this
 *   list of conditions and the following disclaimer.
 * o Redistributions in binary form must reproduce the above copyright notice,
 *   this list of conditions and the following disclaimer in the documentation
 *   and/or other materials provided with the distribution.
 * o Neither the name of the ferite software nor the names of its contributors may
 *   be used to endorse or promote products derived from this software without
 *   specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include "ferite.h"

/**
 * !group Obj Base Class
 * !description These set of functions are used to setup the base class 'Obj' within
 *              the ferite system.
 */

/**
 * !function ferite_add_object_class
 * !declaration void ferite_add_object_class( FeriteScript *script )
 * !brief This adds the Object class that all other classes inherit from.
 * !param FeriteScript *script This script object in which to place the class
 */
void ferite_add_object_class( FeriteScript *script )
{
    FeriteClass *klass = NULL;

    FE_ENTER_FUNCTION;
    klass = ferite_register_inherited_class( script, script->mainns, "Obj", NULL );
    fe_create_cls_fnc( script, klass, "copy", ferite_obj_copy, "", FE_FALSE );
    fe_create_cls_fnc( script, klass, "hash", ferite_obj_hash, "", FE_FALSE );
    FE_LEAVE_FUNCTION(NOWT);
}

/** 
 * !function ferite_obj_copy
 * !brief Return a copy of the object 
 */
FE_NATIVE_FUNCTION( ferite_obj_copy )
{
    FeriteVariable *ptr = NULL;
    FeriteObject *self = VAO(params[0]);
    
    FE_ENTER_FUNCTION;
    ptr = ferite_create_object_variable( script, "copy", FE_STATIC );
    
    VAO(ptr) = fmalloc( sizeof( FeriteObject ) );
    
    VAO(ptr)->name = fstrdup( self->name );
    VAO(ptr)->oid = self->oid;
    VAO(ptr)->odata = NULL;
    VAO(ptr)->refcount = 1;
    VAO(ptr)->klass = self->klass;
    
    /* Copy the variables */
    VAO(ptr)->variables = ferite_duplicate_variable_hash( script, self->variables );
    VAO(ptr)->functions = self->functions;
    
    MARK_VARIABLE_AS_DISPOSABLE( ptr );
    
    ferite_add_to_gc( script, VAO(ptr) );
    
    FE_LEAVE_FUNCTION( ptr );
}

/**
 * !function ferite_obj_hash
 * !brief Create a hash key for the object
 */
FE_NATIVE_FUNCTION( ferite_obj_hash )
{
    FeriteVariable *ptr = NULL;
    FeriteObject *self = VAO(params[0]);
    char buf[1024];
    
    FE_ENTER_FUNCTION;
    sprintf( buf, "Object-%s-%p", self->klass->name, self );
    
    ptr = ferite_create_string_variable_from_ptr( script, "Obj.hash()", buf, strlen(buf), FE_CHARSET_DEFAULT, FE_STATIC );
    MARK_VARIABLE_AS_DISPOSABLE( ptr );
    
    FE_LEAVE_FUNCTION( ptr );
}

/**
 * !end
 */
