/*
 * Copyright (C) 2000-2002 Chris Ross and various contributors
 * Copyright (C) 1999-2000 Chris Ross
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * o Redistributions of source code must retain the above copyright notice, this
 *   list of conditions and the following disclaimer.
 * o Redistributions in binary form must reproduce the above copyright notice,
 *   this list of conditions and the following disclaimer in the documentation
 *   and/or other materials provided with the distribution.
 * o Neither the name of the ferite software nor the names of its contributors may
 *   be used to endorse or promote products derived from this software without
 *   specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include "ferite.h"
#include <math.h>

/**
 * !group Unified Array
 * !description This group is to ferite array's what the String group is to ferite strings.
 */

/**
 * !function ferite_uarray_create
 * !declaration FeriteUnifiedArray *ferite_uarray_create()
 * !brief Create a new empty array
 * !return A newly allocated FeriteUnifiedArray structure
 */
FeriteUnifiedArray *ferite_uarray_create()
{
    FeriteUnifiedArray *out = NULL;

    FE_ENTER_FUNCTION;
    out = fmalloc(sizeof(FeriteUnifiedArray));
    out->size = 0;
    out->actual_size = FE_ARRAY_DEFAULT_SIZE;
    out->hash = ferite_create_hash( NULL, FE_ARRAY_DEFAULT_SIZE );
    out->array = fmalloc( sizeof(FeriteVariable*) * out->actual_size );
    out->iteration = -1;
    out->iterator = NULL;
    out->iterator_type = 0;
    FE_LEAVE_FUNCTION( out );
}

/**
 * !function ferite_uarray_destroy
 * !declaration void ferite_uarray_destroy( FeriteScript *script, FeriteUnifiedArray *array )
 * !brief Destroy the array and free up any memory
 * !param FeriteScript *script The script
 * !param FeriteUnifiedArray *array The array to delete
 * !description
 */
void ferite_uarray_destroy( FeriteScript *script, FeriteUnifiedArray *array )
{
    int i;

    FE_ENTER_FUNCTION;
    FE_ASSERT( array != NULL );
    FUD(("ARRAY: Deleting array, size %d, actual size %d----------------------------\n", array->size, array->actual_size));
    ferite_delete_hash( script, array->hash, NULL );
    for( i = 0; i < array->size; i++ )
    {
        if( array->array[i] != NULL )
        {
            FUD(("ARRYA: Deleting array item %d\n", i ));
            ferite_variable_destroy( script, array->array[i] );
        }
    }
    if( array->iterator != NULL )
      ffree( array->iterator );
    ffree( array->array );
    ffree( array );
    FUD(("ARRAY: Done Deleteing array\n" ));
    FE_LEAVE_FUNCTION( NOWT );
}

/**
 * !function ferite_uarray_add
 * !declaration void ferite_uarray_add( FeriteScript *script, FeriteUnifiedArray *array, FeriteVariable *var, char *id, int pos )
 * !brief Add a variable to the array
 * !param FeriteScript *script The script
 * !param FeriteUnifiedArray *array The array to add to
 * !param FeriteVariable *var The variable to add
 * !param char *id The hash id
 * !param int pos The position in the array where the variable should be put
 * !description The hash id can be NULL - in which case the variable wont be hashed. If it isn't NULL
 *              the variable gets added to the array and also hashed in aswell. The variable can either
 *              be placed at the beginning or at the end of the array. Use either FE_ARRAY_ADD_AT_END
 *              or FE_ARRAY_ADD_AT_START to place the variable at the end or start respectively.
 */
void ferite_uarray_add( FeriteScript *script, FeriteUnifiedArray *array, FeriteVariable *var, char *id, int pos )
{
    long i = 0;

    FE_ENTER_FUNCTION;
    FE_ASSERT( array != NULL );
    if(var->type == F_VAR_OBJ)
    {
        if(VAO(var) != NULL)
        {
            VAO(var)->refcount++;
        }
    }
    if( pos > 0 ) /* we either add from the beginning or the end */
      pos = FE_ARRAY_ADD_AT_END;

    if( id != NULL ) /* add it to the hash */
    {
        ferite_set_variable_name( script, var, id );
        /* We asume that the array is a hash, check against a magic number
         * to see if we need to grow the number of buckets */
        if( array->size > array->hash->size * 20 )
          array->hash = ferite_hash_grow( script, array->hash );
        ferite_hash_add( script, array->hash, var->name, var );
    }
    else
    {
        ferite_set_static_variable_name( script, var, "" );
    }

    if( FE_VAR_IS_DISPOSABLE( var ) )
    {
        UNMARK_VARIABLE_AS_DISPOSABLE( var );
    }

    if( pos == FE_ARRAY_ADD_AT_END )
    {
        if( array->size == array->actual_size ) /* we need to bump up the size of the array */
        {
            FUD(( "  resizing array from %d to %d\n", array->actual_size, array->actual_size * 2 ));
            array->actual_size *= 2;
            array->array = frealloc( array->array, sizeof(FeriteVariable*) * array->actual_size );
        }
        array->array[array->size] = var;
        FUD(( " to end slot %d in array %p\n", array->size, array->array ));
        var->index = array->size;
        array->size++;

        FE_LEAVE_FUNCTION( NOWT );
    }
    if( pos == FE_ARRAY_ADD_AT_START )
    {
        if( array->size == array->actual_size ) /* we need to bump up the size of the array */
        {
            array->actual_size += FE_ARRAY_DEFAULT_SIZE;
            array->array = frealloc( array->array, sizeof(FeriteVariable*) * array->actual_size );
        }
        /* we need to do a memory move here */
        FUD(( " shifting array contents\n" ));
        memmove( array->array + 1, array->array, sizeof(FeriteVariable*) * array->size );
        /* and insert the stuff here */
        array->array[0] = var;
        FUD(( " to slot %d in array %p\n", 0, array->array ));
        array->size++;

        /* we re-index the variables here. this makes it very very expesive. fun. */
        for( i = 0; i < array->size; i++ )
          array->array[i]->index = i;

        FE_LEAVE_FUNCTION( NOWT );
    }
    ferite_error( script, 0, "Invalid add position %d\n", pos );
    FE_LEAVE_FUNCTION( NOWT );
}

/**
 * !function ferite_uarray_get_index
 * !declaration FeriteVariable *ferite_uarray_get_index( FeriteScript *script, FeriteUnifiedArray *array, int index )
 * !brief Get a variable from the array based upon an index
 * !param FeriteScript *script The script
 * !param FeriteUnifiedArray *array The array to get the variable from
 * !param int index The index to obtained
 * !return The variable if it exists, or NULL otherwise
 */
FeriteVariable *ferite_uarray_get_index( FeriteScript *script, FeriteUnifiedArray *array, int index )
{
    FE_ENTER_FUNCTION;
    FUD(( "trying to get index %d\n", index ));
    if( array->size == 0 )
    {
        ferite_error( script, 0,"Invalid index: array size is 0\n");
        FE_LEAVE_FUNCTION( NULL );
    }

    if(index < 0)
    {
        index = array->size + index;
    }

    if( index >= array->size )
    {
        ferite_error( script, 0,"Index %d is out of array's bounds [%d]\n", index, array->size );
        FE_LEAVE_FUNCTION( NULL );
    }
    FE_LEAVE_FUNCTION( array->array[index] );
}

/**
 * !function ferite_uarray_get_from_string
 * !declaration FeriteVariable *ferite_uarray_get_from_string( FeriteScript *script, FeriteUnifiedArray *array, char *id )
 * !brief Get the variable from the array based upon a string
 * !param FeriteScript *script The Script
 * !param FeriteUnifiedArray *array The array to extract the variable from
 * !param char *id The name of the variable to get out of the array
 * !return The variable if it exists or NULL otherwise
 */
FeriteVariable *ferite_uarray_get_from_string( FeriteScript *script, FeriteUnifiedArray *array, char *id )
{
    FeriteVariable *ptr = NULL;
    FE_ENTER_FUNCTION;
    ptr = ferite_hash_get( script, array->hash, id );
    FE_LEAVE_FUNCTION(ptr);
}

/**
 * !function ferite_uarray_delete_from_string
 * !declaration FeriteVariable *ferite_uarray_delete_from_string( FeriteScript *script, FeriteUnifiedArray *array, char *id )
 * !brief Delete a value from the array based upon a string
 * !param FeriteScript *script The Script
 * !param FeriteUnifiedArray *array The array to delete the variable from
 * !param char *id The name of the variable to delete out of the array
 * !return The variable that has been deleted
 */
FeriteVariable *ferite_uarray_delete_from_string( FeriteScript *script, FeriteUnifiedArray *array, char *id )
{
    int real_index = 0;

    FeriteVariable *ptr = NULL;
    FE_ENTER_FUNCTION;
    ptr = ferite_hash_get( script, array->hash, id );
    if( ptr == NULL )
    {
        ferite_error( script, 0, "Unknown index '%s'\n", id );
        FE_LEAVE_FUNCTION(NULL);
    }
    real_index = ptr->index;
    ferite_hash_delete( script, array->hash, id );
    ferite_uarray_del_index( script, array, real_index );
    FE_LEAVE_FUNCTION(ptr);
}

/**
 * !function ferite_uarray_get
 * !declaration FeriteVariable *ferite_uarray_get( FeriteScript *script, FeriteUnifiedArray *array, FeriteVariable *var )
 * !brief Get a variable from an array based upon the value of a FeriteVariable
 * !param FeriteScript *script The script
 * !param FeriteUnifiedArray *array The array to get the value from
 * !param FeriteVariable *var The variable to take the value from
 * !return The variable if it exists or NULL otherwise
 */
FeriteVariable *ferite_uarray_get( FeriteScript *script, FeriteUnifiedArray *array, FeriteVariable *var )
{
    FE_ENTER_FUNCTION;
    switch(var->type)
    {
      case F_VAR_LONG:
        FE_LEAVE_FUNCTION( ferite_uarray_get_index( script, array, VAI(var)) );
        break;
      case F_VAR_DOUBLE:
        FE_LEAVE_FUNCTION( ferite_uarray_get_index( script, array, floor(VAF(var)) ) );
        break;
      case F_VAR_STR:
        FE_LEAVE_FUNCTION( ferite_hash_get( script, array->hash, FE_STR2PTR(var)) );
        break;
    }
    FE_LEAVE_FUNCTION( NULL );
}

FeriteVariable *ferite_uarray_op(FeriteScript *script, FeriteUnifiedArray *array, FeriteVariable *index, void *rhs )
{
    FeriteVariable *ptr = NULL;

    FE_ENTER_FUNCTION;
    if(index->type == F_VAR_VOID && FE_VAR_IS_PLACEHOLDER( index ) )
    {
        FUD(( "adding variable to array\n" ));
        ptr = ferite_create_void_variable( script, "-1", FE_STATIC );
        ferite_uarray_add( script, array, ptr, NULL, FE_ARRAY_ADD_AT_END );
    }
    else
    {
        ptr = ferite_uarray_get( script, array, index );
        if(ptr == NULL)
        {
            ptr = ferite_create_void_variable( script, "uvar", FE_STATIC );
            ferite_uarray_add( script, array, ptr, (index->type == F_VAR_STR ? FE_STR2PTR(index) : NULL ), FE_ARRAY_ADD_AT_END );
        }
    }
    FE_LEAVE_FUNCTION(  ptr );
}

/**
 * !function ferite_uarray_del_var
 * !declaration void ferite_uarray_del_var( FeriteScript *script, FeriteUnifiedArray *array, FeriteVariable *index )
 * !brief Delete a value from an array based upon the value of a FeriteVariable
 * !param FeriteScript *script The script
 * !param FeriteUnifiedArray *array The array to delete the value from
 * !param FeriteVariable *var The variable to take the value from
 */
void ferite_uarray_del_var( FeriteScript *script, FeriteUnifiedArray *array, FeriteVariable *index )
{
    int  real_index = 0;
    FeriteVariable *ptr = NULL;

    FE_ENTER_FUNCTION;
    if( index->type == F_VAR_STR )
    {
        ptr = ferite_hash_get( script, array->hash, FE_STR2PTR(index) );
        if( ptr == NULL )
        {
            ferite_error( script, 0, "Unknown index '%s'\n", FE_STR2PTR(index) );
            FE_LEAVE_FUNCTION(NOWT);
        }
        real_index = ptr->index;
    }
    else if( index->type == F_VAR_LONG )
    {
        real_index = VAI(index);
    }
    else if( index->type == F_VAR_DOUBLE )
    {
        real_index = floor( VAF(index) );
    }
    else
    {
        ferite_error( script, 0, "Invalid index type '%s' on array\n", ferite_variable_id_to_str( script, index->type ) );
        FE_LEAVE_FUNCTION( NOWT );
    }
    ferite_uarray_del_index( script, array, real_index );
    FE_LEAVE_FUNCTION( NOWT );
}

/**
 * !function ferite_uarray_del_index
 * !declaration void ferite_uarray_del_index( FeriteScript *script, FeriteUnifiedArray *array, int index )
 * !brief Delete a value from an array based upon an index
 * !param FeriteScript *script The script
 * !param FeriteUnifiedArray *array The array to delete from
 * !param int index The index to delete
 */
void ferite_uarray_del_index( FeriteScript *script, FeriteUnifiedArray *array, int index )
{
    FeriteVariable *var = NULL;
    long i = 0;

    FE_ENTER_FUNCTION;

    if( index >= array->size || index < 0 )
    {
        ferite_error( script, 0, "Index out of bounds %d, can't delete item\n", index );
        FE_LEAVE_FUNCTION( NOWT );
    }

   /* delete the entry in the array */
    var = array->array[index];
    if( ferite_hash_get( script, array->hash, var->name ) != NULL )
      ferite_hash_delete( script, array->hash, var->name );

    ferite_variable_destroy( script, var );

   /* we shift the items left one */
    memmove( array->array + index,
             array->array + index + 1,
             (array->size - index) * sizeof(FeriteVariable*) );

    array->size--;

   /* we re-index the variables here. this makes it very very expesive. fun. */
    for( i = index; i < array->size; i++ )
      array->array[i]->index = i;

    FE_LEAVE_FUNCTION( NOWT );
}

/**
 * !function ferite_uarray_dup
 * !declaration FeriteUnifiedArray *ferite_uarray_dup( FeriteScript *script, FeriteUnifiedArray *array, void *(*ddup)(FeriteScript*,FeriteVariable *data,void*dup) )
 * !brief Duplicate an array
 * !param FeriteScript *script The script
 * !param FeriteUnifiedArray *array The array to duplicate
 * !param void *dup A function to call to duplicate the variables
 * !return A copy of the array
 */
FeriteUnifiedArray *ferite_uarray_dup( FeriteScript *script, FeriteUnifiedArray *array, void *(*ddup)(FeriteScript*,FeriteVariable *data,void*dup) )
{
    FeriteUnifiedArray *out;
    FeriteVariable *ptr = NULL;
    int i;

    FE_ENTER_FUNCTION;
    out = fmalloc(sizeof(FeriteUnifiedArray));
    out->hash = ferite_create_hash( script, array->hash->size );
    out->size = array->size;
    out->actual_size = array->actual_size;
    out->array = fmalloc( sizeof(FeriteVariable*) * out->actual_size );

   /* this will go through and copy the variables, and where needs be add them to the hash */
    for( i = 0; i < array->size; i++ )
    {
        ptr = (ddup)( script, array->array[i], NULL );
        out->array[i] = ptr;
        if( ptr->index > -1 && ptr->name[0] != '\0' )
          ferite_hash_add( script, out->hash, ptr->name, ptr );
    }
    out->iteration = -1;
    out->iterator = NULL;
    FE_LEAVE_FUNCTION( out );
}

/**
 * !function ferite_uarray_to_str
 * !declaration FeriteString *ferite_uarray_to_str( FeriteScript *script, FeriteUnifiedArray *array)
 * !brief Create a FeriteString based upon the contents of an array
 * !param FeriteScript *script The script
 * !param FeriteUnifiedArray *array The array to covert
 * !return The string version of the array
 * !description This is useful for converting an array to a string to write to disk, or print out.
 *              The function will recusre and handle inbuild arrays correctly.
 */
FeriteString *ferite_uarray_to_str( FeriteScript *script, FeriteUnifiedArray *array )
{
    FeriteVariable *var;
    int i;
    FeriteBuffer *buf;
    FeriteString *str,*s;

    FE_ENTER_FUNCTION;

    buf = ferite_buffer_new(FE_DEFAULT_BUFFER_SIZE);

    ferite_buffer_add_char(buf, '[');

    for(i = 0; i < array->size; i++)
    {
        var = array->array[i];
        s = ferite_variable_to_str( script, var, 1);
        if(strcmp("",var->name))
        {
            ferite_buffer_printf(buf," '%s': %.*s",var->name,s->length,s->data);
        }
        else
        {
            ferite_buffer_add_char(buf, ' ');
            ferite_buffer_add(buf, s->data, s->length);
        }
        ferite_str_destroy( s );
        if(i < array->size - 1)
        {
            ferite_buffer_add_char(buf, ',');
        }
    }
    ferite_buffer_add_char(buf, ' ');
    ferite_buffer_add_char(buf, ']');
    str = ferite_buffer_to_str( buf );
    ferite_buffer_delete(buf);
    FE_LEAVE_FUNCTION( str );
}

/**
 * !function ferite_uarray_push
 * !declaration void ferite_uarray_push( FeriteScript *script, FeriteUnifiedArray *array, FeriteVariable *var )
 * !brief Push a value onto the end array like a stack
 * !param FeriteScript *script The script
 * !param FeriteUnifiedArray *array The array to push the value onto
 * !param FeriteVariable *var The variable to push onto the array
 */
void ferite_uarray_push( FeriteScript *script, FeriteUnifiedArray *array, FeriteVariable *var )
{
    FeriteVariable *ptr = NULL;

    FE_ENTER_FUNCTION;
    ptr = ferite_duplicate_variable( script, var, NULL );
    ferite_uarray_add( script, array, ptr, NULL, FE_ARRAY_ADD_AT_END );
    FE_LEAVE_FUNCTION( NOWT );
}

/**
 * !function ferite_uarray_unshift
 * !declaration void ferite_uarray_unshift( FeriteScript *script, FeriteUnifiedArray *array, FeriteVariable *var )
 * !brief Shift a value onto the front of the array
 * !param FeriteScript *script The script
 * !param FeriteUnifiedArray *array The array to shift the value onto
 * !param FeriteVariable *var The variable to shift onto the array
 */
void ferite_uarray_unshift( FeriteScript *script, FeriteUnifiedArray *array, FeriteVariable *var )
{
    FeriteVariable *v;

    FE_ENTER_FUNCTION;
    v = ferite_duplicate_variable( script, var, NULL );
    ferite_uarray_add( script, array, v, NULL, FE_ARRAY_ADD_AT_START );
    FE_LEAVE_FUNCTION( NOWT );
}

/**
 * !function ferite_uarray_pop
 * !declaration FeriteVariable *ferite_uarray_pop( FeriteScript *script, FeriteUnifiedArray *array )
 * !brief Pop a value off the end of the array and return it
 * !param FeriteScript *script The script
 * !param FeriteUnifiedArray *array The array to pop a value off the end
 */
FeriteVariable *ferite_uarray_pop( FeriteScript *script, FeriteUnifiedArray *array )
{
    FeriteVariable *out = NULL;

    FE_ENTER_FUNCTION;
    if( array->size > 0 )
    {
        out = ferite_duplicate_variable( script, ferite_uarray_get_index( script, array, (array->size) - 1 ), NULL );
        ferite_uarray_del_index( script, array, (array->size)-1);
    }
    else
    {
        ferite_warning( script, "Trying to pop element off an empty array!\n" );
        out = ferite_create_void_variable( script, "no_value", FE_STATIC );
    }
    MARK_VARIABLE_AS_DISPOSABLE( out );
    FE_LEAVE_FUNCTION( out );
}

/**
 * !function ferite_uarray_shift
 * !declaration FeriteVariable *ferite_uarray_shift( FeriteScript *script, FeriteUnifiedArray *array )
 * !brief Shift a value off the front of the array
 * !param FeriteScript *script The script
 * !param FeriteUnifiedArray *array The array to shift a value off the front
 */
FeriteVariable *ferite_uarray_shift( FeriteScript *script, FeriteUnifiedArray *array )
{
    FeriteVariable *out;

    FE_ENTER_FUNCTION;
    if( array->size > 0 )
    {
        out = ferite_duplicate_variable( script, ferite_uarray_get_index( script, array, 0 ), NULL );
        ferite_uarray_del_index( script, array, 0 );
    }
    else
    {
        ferite_warning( script, "Trying to shift element off an empty array!\n" );
        out = ferite_create_void_variable( script, "no_value", FE_STATIC );
    }
    MARK_VARIABLE_AS_DISPOSABLE( out );
    FE_LEAVE_FUNCTION( out );
}

/**
 * !function ferite_uarray_cmp
 * !declaration int ferite_uarray_cmp( FeriteScript *script, FeriteUnifiedArray *left, FeriteUnifiedArray *right )
 * !brief Compare two scripts
 * !param FeriteScript *script The script
 * !param FeriteUnifiedArray *left The first array
 * !param FeriteUnifiedArray *right The second array
 */
int ferite_uarray_cmp( FeriteScript *script, FeriteUnifiedArray *left, FeriteUnifiedArray *right )
{
    int i = 0;

    FE_ENTER_FUNCTION;
    if( left->size != right->size )
    {
        FE_LEAVE_FUNCTION(0);
    }

   /* go through each element in the array */
    for( i = 0; i < left->size; i++ )
    {
    /* check the type of the variables */
        if( left->array[i]->type != right->array[i]->type )
        {
            FE_LEAVE_FUNCTION(0);
        }

#define FE_VAR_TEST( test ) \
   if( !(test) ){           \
    FE_LEAVE_FUNCTION(0); \
   }
    /* check their values */
        switch( left->array[i]->type )
        {
          case F_VAR_LONG:
            FE_VAR_TEST( VAI(left->array[i]) == VAI(right->array[i]) );
            break;
          case F_VAR_DOUBLE:
            FE_VAR_TEST( VAF(left->array[i]) == VAF(right->array[i]) );
            break;
          case F_VAR_STR:
            FE_VAR_TEST( ferite_str_cmp( VAS(left->array[i]), VAS(right->array[i]) ) == 1 );
            break;
          case F_VAR_OBJ:
            FE_VAR_TEST( VAO(left->array[i]) == VAO(right->array[i]) );
            break;
          case F_VAR_UARRAY:
            FE_VAR_TEST( ferite_uarray_cmp( script, VAUA(left->array[i]), VAUA(right->array[i]) ) == 1 );
          default:
            ferite_error( script, 0, "EEEK: unknown type %s in array comparison!\n", ferite_variable_id_to_str( script, left->array[i]->type ) );
            FE_LEAVE_FUNCTION(0);
        }
#undef FE_VAR_TEST
    }
   /* if we have got here they are the same :) */
    FE_LEAVE_FUNCTION( 1 );
}

/**
 * !end
 */
