/*
 * Copyright (C) 2001-2004 Chris Ross
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * o Redistributions of source code must retain the above copyright notice, this
 *   list of conditions and the following disclaimer.
 * o Redistributions in binary form must reproduce the above copyright notice,
 *   this list of conditions and the following disclaimer in the documentation
 *   and/or other materials provided with the distribution.
 * o Neither the name of the ferite software nor the names of its contributors may
 *   be used to endorse or promote products derived from this software without
 *   specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include "ferite.h"
#include "thread_header.h"

FeriteScript *ferite_thread_create_script( FeriteScript *script )
{
    FeriteScript *new_script = NULL;

    FE_ENTER_FUNCTION;

    new_script = fmalloc( sizeof( FeriteScript ) );
    memcpy( new_script, script, sizeof( FeriteScript ) );

    /* Create new cache for this thread */
    ferite_init_cache( new_script );

    /* Create new GC for this thread */
    new_script->gc = NULL;
    ferite_init_gc( new_script );

    /* If we are here then we have not got any errors or warnings reset! :) */
    new_script->error = NULL;
    new_script->warning = NULL;
    new_script->stack_level = 0;

    /* Hook up the parent */
    new_script->parent = script;

    FE_LEAVE_FUNCTION(new_script);
}

void ferite_thread_destroy_script( FeriteScript *parent, FeriteThread *ctx )
{
    FE_ENTER_FUNCTION;

    /* If you want exceptions to be passed on */
    if( ctx->script->error != NULL )
      ferite_reset_errors( ctx->script );

    if( ctx->script->warning != NULL )
      ferite_reset_warnings( ctx->script );

    ferite_free_cache( ctx->script );
    ferite_merge_gc( parent, ctx->script->gc );
    ffree( ctx->script );

    FE_LEAVE_FUNCTION(NOWT);
}

void *ferite_thread_execute( void *ptr )
{
    FeriteThread *thread = (FeriteThread*)ptr;
    FeriteObject *obj = thread->obj;
    FeriteScript *script = thread->script;
    FeriteFunction *function = NULL;
    FeriteVariable **params = NULL;

    FE_ENTER_FUNCTION;

    function = ferite_object_get_function_for_params( script, obj, "run", NULL );
    if( function == NULL )
    {
        ferite_error( script, 0, "No run method found in Thread class!\n" );
        FE_LEAVE_FUNCTION( NULL );
    }

    ferite_thread_group_attach( script, script->thread_group, thread );
    thread->running = FE_TRUE;
    obj->refcount++;

    ferite_variable_destroy( script, ferite_call_function( script, obj, NULL, function, NULL ) );

    obj->refcount--;
    thread->running = FE_FALSE;
    ferite_thread_group_dettach( script, script->thread_group, thread );

    /* If you want exceptions to be passed on */
    if( script->error != NULL )
    {
        if( thread->pass_exceptions )
        {
            char *value = ferite_get_error_string( script );
            ferite_error( script->parent, 0, "Exception in thread: (errors will follow this message)\n%s", value );
            ffree( value );
        }
        ferite_reset_errors( script );
    }

    FE_LEAVE_FUNCTION( NULL );
}
