/* $Id: sprite.c,v 1.52 2004/04/14 15:05:43 bsenders Exp $ */

#include <stdio.h>
#include <stdlib.h>
#include <assert.h>
#include <string.h>
#include <math.h>

#include <SDL.h>

#include "global.h"
#include "sprite.h"
#include "vector.h"

#define GDEBUG(...) DEBUG(DGRAPHICS,"Graphics",__VA_ARGS__)

#ifndef SVG_NORMALIZE
#define SVG_NORMALIZE 1000
#endif

#ifndef SCALE_X
#define SCALE_X 1000
#endif

#ifndef SCALE_Y
#define SCALE_Y 1000
#endif

char *state_names[] = { "dead_player",
                        "unconscious_player",
                        "hit_player",
                        "lthrow_player",
                        "rthrow_player",
                        "ljump_player",
                        "rjump_player",
                        "lwalk_player",
                        "rwalk_player",
                        "lstand_player",
                        "rstand_player",
                        "normal_food",
                        "upgraded_food",
                        "normal_scenery"
};

/*** Rendering code ***/
typedef struct {
  Vector pos;
  float t;
  int kind;
} Graphics_pointinfo;

typedef struct {
  /* 2-dimarray of pointinfo thingies */
  Graphics_pointinfo **points;
  int nrpoints;
  int pparray;
  int arraysize;
} Graphics_scanline;

typedef struct {
  SDL_Surface *surface;
  Uint32 linecol;
  Vector *size;
  /* array of scanlines */
  Graphics_scanline *lines;
} Graphics_render_info;

static Graphics_render_info *
new_graphics_render_info(SDL_Surface *surface,Vector *size,
                         Graphics_sprite_rgba *linecol) {
  Graphics_render_info *result;
  int arraysize;
  result = malloc(sizeof(Graphics_render_info));
  result->surface = surface;
  result->size = size;
  result->linecol = SDL_MapRGBA(surface->format,
                                linecol->red,
                                linecol->green,
                                linecol->blue,
                                linecol->alpha);
  /* Array for every scanline in the picture and one extra for all the points 
   * outside the y of the picture */
  arraysize = (size->y  + 1 ) * sizeof(Graphics_scanline);
  result->lines = malloc(arraysize);
  memset(result->lines,0,arraysize);
  return result;
}

static void
del_graphics_render_info(Graphics_render_info *info) {
  int x,y;
  for (x = 0; x <= info->size->y; x++) {
    for (y = 0 ; y < info->lines[x].arraysize; y++) {
      free(info->lines[x].points[y]);
    }
    free(info->lines[x].points);
  }
  free(info->lines);
  free(info);
}

inline Graphics_pointinfo*
graphics_scanline_getpoint(Graphics_scanline *line,int point) {
  return &(line->points[point/line->pparray][point%line->pparray]);
}

inline void
graphics_scanline_swappoints(Graphics_pointinfo *l,Graphics_pointinfo *r) {
  Graphics_pointinfo tmp;
  tmp = *r;
  *r = *l;
  *l = tmp;
}

int
graphics_scanline_do_partition(Graphics_scanline *line,int  start, int end) {
  Graphics_pointinfo *pivot;
  Graphics_pointinfo *cur;
  int pivotpos = start;
  int j;

  pivot = graphics_scanline_getpoint(line,end);
  for (j = start ; j < end ; j++) {
    cur = graphics_scanline_getpoint(line,j);
    if (cur->pos.x <= pivot->pos.x) {
      if (pivotpos != j) {
        graphics_scanline_swappoints(cur,
                                     graphics_scanline_getpoint(line,pivotpos));
      }
      pivotpos++;
    }
  }
  graphics_scanline_swappoints(pivot,graphics_scanline_getpoint(line,pivotpos));
  return pivotpos;
}

void 
graphics_scanline_do_qsort(Graphics_scanline *line,int start,int end) {
  int pos;
  if (start < end) {
    pos = graphics_scanline_do_partition(line,start,end);
    graphics_scanline_do_qsort(line,start,pos - 1);
    graphics_scanline_do_qsort(line,pos + 1,end);
  }
}

void
graphics_scanline_qsort(Graphics_scanline *line) {
  graphics_scanline_do_qsort(line,0,line->nrpoints - 1);
}

Graphics_pointinfo*
graphics_render_info_addpoint(Graphics_render_info *info,
                              Vector *pos,float t,
                              int kind) {
  int nr;
  Graphics_pointinfo *result;
  Graphics_scanline *scanline;
  if (pos->y < 0 || pos->y >= info->size->y) {
    nr = info->size->y;
  } else {
    nr = pos->y;
  }
  scanline = &(info->lines[nr]);
  if (scanline->nrpoints == scanline->arraysize * scanline->pparray) {
    if (scanline->pparray == 0) {
      scanline->pparray = 20;
    }
    scanline->points = realloc(scanline->points,
                               (scanline->arraysize + 1) * 
                               sizeof(Graphics_pointinfo *));
    scanline->points[scanline->arraysize] = 
      malloc(sizeof(Graphics_pointinfo) * scanline->pparray);
    scanline->arraysize++;
  }
  result = graphics_scanline_getpoint(scanline,scanline->nrpoints);
  scanline->nrpoints++;
  result->pos = *pos;
  result->t = t;
  result->kind = kind;
  return result;
}
              

void
putpixel(SDL_Surface * g, Vector * pos, Uint32 pixel) {
  int bpp = g->format->BytesPerPixel;
  Uint8 *p;
  int x,y;

  x = pos->x;
  y = pos->y;
  if (x >= 0 && y >= 0 && x < g->w && y < g->h) {
    /* Here p is the address to the pixel we want to set */
    p = (Uint8 *) g->pixels + y  * g->pitch + x * bpp;
  } else {
    return;
  }

  switch (bpp) {
    case 1:
      *p = pixel;
      break;
    case 2:
      *(Uint16 *) p = pixel;
      break;
    case 3:
      if (SDL_BYTEORDER == SDL_BIG_ENDIAN) {
        p[0] = (pixel >> 16) & 0xff;
        p[1] = (pixel >> 8) & 0xff;
        p[2] = pixel & 0xff;
      } else {
        p[0] = pixel & 0xff;
        p[1] = (pixel >> 8) & 0xff;
        p[2] = (pixel >> 16) & 0xff;
      }
      break;
    case 4:
      *(Uint32 *) p = pixel;
      break;
  }
}

void 
put_line(SDL_Surface *surf,Uint32 color,int startx, int endx, int y) {
  /* Draws horizontal lines FAST! */
  int bpp = surf->format->BytesPerPixel;
  Uint32 *p,*q;
  Uint32 pixel = 0;
  Uint8 *p8;
  Uint16 *p16;

  startx = MAX(startx,0);
  endx = MIN(endx,surf->w);
  if (endx < startx) return;

  p = surf->pixels + startx * bpp + y * surf->pitch;
  q = surf->pixels + endx * bpp + y * surf->pitch;


  switch (bpp) {
    case 1:
      pixel = (color << 24) | (color << 16) | (color << 8) | color;
      p8 = (Uint8 *)p;
      for (; (unsigned long)p8 & 0x3 && p8 < (Uint8 *)q; p8++ ) {
         *p8 = color;
      }
      for (p = (Uint32 *)p8; p < q-1 ; p++) {
        *p = pixel;
      }
      for (p8 = (Uint8 *)p; p8 < (Uint8 *)q; p8++ ) {
        *p8 = color;
      }
      break;
    case 2:
      pixel = (color << 16) | color;
      for (p16 = (Uint16 *)p ; 
            (unsigned long)p16 & 0x3 && p16 < (Uint16 *) q; p16++ ) {
         *p16 = color;
      }
      for (p = (Uint32 *)p16; p < q-1 ; p++) {
        *p = pixel;
      }
      for (p16 = (Uint16 *)p ; p16 < (Uint16 *)q; p16++ ) {
        *(Uint16 *)p = color;
      }
      break;
    case 3:
      for (; p < q;p++ ) {
        *p = SDL_SwapBE32(color); 
      }
      break;
    case 4:
      for (; p < q ; p++) {
        *p = color;
      }
  }
}

static void
fill_renderinfo(Graphics_render_info *info,Graphics_sprite_rgba *fillcolor) {
  Uint32 color = SDL_MapRGBA(info->surface->format,
                               fillcolor->red,
                               fillcolor->green,
                               fillcolor->blue,
                               fillcolor->alpha);
  int x,y,odd,direction;
  Graphics_scanline *scanline;
  Graphics_pointinfo *point;

  for (y = 0 ; y < info->size->y; y++) {
    odd = FALSE;
    direction = 0;
    scanline = &(info->lines[y]);
    graphics_scanline_qsort(scanline);
    for (x = 0 ; x < scanline->nrpoints - 1; x++) {
      point = graphics_scanline_getpoint(scanline,x);
      if (point->kind == 0) {
        odd = !odd;
      } else if (point->kind == 5 || point->kind == 7) {
        if (direction == 0) {
          direction = point->kind;
        } else if (direction != point->kind) {
          direction = 0;
          odd = !odd;
        } else {
          direction = 0;
        }
      }
      if (odd) {
        put_line(info->surface,color,point->pos.x + 1,
                 graphics_scanline_getpoint(scanline,x+1)->pos.x,
                 y);
      }
    }
  }
}

inline Graphics_pointinfo *
calculate_bezier_point(Graphics_render_info *info,float t,
                       Vector *p0, Vector *p1, Vector *p2, Vector *p3) {
  Vector pos;
  float m0,m1,m2,m3;
  float t3,t2;

  t2 = t * t;
  t3 = t * t2;

  m0 = -1 * t3 + 3 * t2 - 3 * t + 1;
  m1 =  3 * t3 - 6 * t2 + 3 * t;
  m2 = -3 * t3 + 3 * t2;
  m3 =      t3;

  pos.x = m0 * p0->x + m1 * p1->x + m2 * p2->x + m3 * p3->x;
  pos.y = m0 * p0->y + m1 * p1->y + m2 * p2->y + m3 * p3->y;
  return graphics_render_info_addpoint(info,&pos,t,0);
}

static void
render_bezier(Graphics_render_info *info,
              Graphics_pointinfo *lpoint, Graphics_pointinfo *rpoint ,
              Vector *p0, Vector *p1, Vector *p2, Vector *p3) {
  Graphics_pointinfo *mpoint;
  int xdif,ydif;
  float middle;

  xdif = (lpoint->pos.x - rpoint->pos.x);
  ydif = (lpoint->pos.y - rpoint->pos.y);
  if (xdif * xdif + ydif * ydif <= 2) {
    if (lpoint->pos.y > rpoint->pos.y) {
      lpoint->kind += 1;
      rpoint->kind -= 1;
    } else if (lpoint->pos.y < rpoint->pos.y) {
      lpoint->kind -= 1;
      rpoint->kind += 1;
    } else {
      lpoint->kind += 6;
      rpoint->kind += 6;
    }
    return;
  }

  middle = (rpoint->t - lpoint->t)/2.0;
  if (middle == 0) 
    return;
  middle += lpoint->t;
  mpoint = calculate_bezier_point(info,middle,p0,p1,p2,p3);
  putpixel(info->surface,&(mpoint->pos),info->linecol);

  render_bezier(info,lpoint,mpoint,p0,p1,p2,p3);
  render_bezier(info,mpoint,rpoint,p0,p1,p2,p3);
}

static void
render_bezier_element(Graphics_sprite_path_elem *elem,
                      Graphics_render_info *info) {
  Vector *p0,*p1,*p2,*p3;
  Graphics_pointinfo *lpoint,*rpoint;

  /* scale our needed vectors to picture size */
  p0 = vector_mul(elem->bvecs->p0,info->size);
  p0->x = p0->x /1000000 + (p0->x %1000000 < 500000 ? 0 : 1);
  p0->y = p0->y /1000000 + (p0->y %1000000 < 500000 ? 0 : 1);

  p1 = vector_mul(elem->bvecs->p1,info->size);
  p1->x = p1->x /1000000 + (p1->x %1000000 < 500000 ? 0 : 1);
  p1->y = p1->y /1000000 + (p1->y %1000000 < 500000 ? 0 : 1);

  p2 = vector_mul(elem->bvecs->p2,info->size);
  p2->x = p2->x /1000000 + (p2->x %1000000 < 500000 ? 0 : 1);
  p2->y = p2->y /1000000 + (p2->y %1000000 < 500000 ? 0 : 1);

  p3 = vector_mul(elem->bvecs->p3,info->size);
  p3->x = p3->x /1000000 + (p3->x %1000000 < 500000 ? 0 : 1);
  p3->y = p3->y /1000000 + (p3->y %1000000 < 500000 ? 0 : 1);

  lpoint = graphics_render_info_addpoint(info,p0,0,6);
  rpoint = graphics_render_info_addpoint(info,p3,1,6);

  putpixel(info->surface,&(lpoint->pos),info->linecol);
  putpixel(info->surface,&(rpoint->pos),info->linecol);
  render_bezier(info,lpoint,rpoint,p0,p1,p2,p3);
  del_vector(p0); del_vector(p1); del_vector(p2); del_vector(p3);
}

inline Graphics_pointinfo *
calculate_line_point(Graphics_render_info *info,float t,
                       Vector *p0, Vector *p1) {

  Vector pos;
  pos.x = p0->x + t * (p1->x - p0->x);
  pos.y = p0->y + t * (p1->y - p0->y);
  return graphics_render_info_addpoint(info,&pos,t,0);
}

static void
render_line(Graphics_render_info *info,
              Graphics_pointinfo *lpoint, Graphics_pointinfo *rpoint ,
              Vector *p0, Vector *p1) {

  int xdif,ydif;
  float middle;
  Graphics_pointinfo *mpoint;

  xdif = (lpoint->pos.x - rpoint->pos.x);
  ydif = (lpoint->pos.y - rpoint->pos.y);

  if (xdif * xdif + ydif * ydif <= 2) {
    if (lpoint->pos.y > rpoint->pos.y) {
      lpoint->kind += 1;
      rpoint->kind -= 1;
    } else if (lpoint->pos.y < rpoint->pos.y) {
      lpoint->kind -= 1;
      rpoint->kind += 1;
    } else {
      lpoint->kind += 6;
      rpoint->kind += 6;
    }
    return;
  }

  middle = (rpoint->t - lpoint->t)/2.0;
  if (middle == 0) 
    return;
  middle += lpoint->t;
  mpoint = calculate_line_point(info,middle,p0,p1);
  putpixel(info->surface,&(mpoint->pos),info->linecol);

  render_line(info,lpoint,mpoint,p0,p1);
  render_line(info,mpoint,rpoint,p0,p1);
}

static void
render_line_element(Graphics_sprite_path_elem *elem,
                      Graphics_render_info *info) {
    
  Vector *p0,*p1;
  Graphics_pointinfo *lpoint,*rpoint;

  p0 = vector_mul(elem->bvecs->p0,info->size);
  p0->x = p0->x /1000000 + (p0->x %1000000 < 500000 ? 0 : 1);
  p0->y = p0->y /1000000 + (p0->y %1000000 < 500000 ? 0 : 1);

  p1 = vector_mul(elem->bvecs->p3,info->size);
  p1->x = p1->x /1000000 + (p1->x %1000000 < 500000 ? 0 : 1);
  p1->y = p1->y /1000000 + (p1->y %1000000 < 500000 ? 0 : 1);

  lpoint = graphics_render_info_addpoint(info,p0,0,6);
  rpoint = graphics_render_info_addpoint(info,p1,1,6);
  putpixel(info->surface,&(lpoint->pos),info->linecol);
  putpixel(info->surface,&(rpoint->pos),info->linecol);
  render_line(info,lpoint,rpoint,p0,p1);
  del_vector(p0); del_vector(p1);
}

static void
render_rect_element(Graphics_sprite_path_elem *elem,
                      Graphics_render_info *info) {
  Vector *p0,*p1;
  Vector pos;
  int i,start,end;

  p0 = vector_mul(elem->bvecs->p0,info->size);
  p0->x = p0->x /1000000 + (p0->x %1000000 < 500000 ? 0 : 1);
  p0->y = p0->y /1000000 + (p0->y %1000000 < 500000 ? 0 : 1);

  p1 = vector_mul(elem->bvecs->p3,info->size);
  p1->x = p1->x /1000000 + (p1->x %1000000 < 500000 ? 0 : 1);
  p1->y = p1->y /1000000 + (p1->y %1000000 < 500000 ? 0 : 1);

  start = MIN(p0->y,p1->y);
  end   = MAX(p0->y,p1->y);

  pos.x = p0->x;
  for (i = start; i <= end; i++) {
    pos.y = i;
    graphics_render_info_addpoint(info,&pos,0,0);
    putpixel(info->surface,&pos,info->linecol);
  }
  pos.x = p1->x;
  for (i = start; i <= end; i++) {
    pos.y = i;
    graphics_render_info_addpoint(info,&pos,0,0);
    putpixel(info->surface,&pos,info->linecol);
  }

  start = MIN(p0->x,p1->x);
  end   = MAX(p0->x,p1->x);
  pos.y = p0->y;
  for (i = start; i <= end; i++) {
    pos.x = i;
    graphics_render_info_addpoint(info,&pos,0,12);
    putpixel(info->surface,&pos,info->linecol);
  }
  pos.y = p1->y;
  for (i = start; i <= end; i++) {
    pos.x = i;
    graphics_render_info_addpoint(info,&pos,0,12);
    putpixel(info->surface,&pos,info->linecol);
  }
}

static int
render_element(void *data, void *user_data) {
  Graphics_sprite_path_elem *elem = (Graphics_sprite_path_elem *)data;
  Graphics_render_info *info = (Graphics_render_info *)user_data;
  switch (elem->type) {
    case ET_LINE:
      render_line_element(elem,info);
      break;
    case ET_BEZIER:
      render_bezier_element(elem,info);
      break;
    case ET_RECT:
      render_rect_element(elem,info);
      break;
    default:
      WARN("Drawing of type %d not implemented",elem->type);
      break;
  }

  return TRUE;
}

static SDL_Surface *
render_sprite_pic(Graphics_sprite_pic *pic,SDL_Surface *target,Vector *size) {
  SDL_Surface *result;
  List_ptr *paths;
  Graphics_render_info *renderinfo;
  Graphics_sprite_path *path;

  result = SDL_CreateRGBSurface(SDL_SWSURFACE, size->x, size->y,
                                target->format->BitsPerPixel,
                                target->format->Rmask,
                                target->format->Gmask,
                                target->format->Bmask,
                                target->format->Amask);
  if (target->format->palette != NULL) {
    SDL_SetPalette(result,SDL_LOGPAL, target->format->palette->colors,
                   0, target->format->palette->ncolors);
  }
  SDL_SetColorKey(result, SDL_SRCCOLORKEY,SDL_MapRGB(result->format, 
                                                        0xce, 0xf0, 0xf0));
  SDL_FillRect(result, NULL, SDL_MapRGB(result->format, 0xce, 0xf0, 0xf0));
  paths = new_list_ptr(pic->paths);
  assert(list_ptr_first(paths));
  do {
    path = (Graphics_sprite_path *)list_ptr_get_data(paths);
    assert(path != NULL);

    renderinfo = new_graphics_render_info(result,size,path->linecolor);
    list_foreach(path->elems,render_element,renderinfo);
    fill_renderinfo(renderinfo,path->fillcolor);
    del_graphics_render_info(renderinfo);

  } while(list_ptr_next(paths));
  return result;
}

/**** end of rendering code ***/
int
find_state(void *data, void *user_data) {
  Graphics_sprite_state *gss = (Graphics_sprite_state *) data;
  Sprite_state state = (Sprite_state) * ((int *) user_data);

	GDEBUG("Trying to match %d with %d", gss->state, state);
  if (gss->state == state)
    return TRUE;
  else
    return FALSE;
}

Graphics_sprite_anim *
new_graphics_sprite_anim(Vector * size, int numframes, int fpp) {
  Graphics_sprite_anim *result;
  int x;

  result = malloc(sizeof(Graphics_sprite_anim));
  result->numframes = numframes;
  result->anim = malloc(numframes * sizeof(SDL_Surface *));
  result->fpp = fpp;
  for (x = 0; x < numframes; x++) {
    result->anim[x] = NULL;
  }
	GDEBUG("Creating and Sprite anim %p and animation %p", result, result->anim);
  return result;
}

void
del_graphics_sprite_anim(Graphics_sprite_anim * gsa) {
  int i;

	GDEBUG("Removing Sprite anim: %p", gsa);
  for (i = 0; i < gsa->numframes; i++) {
    if (gsa->anim[i] != NULL)
      SDL_FreeSurface(gsa->anim[i]);
  }
  free(gsa->anim);
  free(gsa);
}

SDL_Surface * 
sprite_anim_getsurface(Graphics_sprite_anim *gsa,
                       Sprite_state state, int *framenr) {
  assert(gsa->fpp != 0);
  if (gsa->fpp < 0) {
    *framenr = MIN(*framenr,(abs(gsa->fpp) * gsa->numframes) -1);
  } else {
    *framenr %= gsa->fpp * gsa->numframes;
  }
  return gsa->anim[*framenr/abs(gsa->fpp)];
}


Graphics_sprite_anim *
create_dummy_anim(SDL_Surface *g,Vector *size) {
  Graphics_sprite_anim *result;
  result = new_graphics_sprite_anim(size, 1, 1);
  result->anim[0] = SDL_CreateRGBSurface(SDL_SWSURFACE, size->x, size->y,
                                           g->format->BitsPerPixel,
                                           g->format->Rmask,
                                           g->format->Gmask,
                                           g->format->Bmask,
                                           g->format->Amask);
  if (g->format->palette != NULL) {
      SDL_SetPalette(result->anim[0],SDL_LOGPAL, g->format->palette->colors,
                                          0, g->format->palette->ncolors);
  }
  SDL_FillRect(result->anim[0], NULL,
                 SDL_MapRGB(result->anim[0]->format, 0xff, 0x00, 0x00));
  return result;
}

Graphics_sprite_anim *
draw_sprite_anim(SDL_Surface * g, Vector * size, Sprite_state state,
                 Graphics_sprite_source * gss) {
  Graphics_sprite_anim *result;
  Graphics_sprite_state *spritestate;
  int current_pic = 0;
  List_ptr *spritepics;

  if (gss == NULL) { return create_dummy_anim(g,size); }
  if ((spritestate = list_search(gss->states,find_state,&state)) == NULL
      || list_length(spritestate->pics) == 0) {
    return create_dummy_anim(g,size);
  }
  result = new_graphics_sprite_anim(size,list_length(spritestate->pics),
                                    spritestate->fpp);
  spritepics = new_list_ptr(spritestate->pics);
  assert(list_ptr_first(spritepics));

  /* render the pictures */
  do {
    result->anim[current_pic] = render_sprite_pic(list_ptr_get_data(spritepics),
                                                  g,size);
    current_pic++;
  } while(list_ptr_next(spritepics));
  
  return result;
} 
/******************************************************************************
 * Graphics SVG Series to Graphics Sprite Source
 *****************************************************************************/

Graphics_sprite_source *
new_sprite_source(void) {
  Graphics_sprite_source *gsp_source;

  gsp_source = malloc(sizeof(Graphics_sprite_source));
  gsp_source->states = new_list();
	GDEBUG("Creating Sprite source: %p and list: %p", gsp_source, gsp_source->states);

  return gsp_source;
}

int
del_sprite_source(Graphics_sprite_source * gsp_source) {
  Graphics_sprite_state *gsp_state;

	GDEBUG("Removing Sprite source %p and list %p", gsp_source, gsp_source->states);
	if (list_length(gsp_source->states) > 0) {
	  while ((gsp_state = (Graphics_sprite_state *) list_pop(gsp_source->states))) {
	    del_sprite_state(gsp_state);
	  }
	} 

  del_list(gsp_source->states);

  free(gsp_source);

  return 0;
}

void
sprite_source_add_state(Graphics_sprite_source * gsp_source,
                        Graphics_sprite_state * gsp_state) {
	GDEBUG("Adding state %p to list %p", gsp_state, gsp_source->states);
  list_append(gsp_source->states, gsp_state);
}

/** Sprite state **/

Graphics_sprite_state *
new_sprite_state(void) {
  Graphics_sprite_state *gsp_state;

  gsp_state = malloc(sizeof(Graphics_sprite_state));
  gsp_state->pics = new_list();
	GDEBUG("Creating Sprite state: %p and list: %p", gsp_state, gsp_state->pics);

  return gsp_state;
}

int
del_sprite_state(Graphics_sprite_state * gsp_state) {
  Graphics_sprite_pic *gsp_pic;

	GDEBUG("Removing Sprite state %p and list %p", gsp_state, gsp_state->pics);
  while ((gsp_pic = (Graphics_sprite_pic *) list_pop(gsp_state->pics))) {
    del_sprite_pic(gsp_pic);
  }

  del_list(gsp_state->pics);

  free(gsp_state);

  return 0;
}

void
sprite_state_set_state(Graphics_sprite_state * gsp_state, char *state) {
  int x;

  assert(SS_NRSTATES == (sizeof(state_names)/sizeof(char *)));
  gsp_state->state = -1;
  for (x = 0 ; x < SS_NRSTATES; x++) {
    if (!strcmp(state,state_names[x])) {
        gsp_state->state = x;
    }
  }
}

void
sprite_state_add_pic(Graphics_sprite_state * gsp_state,
                     Graphics_sprite_pic * gsp_pic) {
	GDEBUG("Appending picture %p to list %p of Sprite state %p", gsp_pic, gsp_state->pics, gsp_state);
  list_append(gsp_state->pics, gsp_pic);
}

/** Sprite picture **/

Graphics_sprite_pic *
new_sprite_pic(void) {
  Graphics_sprite_pic *gsp_pic;

  gsp_pic = malloc(sizeof(Graphics_sprite_pic));
  gsp_pic->paths = new_list();
	GDEBUG("Creating Sprite pic: %p and list: %p", gsp_pic, gsp_pic->paths);

  return gsp_pic;
}

int
del_sprite_pic(Graphics_sprite_pic * gsp_pic) {
  Graphics_sprite_path *gsp_path;

	GDEBUG("Removing Sprite pic %p and list %p", gsp_pic, gsp_pic->paths);
  while ((gsp_path = (Graphics_sprite_path *) list_pop(gsp_pic->paths))) {
    del_sprite_path(gsp_path);
  }

  del_list(gsp_pic->paths);

  free(gsp_pic);

  return 0;
}

void sprite_state_set_fpp(Graphics_sprite_state *gsp_state, int fpp) {
	GDEBUG("Setting fpp %d on Sprite state %p", fpp, gsp_state);
  gsp_state->fpp = fpp;
}

void
sprite_pic_add_path(Graphics_sprite_pic * gsp_pic,
                    Graphics_sprite_path * gsp_path) {
	GDEBUG("Appending path %p to list %p of Sprite pic %p", gsp_path, gsp_pic->paths, gsp_pic);
  list_append(gsp_pic->paths, gsp_path);
}

/** Sprite path **/

Graphics_sprite_path *
new_sprite_path(void) {
  Graphics_sprite_path *gsp_path;

  gsp_path = malloc(sizeof(Graphics_sprite_path));
  gsp_path->elems = new_list();
	GDEBUG("Creating Sprite path: %p and list: %p", gsp_path, gsp_path->elems);

  gsp_path->linecolor = NULL;
  gsp_path->fillcolor = NULL;
  return gsp_path;
}

int
del_sprite_path(Graphics_sprite_path * gsp_path) {
  Graphics_sprite_path_elem *gsp_pelem;

	GDEBUG("Removing Sprite anim: %p and list %p", gsp_path, gsp_path->elems);
  while ((gsp_pelem =
          (Graphics_sprite_path_elem *) list_pop(gsp_path->elems))) {
    del_sprite_path_elem(gsp_pelem);
  }
  free(gsp_path->linecolor);
  free(gsp_path->fillcolor);

  del_list(gsp_path->elems);

  free(gsp_path);

  return 0;
}

void
sprite_path_add_elem(Graphics_sprite_path * gsp_path,
                     Graphics_sprite_path_elem * gsp_path_elem) {
	GDEBUG("Appending path element %p to list %p of Sprite path %p", gsp_path_elem, gsp_path->elems, gsp_path);
  list_append(gsp_path->elems, gsp_path_elem);
}

/** Sprite path elem **/

Graphics_sprite_path_elem *
new_sprite_path_elem(void) {
  Graphics_sprite_path_elem *gsp_pelem;

  gsp_pelem = malloc(sizeof(Graphics_sprite_path_elem));
	gsp_pelem->type = -1;
	gsp_pelem->bvecs = NULL; 
	GDEBUG("Creating Sprite path element %p", gsp_pelem);
  return gsp_pelem;
}

int
del_sprite_path_elem(Graphics_sprite_path_elem * gsp_pelem) {

	GDEBUG("Removing Sprite path element %p and vectors %p", gsp_pelem, gsp_pelem->bvecs);
  del_bezier_vectors(gsp_pelem->bvecs);
  free(gsp_pelem);

  return 0;
}

void
sprite_path_elem_set_type(Graphics_sprite_path_elem * gsp_pelem,
                          Graphics_sprite_path_elem_type type) {
  gsp_pelem->type = type;
	GDEBUG("Setting type %d on Sprite path element %p", type, gsp_pelem);
}

void
sprite_path_elem_set_vector(Graphics_sprite_path_elem * gsp_pelem,
                            Bezier_Vectors * v) {
  gsp_pelem->bvecs = new_bezier_vectors(v->p0, v->p1, v->p2, v->p3); 
	GDEBUG("Setting bvecs %p on Sprite path element %p", gsp_pelem->bvecs, gsp_pelem);
}

void sprite_path_set_linecolor(Graphics_sprite_path *gsp_path, Graphics_sprite_rgba *linecolor) {
  gsp_path->linecolor = linecolor;
	GDEBUG("Setting linecolor %p on Sprite path element %p", gsp_path->linecolor, gsp_path);
}

void sprite_path_set_fillcolor(Graphics_sprite_path *gsp_path, Graphics_sprite_rgba *fillcolor) {
  gsp_path->fillcolor = fillcolor;
	GDEBUG("Setting fillcolor %p on Sprite path element %p", gsp_path->fillcolor, gsp_path);
}

/** Bezier functions **/

Bezier_Vectors *
new_bezier_vectors(Vector * p0, Vector * p1, Vector * p2, Vector * p3) {
  Bezier_Vectors *r;

  r = malloc(sizeof(Bezier_Vectors));
	r->p0 = copy_vector(p0);
  r->p1 = copy_vector(p1);
  r->p2 = copy_vector(p2);
	r->p3 = copy_vector(p3);
	GDEBUG("Creating bezier vectors %p and vectors %p, %p, %p, and %p", r, r->p0, r->p1, r->p2, r->p3);

  return r;
}

int
del_bezier_vectors(Bezier_Vectors * bvectors) {
	GDEBUG("Removing bezier vectors %p and vectors %p, %p, %p, and %p", bvectors, bvectors->p0, bvectors->p1, bvectors->p2, bvectors->p3);
  del_vector(bvectors->p0);
  del_vector(bvectors->p1);
  del_vector(bvectors->p2);
  del_vector(bvectors->p3);
  free(bvectors);
  return 0;
}
