/* $Id: object.c,v 1.42 2003/06/07 13:43:44 bsenders Exp $ */

#ifndef _GNU_SOURCE
#define _GNU_SOURCE
#endif

#include <stdio.h>
#include <stdlib.h>
#include <assert.h>
#include <string.h>

#include "global.h"
#include "model.h"
#include "object.h"
#include "food.h"
#include "player.h"
#include "collision.h"

#define MDEBUG(...)      DEBUG(DMODEL, "Model", __VA_ARGS__)


/*****************************************************************************/
/* Model object manipulation functions                                       */
/*****************************************************************************/

Model_object * 
new_object(Model *m, Object_type type, Model_object *owner, 
           int32_t x, int32_t y, int32_t width, int32_t height) {
  Model_object *o;
  
  /* Valid type? */
  assert(type >= OT_FIRST && type < OT_NRTYPES);

  /* Create model object. */
  o = malloc(sizeof(Model_object));
  assert(o != NULL);

  /* Initialize with default settings and parameters. */
  /* Public */
  o->type      = type;
  o->pos       = new_vector(x, y);
  o->size      = new_vector(width, height);
  o->speed     = new_vector(0, 0);
  o->score     = 0;
  o->life      = m->settings->player_max_life;
  o->foodstack = new_list();
  o->lookdir   = (x < (m->width / 2)) ? LD_RIGHT : LD_LEFT;
  o->events    = 0;
  if (IS_FOOD(o))
    o->state.fs = FS_NORMAL;
  else if (IS_PLAYER(o))
    o->state.ps = PS_NORMAL;
  o->gfx_state = GFX_NORMAL;
  /* Private */
  o->active    = FALSE;
  o->disown    = FALSE;
  o->gfx_data  = NULL;
  o->owner     = owner;
  o->accel     = new_vector(0, 0);
  o->des_pos   = new_vector(x, y);
  if (IS_PLAYER(o))
    o->max_speed = copy_vector(m->settings->player_max_speed);
  else if (IS_FOOD(o))
    o->max_speed = copy_vector(m->settings->food_max_speed);
  else
    o->max_speed = new_vector(0, 0);
  o->gravity           = copy_vector(m->settings->gravity);
  /* Limit bounciness to X/Y scale. */
  o->bounciness        = new_vector(MIN(m->settings->bounciness->x,
                                        MODEL_X_SCALE),
                                    MIN(m->settings->bounciness->y,
				        MODEL_Y_SCALE));
  if (IS_PLAYER(o)) {
    o->throw_speed     = copy_vector(m->settings->food_throw_speed);
    o->jump_speed      = copy_vector(m->settings->player_jump_speed);
    o->friction_air    = m->settings->player_air_friction;
    o->friction_ground = m->settings->player_ground_friction;
    o->accel_air       = m->settings->player_air_accel;
    o->accel_ground    = m->settings->player_ground_accel;
  }
  else if (IS_FOOD(o)) {
    o->throw_speed     = new_vector(0, 0);
    o->jump_speed      = new_vector(0, 0);
    o->friction_air    = m->settings->food_air_friction;
    o->friction_ground = m->settings->food_ground_friction;
    o->accel_air       = 0;
    o->accel_ground    = 0;
  }
  else {
    o->throw_speed     = new_vector(0, 0);
    o->jump_speed      = new_vector(0, 0);
    o->friction_air    = 0;
    o->friction_ground = 0;
    o->accel_air       = 0;
    o->accel_ground    = 0;
  }
  o->frame_revive      = 0;
  o->frame_powerup_exp = 0;
  o->left = o->right = o->jump = o->action = o->hit = FALSE;
  o->is_standing = FALSE;

  /* Add object to list of all objects and to right sublist. */
  list_append(m->objects, o);

  if (IS_HOME(o))
    list_append(m->homes, o);
  else if (IS_FOOD(o))
    list_append(m->pas_food, o);
  else if (IS_BOX(o))
    list_append(m->boxes, o);
  else if (IS_PLAYER(o))
    list_append(m->pas_players, o);

  object_dump(o, "Created object: ");

  return o;
}

void
del_object(Model *m, Model_object *o) {
  int r = FALSE;
  Object_type *type;

  assert(IS_DELETED(o) && o->gfx_data == NULL);

  object_dump(o, "Deleting object: ");
  
  /* Remove from models objects list (if it's still it). */
  list_del(m->objects, o);

  /* Remove vectors. */
  del_vector(o->pos);
  del_vector(o->size);
  del_vector(o->speed);
  del_vector(o->accel);
  del_vector(o->des_pos);
  del_vector(o->max_speed);
  del_vector(o->bounciness);
  del_vector(o->throw_speed);
  del_vector(o->jump_speed);
  
  /* Purge food stack */
  while ((type = (Object_type *)list_pop(o->foodstack)))
    free(type);
  del_list(o->foodstack);

  /* Determine type and remove from object sublist in question. */
  if (IS_HOME(o))
    r = list_del(m->homes, o);
  else if (IS_BOX(o))
    r = list_del(m->boxes, o);
  else if (IS_FOOD(o))
    r = IS_ACTIVE(o) ? list_del(m->act_food, o)
                     : list_del(m->pas_food, o);
  else if (IS_PLAYER(o)) {
    if (IS_UNCONSCIOUS(o) || IS_DEAD(o))
      r = IS_ACTIVE(o) ? list_del(m->act_uplayers, o) 
                       : list_del(m->pas_uplayers, o);
    else
      r = IS_ACTIVE(o) ? list_del(m->act_players, o)
                       : list_del(m->pas_players, o);
  }
  assert(r && "Couldn't delete object from sublist.");

  free(o);
}

/* Generates a string representing the stack of the player o
 * and returns it. */
static char *
object_get_stack(Model_object *o) {
  int l;
  char *s;
  
  l = list_length(o->foodstack) + 1;
  s = malloc(l * sizeof(char));

  memset(s, 0, l);
  list_foreach(o->foodstack, food_get_short_string, s);

  return s;
}

int
object_dump(void *object, void *data) {
  char *s = (char *)data;
  Model_object *o = (Model_object *)object;
  char *buf;
  char *buftype;
  char *stack;
  
  if (IS_PLAYER(o)) {
    stack = object_get_stack(o);
    asprintf(&buftype, "state=%s lookdir=%s%s%s stack=%s",
                       player_state_string[o->state.ps],
                       look_dir_string[o->lookdir],
                       IS_ACTIVE(o) ? "" : " passive", 
		       IS_UNCONSCIOUS(o) ? " unconc" : 
			                   (IS_DEAD(o) ? " dead" : ""),
		       stack);
    free(stack);
  }
  else if (IS_FOOD(o))
    asprintf(&buftype, "state=%s", food_state_string[o->state.fs]);
  else
    buftype = NULL;

  asprintf(&buf, 
          "%s<%s:%p size=(%d,%d) pos=(%d,%d) des_pos=(%d, %d) speed=(%d, %d) accel=(%d, %d) gfx=%s events=%X %s>",
	  s == NULL ? "" : s,
	  object_type_string[o->type], o,
	  o->size->x,    o->size->y,
	  o->pos->x,     o->pos->y,
	  o->des_pos->x, o->des_pos->y,
	  o->speed->x,   o->speed->y,
	  o->accel->x,   o->accel->y,
	  graphics_state_string[o->gfx_state],
	  o->events, buftype == NULL ? "" : buftype);

  MDEBUG("%s", buf);

  free(buf);
  free(buftype);

  return TRUE;
}

int
object_set_deleted(void *object, void *user_data) {
  ((Model_object *)object)->gfx_state = GFX_DELETED;

  return TRUE;
}

int
object_check_deleted(void *object, void *model) {
  Model *m = (Model *)model;
  Model_object *o = (Model_object *)object;

  /* Check if object is deleted and remove it, if so. */
  if (IS_DELETED(o) && o->gfx_data == NULL)
    del_object(m, o);

  return TRUE;
}


/*****************************************************************************/
/* Model object active/passive function                                      */
/*****************************************************************************/

/* Move object from right active player/food list to right passive player/food
 * list, unset active flag en disable gravity.
 * This function assumes object is marked active!
 */
static void
object_make_passive(Model *m, Model_object *o) {
  int r = FALSE;
  
  /* Can only make active objects passive. */
  assert(IS_ACTIVE(o));

  /* A passive object should not want to move. */
  o->speed->x = 0;
  o->speed->y = 0;
  
  /* Move object to right list. */
  if (IS_FOOD(o)) {
    r = list_del(m->act_food, o);
    list_append(m->pas_food, o);
  }
  else if (IS_PLAYER(o)) {
    if (IS_UNCONSCIOUS(o) || IS_DEAD(o)) {
      r = list_del(m->act_uplayers, o);
      list_append(m->pas_uplayers, o);
    }
    else {
      r = list_del(m->act_players, o); 
      list_append(m->pas_players, o);
    }
  }
  else
    assert(FALSE && "Invalid type to make passive.");

  assert(r && "Passive object not removed from active sublist.");

  /* Mark object passive. */
  o->active = FALSE;

  /* Disable gravity. */
  o->accel->y = 0;

  object_dump(o, "Made passive: ");
}

/* Move object from right passive player/food list to right active
 * player/food list, set active flag en enable gravity.
 * This function assumes object is marked passive!
 */
void
object_make_active(Model *m, Model_object *o) {
  int r = FALSE;

  /* Can only make passive object active. */
  assert(!IS_ACTIVE(o));

  /* Move object to right list. */
  if (IS_FOOD(o)) {
    r = list_del(m->pas_food, o);
    list_append(m->act_food, o);
  }
  else if (IS_PLAYER(o)) {
    if (IS_UNCONSCIOUS(o) || IS_DEAD(o)) {
      r = list_del(m->pas_uplayers, o);
      list_append(m->act_uplayers, o);
    }
    else {
      r = list_del(m->pas_players, o); 
      list_append(m->act_players, o);
    }
  }
  else
    assert(FALSE && "Invalid type to make active.");

  assert(r && "Active object not removed from passive sublist");

  /* Mark object active. */
  o->active = TRUE;

  /* Enable gravity (except not for powered up banana and orange). */
  if (!IS_POWEREDUP(o) || !(o->type == OT_BANANA || o->type == OT_ORANGE))
    o->accel->y = m->settings->gravity->y;

  object_dump(o, "Made active: ");
}


/*****************************************************************************/
/* Model object mechanics functions                                          */
/*****************************************************************************/

/* Desired position becomes real position of active object. */
static void
object_fix_pos(Model *m, Model_object *o) {
  if (IS_ACTIVE(o)) {
    o->pos->x = o->des_pos->x;
    o->pos->y = o->des_pos->y;
  }
}

/* Accelerate X speed of object until it reaches maximum (in both 
 * directions; left and right). */
static void
object_accel_x(Model_object *o) {
  o->speed->x = MIN(MAX(o->speed->x + o->accel->x, -o->max_speed->x), 
                                                    o->max_speed->x);
}

/* Decelerate X speed of object until it reaches zero. */
static void
object_decel_x(Model_object *o) {
  if (o->speed->x < 0)
    /* Moving left, slow down. */
    o->speed->x = MIN(o->speed->x + o->accel->x, 0);
  else if (o->speed->x > 0)
    /* Moving right, slow down. */
    o->speed->x = MAX(o->speed->x + o->accel->x, 0);
}

/* Change speed on object depending on acceleration. */
static void
object_set_speed(Model_object *o) {
  if (o->speed->x == 0)
    /* Object is still. */
    object_accel_x(o);
  else if (SGN(o->speed->x) == SGN(o->accel->x))
    /* X accerlation is in same direction as speed. */
    object_accel_x(o);
  else
    /* X acceleration is in opposite direction as speed, break. */
    object_decel_x(o);
  
  /* Y acceleration can only be positive (gravity). */
  o->speed->y = MIN(o->speed->y + o->accel->y, o->max_speed->y);
}

/* Calculate desired position of object depending on speed. */
static void
object_set_des_pos(Model_object *o) {
  /* Set new desired position vector. */
  o->des_pos->x += o->speed->x / MODEL_X_SCALE;
  o->des_pos->y += o->speed->y / MODEL_Y_SCALE;
}


/*****************************************************************************/
/* Model object tick function                                                */
/*****************************************************************************/

int
object_passive_tick(void *object, void *model) {
  Model *m = (Model *)model;
  Model_object *o = (Model_object *)object;

  assert(!IS_ACTIVE(o));
  
  /* Make object active if gravity will move the object
   * or if desired position is not current position (object
   * wants to be moved). */
  if (!vectors_equal(o->pos, o->des_pos)) {
    /* Desired position is not current position, not sure if
     * it's standing, let collision resolving make that out. */
    o->is_standing = FALSE;
    object_make_active(m, o);
  }
  else if (has_no_floor(m, o)) {
    /* Gravity will move the object, object is not standing. */
    o->is_standing = FALSE;
    object_make_active(m, o);
  }
  else 
    /* Gravity can not move the object, nor does it desire to move.
     * The object is standing. */
    o->is_standing = TRUE;

  return TRUE;
}

int
object_active_tick(void *object, void *model) {
  Model *m = (Model *)model;
  Model_object *o = (Model_object *)object;

  assert(IS_ACTIVE(o));

  /* Make object passive if there is no movement or object is standing
   * (and not moving). */
  /* FIXME: If an object was just above a box in the previous tick, then
   * gravity in this tick will cause the object to collide with the box, and
   * collision detection will move it back to the original position.  This will
   * cause the object to become passive, while it should actually have bounced.
   * */
  if (vectors_equal(o->pos, o->des_pos) && IS_STANDING(o))
    object_make_passive(m, o);
  else
    /* Object is active, so objets desired position is not current position,
     * move the object. */
    object_fix_pos(m, o); 

  return TRUE;
}

int
object_tick(void *object, void *model) {
  Model *m = (Model *)model;
  Model_object *o = (Model_object *)object;

  /* No events registered yet */
  o->events = 0;

  /* Handle events on players or powerups on food. */
  if (IS_PLAYER(o))
    player_tick(m, o);
  else if (IS_FOOD(o))
    food_tick(m, o);
  else
    return TRUE;

  /* Calculate the new speed. */
  object_set_speed(o);

  /* Calculate the new (desired) position of the object. */
  object_set_des_pos(o);
    
  /* Let collision recheck object stands afterwards. */
  o->is_standing = FALSE;

  return TRUE;
}
