/* Signal management - Be conservative with what finit responds to!
 *
 * Copyright (c) 2008-2010  Claudio Matsuoka <cmatsuoka@gmail.com>
 * Copyright (c) 2008-2021  Joachim Wiberg <troglobit@gmail.com>
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 *
 * Finit is often used on embedded and small Linux systems with BusyBox.
 * For compatibility with its existing toolset (halt, poweroff, reboot),
 * the following signals have been adopted:
 *
 * SIGHUP
 *      Same effect as initctl reload, reloads *.conf in /etc/finit.d/
 *      Also restarts API (initctl) socket, like SysV init and systemd
 *      does on USR1 with their FIFO/D-Bus.
 *
 * SIGUSR1
 *      Restarts API (initctl) socket, like SysV init and systemd does
 *      on USR1 with their FIFO/D-Bus.
 *
 *      Finit <= 4.0 used to perform a system halt on USR1.  This caused
 *      some quite nasty problems on systems with both systemd/sysvinit
 *      installed alongside finit.  For compatibility reasons Finit 4.1
 *      changed to partial SIGHUP (api_exit/init).
 *
 * SIGUSR2
 *      Calls shutdown hooks, including HOOK_SHUTDOWN, stops all running
 *      processes, and unmounts all file systems.  Then tells kernel to
 *      power-off the system, if ACPI or similar exists to actually do
 *      this.  If the kernel fails power-off, Finit falls back to halt.
 *
 *      (SysV init N/A, systemd dumps its internal state to log.)
 *
 * SIGTERM
 *      Like SIGUSR2, but tell kernel to reboot the system when done.
 *
 *      (SysV init N/A, systemd rexecutes itself.)
 *
 * SIGINT
 *      Sent from kernel when the CTRL-ALT-DEL key combo is pressed.
 *      SysV init and systemd default to reboot with `shutdown -r`.
 *      Finit defaults to nothing but sets sys/key/ctrlaltdel condition.
 *
 * SIGWINCH
 *      Sent from kernel when the KeyboardSignal key is pressed.  Some
 *      users bind keys, see loadkeys(1), to create consoles on demand.
 *      Finit currently ignores this signal.
 *
 * SIGPWR
 *      Sent from a power daemon, like powstatd(8), on changes to the
 *      UPS status.  Traditionally SysV init read /etc/powerstatus and
 *      acted on "OK", "FAIL", or "LOW" and then removed the file.
 *      Finit  defaults to nothing but sets sys/pwr/fail condition.
 */

#include <dirent.h>
#include <string.h>		/* strerror() */
#include <sys/reboot.h>
#include <sys/wait.h>
#include <lite/lite.h>

#include "finit.h"
#include "cond.h"
#include "conf.h"
#include "config.h"
#include "helpers.h"
#include "plugin.h"
#include "private.h"
#include "sig.h"
#include "service.h"
#include "util.h"
#include "utmp-api.h"

extern svc_t *wdog;

/*
 * Old-style SysV shutdown sends a setenv cmd INIT_HALT with "=HALT",
 * "=POWERDOWN", or "" to cancel shutdown, before requesting change to
 * runlevel 6 over the /dev/initctl FIFO.
 */
shutop_t halt = SHUT_DEFAULT;

static uev_t sigterm_watcher, sigusr1_watcher, sigusr2_watcher;
static uev_t sighup_watcher,  sigint_watcher,  sigpwr_watcher;
static uev_t sigchld_watcher;

static struct sigmap {
	int   num;
	char *name;
} signames[] = {
	/* ISO C99 signals.  */
	{ SIGINT,    "SIGINT"    },
	{ SIGILL,    "SIGILL"    },
	{ SIGABRT,   "SIGABRT"   },
	{ SIGFPE,    "SIGFPE"    },
	{ SIGSEGV,   "SIGSEGV"   },
	{ SIGTERM,   "SIGTERM"   },
	/* Historical signals specified by POSIX. */
	{ SIGHUP,    "SIGHUP"    },
	{ SIGQUIT,   "SIGQUIT"   },
	{ SIGTRAP,   "SIGTRAP"   },
	{ SIGKILL,   "SIGKILL"   },
	{ SIGBUS,    "SIGBUS"    },
	{ SIGSYS,    "SIGSYS"    },
	{ SIGPIPE,   "SIGPIPE"   },
	{ SIGALRM,   "SIGALRM"   },

	/* New(er) POSIX signals (1003.1-2008, 1003.1-2013). */
	{ SIGURG,    "SIGURG"    },
	{ SIGSTOP,   "SIGSTOP"   },
	{ SIGTSTP,   "SIGTSTP"   },
	{ SIGCONT,   "SIGCONT"   },
	{ SIGCHLD,   "SIGCHLD"   },
	{ SIGTTIN,   "SIGTTIN"   },
	{ SIGTTOU,   "SIGTTOU"   },
	{ SIGPOLL,   "SIGPOLL"   },
	{ SIGXCPU,   "SIGXCPU"   },
	{ SIGXFSZ,   "SIGXFSZ"   },
	{ SIGVTALRM, "SIGVTALRM" },
	{ SIGPROF,   "SIGPROF"   },
	{ SIGUSR1,   "SIGUSR1"   },
	{ SIGUSR2,   "SIGUSR2"   },

	/* Nonstandard signals found in all modern POSIX systems
	   (including both BSD and Linux).  */
	{ SIGWINCH,  "SIGWINCH"  },
#ifdef SIGSTKFLT
	{ SIGSTKFLT, "SIGSTKFLT" },
#endif
	{ SIGPWR,    "SIGPWR"    },

	/* Archaic names for compatibility.  */
#ifdef SIGIO
	{ SIGIO,     "SIGIO"     },
#endif
#ifdef SIGIOT
	{ SIGIOT,    "SIGIOT"    },
#endif
#ifdef SIGCLD
	{ SIGCLD,    "SIGCLD"    },
#endif
};

void mdadm_wait(void);
void unmount_tmpfs(void);
void unmount_regular(void);

/*
 * Kernel threads have no cmdline so fgets() returns NULL for them.  We
 * also skip "special" processes, e.g. mdadm/mdmon or watchdogd that must
 * not be stopped here, for various reasons.
 *
 * https://www.freedesktop.org/wiki/Software/systemd/RootStorageDaemons/
 */
void do_kill(int signo)
{
	DIR *dirp;

	dirp = opendir("/proc");
	if (dirp) {
		struct dirent *d;

		while ((d = readdir(dirp))) {
			int pid;
			FILE *fp;
			char file[LINE_SIZE] = "";

			if (d->d_type != DT_DIR)
				continue;

			pid = atoi(d->d_name);
			if (!pid)
				continue;

			snprintf(file, sizeof(file), "/proc/%s/cmdline", d->d_name);
			fp = fopen(file, "r");
			if (!fp)
				continue;

			if (fgets(file, sizeof(file), fp)) {
				if (strstr(file, "gdbserver"))
					_d("Skipping %s ...", file);
				else if (file[0] == '@')
					_d("Skipping %s ...", &file[1]);
				else
					kill(pid, signo);
			}
			fclose(fp);
		}
		closedir(dirp);
	}
}

void do_shutdown(shutop_t op)
{
	if (sdown)
		run_interactive(sdown, "Calling shutdown hook: %s", sdown);

	/* Update UTMP db */
	utmp_set_halt();

	/*
	 * Tell remaining non-monitored processes to exit, give them
	 * time to exit gracefully, 2 sec was customary, we go for 1.
	 */
	do_kill(SIGTERM);
	do_sleep(1);
	do_kill(SIGKILL);

	/* Exit plugins and API gracefully */
	plugin_exit();
	api_exit();

	/* Reap 'em */
	while (waitpid(-1, NULL, WNOHANG) > 0)
		;

	/* Close all local non-console descriptors */
	for (int fd = 3; fd < 128; fd++)
		close(fd);

	if (vfork()) {
		/*
		 * Put PID 1 aside and let child perform reboot/halt
		 * kernel may exit child and we don't want to exit PID 1
		 * ... causing "aiii killing init" during reboot ...
		 */
		return;
	}

	if (wdog) {
		print(kill(wdog->pid, SIGPWR) == 1, "Advising watchdog, system going down");
		do_sleep(2);
	}

	/* Unmount any tmpfs before unmounting swap ... */
	unmount_tmpfs();
	run("swapoff -e -a");

	/* ... unmount remaining regular file systems. */
	unmount_regular();

	/* We sit on / so we must remount it ro, try all the things! */
	sync();
	run("mount -n -o remount,ro -t dummytype dummydev /");
	run("mount -n -o remount,ro dummydev /");
	run("mount -n -o remount,ro /");

	/* Call mdadm to mark any RAID array(s) as clean before halting. */
	mdadm_wait();

	/* Reboot via watchdog or kernel, or shutdown? */
	if (op == SHUT_REBOOT) {
		if (wdog && wdog->pid > 1) {
			int timeout = 10;

			/* Wait here until the WDT reboots, or timeout with fallback */
			print(kill(wdog->pid, SIGTERM) == 1, "Pending watchdog reboot");
			while (timeout--)
				do_sleep(1);
		}

		_d("Rebooting ...");
		reboot(RB_AUTOBOOT);
	} else if (op == SHUT_OFF) {
		_d("Powering down ...");
		reboot(RB_POWER_OFF);
	}

	/* Also fallback if any of the other two fails */
	_d("Halting ...");
	reboot(RB_HALT_SYSTEM);
}

/*
 * Reload .conf files in /etc/finit.d/
 */
static void sighup_cb(uev_t *w, void *arg, int events)
{
	_d("...");
	if (UEV_ERROR == events) {
		_e("Unrecoverable error in signal watcher");
		return;
	}

	/* Restart initctl API domain socket, similar to systemd/SysV init */
	api_exit();
	api_init(w->ctx);

	/* INIT_CMD_RELOAD: 'init q', 'initctl reload', and SIGHUP */
	service_reload_dynamic();
}

/*
 * SIGINT: generates <sys/key/ctrlaltdel> condition, which the sys.so
 *         plugin picks up and tells Finit to start any service(s) or
 *         task(s) associated with the condition.
 */
static void sigint_cb(uev_t *w, void *arg, int events)
{
	_d("...");
	if (UEV_ERROR == events) {
		_e("Unrecoverable error in signal watcher");
		return;
	}

	cond_set_oneshot_noupdate("sys/key/ctrlaltdel");
}

/*
 * SIGPWR: generates <sys/pwr/fail> condition, which the sys.so plugin
 *         picks up and tells Finit to start any service(s) or task(s)
 *         associated with the condition.
 */
static void sigpwr_cb(uev_t *w, void *arg, int events)
{
	_d("...");
	if (UEV_ERROR == events) {
		_e("Unrecoverable error in signal watcher");
		return;
	}

	cond_set_oneshot_noupdate("sys/pwr/fail");
}

/*
 * SIGUSR1: SysV init/systemd API socket restart
 */
static void sigusr1_cb(uev_t *w, void *arg, int events)
{
	_d("...");
	if (UEV_ERROR == events) {
		_e("Unrecoverable error in signal watcher");
		return;
	}

	/* Restart initctl API domain socket, similar to systemd/SysV init */
	api_exit();
	api_init(w->ctx);
}

/*
 * SIGUSR2: BusyBox style poweroff
 */
static void sigusr2_cb(uev_t *w, void *arg, int events)
{
	_d("...");
	if (UEV_ERROR == events) {
		_e("Unrecoverable error in signal watcher");
		return;
	}

	halt = SHUT_OFF;
	service_runlevel(0);
}

/*
 * SIGTERM: BusyBox style reboot
 */
static void sigterm_cb(uev_t *w, void *arg, int events)
{
	_d("...");
	if (UEV_ERROR == events) {
		_e("Unrecoverable error in signal watcher");
		return;
	}

	halt = SHUT_REBOOT;
	service_runlevel(6);
}

/*
 * SIGCHLD: one of our children has died
 */
static void sigchld_cb(uev_t *w, void *arg, int events)
{
	pid_t pid;
	int status;

	if (UEV_ERROR == events) {
		_e("Unrecoverable error in signal watcher");
		return;
	}

	/* Reap all the children! */
	while (1) {
		pid = waitpid(-1, &status, WNOHANG);
		if (pid <= 0) {
			if (pid == -1 && errno == EINTR)
				continue;
			break;
		}

		_d("Collected child %d", pid);
		service_monitor(pid, status);
	}
}

/*
 * Convert SIGFOO to a number, if it exists
 */
int sig_num(const char *name)
{
	size_t i, offset = 0;

	if (strncasecmp(name, "SIG", 3))
		offset = 3;

	for (i = 0; i < NELEMS(signames); i++) {
		if (strcasecmp(name, &signames[i].name[offset]))
			continue;

		return signames[i].num;
	}

	return -1;
}

const char *sig_name(int signo)
{
	size_t i;

	for (i = 0; i < NELEMS(signames); i++) {
		if (signames[i].num == signo)
			return signames[i].name;
	}

	return "SIGUNKOWN";
}

/*
 * Inital signal setup - ignore everything but SIGCHLD until we're capable of responding
 */
static void chld_handler(int sig, siginfo_t *info, void *ctx)
{
	/* NOP */
}

void sig_init(void)
{
	int i;
	struct sigaction sa;

	for (i = 1; i < NSIG; i++)
		IGNSIG(sa, i, SA_RESTART);

	SETSIG(sa, SIGCHLD, chld_handler, SA_RESTART);
}

/*
 * Unblock all signals blocked by finit when starting children
 */
void sig_unblock(void)
{
	int i;
	sigset_t nmask;
	struct sigaction sa;

	sigemptyset(&nmask);
	sigaddset(&nmask, SIGHUP);
	sigaddset(&nmask, SIGCHLD);
	sigaddset(&nmask, SIGINT);
	sigaddset(&nmask, SIGPWR);
	sigaddset(&nmask, SIGSTOP);
	sigaddset(&nmask, SIGTSTP);
	sigaddset(&nmask, SIGCONT);
	sigaddset(&nmask, SIGTERM);
	sigaddset(&nmask, SIGUSR1);
	sigaddset(&nmask, SIGUSR2);
	sigprocmask(SIG_UNBLOCK, &nmask, NULL);

	/* Reset signal handlers that were set by the parent process */
	for (i = 1; i < NSIG; i++)
		DFLSIG(sa, i, 0);
}

/*
 * Setup limited set of SysV compatible signals to respond to
 */
void sig_setup(uev_ctx_t *ctx)
{
	struct sigaction sa;

	_d("Setup signals");

	/*
	 * Standard SysV init calls ctrl-alt-delete handler
	 * We need to disable kernel default so it sends us SIGINT
	 */
	reboot(RB_DISABLE_CAD);
	uev_signal_init(ctx, &sigint_watcher, sigint_cb, NULL, SIGINT);

	/* BusyBox/SysV init style signals for halt, power-off and reboot. */
	uev_signal_init(ctx, &sigusr1_watcher, sigusr1_cb, NULL, SIGUSR1);
	uev_signal_init(ctx, &sigusr2_watcher, sigusr2_cb, NULL, SIGUSR2);
	uev_signal_init(ctx, &sigpwr_watcher,  sigpwr_cb,  NULL, SIGPWR);
	uev_signal_init(ctx, &sigterm_watcher, sigterm_cb, NULL, SIGTERM);

	/* Some C APIs may need SIGALRM for implementing timers. */
	IGNSIG(sa, SIGALRM, 0);

	/* /etc/inittab not supported yet, instead /etc/finit.d/ is scanned for *.conf */
	uev_signal_init(ctx, &sighup_watcher, sighup_cb, NULL, SIGHUP);

	/* After initial bootstrap of Finit we call the service monitor to reap children */
	uev_signal_init(ctx, &sigchld_watcher, sigchld_cb, NULL, SIGCHLD);

	setsid();
}

/**
 * Local Variables:
 *  indent-tabs-mode: t
 *  c-file-style: "linux"
 * End:
 */
