/*---[ hitview.c ]-----------------------------------------------------
 * Copyright (C) 2002 Tomas Junnonen (majix@sci.fi)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * The Hits page and related functions
 *--------------------------------------------------------------------*/

#include <config.h>
#include <gnome.h>

#include "firestarter.h"
#include "globals.h"
#include "hitview.h"
#include "util.h"
#include "parse.h"
#include "menus.h"
#include "preferences.h"
 
static GtkListStore *hitstore;
static GtkWidget *hitview;
static Hit *last_hit = NULL;


const Hit *
get_last_hit (void)
{
	return last_hit;
}

/* [ create_text_column ]
 * Convinience funtion for creating a text column for a treeview
 */
static GtkTreeViewColumn *
create_text_column (gint number, gchar * title)
{
	GtkTreeViewColumn *column;
	GtkCellRenderer *renderer;

	renderer = gtk_cell_renderer_text_new ();
	column = gtk_tree_view_column_new_with_attributes (title,
		renderer, "text", number, NULL);

	return column;
}

/* [ hitview_clear ]
 * Clears the log CList
 */
void
hitview_clear (void)
{
	if (get_current_status () == STATUS_HIT)
		update_status (STATUS_RUNNING);

	gtk_list_store_clear (hitstore);
}

/* [ unselect_all ]
 * Unselect all entries in the hitview
 */
static void
unselect_all (void)
{
	GtkTreeSelection *s;

	s = gtk_tree_view_get_selection (GTK_TREE_VIEW (hitview));
	gtk_tree_selection_unselect_all (s);
}

/* [ has_selected ]
 * Return true if there are entries selected in the hitview
 */
static gboolean
has_selected (void)
{
	GtkTreeSelection *selection;
	GtkTreeIter iter;
	gboolean has_selected;

	selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (hitview));
	has_selected = gtk_tree_selection_get_selected (selection, NULL, &iter);

	return has_selected;
}

/* [ scroll_to_hit ]
 * Scroll the hitview to the hit iter points to. Only works in ascending sort mode
 */
static void
scroll_to_hit (GtkTreeIter *iter)
{
	static GtkTreeModel *model = NULL;
	gint colid;
	GtkSortType order;
	GtkTreePath *last;

	if (model == NULL)
		model = gtk_tree_view_get_model (GTK_TREE_VIEW (hitview));

	last = gtk_tree_model_get_path (model, iter);
	gtk_tree_sortable_get_sort_column_id (GTK_TREE_SORTABLE (model), &colid, &order);

	if (order == GTK_SORT_ASCENDING)
		gtk_tree_view_scroll_to_cell (GTK_TREE_VIEW (hitview), last, NULL, TRUE, 1.0, 1.0);

	 gtk_tree_path_free (last);
}

/* [ hitview_append_hit ]
 * Append a hit to the hitlist, return true if successful
 */
gboolean
hitview_append_hit (Hit *h)
{
	GtkTreeIter iter;

	if (preferences_get_bool (PREFS_FILTER_REDUNDANT))
		if (compare_to_last_hit (h)) {
			printf ("Hit filtered: Redundant\n");
			return FALSE;
		}

	if (preferences_get_bool (PREFS_FILTER_BROADCAST))
		if (!hit_is_for_me (h)) {
			printf ("Hit filtered: Someone else's problem \n");
			return FALSE;
		}

	if (preferences_get_bool (PREFS_AUTORESOLVE))
		h->source = lookup_ip (g_strdup(h->source));

	gtk_list_store_append (hitstore, &iter);
	gtk_list_store_set (hitstore, &iter,
	                    HITCOL_TIME,        h->time,
	                    HITCOL_IN,          h->in,
	                    HITCOL_OUT,         h->out,
	                    HITCOL_PORT,        h->port,
	                    HITCOL_SOURCE,      h->source,
	                    HITCOL_DESTINATION, h->destination,
	                    HITCOL_LENGTH,      h->length,
	                    HITCOL_TOS,         h->tos,
	                    HITCOL_PROTOCOL,    h->protocol,
	                    HITCOL_SERVICE,     h->service,
	                    -1);

	if (!has_selected ())
		scroll_to_hit (&iter);

	free_hit (last_hit);
	last_hit = copy_hit (h);
	
	return TRUE;
}

/* [ hitview_reload ]
 * Loads the entire kernel log file into the hitlist
 */
void
hitview_reload (void)
{	/* The log entries are compared against this pattern */
	GPatternSpec *pattern = g_pattern_spec_new ("* IN=* OUT=* SRC=*");

	gchar buf[512];
	Hit *h;
	FILE *f;

	f = fopen (get_system_log_path (), "r");
	if (f == NULL)
		return;

	hitview_clear ();

	while (fgets (buf, 512, f) != NULL) {
		if (g_pattern_match_string (pattern, buf)) { /* Is it a firewall entry? */
			h = parse_log_line (buf);
			hitview_append_hit (h);
			free_hit (h);
		}
	}

	fclose (f);
	g_pattern_spec_free (pattern);
}

/* [ create_hitlist_model ]
 * Creates the list for storage of hits
 */
static GtkTreeModel *
create_hitlist_model (void)
{
	hitstore = gtk_list_store_new (10,
		G_TYPE_STRING,
		G_TYPE_STRING,
		G_TYPE_STRING,
		G_TYPE_STRING,
		G_TYPE_STRING,
		G_TYPE_STRING,
		G_TYPE_STRING,
		G_TYPE_STRING,
		G_TYPE_STRING,
		G_TYPE_STRING);

	return GTK_TREE_MODEL (hitstore);
}

/* [ hitview_toggle_column_visibility ]
 * Negate the visibility of a column
 */
void
hitview_toggle_column_visibility (GtkWidget *widget, gint colnum)
{
	GtkTreeViewColumn *column;
	gboolean visible;

	g_assert (colnum < NUM_HITCOLUMNS);

	if (hitview == NULL)
		return;

	column = gtk_tree_view_get_column (GTK_TREE_VIEW (hitview), colnum);
	visible = !gtk_tree_view_column_get_visible (column);
	gtk_tree_view_column_set_visible(column, visible);

	switch (colnum) {
	  case 0: preferences_set_bool (PREFS_HITVIEW_TIME_COL, visible); break;
	  case 1: preferences_set_bool (PREFS_HITVIEW_IN_COL, visible); break;
	  case 2: preferences_set_bool (PREFS_HITVIEW_OUT_COL, visible); break;
	  case 3: preferences_set_bool (PREFS_HITVIEW_PORT_COL, visible); break;
	  case 4: preferences_set_bool (PREFS_HITVIEW_SOURCE_COL, visible); break;
	  case 5: preferences_set_bool (PREFS_HITVIEW_DESTINATION_COL, visible); break;
	  case 6: preferences_set_bool (PREFS_HITVIEW_LENGTH_COL, visible); break;
	  case 7: preferences_set_bool (PREFS_HITVIEW_TOS_COL, visible); break;
	  case 8: preferences_set_bool (PREFS_HITVIEW_PROTOCOL_COL, visible); break;
	  case 9: preferences_set_bool (PREFS_HITVIEW_SERVICE_COL, visible); break;
	}
}

/* [ hitview_add_columns ]
 * Add the columns to the hit TreeView
 */
static void
hitview_add_columns (GtkTreeView *treeview)
{
	GtkTreeViewColumn *column;
	gboolean visible;

	/* column for time */
	column = create_text_column (HITCOL_TIME, _("Time"));
	gtk_tree_view_column_set_sort_column_id (column, HITCOL_TIME);
	gtk_tree_view_append_column (treeview, column);
	visible = preferences_get_bool (PREFS_HITVIEW_TIME_COL);
	gtk_tree_view_column_set_visible (column, visible);

	/* column for in device */
	column = create_text_column (HITCOL_IN, _("In"));
	gtk_tree_view_column_set_sort_column_id (column, HITCOL_IN);
	gtk_tree_view_append_column (treeview, column);
	visible = preferences_get_bool (PREFS_HITVIEW_IN_COL);
	gtk_tree_view_column_set_visible (column, visible);

	/* column for out device */
	column = create_text_column (HITCOL_OUT, _("Out"));
	gtk_tree_view_column_set_sort_column_id (column, HITCOL_OUT);
	gtk_tree_view_append_column (treeview, column);
	visible = preferences_get_bool (PREFS_HITVIEW_OUT_COL);
	gtk_tree_view_column_set_visible (column, visible);

	/* column for port */
	column = create_text_column (HITCOL_PORT, _("Port"));
	gtk_tree_view_column_set_sort_column_id (column, HITCOL_PORT);
	gtk_tree_view_append_column (treeview, column);
	visible = preferences_get_bool (PREFS_HITVIEW_PORT_COL);
	gtk_tree_view_column_set_visible (column, visible);

	/* column for source */
	column = create_text_column (HITCOL_SOURCE, _("Source"));
	gtk_tree_view_column_set_sort_column_id (column, HITCOL_SOURCE);
	gtk_tree_view_append_column (treeview, column);
	visible = preferences_get_bool (PREFS_HITVIEW_SOURCE_COL);
	gtk_tree_view_column_set_visible (column, visible);

	/* column for destination */
	column = create_text_column (HITCOL_DESTINATION, _("Destination"));
	gtk_tree_view_column_set_sort_column_id (column, HITCOL_DESTINATION);
	gtk_tree_view_append_column (treeview, column);
	visible = preferences_get_bool (PREFS_HITVIEW_DESTINATION_COL);
	gtk_tree_view_column_set_visible (column, visible);
	
	/* column for packet length */
	column = create_text_column (HITCOL_LENGTH, _("Length"));
	gtk_tree_view_column_set_sort_column_id (column, HITCOL_LENGTH);
	gtk_tree_view_append_column (treeview, column);
	visible = preferences_get_bool (PREFS_HITVIEW_LENGTH_COL);
	gtk_tree_view_column_set_visible (column, visible);

	/* column for ToS */
	column = create_text_column (HITCOL_TOS, _("TOS"));
	gtk_tree_view_column_set_sort_column_id (column, HITCOL_TOS);
	gtk_tree_view_append_column (treeview, column);
	visible = preferences_get_bool (PREFS_HITVIEW_TOS_COL);
	gtk_tree_view_column_set_visible (column, visible);

	/* column for protocol */
	column = create_text_column (HITCOL_PROTOCOL, _("Protocol"));
	gtk_tree_view_column_set_sort_column_id (column, HITCOL_PROTOCOL);
	gtk_tree_view_append_column (treeview, column);	
	visible = preferences_get_bool (PREFS_HITVIEW_PROTOCOL_COL);
	gtk_tree_view_column_set_visible (column, visible);

	/* column for service */
	column = create_text_column (HITCOL_SERVICE, _("Service"));
	gtk_tree_view_column_set_sort_column_id (column, HITCOL_SERVICE);
	gtk_tree_view_append_column (treeview, column);	
	visible = preferences_get_bool (PREFS_HITVIEW_SERVICE_COL);
	gtk_tree_view_column_set_visible (column, visible);
}

/* [ get_hit ]
 * Retrieve the specific hit iter points to
 */
static Hit *
get_hit (GtkTreeModel *model,
         GtkTreeIter iter)
{
	Hit *h = g_new (Hit, 1);

	gtk_tree_model_get (model, &iter,
	                    HITCOL_TIME,        &h->time,
	                    HITCOL_IN,          &h->in,
	                    HITCOL_OUT,         &h->out,
	                    HITCOL_PORT,        &h->port,
	                    HITCOL_SOURCE,      &h->source,
	                    HITCOL_DESTINATION, &h->destination,
	                    HITCOL_LENGTH,      &h->length,
	                    HITCOL_TOS,         &h->tos,
	                    HITCOL_PROTOCOL,    &h->protocol,
	                    HITCOL_SERVICE,     &h->service,
	                    -1);

	return h; 
}

/* [ hit_activated_cb ]
 * Callback for selecting a row in the hit view
 * TODO: Default action for hits? Hit->Rule helper maybe.
 */
static void 
hit_activated_cb (GtkTreeView *treeview,
                 GtkTreePath *path,
                 GtkTreeViewColumn *arg2,
                 gpointer data)
{
	GtkTreeModel *model;
	GtkTreeIter iter;
	Hit *h;

	model = gtk_tree_view_get_model (treeview);
	gtk_tree_model_get_iter (model, &iter, path);
	h = get_hit (model, iter);

	print_hit (h);
	free_hit (h);

	unselect_all ();
}

/* [ hitview_get_selected_hit ]
 * Get the hit that is currently selected in the hitview
 */
Hit *
hitview_get_selected_hit (void)
{
	GtkTreeSelection *selection;
	GtkTreeModel *model;
	GtkTreeIter iter;
	Hit *h = NULL;
	gboolean has_selected;

	selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (hitview));

	has_selected = gtk_tree_selection_get_selected (selection,
	                                                NULL,
	                                                &iter);
	if (has_selected) {
		model = gtk_tree_view_get_model (GTK_TREE_VIEW (hitview));
		h = get_hit (model, iter);
	}

	return h;
}

GList *
hitview_get_all_hits (void)
{
	GList *hits = NULL;
	Hit *h;
	GtkTreeModel *model;
	GtkTreeIter iter;

	model = gtk_tree_view_get_model (GTK_TREE_VIEW (hitview));
	if (gtk_tree_model_get_iter_first (model, &iter)) {
		do {
			h = get_hit (model, iter);
			hits = g_list_append (hits, h);
		} while (gtk_tree_model_iter_next (model, &iter));
	} else
		printf ("Error compiling list of hits\n");

	return hits;
}

/* [ hitview_button_press_cb ]
 * Pop up an menu when right clicking the hitview
 */
static gboolean
hitview_button_press_cb (GtkWidget* widget, GdkEventButton* bevent)
{
	gboolean retval = FALSE;

	switch (bevent->button) {
		case 3: hitview_popup_menu (bevent);
			retval = TRUE;
			break;
	}

	return retval;
}

/* [ lookup_selected_hit ]
 * Resolve the IP address/hostname from the selected line in hitview
 */
void
lookup_selected_hit (void)
{
	GtkTreeSelection *selection;
	GtkTreeIter iter;
	gboolean has_selected;
	static GtkTreeModel *model = NULL;
	gchar *ip;
	gchar *hostname = NULL;

	selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (hitview));
	has_selected = gtk_tree_selection_get_selected (selection,
	                                                NULL,
	                                                &iter);
	if (!has_selected)
		return;

	if (!model)
		model = gtk_tree_view_get_model (GTK_TREE_VIEW (hitview));

	gtk_tree_model_get (model, &iter,
	                    HITCOL_SOURCE, &ip,
	                    -1);
	hostname = lookup_ip (ip);

	if (hostname != NULL)
		gtk_list_store_set (hitstore, &iter,
	        	            HITCOL_SOURCE, hostname,
	         	           -1);

	/* Resize the columns, good when converting hostname back to ip */
	gtk_tree_view_columns_autosize (GTK_TREE_VIEW (hitview));
}

/* [ month_number ]
 * Convert a three letter month identifier to a number
 */
static int
month_number (gchar *month)
{
	int num = 0;

	if (strcmp (month, "Jan") == 0)
		num = 1;
	else if (strcmp (month, "Feb") == 0)
		num = 2;
	else if (strcmp (month, "Mar") == 0)
		num = 3;
	else if (strcmp (month, "Apr") == 0)
		num = 4;
	else if (strcmp (month, "May") == 0)
		num = 5;
	else if (strcmp (month, "Jun") == 0)
		num = 6;
	else if (strcmp (month, "Jul") == 0)
		num = 7;
	else if (strcmp (month, "Aug") == 0)
		num = 8;
	else if (strcmp (month, "Sep") == 0)
		num = 9;
	else if (strcmp (month, "Oct") == 0)
		num = 10;
	else if (strcmp (month, "Nov") == 0)
		num = 11;
	else if (strcmp (month, "Dec") == 0)
		num = 12;

	return num;
}

/* [ time_sort_func ]
 * Function for sorting the time column
 */
static int
time_sort_func (GtkTreeModel *model, 
	        GtkTreeIter  *a, 
	        GtkTreeIter  *b, 
	        gpointer      user_data)
{
	enum { MONTH, DATE, CLOCK };

	gchar *data1, *data2;
	gchar **time1, **time2;
	gint month1, month2;
	gint day1, day2;
	gint sort = 0;

	gtk_tree_model_get (model, a, HITCOL_TIME, &data1, -1);
	gtk_tree_model_get (model, b, HITCOL_TIME, &data2, -1);

	time1 = g_strsplit (data1, " ", 3);
	time2 = g_strsplit (data2, " ", 3);

	month1 = month_number (time1[MONTH]);
	month2 = month_number (time2[MONTH]); 

	/* Compare first month, then the day, and last the clock */
	if (month1 != month2)
		sort = ((month1 < month2) ? -1:1);
	else {
		day1 = atoi (time1[DATE]);
		day2 = atoi (time2[DATE]);

		if (day1 != day2)
			sort = ((day1 < day2) ? -1:1);
		else
			sort = strcasecmp (time1[CLOCK], time2[CLOCK]);
	}

	g_free (data1);
	g_free (data2);
	g_strfreev (time1);
	g_strfreev (time2);

	return sort;
}

/* [ num_sort_func ]
 * Function for sorting a (text) column with only numbers in it
 */
static int
num_sort_func (GtkTreeModel *model, 
	       GtkTreeIter  *a, 
	       GtkTreeIter  *b, 
	       gpointer      column)
{
	gchar *data1, *data2;
	gint n1, n2;

	gtk_tree_model_get (model, a, (gint)column, &data1, -1);
	gtk_tree_model_get (model, b, (gint)column, &data2, -1);

	n1 = atoi (data1);
	n2 = atoi (data2);

	g_free (data1);
	g_free (data2);

	if (n1 == n2)
		return 0;
	else
		return ((n1 < n2) ? -1:1);
}

/* [ copy_selected_hit ]
 * Copy the selected hit to the clipboard
 */
void
copy_selected_hit (void)
{
	Hit *h;
	gchar *text;
	GtkClipboard *cb;

	h = hitview_get_selected_hit ();

	if (h == NULL)
		return;

	cb = gtk_clipboard_get (GDK_SELECTION_CLIPBOARD);

	text = g_strconcat (
		"Time: ", h->time,
		" Source: ", h->source,
		" Destination: ", h->destination,
		" In IF: ", h->in,
		" Out IF: ", h->out,
		" Port: ", h->port,
		" Length: ", h->length,
		" ToS: ", h->tos,
		" Protocol: ", h->protocol,
		" Service: ", h->service,
		NULL);

	gtk_clipboard_set_text (cb, text, strlen (text));
	g_free (text);
	free_hit (h);
}

/* [ create_hitview_page ]
 * Create the hitview
 */
GtkWidget *
create_hitview_page (void)
{
	GtkWidget *hitpagehbox;
	GtkTreeModel *hitmodel;
	GtkWidget *scrolledwin;

	hitpagehbox = gtk_hbox_new (FALSE, 0);
	hitmodel = create_hitlist_model ();
	hitview = gtk_tree_view_new_with_model (hitmodel);

	scrolledwin = gtk_scrolled_window_new (NULL, NULL);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolledwin),
	                                GTK_POLICY_NEVER,
	                                GTK_POLICY_ALWAYS);
									
	gtk_box_pack_start (GTK_BOX (hitpagehbox), scrolledwin, TRUE, TRUE, 0);

	/* Pack the treeview into the scrolled window  */
	gtk_container_add (GTK_CONTAINER (scrolledwin), hitview);
	hitview_add_columns (GTK_TREE_VIEW (hitview));
	gtk_tree_view_set_rules_hint (GTK_TREE_VIEW (hitview), TRUE);
	gtk_tree_view_set_search_column (GTK_TREE_VIEW (hitview),
		HITCOL_TIME);
  
	g_signal_connect (G_OBJECT (hitview), "row-activated",
	                  G_CALLBACK (hit_activated_cb), NULL);
	g_signal_connect (G_OBJECT (hitview), "button_press_event",
	                  G_CALLBACK (hitview_button_press_cb), NULL);

	/* The list is by default sorted by time */
	gtk_tree_sortable_set_sort_column_id (GTK_TREE_SORTABLE (hitmodel), HITCOL_TIME, GTK_SORT_ASCENDING);

	/* Some of the columns need special functions for sorting */
	gtk_tree_sortable_set_sort_func (GTK_TREE_SORTABLE (hitmodel), HITCOL_TIME,
	                                 time_sort_func, NULL, NULL);

	gtk_tree_sortable_set_sort_func (GTK_TREE_SORTABLE (hitmodel), HITCOL_PORT,
	                                 num_sort_func, (gpointer)HITCOL_PORT, NULL);

	gtk_tree_sortable_set_sort_func (GTK_TREE_SORTABLE (hitmodel), HITCOL_LENGTH,
	                                 num_sort_func, (gpointer)HITCOL_LENGTH, NULL);

	g_object_unref (G_OBJECT (hitmodel));

	return hitpagehbox;
}
