/*---[ parse.c ]------------------------------------------------------
 * Copyright (C) 2000-2002 Tomas Junnonen (majix@sci.fi)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * The text file parse functions.
 *--------------------------------------------------------------------*/
#include <netdb.h>

#include "firestarter.h"
#include "parse.h"
#include "service.h"
#include "util.h"
#include "ruleview.h"

Hit *
parse_log_line (gchar *line)
{
	struct protoent *protocol;

	Hit *h = g_new (Hit, 1);

	/* Take 15 first characters as timestamp */
	h->time = g_strndup (line, 15);

	h->in = get_text_between (line, "IN=", " ");
	h->out = get_text_between (line, "OUT=", " ");
	h->source = get_text_between (line, "SRC=", " ");
	h->destination = get_text_between (line, "DST=", " ");
	h->length = get_text_between (line, "LEN=", " ");
	h->tos = get_text_between (line, "TOS=", " ");
	h->protocol = get_text_between (line, "PROTO=", " ");
	h->port = get_text_between (line, "DPT=", " ");

	/* If the protocol is a number we do a protocol name lookup */
	if (h->protocol != NULL)
		if (g_ascii_isdigit(h->protocol[0])) {
			protocol = getprotobynumber (atoi (h->protocol));
			h->protocol = protocol->p_name;
		}
	h->protocol = g_utf8_strdown (h->protocol, strlen (h->protocol));
	
	/* Determine service used based on the port and protocol */
	if (h->port != NULL && h->protocol != NULL)
		h->service = get_service_name (atoi (h->port), g_strdup (h->protocol));
	else
		h->service = g_strdup ("unknown");

	return h;
}

void
parse_all_rules ()
{
	parse_trusted_hosts ();
	parse_blocked_hosts ();
	parse_forward ();
	parse_open_ports ();
	parse_stealthed_ports ();
	parse_blocked_ports ();
}

void
parse_trusted_hosts ()
{
	gchar buf[512];
	Rule *r = g_new (Rule, 1);
	FILE *f;
	
	f = fopen (get_file_path (RULETYPE_TRUSTED_HOST), "r");
	if (!f) {
		perror("Error reading file of trusted hosts");
		return;
	}

	while (fgets (buf, 512, f)) {
		g_strstrip (buf);

		r->type = RULETYPE_TRUSTED_HOST;
		r->host = g_strdup(buf);
		r->port = "";
		r->active = 1;

		ruleview_insert_rule (r);
	}

	fclose (f);
	g_free (r);
}

void
parse_blocked_hosts ()
{
	gchar buf[512];
	Rule *r = g_new (Rule, 1);
	FILE *f;
	
	f = fopen (get_file_path (RULETYPE_BLOCKED_HOST), "r");
	if (!f) {
		perror("Error reading file of blocked hosts");
		return;
	}

	while (fgets (buf, 512, f)) {
		g_strstrip (buf);

		r->type = RULETYPE_BLOCKED_HOST;
		r->host = g_strdup(buf);
		r->port = "";
		r->active = 1;

		ruleview_insert_rule (r);
	}

	fclose (f);
	g_free (r);
}

void
parse_forward ()
{
	gchar buf[512];
	Rule *r = g_new (Rule, 1);
	Forward *ex = g_new (Forward, 1);
	FILE *f;

	if (!is_capable_of_nat ()) {
		return;
	}
	
	f = fopen (get_file_path (RULETYPE_FORWARD), "r");
	if (!f) {
		perror("Error reading forwarding file");
		return;
	}

	while (fgets (buf, 512, f)) {
		gchar **elements = g_strsplit (buf, " ", 3);;
		g_strstrip (elements[0]);
		g_strstrip (elements[1]);
		g_strstrip (elements[2]);

		r->type = RULETYPE_FORWARD;
		r->host = "";
		r->port = g_strdup (elements[0]);
		r->active = 1;

		ex->int_host = g_strdup (elements[1]);
		ex->int_port = g_strdup (elements[2]);

		r->extension = ex;

		ruleview_insert_rule (r);
		g_strfreev (elements);
	}

	fclose (f);
	g_free (r);
	g_free (ex);
}

void
parse_open_ports ()
{
	gchar buf[512];
	Rule *r = g_new (Rule, 1);
	FILE *f;
	
	f = fopen (get_file_path (RULETYPE_OPEN_PORT), "r");
	if (!f) {
		perror("Error reading file of open ports");
		return;
	}

	while (fgets (buf, 512, f)) {
		g_strstrip (buf);

		r->type = RULETYPE_OPEN_PORT;
		r->host = "";
		r->port = g_strdup(buf);
		r->active = 1;

		ruleview_insert_rule (r);
	}

	fclose (f);
	g_free (r);
}

void
parse_stealthed_ports ()
{
	gchar buf[512];
	Rule *r = g_new (Rule, 1);
	FILE *f;
	
	f = fopen (get_file_path (RULETYPE_STEALTHED_PORT), "r");
	if (!f) {
		perror("Error reading file of stealthed ports");
		return;
	}

	while (fgets (buf, 512, f)) {
		gchar **elements = g_strsplit (buf, " ", 2);;
		g_strstrip (elements[0]);
		g_strstrip (elements[1]);

		r->type = RULETYPE_STEALTHED_PORT;
		r->host = g_strdup(elements[1]);
		r->port = g_strdup(elements[0]);
		r->active = 1;

		ruleview_insert_rule (r);
		g_strfreev (elements);
	}

	fclose (f);
	g_free (r);
}

void
parse_blocked_ports ()
{
	gchar buf[512];
	Rule *r = g_new (Rule, 1);
	FILE *f;
	
	f = fopen (get_file_path (RULETYPE_BLOCKED_PORT), "r");
	if (!f) {
		perror("Error reading file of blocked ports");
		return;
	}

	while (fgets (buf, 512, f)) {
		g_strstrip (buf);

		r->type = RULETYPE_BLOCKED_PORT;
		r->host = "";
		r->port = g_strdup(buf);
		r->active = 1;

		ruleview_insert_rule (r);
	}

	fclose (f);
	g_free (r);
}
