/*---[ util.c ]-------------------------------------------------------
 * Copyright (C) 2000-2002 Tomas Junnonen (majix@sci.fi)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Miscellaneous utility functions
 *--------------------------------------------------------------------*/

#include <sys/stat.h>
#include <stdio.h>
#include <netdb.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <net/if.h>
#include <sys/ioctl.h>

#include "globals.h"
#include "firestarter.h"
#include "util.h"
#include "hitview.h"
#include "preferences.h"

extern int h_errno;

static void
error_dialog_response (GtkDialog *dialog,
                       gint response_id,
                       gpointer data)
{
	gtk_widget_destroy (GTK_WIDGET (dialog));
}

/* [ error_dialog ]
 * Run a dialog with an specified error message
 */
void
error_dialog (gchar *message)
{
	GtkWidget *dialog;
	GtkWidget *hbox;
	GtkWidget *label;
	GdkPixbuf *pixbuf;
	GtkWidget *icon;
	GtkWindow *window = NULL;	

	if (Firestarter.window != NULL)
		window = GTK_WINDOW (Firestarter.window);

	dialog = gtk_dialog_new_with_buttons (
		_("Firestarter error"),
		window,
		GTK_DIALOG_DESTROY_WITH_PARENT | GTK_DIALOG_NO_SEPARATOR,
		GTK_STOCK_OK,
		GTK_RESPONSE_ACCEPT,
		NULL
	);
	gtk_container_set_border_width (GTK_CONTAINER (dialog), 6);

	/* If the main window is displayed we don't block and need a destroyer cb */
	if (window != NULL)
		g_signal_connect (G_OBJECT (dialog), "response", 
		                  G_CALLBACK (error_dialog_response), NULL);

	hbox = gtk_hbox_new (FALSE, 12);
	gtk_container_set_border_width (GTK_CONTAINER (hbox), 6);
	gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->vbox), hbox, FALSE, FALSE, 0);

	pixbuf = gtk_widget_render_icon (dialog, GTK_STOCK_DIALOG_ERROR, GTK_ICON_SIZE_DIALOG, NULL);
	icon = gtk_image_new_from_pixbuf (pixbuf);
	g_object_unref (G_OBJECT(pixbuf));	
	gtk_misc_set_alignment (GTK_MISC (icon), 0.0, 0.0);
	gtk_box_pack_start (GTK_BOX (hbox), icon, FALSE, FALSE, 0);
		
	label = gtk_label_new (message);
	gtk_label_set_use_markup (GTK_LABEL (label), TRUE);
	gtk_label_set_line_wrap (GTK_LABEL (label), TRUE);
	gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 0);

	gtk_widget_show_all (dialog);

	/* If the main window is not displayed we need an new event loop */
	if (window == NULL)
		gtk_dialog_run (GTK_DIALOG (dialog));
}

/* [ detect_netfilter ]
 * Return true if kernel supports netfilter (Linux 2.4 and newer)
 */
gboolean
detect_netfilter (void)
{
	FILE *f;
	gchar buf[512];
	char *ver;

	/* The detection is only done once */
	static gint detected = 0;

	/* The result of the detection */
	static gboolean result = FALSE;

	if (detected)
		return result;

	f = fopen ("/proc/version", "r");

	/* If we can't determine the kernel version */
	if (f == NULL) {
		perror("/proc/version");
		g_print("Could not determine kernel version,"
			"defaulting to 2.4/iptables\n");
		result = TRUE;
	} else {
		while (fgets (buf, 512, f) != NULL) {
			if (strstr (buf, "2.") != NULL){
				ver = strstr (buf, "2.");
				if(ver[2] >= '4') {
					g_print ("NETFILTER detected\n");
					if (access("/sbin/modprobe", R_OK) == 0)
						system ("/sbin/modprobe --autoclean ip_tables");
					else
						system ("`which modprobe` --autoclean ip_tables");
				
					result = TRUE;
				}
			}
		}
		fclose (f);
	}

	detected = 1;
	return result;
}

/* [ get_system_log_path ]
 * Get the correct path to the system log, which may vary with distributions
 */
const gchar *
get_system_log_path (void)
{
	static gchar *path = NULL;

	if (path == NULL) {
		FILE *f;

		/* User has specified the log file location */
		path = preferences_get_string (PREFS_SYSLOG_FILE);
		if (path) {
			return path;
		} else { /* Try to guess some default syslog location */
			f = fopen ("/var/log/messages", "r");
			if (f) {
				path = g_strdup ("/var/log/messages");
				fclose (f);
			}
		}
		
		if (path == NULL) {
			error_dialog (g_strconcat (
				"<span weight=\"bold\" size=\"larger\">",
				_("Failed to open the system log\n\n"),
				"</span>",
				_("No realtime hit information will be available. "
				"Please make sure the syslog daemon is running."),
				NULL));		
		}
	}
	return path;
}

/* [ get_file_path ]
 * Get the correct path to the rule file based on the group number
 */
gchar *
get_file_path (gint rulegroupnum)
{
	gchar *path;

	switch (rulegroupnum) {
	  case RULETYPE_TRUSTED_HOST:
		path = FIRESTARTER_RULES_DIR "/firestarter/trusted-hosts"; break;
	  case RULETYPE_BLOCKED_HOST:
		path = FIRESTARTER_RULES_DIR "/firestarter/blocked-hosts"; break;
	  case RULETYPE_FORWARD:
		path = FIRESTARTER_RULES_DIR "/firestarter/forward"; break;
	  case RULETYPE_OPEN_PORT:
		path = FIRESTARTER_RULES_DIR "/firestarter/open-ports"; break;
	  case RULETYPE_STEALTHED_PORT:
		path = FIRESTARTER_RULES_DIR "/firestarter/stealthed-ports"; break;
	  case RULETYPE_BLOCKED_PORT:
		path = FIRESTARTER_RULES_DIR "/firestarter/blocked-ports"; break;
	  default:
		path = NULL;
	}

	return path;
}

void
print_hit (Hit *h)
{
	printf ("HIT: %s from %s to %s:%s, protocol %s, service %s\n",
		h->time,
		h->source,
		h->destination,
		h->port,
		h->protocol,
		h->service);
}

void
print_rule (Rule *r)
{
	printf ("RULE: type %d, host %s, port %s, active %d\n",
		r->type,
		r->host,
		r->port,
		r->active);
}

Hit *
copy_hit (Hit *h)
{
	Hit *new = g_new (Hit, 1);
	
	new->time = g_strdup (h->time);
	new->in =  g_strdup (h->in);
	new->out = g_strdup (h->out);
	new->port = g_strdup (h->port);
	new->source = g_strdup (h->source);
	new->destination = g_strdup (h->destination);
	new->length = g_strdup (h->length);
	new->tos = g_strdup (h->tos);
	new->protocol = g_strdup (h->protocol);
	new->service = g_strdup (h->service);

	return new;
}

/* [ get_ip_of_interface ]
 * Get the IP address in use by an interface
 */
static gchar* get_ip_of_interface(gchar *itf) {
	int fd;	
	struct ifreq ifreq;
	struct sockaddr_in *sin;
	gchar *ip;

	fd = socket(AF_INET, SOCK_DGRAM, IPPROTO_IP);
	strcpy(ifreq.ifr_name, itf);
	ioctl(fd, SIOCGIFADDR, &ifreq);
	sin = (struct sockaddr_in *)&ifreq.ifr_broadaddr;

	ip = g_strdup(inet_ntoa(sin->sin_addr));
	
	return ip;
}

gboolean
hit_is_for_me (Hit *h)
{
	static gchar *myip = NULL;
	
	if (myip == NULL) {
		myip = get_ip_of_interface (preferences_get_string (PREFS_FW_EXT_IF));
	}

	return (strcmp (myip, h->destination) == 0);
}

/* [ compare_to_last_hit ]
 * Loosely compare a hit to the previous one.
 * Returns true if they are similiar
 */
gboolean
compare_to_last_hit (Hit *new)
{
	const Hit *old = get_last_hit ();
	gboolean same = TRUE;

	if (old == NULL || new == NULL)
		return FALSE;
	
	/* Going from most likely to differ to least */
	if (strcmp (new->port, old->port) != 0) {
		same = FALSE;
	}
	else if (strcmp (new->protocol, old->protocol) != 0) {
		same = FALSE;
	}
	else if (strcmp (new->source, old->source) != 0) {
		same = FALSE;
	}
	else if (strcmp (new->destination, old->destination) != 0) {
		same = FALSE;
	}
/*	else if (strcmp (new->in, old->in) == 0)
		same = FALSE;
	else if (strcmp (new->out, old->out) == 0)
		same = FALSE;
*/
	return same;
}

/* [ create_devicelist ]
 * Create a list of network interfaces
 */
GList*
create_devicelist (void)
{
	GList *devicelist = NULL;
	FILE *f;
	gchar buf[512];
	gchar *name;

	f = fopen ("/proc/net/dev", "r");
	g_assert (f);

	while (fgets (buf, 512, f) != NULL) {
		if (index(buf, ':')) {
			name = g_strndup (buf, index (buf, ':')-buf);
			g_strstrip (name);
			
			if (strcmp (name, "lo")) /* Remove local loopback device from list */
				devicelist = g_list_append (devicelist, name);
		}
	}
	fclose(f);

	return (devicelist);
}

/* [ is_capable_of_nat ]
 * Return true if we can do nat, port forwarding etc.
 */
gboolean
is_capable_of_nat (void)
{
	GList *devices;
	
	devices = (GList *)create_devicelist ();
	return (g_list_length (devices) >= 2);
}

/* [ get_text_between ]
 * Give a string and two subtext markers in the string and the function 
 * returns the text between the markers. Note: Return empty string if fail.
 */
gchar *
get_text_between (const gchar *string, gchar *marker1, gchar *marker2)
{
	gint i = strlen (marker1);
	gchar *text = NULL;

	marker1 = strstr (string, marker1);
	if (marker1 != NULL) {
		marker1 += i;
		marker2 = strstr (marker1, marker2);
		if (marker2 != NULL)
			text = g_strndup (marker1, marker2-marker1);

	}

	if (text == NULL)
		text = g_strdup ("");

	return text;
}

/* [ lookup_ip ]
 * Resolve an IP address given in dotted-decimal notation into
 * an hostname or vice versa.
 */
gchar *
lookup_ip (gchar *ip)
{
	struct hostent *hostentry = NULL;
	struct in_addr address;

	if (inet_aton (ip, &address)) {
		hostentry = gethostbyaddr ((char *)&address,
					   sizeof (address), AF_INET);
		if (hostentry != NULL)
			return hostentry->h_name;
		else
			return ip;
	} else {
		hostentry = gethostbyname (ip);
		if (hostentry != NULL) {
			memcpy (&address.s_addr, hostentry ->h_addr, hostentry->h_length);

			return inet_ntoa(address);
		}
		else
			return ip;
	}
}

/* [ is_a_valid_port ]
 * Test that port is a valid number
 */
gboolean
is_a_valid_port (const gchar *port)
{
	int port_num;

	/* TODO: range detection */
	port_num = atoi (port);
	return (port_num >= 1 && port_num <= 65535);
}

/* [ is_a_valid_host ]
 * _Very_ loose host string checking
 */
gboolean
is_a_valid_host (const gchar *host)
{
	static GPatternSpec *pattern = NULL;
	gboolean valid = FALSE;
	gint length;

	if (pattern == NULL)
		pattern = g_pattern_spec_new ("*.*");

	length = strlen (host);

	/* Host name must contain a dot and be over 3 but less than 255 chars long */
	if (length > 3 && length < 255)
		if (g_pattern_match_string (pattern, host))
			valid = TRUE;

	return valid;
}

void
free_hit (Hit *h)
{
	if (h == NULL)
		return;

	g_free (h->time);
	g_free (h->in);
	g_free (h->out);
	g_free (h->port);
	g_free (h->source);
	g_free (h->destination);
	g_free (h->length);
	g_free (h->tos);
	g_free (h->protocol);
	g_free (h->service);

	g_free (h);
}
