/*---[ wizard.c ]------------------------------------------------------
 * Copyright (C) 2000-2002 Tomas Junnonen (majix@sci.fi)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.de
 *
 * The wizard functions
 *--------------------------------------------------------------------*/
#include "globals.h"
#include "firestarter.h"
#include "wizard.h"
#include "wizard-choices.h"
#include "scriptwriter.h"
#include "util.h"
#include "xpm/firestarter-pixbufs.h"
#include "preferences.h"

enum
{
	RESPONSE_GO_BACK,
	RESPONSE_GO_FORWARD,
	RESPONSE_QUIT,
	RESPONSE_FINISHED
};

static GList *devices = NULL;

/* [ sensitivity_flip_cb ]
 * Flips the sensitivity of a given widget
 */
static void
sensitivity_flip_cb (GtkWidget *widget)
{
	gtk_widget_set_sensitive (widget, !GTK_WIDGET_IS_SENSITIVE (widget));
}

/* [ create_device_page ]
 * Create the contents of the simple device selection page
 */
GtkWidget*
create_device_page (Wizard *data)
{
	GtkWidget *table;
	GtkWidget *label;
	GtkWidget *hbox;
	GtkWidget *combo;

	table = gtk_table_new (5, 2, FALSE);
	gtk_table_set_row_spacings (GTK_TABLE(table), GNOME_PAD_SMALL);
	gtk_table_set_col_spacings (GTK_TABLE(table), GNOME_PAD_SMALL);

	label = gtk_label_new (_(
		"Please select your Internet connected network device from the drop-down\n"
		"list of detected devices."));
	gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.0);
	gtk_label_set_justify (GTK_LABEL (label), GTK_JUSTIFY_LEFT);
	gtk_table_attach (GTK_TABLE (table), label, 0, 1, 0, 1,
		GTK_FILL, GTK_FILL, GNOME_PAD, GNOME_PAD);

	hbox = gtk_hbox_new (FALSE, GNOME_PAD_SMALL);
	gtk_table_attach (GTK_TABLE (table), hbox, 0, 1, 1, 2,
		GTK_FILL, GTK_FILL, GNOME_PAD, 2);

 	label = gtk_label_new (_("Detected device(s):"));
	gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 5);

	combo = gtk_combo_new();
	gtk_tooltips_set_tip (Firestarter.ttips, GTK_COMBO (combo)->entry, _(
		"Select your network device from the dropdown list. "
		"Go with the default if unsure."), "");
		
	if (devices == NULL)
		devices = create_devicelist ();

	gtk_combo_set_popdown_strings (GTK_COMBO (combo), devices);
	gtk_box_pack_start (GTK_BOX (hbox), combo, FALSE, FALSE, 5);

	data->extdevice = (GTK_COMBO (combo)->entry);

	label = gtk_label_new (NULL);
	gtk_label_set_markup (GTK_LABEL (label), g_strconcat (
		"<small>", _(
		"Tip: If you use a modem the device name is likely ppp0. If you have a cable modem or a\n"
		"DSL connection, choose eth0. Choose ppp0 if your cable or DSL operator uses the PPP\n"
		"over Ethernet protocol."
		), "</small>", NULL));
	gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.0);
	gtk_label_set_justify (GTK_LABEL (label), GTK_JUSTIFY_LEFT);
	gtk_table_attach (GTK_TABLE (table), label, 0, 1, 2, 3,
		GTK_FILL, GTK_FILL, GNOME_PAD, GNOME_PAD);


	data->pppcheck = gtk_check_button_new_with_label (_(
		"Start the firewall on dial-out"));
	gtk_tooltips_set_tip (Firestarter.ttips, data->pppcheck, _(
		"Check this option and the firewall will start when "
		"you dial your Internet Service Provider."), "");
	gtk_table_attach (GTK_TABLE (table), data->pppcheck, 0, 2, 3, 4,
		GTK_FILL, GTK_FILL, GNOME_PAD, 0);

	g_signal_emit_by_name (G_OBJECT (GTK_COMBO (combo)->list),
			       "selection-changed", data->pppcheck);

	data->dhcpcheck = gtk_check_button_new_with_label (_(
		"IP address is assigned via DHCP"));
	gtk_tooltips_set_tip (Firestarter.ttips, data->dhcpcheck, _(
		"Check this option if you need to connect to a DHCP server. "
		"Cable modem and DSL users should check this."), "");
	g_object_set_data (G_OBJECT (data->dhcpcheck), "entry",
			     GTK_COMBO (combo)->entry);
	gtk_table_attach (GTK_TABLE (table), data->dhcpcheck, 0, 2, 4, 5,
		GTK_FILL, GTK_FILL, GNOME_PAD, 0);

	return table;
}

/* [ create_masq_page ]
 * Create the contents of the ipmasq setup page
 */
GtkWidget*
create_masq_page (Wizard *data)
{
	GtkWidget *table;
	GtkWidget *table2;
	GtkWidget *table3;
	GtkWidget *label;
	GtkWidget *nat_disabled;
	GtkWidget *nat_enabled;
	GtkWidget *int_range_auto;
	GtkWidget *int_range_manual;
	GtkWidget *entry;
	GtkWidget *combo;
	GtkWidget *hbox;

	table = gtk_table_new (2, 2, FALSE);
	gtk_table_set_row_spacings (GTK_TABLE (table), GNOME_PAD_SMALL);
	gtk_table_set_col_spacings (GTK_TABLE (table), GNOME_PAD_SMALL);

	label = gtk_label_new (_(
		"Network Address Translation (NAT) is a technique that allows all the machines\n"
		"on your network to share a single IP address and Internet connection."));

	gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.0);
	gtk_label_set_justify (GTK_LABEL (label), GTK_JUSTIFY_LEFT);
	gtk_table_attach (GTK_TABLE (table), label, 0, 2, 0, 1,
		GTK_FILL, GTK_FILL, GNOME_PAD, GNOME_PAD_SMALL);

	nat_disabled = gtk_radio_button_new_with_label (NULL,
		_("Disable Network Address Translation"));
	gtk_table_attach (GTK_TABLE (table), nat_disabled, 0, 2, 1, 2,
		GTK_FILL, GTK_FILL, GNOME_PAD, 0);

	nat_enabled = gtk_radio_button_new_with_label_from_widget (GTK_RADIO_BUTTON (nat_disabled),
		_("Enable Network Address Translation"));
	gtk_table_attach (GTK_TABLE (table), nat_enabled, 0, 2, 2, 3,
		GTK_FILL, GTK_FILL, GNOME_PAD, GNOME_PAD_SMALL);

	data->masq = nat_enabled;

	hbox = gtk_hbox_new (FALSE, GNOME_PAD);
	gtk_widget_set_sensitive (hbox, gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (data->masq)));
	gtk_table_attach (GTK_TABLE (table), hbox, 0, 2, 3, 4,
		GTK_FILL, GTK_FILL, GNOME_PAD, 0);

	g_signal_connect_object (G_OBJECT (nat_enabled), "toggled",
		G_CALLBACK (sensitivity_flip_cb), G_OBJECT (hbox), G_CONNECT_SWAPPED);

	table2 = gtk_table_new (2, 5, FALSE);
	gtk_table_set_row_spacings (GTK_TABLE(table2), GNOME_PAD_SMALL);
	gtk_table_set_col_spacings (GTK_TABLE(table2), GNOME_PAD_SMALL);
	gtk_box_pack_start (GTK_BOX (hbox), table2, FALSE, FALSE, 0);

 	label = gtk_label_new (_("Select your internal network device:"));
	gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.0);
	gtk_label_set_justify (GTK_LABEL (label), GTK_JUSTIFY_LEFT);
	gtk_table_attach (GTK_TABLE (table2), label, 0, 1, 0, 1,
		GTK_FILL, GTK_FILL, GNOME_PAD, 0);

	combo = gtk_combo_new();
	gtk_tooltips_set_tip (Firestarter.ttips, GTK_COMBO (combo)->entry, _(
		"This is the device that connects to your LAN. "
		"Go with the default if unsure."), "");

	if (devices == NULL)
		devices = create_devicelist ();

	gtk_combo_set_popdown_strings (GTK_COMBO (combo), devices);
	gtk_list_select_item (GTK_LIST (GTK_COMBO (combo)->list), 1);
	gtk_table_attach (GTK_TABLE (table2), combo, 1, 2, 0, 1,
		GTK_FILL, GTK_FILL, GNOME_PAD, 0);

	int_range_auto = gtk_radio_button_new_with_label (NULL,
		_("Autodetect internal network IP range"));
	gtk_tooltips_set_tip (Firestarter.ttips, int_range_auto, _(
		"Select this and the IP range of your LAN is autodetected."), "");
	gtk_table_attach (GTK_TABLE (table2), int_range_auto, 0, 2, 1, 2,
		GTK_FILL, GTK_FILL, GNOME_PAD, 0);

	int_range_manual = gtk_radio_button_new_with_label_from_widget (GTK_RADIO_BUTTON (int_range_auto),
		_("Specify internal network IP range manually (advanced users only)"));
	gtk_tooltips_set_tip (Firestarter.ttips, int_range_manual, _(
		"Select this only if the automatic detection fails."), "");
	gtk_table_attach (GTK_TABLE (table2), int_range_manual, 0, 2, 2, 3,
		GTK_FILL, GTK_FILL, GNOME_PAD, 0);

	data->auto_int_ip = int_range_manual;
	
	table3 = gtk_table_new (2, 2, FALSE);
	gtk_table_attach (GTK_TABLE (table2), table3, 0, 2, 3, 4,
		GTK_FILL, GTK_FILL, GNOME_PAD, 0);

	g_signal_connect_object (G_OBJECT (int_range_manual), "toggled",
		G_CALLBACK (sensitivity_flip_cb), G_OBJECT (table3), G_CONNECT_SWAPPED);
 	label = gtk_label_new (_("Enter the internal network address range:"));
	gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.0);
	gtk_label_set_justify (GTK_LABEL (label), GTK_JUSTIFY_LEFT);
	gtk_table_attach (GTK_TABLE (table3), label, 0, 1, 0, 1,
		GTK_FILL, GTK_FILL, GNOME_PAD, 0);

	entry = gtk_entry_new ();

	gtk_table_attach (GTK_TABLE (table3), entry, 1, 2, 0, 1,
		GTK_FILL, GTK_FILL, GNOME_PAD, 0);

 	label = gtk_label_new (NULL);
	gtk_label_set_markup (GTK_LABEL (label), g_strconcat (
		"<small>", _(
		"Tip: The network range is given in the form of <i>address/mask</i>. 192.168.0.0/24 is the\n"
		" most common IP range reserved for internal networks."
		), "</small>", NULL));
	gtk_label_set_justify (GTK_LABEL (label), GTK_JUSTIFY_LEFT);
	gtk_table_attach (GTK_TABLE (table3), label, 0, 2, 1, 2,
		GTK_FILL, GTK_FILL, GNOME_PAD, 0);

	data->intdevice = (GTK_COMBO (combo)->entry);
	data->intrange = entry;

	return table;
}

/* [ create_services_page ]
 * Create the contents of the service selection page
 */
GtkWidget*
create_services_page (Wizard *data)
{
	GtkWidget *table;
	GtkWidget *label;
	GtkWidget *hbox;
	GtkWidget *vbox;
	GtkWidget *services_disabled;
	GtkWidget *sercices_enabled;

	table = gtk_table_new (5, 1, FALSE);
	gtk_table_set_row_spacings (GTK_TABLE(table), GNOME_PAD_SMALL);
	gtk_table_set_col_spacings (GTK_TABLE(table), GNOME_PAD_SMALL);

	label = gtk_label_new (NULL);

	gtk_label_set_markup (GTK_LABEL (label), g_strconcat (_(
		"If you run server software on your machine and want to give public access\n"
		"to the services you provide, you can configure them here.\n\n"
		),"<small>", _(
		"Tip: Services can easily be configured later from the main interface, in the Rules view.\n"
		"The Rules view settings override any conflicting choices made here."
		), "</small>", NULL)
	);
	
	gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.0);
	gtk_label_set_justify (GTK_LABEL (label), GTK_JUSTIFY_LEFT);
	gtk_table_attach (GTK_TABLE (table), label, 0, 1, 0, 1,
		GTK_FILL, GTK_FILL, GNOME_PAD, GNOME_PAD_SMALL);

	services_disabled = gtk_radio_button_new_with_label (NULL,
		_("Disable public access to all services"));
	gtk_table_attach (GTK_TABLE (table), services_disabled, 0, 1, 1, 2,
		GTK_FILL, GTK_FILL, GNOME_PAD, 0);

	sercices_enabled = gtk_radio_button_new_with_label_from_widget (GTK_RADIO_BUTTON (services_disabled),
		_("Enable public access to the following services:"));
	gtk_table_attach (GTK_TABLE (table), sercices_enabled, 0, 1, 2, 3,
		GTK_FILL, GTK_FILL, GNOME_PAD, 0);

	data->services = sercices_enabled;

	hbox = gtk_hbox_new (FALSE, GNOME_PAD_BIG);
	gtk_widget_set_sensitive (hbox, gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (data->services)));
	gtk_table_attach (GTK_TABLE (table), hbox, 0, 1, 3, 4,
		GTK_FILL, GTK_FILL, GNOME_PAD, 0);

	g_signal_connect_object (G_OBJECT (sercices_enabled), "toggled",
		G_CALLBACK (sensitivity_flip_cb), G_OBJECT (hbox), G_CONNECT_SWAPPED);

	/* Column 1 */
	vbox = gtk_vbox_new (FALSE, GNOME_PAD_SMALL);

	data->ftp = gtk_check_button_new_with_label ("FTP");
	gtk_box_pack_start (GTK_BOX (vbox), data->ftp, FALSE, FALSE, 0);

	data->ssh = gtk_check_button_new_with_label ("SSH");
	gtk_box_pack_start (GTK_BOX (vbox), data->ssh, FALSE, FALSE, 0);

	data->telnet = gtk_check_button_new_with_label ("Telnet");
	gtk_box_pack_start (GTK_BOX (vbox), data->telnet, FALSE, FALSE, 0);

	data->smtp = gtk_check_button_new_with_label ("SMTP");
	gtk_box_pack_start (GTK_BOX (vbox), data->smtp, FALSE, FALSE, 0);

	data->dns = gtk_check_button_new_with_label ("DNS");
	gtk_box_pack_start (GTK_BOX (vbox), data->dns, FALSE, FALSE, 0);

	data->finger = gtk_check_button_new_with_label ("Finger");
	gtk_box_pack_start (GTK_BOX (vbox), data->finger, FALSE, FALSE, 0);

	gtk_box_pack_start (GTK_BOX (hbox), vbox, FALSE, FALSE, 0);

	/* Column 2 */
	vbox = gtk_vbox_new (FALSE, GNOME_PAD_SMALL);

	data->www = gtk_check_button_new_with_label ("WWW");
	gtk_box_pack_start (GTK_BOX (vbox), data->www, FALSE, FALSE, 0);

	data->sslweb = gtk_check_button_new_with_label ("SSL web");
	gtk_box_pack_start (GTK_BOX (vbox), data->sslweb, FALSE, FALSE, 0);

	data->pop = gtk_check_button_new_with_label ("POP");
	gtk_box_pack_start (GTK_BOX (vbox), data->pop, FALSE, FALSE, 0);

	data->imap = gtk_check_button_new_with_label ("IMAP");
	gtk_box_pack_start (GTK_BOX (vbox), data->imap, FALSE, FALSE, 0);

	data->ident = gtk_check_button_new_with_label ("IDENT");
	gtk_box_pack_start (GTK_BOX (vbox), data->ident, FALSE, FALSE, 0);

	data->nntp = gtk_check_button_new_with_label ("NNTP");
	gtk_box_pack_start (GTK_BOX (vbox), data->nntp, FALSE, FALSE, 0);

	gtk_box_pack_start (GTK_BOX (hbox), vbox, FALSE, FALSE, 0);

	/* Column 3 */
	vbox = gtk_vbox_new (FALSE, GNOME_PAD_SMALL);

	data->ntp = gtk_check_button_new_with_label ("NTP");
	gtk_box_pack_start (GTK_BOX (vbox), data->ntp, FALSE, FALSE, 0);

	data->samba = gtk_check_button_new_with_label ("SAMBA/NetBios");
	gtk_box_pack_start (GTK_BOX (vbox), data->samba, FALSE, FALSE, 0);

	data->routed = gtk_check_button_new_with_label ("Routed");
	gtk_box_pack_start (GTK_BOX (vbox), data->routed, FALSE, FALSE, 0);

	data->nfs = gtk_check_button_new_with_label ("NFS");
	gtk_box_pack_start (GTK_BOX (vbox), data->nfs, FALSE, FALSE, 0);

	data->x = gtk_check_button_new_with_label ("Xwindows");
	gtk_box_pack_start (GTK_BOX (vbox), data->x, FALSE, FALSE, 0);

	data->dhcp = gtk_check_button_new_with_label ("DHCP");
	gtk_box_pack_start (GTK_BOX (vbox), data->dhcp, FALSE, FALSE, 0);

	gtk_box_pack_start (GTK_BOX (hbox), vbox, FALSE, FALSE, 0);
	
	/* Column 4 */
	vbox = gtk_vbox_new (FALSE, GNOME_PAD_SMALL);

	data->ipsec = gtk_check_button_new_with_label ("IPSEC");
	gtk_box_pack_start (GTK_BOX (vbox), data->ipsec, FALSE, FALSE, 0);

	data->upnp = gtk_check_button_new_with_label ("uPNP");
	gtk_box_pack_start (GTK_BOX (vbox), data->upnp, FALSE, FALSE, 0);

  	data->pptp = gtk_check_button_new_with_label ("PPTP");
	gtk_box_pack_start (GTK_BOX (vbox), data->pptp, FALSE, FALSE, 0);

	gtk_box_pack_start (GTK_BOX (hbox), vbox, FALSE, FALSE, 0);

	return table;
}

/* [ create_icmp_page ]
 * Create the contents of the ICMP filtering page - fnk 10/2000
 */
GtkWidget*
create_icmp_page (Wizard *data)
{
	GtkWidget *table;
	GtkWidget *label;
	GtkWidget *hbox;
	GtkWidget *vbox;
	GtkWidget *icmp_disabled;
	GtkWidget *icmp_enabled;

	table = gtk_table_new (4, 1, FALSE);
	gtk_table_set_row_spacings (GTK_TABLE(table), GNOME_PAD_SMALL);
	gtk_table_set_col_spacings (GTK_TABLE(table), GNOME_PAD_SMALL);

	label = gtk_label_new (_(
		"ICMP packet filtering can be useful to prevent some common\n"
		"Denial of Service (DoS) attacks on your network."));
	gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.0);
	gtk_label_set_justify (GTK_LABEL (label), GTK_JUSTIFY_LEFT);
	gtk_table_attach (GTK_TABLE (table), label, 0, 1, 0, 1,
		GTK_FILL, GTK_FILL, GNOME_PAD, GNOME_PAD_SMALL);

	icmp_disabled = gtk_radio_button_new_with_label (NULL,
		_("Disable ICMP filtering"));
	gtk_table_attach (GTK_TABLE (table), icmp_disabled, 0, 1, 1, 2,
		GTK_FILL, GTK_FILL, GNOME_PAD, 0);

	icmp_enabled = gtk_radio_button_new_with_label_from_widget (GTK_RADIO_BUTTON (icmp_disabled),
		_("Enable ICMP filtering related to the following packets:"));
	gtk_table_attach (GTK_TABLE (table), icmp_enabled, 0, 1, 2, 3,
		GTK_FILL, GTK_FILL, GNOME_PAD, 0);

	data->icmp = icmp_enabled;

	hbox = gtk_hbox_new (FALSE, GNOME_PAD_BIG);
	gtk_widget_set_sensitive (hbox, gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (data->icmp)));
	gtk_table_attach (GTK_TABLE (table), hbox, 0, 1, 3, 4,
		GTK_FILL, GTK_FILL, GNOME_PAD, 0);

	g_signal_connect_object (G_OBJECT (icmp_enabled), "toggled",
		G_CALLBACK (sensitivity_flip_cb), G_OBJECT (hbox), G_CONNECT_SWAPPED);

	/* Column 1 */
	vbox = gtk_vbox_new (FALSE, GNOME_PAD_SMALL);

	data->icmp_echo = gtk_check_button_new_with_label (_("Echo"));
	gtk_box_pack_start (GTK_BOX (vbox), data->icmp_echo, FALSE, FALSE, 0);

	data->icmp_traceroute = gtk_check_button_new_with_label (_("Traceroute"));
	gtk_box_pack_start (GTK_BOX (vbox), data->icmp_traceroute, FALSE, FALSE, 0);

	data->icmp_tracert = gtk_check_button_new_with_label (_("MS Traceroute"));
	gtk_box_pack_start (GTK_BOX (vbox), data->icmp_tracert, FALSE, FALSE, 0);

	data->icmp_unreach = gtk_check_button_new_with_label (_("Unreachable"));
	gtk_box_pack_start (GTK_BOX (vbox), data->icmp_unreach, FALSE, FALSE, 0);

	gtk_box_pack_start (GTK_BOX (hbox), vbox, FALSE, FALSE, 0);

	/* Column 2 */
	vbox = gtk_vbox_new (FALSE, GNOME_PAD_SMALL);

	data->icmp_timestamp = gtk_check_button_new_with_label (_("Timestamping"));
	gtk_box_pack_start (GTK_BOX (vbox), data->icmp_timestamp, FALSE, FALSE, 0);

	data->icmp_masking = gtk_check_button_new_with_label (_("Address Masking"));
	gtk_box_pack_start (GTK_BOX (vbox), data->icmp_masking, FALSE, FALSE, 0);

	data->icmp_redir = gtk_check_button_new_with_label (_("Redirection"));
	gtk_box_pack_start (GTK_BOX (vbox), data->icmp_redir, FALSE, FALSE, 0);

	data->icmp_quench = gtk_check_button_new_with_label (_("Source Quenches"));
	gtk_box_pack_start (GTK_BOX (vbox), data->icmp_quench, FALSE, FALSE, 0);

	gtk_box_pack_start (GTK_BOX (hbox), vbox, FALSE, FALSE, 0);

	return table;
}

/* [ create_tos_page ]
 * Create the contents of the Type of Service filtering page - fnk 12/2000
 */
GtkWidget*
create_tos_page (Wizard *data)
{
	GtkWidget *table;
	GtkWidget *label;
	GtkWidget *hbox;
	GtkWidget *vbox;
	GtkWidget *tos_disabled;
	GtkWidget *tos_enabled;

	table = gtk_table_new (4, 1, FALSE);
	gtk_table_set_row_spacings (GTK_TABLE(table), GNOME_PAD_SMALL);
	gtk_table_set_col_spacings (GTK_TABLE(table), GNOME_PAD_SMALL);

	label = gtk_label_new (_(
		"Type of Service filtering allows you to re-prioritize network services\n"
		"to allow higher throughput rates for commonly used services."));
	gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.0);
	gtk_label_set_justify (GTK_LABEL (label), GTK_JUSTIFY_LEFT);
	gtk_table_attach (GTK_TABLE (table), label, 0, 1, 0, 1,
		GTK_FILL, GTK_FILL, GNOME_PAD, GNOME_PAD_SMALL);

	tos_disabled = gtk_radio_button_new_with_label (NULL,
		_("Disable ToS filtering"));
	gtk_table_attach (GTK_TABLE (table), tos_disabled, 0, 1, 1, 2,
		GTK_FILL, GTK_FILL, GNOME_PAD, 0);

	tos_enabled = gtk_radio_button_new_with_label_from_widget (GTK_RADIO_BUTTON (tos_disabled),
		_("Enable ToS filtering related to the following packets:"));
	gtk_table_attach (GTK_TABLE (table), tos_enabled, 0, 1, 2, 3,
		GTK_FILL, GTK_FILL, GNOME_PAD, 0);

	data->tos = tos_enabled;
	
	hbox = gtk_hbox_new (FALSE, GNOME_PAD_BIG);
	gtk_widget_set_sensitive (hbox, gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (data->tos)));
	gtk_table_attach (GTK_TABLE (table), hbox, 0, 1, 3, 4,
		GTK_FILL, GTK_FILL, GNOME_PAD, 0);

	g_signal_connect_object (G_OBJECT (tos_enabled), "toggled",
		G_CALLBACK (sensitivity_flip_cb), G_OBJECT (hbox), G_CONNECT_SWAPPED);

	/* Column 1 */
	vbox = gtk_vbox_new (FALSE, GNOME_PAD_SMALL);

	data->tos_client = gtk_check_button_new_with_label (_("Client Applications"));
	gtk_box_pack_start (GTK_BOX (vbox), data->tos_client, FALSE, FALSE, 0);

	gtk_tooltips_set_tip (Firestarter.ttips, data->tos_client, _(
		"Check this option to enable Type of Service flags for common "
		"client applications, such as WWW, FTP, SSH & E-Mail."), "");

	data->tos_server = gtk_check_button_new_with_label (_("Server Applications"));
	gtk_box_pack_start (GTK_BOX (vbox), data->tos_server, FALSE, FALSE, 0);

	gtk_tooltips_set_tip (Firestarter.ttips, data->tos_server, _(
		"Check this option to enable Type of Service flags for common "
		"server applications, such as SQUID, FTPd, SSHd, SMTP & POP Daemons."), "");

	data->tos_X = gtk_check_button_new_with_label (_("The X Window System"));
	gtk_box_pack_start (GTK_BOX (vbox), data->tos_X, FALSE, FALSE, 0);

	gtk_tooltips_set_tip (Firestarter.ttips, data->tos_X, _(
		"Selecting this option will automatically configure Type of Service "
		"for `Throughput' for both X and SSHd. "
		"You should select this ONLY if you need to run X over your link"), "");

	gtk_box_pack_start (GTK_BOX (hbox), vbox, FALSE, FALSE, 0);

	/* Column 2 */
	vbox = gtk_vbox_new (FALSE, GNOME_PAD_SMALL);

	data->tos_option_throughput = gtk_radio_button_new_with_label (NULL, _("Throughput"));
	gtk_box_pack_start (GTK_BOX (vbox), data->tos_option_throughput, FALSE, FALSE, 0);

	gtk_tooltips_set_tip (Firestarter.ttips, data->tos_option_throughput, _(
		"Selecting this option will configure Type of Service flags for "
		"Maximum Throughput for the options you have selected."), "");

	data->tos_option_reliability = gtk_radio_button_new_with_label_from_widget (
		GTK_RADIO_BUTTON (data->tos_option_throughput), _("Reliability"));
	gtk_box_pack_start (GTK_BOX (vbox), data->tos_option_reliability, FALSE, FALSE, 0);

	gtk_tooltips_set_tip (Firestarter.ttips, data->tos_option_reliability, _(
		"Selecting this option will configure Type of Service flags for "
		"Maximum Reliability for the options you have selected."), "");

	data->tos_option_delay = gtk_radio_button_new_with_label_from_widget (
		GTK_RADIO_BUTTON (data->tos_option_throughput), _("Delay"));
	gtk_box_pack_start (GTK_BOX (vbox), data->tos_option_delay, FALSE, FALSE, 0);

	gtk_tooltips_set_tip (Firestarter.ttips, data->tos_option_delay, _(
		"Selecting this option will configure Type of Service flags for "
		"Minimum Delay for the options you have selected."), "");

	gtk_box_pack_start (GTK_BOX (hbox), vbox, FALSE, FALSE, 0);
     
	return table;
}

/* [ create_welcome_page ]
 * Create the contents of the Type of Service filtering page - fnk 12/2000
 */
static GtkWidget*
create_welcome_page (Wizard *data)
{
	GtkWidget *label;
	GtkWidget *image;
	GtkWidget *hbox;
	GtkWidget *vbox;
	GdkPixbuf *pixbuf;

	hbox = gtk_hbox_new (FALSE, GNOME_PAD_BIG);

 	pixbuf = gdk_pixbuf_new_from_inline (-1, pengo, FALSE, NULL);
	image = gtk_image_new_from_pixbuf (pixbuf);
	g_object_unref (G_OBJECT(pixbuf));
	gtk_box_pack_start (GTK_BOX(hbox), image, FALSE, FALSE, 0);

	vbox = gtk_vbox_new (FALSE, GNOME_PAD_SMALL);
	gtk_box_pack_start (GTK_BOX(hbox), vbox, FALSE, FALSE, 0);


	label = gtk_label_new (NULL);
	gtk_label_set_markup (GTK_LABEL (label), g_strconcat (_(
		"This wizard will help you to set up a firewall for your\n"
		"Linux machine. You will be asked some questions\n"
		"about your network setup in order to customize the\n"
		"firewall for your system.\n\n"),
		"<small>", _(
		"Tip: If you are uncertain of how to answer a question it is\n"
		"best to use the default value supplied.\n\n"
		), "</small>",
		_("Please press the forward button to continue.\n"), NULL)
	);
	gtk_box_pack_start (GTK_BOX(vbox), label, FALSE, FALSE, 0);


	return hbox;
}

/* [ create_page ]
 * Create the default wizard page layout container
 */
static GtkWidget *
create_page (const char *title)
{
	GtkWidget *label;
	GtkWidget *vbox;
	GtkWidget *hbox;
	GtkWidget *alignment;
	GtkWidget *image;
	GdkPixbuf *pixbuf;
	gchar *title_string;

	/* create vbox */
	vbox = gtk_vbox_new (FALSE, GNOME_PAD);

	/* create the titlebar */
	hbox = gtk_hbox_new (FALSE, GNOME_PAD_BIG);
	pixbuf = gdk_pixbuf_new_from_inline (-1, logo, FALSE, NULL);
	image = gtk_image_new_from_pixbuf (pixbuf);
	g_object_unref (G_OBJECT(pixbuf));
	gtk_box_pack_start (GTK_BOX(hbox), image, FALSE, FALSE, 0);

	alignment = gtk_alignment_new (1.0, 1.0, 1.0, 1.0);
	gtk_box_pack_start (GTK_BOX(hbox), alignment, TRUE, TRUE, 0);

	title_string = g_strconcat ("<span size=\"xx-large\" weight=\"ultrabold\">", title ? title : "", "</span>", NULL);
	label = gtk_label_new (title_string);
	gtk_label_set_use_markup (GTK_LABEL(label), TRUE);
	g_free (title_string);
	gtk_container_add (GTK_CONTAINER(alignment), label);
	//gtk_box_pack_start (GTK_BOX(hbox), label, FALSE, FALSE, 0);

	/* pack the titlebar */
	gtk_box_pack_start (GTK_BOX(vbox), hbox, FALSE, FALSE, 0);

	/* pack the separator */
	gtk_box_pack_start (GTK_BOX(vbox), gtk_hseparator_new(), FALSE, FALSE, 0);

	return vbox;
}

/* [ create_page_with_text ]
 * Create a simple page with text only
 */
static GtkWidget *
create_page_with_text (const char *title, const char *text)
{
	GtkWidget *label;
	GtkWidget *page;
       
	page = create_page (title);
	label = gtk_label_new (text);
	gtk_box_pack_start (GTK_BOX(page), label, TRUE, TRUE, 0);

	return page;
}

/* [ dialog_response_cb ]
 * Controls what happens when one of the wizard buttons are pressed
 * Possible page flow control code would go here
 */
static void
dialog_response_cb (GtkDialog *dialog, int response, gpointer data)
{
	Wizard *wizard = (Wizard *)data;
	const int page_qty = wizard->pages->len;
	int current_page = gtk_notebook_get_current_page (GTK_NOTEBOOK(wizard->notebook));

	if (response == RESPONSE_QUIT) {
		if (preferences_get_bool (PREFS_FIRST_RUN) || !script_exists ())
			exit_firestarter ();

		g_free (wizard);
		gtk_widget_destroy (GTK_WIDGET(dialog));
	} else if (response == RESPONSE_FINISHED) {
		FirewallStatus state;

		save_choices (wizard);
		write_script ();

		state = get_current_status ();

		/* Always start firewall on the first run, otherwise only if the firewall
		   has already been started */
		if (preferences_get_bool (PREFS_FIRST_RUN) ||
		    !script_exists () ||
		    state == STATUS_RUNNING ||
		    state == STATUS_HIT)
			start_firewall ();

		g_free (wizard);
		gtk_widget_destroy (GTK_WIDGET(dialog));
	} else {
		if (response == RESPONSE_GO_BACK) {
			if (current_page > 0)
				gtk_notebook_set_current_page (GTK_NOTEBOOK(wizard->notebook), --current_page);
		} else if (response == RESPONSE_GO_FORWARD) {
			if (current_page < (page_qty-1))
				gtk_notebook_set_current_page (GTK_NOTEBOOK(wizard->notebook), ++current_page);
		}

		gtk_dialog_set_response_sensitive (dialog, RESPONSE_GO_BACK, current_page > 0);
		gtk_dialog_set_response_sensitive (dialog, RESPONSE_GO_FORWARD, current_page < (page_qty-1));
		gtk_dialog_set_response_sensitive (dialog, RESPONSE_FINISHED, current_page == (page_qty-1));
	}
}

/* [ devicelist_contains_device ]
 * Return true if a specified device is present
 */
static gboolean
devicelist_contains_device (gchar *device)
{
	GList *e;

	for (e = g_list_first (devices); e != NULL; e = g_list_next (e)) {
		if (strcmp (e->data, device) == 0)
			return TRUE;
	}
	
	return FALSE;
}

/* [ run_wizard ]
 * Run the firewall wizard
 */
void
run_wizard (void)
{
	Wizard *wizard;
	GtkWidget *dialog;
	GtkWidget *notebook;
	GtkWidget *page;
	GtkWidget *alignment;
	gint i;

	wizard = g_new (Wizard, 1);
	
	dialog = gtk_dialog_new_with_buttons (_("Firewall Wizard"),
	                                      NULL, 0,
	                                      GTK_STOCK_GO_BACK, RESPONSE_GO_BACK,
	                                      GTK_STOCK_GO_FORWARD, RESPONSE_GO_FORWARD,
	                                      GTK_STOCK_SAVE, RESPONSE_FINISHED,
	                                      GTK_STOCK_QUIT, RESPONSE_QUIT,
	                                      NULL);

	g_signal_connect (dialog, "response",
	                  G_CALLBACK(dialog_response_cb), wizard);

	gtk_dialog_set_default_response   (GTK_DIALOG(dialog), RESPONSE_GO_FORWARD);
	gtk_dialog_set_response_sensitive (GTK_DIALOG(dialog), RESPONSE_GO_BACK, FALSE);
	gtk_dialog_set_response_sensitive (GTK_DIALOG(dialog), RESPONSE_GO_FORWARD, TRUE);
	gtk_dialog_set_response_sensitive (GTK_DIALOG(dialog), RESPONSE_FINISHED, FALSE);

	/* The wizard is a notebook widget without tabs */
	notebook = wizard->notebook = gtk_notebook_new ();
	gtk_notebook_set_show_tabs (GTK_NOTEBOOK(notebook), FALSE);
	gtk_notebook_set_show_border (GTK_NOTEBOOK(notebook), FALSE);
	gtk_box_pack_start (GTK_BOX(GTK_DIALOG(dialog)->vbox), notebook, TRUE, TRUE, 0);

	devices = create_devicelist ();

	/* Create the basic wizard pages */
	wizard->pages = g_ptr_array_new ();

	page = create_page (_("Welcome to Firestarter"));
	alignment = gtk_alignment_new (0.5, 0.5, 0.5, 0.5);
	gtk_box_pack_start (GTK_BOX(page), alignment, TRUE, TRUE, 0);
	gtk_container_add (GTK_CONTAINER(alignment), create_welcome_page (wizard));
	g_ptr_array_add (wizard->pages, page);

	page = create_page (_("Network device setup"));
	alignment = gtk_alignment_new (0.5, 0.5, 0.5, 0.5);
	gtk_box_pack_start (GTK_BOX(page), alignment, TRUE, TRUE, 0);
	gtk_container_add (GTK_CONTAINER(alignment), create_device_page (wizard));
	g_ptr_array_add (wizard->pages, page);

	page = create_page (_("Internet connection sharing setup"));
	alignment = gtk_alignment_new (0.5, 0.5, 0.5, 0.5);
	gtk_box_pack_start (GTK_BOX(page), alignment, TRUE, TRUE, 0);
	gtk_container_add (GTK_CONTAINER(alignment), create_masq_page (wizard));

	/* Don't show the NAT page if the hardware doesn't support it */
	if (is_capable_of_nat ()) {
		g_ptr_array_add (wizard->pages, page);
	}

	page = create_page (_("Network services setup"));
	alignment = gtk_alignment_new (0.5, 0.5, 0.5, 0.5);
	gtk_box_pack_start (GTK_BOX(page), alignment, TRUE, TRUE, 0);
	gtk_container_add (GTK_CONTAINER(alignment), create_services_page (wizard));
	g_ptr_array_add (wizard->pages, page);

/* FIXME: ToS and ICMP filtering are no longer part of the wizard
   Maybe the code should be moved to the preferences module? */

	page = create_page (_("Type of Service filtering setup"));
	alignment = gtk_alignment_new (0.5, 0.5, 0.5, 0.5);
	gtk_box_pack_start (GTK_BOX(page), alignment, TRUE, TRUE, 0);
	gtk_container_add (GTK_CONTAINER(alignment), create_tos_page (wizard));
/*	g_ptr_array_add (wizard->pages, page); */

	page = create_page (_("ICMP filtering setup"));
	alignment = gtk_alignment_new (0.5, 0.5, 0.5, 0.5);
	gtk_box_pack_start (GTK_BOX(page), alignment, TRUE, TRUE, 0);
	gtk_container_add (GTK_CONTAINER(alignment), create_icmp_page (wizard));
/*	g_ptr_array_add (wizard->pages, page); */

	/* Final page */
	page = create_page_with_text (
		_("Ready to start your firewall"),
		_("The wizard is now ready to start your firewall.\n\n"
		  "Press the save button to continue, or the back button\n"
		  "to review your choices.\n\n"));	
	g_ptr_array_add (wizard->pages, page);	

	/* Load pages into notebook */
	for (i = 0; i < wizard->pages->len; i++) {
		gtk_notebook_append_page (GTK_NOTEBOOK(notebook),
		                          GTK_WIDGET(g_ptr_array_index (wizard->pages,i)),
		                          NULL);
	}

	/* Set some simple defaults when run for the first time, otherwise load previous choices */
	if (preferences_get_bool (PREFS_FIRST_RUN)) {
		if (devicelist_contains_device ("eth0"))
			gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (wizard->dhcpcheck), TRUE);
			
		if (devicelist_contains_device ("ppp0"))
			gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (wizard->pppcheck), TRUE);
	} else {
		load_choices (wizard);
	}
	
	gtk_widget_show_all (dialog);
}
