/* -*- Mode: C; indent-tabs-mode: t; c-basic-offset: 4; tab-width: 4 -*- */
/*
 * fische-3.0
 * Copyright (C) Marcel Ebmer 2009 <marcel@26elf.at>
 * 
 * fische-3.0 is free software: you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 * 
 * fische-3.0 is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License along
 * with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#ifdef WIN32
	// please note that fische will not compile or run on windows
	// this is for development reasons only
	#include <SDL_main.h>
#endif // WIN32

#include <iostream>
#include <cstdlib>
#include <unistd.h>
#include <pthread.h>
#include <stdint.h>
#include "sdlscreen.h"
#include "vectorfield.h"
#include "ringbuffer.h"
#include "blurengine.h"
#include "alsa-samples.h"
#include "wavepainter.h"
#include "analyst.h"
#include "configuration.h"

using namespace std;

int main(int argc, char** argv)
{
	bool paused = false;

	// initialize random number generator
	srand(time(NULL));

	// read options from commandline (or file - TODO)
	Configuration* configuration = new Configuration(argc, argv);

	// create ringbuffer
	RingBuffer* ringbuffer = new RingBuffer(176400 / configuration->fpsTarget());

	// create sampling thread
	static bool run = true;
	pthread_t samplingThread;
	sample_param_t p;
	p.device = configuration->device();
	p.ringbuffer = ringbuffer;
	p.run = &run;
	pthread_create(&samplingThread, NULL, sample_alsa_pcm, (void*)&p);
	
	// create components
	VectorField* vectorfield = new VectorField(configuration->X(), configuration->Y());
	SdlScreen* sdlscreen = new SdlScreen(configuration->screenX(), configuration->screenY(), configuration->fullscreen());
	if((vectorfield->width() > sdlscreen->width()) || (vectorfield->height() > sdlscreen->height()))
	{
		cerr << "ERROR: animation is bigger than screen can allow." << endl;
		exit(EXIT_FAILURE);
	}
	BlurEngine* blurengine = new BlurEngine(sdlscreen, vectorfield, configuration->multiThreaded());
	WavePainter* wavepainter = new WavePainter(sdlscreen, vectorfield, ringbuffer);
	Analyst* analyst = new Analyst(ringbuffer);

	int events;
	int analysis;
	// main loop
	while(!((events = sdlscreen->checkEvents()) & EXIT))
	{
		// check for interesting events
		if(events & TOGGLE_FULLSCREEN) sdlscreen->toggleFullscreen();
		if(events & PAUSE) paused = paused ? false : true;

		// paused? do nothing.
		if(paused)
		{
			usleep(10000);
			continue;
		}

		// wait until ringbuffer is full
		int sleepcount = 0;
		while(!ringbuffer->ready())
		{
			usleep(1000);
			sleepcount ++;
			if(sleepcount > 100) break;
		}

		// ringbuffer not filling? in the meantime, check for new events.
		if(sleepcount > 100) continue;

		// analyse sound data
		analysis = analyst->analyse();
		if(analysis >= 1) wavepainter->changeShape();
		if(analysis >= 2) vectorfield->change();
		if(analysis >= 3) wavepainter->changeColor();

		// have fun
		wavepainter->paint();
		sdlscreen->update();
		blurengine->blur();
	}

	// tell the sound sampling thread to quit
	run = false;
	pthread_join(samplingThread, NULL);

	delete analyst;
	delete wavepainter;
	delete blurengine;
	delete sdlscreen;
	delete vectorfield;
	delete ringbuffer;
	delete configuration;

	exit(EXIT_SUCCESS);
}
