/* -*- Mode: C; indent-tabs-mode: t; c-basic-offset: 4; tab-width: 4 -*- */
/*
 * fische-3.0
 * Copyright (C) Marcel Ebmer 2009 <marcel@26elf.at>
 * 
 * fische-3.0 is free software: you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 * 
 * fische-3.0 is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License along
 * with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <iostream>
#include <stdint.h>
#include <cstdlib>
#include <cmath>
#include "sdlscreen.h"
#include "vectorfield.h"
#include "ringbuffer.h"
#include "wavepainter.h"

using namespace std;

WavePainter::WavePainter(SdlScreen* s, VectorField* v, RingBuffer* r)
{
	height = v->height();
	width = v->width();
	y0 = (s->height() - v->height()) / 2;
	x0 = (s->width() - v->width()) / 2;
	ringbuffer = r;
	sdlscreen = s;
	xCenter = x0 + width / 2;
	yCenter = y0 + height / 2;
	shape = 0;
	shapes = 2;
	color1 = rand() % 0xffffffff;
	color2 = ~color1;
}

WavePainter::~WavePainter()
{
}

void WavePainter::line(int x1, int y1, int x2, int y2, uint32_t color)
{
	int Dx = abs(x1 - x2);
	int Dy = abs(y1 - y2);
	int dx = 1;
	if(x2 < x1) dx = -1;
	int dy = 1;
	if(y2 < y1) dy = -1;
	uint32_t* pixels = (uint32_t*)sdlscreen->pixels();
	int pitch = sdlscreen->width();

	if((Dx == 0) && (Dy == 0)) return;
	
	if(Dx > Dy)
	{
		for(int x = x1; x * dx <= x2 * dx; x += dx)
		{
			int y = (int)((double)y1 + (double)Dy / (double)Dx * (double)dy * (double)abs(x - x1) + 0.5);
			*(pixels + y * pitch + x) = color;
		}
	}
	else
	{
		for(int y = y1; y * dy <= y2 * dy; y += dy)
		{
			int x = (int)((double)x1 + (double)Dx / (double)Dy * (double)dx * (double)abs(y - y1) + 0.5);
			*(pixels + y * pitch + x) = color;
		}
	}
}

void WavePainter::paint()
{
	int16_t* samples = (int16_t*)ringbuffer->get();
	int l = ringbuffer->size() / 4;
	int div = 196605 / height;
	double Pi = acos(-1);
	
	switch(shape)
	{
		case 0:
			for(int i = 0; i < l - 1; i ++)
			{
				int x1 = x0 + (width * i) / l;
				int x2 = x0 + (width * (i + 1)) / l;
				int y1 = yCenter - height / 6 - *(samples + 2 * i) / div;
				int y2 = yCenter - height / 6 - *(samples + 2 * (i + 1)) / div;
				line(x1, y1, x2, y2, color1);
				y1 = yCenter + height / 6 - *(samples + 1 + 2 * i) / div;
				y2 = yCenter + height / 6 - *(samples + 1 + 2 * (i + 1)) / div;
				line(x1, y1, x2, y2, color2);
			}
			return;
		case 1:
			for(int i = 0; i < l - 1; i ++)
			{
				double phi1 = Pi * (0.25 + (double)i / (double)l);
				double phi2 = phi1 + Pi / (double)l;
				double r1 = height / 4 + *(samples + 2 * i) / div;
				double r2 = height / 4 + *(samples + 2 * (i + 1)) / div;
				int x1 = iround(xCenter + r1 * sin(phi1));
				int x2 = iround(xCenter + r2 * sin(phi2));
				int y1 = iround(yCenter + r1 * cos(phi1));
				int y2 = iround(yCenter + r2 * cos(phi2));
				line(x1, y1, x2, y2, color1);
				phi1 += Pi;
				phi2 += Pi;
				r1 = height / 4 + *(samples + 1 + 2 * i) / div;
				r2 = height / 4 + *(samples + 1 + 2 * (i + 1)) / div;
				x1 = iround(xCenter + r1 * sin(phi1));
				x2 = iround(xCenter + r2 * sin(phi2));
				y1 = iround(yCenter + r1 * cos(phi1));
				y2 = iround(yCenter + r2 * cos(phi2));
				line(x1, y1, x2, y2, color2);
			}
			return;
	}
}

void WavePainter::changeShape()
{
	int n = rand() % shapes;
	while(n == shape) n = rand() % shapes;
	shape = n;
}

void WavePainter::changeColor()
{
	color1 = rand() % 0xffffffff;
	color2 = ~color1;
}

int iround(double v)
{
	return (int)(v + 0.5);
}
