/** \file builtin.c
	Functions for executing builtin functions.
*/
#include <stdlib.h>
#include <stdio.h>
#include <wchar.h>
#include <unistd.h>
#include <termios.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <dirent.h>
#include <string.h>
#include <signal.h>
#include <wctype.h>
#include <sys/time.h>

#include "config.h"
#include "util.h"
#include "wutil.h"
#include "builtin.h"
#include "function.h"
#include "complete.h"
#include "proc.h"
#include "parser.h"
#include "reader.h"
#include "env.h"
#include "expand.h"
#include "common.h"
#include "wgetopt.h"
#include "sanity.h"
#include "tokenizer.h"
#include "builtin_help.h"
#include "wildcard.h"

/**
   The default prompt for the read command
*/
#define DEFAULT_READ_PROMPT L"set_color green; echo read; set_color normal; echo \"> \""

/**
   The mode name to pass to history and input
*/

#define READ_MODE_NAME L"fish_read"
/**
   Table of all builtins
*/
static hash_table_t builtin;

int builtin_out_redirect;
int builtin_err_redirect;

/**
   Wrapper for wcsfilecmp providing the correct signature for qsort
*/
static int mycmp( const void *a, const void *b )
{
	return wcsfilecmp( *(wchar_t **)a, *(wchar_t **)b );
}

/** 
	Buffers for storing the output of builtin functions
*/
string_buffer_t *sb_out=0, *sb_err=0;
/**
   Stack containing builtin I/O for recursive builtin calls.
*/
static array_list_t io_stack;

/** 
	The file from which builtin functions should attempt to read, use
	instead of stdin.
*/
static FILE *builtin_stdin;

/**
   Table containing descriptions for all builtins
*/
static hash_table_t *desc=0;

/**
   Counts the number of non null pointers in the specified array
*/
static int count_args( wchar_t **argv )
{
	int argc = 1;
	while( argv[argc] != 0 )
	{
		argc++;
	}
	return argc;
}

/** 
	This function works like wperror, but it prints its result into
	the sb_err string_buffer_t instead of to stderr. Used by the builtin
	commands.
*/
static void builtin_wperror( const wchar_t *s)
{
	if( s != 0 )
	{
		sb_append2( sb_err, s, L": ", 0 );
	}
	char *err = strerror( errno );
	wchar_t *werr = str2wcs( err );
	if( werr )
	{
		sb_append2( sb_err,  werr, L"\n", 0 );	
		free( werr );
	}	
}


/*
  Here follows the definition of all builtin commands. The function
  names are all on the form builtin_NAME where NAME is the name of the
  builtin. so the function name for the builtin 'jobs' is
  'builtin_jobs'.

  Two builtins, 'command' and 'builtin' are not defined here as they
  are part of the parser. (They are not parsed as commands, instead
  they only slightly alter the parser state)

*/


/**
   Noop function. A fake function which successfully does nothing, for
   builtins which are handled by the parser, such as command and
   while.
*/
static int builtin_ignore( wchar_t **argv )
{
	return 0;
}

/**
   Print help for the specified builtin. If \c b is sb_err, also print the line information
*/
static void builtin_print_help( wchar_t *cmd, string_buffer_t *b )
{
	const char *h;

	if( b == sb_err )
	{
		sb_append( sb_err, 
				   parser_current_line() );
	}

	h = builtin_help_get( cmd );

	if( !h )
		return;

	
	
	wchar_t *str = str2wcs(builtin_help_get( cmd ));
	if( str )
	{
		sb_append( b, str );
		free( str );
	}
}

static int builtin_builtin(  wchar_t **argv )
{
	int argc=count_args( argv );
	int list=0;
	
	woptind=0;

	const static struct woption
		long_options[] =
		{
			{
				L"names", no_argument, 0, 'n' 
			}
			,
			{ 
				0, 0, 0, 0 
			}
		}
	;		
		
	while( 1 )
	{
		int opt_index = 0;
		
		int opt = wgetopt_long( argc,
								argv, 
								L"n", 
								long_options, 
								&opt_index );
		if( opt == -1 )
			break;
			
		switch( opt )
		{
			case 0:
				if(long_options[opt_index].flag != 0)
					break;
				sb_append2( sb_err,
							L"functions: Unknown option ",
							long_options[opt_index].name,
							L"\n",
							0);				
				builtin_print_help( argv[0], sb_err );
	
				
				return 1;
				
			case 'n':
				list=1;
				break;
				
			case '?':
				builtin_print_help( argv[0], sb_err );
				
				return 1;
				
		}
		
	}		

	if( list )
	{
		array_list_t names;
		wchar_t **names_arr;
		int i;
		
		al_init( &names );
		builtin_get_names( &names );
		names_arr = list_to_char_arr( &names );
		qsort( names_arr, 
			   al_get_count( &names ), 
			   sizeof(wchar_t *), 
			   &mycmp );
		for( i=0; i<al_get_count( &names ); i++ )
		{
			if( wcscmp( names_arr[i], L"count" ) == 0 )
				continue;
			
			sb_append2( sb_out,
						names_arr[i],
						L"\n",
						0 );			
		}
		free( names_arr );
		al_destroy( &names );			
		return 0;
	}	

}


static int builtin_functions( wchar_t **argv )
{
	int i;
	int erase=0;
	wchar_t *desc=0;	

	array_list_t names;
	wchar_t **names_arr;

	int argc=count_args( argv );
	int list=0;
	
	woptind=0;

	const static struct woption
		long_options[] =
		{
			{
				L"erase", no_argument, 0, 'e' 
			}
			,
			{
				L"description", required_argument, 0, 'd' 
			}
			,
			{
				L"names", no_argument, 0, 'n' 
			}
			,
			{ 
				0, 0, 0, 0 
			}
		}
	;		
		
	while( 1 )
	{
		int opt_index = 0;
		
		int opt = wgetopt_long( argc,
								argv, 
								L"ed:n", 
								long_options, 
								&opt_index );
		if( opt == -1 )
			break;
			
		switch( opt )
		{
			case 0:
				if(long_options[opt_index].flag != 0)
					break;
				sb_append2( sb_err,
							L"functions: Unknown option ",
							long_options[opt_index].name,
							L"\n",
							0);				
				builtin_print_help( argv[0], sb_err );
	
				
				return 1;
				
			case 'e':		
				erase=1;				
				break;

			case 'd':
				desc=woptarg;
				break;				

			case 'n':
				list=1;
				break;
				
			case '?':
				builtin_print_help( argv[0], sb_err );
				
				return 1;
				
		}
		
	}		

	/*
	  Erase, desc and list are mutually exclusive
	*/
	if( (erase + (desc!=0) + list) > 1 )
	{
		sb_append2( sb_err,
					argv[0],
					L": Invalid combination of options\n",
					0);				
		builtin_print_help( argv[0], sb_err );
		
		return 1;
	}
	

	if( erase )
	{
		int i;
		for( i=woptind; i<argc; i++ )
			function_remove( argv[i] );
		return 0;
	}
	else if( desc )
	{
		wchar_t *func;
		
		if( argc-woptind != 1 )
		{
			sb_append2( sb_err,
						L"functions: Expected exactly one function name\n",
						0);				
			builtin_print_help( argv[0], sb_err );
			
			return 1;
		}
		func = argv[woptind];
		if( !function_exists( func ) )
		{
			sb_append2( sb_err,
						L"functions: Function ",
						func,
						L" does not exist\n",
						0);				
			builtin_print_help( argv[0], sb_err );
			
			return 1;			
		}
		
		function_set_desc( func, desc );		
		
		return 0;		
	}
	else if( list )
	{
		al_init( &names );
		function_get_names( &names );
		names_arr = list_to_char_arr( &names );
		qsort( names_arr, 
			   al_get_count( &names ), 
			   sizeof(wchar_t *), 
			   &mycmp );
		for( i=0; i<al_get_count( &names ); i++ )
		{
			sb_append2( sb_out,
						names_arr[i],
						L"\n",
						0 );			
		}
		free( names_arr );
		al_destroy( &names );			
		return 0;
	}
	
	
	switch( argc - woptind )
	{
		case 0:
		{
			sb_append( sb_out, L"Current function definitions are:\n\n" );		
			al_init( &names );
			function_get_names( &names );
			names_arr = list_to_char_arr( &names );
			qsort( names_arr, 
				   al_get_count( &names ), 
				   sizeof(wchar_t *), 
				   &mycmp );
			for( i=0; i<al_get_count( &names ); i++ )
			{
				sb_append2( sb_out,
							L"function ",
							names_arr[i],
							L"\n\t",
							function_get_definition(names_arr[i]),
							L"\nend\n\n",
							0);
			}
			free( names_arr );
			al_destroy( &names );			
			break;
		}
		
		default:
		{
			for( i=woptind; i<argc; i++ )
				sb_append2( sb_out,
							L"function ",
							argv[i],
							L"\n\t",
							function_get_definition(argv[i]),
							L"\nend\n\n",
							0);

			break;
		}
	}
	return 0;
	
	
}


/**
   The function builtin, used for providing subroutines.
   It calls various functions from function.c to perform any heavy lifting.
*/
static int builtin_function( wchar_t **argv )
{	
	int argc = count_args( argv );
	int res=0;
	wchar_t *desc=0;	
	
	woptind=0;

	const static struct woption
		long_options[] =
		{
			{
				L"description", required_argument, 0, 'd' 
			}
			,
			{ 
				0, 0, 0, 0 
			}
		}
	;		
		
	while( 1 )
	{
		int opt_index = 0;
		
		int opt = wgetopt_long( argc,
								argv, 
								L"d:", 
								long_options, 
								&opt_index );
		if( opt == -1 )
			break;
			
		switch( opt )
		{
			case 0:
				if(long_options[opt_index].flag != 0)
					break;
				sb_append2( sb_err,
							L"function: Unknown option ",
							long_options[opt_index].name,
							L"\n",
							0);				
				builtin_print_help( argv[0], sb_err );
				
				return 1;
				
			case 'd':		
				desc=woptarg;				
				break;

			case '?':
				builtin_print_help( argv[0], sb_err );
				
				return 1;
				
		}
		
	}		

	if( argc-woptind != 1 )
	{
		sb_append( sb_err, 
				   L"function: Expected one argument, got " );
		sb_append_int( sb_err, 
					   argc-woptind );
		sb_append( sb_err, 
				   L"\n" );
 		
		res=1;
	}
	else if( !wcsvarname( argv[woptind] ) )
	{ 
		sb_append2( sb_err, 
					L"function: illegal function name \'", 
					argv[woptind], 
					L"\'\n", 
					0 );
		res=1;	
	}	
	else if( parser_reserved(argv[woptind] ) )
    {
		
        sb_append2( sb_err,
                    L"function: the name \'",
                    argv[woptind],
                    L"\' is reserved,\nand can not be used as a function name\n",
                    0 );
		
        res=1;
		
    }
	

	
	if( res )
	{
		int i;
		array_list_t names;
		wchar_t **names_arr;
		int chars=0;
		
		builtin_print_help( argv[0], sb_err );
		
		sb_append( sb_err, L"Current functions are: " );		
		chars += wcslen( L"Current functions are: " );		
		al_init( &names );
		function_get_names( &names );
		names_arr = list_to_char_arr( &names );
		qsort( names_arr, 
			   al_get_count( &names ), 
			   sizeof(wchar_t *), 
			   &mycmp );
		for( i=0; i<al_get_count( &names ); i++ )
		{
			wchar_t *nxt = names_arr[i];
			int l = wcslen( nxt + 2 );
			if( chars+l > reader_get_width() )
			{
				chars = 0;
				sb_append(sb_err, L"\n" );
			}
			
			sb_append2( sb_err,
						nxt, L"  ", 0 );			
		}
		free( names_arr );
		al_destroy( &names );
		sb_append( sb_err, L"\n" );		

		parser_push_block( FAKE );
	}
	else
	{
		parser_push_block( FUNCTION_DEF );
		current_block->function_name=wcsdup(argv[woptind]);
		current_block->function_description=desc?wcsdup(desc):0;
	}
	
	current_block->tok_pos = parser_get_pos();
	current_block->skip = 1;
	
	return 0;
	
}

/**
   The read builtin. Reads from stdin and stores the values in environment variables.
*/
static int builtin_read( wchar_t **argv )
{
	wchar_t *buff=0;
	int buff_len = 0;
	int i, argc = count_args( argv );
	wchar_t *ifs;
	static int place = 0;
	wchar_t *nxt;
	wchar_t *prompt = DEFAULT_READ_PROMPT;
	wchar_t *commandline = L"";
	
	woptind=0;

	while( 1 )
	{
		const static struct woption
			long_options[] =
			{
				{
					L"export", no_argument, &place, ENV_EXPORT 
				}
				,
				{
					L"global", no_argument, &place, ENV_GLOBAL
				}
				,
				{
					L"prompt", required_argument, 0, 'p'
				}
				,
				{
					L"command", required_argument, 0, 'c'
				}
				,
				{ 
					0, 0, 0, 0 
				}
			}
		;		
		
		int opt_index = 0;
		
		int opt = wgetopt_long( argc,
								argv, 
								L"xgp:c:", 
								long_options, 
								&opt_index );
		if( opt == -1 )
			break;
			
		switch( opt )
		{
			case 0:
				if(long_options[opt_index].flag != 0)
					break;
				sb_append2( sb_err, 
							L"read: Unknown option ",
							long_options[opt_index].name,
							L"\n",
							0 );
				builtin_print_help( argv[0], sb_err );

				return 1;
				
			case L'x':		
				place=ENV_EXPORT;
				break;
			case L'g':		
				place=ENV_GLOBAL;
				break;
			case L'p':
				prompt = woptarg;
				break;
			case L'c':
				commandline = woptarg;
				break;
				
			case L'?':
				builtin_print_help( argv[0], sb_err );

				
				return 1;	
		}
		
	}		

	ifs = env_get( L"IFS" );
	if( ifs == 0 )
		ifs = L"";

	/*
	  Check if we should read interactively using \c reader_readline()
	*/
	if( isatty(0) && builtin_stdin == stdin )
	{				
		reader_push( READ_MODE_NAME );		
		reader_set_prompt( prompt );
				
		reader_set_buffer( commandline, wcslen( commandline ) );
		buff = wcsdup(reader_readline( ));
		reader_pop();
	}
	else
	{
		
		int len = fgetws2( &buff, &buff_len, builtin_stdin );
		if ( len == -1 )
		{
			if( buff )
				free( buff );
			builtin_wperror( L"read" );
			sb_append2( sb_err, 
						parser_current_line(),
						L"\n",
						0 );
			return 1;
		}
	}
	
	i=woptind;
	wchar_t *state;
	
	nxt = wcstok( buff, (i<argc-1)?ifs:L"", &state );
	
	while( i<argc )
	{
		env_set( argv[i], nxt != 0 ? nxt: L"", place );		

		i++;
		if( nxt != 0 )
			nxt = wcstok( 0, (i<argc-1)?ifs:L"", &state);		
	}
	
	free( buff );
	return 0;
}

/**
   The commandline builtin. It is used for specifying a new value for
   the commandline.
*/
static int builtin_commandline( wchar_t **argv )
{
	int argc = count_args( argv );
	wchar_t *append_str=0;
	
	if( !shell_is_interactive )
	{
		sb_append2( sb_err,
					argv[0],
					L": Can not set commandline in non-interactive mode\n",
					0 );
		builtin_print_help( argv[0], sb_err );
		return 1;		
	}	

	woptind=0;

	while( 1 )
	{
		const static struct woption
			long_options[] =
			{
				{
					L"append", required_argument, 0, 'a'
				}
				,
				{ 
					0, 0, 0, 0 
				}
			}
		;		
		
		int opt_index = 0;
		
		int opt = wgetopt_long( argc,
								argv, 
								L"a:", 
								long_options, 
								&opt_index );
		if( opt == -1 )
			break;
			
		switch( opt )
		{
			case 0:
				if(long_options[opt_index].flag != 0)
					break;
				sb_append2( sb_err, 
							L"commandline: Unknown option ",
							long_options[opt_index].name,
							L"\n",
							0 );
				builtin_print_help( argv[0], sb_err );

				return 1;
				
			case L'a':
				append_str = woptarg;
				break;
				
			case L'?':
				builtin_print_help( argv[0], sb_err );				
				return 1;	
		}
	}		

	switch(argc-woptind)
	{
		case 0:
			if( append_str )
			{
				wchar_t *old = reader_get_buffer();
				int c = reader_get_cursor_pos();
				int l = wcslen(append_str);
				wchar_t *new = malloc( sizeof(wchar_t)*(wcslen(old)+l+1 ));
				wcscpy( new, old );
				wcscpy( new+c, append_str );
				wcscpy( new+c+l, old+c );
				reader_set_buffer( new, c+l );
				free(new);				
			}
			else
				sb_append2( sb_out, reader_get_buffer(), L"\n", 0 );			
			break;

		case 1:
			reader_set_buffer( argv[1], -1 );
			break;
			
		default:
			sb_append2( sb_err,
							argv[0],
							L": Wrong number of arguments\n",
							0 );
			builtin_print_help( argv[0], sb_err );
			return 1;
			
	}
	
	return 0;
}


/**
   The eval builtin. Concatenates the arguments and calls eval on the
   result.
*/
static int builtin_eval( wchar_t **argv )
{
	wchar_t *tot, **ptr, *next;
	int totlen=0;

	for( ptr = argv+1; *ptr; ptr++ )
	{
		totlen += wcslen( *ptr) + 1;
	}
	tot = malloc( sizeof(wchar_t)*totlen );
	if( !tot )
	{
		sb_append( sb_err,
				   L"fish: out of memory\n" );
		sb_append( sb_err, 
					parser_current_line() );		
		return(1);
	}
	for( ptr = argv+1, next=tot; *ptr; ptr++ )
	{
		int len = wcslen( *ptr );
		wcscpy( next, *ptr );
		next+=len;
		*next++=L' ';
	}
	*(next-1)=L'\0';
	eval( tot, 0, TOP );
	free( tot );
	return proc_get_last_status();
}

/**
   The exit builtin. Calls do_exit to exit and returns the value specified.
*/
static int builtin_exit( wchar_t **argv )
{	
	int argc = count_args( argv );
	
	int ec=0;
	switch( argc )
	{
		case 1:
			break;
		case 2:
		{
			wchar_t *end;
			errno = 0;			
			ec = wcstol(argv[1],&end,10);
			if( errno || *end != 0)
			{
				sb_append2( sb_err, argv[0], L": Argument must be an integer '", argv[1], L"'\n", 0 );
				builtin_print_help( argv[0], sb_err );				
				return 1;
			}
			break;
		}
		
		default:
			sb_append2( sb_err, argv[0], L": Too many arguments\n", 0 );
			builtin_print_help( argv[0], sb_err );				
			return 1;
				
	}
	do_exit();
	return ec;
}

static int set_pwd(wchar_t *env)
{
	char buf[4096];
	char *res = getcwd( buf, 4096 );
	wchar_t *dir_path;
	if( !res )
	{
		builtin_wperror( L"getcwd" );
		return 0;
	}
	dir_path = str2wcs( res );
	if( dir_path )
	{
		env_set( env, dir_path, ENV_EXPORT );
		free( dir_path );
	}
	return 1;
}

/**
   The cd builtin. Changes the current directory to the one specified
   or to $HOME if none is specified. If '-' is the directory specified,
   the directory is changed to the previous working directory. The
   directory can be relative to any directory in the CDPATH variable.
*/
static int builtin_cd( wchar_t **argv )
{
	wchar_t *dir_in;
	wchar_t *dir;
	int free_dir = 0;
	int res=0;

	if( argv[1]  == 0 )
	{
		dir_in = env_get( L"HOME" );
		if( !dir_in )
		{
			sb_append2( sb_err,
						argv[0], 
						L": Could not find home directory\n",
						0 );			
			
		}		
	}	
	else if (wcscmp(argv[1], L"-") == 0)
	{
		dir_in = env_get(L"OLDPWD");
		if (!dir_in)
		{
			sb_append2(sb_err,
			           argv[0],
			           L": Could not find previous working directory.\n",
			           0);
		}
	}
	else 
		dir_in = argv[1];
	
	dir = parser_cdpath_get( dir_in );	
	
	if( !dir )
	{
		sb_append2( sb_err,
					L"cd: ",
					dir_in,
					L" is not a directory or you do not have permission to enter it\n",
					0 );
		sb_append2( sb_err, 
					parser_current_line(),
					0 );			
		return 1;
	}		
	
	if (!set_pwd(L"OLDPWD"))
	{
		res = 1;
	}

	char *dir_str = wcs2str( dir );

	if( chdir( dir_str ) != 0 )
	{
		free( dir_str );
		sb_append2( sb_err,
					L"cd: ",
					dir,
					L" is not a directory\n",
					0 );
		sb_append2( sb_err, 
					parser_current_line(),
					0 );
		
		if( free_dir )
			free( dir );
		
		return 1;
	}
	free( dir_str );

	if (!set_pwd(L"PWD"))
	{
		res=1;
	}
	
	free( dir );

	return res;
}

/**
   The complete builtin. Used for specifying programmable
   tab-completions. Calls the functions in complete.c for any heavy
   lifting.
*/
static int builtin_complete( wchar_t **argv )
{
	
	int argc=0;
	int result_mode=SHARED, long_mode=0;
	int cmd_type=-1;
	int remove = 0;
	int authorative = 1;
	
	wchar_t *cmd=0, short_opt=L'\0', *long_opt=L"", *comp=L"", *desc=L"";
	
	argc = count_args( argv );	
	
	woptind=0;
	
	while( 1 )
	{
		const static struct woption
			long_options[] =
			{
				{
					L"exclusive", no_argument, 0, 'x' 
				}
				,
				{
					L"no-files", no_argument, 0, 'f' 
				}
				,
				{
					L"require-parameter", no_argument, 0, 'r' 
				}
				,
				{
					L"path", required_argument, 0, 'p'
				}
				,					
				{
					L"command", required_argument, 0, 'c' 
				}
				,					
				{
					L"short-option", required_argument, 0, 's' 
				}
				,
				{
					L"long-option", required_argument, 0, 'l'				}
				,
				{
					L"old-option", required_argument, 0, 'o' 
				}
				,
				{
					L"description", required_argument, 0, 'd'
				}
				,
				{
					L"arguments", required_argument, 0, 'a'
				}
				,
				{
					L"erase", no_argument, 0, 'e'
				}
				,
				{
					L"unauthorative", no_argument, 0, 'u'
				}
				,
				{ 
					0, 0, 0, 0 
				}
			}
		;		
		
		int opt_index = 0;
		
		int opt = wgetopt_long( argc,
								argv, 
								L"a:c:p:s:l:o:d:frxeu", 
								long_options, 
								&opt_index );
		if( opt == -1 )
			break;
			
		switch( opt )
		{
			case 0:
				if(long_options[opt_index].flag != 0)
					break;
				sb_append2( sb_err,
							L"complete: Unknown option ",
							long_options[opt_index].name,
							L"\n",
							0 );
//				builtin_print_help( argv[0], sb_err );

				
				return 1;
				
				
			case 'x':					
				result_mode |= EXCLUSIVE;
				break;
					
			case 'f':					
				result_mode |= NO_FILES;
				break;
				
			case 'r':					
				result_mode |= NO_COMMON;
				break;
					
			case 'p':					
				cmd_type = PATH;
				cmd = woptarg;
				break;
					
			case 'c':
				cmd_type = COMMAND;
				cmd = woptarg;
				break;
				
			case 'd':
				desc = woptarg;
				break;
				
			case 'u':
				authorative=0;
				break;
				
			case 's':
				if( wcslen( woptarg ) > 1 )
				{
					sb_append2( sb_err,
								L"complete: Parameter too long ",
								woptarg,
								L"\n",
								0);
//				builtin_print_help( argv[0], sb_err );
					
					return 1;
				}
				
				short_opt = woptarg[0];
				break;
					
			case 'l':
				long_opt = woptarg;
				break;
				
			case 'o':
				long_mode=1;				
				long_opt = woptarg;
				break;

			case 'a':
				comp = woptarg;
				break;
				

			case 'e':
				remove = 1;
				
				break;
				

			case '?':
				//	builtin_print_help( argv[0], sb_err );
				
				return 1;
				
		}
		
	}
	
	if( woptind != argc )
	{
		sb_append( sb_err, 
				   L"complete: Too many arguments\n" );
		//			builtin_print_help( argv[0], sb_err );

		return 1;
	}

	if( cmd == 0 )
	{
		/* No arguments specified, meaning we print the definitions of
		 * all specified completions to stdout.*/
		complete_print( sb_out );		
	}
	else
	{
		if( remove )
		{
			/* Remove the specified completion */
			complete_remove( cmd, 
							 cmd_type, 
							 short_opt,
							 long_opt );
		}
		else
		{
			/* Add the specified completion */
			complete_add( cmd, 
						  cmd_type, 
						  short_opt,
						  long_opt,
						  long_mode, 
						  result_mode, 
						  authorative,
						  comp,
						  desc ); 
		}
	}	
	return 0;
}

/**
   The source builtin. Can be called through either 'source' or
   '.'. Evaluates the contents of a file. 
*/
static int builtin_source( wchar_t ** argv )
{
	int stdin_org;
	int res;
	
	if( (argv[1] == 0) || (argv[2]!=0) )
	{
		sb_append( sb_err, L".: Expected exactly one argument\n" );
		builtin_print_help( argv[0], sb_err );

		return 1;
	}
	
	if( (stdin_org=dup( 0 )) == -1)
	{
		builtin_wperror(L"dup");
		return 1;
	}

	if( close( 0 ) )
	{
		builtin_wperror(L"close");
		return 1;
	}
	
	if( wopen( argv[1], O_RDONLY ) == -1 )
	{		
		builtin_wperror( L"open" );
		res = 1;
	}
	else
	{
		reader_push_current_filename( argv[1] );
		res = reader_read();		
		if( close( 0 ) )
		{
			builtin_wperror(L"close");
			res = errno;
		}
		reader_pop_current_filename();
	}

	if( dup( stdin_org ) == -1)
	{
		builtin_wperror(L"dup");
		res = errno;
		fwprintf( stderr, L"Could not restore stdout\n" );
		sanity_lose();
	}
	
	if( close( stdin_org ) )
	{
		builtin_wperror(L"close");
		res = errno;
		fwprintf( stderr, L"Could not restore stdout\n" );
		sanity_lose();
	}

	return res;
}

/**
   The set builtin. Changes environment variables. Calls the functions in env.c for any heavy lifting.
*/
static int builtin_set( wchar_t **argv )
{
	int export=0, erase=0, global=0, local=0;
	int argc=count_args(argv);
	int list=0;
	
	
	woptind=0;
	
	while( 1 )
	{
		struct woption
			long_options[] =
			{
				{
					L"export", no_argument, &export, 1 
				}
				,
				{
					L"global", no_argument, &global, 1 
				}
				,
				{
					L"local", no_argument, &local, 1 
				}
				,
				{
					L"erase", no_argument, &erase, 1 
				}
				,
				{
					L"erase", no_argument, &erase, 1 
				}
				,
				{
					L"names", no_argument, 0, 'n' 
				}
				,
				{ 
					0, 0, 0, 0 
				}
			}
		;		
		
		int opt_index = 0;
		
		int opt = wgetopt_long( argc,
								argv, 
								L"xgeln", 
								long_options, 
								&opt_index );
		if( opt == -1 )
			break;
			
		switch( opt )
		{
			case 0:
				if(long_options[opt_index].flag != 0)
					break;
				
				sb_append2( sb_err, L"fish: Unknown option ",
							long_options[opt_index].name,
							L"\n",
							0 );				
				builtin_print_help( argv[0], sb_err );
				
				return 1;
				
			case 'e':		
				erase = 1;
				break;
			case 'x':		
				export = 1;
				break;
			case 'l':		
				local = 1;
				break;
			case 'g':		
				global = 1;
				break;
			case 'n':
				list=1;
				break;
			case '?':
				builtin_print_help( argv[0], sb_err );
				
				return 1;	
		}		
	}	

	

	if( local && global )
	{
		sb_append( sb_err, L"set: Variable can\'t be both global and local.\n" );
		builtin_print_help( argv[0], sb_err );
		return 1;
	}	

	if( erase && list )
	{
		sb_append2( sb_err,
					argv[0],
					L": Invalid combination of options\n",
					0);				
		builtin_print_help( argv[0], sb_err );
		
		return 1;
	}	

	if( erase )
	{
		if( woptind != argc-1 )
		{
			sb_append( sb_err, L"set: You must specify the name of the variable to erase\n" );
			builtin_print_help( argv[0], sb_err );
				
			return 1;
		}
		env_remove( argv[woptind], ENV_USER );
	}
	else if( list )
	{		
		array_list_t names;
		wchar_t **names_arr;
		int i;
		
		al_init( &names );
		env_get_names( &names, 
					   (local?ENV_LOCAL:0)|
					   (export?ENV_EXPORT:0)|
					   (global?ENV_GLOBAL:0) );
		names_arr = list_to_char_arr( &names );
		qsort( names_arr, 
			   al_get_count( &names ), 
			   sizeof(wchar_t *), 
			   &mycmp );
		for( i=0; i<al_get_count( &names ); i++ )
		{
			sb_append2( sb_out,
						names_arr[i],
						L"\n", 
						0 );
		}
		free( names_arr );
		al_destroy( &names );					
	}	
	else
	{
		wchar_t *name=0;
		int idx=0;
		int is_arr=0;
		if( argc-woptind > 0 )
		{			
			wchar_t *pos = name = argv[woptind];	

			while( *pos )
			{
				if( !( iswalnum( *pos ) ||
					   (wcschr(L"_", *pos)!= 0)  ) )
					break;
				pos++;
				
			}
			switch( *pos )
			{
				case L'[':
				{
					wchar_t *end;
					*pos = 0;
					
					pos++;
					while( *pos == L' ' )
						pos++;
					
					errno=0;
					idx = wcstol( pos, &end, 10 );
					if( errno )
					{
						sb_append2( sb_err, L"Expected integer for array index", 0 );
				builtin_print_help( argv[0], sb_err );

						return 1;
					}
					pos = end;
					
					while( *pos == L' ' )
						pos++;
					
					if( wcscmp( pos, L"]")!=0 )
					{
						sb_append2( sb_err, L"Expected \']\'", 0 );
				builtin_print_help( argv[0], sb_err );
						return 1;
					}
					is_arr = 1;
					
					break;
				}
				case 0:
				{
					break;
				}
				default:
				{
					sb_append2( sb_err, L"Illegal variable name: ", name, L"\n", 0 );
				builtin_print_help( argv[0], sb_err );
					return 1;
				}
				
			}
						
		}
		
		switch( argc-woptind )
		{
			case 0:
			{
				/* 
				   Print all environment variables
				*/
				array_list_t names;
				wchar_t **names_arr;
				int i;
				
				al_init( &names );
				env_get_names( &names, 
							   (local?ENV_LOCAL:0)|
							   (export?ENV_EXPORT:0)|
							   (global?ENV_GLOBAL:0) );
				names_arr = list_to_char_arr( &names );
				qsort( names_arr, 
					   al_get_count( &names ), 
					   sizeof(wchar_t *), 
					   &mycmp );
				for( i=0; i<al_get_count( &names ); i++ )
				{
					wchar_t *key = expand_escape( wcsdup(names_arr[i]), 1);
					
					wchar_t *val_orig = env_get( names_arr[i] );


					
					sb_append2( sb_out,
								key,
								L" ", 
								0 );
					
					wchar_t * val = expand_escape_variable( val_orig );
					sb_append( sb_out, val );
					free(val);
					
					sb_append( sb_out, 
							   L"\n" );

					free( key );
				}
				free( names_arr );
				al_destroy( &names );			
				break;
			}
			case 1:
			{
				env_set( name,
						 0, 
						 (local?ENV_LOCAL:0)|
						 (export?ENV_EXPORT:0)|
						 (global?ENV_GLOBAL:0)|ENV_USER);

				break;
			}
			
			default:
			{
				
				if( is_arr )
				{
					if( argc-woptind != 2 )
					{
						sb_append( sb_err, L"set: Can not set array element to more than one value\n" );
						builtin_print_help( argv[0], sb_err );
						break;
					}
					
					wchar_t * old = env_get( name );
					array_list_t l;
					string_buffer_t s;
					int i;
					
					al_init( &l );
					
					if( !old )
						old =L"";
					
					//			fwprintf( stderr, L"var %ls, %d characters\n", name, wcslen( old ) );
					expand_variable_array( old, &l );
					//fwprintf( stderr, L"Expanded to %d elements\n", al_get_count( &l ) );
					
					while( al_get_count( &l ) < idx )
						al_push( &l, wcsdup( L""));
					
					//fwprintf( stderr, L"Expanded to %d elements after\n", al_get_count( &l ) );

					free( (void *)al_get( &l, idx - 1 ) );
					al_set( &l, idx-1, argv[woptind+1] );
					
					sb_init( &s );
					for( i=0; i<al_get_count( &l ); i++ )
					{
						if( i != 0 )
							sb_append( &s, ARRAY_SEP_STR );
						
						//fwprintf( stderr, L"add string %ls of len %d\n", (wchar_t *)al_get( &l, i ), wcslen((wchar_t *)al_get( &l, i )));
						
						sb_append( &s, (wchar_t *)al_get( &l, i ) );
						if( al_get( &l, i ) != argv[woptind+1] )
							free( (wchar_t *)al_get( &l, i ) );
					}
										
					env_set( name, (wchar_t *)s.buff, (local?ENV_LOCAL:0)|(export?ENV_EXPORT:0)|(global?ENV_GLOBAL:0)|ENV_USER);
					sb_destroy( &s );
					
				}
				else
				{
					int i;
					
					string_buffer_t s;
					sb_init( &s );
					for( i=woptind+1; i<argc; i++ )
					{
						if( i != (woptind+1) )
							sb_append( &s, ARRAY_SEP_STR );
						
						//fwprintf( stderr, L"add string %ls of len %d\n", (wchar_t *)al_get( &l, i ), wcslen((wchar_t *)al_get( &l, i )));
						
						sb_append( &s, argv[i] );
					}
					

/*					fwprintf( stderr,
  L"Set %ls to %ls\n", 
  argv[woptind], 
  s.buff );
*/				
					env_set( argv[woptind], 
							 (wchar_t *)s.buff, 
							 (local?ENV_LOCAL:0)|(export?ENV_EXPORT:0)|(global?ENV_GLOBAL:0)|ENV_USER );
					sb_destroy( &s );
				}
				
				break;
			}
			
/*			default:
  builtin_print_help( argv[0] );
  return 1;*/

		}
		
	}
	return 0;		
}

/**
   Make the specified job the first job of the job list. Moving jobs
   around in the list makes the list reflect the order in which the
   jobs where used.
*/
static void make_first( job_t *j )
{
	job_t *prev=0;
	job_t *curr;
	for( curr = first_job; curr != j; curr = curr->next )
	{
		prev=curr;
	}
	if( curr == j )
	{
		if( prev == 0 )
			return;
		else
		{
			prev->next = curr->next;
			curr->next = first_job;
			first_job = curr;
		}
	}
}


/**
   Builtin for putting a job in the foreground
*/
static int builtin_fg( wchar_t **argv )
{
	job_t *j;
	
	if( argv[1] == 0 )
	{
		for( j=first_job; ((j!=0) && (!j->constructed)); j=j->next )
			;
	}
	else if( argv[2] != 0 )
	{
		int pid = wcstol( argv[1], 0, 10 );
		j = job_get_from_pid( pid );
		if( j != 0 )
		{
			sb_append( sb_err, L"fg: Ambiguous job\n" );	
		}
		else
		{
			sb_append2( sb_err, L"fg: Not a job (", argv[1], L")\n", 0 );
		}
				builtin_print_help( argv[0], sb_err );
		
		return 1;
	}
	else
	{
		int pid = wcstol( argv[1], 0, 10 );
		j = job_get_from_pid( pid );
	}

	if( j == 0 )
	{
		sb_append( sb_err, L"fg: No suitable job\n" );
				builtin_print_help( argv[0], sb_err );
		return 1;
	}
/*
  It would be nice to print some message before putting the job into
  the foreground, but we can't since the output won't be written until
  after the builtin returns, by which time the command will have
  exited. Maybe we should check if the user is redirecting output and if not print directly to screen.
*/
/*	else
  {
  sb_append( sb_err, L"Set job ");
  sb_append_int( sb_err, j->job_id );
  sb_append( sb_err, L" with pgid " );
  sb_append_int( sb_err, j->pgid );
  sb_append( sb_err, L" to fg\n" );
  }
*/

	wchar_t *ft = tok_first( j->command );
	if( ft != 0 )
		env_set( L"_", ft, ENV_EXPORT );
	free(ft);
	reader_write_title();
/*
  fwprintf( stderr, L"Send job %d, \'%ls\' to foreground\n", 
  j->job_id,
  j->command );
*/
	make_first( j );
	j->fg=1;
	
		
	job_continue( j, job_is_stopped(j) );
	return 0;
}

/**
   Helper function for builtin_bg()
*/
static void send_to_bg( job_t *j, wchar_t *name )
{
	if( j == 0 )
	{
		sb_append2( sb_err, L"bg: Unknown job ", name, L"\n", 0 );
		builtin_print_help( L"bg", sb_err );
		return;
	}	
	else
	{
		sb_append( sb_err, L"Send job ");
		sb_append_int( sb_err, j->job_id );
		sb_append( sb_err, L", \'");
		sb_append( sb_err, j->command);
		sb_append( sb_err, L"\' to background\n" );
	}
	make_first( j );
	j->fg=0;
	job_continue( j, job_is_stopped(j) );
}


/**
   Builtin for putting a job in the background
*/
static int builtin_bg( wchar_t **argv )
{
	if( argv[1] == 0 )
	{
  		job_t *j;
		for( j=first_job; ((j!=0) && (!j->constructed) && (!job_is_stopped(j))); j=j->next )
			;
		send_to_bg( j, L"(default)");
		return 0;
	}
	for( argv++; *argv != 0; argv++ )
	{
		int pid = wcstol( *argv, 0, 10 );
		send_to_bg( job_get_from_pid( pid ), *argv);
	}
	return 0;
}


#ifdef HAVE__PROC_SELF_STAT
/**
   Calculates the cpu usage (in percent) of the specified job.
*/
static int cpu_use( job_t *j )
{
	double u=0;
	process_t *p;
	
	for( p=j->first_process; p; p=p->next )
	{
		struct timeval t;
		int jiffies;
		gettimeofday( &t, 0 );
		jiffies = proc_get_jiffies( p );
		
		double t1 = 1000000.0*p->last_time.tv_sec+p->last_time.tv_usec;
		double t2 = 1000000.0*t.tv_sec+t.tv_usec;
		
/*		fwprintf( stderr, L"t1 %f t2 %f p1 %d p2 %d\n",
  t1, t2, jiffies, p->last_jiffies );
*/	

		u += ((double)(jiffies-p->last_jiffies))/(t2-t1);
	}
	return u*1000000;
}
#endif

/**
   Builtin for printing running jobs
*/
static int builtin_jobs( wchar_t **argv )
{	
	job_t *j;
	int found=0;
	
	for( j= first_job; j; j=j->next )
	{
		/*
		  Ignore unconstructed jobs, i.e. ourself.
		*/
		if( j->constructed )
		{
			if( !found )
			{
				/*
				  Print table header before first job
				*/
				sb_append( sb_out, L"Job\tGroup\t");
#ifdef HAVE__PROC_SELF_STAT
				sb_append( sb_out, L"CPU\t" );
#endif
				sb_append( sb_out, L"State\tCommand\n" );
			}
			
			found = 1;
			sb_append_int( sb_out, j->job_id );			
			sb_append( sb_out, L"\t" );
			sb_append_int( sb_out, j->pgid );
			sb_append( sb_out, L"\t" );
			
#ifdef HAVE__PROC_SELF_STAT
			sb_append_int( sb_out, cpu_use(j) );
			sb_append( sb_out, L"%\t" );
#endif
			sb_append2( sb_out, job_is_stopped(j)?L"stopped\t":L"running\t", 
						j->command, L"\n", 0 );
			
		}
	}
	if( !found )
	{
		sb_append( sb_out, L"jobs: There are no running jobs\n" );
	}
	return 0;
}

/**
   Builtin for looping over a list
*/
static int builtin_for( wchar_t **argv )
{
	int argc = count_args( argv );
	int res=1;
	

	if( argc < 3) 
	{
		sb_append( sb_err, 
				   L"for: Expected at least two arguments\n" );				
		builtin_print_help( argv[0], sb_err );
	}
	else if ( !wcsvarname(argv[1]) )
	{
		sb_append2( sb_err, 
					L"for: \'",
					argv[1],
					L"\' invalid variable name\n",
					0);				
		builtin_print_help( argv[0], sb_err );
	}
	else if (wcscmp( argv[2], L"in") != 0 )
	{
		sb_append( sb_err, 
				   L"for: Second argument must be \'in\'\n" );				
		builtin_print_help( argv[0], sb_err );
	}
	else
	{
		res=0;
	}

	
	if( res )
	{
		parser_push_block( FAKE );
	}
	else
	{
		parser_push_block( FOR );
		al_init( &current_block->for_vars);
		
		int i;
		current_block->tok_pos = parser_get_pos();
		current_block->for_variable = wcsdup( argv[1] );
		
		for( i=argc-1; i>3; i-- )
		{
			al_push( &current_block->for_vars, wcsdup(argv[ i ] ));
		}
		if( argc > 3 )
		{
			env_set( current_block->for_variable, argv[3], 0);
		}
		else
		{
			current_block->skip=1;
		}	
	}
	return res;
}

/**
   Builtin for ending a block of code, such as a for-loop or an if statement
*/
static int builtin_end( wchar_t **argv )
{
	if( !current_block->outer )
	{
		sb_append( sb_err,
				   L"end: Not inside of block\n" );
		builtin_print_help( argv[0], sb_err );
		return 1;
	}
	else
	{
		int kill_block = 1;
		
		switch( current_block->type )
		{
			case WHILE:
			{
				if( !( current_block->skip && (current_block->loop_status != LOOP_CONTINUE )))
				{
					current_block->loop_status = LOOP_NORMAL;
					current_block->skip = 0;
					kill_block = 0;
					parser_set_pos( current_block->tok_pos);
					current_block->while_state = WHILE_TEST_AGAIN;
				}
				
				break;
			}
		
			case IF:
				break;

			case FOR:
			{
				if( current_block->loop_status == LOOP_BREAK )
				{
					while( al_get_count( &current_block->for_vars ) )
					{
						free( (void *)al_pop( &current_block->for_vars ) );
					}
				}
				
				if( al_get_count( &current_block->for_vars ) )
				{
					wchar_t *val = (wchar_t *)al_pop( &current_block->for_vars );
					env_set( current_block->for_variable, val, 0);
					current_block->loop_status = LOOP_NORMAL;
					current_block->skip = 0;
					free(val);
					
					kill_block = 0;
					parser_set_pos( current_block->tok_pos );
/*					
  fwprintf( stderr, 
  L"jump to %d\n",
  current_block->tok_pos );								*/
				}
				break;
			}
		
			case FUNCTION_DEF:
			{
				wchar_t *def = wcsndup( parser_get_buffer()+current_block->tok_pos, parser_get_job_pos()-current_block->tok_pos );
				
//				fwprintf( stderr, L"Function: %ls\n", def );
				function_add( current_block->function_name, 
							  def,
							  current_block->function_description);
				free(def);
			}
			break;
			
		}
		if( kill_block )
		{
			parser_pop_block();
		}
//		fwprintf( stderr, L"End with status %d\n", proc_get_last_status() );
		
		return proc_get_last_status();
	}	
}

/**
   Builtin for executing commands if an if statement is false
*/
static int builtin_else( wchar_t **argv )
{
	if( current_block == 0 || 
		current_block->type != IF ||
		current_block->if_state != 1)
	{
		sb_append( sb_err,
				   L"else: not inside of if block\n" );
		builtin_print_help( argv[0], sb_err );
		return 1;
	}
	else
	{
		current_block->if_state++;
		current_block->skip = !current_block->skip;
		env_pop();
		env_push(0);
	}
	return 0;
}

/**
   this function handles both the 'continue' and the 'break' builtins.
*/
static int builtin_break_continue( wchar_t **argv )
{
	int is_break = (wcscmp(argv[0],L"break")==0);
	int argc = count_args( argv );
	
	block_t *b = current_block;
	
	if( argc != 1 )
	{
		sb_append2( sb_err, 
					argv[0], 
					L": Unknown option \'", argv[1], L"\'", 0 );
		builtin_print_help( argv[0], sb_err );
		return 1;		
	}
	

	while( (b != 0) && 
		   ( b->type != WHILE) && 
		   (b->type != FOR ) )
	{
		b = b->outer;
	}
	if( b == 0 )
	{
		sb_append2( sb_err, 
					argv[0], 
					L": Not inside of loop\n", 0 );
		builtin_print_help( argv[0], sb_err );
		return 1;
	}
	
	b = current_block;
	while( ( b->type != WHILE) && 
		   (b->type != FOR ) )
	{
		b->skip=1;
		b = b->outer;
	}
	b->skip=1;
	b->loop_status = is_break?LOOP_BREAK:LOOP_CONTINUE;
	return 0;
}

/**
   Function for handling the \c return builtin
*/
static int builtin_return( wchar_t **argv )
{
	int argc = count_args( argv );
	int status = 0;	
	
	block_t *b = current_block;
	
	switch( argc )
	{
		case 1:
			break;
		case 2:
		{
			wchar_t *end;
			errno = 0;			
			status = wcstol(argv[1],&end,10);
			if( errno || *end != 0)
			{
				sb_append2( sb_err, argv[0], L": Argument must be an integer '", argv[1], L"'\n", 0 );
				builtin_print_help( argv[0], sb_err );				
				return 1;
			}
//			fwprintf( stderr, L"Return with status %d\n", status );
			break;			
		}
		default:
			sb_append2( sb_err, 
						argv[0], 
						L": Too many arguments\n", 0 );
		builtin_print_help( argv[0], sb_err );
		return 1;		
	}
	

	while( (b != 0) && 
		   ( b->type != FUNCTION_CALL)  )
	{
		b = b->outer;
	}
	
	if( b == 0 )
	{
		sb_append2( sb_err, 
					argv[0], 
					L": Not inside of function\n", 0 );
		builtin_print_help( argv[0], sb_err );
		return 1;
	}
	
	b = current_block;
	while( ( b->type != FUNCTION_CALL))
	{
		b->skip=1;
		b = b->outer;
	}
	b->skip=1;
//	proc_set_last_status( status );
	
	return status;
}

/**
   Builtin for executing one of several blocks of commands depending on the value of an argument.
*/
static int builtin_switch( wchar_t **argv )
{
	int res=0;	
	int argc = count_args( argv );
	
	if( argc != 2 )
	{
		sb_append( sb_err, L"switch: syntax error, expected exactly one argument to switch\n" );
		builtin_print_help( L"switch", sb_err );
		res=1;
		parser_push_block( FAKE );
	}
	else
	{
		parser_push_block( SWITCH );
		current_block->switch_value = wcsdup( argv[1]);
		current_block->skip=1;
		current_block->switch_taken=0;
	}
	
	return res;
}

/**
   Builtin used together with the switch builtin for conditional execution
*/
static int builtin_case( wchar_t **argv )
{
	int argc = count_args( argv );
	int i;
	
	if( current_block->type != SWITCH )
	{
		sb_append( sb_err, L"case: syntax error, case command while not in switch block\n" );
		builtin_print_help( L"case", sb_err );
		return 1;
	}
		
	current_block->skip = 1;

	if( current_block->switch_taken )
	{
		return 0;
	}
	
	for( i=1; i<argc; i++ )
	{
		if( wildcard_match( current_block->switch_value, argv[i], 0 ) )
		{
			current_block->skip = 0;
			current_block->switch_taken = 1;
			break;		
		}
	}
	return 0;		
}


/*
  END OF BUILTIN COMMANDS
  Below are functions for handling the builtin commands
*/
void builtin_init()
{
	al_init( &io_stack );
	hash_init( &builtin, &hash_wcs_func, &hash_wcs_cmp );
	hash_put( &builtin, L"exit", &builtin_exit );	
	hash_put( &builtin, L"builtin", &builtin_builtin );	
	hash_put( &builtin, L"cd", &builtin_cd );	
	hash_put( &builtin, L"function", &builtin_function );	
	hash_put( &builtin, L"functions", &builtin_functions );	
	hash_put( &builtin, L"complete", &builtin_complete );	
	hash_put( &builtin, L"end", &builtin_end );
	hash_put( &builtin, L"else", &builtin_else );
	hash_put( &builtin, L"eval", &builtin_eval );	
	hash_put( &builtin, L"for", &builtin_for );
	hash_put( &builtin, L".", &builtin_source );
	hash_put( &builtin, L"set", &builtin_set );
	hash_put( &builtin, L"fg", &builtin_fg );
	hash_put( &builtin, L"bg", &builtin_bg );
	hash_put( &builtin, L"jobs", &builtin_jobs );
	hash_put( &builtin, L"read", &builtin_read );
	hash_put( &builtin, L"break", &builtin_break_continue );	
	hash_put( &builtin, L"continue", &builtin_break_continue );
	hash_put( &builtin, L"return", &builtin_return );
	hash_put( &builtin, L"commandline", &builtin_commandline );
	hash_put( &builtin, L"switch", &builtin_switch );	
	hash_put( &builtin, L"case", &builtin_case );	
	
	/* 
	   Builtins that are handled directly by the parser. They are
	   bound to a noop function only so that help requests will be
	   honored.
	*/
	hash_put( &builtin, L"command", &builtin_ignore );		
	hash_put( &builtin, L"if", &builtin_ignore );	
	hash_put( &builtin, L"while", &builtin_ignore );	

	/*
	  This is not a builtin, but fish handles it's help display internally.
	*/
	hash_put( &builtin, L"count", &builtin_ignore );	
	
	builtin_help_init();
}

void builtin_destroy()
{
	if( desc )
	{
		hash_destroy( desc );	
		free( desc );
	}
	
	al_destroy( &io_stack );
	hash_destroy( &builtin );
	builtin_help_destroy();
}

int builtin_exists( wchar_t *cmd )
{
	/*
	  Count is not a builtin, but it's help is handled internally by
	  fish, so it is in the hash_table_t.
	*/
	if( wcscmp( cmd, L"count" )==0)
		return 0;
	
	return (hash_get(&builtin, cmd) != 0 );
}

/**
   Return true if the specified builtin should handle it's own help,
   false otherwise.
*/
static int internal_help( wchar_t *cmd )
{
	if( wcscmp( cmd, L"for" ) == 0 ||
		wcscmp( cmd, L"while" ) == 0 ||
		wcscmp( cmd, L"function" ) == 0 ||
		wcscmp( cmd, L"if" ) == 0 ||
		wcscmp( cmd, L"end" ) == 0 ||
		wcscmp( cmd, L"switch" ) == 0 ) 
		return 1;
	return 0;
}

void builtin_exec( wchar_t **argv, int type )
{
	switch( type )
	{
		case BUILTIN_REGULAR:
		{
			int (*cmd)(wchar_t **argv)=0;
			cmd = hash_get( &builtin, argv[0] );
			
			if( argv[1] != 0 && !internal_help(argv[0]) )
			{
				if( argv[2] == 0 && (parser_is_help( argv[1], 0 ) ) )
				{
					builtin_print_help( argv[0], sb_out );
					return;
				}
			}
	
			if( cmd != 0 )
			{
				int status;
				
				status = cmd(argv);
//				fwprintf( stderr, L"Builtin: Set status of %ls to %d\n", argv[0], status );
				
				proc_set_last_status( status );		
			}
			else
			{
				sb_append2( sb_err, L"fish: Unknown builtin: ", argv[0], 0 );	
			}
		}
		break;
		case BUILTIN_FUNCTION:
		{
			wchar_t **arg;
			int i;
			string_buffer_t sb;

			buffer_t output;
			
			
			const wchar_t * def = function_get_definition( argv[0] );
			if( def == 0 )
			{
				fwprintf( stderr, L"Unknown function %ls\n", argv[0] );
				sanity_lose();
				break;
			}
			env_push(1);
			
				
			if( count_args(argv)>1 )
			{
				sb_init( &sb );
				
				for( i=1,arg = argv+1; *arg; i++, arg++ )
				{
					if( i != 1 )
						sb_append( &sb, ARRAY_SEP_STR );
					sb_append( &sb, *arg );
				}

				env_set( L"argv", (wchar_t *)sb.buff, 0 );
				sb_destroy( &sb );
			}
			parser_forbid_function( argv[0] );

			proc_set_last_status( 0 );		
			if( builtin_out_redirect )
			{
				
				b_init( &output );
				
				eval( def, &output, FUNCTION_CALL );
				
				i=0;			
				b_append( &output, &i, 1 );
/*			fwprintf( stderr, 
  L"Read %d bytes: %s\n", 
  output.used, 
  output.buff );*/
				wchar_t *tmp = str2wcs(output.buff);
				if( tmp )
				{
					sb_append( sb_out, tmp );
					free( tmp );
				}				
				b_destroy( &output );
			}
			else
			{
				eval( def, 0, FUNCTION_CALL );				
			}
			parser_allow_function();
			env_pop();
			
		}
		break;
		default:
			fwprintf( stderr, L"Called builtin_exec with invalid type %d\n", type );
			sanity_lose();
			break;
	}	
}

void builtin_get_names( array_list_t *list )
{
 	hash_get_keys( &builtin, list );
}

const wchar_t *builtin_get_desc( const wchar_t *b )
{
	
	if( !desc )
	{
		desc = malloc( sizeof( hash_table_t ) );
		if( !desc) 
			return 0;
		
		hash_init( desc, &hash_wcs_func, &hash_wcs_cmp );

		hash_put( desc, L"exit", L"Exit the shell" );	
		hash_put( desc, L"cd", L"Change working directory" );	
		hash_put( desc, L"function", L"Define a new function" );	
		hash_put( desc, L"functions", L"List or remove functions" );	
		hash_put( desc, L"complete", L"Edit command specific completions" );	
		hash_put( desc, L"end", L"End a block of commands" );
		hash_put( desc, L"else", L"Evaluate block if condition is false" );
		hash_put( desc, L"eval", L"Evaluate parameters as a command" );	
		hash_put( desc, L"for", L"Perform a set of commands multiple times" );
		hash_put( desc, L".", L"Evaluate contents of file" );
		hash_put( desc, L"set", L"Handle environment variables" );
		hash_put( desc, L"fg", L"Send job to foreground" );
		hash_put( desc, L"bg", L"Send job to background" );
		hash_put( desc, L"jobs", L"Print currently running jobs" );
		hash_put( desc, L"read", L"Read a line of input into variables" );
		hash_put( desc, L"break", L"Stop the innermost currently evaluated loop" );	
		hash_put( desc, L"continue", L"Skip the rest of the current lap of the innermost currently evaluated loop" );
		hash_put( desc, L"return", L"Stop the innermost currently evaluated function" );
		hash_put( desc, L"commandline", L"Set the commandline" );
		hash_put( desc, L"switch", L"Conditionally execute a block of commands" );	
		hash_put( desc, L"case", L"Conditionally execute a block of commands" );	
		hash_put( desc, L"builtin", L"Run a builtin command" );	
		hash_put( desc, L"command", L"Run a program" );		
		hash_put( desc, L"if", L"Conditionally execute a command" );	
		hash_put( desc, L"while", L"Perform a command multiple times" );	
	}

	return hash_get( desc, b );	
}


void builtin_push_io( FILE *bi)
{
	if( builtin_stdin != 0 )
	{
		al_push( &io_stack, builtin_stdin );
		al_push( &io_stack, sb_out );
		al_push( &io_stack, sb_err );		
	}
	builtin_stdin = bi;
	sb_out = malloc(sizeof(string_buffer_t));
	sb_err = malloc(sizeof(string_buffer_t));
	sb_init( sb_out );
	sb_init( sb_err );	
}

void builtin_pop_io()
{
	builtin_stdin = 0;
	sb_destroy( sb_out );
	sb_destroy( sb_err );
	free( sb_out);
	free(sb_err);
	
	if( al_get_count( &io_stack ) >0 )
	{
		sb_err = (string_buffer_t *)al_pop( &io_stack );
		sb_out = (string_buffer_t *)al_pop( &io_stack );
		builtin_stdin = (FILE *)al_pop( &io_stack );
	}
	else
	{
		sb_out = sb_err = 0;
		builtin_stdin = 0;
	}
}

