/** \file common.h
	Prototypes for various functions, mostly string utilities, that are used by most parts of fish.
*/

/**
   Under curses, tputs expects an int (*func)(char) as its last parameter, but in ncurses, tputs expects a int (*func)(int) as its last parameter. tputs_arg_t is defined to always be what tputs expects. Hopefully.
*/

#ifdef NCURSES_VERSION
typedef int tputs_arg_t;
#else
typedef char tputs_arg_t;
#endif

/** Save the shell mode on startup so we can restore them on exit */
extern struct termios shell_modes;      

/**
   Take an array_list_t containing wide strings and converts them to a wchar_t **.
*/

wchar_t **list_to_char_arr( array_list_t *l );

/**
   Read a line from the stream f into the buffer buff of length len. If
   buff is to small, it will be reallocated, and both buff and len will
   be updated to reflect this. Returns the number of bytes read or -1
   on failiure. 

   If the carriage return character is encountered, it is
   ignored. fgetws() considers the line to end if reading the file
   results in either a newline (L'\n') character, the null (L'\\0')
   character or the end of file (WEOF) character.
*/
int fgetws2( wchar_t **buff, int *len, FILE *f );

/**
   Sorts a list of wide strings according to the wcsfilecmp-function from the util library
*/
void sort_list( array_list_t *comp );

/**
   Returns a newly allocated wide character string equivalent of the specified multibyte character string
*/
wchar_t *str2wcs( const char *in );

/**
   Returns a newly allocated multibyte character string equivalent of the specified wide character string
*/
char *wcs2str( const wchar_t *in );

/**
   Returns a newly allocated wide character string array equivalent of the specified multibyte character string array
*/
char **wcsv2strv( const wchar_t **in );

/**
   Returns a newly allocated multibyte character string array equivalent of the specified wide character string array
*/
wchar_t **strv2wcsv( const char **in );

/**
   Returns a newly allocated concatenation of the specified wide character strings
*/
wchar_t *wcsdupcat( const wchar_t *a, const wchar_t *b );

/**
   Returns a newly allocated concatenation of the specified wide character strings. The last argument must be a null pointer.
*/
wchar_t *wcsdupcat2( const wchar_t *a, ... );

/**
   Returns a newly allocated wide character string wich is a copy of the string in, but of length c or shorter. The returned string is always null terminated, and the null is not included in the string length.
*/
wchar_t *wcsndup( const wchar_t *in, int c );

/**
   Convert a wide character string to a number in the specified
   base. This functions is the wide character string equivalent of
   strtol. For bases of 10 or lower, 0..9 are used to represent
   numbers. For bases below 36, a-z and A-Z are used to represent
   numbers higher than 9. Higher bases than 36 are not supported.
*/
long wcstol(const wchar_t *nptr,
			wchar_t **endptr,
			int base);

size_t
wcslcat(wchar_t *dst, const wchar_t *src, size_t siz);

size_t
wcslcpy(wchar_t *dst, const wchar_t *src, size_t siz);

/**
   Create a dublicate string. Wide string version of strdup.
*/
wchar_t *wcsdup(const wchar_t *in);

/**
   Case insensitive string compare function. Wide string version of
   strcasecmp.

   This implementation of wcscasecmp does not take into account
   esoteric locales where uppercase and lowercase do not cleanly
   transform between each other. Hopefully this should be fine since
   fish only uses this function with one of the strings supplied by
   fish and guaranteed to be a sane, english word.
*/
int wcscasecmp( const wchar_t *a, const wchar_t *b );

/**
   Test if the given string is a valid variable name
*/

int wcsvarname( wchar_t *str );

/**
   The prototype for this function is missing in some libc
   implementations. Fish has a fallback implementation in case the
   implementation is missing altogether.
*/
int wcwidth( wchar_t c );


/**
   A wcswidth workalike. Fish uses this since the regular wcswidth seems flaky.
*/
int my_wcswidth( const wchar_t *c );

/**
   This functions returns the end of a quoted substring. It can handle nested single and double quotes.
*/
wchar_t *quote_end( const wchar_t *in );

/**
   A call to this function will reset the error counter. Some
   functions print out non-critical error messages. These should check
   the error_count before, and skip printing the message if
   MAX_ERROR_COUNT messages have been printed. The error_reset()
   should be called after each interactive command executes, to allow
   new messages to be printed.
*/
void error_reset();

