#
# This file defines various functions for fish
#

#
# Make ls use colors if we are on a sytem that supports this
#
if ls --help 1>/dev/null 2>/dev/null
	function ls -d "List contents of directory"
		command ls --color=auto --indicator-style=classify $argv
	end
end

#
# These are very common and useful
#
function ll -d "List contents of directory using long format"
	ls -lh $argv
end

function la -d "List contents of directory using long format, showing hidden files"
	ls -lha $argv
end

#
# Print the current working directory. If it is too long, it will be
# ellipsised. This function is used by the default prompt command.
#

function prompt_pwd -d "Print the current working directory, ellipsise it if it is too long"
	set wd (pwd)
	set len (echo $wd|wc -c)
	set max_width (echo $COLUMNS/4|bc)
	if test $len -gt $max_width
	   	printf '\342\200\246'
		echo $wd|cut -c (echo $len-$max_width-1|bc)-
	else
		echo $wd
	end 
end

#
# This is a neat function, stolen from zsh. It allows you to edit the
# value of a variable interactively.
#

function vared -d "Edit variable value"
	if test (count $argv) = 1
	   	switch $argv
			
			case '-h' '--h' '--he' '--hel' '--help'
				 printf "    Synopsis\n\n\tvared VARIABLE\n\n"
				 printf "\tInteractively edit the value of an environment variable\n\n"

			case '-*'
				 printf "vared: Unknown option %s\n" $argv
			
			case '*'
			   	if test (count $$argv ) -lt 2
			   	   	set init ''
				   	if test $$argv
			   	   	   	set init -- $$argv
					end
					set prompt 'set_color green; echo '$argv'; set_color normal; echo "> "'
					read -p $prompt -g -c $init tmp
					if test -n $$argv
				   	   	set -- $argv $tmp
					else
						set -- -g $argv $tmp
					end	
				else
					printf "vared: %s is an array variable. Use " $argv
					set_color $FISH_COLOR_COMMAND
					printf vared
					set_color $FISH_COLOR_NORMAL
					printf " %s[n] to edit the n:th element of %s\n" $argv $argv
				end
		end
	else
		printf "vared: Expected exactly one argument.\n\nSynopsis:\n\t"
		set_color $FISH_COLOR_COMMAND
		printf vared
		set_color $FISH_COLOR_NORMAL
		printf " VARIABLE_NAME\n"
	end
end

#
# This function is bound to ^R, it is used to replace a string in the
# current commandline
#

function replace_commandline -d "Search and replace string in commandline"
	read --prompt "set_color green; echo replace; set_color normal; echo ': '" old
	if test $old
		read --prompt "set_color green; echo with; set_color normal; echo ':	'" new
		if test $new
			commandline (echo $history[1]|sed -e s/$old/$new/) 2>/dev/null
		end
	end
end

#
# This function is bound to Alt-L, it is used to list the contents of
# the directory under the cursor
#

function ls_on_current_token -d "List contents of token under the cursor if it is a directory, otherwise list the contents of the current directory"
	set val (eval echo $end_str)
	if test -d $val
	   	ls $val
	else
		set dir (dirname $val)
		if test $dir != . -a -d $dir
		   	ls $dir
		else
			ls
		end
	end
end

#
# This allows us to use 'open FILENAME' to open a given file in the default 
# application for the file.
#

function open -d "Open file in default application"
	mimedb -l -- $argv
end

#
# Make pwd print out the home directory as a tilde.
#

function pwd -d "Print working directory"
	set out (command pwd $argv)
	if echo $out| grep \^$HOME >/dev/null
		printf \~
		echo $out |cut -b (echo $HOME|wc -c)-
	else
		printf "%s\n" $out
	end
end

#
# help should use 'open' to find a suitable browser, but only 
# if there is a mime database _and_ DISPLAY is set, since the 
# browser will most likely be graphical. Since most systems which
# have a mime databe also have the htmlview program, this is mostly a
# theoretical problem.
#

function help -d "Show help for the fish shell"

	# Declare variables to set correct scope

	set fish_browser 
	set fish_browser_bg 

	#
	# Find a suitable browser for viewing the help pages.  This is needed
	# by the help function defined below.
	#

	set graphical_browsers htmlview x-www-browser firefox galeon mozilla konqueror epiphany opera netscape
	set text_browsers htmlview www-browser links elinks lynx w3m

	if test $BROWSER

	   	# User has manualy set a preferred browser, so we respect that
		set fish_browser $BROWSER

		# If browser is known to be graphical, put into background
	   	for i in $graphical_browsers
			if test $i = $BROWSER
				set fish_browser_bg 1
				break
			end
		end

	else

		# Check for a text-based browser.
		for i in $text_browsers
			if which $i 2>/dev/null >/dev/null
			   	set fish_browser $i
				break
			end
		end

		# If we are in a graphical environment, we check if there is a
		# graphical browser to use instead.
		if test (echo $DISPLAY)
		   	for i in $graphical_browsers
				if which $i 2>/dev/null >/dev/null
				   	set fish_browser $i
					set fish_browser_bg 1
					break
				end
			end
		end
	end

	if test -z $fish_browser
	   	printf "help: Could not find a web browser.\nPlease set the variable $BROWSER to a suitable browser and try again\n\n" 
		return 1
	end

	set fish_help_item $argv[1]
	set fish_help_page index.html

	if test $fish_help_item = .
		set fish_help_page "builtins.html\#source"
	end

	if test $fish_help_item = difference
		set fish_help_page difference.html
	end

	if test $fish_help_item = globbing
		set fish_help_page "index.html\#expand"
	end

	for i in (builtin -n)
		if test $i = $fish_help_item
			set fish_help_page "builtins.html\#"$i
		end
	end

	for i in syntax completion editor job-control todo bugs history killring help color prompt title expand variables builtin-overview changes
		if test $i = $fish_help_item
			set fish_help_page "index.html\#"$i
		end
	end

    if test $fish_browser_bg
		eval $fish_browser file://$__fish_help_dir/$fish_help_page \&
	else
		eval $fish_browser file://$__fish_help_dir/$fish_help_page
	end

end
