/** \file input.c

Functions for reading a character of input from stdin, using the
inputrc information for key bindings.

The inputrc file format was invented for the readline library. The
implementation in fish is as of yet incomplete.

*/

#include "config.h"

#include <wchar.h>

/* 
   On my system, a weird compiler error is issued if this file is
   included further down. Haven't had the time to check what really
   happens...
*/

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <termios.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/ioctl.h>
#include <unistd.h>
#if HAVE_NCURSES_H
#include <ncurses.h>
#else
#include <curses.h>
#endif


#if HAVE_TERMIO_H
#include <termio.h>
#endif

#include <term.h>
#include <signal.h>
#include <dirent.h>
#include <wctype.h>

#include "util.h"
#include "wutil.h"
#include "reader.h"
#include "proc.h"
#include "common.h"
#include "sanity.h"
#include "input.h"
#include "parser.h"
#include "env.h"
#include "expand.h"

/**
   Time in milliseconds to wait for another byte to be available for
   reading after \e is read before assuming that escape key was
   pressed, and not an escape sequence.
*/
#define WAIT_ON_ESCAPE 10

static void input_read_inputrc( wchar_t *fn );

/**
   Array containing characters which have been peeked by the escape
   sequence matching functions and returned
*/

typedef struct
{
	wchar_t *seq; /**< Character sequence which generates this event */
	wchar_t *seq_desc; /**< Description of the character sequence suitable for printing on-screen */
	int type; /**< Type of mapping */
	union
	{
		wchar_t code; /**< character or function generated by this mapping */
		wchar_t *command; /**< command that should be evaluated by this mapping */
	}
	;
	
}
mapping;

/**
   Different kinds of mappings for the mapping datatype.
*/
enum mapping_type
{
	MAPPING_FUNCTION, /**< Mapping to a inputrc function */
	MAPPING_SHELL_COMMAND /**< Mapping to a fish command */
}
;

/**
   Names of all the readline functions supported
*/
const wchar_t *name_arr[] = 
{
	L"beginning-of-line",
	L"end-of-line",
	L"forward-char",
	L"backward-char",
	L"forward-word",
	L"backward-word",
	L"history-search-backward",
	L"history-search-forward",
	L"delete-char",
	L"backward-delete-char",
	L"kill-line",
	L"yank",
	L"yank-pop",
	L"explain",
	L"complete",
	L"beginning-of-history",
	L"end-of-history",
	L"delete-line",
	L"backward-kill-line",
	L"kill-whole-line",
	L"kill-word",
	L"backward-kill-word",
	L"dump-functions",
	L"clear-screen",
	L"exit"
}
;

/**
   Description of each supported readline function
*/
const wchar_t *desc_arr[] =
{
	L"Move to beginning of line",
	L"Move to end of line",
	L"Move forward one character",
	L"Move backward one character",
	L"Move forward one word",
	L"Move backward one word",
	L"Search backward through list of previous commands",
	L"Search forward through list of previous commands",
	L"Delete one character forward",
	L"Delete one character backward",
	L"Move contents from cursor to end of line to killring",
	L"Paste contents of killring",
	L"Rotate to previous killring entry",
	L"Explain possible errors in commandline",
	L"Guess the rest of the next input token",
	L"Move to first item of history",
	L"Move to last item of history",
	L"Clear current line",
	L"Move contents from beginning of line to cursor to killring",
	L"Move entire line to killring",
	L"Move next word to killring",
	L"Move previous word to killring",
	L"Write out keybindings",
	L"Clear entire screen",
	L"Quit the running program"
}
;

/**
   Internal code for each supported readline function
*/
const wchar_t code_arr[] = 
{
	R_BEGINING_OF_LINE,
	R_END_OF_LINE,
	R_FORWARD_CHAR,
	R_BACKWARD_CHAR,
	R_FORWARD_WORD,
	R_BACKWARD_WORD,
	R_HISTORY_SEARCH_BACKWARD,
	R_HISTORY_SEARCH_FORWARD,
	R_DELETE_CHAR,
	R_BACKWARD_DELETE_CHAR,
	R_KILL_LINE,
	R_YANK,
	R_YANK_POP,
	R_EXPLAIN,
	R_COMPLETE,
	R_BEGINNING_OF_HISTORY,
	R_END_OF_HISTORY,
	R_DELETE_LINE,
	R_BACKWARD_KILL_LINE,
	R_KILL_WHOLE_LINE,
	R_KILL_WORD,
	R_BACKWARD_KILL_WORD,
	R_DUMP_FUNCTIONS,
	R_CLEAR_SCREEN,
	R_EXIT
}
;


/**
   List of all key bindings, as mappings from one sequence to either a character or a command
*/
static array_list_t mappings;

/**
   Characters that have been read and returned by the sequence matching code
*/
static wint_t lookahead_arr[32];

/**
   Number of entries in lookahead_arr
*/
static int lookahead_count = 0;

/**
   Number of nested conditional statement levels that are not evaluated
*/
static int inputrc_skip_block_count=0;
/**
   Number of nested conditional statements that have evaluated to true
*/
static int inputrc_block_count=0;

/**
   True if syntax errors where found in the inputrc file
*/
static int inputrc_error = 0;

/**
   Returns the function for the given function name.
*/
static wchar_t input_get_code( wchar_t *name )
{

	int i;
	for( i = 0; i<(sizeof( code_arr )/sizeof(wchar_t)) ; i++ )
	{
		if( wcscmp( name, name_arr[i] ) == 0 )
		{
			return code_arr[i];
		}
	}
	return 0;		
}

/**
   Returns the function name for the given function code.
*/
static const wchar_t *input_get_name( wchar_t c )
{

	int i;
	for( i = 0; i<(sizeof( code_arr )/sizeof(wchar_t)) ; i++ )
	{
		if( c == code_arr[i] )
		{
			return name_arr[i];
		}
	}
	return 0;		
}

/**
   Returns the function description for the given function code.
*/
static const wchar_t *input_get_desc( wchar_t c )
{

	int i;
	for( i = 0; i<(sizeof( code_arr )/sizeof(wchar_t)) ; i++ )
	{
		if( c == code_arr[i] )
		{
			return desc_arr[i];
		}
	}
	return 0;		
}

void add_mapping_function( wchar_t *s, wchar_t * d, wint_t c )
{
	int i;

	if( s == 0 )
		return;
	
	for( i=0; i<al_get_count( &mappings); i++ )
	{
		mapping *m = (mapping *)al_get( &mappings, i );
		if( wcscmp( m->seq, s ) == 0 )
		{
			if( m->type == MAPPING_SHELL_COMMAND )
				free( m->command );
			free( m->seq_desc );
			
			m->seq_desc = wcsdup(d );
			m->type=MAPPING_FUNCTION;			
			m->code=c;
			return;
		}
	}
	
	mapping *m = malloc( sizeof( mapping ) );
	m->seq = wcsdup( s );
	m->seq_desc = wcsdup(d );
	m->type=MAPPING_FUNCTION;			
	m->code=c;
	al_push( &mappings, m );
}

void add_mapping_command( wchar_t *s, wchar_t *d, wchar_t *c )
{
	int i;
	
	if( s == 0 )
		return;
	
	for( i=0; i<al_get_count( &mappings); i++ )
	{
		mapping *m = (mapping *)al_get( &mappings, i );
		if( wcscmp( m->seq, s ) == 0 )
		{
			if( m->type == MAPPING_SHELL_COMMAND )
				free( m->command );
			free( m->seq_desc );

			m->seq_desc = wcsdup(d );
			m->type=MAPPING_SHELL_COMMAND;			
			m->command=wcsdup(c);
			return;
		}
	}
	
	mapping *m = malloc( sizeof( mapping ) );
	m->seq = wcsdup( s );
	m->seq_desc = wcsdup(d );
	m->type=MAPPING_SHELL_COMMAND;			
	m->command=wcsdup(c);
	al_push( &mappings, m );	
}

/**
   Compare sort order for two keyboard mappings. This function is made
   to be suitable for use with the qsort method. 
*/
static int mapping_compare( const void *a, const void *b )
{
	mapping *c = *(mapping **)a;
	mapping *d = *(mapping **)b;

//	fwprintf( stderr, L"%ls %ls\n", c->seq_desc, d->seq_desc );

	return wcscmp( c->seq_desc, d->seq_desc );
	
}



/**
   Print a listing of all keybindings and a description of each
   function. This is used by the dump-functions readline function.
*/
static void dump_functions()
{
	int i;
	fwprintf( stdout, L"\n" );

	qsort(mappings.arr, 
		  al_get_count( &mappings), 
		  sizeof( void*),
		  &mapping_compare );
	
	for( i=0; i<al_get_count( &mappings ); i++ )
	{
		mapping *m = (mapping *)al_get( &mappings, i );
		
		if( m->type == MAPPING_SHELL_COMMAND )
		{
//			write_sequence( m->seq );
			fwprintf( stdout, 
					  L"%ls: evaluate-command %ls (Run the specified shell command)\n", 
					  m->seq_desc, 
					  m->command );
		}
		else
		{
//			write_sequence( m->seq );
			fwprintf( stdout,
					  L"%ls: %ls (%ls)\n", 
					  m->seq_desc,
					  input_get_name(m->code),
					  input_get_desc( m->code ) );
		}
		
	}
	repaint();
}


/**
   Unescape special character from the specified inputrc-style key sequence. 

   \\C-a is expanded to 1, etc.
*/
static wchar_t *input_expand_sequence( wchar_t *in )
{
	wchar_t *res = malloc( sizeof( wchar_t)*(4*wcslen(in)+1));
	wchar_t *out=res;
	int error = 0;
	
	while( *in && !error)
	{
		switch( *in )
		{
			case L'\\':
			{
				in++;
//				fwprintf( stderr, L"Backslash\n" );
				switch( *in )
				{
					case L'\0':
						error = 1;
						break;
						
					case L'e':
						*(out++)=L'\e';
						break;
						
					case L'\\':
					case L'\"':
					case L'\'':
						*(out++)=*in;
						break;

					case L'b':
						*(out++)=L'\b';
						break;
						
					case L'd':
					{
						wchar_t *str = str2wcs( key_dc );
						wchar_t *p=str;
						if( p )
						{
							while( *p )
							{
								*(out++)=*(p++);
							}
							free( str );
						}
						break;
					}
					
					case L'f':
						*(out++)=L'\f';
						break;
						
					case L'n':
						*(out++)=L'\n';
						break;
						
					case L'r':
						*(out++)=L'\r';
						break;
						
					case L't':
						*(out++)=L'\t';
						break;
						
					case L'v':
						*(out++)=L'\v';
						break;
						
					case L'C':
					{
						//fwprintf( stderr, L"Control\n" );
						in++;
						if( *in != L'-' )
						{
							error=1;
//							fwprintf( stderr, L"no dash\n" );
							break;
						}
						in++;
						
						if( (*in >= L'a') && 
							(*in <= L'z') )
						{
							*(out++)=*in-L'a'+1;
							break;
						}
						
						if( (*in >= L'A') && 
							(*in <= L'Z') )
						{
							*(out++)=*in-L'A'+1;
							break;
						}
//						fwprintf( stderr, L"No char after\n" );
						error = 1;
												
						break;
					}
					
					case L'M':
					{
						in++;
						if( *in != L'-' )
						{
							error=1;
//							fwprintf( stderr, L"no dash\n" );
							break;
						}
						in++;
						
						if( (*in >= L'a') && 
							(*in <= L'z') )
						{
							*(out++)=L'\e';
							*(out++)=*in;
							break;
						}
						
						if( (*in >= L'A') && 
							(*in <= L'Z') )
						{
							*(out++)=L'\e';
							*(out++)=*in;
							break;
						}
//						fwprintf( stderr, L"No char after\n" );
						error = 1;
												
						break;
					}
				
					default:
					{
						*(out++)=*in;
						break;
					}
					
				}
				
				break;
			}
			default:
			{
				*(out++)=*in;
				break;
			}
		}
		in++;
	}
	
	if( error )
	{
//		fwprintf( stderr, L"%ls had errors\n", in_orig );
		free( res);
		res=0;
	}
	else
	{
//		fwprintf( stderr, L"%ls translated ok\n", in_orig );
		*out = L'\0';
	}
	
	return res;
}


/**
   Parse a single line of inputrc information. 
*/
static void input_parse_inputrc_line( wchar_t *cmd )
{
	wchar_t *p=cmd;
	
	/* Make all whitespace into space characters */
	while( *p )
	{
		if( *p == L'\t' || *p == L'\r' )
			*p=L' ';
		p++;
	}

	/* Remove spaces at beginning/end */
	while( *cmd == L' ' )
		cmd++;
	
	p = cmd + wcslen(cmd)-1;
	while( (p >= cmd) && (*p == L' ') )
	{
		*p=L'\0';
		p--;
	}
		
	/* Skip comments */
	if( *cmd == L'#' )
		return;

	/* Skip empty lines */
	if( *cmd == L'\0' )
		return;
	
	if( wcscmp( L"$endif", cmd) == 0 )
	{
		if( inputrc_skip_block_count )
		{
			inputrc_skip_block_count--;
/*
  if( !inputrc_skip_block_count )
  fwprintf( stderr, L"Stop skipping\n" );
  else
  fwprintf( stderr, L"Decrease skipping\n" );
*/
		}
		else
		{
			if( inputrc_block_count )
			{
				inputrc_block_count--;
//				fwprintf( stderr, L"End of active block\n" );
			}
			else
			{
				inputrc_error = 1;
				fwprintf( stderr, 
						  L"fish: Mismatched $endif in inputrc file\n" );
			}
		}					
		return;
	}

	if( wcscmp( L"$else", cmd) == 0 )
	{
		if( inputrc_skip_block_count )
		{
			if( inputrc_skip_block_count == 1 )
			{
				inputrc_skip_block_count--;
				inputrc_block_count++;
			}
			
		}
		else
		{
			inputrc_skip_block_count++;
			inputrc_block_count--;
		}
		
		return;
	}
	
	if( inputrc_skip_block_count )
	{
		if( wcsncmp( L"$if ", cmd, wcslen( L"$if " )) == 0 )
			inputrc_skip_block_count++;
//		fwprintf( stderr, L"Skip %ls\n", cmd );
		
		return;
	}
	
	if( *cmd == L'\"' )
	{

		wchar_t *key;
		wchar_t *val;
		wchar_t *sequence;
		wchar_t prev=0;
		

		cmd++;
		key=cmd;
		
		for( prev=0; ;prev=*cmd,cmd++ )
		{
			if( !*cmd )
			{
				fwprintf( stderr, 
						  L"fish: Mismatched quote\n" );
				inputrc_error = 1;
				return;
			}
			
			if(( *cmd == L'\"' ) && prev != L'\\' )
				break;
			
		}
		*cmd=0;
		cmd++;
		if( *cmd != L':' )
		{
			fwprintf( stderr, 
					  L"fish: Expected a \':\'\n" );
			inputrc_error = 1;
			return;
		}
		cmd++;
		while( *cmd == L' ' )
			cmd++;
		
		val = cmd;
		
		sequence = input_expand_sequence( key );
		if( wcsncmp(val, L"evaluate-command ", wcslen(  L"evaluate-command " ) )==0)
		{
			val += wcslen(  L"evaluate-command " );
			add_mapping_command( sequence, key, val );
		}
		else
		{
//			fwprintf( stderr, 
//					  L"code is %ls\n", val );
			
			wchar_t code = input_get_code( val );
			if( code )
				add_mapping_function( sequence, key, code );
		}
		free( sequence );
				
		//fwprintf( stderr, L"Remainder \'%ls\', endchar %d\n", cmd, *cmd );
		
		//fwprintf( stderr, L"%ls -> %ls\n", key, val );
				
		return;	
	}
	else if( wcsncmp( L"$include ", cmd, wcslen(L"$include ") ) == 0 )
	{
		wchar_t *tmp;
		
		cmd += wcslen( L"$include ");
		while( *cmd == L' ' )
			cmd++;		
		tmp=wcsdup(cmd);
		tmp = expand_tilde(tmp);
		if( tmp )
			input_read_inputrc( tmp );			
		free(tmp);		
		return;
	}
	else if( wcsncmp( L"set ", cmd, wcslen( L"set " ) ) == 0 )
    {
		/* 
		   We ignore all variables, so don't bother parsing assignments
		*/
		return;
    }
	else if( wcsncmp( L"$if ", cmd, wcslen( L"$if " )) == 0 )
	{
		wchar_t *term_line = wcsdupcat( L"term=", env_get( L"TERM" ) );
		wchar_t *term_line2 = wcsdup( term_line );
		wchar_t *mode_line = L"mode=emacs";
		wchar_t *app_line = L"fish";
		
		wchar_t *term_line2_end = wcschr( term_line2, L'-' );
		if( term_line2_end )
			*term_line2_end=0;
		
		
		cmd += wcslen( L"$if ");
		while( *cmd == L' ' )
			cmd++;		
		
		if( (wcscmp( cmd, app_line )==0) || 
			(wcscmp( cmd, term_line )==0) || 
			(wcscmp( cmd, mode_line )==0) )
		{
//			fwprintf( stderr, L"Conditional %ls is true\n", cmd );
			inputrc_block_count++;
		}
		else
		{
//			fwprintf( stderr, L"Conditional %ls is false\n", cmd );
			inputrc_skip_block_count++;
		}
		free( term_line );
		free( term_line2 );
		
		return;		
	}
	fwprintf( stderr, L"I don\'t know what %ls means\n", cmd );	
}

/**
   Read the specified inputrc file
*/
static void input_read_inputrc( wchar_t *fn )
{
	FILE *rc;
	wchar_t *buff=0;
	int buff_len=0;
	int error=0;
//	fwprintf( stderr, L"read %ls\n", fn );
		
	rc = wfopen( fn, "r" );
	if( rc )
	{	
		while( !feof( rc ) && (!error))
		{
			switch( fgetws2( &buff, &buff_len, rc ) )
			{
				case -1:
				{
					fwprintf( stderr, L"Error while reading input information from file: %s\n",
							  fn );
						
					wperror( L"fgetws2 (read_ni)" );
					error=1;
					break;
				}
					
				default:
				{
					input_parse_inputrc_line( buff );
					
					if( inputrc_error )
					{
						fwprintf( stderr, L"%ls\n", buff );
						error=1;
					}
				}
			}
		}
		free( buff );
		fclose( rc );
	}
	inputrc_skip_block_count=0;
	inputrc_block_count=0;	
}

/**
   Add a char * based character string mapping.
*/
static void add_terminfo_mapping( char *seq, wchar_t *desc, int func )
{
	if( seq )
	{
		wchar_t *tmp;
		tmp=str2wcs(seq);
		if( tmp )
		{
			add_mapping_function( tmp, desc, func );
			free(tmp);
		}
	}
	
}

int input_init()
{
	wchar_t *fn;
	wchar_t *tmp;
	
	if( setupterm( 0, STDOUT_FILENO, 0) == ERR )
	{
		fwprintf( stderr, L"Could not set up terminal\n" );
		exit(1);
	}
	al_init( &mappings );
	
	/* 
	   Add the default key bindings.

	   Maybe some/most of these should be moved to the keybindings file?
	*/

	/*
	   Many terminals (xterm, screen, etc.) have two different valid escape
	   sequences for arrow keys. One which is defined in terminfo/termcap
	   and one which is actually emitted by the arrow keys. The logic
	   escapes me, but I put in these hardcodes here for that reason.
	*/	

	add_mapping_function( L"\e[A", L"Up", R_HISTORY_SEARCH_BACKWARD );
	add_mapping_function( L"\e[B", L"Down", R_HISTORY_SEARCH_FORWARD );
	add_mapping_function( L"\e[C", L"Right", R_FORWARD_CHAR );
	add_mapping_function( L"\e[D", L"Left", R_BACKWARD_CHAR );

	add_terminfo_mapping( (key_left), L"Left", R_BACKWARD_CHAR );
	
	add_terminfo_mapping( (key_right), L"Right", R_FORWARD_CHAR );
	
	add_terminfo_mapping( (key_up), L"Up", R_HISTORY_SEARCH_BACKWARD );
	
	add_terminfo_mapping( (key_down), L"Down", R_HISTORY_SEARCH_FORWARD );
	
	add_terminfo_mapping( (key_dc), L"Delete", R_DELETE_CHAR );
			
	add_terminfo_mapping( (key_backspace), L"Backspace", R_BACKWARD_DELETE_CHAR );
	
	add_mapping_function( L"\x7f", L"Backspace", R_BACKWARD_DELETE_CHAR );
	
	add_terminfo_mapping( (key_home), L"Home", R_BEGINING_OF_LINE );
	
	add_terminfo_mapping( (key_end), L"End", R_END_OF_LINE );
	
//	add_mapping_function( L"\e\eOA", R_ALT_UP );
//	add_mapping_function( L"\e\eOB", R_ALT_DOWN );
	add_mapping_function( L"\e\eOC", L"Alt-Right", R_FORWARD_WORD );
	add_mapping_function( L"\e\eOD", L"Alt-Left", R_BACKWARD_WORD );
	add_mapping_function( L"\eO3C", L"Alt-Right", R_FORWARD_WORD );
	add_mapping_function( L"\eO3D", L"Alt-Left", R_BACKWARD_WORD );
	
	add_mapping_function( L"\t", L"Tab", R_COMPLETE );
	
	add_mapping_function( tmp=input_expand_sequence(L"\\C-a"), L"Control-a", R_BEGINING_OF_LINE );
	free(tmp);	

	add_mapping_function( tmp=input_expand_sequence(L"\\C-e"), L"Control-e", R_END_OF_LINE );
	free(tmp);
	
	add_mapping_function( tmp=input_expand_sequence(L"\\C-k"), L"Control-k", R_KILL_LINE );
	free(tmp);
	
	add_mapping_function( tmp=input_expand_sequence(L"\\C-y"), L"Control-y", R_YANK );
	free(tmp);
	
	add_mapping_function( tmp=input_expand_sequence(L"\\M-y"), L"Alt-y", R_YANK_POP );
	free(tmp);
	
	add_mapping_function( tmp=input_expand_sequence(L"\\C-h"), L"Control-h", R_BACKWARD_DELETE_CHAR );
	free(tmp);
	
	add_mapping_function( tmp=input_expand_sequence(L"\\C-e"), L"Control-e", R_END_OF_LINE );
	free(tmp);

	add_mapping_function( tmp=input_expand_sequence(L"\\C-w"), L"Control-w", R_BACKWARD_KILL_WORD );
	free(tmp);
	
	add_terminfo_mapping( (key_ppage), L"Page Up", R_BEGINNING_OF_HISTORY );
	add_terminfo_mapping( (key_npage), L"Page Down", R_END_OF_HISTORY );
	
	fn = env_get( L"INPUTRC" );
	
	if( !fn )
		fn = L"~/.inputrc";
	
	fn = expand_tilde( wcsdup( fn ));

	if( fn )
	{	
		input_read_inputrc( fn );
		free(fn);
	}
	
	return 1;

}

void input_destroy()
{
	int i;
	
	for( i=0; i<al_get_count( &mappings ); i++ )
	{
		mapping *m = (mapping *)al_get( &mappings, i );
		free( m->seq );
		free( m->seq_desc );
		
		if( m->type == MAPPING_SHELL_COMMAND )
			free( m->command );
		free(m );
	}
		
	al_destroy( &mappings );
	del_curterm( cur_term );
}


/**
   Internal function used by readch2 to read one byte from fd 1. This function should only be called by
   readch2().
*/
static int readb()
{
	char arr[1];

	int do_loop = 0;
	do
	{
		do_loop = 0;			
		switch( read( 0, arr, 1 ))
		{
			case -1:
			{
				if( errno == EINTR )
				{
					if( job_do_notification() )
						repaint();
					if( reader_interupted() )
					{
						return 3;
					}
					do_loop = 1;
					
				}
				else
				{
					wperror( L"read" );
					exit(1);
				}
				break;
			}
			case 0:
				do_loop=1;
				break;
		}
	}
	while( do_loop );
	
	return arr[0];
}

/**
   Internal function used by input_readch to read bytes from stdin until
   enough bytes have been read to convert them to a
   wchar_t. Conversion is done using mbrtowc. If a character has
   previously been read and then 'unread' using unreadch, that
   character is returned. If timed is true, readch2 will wait at most
   WAIT_ON_ESCAPE milliseconds for a character to be available for
   reading before returning with the value WEOF.
*/
static wchar_t readch2( int timed )
{
	if( lookahead_count == 0 )
	{
		if( timed )
		{
			int count;
			fd_set fds;
			struct timeval tm=
				{
					0,
					1000 * WAIT_ON_ESCAPE
				}
			;
			
			FD_ZERO( &fds );
			FD_SET( 0, &fds );
			count = select(1, &fds, 0, 0, &tm);
			
			switch( count )
			{
				case 0:
					return WEOF;
					
				case -1:
					return WEOF;
					break;
				default:
					break;

			}
		}
		
		wchar_t res;
		static mbstate_t state;

		while(1)
		{
			char b = readb();
			
			int sz = mbrtowc( &res, &b, 1, &state );
			
			switch( sz )
			{
				case -1:
					memset (&state, '\0', sizeof (state));
					fwprintf( stderr, L"fish: Illegal input\n" );					
					repaint();
					return readch2( timed );
				case -2:
					break;
				case 0:
					return 0;
				default:
					
					return res;
			}
		}
	}
	else 
	{
		if( !timed )
		{
			while( (lookahead_count >= 0) && (lookahead_arr[lookahead_count-1] == WEOF) )
				lookahead_count--;
			if( lookahead_count == 0 )
				return readch2(0);
		}
		
		return lookahead_arr[--lookahead_count];
	}
}


void input_unreadch( wint_t ch )
{
	lookahead_arr[lookahead_count++] = ch;
}

wint_t input_readch()
{
	
	wint_t c=0;
	int i, j, k;
	
	/*
	  Clear the interrupted flag
	*/
	reader_interupted();

	for( i=0; i<al_get_count( &mappings); i++ )
	{
		mapping *m = (mapping *)al_get( &mappings, i );
		
		if( m->seq != 0 )
		{
			for( j=0; m->seq[j] != L'\0' && 
					 m->seq[j] == (c=readch2( j>0 )); j++ )
				;
			if( m->seq[j] == L'\0' )
			{
				if( m->type == MAPPING_FUNCTION )
				{

					switch( m->code )
					{
						case R_DUMP_FUNCTIONS:
						{
							dump_functions();
							return input_readch();							
						}
						default:
							return m->code;
					}
				}
				else
				{
					/*
					  This key sequence is bound to a command, which
					  is sent to the parser for evaluation.
					*/
					write( 1, "\n", 1 );
					
					parser_check_command( reader_get_buffer(), reader_get_cursor_pos() );
					
					reader_run_command( m->command );
					
					repaint();
					
					/*
					  We still need to return something to the caller, so we recurse.
					*/
					return input_readch();
				}
			}
			else
			{
				input_unreadch(c);
				for(k=j-1; k>=0; k--)
					input_unreadch(m->seq[k]);
			}
		}	
	}
	return readch2( 0 );
}
