/** \file parser.c
	
The fish parser. Contains functions for parsing code. 

*/

#include <stdlib.h>
#include <stdio.h>
#include <wchar.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <errno.h>
#include <fcntl.h>
#include <termios.h>
#include <pwd.h>
#include <dirent.h>
#include <signal.h>

#include "config.h"
#include "util.h"
#include "common.h"
#include "wutil.h"
#include "parser.h"
#include "tokenizer.h"
#include "proc.h"
#include "exec.h"
#include "wildcard.h"
#include "function.h"
#include "builtin.h"
#include "builtin_help.h"
#include "env.h"
#include "expand.h"
#include "reader.h"
#include "sanity.h"

/** Length of the lineinfo string used for describing the current tokenizer position */
#define LINEINFO_SIZE 128

/**
   Maximum number of block levels in code. This is not the same as
   maximum recursion depth, this only has to do with how many block
   levels are legal in the source code, not at evaluation.
*/
#define BLOCK_MAX_COUNT 64

/** Last error code */
int error_code;

/** Position of last error */

static int err_pos; 

/** Description of last error */
static wchar_t err_str[256];

/** Pointer to the current tokenizer */
static tokenizer *current_tokenizer;

/** String for representing the current line */
static wchar_t lineinfo[LINEINFO_SIZE];

/** This is the position of the beginning of the currently parsed command */
static int current_tokenizer_pos;

/** The current innermost block */
block_t *current_block=0;

/** List of called functions, used to help prevent infinite recursion */
static array_list_t forbidden_function;

/**
   String index where the current job started.
*/
static int job_start_pos;


static int parse_job( process_t *p, 
					  job_t *j,
					  array_list_t *tok_stack);


void parser_push_block( int type )
{
	block_t *new = malloc( sizeof( block_t ));
//	fwprintf( stderr, L"Push block %ls\n", bl[type] );
	
	new->outer = current_block;
	new->type = type;
	new->skip=current_block?current_block->skip:0;
	new->loop_status=LOOP_NORMAL;
	new->out_buffer=current_block?current_block->out_buffer:0;
	
	current_block = new;	
	if( (type != FUNCTION_DEF) && (type != FAKE) && (type != TOP) )
		env_push(0);
}


void parser_pop_block()
{
//	fwprintf( stderr, L"Pop block %ls\n", bl[current_block->type] );
	if( (current_block->type != FUNCTION_DEF ) && (current_block->type != FAKE) && (current_block->type != TOP))
		env_pop();
	
	switch( current_block->type)
	{

		case FOR:
		{
			free( current_block->for_variable );
			al_foreach( &current_block->for_vars, (void (*)(const void *))&free );
			al_destroy( &current_block->for_vars );
			break;
		}

		case SWITCH:
		{
			free( current_block->switch_value );
			break;		
		}

		case FUNCTION_DEF:
		{
			free( current_block->function_name );
			free( current_block->function_description );
			break;
		}		
		
	}
	
	block_t *old = current_block;
	current_block = current_block->outer;
	free( old );
}


int parser_subcommand( const wchar_t *cmd )
{
	return 
		(wcscmp( L"command", cmd )==0) || 
		(wcscmp( L"builtin", cmd )==0) ||
		(wcscmp( L"while", cmd )==0) || 
		(wcscmp( L"if", cmd )==0);
}

wchar_t *parser_cdpath_get( wchar_t *dir )
{
	wchar_t *res = 0;

	if( !dir )
		return 0;
	

	if( dir[0] == L'/' )
	{
		struct stat buf;
		if( wstat( dir, &buf ) == 0 )
		{
			if( S_ISDIR(buf.st_mode) )
			{ 
				res = wcsdup( dir );
			}
		}		
	}
	else
	{
		wchar_t *path = env_get(L"CDPATH");

		if( path == 0 )
		{
			path = L".";
		}
		
		wchar_t *path_cpy = wcsdup( path );
		wchar_t *nxt_path = path;
		wchar_t *state;
		wchar_t *whole_path;

		if( (path_cpy==0) )
		{
			sb_append( sb_err, 
					   L"fish: Out of memory in builtin_cd()\n" );
			return 0;
		}
		
		for( nxt_path = wcstok( path_cpy, ARRAY_SEP_STR, &state );
			 nxt_path != 0; 
			 nxt_path = wcstok( 0, ARRAY_SEP_STR, &state) )
		{
			wchar_t *expanded_path = expand_tilde( wcsdup(nxt_path) );

//			fwprintf( stderr, L"woot %ls\n", expanded_path );

			int path_len = wcslen( expanded_path );
			if( path_len == 0 )
			{
				free(expanded_path );
				continue;
			}
			
			whole_path = wcsdupcat2( expanded_path,
									 ( expanded_path[path_len-1] != L'/' )?L"/":L"",
									 dir, 0 );
			
			free(expanded_path );
			
			struct stat buf;
			if( wstat( whole_path, &buf ) == 0 )
			{
				if( S_ISDIR(buf.st_mode) )
				{ 
					res = whole_path;
					break;
				}
			}
			free( whole_path );
						
		}
		free( path_cpy );
	}
	return res;
}


void parser_forbid_function( wchar_t *function )
{
/*
  if( function )
  fwprintf( stderr, L"Forbid %ls\n", function );
*/
	al_push( &forbidden_function, function?wcsdup(function):0 );
}

void parser_allow_function()
{
/*
  if(  al_peek( &forbidden_function) )
  fwprintf( stderr, L"Allow %ls\n", al_peek( &forbidden_function)  );
*/
	free( (void *) al_pop( &forbidden_function ) );	
}

void error( int ec, const wchar_t *str, int p )
{
	error_code = ec;
	wcsncpy( err_str, str, 256 );
	err_pos = p;
/*	fwprintf( stderr, L"fish: %ls\n", err_str );*/
}

/**
   Wrapper for the error function, which sets the error string to "ec 'ea'".
*/
static void error_arg( int ec, const wchar_t *es, const wchar_t *ea, int p )
{
	wchar_t *msg = wcsdupcat2( es, L" \'", ea, L"\'", 0 );
	error( ec, msg, p );
	free(msg);
}

wchar_t *get_filename( const wchar_t *cmd )
{
	wchar_t *path;
	
	if(wcschr( cmd, '/' ) != 0 ) 
	{
		if( waccess( cmd, X_OK )==0 )
		{
			struct stat buff;
			wstat( cmd, &buff );
			if( S_ISREG(buff.st_mode) )
				return wcsdup( cmd );
			else
				return 0;
		}
	}
	else
	{
		path = env_get(L"PATH");
		if( path != 0 )
		{
			/*Allocate string long enough to hold the whole command*/
			wchar_t *new_cmd = malloc( sizeof(wchar_t)*(wcslen(cmd)+wcslen(path)+2) );
			/* We tokenize a copy of the path, since strtok modifies its arguments */
			wchar_t *path_cpy = wcsdup( path );
			wchar_t *nxt_path = path;
			wchar_t *state;
			
			if( (new_cmd==0) || (path_cpy==0) )
			{
				fwprintf( stderr, L"fish: Out of memory in get_filename()\n" );
				return 0;
			}
			
			for( nxt_path = wcstok( path_cpy, ARRAY_SEP_STR, &state );
				 nxt_path != 0; 
				 nxt_path = wcstok( 0, ARRAY_SEP_STR, &state) )
			{
				int path_len = wcslen( nxt_path );
				wcscpy( new_cmd, nxt_path );
				if( new_cmd[path_len-1] != '/' )
				{
					new_cmd[path_len++]='/';
				}
				wcscpy( &new_cmd[path_len], cmd );
				if( waccess( new_cmd, X_OK )==0 )
				{
					struct stat buff;
					if( wstat( new_cmd, &buff )==-1 )
					{
						if( errno != EACCES )
							wperror( L"stat" );
						continue;
					}					
					if( S_ISREG(buff.st_mode) )
					{
						free( path_cpy );
						return new_cmd;
					}
				}
				else
				{
					switch( errno )
					{
						case ENOENT:
						case ENAMETOOLONG:
						case EACCES:
						case ENOTDIR:
							break;
						default:
							wperror( L"access" );
							fwprintf( stderr, L"Error %d for %ls\n", errno, new_cmd );
					}
				}
			}
			free( path_cpy );
			free( new_cmd );
		}	
	}
	
	return 0;
}

void parser_init()
{
	al_init( &forbidden_function );
}

void parser_destroy()
{
	al_destroy( &forbidden_function );
}

static void print_errors()
{
	if( error_code )
	{
		int tmp;
		
		fwprintf( stderr, L"fish: %ls\n", err_str );
		
		tmp = current_tokenizer_pos;
		current_tokenizer_pos = err_pos;
		
		fwprintf( stderr, L"%ls", parser_current_line() );

		current_tokenizer_pos=tmp;
	}		
}

int eval_args( const wchar_t *line, array_list_t *args )
{
	tokenizer tok;
	tokenizer *previous_tokenizer=current_tokenizer;
	tok_init( &tok, line, 0 );
	int do_loop=1;
	int previous_pos=current_tokenizer_pos;
		
	current_tokenizer=&tok;

	error_code=0;
	
	for(;do_loop && tok_has_next( &tok) ; tok_next( &tok ) )
	{
		current_tokenizer_pos = tok_get_pos( &tok );		
		switch(tok_last_type( &tok ) )
		{			 
			case TOK_STRING:
				if( !expand_string( wcsdup(tok_last( &tok )), args, 0 ) )
				{
					err_pos=tok_get_pos( &tok );					
					do_loop=0;
				}
				
				break;
			case TOK_END:
				break;
				
			case TOK_ERROR:
			{
				error_arg( SYNTAX_ERROR, 
						   L"Tokenizer error",
						   tok_last(&tok),
						   tok_get_pos( &tok ) );
				
				do_loop=0;
				break;					
			}

			default:
				error_arg( SYNTAX_ERROR, 
						   L"Unexpected token of type", 
						   tok_get_desc( tok_last_type(&tok)),
						   tok_get_pos( &tok ) );
				do_loop=0;
				break;
					
		}
	}

	print_errors();	
	tok_destroy( &tok );

	current_tokenizer=previous_tokenizer;
	current_tokenizer_pos = previous_pos;

	return 1;
}

wchar_t *parser_current_line()
{
	int lineno=1;
	
	wchar_t *file = reader_current_filename();
	wchar_t *whole_str = tok_string( current_tokenizer );
	wchar_t *line = whole_str;
	wchar_t *line_end;
	int i;
	int offset;
	int current_line_pos=current_tokenizer_pos;

	lineinfo[0]=0;
	
	/*
	  Calculate line number, line offset, etc.
	*/
	for( i=0; i<current_tokenizer_pos; i++ )
	{
		if( whole_str[i] == L'\n' )
		{
			lineno++;
			current_line_pos = current_tokenizer_pos-i-1;			
			line = &whole_str[i+1];			
		}
	}
	
	/*
	  Copy current line from whole string
	*/
	line_end = wcschr( line, L'\n' );
	if( !line_end )
		line_end = line+wcslen(line);

	line = wcsndup( line, line_end-line );
	
//	fwprintf( stderr, L"Current pos %d, line pos %d, file_length %d\n", current_tokenizer_pos,  current_line_pos, wcslen(whole_str));

	if( !shell_is_interactive )
	{
		swprintf( lineinfo,
				  LINEINFO_SIZE, 
				  L"%ls (line %d): %n", 
				  file,
				  lineno,
				  &offset );
	}
	else
	{
		offset=0;
	}	

	/* Skip printing if we are in interactive mode and the error was on the first character of the line */
	if( offset+current_line_pos )
		swprintf( lineinfo+offset, 
				  LINEINFO_SIZE-offset,
				  L"%ls\n%*c^\n", 
				  line, 
				  offset+current_line_pos,
				  L' ' );

	free( line );
	
	return lineinfo;	
}

int parser_get_pos()
{
	return tok_get_pos( current_tokenizer );
}

int parser_get_job_pos()
{
	return job_start_pos;
}


void parser_set_pos( int p)
{
	tok_set_pos( current_tokenizer, p );
}

const wchar_t *parser_get_buffer()
{
	return tok_string( current_tokenizer );
}


int parser_is_help( wchar_t *s, int min_match )
{
	int len = wcslen(s);
	
	return ( wcscmp( L"-h", s ) == 0 ) ||
		( len >= 3 && (wcsncmp( L"--help", s, len ) == 0) );
}


/**
   Parse options for the specified job
*/
static void parse_job_main_loop( process_t *p,
								 job_t *j,
								 array_list_t *tok_stack, 
								 array_list_t *args )
{
	tokenizer *tok = (tokenizer *)al_get( tok_stack,
										  al_get_count(tok_stack)-1);
	int is_finished=0;

	int proc_is_count=0;
	
	/*
	  Test if this is the 'count' command. We need to special case
	  count, since it should display a help message on 'count .h',
	  but not on 'set foo -h; count $foo'.
	*/
	if( p->actual_cmd )
	{
		wchar_t *woot = wcsrchr( p->actual_cmd, L'/' );
		if( !woot )
			woot = p->actual_cmd;
		else
			woot++;
		
		proc_is_count = wcscmp( woot, L"count" )==0;
	}
	
	while(1)
	{
		if( (is_finished) || (error_code != 0) )
			break;
		
		tok_next( tok );
		
		/*fwprintf( stderr, L"Read token %ls\n", wcsdup(tok_last( tok )) );*/
		
		switch( tok_last_type( tok ) )
		{				
			case TOK_PIPE:
				p->argv = list_to_char_arr( args );
				p->next = calloc( 1, sizeof( process_t ) );
				if( p->next == 0 )
				{
					fwprintf( stderr, L"Malloc error\n" );
					exit(1);
				}
				tok_next( tok );
				if( !parse_job( p->next, j, tok_stack ))
				{
					/*
					  Do not free args content on error - it is
					  already in p->argv and will be freed by job_free
					  later on.
					*/
					
					al_truncate( args, 0 );
				}
				is_finished = 1;
				break;

			case TOK_BACKGROUND:
				j->fg = 0;
			case TOK_END:
			{
				int popped=0;
				
				while( (!tok_has_next( tok )) && (al_get_count( tok_stack )>1 ))
				{
					popped = 1;
					al_pop( tok_stack );
					parser_allow_function();
					tok_destroy( tok );
					free(tok);
					tok = (tokenizer *)al_get( tok_stack, 
											   al_get_count(tok_stack)-1);
				}
				
				if( popped )
				{
					continue;
				}
				else
				{
					p->argv = list_to_char_arr( args );
					if( tok_has_next(tok))
						tok_next(tok);
					
					is_finished = 1;
				}
				break;
			}
			
			case TOK_STRING:
			{
				int skip=0;
				
				if( current_block->skip )
				{
					skip=1;
/*					fwprintf(stderr, L"%d == %d, %ls == %ls, %d\n",
  current_block->type, 
  SWITCH, 
  al_get( args, 0),
  L"case" , 
  p->builtin );
*/				
					if( (current_block->type == SWITCH) && 
						(wcscmp( al_get( args, 0), L"case" )==0) && 
						p->builtin )
					{
						skip=0;
					}
					
				}
				

				if( !skip )
				{
					
				
					if( proc_is_count &&
						(al_get_count( args) == 1) && 
						( parser_is_help( tok_last(tok), 0) ) )
					{
						/*
						  Display help for count
						*/
						p->builtin = 1;
						wcscpy( p->actual_cmd, L"count" );					
					}
//				if(foo)
//				fwprintf( stderr, L"expand %ls\n", tok_last( tok ) );
				
					if( !expand_string( wcsdup(tok_last( tok )),
										args, 
										0 )
						)
					{
						err_pos=tok_get_pos( tok );					
						if( error_code == 0 )
						{
						
							error_arg( SYNTAX_ERROR,
									   L"Could not expand string",
									   tok_last(tok),
									   tok_get_pos( tok ) );
						}
					
					}
				}
				
				break;
			}
			
			case TOK_REDIRECT_OUT:
			case TOK_REDIRECT_IN:
			case TOK_REDIRECT_APPEND:
			case TOK_REDIRECT_FD:
			{
				int type = tok_last_type( tok );
				io_data_t *new_io = malloc( sizeof(io_data_t) );
				wchar_t *target = 0;
				
				new_io->fd = wcstol( tok_last( tok ), 
									 0,
									 10 );
				tok_next( tok );
								
				switch( tok_last_type( tok ) )
				{
					case TOK_STRING:
					{
						target = expand_one( wcsdup( tok_last( tok ) ), 0);
						if( target == 0 && error_code == 0 )
						{
							error_arg( SYNTAX_ERROR, 
									   L"Could not expand string",
									   tok_last( tok ),
									   tok_get_pos( tok ) );
						}
					}
					break;
					default:
						error_arg( SYNTAX_ERROR, 
								   L"Expected redirection, got token of type", 
								   tok_get_desc( tok_last_type(tok)),
								   tok_get_pos( tok ) );
				}
				
				if( target == 0 || wcslen( target )==0 )
				{
					if( error_code == 0 )
						error( SYNTAX_ERROR, 
							   L"Invalid IO redirection",
							   tok_get_pos( tok ) );
					tok_next(tok);
					break;
					
				}
				
				
				switch( type )
				{
					case TOK_REDIRECT_APPEND:
						new_io->io_mode = IO_FILE;
						new_io->flags = O_CREAT | O_APPEND | O_WRONLY;
						new_io->filename = target;
						break;
						
					case TOK_REDIRECT_OUT:
						new_io->io_mode = IO_FILE;
						new_io->flags = O_CREAT | O_WRONLY | O_TRUNC;
						new_io->filename = target;						
						break;
						
					case TOK_REDIRECT_IN:
						new_io->io_mode = IO_FILE;
						new_io->flags = O_RDONLY;
						new_io->filename = target;						
						break;
						
					case TOK_REDIRECT_FD:
						if( wcscmp( target, L"-" ) == 0 )
						{
							new_io->io_mode = IO_CLOSE;
						}
						else
						{
							new_io->io_mode = IO_FD;
							new_io->old_fd = wcstol( target, 
													 0,
													 10 );
							if( ( new_io->old_fd < 0 ) || 
								( new_io->old_fd > 10 ) )
							{
								error_arg( SYNTAX_ERROR,
										   L"Requested redirection to something "
										   L"that is not a file descriptor",
										   target,
										   tok_get_pos( tok ) );
								tok_next(tok);								
							}
							free(target);
						}						
						break;
				}

				j->io = io_add( j->io, new_io );
				
			}
			break;

			case TOK_ERROR:
			{
				error_arg( SYNTAX_ERROR, 
						   L"Tokenizer error",
						   tok_last(tok),
						   tok_get_pos( tok ) );
				
				return;			
			}

			default:
				error_arg( SYNTAX_ERROR, 
						   L"Unexpected token of type", 
						   tok_get_desc( tok_last_type(tok)),
						   tok_get_pos( tok ) );
				tok_next(tok);
				break;
		}
	}	
	return;
}


/**
   Fully parse a single job. Does not call exec.
   
   \param p The process structure that should be used to represent the first process in the job.
   \param j The job structure to contain the parsed job
   \param tok_stack list of tokenizers to read from

   \return 1 on success, 0 on error
*/
static int parse_job( process_t *p, 
					  job_t *j,
					  array_list_t *tok_stack )
{
	tokenizer *tok = (tokenizer *)al_get( tok_stack, al_get_count(tok_stack)-1);
	array_list_t args;
	int use_function = 1;
	int use_builtin = 1;

	al_init( &args );

	current_tokenizer_pos = tok_get_pos( tok );

	while( al_get_count( &args ) == 0 )
	{
		wchar_t *nxt;		
		switch( tok_last_type( tok ))
		{
			case TOK_STRING:
				nxt = expand_one( wcsdup(tok_last( tok )),
								  EXPAND_SKIP_SUBSHELL | EXPAND_SKIP_VARIABLES);			
				if( nxt == 0 )
				{
					error_arg( SYNTAX_ERROR, 
							   L"Unable to parse command name ",
							   tok_last( tok ),
							   tok_get_pos( tok ) );
					al_destroy( &args );
					return 0;
				}
				break;

			case TOK_ERROR:
			{
				error_arg( SYNTAX_ERROR, 
						   L"Tokenizer error",
						   tok_last(tok),
						   tok_get_pos( tok ) );
				
				al_destroy( &args );
				return 0;			
			}

			default:
				error_arg( SYNTAX_ERROR, 
						   L"Expected a command name, got token of type ",
						   tok_get_desc( tok_last_type(tok)),
						   tok_get_pos( tok ) );
				al_destroy( &args );
				return 0;
		}
		
		int mark = tok_get_pos( tok );
		
		if( wcscmp( L"command", nxt )==0 )
		{
			tok_next( tok );			
			if( parser_is_help( tok_last( tok ), 0 ) )
			{				
				tok_set_pos( tok, mark);
			}
			else
			{
				use_function = 0;
				use_builtin=0;
				free( nxt );
				continue;			
			}
		}
		else if(  wcscmp( L"builtin", nxt )==0 )
		{
			tok_next( tok );			
			if( tok_last(tok)[0] == L'-' )
			{
				tok_set_pos( tok, mark);
			}
			else
			{
				use_function = 0;
				free( nxt );
				continue;	
			}
		}
		else if( wcscmp( L"while", nxt ) ==0 )
		{
			tok_next( tok );			

/*			if( parser_is_help( tok_last( tok ), 0 ) )
  {				
  tok_set_pos( tok, mark );
  }
  else */
			{
				int new_block = 0;
				
				if( (current_block->type != WHILE) )
				{
					new_block = 1;
				}
				else if( current_block->while_state == WHILE_TEST_AGAIN )
				{
					current_block->while_state = WHILE_TEST_FIRST;
				}
				else
				{
					new_block = 1;
				}
				
				if( new_block )
				{
					parser_push_block( WHILE );				
					current_block->while_state=WHILE_TEST_FIRST;
					current_block->tok_pos = mark;
				}		
				
				free( nxt );
				continue;
			}
		}
		else if( wcscmp( L"if", nxt ) ==0 )
		{
			tok_next( tok );			
			
			parser_push_block( IF );
			
			current_block->if_state=0;
			current_block->tok_pos = mark;
			
			free( nxt );
			continue;
		}
		
		if( use_function)
		{
			int nxt_forbidden;
			wchar_t *forbid;
			forbid = (wchar_t *)(al_get_count( &forbidden_function)?al_peek( &forbidden_function ):0);
			
			nxt_forbidden = forbid && (wcscmp( forbid, nxt) == 0 );
/*            
			  if( forbid )
			  {
			  fwprintf( stderr, L"%ls %ls %d\n", forbid, nxt, current_tokenizer_pos );
			  }
*/	
			
			if( function_exists( nxt ) && !nxt_forbidden)
			{
				
				/* 
				   Check we are in an nxt function, and not in a conditional block. In that case, we do not 
				*/
				if( al_get_count( &forbidden_function ) > 128 )
				{
					error( SYNTAX_ERROR, 
						   L"Recursion limit exceeded",
						   tok_get_pos( tok ) );
				}
				else
				{
					p->builtin = BUILTIN_FUNCTION;
				}
				
			}
		}		
		al_push( &args, nxt );	
	}
	
	if( error_code == 0 )
	{
		if( !p->builtin )
		{
			if( use_builtin && 
				builtin_exists( (wchar_t *)al_get( &args, 0 ) ) )
				p->builtin = BUILTIN_REGULAR;
		}
		
		if( !p->builtin )
		{
			p->actual_cmd = get_filename( (wchar_t *)al_get( &args, 0 ) );
			/* 
			   Check if the specified command exists
			*/
			if( p->actual_cmd == 0 )
			{
				/* If we are not executing the current block, allow non-existent commands */
				if( current_block->skip )
				{
					p->actual_cmd = wcsdup(L"");
				}
				else
				{
					/*
					  That is not a command! Test if it is a directory, in which case, we use 'cd' as the implicit command.
					*/
					wchar_t *pp = 
						parser_cdpath_get( (wchar_t *)al_get( &args, 0 ) );
					if( pp )
					{
						pp = (wchar_t *)al_get( &args, 0 );
						al_truncate( &args, 0 );
						al_push( &args, wcsdup( L"cd" ) );
						al_push( &args, pp );
						p->builtin = BUILTIN_REGULAR;
					}
					else
					{
						error_arg( EVAL_ERROR, 
								   L"Unknown command",
								   (wchar_t *)al_get( &args, 0 ),
								   tok_get_pos( tok ) );
//						fwprintf( stderr, L"\'%ls\'\n", (wchar_t *)al_get( &args, 0 ) );
					}
				}
			}
		}
	}

	parse_job_main_loop( p, j, tok_stack, &args );
	
	if( error_code != 0)
	{
		al_foreach( &args,
					(void (*)(const void *))&free );
		al_destroy( &args );		
	}
	else
	{
		al_destroy( &args );
	}	

	return !error_code;
}

/**
   The specified job is inside of a skipped block. Perform only block
   level commands.
*/
static void skipped_exec( job_t * j )
{
	process_t *p;
	for( p = j->first_process; p; p=p->next )
	{
		if( p->builtin )
		{
			if(( wcscmp( p->argv[0], L"for" )==0) || 
			   ( wcscmp( p->argv[0], L"switch" )==0) || 
			   ( wcscmp( p->argv[0], L"function" )==0))
			{
				parser_push_block( FAKE );
			}
			else if( wcscmp( p->argv[0], L"end" )==0)
			{
				if(!current_block->outer->skip )
				{
					exec( j );
					return;
				}
				parser_pop_block();
			}
			else if( wcscmp( p->argv[0], L"else" )==0)
			{
				if( (current_block->type == IF ) && 
					(current_block->if_state != 0))
				{
					exec( j );
					return;
				}
			}
			else if( wcscmp( p->argv[0], L"case" )==0)
			{
				if( (current_block->type == SWITCH ) )
				{
					exec( j );
					return;
				}
			}			
		}
	}
	job_free( j );
}

/**
   Evaluates a job from the specified tokenizer. First calls
   parse_job to parse the job and then calls exec to execute it.
*/
static void eval_job( array_list_t *tok_stack )
{
	tokenizer *tok = (tokenizer *)al_get( tok_stack,
										  al_get_count(tok_stack)-1);
	job_t *j;
	job_start_pos = tok_get_pos( tok );	

	switch( tok_last_type( tok ) )
	{
		case TOK_STRING:
		{
			j = job_create();
			j->command=0;
			j->fg=1;
			j->constructed=0;
			
			if( shell_is_interactive )
			{
				if( tcgetattr (0, &j->tmodes) )
				{
					tok_next( tok );
					wperror( L"tcgetattr" );
					job_free( j );
					break;
				}
			}
			
			j->first_process = calloc( 1, sizeof( process_t ) );
			
			/* Copy the command name */
			
			if( parse_job( j->first_process, j, tok_stack ) && 
				j->first_process->argv )
			{
				if( job_start_pos < tok_get_pos( tok ) )
				{
					int start_pos = job_start_pos;
					int stop_pos = tok_get_pos( tok );
					j->command = wcsndup( tok_string(tok)+start_pos, 
										  stop_pos-start_pos );
				}
				else
					j->command = wcsdup( L"" );
				
				if(!current_block->skip )
				{
					exec( j );
				}
				else
				{
					skipped_exec( j );
				}
				
				if( current_block->type == WHILE )
				{
					switch( current_block->while_state )
					{
						case WHILE_TEST_FIRST:
						{
							current_block->skip = proc_get_last_status()!= 0;
							current_block->while_state=WHILE_TESTED;
						}
						break;
					}
				}

				if( current_block->type == IF )
				{
					if( (!current_block->if_state) && 
						(!current_block->skip) )
					{
						/*
						  We need to call job_do_notification,
						  since this is the function which sets
						  the status of the last process to exit
						*/
//							fwprintf( stderr, L"Result of if block is %d\n", proc_get_last_status() );
							
						current_block->skip = proc_get_last_status()!= 0;
						current_block->if_state++;
					}
				}
			}
			else
			{
				job_free( j );
			}
			break;
		}
	
		case TOK_END:
		{
			while( (!tok_has_next( tok )) && (al_get_count( tok_stack )>1 ))
			{
				al_pop( tok_stack );
				parser_allow_function();
				tok_destroy( tok );
				free(tok);
				tok = (tokenizer *)al_get( tok_stack, 
										   al_get_count(tok_stack)-1);
			}
			if( tok_has_next( tok ))
				tok_next( tok );
			break;			
		}
		
		case TOK_ERROR:
		{
			error_arg( SYNTAX_ERROR, 
					   L"Tokenizer error",
					   tok_last(tok),
					   tok_get_pos( tok ) );

			return;			
		}

		default:
		{
			error_arg( SYNTAX_ERROR, 
					   L"Expected a command string, got token of type",
					   tok_get_desc( tok_last_type(tok)),
					   tok_get_pos( tok ) );

			return;			
		}
	}
}

int eval( const wchar_t *cmd, buffer_t *output, int block_type )
{
	int forbid_count;
	int code;
	
	tokenizer *previous_tokenizer=current_tokenizer;
	
	current_tokenizer = malloc( sizeof(tokenizer));

	array_list_t tok_stack;
	block_t *start_current_block = current_block;
	parser_push_block( block_type );
	current_block->out_buffer = output;
	
	forbid_count = al_get_count( &forbidden_function );

	
	tok_init( current_tokenizer, cmd, 0 );
	al_init( &tok_stack );
	al_push( &tok_stack, current_tokenizer );
	error_code = 0;
	
	while( ( al_get_count( &tok_stack ) > 1 || tok_has_next( current_tokenizer )) && 
		   !error_code && 
		   !sanity_check() && 
		   !exit_status() )
		eval_job( &tok_stack );
	
	parser_pop_block();
	
	while( start_current_block != current_block )
	{
		if( current_block == 0 )
		{
			fwprintf( stderr,
					  L"fish: End of block mismatch\n"
					  L"Program terminating. If you can reproduce this behaviour,\n"
					  L"please file a bug report.\n" );
			exit(1);			
			//sanity_lose();
			break;
		}
		
		if( (!exit_status()) && (!proc_get_last_status()) )
		{
			char *h;
			
			//fwprintf( stderr, L"Status %d\n", proc_get_last_status() );
						
			fwprintf( stderr,
					  L"fish: Block missing end\n" );
		
			fwprintf( stderr, L"%ls", parser_current_line() );
			
			h = builtin_help_get( L"end" );
			if( h )
				fwprintf( stderr, L"%s", h );
			
		}		
		parser_pop_block();
	}

	print_errors();
	
	while( al_get_count( &tok_stack) )
	{
		tokenizer *killme = (tokenizer *)al_pop( &tok_stack );
		tok_destroy( killme );
		free( killme );
	}

	while( forbid_count < al_get_count( &forbidden_function ))
		parser_allow_function();
	
	al_destroy( &tok_stack );

	current_tokenizer=previous_tokenizer;
	
	code=error_code;
	error_code=0;
	
	return code;	
}

int parser_reserved( wchar_t *word)
{
	return ( (wcscmp( word, L"for")==0) ||
			 (wcscmp( word, L"while")==0) ||
			 (wcscmp( word, L"if")==0) ||
			 (wcscmp( word, L"function")==0) ||
			 (wcscmp( word, L"switch")==0) );
}


int parser_test( wchar_t * buff, 
				 int babble )
{
	tokenizer tok;
	int had_cmd=0;
	int count = 0;
	int err=0;
	tokenizer *previous_tokenizer=current_tokenizer;
	int previous_pos=current_tokenizer_pos;
	int code;
	int block_pos[BLOCK_MAX_COUNT];
	
	
	current_tokenizer = &tok;
	
	for( tok_init( &tok, buff, 0 );
		 tok_has_next( &tok ) && !err;
		 tok_next( &tok ) )
	{	
		current_tokenizer_pos = tok_get_pos( &tok );

		int last_type = tok_last_type( &tok );
		switch( last_type )
		{
			case TOK_STRING:
			{
				if( !had_cmd )
				{ 
					int mark = tok_get_pos( &tok );
					
					if( wcscmp(tok_last(&tok), L"end")==0)
					{
						tok_next( &tok );
						
//						if( !parser_is_help( tok_last( &tok ), 0 ) )
						{
							count--;
						}
				
						tok_set_pos( &tok, mark );
					}
					else if( parser_reserved( tok_last(&tok) ) )
					{
						if( count >= BLOCK_MAX_COUNT )
						{
							error( SYNTAX_ERROR, L"Too many block levels", tok_get_pos( &tok ) );							
							print_errors();						
						}
						else
						{
							block_pos[count] = current_tokenizer_pos;							
							tok_next( &tok );
							count++;
							tok_set_pos( &tok, mark );
						}
					}
					
					if( count < 0 )
					{
						err = 1;
						if( babble )
						{
							error( SYNTAX_ERROR, 
								   L"'end' command outside of block", 
								   tok_get_pos( &tok ) );
							print_errors();						
	
						}
						
					}
					had_cmd = 1;
				}
				break;
			}
			
			case TOK_REDIRECT_OUT:
			case TOK_REDIRECT_IN:
			case TOK_REDIRECT_APPEND:
			case TOK_REDIRECT_FD:
			{
				if( !had_cmd )
				{
					err = 1;
					if( babble )
					{
						error( SYNTAX_ERROR, 
							   L"Redirection error", 
							   tok_get_pos( &tok ) );
						print_errors();						
					}
				}
				break;
			}
			
			case TOK_END:
			{
				had_cmd = 0;
				break;
			}
			
			case TOK_PIPE:
			case TOK_BACKGROUND:
			{
				if( had_cmd )
				{
					had_cmd = 0;
				}
				else
				{
					err = 1;
					if( babble )
					{

						error( SYNTAX_ERROR, 
							   L"Pipe/bg error", 
							   tok_get_pos( &tok ) );
						print_errors();						
					}
					
				}				
				break;
			}
			
			case TOK_ERROR:
			default:
				err = 1;
				if( babble )
				{
					error_arg( SYNTAX_ERROR, 
							   L"Tokenizer error",
							   tok_last(&tok),
							   tok_get_pos( &tok ) );
					print_errors();
					//fwprintf( stderr, tok_last( &tok) );
				}				
				break;				
		}
	}
	
	tok_destroy( &tok );	
	
	if( babble && count>0 )
	{
		error( SYNTAX_ERROR, 
			   L"Block is missing end\n", 
			   block_pos[count-1] );
		print_errors();
	}
	

	current_tokenizer=previous_tokenizer;
	current_tokenizer_pos = previous_pos;

	error_code=0;	

	return (err?1:0) | (count!=0?2:0);		
}

/**
   Counts how many unbalanced quotes the specified string has
*/
static int is_quoted( const wchar_t *str )
{
	int level=0;
	wchar_t *q_char = L"\'\"";
	int offset=0;

	while(1)
	{
/*		fwprintf( stderr, L"Check %lc\n", *tok->buff );*/
		switch( *str )
		{
			case L'\\':
				str++;
				break;
			case L'\"':
			case L'\'':
				if( level == 0 )
				{
					offset = (*str != L'\"');
					level = 1;
				}
				else
				{
					if( q_char[(level+offset) % 2] == *str )
					{
						level--;
					}
					else
					{
						level++;
					}
				}
				
				break;
		}
		if( (*str == L'\0'))
			break;

		str++;
	}

	return level;
	
}

/**
   This function is an ugly mutant kludge. It takes the string \c val
   and inserts it into the variable \c name. Before doing this, it
   morphs any quoted parts of \c val into a non-quoted form. It Quotes
   are not balanced, this is happily ignored. This is done in the name
   of glorious tab-completion goodness. There has to be a better way.
*/
static void insert_escaped( const wchar_t *name, 
							wchar_t *val )
{
	wchar_t *res = malloc(sizeof(wchar_t)*(wcslen(val)*2+2));
	wchar_t *out = res;
	

	while( *val )
	{
		switch( *val )
		{
			case L'\'':
			case L'\"':
			{
				wchar_t * end = quote_end( val );
				wchar_t * escaped;
				wchar_t prev_val = end?*end:0;
				
				if( !end )
				{
					end = val + wcslen(val);
				}
				else
					*end = 0;
				
				escaped = expand_escape( wcsdup(val+1), 1 );
				wcscpy( out, escaped );				
				out += wcslen( escaped );
				free( escaped );

				*end = prev_val;
				
				val = end;
				break;
			}
			default:
				*out++ = *val++;
		}
	}
	*out = 0;
	
	env_set( name, res, ENV_LOCAL );
	free( res );
		
}

int parser_check_command( const wchar_t *cmd, 
						  int pos )
{
	wchar_t last_cmd[1024];
	wchar_t end_str[1024];
	wchar_t prev_str[1024];
	int prev_cmd=1,new_cmd=1;
	int unfinished=0;
	wchar_t *last_str;
	int is_ok = 1;
	wchar_t *tmp_str;	
	int cmdlen;
	int prev_pos = 0;

	string_buffer_t prev_tokens;
	

	int max_len=0;
	
	tokenizer tok;

	tok_init ( &tok, cmd, TOK_ACCEPT_UNFINISHED );
	prev_pos = tok_get_pos( &tok );
	
//	fwprintf( stderr, L"Check command %ls to pos %d\n", cmd, pos );

	last_cmd[0]=L'\0';
	prev_str[0]=L'\0';
	end_str[0]=L'\0';

	sb_init( &prev_tokens );

	if( !tok_has_next( &tok ) )
	{
		is_ok=0;		
	}
	
	
	while( tok_has_next( &tok ) && is_ok )
	{
		prev_cmd=new_cmd;		
		switch( tok_last_type( &tok ) )
		{
			case TOK_PIPE:
			case TOK_END:
				new_cmd = 1;
				last_cmd[0]=L'\0';
				prev_str[0]=L'\0';
				end_str[0]=L'\0';
				break;

			case TOK_STRING:
				
				tmp_str = wcsdup(tok_last( &tok ));
				if( !tmp_str )
				{
					fwprintf( stderr, L"Could not expand %ls\n", tok_last( &tok ) );
					is_ok=0;
					break;
				}
				else
				{
					if( new_cmd )
					{
						sb_clear( &prev_tokens );
						
						if( !parser_subcommand( tmp_str ) )
						{
							wcsncpy( last_cmd,
									 tmp_str,
									 1024 );
							
							new_cmd=0;
						}					
					}
					else
					{
						wcsncpy( prev_str, end_str, 1024 );					
						wcsncpy( end_str, tmp_str, 1024 );					
					}
				}
				sb_append( &prev_tokens, tmp_str );
				sb_append( &prev_tokens, ARRAY_SEP_STR );

				free( tmp_str );
				break;

			default:
				new_cmd=0;
				
		}
		prev_pos = tok_get_pos( &tok );
		tok_next( &tok );
		if( (!tok_has_next( &tok )) || (tok_get_pos( &tok )> pos) )
		{
			max_len = pos - prev_pos;
			end_str[max_len]=0;
//			fwprintf( stderr, L"Last token truncated after %d characters\n", max_len );
			break;
		}		
	}
	if( is_ok )
	{

		if( prev_tokens.used )
		{
			prev_tokens.used--;
			prev_tokens.buff[prev_tokens.used]=0;
		}
		
		last_str = prev_cmd?last_cmd:end_str;

		/* Is the cursor positioned after the last entered parameter?*/
		/* wprintf( L"Compare \'%ls\' and \'%ls\'\n", last_str, &cmd[wcslen(cmd)-wcslen(last_str)] );*/
		/*unfinished = (wcscmp( last_str, &cmd[wcslen(cmd)-wcslen(last_str)] ) == 0);*/
		cmdlen = wcslen(cmd);
		unfinished = cmdlen==0;

		/*  
			Check if the last token has unbalanced quotes. If so, it has not ended, and will be the contents of the FISH_END_STR variable
		*/
		if( !unfinished )
		{
			unfinished = is_quoted( cmd );
			if( !unfinished )
			{
				int finished = (wcschr( L" \t\n\r", cmd[cmdlen-1] ) != 0 ) &&
					( cmdlen ==1 || cmd[cmdlen-2] != L'\\' );
				unfinished = !finished;
			}
		}
		
		if( !unfinished )
		{
			if( prev_cmd )
			{
				prev_cmd=0;
				prev_str[0] = L'\0';
			}
			else
			{
				wcsncpy( prev_str, end_str, 1024 );					
				end_str[0] = L'\0';
			}
		}

		env_push( 1 );
		
		if( new_cmd || prev_cmd )
		{
			/*  
				If we are on a command, insert the part of it before the cursor into END_STR
				set_color green; echo read; set_color normal; echo \"> \"*/
			wcscpy( end_str, last_cmd );
			end_str[max_len]=0;				
			
			env_set( PARSER_CURSOR_ON_COMMAND, L"1", ENV_LOCAL );
		}		
		else
		{
			env_remove( PARSER_CURSOR_ON_COMMAND, 0 );
		}

		insert_escaped( PARSER_END_STR, end_str );
		insert_escaped( PARSER_WHOLE_STR, (wchar_t *)cmd );
		insert_escaped( PARSER_PREV_STR, prev_str );
		insert_escaped( PARSER_COMMAND, last_cmd );
		insert_escaped( PARSER_PREV_TOKENS, (wchar_t *)prev_tokens.buff  );
		
/*		fwprintf( stderr, L"prev_str: %ls, end_str: %ls, command: %ls, on_command: %ls\n", 
  env_get( L"FISH_PREV_STR" ),
  env_get( L"FISH_END_STR" ),
  env_get( L"FISH_COMMAND" ),
  env_get( L"FISH_CURSOR_ON_COMMAND" ) );
*/	
	}
	tok_destroy( &tok );	
	sb_destroy( &prev_tokens );

	return is_ok;	
}


