/** \file parser.h
	The fish parser. 	
*/

/**
   The environment variable that contains the whole commandline after
   a call to parser_check_commandline
*/
#define PARSER_WHOLE_STR L"whole_str"

/**
   The environment variable that contains the part of the last token
   which comes before the cursor after a call to
   parser_check_commandline
*/
#define PARSER_END_STR L"end_str"

/**
   The environment variable that contains the token before the last
   token after a call to parser_check_commandline
*/
#define PARSER_PREV_STR L"prev_str"

/**
   The environment variable that contains the last command before the
   cursor after a call to parser_check_commandline
*/
#define PARSER_COMMAND L"command"

/**
   The environment variable that is set if the cursor is above a
   command after a call to parser_check_commandline
*/
#define PARSER_CURSOR_ON_COMMAND L"cursor_on_command"


/**
   The environment variable that contains all arguments passed to the
   current before the cursor
*/
#define PARSER_PREV_TOKENS L"prev_tokens" 

/**
   block_t represents a block of commands. 
*/
typedef struct block
{
	int type; /**< Type of block. Can be one of WHILE, FOR, IF and FUNCTION */
	int skip; /**< Whether execution of the commands in this block should be skipped */
	int tok_pos; /**< The start index of the block */

	/**
	   Status for the current loop block. Can be anu of the values from the loop_status enum.
	*/
	int loop_status;

	/**
	   First block type specific variable
	*/
	union 
	{
		int while_state;  /**< True if the loop condition has not yet been evaluated*/
		wchar_t *for_variable; /**< Name of the variable to loop over */
		int if_state; /**< The state of the if block */
		wchar_t *switch_value; /**< The value to test in a switch block */
		wchar_t *function_name; /**< The name of the function to define */
	};	

	/**
	   Second block type specific variable
	*/
	union
	{
		array_list_t for_vars; /**< List of values for a for block */	
		int switch_taken; /**< Whether a switch match has already been found */
		wchar_t *function_description; /**< The description of the function to define */
	};

	buffer_t *out_buffer; 
	
	struct block *outer; /**< Next outer block */
} block_t;

/** 
	Types of blocks
*/
enum block_type
{
	WHILE, /**< While loop block */
	FOR,  /**< For loop block */
	IF, /**< If block */
	FUNCTION_DEF, /**< Function definition block */
	FUNCTION_CALL, /**< Function invocation block */
	SWITCH, /**<Switch block */
	FAKE, /**<Fake block */
	TOP /**<Outermost block */
}
;

/**
   Possible states for a loop
*/
enum loop_status 
{
	LOOP_NORMAL, /**< Current loop block executed as normal */
	LOOP_BREAK, /**< Current loop block should be removed */
	LOOP_CONTINUE, /**< Current loop block should be skipped */
};


/**
   Possible states for a while block
*/
enum while_status
{
	WHILE_TEST_FIRST, /**< This is the first command of the first lap of a while loop */
	WHILE_TEST_AGAIN, /**< This is not the first lap of the while loop, but it is the first command of the loop */
	WHILE_TESTED, /**< This is not the first command in the loop */
}
;



/**
   Errors that can be generated by the parser
*/
enum parser_error 
{
	NO_ERR=0,SYNTAX_ERROR,EVAL_ERROR,OOM,STACK_ERROR,SUBSHELL_ERROR
}
;

/** The current innermost block */
extern block_t *current_block;

/** The current error code */
extern int error_code;



/**
  Finds the full path of an executable in a newly allocated string.
  
  \param cmd The name of the executable.
  \return 0 if the command can not be found, the path of the command otherwise.
*/
wchar_t *get_filename( const wchar_t *cmd );

/**
  Evaluate the expressions contained in cmd.

  \param cmd The string to evaluate
  \param out buffer to insert output to. May be null.
  \param the type of block to push onto the scope stack
  \return 0 on success.
*/
int eval( const wchar_t *cmd, buffer_t *out, int block_type );

/**
  Evaluate line as a list of parameters, i.e. tokenize it and perform parameter expansion and subshell execution on the tokens.
  The output is inserted into output, and should be freed by the caller.

  \param line Line to evaluate
  \param output List to insert output to
*/
int eval_args( const wchar_t *line,
				array_list_t *output );

/**
   Sets the current error

   \param ec The new error code
   \param str The new error message
   \param p The character offset at which the error occured
*/
void error( int ec, const wchar_t *str, int p );

/**
   Tests if the specified commands parameters should be interpreted as another command, which will be true if the command is either 'command' or 'builtin'.  

   \param cmd The command to test
   \return 1 of the command parameter is a command, 0 otherwise
*/
int parser_subcommand( const wchar_t *cmd );

/**
   Returns a string describing the current parser pisition in the format 'FILENAME (line LINE_NUMBER): LINE'.
   Example: 

   init.fish (line 127): ls|grep pancake
*/
wchar_t *parser_current_line();

/**
   Returns the current position in the latest string of the tokenizer.
*/
int parser_get_pos();

/**
   Returns the position where the current job started in the latest string of the tokenizer.
*/
int parser_get_job_pos();

/**
   Set the current position in the latest string of the tokenizer.
*/
void parser_set_pos( int p);

/**
   Get the string currently parsed
*/
const wchar_t *parser_get_buffer();

/**
   Create block of specified type
*/
void parser_push_block( int type);

/**
   Remove the outermost block namespace
*/
void parser_pop_block();

/**
   Test if the specified string can be parsed, or if more bytes need to be read first. 
   The result has the first bit set if the string contains errors, and the second bit is set if the string contains an unclosed block.
*/
int parser_test( wchar_t * buff, int babble );

/**
   Get information about the specified command. parser_check_command()
   will push a new scope of environment variables, and set the
   variables 'end_str', containing the last token starting before
   the cursor, up to the cursor position, 'command', containing the
   last command, and 'cursor_on_command', which will only be set if
   the cursor is over a command.
*/
int parser_check_command( const wchar_t *cmd, int pos );

/**
   Returns the full path of the specified directory. If the \c in is a
   full path to an existing directory, a copy of the string is
   returned. If \c in is a directory relative to one of the
   directories i the CDPATH, the full path is returned. If no
   directory can be found, 0 is returned.
*/
wchar_t *parser_cdpath_get( wchar_t *in );

/**
   Tell the parser that the specified function may not be run if not
   inside of a conditional block. This is to remove some possibilities
   of infinite recursion.
*/
void parser_forbid_function( wchar_t *function );
/**
   Undo last call to parser_forbid_function().
*/
void parser_allow_function();

/**
   Initialize the parser
*/
void parser_init();

/**
   Destroy the parser
*/
void parser_destroy();

/**
   This function checks if the specified string is a help option. 

   \param s the string to test
   \param min_match is the minimum number of characters that must match in a long style option, i.e. the longest common prefix between --help and any other option. If less than 3, 3 will be assumed.
*/
int parser_is_help( wchar_t *s, int min_match );


int parser_reserved( wchar_t *word);

