/*
 *										
 *		File		: configuration.c				
 *										
 *		Title		: Flink email monitoring applet for GNOME 2	
 *										
 *		Description	: Most configuration -(GConf)- related in here	
 *										
 *		Authors		: Marcus Leyman, 2002 <marcus@leyman.nu>	
 *										
 *		License		: GPL						
 *										
 *		Date		: 27 May 2002					
 *										
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <gnome.h>
#include <gconf/gconf-client.h>

#include "callbacks.h"
#include "interface.h"
#include "support.h"
#include "configuration.h"

/********************************************************
 * Function:	flink_config_modify_account		*
 *							*
 * Description:	Updates account information		*
 *							*
 * Arguments:	Account *acc, account to update		*
 *							*
 * Returns:	nothing					*
 ********************************************************/
void
flink_config_modify_account	(Account *acc)
{
  GConfClient	*client;
  Account	*acc_old;
  gchar		*path;

  path = g_strdup_printf("/apps/flink/accounts/%d", acc->id);
	
  acc_old = g_new0(Account, 1);
	
  acc_old = flink_config_get_account_from_path(path);
  client = gconf_client_get_default();

  acc->type = gconf_client_get_int(client, g_strdup_printf("%s_type", path), &acc->err);
	
  if (acc->title) {
    if (strcmp(acc->title, acc_old->title))
      gconf_client_set_string(client, g_strdup_printf("%s_title", path), acc->title, &acc->err);
  }
  if (acc->hostname && acc->type == (1 || 2)) {
    if (strcmp(acc->hostname, acc_old->hostname))
      gconf_client_set_string(client, g_strdup_printf("%s_hostname", path), acc->hostname, &acc->err);
  }
  if (acc->username && acc->type == (1 || 2)) {
    if (strcmp(acc->username, acc_old->username))
      gconf_client_set_string(client, g_strdup_printf("%s_username", path), acc->username, &acc->err);
  }
  if (acc->password && acc->type == (1 || 2)) {
    if (strcmp(acc->password, acc_old->password))
      gconf_client_set_string(client, g_strdup_printf("%s_password", path), acc->password, &acc->err);
  }
  if (acc->folder && acc->type == 2) {
    if (strcmp(acc->folder, acc_old->folder))
      gconf_client_set_string(client, g_strdup_printf("%s_folder", path), acc->folder, &acc->err);
  }
  if (acc->path && acc->type == 3) {
    if (strcmp(acc->path, acc_old->path))
      gconf_client_set_string(client, g_strdup_printf("%s_path", path), acc->path, &acc->err);	
  }
  if (acc->port && acc->type == (1 || 2)) {
    if (acc->port != acc_old->port)
      gconf_client_set_int(client, g_strdup_printf("%s_port", path), acc->port, &acc->err);
  }
  if (acc->interval) {
    if (acc->interval != acc_old->interval)
      gconf_client_set_int(client, g_strdup_printf("%s_interval", path), acc->interval, &acc->err);
  }
}

/********************************************************
 * Function:	flink_config_unset_account		*
 *							*
 * Description:	Unset an accounts gconf keys    	*
 *							*
 * Arguments:	Account *acc, account to unset		*
 *							*
 * Returns:	nothing					*
 ********************************************************/
void
flink_config_unset_account	(Account *acc)
{
  GConfClient	*client;
  gchar		*path;
	
  client = gconf_client_get_default();
	
  /* Unset all keys affiliated to this account */
  path = g_strdup_printf("/apps/flink/accounts/%d", acc->id);

  gconf_client_unset(client, g_strdup_printf("%s_interval", path), &acc->err);
  if(acc->type != 3) {
    gconf_client_unset(client, g_strdup_printf("%s_hostname", path), &acc->err);
    gconf_client_unset(client, g_strdup_printf("%s_port", path), &acc->err);
    gconf_client_unset(client, g_strdup_printf("%s_username", path), &acc->err);
    gconf_client_unset(client, g_strdup_printf("%s_password", path), &acc->err);
  }
	
  if(acc->type == 3)
    gconf_client_unset(client, g_strdup_printf("%s_path", path), &acc->err);
  if(acc->type == 2)
    gconf_client_unset(client, g_strdup_printf("%s_folder", path), &acc->err);	

  gconf_client_unset(client, g_strdup_printf("%s_type", path), &acc->err);

  gconf_client_unset(client, g_strdup_printf("%s_newmail", path), &acc->err);

  gconf_client_unset(client, g_strdup_printf("%s_title", path), &acc->err);
	
  gconf_client_unset(client, g_strdup_printf("%s_id", path), &acc->err);
	
  gconf_client_suggest_sync(client, &acc->err);
}

/********************************************************
 * Function:	flink_config_delete_account		*
 *							*
 * Description:	Deletes an account			*
 *							*
 * Arguments:	Account *acc, account to delete		*
 *							*
 * Returns:	nothing					*
 ********************************************************/

void
flink_config_delete_account	(Account *acc) 
{
  GConfClient	*client;

  gchar		*path;
  guint		accounts;
  guint		current;
	
  client = gconf_client_get_default();

  accounts = gconf_client_get_int(client, "/apps/flink/accounts/accounts", &acc->err);
  flink_config_unset_account(acc);
	
  /* reduce id of accounts after deleted acc's id to keep track of things */
  if (acc->id < accounts) {
    for (current = acc->id + 1; current <= accounts; current++) {
      Account         *tmp_acc;

      tmp_acc = flink_config_get_account_from_path(g_strdup_printf("/apps/flink/accounts/%d", current));
      flink_config_unset_account(tmp_acc);
      tmp_acc->id = current -1;
      flink_config_create_new_account(tmp_acc);
    }
  }
  gconf_client_set_int(client, "/apps/flink/accounts/accounts", accounts -1, &acc->err);
}

/********************************************************
 * Function:	flink_config_get_config			*
 *							*
 * Description:	Fills and returns Flink configuration	*
 *							*
 * Arguments:	none					*
 *							*
 * Returns:	pointer to FlinkConfig			*
 ********************************************************/
FlinkConfig *
flink_config_get_config	() 
{
  GConfClient	*client;
  FlinkConfig	*cfg;
	
  client = gconf_client_get_default();
	
  cfg = g_new0(FlinkConfig, 1);
	
  cfg->accounts		= gconf_client_get_int(client, "/apps/flink/accounts/accounts", &cfg->err);
  cfg->beep 		= gconf_client_get_bool(client, "/apps/flink/beep", &cfg->err);
  cfg->sound		= gconf_client_get_bool(client, "/apps/flink/sound", &cfg->err);
  cfg->soundfile	= gconf_client_get_string(client, "/apps/flink/soundfile", &cfg->err);
	
  cfg->ext_prog_run_auto	= gconf_client_get_bool(client, "/apps/flink/external_autorun", &cfg->err);
  cfg->ext_prog_file	= gconf_client_get_string(client, "/apps/flink/external_file", &cfg->err);
	
  cfg->mail_file	= gconf_client_get_string(client, "/apps/flink/mailimage", &cfg->err);
  cfg->nomail_file	= gconf_client_get_string(client, "/apps/flink/nomailimage", &cfg->err);
	
  return cfg;
}

/********************************************************
 * Function:	flink_config_create_template_account	*
 *							*
 * Description:	Creates template account with bogus info*
 *							*
 * Arguments:	none					*
 *							*
 * Returns:	pointer to Account with bogus info	*
 ********************************************************/
Account *
flink_config_create_template_account	() 
{
  Account *acc;
  acc = g_new0(Account, 1);
	
  acc->type 		= 1;
  acc->id 		= 0;
  acc->title		= "Title";
  acc->hostname 	= "Hostname";
  acc->port 		= 110;
  acc->username		= "username";
  acc->password		= "";
  acc->path		= "path";
  acc->folder		= "folder";
  acc->interval		= 1;	
  return acc;
}	

/********************************************************
 * Function:	flink_config_get_account_from_path	*
 *							*
 * Description:	Gets full account from absolute path	*
 *		i.e. /apps/flink/accounts/[NAME]	*
 *							*
 * Arguments:	gchar *path, see above			*
 *							*
 * Returns:	pointer to Account fetched from gconf	*
 ********************************************************/
Account *
flink_config_get_account_from_path	(gchar *path) 
{
  GConfClient	*client;
  Account	*acc;
  if (path) {

    client = gconf_client_get_default();
    acc = g_new0(Account, 1);
		
    acc->type 	= gconf_client_get_int(client, g_strdup_printf("%s_type", path), &acc->err);
    acc->id 	= gconf_client_get_int(client, g_strdup_printf("%s_id", path), &acc->err);
    acc->title	= gconf_client_get_string(client, g_strdup_printf("%s_title", path), &acc->err);
    acc->interval 	= gconf_client_get_int(client, g_strdup_printf("%s_interval", path), &acc->err);
    if(acc->type != 3) {
      acc->hostname 	= gconf_client_get_string(client, g_strdup_printf("%s_hostname", path), 
						  &acc->err);
      acc->port 	= gconf_client_get_int(client, g_strdup_printf("%s_port", path), 
					       &acc->err);
      acc->username 	= gconf_client_get_string(client, g_strdup_printf("%s_username", path), 
						  &acc->err);
      acc->password 	= gconf_client_get_string(client, g_strdup_printf("%s_password", path), 
						  &acc->err);
    }
    if(acc->type == 3)
      acc->path 	= gconf_client_get_string(client, g_strdup_printf("%s_path", path), 
						  &acc->err);
    if(acc->type == 2)
      acc->folder 	= gconf_client_get_string(client, g_strdup_printf("%s_folder", path), 
						  &acc->err);	
	
    return acc;
  } else {
    return NULL;
  }	
}

/********************************************************
 * Function:	flink_config_create_new_account		*
 *							*
 * Description:	writes new account to gconf database	*
 *							*
 * Arguments:	Account *acc, account to write		*
 *							*
 * Returns:	nothing					*
 ********************************************************/
void
flink_config_create_new_account	(Account *acc)
{
  GConfClient	*client;
  gchar		*path;
  guint		ref;
  gint		accounts;
		
  client = gconf_client_get_default();
  
  accounts = gconf_client_get_int(client, "/apps/flink/accounts/accounts", &acc->err);
  path = g_strdup_printf("/apps/flink/accounts/%d", acc->id);

  gconf_client_set_int(client, g_strdup_printf("%s_type", path), acc->type , &acc->err);
  gconf_client_set_int(client, g_strdup_printf("%s_id", path), acc->id, &acc->err);
  gconf_client_set_string(client, g_strdup_printf("%s_title", path), acc->title, &acc->err);
	
  if(acc->type != 3) {
    gconf_client_set_string(client, g_strdup_printf("%s_hostname", path), acc->hostname, &acc->err);
    gconf_client_set_int(client, g_strdup_printf("%s_port", path), acc->port, &acc->err);
    gconf_client_set_string(client, g_strdup_printf("%s_username", path), acc->username, &acc->err);
    gconf_client_set_string(client, g_strdup_printf("%s_password", path), acc->password, &acc->err);
  }
  if(acc->type == 3)
    gconf_client_set_string(client, g_strdup_printf("%s_path", path), acc->path, &acc->err);
  if(acc->type == 2)
    gconf_client_set_string(client, g_strdup_printf("%s_folder", path), acc->folder, &acc->err);
	
  gconf_client_set_int(client, g_strdup_printf("%s_interval", path), acc->interval, &acc->err);
	
  gconf_client_suggest_sync(client, &acc->err);

  gconf_client_set_int(client, "/apps/flink/accounts/accounts", accounts + 1, &acc->err);

  /*
  if(acc->interval != 0) {
    ref = gtk_timeout_add(((guint) acc->interval * 60000),
			  flink_timeout_reached,
			  (gpointer) acc->id);
  }
 
  gconf_client_notify_add(client,
			  g_strdup_printf("%s_interval", path),
			  flink_timeout_changed,
			  (gpointer) ref,
			  NULL,
			  NULL);
  */
}

/********************************************************
 * Function:	flink_config_add_new_account		*
 *							*
 * Description:	writes new account to gconf database	*
 *							*
 * Arguments:	Account *acc, account to write		*
 *							*
 * Returns:	nothing					*
 ********************************************************/
void
flink_config_add_new_account	(Account *acc) 
{
  /*
  GConfClient     *client;
  gint            accounts;

  client = gconf_client_get_default();
        
  accounts = gconf_client_get_int(client, "/apps/flink/accounts/accounts", &acc->err);
        
  gconf_client_set_int(client, "/apps/flink/accounts/accounts", accounts + 1, &acc->err);
  */    
  flink_config_create_new_account(acc);
}
       
