/**************************************************************************

   Fotoxx      edit photos and manage collections

   Copyright 2007-2014 Michael Cornelison
   Source URL: http://kornelix.com/fotoxx
   Contact: kornelix@posteo.de

   This program is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program. If not, see http://www.gnu.org/licenses/.

***************************************************************************/

#define EX extern                                                          //  enable extern declarations
#include "fotoxx.h"                                                        //  (variables in fotoxx.h are refs)
#include <sys/wait.h>


/**************************************************************************

   Fotoxx image edit - Tools menu functions

***************************************************************************/


//  toggle foreground drawing and mouse cursor color

void  m_fg_color(GtkWidget *, cchar *)
{
   F1_help_topic = "color";
   if (fg_color == white) fg_color = black;
   else if (fg_color == black) fg_color = red;
   else if (fg_color == red) fg_color = green;
   else fg_color = white;
   Fpaint2();
   return;
}


/**************************************************************************/

//  Index Image Files menu function                                        //  v.14.02
//  Manual index request, or missing information, or new install.
//  Dialog to supply top image directories and thumbnails directory.
//  Update the corresponding config. files and generate new image index.

void m_index(GtkWidget *, cchar *)
{
   int index_dialog_event(zdialog *zd, cchar *event);

   zdialog        *zd;
   FILE           *fid;
   char           filespec[200], buff[200], topdirk[200], thumbdirk[200];
   char           *pp;
   GtkWidget      *widget;
   int            ftf, cc, zstat;

   F1_help_topic = "index_files";
   if (checkpend("all")) return;                                           //  check nothing pending
   Fmenulock = 1;                                                          //  lock menus

/***
       _____________________________________________________
      |              Index Image Files                      |
      |                                                     |
      |  Top Image Directories   [add directory]            |
      |  _________________________________________________  |
      | |  /home/<user>/pictures                          | |
      | |  /home/<user>/...                               | |
      | |                                                 | |
      | |                                                 | |
      | |                                                 | |
      | |_________________________________________________| |
      |                                                     |
      | Thumbnails [_____________________________] [Browse] |
      |                                                     |
      |                           [Help] [Proceed] [Cancel] |
      |_____________________________________________________|

***/

   zd = zdialog_new(ZTX("Index Image Files"),Mwin,Bhelp,Bproceed,Bcancel,null);

   zdialog_add_widget(zd,"hbox","hbtop1","dialog",0,"space=4");
   zdialog_add_widget(zd,"label","labtop","hbtop1",ZTX("Top Image Directories"),"space=5");
   zdialog_add_widget(zd,"button","browsetop","hbtop1",ZTX("add directory"),"space=4");
   zdialog_add_widget(zd,"hbox","hbtop2","dialog",0,"expand");
   zdialog_add_widget(zd,"label","space","hbtop2",0,"space=3");
   zdialog_add_widget(zd,"vbox","vbtop2","hbtop2",0,"expand");
   zdialog_add_widget(zd,"label","space","hbtop2",0,"space=3");
   zdialog_add_widget(zd,"frame","frtop","vbtop2",0,"expand");
   zdialog_add_widget(zd,"scrwin","scrtop","frtop",0,"expand");
   zdialog_add_widget(zd,"edit","topdirks","scrtop");
   zdialog_add_widget(zd,"hbox","hbthumb","dialog",0,"space=5");
   zdialog_add_widget(zd,"label","labthumb","hbthumb",ZTX("Thumbnails"),"space=5");
   zdialog_add_widget(zd,"entry","thumbdirk","hbthumb",0,"expand");
   zdialog_add_widget(zd,"button","browsethumb","hbthumb",Bbrowse,"space=3");
   zdialog_add_widget(zd,"label","space","hbthumb",0);

   snprintf(topdirk,200,"%s/Pictures\n",getenv("HOME"));                   //  stuff default top image directory
   zdialog_stuff(zd,"topdirks",topdirk);                                   //    /home/<user>/Pictures

   snprintf(thumbdirk,200,"%s/thumbnails",get_zuserdir());                 //  stuff default thumbnails directory
   zdialog_stuff(zd,"thumbdirk",thumbdirk);                                //    /home/<user>/.fotoxx/thumbnails

   snprintf(filespec,200,"%s/top_directories",index_dirk);                 //  read top image directories file,
   widget = zdialog_widget(zd,"topdirks");                                 //    stuff data into dialog widgets

   fid = fopen(filespec,"r");
   if (fid) {
      zdialog_stuff(zd,"topdirks","");                                     //  clear default
      while (true) {
         pp = fgets_trim(buff,200,fid,1);
         if (! pp) break;
         if (strnEqu(buff,"thumbnails directory: /",23)) {                 //  if "thumbnails directory: /..."
            zdialog_stuff(zd,"thumbdirk",buff+22);                         //  stuff thumbnails directory
            break;                                                         //  last record
         }
         else wprintf(widget,"%s\n",buff);                                 //  stuff next top image directory
      }
      fclose(fid);
   }
   
   zdialog_resize(zd,400,400);                                             //  run dialog
   zdialog_run(zd,index_dialog_event,"parent");

   zstat = zdialog_wait(zd);                                               //  wait for completion
   
   if (zstat != 2) {                                                       //  canceled
      Fmenulock = 0;
      zdialog_free(zd);
      if (! Findexdone) {
         zmessageACK(Mwin,0,ZTX("terminated by user"));
         m_quit(0,0);
      }
      return;
   }

   widget = zdialog_widget(zd,"topdirks");                                 //  get top directories from dialog widget

   snprintf(filespec,200,"%s/top_directories",index_dirk);                 //  open/write top directories file
   fid = fopen(filespec,"w");
   if (! fid) {
      zmessageACK(Mwin,0,"top directories file: \n %s",strerror(errno));
      Fmenulock = 0;
      m_quit(0,0);
   }

   ftf = 1;
   while (true) {
      pp = wscanf(widget,ftf);                                             //  loop widget text lines
      if (! pp) break;
      strncpy0(buff,pp,200);
      strTrim2(buff);                                                      //  remove surrounding blanks
      cc = strlen(buff);
      if (cc < 5) continue;                                                //  ignore blanks or rubbish
      if (buff[cc-1] == '/') buff[cc-1] = 0;                               //  remove trailing '/' 
      fprintf(fid,"%s\n",buff);                                            //  write top directory to output file
   }

   zdialog_fetch(zd,"thumbdirk",buff,200);                                 //  get thumbnails directory from dialog
   strTrim2(buff);                                                         //  remove surrounding blanks
   cc = strlen(buff);
   if (cc && buff[cc-1] == '/') buff[cc-1] = 0;                            //  remove trailing '/'
   if (cc > 10) 
      fprintf(fid,"thumbnails directory: %s\n",buff);                      //  write thumbnails directory to output file
   fclose(fid);

   zdialog_free(zd);
   index_rebuild(1);                                                       //  build image index and thumbnail files

   Fmenulock = 0;
   return;
}


//  index dialog event and completion function

int index_dialog_event(zdialog *zd, cchar *event)
{
   GtkWidget   *widget;
   char        **flist, *thumbdirk;
   int         ii;
   cchar       *topmess = ZTX("Choose top image directories");
   cchar       *thumbmess = ZTX("Choose thumbnail directory");
   
   if (strEqu(event,"browsetop")) {                                        //  [browse] top directories
      flist = zgetfiles(topmess,"folders",getenv("HOME"));                 //  get top directories from user
      if (! flist) return 0;
      widget = zdialog_widget(zd,"topdirks");                              //  add to dialog list
      for (ii = 0; flist[ii]; ii++) {
         wprintf(widget,"%s\n",flist[ii]);
         zfree(flist[ii]);
      }
      zfree(flist);
   }
   
   if (strEqu(event,"browsethumb")) {                                      //  [browse] thumbnail directory
      thumbdirk = zgetfile(thumbmess,"folder",getenv("HOME"));
      if (! thumbdirk) return 0;
      thumbdirk = zstrdup(thumbdirk,12);
      if (! strstr(thumbdirk,"/thumbnails"))                               //  if not containing /thumbnails,
         strcat(thumbdirk,"/thumbnails");                                  //    append /thumbnails
      zdialog_stuff(zd,"thumbdirk",thumbdirk);
      zfree(thumbdirk);
   }

   if (strEqu(event,"enter")) zd->zstat = 2;                               //  [proceed]  v.14.03

   if (! zd->zstat) return 1;                                              //  wait for completion
   
   if (zd->zstat == 1) {                                                   //  [help]
      zd->zstat = 0;
      showz_userguide("index_files");
      return 1;
   }
   
   if (zd->zstat != 2) {                                                   //  dialog canceled
      zdialog_destroy(zd);
      return 1;
   }

   return 1;                                                               //  [proceed]
}


//  Rebuild the image index using top directories                          //  v.14.02
//    and other information in configuration files.
//  Called from main() when Fotoxx is started (menu = 0)
//  Called from menu function m_index() (menu = 1)

zdialog        *zd_index_popup;

void index_rebuild(int menu)
{
   int index_popup_dialog_event(zdialog *zd, cchar *event);
   int index_popup_timeout(void *);
   int image_index_compare(cchar *rec1, cchar *rec2);

   GtkWidget      *poplog;
   FILE           *fid, *fidr, *fidw;
   int            ii, jj, ff, ntop, nthumb, err, cc, ftf;
   int            Nfiledir, dotstep;
   int            newthumb, newindex, Ndirtot;
   int            Nindextot, Nthumbtot;
   int            Nold, Nnew, NF, orec, nrec, nmatch, comp;
   char           *pp, filespec[200], tempfile[200];
   char           **ppv, currdir[maxfcc], buff[maxfcc];
   char           **flist, *file, *thumbfile, *Fupdate;
   sxrec_t        *sxrec_old, *sxrec_new;
   struct stat    statdat;
   PangoFontDescription    *monofont;

   char     *filename, *exifdate, *iptctags, *iptcrating;
   char     *exifsize, *iptccapt, *exifcomms;
   char     *exifcity, *exifcountry, *exiflat, *exiflong;
   char     city2[100], country2[100], lat2[20], long2[20], gtags2[200];

   cchar    *exifkeys[11] = { "FileName", exif_date_key, iptc_keywords_key,
            iptc_rating_key, exif_size_key, exif_comment_key, iptc_caption_key,
            exif_city_key, exif_country_key, exif_latitude_key, exif_longitude_key };

   Findexdone = 0;                                                         //  set = 1 when successful

   //  get current top image directories and thumbnails directory
   //  from /home/<user>/.fotoxx/image_index/top_directories
   
   ntop = nthumb = 0;
   snprintf(filespec,200,"%s/top_directories",index_dirk);                 //  read top directories file

   fid = fopen(filespec,"r");
   if (fid) {
      while (true) {                                                       //  get top image directories
         pp = fgets_trim(buff,200,fid,1);
         if (! pp) break;
         if (strnEqu(buff,"thumbnails directory: /",23)) {
            if (thumbdirk) zfree(thumbdirk);
            thumbdirk = zstrdup(buff+22);
            nthumb = 1;
            break;
         }
         if (topdirks[ntop]) zfree(topdirks[ntop]);
         topdirks[ntop] = zstrdup(buff);                                   //  top directories list in global space
         if (++ntop == maxtopdirks) break;
      }
      fclose(fid);
   }
   
   Ntopdirks = ntop;

   if (! ntop) {                                                           //  if nothing found, must ask user
      zmessageACK(Mwin,0,ZTX("image index is missing"));
      m_index(0,0);
      return;
   }

   for (ii = 0; ii < ntop; ii++) {                                         //  validate top directories
      err = stat(topdirks[ii],&statdat);
      if (err || ! S_ISDIR(statdat.st_mode)) break;
   }

   if (ii < ntop) {
      zmessageACK(Mwin,0,ZTX("invalid top image directory: %s"),topdirks[ii]);
      m_index(0,0);
      return;
   }

   if (! nthumb) {                                                         //  no thumbnails directory, must ask user
      zmessageACK(Mwin,0,ZTX("no thumbnails directory defined"));
      m_index(0,0);
      return;
   }
   
   if (! strstr(thumbdirk,"/thumbnails")) {
      zmessageACK(Mwin,0,ZTX("thumbnails directory not .../thumbnails"));
      m_index(0,0);
      return;
   }

   err = stat(thumbdirk,&statdat);                                         //  create thumbnails dir if needed    v.14.04.1
   if (err || ! S_ISDIR(statdat.st_mode)) {
      err = mkdir(thumbdirk,0750);
      if (err) {
         zmessageACK(Mwin,0,"%s \n %s",thumbdirk,strerror(errno));
         m_index(0,0);
         return;
      }
   }

   //  create popup window for reporting status and statistics
   
   Findexbusy = 1;                                                         //  index process is now busy

   if (zd_index_popup) zdialog_free(zd_index_popup);                       //  make dialog for output log
   zd_index_popup = zdialog_new(ZTX("Index Image Files"),Mwin,"OK",null);
   zdialog_add_widget(zd_index_popup,"frame","frame","dialog",0,"expand");
   zdialog_add_widget(zd_index_popup,"scrwin","scrwin","frame");
   zdialog_add_widget(zd_index_popup,"text","text","scrwin");

   poplog = zdialog_widget(zd_index_popup,"text");
   monofont = pango_font_description_from_string("monospace 9");
   gtk_widget_override_font(poplog,monofont);

   zdialog_resize(zd_index_popup,500,300);
   zdialog_run(zd_index_popup,index_popup_dialog_event,"parent");
   
   wprintf(poplog,"top image directories:\n");                             //  log top image directories
   for (ii = 0; ii < Ntopdirks; ii++)
      wprintf(poplog," %s\n",topdirks[ii]);

   wprintf(poplog,"thumbnails directory: \n %s \n",thumbdirk);             //  and thumbnails directory

   //  read image index file and build "old list" of index recs

   cc = maximages * sizeof(sxrec_t);
   sxrec_old = (sxrec_t *) zmalloc(cc);                                    //  "old" image index recs

   Nold = 0;
   ftf = 1;
   
   wprintf(poplog,"loading image index ...\n");
   
   zsleep(0.1);                                                            //  push outputs up to here            v.14.06
   zmainloop();

   while (true) {
      err = read_sxrec_seq(sxrec_old[Nold],ftf);                           //  read curr. index recs
      if (err) break;
      if (++Nold == maximages) {
         zmessageACK(Mwin,0,ZTX("exceeded max. images: %d"),maximages);
         Findexbusy = 0;
         m_quit(0,0);
      }
   }

   wprintf(poplog,"current index records found: %d \n",Nold);

   //  find all image files and create "new list" of index recs 

   cc = maximages * sizeof(sxrec_t);
   sxrec_new = (sxrec_t *) zmalloc(cc);                                    //  "new" image index recs

   Fupdate = (char *) zmalloc(maximages);                                  //  flags, 1 = update needed

   Nnew = 0;

   for (ii = 0; ii < Ntopdirks; ii++)
   {
      err = find_imagefiles(topdirks[ii],flist,NF,0);
      if (err) {
         zmessageACK(Mwin,0,strerror(errno));
         Findexbusy = 0;
         m_quit(0,0);
      }
      
      if (Nnew + NF > maximages) {
         zmessageACK(Mwin,0,ZTX("exceeded max. images: %d"),maximages);
         Findexbusy = 0;
         m_quit(0,0);
      }
      
      for (jj = 0; jj < NF; jj++) 
      {
         file = flist[jj];
         nrec = Nnew++;
         sxrec_new[nrec].file = zstrdup(file);                             //  image filespec
         stat(file,&statdat);
         compact_time(statdat.st_mtime,sxrec_new[nrec].fdate);             //  file mod date
         sxrec_new[nrec].pdate[0] = 0;                                     //  image date = empty
         strcpy(sxrec_new[nrec].rating,"0");                               //  stars = "0"
         strcpy(sxrec_new[nrec].size,"0x0");                               //  size = "0x0"
         sxrec_new[nrec].tags = 0;                                         //  tags = empty
         sxrec_new[nrec].capt = 0;                                         //  caption = empty
         sxrec_new[nrec].comms = 0;                                        //  comments = empty
         sxrec_new[nrec].gtags = 0;                                        //  geotags = empty
         zfree(file);
      }

      if (NF) zfree(flist);
   }

   wprintf(poplog,"image files found: %d \n",Nnew);
   printz("%d image files found \n",Nnew);

   //  merge and compare lists
   //  if filespecs match and have the same date, then "old" record is OK

   wprintf(poplog,"matching image files with index records \n");

   if (Nold)
      HeapSort((char *) sxrec_old,sizeof(sxrec_t),Nold,image_index_compare);
   if (Nnew)
      HeapSort((char *) sxrec_new,sizeof(sxrec_t),Nnew,image_index_compare);

   nmatch = 0;

   for (orec = nrec = 0; nrec < Nnew; nrec++)
   {
      Fupdate[nrec] = 1;                                                   //  assume nrec update is needed

      if (orec == Nold) continue;                                          //  no more old recs

      while (true)
      {
         if (orec == Nold) break;
         comp = strcasecmp(sxrec_old[orec].file,sxrec_new[nrec].file);     //  compare orec file to nrec file
         if (comp == 0)
            comp = strcmp(sxrec_old[orec].file,sxrec_new[nrec].file);
         if (comp >= 0) break;                                             //  orec >= nrec
         orec++;                                                           //  orec < nrec, next orec
         continue;
      }

      if (comp == 0)                                                       //  orec = nrec (same image file)
      {
         if (strEqu(sxrec_new[nrec].fdate,sxrec_old[orec].fdate))          //  file dates match
         {
            strncpy0(sxrec_new[nrec].pdate,sxrec_old[orec].pdate,15);      //  copy data from old to new
            strncpy0(sxrec_new[nrec].rating,sxrec_old[orec].rating,2);
            strncpy0(sxrec_new[nrec].size,sxrec_old[orec].size,15);        //  v.13.01
            sxrec_new[nrec].tags = sxrec_old[orec].tags;
            sxrec_old[orec].tags = 0;
            sxrec_new[nrec].capt = sxrec_old[orec].capt;
            sxrec_old[orec].capt = 0;
            sxrec_new[nrec].comms = sxrec_old[orec].comms;
            sxrec_old[orec].comms = 0;
            sxrec_new[nrec].gtags = sxrec_old[orec].gtags;
            sxrec_old[orec].gtags = 0;
            Fupdate[nrec] = 0;                                             //  nrec update not needed
            nmatch++;
         }
         orec++;                                                           //  next orec
      }
   }

   wprintf(poplog,"image files matching image index: %d \n",nmatch);

   for (orec = 0; orec < Nold; orec++)                                     //  release old list memory
   {
      zfree(sxrec_old[orec].file);
      if (sxrec_old[orec].tags) zfree(sxrec_old[orec].tags);
      if (sxrec_old[orec].capt) zfree(sxrec_old[orec].capt);
      if (sxrec_old[orec].comms) zfree(sxrec_old[orec].comms);
      if (sxrec_old[orec].gtags) zfree(sxrec_old[orec].gtags);
   }

   zfree(sxrec_old);

   //  Process entries needing update in new index list
   //  (new files or files dated later than image index date).
   //  Get updated metadata from image file EXIF/IPTC data.
   //  Create new thumbnails and refresh stale thumbnails.

   wprintf(poplog,"%s \n",ZTX("updating image index and thumbnails"));

   snprintf(tempfile,200,"%s/newfiles",get_zuserdir());                    //  temp file for new file list        v.13.07
   fidw = fopen(tempfile,"w");
   if (! fidw) {
      zmessLogACK(Mwin,"%s\n terminated by error",strerror(errno));
      Findexbusy = 0;
      m_quit(0,0);
   }
   
   Ndirtot = Nthumbtot = Nindextot = 0;
   Nfiledir = dotstep = 0;
   *currdir = 0;

   for (nrec = 0; nrec < Nnew; nrec++)                                     //  loop all image files
   {
      zmainloop();                                                         //  v.14.04.2

      file = sxrec_new[nrec].file;                                         //  a new or old image file

      pp = strrchr(file,'/');
      filename = pp + 1;                                                   //  file name
      
      cc = pp - file + 1;                                                  //  check if new directory start
      if (strncmp(file,currdir,cc) != 0) {
         if (*currdir) wprintf(poplog," %d \n",Ndirtot);                   //  output prior dirname ... NNN
         Ndirtot = 0;
         strncpy0(currdir,file,cc+1);
         wprintf(poplog," %s ",currdir);                                   //  start new dirname
         dotstep = 0;
         Nfiledir = 0;
      }
      
      newthumb = newindex = 0;                                             //  this file status: no updates

      thumbfile = image_thumbfile(file,&ff);                               //  create thumbnail if needed
      if (thumbfile) zfree(thumbfile);                                     //  OK
      if (ff) newthumb++;                                                  //  note if created

      if (Fupdate[nrec])                                                   //  index update needed
      {
         file = sxrec_new[nrec].file;                                      //  a new/mod file
         fprintf(fidw,"%s\n",file);                                        //  output to temp file                v.13.07

         ppv = exif_get(file,exifkeys,11);                                 //  get iptc/exif metadata
         if (! ppv) {
            zmessLogACK(Mwin,"exif_get() fatal error");
            Findexbusy = 0;
            m_quit(0,0);                                                   //  cannot continue
         }

         if (! ppv[0] || strNeq(ppv[0],filename)) {                        //  image file has no metadata
            printz("exif_get() failure: %s\n",file);
            if (ppv[0]) zfree(ppv[0]);
            continue;                                                      //  skip this image
         }
         else zfree(ppv[0]);

         exifdate = ppv[1];
         iptctags = ppv[2];
         iptcrating = ppv[3];
         exifsize = ppv[4];
         exifcomms = ppv[5];
         iptccapt = ppv[6];
         exifcity = ppv[7];
         exifcountry = ppv[8];
         exiflat = ppv[9];
         exiflong = ppv[10];

         if (exifdate && strlen(exifdate) > 3)                             //  exif date (photo date)
            exif_tagdate(exifdate,sxrec_new[nrec].pdate);
         else strcpy(sxrec_new[nrec].pdate,"null");                        //  not present

         if (iptcrating && strlen(iptcrating)) {                           //  iptc rating
            sxrec_new[nrec].rating[0] = *iptcrating;
            sxrec_new[nrec].rating[1] = 0;
         }
         else strcpy(sxrec_new[nrec].rating,"0");                          //  not present

         if (exifsize && strlen(exifsize))                                 //  exif size                          v.13.01
            strncpy0(sxrec_new[nrec].size,exifsize,15);

         if (iptctags && strlen(iptctags)) {                               //  iptc tags
            sxrec_new[nrec].tags = iptctags;
            iptctags = 0;
         }

         if (iptccapt && strlen(iptccapt)) {                               //  iptc caption
            sxrec_new[nrec].capt = iptccapt;
            iptccapt = 0;
         }

         if (exifcomms && strlen(exifcomms)) {                             //  exif comments
            sxrec_new[nrec].comms = exifcomms;
            exifcomms = 0;
         }

         strcpy(city2,"null");                                             //  geotags = empty
         strcpy(country2,"null");
         strcpy(lat2,"null");
         strcpy(long2,"null");

         if (exifcity) strncpy0(city2,exifcity,99);                        //  get from exif if any
         if (exifcountry) strncpy0(country2,exifcountry,99);
         if (exiflat) strncpy0(lat2,exiflat,10);
         if (exiflong) strncpy0(long2,exiflong,10);

         *gtags2 = 0;
         strncatv(gtags2,200,city2,"^ ",country2,"^ ",lat2,"^ ",long2,null);
         sxrec_new[nrec].gtags = zstrdup(gtags2);

         if (exifdate) zfree(exifdate);                                    //  free EXIF data
         if (iptctags) zfree(iptctags);
         if (iptcrating) zfree(iptcrating);
         if (exifsize) zfree(exifsize);                                    //  v.13.01
         if (exifcomms) zfree(exifcomms);
         if (iptccapt) zfree(iptccapt);
         if (exifcity) zfree(exifcity);
         if (exifcountry) zfree(exifcountry);
         if (exiflat) zfree(exiflat);
         if (exiflong) zfree(exiflong);
         
         newindex++;                                                       //  note index update
      }
      
      Nthumbtot += newthumb;                                               //  total thumbnail updates
      Nindextot += newindex;                                               //  total index updates
      if (newthumb || newindex) Ndirtot++;                                 //  updates in this directory

      if (Nfiledir++ > dotstep) {
         wprintf(poplog,".");                                              //  show progress                      v.13.04.1
         dotstep += 50 + 0.1 * dotstep;                                    //  50 105 165 231 304 384 ...
      }
   }

   wprintf(poplog," %d \n",Ndirtot);                                       //  last dirname ..... NNN

   wprintf(poplog,ZTX("%d image files updated \n"),Nindextot);             //  output statistics                  v.14.03
   printz("%d image files updated \n",Nindextot);
   wprintf(poplog,"%d thumbnails updated \n",Nthumbtot);
   printz("%d thumbnails updated \n",Nthumbtot);

   ftf = 1;
   for (nrec = 0; nrec < Nnew; nrec++) {                                   //  write updated image index recs.
      err = write_sxrec_seq(&sxrec_new[nrec],ftf);
      if (err) {
         zmessLogACK(Mwin,"terminated by error");
         Findexbusy = 0;
         m_quit(0,0);
      }
   }
   write_sxrec_seq(null,ftf);                                              //  close output

   for (nrec = 0; nrec < Nnew; nrec++)                                     //  release new list memory
   {
      zfree(sxrec_new[nrec].file);
      if (sxrec_new[nrec].tags) zfree(sxrec_new[nrec].tags);
      if (sxrec_new[nrec].capt) zfree(sxrec_new[nrec].capt);
      if (sxrec_new[nrec].comms) zfree(sxrec_new[nrec].comms);
      if (sxrec_new[nrec].gtags) zfree(sxrec_new[nrec].gtags);
   }

   zfree(sxrec_new);
   zfree(Fupdate);

   //  if started from menu, find orphan thumbnails and delete them

   if (menu)
   {
      wprintf(poplog,"delete orphan thumbnails \n");
      nthumb = 0;

      snprintf(command,ccc,"find %s -type f",thumbdirk);
      fid = popen(command,"r");
      if (! fid) {
         zmessLogACK(Mwin,"%s\n terminated by error",strerror(errno));
         Findexbusy = 0;
         m_quit(0,0);
      }
      
      while (true)
      {
         thumbfile = fgets_trim(buff,maxfcc,fid);
         if (! thumbfile) break;
         file = thumb2imagefile(thumbfile);
         if (file) { zfree(file); continue; }
         wprintf(poplog," delete: %s \n",thumbfile);
         remove(thumbfile);
         nthumb++;
      }
      
      pclose(fid);
      wprintf(poplog,"%d thumbnails deleted \n",nthumb);
      printz("%d thumbnails deleted \n",nthumb);
   }

   //  generate list of newest files added

   if (Nindextot == 0) {                                                   //  no new files found
      fclose(fidw);
      remove(tempfile);
   }
   
   else if (Nindextot > 199)                                               //  200 or more new files              v.13.07
   {
      fclose(fidw);                                                        //  copy 1st 100 new files
      fidr = fopen(tempfile,"r");                                          //    to newfiles_file
      fidw = fopen(newfiles_file,"w");
      if (! fidr || ! fidw) {
         zmessLogACK(Mwin,"%s\n terminated by error",strerror(errno));
         Findexbusy = 0;
         m_quit(0,0);
      }
      for (ii = 0; ii < 200; ii++) {
         file = fgets_trim(buff,maxfcc,fidr);
         if (! file) break;
         fprintf(fidw,"%s\n",file);
      }
      fclose(fidr);
      fclose(fidw);
      remove(tempfile);
   }

   else                                                                    //  less than 200 new files            v.13.07
   {
      fidr = fopen(newfiles_file,"r");
      if (fidr) {                                                          //  add newest of prior new files
         for (ii = 200 - Nindextot; ii; ii--) {                            //    to reach total of 200
            file = fgets_trim(buff,maxfcc,fidr);                           //  duplicates are not eliminated
            if (! file) break;
            fprintf(fidw,"%s\n",file);
         }
         fclose(fidr);
      }
      fclose(fidw);
      rename(tempfile,newfiles_file);
   }

   if (! menu)                                                             //  automatic mode, make popup
      g_timeout_add_seconds(2,(GSourceFunc) index_popup_timeout,0);        //    go away automatically

   Findexbusy = 0;                                                         //  index complete and OK
   Findexdone = 1;

   wprintf(poplog,"%s\n",ZTX("COMPLETED"));
   printz("index image files completed \n");
   return;
}


//  index popup dialog response function

int index_popup_dialog_event(zdialog *zd, cchar *event)
{
   if (strEqu(event,"timeout")) zd->zstat = 1;                             //  timeout after completion
   if (strEqu(event,"enter")) zd->zstat = 1;                               //  v.14.03

   if (! zd->zstat) return 0;                                              //  dialog still open
   if (zd->zstat == -1 && Findexbusy) {
      zmessageACK(Mwin,0,ZTX("terminated by user"));                       //  user cancelled while index running
      Findexbusy = 0;
      m_quit(0,0);
   }

   if (Findexbusy) zd->zstat = 0;                                          //  keep open during index busy
   else zdialog_free(zd);                                                  //  kill
   zd_index_popup = 0;
   return 0;
}


//  index popup timeout function

int index_popup_timeout(void *)
{
   if (zd_index_popup) 
      zdialog_send_event(zd_index_popup,"timeout");
   return 0;
}


//  sort compare function - compare index record filespecs and return
//   <0 | 0 | >0   for   file1 < | == | > file2

int image_index_compare(cchar *rec1, cchar *rec2)
{
   char * file1 = ((sxrec_t *) rec1)->file;
   char * file2 = ((sxrec_t *) rec2)->file;
   int nn = strcasecmp(file1,file2);                                       //  compare ignoring case
   if (nn) return nn;
   nn = strcmp(file1,file2);                                               //  if equal, use utf8 compare
   return nn;
}


/**************************************************************************/

//  user settings dialog

int      settings_fclone;

void m_settings(GtkWidget *, cchar *)
{
   int   settings_dialog_event(zdialog *zd, cchar *event);

   zdialog  *zd;

/***
       ______________________________________________________________
      |                        Settings                              |
      |  Startup Display                                             |
      |    (o) Recent Files Gallery                                  |
      |    (o) Newest Files Gallery                                  |
      |    (o) Previous Gallery                                      |
      |    (o) Previous Image                                        |
      |    (o) Blank Window                                          |
      |    (o) Directory [________________________________] [browse] |
      |    (o) Image File [_______________________________] [browse] |
      |                                                              |
      |  Menu Style        (o) Icons  (o) Icons + Text               |
      |  Image Pan         (o) Drag  (o) Scroll  [x] Magnified       |
      |  Interpolation     (o) tiles  (o) bilinear  (o) hyperbolic   |
      |  JPEG quality      [___|-+]                                  |
      |  Zooms for 2x      [___|-+]                                  |
      |  RAW command       [_______________________________________] |
      |  RAW file types    [_______________________________________] |
      |                                                              |
      |                                                       [done] |
      |______________________________________________________________|
      
***/

   F1_help_topic = "user_settings";

   if (checkpend("all")) return;                                           //  check nothing pending              v.13.12

   zd = zdialog_new(ZTX("Settings"),Mwin,Bdone,null);

   zdialog_add_widget(zd,"hbox","hbstart","dialog",0,"space=3");
   zdialog_add_widget(zd,"label","labstart","hbstart",ZTX("Startup Display"),"space=5");
   zdialog_add_widget(zd,"hbox","hbrecent","dialog");
   zdialog_add_widget(zd,"check","recent","hbrecent",ZTX("Recent Files Gallery"),"space=20");
   zdialog_add_widget(zd,"hbox","hbnewest","dialog");
   zdialog_add_widget(zd,"check","newest","hbnewest",ZTX("Newest Files Gallery"),"space=20");
   zdialog_add_widget(zd,"hbox","hbprevG","dialog");
   zdialog_add_widget(zd,"check","prevG","hbprevG",ZTX("Previous Gallery"),"space=20");
   zdialog_add_widget(zd,"hbox","hbprevI","dialog");
   zdialog_add_widget(zd,"check","prevI","hbprevI",ZTX("Previous Image"),"space=20");
   zdialog_add_widget(zd,"hbox","hbblank","dialog");
   zdialog_add_widget(zd,"check","blank","hbblank",ZTX("Blank Window"),"space=20");
   zdialog_add_widget(zd,"hbox","hbdirk","dialog");
   zdialog_add_widget(zd,"check","dirk","hbdirk",ZTX("Directory Gallery"),"space=20");
   zdialog_add_widget(zd,"entry","stdirk","hbdirk",0,"expand|space=5");
   zdialog_add_widget(zd,"button","stdbrowse","hbdirk",Bbrowse);
   zdialog_add_widget(zd,"label","space","hbdirk",0,"space=5");
   zdialog_add_widget(zd,"hbox","hbfile","dialog");
   zdialog_add_widget(zd,"check","file","hbfile",ZTX("Image File"),"space=20");
   zdialog_add_widget(zd,"entry","stfile","hbfile",0,"expand|space=5");
   zdialog_add_widget(zd,"button","stfbrowse","hbfile",Bbrowse);
   zdialog_add_widget(zd,"label","space","hbfile",0,"space=5");

   zdialog_add_widget(zd,"hbox","hb1","dialog",0,"space=3");
   zdialog_add_widget(zd,"label","space","hb1",0,"space=3");
   zdialog_add_widget(zd,"vbox","vb1","hb1",0,"homog");
   zdialog_add_widget(zd,"label","space","hb1",0,"space=5");
   zdialog_add_widget(zd,"vbox","vb2","hb1",0,"homog|expand");
   zdialog_add_widget(zd,"label","space","hb1",0,"space=3");

   zdialog_add_widget(zd,"label","labtbs","vb1",ZTX("Menu Style"));
   zdialog_add_widget(zd,"label","labpan","vb1",ZTX("Image Pan"));
   zdialog_add_widget(zd,"label","labintp","vb1","Interpolation");
   zdialog_add_widget(zd,"label","labqual","vb1",ZTX("JPEG quality"));
   zdialog_add_widget(zd,"label","labzoom","vb1",ZTX("Zooms for 2x"));
   zdialog_add_widget(zd,"label","rawlab1","vb1",ZTX("RAW command"));
   zdialog_add_widget(zd,"label","rawlab2","vb1",ZTX("RAW file types"));

   zdialog_add_widget(zd,"hbox","hbmenu","vb2");
   zdialog_add_widget(zd,"radio","icons","hbmenu",ZTX("Icons"));
   zdialog_add_widget(zd,"radio","both","hbmenu",ZTX("Icons + Text"),"space=10");

   zdialog_add_widget(zd,"hbox","hbpan","vb2");
   zdialog_add_widget(zd,"radio","m-drag","hbpan",ZTX("Drag"));
   zdialog_add_widget(zd,"radio","m-scroll","hbpan",ZTX("Scroll"),"space=12");
   zdialog_add_widget(zd,"check","m-mag","hbpan",ZTX("Magnified"));

   zdialog_add_widget(zd,"hbox","hbintp","vb2");
   zdialog_add_widget(zd,"radio","tiles","hbintp","tiles");
   zdialog_add_widget(zd,"radio","bilinear","hbintp","bilinear","space=12");
   zdialog_add_widget(zd,"radio","hyperbolic","hbintp","hyperbolic");

   zdialog_add_widget(zd,"hbox","hbqual","vb2");
   zdialog_add_widget(zd,"spin","quality","hbqual","1|99|1|90");

   zdialog_add_widget(zd,"hbox","hbzoom","vb2");
   zdialog_add_widget(zd,"spin","zoom","hbzoom","1|4|1|2");

   zdialog_add_widget(zd,"entry","rawcomm","vb2","dcraw -w -q 0 -T -6","expand");
   zdialog_add_widget(zd,"entry","rawtypes","vb2",".raw .rw2","expand");

   zdialog_stuff(zd,"stdirk",startdirk);                                   //  stuff present settings into dialog
   zdialog_stuff(zd,"stfile",startfile);

   zdialog_stuff(zd,"recent",0);
   zdialog_stuff(zd,"newest",0);
   zdialog_stuff(zd,"prevG",0);
   zdialog_stuff(zd,"prevI",0);
   zdialog_stuff(zd,"blank",0);
   zdialog_stuff(zd,"dirk",0);
   zdialog_stuff(zd,"file",0);
   zdialog_stuff(zd,startdisplay,1);

   zdialog_stuff(zd,"icons",0);                                            //  default menu style
   zdialog_stuff(zd,"both",0);
   zdialog_stuff(zd,menu_style,1);

   if (Fdragopt == 1) {                                                    //  drag 1x                         v.13.05
      zdialog_stuff(zd,"m-drag",1);
      zdialog_stuff(zd,"m-scroll",0);
      zdialog_stuff(zd,"m-mag",0);
   }

   if (Fdragopt == 2) {                                                    //  scroll 1x
      zdialog_stuff(zd,"m-drag",0);
      zdialog_stuff(zd,"m-scroll",1);
      zdialog_stuff(zd,"m-mag",0);
   }

   if (Fdragopt == 3) {                                                    //  drag magnified
      zdialog_stuff(zd,"m-drag",1);
      zdialog_stuff(zd,"m-scroll",0);
      zdialog_stuff(zd,"m-mag",1);
   }

   if (Fdragopt == 4) {                                                    //  scroll magnified
      zdialog_stuff(zd,"m-drag",0);
      zdialog_stuff(zd,"m-scroll",1);
      zdialog_stuff(zd,"m-mag",1);
   }

   zdialog_stuff(zd,"quality",jpeg_def_quality);                           //  default jpeg file save quality  v.13.08

   if (zoomcount >= 1 && zoomcount <= 4)                                   //  zooms for 2x increase
      zdialog_stuff(zd,"zoom",zoomcount);

   zdialog_stuff(zd,"tiles",0);                                            //  interpolation type              v.13.05
   zdialog_stuff(zd,"bilinear",0);
   zdialog_stuff(zd,"hyperbolic",0);
   zdialog_stuff(zd,interpolation,1);

   zdialog_stuff(zd,"rawcomm",DCrawcommand);
   zdialog_stuff(zd,"rawtypes",RAWfiletypes);

   zdialog_resize(zd,500,0);
   
   settings_fclone = 0;

   zdialog_run(zd,settings_dialog_event);                                  //  run dialog and wait for completion
   zdialog_wait(zd);
   zdialog_free(zd);
   
   if (settings_fclone) {                                                  //  start new and exit current fotoxx
      m_clone(0,0);                                                        //  v.14.06
      m_quit(0,0);
   }

   return;
}


//  settings dialog event function

int settings_dialog_event(zdialog *zd, cchar *event)
{
   int         ii, jj, nn;
   char        *pp, temp[200];
   
   if (strEqu(event,"enter")) zd->zstat = 1;                               //  [done]  v.14.03

   if (zd->zstat && zd->zstat != 1) return 0;                              //  cancel

   if (strstr("recent newest prevG prevI blank dirk file",event)) {        //  make checkboxes work like radio buttons
      zdialog_stuff(zd,"recent",0);                                        //  v.13.11
      zdialog_stuff(zd,"newest",0);
      zdialog_stuff(zd,"prevG",0);
      zdialog_stuff(zd,"prevI",0);
      zdialog_stuff(zd,"blank",0);
      zdialog_stuff(zd,"dirk",0);
      zdialog_stuff(zd,"file",0);
      zdialog_stuff(zd,event,1);
   }

   if (zd->zstat == 1)                                                     //  done, check inputs are valid
   {
      zdialog_fetch(zd,"recent",nn);                                       //  startup display options
      if (nn) {
         zfree(startdisplay);
         startdisplay = zstrdup("recent");                                 //  recent files gallery
      }

      zdialog_fetch(zd,"newest",nn);
      if (nn) {
         zfree(startdisplay);
         startdisplay = zstrdup("newest");                                 //  newest files gallery            v.13.07
      }

      zdialog_fetch(zd,"prevG",nn);
      if (nn) {
         zfree(startdisplay);
         startdisplay = zstrdup("prevG");                                  //  start with gallery from last session
      }

      zdialog_fetch(zd,"prevI",nn);
      if (nn) {
         zfree(startdisplay);
         startdisplay = zstrdup("prevI");                                  //  start with image file from last session
      }

      zdialog_fetch(zd,"blank",nn);
      if (nn) {
         zfree(startdisplay);
         startdisplay = zstrdup("blank");                                  //  start with blank window
      }

      zdialog_fetch(zd,"dirk",nn);
      if (nn) {
         zfree(startdisplay);
         startdisplay = zstrdup("dirk");                                   //  start with directory gallery
      }

      zdialog_fetch(zd,"file",nn);
      if (nn) {
         zfree(startdisplay);
         startdisplay = zstrdup("file");                                   //  start with specified file
      }

      zdialog_fetch(zd,"stdirk",temp,200);                                 //  startup directory gallery
      if (startdirk) zfree(startdirk);
      startdirk = zstrdup(temp);

      zdialog_fetch(zd,"stfile",temp,200);                                 //  startup image file
      if (startfile) zfree(startfile);
      startfile = zstrdup(temp);

      if (strEqu(startdisplay,"dirk")) {                                   //  startup option = directory gallery
         if (image_file_type(startdirk) != 1) {
            zmessageACK(Mwin,0,ZTX("startup directory is invalid"));
            zd->zstat = 0;
            return 0;
         }
      }

      if (strEqu(startdisplay,"file")) {                                   //  startup option = image file
         if (image_file_type(startfile) != 2) {
            zmessageACK(Mwin,0,ZTX("startup file is invalid"));
            zd->zstat = 0;
            return 0;
         }
      }
      
      zdialog_fetch(zd,"icons",nn);                                        //  menu type icons
      if (nn) {
         if (strNeq(menu_style,"icons")) {
            zfree(menu_style);
            menu_style = zstrdup("icons");
            settings_fclone++;
         }
      }

      zdialog_fetch(zd,"both",nn);                                         //  menu type icons + text
      if (nn) {
         if (strNeq(menu_style,"both")) {
            zfree(menu_style);
            menu_style = zstrdup("both");
            settings_fclone++;
         }
      }
      
      zdialog_fetch(zd,"m-drag",nn);                                       //  Fdragopt = 1/2 = drag/scroll
      Fdragopt = 2 - nn;
      zdialog_fetch(zd,"m-mag",nn);                                        //  v.13.05
      Fdragopt += 2 * nn;                                                  //  Fdragopt = 3/4 = drag/scroll magnified

      zdialog_fetch(zd,"quality",jpeg_def_quality);                        //  default jpeg file save quality

      zdialog_fetch(zd,"zoom",zoomcount);                                  //  zooms for 2x image, 1-4
      zoomratio = pow( 2.0, 1.0 / zoomcount);                              //  2.0, 1.4142, 1.2599, 1.1892

      zdialog_fetch(zd,"tiles",nn);                                        //  image interpolation type
      if (nn) {
         zfree(interpolation);
         interpolation = zstrdup("tiles");
      }

      zdialog_fetch(zd,"bilinear",nn);
      if (nn) {
         zfree(interpolation);
         interpolation = zstrdup("bilinear");
      }

      zdialog_fetch(zd,"hyperbolic",nn);
      if (nn) {
         zfree(interpolation);
         interpolation = zstrdup("hyperbolic");
      }

      zdialog_fetch(zd,"rawcomm",temp,200);                                //  command for RAW to tiff using DCraw
      if (DCrawcommand) zfree(DCrawcommand);
      DCrawcommand = zstrdup(temp);

      zdialog_fetch(zd,"rawtypes",temp,200);                               //  RAW file types, .raw .rw2 ...
      pp = zstrdup(temp,100);

      for (ii = jj = 0; temp[ii]; ii++) {                                  //  insure blanks between .raw types
         if (temp[ii] == '.' && temp[ii-1] != ' ') pp[jj++] = ' ';
         pp[jj++] = temp[ii];
      }
      if (pp[jj-1] != ' ') pp[jj++] = ' ';                                 //  insure 1 final blank               v.14.07
      pp[jj] = 0;

      if (RAWfiletypes) zfree(RAWfiletypes);
      RAWfiletypes = pp;

      save_params();                                                       //  done, save modified parameters
      return 0;
   }

   if (strEqu(event,"stdbrowse")) {                                        //  startup directory browse
      zdialog_fetch(zd,"stdirk",temp,200);
      pp = zgetfile(ZTX("Select startup directory"),"folder",temp);
      if (! pp) return 0;
      zdialog_stuff(zd,"stdirk",pp);
      zfree(pp);
   }

   if (strEqu(event,"stfbrowse")) {                                        //  startup file browse
      zdialog_fetch(zd,"stfile",temp,200);
      pp = zgetfile(ZTX("Select startup image file"),"file",temp);
      if (! pp) return 0;
      zdialog_stuff(zd,"stfile",pp);
      zfree(pp);
   }

   return 0;
}


/**************************************************************************/

//  edit bookmarks                                                         //  v.13.03

namespace bookmarknames
{
   #define     maxbmks 40
   char        *bookmarks[maxbmks];                                        //  bookmark names and files
   int         Nbmks;                                                      //  count of entries
   int         bmkposn;                                                    //  current entry, 0-last
   GtkWidget   *bmkswidget;
   zdialog     *zd_goto_bookmarks;
}

void bookmarks_listclick(GtkWidget *widget, int line, int pos);
int  bookmarks_dialog_event(zdialog *zd, cchar *event);
void bookmark_refresh();


void m_edit_bookmarks(GtkWidget *, cchar *)
{
   using namespace bookmarknames;

   int         err;
   char        buff[maxfcc], bmkfile[200];
   char        *pp;
   FILE        *fid;
   zdialog     *zd;
   cchar       *bmk_add = ZTX("Click list position. Click thumbnail to add.");

/***
          _______________________________________________
         |    Edit Bookmarks                             |
         |                                               |
         | Click list position. Click thumbnail to add.  |
         |-----------------------------------------------|
         | bookmarkname1      /topdir/.../filename1.jpg  |
         | bookmarkname2      /topdir/.../filename2.jpg  |
         | bookmarkname3      /topdir/.../filename3.jpg  |
         | bookmarkname4      /topdir/.../filename4.jpg  |
         | bookmarkname5      /topdir/.../filename5.jpg  |
         | bookmarkname6      /topdir/.../filename6.jpg  |
         |-----------------------------------------------|
         | [bookmarkname...] [rename] [delete]           |
         |                                        [done] |
         |_______________________________________________|

***/

   F1_help_topic = "bookmarks";
   if (zd_edit_bookmarks) return;                                          //  already busy
   if (zd_goto_bookmarks) return;

   zd = zdialog_new(ZTX("Edit Bookmarks"),Mwin,Bdone,null);
   zd_edit_bookmarks = zd;
   zdialog_add_widget(zd,"hbox","hbtip","dialog",0,"space=3");
   zdialog_add_widget(zd,"label","labtip","hbtip",bmk_add,"space=5");
   zdialog_add_widget(zd,"hbox","hbbmk","dialog",0,"space=5|expand");
   zdialog_add_widget(zd,"frame","frbmk","hbbmk",0,"space=5|expand");
   zdialog_add_widget(zd,"text","bmklist","frbmk",0,"expand");
   zdialog_add_widget(zd,"hbox","hbname","dialog",0,"space=5");
   zdialog_add_widget(zd,"entry","bmkname","hbname",0,"space=5|scc=30");
   zdialog_add_widget(zd,"button","rename","hbname",Brename,"space=5");
   zdialog_add_widget(zd,"button","delete","hbname",Bdelete,"space=5");

   bmkswidget = zdialog_widget(zd,"bmklist");                              //  connect mouse to bookmark list
   textwidget_set_clickfunc(bmkswidget,bookmarks_listclick);

   Nbmks = 0;
   err = locale_filespec("user","bookmarks",bmkfile);                      //  read bookmarks file
   if (! err) {
      fid = fopen(bmkfile,"r");
      if (fid) {
         while (true) {
            pp = fgets_trim(buff,maxfcc,fid,1);                            //  next bookmark rec.
            if (! pp) break;
            bookmarks[Nbmks] = zstrdup(pp);                                //  fill bookmark list
            if (++Nbmks == maxbmks) break;
         }
         fclose(fid);
      }
   }

   bookmark_refresh();                                                     //  update bookmarks list in dialog
   bmkposn = -1;                                                           //  list position = undefined

   zdialog_resize(zd,300,300);
   zdialog_run(zd,bookmarks_dialog_event,"save");                          //  run dialog, parallel

   m_viewmode(0,"G");                                                      //  show current gallery
   return;
}


//  mouse click function to select existing bookmark from list

void bookmarks_listclick(GtkWidget *widget, int line, int pos)
{
   using namespace bookmarknames;

   char     bookmarkname[32];

   if (! zd_edit_bookmarks) return;
   if (line < 0 || line > Nbmks-1) return;
   bmkposn = line;
   strncpy0(bookmarkname,bookmarks[bmkposn],31);
   strTrim(bookmarkname);
   zdialog_stuff(zd_edit_bookmarks,"bmkname",bookmarkname);
   return;
}


//  mouse click function to receive clicked thumbnails

void bookmarks_Lclick_func(int Nth)
{
   using namespace bookmarknames;

   char     *imagefile, *newbookmark;
   char     *pp, bookmarkname[32];
   int      cc;

   if (! zd_edit_bookmarks) return;
   if (Nth < 0) return;                                                    //  gallery gone ?
   imagefile = gallery(0,"find",Nth);                                      //  get file at clicked position
   if (! imagefile) return;

   pp = strrchr(imagefile,'/');                                            //  get file name or last subdirk name
   if (! pp) return;                                                       //    to use as default bookmark name
   strncpy0(bookmarkname,pp+1,31);                                         //  max. 30 chars. + null

   cc = strlen(imagefile) + 34;                                            //  construct bookmark record:
   newbookmark = (char *) zmalloc(cc);                                     //    filename  /directories.../filename
   sprintf(newbookmark,"%-30s  %s",bookmarkname,imagefile);
   zfree(imagefile);

   if (Nbmks == maxbmks) {                                                 //  if list full, remove first
      zfree(bookmarks[0]);
      Nbmks--;
      for (int ii = 0; ii < Nbmks; ii++)
         bookmarks[ii] = bookmarks[ii+1];
   }

   if (bmkposn < 0) bmkposn = 0;
   if (bmkposn >= Nbmks-1) bmkposn = Nbmks;                                //  use next position

   for (int ii = Nbmks; ii > bmkposn; ii--)                                //  make hole to insert new bookmark
      bookmarks[ii] = bookmarks[ii-1];

   bookmarks[bmkposn] = newbookmark;                                       //  insert
   Nbmks++;

   bookmark_refresh();                                                     //  update bookmarks list in dialog

   zdialog_stuff(zd_edit_bookmarks,"bmkname",bookmarkname);

   return;
}


//  dialog event and completion function

int bookmarks_dialog_event(zdialog *zd, cchar *event)
{
   using namespace bookmarknames;

   char        bmkfile[200];
   char        bookmarkname[32];
   FILE        *fid;
   int         cc;

   if (strEqu(event,"delete"))                                             //  delete bookmark at position
   {
      if (bmkposn < 0 || bmkposn > Nbmks-1) return 1;
      for (int ii = bmkposn; ii < Nbmks-1; ii++)
         bookmarks[ii] = bookmarks[ii+1];
      Nbmks--;
      zdialog_stuff(zd,"bmkname","");                                      //  clear name field
      bookmark_refresh();                                                  //  update bookmarks list in dialog
   }

   if (strEqu(event,"rename"))                                             //  apply new name to bookmark
   {
      if (bmkposn < 0 || bmkposn > Nbmks-1) return 1;
      zdialog_fetch(zd,"bmkname",bookmarkname,31);                         //  get name from dialog
      cc = strlen(bookmarkname);
      if (cc < 30) memset(bookmarkname+cc,' ',30-cc);                      //  blank pad to 30 chars.
      bookmarkname[30] = 0;
      strncpy(bookmarks[bmkposn],bookmarkname,30);                         //  replace name in bookmarks list
      bookmark_refresh();                                                  //  update bookmarks list in dialog
   }

   if (strEqu(event,"enter")) zd->zstat = 1;                               //  [done]  v.14.03

   if (! zd->zstat) return 1;                                              //  wait for completion

   if (zd->zstat == 1)                                                     //  done
   {
      locale_filespec("user","bookmarks",bmkfile);                         //  write bookmarks file
      fid = fopen(bmkfile,"w");
      if (! fid)
         zmessageACK(Mwin,0,ZTX("unable to save bookmarks file"));
      else {
         for (int ii = 0; ii < Nbmks; ii++)
            fprintf(fid,"%s\n",bookmarks[ii]);
         fclose(fid);
      }
   }

   for (int ii = 0; ii < Nbmks; ii++)                                      //  free memory
      zfree(bookmarks[ii]);

   zdialog_free(zd);
   zd_edit_bookmarks = 0;
   return 1;
}


//  private function to update dialog widget with new bookmarks list

void bookmark_refresh()
{
   using namespace bookmarknames;

   if (! zd_edit_bookmarks && ! zd_goto_bookmarks) return;
   wclear(bmkswidget);                                                     //  clear bookmarks list
   for (int ii = 0; ii < Nbmks; ii++) {                                    //  write bookmarks list
      wprintx(bmkswidget,0,bookmarks[ii],"monospace 9");
      wprintx(bmkswidget,0,"\n");
   }
   return;
}


//  select a bookmark and jump gallery to selected bookmark
//  connected to gallery menu "GoTo" button

void m_goto_bookmark(GtkWidget *, cchar *)                                 //  v.13.03
{
   using namespace bookmarknames;

   void goto_bookmark_listclick(GtkWidget *widget, int line, int pos);
   int  goto_bookmark_dialog_event(zdialog *zd, cchar *event);

   int         err, zstat;
   char        buff[maxfcc], bmkfile[200];
   char        *pp1;
   FILE        *fid;
   zdialog     *zd;

/***
                  Gallery Button: [GoTo]
          _______________________________________________
         |      Go to Bookmark                           |
         |-----------------------------------------------|
         | bookmarkname1      /topdir/.../filename1.jpg  |
         | bookmarkname2      /topdir/.../filename2.jpg  |
         | bookmarkname3      /topdir/.../filename3.jpg  |
         | bookmarkname4      /topdir/.../filename4.jpg  |
         | bookmarkname5      /topdir/.../filename5.jpg  |
         | bookmarkname6      /topdir/.../filename6.jpg  |
         |                                               |
         |                              [edit bookmarks] |
         |_______________________________________________|
***/

   F1_help_topic = "bookmarks";
   if (zd_edit_bookmarks) return;                                          //  already busy
   if (zd_goto_bookmarks) return;

   zd = zdialog_new(ZTX("Go To Bookmark"),Mwin,ZTX("Edit Bookmarks"),null);
   zd_goto_bookmarks = zd;

   zdialog_add_widget(zd,"hbox","hbbmk","dialog",0,"space=5|expand");
   zdialog_add_widget(zd,"frame","frbmk","hbbmk",0,"space=5|expand");
   zdialog_add_widget(zd,"text","bmklist","frbmk",0,"expand");

   bmkswidget = zdialog_widget(zd,"bmklist");                              //  connect mouse to bookmark list
   textwidget_set_clickfunc(bmkswidget,goto_bookmark_listclick);

   Nbmks = 0;
   err = locale_filespec("user","bookmarks",bmkfile);                      //  read bookmarks file
   if (! err) {
      fid = fopen(bmkfile,"r");
      if (fid) {
         while (true) {
            pp1 = fgets_trim(buff,maxfcc,fid,1);                           //  fill bookmarks[] bookmark list
            if (! pp1) break;
            bookmarks[Nbmks] = zstrdup(pp1);
            if (++Nbmks == maxbmks) break;
         }
         fclose(fid);
      }
   }

   bookmark_refresh();                                                     //  update bookmarks list in dialog

   zdialog_resize(zd,300,0);
   zdialog_run(zd,0,"mouse");                                              //  run dialog
   zstat = zdialog_wait(zd);
   zdialog_free(zd);
   zd_goto_bookmarks = 0;
   if (zstat == 1) m_edit_bookmarks(0,0);
   return;
}


//  mouse click function to receive clicked bookmarks

void goto_bookmark_listclick(GtkWidget *widget, int line, int pos)
{
   using namespace bookmarknames;

   char        *file;
   int         err;
   struct stat sbuff;
   
   if (checkpend("edit lock")) return;                                     //  v.13.12

   if (! zd_goto_bookmarks) return;
   zdialog_free(zd_goto_bookmarks);                                        //  kill dialog
   zd_goto_bookmarks = 0;

   bmkposn = line;                                                         //  get clicked line
   if (bmkposn < 0 || bmkposn > Nbmks-1) return;
   file = bookmarks[bmkposn] + 32;
   err = stat(file,&sbuff);
   if (err) {
      zmessageACK(Mwin,0,ZTX("file not found"));
      return;
   }

   gallery(file,"init");                                                   //  go to gallery and file position
   gallery(file,"paint");
   m_viewmode(0,"G");

   return;
}


/**************************************************************************/

//  edit keyboard shortcuts                                                //  v.13.03

namespace KBshortcutnames
{
   zdialog     *zd;

   char        *shortcutkey2[maxshortcuts];                                //  KB shortcut keys (or combinations)
   char        *shortcutmenu2[maxshortcuts];                               //  corresponding menu names
   int         Nshortcuts2;                                                //  count of entries

   int         Nreserved = 8;                                              //  reserved shortcuts (hard coded)
   cchar       *reserved[8] = { "F", "G", "W", "F1", "+", "=", "-", "Z" };
   int         Neligible = 25;
   cchar       *eligible[25];                                              //  allowed shortcuts         v.13.04
   cchar       *eligible_en[25] = {
               "Open Image File", "Open Previous File", "Rename Image File",
               "Save to Disk","Trash Image File", "Quit Fotoxx", 
               "Show RGB", "Grid Lines", "Brightness Histogram",
               "Edit Metadata", "View Metadata (short)", "Edit Geotags",
               "Select", "Trim/Rotate", "Resize", "Add Text",
               "Voodoo Enhance", "Retouch Combo", "Tone Mapping", "Brightness Distribution",
               "Sharpen", "Red Eyes", "Undo", "Redo", "Keyboard Shortcuts" };
}

void KBshorts_clickfunc(GtkWidget *widget, int line, int pos);
int  KBshorts_keyfunc(GtkWidget *dialog, GdkEventKey *event);
int  KBshorts_dialog_event(zdialog *zd, cchar *event);


void m_KBshortcuts(GtkWidget *, cchar *)
{
   using namespace KBshortcutnames;

   int         ii, cc;
   GtkWidget   *widget;
   char        textline[100];

/***
          _______________________________________________
         |              Edit KB Shortcuts                |
         |_______________________________________________|
         | Alt+G                  Grid Lines             |
         | Alt+U                  Undo                   |
         | Alt+R                  Redo                   |
         | T                      Trim/Rotate            |
         | ...                    ...                    |
         |_______________________________________________|
         |                                               |
         | shortcut key: [ CTRL+G ] [ (dropdown list)  ] |
         |                                               |
         |                      [delete] [done] [cancel] |
         |_______________________________________________|

***/

   F1_help_topic = "KB_shortcuts";

   zd = zdialog_new(ZTX("Edit KB Shortcuts"),Mwin,Bdelete,Bdone,Bcancel,null);
   zdialog_add_widget(zd,"hbox","hblist","dialog",0,"expand");
   zdialog_add_widget(zd,"frame","frlist","hblist",0,"space=5|expand");
   zdialog_add_widget(zd,"text","shortlist","frlist",0,"expand");
   zdialog_add_widget(zd,"hbox","hbshort","dialog",0,"space=5");
   zdialog_add_widget(zd,"label","labshort","hbshort",ZTX("shortcut key:"),"space=5");
   zdialog_add_widget(zd,"label","shortkey","hbshort",ZTX("(enter key)"));
   zdialog_add_widget(zd,"label","space","hbshort",0,"expand");
   zdialog_add_widget(zd,"comboE","shortmenu","hbshort",0,"space=5");

   widget = zdialog_widget(zd,"shortlist");
   textwidget_set_clickfunc(widget,KBshorts_clickfunc);

   cc = maxshortcuts * sizeof(char *);
   memset(shortcutkey2,0,cc);
   memset(shortcutmenu2,0,cc);

   for (ii = 0; ii < Neligible; ii++)                                      //  use translated menu names
      eligible[ii] = ZTX(eligible_en[ii]);

   for (ii = 0; ii < Nshortcuts; ii++) {                                   //  copy global shortcuts list
      shortcutkey2[ii] = zstrdup(shortcutkey[ii]);                         //    for subsequent editing
      shortcutmenu2[ii] = zstrdup(shortcutmenu[ii]);
   }
   Nshortcuts2 = Nshortcuts;

   wclear(widget);

   for (int ii = 0; ii < Nshortcuts2; ii++) {                              //  show shortcuts list in dialog
      snprintf(textline,100,"%-20s %s \n",shortcutkey2[ii],shortcutmenu2[ii]);
      wprintx(widget,0,textline,"monospace 9");
   }

   for (int ii = 0; ii < Neligible; ii++)                                  //  add eligible menu names to
      zdialog_cb_app(zd,"shortmenu",eligible[ii]);                         //    combobox dropdown list

   widget = zdialog_widget(zd,"dialog");
   G_SIGNAL(widget,"key-press-event",KBshorts_keyfunc,0);
   G_SIGNAL(widget,"key-release-event",KBshorts_keyfunc,0);

   zdialog_resize(zd,420,0);
   zdialog_run(zd,KBshorts_dialog_event,"save");
   return;
}


//  mouse click functions to select existing shortcut from list

void KBshorts_clickfunc(GtkWidget *widget, int line, int pos)
{
   using namespace KBshortcutnames;

   char     *txline;
   char     shortkey[20];
   char     shortmenu[60];

   txline = textwidget_get_line(widget,line,1);
   if (! txline) return;

   strncpy0(shortkey,txline,20);
   strncpy0(shortmenu,txline+21,60);
   zfree(txline);

   strTrim(shortkey);
   strTrim(shortmenu);

   zdialog_stuff(zd,"shortkey",shortkey);
   zdialog_stuff(zd,"shortmenu",shortmenu);

   return;
}


//  intercept KB key events

int KBshorts_keyfunc(GtkWidget *dialog, GdkEventKey *event)
{
   using namespace KBshortcutnames;

   static int  Ctrl = 0, Alt = 0, Shift = 0;
   int         key, type, ii, cc;
   char        keyname[20];

   key = event->keyval;
   type = event->type;

   if (type == GDK_KEY_PRESS) type = 1;
   else type = 0;

   if (key == GDK_KEY_Control_L || key == GDK_KEY_Control_R) {             //  Ctrl key
       Ctrl = type;
       return 1;
   }

   if (key == GDK_KEY_Alt_L || key == GDK_KEY_Alt_R) {                     //  Alt key
       Alt = type;
       return 1;
   }

   if (key == GDK_KEY_Shift_L || key == GDK_KEY_Shift_R) {                 //  Shift key
       Shift = type;
       return 1;
   }

   if (type) return 1;                                                     //  wait for key release

   if (key == GDK_KEY_F1) {                                                //  key is F1 (context help)
      KBstate(event,0);                                                    //  send to main app
      return 1;
   }

   if (key >= GDK_KEY_F2 && key <= GDK_KEY_F9) {                           //  key is F2 to F9
      ii = key - GDK_KEY_F1;
      strcpy(keyname,"F1");
      keyname[1] += ii;
   }

   else if (key > 255) return 1;                                           //  not a simple Ascii key

   else {
      *keyname = 0;                                                        //  build input key combination
      if (Ctrl) strcat(keyname,"Ctrl+");                                   //  [Ctrl+] [Alt+] [Shift+] key
      if (Alt) strcat(keyname,"Alt+");
      if (Shift) strcat(keyname,"Shift+");
      cc = strlen(keyname);
      keyname[cc] = toupper(key);
      keyname[cc+1] = 0;
   }

   for (ii = 0; ii < Nreserved; ii++)
      if (strEqu(keyname,reserved[ii])) break;
   if (ii < Nreserved) {
      zmessageACK(Mwin,0,ZTX("\"%s\"  Reserved, cannot be used"),keyname);
      Ctrl = Alt = 0;
      return 1;
   }

   zdialog_stuff(zd,"shortkey",keyname);                                   //  stuff key name into dialog
   zdialog_stuff(zd,"shortmenu","");                                       //  clear menu choice

   return 1;
}


//  dialog event and completion function

int KBshorts_dialog_event(zdialog *zd, cchar *event)
{
   using namespace KBshortcutnames;

   int         ii, jj, err;
   GtkWidget   *widget;
   char        textline[100];
   char        shortkey[20];
   char        shortmenu[60];
   char        kbfile[200];
   FILE        *fid = 0;

   if (strEqu(event,"shortmenu"))                                          //  menu choice made
   {
      zd->zstat = 0;                                                       //  keep dialog active

      zdialog_fetch(zd,"shortkey",shortkey,20);                            //  get shortcut key and menu
      zdialog_fetch(zd,"shortmenu",shortmenu,60);                          //    from dialog widgets
      if (*shortkey <= ' ' || *shortmenu <= ' ') return 0;

      for (ii = 0; ii < Nshortcuts2; ii++)                                 //  find matching shortcut key in list
         if (strEqu(shortcutkey2[ii],shortkey)) break;

      if (ii < Nshortcuts2) {                                              //  if found, remove from list
         zfree(shortcutkey2[ii]);
         zfree(shortcutmenu2[ii]);
         for (jj = ii; jj < Nshortcuts2; jj++) {
            shortcutkey2[jj] = shortcutkey2[jj+1];
            shortcutmenu2[jj] = shortcutmenu2[jj+1];
         }
         --Nshortcuts2;
      }

      for (ii = 0; ii < Nshortcuts2; ii++)                                 //  find matching shortcut menu in list
         if (strEqu(shortcutmenu2[ii],shortmenu)) break;

      if (ii < Nshortcuts2) {                                              //  if found, remove from list
         zfree(shortcutkey2[ii]);
         zfree(shortcutmenu2[ii]);
         for (jj = ii; jj < Nshortcuts2; jj++) {
            shortcutkey2[jj] = shortcutkey2[jj+1];
            shortcutmenu2[jj] = shortcutmenu2[jj+1];
         }
         --Nshortcuts2;
      }

      if (Nshortcuts2 == maxshortcuts) {
         zmessageACK(Mwin,0,"exceed %d shortcuts",maxshortcuts);
         return 1;
      }

      ii = Nshortcuts2++;                                                  //  add new shortcut to end of list
      shortcutkey2[ii] = zstrdup(shortkey);
      shortcutmenu2[ii] = zstrdup(shortmenu);

      widget = zdialog_widget(zd,"shortlist");
      wclear(widget);

      for (ii = 0; ii < Nshortcuts2; ii++) {                               //  show existing shortcuts in dialog
         snprintf(textline,100,"%-20s %s \n",shortcutkey2[ii],shortcutmenu2[ii]);
         wprintx(widget,0,textline,"monospace 9");
      }

      return 1;
   }
   
   if (! zd->zstat) return 1;                                              //  wait for completion

   if (zd->zstat == 1)                                                     //  delete shortcut
   {
      zd->zstat = 0;                                                       //  keep dialog active

      zdialog_fetch(zd,"shortkey",shortkey,20);                            //  get shortcut key
      if (*shortkey <= ' ') return 0;

      for (ii = 0; ii < Nshortcuts2; ii++)                                 //  find matching shortcut key in list
         if (strEqu(shortcutkey2[ii],shortkey)) break;

      if (ii < Nshortcuts2) {                                              //  if found, remove from list
         zfree(shortcutkey2[ii]);
         zfree(shortcutmenu2[ii]);
         for (jj = ii; jj < Nshortcuts2; jj++) {
            shortcutkey2[jj] = shortcutkey2[jj+1];
            shortcutmenu2[jj] = shortcutmenu2[jj+1];
         }
         --Nshortcuts2;
      }

      widget = zdialog_widget(zd,"shortlist");
      wclear(widget);

      for (ii = 0; ii < Nshortcuts2; ii++) {                               //  show existing shortcuts in dialog
         snprintf(textline,100,"%-20s %s \n",shortcutkey2[ii],shortcutmenu2[ii]);
         wprintx(widget,0,textline,"monospace 9");
      }

      zdialog_stuff(zd,"shortkey","");
      zdialog_stuff(zd,"shortmenu","");
      return 1;
   }

   if (zd->zstat == 2)                                                     //  done - save the shortcuts
   {
      zdialog_free(zd);

      for (ii = 0; ii < Nshortcuts2; ii++) {                               //  convert menunames back to English
         for (jj = 0; jj < Neligible; jj++)
            if (strEqu(eligible[jj],shortcutmenu2[ii])) break;
         if (jj < Neligible) {
            zfree(shortcutmenu2[ii]);
            shortcutmenu2[ii] = zstrdup(eligible_en[jj]);
         }
      }

      err = locale_filespec("user","KB-shortcuts",kbfile);                 //  write shortcuts file
      if (! err) fid = fopen(kbfile,"w");
      if (err || ! fid) {
         zmessageACK(Mwin,0,ZTX("unable to save KB-shortcuts file"));
         return 1;
      }
      for (ii = 0; ii < Nshortcuts2; ii++)
         fprintf(fid,"%-20s %s \n",shortcutkey2[ii],shortcutmenu2[ii]);
      fclose(fid);

      for (ii = 0; ii < Nshortcuts; ii++) {                                //  clear global shortcuts data
         if (shortcutkey[ii]) zfree(shortcutkey[ii]);
         if (shortcutmenu[ii]) zfree(shortcutmenu[ii]);
         shortcutkey[ii] = shortcutmenu[ii] = 0;
      }

      KBshortcuts_load();                                                  //  reload and activate
      return 1;
   }

   zdialog_free(zd);                                                       //  cancel or [x]
   return 1;
}


//  read KB-shortcuts file and load shortcuts table in memory
//  also called from initzfunc() at fotoxx startup

int KBshortcuts_load()                                                     //  v.13.03
{
   int         ii, err;
   char        kbfile[200], buff[200];
   char        *pp1, *pp2;
   FILE        *fid;

   Nshortcuts = 0;
   err = locale_filespec("user","KB-shortcuts",kbfile);
   if (err) return 0;
   ii = 0;
   fid = fopen(kbfile,"r");
   if (! fid) return 0;
   while (true) {
      pp1 = fgets_trim(buff,200,fid,1);                                    //  next record
      if (! pp1) break;
      if (*pp1 == '#') continue;                                           //  comment
      if (*pp1 <= ' ') continue;                                           //  blank
      pp2 = strchr(pp1,' ');
      if (! pp2) continue;
      if (pp2 - pp1 > 20) continue;
      *pp2 = 0;
      shortcutkey[ii] = zstrdup(pp1);                                      //  shortcut key
      pp1 = pp2 + 1;
      while (*pp1 && *pp1 == ' ') pp1++;
      if (! *pp1) continue;
      shortcutmenu[ii] = zstrdup(ZTX(pp1));                                //  menu can be English or translation
      if (++ii == maxshortcuts) break;
   }

   fclose(fid);
   Nshortcuts = ii;
   return 0;
}


/**************************************************************************/

//  Manage Collections - create, view, edit named collections of images

namespace collections
{
   char     *current_collection = 0;                                       //  collection file name
   char     **image_cache = 0;                                             //  images in cache
   int      Ncache = 0;                                                    //  cache count
   zdialog  *zdcoll = 0;
};

void collection_selectfiles();                                             //  select files, add to image cache
void collection_addtocache(char *imagefile);                               //  add image file to end of cache
void collection_cutfile(int posn, int addcache);                           //  remove from collection, add to image cache
void collection_copyfile(int posn);                                        //  add to image cache
void collection_pastecache(int posn, int clear);                           //  paste image cache into current collection
void collection_clearcache();                                              //  clear the image cache
void collection_show();                                                    //  show updated collection (gallery)


void m_manage_coll(GtkWidget *, cchar *)                                   //  v.13.07 overhauled
{
   using namespace collections;

   int      manage_coll_dialog_event(zdialog *zd, cchar *event);           //  manage collectiond dialog event func

   cchar    *popupmess = ZTX("Right-click collection thumbnail to cut/copy \n"
                             "to cache, insert from cache, or remove.");

   F1_help_topic = "manage_collections";

   if (zdcoll) return;                                                     //  already active                     v.13.10

   if (checkpend("all")) return;                                           //  check nothing pending              v.13.12

/***
       ________________________________________________
      |                                                |
      |        Manage Collections                      |
      |                                                |
      |  [New]     Start a new collection, add images  |
      |  [Make]    New collection from current gallery |
      |  [Choose]  Collection to view or edit          |
      |  [Images]  Select images, add to cache         |
      |  [Clear]   Clear image cache                   |
      |  [Delete]  Delete a collection                 |
      |                                                |
      |  Right-click collection thumbnail to cut/copy  |
      |  to cache, insert from cache, or remove.       |
      |                                                |
      |                                      [done]    |
      |________________________________________________|

***/

   zdialog *zd = zdialog_new(ZTX("Manage Collections"),Mwin,Bdone,null);

   zdialog_add_widget(zd,"hbox","hb1","dialog",0,"space=3");
   zdialog_add_widget(zd,"vbox","vb1","hb1",0,"homog|space=5");
   zdialog_add_widget(zd,"vbox","vb2","hb1",0,"homog|space=5");
   zdialog_add_widget(zd,"button","new","vb1",Bnew);
   zdialog_add_widget(zd,"button","make","vb1",Bmake);
   zdialog_add_widget(zd,"button","choose","vb1",Bchoose);
   zdialog_add_widget(zd,"button","images","vb1",Bimages);
   zdialog_add_widget(zd,"button","clear","vb1",Bclear);
   zdialog_add_widget(zd,"button","delete","vb1",Bdelete);
   zdialog_add_widget(zd,"hbox","hbnew","vb2");
   zdialog_add_widget(zd,"label","labnew","hbnew",ZTX("Start a new collection, add images"));
   zdialog_add_widget(zd,"hbox","hbmake","vb2");
   zdialog_add_widget(zd,"label","labmake","hbmake",ZTX("New collection from current gallery"));
   zdialog_add_widget(zd,"hbox","hbchoose","vb2");
   zdialog_add_widget(zd,"label","labchoose","hbchoose",ZTX("Collection to view or edit"));
   zdialog_add_widget(zd,"hbox","hbimages","vb2");
   zdialog_add_widget(zd,"label","labimages","hbimages",ZTX("Select images, add to cache"));
   zdialog_add_widget(zd,"hbox","hbclear","vb2");
   zdialog_add_widget(zd,"label","labclear","hbclear",ZTX("Clear image cache"));
   zdialog_add_widget(zd,"label","Ncache","hbclear",ZTX("(0 images)"),"space=5");
   zdialog_add_widget(zd,"hbox","hbdelete","vb2");
   zdialog_add_widget(zd,"label","labdelete","hbdelete",ZTX("Delete a collection"));
   zdialog_add_widget(zd,"hbox","hbpopup","dialog",0,"space=10");
   zdialog_add_widget(zd,"label","labpopup","hbpopup",popupmess,"space=10");

   zdialog_run(zd,manage_coll_dialog_event,"save");
   zdcoll = zd;

   zdialog_send_event(zdcoll,"Ncache");                                    //  update dialog cache count       v.13.12
   return;
}


//  manage collections dialog event and completion function

int manage_coll_dialog_event(zdialog *zd, cchar *event)
{
   using namespace collections;

   cchar       *newcoll = ZTX("New Collection File Name");
   cchar       *choosecoll = ZTX("Choose Collection");
   cchar       *delcoll = ZTX("Delete Collection");
   cchar       *ncacheFormat = ZTX("(%d images)");
   cchar       *cachewarn = ZTX("%d images remaining in cache");
   char        *cfile, *ifile, ncachetext[40];
   int         Nth, ftype;
   FILE        *fidw;

   if (strEqu(event,"enter")) zd->zstat = 1;                               //  [done]  v.14.03

   if (zd->zstat) {                                                        //  [done] or [x]
      zdialog_free(zd);
      zdcoll = 0;                                                          //  v.13.10
      if (Ncache) zmessageACK(Mwin,0,cachewarn,Ncache);                    //  v.14.01
      return 1;
   }

   if (strEqu(event,"new"))                                                //  start a new collection for editing
   {
      cfile = zgetfile(newcoll,"save",collections_dirk);                   //  choose file name
      if (! cfile) return 1;

      fidw = fopen(cfile,"w");                                             //  open/write empty file
      if (! fidw) {
         zmessageACK(Mwin,0,"%s",strerror(errno));
         return 1;
      }

      fclose(fidw);

      if (current_collection) zfree(current_collection);                   //  set current collection for editing
      current_collection = zstrdup(cfile);
      zfree(cfile);
      
      zdialog_show(zd,0);                                                  //  hide main dialog                   v.14.06
      collection_selectfiles();                                            //  select files, add to image cache
      collection_pastecache(0,1);                                          //  insert cache into collection
      collection_show();
      zmessageACK(Mwin,0,ZTX("New collection created"));                   //  v.13.10
      zdialog_show(zd,1);
   }

   if (strEqu(event,"make"))                                               //  curr. gallery >> new collection
   {
      if (navi::nimages == 0) {
         zmessageACK(Mwin,0,ZTX("gallery is empty"));
         return 1;
      }

      cfile = zgetfile(newcoll,"save",collections_dirk);                   //  choose file name
      if (! cfile) return 1;

      fidw = fopen(cfile,"w");                                             //  open/write collection file
      if (! fidw) {
         zmessageACK(Mwin,0,"%s",strerror(errno));
         return 1;
      }

      for (Nth = 0; Nth < navi::nfiles; Nth++)                             //  add gallery images to collection file
      {
         ifile = gallery(0,"find",Nth);
         if (! ifile) break;
         ftype = image_file_type(ifile);                                   //  must be image or RAW file
         if (ftype != 2 && ftype != 3) {
            zfree(ifile);
            continue;
         }
         fprintf(fidw,"%s\n",ifile);
         zfree(ifile);
      }

      fclose(fidw);

      if (current_collection) zfree(current_collection);                   //  set current collection for editing
      current_collection = zstrdup(cfile);
      zfree(cfile);

      collection_show();
      zmessageACK(Mwin,0,ZTX("New collection created"));                   //  v.13.10
   }

   if (strEqu(event,"choose"))                                             //  choose a collection to view or edit
   {
      cfile = zgetfile(choosecoll,"file",collections_dirk);                //  choose collection file
      if (! cfile) return 1;

      if (current_collection) zfree(current_collection);                   //  set current collection for editing
      current_collection = zstrdup(cfile);
      zfree(cfile);

      if (navi::nfiles == 0)                                               //  collection is empty
         collection_pastecache(0,1);                                       //  insert cache into collection

      collection_show();                                                   //  show collection
   }

   if (strEqu(event,"images"))                                             //  select image files and add to image cache
   {
      zdialog_show(zd,0);                                                  //  hide manage collections dialog

      collection_selectfiles();                                            //  select files, add to image cache

      if (current_collection) {
         collection_show();                                                //  back to edited collection
         if (navi::nfiles == 0)                                            //  collection is empty
            collection_pastecache(0,1);                                    //  insert cache into collection
         collection_show();
      }

      zdialog_show(zd,1);                                                  //  restore manage collections dialog
   }

   if (strEqu(event,"clear"))                                              //  clear image cache
      collection_clearcache();

   if (strEqu(event,"delete"))                                             //  choose collection to delete
   {
      cfile = zgetfile(delcoll,"file",collections_dirk);                   //  choose collection file
      if (! cfile) return 1;
      if (! zmessageYN(Mwin,ZTX("delete %s ?"),cfile)) return 1;

      if (strEqu(cfile,navi::galleryname))                                 //  current gallery is deleted
         m_viewmode(0,"F");

      if (current_collection && strEqu(cfile,current_collection)) {
         zfree(current_collection);
         current_collection = 0;
      }

      remove(cfile);
      zfree(cfile);
   }
   
   if (strEqu(event,"Ncache"))                                             //  update cache count in dialog
   {
      snprintf(ncachetext,40,ncacheFormat,Ncache);
      zdialog_stuff(zd,"Ncache",ncachetext);
   }

   return 1;
}


//  manage collections utility functions

void collection_selectfiles()                                              //  combine cached + newly selected files
{
   using namespace collections;

   int      ii;
   char     **selfiles = 0;

   selfiles = gallery_getfiles(image_cache);                               //  get cached files + new files if any

   if (Ncache) {
      for (ii = 0; ii < Ncache; ii++)                                      //  erase cached files
         zfree(image_cache[ii]);
      zfree(image_cache);
      Ncache = 0;
      zdialog_send_event(zdcoll,"Ncache");                                 //  update dialog cache count       v.13.12
   }

   if (! selfiles) return;                                                 //  nothing selected

   for (ii = 0; selfiles[ii]; ii++) ;                                      //  count selected files
   Ncache = ii;
   image_cache = selfiles;                                                 //  image cache = selected files
   zdialog_send_event(zdcoll,"Ncache");                                    //  update dialog cache count       v.13.12

   return;
}


void collection_addtocache(char *imagefile)                                //  add image file to image cache
{
   using namespace collections;

   int      cc1, cc2;
   char     **newcache;

   cc1 = Ncache * sizeof(char*);
   cc2 = cc1 + sizeof(char *);
   newcache = (char **) zmalloc(cc2);                                      //  new cache, 1 more file than before

   if (Ncache) memcpy(newcache,image_cache,cc1);                           //  copy old image cache to new image cache
   if (Ncache) zfree(image_cache);                                         //    and free old cache
   image_cache = newcache;

   image_cache[Ncache] = zstrdup(imagefile);                               //  add new image file at the end
   Ncache++;
   zdialog_send_event(zdcoll,"Ncache");                                    //  update dialog cache count       v.13.12

   return;
}


void collection_cutfile(int posn, int addcache)                            //  remove from collection, add to cache
{
   using namespace collections;

   int         ii;
   char        *pp, cachebuff[maxfcc];
   FILE        *fidr, *fidw;

   fidw = fopen(collections_copy,"w");                                     //  open/write copy file
   if (! fidw) {
      zmessageACK(Mwin,0,"%s",strerror(errno));
      return;
   }

   fidr = fopen(current_collection,"r");                                   //  open/read collection file
   if (! fidr) {
      zmessageACK(Mwin,0,"%s",strerror(errno));
      fclose(fidw);
      return;
   }

   for (ii = 0; ; ii++)                                                    //  loop collection member files
   {
      pp = fgets_trim(cachebuff,maxfcc,fidr);                              //  collection member file
      if (! pp) break;

      if (ii == posn) {                                                    //  position to remove
         if (addcache) collection_addtocache(pp);                          //  add to image cache if wanted
         continue;                                                         //  omit from copy file
      }

      fprintf(fidw,"%s\n",pp);                                             //  add to copy file
   }

   fclose(fidr);
   fclose(fidw);

   rename(collections_copy,current_collection);                            //  replace collection file with copy

   return;
}


void collection_copyfile(int posn)                                         //  add image file to cache
{
   using namespace collections;

   char *file = gallery(0,"find",posn);                                    //  get gallery file at position
   if (! file) return;
   collection_addtocache(file);                                            //  add to image cache
   return;
}


void collection_pastecache(int posn, int clear)                            //  insert cached files into collection
{                                                                          //    at designated position
   using namespace collections;

   int      ii, jj;
   char     *pp, cachebuff[maxfcc];
   FILE     *fidr, *fidw;
   
   if (! current_collection) return;
   if (Ncache == 0) return;

   fidw = fopen(collections_copy,"w");                                     //  open/write copy file
   if (! fidw) {
      zmessageACK(Mwin,0,"%s",strerror(errno));
      return;
   }

   fidr = fopen(current_collection,"r");                                   //  open/read collection file
   if (! fidr) {
      zmessageACK(Mwin,0,"%s",strerror(errno));
      fclose(fidw);
      return;
   }

   for (ii = 0; ; ii++)                                                    //  loop collection member files
   {
      if (ii == posn)
         for (jj = 0; jj < Ncache; jj++)                                   //  add cached files here
            fprintf(fidw,"%s\n",image_cache[jj]);

      pp = fgets_trim(cachebuff,maxfcc,fidr);                              //  copy collection member file
      if (! pp) break;                                                     //  EOF
      fprintf(fidw,"%s\n",pp);
   }

   fclose(fidr);
   fclose(fidw);

   rename(collections_copy,current_collection);                            //  replace collection file with copy

   if (clear) collection_clearcache();

   return;
}


void collection_clearcache()                                               //  clear the image cache
{
   using namespace collections;

   if (Ncache) {
      for (int ii = 0; ii < Ncache; ii++)
         zfree(image_cache[ii]);
      zfree(image_cache);
      Ncache = 0;
      zdialog_send_event(zdcoll,"Ncache");                                 //  update dialog cache count          v.13.12
   }

   return;
}


void collection_show()                                                     //  show current collection
{
   using namespace collections;

   int      type;
   char     *pp, cachebuff[maxfcc];
   FILE     *fidr, *fidw;

   if (! current_collection) return;

   fidw = fopen(collections_copy,"w");                                     //  open/write copy file
   if (! fidw) {
      zmessageACK(Mwin,0,"%s",strerror(errno));
      return;
   }

   fidr = fopen(current_collection,"r");                                   //  open/read collection file
   if (! fidr) {
      zmessageACK(Mwin,0,"%s",strerror(errno));
      fclose(fidw);
      return;
   }

   while (true)                                                            //  copy collection member files
   {
      pp = fgets_trim(cachebuff,maxfcc,fidr);
      if (! pp) break;

      type = image_file_type(pp);                                          //  remove deleted files               v.14.03
      if (type < 2 || type > 3) {
         printz("remove deleted file from collection: \n %s \n",pp);
         continue;
      }

      fprintf(fidw,"%s\n",pp);
   }

   fclose(fidr);
   fclose(fidw);

   rename(collections_copy,current_collection);                            //  replace collection file with copy

   free_resources();                                                       //  no current file
   navi::gallerytype = 4;
   gallery(current_collection,"initF");
   gallery(0,"paint",-1);
   m_viewmode(0,"G");
   return;
}


//  manage collection right-click popup menus

void coll_popup_remove(GtkWidget *, cchar *menu)                           //  remove image from collection
{
   using namespace collections;

   if (! clicked_file) return;

   if (navi::gallerytype != 4) {
      zfree(clicked_file);                                                 //  reset clicked file           v.13.08
      clicked_file = 0;
      return;
   }

   if (current_collection) zfree(current_collection);                      //  collection being edited
   current_collection = zstrdup(navi::galleryname);

   collection_cutfile(clicked_posn,0);                                     //  remove clicked file

   zfree(clicked_file);                                                    //  reset clicked file
   clicked_file = 0;

   collection_show();
   return;
}


void coll_popup_cutfile(GtkWidget *, cchar *menu)                          //  remove image from collection
{                                                                          //    add to cache
   using namespace collections;

   if (! clicked_file) return;

   if (navi::gallerytype != 4) {
      zfree(clicked_file);                                                 //  reset clicked file           v.13.08
      clicked_file = 0;
      return;
   }

   if (current_collection) zfree(current_collection);                      //  collection being edited
   current_collection = zstrdup(navi::galleryname);

   collection_cutfile(clicked_posn,1);                                     //  remove clicked file, add to cache

   zfree(clicked_file);                                                    //  reset clicked file
   clicked_file = 0;

   collection_show();                                                      //  show revised collection
   return;
}


void coll_popup_copyfile(GtkWidget *, cchar *menu)                         //  add clicked image to cached images
{
   using namespace collections;

   if (! clicked_file) return;                                             //  does not have to be a collection gallery
   collection_copyfile(clicked_posn);                                      //  add clicked file to cache
   zfree(clicked_file);                                                    //  reset clicked file
   clicked_file = 0;
   return;
}


void coll_popup_pastecache(GtkWidget *, cchar *menu)                       //  add cached image files to collection
{                                                                          //    at clicked position
   using namespace collections;                                            //  clear cache if wanted

   int      posn, clear;

   if (! clicked_file) return;

   if (navi::gallerytype != 4) {
      zfree(clicked_file);                                                 //  reset clicked file              v.13.08
      clicked_file = 0;
      return;
   }

   if (current_collection) zfree(current_collection);                      //  collection being edited
   current_collection = zstrdup(navi::galleryname);
   
   posn = clicked_posn;
   if (clicked_width > 50) ++posn;                                         //  thumbnail right side clicked, bump posn

   if (strEqu(menu,"clear")) clear = 1;                                    //  set clear cache option
   else clear = 0;
   collection_pastecache(posn,clear);                                      //  paste cached images at position

   zfree(clicked_file);                                                    //  reset clicked file
   clicked_file = 0;

   collection_show();                                                      //  show revised collection
   return;
}


/**************************************************************************/

//  Change any path seqments for an image collection.
//  Fix collections when images have been moved to new directories.

void m_move_coll(GtkWidget *, cchar *)                                     //  overhauled                         v.13.12
{
   zdialog     *zd;
   char        *pp, *collnames[100], collname[100];
   char        oldseg[200], newseg[200];
   char        buffr[maxfcc], buffw[maxfcc];
   char        collfile[200], tempfile[200];
   int         ii, err, cc, yn;
   int         zstat, Ncoll, contx = 0;
   FILE        *fidr = 0, *fidw = 0;
   cchar       *findcomm = "find -L \"%s\" -type f";
   cchar       *allcolls = ZTX("all collections");

   F1_help_topic = "move_collections";
   if (checkpend("all")) return;                                           //  check nothing pending              v.13.12

   for (contx = ii = 0; ii < 100; ii++)
   {
      pp = command_output(contx,findcomm,collections_dirk);                //  find all collection files
      if (! pp) break;
      collnames[ii] = pp;
   }
   if (contx) command_kill(contx);
   
   Ncoll = ii;
   if (! Ncoll) {
      zmessageACK(Mwin,0,ZTX("no collections found"));
      return;
   }
   if (Ncoll == 100) 
      zmessageACK(Mwin,0,"100 collections limit reached");
   
/***
          ______________________________________
         |         Move Collection              |
         |                                      |
         | collection name [________________|v] |
         | old path segment [_________________] |
         | new path segment [_________________] |
         |                                      |
         |                     [Apply] [Cancel] |
         |______________________________________|

***/

   zd = zdialog_new(ZTX("Move Collection"),Mwin,Bapply,Bcancel,null);
   zdialog_add_widget(zd,"hbox","hbcoll","dialog",0,"space=2");
   zdialog_add_widget(zd,"label","labcoll","hbcoll",ZTX("collection name"),"space=5");
   zdialog_add_widget(zd,"combo","collname","hbcoll",0,"expand");
   zdialog_add_widget(zd,"hbox","hbold","dialog",0,"space=2");
   zdialog_add_widget(zd,"label","labold","hbold",ZTX("old path segment"),"space=5");
   zdialog_add_widget(zd,"entry","oldseg","hbold","/old/path/segment/","scc=20|expand");
   zdialog_add_widget(zd,"hbox","hbnew","dialog",0,"space=2");
   zdialog_add_widget(zd,"label","labnew","hbnew",ZTX("new path segment"),"space=5");
   zdialog_add_widget(zd,"entry","newseg","hbnew","/new/path/seqment/","scc=20|expand");

   zdialog_restore_inputs(zd);

   for (ii = 0; ii < Ncoll; ii++)                                          //  add collection names to dialog list
   {
      pp = strrchr(collnames[ii],'/'); 
      if (! pp) continue;
      zdialog_cb_app(zd,"collname",pp+1);
   }
   zdialog_cb_app(zd,"collname",allcolls);                                 //  add "all collections"

run_again:

   zdialog_run(zd);                                                        //  run dialog
   zstat = zdialog_wait(zd);

   if (zstat != 1) {                                                       //  user cancel
      zdialog_free(zd);
      for (ii = 0; ii < Ncoll; ii++)
         zfree(collnames[ii]);
      return;
   }

   zdialog_fetch(zd,"collname",collname,100);                              //  get collection name
   zdialog_fetch(zd,"oldseg",oldseg,200);                                  //  old and new path segments
   zdialog_fetch(zd,"newseg",newseg,200);

   if (strEqu(collname,allcolls))                                          //  "all collections"
   {
      for (ii = 0; ii < Ncoll; ii++)                                       //  loop all collections
      {
         strcpy(collfile,collnames[ii]);                                   //  collection file
         strcpy(tempfile,collfile);
         strcat(tempfile,".new");
         
         fidr = fopen(collfile,"r");                                       //  open collection and temp files
         if (! fidr) goto error;
         fidw = fopen(tempfile,"w");
         if (! fidw) goto error;
         
         while ((pp = fgets(buffr,maxfcc,fidr)))                           //  loop collection recs
         {
            repl_1str(buffr,buffw,oldseg,newseg);                          //  replace path segment
            err = fputs(buffw,fidw);
            if (err < 0) goto error;
            cc = strlen(buffw);
            if (cc && buffw[cc-1] == '\n') buffw[cc-1] = 0;
         }

         err = fclose(fidr);
         err = fclose(fidw);
         fidr = fidw = 0;
         if (err) goto error;
         err = rename(tempfile,collfile);                                  //  replace collection with temp file
         if (err) goto error;
      }

      goto run_again;
   }
   
   snprintf(collfile,199,"%s/%s",collections_dirk,collname);               //  collection file
   strcpy(tempfile,collfile);
   strcat(tempfile,".new");
   
   fidr = fopen(collfile,"r");                                             //  open collection and temp files
   if (! fidr) goto error;
   fidw = fopen(tempfile,"w");
   if (! fidw) goto error;
   
   write_popup_text("open","",500,300,Mwin);

   while ((pp = fgets(buffr,maxfcc,fidr)))                                 //  loop collection recs
   {
      repl_1str(buffr,buffw,oldseg,newseg);                                //  replace path segment
      err = fputs(buffw,fidw);
      if (err < 0) goto error;
      cc = strlen(buffw);
      if (cc && buffw[cc-1] == '\n') buffw[cc-1] = 0;
      write_popup_text("write",buffw);
   }

   err = fclose(fidr);
   err = fclose(fidw);
   fidr = fidw = 0;
   if (err) goto error;
   
   write_popup_text("top");
   yn = zdialog_choose(0,Mwin,ZTX("Replace Collection?"),Byes,Bno,null);
   write_popup_text("close");

   if (yn == 1) {
      err = rename(tempfile,collfile);                                     //  replace collection with temp file
      if (err) goto error;
   }
   else remove(tempfile);
   goto run_again;

error:
   err = errno;
   zmessageACK(Mwin,0,"%s \n %s",collfile,strerror(err));
   if (fidr) fclose(fidr);
   if (fidw) fclose(fidw);
   write_popup_text("close");
   remove(tempfile);
   goto run_again;
}


/**************************************************************************/

//  Slide Show

int   slideshow_dialog_event(zdialog *zd, cchar *event);                   //  user dialogs
void  ss_transprefs_dialog();                                              //  edit transition preferences
void  ss_imageprefs_dialog();                                              //  edit image preferences
void  ss_loadprefs();                                                      //  load preferences from file
void  ss_saveprefs();                                                      //  write preferences to file
int   ss_timerfunc(void *);                                                //  timer function
int   ss_nextrans();                                                       //  select next transition to use
void  ss_captions();                                                       //  write captions/comments on image
void  ss_blankwindow();                                                    //  blank the window

GdkPixbuf *ss_loadpxb(char *file);                                         //  load image as pixbuf

void  ss_instant();                                                        //  transition functions
void  ss_fadein();
void  ss_rollright();
void  ss_rolldown();
void  ss_venetian();
void  ss_grate();
void  ss_rectangle();
void  ss_radar();
void  ss_jaws();
void  ss_ellipse();
void  ss_raindrops();
void  ss_doubledoor();
void  ss_rotate();
void  ss_zoomin();

char        ss_collfile[300] = "";                                         //  slide show collection file
char        *ss_collname = 0;                                              //  collection name (ss_collfile tail)
int         ss_Nfiles = 0;                                                 //  collection file count
int         ss_secs = 0;                                                   //  standard display time from user
int         ss_cliplim = 10;                                               //  image clipping limit from user
int         ss_showcaps = 0;                                               //  show captions option from user
int         ss_random = 0;                                                 //  use random transitions option from user
char        ss_musicfile[300] = "none";                                    //  music file from user
int         ss_slowdown = 0;                                               //  slowdown factor for transition in use
int         ss_zoom, ss_zsteps;                                            //  zoom-in amount (%) and zoom steps
int         ss_cenx, ss_ceny;                                              //  zoom-in target (50/50 = image midpoint)
int         ss_ww, ss_hh;                                                  //  full screen window size
char        *ss_oldfile, *ss_newfile;                                      //  image files for transition
GdkPixbuf   *ss_pxbold, *ss_pxbnew;                                        //  pixbuf images: old, new
int         ss_rsold, ss_rsnew;                                            //  old and new pixbuf row stride
double      ss_timer = 0;                                                  //  slide show timer
cchar       *ss_state = 0;                                                 //  slide show state
int         ss_Fnext = 0;                                                  //  next file to show
int         ss_blank = 0;                                                  //  flag, user pressed B key (blank screen)
cairo_t     *ss_cr;                                                        //  cairo context

#define     SSNF 13                                                        //  slide show transition types
#define     SSMAXI 10000                                                   //  max. no. slide show images

struct ss_trantab_t {                                                      //  transition table
   char     tranname[32];                                                  //  transition name
   int      enabled;                                                       //  enabled or not
   int      slowdown;                                                      //  slowdown factor
   int      preference;                                                    //  relative preference, 0-99
   void     (*func)();                                                     //  function to perform transition
};

ss_trantab_t  ss_trantab[SSNF];                                            //  specific slide show transition preferences 

int      ss_Tused[SSNF];                                                   //  list of transition types enabled
int      ss_Tlast[SSNF];                                                   //  last transitions used, in order
int      ss_Nused;                                                         //  count of enabled transitions, 0-SSNF
int      ss_Tnext;                                                         //  next transition to use >> last one used

ss_trantab_t  ss_trantab_default[SSNF] =                                   //  transition default preference
{    //   name          enab  slow  pref   function                        //  (enabled, no slowdown, equal preference)
      {  "instant",      1,    0,    10,   ss_instant     },
      {  "fade-in",      1,    0,    10,   ss_fadein      },
      {  "roll-right",   1,    3,    10,   ss_rollright   },               //  NO BLANKS IN TRANSITION NAMES
      {  "roll-down",    1,    3,    10,   ss_rolldown    },
      {  "venetian",     1,    0,    10,   ss_venetian    },
      {  "grate",        1,    0,    10,   ss_grate       },
      {  "rectangle",    1,    0,    10,   ss_rectangle   },
      {  "radar",        1,    0,    10,   ss_radar       },
      {  "jaws",         1,    0,    10,   ss_jaws        },
      {  "ellipse",      1,    0,    10,   ss_ellipse     },
      {  "raindrops",    1,    0,    10,   ss_raindrops   },
      {  "doubledoor",   1,    3,    10,   ss_doubledoor  },
      {  "rotate",       1,    0,    10,   ss_rotate      }
};

struct ss_imagetab_t   {                                                   //  image table
   char     *imagefile;                                                    //  image file
   char     tranname[32];                                                  //  transition type to use
   int      Ftone;                                                         //  flag, play tone when shown
   int      bzsecs;                                                        //  seconds to show image before zoom
   int      zoom, zsteps;                                                  //  opt. zoom-in size (%) and steps
   int      cenx, ceny;                                                    //  zoom-in target (50/50 = image midpoint)
   int      azsecs;                                                        //  seconds to show image after zoom
};

ss_imagetab_t  ss_imagetab[SSMAXI];                                        //  specific slide show image table


//  menu function - start or stop a slide show

void m_slideshow(GtkWidget *, cchar *)                                     //  overhauled                      v.14.04
{
   zdialog     *zd;
   int         zstat, ii;
   char        countmess[50];
   cchar       *esc_message = ZTX("Press ESC to exit slide show");

   strncpy0(ss_trantab_default[0].tranname,ZTX("instant"),32);             //  get translations into the .po file
   strncpy0(ss_trantab_default[1].tranname,ZTX("fade-in"),32);
   strncpy0(ss_trantab_default[2].tranname,ZTX("roll-right"),32);
   strncpy0(ss_trantab_default[3].tranname,ZTX("roll-down"),32);
   strncpy0(ss_trantab_default[4].tranname,ZTX("venetian"),32);
   strncpy0(ss_trantab_default[5].tranname,ZTX("grate"),32);
   strncpy0(ss_trantab_default[6].tranname,ZTX("rectangle"),32);
   strncpy0(ss_trantab_default[7].tranname,ZTX("radar"),32);
   strncpy0(ss_trantab_default[8].tranname,ZTX("jaws"),32);
   strncpy0(ss_trantab_default[9].tranname,ZTX("ellipse"),32);
   strncpy0(ss_trantab_default[10].tranname,ZTX("raindrops"),32);
   strncpy0(ss_trantab_default[11].tranname,ZTX("doubledoor"),32);
   strncpy0(ss_trantab_default[12].tranname,ZTX("rotate"),32);

   F1_help_topic = "slide_show";
   if (checkpend("all")) return;                                           //  check nothing pending

/**
       ________________________________________________________
      | [x] [-] [ ]   Slide Show                               |  
      |                                                        |
      | [Select] collection-name                123 images     |
      | Seconds [__]  Clip Limit (%) [__]  [x] Show Captions   |
      | Music file [_____________________________] [Browse]    |
      |                                                        |
      | Customize: [transitions]  [image files]                |
      | Press ESC to exit slide show                           |
      |                                     [Proceed] [Cancel] |
      |________________________________________________________|

**/

   zd = zdialog_new(ZTX("Slide Show"),Mwin,Bproceed,Bcancel,null);         //  user dialog

   zdialog_add_widget(zd,"hbox","hbss","dialog",0,"space=3");
   zdialog_add_widget(zd,"button","selectcoll","hbss",Bselect,"space=5");
   zdialog_add_widget(zd,"label","collname","hbss",Bnoselection,"space=5");
   zdialog_add_widget(zd,"label","nfiles","hbss",Bnoimages,"space=5");

   zdialog_add_widget(zd,"hbox","hbprefs","dialog");
   zdialog_add_widget(zd,"label","labsecs","hbprefs",Bseconds,"space=5");
   zdialog_add_widget(zd,"spin","seconds","hbprefs","1|99|1|3");
   zdialog_add_widget(zd,"label","space","hbprefs",0,"space=5");
   zdialog_add_widget(zd,"label","labclip","hbprefs",ZTX("Clip Limit"),"space=5");
   zdialog_add_widget(zd,"spin","cliplim","hbprefs","0|50|1|0");
   zdialog_add_widget(zd,"check","showcaps","hbprefs",ZTX("Show Captions"),"space=10");

   zdialog_add_widget(zd,"hbox","hbmf","dialog",0,"space=3");
   zdialog_add_widget(zd,"label","labmf","hbmf",ZTX("Music File"),"space=5");
   zdialog_add_widget(zd,"entry","musicfile","hbmf",0,"expand|space=5");
   zdialog_add_widget(zd,"button","browse","hbmf",Bbrowse,"space=5");

   zdialog_add_widget(zd,"hbox","hbcust","dialog",0,"space=3");
   zdialog_add_widget(zd,"label","labprefs","hbcust",ZTX("Customize:"),"space=5");
   zdialog_add_widget(zd,"button","transprefs","hbcust",ZTX("transitions"),"space=5");
   zdialog_add_widget(zd,"button","imageprefs","hbcust",ZTX("image files"),"space=5");

   zdialog_add_widget(zd,"hbox","hbesc","dialog",0,"space=3");
   zdialog_add_widget(zd,"label","labesc","hbesc",esc_message,"space=5");

   if (ss_Nfiles) {                                                        //  if collection file available,
      ss_loadprefs();                                                      //    load slide show data or defaults
      zdialog_stuff(zd,"collname",ss_collname);
      snprintf(countmess,50,ZTX("%d images"),ss_Nfiles);
      zdialog_stuff(zd,"nfiles",countmess);
      zdialog_stuff(zd,"seconds",ss_secs);
      zdialog_stuff(zd,"cliplim",ss_cliplim);
      zdialog_stuff(zd,"showcaps",ss_showcaps);
      zdialog_stuff(zd,"musicfile",ss_musicfile);

      navi::gallerytype = 4;                                               //  open gallery with slide show collection
      gallery(ss_collfile,"initF");
      gallery(0,"paint",ss_Fnext);
      m_viewmode(0,"G");
   }
   
   zdialog_run(zd,slideshow_dialog_event);                                 //  run dialog
   zstat = zdialog_wait(zd);                                               //  wait for completion

   if (zstat != 1) {                                                       //  cancel
      zdialog_free(zd);
      return;
   }
      
   if (! ss_Nfiles) {                                                      //  no selection
      zmessageACK(Mwin,0,ZTX("invalid collection"));
      return;
   }

   zdialog_fetch(zd,"seconds",ss_secs);                                    //  timer interval, seconds
   if (ss_Nused == 0) ss_secs = 9999;                                      //  if only arrow-keys used, huge interval
   zdialog_fetch(zd,"cliplim",ss_cliplim);                                 //  image clipping limit
   zdialog_fetch(zd,"showcaps",ss_showcaps);
   zdialog_fetch(zd,"musicfile",ss_musicfile,300);

   strTrim2(ss_musicfile);                                                 //  if music file blanked, use "none"
   if (! *ss_musicfile || *ss_musicfile == ' ')
      strcpy(ss_musicfile,"none");

   zdialog_free(zd);                                                       //  kill dialog

   ss_saveprefs();                                                         //  save preference changes

   if (curr_file) {                                                        //  start at curr. file 
      for (ii = 0; ii < ss_Nfiles; ii++)                                   //    if member of file list
         if (strEqu(curr_file,ss_imagetab[ii].imagefile)) break;
      if (ii == ss_Nfiles) ii = 0;
   }
   else ii = 0;
   ss_Fnext = ii;                                                          //  next file in list to show

   m_viewmode(0,"F");                                                      //  insure tab F

   win_fullscreen(1);                                                      //  full screen, hide menu and panel
   ss_ww = gdk_window_get_width(gdkwin);                                   //  drawing window size
   ss_hh = gdk_window_get_height(gdkwin);

   ss_newfile = 0;                                                         //  no new image
   ss_pxbnew = 0;
   ss_oldfile = 0;                                                         //  no old (prior) image
   ss_pxbold = 0;

   Fslideshow = 1;                                                         //  slideshow active for KB events
   ss_blank = 0;                                                           //  not blank window
   ss_state = "first";
   Fblowup = 1;                                                            //  expand small images
   Fzoom = 0;                                                              //  fit window
   Fcaptions = ss_showcaps;                                                //  show captions option

   if (*ss_musicfile == '/')                                               //  if music file present, start it up
      shell_ack("xdg-open \"%s\" ",ss_musicfile);

   g_timeout_add(100,ss_timerfunc,0);
   return;
}


//  dialog event function - file chooser for images to show and music file

int slideshow_dialog_event(zdialog *zd, cchar *event)
{
   char        *file, *pp;
   char        countmess[50];

   if (strEqu(event,"focus"))
      F1_help_topic = "slide_show";

   if (strEqu(event,"enter")) zd->zstat = 1;                               //  [proceed]

   if (zd->zstat == 1)                                                     //  [proceed]
   {
      if (! ss_Nfiles) {
         zmessageACK(Mwin,0,ZTX("invalid collection"));
         zd->zstat = 0;
         return 1;
      }
   }

   if (strEqu(event,"selectcoll"))                                         //  select a slide show collection
   {
      ss_Nfiles = 0;                                                       //  reset collection data
      ss_collfile[0] = 0;
      zdialog_stuff(zd,"collname",Bnoselection);
      zdialog_stuff(zd,"nfiles",Bnoimages);

      file = zgetfile(ZTX("open collection"),"file",collections_dirk);
      if (! file) return 1;
      if (strlen(file) > 299) {
         zmessageACK(Mwin,0,"file name too long");
         return 1;
      }

      strcpy(ss_collfile,file);
      pp = strrchr(ss_collfile,'/');                                       //  get collection name
      ss_collname = pp + 1;
      zfree(file);
      
      ss_loadprefs();                                                      //  get slide show prefs or defaults

      if (! ss_Nfiles) {
         zmessageACK(Mwin,0,ZTX("invalid collection"));
         return 1;
      }

      zdialog_stuff(zd,"collname",ss_collname);                            //  update dialog collection data
      snprintf(countmess,50,ZTX("%d images"),ss_Nfiles);
      zdialog_stuff(zd,"nfiles",countmess);
      zdialog_stuff(zd,"seconds",ss_secs);
      zdialog_stuff(zd,"cliplim",ss_cliplim);
      zdialog_stuff(zd,"showcaps",ss_showcaps);
      zdialog_stuff(zd,"musicfile",ss_musicfile);

      navi::gallerytype = 4;                                               //  open gallery with slide show collection
      gallery(ss_collfile,"initF");
      gallery(0,"paint",0);
      m_viewmode(0,"G");
   }
   
   if (! ss_Nfiles) return 1;
   
   if (strEqu(event,"transprefs")) ss_transprefs_dialog();                 //  edit transition preferences
   if (strEqu(event,"imageprefs")) ss_imageprefs_dialog();                 //  edit image preferences

   if (strEqu(event,"browse"))                                             //  look for music file
   {
      file = zgetfile(ZTX("Select music file or playlist"),"file",ss_musicfile);
      if (! file) return 0;
      zdialog_stuff(zd,"musicfile",file);
      zfree(file);
   }
   
   return 1;
}


//  set transitions preferences for specific slide show

void ss_transprefs_dialog() 
{
   zdialog     *zd;
   int         ii, jj;
   char        nameii[SSNF], enabii[SSNF], timeii[SSNF], prefii[SSNF];
   cchar       *randmess = ZTX("select random (if 5+ enabled)");

/***
          _______________________________________________
         |        Transition Preferences                 |
         |                                               |
         |  [x] select random (if 5+ enabled)            |
         |                                               |
         |  transition    enabled  slowdown  preference  |
         |  instant         [x]     [ 1 ]      [ 10 ]    |
         |  fade-in         [x]     [ 1 ]      [ 20 ]    |
         |  roll-right      [x]     [ 1 ]      [  0 ]    |
         |  ....            ...     ...        ...       |
         |                                               |
         |                                       [done]  |
         |_______________________________________________|
         
***/

   if (! ss_Nfiles) {
      zmessageACK(Mwin,0,ZTX("invalid collection"));
      return;
   }

   zd = zdialog_new(ZTX("Transition Preferences"),Mwin,Bdone,null);
   
   zdialog_add_widget(zd,"hbox","hbrand","dialog",0,"space=3");
   zdialog_add_widget(zd,"check","rand","hbrand",randmess,"space=3");
   zdialog_add_widget(zd,"hbox","hb1","dialog",0,"space=3");
   zdialog_add_widget(zd,"vbox","vb1","hb1",0,"space=3|homog");
   zdialog_add_widget(zd,"vbox","vb2","hb1",0,"space=3|homog");
   zdialog_add_widget(zd,"vbox","vb3","hb1",0,"space=3|homog");
   zdialog_add_widget(zd,"vbox","vb4","hb1",0,"space=3|homog");
   zdialog_add_widget(zd,"label","labname","vb1",ZTX("transition"));
   zdialog_add_widget(zd,"label","labenab","vb2",ZTX("enabled"));
   zdialog_add_widget(zd,"label","labtime","vb3",ZTX("slowdown"));
   zdialog_add_widget(zd,"label","labpref","vb4",ZTX("preference"));
   
   zdialog_stuff(zd,"rand",ss_random);                                     //  stuff random checkbox
   
   for (ii = 0; ii < SSNF; ii++) {                                         //  build input dialog for transition prefs
      sprintf(nameii,"name_%d",ii);
      sprintf(enabii,"enab_%d",ii);
      sprintf(timeii,"time_%d",ii);
      sprintf(prefii,"pref_%d",ii);
      zdialog_add_widget(zd,"label",nameii,"vb1","transition");
      zdialog_add_widget(zd,"check",enabii,"vb2",0,"scc=3");
      zdialog_add_widget(zd,"spin",timeii,"vb3","0|99|1|1","scc=3");
      zdialog_add_widget(zd,"spin",prefii,"vb4","0|99|1|10","scc=3");
      zdialog_stuff(zd,nameii,ss_trantab[ii].tranname);                    //  stuff current transition prefs
      zdialog_stuff(zd,enabii,ss_trantab[ii].enabled);
      zdialog_stuff(zd,timeii,ss_trantab[ii].slowdown);
      zdialog_stuff(zd,prefii,ss_trantab[ii].preference);
   }
   
   zdialog_run(zd);                                                        //  run dialog, wait for completion
   zdialog_wait(zd);
   
   zdialog_fetch(zd,"rand",ss_random);                                     //  get mode, 0/1 = sequential/random
   
   for (ii = 0; ii < SSNF; ii++) {
      sprintf(enabii,"enab_%d",ii);
      sprintf(timeii,"time_%d",ii);
      sprintf(prefii,"pref_%d",ii);
      zdialog_fetch(zd,enabii,ss_trantab[ii].enabled);
      zdialog_fetch(zd,timeii,ss_trantab[ii].slowdown);
      zdialog_fetch(zd,prefii,ss_trantab[ii].preference);
   }
   
   zdialog_free(zd);

   ss_saveprefs();

   for (ii = jj = 0; ii < SSNF; ii++) {                                    //  initialize list of enabled 
      if (ss_trantab[ii].enabled) {                                        //    and last used transition types
         ss_Tused[jj] = ii;
         jj++;
      }
      ss_Tlast[ii] = 0;
   }

   ss_Nused = jj;                                                          //  no. enabled transition types
   ss_Tnext = 0;                                                           //  next one to use (first)
   
   return;
}


//  set image preferences for specific slide show

void ss_imageprefs_dialog()                                                //  v.14.04
{
   int ss_imageprefs_dialog_event(zdialog *zd, cchar *event);

   zdialog  *zd;
   int      ii;
   char     *pp, imagefile[100];

/***
       ____________________________________________
      |        Image Preferences                   |
      |                                            |
      |  Image File: /.../filename.jpg             |
      |  Play Tone [x]  Transition: [ rotate |v]   |
      |  +Seconds [ 99 ]  before zoom              |
      |  Zoom size % [50]  Steps [200 |v]          |
      |  Zoom center  width [40]  height [60]      |
      |  +Seconds [ 99 ]  after zoom               |
      |                                            |
      |                                    [done]  |
      |____________________________________________|

***/

   if (! ss_Nfiles) {
      zmessageACK(Mwin,0,ZTX("invalid collection"));
      return;
   }

   zd = zdialog_new(ZTX("Image Preferences"),Mwin,Bdone,null);
   zd_ss_imageprefs = zd;

   zdialog_add_widget(zd,"hbox","hbimf","dialog",0,"space=5");
   zdialog_add_widget(zd,"label","labimf","hbimf",ZTX("Image File:"),"space=3");

   zdialog_add_widget(zd,"hbox","hbtt","dialog");
   zdialog_add_widget(zd,"check","tone","hbtt",ZTX("Play tone"),"space=3");
   zdialog_add_widget(zd,"label","space","hbtt",0,"space=10");
   zdialog_add_widget(zd,"label","labtr","hbtt",ZTX("Transition"),"space=3");
   zdialog_add_widget(zd,"combo","trans","hbtt",0,"expand");
   zdialog_add_widget(zd,"label","space","hbtt",0,"space=10");

   zdialog_add_widget(zd,"hbox","hbbzs","dialog");
   zdialog_add_widget(zd,"label","labbzs1","hbbzs",ZTX("+Seconds"),"space=3");
   zdialog_add_widget(zd,"spin","bzsecs","hbbzs","0|999|1|0","space=3");
   zdialog_add_widget(zd,"label","labbzs2","hbbzs",ZTX("before zoom"),"space=3");

   zdialog_add_widget(zd,"hbox","hbz1","dialog");
   zdialog_add_widget(zd,"label","labz","hbz1",ZTX("Zoom size %"),"space=3");
   zdialog_add_widget(zd,"spin","zoom","hbz1","0|99|1|0","space=3");
   zdialog_add_widget(zd,"label","space","hbz1",0,"space=5");
   zdialog_add_widget(zd,"label","labst","hbz1",ZTX("Steps"),"space=3");
   zdialog_add_widget(zd,"spin","zsteps","hbz1","50|999|1|200");

   zdialog_add_widget(zd,"hbox","hbz2","dialog",0,"space=5");
   zdialog_add_widget(zd,"label","labcx","hbz2",ZTX("Zoom center"),"space=3");
   zdialog_add_widget(zd,"label","labcx","hbz2",Bwidth,"space=8");
   zdialog_add_widget(zd,"label","cenx","hbz2"," 50 ");
   zdialog_add_widget(zd,"label","space","hbz2",0,"space=5");
   zdialog_add_widget(zd,"label","labcy","hbz2",Bheight,"space=8");
   zdialog_add_widget(zd,"label","ceny","hbz2"," 50 ");

   zdialog_add_widget(zd,"hbox","hbazs","dialog");
   zdialog_add_widget(zd,"label","labazs1","hbazs",ZTX("+Seconds"),"space=3");
   zdialog_add_widget(zd,"spin","azsecs","hbazs","0|999|1|0","space=3");
   zdialog_add_widget(zd,"label","labazs2","hbazs",ZTX("after zoom"),"space=3");

   zdialog_cb_app(zd,"trans",ZTX("next"));                                 //  default transition

   for (ii = 0; ii < SSNF; ii++)                                           //  add all transitions to dropdown list
      zdialog_cb_app(zd,"trans",ss_trantab[ii].tranname);
   
   if (curr_file) {                                                        //  set to curr. file 
      for (ii = 0; ii < ss_Nfiles; ii++)                                   //    if member of collection
         if (strEqu(curr_file,ss_imagetab[ii].imagefile)) break;
      if (ii == ss_Nfiles) ii = 0;
   }
   else ii = 0;
   ss_Fnext = ii;

   pp = strrchr(ss_imagetab[ii].imagefile,'/');                            //  stuff image file prefs into dialog
   if (pp) pp++;
   snprintf(imagefile,100,"%s %s",ZTX("Image File:"),pp); 
   zdialog_stuff(zd,"labimf",imagefile);

   zdialog_stuff(zd,"trans",ss_imagetab[ii].tranname);
   zdialog_stuff(zd,"tone",ss_imagetab[ii].Ftone);
   zdialog_stuff(zd,"bzsecs",ss_imagetab[ii].bzsecs);
   zdialog_stuff(zd,"zoom",ss_imagetab[ii].zoom);
   zdialog_stuff(zd,"zsteps",ss_imagetab[ii].zsteps);
   zdialog_stuff(zd,"cenx",ss_imagetab[ii].cenx);
   zdialog_stuff(zd,"ceny",ss_imagetab[ii].ceny);
   zdialog_stuff(zd,"azsecs",ss_imagetab[ii].azsecs);

   zdialog_run(zd,ss_imageprefs_dialog_event);
   zdialog_wait(zd);
   zdialog_free(zd);

   zd_ss_imageprefs = 0;
   ss_saveprefs();

   return;
}


//  image prefs dialog event and completion function

int  ss_imageprefs_dialog_event(zdialog *zd, cchar *event)                 //  v.14.04
{
   int      ii, jj;
   int      addsecs, tone, zoom, zsteps, cenx, ceny;
   char     tranname[32];
   
   ii = ss_Fnext;
   if (ii >= ss_Nfiles) return 1;
   
   if (strEqu(event,"trans")) {
      zdialog_fetch(zd,"trans",tranname,32);
      if (strNeq(tranname,"next")) {
         for (jj = 0; jj < SSNF; jj++)
            if (strEqu(tranname,ss_trantab[jj].tranname)) break;
         if (jj == SSNF) return 1;
      }
      strncpy0(ss_imagetab[ii].tranname,tranname,32);
   }

   if (strEqu(event,"tone")) {
      zdialog_fetch(zd,"tone",tone);
      ss_imagetab[ii].Ftone = tone;
   }

   if (strEqu(event,"bzsecs")) {
      zdialog_fetch(zd,"bzsecs",addsecs);
      ss_imagetab[ii].bzsecs = addsecs;
   }

   if (strEqu(event,"zoom")) {
      zdialog_fetch(zd,"zoom",zoom);
      ss_imagetab[ii].zoom = zoom;
   }

   if (strEqu(event,"zsteps")) {
      zdialog_fetch(zd,"zsteps",zsteps);
      ss_imagetab[ii].zsteps = zsteps;
   }

   if (strEqu(event,"cenx")) {
      zdialog_fetch(zd,"cenx",cenx);
      ss_imagetab[ii].cenx = cenx;
   }

   if (strEqu(event,"ceny")) {
      zdialog_fetch(zd,"ceny",ceny);
      ss_imagetab[ii].ceny = ceny;
   }

   if (strEqu(event,"azsecs")) {
      zdialog_fetch(zd,"azsecs",addsecs);
      ss_imagetab[ii].azsecs = addsecs;
   }

   return 1;
}


//  response function for gallery thumbnail left-click

void slideshow_Lclick_func(int Nth)                                        //  v.14.04
{
   zdialog     *zd;
   int         ii;
   char        *pp, imagefile[100];
   
   if (! clicked_file) return;
   zfree(clicked_file);
   clicked_file = 0;

   zd = zd_ss_imageprefs;
   if (! zd) return;

   if (! ss_Nfiles) {
      zmessageACK(Mwin,0,ZTX("invalid collection"));
      return;
   }

   ii = Nth;
   if (ii >= ss_Nfiles) return;
   ss_Fnext = ii;
   
   ss_imagetab[ii].cenx = clicked_width;                                   //  set zoom center from thumbnail
   ss_imagetab[ii].ceny = clicked_height;                                  //    click position
   
   pp = strrchr(ss_imagetab[ii].imagefile,'/');                            //  stuff image file prefs into dialog
   if (pp) pp++;
   snprintf(imagefile,100,"%s %s",ZTX("Image File:"),pp); 
   zdialog_stuff(zd,"labimf",imagefile);
   zdialog_stuff(zd,"trans",ss_imagetab[ii].tranname);
   zdialog_stuff(zd,"tone",ss_imagetab[ii].Ftone);
   zdialog_stuff(zd,"bzsecs",ss_imagetab[ii].bzsecs);
   zdialog_stuff(zd,"zoom",ss_imagetab[ii].zoom);
   zdialog_stuff(zd,"zsteps",ss_imagetab[ii].zsteps);
   zdialog_stuff(zd,"cenx",ss_imagetab[ii].cenx);
   zdialog_stuff(zd,"ceny",ss_imagetab[ii].ceny);
   zdialog_stuff(zd,"azsecs",ss_imagetab[ii].azsecs);
   
   return;
}


//  Load all data for a specific slide show from a slide show preferences file.
//  Set defaults if no data previously defined.

void ss_loadprefs()                                                        //  v.14.04
{
   FILE        *fid;
   char        buff[maxfcc];
   char        prefsfile[300], *pp;
   int         ii, jj, nn, ftype, format;
   char        tranname[32];
   int         n1, n2, n3, n4, n5, n6, n7;
   
   for (ii = 0; ii < ss_Nfiles; ii++)                                      //  free prior image data if any
      zfree(ss_imagetab[ii].imagefile);

   ss_Nfiles = 0;
   
   fid = fopen(ss_collfile,"r");                                           //  open collection file
   if (! fid) {
      zmessageACK(Mwin,0,ZTX("invalid collection"));
      return;
   }

   for (ii = 0; ii < SSMAXI; ) {                                           //  read all image file names
      pp = fgets_trim(buff,maxfcc,fid);
      if (! pp) break;
      ftype = image_file_type(pp);                                         //  screen out deleted image files
      if (ftype != 2) continue;
      ss_imagetab[ii].imagefile = zstrdup(pp);                             //  add to image table
      ii++;
   }

   fclose(fid);
   ss_Nfiles = ii;

   if (! ss_Nfiles) {
      zmessageACK(Mwin,0,ZTX("invalid collection"));
      return;
   }

   navi::gallerytype = 4;                                                  //  open gallery with slide show collection
   gallery(ss_collfile,"initF");
   gallery(0,"paint",0);
   m_viewmode(0,"G");

   ss_secs = 3;                                                            //  defaults: image display time
   ss_cliplim = 0;                                                         //  image clip limit = no clipping
   ss_showcaps = 0;                                                        //  show captions OFF
   strcpy(ss_musicfile,"none");                                            //  music file = NONE

   for (ii = 0; ii < SSNF; ii++)                                           //  initialize transitions table
      ss_trantab[ii] = ss_trantab_default[ii];                             //    with default preferences
   
   for (ii = 0; ii < SSNF; ii++) {
      ss_Tused[ii] = ii;                                                   //  all transition types are used
      ss_Tlast[ii] = 0;                                                    //  last used list is empty
   }

   ss_random = 0;                                                          //  random transitions = NO
   ss_Nused = SSNF;                                                        //  used transitions = all
   ss_Tnext = 0;                                                           //  next = first

   for (ii = 0; ii < ss_Nfiles; ii++) {                                    //  initialize image table with
      strcpy(ss_imagetab[ii].tranname,"next");                             //    default preferences
      ss_imagetab[ii].Ftone = 0;
      ss_imagetab[ii].bzsecs = 0;
      ss_imagetab[ii].zoom = 0;
      ss_imagetab[ii].zsteps = 200;
      ss_imagetab[ii].cenx = 50;
      ss_imagetab[ii].ceny = 50;
      ss_imagetab[ii].azsecs = 0;
   }

   snprintf(prefsfile,300,"%s/%s",slideshow_dirk,ss_collname);
   fid = fopen(prefsfile,"r");                                             //  open slide show prefs file
   if (! fid) return;

   format = 0;
   
   while (true)
   {
      pp = fgets_trim(buff,300,fid,1);
      if (! pp) break;
      
      if (strnEqu(pp,"overall:",8)) {
         format = 1;
         continue;
      }

      if (strnEqu(pp,"transitions:",12)) {
         format = 2;
         continue;
      }

      if (strnEqu(pp,"images:",7)) {
         format = 3;
         continue;
      }
      
      if (format == 1)                                                     //  overall preferences
      {
         nn = sscanf(buff,"%d %d %d %d \"%m[^\"] ",&n1,&n2,&n3,&n4,&pp); 
         if (nn != 5) goto format_error;
         ss_secs = n1;
         ss_cliplim = n2;
         ss_showcaps = n3;
         ss_random = n4;
         if (*pp == '/') strncpy0(ss_musicfile,pp,300);
         zfree(pp);
      }
      
      if (format == 2)                                                     //  transition preferences
      {
         nn = sscanf(buff,"%31s %d %d %d ",tranname,&n1,&n2,&n3);
         if (nn != 4) goto format_error;
         for (ii = 0; ii < SSNF; ii++)
            if (strEqu(tranname,ss_trantab[ii].tranname)) break;
         if (ii == SSNF) goto format_error;
         ss_trantab[ii].enabled = n1;
         ss_trantab[ii].slowdown = n2;
         ss_trantab[ii].preference = n3;
      }
      
      if (format == 3)                                                     //  image file preferences
      {
         pp = strrchr(buff,'"');                                           //  rec. 1 = "/.../filename.jpg"
         if (pp) *pp = 0;                                                  //  remove trailing quote
         pp = buff+1;                                                      //  and leading quote
         for (ii = 0; ii < ss_Nfiles; ii++)                                //  search collection for matching image
            if (strEqu(pp,ss_imagetab[ii].imagefile)) break;
         if (ii == ss_Nfiles) {                                            //  image not in collection, ignore
            fgets_trim(buff,300,fid,1);                                    //  discard rec. 2
            continue;                                                      //  next image
         }

         pp = fgets_trim(buff,300,fid,1);                                  //  rec. 2 = preferences
         if (! pp) goto format_error;
         nn = sscanf(buff,"%31s %d %d %d %d %d %d %d ",tranname,&n1,&n2,&n3,&n4,&n5,&n6,&n7);
         if (nn != 8) goto format_error;
         strncpy0(ss_imagetab[ii].tranname,tranname,32);
         ss_imagetab[ii].Ftone = n1;
         ss_imagetab[ii].bzsecs = n2;
         ss_imagetab[ii].zoom = n3;
         ss_imagetab[ii].zsteps = n4;
         ss_imagetab[ii].cenx = n5;
         ss_imagetab[ii].ceny = n6;
         ss_imagetab[ii].azsecs = n7;
      }
   }

   fclose(fid);
   
   for (ii = jj = 0; ii < SSNF; ii++) {                                    //  initialize list of enabled 
      if (ss_trantab[ii].enabled) {                                        //    transition types
         ss_Tused[jj] = ii;
         jj++;
      }
   }

   ss_Nused = jj;                                                          //  no. enabled transition types
   return;

format_error:
   zmessageACK(Mwin,0,ZTX("file format error"));
   printf("%s \n",buff);
   fclose(fid);
   return;
}


//  Save all data for a specific slide show to a slide show preferences file.

void ss_saveprefs()                                                        //  v.14.04
{
   FILE        *fid;
   char        prefsfile[300];
   int         ii;

   if (! ss_Nfiles) {
      zmessageACK(Mwin,0,ZTX("invalid collection"));
      return;
   }

   snprintf(prefsfile,300,"%s/%s",slideshow_dirk,ss_collname);
   fid = fopen(prefsfile,"w");                                             //  open slide show prefs file
   if (! fid) {
      zmessageACK(Mwin,0,strerror(errno));
      return;
   }
   
   fprintf(fid,"overall:\n %d %d %d %d \"%s\" \n",
            ss_secs, ss_cliplim, ss_showcaps, ss_random, ss_musicfile);
   
   fprintf(fid,"transitions:\n");

   for (ii = 0; ii < SSNF; ii++)                                           //  transition1: tranname  1  2  10
      fprintf(fid,"%s %d %d %d \n", ss_trantab[ii].tranname,
              ss_trantab[ii].enabled, ss_trantab[ii].slowdown, 
              ss_trantab[ii].preference);

   fprintf(fid,"images:\n");

   for (ii = 0; ii < ss_Nfiles; ii++) {
      fprintf(fid,"\"%s\"\n",ss_imagetab[ii].imagefile);
      fprintf(fid,"%s %d %d %d %d %d %d %d \n",
               ss_imagetab[ii].tranname, ss_imagetab[ii].Ftone, 
               ss_imagetab[ii].bzsecs, 
               ss_imagetab[ii].zoom, ss_imagetab[ii].zsteps, 
               ss_imagetab[ii].cenx, ss_imagetab[ii].ceny,
               ss_imagetab[ii].azsecs);
   }
   
   fclose(fid);
   
   return;
}


//  Show next slide when time is up or user navigates with arrow keys.
//  Cycles every 0.1 seconds when slide show is active.

int ss_timerfunc(void *)                                                   //  v.14.04
{
   int         ii, jj;
   char        buff[300];

   if (checkpend("all")) return 0;                                         //  conflict, quit

   if (ss_escape || FGW != 'F') {
      if (ss_pxbold) g_object_unref(ss_pxbold);                            //  free memory
      if (ss_pxbnew) g_object_unref(ss_pxbnew);
      ss_pxbold = ss_pxbnew = 0;
      win_unfullscreen();                                                  //  restore old window size, menu etc.
      Fslideshow = 0;                                                      //  reset flags
      Fblankwindow = 0;  
      Fblowup = 0;
      ss_escape = 0;
      m_slideshow(0,0);                                                    //  return to slide show dialog
      return 0;                                                            //  stop the timer
   }
   
   if (ss_spacebar) {                                                      //  pause/resume
      ss_spacebar = 0;
      ss_blank = 0;
      if (strEqu(ss_state,"pause")) ss_state = "next";
      else ss_state = "pause";
      ss_timer = get_seconds();                                            //  next image immediately
      return 1;
   }

   if (ss_Bkey) {                                                          //  blank/unblank screen
      ss_Bkey = 0;
      ss_blank = 1 - ss_blank;
      if (ss_blank) ss_blankwindow();
      else ss_instant();
      ss_state = "pause";
      return 1;
   }
   
   if (ss_Larrow) {                                                        //  go back one image
      ss_Larrow = 0;
      ss_Fnext--;
      if (ss_Fnext < 0) ss_Fnext = ss_Nfiles - 1;

      ii = ss_Fnext;
      ss_oldfile = ss_newfile;                                             //  old file = new
      if (ss_pxbold) g_object_unref(ss_pxbold);
      ss_pxbold = ss_pxbnew;                                               //  old pixbuf = new
      ss_rsold = ss_rsnew;
      ss_newfile = ss_imagetab[ii].imagefile;                              //  new new file
      ss_pxbnew = ss_loadpxb(ss_newfile);                                  //  new new pixbuf from new new file
      if (! ss_pxbnew) return 0;                                           //  failure
      ss_rsnew = gdk_pixbuf_get_rowstride(ss_pxbnew);                      //  row stride

      ss_instant();
      ss_blank = 0;
      ss_state = "pause";
      return 1;
   }
   
   if (ss_Rarrow) {                                                        //  go forward one image
      ss_Rarrow = 0;
      ss_Fnext++;
      if (ss_Fnext >= ss_Nfiles) ss_Fnext = 0;

      ii = ss_Fnext;
      ss_oldfile = ss_newfile;                                             //  old file = new
      if (ss_pxbold) g_object_unref(ss_pxbold);
      ss_pxbold = ss_pxbnew;                                               //  old pixbuf = new
      ss_rsold = ss_rsnew;
      ss_newfile = ss_imagetab[ii].imagefile;                              //  new new file
      ss_pxbnew = ss_loadpxb(ss_newfile);                                  //  new new pixbuf from new new file
      if (! ss_pxbnew) return 0;                                           //  failure
      ss_rsnew = gdk_pixbuf_get_rowstride(ss_pxbnew);                      //  row stride

      ss_instant();
      ss_blank = 0;
      ss_state = "pause";
      return 1;
   }
   
   if (strEqu(ss_state,"pause")) return 1;                                 //  do nothing
   
   if (strEqu(ss_state,"first")) {                                         //  first image
      ss_Fnext--;                                                          //  make next image = same
      ss_state = "next";
      ss_timer = get_seconds();
   }
   
   if (strEqu(ss_state,"next"))                                            //  next image (or first)
   {
      if (get_seconds() < ss_timer) return 1;                              //  wait for my time

      ss_Fnext++;                                                          //  set next image to show
      if (ss_Fnext >= ss_Nfiles) ss_Fnext = 0;

      ii = ss_Fnext;
      ss_oldfile = ss_newfile;                                             //  old file = new
      if (ss_pxbold) g_object_unref(ss_pxbold);
      ss_pxbold = ss_pxbnew;                                               //  old pixbuf = new
      ss_rsold = ss_rsnew;
      ss_newfile = ss_imagetab[ii].imagefile;                              //  new new file
      ss_pxbnew = ss_loadpxb(ss_newfile);                                  //  new new pixbuf from new new file
      if (! ss_pxbnew) return 0;                                           //  failure
      ss_rsnew = gdk_pixbuf_get_rowstride(ss_pxbnew);                      //  row stride
      
      if (! ss_pxbold) ss_instant();                                       //  no prior image, use instant transition
      else {
         jj = ss_nextrans();                                               //  select next transition type
         ss_slowdown = ss_trantab[jj].slowdown;                            //  set slowdown factor for transition
         ss_trantab[jj].func();                                            //  call transition function
      }
      
      if (Fcaptions) ss_captions();                                        //  show captions/comments on image 

      if (ss_imagetab[ii].Ftone) {                                         //  play tone if specified             v.14.04
         snprintf(buff,300,"paplay %s/slideshow-tone.oga",slideshow_dirk);
         shell_quiet(buff);
      }

      ss_state = "bzwait";
      ss_timer = get_seconds() + ss_imagetab[ii].bzsecs;
      return 1;
   }
   
   if (strEqu(ss_state,"bzwait")) {
      if (get_seconds() < ss_timer) return 1;                              //  wait for my time
      ii = ss_Fnext;
      ss_zoom = ss_imagetab[ii].zoom;                                      //  zoom % size increase
      ss_zsteps = ss_imagetab[ii].zsteps;                                  //  zoom steps 
      ss_cenx = ss_imagetab[ii].cenx;                                      //  target location for final center
      ss_ceny = ss_imagetab[ii].ceny;                                      //  (0-100% of image, 50/50 = middle)
      if (ss_zoom) ss_zoomin();
      ss_state = "azwait";
      ss_timer = get_seconds() + ss_imagetab[ii].azsecs;
      return 1;
   }

   if (strEqu(ss_state,"azwait")) {
      if (get_seconds() < ss_timer) return 1;                              //  wait for my time
      ss_state = "sswait";
      ss_timer = get_seconds() + ss_secs;
      return 1;
   }
   
   if (strEqu(ss_state,"sswait")) {
      if (get_seconds() < ss_timer) return 1;                              //  wait for my time
      ss_state = "next";
      return 1;
   }
      
   return 1;
}


//  select next transision type to use
//  mode = sequential: use each enabled transition type in sequence
//  mode = random: exclude recently used, choose random from remaining

int ss_nextrans()
{
   int      ii, jj, maxii, maxjj, next;
   float    maxrank, rank;
   
   ii = ss_Fnext;
   if (strNeq(ss_imagetab[ii].tranname,"next")) {                          //  image transition not "next"
      for (jj = 0; jj < SSNF; jj++)
         if (strEqu(ss_trantab[jj].tranname,ss_imagetab[ii].tranname)) break;
      if (jj < SSNF) {
         next = jj;                                                        //  assigned transition type
         for (ii = ss_Nused - 1; ii > 0; ii--)                             //  >> most recently used
            ss_Tlast[ii] = ss_Tlast[ii-1];
         ss_Tlast[0] = next;
         return next;
      }
   }
   
   if (ss_Nused < 5 || ss_random == 0)                                     //  few enabled transition types
   {                                                                       //    or sequential mode
      ss_Tnext++;
      if (ss_Tnext == ss_Nused) ss_Tnext = 0;                              //  select transition types sequentially
      next = ss_Tused[ss_Tnext];
   }
   
   else                                                                    //  select transition types randomly
   {
      maxrank = 0;
      maxii = 0;
      maxjj = ss_Nused / 2;                                                //  most recently used to exclude
      if (maxjj > 4) maxjj = 4;                                            //  max. 4 

      for (ii = 0; ii < ss_Nused; ii++)                                    //  search enabled transitions
      {
         for (jj = 0; jj < maxjj; jj++)                                    //  exclude most recently used 50%
            if (ss_Tused[ii] == ss_Tlast[jj]) break;
         if (jj < maxjj) continue;
         jj = ss_Tused[ii];
         rank = ss_trantab[jj].preference * drandz();                      //  rank = preference * random value
         if (rank > maxrank) {
            maxrank = rank;                                                //  remember highest rank
            maxii = ii;
         }
      }

      next = ss_Tused[maxii];                                              //  transition to use

      for (ii = ss_Nused - 1; ii > 0; ii--)                                //  make it most recent
         ss_Tlast[ii] = ss_Tlast[ii-1];
      ss_Tlast[0] = next;  
   }

   return next;
}


//  write captions and comments at the top of the image

void ss_captions()                                                         //  v.14.02
{
   cchar        *keynames[2] = { iptc_caption_key, exif_comment_key };
   char         **keyvals;
   char         caption[200], comment[200];
   static char  text[402];

   static PangoFontDescription   *pangofont = null;
   static PangoLayout            *pangolayout = null;
   int                           ww, hh;

   *caption = *comment = 0;

   keyvals = exif_get(ss_newfile,keynames,2);                              //  get captions and comments metadata

   if (keyvals[0]) {
      strncpy0(caption,keyvals[0],200);
      zfree(keyvals[0]);
   }

   if (keyvals[1]) {
      strncpy0(comment,keyvals[1],200);
      zfree(keyvals[1]);
   }

   *text = 0;

   if (*caption) strcpy(text,caption);
   if (*caption && *comment) strcat(text,"\n");
   if (*comment) strcat(text,comment);

   if (! *text) return;

   pangofont = pango_font_description_from_string("Sans 12");              //  make pango layout for font
   pangolayout = gtk_widget_create_pango_layout(Cdraw,0);
   pango_layout_set_font_description(pangolayout,pangofont);
   pango_layout_set_text(pangolayout,text,-1);                             //  add text to layout
   pango_layout_get_pixel_size(pangolayout,&ww,&hh);
   
   ss_cr = gdk_cairo_create(gdkwin); 

   cairo_set_line_width(ss_cr,1);
   cairo_set_source_rgb(ss_cr,1,1,1);                                      //  draw white background
   cairo_rectangle(ss_cr,10,10,ww,hh);
   cairo_fill(ss_cr);

   cairo_move_to(ss_cr,10,10);                                             //  draw layout with text
   cairo_set_source_rgb(ss_cr,0,0,0);
   pango_cairo_show_layout(ss_cr,pangolayout);

   cairo_destroy(ss_cr);
   return;
}


//  Load image and rescale to fit in window size.
//  If image aspect ratio is close enough to window ratio, 
//  truncate to avoid having margins around around the image.

GdkPixbuf * ss_loadpxb(char *file)                                         //  v.14.02
{
   GdkPixbuf   *pxbin, *pxbtemp, *pxbout;
   int         ww1, hh1, ww2, hh2, cc;
   int         Iorgx, Iorgy, Worgx, Worgy;
   float       Rm, Rw, dR;
   int         px, py, px1, py1, px2, py2;
   uint8       *pixels1, *pixels2, *pix1, *pix2;
   int         rs1, rs2, nch1;
   GError      *gerror = 0;
   
   pxbin = gdk_pixbuf_new_from_file(file,&gerror);                         //  load image file into pixbuf
   if (! pxbin) {
      zmessageACK(Mwin,0,"%s",gerror->message);
      return 0;
   }
   
   ww1 = gdk_pixbuf_get_width(pxbin);                                      //  image dimensions
   hh1 = gdk_pixbuf_get_height(pxbin);
   
   ww2 = ss_ww;                                                            //  window dimensions
   hh2 = ss_hh;
   
   Rm = 1.0 * ww1 / hh1;                                                   //  image width/height ratio
   Rw = 1.0 * ww2 / hh2;                                                   //  window width/height ratio
   dR = fabsf(Rm - Rw) / Rw;                                               //  discrepancy ratio
   
   if (dR <= 0.01 * ss_cliplim) {                                          //  discrepancy within user limit
      if (Rw >= Rm) {
         ww1 = ww2;                                                        //  height will be clipped
         hh1 = ww1 / Rm;
      }
      else {
         hh1 = hh2;                                                        //  width will be clipped
         ww1 = hh1 * Rm;
      }
   }
   else {                                                                  //  discrepancy too great
      if (Rw >= Rm) {
         hh1 = hh2;                                                        //  ratio image to fit in window
         ww1 = hh1 * Rm;
      }
      else {
         ww1 = ww2;
         hh1 = ww1 / Rm;
      }
   }

   pxbtemp = gdk_pixbuf_scale_simple(pxbin,ww1,hh1,BILINEAR);              //  rescale image
   g_object_unref(pxbin);
   
   Iorgx = (ww1 - ww2) / 2.0;                                              //  top left corner of image to copy from
   if (Iorgx < 0) Iorgx = 0;
   Iorgy = (hh1 - hh2) / 2.0;
   if (Iorgy < 0) Iorgy = 0;
   
   Worgx = (ww2 - ww1) / 2.0;                                              //  top left corner of window to copy to
   if (Worgx < 0) Worgx = 0;
   Worgy = (hh2 - hh1) / 2.0;
   if (Worgy < 0) Worgy = 0;

   if (ww2 < ww1) ww1 = ww2;                                               //  copy width
   if (hh2 < hh1) hh1 = hh2;                                               //  copy height
   
   pixels1 = gdk_pixbuf_get_pixels(pxbtemp);                               //  input pixels location
   rs1 = gdk_pixbuf_get_rowstride(pxbtemp);                                //  row stride
   nch1 = gdk_pixbuf_get_n_channels(pxbtemp);                              //  colors (1 or 3 channels)

   pxbout = gdk_pixbuf_new(GDKRGB,0,8,ww2,hh2);                            //  create output pixbuf = window size
   pixels2 = gdk_pixbuf_get_pixels(pxbout);                                //  pixels location
   rs2 = gdk_pixbuf_get_rowstride(pxbout);                                 //  row stride

   cc = rs2 * hh2;                                                         //  clear output pixels black
   memset(pixels2,0,cc);

   for (py = 0; py < hh1; py++)                                            //  copy image to output pixmap
   for (px = 0; px < ww1; px++)                                            //    and center the image
   {
      px1 = px + Iorgx;
      py1 = py + Iorgy;
      px2 = px + Worgx;
      py2 = py + Worgy;
      pix1 = pixels1 + py1 * rs1 + px1 * nch1;
      pix2 = pixels2 + py2 * rs2 + px2 * 3;
      if (nch1 > 2) {
         pix2[0] = pix1[0];
         pix2[1] = pix1[1];
         pix2[2] = pix1[2];
      }
      else pix2[0] = pix2[1] = pix2[2] = pix1[0];
   }
   
   g_object_unref(pxbtemp);

   return pxbout;
}


//  write black to entire window

void ss_blankwindow()
{   
   ss_cr = gdk_cairo_create(gdkwin);   
   gdk_cairo_set_source_rgba(ss_cr,&GDKdark);
   cairo_paint(ss_cr);
   cairo_destroy(ss_cr);
   gdk_flush();
   return;
}


//  instant transition (also used for keyboard arrow keys)

void ss_instant()
{      
   ss_cr = gdk_cairo_create(gdkwin);   
   gdk_cairo_set_source_pixbuf(ss_cr,ss_pxbnew,0,0);
   cairo_paint(ss_cr);
   cairo_destroy(ss_cr);
   gdk_flush();
   return;
}


//  fade-out / fade-in transition

void ss_fadein()
{
   GdkPixbuf   *pxbmix;
   int         ii, jj, kk, px, py, rs, iinc;
   float       newpart, oldpart;
   uint8       *pixels1, *pixels2, *pixels3;
   uint8       *pix1, *pix2, *pix3;

   ss_cr = gdk_cairo_create(gdkwin);   
   pxbmix = gdk_pixbuf_copy(ss_pxbold);
   rs = gdk_pixbuf_get_rowstride(pxbmix);

   pixels1 = gdk_pixbuf_get_pixels(ss_pxbold);
   pixels2 = gdk_pixbuf_get_pixels(ss_pxbnew);
   pixels3 = gdk_pixbuf_get_pixels(pxbmix);
   
   iinc = 10.0 / (1 + ss_slowdown / 2.0);                                  //  slowdown factor                    v.14.01
   if (iinc < 1) iinc = 1;
   
   for (ii = 0; ii <= 100; ii += iinc)
   {
      newpart = 0.01 * ii;
      oldpart = 1.0 - newpart;

      for (jj = 0; jj < 2; jj++)                                           //  four passes, each modifies 25%
      for (kk = 0; kk < 2; kk++)                                           //    of the pixels (visually smoother)
      {
         for (py = jj; py < ss_hh; py += 2)
         for (px = kk; px < ss_ww; px += 2)
         {
            pix1 = pixels1 + py * ss_rsold + px * 3;
            pix2 = pixels2 + py * ss_rsnew + px * 3;
            pix3 = pixels3 + py * rs + px * 3;
            pix3[0] = newpart * pix2[0] + oldpart * pix1[0];
            pix3[1] = newpart * pix2[1] + oldpart * pix1[1];
            pix3[2] = newpart * pix2[2] + oldpart * pix1[2];
         }

         gdk_cairo_set_source_pixbuf(ss_cr,pxbmix,0,0);
         cairo_paint(ss_cr);
         gdk_flush();                                                      //  avoid jerky output (Ubuntu 13.04)  v.13.05
      }
   }

   g_object_unref(pxbmix);
   cairo_destroy(ss_cr);
   return;
}


//  new image rolls over prior image from left to right

void ss_rollright()
{
   GdkPixbuf   *pixbuf;
   int         px;
   float       delay = 1.0 * ss_slowdown / ss_ww;                          //  v.14.01
   uint8       *pixels, *pix3;

   ss_cr = gdk_cairo_create(gdkwin);   
   pixels = gdk_pixbuf_get_pixels(ss_pxbnew);

   for (px = 0; px < ss_ww-4; px += 4)                                     //  4-wide     v.13.09
   {
      pix3 = pixels + px * 3;
      pixbuf = gdk_pixbuf_new_from_data(pix3,GDKRGB,0,8,4,ss_hh,ss_rsnew,0,0);
      gdk_cairo_set_source_pixbuf(ss_cr,pixbuf,px,0);
      cairo_paint(ss_cr);
      g_object_unref(pixbuf);
      gdk_flush();                                                         //  avoid jerky output (Ubuntu 13.04)  v.13.05
      zsleep(delay);
   }

   cairo_destroy(ss_cr);
   return;
}


//  new image rolls over prior image from top down

void ss_rolldown()
{
   GdkPixbuf   *pixbuf;
   int         py;
   float       delay = 1.0 * ss_slowdown / ss_hh;                          //  v.14.01
   uint8       *pixels, *pix3;

   ss_cr = gdk_cairo_create(gdkwin);
   pixels = gdk_pixbuf_get_pixels(ss_pxbnew);

   for (py = 0; py < ss_hh-2; py += 4)                                     //  4-deep        v.13.09
   {
      pix3 = pixels + py * ss_rsnew;
      pixbuf = gdk_pixbuf_new_from_data(pix3,GDKRGB,0,8,ss_ww,4,ss_rsnew,0,0);
      gdk_cairo_set_source_pixbuf(ss_cr,pixbuf,0,py);
      cairo_paint(ss_cr);
      g_object_unref(pixbuf);
      gdk_flush();                                                         //  avoid jerky output (Ubuntu 13.04)  v.13.05
      zsleep(delay);
   }

   cairo_destroy(ss_cr);
   return;
}


//  new image opens up in horizontal rows like venetian blinds

void ss_venetian()
{
   GdkPixbuf   *pixbuf;
   int         py1, py2;
   uint8       *pixels, *pix3;
   int         louver, Nlouvers = 20;
   int         louversize = ss_hh / Nlouvers;
   float       delay = 1.0 / louversize * (1 + ss_slowdown / 2.0);         //  v.14.01

   ss_cr = gdk_cairo_create(gdkwin);
   pixels = gdk_pixbuf_get_pixels(ss_pxbnew);

   for (py1 = 0; py1 < louversize; py1++)                                  //  y-row within each louver
   {
      for (louver = 0; louver < Nlouvers; louver++)                        //  louver, first to last
      {
         py2 = py1 + louver * louversize;
         if (py2 >= ss_hh) break;
         pix3 = pixels + py2 * ss_rsnew;
         pixbuf = gdk_pixbuf_new_from_data(pix3,GDKRGB,0,8,ss_ww,1,ss_rsnew,0,0);
         gdk_cairo_set_source_pixbuf(ss_cr,pixbuf,0,py2);
         cairo_paint(ss_cr);
         g_object_unref(pixbuf);
      }

      gdk_flush();                                                         //  avoid jerky output (Ubuntu 13.04)  v.13.05
      zsleep(delay);
   }

   cairo_destroy(ss_cr);
   return;
}


//  a grate opens up to show new image

void ss_grate()
{
   GdkPixbuf   *pixbuf;
   int         px1, px2, py1, py2;
   uint8       *pixels, *pix3;
   int         row, col, Nrow, Ncol;                                       //  rows and columns
   int         boxww, boxhh;
   float       delay;

   ss_cr = gdk_cairo_create(gdkwin);
   pixels = gdk_pixbuf_get_pixels(ss_pxbnew);

   Ncol = 20;                                                              //  20 columns
   boxww = boxhh = ss_ww / Ncol;                                           //  square boxes
   Nrow = ss_hh / boxhh;                                                   //  corresp. rows
   Ncol++;                                                                 //  round up
   Nrow++;
   delay = 1.0 / boxhh * (1 + ss_slowdown / 2.0);                          //  v.14.01

   for (py1 = 0; py1 < boxhh; py1++)
   {
      for (row = 0; row < Nrow; row++)
      {
         py2 = py1 + row * boxhh;
         if (py2 >= ss_hh) break;
         pix3 = pixels + py2 * ss_rsnew;
         pixbuf = gdk_pixbuf_new_from_data(pix3,GDKRGB,0,8,ss_ww,1,ss_rsnew,0,0);
         gdk_cairo_set_source_pixbuf(ss_cr,pixbuf,0,py2);
         cairo_paint(ss_cr);
         g_object_unref(pixbuf);
      }

      px1 = py1;

      for (col = 0; col < Ncol; col++)
      {
         px2 = px1 + col * boxww;
         if (px2 >= ss_ww) break;
         pix3 = pixels + px2 * 3;
         pixbuf = gdk_pixbuf_new_from_data(pix3,GDKRGB,0,8,1,ss_hh,ss_rsnew,0,0);
         gdk_cairo_set_source_pixbuf(ss_cr,pixbuf,px2,0);
         cairo_paint(ss_cr);
         g_object_unref(pixbuf);
      }

      gdk_flush();                                                         //  avoid jerky output (Ubuntu 13.04)  v.13.05
      zsleep(delay);
   }

   cairo_destroy(ss_cr);
   return;
}


//  A rectangle opens up from the center and expands outward

void ss_rectangle()
{
   GdkPixbuf   *pixbuf;
   int         px1, py1, px2, py2, px3, py3;
   int         ww1, hh1, ww2, hh2;
   uint8       *pixels, *pix3;
   int         step, Nsteps = 200;
   float       delay = 1.0 / Nsteps * (1 + ss_slowdown / 2.0);             //  v.14.01

   ss_cr = gdk_cairo_create(gdkwin);
   pixels = gdk_pixbuf_get_pixels(ss_pxbnew);

   for (step = 1; step < Nsteps; step++)
   {
      ww1 = ss_ww * step / Nsteps;
      hh1 = ww1 * ss_hh / ss_ww;
      ww2 = ss_ww / Nsteps / 2 + 1;
      hh2 = ss_hh / Nsteps / 2 + 1;

      px1 = (ss_ww - ww1) / 2;
      py1 = (ss_hh - hh1) / 2;
      px2 = px1 + ww1 - ww2;
      py2 = py1;
      px3 = px1;
      py3 = py1 + hh1 - hh2;

      pix3 = pixels + py1 * ss_rsnew + px1 * 3;
      pixbuf = gdk_pixbuf_new_from_data(pix3,GDKRGB,0,8,ww1+1,hh2+1,ss_rsnew,0,0);
      gdk_cairo_set_source_pixbuf(ss_cr,pixbuf,px1,py1);
      cairo_paint(ss_cr);
      g_object_unref(pixbuf);

      pix3 = pixels + py2 * ss_rsnew + px2 * 3;
      pixbuf = gdk_pixbuf_new_from_data(pix3,GDKRGB,0,8,ww2+1,hh1+1,ss_rsnew,0,0);
      gdk_cairo_set_source_pixbuf(ss_cr,pixbuf,px2,py2);
      cairo_paint(ss_cr);
      g_object_unref(pixbuf);

      pix3 = pixels + py3 * ss_rsnew + px3 * 3;
      pixbuf = gdk_pixbuf_new_from_data(pix3,GDKRGB,0,8,ww1+1,hh2+1,ss_rsnew,0,0);
      gdk_cairo_set_source_pixbuf(ss_cr,pixbuf,px3,py3);
      cairo_paint(ss_cr);
      g_object_unref(pixbuf);

      pix3 = pixels + py1 * ss_rsnew + px1 * 3;
      pixbuf = gdk_pixbuf_new_from_data(pix3,GDKRGB,0,8,ww2+1,hh1+1,ss_rsnew,0,0);
      gdk_cairo_set_source_pixbuf(ss_cr,pixbuf,px1,py1);
      cairo_paint(ss_cr);
      g_object_unref(pixbuf);

      gdk_flush();                                                         //  avoid jerky output (Ubuntu 13.04)  v.13.05
      zsleep(delay);
   }

   gdk_cairo_set_source_pixbuf(ss_cr,ss_pxbnew,0,0);                       //  v.14.03
   cairo_paint(ss_cr);
   cairo_destroy(ss_cr);
   return;
}


//  New image sweeps into view like a circular radar image

void ss_radar()
{
   GdkPixbuf   *pixbuf;
   int         px = 0, py = 0, Npx, Npy, Np = 10;
   float       R, Rmax, T, Tmax, dT, delay;
   float       cosT, sinT;
   float       ww2 = ss_ww / 2, hh2 = ss_hh / 2;
   uint8       *pixels, *pix3;

   ss_cr = gdk_cairo_create(gdkwin);
   pixels = gdk_pixbuf_get_pixels(ss_pxbnew);

   Rmax = sqrt(ww2 * ww2 + hh2 * hh2);
   Tmax = PI;
   dT = 0.6 * asinf(Np / Rmax);
   delay = 0.001 * (ss_slowdown / 2.0);                                    //  v.14.01

   for (T = 0; T < Tmax + dT; T += dT)
   {
      cosT = cosf(T);
      sinT = sinf(T);

      for (R = -Rmax; R < Rmax; R += Np)
      {
         px = ww2 + R * cosT;
         py = hh2 - R * sinT;
         if (px < -Np) continue;
         if (py < -Np) continue;
         if (px > ss_ww-1) continue;
         if (py > ss_hh-1) continue;
         Npx = Npy = Np;
         if (px < 0) px = 0;
         if (py < 0) py = 0;
         if (px + Npx > ss_ww-1) Npx = ss_ww-1 - px;
         if (py + Npy > ss_hh-1) Npy = ss_hh-1 - py;
         if (Npx < 1 || Npy < 1) continue;
         pix3 = pixels + py * ss_rsnew + px * 3;
         pixbuf = gdk_pixbuf_new_from_data(pix3,GDKRGB,0,8,Npx,Npy,ss_rsnew,0,0);
         gdk_cairo_set_source_pixbuf(ss_cr,pixbuf,px,py);
         cairo_paint(ss_cr);
         g_object_unref(pixbuf);
      }

      zsleep(delay);
   }

   cairo_destroy(ss_cr);
   return;
}


//  New image closes in from top and bottom with jagged teeth

void ss_jaws()
{
   GdkPixbuf   *pixbuf;
   int         nteeth = 20, Np = 10;
   int         tbase1, tbase2, twidth, tlength, tooth, tpos;
   int         ii, px, py, ww, ww2;
   uint8       *pixels, *pix3;

   ss_cr = gdk_cairo_create(gdkwin);
   pixels = gdk_pixbuf_get_pixels(ss_pxbnew);

   twidth = ss_ww / nteeth;
   tlength = twidth;
   
   Np = Np / (1.0 + ss_slowdown / 4.0);                                    //  v.14.01
   if (Np < 1) Np = 1;

   for (ii = 0; ii <= ss_hh/2 - tlength/2 + 1; ii += Np)
   {
      tbase1 = ii;                                                         //  tooth base from window top to middle
      tbase2 = ss_hh - tbase1 - 1;                                         //  tooth base from window bottom to middle

      for (tooth = 0; tooth <= nteeth; tooth++)                            //  tooth first to last + 1
      {
         for (tpos = 0; tpos < tlength; tpos += Np)                        //  tooth position from base to point
         {
            ww = twidth * (tlength - tpos) / tlength;                      //  tooth width at scan line
            if (ww < 2) break;

            py = tbase1 + tpos;                                            //  top teeth scan line y
            px = twidth / 2 + tooth * twidth - ww / 2;                     //  scan line x to x + ww
            if (px < ss_ww) {
               ww2 = ww;
               if (px + ww2 > ss_ww) ww2 = ss_ww - px;
               pix3 = pixels + py * ss_rsnew + px * 3;
               pixbuf = gdk_pixbuf_new_from_data(pix3,GDKRGB,0,8,ww2,Np,ss_rsnew,0,0);
               gdk_cairo_set_source_pixbuf(ss_cr,pixbuf,px,py);
               cairo_paint(ss_cr);
               g_object_unref(pixbuf);

            }

            py = tbase2 - tpos;                                            //  bottom teeth scan line y
            py = py - Np;
            px = tooth * twidth - ww / 2;                                  //  scan line x to x + ww
            if (tooth == 0) {
               px = 0;                                                     //  leftmost tooth is half
               ww = ww / 2;
            }
            if (px < ss_ww) {
               ww2 = ww;
               if (px + ww2 > ss_ww) ww2 = ss_ww - px;
               pix3 = pixels + py * ss_rsnew + px * 3;
               pixbuf = gdk_pixbuf_new_from_data(pix3,GDKRGB,0,8,ww2,Np,ss_rsnew,0,0);
               gdk_cairo_set_source_pixbuf(ss_cr,pixbuf,px,py);
               cairo_paint(ss_cr);
               g_object_unref(pixbuf);

            }
         }
      }
   }

   gdk_cairo_set_source_pixbuf(ss_cr,ss_pxbnew,0,0);                       //  v.14.03
   cairo_paint(ss_cr);
   cairo_destroy(ss_cr);
   return;
}


//  An ellipse opens up from the center and expands outward

void ss_ellipse()
{
   GdkPixbuf   *pixbuf;
   uint8       *pixels, *pix3;
   int         step, Nsteps = 100;
   int         px1, py1, ww;
   float       delay = 0.1 / Nsteps;
   float       a, b, a2, b2, px, py, px2, py2;
   float       ww2 = ss_ww / 2, hh2 = ss_hh / 2;

   delay = delay * (1 + ss_slowdown / 4.0);                                //  v.14.01

   ss_cr = gdk_cairo_create(gdkwin);
   pixels = gdk_pixbuf_get_pixels(ss_pxbnew);

   for (step = 1; step < 1.4 * Nsteps; step++)
   {
      a = ww2 * step / Nsteps;                                             //  ellipse a and b constants
      b = a * ss_hh / ss_ww;                                               //    from tiny to >> image size
      a2 = a * a;
      b2 = b * b;

      for (py = -b; py <= +b; py += 3)                                     //  py from top of ellipse to bottom
      {
         while (py < -(hh2-2)) py += 3;
         if (py > hh2-2) break;
         py2 = py * py;
         px2 = a2 * (1.0 - py2 / b2);                                      //  corresponding px value,
         px = sqrt(px2);                                                   //  (+/- from center of ellipse)
         if (px > ww2) px = ww2;
         ww = 2 * px;                                                      //  length of line thru ellipse
         if (ww < 2) continue;
         px1 = ww2 - px;                                                   //  relocate origin
         py1 = py + hh2;
         if (px1 + ww > ss_ww) px1 = ss_ww - ww;                           //  insurance
         if (py1 + 3 > ss_hh) py1 = ss_hh - 3;
         pix3 = pixels + py1 * ss_rsnew + px1 * 3;
         pixbuf = gdk_pixbuf_new_from_data(pix3,GDKRGB,0,8,ww,3,ss_rsnew,0,0);
         gdk_cairo_set_source_pixbuf(ss_cr,pixbuf,px1,py1);
         cairo_paint(ss_cr);
         g_object_unref(pixbuf);
      }

      gdk_flush();                                                         //  avoid jerky output (Ubuntu 13.04)  v.13.05
      zsleep(delay);
   }
   
   gdk_cairo_set_source_pixbuf(ss_cr,ss_pxbnew,0,0);                       //  v.14.03
   cairo_paint(ss_cr);
   cairo_destroy(ss_cr);
   return;
}


//  new image splats onto old image one drop at a time

void ss_raindrops()
{
   GdkPixbuf   *pxbmix, *pxbdrop;
   int         rsmix;
   int         px, py, px1, py1, px2, py2, cx, cy;
   int         Rmin, Rmax, R, R2, dist2, Ndrops;
   uint8       *pixels2, *pixels3;
   uint8       *pix2, *pix3 = 0;
   float       dtime;

   ss_cr = gdk_cairo_create(gdkwin);
   pixels2 = gdk_pixbuf_get_pixels(ss_pxbnew);                             //  source image

   pxbmix = gdk_pixbuf_new(GDKRGB,0,8,ss_ww,ss_hh);                        //  destination image
   pixels3 = gdk_pixbuf_get_pixels(pxbmix);
   rsmix = gdk_pixbuf_get_rowstride(pxbmix);
   memset(pixels3,0,ss_hh * rsmix);                                        //  clear dest. to black

   gdk_cairo_set_source_pixbuf(ss_cr,pxbmix,0,0);                          //  paint black window
   cairo_paint(ss_cr);

   Rmin = ss_ww * 0.01;                                                    //  drop size range                    v.13.12
   Rmax = ss_ww * 0.02;
   Ndrops = 3000;

   for (int ii = 0; ii < Ndrops; ii++)
   {
      cx = drandz() * ss_ww;                                               //  drop location on image
      cy = drandz() * ss_hh;
      R = drandz() * Rmax + Rmin;                                          //  drop size
      R2 = R * R;
      px1 = cx - R;
      if (px1 < 0) px1 = 0;
      py1 = cy - R;
      if (py1 < 0) py1 = 0;
      px2 = cx + R;
      if (px2 >= ss_ww) px2 = ss_ww;
      py2 = cy + R;
      if (py2 > ss_hh) py2 = ss_hh;

      for (py = py1; py < py2; py++)                                       //  copy drop area from new image
      for (px = px1; px < px2; px++)                                       //    to old image
      {
         dist2 = (px-cx) * (px-cx) + (py-cy) * (py-cy);
         if (dist2 > R2) continue;
         pix2 = pixels2 + py * ss_rsnew + px * 3;
         pix3 = pixels3 + py * rsmix + px * 3;
         pix3[0] = pix2[0];
         pix3[1] = pix2[1];
         pix3[2] = pix2[2];
      }

      pxbdrop = gdk_pixbuf_new_subpixbuf(pxbmix,px1,py1,px2-px1,py2-py1);
      gdk_cairo_set_source_pixbuf(ss_cr,pxbdrop,px1,py1);
      cairo_paint(ss_cr);
      g_object_unref(pxbdrop);
      dtime = 0.001 * (1.0 - pow(1.0*ii/Ndrops,0.1));                      //  v.13.12
      dtime = dtime * (1.0 + ss_slowdown);                                 //  v.14.01
      zsleep(dtime);
      gdk_flush();
   }
   
   gdk_cairo_set_source_pixbuf(ss_cr,ss_pxbnew,0,0);                       //  fill bits that are still missing   v.14.02
   cairo_paint(ss_cr);
   g_object_unref(pxbmix);
   cairo_destroy(ss_cr);
   gdk_flush();
   return;
}


//  new image spreads from the middle to left and right edges
//  like a double-door swinging open

void ss_doubledoor()                                                       //  v.13.12
{
   #define GPNFD(pix,ww,hh) gdk_pixbuf_new_from_data(pix,GDKRGB,0,8,ww,hh,ss_rsnew,0,0)

   GdkPixbuf   *pixbuf;
   int         bx, px;
   uint8       *pixels, *pix3;
   float       delay = 0.2 / ss_ww;
   
   delay = delay * (1 + ss_slowdown / 2.0);                                //  v.14.01

   ss_cr = gdk_cairo_create(gdkwin);
   pixels = gdk_pixbuf_get_pixels(ss_pxbnew);
   
   for (bx = 0; bx < ss_ww/2; bx++)                                        //  bx = 0 ... ww/2
   {
      px = ss_ww / 2 - bx;
      pix3 = pixels + 3 * px;                                              //  line from (ww/2-bx,0) to (ww/2-bx,hh-1)
      pixbuf = GPNFD(pix3,1,ss_hh);
      gdk_cairo_set_source_pixbuf(ss_cr,pixbuf,px,0);
      cairo_paint(ss_cr);
      g_object_unref(pixbuf);

      px = ss_ww / 2 + bx;
      pix3 = pixels + 3 * px;                                              //  line from (ww/2+bx,0) to (ww/2+bx,hh-1)
      pixbuf = GPNFD(pix3,1,ss_hh);
      gdk_cairo_set_source_pixbuf(ss_cr,pixbuf,px,0);
      cairo_paint(ss_cr);
      g_object_unref(pixbuf);
      
      gdk_flush();
      zsleep(delay);
   }

   cairo_destroy(ss_cr);
   return;
}


//  Rotate from old image to new image.

namespace ss_rotate_names
{
   int      cx1, cy1, cx2, cy2, cy3, cy4;
   uint8    *pixels1, *pixels2, *pixels3;
   int      rsmix, tbusy[max_threads];
}

void ss_rotate()                                                           //  v.14.02
{
   using namespace ss_rotate_names;

   void * ss_rotate_thread1(void *arg);
   void * ss_rotate_thread2(void *arg);

   GdkPixbuf   *pxbmix;
   float       R, step, stepx = 1, Nsteps = 40;
   int         ii;
   
   ss_cr = gdk_cairo_create(gdkwin);

   pixels1 = gdk_pixbuf_get_pixels(ss_pxbold);                             //  source images
   pixels2 = gdk_pixbuf_get_pixels(ss_pxbnew);

   pxbmix = gdk_pixbuf_new(GDKRGB,0,8,ss_ww,ss_hh);                        //  destination image
   rsmix = gdk_pixbuf_get_rowstride(pxbmix);
   pixels3 = gdk_pixbuf_get_pixels(pxbmix);
   
   Nsteps = Nsteps * (1.0 + ss_slowdown / 2.0);

   for (step = 0; step < Nsteps; step += stepx)
   {
      R = 1.0 * step / Nsteps;
      stepx = 2 - R;                                                       //  2 ... 1

      if (step + stepx >= Nsteps) {
         step = Nsteps;
         R = 1.0;
      }

      cx1 = R * ss_ww / 2.0;                                               //  corners of shrinking trapezoid
      cy1 = 0.3 * R * ss_hh;
      cx2 = ss_ww - cx1;
      cy2 = 0;
      cy3 = ss_hh;
      cy4 = ss_hh - cy1;

      memset(pixels3,0,ss_hh * rsmix);

      for (ii = 0; ii < Nwt; ii++) {                                       //  start worker threads
         tbusy[ii] = 1;
         start_detached_thread(ss_rotate_thread1,&Nval[ii]);
      }

      for (ii = 0; ii < Nwt; ii++)
         while(tbusy[ii]) zsleep(0.001);

      gdk_cairo_set_source_pixbuf(ss_cr,pxbmix,0,0);
      cairo_paint(ss_cr);
      gdk_flush();
   }

   for (step = 0; step < Nsteps; step += stepx)
   {
      R = 1.0 * step / Nsteps;                                             
      stepx = 1 + R;                                                       //  1 ... 2

      if (step + stepx >= Nsteps) {
         step = Nsteps;
         R = 1.0;
      }

      cx1 = ss_ww * (0.5 + 0.5 * R);                                       //  corners of expanding trapezoid
      cy1 = ss_hh * (0.3 - 0.3 * R);
      cx2 = ss_ww * (0.5 - 0.5 * R);
      cy2 = 0;
      cy3 = ss_hh;
      cy4 = ss_hh - cy1;

      memset(pixels3,0,ss_hh * rsmix);

      for (ii = 0; ii < Nwt; ii++) {                                       //  start worker threads
         tbusy[ii] = 1;
         start_detached_thread(ss_rotate_thread2,&Nval[ii]);
      }

      for (ii = 0; ii < Nwt; ii++)
         while(tbusy[ii]) zsleep(0.001);

      gdk_cairo_set_source_pixbuf(ss_cr,pxbmix,0,0);
      cairo_paint(ss_cr);
      gdk_flush();
   }

   g_object_unref(pxbmix);
   cairo_destroy(ss_cr);
   return;
}

void * ss_rotate_thread1(void *arg)
{
   using namespace ss_rotate_names;

   int      index = *((int *) (arg));
   int      px, py, ylo, yhi, vpx, vpy;
   uint8    *pix1, *pix3;
   float    Rx, Ry;
   
   for (px = cx1 + index; px < cx2; px += Nwt)
   {
      Rx = 1.0 * (px - cx1) / (cx2 - cx1);
      ylo = cy1 + Rx * (cy2 - cy1); 
      yhi = cy4 + Rx * (cy3 - cy4);

      for (py = ylo; py < yhi; py++)
      {
         Ry = 1.0 * (py - ylo) / (yhi - ylo);
         vpx = Rx * (ss_ww - 1);
         vpy = Ry * (ss_hh - 1);

         pix1 = pixels1 + vpy * ss_rsold + vpx * 3;
         pix3 = pixels3 + py * rsmix + px * 3;

         pix3[0] = pix1[0];
         pix3[1] = pix1[1];
         pix3[2] = pix1[2];
      }
   }

   tbusy[index] = 0;
   pthread_exit(0);
   return 0;
}

void * ss_rotate_thread2(void *arg)
{
   using namespace ss_rotate_names;

   int      index = *((int *) (arg));
   int      px, py, ylo, yhi, vpx, vpy;
   uint8    *pix2, *pix3;
   float    Rx, Ry;

   for (px = cx2 + index; px < cx1; px += Nwt)
   {
      Rx = 1.0 * (px - cx2) / (cx1 - cx2);
      ylo = cy2 + Rx * (cy1 - cy2); 
      yhi = cy3 + Rx * (cy4 - cy3);

      for (py = ylo; py < yhi; py++)
      {
         Ry = 1.0 * (py - ylo) / (yhi - ylo);
         vpx = Rx * (ss_ww - 1);
         vpy = Ry * (ss_hh - 1);

         pix2 = pixels2 + vpy * ss_rsnew + vpx * 3;
         pix3 = pixels3 + py * rsmix + px * 3;

         pix3[0] = pix2[0];
         pix3[1] = pix2[1];
         pix3[2] = pix2[2];
      }
   }

   tbusy[index] = 0;
   pthread_exit(0);
   return 0;
}


//  slowly zoom-in on the image (Ken Burns effect)

void ss_zoomin()                                                           //  v.14.04
{
   float       zoom = 0.01 * ss_zoom;                                      //  zoom %, 0.0 to 1.0 = 2x
   float       cx = 0.01 * ss_cenx;                                        //  zoom target, 0.5/0.5 = image midpoint
   float       cy = 0.01 * ss_ceny;
   float       nn, Nsteps = ss_zsteps;

   GdkPixbuf   *pxb1, *pxb2, *pxb3;
   GError      *gerror = 0;
   float       Rm, Rw, dR;
   int         ww1, hh1, ww2, hh2, ww3, hh3;
   int         px1, py1, px2, py2, px3, py3;
   int         px4, py4, px5, py5;
   int         Iorgx, Iorgy, Worgx, Worgy;
   int         rs1, nch1, rs2, cc;
   uint8       *pixs1, *pixs2, *pix1, *pix2;
   
   pxb1 = gdk_pixbuf_new_from_file(ss_newfile,&gerror);
   if (! pxb1) {
      zmessageACK(Mwin,0,gerror->message);
      return;
   }

   ww1 = gdk_pixbuf_get_width(pxb1);                                       //  image dimensions
   hh1 = gdk_pixbuf_get_height(pxb1);
   
   ww2 = ss_ww;                                                            //  window dimensions
   hh2 = ss_hh;

/***   
   ww1 = ww1 * (1 + 0.5 * zoom);       /////////////  causes dual-speed gdk_pixbuf_scale_simple()
   hh1 = hh1 * (1 + 0.5 * zoom);
   ww2 = ww2 * (1 + 0.5 * zoom);
   hh2 = hh2 * (1 + 0.5 * zoom);
***/

   Rm = 1.0 * ww1 / hh1;                                                   //  image width/height ratio
   Rw = 1.0 * ww2 / hh2;                                                   //  window width/height ratio
   dR = fabsf(Rm - Rw) / Rw;                                               //  discrepancy ratio
   
   if (dR <= 0.01 * ss_cliplim) {                                          //  discrepancy is within user limit
      if (Rw >= Rm) {
         ww1 = ww2;                                                        //  width fits window
         hh1 = ww1 / Rm;                                                   //  height will be more
      }
      else {
         hh1 = hh2;                                                        //  height fits window
         ww1 = hh1 * Rm;                                                   //  width will be more
      }
   }
   else {                                                                  //  discrepancy is too great
      if (Rw >= Rm) {
         hh1 = hh2;                                                        //  height fits window
         ww1 = hh1 * Rm;                                                   //  width will be less
      }
      else {
         ww1 = ww2;                                                        //  width fits window
         hh1 = ww1 / Rm;                                                   //  height will be less
      }
   }
   
   cx = cx * ww1;                                                          //  zoom center target position
   cy = cy * hh1;

   pxb2 = gdk_pixbuf_scale_simple(pxb1,ww1,hh1,BILINEAR);                  //  rescaled image
   g_object_unref(pxb1);
   pxb1 = pxb2;
   
   Iorgx = (ww1 - ww2) / 2.0;                                              //  top left corner of image to copy from
   if (Iorgx < 0) Iorgx = 0;
   Iorgy = (hh1 - hh2) / 2.0;
   if (Iorgy < 0) Iorgy = 0;
   
   Worgx = (ww2 - ww1) / 2.0;                                              //  top left corner of image to copy to
   if (Worgx < 0) Worgx = 0;
   Worgy = (hh2 - hh1) / 2.0;
   if (Worgy < 0) Worgy = 0;

   if (ww2 < ww1) ww1 = ww2;                                               //  copy width
   if (hh2 < hh1) hh1 = hh2;                                               //  copy height
   
   pixs1 = gdk_pixbuf_get_pixels(pxb1);                                    //  input pixels location
   rs1 = gdk_pixbuf_get_rowstride(pxb1);                                   //  row stride
   nch1 = gdk_pixbuf_get_n_channels(pxb1);                                 //  colors (1 or 3 channels)

   pxb2 = gdk_pixbuf_new(GDKRGB,0,8,ww2,hh2);                              //  output pixbuf matching window w/h
   pixs2 = gdk_pixbuf_get_pixels(pxb2);                                    //  pixels location
   rs2 = gdk_pixbuf_get_rowstride(pxb2);                                   //  row stride

   cc = rs2 * hh2;                                                         //  clear output image black
   memset(pixs2,0,cc);

   for (int py = 0; py < hh1; py++)                                        //  copy input image to output image
   for (int px = 0; px < ww1; px++)                                        //    and center the image
   {
      px1 = px + Iorgx;
      py1 = py + Iorgy;
      px2 = px + Worgx;
      py2 = py + Worgy;
      pix1 = pixs1 + py1 * rs1 + px1 * nch1;
      pix2 = pixs2 + py2 * rs2 + px2 * 3;
      if (nch1 > 2) {
         pix2[0] = pix1[0];
         pix2[1] = pix1[1];
         pix2[2] = pix1[2];
      }
      else pix2[0] = pix2[1] = pix2[2] = pix1[0];
   }
   
   g_object_unref(pxb1);                                                   //  pxb1 = input image, clipped
   pxb1 = pxb2;                                                            //    or with added margins         

   ww1 = ww2;                                                              //  input image size = window size
   hh1 = hh2;

   ww2 = ww2 / (1.0 + zoom);                                               //  final zoomed image size
   hh2 = hh2 / (1.0 + zoom);                                               //  (will expand to window size)

   cx = cx - Iorgx + Worgx;                                                //  zoom target center, adjusted
   cy = cy - Iorgy + Worgy;                                                //    for clipping and margins

   px2 = cx - ww2/2;                                                       //  upper left corner of final
   py2 = cy - hh2/2;                                                       //    image segment to display

   if (px2 + ww2 > ww1) px2 = ww1 - ww2;                                   //  shift final image segment
   if (px2 < 0) px2 = 0;                                                   //    to stay within image bounds
   if (py2 + hh2 > hh1) py2 = hh1 - hh2;
   if (py2 < 0) py2 = 0;
   
   px4 = px2 + ww2;                                                        //  lower right corner of 
   py4 = py2 + hh2;                                                        //    final image segment
   
   ss_cr = gdk_cairo_create(gdkwin);
   
   for (nn = 0; nn < Nsteps; nn++)                                         //  loop reducing image segment
   {
      px3 = px2 * nn / Nsteps;                                             //  px3/py3 from 0/0 to px2/py2
      py3 = py2 * nn / Nsteps;
      px5 = ww1 - (ww1 - px4) * nn / Nsteps;                               //  px5/py5 from ww1/hh1 to px4/py4
      py5 = hh1 - (hh1 - py4) * nn / Nsteps;
      ww3 = px5 - px3;                                                     //  ww3/hh3 from ww1/hh1 to ww2/hh2
      hh3 = py5 - py3;
      
      pxb2 = gdk_pixbuf_new_subpixbuf(pxb1,px3,py3,ww3,hh3);               //  smaller image segment to display
      pxb3 = gdk_pixbuf_scale_simple(pxb2,ss_ww,ss_hh,BILINEAR);           //  scale up to window size
      
      gdk_cairo_set_source_pixbuf(ss_cr,pxb3,0,0);                         //  paint
      cairo_paint(ss_cr);
      gdk_flush();
      
      if (nn + 1 < Nsteps) {                                               //  not done yet
         g_object_unref(pxb2);
         g_object_unref(pxb3);
         continue;
      }
      
      g_object_unref(ss_pxbnew);
      ss_pxbnew = pxb3;                                                    //  retain final zoomed image
      g_object_unref(pxb2);                                                //    for next transition
      g_object_unref(pxb1);
      break;
   }

   cairo_destroy(ss_cr);
   return;
}


/**************************************************************************/

//  Batch file rename, convert, resize, move

namespace batch_convert                                                    //  overhauled v.14.02
{
   char     **filelist;
   int      Fsametype, Fsamesize, maxww, maxhh;
   int      Fdelete, Fcopymeta, Fupright, Fsharpen;
   char     newloc[400], newname[200], newext[8];
   int      filecount, baseseq, addseq;
   int      amount, thresh;
};


void m_batch_convert(GtkWidget *, cchar *)
{
   using namespace batch_convert;

   int  batch_convert_dialog_event(zdialog *zd, cchar *event);
   int  batch_sharp_func(PXM *pxm, int amount, int thresh);

   zdialog        *zd;
   int            zstat;
   char           *save_curr_file = 0;
   char           *infile, *outfile, *tempfile;
   char           *inloc, *inname, *inext;
   char           *outloc, *outname, *outext;
   char           seqnum[20];
   char           *pp1, *pp2, **ppv;
   int            ii, jj, cc, err;
   int            ww, hh, bpc, delinput;
   char           orientation[4] = "1";
   float          scale, wscale, hscale;
   PXM            *pxmin, *pxmout;
   struct stat    statdat;
   cchar          *exifkey[1] = { exif_orientation_key };
   cchar          *exifdata[1];

   F1_help_topic = "batch_convert";

   if (checkpend("all")) return;                                           //  check nothing pending              v.13.12
   Fmenulock = 1;

/***
       ____________________________________________________________________
      |                    Batch Convert                                   |
      |                                                                    |
      |  [Select Files]  N files selected                                  |
      |                                                                    |
      |  New Name [____________________________]  base [____]  adder [__]  |
      |  New Location [________________________________________] [browse]  |
      |  New File Type  (o) JPG  (o) PNG  (o) TIF  (o) no change           |
      |  Max. Width [____]  Height [____]  [x] no change                   |
      |  [x] Delete Originals  [x] Copy Metadata  [x] Upright              |
      |  [x] Sharpen   amount [___]  threshold [___]                       |
      |                                                                    |
      |                                               [proceed] [cancel]   |
      |____________________________________________________________________|
       
***/

   zd = zdialog_new(ZTX("Batch Convert"),Mwin,Bproceed,Bcancel,null);

   zdialog_add_widget(zd,"hbox","hbf","dialog",0,"space=5");
   zdialog_add_widget(zd,"button","files","hbf",Bselectfiles,"space=5");
   zdialog_add_widget(zd,"label","fcount","hbf",Bnofileselected,"space=10");

   zdialog_add_widget(zd,"hbox","hbname","dialog");
   zdialog_add_widget(zd,"label","labname","hbname",ZTX("New Name"),"space=5");
   zdialog_add_widget(zd,"entry","newname","hbname",0,"expand");
   zdialog_add_widget(zd,"label","labbase","hbname",ZTX("base"),"space=5");
   zdialog_add_widget(zd,"entry","baseseq","hbname",0,"scc=5");
   zdialog_add_widget(zd,"label","labadder","hbname",ZTX("adder"),"space=5");
   zdialog_add_widget(zd,"entry","addseq","hbname",0,"scc=3");

   zdialog_add_widget(zd,"hbox","hbloc","dialog",0,"expand");
   zdialog_add_widget(zd,"label","labloc","hbloc",ZTX("New Location"),"space=5");
   zdialog_add_widget(zd,"entry","newloc","hbloc",0,"expand");
   zdialog_add_widget(zd,"button","browse","hbloc",Bbrowse,"space=5");

   zdialog_add_widget(zd,"hbox","hbtyp","dialog");
   zdialog_add_widget(zd,"label","labtyp","hbtyp",ZTX("New File Type"),"space=5");
   zdialog_add_widget(zd,"radio","jpg","hbtyp","JPG","space=6");
   zdialog_add_widget(zd,"radio","png","hbtyp","PNG","space=6");
   zdialog_add_widget(zd,"radio","tif","hbtyp","TIF","space=6");
   zdialog_add_widget(zd,"radio","sametype","hbtyp",ZTX("no change"),"space=6");

   zdialog_add_widget(zd,"hbox","hbwh","dialog");
   zdialog_add_widget(zd,"label","labw","hbwh",ZTX("max. Width"),"space=5");
   zdialog_add_widget(zd,"entry","maxww","hbwh","1000","scc=5");
   zdialog_add_widget(zd,"label","space","hbwh",0,"space=5");
   zdialog_add_widget(zd,"label","labh","hbwh",ZTX("Height"),"space=5");
   zdialog_add_widget(zd,"entry","maxhh","hbwh","700","scc=5");
   zdialog_add_widget(zd,"check","samesize","hbwh",ZTX("no change"),"space=12");

   zdialog_add_widget(zd,"hbox","hbopts","dialog");
   zdialog_add_widget(zd,"check","delete","hbopts",ZTX("Delete Originals"),"space=3");
   zdialog_add_widget(zd,"check","copymeta","hbopts",ZTX("Copy Metadata"),"space=3");
   zdialog_add_widget(zd,"check","upright","hbopts",ZTX("Upright"),"space=3");

   zdialog_add_widget(zd,"hbox","hbsharp","dialog");
   zdialog_add_widget(zd,"check","sharpen","hbsharp",ZTX("Sharpen"),"space=3");
   zdialog_add_widget(zd,"label","space","hbsharp",0,"space=8");
   zdialog_add_widget(zd,"label","labamount","hbsharp",ZTX("amount"),"space=3");
   zdialog_add_widget(zd,"spin","amount","hbsharp","0|400|1|100");
   zdialog_add_widget(zd,"label","space","hbsharp",0,"space=8");
   zdialog_add_widget(zd,"label","labthresh","hbsharp",ZTX("threshold"),"space=3");
   zdialog_add_widget(zd,"spin","thresh","hbsharp","0|100|1|20");

   zdialog_stuff(zd,"sametype",1);                                         //  same file type
   zdialog_stuff(zd,"samesize",1);                                         //  same size
   zdialog_stuff(zd,"delete",0);                                           //  delete originals - no
   zdialog_stuff(zd,"copymeta",0);                                         //  copy metadata - no
   zdialog_stuff(zd,"upright",1);                                          //  upright rotation - yes
   zdialog_stuff(zd,"sharpen",0);                                          //  sharpen - no
   zdialog_stuff(zd,"amount",100);
   zdialog_stuff(zd,"thresh",20);
   
   filelist = 0;
   filecount = 0;
   *newloc = 0;

   zdialog_restore_inputs(zd);                                             //  preload prior user inputs
   zdialog_run(zd,batch_convert_dialog_event,"parent");                    //  run dialog
   zstat = zdialog_wait(zd);                                               //  wait for completion
   zdialog_free(zd);
   if (zstat != 1) goto cleanup;                                           //  canceled
   if (! filecount) goto cleanup;

   if (curr_file)                                                          //  v.14.02
      save_curr_file = zstrdup(curr_file);
   free_resources();

   m_viewmode(0,"F");
   gallery_monitor("stop");                                                //  stop tracking gallery updates

   write_popup_text("open","Processing files",500,200,Mwin);               //  status monitor popup window
   
   for (ii = 0; ii < filecount; ii++)                                      //  loop selected files
   {
      infile = filelist[ii];                                               //  input file

      parsefile(infile,&inloc,&inname,&inext);                             //  parse directory, filename, .ext
      if (! inloc || ! inname || ! inext) continue;

      outloc = zstrdup(inloc);                                             //  initial output = input file
      outname = zstrdup(inname);
      outext = (char *) zmalloc(8);

      cc = strlen(outloc) - 1;                                             //  remove trailing '/'
      if (outloc[cc] == '/') outloc[cc] = 0;

      if (*newname)                                                        //  new file name was given
      {
         pp1 = strchr(newname,'#');                                        //  .....#####.....
         pp2 = pp1 + 1;                                                    //       |    |
         while (*pp2 == '#') pp2++;                                        //       pp1  pp2
         cc = pp2 - pp1;
         jj = baseseq + ii * addseq;                                       //  new sequence number nnnnnn
         sprintf(seqnum,"%0*d",cc,jj);
         zfree(outname);
         outname = zstrdup(newname,8);                                     //  .....#####.....
         pp1 = strchr(outname,'#');
         strcpy(pp1,seqnum);
         strcat(pp1,pp2);                                                  //  .....nnnnn.....
      }
      
      if (*newloc) {                                                       //  new location was given
         zfree(outloc);
         outloc = zstrdup(newloc);
      }
      
      if (Fsametype) {
         if (strcasestr(".jpg .jpeg",inext)) strcpy(outext,".jpg");        //  new .ext from existing .ext
         else if (strcasestr(".png",inext)) strcpy(outext,".png");
         else if (strcasestr(".tif .tiff",inext)) strcpy(outext,".tif");
         else strcpy(outext,".jpg");                                       //  unknown >> .jpg
      }
      else strcpy(outext,newext);                                          //  new .ext was given

      cc = strlen(outloc) + strlen(outname) + strlen(outext) + 4;
      outfile = (char *) zmalloc(cc);
      sprintf(outfile,"%s/%s%s",outloc,outname,outext);
      
      zfree(outloc);
      zfree(outname);
      zfree(outext);

      write_popup_text("write",outfile);                                   //  log each output file
      zmainloop();

      if (*newloc) {
         err = stat(outfile,&statdat);                                     //  check if file exists in new location
         if (! err) {
            write_popup_text("write",ZTX("file already exists"));
            zfree(outfile);
            continue;
         }
      }

      pxmin = PXM_load(infile,0);                                          //  read input file
      if (! pxmin) {
         write_popup_text("write",ZTX("file type not supported"));
         zfree(outfile);
         continue;
      }

      if (Fupright)                                                        //  upright image if turned
      {
         ppv = exif_get(infile, exifkey, 1);                               //  get EXIF: Orientation
         if (ppv[0]) {
            orientation[0] = *ppv[0];                                      //  single character
            zfree(ppv[0]);
         }
         else orientation[0] = '1';                                        //  if missing assume unrotated

         pxmout = 0;
         if (orientation[0] == '6')                                        //  rotate clockwise 90 deg.
            pxmout = PXM_rotate(pxmin,+90);
         if (orientation[0] == '8')
            pxmout = PXM_rotate(pxmin,-90);                                //  counterclockwise
         if (pxmout) {
            PXM_free(pxmin);                                               //  input image unrotated
            pxmin = pxmout;
         }
      }
      
      bpc = f_load_bpc;                                                    //  input file bits/color
      ww = pxmin->ww;                                                      //  input file size
      hh = pxmin->hh;

      if (Fsamesize) pxmout = pxmin;                                       //  same size, output = input
      else {
         wscale = hscale = 1.0;
         if (ww > maxww) wscale = 1.0 * maxww / ww;                        //  compute new size
         if (hh > maxhh) hscale = 1.0 * maxhh / hh;
         if (wscale < hscale) scale = wscale;
         else scale = hscale;
         if (scale > 0.999) pxmout = pxmin;                                //  no change
         else {
            ww = ww * scale;
            hh = hh * scale;
            pxmout = PXM_rescale(pxmin,ww,hh);                             //  rescaled output file
            PXM_free(pxmin);                                               //  free memory
         }
      }

      if (Fsharpen)                                                        //  auto sharpen output image          v.14.06
         batch_sharp_func(pxmout,amount,thresh);
      
      tempfile = zstrdup(infile,12);                                       //  temp file needed for EXIF/IPTC copy
      pp1 = strrchr(outfile,'.');
      pp2 = strrchr(tempfile,'.');
      strcpy(pp2,"-temp");
      strcpy(pp2+5,pp1);

      if (strEqu(".tif",pp1))                                              //  write output file to temp file
         err = PXM_TIFF_save(pxmout,tempfile,bpc);
      else if (strEqu(".png",pp1))
         err = PXM_PNG_save(pxmout,tempfile,bpc);
      else
         err = PXM_ANY_save(pxmout,tempfile);

      if (err) {
         write_popup_text("write",ZTX("cannot create new file"));
         zfree(outfile);
         zfree(tempfile);
         PXM_free(pxmout);
         continue;
      }

      if (Fcopymeta)                                                       //  copy EXIF/IPTC if requested
      {
         if (Fupright) {                                                   //  if image possibly uprighted
            exifdata[0] = "";                                              //    remove exif:orientation
            exif_copy(infile,tempfile,exifkey,exifdata,1);                 //  (set = 1 does not work) 
         }
         else exif_copy(infile,tempfile,0,0,0);                            //  else no EXIF change
      }

      err = shell_ack("cp -p \"%s\" \"%s\" ",tempfile,outfile);            //  copy tempfile to output file
      if (err) write_popup_text("write",wstrerror(err));
      remove(tempfile);                                                    //  remove tempfile

      delinput = 0;                                                        //  figure out if input file can be deleted
      if (Fdelete) delinput = 1;                                           //  user says yes
      if (err) delinput = 0;                                               //  not if error
      if (strEqu(infile,outfile)) delinput = 0;                            //  not if overwritten by output
      
      if (delinput) {                                                      //  delete input file
         remove(infile);
         delete_image_index(infile);                                       //  remove from image index
      }

      if (! err) {
         load_filemeta(outfile);                                           //  update image index for output file
         update_image_index(outfile);
      }

      zfree(outfile);
      zfree(tempfile);
      PXM_free(pxmout);
   }

   write_popup_text("write","COMPLETED");
   Fmenulock = 0;

   if (save_curr_file) {                                                   //  re-open curr. file
      err = f_open(save_curr_file,0,0,0);                                  //    (may be gone or changed)
      if (err) {
         free_resources();
         gallery(curr_dirk,"init");                                        //  update gallery file list
      }
      else gallery(curr_file,"init");
      zfree(save_curr_file);
   }

   gallery_monitor("start");                                               //  resume tracking gallery updates

cleanup:

   if (filecount) {                                                        //  free memory
      for (ii = 0; ii < filecount; ii++)
         zfree(filelist[ii]);
      zfree(filelist);
      filelist = 0;
      filecount = 0;
   }

   Fmenulock = 0;
   return;
}


//  dialog event and completion callback function

int batch_convert_dialog_event(zdialog *zd, cchar *event)
{
   using namespace batch_convert;

   char         countmess[50];
   char         *ploc;
   int          ii, cc, yn, err;
   
   if (strEqu(event,"files"))                                              //  select images to convert
   {
      if (filelist) {                                                      //  free prior list
         for (ii = 0; ii < filecount; ii++)
            zfree(filelist[ii]);
         zfree(filelist);
         filelist = 0;
         filecount = 0;
      }

      zdialog_show(zd,0);                                                  //  hide parent dialog
      filelist = gallery_getfiles();                                       //  get list of files to convert
      zdialog_show(zd,1);

      ii = 0;
      if (filelist)                                                        //  count files selected
         for (ii = 0; filelist[ii]; ii++);
      filecount = ii;

      snprintf(countmess,50,ZTX("%d files selected"),ii);                  //  update dialog
      zdialog_stuff(zd,"fcount",countmess);
   }
   
   if (strEqu(event,"browse")) {
      ploc = zgetfile(ZTX("Select directory"),"folder",curr_dirk);         //  new location browse
      if (! ploc) return 1;
      zdialog_stuff(zd,"newloc",ploc);
      zfree(ploc);
   }
   
   if (strstr("maxhh maxww",event))                                        //  if max width/height changed,
      zdialog_stuff(zd,"samesize",0);                                      //    reset "no change"
   
   if (strEqu(event,"samesize")) {                                         //  if "no change" set,
      zdialog_fetch(zd,"samesize",Fsamesize);                              //    clear max width / height
      if (Fsamesize) {
         zdialog_stuff(zd,"maxww","");
         zdialog_stuff(zd,"maxhh","");
      }
   }

   if (zd->zstat != 1) return 1;                                           //  wait for [proceed]
   
   zd->zstat = 0;                                                          //  dialog active until inputs OK

   zdialog_fetch(zd,"newname",newname,100);                                //  new file name
   zdialog_fetch(zd,"baseseq",baseseq);                                    //  base sequence number
   zdialog_fetch(zd,"addseq",addseq);                                      //  sequence number adder
   zdialog_fetch(zd,"newloc",newloc,400);                                  //  new location (directory)
   zdialog_fetch(zd,"maxww",maxww);                                        //  new max width
   zdialog_fetch(zd,"maxhh",maxhh);                                        //  new max height
   zdialog_fetch(zd,"samesize",Fsamesize);                                 //  keep same width/height
   zdialog_fetch(zd,"sametype",Fsametype);                                 //  keep same file type
   zdialog_fetch(zd,"delete",Fdelete);                                     //  delete originals
   zdialog_fetch(zd,"copymeta",Fcopymeta);                                 //  copy metadata
   zdialog_fetch(zd,"upright",Fupright);                                   //  upright rotation
   zdialog_fetch(zd,"sharpen",Fsharpen);                                   //  auto sharpen                       v.14.02
   zdialog_fetch(zd,"amount",amount);                                      //  sharpen amount                     v.14.06
   zdialog_fetch(zd,"thresh",thresh);                                      //  sharpen threshold

   if (! filecount) {
      zmessageACK(Mwin,0,Bnofileselected);
      return 1;
   }

   strTrim2(newname);                                                      //  check new name, baseseq, addseq
   if (! blank_null(newname)) {
      err = 0;
      if (strlen(newname) < 2 || baseseq < 1 || addseq < 1) err++;
      if (! strchr(newname,'#')) err++;
      if (err) {
         zmessageACK(Mwin,0,ZTX("new name/base/adder unreasonable"
                                "\n e.g. newname ###   100  10"));
         return 1;
      }
   }
   
   strTrim2(newloc);                                                       //  check location
   if (! blank_null(newloc)) {
      cc = strlen(newloc) - 1;
      if (newloc[cc] == '/') newloc[cc] = 0;                               //  remove trailing '/'
      err = check_create_dir(newloc);                                      //  create if needed                   v.14.03
      if (err) return 1;
   }
   
   if (! Fsametype) {                                                      //  file type change
      strcpy(newext,".jpg");                                               //  get new .ext
      zdialog_fetch(zd,"png",ii);
      if (ii) strcpy(newext,".png");
      zdialog_fetch(zd,"tif",ii);
      if (ii) strcpy(newext,".tif");
   }

   if (! Fsamesize && (maxww < 20 || maxhh < 20)) {
      zmessageACK(Mwin,0,ZTX("max. size %d x %d is not reasonable"),maxww,maxhh);
      return 1;
   }
   
   /**   Convert NN image files                    0
           Rename to xxxxxxx                       1
           Convert to .ext                         2
           Resize within NNxNN                     3
           Output to /.../...                      4
           Copy Metadata  Upright  Sharpen         5
           Delete Originals                        6
         PROCEED?                                  7
   **/

   char  mess0[60], mess1[100], mess2[60], mess3[60], mess4[200], mess5[80], mess6[60], mess7[40];
   char  warnmess[600];

   *mess0 = *mess1 = *mess2 = *mess3 = *mess4 = *mess5 = *mess6 = *mess7 = 0;

   snprintf(mess0,60,ZTX("Convert %d image files"),filecount);
   if (*newname) snprintf(mess1,100,"\n  %s %s",ZTX("Rename to"),newname);
   if (! Fsametype) snprintf(mess2,60,"\n  %s %s",ZTX("Convert to"),newext);
   if (! Fsamesize) snprintf(mess3,60,"\n  %s %dx%d",ZTX("Resize within"),maxww,maxhh);
   if (*newloc) snprintf(mess4,200,"\n  %s  %s",ZTX("Output to"),newloc);
   if (Fcopymeta || Fupright || Fsharpen) strcat(mess5,"\n  ");
   if (Fcopymeta) { strcat(mess5,ZTX("Copy Metadata")); strcat(mess5,"  "); }
   if (Fupright) { strcat(mess5,ZTX("Upright")); strcat(mess5,"  "); }
   if (Fsharpen) { strcat(mess5,ZTX("Sharpen")); strcat(mess5,"  "); }
   if (Fdelete) snprintf(mess6,60,"\n  %s",ZTX("Delete Originals"));
   snprintf(mess7,40,"\n\n%s",ZTX("PROCEED?"));

   snprintf(warnmess,600,"%s %s %s %s %s %s %s %s",mess0,mess1,mess2,mess3,mess4,mess5,mess6,mess7);

   yn = zmessageYN(Mwin,warnmess);
   if (! yn) return 1;

   zd->zstat = 1;                                                          //  [proceed]
   return 1;
}


/**************************************************************************/

//  Batch upright image files.
//  Look for files rotated 90˚ (according to EXIF) and upright them.

char     **bup_filelist;
int      bup_filecount;

void m_batch_upright(GtkWidget *, cchar *)
{
   int  batch_upright_dialog_event(zdialog *zd, cchar *event);

   zdialog        *zd;
   int            zstat;
   char           *save_curr_file = 0;
   char           *infile, *tempfile;
   char           *pp1, *pp2, **ppv;
   int            ii, err, bpc;
   char           orientation;
   PXM            *pxmin, *pxmout;
   cchar          *exifkey[1] = { exif_orientation_key };
   cchar          *exifdata[1];

   F1_help_topic = "batch_upright";

   if (checkpend("all")) return;                                           //  check nothing pending              v.13.12
   Fmenulock = 1;

/***
       ___________________________________
      |       Batch Upright               |
      |                                   |
      |  [Select Files]  N files selected |
      |                                   |
      |               [proceed] [cancel]  |
      |___________________________________|
       
***/

   zd = zdialog_new(ZTX("Batch Upright"),Mwin,Bproceed,Bcancel,null);

   zdialog_add_widget(zd,"hbox","hbf","dialog",0,"space=5");
   zdialog_add_widget(zd,"button","files","hbf",Bselectfiles,"space=5");
   zdialog_add_widget(zd,"label","fcount","hbf",Bnofileselected,"space=10");

   bup_filelist = 0;
   bup_filecount = 0;

   zdialog_run(zd,batch_upright_dialog_event,"parent");                    //  run dialog
   zstat = zdialog_wait(zd);                                               //  wait for completion
   zdialog_free(zd);
   if (zstat != 1) goto cleanup;                                           //  canceled
   if (! bup_filecount) goto cleanup;

   if (curr_file)                                                          //  v.14.02
      save_curr_file = zstrdup(curr_file);
   free_resources();

   m_viewmode(0,"F");
   gallery_monitor("stop");                                                //  stop tracking gallery updates

   write_popup_text("open","Processing files",500,200,Mwin);               //  status monitor popup window
   
   for (ii = 0; ii < bup_filecount; ii++)                                      //  loop selected files
   {
      infile = bup_filelist[ii];                                           //  input file
      write_popup_text("write",infile);                                    //  log each output file
      zmainloop();

      pxmin = PXM_load(infile,0);                                          //  read input file
      if (! pxmin) {
         write_popup_text("write",ZTX("file type not supported"));
         continue;
      }

      ppv = exif_get(infile,exifkey,1);                                    //  get EXIF: Orientation
      if (! ppv[0]) {
         PXM_free(pxmin);                                                  //  none, assume not rotated
         continue;
      }

      orientation = *ppv[0];                                               //  single character

      pxmout = 0;
      if (orientation == '6')                                              //  rotate clockwise 90 deg.
         pxmout = PXM_rotate(pxmin,+90);
      else if (orientation == '8')
         pxmout = PXM_rotate(pxmin,-90);                                   //  counterclockwise

      PXM_free(pxmin);
      if (! pxmout) continue;                                              //  not rotated

      tempfile = zstrdup(infile,12);                                       //  temp file needed for EXIF/IPTC copy
      pp1 = strrchr(infile,'.');
      pp2 = strrchr(tempfile,'.');
      strcpy(pp2,"-temp");
      strcpy(pp2+5,pp1);

      bpc = f_load_bpc;                                                    //  input file bits/color
      
      if (strEqu(f_load_type,"tif"))
         err = PXM_TIFF_save(pxmout,tempfile,bpc);
      else if (strEqu(f_load_type,"png"))
         err = PXM_PNG_save(pxmout,tempfile,bpc);
      else
         err = PXM_ANY_save(pxmout,tempfile);

      PXM_free(pxmout);

      if (err) {
         write_popup_text("write","*** upright failed");
         zfree(tempfile);
         continue;
      }

      exifdata[0] = "";                                                    //  remove exif:orientation
      exif_copy(infile,tempfile,exifkey,exifdata,1);                       //  (set = 1 does not work)

      err = shell_ack("cp -p \"%s\" \"%s\" ",tempfile,infile);             //  copy temp file to input file
      if (err) write_popup_text("write",wstrerror(err));
      remove(tempfile);                                                    //  remove tempfile
      zfree(tempfile);

      if (! err) {
         load_filemeta(infile);                                            //  update image index
         update_image_index(infile);
         write_popup_text("write","  uprighted");
      }
   }

   write_popup_text("write","COMPLETED");
   Fmenulock = 0;

   if (save_curr_file) {                                                   //  re-open curr. file
      err = f_open(save_curr_file,0,0,0);                                  //    (may be gone or changed)
      if (err) {
         free_resources();
         gallery(curr_dirk,"init");                                        //  update gallery file list
      }
      else gallery(curr_file,"init");
      zfree(save_curr_file);
   }

   gallery_monitor("start");                                               //  resume tracking gallery updates

cleanup:

   if (bup_filecount) {                                                    //  free memory
      for (ii = 0; ii < bup_filecount; ii++)
         zfree(bup_filelist[ii]);
      zfree(bup_filelist);
      bup_filelist = 0;
      bup_filecount = 0;
   }

   Fmenulock = 0;
   return;
}


//  dialog event and completion callback function

int batch_upright_dialog_event(zdialog *zd, cchar *event)
{
   char         countmess[50];
   int          ii;
   
   if (strEqu(event,"files"))                                              //  select images to convert
   {
      if (bup_filelist) {                                                  //  free prior list
         for (ii = 0; ii < bup_filecount; ii++)
            zfree(bup_filelist[ii]);
         zfree(bup_filelist);
         bup_filelist = 0;
         bup_filecount = 0;
      }

      zdialog_show(zd,0);                                                  //  hide parent dialog
      bup_filelist = gallery_getfiles();                                   //  get list of files to convert
      zdialog_show(zd,1);

      ii = 0;
      if (bup_filelist)                                                    //  count files selected
         for (ii = 0; bup_filelist[ii]; ii++);
      bup_filecount = ii;

      snprintf(countmess,50,ZTX("%d files selected"),ii);                  //  update dialog
      zdialog_stuff(zd,"fcount",countmess);
   }
   
   if (zd->zstat != 1) return 1;                                           //  wait for [proceed]
   
   if (! bup_filecount) {                                                  //  nothing selected
      zmessageACK(Mwin,0,Bnofileselected);
      zd->zstat = 0;                                                       //  keep dialog active
      return 1;
   }

   return 1;
}


/**************************************************************************/

//  convert multiple RAW files to tiff, jpeg, or png using DCraw

namespace batch_dcraw
{
   char     location[400];
   char     **filelist;
   int      filecount;
   char     dcraw_command[100] = "";
   cchar    *filetype;
   int      bpc;
   float    resize;
   int      Fsharpen;
   int      amount, thresh;
};


void  m_batch_dcraw(GtkWidget *, cchar *menu)
{
   using namespace batch_dcraw;

   int  batch_dcraw_dialog_event(zdialog *zd, cchar *event);

   zdialog     *zd;
   cchar       *title = ZTX("Batch Convert RAW Files (DCraw)");
   char        *tiffile, *infile, *outfile, *pp;
   int         zstat, ii, err, ftype;
   int         cc, ww2, hh2;
   PXM         *pxm1, *pxm2;

   F1_help_topic = "batch_convert_raw";

   if (! Fdcraw) {                                                         //  v.14.07
      zmessageACK(Mwin,0,ZTX("DCraw not installed"));
      return;
   }

   if (checkpend("all")) return;                                           //  check nothing pending              v.13.12
   Fmenulock = 1;
   

/***
       _____________________________________________________________________________
      | [x] [-] [_]       Batch Convert RAW Files (DCraw)                           |
      |                                                                             |
      | [Select Files]  N files selected                                            |
      | output location [_________________________________________________]         |
      | output file type  (o) JPEG  (o) PNG-8  (o) PNG-16  (o) TIFF-8  (o) TIFF-16  |
      | resize  (o) 1.0  (o) 3/4  (o) 2/3  (o) 1/2  (o) 1/3                         |
      | [x] Sharpen   amount [___]   threshold [___]                                |
      | --------------------------------------------------------------------------- |
      | white balance  (o) camera  (o) fixed  (o) calculated                        |
      | interpolation  (o) bilinear  (o) VNG  (o) PPG  (o) AHD                      |
      | color space    (o) raw  (o) sRGB  (o) Adobe  (o) wide  (o) Kodak  (o) XYZ   |
      | gamma curve    (o) default  (o) sRGB  (o) linear                            |
      | [ dcraw -T -6 -w -q 0 -o 1 -g 2.222 4.5       ]   [defaults]                |
      |                                                                             |
      |                                                         [proceed] [cancel]  |
      |_____________________________________________________________________________|
      
***/

   zd = zdialog_new(title,Mwin,Bproceed,Bcancel,null);

   zdialog_add_widget(zd,"hbox","hb1","dialog",0,"space=3");
   zdialog_add_widget(zd,"button","files","hb1",Bselectfiles,"space=5");
   zdialog_add_widget(zd,"label","fcount","hb1",Bnofileselected,"space=10");

   zdialog_add_widget(zd,"hbox","hbout","dialog");
   zdialog_add_widget(zd,"label","labout","hbout",ZTX("output location"),"space=5");
   zdialog_add_widget(zd,"entry","location","hbout",0,"space=5|expand");
   zdialog_add_widget(zd,"button","browse","hbout",Bbrowse,"space=5");

   zdialog_add_widget(zd,"hbox","hb2","dialog");
   zdialog_add_widget(zd,"label","labtype","hb2",ZTX("output file type"),"space=5");
   zdialog_add_widget(zd,"radio","JPEG","hb2","JPEG","space=5");
   zdialog_add_widget(zd,"radio","PNG-8","hb2","PNG-8","space=5");
   zdialog_add_widget(zd,"radio","PNG-16","hb2","PNG-16","space=5");
   zdialog_add_widget(zd,"radio","TIFF-8","hb2","TIFF-8","space=5");
   zdialog_add_widget(zd,"radio","TIFF-16","hb2","TIFF-16","space=5");

   zdialog_add_widget(zd,"hbox","hbsize","dialog");                                 //  v.14.06
   zdialog_add_widget(zd,"label","labsize","hbsize",ZTX("resize"),"space=5");
   zdialog_add_widget(zd,"label","space","hbsize",0,"space=5");
   zdialog_add_widget(zd,"radio","1.0","hbsize","1.0","space=5");
   zdialog_add_widget(zd,"radio","3/4","hbsize","3/4","space=5");
   zdialog_add_widget(zd,"radio","2/3","hbsize","2/3","space=5");
   zdialog_add_widget(zd,"radio","1/2","hbsize","1/2","space=5");
   zdialog_add_widget(zd,"radio","1/3","hbsize","1/3","space=5");
   
   zdialog_add_widget(zd,"hbox","hbsharp","dialog");                                //  v.14.06
   zdialog_add_widget(zd,"check","sharpen","hbsharp",ZTX("Sharpen"),"space=3");
   zdialog_add_widget(zd,"label","space","hbsharp",0,"space=8");
   zdialog_add_widget(zd,"label","labamount","hbsharp",ZTX("amount"),"space=3");
   zdialog_add_widget(zd,"spin","amount","hbsharp","0|400|1|100");
   zdialog_add_widget(zd,"label","space","hbsharp",0,"space=8");
   zdialog_add_widget(zd,"label","labthresh","hbsharp",ZTX("threshold"),"space=3");
   zdialog_add_widget(zd,"spin","thresh","hbsharp","0|100|1|20");

   zdialog_add_widget(zd,"hsep","hsep1","dialog",0,"space=5");

   zdialog_add_widget(zd,"hbox","hbvb","dialog");
   zdialog_add_widget(zd,"vbox","vb1","hbvb",0,"space=5|homog");
   zdialog_add_widget(zd,"vbox","vb2","hbvb",0,"homog");

   zdialog_add_widget(zd,"label","labwbal","vb1",ZTX("white balance"));
   zdialog_add_widget(zd,"label","labinterp","vb1",ZTX("interpolation"));
   zdialog_add_widget(zd,"label","labcolor","vb1",ZTX("color space"));
   zdialog_add_widget(zd,"label","labgamma","vb1",ZTX("gamma curve"));

   zdialog_add_widget(zd,"hbox","hb3","vb2");
   zdialog_add_widget(zd,"radio","-w","hb3",ZTX("camera"),"space=3");
   zdialog_add_widget(zd,"radio","-W","hb3",ZTX("fixed"),"space=3");
   zdialog_add_widget(zd,"radio","-a","hb3",ZTX("calculated"),"space=3");

   zdialog_add_widget(zd,"hbox","hb4","vb2");
   zdialog_add_widget(zd,"radio","-q 0","hb4","bilinear","space=3");
   zdialog_add_widget(zd,"radio","-q 1","hb4","VNG","space=3");
   zdialog_add_widget(zd,"radio","-q 2","hb4","PPG","space=3");
   zdialog_add_widget(zd,"radio","-q 3","hb4","AHD","space=3");

   zdialog_add_widget(zd,"hbox","hb5","vb2");
   zdialog_add_widget(zd,"radio","-o 0","hb5","raw","space=3");
   zdialog_add_widget(zd,"radio","-o 1","hb5","sRGB","space=3");
   zdialog_add_widget(zd,"radio","-o 2","hb5","Adobe","space=3");
   zdialog_add_widget(zd,"radio","-o 3","hb5","Wide","space=3");
   zdialog_add_widget(zd,"radio","-o 4","hb5","Kodak","space=3");
   zdialog_add_widget(zd,"radio","-o 5","hb5","XYZ","space=3");

   zdialog_add_widget(zd,"hbox","hb6","vb2");
   zdialog_add_widget(zd,"radio","-g 2.222 4.5","hb6",ZTX("default"),"space=3");
   zdialog_add_widget(zd,"radio","-g 2.4 12.92","hb6","sRGB","space=3");
   zdialog_add_widget(zd,"radio","-g 1.0 1.0","hb6","linear","space=3");

   zdialog_add_widget(zd,"hbox","hb7","dialog",0,"space=3");
   zdialog_add_widget(zd,"entry","dcraw_command","hb7",0,"space=5|expand");
   zdialog_add_widget(zd,"button","defaults","hb7",ZTX("defaults"),"space=5");

   filelist = 0;                                                           //  set no files selected
   filecount = 0;
   *location = 0;

   zdialog_stuff(zd,"JPEG",1);                                             //  compensate GTK bug        v.13.07.1
   batch_dcraw_dialog_event(zd,"defaults");                                //  set defaults
   zdialog_restore_inputs(zd);                                             //  get prior inputs if any

   zdialog_run(zd,batch_dcraw_dialog_event,"save");                        //  run dialog
   zstat = zdialog_wait(zd);                                               //  wait for completion
   zdialog_free(zd);
   if (zstat != 1) goto cleanup;
   if (! filecount) goto cleanup;

   printz("dcraw command: %s \n",dcraw_command);

   m_viewmode(0,"F");
   gallery_monitor("stop");                                                //  stop tracking gallery updates

   write_popup_text("open","Converting RAW files",500,200,Mwin);           //  status monitor popup window
   zmainloop();

   rawfile = 0;

   for (ii = 0; ii < filecount; ii++)                                      //  loop all RAW files
   {
      rawfile = filelist[ii];                                              //  filename.raw

      write_popup_text("write",rawfile);                                   //  write input raw file to log window
      zmainloop();

      ftype = image_file_type(rawfile);
      if (ftype != 3) {
         write_popup_text("write"," *** unknown RAW file type");
         continue;
      }
      
      err = shell_ack("%s \"%s\" ",dcraw_command,rawfile);                 //  convert filename.raw to filename.tiff
      if (err) {
         write_popup_text("write"," *** raw conversion failed");           //  conversion failed
         write_popup_text("write",wstrerror(err));
         zmainloop();
         continue;
      }
      
      tiffile = raw_to_tiff(rawfile);                                      //  converted file

      pxm1 = PXM_load(tiffile,0);                                          //  load tiff-16 output file
      if (! pxm1) {
         write_popup_text("write"," *** no dcraw .tiff output");
         zmainloop();
         zfree(tiffile);
         continue;
      }

      if (resize < 1.0)                                                    //  resize down if wanted
      {
         ww2 = resize * pxm1->ww;
         hh2 = resize * pxm1->hh;
         pxm2 = PXM_rescale(pxm1,ww2,hh2);
         PXM_free(pxm1);
         pxm1 = pxm2;

         if (! pxm1) {
            write_popup_text("write"," *** resize failed");
            zmainloop();
            zfree(tiffile);
            continue;
         }
      }
      
      if (Fsharpen)                                                        //  sharpen if wanted
         batch_sharp_func(pxm1,amount,thresh);
      
      outfile = zstrdup(tiffile);                                          //  have filename.tiff, 16 bits/color

      pp = strrchr(outfile,'.');                                           //  make *.jpg or *.png if wanted
      if (pp) strcpy(pp+1,filetype);
      
      if (strnEqu(filetype,"tif",3))                                       //  output .tif file, 8 bits/color
         err = PXM_TIFF_save(pxm1,outfile,bpc);                            //    or .tiff file, 16 bits/color

      else if (strEqu(filetype,"png"))                                     //  output .png file, 8/16 bits
         err = PXM_PNG_save(pxm1,outfile,bpc);

      else err = PXM_ANY_save(pxm1,outfile);                               //  output .jpg file, 8 bits
      
      if (err) {
         write_popup_text("write"," *** file type conversion failed");
         zmainloop();
         zfree(tiffile);
         zfree(outfile);
         continue;
      }

      if (strNeq(outfile,tiffile))                                         //  if conversion was made,
         remove(tiffile);                                                  //    delete tiff-16 file

      exif_copy(rawfile,outfile,0,0,0);                                    //  copy metadata from RAW file     v.13.02
      
      if (*location && ! samedirk(location,outfile)) {
         infile = zstrdup(outfile);                                        //  copy to new location            v.14.07
         zfree(outfile);
         pp = strrchr(infile,'/');                                         //  /raw-location/filename.ext
         cc = strlen(pp);                                                  //               |
         outfile = zstrdup(location,cc+1);                                 //               pp
         strcat(outfile,pp);                                               //  /new-location/filename.ext
         err = shell_ack("cp -p \"%s\" \"%s\" ",infile,outfile);           //  copy to new location
         if (err) write_popup_text("write",wstrerror(err));
         remove(infile);                                                   //  remove tempfile
         zfree(infile);
      }

      load_filemeta(outfile);                                              //  update image file index
      update_image_index(outfile);

      f_open(outfile,0,0,0);                                               //  open converted file

      write_popup_text("write",outfile);                                   //  write output file to log window
      zfree(outfile);
      zfree(tiffile);
      zmainloop();
   }

   write_popup_text("write","COMPLETED");

   if (curr_file) gallery(curr_file,"init");                               //  update gallery file list
   gallery_monitor("start");                                               //  resume tracking gallery updates

cleanup:                                                                   //  clean up and return

   if (filecount) {
      for (ii = 0; ii < filecount; ii++)                                   //  free memory
         zfree(filelist[ii]);
      zfree(filelist);
      filelist = 0;
      filecount = 0;
   }

   rawfile = 0;
   Fmenulock = 0;
   return;
}


//  dialog event and completion callback function

int batch_dcraw_dialog_event(zdialog *zd, cchar *event)
{
   using namespace batch_dcraw;

   int            ii, Fupdate = 0, err, cc;
   char           countmess[50], *ploc;
   static cchar   *wbal = 0, *intp = 0, *color = 0, *gamma = 0;

   if (! wbal) {                                                           //  get current options if not already
      wbal = "-w";
      zdialog_fetch(zd,"-W",ii);
      if (ii) wbal = "-W";
      zdialog_fetch(zd,"-a",ii);
      if (ii) wbal = "-a";
      Fupdate++;
   }

   if (! intp) {
      intp = "-q 0";
      zdialog_fetch(zd,"-q 1",ii);
      if (ii) intp = "-q 1";
      zdialog_fetch(zd,"-q 2",ii);
      if (ii) intp = "-q 2";
      zdialog_fetch(zd,"-q 3",ii);
      if (ii) intp = "-q 3";
      Fupdate++;
   }

   if (! color) {
      color = "-o 1";
      zdialog_fetch(zd,"-o 0",ii);
      if (ii) color = "-o 0";
      zdialog_fetch(zd,"-o 2",ii);
      if (ii) color = "-o 2";
      zdialog_fetch(zd,"-o 3",ii);
      if (ii) color = "-o 3";
      zdialog_fetch(zd,"-o 4",ii);
      if (ii) color = "-o 4";
      zdialog_fetch(zd,"-o 5",ii);
      if (ii) color = "-o 5";
      Fupdate++;
   }

   if (! gamma) {
      gamma = "-g 2.222 4.5";
      zdialog_fetch(zd,"-g 2.4 12.92",ii);
      if (ii) gamma = "-g 2.4 12.92";
      zdialog_fetch(zd,"-g 1.0 1.0",ii);
      if (ii) gamma = "-g 1.0 1.0";
      Fupdate++;
   }

   if (strEqu(event,"files"))                                              //  select images to convert
   {
      if (filecount) {                                                     //  free prior list
         for (ii = 0; ii < filecount; ii++)
            zfree(filelist[ii]);
         zfree(filelist);
         filelist = 0;
         filecount = 0;
      }

      zdialog_show(zd,0);                                                  //  hide parent dialog
      filelist = gallery_getfiles();                                       //  get list of files to convert
      zdialog_show(zd,1);

      ii = 0;
      if (filelist)                                                        //  count files in list
         for (ii = 0; filelist[ii]; ii++);
      filecount = ii;

      snprintf(countmess,50,ZTX("%d files selected"),ii);                  //  stuff count into dialog
      zdialog_stuff(zd,"fcount",countmess);
   }

   if (strEqu(event,"browse")) {
      ploc = zgetfile(ZTX("Select directory"),"folder",curr_dirk);         //  new location browse
      if (! ploc) return 1;
      zdialog_stuff(zd,"location",ploc);
      zfree(ploc);
   }

   if (strEqu(event,"defaults"))                                           //  default all parameters
   {
      zdialog_stuff(zd,"JPEG",0);
      zdialog_stuff(zd,"PNG-8",0);
      zdialog_stuff(zd,"PNG-16",0);
      zdialog_stuff(zd,"TIFF-8",0);
      zdialog_stuff(zd,"TIFF-16",1);                                       //  output = TIFF-16

      zdialog_stuff(zd,"-w",1);                                            //  color balance = camera value
      zdialog_stuff(zd,"-W",0);
      zdialog_stuff(zd,"-a",0);
      wbal = "-w";

      zdialog_stuff(zd,"-q 0",1);                                          //  interpolation = bilinear
      zdialog_stuff(zd,"-q 1",0);
      zdialog_stuff(zd,"-q 2",0);
      zdialog_stuff(zd,"-q 3",0);
      intp = "-q 0";

      zdialog_stuff(zd,"-o 0",0);
      zdialog_stuff(zd,"-o 1",1);                                          //  color space = sRGB
      zdialog_stuff(zd,"-o 2",0);
      zdialog_stuff(zd,"-o 3",0);
      zdialog_stuff(zd,"-o 4",0);
      zdialog_stuff(zd,"-o 5",0);
      color = "-o 1";

      zdialog_stuff(zd,"-g 2.222 4.5",1);                                  //  gamma = default
      zdialog_stuff(zd,"-g 2.4 12.92",0);
      zdialog_stuff(zd,"-g 1.0 1.0",0);
      gamma = "-g 2.222 4.5";

      Fupdate++;
   }

   if (strstr("-w -W -a",event)) {                                         //  event is one of: -w -W -a
      zdialog_fetch(zd,event,ii);
      if (ii) {
         wbal = event;
         Fupdate++;
      }
   }

   if (strstr(event,"-q")) {                                               //  event is: -q N
      zdialog_fetch(zd,event,ii);
      if (ii) {
         intp = event;
         Fupdate++;
      }
   }

   if (strstr(event,"-o")) {                                               //  event is: -o N
      zdialog_fetch(zd,event,ii);
      if (ii) {
         color = event;
         Fupdate++;
      }
   }

   if (strstr(event,"-g")) {                                               //  event is: -g N1 N2
      zdialog_fetch(zd,event,ii);
      if (ii) {
         gamma = event;
         Fupdate++;
      }
   }

   if (Fupdate || ! *dcraw_command)                                        //  initialize or update dcraw command
   {
      sprintf(dcraw_command,"dcraw -T -6 %s %s %s %s",wbal,intp,color,gamma);
      zdialog_stuff(zd,"dcraw_command",dcraw_command);
   }

   if (zd->zstat != 1) return 0;                                           //  wait for [proceed]
   
   zd->zstat = 0;                                                          //  keep dialog active until inputs OK
   
   if (! filecount) {
      zmessageACK(Mwin,0,Bnofileselected);
      return 1;
   }
   
   zdialog_fetch(zd,"location",location,400);                              //  output location
   strTrim2(location);
   if (! blank_null(location)) {
      cc = strlen(location) - 1;
      if (location[cc] == '/') location[cc] = 0;                           //  remove trailing '/'
      err = check_create_dir(location);                                    //  create if needed
      if (err) return 1;
   }

   zdialog_fetch(zd,"JPEG",ii);                                            //  file type, color depth
   if (ii) { filetype = "jpg"; bpc = 8; }
   zdialog_fetch(zd,"PNG-8",ii);
   if (ii) { filetype = "png"; bpc = 8; }
   zdialog_fetch(zd,"PNG-16",ii);
   if (ii) { filetype = "png"; bpc = 16; }
   zdialog_fetch(zd,"TIFF-8",ii);
   if (ii) { filetype = "tif"; bpc = 8; }
   zdialog_fetch(zd,"TIFF-16",ii);
   if (ii) { filetype = "tiff"; bpc = 16; }
   
   resize = 1.0;
   zdialog_fetch(zd,"1.0",ii);                                             //  resize option                      v.14.06
   if (ii) resize = 1.0;
   zdialog_fetch(zd,"3/4",ii);
   if (ii) resize = 0.75;
   zdialog_fetch(zd,"2/3",ii);
   if (ii) resize = 0.666667;
   zdialog_fetch(zd,"1/2",ii);
   if (ii) resize = 0.50;
   zdialog_fetch(zd,"1/3",ii);
   if (ii) resize = 0.333333;
   
   zdialog_fetch(zd,"sharpen",Fsharpen);                                   //  sharpen option                     v.14.06
   zdialog_fetch(zd,"amount",amount);
   zdialog_fetch(zd,"thresh",thresh);

   zd->zstat = 1;                                                          //  dialog complete
   return 1;
}


/**************************************************************************/

//  convert multiple RAW files to tiff, jpeg, or png using Raw Therapee

namespace batch_rawtherapee
{
   char     location[400];
   char     **filelist;
   int      filecount;
   char     rawtherapee_command[100] = "";
   cchar    *filetype;
   int      bpc;
   float    resize;
   int      Fsharpen;
   int      amount, thresh;
};


void  m_batch_rawtherapee(GtkWidget *, cchar *menu)                        //  v.14.07
{
   using namespace batch_rawtherapee;

   int  batch_rawtherapee_dialog_event(zdialog *zd, cchar *event);

   zdialog     *zd;
   cchar       *title = ZTX("Batch Convert RAW Files (Raw Therapee)");
   char        *tiffile, *infile, *outfile, *pp;
   int         zstat, ii, err, ftype;
   int         cc, ww2, hh2;
   PXM         *pxm1, *pxm2;

   F1_help_topic = "batch_convert_raw";

   if (! Frawtherapee) {                                                   //  v.14.07
      zmessageACK(Mwin,0,ZTX("Raw Therapee not installed"));
      return;
   }

   if (checkpend("all")) return;                                           //  check nothing pending              v.13.12
   Fmenulock = 1;

/***
       _____________________________________________________________________________
      | [x] [-] [_]       Batch Convert RAW Files (Raw Therapee)                    |
      |                                                                             |
      | [Select Files]  N files selected                                            |
      | output location [_________________________________________________]         |
      | output file type  (o) JPEG  (o) PNG-8  (o) PNG-16  (o) TIFF-8  (o) TIFF-16  |
      | resize  (o) 1.0  (o) 3/4  (o) 2/3  (o) 1/2  (o) 1/3                         |
      | [x] Sharpen   amount [___]   threshold [___]                                |
      |                                                         [proceed] [cancel]  |
      |_____________________________________________________________________________|
      
***/

   zd = zdialog_new(title,Mwin,Bproceed,Bcancel,null);

   zdialog_add_widget(zd,"hbox","hb1","dialog",0,"space=3");
   zdialog_add_widget(zd,"button","files","hb1",Bselectfiles,"space=5");
   zdialog_add_widget(zd,"label","fcount","hb1",Bnofileselected,"space=10");

   zdialog_add_widget(zd,"hbox","hbout","dialog");
   zdialog_add_widget(zd,"label","labout","hbout",ZTX("output location"),"space=5");
   zdialog_add_widget(zd,"entry","location","hbout",0,"space=5|expand");
   zdialog_add_widget(zd,"button","browse","hbout",Bbrowse,"space=5");

   zdialog_add_widget(zd,"hbox","hb2","dialog");
   zdialog_add_widget(zd,"label","labtype","hb2",ZTX("output file type"),"space=5");
   zdialog_add_widget(zd,"radio","JPEG","hb2","JPEG","space=5");
   zdialog_add_widget(zd,"radio","PNG-8","hb2","PNG-8","space=5");
   zdialog_add_widget(zd,"radio","PNG-16","hb2","PNG-16","space=5");
   zdialog_add_widget(zd,"radio","TIFF-8","hb2","TIFF-8","space=5");
   zdialog_add_widget(zd,"radio","TIFF-16","hb2","TIFF-16","space=5");

   zdialog_add_widget(zd,"hbox","hbsize","dialog");                                 //  v.14.06
   zdialog_add_widget(zd,"label","labsize","hbsize",ZTX("resize"),"space=5");
   zdialog_add_widget(zd,"label","space","hbsize",0,"space=5");
   zdialog_add_widget(zd,"radio","1.0","hbsize","1.0","space=5");
   zdialog_add_widget(zd,"radio","3/4","hbsize","3/4","space=5");
   zdialog_add_widget(zd,"radio","2/3","hbsize","2/3","space=5");
   zdialog_add_widget(zd,"radio","1/2","hbsize","1/2","space=5");
   zdialog_add_widget(zd,"radio","1/3","hbsize","1/3","space=5");
   
   zdialog_add_widget(zd,"hbox","hbsharp","dialog");                                //  v.14.06
   zdialog_add_widget(zd,"check","sharpen","hbsharp",ZTX("Sharpen"),"space=3");
   zdialog_add_widget(zd,"label","space","hbsharp",0,"space=8");
   zdialog_add_widget(zd,"label","labamount","hbsharp",ZTX("amount"),"space=3");
   zdialog_add_widget(zd,"spin","amount","hbsharp","0|400|1|100");
   zdialog_add_widget(zd,"label","space","hbsharp",0,"space=8");
   zdialog_add_widget(zd,"label","labthresh","hbsharp",ZTX("threshold"),"space=3");
   zdialog_add_widget(zd,"spin","thresh","hbsharp","0|100|1|20");

   filelist = 0;                                                           //  set no files selected
   filecount = 0;
   *location = 0;

   zdialog_stuff(zd,"JPEG",1);                                             //  compensate GTK bug        v.13.07.1
   zdialog_restore_inputs(zd);                                             //  get prior inputs if any

   zdialog_run(zd,batch_rawtherapee_dialog_event,"save");                  //  run dialog
   zstat = zdialog_wait(zd);                                               //  wait for completion
   zdialog_free(zd);
   if (zstat != 1) goto cleanup;
   if (! filecount) goto cleanup;

   m_viewmode(0,"F");
   gallery_monitor("stop");                                                //  stop tracking gallery updates

   write_popup_text("open","Converting RAW files",500,200,Mwin);           //  status monitor popup window
   zmainloop();

   rawfile = 0;

   for (ii = 0; ii < filecount; ii++)                                      //  loop all RAW files
   {
      rawfile = filelist[ii];                                              //  filename.raw

      write_popup_text("write",rawfile);                                   //  write input raw file to log window
      zmainloop();

      ftype = image_file_type(rawfile);
      if (ftype != 3) {
         write_popup_text("write"," *** unknown RAW file type");
         continue;
      }
      
      sprintf(rawtherapee_command,"rawtherapee -t -Y -c ");

      err = shell_ack("%s \"%s\" ",rawtherapee_command,rawfile);           //  convert filename.raw to filename.tiff
      if (err) {
         write_popup_text("write"," *** raw conversion failed");           //  conversion failed
         write_popup_text("write",wstrerror(err));
         zmainloop();
         continue;
      }
      
      outfile = zstrdup(rawfile,5);                                        //  output is rawfilename.tif
      pp = strrchr(outfile,'.');
      strcpy(pp,".tif");
      tiffile = raw_to_tiff(rawfile);                                      //  rawfilename.tiff
      rename(outfile,tiffile);
      zfree(outfile);

      pxm1 = PXM_load(tiffile,0);                                          //  load tiff-16 output file
      if (! pxm1) {
         write_popup_text("write"," *** no Raw Therapee .tiff output");
         zmainloop();
         zfree(tiffile);
         continue;
      }

      if (resize < 1.0)                                                    //  resize down if wanted
      {
         ww2 = resize * pxm1->ww;
         hh2 = resize * pxm1->hh;
         pxm2 = PXM_rescale(pxm1,ww2,hh2);
         PXM_free(pxm1);
         pxm1 = pxm2;

         if (! pxm1) {
            write_popup_text("write"," *** resize failed");
            zmainloop();
            zfree(tiffile);
            continue;
         }
      }
      
      if (Fsharpen)                                                        //  sharpen if wanted
         batch_sharp_func(pxm1,amount,thresh);
      
      outfile = zstrdup(tiffile);                                          //  have filename.tiff, 16 bits/color

      pp = strrchr(outfile,'.');                                           //  make *.jpg or *.png if wanted
      if (pp) strcpy(pp+1,filetype);
      
      if (strnEqu(filetype,"tif",3))                                       //  output .tif file, 8 bits/color
         err = PXM_TIFF_save(pxm1,outfile,bpc);                            //    or .tiff file, 16 bits/color

      else if (strEqu(filetype,"png"))                                     //  output .png file, 8/16 bits
         err = PXM_PNG_save(pxm1,outfile,bpc);

      else err = PXM_ANY_save(pxm1,outfile);                               //  output .jpg file, 8 bits
      
      if (err) {
         write_popup_text("write"," *** file type conversion failed");
         zmainloop();
         zfree(tiffile);
         zfree(outfile);
         continue;
      }

      if (strNeq(outfile,tiffile))                                         //  if conversion was made,
         remove(tiffile);                                                  //    delete tiff-16 file

      exif_copy(rawfile,outfile,0,0,0);                                    //  copy metadata from RAW file     v.13.02
      
      if (*location && ! samedirk(location,outfile)) { 
         infile = zstrdup(outfile);                                        //  copy to new location
         zfree(outfile);
         pp = strrchr(infile,'/');                                         //  /raw-location/filename.ext
         cc = strlen(pp);                                                  //               |
         outfile = zstrdup(location,cc+1);                                 //               pp
         strcat(outfile,pp);                                               //  /new-location/filename.ext
         err = shell_ack("cp -p \"%s\" \"%s\" ",infile,outfile);           //  copy to new location
         if (err) write_popup_text("write",wstrerror(err));
         remove(infile);                                                   //  remove tempfile
         zfree(infile);
      }

      load_filemeta(outfile);                                              //  update image file index
      update_image_index(outfile);

      f_open(outfile,0,0,0);                                               //  open converted file

      write_popup_text("write",outfile);                                   //  write output file to log window
      zfree(outfile);
      zfree(tiffile);
      zmainloop();
   }

   write_popup_text("write","COMPLETED");

   if (curr_file) gallery(curr_file,"init");                               //  update gallery file list
   gallery_monitor("start");                                               //  resume tracking gallery updates

cleanup:                                                                   //  clean up and return

   if (filecount) {
      for (ii = 0; ii < filecount; ii++)                                   //  free memory
         zfree(filelist[ii]);
      zfree(filelist);
      filelist = 0;
      filecount = 0;
   }

   rawfile = 0;
   Fmenulock = 0;
   return;
}


//  dialog event and completion callback function

int batch_rawtherapee_dialog_event(zdialog *zd, cchar *event)
{
   using namespace batch_rawtherapee;

   int            ii, err, cc;
   char           countmess[50], *ploc;

   if (strEqu(event,"files"))                                              //  select images to convert
   {
      if (filecount) {                                                     //  free prior list
         for (ii = 0; ii < filecount; ii++)
            zfree(filelist[ii]);
         zfree(filelist);
         filelist = 0;
         filecount = 0;
      }

      zdialog_show(zd,0);                                                  //  hide parent dialog
      filelist = gallery_getfiles();                                       //  get list of files to convert
      zdialog_show(zd,1);

      ii = 0;
      if (filelist)                                                        //  count files in list
         for (ii = 0; filelist[ii]; ii++);
      filecount = ii;

      snprintf(countmess,50,ZTX("%d files selected"),ii);                  //  stuff count into dialog
      zdialog_stuff(zd,"fcount",countmess);
   }

   if (strEqu(event,"browse")) {
      ploc = zgetfile(ZTX("Select directory"),"folder",curr_dirk);         //  new location browse
      if (! ploc) return 1;
      zdialog_stuff(zd,"location",ploc);
      zfree(ploc);
   }

   if (zd->zstat != 1) return 0;                                           //  wait for [proceed]
   
   zd->zstat = 0;                                                          //  keep dialog active until inputs OK
   
   if (! filecount) {
      zmessageACK(Mwin,0,Bnofileselected);
      return 1;
   }
   
   zdialog_fetch(zd,"location",location,400);                              //  output location
   strTrim2(location);
   if (! blank_null(location)) {
      cc = strlen(location) - 1;
      if (location[cc] == '/') location[cc] = 0;                           //  remove trailing '/'
      err = check_create_dir(location);                                    //  create if needed
      if (err) return 1;
   }
   
   zdialog_fetch(zd,"JPEG",ii);                                            //  file type, color depth
   if (ii) { filetype = "jpg"; bpc = 8; }
   zdialog_fetch(zd,"PNG-8",ii);
   if (ii) { filetype = "png"; bpc = 8; }
   zdialog_fetch(zd,"PNG-16",ii);
   if (ii) { filetype = "png"; bpc = 16; }
   zdialog_fetch(zd,"TIFF-8",ii);
   if (ii) { filetype = "tif"; bpc = 8; }
   zdialog_fetch(zd,"TIFF-16",ii);
   if (ii) { filetype = "tiff"; bpc = 16; }
   
   resize = 1.0;
   zdialog_fetch(zd,"1.0",ii);                                             //  resize option                      v.14.06
   if (ii) resize = 1.0;
   zdialog_fetch(zd,"3/4",ii);
   if (ii) resize = 0.75;
   zdialog_fetch(zd,"2/3",ii);
   if (ii) resize = 0.666667;
   zdialog_fetch(zd,"1/2",ii);
   if (ii) resize = 0.50;
   zdialog_fetch(zd,"1/3",ii);
   if (ii) resize = 0.333333;
   
   zdialog_fetch(zd,"sharpen",Fsharpen);                                   //  sharpen option                     v.14.06
   zdialog_fetch(zd,"amount",amount);
   zdialog_fetch(zd,"thresh",thresh);

   zd->zstat = 1;                                                          //  dialog complete
   return 1;
}

/**************************************************************************/

//  callable sharpen function for batch_convert and batch_RAW_convert
//  amount: 0 to 400    strength of applied algorithm
//  thresh: 0 to 100    contrast level below which sharpen is diminished

namespace batch_sharp_names
{
   PXM      *pxm1, *pxm2;
   int      BSamount, BSthresh;
}


int batch_sharp_func(PXM *pxm, int amount, int thresh)                     //  14.06
{
   using namespace batch_sharp_names;
   
   void * batch_sharp_wthread(void *arg);
   
   pxm2 = pxm;                                                             //  output
   pxm1 = PXM_copy(pxm2);                                                  //  input
   
   BSamount = amount;
   BSthresh = thresh;

   for (int ii = 0; ii < Nwt; ii++)                                        //  start worker threads
      start_wthread(batch_sharp_wthread,&Nval[ii]);
   wait_wthreads();                                                        //  wait for completion
   
   PXM_free(pxm1);
   return 1;
}


void * batch_sharp_wthread(void *arg)                                      //  worker thread function
{
   using namespace batch_sharp_names;

   float       *pix1, *pix2;
   int         px, py;   
   float       amount, thresh;
   float       b1, b1x, b1y, b2x, b2y, b2, bf, f1, f2;
   float       red1, green1, blue1, red2, green2, blue2;

   int         index = *((int *) arg);

   amount = 1 + 0.01 * BSamount;                                           //  1.0 - 5.0
   thresh = BSthresh;                                                      //  0 - 100
   
   for (py = index + 1; py < pxm1->hh; py += Nwt)                          //  loop all image pixels
   for (px = 1; px < pxm1->ww; px++)
   {
      pix1 = PXMpix(pxm1,px,py);                                           //  input pixel
      pix2 = PXMpix(pxm2,px,py);                                           //  output pixel

      b1 = pixbright(pix1);                                                //  pixel brightness, 0 - 256
      if (b1 == 0) continue;                                               //  black, don't change                v.13.10
      b1x = b1 - pixbright(pix1-3);                                        //  horiz. brightness gradient
      b1y = b1 - pixbright(pix1-3 * pxm1->ww);                             //  vertical
      f1 = fabsf(b1x + b1y);
      
      if (f1 < thresh)                                                     //  moderate brightness change for
         f1 = f1 / thresh;                                                 //    pixels below threshold gradient
      else  f1 = 1.0;
      f2 = 1.0 - f1;

      b1x = b1x * amount;                                                  //  amplified gradient
      b1y = b1y * amount;

      b2x = pixbright(pix1-3) + b1x;                                       //  + prior pixel brightness
      b2y = pixbright(pix1-3 * pxm2->ww) + b1y;                            //  = new brightness
      b2 = 0.5 * (b2x + b2y);

      b2 = f1 * b2 + f2 * b1;                                              //  possibly moderated

      bf = b2 / b1;                                                        //  ratio of brightness change
      if (bf < 0) bf = 0;
      if (bf > 4) bf = 4;
      
      red1 = pix1[0];                                                      //  input RGB
      green1 = pix1[1];
      blue1 = pix1[2];
      
      red2 = bf * red1;                                                    //  output RGB
      if (red2 > 255.9) red2 = 255.9;
      green2 = bf * green1;
      if (green2 > 255.9) green2 = 255.9;
      blue2 = bf * blue1;
      if (blue2 > 255.9) blue2 = 255.9;

      pix2[0] = red2;
      pix2[1] = green2;
      pix2[2] = blue2;
   }

   exit_wthread();
   return 0;                                                               //  not executed, avoid gcc warning
}


/**************************************************************************/

//  create or update brightness distribution graph

namespace brdist_names
{
   GtkWidget   *drawwin_dist, *drawwin_scale;                              //  brightness distribution graph widgets
   int         RGBW[4] = { 0, 0, 0, 1 };                                   //     "  colors: red/green/blue/white (all)
}


//  menu function

void m_show_brdist(GtkWidget *, cchar *menu)                               //  menu function       v.13.02
{
   using namespace brdist_names;

   int  show_brdist_dialog_event(zdialog *zd, cchar *event);

   GtkWidget   *frdist, *frscale;
   zdialog     *zd;

   if (! Fpxb) return;

   if (menu && strEqu(menu,"kill")) {                                      //  v.13.05
      if (zdbrdist) zdialog_destroy(zdbrdist);
      zdbrdist = 0;
      return;
   }

   if (zdbrdist) {                                                         //  dialog already present
      gtk_widget_queue_draw(drawwin_dist);                                 //  refresh drawing windows
      return;
   }

   if (menu) F1_help_topic = "show_brdist";

   zd = zdialog_new(ZTX("Brightness Distribution"),Mwin,null);
   zdialog_add_widget(zd,"frame","frdist","dialog",0,"expand");            //  frames for 2 drawing areas
   zdialog_add_widget(zd,"frame","frscale","dialog");
   frdist = zdialog_widget(zd,"frdist");
   frscale = zdialog_widget(zd,"frscale");

   drawwin_dist = gtk_drawing_area_new();                                  //  histogram drawing area
   gtk_container_add(GTK_CONTAINER(frdist),drawwin_dist);
   G_SIGNAL(drawwin_dist,"draw",brdist_drawgraph,RGBW);

   drawwin_scale = gtk_drawing_area_new();                                 //  brightness scale under histogram
   gtk_container_add(GTK_CONTAINER(frscale),drawwin_scale);
   gtk_widget_set_size_request(drawwin_scale,300,12);
   G_SIGNAL(drawwin_scale,"draw",brdist_drawscale,0);

   zdialog_add_widget(zd,"hbox","hbcolors","dialog");
   zdialog_add_widget(zd,"check","all","hbcolors",Ball,"space=5");
   zdialog_add_widget(zd,"check","red","hbcolors",Bred,"space=5");
   zdialog_add_widget(zd,"check","green","hbcolors",Bgreen,"space=5");
   zdialog_add_widget(zd,"check","blue","hbcolors",Bblue,"space=5");

   zdialog_stuff(zd,"red",RGBW[0]);
   zdialog_stuff(zd,"green",RGBW[1]);
   zdialog_stuff(zd,"blue",RGBW[2]);
   zdialog_stuff(zd,"all",RGBW[3]);

   zdialog_resize(zd,300,250);
   zdialog_run(zd,show_brdist_dialog_event,"save");

   zdbrdist = zd;
   return;
}


//  dialog event and completion function

int show_brdist_dialog_event(zdialog *zd, cchar *event)
{
   using namespace brdist_names;

   if (zd->zstat) {
      zdialog_free(zd);
      zdbrdist = 0;
      return 0;
   }

   if (strstr("all red green blue",event)) {                               //  update chosen colors
      zdialog_fetch(zd,"red",RGBW[0]);
      zdialog_fetch(zd,"green",RGBW[1]);
      zdialog_fetch(zd,"blue",RGBW[2]);
      zdialog_fetch(zd,"all",RGBW[3]);
      gtk_widget_queue_draw(drawwin_dist);                                 //  refresh drawing window
   }

   return 0;
}


//  draw the histogram and the brightness scale underneath

void brdist_drawgraph(GtkWidget *drawin, cairo_t *cr, int *rgbw)           //  v.14.04
{
   int         bin, Nbins = 85, brdist[85][4];                             //  bin count, R/G/B/all
   int         px, py, dx, dy;
   int         ww, hh, winww, winhh;
   int         ii, rgb, maxdist, bright;
   float       *pixf, pixg[3];
   uint8       *pixi;
   
   TRACE

   if (! Fpxb) return;
   if (rgbw[0]+rgbw[1]+rgbw[2]+rgbw[3] == 0) return;

   winww = gtk_widget_get_allocated_width(drawin);                         //  drawing window size
   winhh = gtk_widget_get_allocated_height(drawin);

   for (bin = 0; bin < Nbins; bin++)                                       //  clear brightness distribution
   for (rgb = 0; rgb < 4; rgb++)
      brdist[bin][rgb] = 0;

   mutex_lock(&Fpixmap_lock);

   ww = Fpxb->ww;
   hh = Fpxb->hh;

   for (ii = 0; ii < ww * hh; ii++)                                        //  overhaul                     v.13.05
   {
      if (sa_stat == 3 && ! sa_pixmap[ii]) continue;                       //  stay within active select area

      py = ii / ww;                                                        //  image pixel
      px = ii - ww * py;

      dx = Mscale * px - Morgx + Dorgx;                                    //  stay within visible window
      if (dx < 0 || dx > Dww-1) continue;                                  //    for zoomed image
      dy = Mscale * py - Morgy + Dorgy;
      if (dy < 0 || dy > Dhh-1) continue;

      pixi = PXBpix(Fpxb,px,py);                                           //  use displayed image
      pixg[0] = pixi[0];                                                   //  convert uint8 RGB to float
      pixg[1] = pixi[1];
      pixg[2] = pixi[2];
      pixf = pixg;

      for (rgb = 0; rgb < 3; rgb++) {                                      //  get R/G/B brightness levels
         bright = pixf[rgb] * Nbins / 256.0;                               //  scale 0 to Nbins-1
         if (bright < 0 || bright > 255) {
            printz("pixel %d/%d: %.2f %.2f %.2f \n",px,py,pixf[0],pixf[1],pixf[2]);
            mutex_unlock(&Fpixmap_lock);
            return;
         }
         ++brdist[bright][rgb];
      }

      bright = (pixf[0] + pixf[1] + pixf[2]) * 0.333 * Nbins / 256.0;      //  R+G+B, 0 to Nbins-1
      ++brdist[bright][3];
   }

   mutex_unlock(&Fpixmap_lock);
   
   maxdist = 0;

   for (bin = 1; bin < Nbins-1; bin++)                                     //  find max. bin over all RGB
   for (rgb = 0; rgb < 3; rgb++)                                           //    omit bins 0 and last
      if (brdist[bin][rgb] > maxdist)                                      //      which can be huge
         maxdist = brdist[bin][rgb];

   for (rgb = 0; rgb < 4; rgb++)                                           //  R/G/B/white (all)
   {
      if (! rgbw[rgb]) continue;                                           //  color not selected for graph
      if (rgb == 0) cairo_set_source_rgb(cr,1,0,0);
      if (rgb == 1) cairo_set_source_rgb(cr,0,1,0);
      if (rgb == 2) cairo_set_source_rgb(cr,0,0,1);
      if (rgb == 3) cairo_set_source_rgb(cr,0,0,0);                        //  color "white" = R+G+B uses black line

      cairo_move_to(cr,0,winhh-1);                                         //  start at (0,0)

      for (px = 0; px < winww; px++)                                       //  x from 0 to window width
      {
         bin = Nbins * px / winww;                                         //  bin = 0-Nbins for x = 0-width
         py = 0.9 * winhh * brdist[bin][rgb] / maxdist;                    //  height of bin in window
         py = winhh * sqrt(1.0 * py / winhh);
         py = winhh - py - 1;
         cairo_line_to(cr,px,py);                                          //  draw line from bin to bin
      }

      cairo_stroke(cr);
   }

   return;
}


/**************************************************************************/

//  Paint a horizontal stripe drawing area with a color progressing from
//  black to white. This represents a brightness scale from 0 to 255.

void brdist_drawscale(GtkWidget *drawarea, cairo_t *cr, int *)
{
   int      px, ww, hh;
   float    fbright;

   ww = gtk_widget_get_allocated_width(drawarea);                          //  drawing area size
   hh = gtk_widget_get_allocated_height(drawarea);

   for (px = 0; px < ww; px++)                                             //  draw brightness scale
   {
      fbright = 1.0 * px / ww;
      cairo_set_source_rgb(cr,fbright,fbright,fbright);
      cairo_move_to(cr,px,0);
      cairo_line_to(cr,px,hh-1);
      cairo_stroke(cr);
   }

   return;
}


/**************************************************************************/

//  setup x and y grid lines - count/spacing, enable/disable, offsets

void m_gridlines(GtkWidget *widget, cchar *menu)
{
   int gridlines_dialog_event(zdialog *zd, cchar *event);

   zdialog     *zd;
   int         G;

   if (menu && strnEqu(menu,"grid ",5))                                    //  grid N from some edit functions
      currgrid = menu[5] - '0';                                            //  v.13.03

   else if (! widget) {                                                    //  from KB shortcut
      toggle_grid(2);                                                      //  toggle grid lines on/off
      return;
   }

   else currgrid = 0;                                                      //  must be menu call

   if (currgrid == 0)
      F1_help_topic = "grid_lines";

   zd = zdialog_new(ZTX("Grid Lines"),Mwin,Bdone,null);

   zdialog_add_widget(zd,"hbox","hb0","dialog",0,"space=10");
   zdialog_add_widget(zd,"vbox","vb1","hb0",0,"homog|space=5");
   zdialog_add_widget(zd,"vbox","vb2","hb0",0,"homog");
   zdialog_add_widget(zd,"vbox","vbspace","hb0",0,"space=5");
   zdialog_add_widget(zd,"vbox","vb3","hb0",0,"homog|space=5");
   zdialog_add_widget(zd,"vbox","vb4","hb0",0,"homog");

   zdialog_add_widget(zd,"label","lab1x","vb1",ZTX("x-spacing"));
   zdialog_add_widget(zd,"label","lab2x","vb1",ZTX("x-count"));
   zdialog_add_widget(zd,"label","lab4x","vb1",ZTX("x-enable"));

   zdialog_add_widget(zd,"spin","spacex","vb2","10|200|1|50");
   zdialog_add_widget(zd,"spin","countx","vb2","0|100|1|2");
   zdialog_add_widget(zd,"check","enablex","vb2",0);

   zdialog_add_widget(zd,"label","lab1y","vb3",ZTX("y-spacing"));
   zdialog_add_widget(zd,"label","lab2y","vb3",ZTX("y-count"));
   zdialog_add_widget(zd,"label","lab4y","vb3",ZTX("y-enable"));

   zdialog_add_widget(zd,"spin","spacey","vb4","10|200|1|50");
   zdialog_add_widget(zd,"spin","county","vb4","0|100|1|2");
   zdialog_add_widget(zd,"check","enabley","vb4",0);

   zdialog_add_widget(zd,"hbox","hboffx","dialog");
   zdialog_add_widget(zd,"label","lab3x","hboffx",Bxoffset,"space=7");
   zdialog_add_widget(zd,"hscale","offsetx","hboffx","0|100|1|0","expand");
   zdialog_add_widget(zd,"label","space","hboffx",0,"space=20");

   zdialog_add_widget(zd,"hbox","hboffy","dialog");
   zdialog_add_widget(zd,"label","lab3y","hboffy",Byoffset,"space=7");
   zdialog_add_widget(zd,"hscale","offsety","hboffy","0|100|1|0","expand");
   zdialog_add_widget(zd,"label","space","hboffy",0,"space=20");

   G = currgrid;                                                           //  grid logic overhaul    v.13.03

   if (! gridsettings[G][GXS])                                             //  if [G] never set, get defaults
      for (int ii = 0; ii < 9; ii++)
         gridsettings[G][ii] = gridsettings[0][ii];

   zdialog_stuff(zd,"enablex",gridsettings[G][GX]);                        //  current settings >> dialog widgets
   zdialog_stuff(zd,"enabley",gridsettings[G][GY]);
   zdialog_stuff(zd,"spacex",gridsettings[G][GXS]);
   zdialog_stuff(zd,"spacey",gridsettings[G][GYS]);
   zdialog_stuff(zd,"countx",gridsettings[G][GXC]);
   zdialog_stuff(zd,"county",gridsettings[G][GYC]);
   zdialog_stuff(zd,"offsetx",gridsettings[G][GXF]);
   zdialog_stuff(zd,"offsety",gridsettings[G][GYF]);

   zdialog_run(zd,gridlines_dialog_event);
   zdialog_wait(zd);
   zdialog_free(zd);
   return;
}


//  dialog event function

int gridlines_dialog_event(zdialog *zd, cchar *event)
{
   int      G = currgrid;

   if (strEqu(event,"enter")) zd->zstat = 1;                               //  [done]  v.14.03

   if (zd->zstat)                                                          //  done or cancel
   {
      if (gridsettings[G][GX] || gridsettings[G][GY])                      //  v.13.03
         gridsettings[G][GON] = 1;
      else gridsettings[G][GON] = 0;
      Fpaint2();
      return 0;
   }

   if (strEqu(event,"enablex"))                                            //  x/y grid enable or disable
      zdialog_fetch(zd,"enablex",gridsettings[G][GX]);

   if (strEqu(event,"enabley"))
      zdialog_fetch(zd,"enabley",gridsettings[G][GY]);

   if (strEqu(event,"spacex"))                                             //  x/y grid spacing (if counts == 0)
      zdialog_fetch(zd,"spacex",gridsettings[G][GXS]);

   if (strEqu(event,"spacey"))
      zdialog_fetch(zd,"spacey",gridsettings[G][GYS]);

   if (strEqu(event,"countx"))                                             //  x/y grid line counts
      zdialog_fetch(zd,"countx",gridsettings[G][GXC]);

   if (strEqu(event,"county"))
      zdialog_fetch(zd,"county",gridsettings[G][GYC]);

   if (strEqu(event,"offsetx"))                                            //  x/y grid starting offsets
      zdialog_fetch(zd,"offsetx",gridsettings[G][GXF]);

   if (strEqu(event,"offsety"))
      zdialog_fetch(zd,"offsety",gridsettings[G][GYF]);

   if (gridsettings[G][GX] || gridsettings[G][GY])                         //  if either grid enabled, show grid
      gridsettings[G][GON] = 1;

   Fpaint2();
   return 0;
}


//  toggle grid lines on or off
//  action: 0 = off, 1 = on, 2 = toggle: on > off, off > on

void toggle_grid(int action)
{
   int   G = currgrid;

   if (action == 0) gridsettings[G][GON] = 0;                              //  grid off
   if (action == 1) gridsettings[G][GON] = 1;                              //  grid on
   if (action == 2) gridsettings[G][GON] = 1 - gridsettings[G][GON];       //  toggle grid

   if (gridsettings[G][GON])
      if (! gridsettings[G][GX] && ! gridsettings[G][GY])                  //  if grid on and x/y both off,
         gridsettings[G][GX] = gridsettings[G][GY] = 1;                    //    set both grids on

   Fpaint2();
   return;
}


/**************************************************************************/

//  Show RGB values for 1-10 pixels selected with mouse-clicks.

void  show_RGB_mousefunc();
int   show_RGB_timefunc(void *);

zdialog     *RGBSzd;
int         RGBSpixel[10][2];                                              //  last 10 pixels clicked
int         RGBSnpix;                                                      //  count of pixels
int         RGBSmetric = 1;                                                //  1/2/3 = /RGB/EV/OD
int         RGBSdelta = 0;                                                 //  abs/delta mode
int         RGBSlabels = 0;                                                //  pixel labels on/off
int         RGBSupdate = 0;                                                //  window update needed


void m_show_RGB(GtkWidget *, cchar *menu)
{
   int   show_RGB_event(zdialog *zd, cchar *event);

   PangoFontDescription    *pfontdesc;
   GtkWidget   *widget;
   cchar       *mess = ZTX("Click image to select pixels.");
   cchar       *header = " Pixel            Red     Green   Blue";
   char        hbx[8] = "hbx", pixx[8] = "pixx";                           //  last char. is '0' to '9'
   int         ii;

   F1_help_topic = "show_RGB";

   if (! curr_file) return;                                                //  no image file
   RGBSnpix = 0;                                                           //  no pixels yet
   pfontdesc = pango_font_description_from_string("Monospace 9");          //  monospace font

/***
    _________________________________________
   |                                         |
   |  Click image to select pixels.          |
   |  [x] delta  [x] labels                  |
   |   Metric: (o) RGB  (o) EV               |
   |   Pixel           Red    Green  Blue    |
   |   A xxxxx xxxxx   xxxxx  xxxxx  xxxxx   |
   |   B xxxxx xxxxx   xxxxx  xxxxx  xxxxx   |
   |   C xxxxx xxxxx   xxxxx  xxxxx  xxxxx   |
   |   D xxxxx xxxxx   xxxxx  xxxxx  xxxxx   |
   |   E xxxxx xxxxx   xxxxx  xxxxx  xxxxx   |
   |   F xxxxx xxxxx   xxxxx  xxxxx  xxxxx   |
   |   G xxxxx xxxxx   xxxxx  xxxxx  xxxxx   |
   |   H xxxxx xxxxx   xxxxx  xxxxx  xxxxx   |
   |   I xxxxx xxxxx   xxxxx  xxxxx  xxxxx   |
   |   J xxxxx xxxxx   xxxxx  xxxxx  xxxxx   |
   |     xxxxx xxxxx   xxxxx  xxxxx  xxxxx   |
   |                                         |
   |                        [clear] [done]   |
   |_________________________________________|

***/

   if (RGBSzd) zdialog_free(RGBSzd);                                       //  delete previous if any
   zdialog *zd = zdialog_new(ZTX("Show RGB"),Mwin,Bclear,Bdone,null);
   RGBSzd = zd;

   zdialog_add_widget(zd,"hbox","hbmess","dialog",0,"space=3");
   zdialog_add_widget(zd,"label","labmess","hbmess",mess,"space=5");

   zdialog_add_widget(zd,"hbox","hbmym","dialog");
   zdialog_add_widget(zd,"check","delta","hbmym","delta","space=8");
   zdialog_add_widget(zd,"check","labels","hbmym","labels","space=8");

   if (RGBSdelta && E3pxm) zdialog_stuff(zd,"delta",1);

   zdialog_add_widget(zd,"hbox","hbmetr","dialog",0,"space=3");
   zdialog_add_widget(zd,"label","labmetr","hbmetr","Metric:","space=5");
   zdialog_add_widget(zd,"radio","radRGB","hbmetr","RGB","space=3");
   zdialog_add_widget(zd,"radio","radEV","hbmetr","EV","space=3");

   if (RGBSmetric == 1) zdialog_stuff(zd,"radRGB",1);                      //  get which metric to use
   if (RGBSmetric == 2) zdialog_stuff(zd,"radEV",1);

   zdialog_add_widget(zd,"vbox","vbdat","dialog");                         //  vbox for current pixel values
   zdialog_add_widget(zd,"hbox","hbpix","vbdat");
   zdialog_add_widget(zd,"label","labheader","hbpix",header);              //  Pixel        Red    Green  Blue
   widget = zdialog_widget(zd,"labheader");
   gtk_widget_override_font(widget,pfontdesc);

   for (ii = 0; ii < 10; ii++)
   {                                                                       //  10 labels for pixel data output
      hbx[2] = '0' + ii;
      pixx[3] = '0' + ii;
      zdialog_add_widget(zd,"hbox",hbx,"vbdat");
      zdialog_add_widget(zd,"label",pixx,hbx);
      widget = zdialog_widget(zd,pixx);                                    //  use monospace font
      gtk_widget_override_font(widget,pfontdesc);
   }

   zdialog_run(zd,show_RGB_event,"save");                                  //  run dialog
   takeMouse(show_RGB_mousefunc,dragcursor);                               //  connect mouse function
   g_timeout_add(200,show_RGB_timefunc,0);                                 //  start timer function, 200 ms

   return;
}


//  dialog event function

int show_RGB_event(zdialog *zd, cchar *event)
{
   int      button;

   if (strEqu(event,"enter")) zd->zstat = 2;                               //  [done]  v.14.03

   if (zd->zstat) {
      if (zd->zstat == 1) {                                                //  clear
         zd->zstat = 0;                                                    //  keep dialog active
         RGBSnpix = 0;                                                     //  point count = 0
         erase_toptext(102);                                               //  erase labels on image
         RGBSupdate++;                                                     //  update display window
      }
      else {                                                               //  done or kill
         freeMouse();                                                      //  disconnect mouse function
         zdialog_free(RGBSzd);                                             //  kill dialog
         RGBSzd = 0;
         erase_toptext(102);
      }
      Fpaint2();
   }

   if (strEqu(event,"focus"))                                              //  toggle mouse capture
      takeMouse(show_RGB_mousefunc,dragcursor);                            //  connect mouse function

   if (strEqu(event,"delta")) {                                            //  set absolute/delta mode
      zdialog_fetch(zd,"delta",RGBSdelta);
      if (RGBSdelta && ! E3pxm) {
         RGBSdelta = 0;                                                    //  block delta mode if no edit underway
         zdialog_stuff(zd,"delta",0);
         zmessageACK(Mwin,0,ZTX("Edit function must be active"));          //  13.02.1
      }
   }

   if (strEqu(event,"labels")) {                                           //  get labels on/off
      zdialog_fetch(zd,"labels",RGBSlabels);
      RGBSupdate++;                                                        //  update window
   }

   if (strEqu(event,"radRGB")) {                                           //  metric = RGB
      zdialog_fetch(zd,event,button);
      if (button) RGBSmetric = 1;
   }

   if (strEqu(event,"radEV")) {                                            //  metric = EV
      zdialog_fetch(zd,event,button);
      if (button) RGBSmetric = 2;
   }

   return 0;
}


//  mouse function
//  table positions [0] to [RGBSnpix-1] for labeled pixel positions, fixed
//  table position [RGBSnpix] = values from current mouse position

void show_RGB_mousefunc()                                                  //  mouse function
{
   int      ii;
   PXM      *pxm;

   if (E3pxm) pxm = E3pxm;                                                 //  report image being edited
   else if (E1pxm) pxm = E1pxm;
   else if (E0pxm) pxm = E0pxm;
   else {
      E0pxm = PXM_load(curr_file,1);                                       //  never edited, 
      if (! E0pxm) return;                                                 //  create E0 image for my use
      pxm = E0pxm;
   }

   if (Mxposn < 0 || Mxposn > pxm->ww-1) return;                           //  mouse outside image, ignore
   if (Myposn < 0 || Myposn > pxm->hh-1) return;

   if (LMclick)                                                            //  left click, add labeled position
   {
      LMclick = 0;

      if (RGBSnpix == 9) {                                                 //  if all 9 labeled positions filled,
         for (ii = 1; ii < 9; ii++) {                                      //    remove first (oldest) and
            RGBSpixel[ii-1][0] = RGBSpixel[ii][0];                         //      push the rest back
            RGBSpixel[ii-1][1] = RGBSpixel[ii][1];
         }
         RGBSnpix = 8;                                                     //  position for newest
      }

      ii = RGBSnpix;                                                       //  next position to fill
      RGBSpixel[ii][0] = Mxclick;                                          //  save newest pixel
      RGBSpixel[ii][1] = Myclick;
      RGBSnpix++;
   }
   
   ii = RGBSnpix;                                                          //  fill next position from active mouse
   RGBSpixel[ii][0] = Mxposn;                                              //  v.14.01
   RGBSpixel[ii][1] = Myposn;

   RGBSupdate++;                                                           //  update window
   return;
}


//  timer function - continuously display RGB values for selected pixels

int show_RGB_timefunc(void *arg)                                           //  up to 10 pixels, live update
{
   static char    label[10][4] = { " A ", " B ", " C ", " D ", " E ",
                                   " F ", " G ", " H ", " I ", " J " };
   PXM         *pxm = 0;
   int         ii, jj, px, py, delta;
   int         ww, hh;
   float       red1, green1, blue1;
   float       red3, green3, blue3;
   float       *ppixa, *ppixb;
   char        text[100], pixx[8] = "pixx";

   #define EVfunc(rgb) (log2(rgb) - 7)                                     //  RGB units to EV (128 == 0 EV)
   
   if (! RGBSzd) return 0;                                                 //  user quit, cancel timer            v.14.01

   if (RGBSdelta && E3pxm) delta = 1;                                      //  delta mode only if edit underway
   else delta = 0;

   if (E3pxm) pxm = E3pxm;                                                 //  report image being edited
   else if (E1pxm) pxm = E1pxm;
   else if (E0pxm) pxm = E0pxm;
   else return 1;

   ww = pxm->ww;
   hh = pxm->hh;

   if (RGBSnpix)                                                           //  some labeled positions to report
   {
      for (ii = 0; ii < 10; ii++)
      {
         px = RGBSpixel[ii][0];                                            //  next pixel to report
         py = RGBSpixel[ii][1];
         if (px >= 0 && px < ww && py >= 0 && py < hh) continue;           //  within image limits
         for (jj = ii+1; jj < 10; jj++) {
            RGBSpixel[jj-1][0] = RGBSpixel[jj][0];                         //  remove pixel outside limits
            RGBSpixel[jj-1][1] = RGBSpixel[jj][1];                         //    and pack the remaining down
         }
         RGBSpixel[9][0] = RGBSpixel[9][1] = 0;                            //  last position is empty
         RGBSnpix--;
         ii--;
         RGBSupdate++;                                                     //  update window
      }
   }

   if (RGBSupdate)                                                         //  refresh window labels
   {
      RGBSupdate = 0;
      erase_toptext(102);

      if (RGBSlabels) {
         for (ii = 0; ii < RGBSnpix; ii++) {                               //  show pixel labels on image
            px = RGBSpixel[ii][0];
            py = RGBSpixel[ii][1];
            add_toptext(102,px,py,label[ii],"Sans 8");
         }
      }

      Fpaint2();
   }

   red1 = green1 = blue1 = 0;

   for (ii = 0; ii < 10; ii++)                                             //  loop positons 0 to 9
   {
      pixx[3] = '0' + ii;                                                  //  widget names "pix0" ... "pix9"

      px = RGBSpixel[ii][0];                                               //  next pixel to report
      py = RGBSpixel[ii][1];
      if (ii > RGBSnpix) {                                                 //  no pixel there yet
         zdialog_stuff(RGBSzd,pixx,"");                                    //  blank report line
         continue;
      }

      ppixa = PXMpix(pxm,px,py);                                           //  get pixel RGB values
      red3 = ppixa[0];
      green3 = ppixa[1];
      blue3 = ppixa[2];

      if (delta && E1pxm && px < E1pxm->ww && py < E1pxm->hh) {            //  delta RGB for ongoing edited image
         ppixb = PXMpix(E1pxm,px,py);                                      //  "before" image E1
         red1 = ppixb[0];
         green1 = ppixb[1];
         blue1 = ppixb[2];
      }

      if (ii == RGBSnpix)
         sprintf(text,"   %5d %5d  ",px,py);                               //  mouse pixel, format "   xxxx yyyy"
      else sprintf(text," %c %5d %5d  ",'A'+ii,px,py);                     //  fixed pixel, format " A xxxx yyyy"

      if (RGBSmetric == 1) {                                               //  output RGB values
         if (delta) {
            red3 -= red1;                                                  //  delta RGB
            green3 -= green1;
            blue3 -= blue1;
         }
         sprintf(text+14,"   %6.2f  %6.2f  %6.2f ",red3,green3,blue3);     //  xxx.xx
      }

      if (RGBSmetric == 2) {                                               //  output EV values
         red3 = EVfunc(red3);
         green3 = EVfunc(green3);
         blue3 = EVfunc(blue3);
         if (delta) {
            red3 -= EVfunc(red1);                                          //  delta EV
            green3 -= EVfunc(green1);
            blue3 -= EVfunc(blue1);
         }
         sprintf(text+14,"   %6.3f  %6.3f  %6.3f ",red3,green3,blue3);     //  x.xxx
      }

      zdialog_stuff(RGBSzd,pixx,text);                                     //  pixel and RGB values >> dialog
   }

   return 1;
}


/**************************************************************************/

//  dark_brite menu function
//  highlight darkest and brightest pixels

namespace darkbrite {
   float    darklim = 0;
   float    brightlim = 255;
}

void m_darkbrite(GtkWidget *, const char *)                                //  v.13.11
{
   using namespace darkbrite;

   int    darkbrite_dialog_event(zdialog* zd, const char *event);

   cchar    *title = ZTX("Darkest and Brightest Pixels");

   F1_help_topic = "darkbrite";

   if (! curr_file) return;                                                //  no image file

/**
       ______________________________________
      |    Darkest and Brightest Pixels      |
      |                                      |
      |  Dark Limit   ===[]============ NNN  |
      |  Bright Limit ============[]=== NNN  |
      |                                      |
      |                              [Done]  |
      |______________________________________|
      
**/
   
   zdialog *zd = zdialog_new(title,Mwin,Bdone,null);                       //  darkbrite dialog
   zdialog_add_widget(zd,"hbox","hb1","dialog",0,"space=5");
   zdialog_add_widget(zd,"vbox","vb1","hb1",0,"space=3");
   zdialog_add_widget(zd,"vbox","vb2","hb1",0,"space=3|expand");
   zdialog_add_widget(zd,"vbox","vb3","hb1",0,"space=3");
   zdialog_add_widget(zd,"label","labD","vb1",ZTX("Dark Limit"));
   zdialog_add_widget(zd,"label","labB","vb1",ZTX("Bright Limit"));
   zdialog_add_widget(zd,"hscale","limD","vb2","0|255|1|0","expand");
   zdialog_add_widget(zd,"hscale","limB","vb2","0|255|1|255","expand");
   zdialog_add_widget(zd,"label","valD","vb3");
   zdialog_add_widget(zd,"label","valB","vb3");

   zdialog_stuff(zd,"limD",darklim);                                       //  start with prior values
   zdialog_stuff(zd,"limB",brightlim);

   zdialog_resize(zd,300,0);
   zdialog_run(zd,darkbrite_dialog_event,"save");                          //  run dialog - parallel
   zddarkbrite = zd;                                                       //  global pointer for Fpaint*()

   zdialog_send_event(zd,"limD");                                          //  initz. NNN labels
   zdialog_send_event(zd,"limB");

   return;
}


//  darkbrite dialog event and completion function

int darkbrite_dialog_event(zdialog *zd, const char *event)                 //  darkbrite dialog event function
{
   using namespace darkbrite;

   char     text[8];

   if (strEqu(event,"enter")) zd->zstat = 1;                               //  [done]  v.14.03

   if (zd->zstat)
   {
      zdialog_free(zd);
      zddarkbrite = 0;
      Fpaint2();
      return 0;
   }

   if (strEqu(event,"limD")) {
      zdialog_fetch(zd,"limD",darklim);
      sprintf(text,"%.0f",darklim);
      zdialog_stuff(zd,"valD",text);
   }

   if (strEqu(event,"limB")) {
      zdialog_fetch(zd,"limB",brightlim);
      sprintf(text,"%.0f",brightlim);
      zdialog_stuff(zd,"valB",text);
   }
   
   Fpaint2();
   return 0;
}


//  this function called by Fpaint() if zddarkbrite dialog active

void darkbrite_paint()                                                     //  v.14.05
{
   using namespace darkbrite;
   
   int         px, py;
   uint8       *pix;
   float       P, D = darklim, B = brightlim;

   for (py = 0; py < Mpxb->hh; py++)                                       //  loop all image pixels
   for (px = 0; px < Mpxb->ww; px++)
   {
      pix = PXBpix(Mpxb,px,py);
      P = pixbright(pix); 
      if (P < D) pix[0] = 100;                                             //  dark pixel = mostly red
      else if (P > B) pix[0] = 0;                                          //  bright pixel = mostly green/blue
   }

   return;                                                                 //  not executed, avoid gcc warning
}


/**************************************************************************/

//  monitor test function

void m_moncheck(GtkWidget *, cchar *)
{
   char     file[200];
   cchar    *message = ZTX("Brightness should show a gradual ramp \n"
                           "extending all the way to the edges.");

   F1_help_topic = "check_monitor";
   snprintf(file,200,"%s/images/colorchart.png",get_zdatadir());           //  color chart .png file
   temp_image(file,"check monitor",message);
   return;
}


/**************************************************************************/

//  Show a temporary image and return to prior image when done

void temp_image(cchar *file, cchar *title, cchar *message)
{
   int         err;
   zdialog     *zd;
   char        *savecurrfile = 0;
   char        *savecurrdirk = 0;
   
   if (checkpend("all")) return;                                           //  check nothing pending              v.13.12
   Fmenulock = 1;

   if (curr_file) savecurrfile = zstrdup(curr_file);
   if (curr_dirk) savecurrdirk = zstrdup(curr_dirk);
   
   err = f_open(file);
   if (err) goto restore;

   Fzoom = 1;
   gtk_window_set_title(MWIN,title);

   zd = zdialog_new(title,Mwin,Bdone,null);                                //  start user dialog
   if (message) {
      zdialog_add_widget(zd,"hbox","hb1","dialog",0,"space=5");
      zdialog_add_widget(zd,"label","lab1","hb1",message,"space=5");
   }
   
   zdialog_resize(zd,300,0);
   zdialog_run(zd,0,"0/0");
   zdialog_wait(zd);                                                       //  wait for dialog complete
   zdialog_free(zd);

restore:

   if (savecurrdirk) {                                                     //  v.13.05.1
      curr_dirk = zstrdup(savecurrdirk);
      zfree(savecurrdirk);
   }

   if (savecurrfile) {
      f_open(savecurrfile);                                                //  v.13.05.1
      zfree(savecurrfile);
   }
   else {
      curr_file = 0;
      if (curr_dirk) gallery(curr_dirk,"init",-1);
   }

   Fmenulock = 0;
   return;
}


/**************************************************************************/

//  check and adjust monitor gamma

void m_mongamma(GtkWidget *, cchar *)
{
   int   mongamma_event(zdialog *zd, cchar *event);

   int         err;
   char        gammachart[200];
   zdialog     *zd;
   char        *savecurrent = 0;

   cchar       *permit = "Chart image courtesy of Norman Koren";
   cchar       *website = "http://www.normankoren.com";

   F1_help_topic = "monitor_gamma";

   err = shell_quiet("which xgamma");                                      //  check for xgamma
   if (err) {
      zmessageACK(Mwin,0,"xgamma program is not installed");
      return;
   }

   if (curr_file) savecurrent = zstrdup(curr_file);

   snprintf(gammachart,200,"%s/images/gammachart.png",get_zdatadir());     //  gamma chart .png file
   err = f_open(gammachart);
   if (err) goto restore;

   Fzoom = 1;                                                              //  scale 100% (required)
   gtk_window_set_title(MWIN,"adjust gamma chart");

   zd = zdialog_new(ZTX("Monitor Gamma"),Mwin,Bdone,null);                 //  start user dialog
   zdialog_add_widget(zd,"hbox","hb1","dialog",0,"space=8");
   zdialog_add_widget(zd,"label","labgamma","hb1","gamma","space=5");
   zdialog_add_widget(zd,"hscale","gamma","hb1","0.6|1.4|0.02|1.0","expand");
   zdialog_add_widget(zd,"hbox","hb2","dialog");
   zdialog_add_widget(zd,"label","permit","hb2",permit,"space=5");
   zdialog_add_widget(zd,"hbox","hb3","dialog");
   zdialog_add_widget(zd,"link","nolabel","hb3",website);

   zdialog_resize(zd,200,0);
   zdialog_restore_inputs(zd);                                             //  preload prior user inputs
   zdialog_run(zd,mongamma_event);
   zdialog_wait(zd);                                                       //  wait for dialog complete
   zdialog_free(zd);

restore:

   if (savecurrent) {
      f_open(savecurrent);                                                 //  back to current file
      zfree(savecurrent);
   }

   return;
}


//  dialog event function

int mongamma_event(zdialog *zd, cchar *event)
{
   double   gamma;

   if (strEqu(event,"enter")) zd->zstat = 1;                               //  [done]  v.14.03

   if (strEqu(event,"gamma")) {
      zdialog_fetch(zd,"gamma",gamma);
      shell_ack("xgamma -quiet -gamma %.2f",gamma);
   }

   return 0;
}


/**************************************************************************/

//  set GUI language

void  m_lang(GtkWidget *, cchar *)                                         //  overhauled
{
   zdialog     *zd;
   int         ii, cc, val, zstat;
   char        progexe[300], lang1[8], *pp;

   cchar  *langs[11] = { "en English",                                     //  english first
                         "ca Catalan", "de German", "es Spanish",
                         "fr French", "it Italian", "nl Dutch", 
                         "pt Portuguese", "ru Russian", 
                         "sv Swedish", null  };

   cchar  *title = ZTX("Available Translations");

   F1_help_topic = "language";

   zd = zdialog_new(ZTX("Set Language"),Mwin,Bdone,Bcancel,null);
   zdialog_add_widget(zd,"label","title","dialog",title,"space=5");
   zdialog_add_widget(zd,"hbox","hb1","dialog");
   zdialog_add_widget(zd,"vbox","vb1","hb1");

   for (ii = 0; langs[ii]; ii++)                                           //  make radio button per language
      zdialog_add_widget(zd,"radio",langs[ii],"vb1",langs[ii]);

   cc = strlen(zfuncs::zlang);                                             //  current language
   for (ii = 0; langs[ii]; ii++)                                           //  match on lc_RC
      if (strnEqu(zfuncs::zlang,langs[ii],cc)) break;
   if (! langs[ii])
      for (ii = 0; langs[ii]; ii++)                                        //  failed, match on lc alone
         if (strnEqu(zfuncs::zlang,langs[ii],2)) break;
   if (! langs[ii]) ii = 0;                                                //  failed, default english
   zdialog_stuff(zd,langs[ii],1);

   zdialog_resize(zd,200,0);
   zdialog_run(zd);                                                        //  run dialog
   zstat = zdialog_wait(zd);

   for (ii = 0; langs[ii]; ii++) {                                         //  get active radio button
      zdialog_fetch(zd,langs[ii],val);
      if (val) break;
   }

   zdialog_free(zd);                                                       //  kill dialog

   if (zstat != 1) return;                                                 //  user cancel
   if (! val) return;                                                      //  no selection

   strncpy0(lang1,langs[ii],8);
   pp = strchr(lang1,' ');                                                 //  isolate lc_RC part
   *pp = 0;

   cc = readlink("/proc/self/exe",progexe,300);                            //  get own program path      v.13.01
   if (cc <= 0) {
      zmessageACK(Mwin,0,"cannot get /proc/self/exe");
      return;
   }
   progexe[cc] = 0;

   shell_ack("%s -l %s -p &",progexe,lang1);                               //  start new fotoxx with language

   m_quit(0,0);                                                            //  exit
   return;
}


/**************************************************************************/

//  report missing translations in a popup window

void  m_untranslated(GtkWidget *, cchar *)                                 //  v.13.07
{
   int      ftf = 1, Nmiss = 0;
   cchar    *missing;
   char     text[40];

   F1_help_topic = "missing_translations";

   write_popup_text("open","missing translations",300,200,Mwin);

   while (true)
   {
      missing = ZTX_missing(ftf);
      if (! missing) break;
      write_popup_text("write",missing);
      Nmiss++;
   }

   sprintf(text,"%d missing translations",Nmiss);
   write_popup_text("write",text);
   write_popup_text("top",0);

   return;
}


/**************************************************************************/

//  create desktop icon / launcher

void  m_menu_launcher(GtkWidget *, cchar *)
{
   char  *command;

   F1_help_topic = "menu_launcher";
   command = zdialog_text(Mwin,ZTX("Make Launcher"),"fotoxx -recent");
   if (! command || ! *command) return;
   zmake_menu_launcher(command,"Graphics","Image Editor");
   zfree(command);                                                         //  v.13.01
   return;
}


/**************************************************************************/

//  burn images to CD/DVD

void  m_burn(GtkWidget *, cchar *)
{
   int      ii, cc;
   char     **filelist, *imagefile, *bcommand;
   
   if (! Fbrasero) {                                                       //  v.14.07
      zmessageACK(Mwin,0,ZTX("Brasero not installed"));
      return;
   }

   if (checkpend("all")) return;                                           //  check nothing pending              v.13.12

   F1_help_topic = "burn";

   filelist = gallery_getfiles();                                          //  get list of files to burn
   if (! filelist) return;                                                 //  nothing selected
   m_viewmode(0,"F");

   cc = 0;
   for (ii = 0; filelist[ii]; ii++)                                        //  get memory for brasero command line
      cc += strlen(filelist[ii]) + 4;

   bcommand = (char *) zmalloc(cc+20);
   strcpy(bcommand,"brasero ");
   cc = strlen(bcommand);

   for (ii = 0; filelist[ii]; ii++)                                        //  copy files to command line
   {
      imagefile = filelist[ii];
      strcpy(bcommand+cc," \"");
      cc += 2;
      strcpy(bcommand+cc,imagefile);
      cc += strlen(imagefile);
      strcpy(bcommand+cc,"\"");
      cc += 1;
      zfree(imagefile);
   }

   zfree(filelist);

   strcat(bcommand," &");                                                  //  brasero command in background
   shell_ack(bcommand);
   zfree(bcommand);

   return;
}


/**************************************************************************/

//  Dump CPU usage, zdialog statistics, memory statistics
//  counters are reset

void m_resources(GtkWidget *, cchar *)
{
   zmalloc_report();
   return;
}


/**************************************************************************/

//  zappcrash test - make a segment fault

char     *zapptext = null;

void m_zappcrash(GtkWidget *, cchar *)                                     //  v.14.01
{
   int      ii, jj;

   TRACE
   
   for (ii = 0; ii < 100; ii++) {                                          //  fake-out compiler diagnostics
      jj = 1000.0 * drandz();
      zapptext[jj] = 'a' + ii / 10;                                        //  write to address 0 + jj
   }

   printz("zapptext: %s \n",zapptext);
   return;
}




