/**************************************************************************

   Fotoxx      edit photos and manage collections

   Copyright 2007-2014 Michael Cornelison
   Source URL: http://kornelix.com/fotoxx
   Contact: kornelix@posteo.de
   
   This program is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program. If not, see http://www.gnu.org/licenses/.

***************************************************************************

   main                 main program, set up defaults
   initzfunc            initializations, carry out command line options
   gtimefunc            periodic function
   paint_busy           show busy indicator on window when something is running
   update_Fpanel        update status parameters on F window top panel

   delete_event         response function for main window delete event
   destroy_event        response function for main window delete event
   state_event          response function for main window fullscreen state change

   FWpaint              main / drawing window refresh (draw signal response function)
   Fpaint2              trigger FWpaint, callable from theads
   Fpaintnow            create cairo context and call FWpaint
   Fpaint3              update drawing window section from updated E3 section
   Fpaint4              update drawing window visible portion

   mouse_event          mouse event response function
   mouse_convert        convert mouse/window space to image space
   m_zoom               main window zoom in/out function
   KBstate              send KB key from dialog to main window
   KBpress              KB key press event function
   KBrelease            KB key release event function
   win_fullscreen       set main window full screen status
   win_unfullscreen     restore main window to former size

   draw_pixel           draw one overlay pixel using image space
   erase_pixel          erase one pixel
   draw_line            draw overlay line in image space
   draw_dotline         draw overlay dotted-line in image space
   erase_line           erase line
   draw_toplines        draw or redraw a set of overlay lines
   draw_gridlines       draw grid lines over image
   add_toptext          add to set of overlay text strings
   draw_toptext         draw set of overlay text strings
   erase_toptext        remove text from set of overlay text strings
   draw_text            draw text on window in image space
   add_topcircle        add a circle to set of overlay circles
   draw_topcircles      draw the set of overlay circles in window space
   erase_topcircles     erase the set of overlay circles
   draw_mousearc        draw a circle or ellipse centered on mouse, image space

   splcurve_init        set up a spline curve drawing area
   splcurve_adjust      mouse event function to manipulate curve nodes
   splcurve_draw        draw curve through nodes
   splcurve_generate    generate x/y table of values from curve
   splcurve_yval        get curve y-value for given x-value
   splcurve_load        load curve data from a saved file
   splcurve_save        save curve data to a file

   edit_setup()         start an image edit function
   edit_cancel()        cancle image edit
   edit_done()          finish image edit
   edit_undo()          undo current edit (reset)
   edit_redo()          redo current edit
   edit_reset()         reset all edit changes
   edit_fullsize()      convert preview to full size edit

   m_undo               undo one edit in a series
   m_redo               redo one edit in a series
   undo_all             undo all edits for image
   save_undo            save image in the undo stack
   load_undo            load image from the undo stack
   checkpend()          check list: edit mods lock any quiet
   takemouse            set mouse event function and special cursor
   freemouse            remove mouse event function, set normal cursor

   start_thread            start thread running
   signal_thread           signal thread that work is pending
   wait_thread_idle        wait for pending work complete
   wrapup_thread           wait for thread exit or command thread exit
   wrapup_thread_noblock   same with periodic zmainloop()
   thread_idle_loop        wait for pending work, exit if commanded
   thread_exit             exit thread unconditionally
   start_wthread           start a working thread per SMP processor
   exit_wthread            exit a working thread
   wait_wthreads           wait for all working threads to exit

   m_favorites          function to generate favorites menu
   favorites_callback   response function for clicked menu
   m_help               response function for help menu
   save_params          save parameters when fotoxx exits
   load_params          load parameters at fotoxx startup
   free_resources       free resources for the current image file
   vpixel               get virtual pixel at any image location (float)
   sigdiff              compare 2 floats with margin of significance
   PXM_audit            check PXM image for pixel values > 255.99...
   PXM_make             create new PXM pixmap image
   PXM_free             destroy PXM and free memory
   PXM_copy             create a copy of a PXM image
   PXM_copy_area        create a copy of a rectangular area within a PXM
   PXM_fixblue          replace blue = 0 pixels with blue = 1
   PXM_rescale          create a rescaled copy of a PXM image
   PXM_rotate           create a rotated copy of a PXM image
   
   PXB_make             create a new PXB image, empty for from GdkPixbuf
   PXB_free             destroy PXB and free memory
   PXB_copy             create a copy of a PXB image
   PXB_rescale          create a rescaled copy of a PXB image
   PXM_PXB_copy         create a PXB copy of a PXM image
   PXM_PXB_update       update a rectangular area in a PXB from a corresp. PXM
   PXB_PXB_update       update a rectangular area in a PXB from another PXB

***************************************************************************/


#define EX                                                                 //  disable extern declarations
#include "fotoxx.h"                                                        //  (variables in fotoxx.h are defined)


//  fotoxx main program

int main(int argc, char *argv[])
{   
   char           lang[8] = "";
   int            Fclone=0, cloxx=0, cloyy=0, cloww=0, clohh=0;
   int            err;
   struct stat    statb;
   
   if (argc > 1 && strEqu(argv[1],"-v")) {
      printz(Frelease "\n");                                               //  print version and exit
      return 0;
   }

   gtk_init(&argc,&argv);                                                  //  initz. GTK & strip GTK params
   zinitapp("fotoxx",null);                                                //  initz. app directories
   err = chdir(get_zuserdir());                                            //  set to /home/<user>/.fotoxx        v.14.02

   //  initialize externals to default values 
   //  (saved parameters will override some of these)

   strcpy(zfuncs::zappname,Frelease);                                      //  app name and version
   Ffirsttime = 1;                                                         //  first time startup (params override)
   Prelease = zstrdup("unknown");                                          //  prev. fotoxx release (params override)
   Nrelease = zstrdup("unknown");                                          //  last notified new release
   for (int ii = 0; ii < 10; ii++) Nval[ii] = ii;                          //  integer values 0-9
   mwgeom[0] = mwgeom[1] = 100;                                            //  default main window geometry
   mwgeom[2] = 1200; mwgeom[3] = 800;                                      //  v.13.05
   *paneltext = 0;                                                         //  no status bar text                 v.14.03
   panelfont = pango_font_description_from_string("Monospace 9px");        //  top panel font, 7 pixels per character
   trimsize[0] = 1600;                                                     //  default initial image trim size
   trimsize[1] = 1000;
   trimbuttons[0] = zstrdup("1:1");  trimratios[0] = zstrdup("1:1");       //  default trim ratio buttons         v.13.10
   trimbuttons[1] = zstrdup("2:1");  trimratios[1] = zstrdup("2:1");
   trimbuttons[2] = zstrdup("3:2");  trimratios[2] = zstrdup("3:2");
   trimbuttons[3] = zstrdup("4:3");  trimratios[3] = zstrdup("4:3");
   trimbuttons[4] = zstrdup("16:9");  trimratios[4] = zstrdup("16:9");
   trimbuttons[5] = zstrdup("gold");  trimratios[5] = zstrdup("1.62:1");
   editresize[0] = 1600;                                                   //  default initial resize size
   editresize[1] = 1200;
   currgrid = 0;                                                           //  default initial grid settings      v.13.02
   gridsettings[0][GON] = 0;                                               //  grid off
   gridsettings[0][GX] = gridsettings[0][GY] = 1;                          //  x/y grid lines enabled
   gridsettings[0][GXS] = gridsettings[0][GYS] = 200;                      //  x/y spacing
   gridsettings[0][GXC] = gridsettings[0][GYC] = 5;                        //  x/y count
   menu_style = zstrdup("both");                                           //  default menu style (icons + text)
   interpolation = zstrdup("bilinear");                                    //  default image interpolation        v.13.05
   startdisplay = zstrdup("blank");                                        //  start with blank window
   Fdragopt = 2;                                                           //  image drag is magnified scroll
   zoomcount = 2;                                                          //  zooms to reach 2x image size
   zoomratio = sqrtf(2);                                                   //  corresp. zoom ratio 
   thumbdirk = 0;                                                          //  no thumbnail directory
   jpeg_def_quality = 90;                                                  //  default .jpeg save quality         v.13.08
   jpeg_1x_quality = 90;
   lens_mm = 35;                                                           //  pano lens parameters
   lens_bow = 0;
   DCrawcommand = zstrdup("dcraw -T -6 -w -q 0 -o 1");                     //  default RAW to tiff command using DCraw
   RAWfiletypes = zstrdup(".arw .crw .cr2 .dng .erf .iiq .mef .mos "       //  some of the known RAW file types
                          ".mpo .nef .nrw .orf .pef .ptx .raw "            //  (case is not significant here)
                          ".rw2 .rwl .srf .srw .sr2 ");
   imagefiletypes = zstrdup(".jpg .jpeg .png .tif .tiff .bmp .ico "        //  supported image file types         v.14.01
                            ".gif .svg .xpm .tga ");

   black[0] = black[1] = black[2] = 0;                                     //  define RGB colors
   white[0] = white[1] = white[2] = 255;
   red[0] = 255; red[1] = red[2] = 0;
   green[1] = 255; green[0] = green[2] = 0;
   fg_color = green;                                                       //  initial foreground drawing color

   GDKdark.red = GDKdark.green = GDKdark.blue = 0.2;                       //  GDK color for window backgrounds   v.14.04.2
   GDKdark.alpha = 1;

   arrowcursor = gdk_cursor_new(GDK_TOP_LEFT_ARROW);                       //  cursor for selection
   dragcursor = gdk_cursor_new(GDK_CROSSHAIR);                             //  cursor for dragging
   drawcursor = gdk_cursor_new(GDK_PENCIL);                                //  cursor for drawing lines

   //  process command line parameters
   
   for (int ii = 1; ii < argc; ii++)                                       //  command line parameters
   {
      char *pp = argv[ii];
      
      if (strcmpv(pp,"-debug","-d",null))                                  //  -d -debug
         Fdebug = 1;
      else if (strcmpv(pp,"-lang","-l",null) && argc > ii+1)               //  -l -lang lc_RC   language/region code
         strncpy0(lang,argv[++ii],7);
      else if (strcmpv(pp,"-clone","-c",null) && argc > ii+4) {            //  -c -clone        clone new instance
         Fclone = 1;
         cloxx = atoi(argv[ii+1]);                                         //  window position and size
         cloyy = atoi(argv[ii+2]);                                         //    passed from parent instance
         cloww = atoi(argv[ii+3]);
         clohh = atoi(argv[ii+4]);
         ii += 4;
      }
      else if (strcmpv(pp,"-recent","-r",null))                            //  -r -recent       recent files gallery
         Frecent = 1;
      else if (strcmpv(pp,"-new","-n",null))                               //  -n -new          new added files gallery
         Fnew = 1;
      else if (strcmpv(pp,"-previous","-prev","-p",null))                  //  -p -prev         open previous file
         Fprev = 1;
      else if (strcmpv(pp,"-blank","-b",null))                             //  -b -blank        start with blank window
         Fblank = 1;
      else if (strEqu(pp,"-menu") && argc > ii+1)                          //  -menu menuname   starting menu
         startmenu = zstrdup(argv[++ii]);
      else  initial_file = zstrdup(pp);                                    //  must be initial file or directory
   }

   ZTXinit(lang);                                                          //  setup locale and translations
   setlocale(LC_NUMERIC,"en_US.UTF-8");                                    //  stop comma decimal points

   //  get files in user directory /home/<user>/.fotoxx/*

   snprintf(index_dirk,199,"%s/image_index",get_zuserdir());               //  image index directory              v.14.02
   snprintf(tags_defined_file,199,"%s/tags_defined",get_zuserdir());       //  defined tags file
   snprintf(recentfiles_file,199,"%s/recent_files",get_zuserdir());        //  recent files file (index func)
   snprintf(newfiles_file,199,"%s/newest_files",get_zuserdir());           //  newest files file (index func)
   snprintf(saved_areas_dirk,199,"%s/saved_areas",get_zuserdir());         //  saved areas directory
   snprintf(collections_dirk,199,"%s/collections",get_zuserdir());         //  collections directory
   snprintf(collections_copy,199,"%s/collections_copy",get_zuserdir());    //  collections copy file
   snprintf(saved_curves_dirk,199,"%s/saved_curves",get_zuserdir());       //  saved curves directory
   snprintf(writetext_dirk,199,"%s/write_text",get_zuserdir());            //  write_text directory
   snprintf(geotags_dirk,199,"%s/geotags",get_zuserdir());                 //  geotags directory
   snprintf(worldmap_file,200,"%s/world-map-20K.jpg",geotags_dirk);        //  world map file                     v.14.07
   snprintf(favorites_dirk,199,"%s/favorites",get_zuserdir());             //  favorites directory
   snprintf(locales_dirk,199,"%s/locales",get_zuserdir());                 //  locales directory
   snprintf(mashup_dirk,199,"%s/mashup",get_zuserdir());                   //  mashup projects directory
   locale_filespec("data","quickstart.html",quickstart_file);              //  quickstart html file
   snprintf(latest_release_file,199,"%s/latest_release",get_zuserdir());   //  latest fotoxx release file
   snprintf(slideshow_dirk,199,"%s/slideshows",get_zuserdir());            //  slide show directory
   snprintf(pattern_dirk,199,"%s/patterns",get_zuserdir());                //  pattern files directory            v.14.04
   snprintf(retouch_combo_dirk,199,"%s/retouch_combo",get_zuserdir());     //  retouch combo settings directory   v.14.05
   snprintf(custom_kernel_dirk,199,"%s/custom_kernel",get_zuserdir());     //  custom kernel files directory      v.14.06
   
   err = stat(index_dirk,&statb);                                          //  create any missing directories
   if (err) mkdir(index_dirk,0750);
   err = stat(saved_areas_dirk,&statb);
   if (err) mkdir(saved_areas_dirk,0750);
   err = stat(collections_dirk,&statb);
   if (err) mkdir(collections_dirk,0750);
   err = stat(saved_curves_dirk,&statb);
   if (err) mkdir(saved_curves_dirk,0750);
   err = stat(writetext_dirk,&statb);
   if (err) mkdir(writetext_dirk,0750);
   err = stat(geotags_dirk,&statb);
   if (err) mkdir(geotags_dirk,0750);
   err = stat(locales_dirk,&statb);                                        //  v.13.02
   if (err) mkdir(locales_dirk,0750);
   err = stat(mashup_dirk,&statb);                                         //  v.13.09
   if (err) mkdir(mashup_dirk,0750);
   err = stat(slideshow_dirk,&statb);                                      //  v.14.04
   if (err) mkdir(slideshow_dirk,0750);
   
   //  restore parameters from last session
   
   load_params();

   //  build widgets and menus for F/G/W view modes, set F mode
   
   build_widgets();
   m_viewmode(0,"F");
   zmainloop();                                                            //  GTK bug workaround                 v.14.07.1

   //  open main window and start GTK main loop

   if (Fclone) {                                                           //  clone: open new window 
      gtk_window_move(MWIN,cloxx+10,cloyy+10);                             //    slightly offset from old window
      gtk_window_resize(MWIN,cloww,clohh);
   }
   else {
      gtk_window_move(MWIN,mwgeom[0],mwgeom[1]);                           //  main window geometry
      gtk_window_resize(MWIN,mwgeom[2],mwgeom[3]);                         //  defaults or last session params
   }

   //  start GTK main loop

   g_timeout_add(100,initzfunc,0);                                         //  initz. call from gtk_main()  v.13.01
   gtk_main();                                                             //  start processing window events
   return 0;
}


/**************************************************************************/

//  Initial function called from gtk_main() at startup.
//  This function MUST return 0.

int initzfunc(void *)
{
   int            Fexiftool = 0, Fxdgopen = 0;
   int            ii, err, npid, contx;
   int            ftype, Fdirk = 0;
   char           Lrelease[20], *pp, *pp2;
   char           procfile[20], buff[200];
   char           KBshortsU[200], KBshortsI[200];
   char           tonefile[200], badnews[200];
   double         freememory, cachememory;
   float          exifver = 0;
   FILE           *fid;
   struct stat    statb;

   cchar *ftmess = ZTX("First time Fotoxx startup. \n"
                       "Latest version is available at: www.kornelix.com/fotoxx \n"
                       "An image file index will be created. Your image files will \n"
                       "not be changed, moved, or copied. This process may need \n"
                       "considerable time if you have thousands of image files.");

   cchar *kornelix_fotoxx_release = 
         "http://www.kornelix.com/uploads/1/3/0/3/13035936/fotoxx-release";

   cchar *release_notify = "new release  %s  available at \n"
                           "http://www.kornelix.com/packages";

   err = chdir(get_zuserdir());                                            //  working directory .../.fotoxx      v.14.04

   printz(Frelease "\n");                                                  //  print Fotoxx release version

   int v1 = gtk_get_major_version();                                       //  get GTK release version            v.14.04.1
   int v2 = gtk_get_minor_version();
   int v3 = gtk_get_micro_version();
   sprintf(gtk_version,"%d.%02d.%02d",v1,v2,v3);
   printz("GTK version: %s \n",gtk_version);

   //  find and save the main window and pointer device                    //  v.13.08

   Fwindow = gtk_widget_get_window(Mwin);                                  //  gdk window for main window
   if (Fwindow) Fdisplay = gdk_window_get_display(Fwindow);
   if (Fwindow) Fscreen = gdk_window_get_screen(Fwindow);
   if (Fdisplay) Fmanager = gdk_display_get_device_manager(Fdisplay);
   if (Fmanager) Fpointer = gdk_device_manager_get_client_pointer(Fmanager);
   if (! Fscreen || ! Fpointer) {
      printz("GTK/GDK failure to find hardware %p %p %p %p %p \n",
                     Fwindow,Fdisplay,Fscreen,Fmanager,Fpointer);
      m_quit(0,0);
      exit(1);
   }

   //  check that necessary programs are installed

   fid = popen("exiftool -ver","r");                                       //  check exiftool version
   if (fid) {
      ii = fscanf(fid,"%s",buff);
      pclose(fid);                                                         //  bugfix: accept period or comma
      convSF(buff,exifver);
      printz("exiftool version: %.2f \n",exifver);
      if (exifver >= 8.60) Fexiftool = 1;
   }

   err = shell_quiet("which xdg-open");                                    //  check for xdg-open
   if (! err) Fxdgopen = 1;
   err = shell_quiet("which dcraw");                                       //  check for DCraw
   if (! err) Fdcraw = 1;
   err = shell_quiet("which ufraw");                                       //  check for UFraw
   if (! err) Fufraw = 1;
   err = shell_quiet("which rawtherapee");                                 //  check for Raw Therapee
   if (! err) Frawtherapee = 1;
   err = shell_quiet("which brasero");                                     //  check for Brasero
   if (! err) Fbrasero = 1;

   if (Fexiftool + Fxdgopen < 2) {                                         //  check mandatory dependencies
      strcpy(badnews,ZTX("Please install missing programs:"));
      if (! Fexiftool) strcat(badnews,"\n exiftool 8.6 or later");         //  refuse to start if any are missing
      if (! Fxdgopen) strcat(badnews,"\n xdg-utils");
      if (! Fdcraw) strcat(badnews,"\n DCraw");
      zmessageACK(Mwin,0,badnews);
      m_quit(0,0);
   }
   
   if (! Fufraw) printz("UFraw not installed \n");                         //  optional
   if (! Frawtherapee) printz("Raw Therapee not installed \n");
   if (! Fbrasero) printz("Brasero not installed \n");
   
   //  check for first time Fotoxx install or new release install
   
   if (Ffirsttime) {                                                       //  new fotoxx install                 v.14.04
      showz_html(quickstart_file);
      Prelease = zstrdup(Frelease);
   }

   if (! Ffirsttime && strNeq(Prelease,Frelease)) {                        //  Fotoxx release change
      zmessLogACK(Mwin,"fotoxx new release %s",Frelease);
      Prelease = zstrdup(Frelease);
      showz_textfile("doc","changelog");
   }

   //  wget is used to increment a web page counter.
   //  (no information goes in either direction).

   usagecounter += 1;
   if (usagecounter == 10) {                                               //  when usage counter reaches 10      v.14.05
      snprintf(command,ccc,"wget -b -T 20 -o /dev/null -O /dev/null "      //    count as a real user
                           "http://www.kornelix.com/fotoxx_users");
      err = system(command);
   }

   //  check for available fotoxx release > installed release
      
   fid = fopen(latest_release_file,"r");
   if (fid) {
      fgets_trim(Lrelease,20,fid,1);                                       //  latest release on web site
      fclose(fid);
      if (strnEqu(Lrelease,"Fotoxx",6)) {                                  //  legitimate
         printz("latest release: %s \n",Lrelease);
         if (strcmp(Frelease,Lrelease) < 0                                 //  notify if release < latest release
               && strNeq(Nrelease,Lrelease)) {
            zmessageACK(Mwin,0,release_notify,Lrelease);
            Nrelease = zstrdup(Lrelease);                                  //  only once - save notified release
         }
      }
   }
   snprintf(command,ccc,"wget -b -T 20 -o /dev/null -O %s %s",             //  start retrieval of latest release
               latest_release_file, kornelix_fotoxx_release);              //  (no wait, times out in 20 secs.)
   err = system(command);                                                  //  may return junk if failure
 
   //  delete fotoxx tempdir files if parent process is no longer running
   
   contx = 0;                                                              //  v.13.12
   while ((pp = command_output(contx,"find /tmp/fotoxx-*",0))) {
      pp2 = strchr(pp,'-');
      if (! pp2) continue;
      npid = atoi(pp2+1);                                                  //  pid of fotoxx owner process
      snprintf(procfile,20,"/proc/%d",npid);
      err = stat(procfile,&statb);
      if (! err) continue;                                                 //  pid is active, keep
      shell_quiet("rm -R -f -v %s",pp);                                    //  delete
   }

   //  set up temp directory /tmp/fotoxx-nnnnnn
   
   snprintf(PIDstring,11,"%06d",getpid());                                 //  get process PID 6-digits
   sprintf(tempdir,"/tmp/fotoxx-%s",PIDstring);
   err = mkdir(tempdir,0750);
   if (err) {
      zmessageACK(Mwin,0,"cannot create %s: %s",tempdir,strerror(errno));
      exit(1);
   }
   printz("tempdir: %s \n",tempdir);
   
   //  file name template for undo/redo files
   
   snprintf(URS_filename,200,"%s/undo_nn",tempdir);                        //  /tmp/fotoxx-nnnnnn/undo_nn

   //  check free memory and suggest image size limits

   parseprocfile("/proc/meminfo","MemFree:",&freememory,null);             //  get amount of free memory
   parseprocfile("/proc/meminfo","Cached:",&cachememory,null);
   freememory = (freememory + cachememory) / 1024;                         //  megabytes
   printz("free memory: %.0f MB \n",freememory);
   printz("image size limits for good performance: \n");                   //  v.13.04
   printz("  view: %.0f megapixels \n",(freememory-100)/6);                //  F + preview, 3 bytes/pixel each
   printz("  edit: %.0f megapixels \n",(freememory-100)/54);               //  + E0/E1/E3/ER, 12 bytes/pixel each

   //  update KB shortcuts if needed

   err = locale_filespec("user","KB-shortcuts",KBshortsU);                 //  user KB shortcuts
   if (err) {                                                              //  not found
      locale_filespec("data","KB-shortcuts",KBshortsI);
      shell_quiet("cp %s %s",KBshortsI,KBshortsU);                         //  copy installed KB shortcuts        v.14.07
      printz("keyboard shortcuts installed \n");
   }
   KBshortcuts_load();                                                     //  load KB shortcuts from file
   
   //  copy default data files to user directories if not already

   err = stat(pattern_dirk,&statb);                                        //  pattern files                      v.14.06
   if (err) {
      printz("copy default pattern files \n");
      mkdir(pattern_dirk,0750);
      shell_quiet("cp -n %s/patterns/* %s",get_zdatadir(),pattern_dirk);
   }
   
   err = stat(retouch_combo_dirk,&statb);                                  //  retouch combo settings
   if (err) {
      printz("copy default retouch_combo files \n");
      mkdir(retouch_combo_dirk,0750);
      shell_quiet("cp -n %s/retouch_combo/* %s",get_zdatadir(),retouch_combo_dirk);
   }
   
   err = stat(custom_kernel_dirk,&statb);                                  //  custom convolution kernels
   if (err) {
      printz("copy default custom_kernel files \n");
      mkdir(custom_kernel_dirk,0750);
      shell_quiet("cp -n %s/custom_kernel/* %s",get_zdatadir(),custom_kernel_dirk);
   }
   
   err = stat(favorites_dirk,&statb);                                      //  favorites menu
   if (err) {
      printz("copy default favorites menu \n");
      mkdir(favorites_dirk,0750);
      shell_quiet("cp -n %s/favorites/* %s",get_zdatadir(),favorites_dirk);
   }
   
   err = stat(tags_defined_file,&statb);                                   //  tags_defined file
   if (err) {
      printz("create default tags_defined file \n");
      shell_quiet("cp %s/tags_defined %s",get_zdatadir(),tags_defined_file);
   }

   snprintf(tonefile,200,"%s/slideshow-tone.oga",slideshow_dirk);          //  default slide show pause tone file
   err = stat(tonefile,&statb);
   if (err) {
      printz("create default slideshow pause tone file \n");
      shell_quiet("cp %s/slideshow-tone.oga %s",get_zdatadir(),tonefile);
   }

   //  miscellaneous

   Nwt = get_nprocs();                                                     //  get SMP CPU count
   if (Nwt <= 0) Nwt = 2;
   if (Nwt > max_threads) Nwt = max_threads;                               //  compile time limit
   printz("using %d threads \n",Nwt);

   mutex_init(&Fpixmap_lock,0);                                            //  setup lock for edit pixmaps
   mutex_init(&preload_lock,0);                                            //  setup lock for file preload function

   zdialog_inputs("load");                                                 //  load saved dialog inputs
   zdialog_positions("load");                                              //  load saved dialog positions

   exiftool_server(null);                                                  //  kill orphan exiftool process

   g_timeout_add(100,gtimefunc,0);                                         //  start periodic function (100 ms)   v.13.11

   //  create or update image index file                                   //  v.14.02
   //  cannot continue until completed OK

   if (Ffirsttime)                                                         //  if first startup, explain
      zmessageACK(Mwin,0,ftmess);                                          //    image file index process

   Findexdone = 0;                                                         //  insist on index rebuild            v.14.02
   while (! Findexdone) index_rebuild(0);                                  //    until successfully completed

   printz("RAW file types found: %s \n",myRAWtypes);                       //  list RAW types found               v.14.07

   Ffirsttime = 0;                                                         //  reset first time flag
   save_params();                                                          //  save parameters now

   //  set current directory/file from command line if present

   curr_dirk = zstrdup(topdirks[0]);                                       //  default 1st top image directory
   
   if (initial_file) {                                                     //  from command line
      pp = realpath(initial_file,0);                                       //  prepend directory if needed
      if (pp) {
         zfree(initial_file);
         initial_file = zstrdup(pp);
         zfree(pp);
      }
      else {
         printz("invalid file: %s \n",initial_file);                       //  invalid file path
         zfree(initial_file);
         initial_file = 0;
      }
   }

   if (initial_file) {
      ftype = image_file_type(initial_file);
      if (ftype == 0) {                                                    //  non-existent file
         printz("invalid file: %s \n",initial_file);
         zfree(initial_file);
         initial_file = 0;
      }
      else if (ftype == 1) {                                               //  is a directory
         if (curr_dirk) zfree(curr_dirk);
         curr_dirk = initial_file;
         initial_file = 0;
         Fdirk = 1;
      }
      else {
         if (curr_dirk) zfree(curr_dirk);
         curr_dirk = zstrdup(initial_file);                                //  set current directory from file
         pp = strrchr(curr_dirk,'/');
         if (pp) *pp = 0;
      }
   }
   
   if (initial_file)                                                       //  open initial file
      f_open(initial_file);
   
   else if (Fdirk) {                                                       //  open initial directory - gallery
      gallery(curr_dirk,"init");                                           //  v.13.01
      gallery(0,"paint");
      m_viewmode(0,"G"); 
   }
   
   else if (Fprev)                                                         //  start with previous file
      m_previous(0,0);
   
   else if (Frecent)                                                       //  start with recent files
      m_recentfiles(0,0);
   
   else if (Fnew)                                                          //  start with newly added files
      m_newfiles(0,0);                                                     //  v.13.07
   
   else if (Fblank) { /* do nothing */ }                                   //  start with blank window

   //  if no command line option, get startup option from user settings
   
   else if (strEqu(startdisplay,"recent"))                                 //  recent files gallery
      m_recentfiles(0,0);

   else if (strEqu(startdisplay,"newest"))                                 //  newest files gallery
      m_newfiles(0,0);

   else if (strEqu(startdisplay,"prevG")) {                                //  previous gallery
      m_previous(0,0);
      printf("prev gallery \n");
      gallery(curr_dirk,"init",0);
      gallery(0,"paint");
      m_viewmode(0,"G"); 
   }

   else if (strEqu(startdisplay,"prevI"))                                  //  previous image file                v.14.03
      m_previous(0,"first");

   else if (strEqu(startdisplay,"dirk")) {                                 //  given directory
      if (startdirk && *startdirk == '/') {
         if (curr_dirk) zfree(curr_dirk);
         curr_dirk = zstrdup(startdirk);
         navi::thumbsize = 128;
         gallery(curr_dirk,"init",0);                                      //  v.13.01
         gallery(0,"paint");
         m_viewmode(0,"G"); 
      }
   }

   else if (strEqu(startdisplay,"file"))                                   //  given image file
      f_open(startfile);

   if (startmenu) {                                                        //  startup menu on command line
      printz("startmenu: %s \n",startmenu);
      for (ii = 0; ii < Nmenus; ii++)                                      //  convert menu name to menu function
         if (strcasecmp(ZTX(startmenu),ZTX(menutab[ii].menu)) == 0) break;
      if (ii < Nmenus) menutab[ii].func(0,menutab[ii].arg);                //  call the menu function
   }

   zmainloop();                                                            //  mysterious, avoid blank window
   return 0;                                                               //  don't come back
}


/**************************************************************************/

//  Periodic function (100 milliseconds)
//  Avoid any thread usage of gtk/gdk functions.

int gtimefunc(void *arg)
{
   static int     domore = 0;
   
   if (Frefresh && Cdraw) {                                                //  window update needed             v.13.04
      if (mutex_trylock(&Fpixmap_lock) != 0) return 1;                     //  pixmaps locked, come back later
      mutex_unlock(&Fpixmap_lock);                                         //  (FWpaint will lock)
      gtk_widget_queue_draw(Cdraw);
   }
   
   if (zd_thread && zd_thread_event) {                                     //  send dialog event from thread   v.13.09
      zdialog_send_event(zd_thread,zd_thread_event);
      zd_thread_event = 0;
   }
   
   if (--domore > 0) return 1;                                             //  do rest 1/5 as frequently
   domore = 5;
   
   update_Fpanel();                                                        //  update panel information
   paint_busy();                                                           //  paint "BUSY" on window if something busy
   
   return 1;
}


/**************************************************************************/

//  Watch globals Fthreadbusy, Ffuncbusy, Findexbusy.
//  Display spinner if any busy flag > 0.
//  Add caller message if present.
//  Add percent complete if Fbusy_goal > 0.
//  window title bar: [x] - [_]  window title ...    ✻ caller message 99 % 

void paint_busy(cchar *message)
{
   #define GSFNORMAL GTK_STATE_FLAG_NORMAL

   static GtkWidget  *mlabel = 0, *plabel = 0, *spinner = 0;
   static char       mmess[100] = "", pmess[12];
   int               busy, percent;
   GdkRGBA           red = { 1, 0, 1, 1 };
   
   busy = Fthreadbusy + Ffuncbusy + Findexbusy;                            //  v.13.12
   
   if (! busy) {                                                           //  nothing busy
      if (! spinner) return;
      gtk_widget_destroy(spinner);
      if (plabel) gtk_widget_destroy(plabel);
      if (mlabel) gtk_widget_destroy(mlabel);
      spinner = plabel = mlabel = 0;
      *mmess = 0;
      return;
   }
   
   if (! spinner)
   {
      spinner = gtk_spinner_new();                                         //  add spinning icon to window
      gtk_widget_set_size_request(spinner,14,14);
      gtk_widget_override_color(spinner,GSFNORMAL,&red); 
      gtk_spinner_start(GTK_SPINNER(spinner));
      gtk_box_pack_start(GTK_BOX(Fpanel),spinner,0,0,8);

      if (message) {                                                       //  new message
         strncpy0(mmess,message,99);
         if (mlabel) gtk_widget_destroy(mlabel);
         mlabel = gtk_label_new(mmess);
         gtk_widget_override_font(mlabel,panelfont);
         gtk_box_pack_start(GTK_BOX(Fpanel),mlabel,0,0,0);
      }
      
      if (Fbusy_goal) {
         percent = 100 * (Fbusy_done / Fbusy_goal);                        //  add NN % 
         sprintf(pmess,"%02d %c",percent,'%');
         if (plabel) gtk_widget_destroy(plabel);
         plabel = gtk_label_new(pmess);
         gtk_widget_override_font(plabel,panelfont);
         gtk_box_pack_start(GTK_BOX(Fpanel),plabel,0,0,0);
      }
      
      gtk_widget_show_all(Fpanel);
   }

   if (Fbusy_goal) {
      percent = 100 * (Fbusy_done / Fbusy_goal);                           //  update NN %
      sprintf(pmess,"%02d %c",percent,'%');
      gtk_label_set_label(GTK_LABEL(plabel),pmess);
   }      

   return;
}


/**************************************************************************/

//  update F window top panel with current status information
//  called from timer function

void update_Fpanel()
{
   static double  time1 = 0, time2, cpu1 = 0, cpu2, cpuload;
   char           text1[300], text2[100];
   static char    ptext1[300] = "";
   int            faultrate, ww, hh, scale, bpc;
   static int     ftf = 1;
   double         file_MB = 1.0 / mega * curr_file_size;
   static cchar   *reduced, *areaactive, *dialogopen;
   static cchar   *funcbusy, *indexbusy, *menulock;

   if (ftf) {
      ftf = 0;
      reduced = ZTX("(reduced)");
      areaactive = ZTX("area active");
      dialogopen = ZTX("dialog open");
      funcbusy = ZTX("function busy");
      indexbusy = ZTX("file index busy");
      menulock = ZTX("menu locked");
   }

   if (FGW != 'F') return;

   *text1 = *text2 = 0;
   
   if (! time1) {
      time1 = get_seconds();
      cpu1 = jobtime();
   }

   time2 = get_seconds();                                                  //  compute process cpu load %
   if (time2 - time1 > 1.0) {                                              //    at 1 second intervals
      cpu2 = jobtime();
      cpuload = 100.0 * (cpu2 - cpu1) / (time2 - time1);
      time1 = time2;
      cpu1 = cpu2;
   }

   faultrate = pagefaultrate();                                            //  recent hard page fault rate        v.14.03

   sprintf(text1,"CPU %03.0f%c  PF %d",cpuload,'%',faultrate);             //  CPU 023%  PF 0
   
   if (curr_file && Fpxb)                                                  //  bugfix  v.13.02.1
   {
      if (E3pxm) {
         ww = E3pxm->ww;
         hh = E3pxm->hh;
      }
      else {
         ww = Fpxb->ww;
         hh = Fpxb->hh;
      }

      bpc = curr_file_bpc;

      snprintf(text2,99,"  %dx%dx%d",ww,hh,bpc);                           //  2345x1234x16 (preview) 1.56MB 45%
      strcat(text1,text2);
      if (CEF && CEF->Fpreview) strncatv(text1,300," ",reduced,0);         //  v.13.04
      sprintf(text2,"  %.2fMB",file_MB);
      strcat(text1,text2);
      scale = Mscale * 100 + 0.5;
      sprintf(text2,"  %d%c",scale,'%');
      strcat(text1,text2);
   
      if (URS_pos) {                                                       //  edit undo/redo stack depth 
         snprintf(text2,99,"  %s: %d",ZTX("edits"),URS_pos);
         strcat(text1,text2);
      }
   }

   if (sa_stat == 3) strncatv(text1,300,"  ",areaactive,0);
   if (zfuncs::zdialog_busy) strncatv(text1,300,"  ",dialogopen,0);

   if (Ffuncbusy) strncatv(text1,300,"  ",funcbusy,0);                     //  v.13.12
   if (Findexbusy) strncatv(text1,300,"  ",indexbusy,0);
   if (Fmenulock) strncatv(text1,300,"  ",menulock,0);
   
   if (*paneltext) {                                                       //  application text
      strcat(text1,"  ");
      strcat(text1,paneltext);
   }
   
   if (strEqu(text1,ptext1)) return;                                       //  if no change, don't output         v.13.10

   gtk_label_set_label(GTK_LABEL(Fpanlab),text1);                          //  update panel bar
   strcpy(ptext1,text1);

   return;
}


/**************************************************************************/

//  functions for main window event signals

int delete_event()                                                         //  main window closed
{
   int      yn;

   printz("main window delete event \n");
   if (checkpend("mods")) return 1;                                        //  allow user bailout              v.13.12
   if (zfuncs::zdialog_busy) {
      yn = zmessageYN(Mwin,ZTX("Kill active dialog?"));                    //  allow user bailout              v.14.01
      if (! yn) return 1;
   }
   quitxx();
   return 0;
}

int destroy_event()                                                        //  main window destroyed
{
   printz("main window destroy event \n");
   quitxx();
   return 0;
}

int state_event(GtkWidget *, GdkEvent *event)                              //  main window state changed
{
   int state = ((GdkEventWindowState *) event)->new_window_state;          //  track window fullscreen status
   if (state & GDK_WINDOW_STATE_FULLSCREEN) Ffullscreen = 1;
   else Ffullscreen = 0;
   return 0;
}


/**************************************************************************/

//  GTK3 "draw" function for F and W mode drawing windows.
//  Paint window when created, exposed, resized, or image modified (edited).
//  Update window image if scale change or window size change.
//  Otherwise do only the draw function (triggered from GTK).
//  Draw the image section currently within the visible window. 
//  May NOT be called from threads. See Fpaint2() for threads.

int FWpaint(GtkWidget *Cdraw, cairo_t *cr)
{
   GdkPixbuf      *Dpxb;
   uint8          *pixels;
   static int     pdww = 0, pdhh = 0;                                      //  prior window size
   float          wscale, hscale;
   int            fww, fhh;                                                //  current image size at 1x
   int            mww, mhh;                                                //  scaled image size
   int            centerx, centery; 
   int            quitnow = 0;
   int            refresh = 0;                                             //  flag, full image refesh needed
   int            mousex, mousey;                                          //  mouse position after zoom
   int            rs;
   float          magx, magy;                                              //  mouse drag, magnification ratios

   TRACE   

   if (Fshutdown) quitnow++;                                               //  shutdown underway
   if (Fblankwindow) quitnow++;                                            //  blank window
   if (! Cdraw) quitnow++;
   if (! gdkwin) quitnow++;
   if (! Cstate || ! Cstate->fpxb) quitnow++;                              //  no image

   if (quitnow) {
      Frefresh = 0;                                                        //  reset paint request
      return 0;
   }
   
   if (mutex_trylock(&Fpixmap_lock) != 0) return 0;                        //  cannot lock pixmaps (causes window blink)
   
   Dww = gdk_window_get_width(gdkwin);                                     //  (new) drawing window size
   Dhh = gdk_window_get_height(gdkwin);
   if (Dww < 20 || Dhh < 20) return 0;                                     //  too small
   
   if (Dww != pdww || Dhh != pdhh) {                                       //  window size changed
      refresh++;                                                           //  image refresh needed
      pdww = Dww; 
      pdhh = Dhh;
   }

   if (Frefresh) {                                                         //  image refresh needed
      if (FGW == 'F' && (E0pxm || E3pxm)) {                                //  insure window F
         PXB *pxb1;                                                        //  update Fpxb from E0/E3
         if (E3pxm) pxb1 = PXM_PXB_copy(E3pxm);                            //  use image now being edited
         else pxb1 = PXM_PXB_copy(E0pxm);                                  //  or use already edited image
         PXB_free(Cstate->fpxb);
         Cstate->fpxb = pxb1;
      }
      Frefresh = 0;                                                        //  reset image paint request
      refresh++;                                                           //  full image refresh needed
   }

   centerx = (Cstate->morgx + 0.5 * dww) / Cstate->mscale;                 //  center of window, image space
   centery = (Cstate->morgy + 0.5 * dhh) / Cstate->mscale;                 //  (before window or scale change)

   fww = Cstate->fpxb->ww;                                                 //  1x image size
   fhh = Cstate->fpxb->hh;

   if (Cstate->fzoom == 0) {                                               //  scale to fit window
      wscale = 1.0 * Dww / fww;
      hscale = 1.0 * Dhh / fhh;
      if (wscale < hscale) Cstate->mscale = wscale;                        //  use greatest ww/hh ratio
      else  Cstate->mscale = hscale;
      if (fww <= Dww && fhh <= Dhh && ! Fblowup) Cstate->mscale = 1.0;     //  small image 1x unless Fblowup
      zoomx = zoomy = 0;                                                   //  no zoom target
   }
   else Cstate->mscale = Cstate->fzoom;                                    //  scale to fzoom level
   
   mww = fww * Cstate->mscale;                                             //  scaled image size for window
   mhh = fhh * Cstate->mscale;
   
   dww = Dww;                                                              //  image fitting inside drawing window
   if (dww > mww) dww = mww;                                               //  image < window size
   dhh = Dhh;
   if (dhh > mhh) dhh = mhh;

   if (Cstate->mscale != Cstate->pscale) {                                 //  scale changed
      Cstate->morgx = Cstate->mscale * centerx - 0.5 * dww;                //  change origin to keep same center
      Cstate->morgy = Cstate->mscale * centery - 0.5 * dhh;                //  (subject to later rules)
      Cstate->pscale = Cstate->mscale;                                     //  remember scale
      refresh++;                                                           //  full image refresh needed
   }
   
   if (! Cstate->mpxb) refresh++;

   if (refresh) {                                                          //  image refresh needed
      if (Cstate->mpxb) PXB_free(Cstate->mpxb);
      if (Cstate->mscale == 1) Cstate->mpxb = PXB_copy(Cstate->fpxb);      //  fast 1x zoom 
      else Cstate->mpxb = PXB_rescale(Cstate->fpxb,mww,mhh);
   }

   if ((Mxdrag || Mydrag)) {                                               //  pan/scroll via mouse drag
      zoomx = zoomy = 0;                                                   //  no zoom target
      magx = 1.0 * (mww - dww) / dww;
      magy = 1.0 * (mhh - dhh) / dhh;                                      //  needed magnification of mouse movement
      if (magx < 1) magx = 1;                                              //  retain a minimum speed
      if (magy < 1) magy = 1;

      if (Fdragopt == 1) {
         Cstate->morgx -= round(Mwdragx);                                  //  same direction
         Cstate->morgy -= round(Mwdragy);
      }
      if (Fdragopt == 2) {
         Cstate->morgx += round(Mwdragx);                                  //  opposite direction
         Cstate->morgy += round(Mwdragy);
      }
      if (Fdragopt == 3) {
         Cstate->morgx -= round(Mwdragx * magx);                           //  same direction, magnified
         Cstate->morgy -= round(Mwdragy * magy);
      }
      if (Fdragopt == 4) {
         Cstate->morgx += round(Mwdragx * magx);                           //  opposite direction, magnified 
         Cstate->morgy += round(Mwdragy * magy);
      }
   }

   if (dww < Dww) Cstate->dorgx = 0.5 * (Dww - dww);                       //  if scaled image width < window width
   else Cstate->dorgx = 0;                                                 //    center image in window
   if (dhh < Dhh) Cstate->dorgy = 0.5 * (Dhh - dhh);
   else Cstate->dorgy = 0;

   if (zoomx || zoomy) {                                                   //  requested zoom center
      Cstate->morgx = Cstate->mscale * zoomx - 0.5 * dww;                  //  corresp. window position within image
      Cstate->morgy = Cstate->mscale * zoomy - 0.5 * dhh;
   }
   
   if (Cstate->morgx < 0) Cstate->morgx = 0;                               //  maximize image within window
   if (Cstate->morgy < 0) Cstate->morgy = 0;                               //  (no unused margins)
   if (Cstate->morgx + dww > mww) Cstate->morgx = mww - dww;
   if (Cstate->morgy + dhh > mhh) Cstate->morgy = mhh - dhh;
   
   if (zoomx || zoomy) {                                                   //  zoom target
      mousex = zoomx * Cstate->mscale - Cstate->morgx + Cstate->dorgx;
      mousey = zoomy * Cstate->mscale - Cstate->morgy + Cstate->dorgy;     //  mouse pointer follows target
      move_pointer(Cdraw,mousex,mousey);
      zoomx = zoomy = 0;                                                   //  reset zoom target
   }
   
   if (zddarkbrite) darkbrite_paint();                                     //  update dark/bright pixels
   
   pixels = PXBpix(Cstate->mpxb,Cstate->morgx,Cstate->morgy);              //  mpxb image section within drawing window
   rs = Cstate->mpxb->rs;                                                  //  (morgx/y, dww/hh, rowstride rs)
   Dpxb = gdk_pixbuf_new_from_data(pixels,GDKRGB,0,8,dww,dhh,rs,0,0);      //  make a pixbuf from section

   gdk_cairo_set_source_pixbuf(cr,Dpxb,Cstate->dorgx,Cstate->dorgy);       //  paint mpxb image section
   cairo_paint(cr);
   g_object_unref(Dpxb);

   mwcr = cr;                                                              //  cairo context for draw funcs below
   cairo_set_line_width(mwcr,1);

   if (Cstate == &Fstate) {                                                //  view mode image
      if (Ntoplines) draw_toplines(1);                                     //  draw line overlays
      if (gridsettings[currgrid][GON]) draw_gridlines();                   //  draw grid lines
      if (Ntoptext) draw_toptext();                                        //  draw text strings
      if (Ntopcircles) draw_topcircles();                                  //  draw circles
      if (Fshowarea) sa_show(1);                                           //  draw select area outline
   }

   mwcr = 0;                                                               //  cairo context invalid
   mutex_unlock(&Fpixmap_lock);                                            //  unlock pixmaps
   
   if (refresh) 
      if (zdbrdist) m_show_brdist(0,0);                                    //  update brightness distribution

   return 0;
}


/**************************************************************************/

//  Cause (modified) output image to get repainted soon.
//  Same as FWpaint() but delayed. MAY be called from threads.
//  FWpaint() will be called by gtimefunc() next timer cycle.

void Fpaint2()
{
   Frefresh++;
   return;
}


//  Repaint modified image synchrounously. 
//  May NOT be called from threads.

void Fpaintnow()                                                           //  v.13.04
{
   cairo_t     *ct;                                                        //  do not use mwcr

   TRACE   

   if (! Cdraw) return;
   if (! gdkwin) return;
   if (! Cstate || ! Cstate->fpxb) return;                                 //  no image

   while (mutex_trylock(&Fpixmap_lock) != 0) zsleep(0.02);                 //  wait for thread to unlock
   mutex_unlock(&Fpixmap_lock);                                            //  (FWpaint will lock)

   gdk_window_set_background_rgba(gdkwin,&GDKdark);                        //  background for F and W view modes

   Frefresh++;
   ct = gdk_cairo_create(gdkwin);
   FWpaint(Cdraw,ct);
   cairo_destroy(ct);

   return;
}


//  Update a section of fpxb and mpxb from an updated section of E3pxm,
//  then update the corresponding section of the drawing window.
//  This avoids a full image refresh, E3pxm > fpxb > mpxb > drawing window.
//  px3, py3, ww3, hh3: modified section within E3pxm to be propagated.
//  May NOT be called from threads.

void Fpaint3(int px3, int py3, int ww3, int hh3)
{
   TRACE   
   PXM_PXB_update(E3pxm,Fpxb,px3,py3,ww3,hh3);                             //  E3pxm > Fpxb, both 1x scale 
   PXB_PXB_update(Fpxb,Mpxb,px3,py3,ww3,hh3);                              //  Fpxb > Mpxb, scaled up or down
   Fpaint4(px3,py3,ww3,hh3);                                               //  update drawing window from Mpxb
   return;
}


//  Repaint a section of the mpxb image in the visible window.
//  px3, py3, ww3, hh3: area to be repainted (in 1x image space).
//  May NOT be called from threads.

void Fpaint4(int px3, int py3, int ww3, int hh3) 
{
   GdkPixbuf   *pixbuf;
   uint8       *pixels;
   int         px2, py2, ww2, hh2, rs;
   int         crflag = 0;
   
   TRACE   

   if (! Cdraw) return;
   if (! gdkwin) return;
   if (! Cstate || ! Cstate->fpxb) return;                                 //  no image

   px2 = Mscale * px3 - 2;                                                 //  1x image space to mpxb space
   py2 = Mscale * py3 - 2;                                                 //  (expanded)
   ww2 = Mscale * ww3 + 2 / Mscale + 4;
   hh2 = Mscale * hh3 + 2 / Mscale + 4;

   if (px2 + ww2 > Mpxb->ww) ww2 = Mpxb->ww - px2;                         //  avoid overshoot
   if (py2 + hh2 > Mpxb->hh) hh2 = Mpxb->hh - py2;
   
   if (px2 < Morgx) {                                                      //  reduce to currently visible window
      ww2 = ww2 - (Morgx - px2);
      px2 = Morgx;
   }

   if (py2 < Morgy) {
      hh2 = hh2 - (Morgy - py2);
      py2 = Morgy;
   }

   if (ww2 <= 0 || hh2 <= 0) return;                                       //  completely outside visible window
   
   if (! mwcr) {
      mwcr = gdk_cairo_create(gdkwin);                                     //  create cairo context if not already
      crflag = 1;
   }

   pixels = PXBpix(Mpxb,px2,py2);
   rs = Mpxb->rs;                                                          //  row stride
   pixbuf = gdk_pixbuf_new_from_data(pixels,GDKRGB,0,8,ww2,hh2,rs,0,0);    //  pixbuf for image within window

   px2 = px2 - Morgx + Dorgx;                                              //  section position in window
   py2 = py2 - Morgy + Dorgy;

   gdk_cairo_set_source_pixbuf(mwcr,pixbuf,px2,py2);                       //  draw area to window
   cairo_paint(mwcr);
   g_object_unref(pixbuf);
   
   if (Fshowarea) { 
      px3 = (px2 - Dorgx + Morgx) / Mscale;                                //  back to image scale, expanded
      py3 = (py2 - Dorgy + Morgy) / Mscale;
      ww3 = ww2 / Mscale + 2;
      hh3 = hh2 / Mscale + 2;
      sa_show_rect(px3,py3,ww3,hh3);                                       //  refresh select area outline
   }

   if (crflag) {
      cairo_destroy(mwcr);
      mwcr = 0;
   }

   return;
}


/**************************************************************************/

//  mouse event function - capture buttons and drag movements

void mouse_event(GtkWidget *widget, GdkEventButton *event, void *)
{
   void mouse_convert(int xpos1, int ypos1, int &xpos2, int &ypos2);

   int            button, time, type, scroll;
   static int     bdtime = 0, butime = 0, mbusy = 0;
   static int     mdragx0, mdragy0;
   
   if (! Cdraw) return;
   if (! gdkwin) return;
   if (! Cstate || ! Cstate->fpxb) return;                                 //  no image
   
   type = event->type;
   button = event->button;                                                 //  button, 1/2/3 = left/center/right
   time = event->time;
   Mwxposn = event->x;                                                     //  mouse position in window
   Mwyposn = event->y;
   mouse_convert(Mwxposn,Mwyposn,Mxposn,Myposn);                           //  convert to image space

   scroll = ((GdkEventScroll *) event)->direction;                         //  scroll wheel event
   
   KBcontrolkey = KBshiftkey = KBaltkey = 0;
   if (event->state & GDK_CONTROL_MASK) KBcontrolkey = 1;                  //  v.13.08
   if (event->state & GDK_SHIFT_MASK) KBshiftkey = 1;
   if (event->state & GDK_MOD1_MASK) KBaltkey = 1;

   if (type == GDK_SCROLL) {                                               //  scroll wheel = zoom 
      zoomx = Mxposn;                                                      //  zoom center = mouse position
      zoomy = Myposn;
      if (scroll == GDK_SCROLL_UP) m_zoom(null,"in");
      if (scroll == GDK_SCROLL_DOWN) m_zoom(null,"out");
      return;
   }

   if (type == GDK_MOTION_NOTIFY) {
      if (mbusy) return;                                                   //  discard excess motion events
      mbusy++;
      zmainloop();
      mbusy = 0;
   }

   if (type == GDK_BUTTON_PRESS) {                                         //  button down
      Mdrag++;                                                             //  possible drag start
      bdtime = time;                                                       //  time of button down
      Mbutton = button;
      mdragx0 = Mwxposn;                                                   //  window position at button down
      mdragy0 = Mwyposn;
      Mxdown = Mxposn;                                                     //  image position at button down
      Mydown = Myposn;
      Mxdrag = Mydrag = 0;
   }

   if (type == GDK_MOTION_NOTIFY && Mdrag) {                               //  drag underway
      Mwdragx = Mwxposn - mdragx0;                                         //  drag increment, window space
      Mwdragy = Mwyposn - mdragy0;
      mdragx0 = Mwxposn;                                                   //  new drag origin = current position
      mdragy0 = Mwyposn;
      Mxdrag = Mxposn;                                                     //  drag position, image space
      Mydrag = Myposn;
   }

   if (type == GDK_BUTTON_RELEASE) {                                       //  button up
      Mxclick = Myclick  = 0;                                              //  reset click status
      butime = time;                                                       //  time of button up
      if (butime - bdtime < 400) {                                         //  less than 0.4 secs down
         if (Mxposn == Mxdown && Myposn == Mydown) {                       //       and not moving
            if (Mbutton == 1) LMclick++;                                   //  left mouse click
            if (Mbutton == 3) RMclick++;                                   //  right mouse click
            Mxclick = Mxdown;                                              //  click = button down position
            Myclick = Mydown;
            if (button == 2) {                                             //  center button click
               zoomx = Mxposn;                                             //  re-center at mouse (Doriano)
               zoomy = Myposn;
               gtk_widget_queue_draw(Cdraw);
            }
         }
      }
      Mxdown = Mydown = Mxdrag = Mydrag = Mdrag = Mbutton = 0;             //  forget buttons and drag
   }

   Fmousemain = 1;                                                         //  mouse acts on main window
   if (Mcapture) Fmousemain = 0;                                           //  curr. function handles mouse 
   if (mouseCBfunc) Fmousemain = 0;                                        //  mouse owned by callback function
   if (KBcontrolkey) Fmousemain = 1;                                       //  mouse acts on main window
   
   if (mutex_trylock(&Fpixmap_lock) != 0) return;                          //  block mouse if image transition
   mutex_unlock(&Fpixmap_lock);                                            //  moved here

   if (mouseCBfunc && ! Fmousemain) {                                      //  pass to handler function
      if (mbusy) return;
      mbusy++;                                                             //  stop re-entrance
      (* mouseCBfunc)();
      mbusy = 0;     
      Fmousemain = 1;                                                      //  click/drag params are processed
   }                                                                       //    if not reset by handler func.
   
   if (! Fmousemain) return;                                               //  curr. function handles mouse
   
   if (LMclick) {                                                          //  left click = zoom request
      LMclick = 0;
      zoomx = Mxclick;                                                     //  zoom center = mouse
      zoomy = Myclick;
      m_zoom(null,"in");
   }

   if (RMclick) {                                                          //  right click
      RMclick = 0;
      if (Cstate->fzoom) {                                                 //  if zoomed image, reset to fit window
         zoomx = zoomy = 0;
         m_zoom(null,"fit");
      }
      else if (curr_file && FGW == 'F')
         image_Rclick_popup(curr_file_posn);                               //  image right-click popup menu
   }
   
   if ((Mxdrag || Mydrag))                                                 //  drag = scroll by mouse
      gtk_widget_queue_draw(Cdraw);

   return;
}


//  convert mouse position from window space to image space

void mouse_convert(int xpos1, int ypos1, int &xpos2, int &ypos2)
{
   xpos2 = (xpos1 - Cstate->dorgx + Cstate->morgx) / Cstate->mscale + 0.5;
   ypos2 = (ypos1 - Cstate->dorgy + Cstate->morgy) / Cstate->mscale + 0.5;

   if (xpos2 < 0) xpos2 = 0;                                               //  if outside image put at edge
   if (ypos2 < 0) ypos2 = 0;

   if (E3pxm) { 
      if (xpos2 >= E3pxm->ww) xpos2 = E3pxm->ww-1;
      if (ypos2 >= E3pxm->hh) ypos2 = E3pxm->hh-1;
   }
   else {
      if (xpos2 >= Cstate->fpxb->ww) xpos2 = Cstate->fpxb->ww-1;
      if (ypos2 >= Cstate->fpxb->hh) ypos2 = Cstate->fpxb->hh-1;
   }
   
   return;
}


/**************************************************************************/

//  set new image zoom level or magnification
//  zoom:   "in"        zoom in in steps
//          "out"       zoom out in steps
//          "fit"       zoom out to fit window
//          "100"       toggle 100% and fit window

void m_zoom(GtkWidget *, cchar *zoom)
{
   int         fww, fhh;
   float       scalew, scaleh, fitscale, fzoom2, zratio = 1;
   float       Rzoom, pixels;
   
   if (! Cstate || ! Cstate->fpxb) return;
   if (! Cdraw) return;
   if (! gdkwin) return;
   if (! Cstate || ! Cstate->fpxb) return;                                 //  no image

   Rzoom = Cstate->fzoom;                                                  //  current zoom ratio
   
   if (E3pxm) {
      fww = E3pxm->ww;                                                     //  1x image size
      fhh = E3pxm->hh;
   }
   else  {
      fww = Cstate->fpxb->ww;
      fhh = Cstate->fpxb->hh;
   }

   if (fww > Dww || fhh > Dhh) {                                           //  get window fit scale
      scalew = 1.0 * Dww / fww;
      scaleh = 1.0 * Dhh / fhh;
      if (scalew < scaleh) fitscale = scalew;
      else fitscale = scaleh;                                              //  window/image, < 1.0
   }
   else fitscale = 1.0;                                                    //  if image < window use 100%
   
   if (strEqu(zoom,"Zoom+")) zoom = "in";                                  //  menu button: + = zoom in
   if (strEqu(zoom,"Zoom-")) zoom = "fit";                                 //               - = fit window

   if (strEqu(zoom,"fit")) Cstate->fzoom = 0;                              //  zoom to fit window

   if (strEqu(zoom,"100")) {
      if (Cstate->fzoom != 0) Cstate->fzoom = 0;                           //  toggle 100% and fit window
      else  Cstate->fzoom = 1;
   }

   if (strstr("in out",zoom))                                              //  caller: zoom in or out
   {
      if (! Cstate->fzoom) Cstate->fzoom = fitscale;                           //  current zoom scale
      for (fzoom2 = 0.125; fzoom2 < 4.0; fzoom2 *= zoomratio)              //  find nearest natural ratio
         if (Cstate->fzoom < fzoom2 * sqrt(zoomratio)) break;
      if (strEqu(zoom,"in")) zratio = zoomratio;                           //  zoom in, make image larger
      if (strEqu(zoom,"out")) zratio = 1.0 / zoomratio;                    //  zoom out, make image smaller
      Cstate->fzoom = fzoom2 * zratio;

      if (Cstate->fzoom > 0.124 && Cstate->fzoom < 0.126)                  //  hit these ratios exactly
         Cstate->fzoom = 0.125;
      if (Cstate->fzoom > 0.24  && Cstate->fzoom < 0.26)  
         Cstate->fzoom = 0.25;
      if (Cstate->fzoom > 0.49  && Cstate->fzoom < 0.51)  
         Cstate->fzoom = 0.50;
      if (Cstate->fzoom > 0.99  && Cstate->fzoom < 1.01)  
         Cstate->fzoom = 1.00;
      if (Cstate->fzoom > 1.99  && Cstate->fzoom < 2.01)  
         Cstate->fzoom = 2.00;
      if (Cstate->fzoom > 3.99) Cstate->fzoom = 4.0;                       //  max. allowed zoom
      if (Cstate->fzoom < fitscale) Cstate->fzoom = 0;                     //  image < window
   }
   
   if (FGW == 'W') {                                                       //  optimize for world map
      if (strEqu(zoom,"in")) Cstate->fzoom = 1.0;                          //  zoom to 100% directly
      if (strEqu(zoom,"out")) Cstate->fzoom = 0.0;                         //  zoom out = fit window directly     v.13.10
      if (Cstate->fzoom == 1.0 && Rzoom == 1.0) {
         gtk_widget_queue_draw(Cdraw);                                     //  if already, move mouse pointer only
         return;
      }
   }

   if (Cstate->fzoom > 1.0) {                                              //  limit image in window to 1 gigabyte
      pixels = Cstate->fzoom * fww * Cstate->fzoom * fhh;                  //  (333 megapixels x 3 bytes/pixel)
      if (pixels > 333 * mega) {                                           //  if size is at maximum,
         Cstate->fzoom = Rzoom;                                            //    move mouse pointer only
         gtk_widget_queue_draw(Cdraw);
         return;
      }
   }

   if (! Cstate->fzoom) zoomx = zoomy = 0;                                 //  no requested zoom center
   Fpaint2();                                                              //  refresh window
   return;
}


/**************************************************************************/

//  function for dialogs to call to send KB keys for processing by main app

void KBstate(GdkEventKey *event, int state)
{
   if (state) KBpress(0,event,0);
   else KBrelease(0,event,0);
   return;
}


//  keyboard event functions
//  GDK key symbols: /usr/include/gtk-3.0/gdk/gdkkeysyms.h

namespace trimrotate { void KBfunc(int key); }                             //  keyboard functions called from here
namespace perspective { void KBfunc(int key); }
namespace mashup { void KBfunc(int key); }

int KBpress(GtkWidget *win, GdkEventKey *event, void *)                    //  prevent propagation of key-press
{                                                                          //    events to menu buttons
   KBkey = event->keyval;

   KBcontrolkey = KBshiftkey = KBaltkey = 0;
   if (event->state & GDK_CONTROL_MASK) KBcontrolkey = 1;                  //  v.13.08
   if (event->state & GDK_SHIFT_MASK) KBshiftkey = 1;
   if (event->state & GDK_MOD1_MASK) KBaltkey = 1;

   return 1;
}

int KBrelease(GtkWidget *win, GdkEventKey *event, void *)                  //  some key was released
{
   int      ii, jj, cc;
   char     matchkey[20];
   
   TRACE   

   KBcontrolkey = KBshiftkey = KBaltkey = 0;
   if (event->state & GDK_CONTROL_MASK) KBcontrolkey = 1;                  //  v.13.08
   if (event->state & GDK_SHIFT_MASK) KBshiftkey = 1;
   if (event->state & GDK_MOD1_MASK) KBaltkey = 1;
   
   KBkey = event->keyval;

   if (! KBcontrolkey && ! KBshiftkey && ! KBaltkey) {                     //  F/G/W keys set corresp. window mode
      if (KBkey == GDK_KEY_f) { m_viewmode(0,"F"); goto KBend; }           //  Ctrl/Alt + F/G/W can also be used for shortcuts
      if (KBkey == GDK_KEY_g) { m_viewmode(0,"G"); goto KBend; }
      if (KBkey == GDK_KEY_w) { m_viewmode(0,"W"); goto KBend; }
   }

   if (KBcapture) return 1;                                                //  let current function handle it

//  reserved shortcuts (not user configurable)
   
   if (KBkey == GDK_KEY_F1) {                                              //  F1  >>  user guide
      showz_userguide(F1_help_topic);                                      //  show topic if there, or page 1
      goto KBend;
   }
   
   if (KBkey == GDK_KEY_F11) {                                             //  F11  >>  toggle main window
      if (! Ffullscreen) win_fullscreen(0);                                //    to full-screen mode and back
      else win_unfullscreen();
      if (Fslideshow) ss_escape = 1;                                       //  quit slideshow if active
      goto KBend;
   }

   if (KBkey == GDK_KEY_Escape && Ffullscreen) {                           //  ESC key >> exit full screen
      win_unfullscreen();
      if (Fslideshow) ss_escape = 1;                                       //  quit slideshow if active
      goto KBend;
   }

   if (FGW == 'G') {                                                       //  window mode G
      navi::KBrelease(win,event,0);                                        //  pass to gallery KB function
      goto KBend;                                                          //  (keys below not for gallery)
   }

   if (KBkey == GDK_KEY_z || KBkey == GDK_KEY_Z) {                         //  Z key
      if (! KBcontrolkey) m_zoom(null,"100");                              //  if no CTRL, toggle zoom 1x
      goto KBend;
   }

   if (FGW == 'W') goto KBend;                                             //  window mode W, no other KB events

   if (KBkey == GDK_KEY_plus) m_zoom(null,"in");                           //  + key  >>  zoom in
   if (KBkey == GDK_KEY_minus) m_zoom(null,"fit");                         //  - key  >>  fit to window
   if (KBkey == GDK_KEY_equal) m_zoom(null,"in");                          //  = key: same as +
   if (KBkey == GDK_KEY_KP_Add) m_zoom(null,"in");                         //  keypad +
   if (KBkey == GDK_KEY_KP_Subtract) m_zoom(null,"fit");                   //  keypad -

   if (Fslideshow) {                                                       //  slide show mode
      if (KBkey == GDK_KEY_Left) ss_Larrow = 1;                            //  arrow keys = prev/next image
      if (KBkey == GDK_KEY_Right) ss_Rarrow = 1;
      if (KBkey == GDK_KEY_space) ss_spacebar = 1;                         //  spacebar = pause/resume
      if (KBkey == GDK_KEY_B || KBkey == GDK_KEY_b) ss_Bkey = 1;           //  B-key = blank screen + pause
      goto KBend;
   }

   if (CEF && CEF->menufunc == m_trimrotate) {                             //  trim_rotate active, pass KB key    v.14.06
      trimrotate::KBfunc(KBkey);
      goto KBend;
   }
   
   if (CEF && CEF->menufunc == m_mashup) {                                 //  mashup active, pass KB key         v.13.12
      mashup::KBfunc(KBkey);
      goto KBend;
   }
   
   if (CEF && CEF->menufunc == m_perspective) {                            //  perspective active, pass KB key    v.14.06
      perspective::KBfunc(KBkey);
      goto KBend;
   }

   if (KBkey == GDK_KEY_Left) m_prev(0,0);                                 //  arrow keys = prev/next image
   if (KBkey == GDK_KEY_Right) m_next(0,0);
   if (KBkey == GDK_KEY_Page_Up) m_prev(0,0);                              //  page up/down = prev/next image
   if (KBkey == GDK_KEY_Page_Down) m_next(0,0);

   if (KBkey == GDK_KEY_Delete) m_trash(0,0);                              //  delete  >>  trash

//  look for configurable shortcut keys in shortcut table

   if (KBkey >= GDK_KEY_F1 && KBkey <= GDK_KEY_F9) {                       //  input key is F1 to F9
      ii = KBkey - GDK_KEY_F1;
      strcpy(matchkey,"F1");
      matchkey[1] += ii;
   }
   
   else if (KBkey > 255) goto KBend;                                       //  not a single letter or digit 

   else {
      *matchkey = 0;                                                       //  build input key combination
      if (KBcontrolkey) strcat(matchkey,"Ctrl+");                          //  [Ctrl+] [Alt+] [Shift+] key
      if (KBaltkey) strcat(matchkey,"Alt+");
      if (KBshiftkey) strcat(matchkey,"Shift+");
      cc = strlen(matchkey);
      matchkey[cc] = KBkey;
      matchkey[cc+1] = 0;
   }
   
   for (ii = 0; ii < Nshortcuts; ii++)                                     //  convert key combination to menu name
      if (strcasecmp(matchkey,shortcutkey[ii]) == 0) break;
   
   if (ii < Nshortcuts) {                                                  //  convert menu name to menu function
      for (jj = 0; jj < Nmenus; jj++)
         if (strcasecmp(shortcutmenu[ii],ZTX(menutab[jj].menu)) == 0) break;
      if (jj < Nmenus) menutab[jj].func(0,menutab[jj].arg);                //  call the menu function
   }

KBend:
   KBkey = 0;
   return 1;
}


/**************************************************************************/

//  set the main window to fullscreen status 
//  (with no menu or panel)

void win_fullscreen(int hidemenu)                                          //  v.14.06
{
   if (FGW == 'F' && hidemenu) {                                           //  if F window, hide menu and panel
      gtk_widget_hide(Fmenu);
      gtk_widget_hide(Fpanel);
   }

   gtk_window_fullscreen(MWIN);

   for (int ii = 0; ii < 20; ii++) {                                       //  wait for window manager to act
      zmainloop();
      if (Ffullscreen) break;
      zsleep(0.1);
   }

   return;
}


//  restore window to former size and restore menu etc.

void win_unfullscreen()
{
   gtk_window_unfullscreen(MWIN);                                          //  restore old window size
   gtk_widget_show(Fmenu);
   gtk_widget_show(Fpanel);
   Ffullscreen = 0;
   return;
}


/**************************************************************************/

//  draw a pixel using foreground color.
//  px, py are image space.

void draw_pixel(int px, int py)
{
   int               qx, qy, crflag = 0;
   static int        pqx, pqy;
   static uint8      pixel[12];                                            //  2x2 block of pixels
   static GdkPixbuf  *pixbuf1 = 0, *pixbuf4 = 0;
   
   if (! Cdraw) return;
   if (! gdkwin) return;
   if (! Cstate || ! Cstate->fpxb) return;                                 //  no image

   if (! pixbuf1) {
      pixbuf1 = gdk_pixbuf_new_from_data(pixel,GDKRGB,0,8,1,1,3,0,0);      //  1x1 pixels
      pixbuf4 = gdk_pixbuf_new_from_data(pixel,GDKRGB,0,8,2,2,6,0,0);      //  2x2 pixels
   }

   qx = Mscale * px - Morgx;                                               //  image to window space
   qy = Mscale * py - Morgy;

   if (qx == pqx && qy == pqy) return;                                     //  avoid redundant points

   pqx = qx;
   pqy = qy;
   
   if (qx < 0 || qx > dww-2) return;                                       //  keep off image edges
   if (qy < 0 || qy > dhh-2) return;
   
   if (! mwcr) {
      mwcr = gdk_cairo_create(gdkwin);                                     //  create cairo context if not already
      crflag = 1;
   }

   if (Mscale < 1.99) {                                                    //  write 1x1 pixels
      pixel[0] = fg_color[0];
      pixel[1] = fg_color[1];
      pixel[2] = fg_color[2];

      gdk_cairo_set_source_pixbuf(mwcr,pixbuf1,qx+Dorgx,qy+Dorgy);
      cairo_paint(mwcr);
   }
   
   else {                                                                  //  write 2x2 fat pixels
      pixel[0] = pixel[3] = pixel[6] = pixel[9] = fg_color[0];
      pixel[1] = pixel[4] = pixel[7] = pixel[10] = fg_color[1];
      pixel[2] = pixel[5] = pixel[8] = pixel[11] = fg_color[2];

      gdk_cairo_set_source_pixbuf(mwcr,pixbuf4,qx+Dorgx,qy+Dorgy);
      cairo_paint(mwcr);
   }

   if (crflag) {
      cairo_destroy(mwcr);
      mwcr = 0;
   }

   return;
}


//  erase one drawn pixel - restore from window image mpxb.
//  px, py are image space.

void erase_pixel(int px, int py)
{
   static PXB     *pxb2 = 0;
   static uint8   *pix1, *pix2;
   static int     rs2, pqx, pqy;
   int            qx, qy, crflag = 0;

   if (! Cdraw) return;
   if (! gdkwin) return;
   if (! Cstate || ! Cstate->fpxb) return;                                 //  no image
   
   if (! pxb2) {
      pxb2 = PXB_make(2,2);                                                //  2x2 pixbuf
      pix2 = pxb2->pixels;
      rs2 = pxb2->rs;
   }

   qx = Mscale * px;                                                       //  image to window space
   qy = Mscale * py;

   if (qx == pqx && qy == pqy) return;                                     //  avoid same target pixel

   pqx = qx;
   pqy = qy;
   
   if (qx < 0 || qx > Mpxb->ww-2) return;                                  //  pixel outside scaled image
   if (qy < 0 || qy > Mpxb->hh-2) return;
   
   if (qx < Morgx || qx > Morgx + dww-2) return;                           //  pixel outside drawing window
   if (qy < Morgy || qy > Morgy + dhh-2) return;

   if (! mwcr) {
      mwcr = gdk_cairo_create(gdkwin);                                     //  create cairo context if not already
      crflag = 1;
   }

   pix1 = PXBpix(Mpxb,qx,qy);                                              //  source pixels, 2x2 block
   memcpy(pix2,pix1,6);                                                    //  copy 2 upper pixels, 6 bytes
   memcpy(pix2+rs2,pix1+Mpxb->rs,6);                                       //  copy 2 lower pixels

   qx = qx - Morgx + Dorgx;                                                //  target pixel in window
   qy = qy - Morgy + Dorgy;
   gdk_cairo_set_source_pixbuf(mwcr,pxb2->pixbuf,qx,qy);
   cairo_paint(mwcr);

   if (crflag) {
      cairo_destroy(mwcr);
      mwcr = 0;
   }

   return;
}


/**************************************************************************/

//  draw line.
//  coordinates are image space.

void draw_line(int x1, int y1, int x2, int y2)
{
   float       px1, py1, px2, py2;
   int         crflag = 0;

   if (! Cdraw) return;
   if (! gdkwin) return;
   if (! Cstate || ! Cstate->fpxb) return;                                 //  no image

   px1 = Mscale * x1 - Morgx + Dorgx;                                      //  image to window space
   py1 = Mscale * y1 - Morgy + Dorgy;
   px2 = Mscale * x2 - Morgx + Dorgx;
   py2 = Mscale * y2 - Morgy + Dorgy;
   
   if (! mwcr) {
      mwcr = gdk_cairo_create(gdkwin);                                     //  create cairo context if not already
      cairo_set_line_width(mwcr,1);
      crflag = 1;
   }

   cairo_set_source_rgb(mwcr,1,1,1);
   cairo_move_to(mwcr,px1,py1);
   cairo_line_to(mwcr,px2,py2);
   cairo_stroke(mwcr);
   
   if (crflag) {
      cairo_destroy(mwcr);
      mwcr = 0;
   }

   return;
}


//  draw dotted line.
//  coordinates are image space.

void draw_dotline(int x1, int y1, int x2, int y2)
{
   float       px1, py1, px2, py2;
   double      dashes[2] = { 4, 4 };
   int         crflag = 0;
   
   if (! Cdraw) return;
   if (! gdkwin) return;
   if (! Cstate || ! Cstate->fpxb) return;                                 //  no image

   px1 = Mscale * x1 - Morgx + Dorgx;                                      //  image to window space
   py1 = Mscale * y1 - Morgy + Dorgy;
   px2 = Mscale * x2 - Morgx + Dorgx;
   py2 = Mscale * y2 - Morgy + Dorgy;
   
   if (px1 > Dww-2) px1 = Dww-2;
   if (py1 > Dhh-2) py1 = Dhh-2;
   if (px2 > Dww-2) px2 = Dww-2;
   if (py2 > Dhh-2) py2 = Dhh-2;
   
   if (! mwcr) {
      mwcr = gdk_cairo_create(gdkwin);                                     //  create cairo context if not already
      cairo_set_line_width(mwcr,1);
      crflag = 1;
   }
   else cairo_save(mwcr);

   cairo_set_source_rgb(mwcr,1,1,1);
   cairo_set_dash(mwcr,dashes,2,0);
   cairo_move_to(mwcr,px1,py1);
   cairo_line_to(mwcr,px2,py2);
   cairo_stroke(mwcr);
   
   if (crflag) {
      cairo_destroy(mwcr);
      mwcr = 0;
   }
   else cairo_restore(mwcr);

   return;
}


//  erase line. refresh line path from mpxb window image.
//  double line width is erased.
//  coordinates are image space.

void erase_line(int x1, int y1, int x2, int y2)
{
   float       pxm, pym, slope;
   int         crflag = 0;

   if (! Cdraw) return;
   if (! gdkwin) return;
   if (! Cstate || ! Cstate->fpxb) return;                                 //  no image

   if (! mwcr) {
      mwcr = gdk_cairo_create(gdkwin);                                     //  create cairo context if not already
      crflag = 1;
   }

   if (abs(y2 - y1) > abs(x2 - x1)) {
      slope = 1.0 * (x2 - x1) / (y2 - y1);
      for (pym = y1; pym <= y2; pym++) {
         pxm = x1 + slope * (pym - y1);
         erase_pixel(pxm,pym);
      }
   }

   else {
      slope = 1.0 * (y2 - y1) / (x2 - x1);
      for (pxm = x1; pxm <= x2; pxm++) {
         pym = y1 + slope * (pxm - x1);
         erase_pixel(pxm,pym);
      }
   }
   
   if (crflag) {
      cairo_destroy(mwcr);
      mwcr = 0;
   }

   return;
}


/**************************************************************************/

//  draw pre-set overlay lines on top of image
//  arg = 1:   paint lines only (because window repainted)
//        2:   erase lines and forget them
//        3:   erase old lines, paint new lines, save new in old

void draw_toplines(int arg)
{
   int         ii, crflag = 0;

   if (! Cdraw) return;
   if (! gdkwin) return;
   if (! Cstate || ! Cstate->fpxb) return;                                 //  no image

   if (! mwcr) {
      mwcr = gdk_cairo_create(gdkwin);                                     //  create cairo context if not already
      cairo_set_line_width(mwcr,1);
      crflag = 1;
   }

   if (arg == 2 || arg == 3)                                               //  erase old lines
      for (ii = 0; ii < Nptoplines; ii++)
         erase_line(ptoplinex1[ii],ptopliney1[ii],ptoplinex2[ii],ptopliney2[ii]);
   
   if (arg == 1 || arg == 3)                                               //  draw new lines
      for (ii = 0; ii < Ntoplines; ii++)
         draw_dotline(toplinex1[ii],topliney1[ii],toplinex2[ii],topliney2[ii]);

   if (crflag) {
      cairo_destroy(mwcr);
      mwcr = 0;
   }

   if (arg == 2) {
      Nptoplines = Ntoplines = 0;                                          //  forget lines
      return;
   }

   for (ii = 0; ii < Ntoplines; ii++)                                      //  save for future erase
   {
      ptoplinex1[ii] = toplinex1[ii];
      ptopliney1[ii] = topliney1[ii];
      ptoplinex2[ii] = toplinex2[ii];
      ptopliney2[ii] = topliney2[ii];
   }

   Nptoplines = Ntoplines;

   return;
}


/**************************************************************************/

//  draw a grid of horizontal and vertical lines.
//  grid line spacings are in window space.

void draw_gridlines()
{
   int         crflag = 0, G = currgrid;
   int         px, py, gww, ghh;
   int         startx, starty, endx, endy, stepx, stepy;
   int         startx1, starty1;

   if (! Cdraw) return;
   if (! gdkwin) return;
   if (! Cstate || ! Cstate->fpxb) return;                                 //  no image

   if (! gridsettings[G][GON]) return;                                     //  grid lines off
   
   gww = dww;                                                              //  grid box size
   ghh = dhh;
   startx = Dorgx;                                                         //  starting corner (top left)
   starty = Dorgy;
   
   if (CEF && strEqu(CEF->funcname,"trim_rotate")) {                       //  trim/rotate function is active     v.13.11
      gww = Mscale * (trimrect[2] - trimrect[0]);                          //  fit grid box to trim rectangle
      ghh = Mscale * (trimrect[3] - trimrect[1]);
      startx = Mscale * trimrect[0] - Morgx + Dorgx;
      starty = Mscale * trimrect[1] - Morgy + Dorgy;
   }

   endx = startx + gww;
   endy = starty + ghh;

   stepx = gridsettings[G][GXS];                                           //  space between grid lines
   stepy = gridsettings[G][GYS];                                           //  (window space)

   if (gridsettings[G][GXC]) 
      stepx = gww / (1 + gridsettings[G][GXC]);                            //  if line counts specified,
   if (gridsettings[G][GYC])                                               //    set spacing accordingly
      stepy = ghh / ( 1 + gridsettings[G][GYC]);

   startx1 = startx + stepx * gridsettings[G][GXF] / 100;                  //  variable starting offsets
   starty1 = starty + stepy * gridsettings[G][GYF] / 100;
   
   if (! mwcr) {
      mwcr = gdk_cairo_create(gdkwin);                                     //  create cairo context if not already
      cairo_set_line_width(mwcr,1);
      crflag = 1;
   }

   cairo_set_source_rgb(mwcr,1,1,1);                                       //  white lines

   if (gridsettings[G][GX] && stepx)
      for (px = startx1; px < endx; px += stepx) {
         cairo_move_to(mwcr,px,starty);
         cairo_line_to(mwcr,px,endy);
      }

   if (gridsettings[G][GY] && stepy)
      for (py = starty1; py < endy; py += stepy) {
         cairo_move_to(mwcr,startx,py);
         cairo_line_to(mwcr,endx,py);
      }

   cairo_stroke(mwcr);

   cairo_set_source_rgb(mwcr,0,0,0);                                       //  adjacent black lines

   if (gridsettings[G][GX] && stepx) 
      for (px = startx1+1; px < endx+1; px += stepx) {
         cairo_move_to(mwcr,px,starty);
         cairo_line_to(mwcr,px,endy);
      }

   if (gridsettings[G][GY] && stepy)
      for (py = starty1+1; py < endy+1; py += stepy) {
         cairo_move_to(mwcr,startx,py);
         cairo_line_to(mwcr,endx,py);
      }

   cairo_stroke(mwcr);

   if (crflag) {
      cairo_destroy(mwcr);
      mwcr = 0;
   }

   return;
}


/**************************************************************************/

//  maintain a set of text strings written over the image in the window.
//  add a new text string to the list.
//  multiple text strings can be added with the same ID.
//  px and py are image space.

void add_toptext(int ID, int px, int py, cchar *text, cchar *font)
{
   if (Ntoptext == maxtoptext) {
      printz("*** maxtoptext exceeded \n");
      return;
   }
   
   int ii = Ntoptext++;
   toptext[ii].ID = ID;
   toptext[ii].px = px;
   toptext[ii].py = py;
   toptext[ii].text = text;
   toptext[ii].font = font;
   
   return;
}


//  draw current text strings over the image in window.
//  called from FWpaint().

void draw_toptext() 
{
   int         crflag = 0;

   if (! Cdraw) return;
   if (! gdkwin) return;
   if (! Cstate || ! Cstate->fpxb) return;                                 //  no image

   if (! mwcr) {
      mwcr = gdk_cairo_create(gdkwin);                                     //  create cairo context if not already
      cairo_set_line_width(mwcr,1);
      crflag = 1;
   }

   for (int ii = 0; ii < Ntoptext; ii++)
      draw_text(toptext[ii].px,toptext[ii].py,toptext[ii].text,toptext[ii].font);

   if (crflag) {
      cairo_destroy(mwcr);
      mwcr = 0;
   }

   return;
}


//  delete text strings having the given ID from the list

void erase_toptext(int ID)
{
   int      ii, jj;

   for (ii = jj = 0; ii < Ntoptext; ii++)
   {
      if (toptext[ii].ID == ID) continue;
      else toptext[jj++] = toptext[ii];
   }
   
   Ntoptext = jj;
   return;
}


//  draw text on window, black on white background
//  coordinates are image space

void draw_text(int px, int py, cchar *text, cchar *font)
{
   static PangoFontDescription   *pangofont = null;
   static PangoLayout            *pangolayout = null;
   static char                   priorfont[40] = "";
   int                           ww, hh, crflag = 0;

   if (! Cdraw) return;
   if (! gdkwin) return;

   px = Mscale * px - Morgx + Dorgx;                                       //  image to window space
   py = Mscale * py - Morgy + Dorgy;

   if (strNeq(font,priorfont)) {                                           //  change font
      strncpy0(priorfont,font,40);
      if (pangofont) pango_font_description_free(pangofont);
      if (pangolayout) g_object_unref(pangolayout);
      pangofont = pango_font_description_from_string(font);                //  make pango layout for font
      pangolayout = gtk_widget_create_pango_layout(Cdraw,0);
      pango_layout_set_font_description(pangolayout,pangofont);
   }
   
   pango_layout_set_text(pangolayout,text,-1);                             //  add text to layout
   pango_layout_get_pixel_size(pangolayout,&ww,&hh);
   
   if (! mwcr) {
      mwcr = gdk_cairo_create(gdkwin);                                     //  create cairo context if not already
      cairo_set_line_width(mwcr,1);
      crflag = 1;
   }

   cairo_set_source_rgb(mwcr,1,1,1);                                       //  draw white background
   cairo_rectangle(mwcr,px,py,ww,hh);
   cairo_fill(mwcr);
   
   cairo_move_to(mwcr,px,py);                                              //  draw layout with text
   cairo_set_source_rgb(mwcr,0,0,0);
   pango_cairo_show_layout(mwcr,pangolayout);

   if (crflag) {
      cairo_destroy(mwcr);
      mwcr = 0;
   }

   return;
}


/**************************************************************************/

//  maintain a set of circles drawn over the image in the window
//  px, py are image space, radius is window space
//  color = 1/2/3/4 = white/black/red/green

void add_topcircle(int px, int py, int radius, int color)
{
   if (Ntopcircles == maxtopcircles) {
      printz("*** maxtopcircles exceeded \n");
      return;
   }
   
   int ii = Ntopcircles++;
   topcircles[ii].px = px;
   topcircles[ii].py = py;
   topcircles[ii].radius = radius;
   topcircles[ii].color = color;
   
   return;
}


//  draw current circles over the image in the window
//  called from FWpaint()

void draw_topcircles()
{
   int         ii, px, py, rad, color;
   int         crflag = 0;

   if (! Cdraw) return;
   if (! gdkwin) return;
   if (! Cstate || ! Cstate->fpxb) return;                                 //  no image

   if (! mwcr) {
      mwcr = gdk_cairo_create(gdkwin);                                     //  create cairo context if not already
      cairo_set_line_width(mwcr,1);
      crflag = 1;
   }

   for (ii = 0; ii < Ntopcircles; ii++)
   {
      px = topcircles[ii].px * Mscale - Morgx + Dorgx;                     //  image to window space
      py = topcircles[ii].py * Mscale - Morgy + Dorgy;
      rad = topcircles[ii].radius;                                         //  radius is window space
      color = topcircles[ii].color;                                        //  1/2/3/4 = white/black/red/green

      if (color == 1) cairo_set_source_rgb(mwcr,1,1,1);                    //  set foreground color
      else if (color == 2) cairo_set_source_rgb(mwcr,0,0,0);
      else if (color == 3) cairo_set_source_rgb(mwcr,1,0,0);
      else cairo_set_source_rgb(mwcr,0,1,0);

      cairo_arc(mwcr,px,py,rad,0,2*PI);                                    //  draw 360 deg. arc
      cairo_stroke(mwcr);
   }
   
   if (crflag) {
      cairo_destroy(mwcr);
      mwcr = 0;
   }

   return;
}


//  erase top circles (next window repaint)

void erase_topcircles()
{
   Ntopcircles = 0;
   return;
}


/**************************************************************************/

//  Draw circle/ellipse around the mouse pointer.
//  Prior circle/ellipse will be erased first.
//  Used for mouse/brush radius in select and paint functions.
//  cx, cy, ww, hh: center and axes of ellipse (circle: ww = hh)
//  if Ferase, then erase previous circle/ellipse only.

void draw_mousearc(int cx, int cy, int cww, int chh, int Ferase)           //  simplified v.14.03
{
   int         px3, py3, ww3, hh3;
   static int  ppx3, ppy3, pww3 = 0, phh3;
   int         px, py;
   int         crflag = 0;
   float       a, b, a2, b2;
   float       x, y, x2, y2;
   
   zthreadcrash();

   if (! Cdraw) return;
   if (! gdkwin) return;
   if (! Cstate || ! Cstate->fpxb) return;                                 //  no image
   
   gdk_window_freeze_updates(gdkwin);                                      //  visually smoother                  v.14.02 
   
   if (pww3 > 0) {                                                         //  erase prior 
      Fpaint4(ppx3,ppy3,pww3,phh3);                                        //  refresh from Mpxb
      pww3 = 0;
   }

   if (Ferase) {
      gdk_window_thaw_updates(gdkwin);                                     //  erase only was wanted
      return;
   }
   
   px3 = cx - (cww + 2) / 2;                                               //  convert pointer center + radius
   py3 = cy - (chh + 2) / 2;                                               //    to block position, width, length
   ww3 = cww + 2;
   hh3 = chh + 2;

   ppx3 = px3;                                                             //  remember pixel block area
   ppy3 = py3;                                                             //    to erase in next call
   pww3 = ww3;
   phh3 = hh3;

   if (! mwcr) {
      mwcr = gdk_cairo_create(gdkwin);                                     //  create cairo context if not already
      crflag = 1;
   }

   a = cww / 2;                                                            //  ellipse constants from
   b = chh / 2;                                                            //    enclosing rectangle
   a2 = a * a;  
   b2 = b * b;

   for (y = -b; y < b; y++)                                                //  step through y values, omitting
   {                                                                       //    curve points covered by x values
      y2 = y * y;
      x2 = a2 * (1 - y2 / b2);
      x = sqrtf(x2);                                                       //  corresp. x values, + and -
      py = y + cy;
      px = cx - x + 0.5;
      draw_pixel(px,py);                                                   //  draw 2 points on ellipse
      px = cx + x + 0.5;
      draw_pixel(px,py);
   }

   for (x = -a; x < a; x++)                                                //  step through x values, omitting
   {                                                                       //    curve points covered by y values
      x2 = x * x;
      y2 = b2 * (1 - x2 / a2);
      y = sqrtf(y2);                                                       //  corresp. y values, + and -
      px = x + cx;
      py = cy - y + 0.5;
      draw_pixel(px,py);                                                   //  draw 2 points on ellipse
      py = cy + y + 0.5;
      draw_pixel(px,py);
   }

   if (crflag) {
      cairo_destroy(mwcr);
      mwcr = 0;
   }

   gdk_window_thaw_updates(gdkwin);                                        //  v.14.02
   return;
}


//  duplicate for drawing and tracking a 2nd mouse arc
//  (used by paint/clone to track source pixels being cloned)

void draw_mousearc2(int cx, int cy, int cww, int chh, int Ferase)          //  simplified v.14.03
{
   int         px3, py3, ww3, hh3;
   static int  ppx3, ppy3, pww3 = 0, phh3;
   int         px, py;
   int         crflag = 0;
   float       a, b, a2, b2;
   float       x, y, x2, y2;
   
   zthreadcrash();

   if (! Cdraw) return;
   if (! gdkwin) return;
   if (! Cstate || ! Cstate->fpxb) return;                                 //  no image
   
   gdk_window_freeze_updates(gdkwin);                                      //  visually smoother                  v.14.02 
   
   if (pww3 > 0) {                                                         //  erase prior 
      Fpaint4(ppx3,ppy3,pww3,phh3);                                        //  refresh from Mpxb
      pww3 = 0;
   }

   if (Ferase) {
      gdk_window_thaw_updates(gdkwin);                                     //  erase only was wanted
      return;
   }
   
   px3 = cx - (cww + 2) / 2;                                               //  convert pointer center + radius
   py3 = cy - (chh + 2) / 2;                                               //    to block position, width, length
   ww3 = cww + 2;
   hh3 = chh + 2;

   ppx3 = px3;                                                             //  remember pixel block area
   ppy3 = py3;                                                             //    to erase in next call
   pww3 = ww3;
   phh3 = hh3;

   if (! mwcr) {
      mwcr = gdk_cairo_create(gdkwin);                                     //  create cairo context if not already
      crflag = 1;
   }

   a = cww / 2;                                                            //  ellipse constants from
   b = chh / 2;                                                            //    enclosing rectangle
   a2 = a * a;  
   b2 = b * b;

   for (y = -b; y < b; y++)                                                //  step through y values, omitting
   {                                                                       //    curve points covered by x values
      y2 = y * y;
      x2 = a2 * (1 - y2 / b2);
      x = sqrtf(x2);                                                       //  corresp. x values, + and -
      py = y + cy;
      px = cx - x + 0.5;
      draw_pixel(px,py);                                                   //  draw 2 points on ellipse
      px = cx + x + 0.5;
      draw_pixel(px,py);
   }

   for (x = -a; x < a; x++)                                                //  step through x values, omitting
   {                                                                       //    curve points covered by y values
      x2 = x * x;
      y2 = b2 * (1 - x2 / a2);
      y = sqrtf(y2);                                                       //  corresp. y values, + and -
      px = x + cx;
      py = cy - y + 0.5;
      draw_pixel(px,py);                                                   //  draw 2 points on ellipse
      py = cy + y + 0.5;
      draw_pixel(px,py);
   }

   if (crflag) {
      cairo_destroy(mwcr);
      mwcr = 0;
   }

   gdk_window_thaw_updates(gdkwin);                                        //  v.14.02
   return;
}


/**************************************************************************

   spline curve setup and edit functions

   Support multiple frames with curves in parallel
   (edit curve(s) and image mask curve)

   Usage:
   Add frame widget to dialog or zdialog.
   Set up drawing in frame: 
      sd = splcurve_init(frame, callback_func)
   Initialize no. of curves in frame (1-10): 
      sd->Nspc = n
   Initialize vert/horz flag for curve spc: 
      sd->vert[spc] = hv
   Initialize anchor points for curve spc: 
      sd->nap[spc], sd->apx[spc][xx], sd->apy[spc][yy]
   Generate data for curve spc:  
      splcurve_generate(sd,spc)
   Curves will now be shown and edited inside the frame when window is realized.
   The callback_func(spc) will be called when curve spc is edited.
   Change curve in program: 
      set anchor points, call splcurve_generate(sd,spc)
   Get y-value (0-1) for curve spc and given x-value (0-1):
      yval = splcurve_yval(sd,spc,xval)
   If faster access to curve is needed (no interpolation):
      kk = 1000 * xval;
      if (kk > 999) kk = 999;
      yval = sd->yval[spc][kk];

***/

//  initialize for spline curve editing
//  initial anchor points are pre-loaded into spldat before window is realized

spldat * splcurve_init(GtkWidget *frame, void func(int spc))
{
   int      cc = sizeof(spldat);                                           //  allocate spc curve data area
   spldat * sd = (spldat *) zmalloc(cc);
   memset(sd,0,cc);

   sd->drawarea = gtk_drawing_area_new();                                  //  drawing area for curves
   gtk_container_add(GTK_CONTAINER(frame),sd->drawarea);
   sd->spcfunc = func;                                                     //  user callback function

   gtk_widget_add_events(sd->drawarea,GDK_BUTTON_PRESS_MASK);              //  connect mouse events to drawing area
   gtk_widget_add_events(sd->drawarea,GDK_BUTTON_RELEASE_MASK);
   gtk_widget_add_events(sd->drawarea,GDK_BUTTON1_MOTION_MASK); 
   G_SIGNAL(sd->drawarea,"motion-notify-event",splcurve_adjust,sd);
   G_SIGNAL(sd->drawarea,"button-press-event",splcurve_adjust,sd);
   G_SIGNAL(sd->drawarea,"draw",splcurve_draw,sd);
   
   return sd;
}


//  modify anchor points in curve using mouse

int splcurve_adjust(void *, GdkEventButton *event, spldat *sd)
{
   int            ww, hh, kk;
   int            mx, my, button, evtype;
   static int     spc, ap, mbusy = 0, Fdrag = 0;                           //  drag continuation logic
   int            minspc, minap;
   float          mxval, myval, cxval, cyval;
   float          dist2, mindist2 = 0;
   float          dist, dx, dy;
   
   mx = event->x;                                                          //  mouse position in drawing area
   my = event->y;
   evtype = event->type;
   button = event->button;

   if (evtype == GDK_MOTION_NOTIFY) {
      if (mbusy) return 0;                                                 //  discard excess motion events
      mbusy++;
      zmainloop();
      mbusy = 0;
   }

   if (evtype == GDK_BUTTON_RELEASE) {
      Fdrag = 0;
      return 0;
   }

   ww = gtk_widget_get_allocated_width(sd->drawarea);                      //  drawing area size
   hh = gtk_widget_get_allocated_height(sd->drawarea);
   
   if (mx < 0) mx = 0;                                                     //  limit edge excursions
   if (mx > ww) mx = ww;
   if (my < 0) my = 0;
   if (my > hh) my = hh;
   
   if (evtype == GDK_BUTTON_PRESS) Fdrag = 0;                              //  left or right click
   
   if (Fdrag)                                                              //  continuation of drag
   {
      if (sd->vert[spc]) {
         mxval = 1.0 * my / hh;                                            //  mouse position in curve space
         myval = 1.0 * mx / ww;
      }
      else {
         mxval = 1.0 * mx / ww;
         myval = 1.0 * (hh - my) / hh;
      }

      if (ap < sd->nap[spc]-1) {                                           //  not the last anchor point          v.14.04
         dx = sd->apx[spc][ap+1] - mxval;                                  //  get distance to next anchor point
         if (dx < 0.01) return 0;                                          //  x-value not increasing, forbid
         dy = sd->apy[spc][ap+1] - myval;
         dist = sqrtf(dx * dx + dy * dy);
         if (dist < 0.05) return 0;                                        //  too close, forbid
      }
      if (ap > 0) {                                                        //  not the first anchor point
         dx = mxval - sd->apx[spc][ap-1];                                  //  get distance to prior anchor point
         if (dx < 0.01) return 0;                                          //  x-value not increasing, forbid
         dy = myval - sd->apy[spc][ap-1];
         dist = sqrtf(dx * dx + dy * dy);
         if (dist < 0.05) return 0;                                        //  too close, forbid
      }
   }
   
   else                                                                    //  mouse click or new drag begin
   {
      minspc = minap = -1;                                                 //  find closest curve/anchor point
      mindist2 = 999999;

      for (spc = 0; spc < sd->Nspc; spc++)                                 //  loop curves
      {
         if (sd->vert[spc]) {
            mxval = 1.0 * my / hh;                                         //  mouse position in curve space
            myval = 1.0 * mx / ww;
         }
         else {
            mxval = 1.0 * mx / ww;
            myval = 1.0 * (hh - my) / hh;
         }

         for (ap = 0; ap < sd->nap[spc]; ap++)                             //  loop anchor points
         {
            cxval = sd->apx[spc][ap];
            cyval = sd->apy[spc][ap];
            dist2 = (mxval-cxval)*(mxval-cxval) 
                  + (myval-cyval)*(myval-cyval);
            if (dist2 < mindist2) {
               mindist2 = dist2;                                           //  remember closest anchor point
               minspc = spc;
               minap = ap;
            }
         }
      }

      if (minspc < 0) return 0;                                            //  impossible
      spc = minspc;                                                        //  nearest curve
      ap = minap;                                                          //  nearest anchor point
   }
   
   if (evtype == GDK_BUTTON_PRESS && button == 3)                          //  right click, remove anchor point
   {
      if (sqrtf(mindist2) > 0.05) return 0;                                //  not close enough
      if (sd->nap[spc] < 3) return 0;                                      //  < 2 anchor points would remain
      sd->nap[spc]--;                                                      //  decr. before loop
      for (kk = ap; kk < sd->nap[spc]; kk++) {
         if (! kk) printz("meaningless reference %d",kk);                  //  stop gcc optimization bug   /////
         sd->apx[spc][kk] = sd->apx[spc][kk+1];
         sd->apy[spc][kk] = sd->apy[spc][kk+1];
      }
      splcurve_generate(sd,spc);                                           //  regenerate data for modified curve
      gtk_widget_queue_draw(sd->drawarea);
      sd->spcfunc(spc);                                                    //  call user function
      return 0;
   }

   if (! Fdrag)                                                            //  new drag or left click
   {
      if (sd->vert[spc]) {
         mxval = 1.0 * my / hh;                                            //  mouse position in curve space
         myval = 1.0 * mx / ww;
      }
      else {
         mxval = 1.0 * mx / ww;
         myval = 1.0 * (hh - my) / hh;
      }

      if (sqrtf(mindist2) < 0.05)                                          //  anchor point close enough,         v.14.04
      {                                                                    //    move this one to mouse position
         if (ap < sd->nap[spc]-1) {                                        //  not the last anchor point
            dx = sd->apx[spc][ap+1] - mxval;                               //  get distance to next anchor point
            if (dx < 0.01) return 0;                                       //  x-value not increasing, forbid
            dy = sd->apy[spc][ap+1] - myval;
            dist = sqrtf(dx * dx + dy * dy);
            if (dist < 0.05) return 0;                                     //  too close, forbid
         }
         if (ap > 0) {                                                     //  not the first anchor point
            dx = mxval - sd->apx[spc][ap-1];                               //  get distance to prior anchor point
            if (dx < 0.01) return 0;                                       //  x-value not increasing, forbid
            dy = myval - sd->apy[spc][ap-1];
            dist = sqrtf(dx * dx + dy * dy);
            if (dist < 0.05) return 0;                                     //  too close, forbid
         }
      }
   
      else                                                                 //  none close, add new anchor point
      {
         minspc = -1;                                                      //  find closest curve to mouse
         mindist2 = 999999;

         for (spc = 0; spc < sd->Nspc; spc++)                              //  loop curves
         {
            if (sd->vert[spc]) {
               mxval = 1.0 * my / hh;                                      //  mouse position in curve space
               myval = 1.0 * mx / ww;
            }
            else {
               mxval = 1.0 * mx / ww;
               myval = 1.0 * (hh - my) / hh;
            }

            cyval = splcurve_yval(sd,spc,mxval);
            dist2 = fabsf(myval - cyval);
            if (dist2 < mindist2) {
               mindist2 = dist2;                                           //  remember closest curve
               minspc = spc;
            }
         }

         if (minspc < 0) return 0;                                         //  impossible
         if (mindist2 > 0.05) return 0;                                    //  not close enough to any curve
         spc = minspc;

         if (sd->nap[spc] > 49) {
            zmessageACK(Mwin,0,ZTX("Exceed 50 anchor points"));
            return 0;
         }

         if (sd->vert[spc]) {
            mxval = 1.0 * my / hh;                                         //  mouse position in curve space
            myval = 1.0 * mx / ww;
         }
         else {
            mxval = 1.0 * mx / ww;
            myval = 1.0 * (hh - my) / hh;
         }

         for (ap = 0; ap < sd->nap[spc]; ap++)                             //  find anchor point with next higher x
            if (mxval <= sd->apx[spc][ap]) break;                          //    (ap may come out 0 or nap)

         if (ap < sd->nap[spc] && sd->apx[spc][ap] - mxval < 0.05)         //  disallow < 0.05 from next
               return 0;                                                   //    or prior anchor point
         if (ap > 0 && mxval - sd->apx[spc][ap-1] < 0.05) return 0;

         for (kk = sd->nap[spc]; kk > ap; kk--) {                          //  make hole for new point
            sd->apx[spc][kk] = sd->apx[spc][kk-1];
            sd->apy[spc][kk] = sd->apy[spc][kk-1];
         }

         sd->nap[spc]++;                                                   //  up point count
      }
   }

   if (evtype == GDK_MOTION_NOTIFY) Fdrag = 1;                             //  remember drag is underway

   sd->apx[spc][ap] = mxval;                                               //  new or moved anchor point
   sd->apy[spc][ap] = myval;                                               //    at mouse position

   splcurve_generate(sd,spc);                                              //  regenerate data for modified curve
   if (sd->drawarea) gtk_widget_queue_draw(sd->drawarea);                  //  redraw graph
   if (sd->spcfunc) sd->spcfunc(spc);                                      //  call user function
   
   return 0;
}


//  for expose event or when a curve is changed
//  draw all curves based on current anchor points

int splcurve_draw(GtkWidget *drawarea, cairo_t *cr, spldat *sd)
{
   int         ww, hh, spc, ap;
   float       xval, yval, px, py, qx, qy;
   
   ww = gtk_widget_get_allocated_width(sd->drawarea);                      //  drawing area size
   hh = gtk_widget_get_allocated_height(sd->drawarea);
   if (ww < 50 || hh < 50) return 0;

   cairo_set_line_width(cr,1);
   cairo_set_source_rgb(cr,0.7,0.7,0.7);

   for (int ii = 0; ii < sd->Nscale; ii++)                                 //  draw y-scale lines if any
   {
      px = ww * sd->xscale[0][ii];
      py = hh - hh * sd->yscale[0][ii];
      qx = ww * sd->xscale[1][ii];
      qy = hh - hh * sd->yscale[1][ii];
      cairo_move_to(cr,px,py);
      cairo_line_to(cr,qx,qy);
   }
   cairo_stroke(cr);

   cairo_set_source_rgb(cr,0,0,0);

   for (spc = 0; spc < sd->Nspc; spc++)                                    //  loop all curves
   {
      if (sd->vert[spc])                                                   //  vert. curve
      {
         for (py = 0; py < hh; py++)                                       //  generate all points for curve
         {
            xval = 1.0 * py / hh;
            yval = splcurve_yval(sd,spc,xval);
            px = ww * yval;
            if (py == 0) cairo_move_to(cr,px,py);
            cairo_line_to(cr,px,py);
         }
         cairo_stroke(cr);
         
         for (ap = 0; ap < sd->nap[spc]; ap++)                             //  draw boxes at anchor points
         {
            xval = sd->apx[spc][ap];
            yval = sd->apy[spc][ap];
            px = ww * yval;
            py = hh * xval;
            cairo_rectangle(cr,px-2,py-2,4,4);
         }
         cairo_fill(cr);
      }
      else                                                                 //  horz. curve
      {
         for (px = 0; px < ww; px++)                                       //  generate all points for curve
         {
            xval = 1.0 * px / ww;
            yval = splcurve_yval(sd,spc,xval);
            py = hh - hh * yval;
            if (px == 0) cairo_move_to(cr,px,py);
            cairo_line_to(cr,px,py);
         }
         cairo_stroke(cr);
         
         for (ap = 0; ap < sd->nap[spc]; ap++)                             //  draw boxes at anchor points
         {
            xval = sd->apx[spc][ap];
            yval = sd->apy[spc][ap];
            px = ww * xval;
            py = hh - hh * yval;
            cairo_rectangle(cr,px-2,py-2,4,4);
         }
         cairo_fill(cr);
      }
   }

   return 0;
}


//  generate all curve data points when anchor points are modified

int splcurve_generate(spldat *sd, int spc)
{
   int      kk, kklo, kkhi;
   float    xval, yvalx;
   
   spline1(sd->nap[spc],sd->apx[spc],sd->apy[spc]);                        //  compute curve fitting anchor points

   kklo = 1000 * sd->apx[spc][0] - 30;                                     //  xval range = anchor point range
   if (kklo < 0) kklo = 0;                                                 //    + 0.03 extra below/above
   kkhi = 1000 * sd->apx[spc][sd->nap[spc]-1] + 30;
   if (kkhi > 1000) kkhi = 1000;

   for (kk = 0; kk < 1000; kk++)                                           //  generate all points for curve
   {
      xval = 0.001 * kk;                                                   //  remove anchor point limits
      yvalx = spline2(xval);
      if (yvalx < 0) yvalx = 0;                                            //  yval < 0 not allowed, > 1 OK
      sd->yval[spc][kk] = yvalx;
   }

   return 0;
}


//  Retrieve curve data using interpolation of saved table of values

float splcurve_yval(spldat *sd, int spc, float xval)
{
   int      ii;
   float    x1, x2, y1, y2, y3;
   
   if (xval <= 0) return sd->yval[spc][0];
   if (xval >= 0.999) return sd->yval[spc][999];
   
   x2 = 1000.0 * xval;
   ii = x2;
   x1 = ii;
   y1 = sd->yval[spc][ii];
   y2 = sd->yval[spc][ii+1];
   y3 = y1 + (y2 - y1) * (x2 - x1);
   return y3;
}


//  load curve data from a file
//  returns 0 if fail (invalid file data), sd not modified
//  returns 1 if succcess, sd is initialized from file data

int splcurve_load(spldat *sd, cchar *file)
{
   char        *pfile;
   int         nn, ii, jj;
   FILE        *fid = 0;
   int         Nspc, vert[10], nap[10];
   float       apx[10][50], apy[10][50];

   F1_help_topic = "curve_edit";

   if (file) pfile = zstrdup(file);
   else pfile = zgetfile(ZTX("load curve from a file"),"file",saved_curves_dirk);
   if (! pfile) return 0;

   fid = fopen(pfile,"r");
   zfree(pfile);
   if (! fid) goto fail;
   
   nn = fscanf(fid,"%d ",&Nspc);                                           //  no. of curves
   if (nn != 1) goto fail;
   if (Nspc < 1 || Nspc > 10) goto fail;
   if (Nspc != sd->Nspc) goto fail2;

   for (ii = 0; ii < Nspc; ii++)                                           //  loop each curve
   {
      nn = fscanf(fid,"%d  %d ",&vert[ii],&nap[ii]);                       //  vertical flag, no. anchor points
      if (nn != 2) goto fail;
      if (vert[ii] < 0 || vert[ii] > 1) goto fail;
      if (nap[ii] < 2 || nap[ii] > 50) goto fail;

      for (jj = 0; jj < nap[ii]; jj++)                                     //  anchor point values 
      {
         nn = fscanf(fid,"%f/%f  ",&apx[ii][jj],&apy[ii][jj]);
         if (nn != 2) goto fail;
         if (apx[ii][jj] < 0 || apx[ii][jj] > 1) goto fail;
         if (apy[ii][jj] < 0 || apy[ii][jj] > 1) goto fail;
      }
   }
   
   fclose(fid);
   
   sd->Nspc = Nspc;                                                        //  copy curve data to caller's arg

   for (ii = 0; ii < Nspc; ii++)
   {
      sd->vert[ii] = vert[ii];
      sd->nap[ii] = nap[ii];
      
      for (jj = 0; jj < nap[ii]; jj++)
      {
         sd->apx[ii][jj] = apx[ii][jj];
         sd->apy[ii][jj] = apy[ii][jj];
      }
   }

   for (ii = 0; ii < Nspc; ii++)                                           //  generate curve data from anchor points
      splcurve_generate(sd,ii);
   
   if (sd->drawarea)                                                       //  redraw all curves
      gtk_widget_queue_draw(sd->drawarea);

   return 1;
   
fail:
   if (fid) fclose(fid);
   printz("curve file is invalid \n");
   return 0;

fail2:
   if (fid) fclose(fid);
   printz("curve file has different no. of curves \n");
   return 0;
}


//  save curve data to a file

int splcurve_save(spldat *sd, cchar *file) 
{
   char        *pfile, *pp;
   int         ii, jj;
   FILE        *fid;

   F1_help_topic = "curve_edit";

   if (file) pp = zstrdup(file);
   else pp = zgetfile(ZTX("save curve to a file"),"save",saved_curves_dirk);
   if (! pp) return 0;

   pfile = zstrdup(pp,8);
   zfree(pp);

   pp = strrchr(pfile,'/');                                                //  force .curve extension
   if (pp) pp = strrchr(pp,'.');
   if (pp) strcpy(pp,".curve");
   else strcat(pfile,".curve");

   fid = fopen(pfile,"w");
   zfree(pfile);
   if (! fid) return 0;
   
   fprintf(fid,"%d \n",sd->Nspc);                                          //  no. of curves

   for (ii = 0; ii < sd->Nspc; ii++)                                       //  loop each curve
   {
      fprintf(fid,"%d  %d \n",sd->vert[ii],sd->nap[ii]);                   //  vertical flag, no. anchor points
      for (jj = 0; jj < sd->nap[ii]; jj++)                                 //  anchor point values 
         fprintf(fid,"%.4f/%.4f  ",sd->apx[ii][jj],sd->apy[ii][jj]);
      fprintf(fid,"\n");
   }
   
   fclose(fid);
   return 0;
}


/**************************************************************************

   edit transaction and thread support functions

   edit transaction management
      edit_setup()                  get E0 if none, E0 > E1 > E3
      edit_cancel()                 free (E1 E3 ER)
      edit_done()                   E3 > E0, free (E1 ER) add to undo stack
      edit_undo()                   E3 > ER, E1 > E3
      edit_redo()                   ER > E3
      edit_reset()                  free ER, E1 > E3
      edit_fullsize()               free (E1 E3) E0 > E1 > E3, signal thread

***************************************************************************/

/**************************************************************************

  Setup for a new edit transaction
  Create E1 (edit input) and E3 (edit output) pixmaps from
  previous edit (E0) or image file (new E0).

  FprevReq   0     edit full-size image
             1     edit preview image unless select area exists

  Farea      0     select_area is invalid and will be deleted (e.g. rotate)
             1     select_area not used but remains valid (e.g. red-eye)
             2     select_area can be used and remains valid (e.g. gamma)

***************************************************************************/

int edit_setup(editfunc &EF)
{
   int      yn, rww, rhh;
   int      Fpreview;
   
   TRACE

   if (! curr_file) return 0;                                              //  no image file
   if (FGW != 'F') return 0;
   if (checkpend("lock")) return 0;                                        //  menu lock                          v.13.12
   
   if (CEF && CEF->zd)                                                     //  if pending edit, complete it       v.13.12
      zdialog_send_event(CEF->zd,"done");
   if (checkpend("edit")) return 0;                                        //  failed (HDR etc.)                  v.13.12

   if (URS_pos > maxedits-2) {                                             //  undo/redo stack capacity reached
      zmessageACK(Mwin,0,ZTX("Too many edits, please save image"));
      return 0;
   }
   
   free_geolocs();                                                         //  free memory for use by edit        v.14.07

   sa_validate();                                                          //  delete area if not valid

   if (EF.Farea == 0 && sa_stat) {                                         //  select area will be lost, warn user
      yn = zmessageYN(Mwin,ZTX("Select area cannot be kept.\n"
                               "Continue?"));
      if (! yn) return 0;
      sa_unselect();                                                       //  unselect area
      if (zdsela) zdialog_free(zdsela);
   }
   
   if (EF.Farea == 2 && sa_stat && sa_stat != 3) {                         //  select area exists and can be used,
      yn = zmessageYN(Mwin,ZTX("Select area not active.\n"                 //    but not active, ask user
                               "Continue?"));
      if (! yn) return 0;
   }

   if (! E0pxm) {                                                          //  first edit for this file
      E0pxm = PXM_load(curr_file,1);                                       //  get E0 image (poss. 16-bit color)
      if (! E0pxm) return 0;
      PXM_fixblue(E0pxm);                                                  //  blue=0 >> blue=1 for vpixel()
      curr_file_bpc = f_load_bpc;
   }

   if (URS_pos == 0) save_undo();                                          //  initial image >> undo/redo stack

   mutex_lock(&Fpixmap_lock);                                              //  lock pixmaps

   Fpreview = 0;                                                           //  assume no preview

   if (EF.FprevReq && ! Fzoom)                                             //  preview requested by edit func.
      Fpreview = 1;

   if (EF.Farea == 2 && sa_stat == 3)                                      //  not if select area active
      Fpreview = 0;

   if (E0pxm->ww * E0pxm->hh < 2000000)                                    //  if image is small, don't use preview
      Fpreview = 0;                                                        //       v.13.04
   
   if (E0pxm->ww < 1.4 * Dww && E0pxm->hh < 1.4 * Dhh)                     //  if image slightly larger than window,
      Fpreview = 0;                                                        //       don't use preview
   
   if (Fpreview) {
      if (Fpxb->ww * Dhh > Fpxb->hh * Dww) {                               //  use preview image 1.4 * window size 
         rww = 1.4 * Dww;
         if (rww < 1200) rww = 1200;                                       //  at least 1200 on one side
         rhh = 1.0 * rww * Fpxb->hh / Fpxb->ww + 0.5;
      }
      else {
         rhh = 1.4 * Dhh;
         if (rhh < 1200) rhh = 1200;
         rww = 1.0 * rhh * Fpxb->ww / Fpxb->hh + 0.5;
      }
      if (rww > Fpxb->ww) Fpreview = 0;
   }

   if (Fpreview) {
      E1pxm = PXM_rescale(E0pxm,rww,rhh);                                  //  scale image to preview size
      sa_show(0);                                                          //  hide select area if present
   }
   else E1pxm = PXM_copy(E0pxm);                                           //  else use full size image

   E3pxm = PXM_copy(E1pxm);                                                //  E1 >> E3

   CEF = &EF;                                                              //  set current edit function
   CEF->Fmods = 0;                                                         //  image not modified yet             v.13.04
   CEF->Fpreview = Fpreview;

   CEF->thread_command = CEF->thread_status = 0;                           //  no thread running
   CEF->thread_pend = CEF->thread_done = CEF->thread_hiwater = 0;          //  no work pending or done
   if (CEF->threadfunc) start_thread(CEF->threadfunc,0);                   //  start edit thread
   
   mutex_unlock(&Fpixmap_lock);                                            //  moved down                         v.14.02
   Fpaintnow();                                                            //  update image data synchronous      v.13.04
   return 1;
}


/**************************************************************************/

//  process edit cancel
//  keep: retain zdialog, mousefunc, curves

void edit_cancel(int keep)
{
   TRACE                                                                   //  v.14.04.2

   if (! CEF) return;
   
   wrapup_thread(9);                                                       //  tell thread to quit now, wait
   
   mutex_lock(&Fpixmap_lock);

   PXM_free(E1pxm);                                                        //  free E1, E3, ER
   PXM_free(E3pxm);
   PXM_free(ERpxm);
   PXM_free(E9pxm);                                                        //  v.13.10

   if (! keep)                                                             //  v.13.12
   {
      if (CEF->zd == zd_thread) zd_thread = 0;                             //  bugfix                             v.14.06
      if (CEF->zd) zdialog_free(CEF->zd);                                  //  kill dialog
      if (CEF->mousefunc == mouseCBfunc) freeMouse();                      //  if my mouse, free mouse
      if (CEF->curves) zfree(CEF->curves);                                 //  free curves data
   }
   
   CEF = 0;                                                                //  no current edit func
   mutex_unlock(&Fpixmap_lock);                                            //  moved down                         v.14.02
   Fpaintnow();                                                            //  update image data synchronous      v.13.04
   return;
}   


/**************************************************************************/

//  process edit done
//  keep: retain zdialog, mousefunc, curves

void edit_done(int keep)
{
   TRACE                                                                   //  v.14.04.2

   if (! CEF) return; 

   wait_thread_idle();                                                     //  wait for thread to finish
   
   if (CEF->Fpreview && CEF->Fmods) {                                      //  preview image was edited
      Fzoom = 0;
      edit_fullsize();                                                     //  update full image
   }
   
   wrapup_thread(8);                                                       //  tell thread to finish and exit

   mutex_lock(&Fpixmap_lock);

   if (CEF->Fmods) {                                                       //  image was modified
      PXM_free(E0pxm);
      E0pxm = E3pxm;                                                       //  E3 updated image >> E0
      E3pxm = 0;
      PXM_free(E1pxm);                                                     //  free E1, ER
      PXM_free(ERpxm);
      PXM_free(E9pxm);                                                     //  v.13.10
      URS_pos++;                                                           //  next undo/redo stack position
      URS_max = URS_pos;                                                   //  image modified - higher mods now obsolete
      save_undo();                                                         //  save undo state (for following undo)
   }
   
   else {                                                                  //  not modified
      PXM_free(E1pxm);                                                     //  free E1, E3, ER
      PXM_free(E3pxm);
      PXM_free(ERpxm);
      PXM_free(E9pxm);                                                     //  v.13.10
   }
   
   if (! keep)                                                             //  v.13.12
   {
      if (CEF->zd == zd_thread) zd_thread = 0;                             //  bugfix                             v.14.06
      if (CEF->zd) zdialog_free(CEF->zd);                                  //  kill dialog
      if (CEF->mousefunc == mouseCBfunc) freeMouse();                      //  if my mouse, free mouse
      if (CEF->curves) zfree(CEF->curves);                                 //  free curves data
   }

   CEF = 0;                                                                //  no current edit func

   mutex_unlock(&Fpixmap_lock);                                            //  moved down                         v.14.02
   Fpaintnow();                                                            //  update image data synchronous      v.13.04
   return;
}


/**************************************************************************/

//  edit undo, redo, reset functions
//  these apply within an active edit function

void edit_undo()
{
   TRACE                                                                   //  v.14.04.2

   if (! CEF) return;
   if (CEF->thread_status == 2) return;                                    //  thread busy
   if (! CEF->Fmods) return;                                               //  not modified
   
   mutex_lock(&Fpixmap_lock);
   PXM_free(ERpxm);                                                        //  E3 >> redo copy
   ERpxm = E3pxm;
   E3pxm = PXM_copy(E1pxm);                                                //  E1 >> E3
   CEF->Fmods = 0;                                                         //  reset image modified status
   mutex_unlock(&Fpixmap_lock);
   Fpaintnow();                                                            //  update image data synchronous      v.13.04
   return;
}


void edit_redo()
{
   TRACE                                                                   //  v.14.04.2

   if (! CEF) return;
   if (CEF->thread_status == 2) return;                                    //  thread busy
   if (! ERpxm) return;                                                    //  no prior undo

   mutex_lock(&Fpixmap_lock);
   PXM_free(E3pxm);                                                        //  redo copy >> E3
   E3pxm = ERpxm;
   ERpxm = 0;
   CEF->Fmods = 1;                                                         //  image modified
   mutex_unlock(&Fpixmap_lock);
   Fpaintnow();                                                            //  update image data synchronous      v.13.04
   return;
}


void edit_reset()                                                          //  reset E3 to E1 status
{
   TRACE                                                                   //  v.14.04.2

   if (! CEF) return;
   wait_thread_idle();                                                     //  v.13.04
   if (! CEF->Fmods) return;                                               //  not modified

   mutex_lock(&Fpixmap_lock);
   PXM_free(ERpxm);                                                        //  delete redo copy
   PXM_free(E3pxm);
   E3pxm = PXM_copy(E1pxm);                                                //  E1 >> E3
   CEF->Fmods = 0;                                                         //  reset image modified status        v.13.04
   mutex_unlock(&Fpixmap_lock);
   Fpaintnow();                                                            //  update image data synchronous      v.13.04
   return;
}


/**************************************************************************/

//  Convert from preview mode (window-size pixmaps) to full-size pixmaps.
//  If edit function has standard thread, it is signaled to go to work.
//  Called by edit_done() if preview mode active.
//  Can be called by edit function prior to edit_done().

void edit_fullsize()
{
   TRACE                                                                   //  v.14.04.2

   if (! CEF) return;
   if (! CEF->Fpreview) return;
   
   wait_thread_idle();                                                     //  v.13.04

   mutex_lock(&Fpixmap_lock);
   PXM_free(E1pxm);                                                        //  free preview pixmaps
   PXM_free(E3pxm);
   E1pxm = PXM_copy(E0pxm);                                                //  E0 >> E1, full size image
   E3pxm = PXM_copy(E1pxm);                                                //  E1 >> E3
   PXB_free(Cstate->fpxb);
   Cstate->fpxb = PXM_PXB_copy(E3pxm);                                     //  update Fpxb from image E3
   Fzoom = 0;
   CEF->Fpreview = 0;                                                      //  this prevents re-entry
   mutex_unlock(&Fpixmap_lock);

   if (! CEF->Fmods) return;                                               //  no change
   
   if (CEF->thread_status) {
      signal_thread();                                                     //  signal thread, repeat edit
      wait_thread_idle(); 
   }

   return;
}


/**************************************************************************/

//  Load/save all zdialog widgets data from/to a file.
//  Funcname corresponds to /home/<user>/.appname/funcname
//    where zdialog data is saved for the respective function.
//  Supply spldat for curve data if edit curves are used.
//  Freload = 1 to reload prior data without user input.

void edit_load_widgets(editfunc *EF, int Freload)                          //  v.14.05
{
   using namespace zfuncs;
   
   static char *filename = 0;
   cchar       *mess = ZTX("Load settings from file");

   FILE     *fid;
   cchar    *funcname;
   zdialog  *zd;
   spldat   *sd;
   char     dirname[200], buff[1000];
   char     *wname, *wdata, wdata2[1000];
   char     *pp, *pp1, *pp2;
   int      ii, jj, kk, cc1, cc2;
   int      Nspc, vert, nap, err;
   float    apx, apy;
   
   TRACE

   funcname = EF->funcname;
   zd = EF->zd;
   sd = EF->curves;
   
   if (! Freload || ! filename) {                                          //  get filename from user unless Freload
      if (filename) zfree(filename);
      snprintf(dirname,200,"%s/%s",get_zuserdir(),funcname);               //  /home/<user>/.appname/funcname
      filename = zgetfile(mess,"file",dirname,0);                          //  open data file
      if (! filename) return;                                              //  user cancel
   }
   
   fid = fopen(filename,"r");
   if (! fid) {
      zmessageACK(0,0,"%s \n %s",filename,strerror(errno));
      return;
   }

   for (ii = 0; ii < zdmaxwidgets; ii++)                                   //  read widget data recs
   {
      pp = fgets_trim(buff,1000,fid,1);
      if (! pp) break;
      if (strEqu(pp,"curves")) break;                                      //  start of curves data
      pp1 = pp;
      pp2 = strstr(pp1," ==");
      if (! pp2) continue;                                                 //  widget has no data
      cc1 = pp2 - pp1;
      if (cc1 > 100) continue;
      pp1[cc1] = 0;
      wname = pp1;                                                         //  widget name
      pp2 += 3;
      if (*pp2 == ' ') pp2++;
      wdata = pp2;                                                         //  widget data
      cc2 = strlen(wdata);
      if (cc2 < 1) wdata = (char *) "";
      if (cc2 > 300) continue;
      repl_1str(wdata,wdata2,"\\n","\n");                                  //  replace "\n" with newline chars.
      zdialog_put_data(zd,wname,wdata2);
   }
   
   if (sd)                                                                 //  read curve data
   {
      pp = fgets_trim(buff,200,fid,1);                                     //  curve count
      if (! pp) goto baddata;
      Nspc = atoi(pp);
      if (Nspc != sd->Nspc) goto baddata;

      for (ii = 0; ii < Nspc; ii++)                                        //  loop curves
      {
         pp = fgets_trim(buff,1000,fid,1);                                 //  vert flag, node count
         if (! pp) goto baddata;
         pp = (char *) strField(buff,' ',1);                               //  vert flag, 0 or 1
         if (! pp) goto baddata;
         vert = atoi(pp);
         if (vert > 1) goto baddata;
         sd->vert[ii] = vert;
         pp = (char *) strField(buff,' ',2);                               //  node count
         if (! pp) goto baddata;
         nap = atoi(pp);
         if (nap < 2 || nap > 30) goto baddata;
         sd->nap[ii] = nap;
         
         for (jj = 0; jj < nap; jj++)                                      //  loop nodes
         {
            kk = 2 * jj + 3;
            pp = (char *) strField(buff," /",kk);                          //  get x-value with '/' delimiter
            if (! pp) goto baddata;
            err = convSF(pp,apx,0.0,1.0,0);
            if (err > 1) goto baddata;
            pp = (char *) strField(buff," /",kk+1);                        //  get y-value with ' ' delimiter
            if (! pp) goto baddata;
            err = convSF(pp,apy,0.0,1.0,0);
            if (err > 1) goto baddata;
            sd->apx[ii][jj] = apx;
            sd->apy[ii][jj] = apy;
         }
         
         splcurve_generate(sd,ii);
      }
   }
   
   fclose(fid);
   return;

baddata:
   zmessageACK(0,0,ZTX("file data does not fit dialog"));
   return;
}


void edit_save_widgets(editfunc *EF)                                       //  v.14.05
{
   using namespace zfuncs;
   
   cchar    *funcname;
   zdialog  *zd;
   spldat   *sd;
   FILE     *fid;
   char     *filename, dirname[200];
   char     *wtype, *wname, *wdata, wdata2[1000];
   int      ii, jj, cc;
   int      Nspc, vert, nap;
   float    apx, apy;
   
   cchar    *editwidgets = "entry edit text togbutt check combo comboE "
                           "radio spin hscale vscale colorbutt";
   
   TRACE

   funcname = EF->funcname;
   zd = EF->zd;
   sd = EF->curves;

   snprintf(dirname,200,"%s/%s",get_zuserdir(),funcname);                  //  /home/<user>/.appname/funcname
   filename = zgetfile(ZTX("Save settings to a file"),"save",dirname,0);   //  open data file
   if (! filename) return;                                                 //  user cancel
   
   fid = fopen(filename,"w");
   zfree(filename);
   if (! fid) {
      zmessageACK(0,0,"%s \n %s",filename,strerror(errno));
      return;
   }
   
   for (ii = 0; ii < zdmaxwidgets; ii++)
   {
      wtype = (char *) zd->widget[ii].type;
      if (! wtype) break;
      if (! strstr(editwidgets,wtype)) continue;
      wname = (char *) zd->widget[ii].name;                                //  write widget data recs:
      wdata = zd->widget[ii].data;                                         //    widgetname == widgetdata
      if (! wdata) continue;
      cc = strlen(wdata);
      if (cc > 900) continue;
      repl_1str(wdata,wdata2,"\n","\\n");
      fprintf(fid,"%s == %s \n",wname,wdata);
   }

   if (sd)                                                                 //  write curve data
   {
      fprintf(fid,"curves\n");

      Nspc = sd->Nspc;                                                     //  curve count
      fprintf(fid,"%d \n",Nspc);

      for (ii = 0; ii < Nspc; ii++)                                        //  loop curves
      {
         vert = sd->vert[ii];
         nap = sd->nap[ii];
         fprintf(fid,"%d %d ",vert,nap);                                   //  write vert flag and node count

         for (jj = 0; jj < nap; jj++)                                      //  loop nodes
         {
            apx = sd->apx[ii][jj];
            apy = sd->apy[ii][jj];
            fprintf(fid,"%.3f/%.3f ",apx,apy);                             //  write x-value / y-value pairs
         }
         
         fprintf(fid,"\n");
      }
   }
   
   fclose(fid);
   return;
}


/**************************************************************************
      undo / redo menu buttons
***************************************************************************/

//  [undo] menu function - reinstate previous edit in undo/redo stack

void m_undo(GtkWidget *, cchar *)
{
   TRACE                                                                   //  v.14.04.2

   if (CEF) {                                                              //  undo active edit
      edit_undo();
      return;
   }

   if (URS_pos == 0) return;                                               //  undo past edit
   URS_pos--;
   load_undo();
   return;
}


//  [redo] menu function - reinstate next edit in undo/redo stack

void m_redo(GtkWidget *, cchar *)
{
   TRACE                                                                   //  v.14.04.2

   if (CEF) {                                                              //  redo active edit
      edit_redo();
      return;
   }

   if (URS_pos == URS_max) return;                                         //  redo past edit
   URS_pos++;
   load_undo();
   return;
}


//  [undo/redo] menu function
//  call m_undo() or m_redo() depending on left/right mouse click on icon

void m_undo_redo(GtkWidget *, cchar *)                                     //  v.14.07
{
   int button = zfuncs::vmenuclickbutton;
   if (button == 1) m_undo(0,0);
   else m_redo(0,0);
   return;
}


//  undo all edits of the current image
//  (discard modifications)

void undo_all() 
{
   TRACE                                                                   //  v.14.04.2

   if (CEF) return;                                                        //  not if edit active        v.13.04
   if (URS_pos == 0) return;

   URS_pos = 0;
   load_undo();
   return;
}


//  Save E0 to undo/redo file stack
//  stack position = URS_pos

void save_undo()
{
   char     *pp, buff[24];
   int      fid, cc, cc2;
   
   TRACE                                                                   //  v.14.04.2

   pp = strstr(URS_filename,"undo_");                                      //  get undo/redo stack filename to use
   if (! pp) zappcrash("undo/redo stack corrupted");                       //  v.13.12
   snprintf(pp+5,3,"%02d",URS_pos);
   
   fid = open(URS_filename,O_WRONLY|O_CREAT|O_TRUNC,0640);                 //  create or overwrite
   if (! fid) goto writefail;

   snprintf(buff,24," %05d %05d fotoxx ",E0pxm->ww,E0pxm->hh);             //  write header
   cc = write(fid,buff,20);
   if (cc != 20) goto writefail;
   
   cc = E0pxm->ww * E0pxm->hh * 3 * sizeof(float);                         //  write image, ww * hh RGB pixels
   cc2 = write(fid,E0pxm->pixels,cc);
   if (cc2 != cc) goto writefail;
   close(fid);
   
   if (URS_pos == 0) {                                                     //  stack posn. 0 = original image
      strcpy(URS_funcs[0],"original");                                     //  function name for original image
      URS_saved[0] = 1;                                                    //  original image already on disk
   }
   else {                                                                  //  stack position 
      if (! CEF) zappcrash("save_undo() no CEF");                          //  must have an edit function
      strcpy(URS_funcs[URS_pos],CEF->funcname);                            //  edit function name
      URS_saved[URS_pos] = 0;                                              //  not yet saved to disk
   }

   return;
   
writefail:                                                                 //  v.14.06
   zmessageACK(Mwin,0,"undo/redo stack write failure: %d \n"
                      "(maybe out of disk space)",errno);
   shell_quiet("rm -R %s",tempdir);
   exit(errno);
}


//  Load E0 from undo/redo file stack
//  stack position = URS_pos

void load_undo()
{
   char     *pp, buff[24], fotoxx[8];
   int      fid, ww, hh, cc, cc2;

   TRACE                                                                   //  v.14.04.2

   pp = strstr(URS_filename,"undo_");                                      //  v.13.12
   if (! pp) zappcrash("undo/redo stack corrupted 1");
   snprintf(pp+5,3,"%02d",URS_pos);
   
   fid = open(URS_filename,O_RDONLY);
   if (! fid) zappcrash("undo/redo stack corrupted 2");
   
   *fotoxx = 0;
   cc = read(fid,buff,20);
   sscanf(buff," %d %d %8s ",&ww, &hh, fotoxx);
   if (! strEqu(fotoxx,"fotoxx")) zappcrash("undo/redo stack corrupted 4");

   mutex_lock(&Fpixmap_lock);

   PXM_free(E0pxm);
   E0pxm = PXM_make(ww,hh);
   cc = ww * hh * 12;
   cc2 = read(fid,E0pxm->pixels,cc);
   if (cc2 != cc) zappcrash("undo/redo stack corrupted 5");
   close(fid);
   
   sa_validate();                                                          //  delete area if invalid

   mutex_unlock(&Fpixmap_lock);                                            //  moved down                         v.14.02
   Fpaintnow();                                                            //  update image data synchronous      v.13.04
   return;
}


/**************************************************************************/

//  check for various pending or busy conditions and message the user
//  returns  1  if something is pending or busy
//           0  if nothing is pending or user decides to discard mods
//  conditions:
//       edit    edit function is active (CEF not null)
//       mods    current file has unsaved modifications
//       lock    a menu lock is active
//       all     check all of the above
//       quiet   suppress user message

int checkpend(cchar *list)                                                 //  v.13.12
{
   int      edit, mods, lock, all, quiet, pend, choice;
   cchar    *modmess = ZTX("This action will discard changes\n");
   cchar    *keep = ZTX("Keep");
   cchar    *discard = ZTX("Discard");
   
   edit = mods = lock = all = quiet = pend = 0;
   
   if (strstr(list,"edit")) edit = 1;
   if (strstr(list,"mods")) mods = 1;
   if (strstr(list,"lock")) lock = 1;
   if (strstr(list,"all")) all = 1;
   if (strstr(list,"quiet")) quiet = 1;
   
   if (all) edit = mods = lock = 1;
   
   if (edit && CEF) {
      pend = 1;
      if (! quiet) zmessageACK(Mwin,0,ZTX("prior function active"));
   }

   if (! pend && mods) {
      if (CEF && CEF->Fmods && ! CEF->Fsaved) pend = 1;                    //  active edits unsaved
      if (URS_pos > 0 && URS_saved[URS_pos] == 0) pend = 1;                //  completed edits unsaved
      if (Fmetachanged) pend = 1;                                          //  metadata edit unsaved
      if (pend && ! quiet) {
         choice = zdialog_choose(0,Mwin,modmess,keep,discard,null);        //  ask user
         if (choice == 2) {                                                //  choice is discard
            if (CEF) edit_cancel(0);                                       //  discard current edit
            if (URS_pos > 0) undo_all();                                   //  undo prior edits
            Fmetachanged = 0;                                              //  discard metadata edits
            pend = 0;
         }
      }
   }
   
   if (! pend && lock && Fmenulock) {
      pend = 1;
      if (! quiet) zmessageACK(Mwin,0,ZTX("prior function active"));
   }

   return pend;                                                            //  1 if something pending
}


/**************************************************************************/

//  zdialog mouse capture and release

void takeMouse(mcbFunc func, GdkCursor *cursor)                            //  capture mouse for dialog
{
   if (! Cdraw) return;
   if (! gdkwin) return;

   freeMouse();
   mouseCBfunc = func;
   Mcapture = 1;
   gdk_window_set_cursor(gdkwin,cursor);
   return;
}

void freeMouse()                                                           //  free mouse for main window
{
   if (! Cdraw) return;
   if (! gdkwin) return;

   if (! Mcapture) return;
   mouseCBfunc = 0;
   Mcapture = 0;
   draw_mousearc(0,0,0,0,1);                                               //  remove mouse circle
   gdk_window_set_cursor(gdkwin,0);                                        //  set normal cursor
   return;
}


/**************************************************************************

   functions to manage working threads

   main level thread management
      start_thread(func,arg)           start thread running
      signal_thread()                  signal thread that work is pending
      wait_thread_idle()               wait for pending work complete
      wrapup_thread(command)           wait for exit or command thread exit

   thread function
      thread_idle_loop()               wait for pending work, exit if commanded
      thread_exit()                    exit thread unconditionally

   thread_status (thread ownership
      0     no thread is running
      1     thread is running and idle (no work)
      2     thread is working
      0     thread has exited

   thread_command (main program ownership)
      0     idle, initial status
      8     exit when pending work is done
      9     exit now, unconditionally

   thread_pend       work requested counter
   thread_done       work done counter
   thread_hiwater    high water mark

***************************************************************************/

//  start thread that does the edit work

void start_thread(threadfunc func, void *arg)
{
   TRACE                                                                   //  v.14.04.2
   CEF->thread_status = 1;                                                 //  thread is running
   CEF->thread_command = CEF->thread_pend = CEF->thread_done 
                       = CEF->thread_hiwater = 0;                          //  nothing pending
   start_detached_thread(func,arg);
   return;
}


//  signal thread that new work is pending

void signal_thread()
{
   TRACE
   if (CEF->thread_status > 0) CEF->thread_pend++;
   return;
}


//  wait for edit thread to complete pending work and become idle

void wait_thread_idle()
{
   TRACE                                                                   //  v.14.04.2
   if (! CEF->thread_status) return;                                       //  v.13.10
   if (CEF->thread_pend == CEF->thread_done) return;                       //  v.13.10
   while (CEF->thread_status && CEF->thread_pend > CEF->thread_done)       //  no zmainloop()
      zsleep(0.01);
   return;
}


//  non-blocking version                                                   //  v.13.09.4  *** kludge ***

void wrapup_thread_noblock(int command)
{
   TRACE                                                                   //  v.14.04.2
   CEF->thread_command = command;                                          //  tell thread to quit or finish
   while (CEF->thread_status > 0) {                                        //  wait for thread to finish
      zsleep(0.01);
      zmainloop();
      if (! CEF) zappcrash("wrapup_thread() CEF = 0");
   }
   return;
}


//  wait for thread exit or command thread exit
//  command = 0    wait for normal completion
//            8    finish pending work and exit
//            9    quit, exit now

void wrapup_thread(int command)
{
   TRACE                                                                   //  v.14.04.2
   CEF->thread_command = command;                                          //  tell thread to quit or finish
   while (CEF->thread_status > 0)                                          //  wait for thread to finish
      zsleep(0.01);                                                        //  no zmainloop()
   return;
}


//  called only from edit threads
//  idle loop - wait for work request or exit command

void thread_idle_loop()
{
   TRACE

   if (CEF->thread_status == 2) zadd_locked(Fthreadbusy,-1);
   CEF->thread_status = 1;                                                 //  status = idle
   CEF->thread_done = CEF->thread_hiwater;                                 //  work done = high-water mark

   while (true)
   {
      if (CEF->thread_command == 9) thread_exit();                         //  quit now command
      if (CEF->thread_command == 8)                                        //  finish work and exit
         if (CEF->thread_pend <= CEF->thread_done) thread_exit();
      if (CEF->thread_pend > CEF->thread_done) break;                      //  wait for work request
      zsleep(0.01);
   }
   
   CEF->thread_hiwater = CEF->thread_pend;                                 //  set high-water mark
   CEF->thread_status = 2;                                                 //  thread is working
   zadd_locked(Fthreadbusy,+1);
   return;                                                                 //  perform edit
}


//  exit thread unconditionally, called only from edit threads

void thread_exit()
{
   TRACE                                                                   //  v.14.04.2
   wait_wthreads();                                                        //  wait for worker threads if any     v.13.04
   if (! CEF) zappcrash("thread_exit() CEF = 0");
   if (CEF->thread_status == 2) zadd_locked(Fthreadbusy,-1);
   CEF->thread_pend = CEF->thread_done = CEF->thread_hiwater = 0;
   CEF->thread_status = 0;
   pthread_exit(0);                                                        //  "return" cannot be used here
}


/**************************************************************************/

//  edit support functions for worker threads (per processor core)

void start_wthread(threadfunc func, void *arg)                             //  start thread and increment busy count
{                                                                          //  may be called from a thread
   TRACE                                                                   //  v.14.04.2
   zadd_locked(Fthreadbusy,+1);
   zadd_locked(wthreads_busy,+1);
   start_detached_thread(func,arg);
   return;
}


void exit_wthread()                                                        //  decrement busy count and exit thread
{
   TRACE                                                                   //  v.14.04.2
   zadd_locked(Fthreadbusy,-1);
   zadd_locked(wthreads_busy,-1);
   pthread_exit(0);                                                        //  "return" cannot be used here
}


void wait_wthreads(int block)                                              //  wait for all worker threads done
{                                                                          //  may be called from thread or non-thread
   TRACE                                                                   //  v.14.04.2
   
   if (CEF) {
      while (wthreads_busy) {
         zsleep(0.01);
         if (block) continue;                                              //  v.14.07
         zmainloop();
      }
      if (! CEF) zappcrash("wait_wthreads() CEF = 0");
   }
   else {
      while (wthreads_busy) {
         zsleep(0.01);
         if (block) continue;
         zmainloop();
      }
   }
   return;
}


/**************************************************************************
      other support functions
***************************************************************************/


//  favorites menu - popup graphic menu with user's favorites

void m_favorites(GtkWidget *, cchar *)
{
   void  favorites_callback(cchar *menu);
   char  menuconfigfile[200];
   
   F1_help_topic = "favorites";
   snprintf(menuconfigfile,200,"%s/menu-config",favorites_dirk);
   gmenuz(Mwin,ZTX("Favorite Functions"),menuconfigfile,favorites_callback);
   return;
}


//  response function for clicked menu
//  a menu function is called as from the text menus

void favorites_callback(cchar *menu)
{
   int      ii;

   m_viewmode(0,"F"); 

   for (ii = 0; ii < Nmenus; ii++)
      if (strcasecmp(ZTX(menu),ZTX(menutab[ii].menu)) == 0) break;
   
   if (ii < Nmenus) menutab[ii].func(0,menu);
   return;
}


//  help menu function

void m_help(GtkWidget *, cchar *menu)
{
   if (strEqu(menu,ZTX("Quick Start")))
      showz_html(quickstart_file);                                         //  v.13.06

   if (strEqu(menu,ZTX("User Guide"))) 
      showz_userguide();
   
   if (strEqu(menu,ZTX("User Guide Changes")))
      showz_userguide("changes");

   if (strEqu(menu,ZTX("README")))
      showz_textfile("doc","README");

   if (strEqu(menu,ZTX("Edit Functions Summary")))
      showz_textfile("data","edit-menus");

   if (strEqu(menu,ZTX("Change Log")))
      showz_textfile("doc","changelog");
   
   if (strEqu(menu,ZTX("Log File")))
      showz_logfile();

   if (strEqu(menu,ZTX("Translations")))
      showz_textfile("doc","translations");
      
   if (strEqu(menu,ZTX("Home Page")))
      showz_html(Fhomepage);

   if (strEqu(menu,ZTX("About")))                                          //  v.13.02
      zmessageACK(Mwin,"Fotoxx"," %s\n %s\n %s\n %s\n %s\n %s\n",
         Frelease,Flicense,Fhomepage,Fsoftware,Fcontact,Ftranslators);
      
   if (strEqu(menu,ZTX("Help")))                                           //  menu button
      showz_userguide(F1_help_topic);                                      //  show topic if there, or page 1

   return;
}


/**************************************************************************/

//  table for loading and saving adjustable parameters between sessions

typedef struct {
   char     name[20];
   char     type[12];
   int      count;
   void     *location;
}  param;

#define Nparms 28
param paramTab[Nparms] = {
//     name                    type        count      location
{     "fotoxx release",       "char",        1,       &Prelease               },
{     "notified release",     "char",        1,       &Nrelease               },
{     "first time",           "int",         1,       &Ffirsttime             },
{     "usage counter",        "int",         1,       &usagecounter           },
{     "window geometry",      "int",         4,       &mwgeom                 },
{     "thumbnail size",       "int",         1,       &navi::thumbsize        },
{     "menu style",           "char",        1,       &menu_style             },
{     "drag option",          "int",         1,       &Fdragopt               },
{     "zoom count",           "int",         1,       &zoomcount              },
{     "interpolation",        "char",        1,       &interpolation          },
{     "startup display",      "char",        1,       &startdisplay           },
{     "start image file",     "char",        1,       &startfile              },
{     "start directory",      "char",        1,       &startdirk              },
{     "grid base",            "int",         10,      &gridsettings[0]        },
{     "grid trim/rotate",     "int",         10,      &gridsettings[1]        },
{     "grid unbend",          "int",         10,      &gridsettings[2]        },
{     "grid warp curved",     "int",         10,      &gridsettings[3]        },
{     "grid warp linear",     "int",         10,      &gridsettings[4]        },
{     "grid warp affine",     "int",         10,      &gridsettings[5]        },
{     "RAW command",          "char",        1,       &DCrawcommand           },
{     "RAW file types",       "char",        1,       &RAWfiletypes           },
{     "trim size",            "int",         2,       &trimsize               },
{     "trim buttons",         "char",        6,       &trimbuttons            },
{     "trim ratios",          "char",        6,       &trimratios             },
{     "edit resize",          "int",         2,       &editresize             },
{     "jpeg def quality",     "int",         1,       &jpeg_def_quality       },
{     "lens mm",              "float",       1,       &lens_mm                },
{     "lens bow",             "float",       1,       &lens_bow               }  };

//  save parameters to file /.../.fotoxx/parameters

void save_params()
{
   FILE        *fid;
   char        buff[1050], text[1000];                                     //  limit for character data cc
   char        *name, *type;
   int         count;
   void        *location;
   char        **charloc;
   int         *intloc;
   float       *floatloc;

   snprintf(buff,199,"%s/parameters",get_zuserdir());                      //  open output file
   fid = fopen(buff,"w");
   if (! fid) return;

   for (int ii = 0; ii < Nparms; ii++)                                     //  write table of state data
   {
      name = paramTab[ii].name;
      type = paramTab[ii].type;
      count = paramTab[ii].count;
      location = paramTab[ii].location;
      charloc = (char **) location;
      intloc = (int *) location;
      floatloc = (float *) location;

      fprintf(fid,"%-20s  %-8s  %02d  ",name,type,count);                  //  write parm name, type, count

      for (int kk = 0; kk < count; kk++)                                   //  write "value" "value" ...
      {
         if (strEqu(type,"char")) {
            if (! *charloc) break;                                         //  missing, discontinue this parameter
            repl_1str(*charloc++,text,"\n","\\n");                         //  replace newlines with "\n"
            fprintf(fid,"  \"%s\"",text);
         }
         if (strEqu(type,"int"))
            fprintf(fid,"  \"%d\"",*intloc++);

         if (strEqu(type,"float"))
            fprintf(fid,"  \"%.2f\"",*floatloc++);
      }

      fprintf(fid,"\n");                                                   //  write EOR
   }
   
   fprintf(fid,"\n");
   fclose(fid);                                                            //  close file

   return;
}


//  load parameters from file /.../.fotoxx/parameters

void load_params()
{
   FILE        *fid;
   int         ii, err, pcount;
   int         Idata;
   float       Fdata;
   char        buff[1000], text[1000], *pp;
   char        name[20], type[12], count[8], data[1000];
   void        *location;
   char        **charloc;
   int         *intloc;
   float       *floatloc;
   
   snprintf(buff,199,"%s/parameters",get_zuserdir());                      //  open parameters file
   fid = fopen(buff,"r");
   if (! fid) return;                                                      //  none, defaults are used

   while (true)                                                            //  read parameters
   {
      pp = fgets_trim(buff,999,fid,1);
      if (! pp) break;
      if (*pp == '#') continue;                                            //  comment
      if (strlen(pp) < 40) continue;                                       //  rubbish

      err = 0;

      strncpy0(name,pp,20);                                                //  parm name
      strTrim2(name);
      
      strncpy0(type,pp+22,8);                                              //  parm type
      strTrim2(type);
      
      strncpy0(count,pp+32,4);                                             //  parm count
      strTrim2(count);
      err = convSI(count,pcount);
      
      strncpy0(data,pp+38,1000);                                           //  parm value(s)
      strTrim2(data);
      
      for (ii = 0; ii < Nparms; ii++)                                      //  match file record to param table
      {
         if (strNeq(name,paramTab[ii].name)) continue;                     //  parm name
         if (strNeq(type,paramTab[ii].type)) continue;                     //  parm type
         if (paramTab[ii].count != pcount) continue;                       //  parm count
         break;
      }
      
      if (ii == Nparms) continue;                                          //  not found, ignore file param
      
      location = paramTab[ii].location;                                    //  get parameter memory location
      charloc = (char **) location;
      intloc = (int *) location;
      floatloc = (float *) location;
      
      for (ii = 1; ii <= pcount; ii++)                                     //  get parameter value(s)
      {
         pp = (char *) strField(data,' ',ii);
         if (! pp) break;

         if (strEqu(type,"char")) {
            repl_1str(pp,text,"\\n","\n");                                 //  replace "\n" with real newlines
            *charloc++ = zstrdup(text);
         }

         if (strEqu(type,"int")) {
            err = convSI(pp,Idata);
            if (err) continue;
            *intloc++ = Idata;
         }

         if (strEqu(type,"float")) {
            err = convSF(pp,Fdata);
            if (err) continue;
            *floatloc++ = Fdata;
         }
      }
   }

   fclose(fid);

   for (ii = 0; ii < Nparms; ii++)                                         //  set any null strings to ""
   {
      if (strNeq(paramTab[ii].type,"char")) continue;
      charloc = (char **) paramTab[ii].location;
      pcount = paramTab[ii].count;
      for (int jj = 0; jj < pcount; jj++)
         if (! charloc[jj]) 
            charloc[jj] = zstrdup("",20);
   }
   
   zoomratio = pow( 2.0, 1.0 / zoomcount);                                 //  set zoom ratio from zoom count

   return;
}


/**************************************************************************/

//  free all resources associated with the current image file

void free_resources(int fkeepundo)
{
   TRACE

   mutex_lock(&Fpixmap_lock);                                              //  lock pixmaps
   
   if (! fkeepundo)
      shell_quiet("rm -f %s/undo_*",tempdir);                              //  remove undo/redo files          v.13.12

   if (Fshutdown) {                                                        //  stop here if shutdown mode
      mutex_unlock(&Fpixmap_lock);
      return;
   }
   
   URS_pos = URS_max = 0;                                                  //  reset undo/redo stack           v.13.04
   Fmetachanged = 0;                                                       //  no unsaved metadata changes     v.13.08
   sa_unselect();                                                          //  unselect select area

   draw_toplines(2);                                                       //  no toplines
   erase_topcircles();                                                     //  no topcircles
   Ntoptext = 0;                                                           //  no toptext                      v.13.08
   Fbusy_goal = 0;
   
   if (curr_file) {
      if (zdsela) zdialog_free(zdsela);                                    //  kill select area dialog if active
      freeMouse();                                                         //  free mouse                      v.13.09
      zfree(curr_file);                                                    //  free image file
      curr_file = 0;
      *paneltext = 0;
   }

   PXB_free(Fpxb);
   PXM_free(E0pxm);
   PXM_free(E1pxm);
   PXM_free(E3pxm);
   PXM_free(ERpxm);
   PXM_free(E8pxm);
   PXM_free(E9pxm);

   mutex_unlock(&Fpixmap_lock);
   return;
}


/**************************************************************************/

//  Get a virtual pixel at location (px,py) (real) in a PXM pixmap.
//  Get the overlapping real pixels and build a composite.
//  Output vpix is float[3] supplied by caller.
//  Returns 1 if OK, 0 if px/py out of limits for pxm.

int vpixel(PXM *pxm, float px, float py, float *vpix)
{
   int            ww, hh, px0, py0;
   float          *pix0, *pix1, *pix2, *pix3;
   float          f0, f1, f2, f3;
   float          red, green, blue;
   
   ww = pxm->ww;
   hh = pxm->hh;

   px0 = px;                                                               //  integer pixel containing (px,py)
   py0 = py;

   if (px0 < 1 || py0 < 1) return 0;                                       //  void edge pixels
   if (px0 > ww-2 || py0 > hh-2) return 0;
   
   pix0 = PXMpix(pxm,px0,py0);                                             //  4 pixels based at (px0,py0)
   pix1 = pix0 + ww * 3;
   pix2 = pix0 + 3;
   pix3 = pix0 + ww * 3 + 3;

   f0 = (px0+1 - px) * (py0+1 - py);                                       //  overlap of (px,py)
   f1 = (px0+1 - px) * (py - py0);                                         //   in each of the 4 pixels
   f2 = (px - px0) * (py0+1 - py);
   f3 = (px - px0) * (py - py0);
   
   red =   f0 * pix0[0] + f1 * pix1[0] + f2 * pix2[0] + f3 * pix3[0];      //  sum the weighted inputs
   green = f0 * pix0[1] + f1 * pix1[1] + f2 * pix2[1] + f3 * pix3[1];
   blue =  f0 * pix0[2] + f1 * pix1[2] + f2 * pix2[2] + f3 * pix3[2];
   if (blue < 0.5) blue = 0;                                               //  edge or voided pixel
   
   vpix[0] = red;
   vpix[1] = green;
   vpix[2] = blue;
   
   if (! blue) return 0;
   return 1;
}


/**************************************************************************/

//  compare two floats for significant difference
//  return:  0  difference not significant
//          +1  d1 > d2
//          -1  d1 < d2

int sigdiff(float d1, float d2, float signf)
{
   float diff = fabsf(d1-d2);
   if (diff == 0.0) return 0;
   diff = diff / (fabsf(d1) + fabsf(d2));
   if (diff < signf) return 0;
   if (d1 > d2) return 1;
   else return -1;
}


/**************************************************************************
      PXM pixmap functions - RGB float pixel map
      pixel RGB values may range from 0.0 to 255.99
***************************************************************************/


//  audit the contents of a PXM pixmap                                     //  v.13.10

void PXM_audit(PXM *pxm)
{   
   int      px, py;
   float    bright, *pix;

   for (py = 0; py < pxm->hh; py++)
   for (px = 0; px < pxm->ww; px++)
   {
      pix = PXMpix(pxm,px,py);
      bright = pixbright(pix);
      if (int(bright) < 0 || int(bright) > 255)
         zappcrash("PXM_audit: px/py %d/%d  RGB %.2f %.2f %.2f  bright %.2f %d \n",
                                px,py,pix[0],pix[1],pix[2],bright,int(bright));
   }
   return;
}


//  initialize PXM pixmap - allocate memory

PXM * PXM_make(int ww, int hh)
{
   uint     cc;

   TRACE
   
   if (ww < 5 || hh < 5) {                                                 //  impose reasonableness limits
      zmessageACK(Mwin,0,"image too small: %d %d",ww,hh);
      return 0;
   }

   cc = ww * hh;
   if (ww > wwhh_limit || hh > wwhh_limit || cc > 178956000) {             //  avoid image > 2 GB memory          v.14.07
      zmessageACK(Mwin,0,"image too big: %d %d",ww,hh);
      exit(12);
   }

   PXM *pxm = (PXM *) zmalloc(sizeof(PXM));
   pxm->ww = ww;
   pxm->hh = hh;
   strcpy(pxm->wmi,"pxmpix");

   cc = cc * 12;                                                           //  float[3] per pixel
   pxm->pixels = (float *) zmalloc(cc);
   memset(pxm->pixels,0,cc);                                               //  clear pixels to zero

   return pxm;
}


//  free PXM pixmap

void PXM_free(PXM *&pxm)
{
   TRACE

   if (! pxm) return;
   if (! strEqu(pxm->wmi,"pxmpix")) 
      zappcrash("PXM_free(), bad PXM");
   strcpy(pxm->wmi,"xxxxxx");
   zfree(pxm->pixels);
   zfree(pxm);
   pxm = 0;
   return;
}


//  create a copy of a PXM pixmap

PXM * PXM_copy(PXM *pxm1)
{
   int      cc;
   PXM      *pxm2;

   TRACE
   pxm2 = PXM_make(pxm1->ww, pxm1->hh);
   cc = pxm1->ww * pxm1->hh * 3 * sizeof(float);
   memcpy(pxm2->pixels,pxm1->pixels,cc);
   return pxm2;
}


//  create a copy of a PXM rectangular area

PXM * PXM_copy_area(PXM *pxm1, int orgx, int orgy, int ww2, int hh2)
{
   float    *pix1, *pix2;
   PXM      *pxm2 = 0;
   int      px1, py1, px2, py2;

   TRACE

   pxm2 = PXM_make(ww2,hh2);
  
   for (py1 = orgy, py2 = 0; py2 < hh2; py1++, py2++) 
   {
      pix1 = PXMpix(pxm1,orgx,orgy);
      pix2 = PXMpix(pxm2,0,py2);

      for (px1 = orgx, px2 = 0; px2 < ww2; px1++, px2++)
      {
         pix2[0] = pix1[0];
         pix2[1] = pix1[1];
         pix2[2] = pix1[2];
         pix1 += 3;
         pix2 += 3;
      }
   }

   return pxm2;
}


//  replace blue = 0 pixels with blue = 1
//  (blue = 0 reserved for pixels voided by warp or overlay offsets)
//  needs about 0.013 seconds for 10 megapixel image and 3.3 GHz CPU

void PXM_fixblue(PXM *pxm)
{
   int      size;
   float    *pixel, *pixel0, *pixelN;

   size = pxm->ww * pxm->hh * 3;
   pixel0 = pxm->pixels + 2;
   pixelN = pixel0 + size;

   for (pixel = pixel0; pixel < pixelN; pixel += 3)                        //  +3 is +12 bytes
      if (! *pixel) *pixel = 1.0;

   return;
}


/**************************************************************************

   Rescale PXM image to new width and height.
   The scale ratios may be different for width and height.

   Method: 
   The input and output images are overlayed, stretching or shrinking the
   output pixels as needed. The contribution of each input pixel overlapping
   an output pixel is proportional to the area of the output pixel covered by
   the input pixel. The contributions of all overlaping input pixels are added.
   The work is spread among Nwt threads to reduce time on SMP processors.

   Example: if the output image is 40% of the input image, then:
     outpix[0,0] = 0.16 * inpix[0,0] + 0.16 * inpix[1,0] + 0.08 * inpix[2,0]
                 + 0.16 * inpix[0,1] + 0.16 * inpix[1,1] + 0.08 * inpix[2,1]
                 + 0.08 * inpix[0,2] + 0.08 * inpix[1,2] + 0.04 * inpix[2,2]

***************************************************************************/

namespace pxmrescale {
   float    *bmp1;
   float    *bmp2;
   int      ww1;
   int      hh1;
   int      ww2;
   int      hh2;
   int      *px1L;
   int      *py1L;
   float    *pxmap;
   float    *pymap;
   int      maxmapx;
   int      maxmapy;
   int      busy[max_threads];
}

PXM * PXM_rescale(PXM *pxm1, int ww, int hh)
{
   using namespace pxmrescale;

   void * pxm_rescale_thread(void *arg);

   PXM         *pxm2;
   int         px1, py1, px2, py2;
   int         pxl, pyl, pxm, pym, ii;
   float       scalex, scaley;
   float       px1a, py1a, px1b, py1b;
   float       fx, fy;

   TRACE

   ww1 = pxm1->ww;                                                         //  input PXM
   hh1 = pxm1->hh;
   bmp1 = pxm1->pixels;
   
   pxm2 = PXM_make(ww,hh);                                                 //  output PXM
   if (! pxm2) return 0;
   ww2 = ww;
   hh2 = hh;
   bmp2 = pxm2->pixels;
   
   memset(bmp2, 0, ww2 * hh2 * 3 * sizeof(float));                         //  clear output pixmap

   scalex = 1.0 * ww1 / ww2;                                               //  compute x and y scales
   scaley = 1.0 * hh1 / hh2;
   
   if (scalex <= 1) maxmapx = 2;                                           //  compute max input pixels
   else maxmapx = scalex + 2;                                              //    mapping into output pixels
   maxmapx += 1;                                                           //      for both dimensions
   if (scaley <= 1) maxmapy = 2;                                           //  (pixels may not be square)
   else maxmapy = scaley + 2;
   maxmapy += 1;                                                           //  (extra entry for -1 flag)
   
   pymap = (float *) zmalloc(hh2 * maxmapy * sizeof(float));               //  maps overlap of < maxmap input
   pxmap = (float *) zmalloc(ww2 * maxmapx * sizeof(float));               //    pixels per output pixel

   py1L = (int *) zmalloc(hh2 * sizeof(int));                              //  maps first (lowest) input pixel
   px1L = (int *) zmalloc(ww2 * sizeof(int));                              //    per output pixel

   for (py2 = 0; py2 < hh2; py2++)                                         //  loop output y-pixels
   {
      py1a = py2 * scaley;                                                 //  corresponding input y-pixels
      py1b = py1a + scaley;
      if (py1b >= hh1) py1b = hh1 - 0.001;                                 //  fix precision limitation
      pyl = py1a;
      py1L[py2] = pyl;                                                     //  1st overlapping input pixel

      for (py1 = pyl, pym = 0; py1 < py1b; py1++, pym++)                   //  loop overlapping input pixels
      {
         if (py1 < py1a) {                                                 //  compute amount of overlap
            if (py1+1 < py1b) fy = py1+1 - py1a;                           //    0.0 to 1.0 
            else fy = scaley;
         }
         else if (py1+1 > py1b) fy = py1b - py1;
         else fy = 1;

         ii = py2 * maxmapy + pym;                                         //  save it
         pymap[ii] = 0.9999 * fy / scaley;
      }
      ii = py2 * maxmapy + pym;                                            //  set an end marker after
      pymap[ii] = -1;                                                      //    last overlapping pixel
   }
   
   for (px2 = 0; px2 < ww2; px2++)                                         //  do same for x-pixels
   {
      px1a = px2 * scalex;
      px1b = px1a + scalex;
      if (px1b >= ww1) px1b = ww1 - 0.001;
      pxl = px1a;
      px1L[px2] = pxl;

      for (px1 = pxl, pxm = 0; px1 < px1b; px1++, pxm++)
      {
         if (px1 < px1a) {
            if (px1+1 < px1b) fx = px1+1 - px1a;
            else fx = scalex;
         }
         else if (px1+1 > px1b) fx = px1b - px1;
         else fx = 1;

         ii = px2 * maxmapx + pxm;
         pxmap[ii] = 0.9999 * fx / scalex;
      }
      ii = px2 * maxmapx + pxm;
      pxmap[ii] = -1;
   }

   for (ii = 0; ii < Nwt; ii++) {                                          //  start working threads
      busy[ii] = 1;
      start_detached_thread(pxm_rescale_thread,&Nval[ii]);
   }
   
   for (ii = 0; ii < Nwt; ii++)                                            //  wait for all done
      while (busy[ii]) zsleep(0.004);

   zfree(px1L);
   zfree(py1L);
   zfree(pxmap);
   zfree(pymap);

   return pxm2;
}

void * pxm_rescale_thread(void *arg)                                       //  worker thread function
{
   using namespace pxmrescale;

   int         index = *((int *) arg);
   int         px1, py1, px2, py2;
   int         pxl, pyl, pxm, pym, ii;
   float       *pixel1, *pixel2;
   float       fx, fy, ftot;
   float       red, green, blue;

   for (py2 = index; py2 < hh2; py2 += Nwt)                                //  loop output y-pixels
   {
      pyl = py1L[py2];                                                     //  corresp. 1st input y-pixel

      for (px2 = 0; px2 < ww2; px2++)                                      //  loop output x-pixels
      {
         pxl = px1L[px2];                                                  //  corresp. 1st input x-pixel

         red = green = blue = 0;                                           //  initz. output pixel

         for (py1 = pyl, pym = 0; ; py1++, pym++)                          //  loop overlapping input y-pixels
         {
            ii = py2 * maxmapy + pym;                                      //  get y-overlap
            fy = pymap[ii];
            if (fy < 0) break;                                             //  no more pixels

            for (px1 = pxl, pxm = 0; ; px1++, pxm++)                       //  loop overlapping input x-pixels
            {
               ii = px2 * maxmapx + pxm;                                   //  get x-overlap
               fx = pxmap[ii];
               if (fx < 0) break;                                          //  no more pixels

               ftot = fx * fy;                                             //  area overlap = x * y overlap
               pixel1 = bmp1 + (py1 * ww1 + px1) * 3;
               red += pixel1[0] * ftot;                                    //  add input pixel * overlap
               green += pixel1[1] * ftot;
               blue += pixel1[2] * ftot;
            }

            pixel2 = bmp2 + (py2 * ww2 + px2) * 3;                         //  save output pixel
            pixel2[0] = red;
            pixel2[1] = green;
            pixel2[2] = blue;
         }
      }
   }

   busy[index] = 0;
   return 0;
}


/**************************************************************************

      PXM *pxm2 = PXM_rotate(PXM *pxm1, float angle)

      Rotate PXM pixmap through an arbitrary angle (degrees).

      The returned image has the same size as the original, but the
      pixmap size is increased to accomodate the rotated image.
      (e.g. a 100x100 image rotated 45 deg. needs a 142x142 pixmap).

      The space added around the rotated image is black (RGB 0,0,0).
      Angle is in degrees. Positive direction is clockwise.
      Speed is about 28 million pixels/sec/thread for a 3.3 GHz CPU.
      Loss of resolution is less than 1 pixel.
      
      Work is divided among Nwt threads to gain speed.
      
***************************************************************************/

namespace pxmrotate {
   int      busy = 0;
   float    *bmp1;
   float    *bmp2;
   int      ww1;
   int      hh1;
   int      ww2;
   int      hh2;
   float    angle;
}

PXM * PXM_rotate(PXM *pxm1, float anglex)
{
   using namespace pxmrotate;

   void     *PXM_rotate_thread(void *);

   int      ii;
   PXM      *pxm2;
   
   TRACE

   ww1 = pxm1->ww;                                                         //  input PXM
   hh1 = pxm1->hh;
   bmp1 = pxm1->pixels;
   angle = anglex;

   while (angle < -180) angle += 360;                                      //  normalize, -180 to +180
   while (angle > 180) angle -= 360;
   angle = angle * PI / 180;                                               //  radians, -PI to +PI
   
   if (fabsf(angle) < 0.001) {                                             //  angle = 0 within my precision
      pxm2 = PXM_copy(pxm1);                                               //  return a copy of the input PXM
      return pxm2;
   }

   ww2 = ww1*fabsf(cosf(angle)) + hh1*fabsf(sinf(angle));                  //  rectangle containing rotated image
   hh2 = ww1*fabsf(sinf(angle)) + hh1*fabsf(cosf(angle));
   
   pxm2 = PXM_make(ww2,hh2);                                               //  output PXM
   bmp2 = pxm2->pixels;

   for (ii = 0; ii < Nwt; ii++)                                            //  start worker threads
      start_detached_thread(PXM_rotate_thread,&Nval[ii]);
   zadd_locked(busy,+Nwt);

   while (busy) zsleep(0.004);                                             //  wait for completion
   return pxm2;
}

void * PXM_rotate_thread(void *arg)
{
   using namespace pxmrotate;

   int      index = *((int *) (arg));
   int      px2, py2, px0, py0;
   float    *pix0, *pix1, *pix2, *pix3;
   float    px1, py1;
   float    f0, f1, f2, f3, red, green, blue;
   float    a, b, d, e, ww15, hh15, ww25, hh25;

   ww15 = 0.5 * ww1;
   hh15 = 0.5 * hh1;
   ww25 = 0.5 * ww2;
   hh25 = 0.5 * hh2;

   a = cosf(angle);
   b = sinf(angle);
   d = - sinf(angle);
   e = cosf(angle);
   
   for (py2 = index; py2 < hh2; py2 += Nwt)                                //  loop through output pixels
   for (px2 = 0; px2 < ww2; px2++)
   {
      px1 = a * (px2 - ww25) + b * (py2 - hh25) + ww15;                    //  (px1,py1) = corresponding
      py1 = d * (px2 - ww25) + e * (py2 - hh25) + hh15;                    //    point within input pixels

      px0 = px1;                                                           //  pixel containing (px1,py1)
      py0 = py1;
      
      if (px0 < 0 || px0 > ww1-2 || py0 < 0 || py0 > hh1-2) {              //  if outside input pixel array       v.13.09
         pix2 = bmp2 + (py2 * ww2 + px2) * 3;                              //    output is black
         pix2[0] = pix2[1] = pix2[2] = 0;
         continue;
      }

      pix0 = bmp1 + (py0 * ww1 + px0) * 3;                                 //  4 input pixels based at (px0,py0)
      pix1 = pix0 + ww1 * 3;
      pix2 = pix0 + 3;
      pix3 = pix1 + 3;

      f0 = (px0+1 - px1) * (py0+1 - py1);                                  //  overlap of (px1,py1)
      f1 = (px0+1 - px1) * (py1 - py0);                                    //    in each of the 4 pixels
      f2 = (px1 - px0) * (py0+1 - py1);
      f3 = (px1 - px0) * (py1 - py0);
   
      red =   f0 * pix0[0] + f1 * pix1[0] + f2 * pix2[0] + f3 * pix3[0];   //  sum the weighted inputs
      green = f0 * pix0[1] + f1 * pix1[1] + f2 * pix2[1] + f3 * pix3[1];
      blue =  f0 * pix0[2] + f1 * pix1[2] + f2 * pix2[2] + f3 * pix3[2];
      
      pix2 = bmp2 + (py2 * ww2 + px2) * 3;                                 //  output pixel
      pix2[0] = red;
      pix2[1] = green;
      pix2[2] = blue;
   }
   
   zadd_locked(busy,-1);
   return 0;
}


/**************************************************************************
      PXB pixmap functions - RGB uint8 pixel map and GdkPixbuf wrapper
***************************************************************************/

//  Create PXB pixmap with pixels cleared to zero

PXB * PXB_make(int ww, int hh)
{
   TRACE

   PXB *pxb = (PXB *) zmalloc(sizeof(PXB));
   strcpy(pxb->wmi,"pxbpix");
   pxb->pixbuf = gdk_pixbuf_new(GDKRGB,0,8,ww,hh);
   if (! pxb->pixbuf) {
      zmessageACK(Mwin,0,"memory allocation failure");                     //  v.14.03
      exit(12);
   }
   pxb->pixels = gdk_pixbuf_get_pixels(pxb->pixbuf);
   pxb->ww = ww;
   pxb->hh = hh;
   pxb->rs = gdk_pixbuf_get_rowstride(pxb->pixbuf);
   int cc = hh * pxb->rs;
   memset(pxb->pixels,0,cc);
   return pxb;
}


//  Create a PXB pixmap from a GdkPixbuf
//  The pixbuf is used directly and is not duplicated

PXB * PXB_make(GdkPixbuf *pixbuf1) 
{
   TRACE

   PXB *pxb2 = (PXB *) zmalloc(sizeof(PXB));
   strcpy(pxb2->wmi,"pxbpix");
   pxb2->pixbuf = pixbuf1;
   pxb2->ww = gdk_pixbuf_get_width(pixbuf1);
   pxb2->hh = gdk_pixbuf_get_height(pixbuf1);
   pxb2->rs = gdk_pixbuf_get_rowstride(pixbuf1);
   pxb2->pixels = gdk_pixbuf_get_pixels(pixbuf1);
   return pxb2;
}


//  Free PXB pixmap - release memory

void PXB_free(PXB *&pxb)
{
   TRACE

   if (! pxb) return;
   if (! strEqu(pxb->wmi,"pxbpix")) 
      zappcrash("PXB_free(), bad PXB");
   strcpy(pxb->wmi,"xxxxxx");
   g_object_unref(pxb->pixbuf);
   zfree(pxb);
   pxb = 0;
   return;
}


//  Copy a PXB pixmap to a new PXB pixmap

PXB * PXB_copy(PXB *pxb1)
{
   TRACE

   PXB *pxb2 = (PXB *) zmalloc(sizeof(PXB));
   strcpy(pxb2->wmi,"pxbpix");
   pxb2->pixbuf = gdk_pixbuf_copy(pxb1->pixbuf);
   if (! pxb2->pixbuf) {
      zmessageACK(Mwin,0,"memory allocation failure");                     //  v.14.03
      exit(12);
   }
   pxb2->ww = pxb1->ww;
   pxb2->hh = pxb1->hh;
   pxb2->rs = pxb1->rs;
   pxb2->pixels = gdk_pixbuf_get_pixels(pxb2->pixbuf);
   return pxb2;
}


//  Rescale PXB pixmap to given width and height.

PXB * PXB_rescale(PXB *pxb1, int ww2, int hh2)
{
   GdkPixbuf   *pixbuf2 = 0;
   
   TRACE

   if (strEqu(interpolation,"tiles"))                                      //  v.13.05
      pixbuf2 = gdk_pixbuf_scale_simple(pxb1->pixbuf,ww2,hh2,GDK_INTERP_TILES);
   else if (strEqu(interpolation,"hyperbolic"))
      pixbuf2 = gdk_pixbuf_scale_simple(pxb1->pixbuf,ww2,hh2,GDK_INTERP_HYPER);
   else pixbuf2 = gdk_pixbuf_scale_simple(pxb1->pixbuf,ww2,hh2,GDK_INTERP_BILINEAR);

   if (! pixbuf2) {
      zmessageACK(Mwin,0,"memory allocation failure");                     //  v.14.03
      exit(12);
   }

   PXB *pxb2 = PXB_make(pixbuf2);
   return pxb2;
}


//  Copy a PXM image (RGB float) to a PXB image (RGB uint8).

PXB * PXM_PXB_copy(PXM *pxm1)
{
   float    *pix1;
   uint8    *pix2;
   PXB      *pxb2;
   int      ww, hh, px, py;

   TRACE

   ww = pxm1->ww;
   hh = pxm1->hh;

   pxb2 = PXB_make(ww,hh);

   for (py = 0; py < hh; py++) 
   {
      pix1 = PXMpix(pxm1,0,py); 
      pix2 = PXBpix(pxb2,0,py); 

      for (px = 0; px < ww; px++)
      {
         pix2[0] = pix1[0];
         pix2[1] = pix1[1];
         pix2[2] = pix1[2];
         pix1 += 3;
         pix2 += 3;
      }
   }

   return pxb2;
}


//  Update a PXB section from an updated PXM section.
//  The PXM and PXB must have the same dimensions.
//  px3, py3, ww3, hh3: modified section within pxm1 to propagate to pxb2;

void PXM_PXB_update(PXM *pxm1, PXB *pxb2, int px3, int py3, int ww3, int hh3)
{
   float    *pix1;
   uint8    *pix2;
   int      px, py;
   int      lox, hix, loy, hiy;
   
   TRACE

   lox = px3;
   hix = px3 + ww3;
   if (lox < 0) lox = 0;
   if (hix > pxb2->ww) hix = pxb2->ww;
   
   loy = py3;
   hiy = py3 + hh3;
   if (loy < 0) loy = 0;
   if (hiy > pxb2->hh) hiy = pxb2->hh;
   
   for (py = loy; py < hiy; py++)
   {
      pix1 = PXMpix(pxm1,lox,py);
      pix2 = PXBpix(pxb2,lox,py);
      
      for (px = lox; px < hix; px++)
      {
         pix2[0] = pix1[0];                                                //  0.0 - 255.99  >>  0 - 255
         pix2[1] = pix1[1];
         pix2[2] = pix1[2];
         pix1 += 3;
         pix2 += 3;
      }
   }
   
   return;
}


//  Update an output PXB section from a corresponding input PXB section.
//  The two PXBs represent the same image at different scales (width/height).
//  px3, py3, ww3, hh3: modified section within pxb1 to propagate to pxb2;

void PXB_PXB_update(PXB *pxb1, PXB *pxb2, int px3, int py3, int ww3, int hh3)
{
   static int     pww1 = 0, phh1 = 0, pww2 = 0, phh2 = 0;
   static int     *px1L = 0, *py1L = 0;
   static float   scalex = 1, scaley = 1;
   static float   *pxmap = 0, *pymap = 0;
   static int     maxmapx = 0, maxmapy = 0;

   uint8       *bmp1, *bmp2;
   int         ww1, hh1, ww2, hh2, rs1, rs2;
   int         px1, py1, px2, py2;
   int         pxl, pyl, pxm, pym, ii;
   float       px1a, py1a, px1b, py1b;
   float       fx, fy, ftot;
   uint8       *pixel1, *pixel2;
   float       red, green, blue;
   int         lox, hix, loy, hiy;
   
   TRACE

   ww1 = pxb1->ww;
   hh1 = pxb1->hh;
   rs1 = pxb1->rs;
   bmp1 = pxb1->pixels;
   
   ww2 = pxb2->ww;
   hh2 = pxb2->hh;
   rs2 = pxb2->rs;
   bmp2 = pxb2->pixels;
   
   if (ww1 == pww1 && hh1 == phh1 && ww2 == pww2 && hh2 == phh2)           //  if the sizes are the same as before,
      goto copy_pixels;                                                    //    the pixel mapping math can be avoided.
   
   pww1 = ww1;
   phh1 = hh1;
   pww2 = ww2;
   phh2 = hh2;
   
   if (px1L) {                                                             //  unless this is the first call,
      zfree(px1L);                                                         //    free prior map memory
      zfree(py1L);
      zfree(pxmap);
      zfree(pymap);
   }

   scalex = 1.0 * ww1 / ww2;                                               //  compute x and y scales
   scaley = 1.0 * hh1 / hh2;

   if (scalex <= 1) maxmapx = 2;                                           //  compute max input pixels
   else maxmapx = scalex + 2;                                              //    mapping into output pixels
   maxmapx += 1;                                                           //      for both dimensions
   if (scaley <= 1) maxmapy = 2;                                           //  (pixels may not be square)
   else maxmapy = scaley + 2;
   maxmapy += 1;                                                           //  (extra entry for -1 flag)

   pymap = (float *) zmalloc(hh2 * maxmapy * sizeof(float));               //  maps overlap of < maxmap input
   pxmap = (float *) zmalloc(ww2 * maxmapx * sizeof(float));               //    pixels per output pixel

   py1L = (int *) zmalloc(hh2 * sizeof(int));                              //  maps first (lowest) input pixel
   px1L = (int *) zmalloc(ww2 * sizeof(int));                              //    per output pixel

   for (py2 = 0; py2 < hh2; py2++)                                         //  loop output y-pixels
   {
      py1a = py2 * scaley;                                                 //  corresponding input y-pixels
      py1b = py1a + scaley;
      if (py1b >= hh1) py1b = hh1 - 0.001;                                 //  fix precision limitation
      pyl = py1a;
      py1L[py2] = pyl;                                                     //  1st overlapping input pixel

      for (py1 = pyl, pym = 0; py1 < py1b; py1++, pym++)                   //  loop overlapping input pixels
      {
         if (py1 < py1a) {                                                 //  compute amount of overlap
            if (py1+1 < py1b) fy = py1+1 - py1a;                           //    0.0 to 1.0 
            else fy = scaley;
         }
         else if (py1+1 > py1b) fy = py1b - py1;
         else fy = 1;

         ii = py2 * maxmapy + pym;                                         //  save it
         pymap[ii] = 0.9999 * fy / scaley;
      }
      ii = py2 * maxmapy + pym;                                            //  set an end marker after
      pymap[ii] = -1;                                                      //    last overlapping pixel
   }
   
   for (px2 = 0; px2 < ww2; px2++)                                         //  do same for x-pixels
   {
      px1a = px2 * scalex;
      px1b = px1a + scalex;
      if (px1b >= ww1) px1b = ww1 - 0.001;
      pxl = px1a;
      px1L[px2] = pxl;

      for (px1 = pxl, pxm = 0; px1 < px1b; px1++, pxm++)
      {
         if (px1 < px1a) {
            if (px1+1 < px1b) fx = px1+1 - px1a;
            else fx = scalex;
         }
         else if (px1+1 > px1b) fx = px1b - px1;
         else fx = 1;

         ii = px2 * maxmapx + pxm;
         pxmap[ii] = 0.9999 * fx / scalex;
      }
      ii = px2 * maxmapx + pxm;
      pxmap[ii] = -1;
   }

copy_pixels:

   px3 = px3 / scalex;                                                     //  convert input area to output area
   py3 = py3 / scaley;
   ww3 = ww3 / scalex + 2;
   hh3 = hh3 / scaley + 2;
   
   lox = px3;
   hix = px3 + ww3;
   if (lox < 0) lox = 0;
   if (hix > ww2) hix = ww2;
   
   loy = py3;
   hiy = py3 + hh3;
   if (loy < 0) loy = 0;
   if (hiy > hh2) hiy = hh2;
   
   for (py2 = loy; py2 < hiy; py2++)                                       //  loop output y-pixels
   {
      pyl = py1L[py2];                                                     //  corresp. 1st input y-pixel

      for (px2 = lox; px2 < hix; px2++)                                    //  loop output x-pixels
      {
         pxl = px1L[px2];                                                  //  corresp. 1st input x-pixel

         red = green = blue = 0;                                           //  initz. output pixel

         for (py1 = pyl, pym = 0; ; py1++, pym++)                          //  loop overlapping input y-pixels
         {
            ii = py2 * maxmapy + pym;                                      //  get y-overlap
            fy = pymap[ii];
            if (fy < 0) break;                                             //  no more pixels

            for (px1 = pxl, pxm = 0; ; px1++, pxm++)                       //  loop overlapping input x-pixels
            {
               ii = px2 * maxmapx + pxm;                                   //  get x-overlap
               fx = pxmap[ii];
               if (fx < 0) break;                                          //  no more pixels

               ftot = fx * fy;                                             //  area overlap = x * y overlap
               pixel1 = bmp1 + py1 * rs1 + px1 * 3;
               red += pixel1[0] * ftot;                                    //  add input pixel * overlap
               green += pixel1[1] * ftot;
               blue += pixel1[2] * ftot;
            }

            pixel2 = bmp2 + py2 * rs2 + px2 * 3;                           //  save output pixel
            pixel2[0] = red;                                               //  0.0 - 255.996  >>  0 - 255
            pixel2[1] = green;
            pixel2[2] = blue;
         }
      }
   }

   return;
}



