/**************************************************************************

   Fotoxx      edit photos and manage collections

   Copyright 2007-2014 Michael Cornelison
   Source URL: http://kornelix.com/fotoxx
   Contact: kornelix@posteo.de
   
   This program is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program. If not, see http://www.gnu.org/licenses/.

***************************************************************************

   Fotoxx image edit - file menu functions

   m_clone                 start a new parallel instance of Fotoxx
   m_open                  open image file menu function
   m_open_drag             open drag/drop image file
   m_previous              open the previous image file
   m_recentfiles           show gallery of recently viewed files
   m_newfiles              show gallery of new (recently added) image files
   m_prev_next             open previous or next file in current gallery
   m_ufraw                 open a camera RAW file using the Ufraw program
   m_rawtherapee           open a camera RAW file using the RawTherapee program
   m_file_save             save a (modified) image file to disk
   file_save_as            dialog to save an image file with a designated file name
   m_rename                rename current image file or clicked thumbnail
   m_create                create a new monocolor image
   create_blank_file       callable function to create a new monocolor image
   m_copytoloc             copy an image file to a new location (duplicate)
   m_movetoloc             move an image file to a new location
   copy_move               callable function to copy or move an image file
   m_trash                 move an image file to trash location
   m_delete                delete an image file
   m_print                 print an image file
   m_quit                  controlled exit with warning of unsaved changes
   quitxx                  unconditional exit
   copyto_clip             copy current image file to the clipboard
   add_recent_file         add an image file to the list of recent files
   set_mwin_title          update the main window title bar
   find_imagefiles         find all image files under a given directory path
   raw_to_tiff             convert a RAW file name to equivalent .tiff name
   f_preload               preload image files ahead of need
   f_open                  open and display an image file, initz. internal parameters
   f_open_saved            open the last image file saved
   f_save                  save an image file to disk (replace, new version, new file)
   PXB_load                load an image file into a PXB pixmap structure (8-bit RGB)
   PXM_load                load an image file into a PXM pixmap structure (float RGB)
   TIFF_PXB_load           load a .tiff file into a PXB pixmap structure (8-bit RGB)
   TIFF_PXM_load           load a .tiff file into a PXM pixmap structure (float RGB)
   PXM_TIFF_save           save a PXM pixmap to a .tiff file (8/16-bit RGB)
   PNG_PXB_load            load a .png file into a PXB pixmap structure (8-bit RGB)
   PNG_PXM_load            load a .png file into a PXM pixmap structure (float RGB)
   PXM_PNG_save            save a PXM pixmap to a .png file (8/16-bit RGB)
   ANY_PXB_load            load other image file into a PXB pixmap structure (8-bit RGB)
   ANY_PXM_load            load other image file into a PXM pixmap structure (float RGB)
   PXM_ANY_save            save a PXM pixmap to other file type (8-bit RGB)
   RAW_PXB_load            load a RAW file into a PXB pixmap structure (8-bit RGB)
   RAW_PXM_load            load a RAW file into a PXM pixmap structure (float RGB)
   gallery                 create/update/search/paint an image gallery
   navi::navigate             create/update/sort/search/paint an image gallery
   navi::gallery_comp         special file compare function for galleries
   navi::gallery_paint        gallery window paint function
   navi::gallery_paintmeta      "" with selected metadata
   navi::draw_text            draw thumbnail text in gallery window
   navi::menufuncx            gallery menu function (scroll, jump, open folder ...)
   navi::changedirk           show gallery from mouse click on a parent directory
   navi::newtop               show gallery from selected top image directory
   navi::getcoll              list collections and display gallery for selected one
   navi::gallery_sort         sort gallery by name, file date, image date
   navi::mouse_event          handle mouse click and drag in a gallery
   navi::KBrelease            handle KB navigation for a gallery
   set_gwin_title          update the gallery window title bar
   gallery_find            get Nth file in a gallery
   gallery_position        get a file position within a gallery
   image_file_type         determine file type (directory, image, RAW, thumbnail, other)
   thumb2imagefile         get corresponding image file for given thumbnail file
   image2thumbfile         get corresponding thumbnail file for given image file
   image_thumbfile         retrieve or create the thumbnail file for given image file
   image_thumbnail         get thumbnail pixbuf for an image file from cache or disk
   get_thumbnail_pixbuf    make thumbnail pixbuf from an image file
   popimage                popup a larger image from a clicked thumbnail
   gallery_monitor         monitor directory for file changes and update gallery
   gallery_getfiles        gallery file selection function and dialog

***************************************************************************/

#define EX extern                                                          //  disable extern declarations
#include "fotoxx.h"                                                        //  (variables in fotoxx.h are refs)

/**************************************************************************/


//  start a new parallel instance of fotoxx
//  new window is slightly down and right from old window

void m_clone(GtkWidget *, cchar *)
{
   int      cc, xx, yy, ww, hh;
   char     progexe[300];

   F1_help_topic = "clone";
   
   gtk_window_get_position(MWIN,&xx,&yy);                                  //  get window position and size
   gtk_window_get_size(MWIN,&ww,&hh);

   cc = readlink("/proc/self/exe",progexe,300);                            //  get own program path
   if (cc <= 0) {
      zmessageACK(Mwin,0,"cannot get /proc/self/exe");
      return;
   }
   progexe[cc] = 0;

   snprintf(command,ccc,"%s -c %d %d %d %d -lang %s",progexe,xx,yy,ww,hh,zfuncs::zlang);
   if (curr_file) strncatv(command,ccc," \"",curr_file,"\"",null);

   strcat(command," &");                                                   //  start new instance and pass
   shell_ack(command);                                                     //    my window posn and size
   return;
}


/**************************************************************************/

//  open file menu function

void m_open(GtkWidget *, cchar *)
{
   F1_help_topic = "open_image_file";
   f_open(null);
   return;
}


/**************************************************************************/

//  open drag-drop file

void m_open_drag(int x, int y, char *file)
{
   F1_help_topic = "open_image_file";
   f_open(file);
   return;
}


/**************************************************************************/

//  open the previous file opened (not the same as toolbar [prev] button)
//  repeated use will cycle back and forth between two most recent files

void m_previous(GtkWidget *, cchar *menu)                                  //  v.14.03
{
   FILE     *fid;
   char     *file, buff[maxfcc];
   int      Nth = 0, err;
   float    gzoom;

   F1_help_topic = "open_previous_file";
   
   gzoom = Cstate->fzoom;

   fid = fopen(recentfiles_file,"r");
   if (! fid) return;

   if (menu && strNeq(menu,"first"))                                       //  if not first file, skip one
      file = fgets_trim(buff,maxfcc,fid,1);

   while (true) 
   {
      file = fgets_trim(buff,maxfcc,fid,1);
      if (! file) break;
      err = f_open(file,Nth,0,0,0);                                        //  v.14.03
      if (! err) break;
   }
   
   fclose(fid);
   
   Cstate->fzoom = gzoom;                                                  //  keep zoom size                  v.14.08
   Fpaint2();
   return;
}


/**************************************************************************/

//  show recently seen image files (edited or only viewed)
//  optionally update gallery list only but no change to G view

void m_recentfiles(GtkWidget *, cchar *menu)                               //  v.14.03
{
   navi::gallerytype = 5;                                                  //  gallery type = recent files
   gallery(recentfiles_file,"initF");                                      //  generate gallery of recent files
   gallery(0,"paint",0);
   F1_help_topic = "recent_images";
   m_viewmode(0,"G");
   return;
}


/**************************************************************************/

//  show most recently added image files (from most recent image file index)

void m_newfiles(GtkWidget *, cchar *)
{
   FILE     *fidr, *fidw;
   int      ftyp, count = 0;
   char     *file, buff[maxfcc], tempfile[200];
   
   F1_help_topic = "new images";

   fidr = fopen(newfiles_file,"r");
   if (! fidr) {
      zmessageACK(Mwin,0,strerror(errno));
      return;
   }
   
   snprintf(tempfile,200,"%s/newfiles",get_zuserdir());

   fidw = fopen(tempfile,"w");
   if (! fidw) {
      zmessageACK(Mwin,0,strerror(errno));
      return;
   }
   
   while (true)
   {
      file = fgets_trim(buff,maxfcc,fidr);
      if (! file) break;
      ftyp = image_file_type(file);
      if (ftyp != 2 && ftyp != 3) continue;
      fprintf(fidw,"%s\n",file);
      count++;
   }

   fclose(fidr);
   fclose(fidw);
   
   if (count > 0)
      rename(tempfile,newfiles_file);
   else {
      remove(tempfile);
      zmessageACK(Mwin,0,ZTX("no newly added files found"));
      return;
   }

   navi::gallerytype = 6;                                                  //  gallery type = new files
   gallery(newfiles_file,"initF");                                         //  generate gallery of new files
   m_viewmode(0,"G");                                                      //  position at curr. file
   gallery(0,"paint",0);                                                   //  position at top

   return;
}


/**************************************************************************/

//  open previous or next file in current gallery list
//  index is -1 or +1

void m_prev_next(int index)
{
   int      err, Nth, loNth, hiNth, ftype;
   char     *pfile;
   STATB    statb;

   loNth = navi::nfiles - navi::nimages;                                   //  1st image file (after directories)
   hiNth = navi::nfiles - 1;                                               //  last 
   Nth = curr_file_posn;                                                   //  get prev/next file from here 
   
   for (Nth += index; ; Nth += index)
   {
      if (Nth < loNth || Nth > hiNth) goto nomoreimages;                   //  reached the end

      pfile = gallery(0,"find",Nth);                                       //  next file in gallery
      if (! pfile) continue;
      ftype = image_file_type(pfile);
      if (ftype != 2 && ftype != 3) {                                      //  not an image or RAW file
         zfree(pfile);
         continue;
      }

      err = f_open(pfile,Nth);                                             //  open image or RAW file             v.14.02
      zfree(pfile);
      if (err) return;
      f_preload(index);                                                    //  preload next image
      return;
   }

nomoreimages:

   if (! curr_file) Fblankwindow = 1;
   err = stat(curr_file,&statb);
   if (err) {
      Fblankwindow = 1;
      Fpaint2();
   }

   poptext_window(ZTX("no more images"),MWIN,200,200,0,3); 
   return;
}


void m_prev(GtkWidget *, cchar *menu)
{
   if (menu) F1_help_topic = "open_image_file";
   m_prev_next(-1);                                                        //  search from curr_file -1 to first file
   return;
}


void m_next(GtkWidget *, cchar *menu) 
{
   if (menu) F1_help_topic = "open_image_file";
   m_prev_next(+1);                                                        //  search from curr_file +1 to last file
   return;
}


void m_prev_next(GtkWidget *, cchar *menu)                                 //  prev/next if left/right mouse click on icon
{                                                                          //  v.14.07
   int button = zfuncs::vmenuclickbutton;
   if (button == 1) m_prev(0,0);
   else m_next(0,0);
   return;
}


/**************************************************************************/

//  menu function: open a camera RAW file and edit with the ufraw GUI
//  opens 'clicked_file' if present or 'rawfile' if not

void m_ufraw(GtkWidget *, cchar *menu)
{
   char     *pp;
   char     *tiffile;
   int      err;
   STATB    statb;

   cchar *ufrawcommand = "ufraw --exposure=auto --interpolation=bilinear"
                         " --out-type=tiff --out-depth=16 --overwrite"
                         " --output=\"%s\" \"%s\"";

   F1_help_topic = "open_raw_ufraw";

   if (! Fufraw) {                                                         //  v.14.07
      zmessageACK(Mwin,0,ZTX("UFraw not installed"));
      return;
   }

   if (checkpend("all")) return;

   if (rawfile) zfree(rawfile);

   if (clicked_file) {
      rawfile = clicked_file;
      clicked_file = 0;
   }
   
   if (! rawfile) {
      pp = curr_file;                                                      //  directory for file open dialog
      if (! pp) pp = curr_dirk;
      rawfile = zgetfile(ZTX("Open RAW file (ufraw)"),"file",pp);          //  dialog to get RAW filespec
      if (! rawfile) return;
   }
   
   if (image_file_type(rawfile) != 3) {
      zmessageACK(Mwin,0,ZTX("RAW type not registered in User Settings"));
      zfree(rawfile);
      rawfile = 0;
      return;
   }
   
   zmainloop();
   Fmenulock = 1;
   tiffile = raw_to_tiff(rawfile);
   shell_quiet(ufrawcommand,tiffile,rawfile);                              //  start ufraw command, convert to tiff
   Fmenulock = 0;

   zfree(rawfile);
   rawfile = 0;

   err = stat(tiffile,&statb);
   if (err) {
      zfree(tiffile);
      printz("ufraw produced no tiff-16 file\n");
      return;
   }
   
   load_filemeta(tiffile);                                                 //  update image index
   update_image_index(tiffile);

   f_open(tiffile);                                                        //  open tiff file if present
   zfree(tiffile);
   return;
}


/**************************************************************************/

//  menu function: open a camera RAW file and edit with the Raw Therapee GUI
//  opens 'clicked_file' if present or 'rawfile' if not

void m_rawtherapee(GtkWidget *, cchar *menu)                               //  v.14.07
{
   char     *pp;
   char     *tiffile, *temp;
   int      err;
   STATB    statb;

   cchar *command = "rawtherapee -o \"%s\" -t -Y \"%s\" ";

   F1_help_topic = "open_raw";

   if (! Frawtherapee) {                                                   //  v.14.07
      zmessageACK(Mwin,0,ZTX("Raw Therapee not installed"));
      return;
   }

   if (checkpend("all")) return;

   if (rawfile) zfree(rawfile);

   if (clicked_file) {
      rawfile = clicked_file;
      clicked_file = 0;
   }
   
   if (! rawfile) {
      pp = curr_file;                                                      //  directory for file open dialog
      if (! pp) pp = curr_dirk;
      rawfile = zgetfile(ZTX("Open RAW file (Raw Therapee)"),"file",pp);   //  dialog to get RAW filespec
      if (! rawfile) return;
   }
   
   if (image_file_type(rawfile) != 3) {
      zmessageACK(Mwin,0,ZTX("RAW type not registered in User Settings"));
      zfree(rawfile);
      rawfile = 0;
      return;
   }
   
   zmainloop();
   Fmenulock = 1;
   tiffile = raw_to_tiff(rawfile);
   shell_quiet(command,tiffile,rawfile);                                   //  start command, convert to tiff
   Fmenulock = 0;

   zfree(rawfile);
   rawfile = 0;

   temp = zstrdup(tiffile);                                                //  Raw Therapee ignores specified .tiff
   pp = strstr(temp,".tif");                                               //    and outputs .tif instead
   pp[4] = 0;
   rename(temp,tiffile);
   zfree(temp);

   err = stat(tiffile,&statb);
   if (err) {
      zfree(tiffile);
      printz("Raw Therapee produced no tiff-16 file\n");
      return;
   }
   
   load_filemeta(tiffile);                                                 //  update image index
   update_image_index(tiffile);

   f_open(tiffile);                                                        //  open tiff file if present
   zfree(tiffile);
   return;
}


/**************************************************************************/

//  save (modified) image file to disk

void m_file_save(GtkWidget *, cchar *menu) 
{
   int  file_save_dialog_event(zdialog *zd, cchar *event);

   cchar          *pp;
   zdialog        *zd;

   F1_help_topic = "save_file";
   if (! curr_file) {
      if (zdfilesave) zdialog_destroy(zdfilesave);
      zdfilesave = 0;
      return;
   }

   if (strEqu(curr_file_type,"other"))                                     //  if unsupported type, use jpg
      strcpy(curr_file_type,"jpg");

/***
          _______________________________________________
         |                                               |
         |        Save Image File                        |
         |                                               |
         |   filename.jpg                                |
         |                                               |
         |  [version] save as new file version           |
         |  [new file] save as new file name or type     |
         |  [replace] replace file (OVERWRITE)           |
         |                                               |
         |                                    [cancel]   |
         |_______________________________________________|

***/

   if (! zdfilesave)
   {
      zd = zdialog_new(ZTX("Save Image File"),Mwin,Bcancel,null);
      zdfilesave = zd;

      zdialog_add_widget(zd,"hbox","hbf","dialog",0,"space=3");
      zdialog_add_widget(zd,"label","filename","hbf",0,"space=10");

      zdialog_add_widget(zd,"hbox","hb0","dialog");
      zdialog_add_widget(zd,"vbox","vb1","hb0",0,"space=3|homog");
      zdialog_add_widget(zd,"vbox","vb2","hb0",0,"space=3|homog");
      
      zdialog_add_widget(zd,"button","newvers","vb1",ZTX("new version"));
      zdialog_add_widget(zd,"button","newfile","vb1",ZTX("new file"));
      zdialog_add_widget(zd,"button","replace","vb1",ZTX("replace file"));

      zdialog_add_widget(zd,"hbox","hb1","vb2");
      zdialog_add_widget(zd,"hbox","hb2","vb2");
      zdialog_add_widget(zd,"hbox","hb3","vb2");

      zdialog_add_widget(zd,"label","labvers","hb1",ZTX("save as new file version"));
      zdialog_add_widget(zd,"label","labfile","hb2",ZTX("save as new file name or type"));
      zdialog_add_widget(zd,"image","warning","hb3","warning.png");
      zdialog_add_widget(zd,"label","labrepl","hb3",ZTX("replace old file (OVERWRITE)"),"space=3");
   }
   
   zd = zdfilesave;

   pp = strrchr(curr_file,'/');
   if (pp) zdialog_stuff(zd,"filename",pp+1);

   zdialog_run(zd,file_save_dialog_event,"mouse");
   return;
}


//  dialog event and completion function

int file_save_dialog_event(zdialog *zd, cchar *event)
{
   int  file_save_as();

   static char *newfilename = 0;
   char        *pfile, *pext, *pvers;
   cchar       *delim;
   int         ii, err, nvers;
   STATB       fstat;
   char        event2[12];

   if (zd->zstat) {                                                        //  cancel
      zdialog_free(zd);                                                    //  kill dialog
      zdfilesave = 0;
      return 1;
   }
   
   if (! strstr("newvers newfile replace",event)) return 1;                //  ignore other events

   strncpy0(event2,event,12);                                              //  preserve event

   zdialog_free(zd);                                                       //  kill dialog
   zdfilesave = 0;
   
   if (strstr("newvers replace",event2)) {
      if (strEqu(curr_file_type,"RAW")) {
         zmessageACK(Mwin,0,ZTX("cannot save as RAW type"));
         return 1;
      }
   }

   if (strEqu(event2,"newvers"))
   {
      if (newfilename) zfree(newfilename);
      newfilename = zstrdup(curr_file,12);
      pfile = strrchr(newfilename,'/');                                    //  get filename
      if (! pfile) return 1;
      pext = strrchr(pfile,'.');                                           //  get .ext
      if (pext && strlen(pext) > 5) pext = 0;
      if (! pext) pext = newfilename + strlen(newfilename);                //  unknown, none

      pvers = pext - 4;                                                    //  get curr. version: filename.vNN.ext
      if (! strnEqu(pvers,".v",2)) nvers = 0;
      else {
         err = convSI(pvers+2,nvers,1,98,&delim);                          //  convert NN to number 1-98
         if (err > 1) nvers = 0;                                           //  conversion error
         if (delim != pext) nvers = 0;                                     //  check format is .vNN
      }

      if (nvers == 0) {                                                    //  no version in file name
         pvers = pext;
         pext += 4;
         memmove(pext,pvers,6);                                            //  make space for .vNN before .ext
         strncpy(pvers,".vNN",4);
      }

      for (ii = 98; ii > nvers; ii--)                                      //  look for higher file versions
      {
         pvers[2] = ii/10 + '0';                                           //  build filename.vNN.ext
         pvers[3] = ii - 10 * (ii/10) + '0';
         err = stat(newfilename,&fstat);
         if (! err) break;
      }

      ii++;                                                                //  use next version 1-99
      nvers = ii;                                                          //  (vers 99 >> 99)
      pvers[2] = ii/10 + '0';                                              //  build filename.vNN.ext
      pvers[3] = ii - 10 * (ii/10) + '0';

      err = f_save(newfilename,curr_file_type,curr_file_bpc);              //  save file (fails at 99 versions)
      if (! err) f_open_saved();                                           //  open saved file with edit hist
   }

   if (strEqu(event2,"replace"))
   {
      err = f_save(curr_file,curr_file_type,curr_file_bpc);                //  save file with curr. edits applied
      if (err) return 1;
      strcpy(curr_file_type,f_save_type);                                  //  update curr_file_xxx from f_save_xxx
      curr_file_size = f_save_size;
   }
   
   if (strEqu(event2,"newfile"))
      err = file_save_as();                                                //  save-as file chooser dialog
   
   return 1;
}


//  save (modified) image to new file name or type
//  confirm if overwrite of existing file
//  returns 0 if OK, 1 if cancel or error

GtkWidget      *saveas_fchooser;

int file_save_as() 
{
   int  file_save_as_dialog_event(zdialog *zd, cchar *event);

   zdialog        *zd;
   static char    *save_dirk = 0;
   cchar          *type;
   char           *outfile = 0, *outfile2 = 0, *pp, *pext;
   int            ii, zstat, err, yn;
   int            bpc, mkcurr = 0;
   STATB          fstat;

/***
       ____________________________________________________________________
      |                                                                    |
      |   Save as New File Name or Type                                    |
      |                                                                    |
      |   [ file chooser dialog goes here ]                                |
      |                                                                    |
      |  ꖴ tif-8  ꖴ tif-16  ꖴ png-8  ꖴ png-16  ꖴ bmp  ꖴ jpg quality [90]  |
      |  [x] make current                                                  |
      |                                                                    |
      |                                                 [save] [cancel]    |
      |____________________________________________________________________|

***/

   zd = zdialog_new(ZTX("save as new file name or type"),Mwin,Bsave,Bcancel,null);

   zdialog_add_widget(zd,"hbox","hbfc","dialog",0,"expand");
   zdialog_add_widget(zd,"vbox","vbfc","hbfc",0,"space=3");
   saveas_fchooser = gtk_file_chooser_widget_new(GTK_FILE_CHOOSER_ACTION_SAVE);
   gtk_container_add(GTK_CONTAINER(zdialog_widget(zd,"hbfc")),saveas_fchooser);

   zdialog_add_widget(zd,"hbox","hbft","dialog",0,"space=3");
   zdialog_add_widget(zd,"radio","tif-8","hbft","tif-8","space=3");
   zdialog_add_widget(zd,"radio","tif-16","hbft","tif-16","space=3");
   zdialog_add_widget(zd,"radio","png-8","hbft","png-8","space=3");
   zdialog_add_widget(zd,"radio","png-16","hbft","png-16","space=3");
   zdialog_add_widget(zd,"radio","bmp","hbft","bmp","space=3");
   zdialog_add_widget(zd,"radio","jpg","hbft","jpg","space=3");
   zdialog_add_widget(zd,"label","labqual","hbft","quality");
   zdialog_add_widget(zd,"entry","jpgqual","hbft","90","space=2|scc=2");
   zdialog_add_widget(zd,"check","mkcurr","hbft",ZTX("make current"),"space=3");

   if (! save_dirk) {                                                      //  default from curr. file
      save_dirk = zstrdup(curr_file);
      pp = strrchr(save_dirk,'/');
      *pp = 0;
   }

   gtk_file_chooser_set_current_folder(GTK_FILE_CHOOSER(saveas_fchooser),save_dirk);
   char *fname = strrchr(curr_file,'/') + 1;
   gtk_file_chooser_set_current_name(GTK_FILE_CHOOSER(saveas_fchooser),fname);

   zdialog_stuff(zd,"tif-8",0);                                            //  turn off all radio buttons
   zdialog_stuff(zd,"tif-16",0);                                           //  GTK bug: programatic selection of one
   zdialog_stuff(zd,"png-8",0);                                            //    does not deselect the alternatives
   zdialog_stuff(zd,"png-16",0);                                           //      (this works for GUI selection only)
   zdialog_stuff(zd,"bmp",0);
   zdialog_stuff(zd,"jpg",0);

   zdialog_stuff(zd,"jpgqual",jpeg_def_quality);                           //  default jpeg quality, user setting

   if (strEqu(curr_file_type,"tif")) {                                     //  if curr. file type is tif, 
      if (curr_file_bpc == 16)                                             //  select tif-8 or tif-16 button
         zdialog_stuff(zd,"tiff-16",1);
      else zdialog_stuff(zd,"tiff-8",1);
   }

   else if (strEqu(curr_file_type,"png")) {                                //  same for png
      if (curr_file_bpc == 16)
         zdialog_stuff(zd,"png-16",1);
      else zdialog_stuff(zd,"png-8",1);
   }

   else if (strEqu(curr_file_type,"bmp"))                                  //  and bmp
      zdialog_stuff(zd,"bmp",1);
   
   else zdialog_stuff(zd,"jpg",1);                                         //  else default jpg
   
   zdialog_stuff(zd,"mkcurr",0);                                           //  deselect "make current"

   zdialog_resize(zd,500,500);
   zdialog_run(zd,file_save_as_dialog_event);

zdialog_wait:

   zstat = zdialog_wait(zd);
   if (zstat != 1) {                                                       //  user cancel
      zdialog_free(zd);
      return 1;
   }

   outfile2 = gtk_file_chooser_get_filename(GTK_FILE_CHOOSER(saveas_fchooser));
   if (! outfile2) {
      zd->zstat = 0;
      goto zdialog_wait;
   }

   if (outfile) zfree(outfile);
   outfile = zstrdup(outfile2,12);                                         //  add space for possible .vNN and .ext
   g_free(outfile2);
   
   zfree(save_dirk);                                                       //  remember save dirk for next time
   save_dirk = zstrdup(outfile);
   pp = strrchr(save_dirk,'/');
   *(pp+1) = 0;                                                            //  keep trailing '/'               v.14.02

   type = "jpg";                                                           //  default output type
   bpc = 8;

   zdialog_fetch(zd,"tif-8",ii);                                           //  get selected radio button
   if (ii) type = "tif";

   zdialog_fetch(zd,"tif-16",ii);
   if (ii) {
      type = "tif";
      bpc = 16;
   }

   zdialog_fetch(zd,"png-8",ii);
   if (ii) type = "png";

   zdialog_fetch(zd,"png-16",ii);
   if (ii) {
      type = "png";
      bpc = 16;
   }

   zdialog_fetch(zd,"bmp",ii);
   if (ii) type = "bmp";

   if (strEqu(type,"jpg")) {                                               //  set non-default jpeg save quality
      zdialog_fetch(zd,"jpgqual",ii);                                      //  will be used only one time
      if (ii < 1 || ii > 100) {
         zmessageACK(Mwin,0,ZTX("jpeg quality must be 1-100"));
         zd->zstat = 0;
         goto zdialog_wait;
      }
      jpeg_1x_quality = ii;
   }
   
   pext = strrchr(outfile,'/');                                            //  locate file .ext
   if (pext) pext = strrchr(pext,'.');

   if (pext) {                                                             //  validate .ext OK for type
      if (strEqu(type,"jpg") && ! strcasestr(".jpg .jpeg",pext)) *pext = 0;
      if (strEqu(type,"tif") && ! strcasestr(".tif .tiff",pext)) *pext = 0;
      if (strEqu(type,"png") && ! strcasestr(".png",pext)) *pext = 0;
      if (strEqu(type,"bmp") && ! strcasestr(".bmp",pext)) *pext = 0;
   }
   
   if (! pext || ! *pext) {
      pext = outfile + strlen(outfile);                                    //  wrong or missing, add new .ext
      *pext = '.';                                                         //  no replace .JPG with .jpg etc.
      strcpy(pext+1,type);
   }

   zdialog_fetch(zd,"mkcurr",mkcurr);                                      //  get make current option
   
   err = stat(outfile,&fstat);                                             //  check if file exists
   if (! err) {
      yn = zmessageYN(Mwin,ZTX("Overwrite file? \n %s"),outfile);          //  confirm overwrite
      if (! yn) {
         zd->zstat = 0;
         goto zdialog_wait;
      }
   }

   zdialog_free(zd);                                                       //  zdialog_free(zd);

   err = f_save(outfile,type,bpc);                                         //  save the file
   if (err) {
      zfree(outfile);
      return 1;
   }
   
   if (samedirk(outfile,navi::galleryname)) {                              //  if saving into current gallery
      gallery(outfile,"init");                                             //    refresh gallery list
      curr_file_posn = gallery_position(curr_file,curr_file_posn);         //    update curr. file position       v.14.02
      set_mwin_title();                                                    //    update window title
   }
   
   if (mkcurr) f_open_saved();                                             //  open saved file with edit hist

   zfree(outfile);
   return 0;
}


//  set dialog file type from user selection of file type radio button

int file_save_as_dialog_event(zdialog *zd, cchar *event)
{
   int      ii;
   cchar    *filetypes[6] = { "tif-8", "tif-16", "png-8", "png-16", "bmp", "jpg" };
   char     ext[4];
   char     *filespec;
   char     *filename, *pp;
   
   if (strEqu(event,"enter")) zd->zstat = 1;                               //  v.14.03

   if (strEqu(event,"jpgqual")) {                                          //  if jpg quality edited, set jpg .ext
      zdialog_stuff(zd,"jpg",1);
      event = "jpg";
   }
   
   for (ii = 0; ii < 6; ii++)                                              //  detect file type selection
      if (strEqu(event,filetypes[ii])) break;
   if (ii == 6) return 1;

   zdialog_stuff(zd,"tif-8",0);                                            //  turn off unselected types first
   zdialog_stuff(zd,"tif-16",0);                                           //  (see GTK bug note above)
   zdialog_stuff(zd,"png-8",0);
   zdialog_stuff(zd,"png-16",0);
   zdialog_stuff(zd,"bmp",0);
   zdialog_stuff(zd,"jpg",0);
   zdialog_stuff(zd,event,1);                                              //  lastly, turn on selected type

   strncpy0(ext,event,4);                                                  //  "tif-16" >> "tif" etc.

   filespec = gtk_file_chooser_get_filename(GTK_FILE_CHOOSER(saveas_fchooser));
   if (! filespec) return 1;
   filename = strrchr(filespec,'/');                                       //  revise file .ext in chooser dialog
   if (! filename) return 1;
   filename = zstrdup(filename+1,6);
   pp = strrchr(filename,'.');
   if (! pp || strlen(pp) > 5) pp = filename + strlen(filename);
   *pp = '.';
   strcpy(pp+1,ext);
   gtk_file_chooser_set_current_name(GTK_FILE_CHOOSER(saveas_fchooser),filename);
   zfree(filename);
   g_free(filespec);

   return 1;
}


/**************************************************************************/

//  rename menu function
//  activate rename dialog, stuff data from current or clicked file
//  dialog remains active when new file is opened

char     rename_old[200] = "";
char     rename_new[200] = "";
char     rename_prev[200] = "";
char     *rename_file = 0;

void m_rename(GtkWidget *, cchar *menu)
{
   int rename_dialog_event(zdialog *zd, cchar *event);
   
   char     *pdir, *pfile, *pext;

   if (menu) F1_help_topic = "rename";
   
   if (checkpend("all")) return;
   
   if (rename_file) zfree(rename_file);
   rename_file = 0;
   
   if (clicked_file) {                                                     //  use clicked file if present
      rename_file = clicked_file;
      clicked_file = 0;
   }
   else if (curr_file)                                                     //  else current file
      rename_file = zstrdup(curr_file);
   else return;

/***
       ______________________________________
      |         Rename Image File            |
      |                                      |
      | Old Name  [_______________________]  |
      | New Name  [_______________________]  |
      |           [previous name] [add 1]    |
      |                                      |
      |                   [apply] [cancel]   |
      |______________________________________|
      
***/

   if (! zdrename)                                                         //  restart dialog
   {
      zdrename = zdialog_new(ZTX("Rename Image File"),Mwin,Bapply,Bcancel,null);
      zdialog *zd = zdrename;

      zdialog_add_widget(zd,"hbox","hb1","dialog",0,"space=3");
      zdialog_add_widget(zd,"vbox","vb1","hb1",0,"homog|space=5");
      zdialog_add_widget(zd,"vbox","vb2","hb1",0,"homog|expand");

      zdialog_add_widget(zd,"label","Lold","vb1",ZTX("Old Name"));
      zdialog_add_widget(zd,"label","Lnew","vb1",ZTX("New Name"));
      zdialog_add_widget(zd,"label","space","vb1");

      zdialog_add_widget(zd,"hbox","hb2","vb2");
      zdialog_add_widget(zd,"label","oldname","hb2");
      zdialog_add_widget(zd,"label","space","hb2",0,"expand");

      zdialog_add_widget(zd,"entry","newname","vb2",0,"scc=30");
      zdialog_add_widget(zd,"hbox","hb3","vb2",0,"space=3");
      zdialog_add_widget(zd,"button","Bprev","hb3",ZTX("previous name"));
      zdialog_add_widget(zd,"button","Badd1","hb3",ZTX("add 1"),"space=8");

      zdialog_run(zd,rename_dialog_event);                                 //  run dialog
   }
   
   zdialog *zd = zdrename;
   parsefile(rename_file,&pdir,&pfile,&pext);
   strncpy0(rename_old,pfile,199);
   strncpy0(rename_new,pfile,199);
   zdialog_stuff(zd,"oldname",rename_old);                                 //  current file name
   zdialog_stuff(zd,"newname",rename_new);                                 //  entered file name (same)

   return;
}


//  dialog event and completion callback function

int rename_dialog_event(zdialog *zd, cchar *event)
{
   char     *pp, *pdir, *pfile, *pext;
   char     *newfile, *thumbfile;
   int      nseq, digits, ccp, ccn, ccx, Nth, err, fnew;
   STATB    statb;
   
   if (strEqu(event,"enter")) zd->zstat = 1;                               //  [apply]                            v.14.02

   if (strEqu(event,"Bprev"))                                              //  previous name >> new name
      if (strlen(rename_prev) > 0)
         zdialog_stuff(zd,"newname",rename_prev);

   if (strEqu(event,"Badd1"))                                              //  increment sequence number
   {
      zdialog_fetch(zd,"newname",rename_new,194);                          //  get entered filename
      pp = rename_new + strlen(rename_new);
      digits = 0;
      while (pp[-1] >= '0' && pp[-1] <= '9') {
         pp--;                                                             //  look for NNN in filenameNNN
         digits++;
      }
      nseq = 1 + atoi(pp);                                                 //  NNN + 1
      if (nseq > 9999) nseq = 0;
      if (digits < 2) digits = 2;                                          //  keep digit count if enough
      if (nseq > 99 && digits < 3) digits = 3;                             //  use leading zeros
      if (nseq > 999 && digits < 4) digits = 4;
      snprintf(pp,digits+1,"%0*d",digits,nseq);
      zdialog_stuff(zd,"newname",rename_new);
   }
   
   if (zd->zstat == 0) return 0;                                           //  not finished
   
   if (zd->zstat != 1) {                                                   //  canceled
      zdialog_free(zd);                                                    //  kill dialog
      zdrename = 0;
      if (rename_file) zfree(rename_file);
      rename_file = 0;
      return 0;
   }

   zd->zstat = 0;                                                          //  apply - keep dialog active

   gtk_window_present(MWIN);                                               //  keep focus on main window          v.14.09

   if (! rename_file) return 0;
   if (checkpend("all")) return 0;

   parsefile(rename_file,&pdir,&pfile,&pext);                              //  existing /directories/file.ext

   zdialog_fetch(zd,"newname",rename_new,194);                             //  new file name from user

   ccp = strlen(pdir);                                                     //  length of /directories/
   ccn = strlen(rename_new);                                               //  length of file
   if (pext) ccx = strlen(pext);                                           //  length of .ext
   else ccx = 0;

   newfile = (char *) zmalloc(ccp + ccn + ccx + 1);                        //  put it all together
   strncpy(newfile,rename_file,ccp);                                       //   /directories.../newfilename.ext
   strcpy(newfile+ccp,rename_new);
   if (ccx) strcpy(newfile+ccp+ccn,pext);
   
   err = stat(newfile,&statb);                                             //  check if new name exists
   if (! err) {
      zmessageACK(Mwin,0,ZTX("target file already exists"));
      zfree(newfile);
      return 0;
   }
   
   err = rename(rename_file,newfile);                                      //  v.14.03
   if (err) {
      zmessageACK(Mwin,0,"file error: %s",strerror(errno));
      zdialog_free(zd);
      zdrename = 0;
      zfree(newfile);
      zfree(rename_file);
      rename_file = 0;
      return 0;
   }
   
   thumbfile = image_thumbfile(newfile,&fnew);                             //  stop use of leftover thumbnail     v.14.07
   if (fnew != 1) remove(thumbfile);
   zfree(thumbfile);

   strncpy0(rename_prev,rename_new,199);                                   //  save new name to previous name

   load_filemeta(newfile);                                                 //  add new file to image index
   update_image_index(newfile);

   if (navi::gallerytype == 1) gallery(curr_file,"init");                  //  update curr. gallery list          v.14.03
   set_mwin_title();                                                       //  update window title posn, count
   add_recent_file(newfile);                                               //  first in recent files list
   zfree(newfile);

   delete_image_index(rename_file);                                        //  delete old file in image index
   Nth = gallery_position(rename_file,0);                                  //  find in gallery list
   if (Nth >= 0) gallery(0,"delete",Nth);                                  //  delete from gallery list

   zfree(rename_file);
   rename_file = 0;

   if (FGW == 'F')                                                         //  F window, curr. file was renamed
      m_next(0,0);
   else {                                                                  //  gallery window, kill dialog
      zdialog_free(zd);
      zdrename = 0;
   }

   return 0;
}


/**************************************************************************/

//  create a new blank image with desired background color

void m_create(GtkWidget *, cchar *)
{
   int   create_dialog_event(zdialog *zd, cchar *event);

   char        color[20], fname[100], fext[8], *filespec;
   int         zstat, err, cc, ww, hh, RGB[3];
   zdialog     *zd;
   cchar       *pp;

   F1_help_topic = "create_blank_image";
   
   if (checkpend("all")) return;

//    file name [___________________________] [.jpg|v]
//    width [____]  height [____] (pixels)
//    color [____]
   
   zd = zdialog_new(ZTX("Create Blank Image"),Mwin,Bdone,Bcancel,null);
   zdialog_add_widget(zd,"hbox","hbf","dialog",0,"space=1");
   zdialog_add_widget(zd,"label","labf","hbf",ZTX("file name"),"space=3");
   zdialog_add_widget(zd,"entry","file","hbf",0,"space=3|expand");
   zdialog_add_widget(zd,"combo","ext","hbf",".jpg");
   zdialog_add_widget(zd,"hbox","hbz","dialog",0,"space=1");
   zdialog_add_widget(zd,"label","labw","hbz",ZTX("width"),"space=5");
   zdialog_add_widget(zd,"spin","width","hbz","100|9999|1|1600");
   zdialog_add_widget(zd,"label","space","hbz",0,"space=5");
   zdialog_add_widget(zd,"label","labh","hbz",ZTX("height"),"space=5");
   zdialog_add_widget(zd,"spin","height","hbz","100|9999|1|1000");
   zdialog_add_widget(zd,"label","labp","hbz","(pixels)","space=3");
   zdialog_add_widget(zd,"hbox","hbc","dialog",0,"space=1");
   zdialog_add_widget(zd,"label","labc","hbc",ZTX("color"),"space=5");
   zdialog_add_widget(zd,"colorbutt","color","hbc","200|200|200");
   
   zdialog_cb_app(zd,"ext",".jpg");
   zdialog_cb_app(zd,"ext",".png");
   zdialog_cb_app(zd,"ext",".tif");
   zdialog_cb_app(zd,"ext",".bmp");
   
   zdialog_restore_inputs(zd);                                             //  preload prior user inputs
   zdialog_stuff(zd,"file","");                                            //  force input of new name            v.14.07

   zdialog_run(zd);                                                        //  run dialog
   zstat = zdialog_wait(zd);                                               //  wait for completion

   if (zstat != 1) {                                                       //  cancel
      zdialog_free(zd);
      return;
   }

   zdialog_fetch(zd,"file",fname,92);                                      //  get new file name
   strTrim2(fname);
   if (*fname <= ' ') strcpy(fname,"no-name");                             //  no name input, use "no-name"

   zdialog_fetch(zd,"ext",fext,8);                                         //  add extension
   strcat(fname,fext);
   
   cc = strlen(fname);
   filespec = zstrdup(curr_dirk,cc+4);                                     //  make full filespec
   strcat(filespec,"/");
   strcat(filespec,fname);
   
   zdialog_fetch(zd,"width",ww);                                           //  get image dimensions
   zdialog_fetch(zd,"height",hh);

   RGB[0] = RGB[1] = RGB[2] = 255;
   zdialog_fetch(zd,"color",color,19);                                     //  get image color
   pp = strField(color,"|",1);
   if (pp) RGB[0] = atoi(pp);
   pp = strField(color,"|",2);
   if (pp) RGB[1] = atoi(pp);
   pp = strField(color,"|",3);
   if (pp) RGB[2] = atoi(pp);

   zdialog_free(zd);

   err = create_blank_file(filespec,ww,hh,RGB);
   if (err) return;

   f_open(filespec);                                                       //  make it the current file
   return;
}


//  function to create a new blank image file
//  file extension must be one of: .jpg .tif .png .bmp
//  RGB args are in the range 0 - 255
//  if file exists it is overwritten
//  returns 0 if OK, +N if error

int create_blank_file(cchar *file, int ww, int hh, int RGB[3])
{
   cchar          *pp;
   cchar          *fext;
   int            cstat;
   PXB            *tempxb;
   GError         *gerror = 0;
   uint8          *pixel;

   pp = strrchr(file,'.');                                                 //  get file .ext
   if (! pp || strlen(pp) > 4) return 1;
   
   if (strEqu(pp,".jpg")) fext = "jpeg";                                   //  validate and set pixbuf arg.
   else if (strEqu(pp,".png")) fext = "png";
   else if (strEqu(pp,".tif")) fext = "tiff";
   else if (strEqu(pp,".bmp")) fext = "bmp";
   else return 1;
   
   tempxb = PXB_make(ww,hh);                                               //  create pixbuf image
   if (! tempxb) zappcrash("out of memory");
   
   for (int py = 0; py < hh; py++)                                         //  fill with color
   for (int px = 0; px < ww; px++)
   {
      pixel = PXBpix(tempxb,px,py);
      pixel[0] = RGB[0];
      pixel[1] = RGB[1];
      pixel[2] = RGB[2];
   }

   cstat = gdk_pixbuf_save(tempxb->pixbuf,file,fext,&gerror,null);
   if (! cstat) {
      zmessageACK(Mwin,0,"error: %s",gerror->message);
      PXB_free(tempxb);
      return 2;
   }

   PXB_free(tempxb);
   return 0;
}


/**************************************************************************/

//  copy an image file to a new location (duplicate)

void m_copytoloc(GtkWidget *, cchar *) 
{
   copy_move("copy");
   return;
}


//  move an image file to a new location (delete original)

void m_movetoloc(GtkWidget *, cchar *)
{
   copy_move("move");
   return;
}


//  copy an image to a new location and optionally delete the original

char     *copymove_file = 0;
char     *copymove_loc = 0;
int      copymove_Fdelete;

void copy_move(cchar *menu)
{
   int copymove_dialog_event(zdialog *zd, cchar *event);
   
   static char menux[8];
   
   if (menu) strcpy(menux,menu);                                           //  remember for re-entry with null menu
   
   F1_help_topic = menux;
   
   if (checkpend("all")) return;
   
   if (copymove_file) zfree(copymove_file);
   copymove_file = 0;
   
   if (clicked_file) {                                                     //  use clicked file if present
      copymove_file = clicked_file;
      clicked_file = 0;
   }
   else if (curr_file)                                                     //  else use current file
      copymove_file = zstrdup(curr_file);
   else return;

/***
               XXXX Image File

         Image File: [________________]
         new location [______________] [browse]

                         [apply] [cancel]
***/

   if (! zdcopymove)
   {
      if (strEqu(menux,"copy"))
         zdcopymove = zdialog_new(ZTX("Copy Image File"),Mwin,Bapply,Bcancel,null);
      else if (strEqu(menux,"move"))
         zdcopymove = zdialog_new(ZTX("Move Image File"),Mwin,Bapply,Bcancel,null);
      else return;

      zdialog *zd = zdcopymove;

      zdialog_add_widget(zd,"hbox","hb1","dialog",0,"space=3");
      zdialog_add_widget(zd,"label","labf","hb1",ZTX("Image File:"),"space=5");
      zdialog_add_widget(zd,"label","file","hb1");
      zdialog_add_widget(zd,"label","space","hb1",0,"expand");

      zdialog_add_widget(zd,"hbox","hb2","dialog");
      zdialog_add_widget(zd,"label","labl","hb2",ZTX("new location"),"space=5");
      zdialog_add_widget(zd,"entry","newloc","hb2",0,"expand");
      zdialog_add_widget(zd,"button","browse","hb2",Bbrowse,"space=5");

      zdialog_resize(zd,400,0);
      zdialog_run(zd,copymove_dialog_event);                               //  run dialog
   }

   zdialog *zd = zdcopymove;
   
   char *pfile = strrchr(copymove_file,'/');
   if (pfile) pfile++;
   else pfile = copymove_file;

   zdialog_stuff(zd,"file",pfile);
   if (copymove_loc)
      zdialog_stuff(zd,"newloc",copymove_loc);

   if (strEqu(menux,"copy")) copymove_Fdelete = 0;                         //  copy, no delete
   else copymove_Fdelete = 1;                                              //  move, delete original

   return;
}


//  dialog event and completion callback function

int copymove_dialog_event(zdialog *zd, cchar *event)
{
   char     *newloc, *pfile, *newfile;
   int      cc, err, Nth;
   STATB    statb;
   
   if (strEqu(event,"browse")) {                                           //  browse for new location
      newloc = zgetfile(ZTX("Select directory"),"folder",copymove_loc);
      if (! newloc) return 0;
      zdialog_stuff(zd,"newloc",newloc);
      if (copymove_loc) zfree(copymove_loc);
      copymove_loc = newloc;
      return 0;
   }
   
   if (zd->zstat == 0) return 0;                                           //  wait for dialog finished
   
   if (zd->zstat != 1) {                                                   //  canceled
      zdialog_free(zd);                                                    //  kill dialog
      zdcopymove = 0;
      if (copymove_file) zfree(copymove_file);
      copymove_file = 0;
      return 0;
   }

   zd->zstat = 0;                                                          //  apply - keep dialog active

   gtk_window_present(MWIN);                                               //  keep focus on main window          v.14.09

   if (! copymove_file) return 0;
   if (checkpend("all")) return 0;

   if (copymove_loc) zfree(copymove_loc);                                  //  get new location from dialog
   copymove_loc = (char *) zmalloc(maxfcc);
   zdialog_fetch(zd,"newloc",copymove_loc,maxfcc);
   
   stat(copymove_loc,&statb);                                              //  check for valid directory
   if (! S_ISDIR(statb.st_mode)) {
      zmessageACK(Mwin,0,ZTX("new location is not a directory"));
      return 0;
   }
   
   pfile = strrchr(copymove_file,'/');                                     //  isolate source file filename.ext
   if (pfile) pfile++;
   else pfile = copymove_file;
   cc = strlen(copymove_loc) + strlen(pfile) + 2;                          //  new file = /new/location/filename.ext
   newfile = (char *) zmalloc(cc);
   strcpy(newfile,copymove_loc);
   strcat(newfile,"/");
   strcat(newfile,pfile);

   err = stat(newfile,&statb);                                             //  check if new file exists
   if (! err) {
      zmessageACK(Mwin,0,ZTX("target file already exists"));
      zfree(newfile);
      return 0;
   }

   err = shell_ack("cp -p \"%s\" \"%s\"",copymove_file,newfile);           //  copy source file to new file
   if (err) {
      zfree(newfile);
      zdialog_free(zd);
      zdcopymove = 0;
      zfree(copymove_file);
      copymove_file = 0;
      return 0;
   }

   load_filemeta(newfile);                                                 //  update image index for new file
   update_image_index(newfile);
   zfree(newfile);
   
   if (copymove_Fdelete) {                                                 //  move - delete source file
      err = remove(copymove_file);
      if (err) {
         zmessageACK(Mwin,0,ZTX("delete failed: \n %s"),wstrerror(err));
         zdialog_free(zd);
         zdcopymove = 0;
         zfree(copymove_file);
         copymove_file = 0;
         return 0;
      }

      delete_image_index(copymove_file);                                   //  delete in image index
      Nth = gallery_position(copymove_file,0);                             //  find in gallery list
      if (Nth >= 0) gallery(0,"delete",Nth);                               //  delete from gallery list
      
      if (FGW == 'F') {                                                    //  F window                           v.14.06
         Nth = curr_file_posn;                                             //  file now at same gallery position
         newfile = gallery(0,"find",Nth);                                  //  open this file
         if (newfile) {
            f_open(newfile,Nth);
            zfree(newfile);
         }
         else {                                                            //  end of gallery
            Fblankwindow = 1;
            Fpaint2();
            zdialog_free(zd);                                              //  kill dialog
            zdcopymove = 0;
            zfree(copymove_file);
            copymove_file = 0;
         }
      }
   }
   
   else {                                                                  //  copy - keep source file
      if (FGW == 'F')                                                      //  F window
         m_next(0,0);                                                      //  move to next file
   }

   if (FGW == 'G') {                                                       //  G window
      zdialog_free(zd);                                                    //  kill dialog
      zdcopymove = 0;
      zfree(copymove_file);
      copymove_file = 0;
   }

   return 0;
}


/**************************************************************************/

//  Trash image file - move curr_file to trash.
//  Use the Linux standard trash function. 
//  If not available, revert to Desktop folder.

char     *trash_file = 0;

void m_trash(GtkWidget *, cchar *)
{
   int trash_dialog_event(zdialog *zd, cchar *event);
   
   cchar  *title = ZTX("Trash Image File");
   cchar  *autostep = ZTX("(automatic step to next image)");

   F1_help_topic = "trash";
   
   if (checkpend("all")) return;
   
   if (trash_file) zfree(trash_file);
   trash_file = 0;
   
   if (clicked_file) {                                                     //  use clicked file if present
      trash_file = clicked_file;
      clicked_file = 0;
   }
   else if (curr_file)                                                     //  else use current file
      trash_file = zstrdup(curr_file);
   else return;

/***
          ______________________________________
         |      Trash Image File                |
         |                                      |
         | Image File: [______________________] |
         |                                      |
         | (automatic step to next image)       |
         |                                      |
         |              [trash] [keep] [cancel] |
         |______________________________________|
         
***/

   if (! zdtrash)                                                          //  start dialog if not already
   {
      zdtrash = zdialog_new(title,Mwin,Btrash,Bkeep,Bcancel,null);
      zdialog *zd = zdtrash;

      zdialog_add_widget(zd,"hbox","hb1","dialog",0,"space=3");
      zdialog_add_widget(zd,"label","labf","hb1",ZTX("Image File:"),"space=3");
      zdialog_add_widget(zd,"label","file","hb1",0,"space=3");
      if (FGW == 'F') zdialog_add_widget(zd,"label","labnext","dialog",autostep);

      zdialog_resize(zd,300,0);
      zdialog_run(zd,trash_dialog_event);                                  //  run dialog
   }

   char *pfile = strrchr(trash_file,'/');
   if (pfile) pfile++;
   else pfile = trash_file;
   zdialog_stuff(zdtrash,"file",pfile);

   return;
}


//  dialog event and completion callback function

int trash_dialog_event(zdialog *zd, cchar *event)
{
   int         err = 0, yn, nn, Nth, gstat;
   char        *file, dtdirk[200], trashdir[200];
   GError      *gerror = 0;
   GFile       *gfile = 0;
   STATB       statb;
   static int  trashworks = 1, desktopworks = 1;                           //  assume OK until found otherwise
   FILE        *fid;

   cchar    *mess1 = ZTX("Linux standard trash does not work. \n"
                         "Desktop trash folder will be created.");

   cchar    *mess2 = ZTX("Linux and Desktop trash do not work. \n"
                         "Permanently delete the image file?");
   
   cchar    *mess3 = ZTX("Cannot create trash folder: %s");

   if (checkpend("all")) goto KILL;                                        //  cancel dialog

   if (zd->zstat == 0) return 1;                                           //  wait for dialog finished
   if (zd->zstat == 2) goto NEXT;                                          //  [keep] button
   if (zd->zstat != 1) goto KILL;                                          //  [cancel] or [x]
   if (! trash_file) goto KILL;                                            //  [trash] button

   err = stat(trash_file,&statb);                                          //  get file status
   if (err) goto KILL;
   if (! S_ISREG(statb.st_mode)) goto KILL;

   if (! (statb.st_mode & S_IWUSR)) {                                      //  check permission
      yn = zmessageYN(Mwin,ZTX("Move read-only file to trash?"));
      if (! yn) goto NEXT;                                                 //  keep file
      statb.st_mode |= S_IWUSR;                                            //  make writable if needed
      chmod(trash_file,statb.st_mode);
   }

   if (trashworks)                                                         //  try Linux standard trash
   {
      gfile = g_file_new_for_path(trash_file);
      gstat = g_file_trash(gfile,0,&gerror);                               //  move file to trash
      g_object_unref(gfile);
      if (! gstat) {
         printz("standard trash failed: %s \n",gerror->message);           //  did not work
         zmessageACK(Mwin,0,mess1);
         trashworks = 0;
      }
   }

   if (! trashworks && desktopworks)
   {
      fid = popen("xdg-user-dir DESKTOP","r");                             //  get desktop for user locale        v.14.05
      if (! fid) {
         zmessLogACK(Mwin,mess3,strerror(errno));
         desktopworks = 0;
      }
      else {
         nn = fscanf(fid,"%s",dtdirk);
         pclose(fid);
         if (nn != 1) {
            zmessLogACK(Mwin,mess3,strerror(errno));
            desktopworks = 0;
         }
      }
      
      if (desktopworks) {
         snprintf(trashdir,200,"%s/fotoxx-trash",dtdirk);                  //  check if trash directory exists
         err = stat(trashdir,&statb);
         if (! S_ISDIR(statb.st_mode)) {
            err = mkdir(trashdir,0750);                                    //  create if not already
            if (err) {
               zmessLogACK(Mwin,mess3,strerror(errno));
               desktopworks = 0;
            }
         }
      }
      
      if (desktopworks)
      {
         err = shell_ack("cp \"%s\" \"%s\" ",trash_file,trashdir);         //  copy image file to desktop trash
         if (err) desktopworks = 0;
         else {
            err = remove(trash_file);                                      //  remove original file
            if (err) {
               zmessLogACK(Mwin,ZTX("error: %s"),wstrerror(errno));        //  cannot delete file
               goto KILL;
            }
         }
      }
   }

   if (! trashworks && ! desktopworks)
   {
      yn = zmessageYN(Mwin,mess2);                                         //  nothing works, ask to delete 
      if (! yn) goto NEXT;                                                 //  keep file
      err = remove(trash_file);                                            //  delete file
      if (err) {
         zmessLogACK(Mwin,ZTX("delete failed: \n %s"),wstrerror(errno));
         goto KILL;
      }
   }

   delete_image_index(trash_file);                                         //  delete file in image index

   Nth = gallery_position(trash_file,0);                                   //  find in gallery list
   if (Nth >= 0) gallery(0,"delete",Nth);                                  //  delete from gallery list

   if (curr_file && strEqu(trash_file,curr_file))                          //  current file was trashed
      free_resources();

   if (FGW == 'F') {                                                       //  F window
      file = gallery(0,"find",curr_file_posn);                             //  new current file
      if (! file) 
         poptext_window(ZTX("no more images"),MWIN,200,200,0,3);           //  v.14.09
      else {
         f_open(file);
         zfree(file);
      }
      zd->zstat = 0;                                                       //  v.14.03
      gtk_window_present(MWIN);                                            //  keep focus on main window          v.14.09
      return 1;
   }

NEXT:
   if (FGW != 'F') goto KILL;                                              //  G gallery window
   m_next(0,0);                                                            //  open next image
   gtk_window_present(MWIN);                                               //  keep focus on main window          v.14.09
   zd->zstat = 0;                                                          //  keep dialog active
   return 1;

KILL:
   zdialog_free(zd);                                                       //  kill dialog
   zdtrash = 0;
   if (trash_file) zfree(trash_file);                                      //  free memory
   trash_file = 0;
   return 1;
}


/**************************************************************************/

//  delete an image file, cannot be reversed

char     *delete_file = 0;

void m_delete(GtkWidget *, cchar *)
{
   int delete_dialog_event(zdialog *zd, cchar *event);
   
   cchar  *title = ZTX("Delete Image File - CANNOT BE REVERSED");
   cchar  *autostep = ZTX("(automatic step to next image)");
   
   F1_help_topic = "delete";
   
   if (checkpend("all")) return;
   
   if (delete_file) zfree(delete_file);
   delete_file = 0;
   
   if (clicked_file) {                                                     //  use clicked file if present
      delete_file = clicked_file;
      clicked_file = 0;
   }
   else if (curr_file)                                                     //  else use current file
      delete_file = zstrdup(curr_file);
   else return;

/***
          ______________________________________
         |      Delete Image File               |
         |      CANNOT BE REVERSED              |
         |                                      |
         | Image File: [______________________] |
         |                                      |
         | (automatic step to next image)       |
         |                                      |
         |            [delete] [keep] [cancel]  |
         |______________________________________|
         
***/

   if (! zddelete)
   {
      zddelete = zdialog_new(title,Mwin,Bdelete,Bkeep,Bcancel,null);
      zdialog *zd = zddelete;

      zdialog_add_widget(zd,"hbox","hb1","dialog",0,"space=3");
      zdialog_add_widget(zd,"label","labf","hb1",ZTX("Image File:"),"space=5");
      zdialog_add_widget(zd,"label","file","hb1");
      if (FGW == 'F') zdialog_add_widget(zd,"label","labnext","dialog",autostep);

      zdialog_resize(zd,400,0);
      zdialog_run(zd,delete_dialog_event);                                 //  run dialog
   }

   char *pfile = strrchr(delete_file,'/');
   if (pfile) pfile++;
   else pfile = delete_file;
   zdialog_stuff(zddelete,"file",pfile);

   return;
}


//  dialog event and completion callback function

int delete_dialog_event(zdialog *zd, cchar *event) 
{
   int      err, Nth, yn;
   char     *file;
   STATB    statb;
   
   if (checkpend("all")) goto KILL;                                        //  cancel dialog

   if (zd->zstat == 0) return 1;                                           //  wait for dialog finished
   if (zd->zstat == 2) goto NEXT;                                          //  [keep] button
   if (zd->zstat != 1) goto KILL;                                          //  [cancel] or [x]
   if (! delete_file) goto KILL;                                           //  [delete] button

   err = stat(delete_file,&statb);                                         //  file exists?                       v.14.10
   if (err) goto KILL;
   if (! S_ISREG(statb.st_mode)) goto KILL;
   
   if (! (statb.st_mode & S_IWUSR)) {                                      //  check permission                   v.14.10
      yn = zmessageYN(Mwin,ZTX("Delete read-only file?"));
      if (! yn) goto NEXT;                                                 //  keep file
      statb.st_mode |= S_IWUSR;                                            //  make writable if needed
      chmod(delete_file,statb.st_mode);
   }

   err = remove(delete_file);                                              //  delete file
   if (err) {
      zmessageACK(Mwin,0,ZTX("delete failed: \n %s"),wstrerror(err));
      goto KILL;
   }

   delete_image_index(delete_file);                                        //  delete old file in image index

   Nth = gallery_position(delete_file,0);                                  //  delete from gallery list
   if (Nth >= 0) gallery(0,"delete",Nth);

   if (curr_file && strEqu(delete_file,curr_file))                         //  current file was deleted
      free_resources();

   if (FGW == 'F') {                                                       //  F window
      file = gallery(0,"find",curr_file_posn);                             //  new current file
      if (! file) 
         poptext_window(ZTX("no more images"),MWIN,200,200,0,3);           //  v.14.09
      else {
         f_open(file);
         zfree(file);
      }
      zd->zstat = 0;                                                       //  v.14.03
      gtk_window_present(MWIN);                                            //  keep focus on main window          v.14.09
      return 1;
   }
   
NEXT:
   if (FGW != 'F') goto KILL;                                              //  v.14.10
   m_next(0,0);                                                            //  open next image
   gtk_window_present(MWIN);                                               //  keep focus on main window          v.14.09
   zd->zstat = 0;                                                          //  keep dialog active
   return 1;

KILL:
   zdialog_free(zd);                                                       //  kill dialog
   zddelete = 0;
   if (delete_file) zfree(delete_file);                                    //  free memory
   delete_file = 0;
   return 1;
}


/**************************************************************************/

//  print image file

void m_print(GtkWidget *, cchar *)                                         //  use GTK print
{
   int print_addgrid(PXB *Ppxb);

   int      pstat;
   char     *printfile = 0;
   PXB      *Ppxb;
   GError   *gerror = 0;
   
   F1_help_topic = "print";
   
   if (clicked_file) {                                                     //  use clicked file if present
      printfile = clicked_file;
      clicked_file = 0;
   }
   else if (curr_file)                                                     //  else use current file
      printfile = zstrdup(curr_file);
   else return;
   
   Ppxb = PXB_load(printfile,1);                                           //  error > zmessageACK()
   if (! Ppxb) return;

   zfree(printfile);
   printfile = 0;

   print_addgrid(Ppxb);                                                    //  add grid lines if wanted
   
   printfile = zstrdup(tempdir,20);                                        //  make temp print file:
   strcat(printfile,"/printfile.bmp");                                     //    /.../fotoxx-nnnnnn/printfile.bmp

   pstat = gdk_pixbuf_save(Ppxb->pixbuf,printfile,"bmp",&gerror,null);     //  bmp much faster than png
   if (! pstat) {
      zmessageACK(Mwin,0,"error: %s",gerror->message);
      PXB_free(Ppxb);
      zfree(printfile);
      return;
   }

   print_image_file(Mwin,printfile);
   
   PXB_free(Ppxb);
   zfree(printfile);
   return;
}


//  add grid lines to print image if wanted

int print_addgrid(PXB *Ppxb)
{
   uint8    *pix;
   int      px, py, ww, hh;
   int      startx, starty, stepx, stepy;
   int      G = currgrid;
   
   if (! gridsettings[G][GON]) return 0;                                   //  grid lines off

   ww = Ppxb->ww;
   hh = Ppxb->hh;

   stepx = gridsettings[G][GXS];                                           //  space between grid lines
   stepy = gridsettings[G][GYS];
   
   stepx = stepx / Mscale;                                                 //  window scale to image scale
   stepy = stepy / Mscale;
   
   if (gridsettings[G][GXC])                                               //  if line counts specified,
      stepx = ww / (1 + gridsettings[G][GXC]);                             //    set spacing accordingly
   if (gridsettings[G][GYC])
      stepy = hh / ( 1 + gridsettings[G][GYC]);
   
   startx = stepx * gridsettings[G][GXF] / 100;                            //  variable offsets
   if (startx < 0) startx += stepx;

   starty = stepy * gridsettings[G][GYF] / 100;
   if (starty < 0) starty += stepy;

   if (gridsettings[G][GX]) {                                              //  x-grid enabled
      for (px = startx; px < ww-1; px += stepx)
      for (py = 0; py < hh; py++)
      {
         pix = PXBpix(Ppxb,px,py);
         pix[0] = pix[1] = pix[2] = 255;
         pix[3] = pix[4] = pix[5] = 0;
      }
   }

   if (gridsettings[G][GY]) {                                              //  y-grid enabled
      for (py = starty; py < hh-1; py += stepy)
      for (px = 0; px < ww; px++)
      {
         pix = PXBpix(Ppxb,px,py);
         pix[0] = pix[1] = pix[2] = 255;
         pix = PXBpix(Ppxb,px,py+1);                                       //  bugfix v.14.10
         pix[0] = pix[1] = pix[2] = 0;
      }
   }

   return 1;
}


/**************************************************************************/

//  normal quit menu function

void m_quit(GtkWidget *, cchar *)
{
   int      busy;

   printz("quit \n");
   Fshutdown++;

   for (int ii = 0; ii < 20; ii++) {                                       //  wait up to 2 secs. if something running
      busy = checkpend("all quiet");
      if (! busy) break;
      zmainloop();
      zsleep(0.1);
   }
   
   if (busy) printz("busy function killed \n");

   quitxx();                                                               //  does not return
   return;
}

//  used also for main window delete and destroy events
//  does not return

void quitxx()
{
   Fshutdown++;
   gtk_window_get_position(MWIN,&mwgeom[0],&mwgeom[1]);                    //  get last window position 
   gtk_window_get_size(MWIN,&mwgeom[2],&mwgeom[3]);                        //    and size for next session
   save_params();                                                          //  save state for next session
   zdialog_inputs("save");                                                 //  save dialog inputs 
   zdialog_positions("save");                                              //  save dialog positions
   free_resources();                                                       //  delete temp files
   fflush(null);                                                           //  flush stdout, stderr
   exiftool_server(0);                                                     //  kill exiftool_server process
   shell_quiet("rm -R -f %s",tempdir);                                     //  delete temp files                  v.14.10
   gtk_main_quit();                                                        //  gone forever
   exit(1);                                                                //  just in case
}


/**************************************************************************/

//  function to copy an image file to the clipboard
//  used in thumbnail right-click popup menu
//  uses and frees clicked_file if present
//  else uses Fpxb->pixbuf (current image in window)

void copyto_clip(GtkWidget *, cchar *)
{
   GtkClipboard   *clipboard;
   GdkPixbuf      *pixbuf;
   GError         *gerror = 0;
   int            ww, hh, rs;
   uint8          *pixels;
   
   clipboard = gtk_clipboard_get(GDK_SELECTION_CLIPBOARD);
   gtk_clipboard_clear(clipboard);

   if (clicked_file) 
   {
      pixbuf = gdk_pixbuf_new_from_file(clicked_file,&gerror);
      zfree(clicked_file);
      clicked_file = 0;
      if (! pixbuf) return;
      gtk_clipboard_set_image(clipboard,pixbuf);
      g_object_unref(pixbuf);
   }
   else 
   {
      if (! Fpxb) return;
      pixels = Fpxb->pixels;
      ww = Fpxb->ww;
      hh = Fpxb->hh;
      rs = Fpxb->rs;
      pixbuf = gdk_pixbuf_new_from_data(pixels,GDKRGB,0,8,ww,hh,rs,0,0);
      gtk_clipboard_set_image(clipboard,pixbuf);
      g_object_unref(pixbuf);
   }

   return;
}


/**************************************************************************/

//  add a new file to the list of recent files, first position

void add_recent_file(cchar *newfile)                                       //  v.14.03
{
   FILE     *fidr, *fidw;
   char     *pp, tempfile[200], buff[maxfcc];
   int      ii, err;

   F1_help_topic = "recent_images";

   strcpy(tempfile,recentfiles_file);
   strcat(tempfile,"_temp");   
   
   fidw = fopen(tempfile,"w");                                             //  open output temp file
   if (! fidw) {
      zmessageACK(Mwin,0,"file error: %s",strerror(errno));
      return;
   }

   fprintf(fidw,"%s\n",newfile);                                           //  add new file as first in list
   
   fidr = fopen(recentfiles_file,"r");
   if (fidr) {
      for (ii = 0; ii < maxrecentfiles; ii++) {                            //  read list of recent files
         pp = fgets_trim(buff,maxfcc,fidr,1);
         if (! pp) break;
         if (strEqu(pp,newfile)) continue;                                 //  skip new file if present
         if (*pp == '/') fprintf(fidw,"%s\n",pp);                          //  write to output file
      }
      fclose(fidr);
   }
   
   fclose(fidw);

   err = rename(tempfile,recentfiles_file); 
   if (err) zmessageACK(Mwin,0,"file error: %s",strerror(errno));

   return;
}


/**************************************************************************/

//  update information in main window title bar

void set_mwin_title()
{
   int         cc, gtype, fposn, nfiles, nimages;
   char        *pp, titlebar[250];
   char        fname[100], gname[100], fdirk[100];

   if (! curr_file || *curr_file != '/') {                                 //  v.14.03
      gtk_window_set_title(MWIN,Frelease);
      return;
   }

   pp = (char *) strrchr(curr_file,'/');
   strncpy0(fname,pp+1,99);                                                //  file name
   cc = pp - curr_file;
   if (cc < 99) strncpy0(fdirk,curr_file,cc+2);                            //  get dirk/path/ if short enough
   else {
      strncpy(fdirk,curr_file,96);                                         //  or use /dirk/path...
      strcpy(fdirk+95,"...");
   }

   nfiles = navi::nfiles;                                                  //  total gallery files (incl. directories)
   nimages = navi::nimages;                                                //  total image files

   fposn = gallery_position(curr_file,curr_file_posn);                     //  curr. file in curr. gallery?       v.14.05
   if (fposn >= 0) {
      curr_file_posn = fposn;
      fposn = fposn + 1 - nfiles + nimages;                                //  position among images, 1-based
   }

   gtype = navi::gallerytype;

   if (gtype == 1)                                                         //  gallery name = directory
      snprintf(titlebar,250,"%s  %d/%d  %s  %s",
                   Frelease,fposn,nimages,fname,fdirk);
   else {
      if (gtype == 2 || gtype == 3)
         strcpy(gname,"SEARCH RESULTS");
      else if (gtype == 4) {
         pp = strrchr(navi::galleryname,'/');
         if (! pp) pp = navi::galleryname;
         else pp++;
         strcpy(gname,"COLLECTION: ");
         strncpy0(gname+12,pp,87);
      }
      else if (gtype == 5)
         strcpy(gname,"RECENT FILES");
      else if (gtype == 6)
         strcpy(gname,"NEWEST FILES");
      else
         strcpy(gname,"NO GALLERY");

      if (fposn > 0) 
         snprintf(titlebar,250,"%s  %s  %d/%d  %s  %s",                    //  window title bar
                Frelease,gname,fposn,nimages,fname,fdirk);
      else 
         snprintf(titlebar,250,"%s  %s  (*)/%d  %s  %s",                   //  image not in gallery               v.14.06
                Frelease,gname,nimages,fname,fdirk);
   }

   gtk_window_set_title(MWIN,titlebar);                                    //  zmainloop() removed (leaky)

   return;
}


/**************************************************************************

   Find all image files within given path.                     v.14.02

      dirk        directory path to search
      NF          count of filespecs returned 
      flist       list of filespecs returned 
                  (caller zfree() flist[NF] and flist)
      Fthumbs     include thumbnail files if Fthumbs = 1       v.14.03
      Finit       initialize flag (omit, default = 1)
                  (0 is used for internal recursive calls)

   Hidden directories and files (dotfiles) are suppressed.
   Returns 0 if OK, +N if error (errno is set).

***************************************************************************/

char  **fif_filelist;                  //  list of filespecs returned
int   fif_count1;                      //  filelist slots allocated
int   fif_count2;                      //  filelist slots filled

int find_imagefiles(cchar *dirk, char **&flist, int &NF, int Fthumbs, int Finit)
{
   int      flags = GLOB_NOSORT;
   int      err, cc, ftype;
   char     *file, *mdirk, **templist;

   glob_t   globdata;
   STATB    statdat;
   
   if (Finit) fif_count1 = fif_count2 = 0;

   globdata.gl_pathc = 0;                                                  //  glob() setup
   globdata.gl_offs = 0;
   globdata.gl_pathc = 0;
   NF = 0;
   flist = 0;

   mdirk = zstrdup(dirk,4);                                                //  append /* to input directory
   strcat(mdirk,"/*");

   err = glob(mdirk,flags,null,&globdata);                                 //  find all files in directory

   if (err) {
      if (err == GLOB_NOMATCH) err = 0;
      else if (err == GLOB_ABORTED) err = 1;
      else if (err == GLOB_NOSPACE) err = 2;
      else err = 3;
      goto fif_return;
   }
   
   for (uint ii = 0; ii < globdata.gl_pathc; ii++)                         //  loop found files
   {
      file = globdata.gl_pathv[ii];
      err = stat(file,&statdat);
      if (err) continue;

      if (S_ISDIR(statdat.st_mode)) {                                      //  directory, call recursively
         err = find_imagefiles(file,flist,NF,Fthumbs,0);
         if (err) goto fif_return;
         continue;
      }

      ftype = image_file_type(file);                                       //  check file type is image,
      if (ftype < 2 || ftype > 4) continue;                                //    RAW, or thumbnail file
      if (! Fthumbs && ftype == 4) continue;                               //  exclude thumbnail files

      if (fif_count2 == fif_count1) {                                      //  output list is full
         if (fif_count1 == 0) {
            fif_count1 = 1000;                                             //  initial space, 1000 files
            cc = fif_count1 * sizeof(char *);
            fif_filelist = (char **) zmalloc(cc);
         }
         else {
            templist = fif_filelist;                                       //  expand by 2x each time needed
            cc = fif_count1 * sizeof(char *);
            fif_filelist = (char **) zmalloc(cc+cc);
            memcpy(fif_filelist,templist,cc);
            memset(fif_filelist+fif_count1,0,cc);
            zfree(templist);
            fif_count1 *= 2;
         }
      }
      
      fif_filelist[fif_count2] = zstrdup(file);                            //  add file to output list
      fif_count2 += 1;
   }
   
   err = 0;

fif_return:

   globfree(&globdata);                                                    //  free memory
   zfree(mdirk);

   if (Finit) {
      NF = fif_count2;
      if (NF) flist = fif_filelist;
   }

   return err;
}   


/**************************************************************************
   Image file load and save functions
   PXM pixmap or PXB pixbuf <--> file on disk
***************************************************************************/

//  get the equivalent .tiff file name for a given RAW file
//  returned file name is subject to zfree()

char * raw_to_tiff(cchar *rawfile)
{
   char     *ptiff, *pext;

   ptiff = zstrdup(rawfile,6);
   pext = strrchr(ptiff,'.');
   if (! pext) pext = ptiff + strlen(ptiff);
   strcpy(pext,".tiff");
   return ptiff;
}


/**************************************************************************/

//  Function to preload image files hopefully ahead of need.
//  Usage: f_preload(next)
//    next = -1 / +1  to read previous / next image file in curr. gallery
//    preload_file will be non-zero if and when preload_pxb is available.

char     *preload_tryfile;
char     *preload_file;
char     preload_type[8];
int      preload_bpc;
int      preload_size;
int      preload_Nth;
PXB      *preload_pxb;

void f_preload(int next)
{
   void * preload_thread(void *);

   int      Nth, ftype = 0, err;
   int      nfiles = navi::nfiles;                                         //  gallery files (incl. directories)
   char     *file = 0;
   STATB    statb;

   while (mutex_trylock(&preload_lock) != 0) zsleep(0.1);                  //  wait if preload busy
   mutex_unlock(&preload_lock);

   if (preload_pxb) PXB_free(preload_pxb);
   if (preload_file) zfree(preload_file);                                  //  v.14.02.2
   preload_pxb = 0;
   preload_file = 0;
   preload_tryfile = 0;

   if (! curr_file) return;

   if (next == 1) {
      for (Nth = curr_file_posn+1; Nth < nfiles; Nth++) {
         file = gallery_find(Nth);
         if (! file) continue;
         ftype = image_file_type(file);
         if (ftype == 2 || ftype == 3) break;
         zfree(file);                                                      //  v.14.02.2
         file = 0;
      }
   }
   
   if (next == -1) {
      for (Nth = curr_file_posn-1; Nth >= 0; Nth--) {
         file = gallery_find(Nth);
         if (! file) continue;
         ftype = image_file_type(file);
         if (ftype == 2 || ftype == 3) break;
         zfree(file);                                                      //  v.14.02.2
         file = 0;
      }
   }
   
   if (! file) return;
   err = stat(file,&statb);
   if (err) return;

   if (strEqu(file,curr_file)) { 
      zfree(file);                                                         //  v.14.02.2
      return;
   }
   
   mutex_lock(&preload_lock);                                              //  lock now, thread unlocks
   preload_tryfile = file;
   preload_Nth = Nth;
   start_detached_thread(preload_thread,0);
   return;
}

void * preload_thread(void *)
{
   int            ftyp;
   char           *file;

   file = preload_tryfile;
   ftyp = image_file_type(file);

   if (ftyp == 2)
      preload_pxb = PXB_load(file,0);                                      //  image file
   
   else if (ftyp == 3)                                                     //  RAW file
      preload_pxb = PXB_load(file,0);

   if (preload_pxb) {
      preload_file = file;                                                 //  file is now available
      strcpy(preload_type,f_load_type);
      preload_bpc = f_load_bpc;
      preload_size = f_load_size;
   }
   else zfree(file);                                                       //  v.14.02.2

   mutex_unlock(&preload_lock);
   pthread_exit(0);
}


/**************************************************************************/

//  Open a file and initialize Fpxb pixbuf.
//
//  Nth:    if Nth matches file position in current gallery, curr_file_posn 
//          is set to Nth, otherwise it is searched and set correctly.
//          (a file can be present multiple times in a collection gallery).
//  Fkeep:  edit undo/redo stack is not purged, and current edits are kept 
//          after opening the new file (used by file_save()).
//  Fack:   failure will cause a popup ACK dialog.
//  zoom:   keep current zoom level and position, otherwise fit window.
//  
//  Following are set: curr_file_type, curr_file_bpc, curr_file_size.
//  Returns: 0 = OK, +N = error.
//  errors: 1  func busy or edit func not restartable
//          2  unsaved edits
//          3  file not found or user cancel
//          4  unsupported file type or open failure

int f_open(cchar *filespec, int Nth, int Fkeep, int Fack, int zoom)
{
   PXB         *tempxb = 0;
   int         err, fposn, ftyp, retcode = 0;
   static int  Fbusy = 0;
   char        *pp, *file;
   void        (*menufunc)(GtkWidget *, cchar *);
   STATB       statb;
   
   if (clicked_file) zfree(clicked_file);                                  //  v.14.03
   clicked_file = 0;

   if (Fbusy) return 1;
   Fbusy++;
   if (CEF && ! CEF->Frestart) goto ret1;                                  //  edit function not restartable
   if (Fthreadbusy) goto ret1;                                             //  do not interrupt thread 
   if (CEF) menufunc = CEF->menufunc;                                      //  active edit, save menu function
   else menufunc = 0;
   if (checkpend("mods")) goto ret2;                                       //  keep unsaved changes
   if (CEF) edit_cancel(0);                                                //  cancel pending edit
   
   sa_unselect();                                                          //  unselect area if any 

   if (filespec)  
      file = zstrdup(filespec);                                            //  use passed filespec
   else {
      pp = curr_file;                                                      //  use file open dialog
      if (! pp) pp = curr_dirk;
      file = zgetfile(ZTX("Open Image File"),"file",pp);
      if (! file) goto ret3;
   }

   err = stat(file,&statb);                                                //  check file exists
   if (err) {
      if (Fack) zmessageACK(Mwin,0,"%s \n %s",file,strerror(errno));       //  v.14.03
      zfree(file);
      goto ret3;
   }

   ftyp = image_file_type(file);                                           //  must be image or RAW file type
   if (ftyp == 4) {
      if (Fack) zmessageACK(Mwin,0,"thumbnail file");
      goto ret4;
   }

   if (ftyp != 2 && ftyp != 3) {                                           //  must be supported image file type
      if (Fack) zmessageACK(Mwin,0,ZTX("unknown file type"));              //    or RAW file 
      zfree(file);
      goto ret4;
   }
   
   Ffuncbusy = 1;                                                          //  may be large or RAW file, slow CPU

   while (mutex_trylock(&preload_lock) != 0) zsleep(0.05);                 //  wait if preload busy
   mutex_unlock(&preload_lock);

   if (preload_file && strEqu(file,preload_file)) {                        //  use preload file if there
      tempxb = preload_pxb;
      zfree(preload_file);                                                 //  v.14.02.2
      preload_file = 0;
      preload_pxb = 0;
      strcpy(f_load_type,preload_type);                                    //  file poop from preloader
      f_load_bpc = preload_bpc;
      f_load_size = preload_size;
   }
   
   else tempxb = PXB_load(file,1);                                         //  load image as PXB pixbuf

   Ffuncbusy = 0;

   if (! tempxb) {                                                         //  PXB_load() messages user
      zfree(file);
      goto ret4;
   }

   free_resources(Fkeep);                                                  //  free resources for old image file

   curr_file = file;                                                       //  new current file

   if (curr_dirk) zfree(curr_dirk);                                        //  set current directory
   curr_dirk = zstrdup(curr_file);                                         //    for new current file
   pp = strrchr(curr_dirk,'/');
   *pp = 0;

   Fpxb = tempxb;                                                          //  pixmap for current image

   strcpy(curr_file_type,f_load_type);                                     //  set curr_file_xxx from f_load_xxx
   curr_file_bpc = f_load_bpc;
   curr_file_size = f_load_size;

   fposn = gallery_position(file,Nth);                                     //  file position in gallery list
   if (fposn < 0) {                                                        //  not there
      gallery(curr_file,"init");                                           //  generate new gallery list 
      fposn = gallery_position(curr_file,0);                               //  position and count in gallery list
   }
   curr_file_posn = fposn;                                                 //  keep track of file position
   
   URS_reopen_pos = 0;                                                     //  not f_open_saved() 

   if (! zoom) {                                                           //  discard zoom state
      Fzoom = 0;                                                           //  zoom level = fit window
      zoomx = zoomy = 0;                                                   //  no zoom center
   }

   set_mwin_title();                                                       //  set win title from curr_file info 
   Fblankwindow = 0;

   add_recent_file(curr_file);                                             //  most recent file opened

   if (zdrename) m_rename(0,0);                                            //  update active rename dialog
   if (zdcopymove) copy_move(0);                                           //    "  copy/move dialog
   if (zddelete) m_delete(0,0);                                            //    "  delete dialog
   if (zdtrash) m_trash(0,0);                                              //    "  trash dialog
   if (zdexifview) meta_view(0);                                           //    "  EXIF/IPTC data view window
   if (zdeditmeta) m_edit_metadata(0,0);                                   //    "  edit main metadata dialog
   if (zdexifedit) m_meta_edit_any(0,0);                                   //    "  edit any metadata dialog
   if (zdeditgeotags) m_edit_geotags(0,0);                                 //    "  edit geotags dialog
   if (Fcaptions) m_captions(0,0);                                         //  show caption/comments at top 

   m_viewmode(0,"F");

   if (menufunc) menufunc(0,0);                                            //  restart edit function
   goto ret0;

   ret4: retcode++;
   ret3: retcode++;
   ret2: retcode++;
   ret1: retcode++;
   ret0:
   Fbusy = 0;
   return retcode;
}


/**************************************************************************/

//  Open a file that was just saved. Used by file_save().
//  The edit undo/redo stack is not purged and current edits are kept.
//  Following are set: 
//    curr_file  *_dirk  *_file_posn  *_file_type  *_file_bpc  *_file_size
//  Returns: 0 = OK, +N = error.

int f_open_saved()
{
   int      Nth = -1, fposn;

   if (clicked_file) zfree(clicked_file);                                  //  v.14.03
   clicked_file = 0;
   
   if (E0pxm) {                                                            //  edits were made                    v.14.09
      PXB_free(Fpxb);                                                      //  new window image
      Fpxb = PXM_PXB_copy(E0pxm);
   }
   
   if (curr_file) zfree(curr_file);
   curr_file = zstrdup(f_save_file);                                       //  new current file

   fposn = gallery_position(curr_file,Nth);                                //  file position in gallery list      v.14.04.2
   if (fposn < 0) {                                                        //  not there                          bugfix
      gallery(curr_file,"init");                                           //  generate new gallery list
      fposn = gallery_position(curr_file,0);                               //  position and count in gallery list
   }
   curr_file_posn = fposn;                                                 //  keep track of file position

   strcpy(curr_file_type,f_save_type);                                     //  set curr_file_xxx from f_save_xxx
   curr_file_bpc = f_save_bpc;
   curr_file_size = f_save_size;

   URS_reopen_pos = URS_pos;                                               //  track undo/redo stack position

   zoomx = zoomy = 0;                                                      //  no zoom center
   set_mwin_title();                                                       //  set win title from curr_file info 
   gtk_window_present(MWIN);

   if (zdexifview) meta_view(0);                                           //  update EXIF/IPTC view window
   return 0;
}


/**************************************************************************/

//  save current image to specified disk file (same or new).
//  set f_save_file, f_save_type, f_save_bpc, f_save_size.
//  update image index file.
//  returns 0 if OK, else +N.
//  If Fack is true, failure will cause a popup ACK dialog.

int f_save(char *outfile, cchar *type, int bpc, int Fack)                  //  use tiff/png/pixbuf libraries
{
   cchar    *exifkey[2] = { exif_orientation_key, exif_editlog_key };
   cchar    *exifdata[2];
   char     **ppv, *tempfile, *pext, *thumbfile;
   char     edithist[exif_maxcc];                                          //  edit history trail
   int      nkeys, err, cc1, cc2, ii, ii0;
   int      Fmodified;
   void     (*menufunc)(GtkWidget *, cchar *);
   STATB    fstat;

   if (! curr_file) return 1;
   if (checkpend("lock")) return 1;                                        //  v.14.09
   if (Fthreadbusy) return 1;                                              //  poss. edit thread running
   
   if (strEqu(type,"RAW")) {                                               //  disallow saving as RAW type
      zmessageACK(Mwin,0,ZTX("cannot save as RAW type"));
      return 1;
   }

   Fmodified = 0;
   menufunc = 0;

   if (CEF) {                                                              //  edit function active
      if (CEF->Fmods) Fmodified = 1;                                       //  active edits
      menufunc = CEF->menufunc;                                            //  save menu function for restart
      if (CEF->zd) zdialog_send_event(CEF->zd,"done");                     //  tell it to finish
      if (CEF) return 1;                                                   //  failed (HDR etc.)                  v.14.07
   }

   if (URS_pos > 0 && URS_saved[URS_pos] == 0) Fmodified = 1;              //  completed edits not saved          v.14.09

   outfile = zstrdup(outfile,6);                                           //  output file name

   pext = strrchr(outfile,'/');                                            //  force compatible file extension
   if (pext) pext = strrchr(pext,'.');                                     //    if not already
   if (! pext) pext = outfile + strlen(outfile);
   
   if (strEqu(type,"jpg") && ! strcasestr(".jpg .jpeg",pext))              //  standardize output file .ext
      strcpy(pext,".jpg");                                                 //    .jpg .tif .png .bmp 
   if (strEqu(type,"tif") && ! strcasestr(".tif .tiff",pext))
      strcpy(pext,".tif");
   if (strEqu(type,"png") && ! strcasestr(".png",pext))
      strcpy(pext,".png");
   if (strEqu(type,"bmp") && ! strcasestr(".bmp",pext))
      strcpy(pext,".bmp");
   
   if (strNeq(curr_file,outfile)) Fmodified = 1;                           //  bugfix                             v.14.10.2

   if (! Fmodified) {                                                      //  no unsaved edits
      if (strEqu(curr_file,outfile)) goto metadata;                        //  output = input, skip save image
      err = shell_quiet("cp \"%s\" \"%s\"",curr_file,outfile);             //  copy file directly to output       v.14.09
      if (err) return 1;                                                   //  (avoid jpeg re-compress)
      goto metadata;
   }
   
   Ffuncbusy = 1;                                                          //  may be large file, slow CPU

   if (! E0pxm) E0pxm = PXM_load(curr_file,1);                             //  no prior edits, load image file
   if (! E0pxm) {
      zfree(outfile);
      Ffuncbusy = 0;                                                       //  PXM_load() diagnoses error
      return 1;
   }

   tempfile = zstrdup(outfile,20);                                         //  temp file in same directory
   strcat(tempfile,"-temp.");
   strcat(tempfile,type);

   if (strEqu(type,"tif"))                                                 //  save as tif file (bpc = 8 or 16)
      err = PXM_TIFF_save(E0pxm,tempfile,bpc);

   else if (strEqu(type,"png"))                                            //  save as png file (bpc = 8 or 16)
      err = PXM_PNG_save(E0pxm,tempfile,bpc);

   else {                                                                  //  save as .bmp .jpg 
      err = PXM_ANY_save(E0pxm,tempfile);                                  //  (bpc = 8 always)
      bpc = 8;
   }

   if (err) {
      if (*file_errmess)                                                   //  pass error to user
         if (Fack) zmessageACK(Mwin,0,file_errmess);
      remove(tempfile);                                                    //  failure, clean up
      zfree(tempfile);
      zfree(outfile);
      Ffuncbusy = 0;
      return 1;
   }
   
   ppv = exif_get(curr_file,&exifkey[1],1);                                //  get prior edit history in EXIF
   if (ppv[0]) {
      strncpy0(edithist,ppv[0],exif_maxcc-2);
      zfree(ppv[0]);
      cc1 = strlen(edithist);                                              //  edits made before this file was opened
   }
   else cc1 = 0;                                                           //  none

   if ((CEF && CEF->Fmods) || URS_pos > 0)                                 //  active or completed edits
   {
      strcpy(edithist+cc1," ");                                            //  update edit history
      strcpy(edithist+cc1+1,"Fotoxx:");                                    //  append " Fotoxx:"
      cc1 += 8;
      
      if (URS_reopen_pos > 0) ii0 = URS_reopen_pos + 1;                    //  if last file saved was kept open,
      else ii0 = 1;                                                        //    these edits are already in history

      for (ii = ii0; ii <= URS_pos; ii++)                                  //  append list of edits from undo/redo stack
      {                                                                    //  (omit index 0 = initial image)
         cc2 = strlen(URS_funcs[ii]);
         strcpy(edithist+cc1,URS_funcs[ii]);
         strcpy(edithist+cc1+cc2,"|");
         cc1 += cc2 + 1;
      }

      if (CEF && CEF->Fmods) {                                             //  save during active edit function
         cc2 = strlen(CEF->funcname);                                      //  add curr. edit to history list
         strcpy(edithist+cc1,CEF->funcname);
         strcpy(edithist+cc1+cc2,"|");
         cc1 += cc2 + 1;
      }
   }

   exifdata[0] = "";                                                       //  remove EXIF orientation if present
   nkeys = 1;                                                              //  (assume saved file is upright)

   if (cc1) {                                                              //  prior and/or curr. edit hist
      exifdata[1] = edithist;
      nkeys = 2;
   }
   
   err = exif_copy(curr_file,tempfile,exifkey,exifdata,nkeys);             //  copy all EXIF/IPTC data to
   if (err && Fack)                                                        //    temp file with above revisions
      zmessageACK(Mwin,0,ZTX("Unable to copy EXIF/IPTC data"));

   err = rename(tempfile,outfile);                                         //  rename temp file to output file
   if (err) {
      if (Fack) zmessageACK(Mwin,0,wstrerror(err));
      remove(tempfile);                                                    //  delete temp file
      zfree(tempfile);
      zfree(outfile);
      Ffuncbusy = 0;
      return 2;                                                            //  could not save
   }

   zfree(tempfile);                                                        //  free memory
   
   for (ii = 0; ii <= URS_pos; ii++)                                       //  mark all prior edits as saved
      URS_saved[ii] = 1;

   thumbfile = image_thumbfile(outfile);                                   //  update thumbnail file
   if (thumbfile) zfree(thumbfile);

metadata:

   if (Fmetachanged) save_filemeta(outfile);                               //  update metadata and index          v.14.09
   else {                                                                  //  if no changes,
      load_filemeta(curr_file);                                            //    load metadata for index          v.14.10
      update_image_index(outfile);                                         //    update index file
   }
   
   if (navi::gallerytype == 1 && samedirk(outfile,curr_file)) {            //  output to curr. directory          v.14.03
      gallery(curr_file,"init");                                           //  update curr. gallery list
      set_mwin_title();                                                    //  update posn, count in title
   }

   add_recent_file(outfile);                                               //  first in recent files list
   
   if (f_save_file) zfree(f_save_file);                                    //  actual file saved (.ext may change)
   f_save_file = outfile;
   strcpy(f_save_type,type);                                               //  update f_save_xxx data
   f_save_bpc = bpc;
   err = stat(outfile,&fstat);
   f_save_size = fstat.st_size;

   if (menufunc) menufunc(0,0);                                            //  restart edit function

   Ffuncbusy = 0;
   return 0;
}


/**************************************************************************/

//  Load an image file into a PXB pixmap (pixbuf 8 bit color).
//  Also sets the following file scope variables:
//    f_load_type = "jpg" "tif" "png" "bmp" "ico" "other"
//    f_load_bpc = disk file bits per color = 1/8/16
//    f_load_size = disk file size
//  If Fack is true, failure will cause a popup ACK dialog
//  Do not call from a thread with Fack true.

PXB * PXB_load(cchar *file, int Fack)
{
   int      err, ftyp;
   cchar    *pext;
   PXB      *pxb;
   STATB    fstat;
   
   err = stat(file,&fstat);
   if (err) {
      if (Fack) zmessageACK(Mwin,0,ZTX("file not found: %s"),file);
      goto f_load_fail;
   }

   ftyp = image_file_type(file);
   if (ftyp != 2 && ftyp != 3) {
      if (Fack) zmessageACK(Mwin,0,ZTX("file type not supported: %s"),file);
      goto f_load_fail;
   }

   f_load_size = fstat.st_size;                                            //  set f_load_size

   pext = strrchr(file,'/');
   if (! pext) pext = file;
   pext = strrchr(pext,'.');
   if (! pext) pext = "";
   
   if (ftyp == 2) {
      if (strcasestr(".jpg .jpeg",pext)) strcpy(f_load_type,"jpg");        //  f_load_type = jpg/tif/png/...
      else if (strcasestr(".tif .tiff",pext)) strcpy(f_load_type,"tif");
      else if (strcasestr(".png",pext)) strcpy(f_load_type,"png");
      else if (strcasestr(".bmp",pext)) strcpy(f_load_type,"bmp");
      else if (strcasestr(".ico",pext)) strcpy(f_load_type,"ico");
      else strcpy(f_load_type,"other");
   }

   if (ftyp == 3) strcpy(f_load_type,"RAW");                               //  f_load_type = RAW
   
   if (strEqu(f_load_type,"tif"))                                          //  use loader for file type
      pxb = TIFF_PXB_load(file);                                           //  f_load_bpc = file bpc

   else if (strEqu(f_load_type,"png")) 
      pxb = PNG_PXB_load(file);

   else if (strEqu(f_load_type,"RAW"))                                     //  RAW file
      pxb = RAW_PXB_load(file);

   else pxb = ANY_PXB_load(file);

   if (pxb) return pxb;

   if (Fack && *file_errmess)                                              //  pass error to user
      zmessageACK(Mwin,0,file_errmess);

f_load_fail:
   *f_load_type = f_load_size = f_load_bpc = 0;
   return 0;
}


/**************************************************************************/

//  Load an image file into a PXM pixmap (float color).
//  Also sets the following file scope variables:
//    f_load_type = "jpg" "tif" "png" "bmp" "ico" "other"
//    f_load_bpc = disk file bits per color = 1/8/16
//    f_load_size = disk file size
//  If Fack is true, failure will cause a popup ACK dialog
//  Do not call from a thread with Fack true.

PXM * PXM_load(cchar *file, int Fack)
{
   int      err, ftyp;
   cchar    *pext;
   PXM      *pxm;
   STATB    fstat;
   
   err = stat(file,&fstat);
   if (err) {
      if (Fack) zmessageACK(Mwin,0,ZTX("file not found: %s"),file);
      goto f_load_fail;
   }

   ftyp = image_file_type(file);
   if (ftyp != 2 && ftyp != 3) {
      if (Fack) zmessageACK(Mwin,0,ZTX("file type not supported: %s"),file);
      goto f_load_fail;
   }

   f_load_size = fstat.st_size;                                            //  set f_load_size
   
   pext = strrchr(file,'/');
   if (! pext) pext = file;
   pext = strrchr(pext,'.');
   if (! pext) pext = "";
   
   if (strcasestr(".jpg .jpeg",pext)) strcpy(f_load_type,"jpg");           //  set f_load_type
   else if (strcasestr(".tif .tiff",pext)) strcpy(f_load_type,"tif");
   else if (strcasestr(".png",pext)) strcpy(f_load_type,"png");
   else if (strcasestr(".bmp",pext)) strcpy(f_load_type,"bmp");
   else if (strcasestr(".ico",pext)) strcpy(f_load_type,"ico");
   else strcpy(f_load_type,"other");
   
   if (strEqu(f_load_type,"tif"))                                          //  use loader for file type
      pxm = TIFF_PXM_load(file);                                           //  f_load_bpc = file bpc

   else if (strEqu(f_load_type,"png"))  
      pxm = PNG_PXM_load(file); 

   else if (ftyp == 3)                                                     //  RAW file
      pxm = RAW_PXM_load(file);

   else pxm = ANY_PXM_load(file);
   
   if (pxm) return pxm;

   if (Fack && *file_errmess)                                              //  pass error to user
      zmessageACK(Mwin,0,file_errmess);

f_load_fail:
   *f_load_type = f_load_size = f_load_bpc = 0;
   return 0;
}


/**************************************************************************
   TIFF file read and write functions
***************************************************************************/

//  Intercept TIFF warning messages (many)

void tiffwarninghandler(cchar *module, cchar *format, va_list ap)
{
   return;                                                                 //  stop flood of crap
   char  message[200];
   vsnprintf(message,199,format,ap);
   printz("TIFF warning: %s %s \n",module,message);
   return;
}


//  Load PXB from TIFF file using TIFF library. 
//  Native TIFF file bits/pixel >> f_load_bpc

PXB * TIFF_PXB_load(cchar *file)
{
   static int  ftf = 1;
   TIFF        *tiff;
   PXB         *pxb;
   char        *tiffbuff;
   uint8       *tiff8;
   uint16      *tiff16;
   uint8       *pix;
   uint16      bpc, nch, fmt; 
   int         ww, hh, rps, stb, nst;                                      //  int not uint
   int         tiffstat, row, col, strip;
   int         row1, row2, cc;
   
   if (ftf) {
      TIFFSetWarningHandler(tiffwarninghandler);                           //  intercept TIFF warning messages
      ftf = 0;
   }
   
   *file_errmess = 0;

   tiff = TIFFOpen(file,"r");
   if (! tiff) {
      snprintf(file_errmess,999,"TIFF file read error: %s",file);
      printz("%s\n",file_errmess);
      return 0;
   }

   TIFFGetField(tiff, TIFFTAG_IMAGEWIDTH, &ww);                            //  width
   TIFFGetField(tiff, TIFFTAG_IMAGELENGTH, &hh);                           //  height
   TIFFGetField(tiff, TIFFTAG_BITSPERSAMPLE, &bpc);                        //  bits per color, 1/8/16
   TIFFGetField(tiff, TIFFTAG_SAMPLESPERPIXEL, &nch);                      //  no. channels (colors), 1/3/4
   TIFFGetField(tiff, TIFFTAG_ROWSPERSTRIP, &rps);                         //  rows per strip
   TIFFGetField(tiff, TIFFTAG_SAMPLEFORMAT, &fmt);                         //  data format
   stb = TIFFStripSize(tiff);                                              //  strip size
   nst = TIFFNumberOfStrips(tiff);                                         //  number of strips
   
   if (ww > wwhh_limit || hh > wwhh_limit) {
      snprintf(file_errmess,999,"image too big, %dx%d: %s",ww,hh,file);
      printz("%s\n",file_errmess);
      TIFFClose(tiff);
      return 0;
   }

   if (bpc > 8 && bpc != 16) {                                             //  check for supported bits/color
      printz("TIFF bits=%d not supported: %s \n",bpc,file);
      TIFFClose(tiff);
      return ANY_PXB_load(file);                                           //  fallback to pixbuf loader
   }
   
   f_load_bpc = bpc;                                                       //  file bpc 1/8/16

   if (bpc <= 8)                                                           //  use universal TIFF reader
   {                                                                       //    if bits/color <= 8
      tiffbuff = (char *) zmalloc(ww*hh*4);
      tiffstat = TIFFReadRGBAImage(tiff, ww, hh, (uint *) tiffbuff, 0);
      TIFFClose(tiff);

      if (tiffstat != 1) {
         snprintf(file_errmess,999,"TIFF file read error: %s",file);
         printz("%s\n",file_errmess);
         zfree(tiffbuff);
         return 0;
      }

      pxb = PXB_make(ww,hh);                                               //  create PXB

      tiff8 = (uint8 *) tiffbuff;

      for (row = 0; row < hh; row++)                                       //  convert RGBA to RGB
      {
         pix = pxb->pixels + (hh-1 - row) * pxb->rs;

         for (col = 0; col < ww; col++)
         {
            pix[0] = tiff8[0];                                             //  0 - 255  >> 0 - 255
            pix[1] = tiff8[1];
            pix[2] = tiff8[2];
            pix += 3;
            tiff8 += 4;
         }
      }

      zfree(tiffbuff);
      return pxb;
   }

   pxb = PXB_make(ww,hh);                                                  //  TIFF file has 16 bits/color
   
   stb += 1000000;                                                         //  reduce risk of crash
   tiffbuff = (char *) zmalloc(stb);                                       //  read encoded strips 

   for (strip = 0; strip < nst; strip++)
   {
      cc = TIFFReadEncodedStrip(tiff,strip,tiffbuff,stb);
      if (cc < 0) {
         snprintf(file_errmess,999,"TIFF file read error: %s",file);
         printz("%s\n",file_errmess);
         TIFFClose(tiff);
         zfree(tiffbuff);
         PXB_free(pxb);
         return 0;
      }
      
      if (cc == 0) break;
      
      tiff16 = (uint16 *) tiffbuff;

      row1 = strip * rps;                                                  //  rows in strip
      row2 = row1 + cc / (ww * nch * 2);
      if (row2 > hh) row2 = hh;                                            //  rps unreliable if nst = 1
      
      for (row = row1; row < row2; row++)
      {
         pix = pxb->pixels + row * pxb->rs;

         for (col = 0; col < ww; col++)
         {
            pix[0] = tiff16[0] / 256;                                      //  0 - 65535  >>  0 - 255
            pix[1] = tiff16[1] / 256;
            pix[2] = tiff16[2] / 256;
            pix += 3;
            tiff16 += nch;
         }
      }
   }

   TIFFClose(tiff);
   zfree(tiffbuff);
   return pxb;
}


//  Load PXM from TIFF file using TIFF library. 
//  Native TIFF file bits/pixel >> f_load_bpc

PXM * TIFF_PXM_load(cchar *file)
{
   static int  ftf = 1;
   TIFF        *tiff;
   PXM         *pxm;
   char        *tiffbuff;
   uint8       *tiff8;
   uint16      *tiff16;
   float       *pix;
   uint16      bpc, nch, fmt; 
   int         ww, hh, rps, stb, nst;                                      //  int not uint
   int         tiffstat, row, col, strip, cc;

   if (ftf) {
      TIFFSetWarningHandler(tiffwarninghandler);                           //  intercept TIFF warning messages
      ftf = 0;
   }

   *file_errmess = 0;

   tiff = TIFFOpen(file,"r");
   if (! tiff) {
      snprintf(file_errmess,999,"TIFF file read error: %s",file);
      printz("%s\n",file_errmess);
      return 0;
   }

   TIFFGetField(tiff, TIFFTAG_IMAGEWIDTH, &ww);                            //  width
   TIFFGetField(tiff, TIFFTAG_IMAGELENGTH, &hh);                           //  height
   TIFFGetField(tiff, TIFFTAG_BITSPERSAMPLE, &bpc);                        //  bits per color, 1/8/16
   TIFFGetField(tiff, TIFFTAG_SAMPLESPERPIXEL, &nch);                      //  no. channels (colors), 1/3/4
   TIFFGetField(tiff, TIFFTAG_ROWSPERSTRIP, &rps);                         //  rows per strip
   TIFFGetField(tiff, TIFFTAG_SAMPLEFORMAT, &fmt);                         //  data format
   stb = TIFFStripSize(tiff);                                              //  strip size
   nst = TIFFNumberOfStrips(tiff);                                         //  number of strips

   if (ww > wwhh_limit || hh > wwhh_limit) {
      snprintf(file_errmess,999,"image too big, %dx%d: %s",ww,hh,file);
      printz("%s\n",file_errmess);
      TIFFClose(tiff);
      return 0;
   }

   if (bpc > 8 && bpc != 16) {                                             //  check for supported bits/color
      printz("TIFF bits=%d not supported: %s \n",bpc,file);
      TIFFClose(tiff);
      return ANY_PXM_load(file);                                           //  fallback to pixbuf loader
   }

   f_load_bpc = bpc;                                                       //  file bpc 1/8/16

   if (bpc <= 8)                                                           //  use universal TIFF reader
   {                                                                       //    if bits/color <= 8
      tiffbuff = (char *) zmalloc(ww*hh*4);
      tiffstat = TIFFReadRGBAImage(tiff, ww, hh, (uint *) tiffbuff, 0);
      TIFFClose(tiff);

      if (tiffstat != 1) {
         snprintf(file_errmess,999,"TIFF file read error: %s",file);
         printz("%s\n",file_errmess);
         zfree(tiffbuff);
         return 0;
      }

      pxm = PXM_make(ww,hh);                                               //  create PXM

      tiff8 = (uint8 *) tiffbuff;

      for (row = 0; row < hh; row++)                                       //  convert RGBA to RGB
      {
         pix = pxm->pixels + (hh-1 - row) * ww * 3;

         for (col = 0; col < ww; col++)
         {
            pix[0] = tiff8[0];                                             //  0 - 255  >>  0.0 - 255.0 
            pix[1] = tiff8[1];
            pix[2] = tiff8[2];
            pix += 3;
            tiff8 += 4;
         }
      }

      zfree(tiffbuff);
      return pxm;
   }

   pxm = PXM_make(ww,hh);                                                  //  TIFF file has 16 bits/color
   
   stb += 1000000;                                                         //  reduce risk of crash
   tiffbuff = (char *) zmalloc(stb);                                       //  read encoded strips 

   for (strip = 0; strip < nst; strip++)
   {
      cc = TIFFReadEncodedStrip(tiff,strip,tiffbuff,stb);
      if (cc < 0) {
         snprintf(file_errmess,999,"TIFF file read error: %s",file);
         printz("%s\n",file_errmess);
         TIFFClose(tiff);
         zfree(tiffbuff);
         PXM_free(pxm);
         return 0;
      }
      
      if (cc == 0) break;
      
      tiff16 = (uint16 *) tiffbuff;
      row = strip * rps;
      pix = pxm->pixels + row * ww * 3;

      while (cc >= 6)
      {
         pix[0] = tiff16[0] / 256.0;                                       //  0 - 65535  >>  0.0 - 255.996
         pix[1] = tiff16[1] / 256.0;
         pix[2] = tiff16[2] / 256.0;
         pix += 3;
         tiff16 += nch;
         cc -= nch * 2;
      }
   }

   TIFFClose(tiff);
   zfree(tiffbuff);
   return pxm;
}


//  Write PXM to TIFF file using TIFF library. 
//  TIFF file bpc is 8 or 16.
//  Returns 0 if OK, +N if error.

int PXM_TIFF_save(PXM *pxm, cchar *file, int bpc)
{
   static int  ftf = 1;
   TIFF        *tiff;
   uint8       *tiff8;
   uint16      *tiff16;
   float       *pix;
   int         tiffstat = 0;
   int         ww, hh, row, col, rowcc;                                    //  int not uint
   int         nch, pm = 2, pc = 1, comp = 5; 
   char        *tiffbuff;
   
   if (ftf) {
      TIFFSetWarningHandler(tiffwarninghandler);                           //  intercept TIFF warning messages
      ftf = 0;
   }

   *file_errmess = 0;

   tiff = TIFFOpen(file,"w");
   if (! tiff) {
      snprintf(file_errmess,999,"TIFF file write error: %s",file);
      printz("%s\n",file_errmess);
      return 0;
   }
   
   ww = pxm->ww;
   hh = pxm->hh;
   nch = 3;                                                                //  alpha channel removed

   TIFFSetField(tiff, TIFFTAG_IMAGEWIDTH, ww);
   TIFFSetField(tiff, TIFFTAG_BITSPERSAMPLE, bpc);
   TIFFSetField(tiff, TIFFTAG_SAMPLESPERPIXEL, nch);
   TIFFSetField(tiff, TIFFTAG_PHOTOMETRIC, pm);                            //  RGB
   TIFFSetField(tiff, TIFFTAG_PLANARCONFIG, pc);
   TIFFSetField(tiff, TIFFTAG_COMPRESSION, comp);                          //  LZW

   rowcc = TIFFScanlineSize(tiff);
   tiffbuff = (char*) zmalloc(rowcc);

   for (row = 0; row < hh; row++)
   {
      if (bpc == 8)                                                        //  write 8-bit TIFF
      {
         tiff8 = (uint8 *) tiffbuff;
         pix = pxm->pixels + row * ww * 3;

         for (col = 0; col < ww; col++)
         {
            tiff8[0] = pix[0];                                             //  0.0 - 255.996  >>  0 - 255
            tiff8[1] = pix[1];
            tiff8[2] = pix[2];
            pix += 3;
            tiff8 += nch;
         }
      }

      if (bpc == 16)                                                       //  write 16-bit TIFF
      {
         tiff16 = (uint16 *) tiffbuff;
         pix = pxm->pixels + row * ww * 3;

         for (col = 0; col < ww; col++)
         {
            tiff16[0] = 256.0 * pix[0];                                    //  0.0 - 255.996  >>  0.0 >> 65535
            tiff16[1] = 256.0 * pix[1];
            tiff16[2] = 256.0 * pix[2];
            pix += 3;
            tiff16 += nch;
         }
      }

      tiffstat = TIFFWriteScanline(tiff,tiffbuff,row,0);
      if (tiffstat != 1) break;
   }

   TIFFClose(tiff);
   zfree(tiffbuff);
   
   if (tiffstat == 1) return 0;
   snprintf(file_errmess,999,"TIFF file write error: %s",file);
   printz("%s\n",file_errmess);
   return 2;
}


/**************************************************************************
   PNG file read and write functions
***************************************************************************/

//  Load PXB from PNG file using PNG library. 
//  Native PNG file bits/pixel >> f_load_bpc

PXB * PNG_PXB_load(cchar *file) 
{
   png_structp    pngstruct = 0;
   png_infop      pnginfo = 0;
   FILE           *fid;
   int            ww, hh, bpc, nch, colortype;
   uchar          **pngrows;
   uint8          *png8;
   uint16         *png16;
   PXB            *pxb;
   uint8          *pix;
   int            row, col, bit, ii, rgbval;
   static int     ftf = 1;
   static int     mask[9] = { 0, 0x80, 0xc0, 0xe0, 0xf0, 0xf8, 0xfc, 0xfe, 0xff };
   static float   scale[9];
   
   if (ftf) {
      ftf = 0;
      for (ii = 1; ii < 9; ii++)                                           //  used for < 8 bits per color
         scale[ii] = 255.0 / mask[ii];
   }

   *file_errmess = 0;

   fid = fopen(file,"r");                                                  //  open file
   if (! fid) goto erret;

   pngstruct = png_create_read_struct(PNG_LIBPNG_VER_STRING,0,0,0);        //  create png structs
   if (! pngstruct) goto erret;
   pnginfo = png_create_info_struct(pngstruct);
   if (! pnginfo) goto erret;
   if (setjmp(png_jmpbuf(pngstruct))) goto erret;                          //  setup error handling

   png_init_io(pngstruct,fid);                                             //  read png file
   png_read_png(pngstruct,pnginfo,PNG_TRANSFORM_SWAP_ENDIAN,0);
   fclose(fid);

   ww = png_get_image_width(pngstruct,pnginfo);                            //  width
   hh = png_get_image_height(pngstruct,pnginfo);                           //  height
   bpc = png_get_bit_depth(pngstruct,pnginfo);                             //  bits per color (channel)
   nch = png_get_channels(pngstruct,pnginfo);                              //  no. channels
   colortype = png_get_color_type(pngstruct,pnginfo);                      //  color type
   pngrows = png_get_rows(pngstruct,pnginfo);                              //  array of png row pointers

   if (ww > wwhh_limit || hh > wwhh_limit) {
      png_destroy_read_struct(&pngstruct,&pnginfo,0);
      snprintf(file_errmess,999,"image too big, %dx%d: %s",ww,hh,file);
      printz("%s\n",file_errmess);
      return 0;
   }

   if (bpc > 8 && bpc != 16) {
      png_destroy_read_struct(&pngstruct,&pnginfo,0);
      printz("PNG bpc %d not supported: %s \n",bpc,file);
      return ANY_PXB_load(file);                                           //  fallback to pixbuf loader
   }
   
   if (colortype != PNG_COLOR_TYPE_GRAY && colortype != PNG_COLOR_TYPE_RGB) {
      png_destroy_read_struct(&pngstruct,&pnginfo,0);
      printz("PNG file not RGB or gray scale: %s \n",file);
      return ANY_PXB_load(file);                                           //  fallback to pixbuf loader
   }

   pxb = PXB_make(ww,hh);                                                  //  create PXB

   if (bpc == 8)                                                           //  8 bits/channel
   {
      for (row = 0; row < hh; row++)
      {
         png8 = (uint8 *) pngrows[row];
         pix = pxb->pixels + row * pxb->rs;
         
         for (col = 0; col < ww; col++)
         {
            if (nch < 3) {                                                 //  gray or gray + alpha
               pix[0] = pix[1] = pix[2] = *png8;
            }
            else {                                                         //  RGB or RGB + alpha
               pix[0] = png8[0];
               pix[1] = png8[1];
               pix[2] = png8[2];
            }
            pix += 3;
            png8 += nch;
         }
      }
   }

   else if (bpc < 8)                                                       //  1, 2, 4  bits/channel
   {
      for (row = 0; row < hh; row++)
      {
         png8 = (uint8 *) pngrows[row];
         pix = pxb->pixels + row * pxb->rs;
         bit = 0;
         
         for (col = 0; col < ww; col++)
         {
            if (nch < 3) {                                                 //  gray or gray + alpha
               rgbval = *png8 << bit;
               rgbval = (rgbval & mask[bpc]) * scale[bpc];
               pix[0] = pix[1] = pix[2] = rgbval;
               for (ii = 0; ii < nch; ii++)
               {
                  bit += bpc;
                  if (bit > 7) {
                     bit = bit - 8;
                     png8++;
                  }
               }
            }
            else {
               for (ii = 0; ii < 3; ii++)                                  //  RGB or RGB + alpha
               {
                  rgbval = *png8 << bit;
                  rgbval = (rgbval & mask[bpc]) * scale[bpc];
                  pix[ii] = rgbval;
                  bit += bpc;
                  if (bit > 7) {
                     bit = bit - 8;
                     png8++;
                  }
               }
               if (nch == 4) {
                  bit += bpc;
                  if (bit > 7) {
                     bit = bit - 8;
                     png8++;
                  }
               }
            }
            pix += 3;
         }
      }
   }

   else if (bpc == 16)                                                     //  16 bits/channel
   {
      for (row = 0; row < hh; row++)
      {
         png16 = (uint16 *) pngrows[row];
         pix = pxb->pixels + row * pxb->rs;

         for (col = 0; col < ww; col++)
         {
            if (nch < 3) {                                                 //  gray or gray + alpha
               pix[0] = pix[1] = pix[2] = (*png16) >> 8;
               png16 += nch;
            }
            else {
               pix[0] = png16[0] >> 8;                                     //  16 bits to 8 bits
               pix[1] = png16[1] >> 8;
               pix[2] = png16[2] >> 8;
               png16 += nch;
            }
            pix += 3;
         }
      }
   }

   else zappcrash("PNGread() bug %d %d",bpc,nch);

   png_destroy_read_struct(&pngstruct,&pnginfo,0);                         //  release memory

   f_load_bpc = bpc;                                                       //  file bpc 1/8/16
   return pxb;

erret:
   if (fid) fclose(fid);
   snprintf(file_errmess,999,"PNG file read error: %s",file);
   printz("%s\n",file_errmess);
   return 0;
}


//  Load PXM from PNG file using PNG library. 
//  Native PNG file bits/pixel >> f_load_bpc

PXM * PNG_PXM_load(cchar *file)
{
   png_structp    pngstruct = 0;
   png_infop      pnginfo = 0;
   FILE           *fid;
   int            ww, hh, bpc, nch, colortype;
   uchar          **pngrows;
   uint8          *png8;
   uint16         *png16;
   PXM            *pxm;
   float          *pix, f256 = 1.0 / 256.0;
   int            row, col, bit, ii, rgbval;
   static int     ftf = 1;
   static int     mask[9] = { 0, 0x80, 0xc0, 0xe0, 0xf0, 0xf8, 0xfc, 0xfe, 0xff };
   static float   scale[9];
   
   if (ftf) {
      ftf = 0;
      for (ii = 1; ii < 9; ii++)                                           //  used for < 8 bits per color
         scale[ii] = 255.0 / mask[ii];
   }

   *file_errmess = 0;

   fid = fopen(file,"r");                                                  //  open file
   if (! fid) goto erret;

   pngstruct = png_create_read_struct(PNG_LIBPNG_VER_STRING,0,0,0);        //  create png structs
   if (! pngstruct) goto erret;
   pnginfo = png_create_info_struct(pngstruct);
   if (! pnginfo) goto erret;
   if (setjmp(png_jmpbuf(pngstruct))) goto erret;                          //  setup error handling

   png_init_io(pngstruct,fid);                                             //  read png file
   png_read_png(pngstruct,pnginfo,PNG_TRANSFORM_SWAP_ENDIAN,0);
   fclose(fid);
   
   ww = png_get_image_width(pngstruct,pnginfo);                            //  width
   hh = png_get_image_height(pngstruct,pnginfo);                           //  height
   bpc = png_get_bit_depth(pngstruct,pnginfo);                             //  bits per color (channel)
   nch = png_get_channels(pngstruct,pnginfo);                              //  no. channels
   colortype = png_get_color_type(pngstruct,pnginfo);                      //  color type
   pngrows = png_get_rows(pngstruct,pnginfo);                              //  array of png row pointers
   
   if (ww > wwhh_limit || hh > wwhh_limit) {
      png_destroy_read_struct(&pngstruct,&pnginfo,0);
      snprintf(file_errmess,999,"image too big, %dx%d: %s",ww,hh,file);
      printz("%s\n",file_errmess);
      return 0;
   }

   if (bpc > 8 && bpc != 16) {
      png_destroy_read_struct(&pngstruct,&pnginfo,0);
      printz("PNG bpc %d not supported: %s \n",bpc,file);
      return ANY_PXM_load(file);                                           //  fallback to pixbuf loader
   }

   if (colortype != PNG_COLOR_TYPE_GRAY && colortype != PNG_COLOR_TYPE_RGB && colortype != PNG_COLOR_TYPE_RGB_ALPHA) {
      png_destroy_read_struct(&pngstruct,&pnginfo,0);
      printz("PNG color type %d not supported: %s \n",colortype,file);
      return ANY_PXM_load(file);                                           //  fallback to pixbuf loader
   }

   pxm = PXM_make(ww,hh);                                                  //  create PXM

   if (bpc == 8)                                                           //  8 bits/channel
   {
      for (row = 0; row < hh; row++)
      {
         png8 = (uint8 *) pngrows[row];
         pix = pxm->pixels + row * ww * 3;
         
         for (col = 0; col < ww; col++)
         {
            if (nch < 3) {                                                 //  gray or gray + alpha
               pix[0] = pix[1] = pix[2] = *png8;
            }
            else {                                                         //  RGB or RGB + alpha
               pix[0] = png8[0];
               pix[1] = png8[1];
               pix[2] = png8[2];
            }
            pix += 3;
            png8 += nch;
         }
      }
   }

   else if (bpc < 8)                                                       //  1, 2, 4  bits/channel
   {
      for (row = 0; row < hh; row++)
      {
         png8 = (uint8 *) pngrows[row];
         pix = pxm->pixels + row * ww * 3;
         bit = 0;
         
         for (col = 0; col < ww; col++)
         {
            if (nch < 3) {                                                 //  gray or gray + alpha
               rgbval = *png8 << bit;
               rgbval = (rgbval & mask[bpc]) * scale[bpc];
               pix[0] = pix[1] = pix[2] = rgbval;
               for (ii = 0; ii < nch; ii++)
               {
                  bit += bpc;
                  if (bit > 7) {
                     bit = bit - 8;
                     png8++;
                  }
               }
            }
            else {
               for (ii = 0; ii < 3; ii++)                                  //  RGB or RGB + alpha
               {
                  rgbval = *png8 << bit;
                  rgbval = (rgbval & mask[bpc]) * scale[bpc];
                  pix[ii] = rgbval;
                  bit += bpc;
                  if (bit > 7) {
                     bit = bit - 8;
                     png8++;
                  }
               }
               if (nch == 4) {
                  bit += bpc;
                  if (bit > 7) {
                     bit = bit - 8;
                     png8++;
                  }
               }
            }
            pix += 3;
         }
      }
   }

   else if (bpc == 16)                                                     //  16 bits/channel
   {
      for (row = 0; row < hh; row++)
      {
         png16 = (uint16 *) pngrows[row];
         pix = pxm->pixels + row * ww * 3;
         
         for (col = 0; col < ww; col++)
         {
            if (nch < 3) {                                                 //  gray or gray + alpha
               pix[0] = pix[1] = pix[2] = f256 * (*png16);
               png16 += nch;
            }
            else {                                                         //  RGB or RGB + alpha
               pix[0] = f256 * png16[0];                                   //  0-65535 >> 0-255.996
               pix[1] = f256 * png16[1];
               pix[2] = f256 * png16[2];
               png16 += nch;
            }
            pix += 3;
         }
      }
   }

   else zappcrash("PNGread() bug %d %d",bpc,nch);

   png_destroy_read_struct(&pngstruct,&pnginfo,0);                         //  release memory

   f_load_bpc = bpc;                                                       //  file bpc 1/8/16
   return pxm;

erret:
   if (fid) fclose(fid);
   snprintf(file_errmess,999,"PNG file read error: %s",file);
   printz("%s\n",file_errmess);
   return 0;
}


//  Write PXM to PNG file using PNG library. 
//  File bpc is 8 or 16.
//  returns 0 if OK, +N if error.

int PXM_PNG_save(PXM *pxm, cchar *file, int bpc)
{
   png_structp    pngstruct;
   png_infop      pnginfo;
   FILE           *fid;
   uchar          **pngrows;
   uint8          *png8;
   uint16         *png16;
   float          *pix;
   int            ww, hh, cc, row, col;
   
   if (bpc != 8 && bpc != 16) zappcrash("PNG write bpc: %d",bpc);
   
   *file_errmess = 0;

   ww = pxm->ww;
   hh = pxm->hh;
   
   fid = fopen(file,"w");                                                  //  open output file
   if (! fid) goto erret;

   pngstruct = png_create_write_struct(PNG_LIBPNG_VER_STRING,0,0,0);       //  create png structs
   if (! pngstruct) goto erret;
   pnginfo = png_create_info_struct(pngstruct);
   if (! pnginfo) goto erret;
   if (setjmp(png_jmpbuf(pngstruct))) goto erret;                          //  setup error handling

   png_init_io(pngstruct,fid);                                             //  initz. for writing file
   
   png_set_compression_level(pngstruct,4);                                 //  1...9 = faster...smaller size

   png_set_IHDR(pngstruct,pnginfo,ww,hh,bpc,PNG_COLOR_TYPE_RGB,0,0,0);     //  8 or 18 bit RGB
   
   pngrows = (uchar **) zmalloc(hh * sizeof(char *));                      //  allocate rows of RGB data
   if (bpc == 8) cc = ww * 3;                                              //  3 bytes per RGB pixel
   else cc = ww * 6;                                                       //  6 bytes per RGB pixel
   for (row = 0; row < hh; row++) 
      pngrows[row] = (uchar *) zmalloc(cc);

   png_set_rows(pngstruct,pnginfo,pngrows);                                //  array of png row pointers
   
   if (bpc == 8)                                                           //  8 bit RGB
   {
      for (row = 0; row < hh; row++)
      {
         png8 = (uint8 *) pngrows[row];
         pix = pxm->pixels + row * ww * 3;

         for (col = 0; col < ww; col++)
         {
            png8[0] = pix[0];
            png8[1] = pix[1];
            png8[2] = pix[2];
            png8 += 3;
            pix += 3;
         }
      }
   }

   else if (bpc == 16)                                                     //  16 bit RGB
   {
      for (row = 0; row < hh; row++)
      {
         png16 = (uint16 *) pngrows[row];
         pix = pxm->pixels + row * ww * 3;
         
         for (col = 0; col < ww; col++)
         {
            png16[0] = 256.0 * pix[0];
            png16[1] = 256.0 * pix[1];
            png16[2] = 256.0 * pix[2];
            png16 += 3;
            pix += 3;
         }
      }
   }
   
   png_write_png(pngstruct,pnginfo,PNG_TRANSFORM_SWAP_ENDIAN,0);           //  write the file 
   fclose(fid);

   png_destroy_write_struct(&pngstruct,&pnginfo);                          //  release memory

   for (row = 0; row < hh; row++)
      zfree(pngrows[row]);
   zfree(pngrows);

   return 0;

erret:
   if (fid) fclose(fid);
   snprintf(file_errmess,999,"PNG file write error: %s",file);
   printz("%s\n",file_errmess);
   return 2;
}


/**************************************************************************
   JPEG and other file types read and write functions
***************************************************************************/

//  Load PXB from jpeg and other file types using the pixbuf library. 
//  bpc = 8.

PXB * ANY_PXB_load(cchar *file) 
{
   GError         *gerror = 0;
   PXB            *pxb;
   GdkPixbuf      *pixbuf;
   int            ww, hh, px, py, nch, rowst;
   uint8          *bmp1, *pix1;
   uint8          *bmp2, *pix2;
   
   *file_errmess = 0;

   pixbuf = gdk_pixbuf_new_from_file(file,&gerror);
   if (! pixbuf) {
      if (gerror) printz("%s\n",gerror->message);
      snprintf(file_errmess,999,"(pixbuf) file read error: %s",file);
      printz("%s\n",file_errmess);
      return 0;
   }

   ww = gdk_pixbuf_get_width(pixbuf);
   hh = gdk_pixbuf_get_height(pixbuf);
   nch = gdk_pixbuf_get_n_channels(pixbuf);
   rowst = gdk_pixbuf_get_rowstride(pixbuf);
   bmp1 = gdk_pixbuf_get_pixels(pixbuf);
   
   if (ww > wwhh_limit || hh > wwhh_limit) {
      snprintf(file_errmess,999,"image too big, %dx%d: %s",ww,hh,file);
      printz("%s\n",file_errmess);
      g_object_unref(pixbuf);
      return 0;
   }

   pxb = PXB_make(ww,hh);
   bmp2 = pxb->pixels;
   
   for (py = 0; py < hh; py++)
   {
      pix1 = bmp1 + rowst * py;
      pix2 = bmp2 + py * pxb->rs;
      
      if (nch >= 3)
      {
         for (px = 0; px < ww; px++)
         {
            pix2[0] = pix1[0];
            pix2[1] = pix1[1];
            pix2[2] = pix1[2];
            pix1 += nch;
            pix2 += 3;
         }
      }
      
      else
      {
         for (px = 0; px < ww; px++)
         {
            pix2[0] = pix2[1] = pix2[2] = pix1[0];
            pix1 += nch;
            pix2 += 3;
         }
      }
   }
   
   f_load_bpc = 8;                                                         //  file bits per color
   g_object_unref(pixbuf);

   return pxb;
}


//  Load PXM from jpeg and other file types using the pixbuf library. 
//  bpc = 8.

PXM * ANY_PXM_load(cchar *file)
{
   GError         *gerror = 0;
   PXM            *pxm;
   GdkPixbuf      *pixbuf;
   int            ww, hh, px, py, nch, rowst;
   uint8          *bmp1, *pix1;
   float          *bmp2, *pix2;
   static int     ftf = 1;
   static float   ftab[256];
   
   if (ftf) {                                                              //  table lookup for speed
      ftf = 0;
      for (int ii = 0; ii < 256; ii++)
         ftab[ii] = ii;
   }
   
   *file_errmess = 0;

   pixbuf = gdk_pixbuf_new_from_file(file,&gerror);
   if (! pixbuf) {
      if (gerror) printz("%s\n",gerror->message);
      snprintf(file_errmess,999,"(pixbuf) file read error: %s",file);
      printz("%s\n",file_errmess);
      return 0;
   }

   ww = gdk_pixbuf_get_width(pixbuf);
   hh = gdk_pixbuf_get_height(pixbuf);
   nch = gdk_pixbuf_get_n_channels(pixbuf);
   rowst = gdk_pixbuf_get_rowstride(pixbuf);
   bmp1 = gdk_pixbuf_get_pixels(pixbuf);
   
   if (ww > wwhh_limit || hh > wwhh_limit) {
      snprintf(file_errmess,999,"image too big, %dx%d: %s",ww,hh,file);
      printz("%s\n",file_errmess);
      g_object_unref(pixbuf);
      return 0;
   }

   pxm = PXM_make(ww,hh);
   bmp2 = pxm->pixels;
   
   for (py = 0; py < hh; py++)
   {
      pix1 = bmp1 + rowst * py;
      pix2 = bmp2 + py * ww * 3;
      
      if (nch >= 3)
      {
         for (px = 0; px < ww; px++)
         {
            pix2[0] = ftab[pix1[0]];                                       //  0-255  >>  0.0-255.0
            pix2[1] = ftab[pix1[1]];
            pix2[2] = ftab[pix1[2]];
            pix1 += nch;
            pix2 += 3;
         }
      }
      
      else
      {
         for (px = 0; px < ww; px++)
         {
            pix2[0] = pix2[1] = pix2[2] = ftab[pix1[0]];
            pix1 += nch;
            pix2 += 3;
         }
      }
   }
   
   f_load_bpc = 8;                                                         //  file bits per color
   g_object_unref(pixbuf);

   return pxm;
}


//  Write PXM to jpeg and other file types using the pixbuf library.
//  bpc = 8.
//  returns 0 if OK, +N if error.

int PXM_ANY_save(PXM *pxm, cchar *file)
{
   int         ww, hh, px, py, rowst;
   float       *bmp1, *pix1;
   char        txqual[8];
   uint8       *bmp2, *pix2;
   GdkPixbuf   *pixbuf;
   cchar       *pext;
   GError      *gerror = 0;
   int         pxbstat;

   *file_errmess = 0;

   ww = pxm->ww;
   hh = pxm->hh;

   pixbuf = gdk_pixbuf_new(GDKRGB,0,8,ww,hh);
   if (! pixbuf) zappcrash("pixbuf allocation failure");

   bmp1 = pxm->pixels;
   bmp2 = gdk_pixbuf_get_pixels(pixbuf);
   rowst = gdk_pixbuf_get_rowstride(pixbuf);

   for (py = 0; py < hh; py++)
   {
      pix1 = bmp1 + py * ww * 3;
      pix2 = bmp2 + rowst * py;

      for (px = 0; px < ww; px++)
      {
         pix2[0] = pix1[0];                                                //  0.0 - 255.996  >>  0 - 255
         pix2[1] = pix1[1];
         pix2[2] = pix1[2];
         pix1 += 3;
         pix2 += 3;
      }
   }

   pext = strrchr(file,'/');
   if (! pext) pext = file;
   pext = strrchr(pext,'.');
   if (! pext) pext = "";

   if (strstr(".png .PNG",pext))
      pxbstat = gdk_pixbuf_save(pixbuf,file,"png",&gerror,"compression","4",null);

   else if (strstr(".bmp .BMP",pext))
      pxbstat = gdk_pixbuf_save(pixbuf,file,"bmp",&gerror,null);

   else {
      snprintf(txqual,8,"%d",jpeg_1x_quality);
      pxbstat = gdk_pixbuf_save(pixbuf,file,"jpeg",&gerror,"quality",txqual,null);
      jpeg_1x_quality = jpeg_def_quality;                                  //  reset to default after use
   }

   g_object_unref(pixbuf);
   if (pxbstat) return 0;

   if (gerror) printz("%s\n",gerror->message);
   snprintf(file_errmess,999,"(pixbuf) file write error: %s",file);
   printz("%s\n",file_errmess);
   return 2;
}


/**************************************************************************
   RAW file read functions (there are no write functions)
***************************************************************************/

//  Load PXB (pixbuf, 8 bit color) from RAW file using DCraw program
//  f_load_bpc is set to 16

PXB * RAW_PXB_load(cchar *rawfile)
{
   PXB      *pxb;
   int      handle, err;
   char     *pp, rawcommand[200];
   
   strcpy(rawcommand,DCrawcommand);                                        //  change default DCraw command to
   pp = strstr(rawcommand,"-6 ");                                          //    output tiff-8 instead of tiff-16
   if (pp) strcpy(pp, DCrawcommand + (pp+3-rawcommand));

   handle = shell_asynch("%s -c \"%s\" >temp.tiff",rawcommand,rawfile);    //  convert RAW to tiff-8 file
                           
   while (true) {
      err = shell_asynch_status(handle);
      if (err != -1) break;
      zsleep(0.1);
      zmainloop();
   }

   pxb = TIFF_PXB_load("temp.tiff");                                       //  load the tiff file
   f_load_bpc = 16;                                                        //  RAW file bpc
   remove("temp.tiff");                                                    //  delete the tiff file
   return pxb;
}


//  Load PXM (float color) from RAW file using DCraw program
//  f_load_bpc is set to 16

PXM * RAW_PXM_load(cchar *rawfile)
{
   PXM      *pxm;
   int      handle, err;
   
   handle = shell_asynch("%s -c \"%s\" >temp.tiff",DCrawcommand,rawfile);  //  convert RAW to tiff-16 file
   while (true) {
      err = shell_asynch_status(handle);
      if (err != -1) break;
      zsleep(0.1);
      zmainloop();
   }

   pxm = TIFF_PXM_load("temp.tiff");                                       //  load the tiff file
   f_load_bpc = 16;
   remove("temp.tiff");                                                    //  delete the tiff file
   return pxm;
}


/**************************************************************************
   directory navigation and thumbnail gallery functions
***************************************************************************/

namespace navi 
{
   #define thumbnail_cachesize 10000                                       //  max thumbnails cached in memory
   #define TEXTWIN GTK_TEXT_WINDOW_TEXT                                    //  GDK window of GTK text view
   #define NEVER GTK_POLICY_NEVER
   #define ALWAYS GTK_POLICY_ALWAYS
                                                               
   #define     thumbxx 6                                                   //  thumbx array size
   int         thumbx[6] = { 512, 360, 256, 180, 128, 90 };                //  thumbnail sizes

   typedef struct {                                                        //  current gallery list in memory
      char     *file;                                                      //  /directory.../filename
      char     fdate[16];                                                  //  file date: yyyymmddhhmmss
      char     pdate[16];                                                  //  photo date: yyyy:mm:dd
      char     size[16];                                                   //  image size: 2345x1234
   }  glist_t;

   glist_t     *glist;

   //  gallery type: 1-6 = directory, search results, search metadata results, 
   //                                 collection, recent images, newest images
   int         gallerytype = 0;
   char        *galleryname = 0;                                           //  image directory or file list name
   int         gallerysort = 1;                                            //  1/2/3 = filename/filedate/photodate
   int         galleryseq = 1;                                             //  1/2 = ascending/descending
   int         gallerypainted = 0;                                         //  gallery is finished painting
   int         nfiles = 0;                                                 //  gallery file count (incl. subdirks)
   int         nimages = 0;                                                //  gallery image file count
   char        **mdlist = 0;                                               //  corresp. metadata list
   int         mdrows = 0;                                                 //  text rows in metadata list
   int         xwinW = 1000, xwinH = 700;                                  //  gallery window initial size
   int         xwinX, xwinY;                                               //  gallery window initial position
   int         thumbsize = 128;                                            //  initial thumbnail image size
   int         thumbW, thumbH;                                             //  gallery window thumbnail cell size
   int         fontsize = 9;                                               //  font size for text in gallery window
   int         xrows, xcols;                                               //  gallery window thumbnail rows, cols
   int         margin = 5;                                                 //  cell margin from left and top edge
   int         texthh;
   int         genthumbs = 0;                                              //  count newly generated thumbnails
   int         scrollp = 0;                                                //  scroll position
   int         maxscroll;                                                  //  max. scroll position
   int         topfileposn = 0;                                            //  scroll-to file position (Nth)
   int         Fslowscroll = 0;                                            //  gallery slow scroll-down engaged

   //  private functions
   char * navigate(cchar *filez, cchar *action, int Nth = 0);              //  image file list setup and navigate
   int    gallery_comp(cchar *rec1, cchar *rec2);                          //  gallery record compare for sort options
   int    gallery_paint(GtkWidget *, cairo_t *);                           //  gallery window paint function
   int    gallery_paintmeta(GtkWidget *Gdrawin, cairo_t *cr);              //  same, for metadata report
   void   draw_text(cairo_t *cr, char *text, int x, int y, int ww);        //  draw text in gallery window
   void   menufuncx(GtkWidget *win, cchar *menu);                          //  function for gallery window buttons
   void   changedirk(GtkWidget *widget, GdkEventButton *event);            //  function for gallery directory change
   void   newtop(GtkWidget *widget, GdkEventButton *event);                //  function for top image directory change
   int    newtop_dialog_event(zdialog *zd, cchar *event);                  //   "" dialog event function
   void   getcoll();                                                       //  function to get gallery from a collection
   int    getcoll_dialog_event(zdialog *zd, cchar *event);                 //   "" dialog event function
   void   gallery_sort();                                                  //  choose gallery sort order
   void   mouse_event(GtkWidget *widget, GdkEvent *event, void *);         //  gallery window mouse event function
   int    KBrelease(GtkWidget *, GdkEventKey *, void *);                   //  gallery window key release event
}


/**************************************************************************

   public function to create/update image gallery (thumbnail window)

   Make a scrolling window of thumbnails for a list of files
   Handle window buttons (up row, down page, open file, etc.)
   Call external functions in response to thumbnail mouse-clicks.

   char * gallery(cchar *filez, cchar *action, int Nth)

   filez: image file or directory, or file with list of image files

   action:  init:     filez = initial file or directory 
            initF:    filez = file with list of image files to use
            sort:     sort the file list, directories first, ignore case
            insert:   insert filez into file list at position Nth (0 to last+1)
            delete:   delete Nth file in list
            find:     return Nth file (0 base) or null if Nth > last
            get1st:   return 1st image file or null if none
            paint:    paint gallery window, filez in top row, Nth position if match
                      (if filez null, no change in rop row)
   
   Nth: file position in gallery (action = insert/delete/find/paint) (0 base)
   (Nth is optional argument, default = 0)

   thumbnail click functions: 
      gallery_Lclick_func()                     default function (open file)
      gallery_Rclick_popup()                    default function (popup menu)
      gallery_getfiles_Lclick_func()            gallery_getfiles active
      gallery_getfiles_Rclick_func()            gallery_getfiles active
      bookmarks_Lclick_func                     edit bookmarks active
      passes Nth of clicked thumbnail (0 to last)
      passes -1 if gallery window is closed
      passes -2 if key F1 is pressed (for context help)
      
   Returned values:
      find: filespec,    others: null
      The returned file belongs to caller and is subject for zfree().

***************************************************************************/

char * gallery(cchar *filez, cchar *action, int Nth)                       //  overhauled
{
   using namespace navi;
   
   int      nth = Nth;
   
   zthreadcrash();                                                         //  thread usage not allowed
   
   if (strstr("init initF",action)) {                                      //  generate new gallery file list
      gallerypainted = 0;                                                  //    from directory/file
      navigate(filez,action);
      return 0;
   }   

   if (strstr("sort insert delete",action)) {                              //  revise gallery file list
      navigate(filez,action,Nth);
      gallerypainted = 0;
      return 0;
   }

   if (strEqu(action,"find"))                                              //  find Nth file in file list
      return navigate(0,action,Nth);
   
   if (strEqu(action,"get1st"))                                            //  get 1st image file in file list
      return navigate(0,action);

   if (strEqu(action,"paint")) {                                           //  paint gallery window
      gallerypainted = 0;
      if (filez) nth = gallery_position(filez,Nth);                        //  use Nth if valid, else find Nth    v.14.10
      topfileposn = nth;                                                   //  filez or -1 or caller Nth
      gtk_widget_queue_draw(Gdrawin);                                      //  draw gallery window
      return 0;
   }
   
   zappcrash("gallery() action: %s",action);                               //  bad call
   return 0;
}


/**************************************************************************

   private function - manage list of image files within a directory

   char * navi::navigate(cchar *filez, cchar *action, int Nth)

   action:  init:    file list = directories and image files in directory filez
            initF:   file list = filez = list of image files to use (cchar **)
            sort:    sort the file list, directories first, ignore case
            insert:  insert filez into file list at position Nth (0 to last+1)
            delete:  delete Nth file in list
            find:    returns Nth file or null if Nth > last
            get1st:  return 1st image file or null if none

   Nth: file to find/insert/delete (0 base)
   (used for initF galleries: collection or search results)
   
   Returned values:
      find: filespec, else null
      The returned file belongs to caller and is subject for zfree().

***************************************************************************/

char * navi::navigate(cchar *filez, cchar *action, int Nth)
{
   using namespace navi;

   char     *buff, *findname;
   cchar    *findcommand = "find -L \"%s\" -maxdepth 1";                   //  -L added
   char     *pp, *file2, fdate[16], pdate[16], size[16];
   int      err, ii, cc, ftyp, contx = 0, fposn;
   FILE     *fid;
   STATB    statbuf;

   if (! strstr("init initF sort insert delete find get1st",action))
         zappcrash("navigate %s",action);
   
   if (strnEqu(action,"init",4))                                           //  init or initF
   {
      if (! filez || *filez == 0) return 0;                                //  should not happen

      if (glist) {
         for (ii = 0; ii < nfiles; ii++)                                   //  free prior gallery list
            zfree(glist[ii].file);
         zfree(glist);
         glist = 0;
      }

      if (mdlist) {                                                        //  clear prior metadata if any
         for (ii = 0; ii < nfiles; ii++) 
            if (mdlist[ii]) zfree(mdlist[ii]);
         zfree(mdlist);
         mdlist = 0; 
      }

      cc = maximages * sizeof(glist_t);      
      glist = (glist_t *) zmalloc(cc);                                     //  gallery file list 

      nfiles = nimages = 0;                                                //  no files
      fposn = 0;

      if (galleryname && filez != galleryname)                             //  don't destroy input
         zfree(galleryname);
      galleryname = zstrdup(filez);
   }

   if (strEqu(action,"init"))                                              //  initialize from given directory
   {
      gallerytype = 1;                                                     //  gallery type = directory
      
      err = stat(galleryname,&statbuf);
      if (err) {
         pp = (char *) strrchr(galleryname,'/');                           //  bad file, check directory part  
         if (! pp) return 0;
         pp[1] = 0;
         err = stat(galleryname,&statbuf);
         if (err) return 0;                                                //  give up, empty file list
      }

      if (S_ISREG(statbuf.st_mode)) {                                      //  if a file, get directory part
         pp = (char *) strrchr(galleryname,'/');
         if (! pp) return 0;
         pp[1] = 0;
      }
      
      if (strchr(galleryname,'"')) {                                       //  if galleryname has embedded quotes,
         cc = strlen(galleryname);                                         //    they must be escaped for "find"
         findname = (char *) zmalloc(cc+40);
         repl_1str(galleryname,findname,"\"","\\\"");
      }
      else findname = zstrdup(galleryname);

      while ((buff = command_output(contx,findcommand,findname)))          //  find all files
      {
         if (strEqu(buff,galleryname)) {                                   //  skip self directory
            zfree(buff);
            continue;
         }
         
         if (! Fshowhidden && strstr(buff,"/.")) {                         //  skip hidden directory              v.14.02
            zfree(buff);                                                   //    unless "show" option             v.14.06
            continue;
         }

         if (nfiles == maximages) {
            zmessageACK(0,0,Btoomanyfiles,maximages);
            break;
         }

         ftyp = image_file_type(buff);

         if (ftyp == 1) {                                                  //  subdirectory
            glist[nfiles].file = buff;                                     //  add to file list
            glist[nfiles].file[0] = '!';                                   //  if directory, make it sort first
            glist[nfiles].fdate[0] = 0;                                    //  no file date
            glist[nfiles].pdate[0] = 0;                                    //  no photo date
            nfiles++;
         }

         else if (ftyp == 2 || ftyp == 3) {                                //  supported image or RAW file
            err = stat(buff,&statbuf);
            if (err) continue;
            glist[nfiles].file = buff;                                     //  add to file list
            get_sxrec_min(buff,fdate,pdate,size);                          //  file date, photo date, size
            strcpy(glist[nfiles].fdate,fdate);
            strcpy(glist[nfiles].pdate,pdate);
            nfiles++;
            nimages++; 
         }

         else {
            zfree(buff);                                                   //  (thumbnails not ftyp 1)
            continue;
         }
      }
      
      zfree(findname);
      
      if (nfiles > 1)                                                      //  sort the glist records
         HeapSort((char *) glist, sizeof(glist_t), nfiles, gallery_comp);
      
      gallery_monitor("start");                                            //  monitor gallery file changes
      curr_file_count = nimages;                                           //  gallery image file count
      return 0;
   }

   if (strEqu(action,"initF"))                                             //  initialize from given list
   {
      if (gallerytype < 2) zappcrash("gallerytype %d",gallerytype);        //  gallery type from caller
      
      fid = fopen(galleryname,"r");                                        //  open file
      if (! fid) return 0;

      buff = (char *) zmalloc(maxfcc);

      while (true)                                                         //  read list of files
      {
         pp = fgets_trim(buff,maxfcc-1,fid,1);
         if (! pp) break;
         err = stat(pp,&statbuf);                                          //  check file exists
         if (err) continue;
         glist[nfiles].file = zstrdup(pp);                                 //  add to file list
         get_sxrec_min(pp,fdate,pdate,size);                               //  file date, photo date, size
         strcpy(glist[nfiles].fdate,fdate);
         strcpy(glist[nfiles].pdate,pdate);
         nfiles++;
         if (nfiles == maximages) {
            zmessageACK(0,0,Btoomanyfiles,maximages);
            break;
         }
      }
      
      fclose(fid);
      zfree(buff);

      nimages = nfiles;
      curr_file_count = nimages;                                           //  gallery image file count
      
      if (curr_file) {
         Nth = gallery_position(curr_file,0);                              //  current file in the new gallery?   v.14.10
         curr_file_posn = Nth;                                             //  set curr. file posn. or -1 if not
      }

      return 0;
   }
   
   if (strEqu(action,"sort"))                                              //  sort the file list from init
   {
      if (nfiles < 2) return 0;
      HeapSort((char *) glist, sizeof(glist_t), nfiles, gallery_comp);     //  sort the glist records
      return 0;
   }

   if (strEqu(action,"insert"))                                            //  insert new file into list
   {
      if (gallerytype == 3) return 0;                                      //  metadata report
      fposn = Nth;                                                         //  file position from caller
      if (fposn < 0) fposn = 0;                                            //  limit to allowed range
      if (fposn > nfiles) fposn = nfiles;

      if (nfiles == maximages-1) {                                         //  no room
         zmessageACK(0,0,Btoomanyfiles,maximages);
         return 0;
      }

      for (ii = nfiles; ii > fposn; ii--)                                  //  create hole is list
         glist[ii] = glist[ii-1];

      glist[fposn].file = zstrdup(filez);                                  //  put new file in hole
      get_sxrec_min(filez,fdate,pdate,size);                               //  file date, photo date, size
      strcpy(glist[nfiles].fdate,fdate);
      strcpy(glist[nfiles].pdate,pdate);
      nfiles++;
      nimages++;                                                           //  bugfix
   }

   if (strEqu(action,"delete"))                                            //  delete file from list
   {
      fposn = Nth;                                                         //  file position from caller must be OK
      if (fposn < 0 || fposn > nfiles-1) return 0;
      nfiles--;
      if (*glist[fposn].file != '!') nimages--;                            //  not a directory, reduce image count
      zfree(glist[fposn].file);                                            //  remove glist record
      for (ii = fposn; ii < nfiles; ii++) {                                //  close the hole
         if (nfiles < 0) printz("meaningless reference %d",ii);            //  stop g++ optimization bug    ////
         glist[ii] = glist[ii+1];
      }
      if (mdlist) {
         if (mdlist[fposn]) zfree(mdlist[fposn]);                          //  delete corresp. metadata
         for (ii = fposn; ii < nfiles; ii++) {                             //  close the hole
            if (nfiles < 0) printz("meaningless reference %d",ii);         //  stop g++ optimization bug    ////
            mdlist[ii] = mdlist[ii+1];
         }
      }
   }

   if (strEqu(action,"find"))                                              //  return Nth file in gallery
   {
      fposn = Nth;                                                         //  file position from caller must be OK
      if (fposn < 0 || fposn > nfiles-1) return 0;
      file2 = zstrdup(glist[fposn].file);                                  //  get Nth file
      file2[0] = '/';                                                      //  restore initial '/'
      err = stat(file2,&statbuf);
      if (! err) return file2;
      zfree(file2);
   }

   if (strEqu(action,"get1st"))                                            //  return 1st image file in gallery
   {
      for (Nth = 0; Nth < nfiles; Nth++) 
      {
         if (glist[Nth].file[0] == '!') continue;                          //  subdirectory
         file2 = zstrdup(glist[Nth].file);                                  //  get Nth file
         err = stat(file2,&statbuf);
         if (! err) return file2;
         zfree(file2);
      }
      return 0;
   }
   
   return 0;
}


//  private function for special file name compare
//  directories sort first and upper/lower case is ignored

int navi::gallery_comp(cchar *rec1, cchar *rec2)
{
   int      nn;
   glist_t  *grec1, *grec2;
   
   if (galleryseq == 1) {                                                  //  ascending
      grec1 = (glist_t *) rec1;
      grec2 = (glist_t *) rec2;
   }
   else {                                                                  //  descending
      grec1 = (glist_t *) rec2;
      grec2 = (glist_t *) rec1;
   }
   
   switch (gallerysort) {
   
      case 1: {                                                            //  file name
         nn = strcasecmp(grec1->file,grec2->file);
         if (nn != 0) return nn;
         nn = strcmp(grec1->file,grec2->file);                             //  if equal, use utf8 compare
         return nn;
      }

      case 2: {                                                            //  file mod date/time
         nn = strcmp(grec1->fdate,grec2->fdate);
         return nn;
      }

      case 3: {                                                            //  photo date/time
         nn = strcmp(grec1->pdate,grec2->pdate);                           //  (EXIF DateTimeOriginal)
         return nn;
      }
      
      default: return 0;
   }
}


//  private function
//  paint gallery window - draw all thumbnail images that can fit

int navi::gallery_paint(GtkWidget *drwin, cairo_t *cr)
{
   using namespace navi;

   GdkPixbuf   *pxbT;
   double      x1, y1, x2, y2;
   int         ii, nrows, row, col;
   int         row1, row2, ftyp, ww, hh;
   int         drwingW, drwingH;
   int         thumx, thumy;
   char        *pp, *fname, p0;
   char        text[200], fdate[16], pdate[16], size[16];
   
   gallerypainted = 0;
   
   if (! galleryname) {
      if (! curr_dirk) return 1;
      gallery(curr_dirk,"init");
   }
   
   set_gwin_title();                                                       //  main window title = gallery name
   gtk_entry_set_text(GTK_ENTRY(Gpath),galleryname);                       //  set current directory label

   if (gallerytype == 3) {                                                 //  metadata report
      gallery_paintmeta(drwin,cr);
      return 1;
   }
   
   fontsize = 7 + thumbsize / 128;                                         //  font size from 7 to 10
   if (! thumbsize) fontsize = 9;                                          //  text only view
   texthh = 3.5 * fontsize + 4;                                            //  two text lines
   thumbW = thumbsize + 10;                                                //  thumbnail cell size
   thumbH = thumbsize + texthh + thumbsize/24 + 10;

   if (! thumbsize) {
      thumbW = 400;                                                        //  zero, list view
      thumbH = 40;
   }

   xwinW = gtk_widget_get_allocated_width(Gscroll);                        //  drawing window size
   xwinH = gtk_widget_get_allocated_height(Gscroll);

   xrows = int(0.1 + 1.0 * xwinH / thumbH);                                //  get thumbnail rows and cols that
   xcols = int(0.1 + 1.0 * xwinW / thumbW);                                //    (almost) fit in window 
   if (xrows < 1) xrows = 1;
   if (xcols < 1) xcols = 1;
   nrows = (nfiles+xcols-1) / xcols;                                       //  thumbnail rows, 1 or more
   if (nrows < 1) nrows = 1;

   drwingW = xcols * thumbW + margin + 10;                                 //  layout size for entire gallery
   drwingH = (nrows + 1) * thumbH;                                         //  last row
   if (drwingH < xwinH) drwingH = xwinH;                                   //  at least current size
   
   gtk_widget_get_size_request(drwin,&ww,&hh);                             //  current size
   if (ww != drwingW || hh != drwingH)
      gtk_widget_set_size_request(drwin,-1,drwingH);                       //  needs to change

   maxscroll = nrows * thumbH;                                             //  too far but necessary              v.14.10
   if (maxscroll < xwinH) maxscroll = xwinH;                               //  compensate GTK bug                 v.14.10

   gtk_adjustment_set_step_increment(Gadjust,thumbH);                      //  scrollbar works in row steps
   gtk_adjustment_set_page_increment(Gadjust,thumbH * xrows);              //  and in page steps
   
   if (topfileposn >= 0) {                                                 //  new target file position (Nth)
      scrollp = topfileposn / xcols * thumbH;                              //  scroll position for target file
      if (scrollp > maxscroll) scrollp = maxscroll;                        //    in top row of window
      gtk_adjustment_set_upper(Gadjust,maxscroll);                         //  v.14.10
      gtk_adjustment_set_value(Gadjust,scrollp);                           //  will cause re-entrance
      gtk_widget_queue_draw(drwin);
      topfileposn = -1;                                                    //  keep scroll position next time
      return 1;
   }
   
   else {
      cairo_clip_extents(cr,&x1,&y1,&x2,&y2);                              //  window region to paint
      row1 = y1 / thumbH;
      row2 = y2 / thumbH;
   }
   
   for (row = row1; row <= row2; row++)                                    //  draw file thumbnails
   {
      for (col = 0; col < xcols; col++)                                    //  draw all columns in row
      {
         ii = row * xcols + col;                                           //  next file
         if (ii >= nfiles) goto endloops;                                  //  exit 2 nested loops

         p0 = *glist[ii].file;                                             //  replace possible ! with /
         *glist[ii].file = '/';

         fname = glist[ii].file;                                           //  filespec
         pp = strrchr(fname,'/');                                          //  get file name only
         if (pp) fname = pp + 1;

         thumx = col * thumbW + margin;                                    //  upper left corner in drawing area
         thumy = row * thumbH + margin;

         if (curr_file && strEqu(glist[ii].file,curr_file)) {              //  yellow background for curr. image  v.14.07
            cairo_set_source_rgb(cr,1,1,0.5);
            cairo_rectangle(cr,thumx-3,thumy-3,thumbW-3,texthh);
            cairo_fill(cr);
         }

         ftyp = image_file_type(glist[ii].file);
         if (ftyp == 2 || ftyp == 3) {                                     //  image or RAW file
            get_sxrec_min(glist[ii].file,fdate,pdate,size);                //  filename, file/photo date, size
            if (gallerysort == 2) pp = fdate;
            else pp = pdate;                                               //  use file or photo date based on sort
            if (strNeq(pp,"undated")) {
               memcpy(pp+8,pp+6,2);
               memcpy(pp+5,pp+4,2);                                        //  convert yyyymmdd to yyyy-mm-dd
               pp[4] = pp[7] = '-';
               pp[10] = 0;
            }
            snprintf(text,200,"%s\n%-10s  %s",fname,pp,size);
            draw_text(cr,text,thumx,thumy,thumbW-5);
            thumy += texthh;                                               //  position below text
         }

         else {                                                            //  other file type
            snprintf(text,200,"%s",fname);
            draw_text(cr,text,thumx,thumy,thumbW-5);
            thumy += texthh / 2;                                           //  position below text
         }

         if (thumbsize) {                                                  //  zero >> list view
            pxbT = image_thumbnail(glist[ii].file,thumbsize);              //  get thumbnail
            if (pxbT) {
               gdk_cairo_set_source_pixbuf(cr,pxbT,thumx,thumy);
               cairo_paint(cr);
               g_object_unref(pxbT);
            }
         }

         *glist[ii].file = p0;                                             //  restore !
      }
   }

   endloops:
   gallerypainted = 1;
   return 1;
}


//  private function
//  paint metadata report - draw thumbnail images + metadata

int navi::gallery_paintmeta(GtkWidget *drwin, cairo_t *cr)
{
   using namespace navi;

   GdkPixbuf      *pxbT;
   double      x1, y1, x2, y2;
   int         ii, nrows, row, col;
   int         row1, row2, ww, hh;
   int         drwingW, drwingH;
   int         thumx, thumy, textww;
   char        p0;
   
   if (thumbsize < 128) thumbsize = 128;

   fontsize = 7 + thumbsize / 128;                                         //  font size from 7 to 10

   thumbW = thumbsize + 10;                                                //  thumbnail layout size
   thumbH = thumbsize + 20;

   texthh = mdrows * fontsize * 1.8 + 20;                                  //  space for metadata text
   if (texthh > thumbH) thumbH = texthh;

   xwinW = gtk_widget_get_allocated_width(Gscroll);                        //  drawing window size
   xwinH = gtk_widget_get_allocated_height(Gscroll);

   xrows = int(0.1 + 1.0 * xwinH / thumbH);                                //  get thumbnail rows fitting in window
   if (xrows < 1) xrows = 1;
   xcols = 1;                                                              //  force cols = 1
   nrows = nfiles;                                                         //  thumbnail rows

   drwingW = xwinW;                                                        //  layout size for entire file list
   if (drwingW < 800) drwingW = 800;
   drwingH = (nrows + 1) * thumbH;                                         //  last row
   if (drwingH < xwinH) drwingH = xwinH;

   gtk_widget_get_size_request(drwin,&ww,&hh);                             //  current size
   if (ww != drwingW || hh != drwingH)
      gtk_widget_set_size_request(drwin,-1,drwingH);                       //  needs to change

   maxscroll = nrows * thumbH;                                             //  too far but necessary              v.14.10
   if (maxscroll < xwinH) maxscroll = xwinH;                               //  compensate GTK bug                 v.14.10

   gtk_adjustment_set_step_increment(Gadjust,thumbH);                      //  scrollbar works in row steps
   gtk_adjustment_set_page_increment(Gadjust,thumbH * xrows);              //  and in page steps

   if (topfileposn >= 0) {                                                 //  new target file position (Nth)
      scrollp = topfileposn / xcols * thumbH;                              //  scroll position for target file
      if (scrollp > maxscroll) scrollp = maxscroll;                        //    in top row of window
      gtk_adjustment_set_upper(Gadjust,maxscroll);                         //  v.14.10
      gtk_adjustment_set_value(Gadjust,scrollp);                           //  will cause re-entrance
      gtk_widget_queue_draw(drwin);
      topfileposn = -1;                                                    //  keep scroll position next time
      return 1;
   }
   
   else {
      cairo_clip_extents(cr,&x1,&y1,&x2,&y2);                              //  window region to paint
      row1 = y1 / thumbH;
      row2 = y2 / thumbH;
   }

   textww = drwingW - thumbW - 2 * margin;                                 //  space for text right of thumbnail

   for (row = row1; row <= row2; row++)                                    //  draw file thumbnails
   {
      for (col = 0; col < xcols; col++)
      {
         ii = row * xcols + col;                                           //  next file
         if (ii >= nfiles) goto endloops;                                  //  exit 2 nested loops

         p0 = *glist[ii].file;                                             //  replace possible ! with /
         *glist[ii].file = '/';

         thumx = col * thumbW + margin;                                    //  upper left corner in window space
         thumy = row * thumbH + margin;
         
         pxbT = image_thumbnail(glist[ii].file,thumbsize);                 //  get thumbnail
         if (pxbT) {
            gdk_cairo_set_source_pixbuf(cr,pxbT,thumx,thumy);
            cairo_paint(cr);
            g_object_unref(pxbT);
         }

         draw_text(cr,glist[ii].file,thumbW+margin,thumy,textww);          //  write filespec to right of thumbnail
         
         if (mdlist && mdlist[ii])                                         //  write metadata if present
            draw_text(cr,mdlist[ii],thumbW+margin,thumy+20,textww);
         
         *glist[ii].file = p0;                                             //  restore !
      }
   }

   endloops:
   gallerypainted = 1;
   return 1;
}


//  private function
//  write text for thumbnail limited by width of thumbnail

void navi::draw_text(cairo_t *cr, char *text, int px, int py, int ww)
{
   using namespace navi;

   static PangoFontDescription   *pfont = 0;
   static PangoLayout            *playout = 0;
   
   static int     pfontsize = -1;
   static char    thumbfont[12] = "";
   
   if (fontsize != pfontsize) {                                            //  adjust for curr. font size
      pfontsize = fontsize;
      snprintf(thumbfont,12,"sans %d",fontsize);
      if (pfont) pango_font_description_free(pfont);                       //  free memory                        v.14.07
      pfont = pango_font_description_from_string(thumbfont);
      if (playout) g_object_unref(playout);                                //  free memory                        v.14.07
      playout = pango_cairo_create_layout(cr);
      pango_layout_set_font_description(playout,pfont);
   }
   
   pango_layout_set_width(playout,ww*PANGO_SCALE);                         //  limit width to avail. space
   pango_layout_set_ellipsize(playout,PANGO_ELLIPSIZE_END);
   pango_layout_set_text(playout,text,-1);

   cairo_move_to(cr,px,py);
   cairo_set_source_rgb(cr,0,0,0);
   pango_cairo_show_layout(cr,playout);
   return;
}


//  private function - menu function for gallery window
//    - scroll window as requested
//    - jump to new file or folder as requested

void navi::menufuncx(GtkWidget *, cchar *menu)
{
   using namespace navi;

   int         ii, scroll1, scroll2;
   int         step = thumbH / 8;                                          //  row-scroll step size
   char        *file;
   
   if (strNeq(menu,ZTX("Scroll"))) Fslowscroll = 0;                        //  v.14.04
   
   if (strEqu(menu,ZTX("Sync.G")) || strEqu(menu,ZTX("Sync Gallery")))     //  gallery <-- directory of curr. file
   {
      if (! curr_file) return;
      m_viewmode(0,"G"); 
      gallery(curr_file,"init");                                           //  v.14.10
      gallery(curr_file,"paint");
      return;
   }

   if (! gallerypainted) return;                                           //  wait for pending paint

   if (strEqu(menu,ZTX("Open"))) {                                         //  change directory
      file = zgetfile(ZTX("change directory"),"folder",curr_dirk);
      if (! file) return;
      gallery(file,"init");
      gallery(0,"paint",0);
      zfree(file);
      return;
   }
   
   if (strEqu(menu,ZTX("GoTo"))) {
      m_goto_bookmark(0,0);
      return;
   }

   if (strEqu(menu,ZTX("Sort"))) {                                         //  choose gallery sort order 
      gallery_sort();
      return;
   }

   scrollp = gtk_adjustment_get_value(Gadjust);                            //  current scroll position

   if (strEqu(menu,ZTX("Zoom+")))  {                                       //  next bigger thumbnail size
      for (ii = 0; ii < thumbxx; ii++) 
         if (thumbsize == thumbx[ii]) break;
      if (ii == 0) return;
      thumbsize = thumbx[ii-1];
      topfileposn = scrollp / thumbH * xcols;                              //  keep top row position
      gtk_widget_queue_draw(Gdrawin);
      return;
   }

   if (strEqu(menu,ZTX("Zoom-")))  {                                       //  next smaller
      for (ii = 0; ii < thumbxx; ii++) 
         if (thumbsize == thumbx[ii]) break;
      if (ii >= thumbxx-1) thumbsize = 0;                                  //  no thumbs, list view
      else  thumbsize = thumbx[ii+1];
      if (thumbsize < 128 && gallerytype == 3)                             //  min. for metadata report
         thumbsize = 128;
      topfileposn = scrollp / thumbH * xcols;                              //  keep top row position
      gtk_widget_queue_draw(Gdrawin);
      return;
   }
   
   if (strEqu(menu,ZTX("Row↑"))) {                                        //  scroll 1 row in small steps
      scroll1 = scrollp / thumbH * thumbH;                                 //  faster/smoother
      scroll2 = scroll1 - thumbH;
      if (scroll2 < 0) scroll2 = 0;
      for (scrollp = scroll1; scrollp >= scroll2; scrollp -= step) {
         if (scrollp - scroll2 < step) scrollp = scroll2;
         gtk_adjustment_set_value(Gadjust,scrollp);
         zsleep(0.01);                                                     //  v.14.08
         zmainloop();
      }
      return;
   }
      
   if (strEqu(menu,ZTX("Row↓"))) {
      scroll1 = (scrollp+thumbH/3) / thumbH * thumbH;
      scroll2 = scroll1 + thumbH;
      if (scroll2 > maxscroll) scroll2 = maxscroll;
      for (scrollp = scroll1; scrollp <= scroll2; scrollp += step) {
         if (scroll2 - scrollp < step) scrollp = scroll2;
         gtk_adjustment_set_value(Gadjust,scrollp);
         zsleep(0.01);                                                     //  v.14.08
         zmainloop();
      }
      return;
   }
   
   if (strEqu(menu,ZTX("Scroll"))) {                                       //  start/stop slow scroll down        v.14.04
      Fslowscroll = 1 - Fslowscroll;
      while (Fslowscroll) {
         scroll1 = (scrollp+thumbH/3) / thumbH * thumbH;
         scroll2 = scroll1 + thumbH;
         if (scroll2 > maxscroll) scroll2 = maxscroll;
         for (scrollp = scroll1; scrollp <= scroll2; scrollp++) {
            gtk_adjustment_set_value(Gadjust,scrollp);
            for (ii = 0; ii < 4; ii++) {
               g_usleep(500);                                              //  v.14.10
               zmainloop();
            }
            if (! Fslowscroll) break;
         }
         if (scrollp >= maxscroll) Fslowscroll = 0;
         if (FGW != 'G') Fslowscroll = 0;                                  //  v.14.04.1
      }
      return;
   }
      
   if (strEqu(menu,ZTX("First"))) scrollp = 0;                             //  jump to page
   if (strEqu(menu,ZTX("Last"))) scrollp = maxscroll;
   if (strEqu(menu,ZTX("Page↑"))) scrollp -= thumbH * xrows;
   if (strEqu(menu,ZTX("Page↓"))) scrollp += thumbH * xrows;

   if (scrollp < 0) scrollp = 0;                                           //  enforce limits
   if (scrollp > maxscroll) scrollp = maxscroll;

   topfileposn = scrollp / thumbH * xcols;
   gtk_widget_queue_draw(Gdrawin);
   return;
}


//  private function for mouse clicks on gallery path widget
//  directory selection - get new directory from mouse-click position

void navi::changedirk(GtkWidget *widget, GdkEventButton *event)
{
   using namespace navi;

   int      cc, pos;
   char     *pp, directory[500];
   cchar    *ppc;

   Fslowscroll = 0;
   if (gallerytype > 1) return;                                            //  not a directory gallery
   
   ppc = gtk_entry_get_text(GTK_ENTRY(widget));                            //  current directory 
   if (! ppc) return;                                                      //  (ppc must not be freed)
   strncpy0(directory,ppc,500);
   cc = strlen(directory);
   if (cc < 2) return;
   pos = 1 + int(event->x);                                                //  mouse position within text, pixels
   pos = pos / monitorscale;                                               //  adjust for monitor scale           v.14.10
   pos = pos / panelfontchww;                                              //  mouse to character position        v.14.10
   if (pos >= cc) pos = cc - 1;
   if (pos < 2) return;
   pp = strchr(directory+pos,'/');                                         //  truncate following directories
   if (! pp) return;
   *pp = 0;
   gallery(directory,"init");                                              //  initialize new gallery
   gallery(0,"paint",0);                                                   //  paint new gallery 
   return;
}


//  private function - [Top] button: select new top directory

void navi::newtop(GtkWidget *widget, GdkEventButton *event)
{
   zdialog     *zd;
   char        dirk[200], *pp;

   Fslowscroll = 0;

   zd = zdialog_new(ZTX("Choose image directory"),Mwin,null);              //  popup dialog with top image directories
   zdialog_add_widget(zd,"combo","top","dialog");
   for (int ii = 0; ii < Ntopdirks; ii++)                                  //  insert all top image directories
      zdialog_cb_app(zd,"top",topdirks[ii]);
   zdialog_cb_app(zd,"top","/");                                           //  add "/" and "HOME"
   zdialog_cb_app(zd,"top","HOME");
   zdialog_cb_app(zd,"top",ZTX("recent"));                                 //  add "recent" and "newest"
   zdialog_cb_app(zd,"top",ZTX("newest"));
   zdialog_cb_app(zd,"top",ZTX("collections"));                            //  add "collections"
   zdialog_resize(zd,200,0);
   zdialog_run(zd,newtop_dialog_event,"mouse");                            //  run dialog, wait for response
   zdialog_cb_popup(zd,"top");
   zdialog_wait(zd);

   if (zd->zstat == 1) 
      zdialog_fetch(zd,"top",dirk,200);                                    //  get user choice
   else *dirk = 0;
   zdialog_free(zd);
   if (! *dirk) return;
   zmainloop();

   if (strEqu(dirk,ZTX("recent"))) { 
      m_recentfiles(0,0);
      return;
   }

   if (strEqu(dirk,ZTX("newest"))) {
      m_newfiles(0,0);
      return;
   }

   if (strEqu(dirk,"HOME"))                                                //  if HOME, get /home/<user> directory
      if ((pp = getenv("HOME")))
         if (pp) strncpy0(dirk,pp,200);

   if (strEqu(dirk,ZTX("collections"))) {                                  //  choose a collection 
      getcoll();
      return;
   }

   gallery(dirk,"init");
   gallery(0,"paint",0);                                                   //  paint new gallery
   return;
}

int navi::newtop_dialog_event(zdialog *zd, cchar *event)                   //  dialog event function
{
   if (strEqu(event,"top")) zd->zstat = 1;                                 //  finish dialog when choice is made
   else zd->zstat = 2;
   return 1;
}


//  private function
//  show a list of collections and choose a one which becomes current gallery
//  called by [Top] button menu in gallery view

void navi::getcoll()
{
   cchar          *findcomm = "find -L \"%s\" -type f";
   char           *collections[100], collname[100], collfile[200];
   char           *buff, *pp;
   int            ii, contx = 0, count = 0;
   zdialog        *zd;

   Fslowscroll = 0;

   while ((buff = command_output(contx,findcomm,collections_dirk)))        //  find all collection files
   {
      if (count > 99) {
         zmessageACK(Mwin,0,Btoomanyfiles,100);
         break;
      }
      
      pp = strrchr(buff,'/');
      if (! pp) continue;
      collections[count] = zstrdup(pp+1);
      zfree(buff); 
      count++;
   }
   
   if (buff) command_kill(contx);

   if (! count) {
      zmessageACK(Mwin,0,ZTX("no collections found"));
      return;
   }

   if (count > 1)                                                          //  sort collection names
      HeapSort(collections,count);
   
   zd = zdialog_new(ZTX("Choose collection"),Mwin,null);                   //  popup dialog with collections list
   zdialog_add_widget(zd,"combo","colls","dialog");

   for (ii = 0; ii < count; ii++)                                          //  insert collection file names
      zdialog_cb_app(zd,"colls",collections[ii]);

   zdialog_resize(zd,250,0);                                               //  v.14.10
   zdialog_run(zd,getcoll_dialog_event,"mouse");                           //  run dialog, wait for response
   zdialog_cb_popup(zd,"colls");
   zdialog_wait(zd);

   if (zd->zstat != 1) {
      zdialog_free(zd);
      for (ii = 0; ii < count; ii++) 
         zfree(collections[ii]);
      return;
   }

   zdialog_fetch(zd,"colls",collname,100);                                 //  get user choice
   zdialog_free(zd);
   for (ii = 0; ii < count; ii++) 
      zfree(collections[ii]);

   snprintf(collfile,200,"%s/%s",collections_dirk,collname);
   navi::gallerytype = 4;
   gallery(collfile,"initF");
   gallery(0,"paint",0);
   return;
}

int navi::getcoll_dialog_event(zdialog *zd, cchar *event)                  //  dialog event function
{
   if (strEqu(event,"colls")) zd->zstat = 1;                               //  finish dialog when choice is made
   else zd->zstat = 2;
   return 1;
}


//  private function
//  gallery sort by file name, file date, or photo date (exif)

void navi::gallery_sort()
{
   zdialog     *zd;
   int         nn;
   
   Fslowscroll = 0;

   zd = zdialog_new(ZTX("Gallery Sort"),Mwin,Bapply,null);                 //  user dialog
   zdialog_add_widget(zd,"hbox","hb1","dialog");
   zdialog_add_widget(zd,"label","space","hb1",0,"space=2");
   zdialog_add_widget(zd,"vbox","vb1","hb1");
   zdialog_add_widget(zd,"radio","filename","vb1",ZTX("File Name"));
   zdialog_add_widget(zd,"radio","filedate","vb1",ZTX("File Mod Date/Time"));
   zdialog_add_widget(zd,"radio","photodate","vb1",ZTX("Photo Date/Time (EXIF)"));
   zdialog_add_widget(zd,"hbox","hb2","dialog",0,"space=5");
   zdialog_add_widget(zd,"radio","ascending","hb2",ZTX("ascending"),"space=4");
   zdialog_add_widget(zd,"radio","descending","hb2",ZTX("descending"),"space=2");
   
   if (gallerysort == 1) zdialog_stuff(zd,"filename",1);                   //  stuff current sort order
   if (gallerysort == 2) zdialog_stuff(zd,"filedate",1);
   if (gallerysort == 3) zdialog_stuff(zd,"photodate",1);

   if (galleryseq == 1) zdialog_stuff(zd,"ascending",1);                   //  ascending/descending
   else zdialog_stuff(zd,"descending",1);

   zdialog_run(zd,0,"mouse");                                              //  run dialog, wait for completion
   zdialog_wait(zd);

   zdialog_fetch(zd,"filename",nn);                                        //  get user choice
   if (nn) gallerysort = 1;
   zdialog_fetch(zd,"filedate",nn);
   if (nn) gallerysort = 2;
   zdialog_fetch(zd,"photodate",nn);
   if (nn) gallerysort = 3;
   zdialog_fetch(zd,"ascending",nn);
   if (nn) galleryseq = 1;
   else galleryseq = 2;

   zdialog_free(zd);
   
   gallery(0,"sort");                                                      //  sort the gallery
   gallery(0,"paint",0);
   return;
}


//  private function
//  mouse event function for gallery window - get selected thumbnail and file
//  user function receives clicked file, which is subject for zfree()

void navi::mouse_event(GtkWidget *widget, GdkEvent *event, void *)         //  v.14.08
{
   using namespace navi;

   GdkEventButton *eventB;
   GdkPixbuf      *pxbT;
   static int     Bdown, Fdrag, bdNth, bdtime;
   int            evtype, evtime, mousex, mousey, mousebutt;
   int            row, col, nrows, tww, thh, top, bottom;
   int            Nth, poswidth, posheight, err;
   char           *filez;
   STATB          statb;
   
   if (! nfiles) return;                                                   //  empty gallery
   if (! gallerypainted) return;                                           //  not initialized
   
   evtype = event->type;
   evtime = ((GdkEventButton *) event)->time;
   mousex = int(((GdkEventButton *) event)->x);
   mousey = int(((GdkEventButton *) event)->y);
   mousebutt = ((GdkEventButton *) event)->button;
   if (mousex < margin) return;
   if (mousey < margin) return;
   
   row = (mousey - margin) / thumbH;                                       //  find selected row, col
   col = (mousex - margin) / thumbW;

   if (thumbsize) {                                                        //  bugfix                             v.14.07
      poswidth = (mousex - margin) - thumbW * col;                         //  mouse position within thumbnail
      poswidth = 100 * poswidth / thumbsize;                               //  0-100 = left to right edge
      posheight = (mousey - texthh - margin) - thumbH * row;
      posheight = 100 * posheight / thumbsize;                             //  0-100 = top to bottom edge         v.14.04
   }
   else poswidth = posheight = 0;

   if (! xcols) return;
   nrows = 1 + (nfiles-1) / xcols;                                         //  total thumbnail rows, 1 or more
   if (col < 0 || col >= xcols) return;                                    //  mouse not on a thumbnail
   if (row < 0 || row >= nrows) return;
   Nth = xcols * row + col;                                                //  mouse at this thumbnail (image file)
   if (Nth >= nfiles) return;
   
   if (evtype == GDK_MOTION_NOTIFY) {
      if (! Bdown) return;
      if (Fdrag) goto scroll;                                              //  continue collection drag
      if (bdNth < 0) return;
      if (evtime - bdtime < 500) return;
      coll_drag_start(bdNth);                                              //  start collection image drag
      Fdrag = 1;
      bdNth = -1;
      return;
   }

   if (evtype == GDK_BUTTON_PRESS)                                         //  v.14.08
   {
      Fslowscroll = 0;

      if (Fdrag) coll_drag_kill();
      Fdrag = 0;

      if (mousebutt == 1) {
         Bdown = 1;
         bdNth = Nth;                                                      //  image at mouse when button pressed
         bdtime = evtime;
      }
      else {
         Bdown = 0;
         bdNth = -1;
      }
      return;
   }
   
   if (evtype == GDK_BUTTON_RELEASE)
   {
      Fslowscroll = 0;
      Bdown = 0;

      if (Fdrag) {                                                         //  drag done, drop image here         v.14.08
         if (poswidth > 50) Nth++;
         coll_drag_drop(Nth);
         Fdrag = 0;
         bdNth = -1;
         return;
      }

      filez = zstrdup(glist[Nth].file);                                    //  file (thumbnail) at mouse position
      *filez = '/';

      err = stat(filez,&statb);
      if (err) {                                                           //  file is gone?
         zfree(filez);
         return;
      }
   
      if (S_ISDIR(statb.st_mode)) {                                        //  if directory, go there
         gallery(filez,"init");
         gallery(0,"paint",0);                                             //  paint new gallery
         zfree(filez);
         return;
      }
      
      if (clicked_file) zfree(clicked_file);                               //  save clicked file and gallery position
      clicked_file = filez;
      clicked_posn = Nth;
      clicked_width = poswidth;                                            //  normalized 0-100
      clicked_height = posheight;
      
      pxbT = image_thumbnail(filez,thumbsize);                             //  get thumbnail image
      if (pxbT) {
         tww = gdk_pixbuf_get_width(pxbT);                                 //  thumbnail width and height
         thh = gdk_pixbuf_get_height(pxbT);
         g_object_unref(pxbT);
         clicked_width = poswidth * thumbsize / tww;                       //  clicked position is relative       v.14.04
         clicked_height = posheight * thumbsize / thh;                     //    to thumbnail dimensions
         if (clicked_width < 0) clicked_width = 0;
         if (clicked_width > 100) clicked_width = 100;
         if (clicked_height < 0) clicked_height = 0;
         if (clicked_height > 100) clicked_height = 100;
      }

      eventB = (GdkEventButton *) event;                                   //  set KB key status 
      KBcontrolkey = KBshiftkey = KBaltkey = 0;
      if (eventB->state & GDK_CONTROL_MASK) KBcontrolkey = 1;
      if (eventB->state & GDK_SHIFT_MASK) KBshiftkey = 1;
      if (eventB->state & GDK_MOD1_MASK) KBaltkey = 1;

      if (mousebutt == 1) {                                                //  left click 
         if (zd_gallery_getfiles) gallery_getfiles_Lclick_func(Nth);       //  send to gallery_getfiles()
         else if (zd_edit_bookmarks) bookmarks_Lclick_func(Nth);           //  send to bookmarks editor
         else if (zd_ss_imageprefs) slideshow_Lclick_func(Nth);            //  send to slide show editor          v.14.04
         else gallery_Lclick_func(Nth);                                    //  open the file
         return;
      }
         
      if (mousebutt == 3) {                                                //  right click
         if (zd_gallery_getfiles) gallery_getfiles_Rclick_func(Nth);       //  send to gallery_getfiles()
         else gallery_Rclick_popup(Nth);                                   //  send to gallery thumbnail popup menu
         return;
      }
   }

scroll:                                                                    //  collection image drag and drop     v.14.08

   top = gtk_adjustment_get_value(Gadjust);                                //  drawing window range in visible window
   bottom = top + xwinH;

   while (mousey < top+20 && top > 0) {                                    //  mouse above range, scroll up
      top -= 30;
      if (top < 30) top = 0;
      gtk_adjustment_set_value(Gadjust,top);
   }

   if (mousey > bottom-20 && top < maxscroll) {                            //  mouse below range, scroll down
      top += 30;
      if (top > maxscroll-30) top = maxscroll;
      gtk_adjustment_set_value(Gadjust,top);
   }

   return;
}


//  Private function - respond to keyboard navigation keys.
//  KBrelease() for main window calls this function when G view is active.
//  key definitions: /usr/include/gtk-2.0/gdk/gdkkeysyms.h

int navi::KBrelease(GtkWidget *win, GdkEventKey *event, void *)
{
   using namespace navi;

   int      KBkey;
   
   KBkey = event->keyval;

   Fslowscroll = 0;

   if (KBkey == GDK_KEY_plus) menufuncx(win,ZTX("Zoom+"));                 //  +/- = bigger/smaller thumbnails
   if (KBkey == GDK_KEY_equal) menufuncx(win,ZTX("Zoom+"));
   if (KBkey == GDK_KEY_minus) menufuncx(win,ZTX("Zoom-"));
   if (KBkey == GDK_KEY_KP_Add) menufuncx(win,ZTX("Zoom+"));               //  keypad +/- also 
   if (KBkey == GDK_KEY_KP_Subtract) menufuncx(win,ZTX("Zoom-"));

   if (KBkey == GDK_KEY_Left) menufuncx(win,ZTX("Page↑"));                 //  left arrow = previous page
   if (KBkey == GDK_KEY_Right) menufuncx(win,ZTX("Page↓"));                //  right arrow = next page
   if (KBkey == GDK_KEY_Up) menufuncx(win,ZTX("Row↑"));                    //  up arrow = previous row
   if (KBkey == GDK_KEY_Down) menufuncx(win,ZTX("Row↓"));                  //  down arrow = next row
   
   if (KBkey == GDK_KEY_Home) menufuncx(win,ZTX("First"));                 //  keys added 
   if (KBkey == GDK_KEY_End) menufuncx(win,ZTX("Last"));
   if (KBkey == GDK_KEY_Page_Up) menufuncx(win,ZTX("Page↑"));
   if (KBkey == GDK_KEY_Page_Down) menufuncx(win,ZTX("Page↓"));
   
   return 1;
}


//  set the window title for the gallery window
//  window title = gallery name

void set_gwin_title()
{
   using namespace navi;

   char     *pp, title[200];
   int      gtype;

   if (FGW != 'G') return;
   
   gtype = navi::gallerytype;
   
   if (gtype == 1) 
      snprintf(title,200,"DIRECTORY   %s  %d files",galleryname,nfiles);

   else if (gtype == 2 || gtype == 3)
      snprintf(title,200,"SEARCH RESULTS   %d files",nimages);

   else if (gtype == 4) {
      pp = strrchr(navi::galleryname,'/');
      if (! pp) pp = navi::galleryname;
      else pp++;
      snprintf(title,200,"COLLECTION   %s  %d files",pp,nimages);
   }

   else if (gtype == 5)
      strcpy(title,"RECENT FILES");

   else if (gtype == 6)
      strcpy(title,"NEWEST FILES");

   else strcpy(title,"UNKNOWN");

   gtk_window_set_title(MWIN,title);
   return;
}


//  find function for use by f_preload_thread()

char * gallery_find(int Nth)
{
   using namespace navi;

   char     *file2;
   int      err, fposn;
   STATB    statbuf;
   
   fposn = Nth;                                                            //  file position from caller must be OK
   if (fposn < 0 || fposn > nfiles-1) return 0;
   file2 = zstrdup(glist[fposn].file);                                     //  get Nth file 
   file2[0] = '/';                                                         //  restore initial '/'
   err = stat(file2,&statbuf);
   if (! err) return file2;
   zfree(file2);
   return 0;
}


//  Get file position in file list.
//  If Nth position matches file, this is returned.
//  Otherwise the list is searched from position 0.
//  Position 0-last is returned if found, or -1 if not.

int gallery_position(cchar *file, int Nth)
{
   using namespace navi;
   
   int      ii;
   
   if (! nfiles) return -1;
   if (! file) return -1;

   if (Nth >= 0 && Nth < nfiles) ii = Nth;
   else ii = 0;
   
   if (strEqu(file+1,glist[ii].file+1)) return ii;                         //  file[0] may be !

   for (ii = 0; ii < nfiles; ii++)
      if (strEqu(file+1,glist[ii].file+1)) break;

   if (ii < nfiles) return ii;
   return -1;
}


//  Determine if a file is a directory or a supported image file type
//  Return: 0 = file not found
//          1 = directory
//          2 = image file
//          3 = RAW file
//          4 = thumbnail
//          5 = other

int image_file_type(cchar *file)
{
   using namespace navi;

   int         err, xcc, tcc;
   static int  ftf = 1, tdcc = 0;
   cchar       *ppx;
   char        ppx2[8], *ppt;
   STATB       statbuf;
   
   if (! file) return 0;
   err = stat(file,&statbuf);
   if (err) return 0;
   
   if (S_ISDIR(statbuf.st_mode)) return 1;                                 //  directory

   if (! S_ISREG(statbuf.st_mode)) return 5;                               //  not a regular file

   if (ftf) {                                                              //  v.14.02
      if (thumbdirk && *thumbdirk == '/')
         tdcc = strlen(thumbdirk);
      myRAWtypes = zstrdup(" ");
      ftf = 0;
   }

   if (tdcc && strnEqu(file,thumbdirk,tdcc)) return 4;                     //  fotoxx thumbnail                   v.14.02
   
   ppx = strrchr(file,'.');
   if (! ppx) return 5;                                                    //  no file .ext

   xcc = strlen(ppx);
   if (xcc > 5) return 5;                                                  //  file .ext > 5 chars.
   
   strcpy(ppx2,ppx);                                                       //  add trailing blank: ".ext "
   strcpy(ppx2+xcc," ");
   
   if (strcasestr(imagefiletypes,ppx2)) return 2;                          //  supported image type
   if (strcasestr(myRAWtypes,ppx2)) return 3;                              //  one of my RAW types

   if (strcasestr(RAWfiletypes,ppx2)) {                                    //  found in list of known RAW types
      tcc = strlen(myRAWtypes) + xcc + 2;
      ppt = (char *) zmalloc(tcc);                                         //  add to cache of my RAW types       v.14.07
      strcpy(ppt,ppx2);
      strcpy(ppt+xcc+1,myRAWtypes);
      zfree(myRAWtypes);
      myRAWtypes = ppt;
      return 3;
   }
   
   return 5;                                                               //  not a known image file type
}


//  Given a thumbnail file, get the corresponding image file.
//  Returns null if no image file found.
//  Returned file is subject for zfree().
//  image file:  /image/dirk/file.xxx                                      //  .jpg .png .tiff etc.
//  thumb dirk:  /thumb/dirk                                               //  if thumb dirk defined
//  thumb file:  /thumb/dirk/image/dirk/file.xxx.jpeg                      //     thumb file is like this
//  thumb file:  /image/dirk/.thumbnails/file.xxx.jpeg                     //  else like this

char * thumb2imagefile(cchar *thumbfile)
{
   STATB    statb;
   int      err, cc;
   char     *imagefile;
   char     *pp;
   
   if (thumbdirk && *thumbdirk == '/')                                     //  remove /thumb/dirk from the front
   {
      cc = strlen(thumbdirk);
      imagefile = zstrdup(thumbfile+cc);                                   //  have /image/dirk/file.xxx.jpeg
      cc = strlen(imagefile);
      imagefile[cc-5] = 0;                                                 //  /image/dirk/file.xxx
   }

   else                                                                    //  remove /.thumbnails from between
   {
      pp = (char *) strrchr(thumbfile,'/');                                //  /image/dirk/.thumbnails/file.xxx.jpeg
      if (! pp) return 0;                                                  //             |           |             |
      pp = pp - 12;                                                        //             pp          +12           +cc
      if (! strnEqu(pp,"/.thumbnails/",13)) return 0;
      cc = strlen(pp+12);
      if (! strEqu(pp+12+cc-5,".jpeg")) return 0;
      imagefile = zstrdup(thumbfile);
      pp = imagefile + (pp - thumbfile);
      memmove((char *) pp,pp+12,cc-5);                                     //  /image/dirk/file.xxx
      pp[cc-5] = 0;
   }

   err = stat(imagefile,&statb);                                           //  check file exists
   if (! err) return imagefile;                                            //  return image file
   zfree(imagefile);                                                       //  not found
   return 0;
}


//  Given an image file, get the corresponding thumbnail file.
//  The filespec is returned whether or not the file exists.
//  Returned file is subject for zfree().

char * image2thumbfile(cchar *imagefile)
{
   int      cc, cc1, cc2;
   char     *pp, *thumbfile;

   if (thumbdirk && *thumbdirk == '/')                                     //  use defined thumbnail directory
   {
      cc1 = strlen(thumbdirk);
      cc2 = strlen(imagefile);
      thumbfile = (char *) zmalloc(cc1+cc2+6);
      strcpy(thumbfile,thumbdirk);                                         //  /thumb/dirk
      strcpy(thumbfile+cc1,imagefile);                                     //  /thumb/dirk/image/dirk/file.xxx
      strcpy(thumbfile+cc1+cc2,".jpeg");                                   //  /thumb/dirk/image/dirk/file.xxx.jpeg
      return thumbfile;
   }

   else                                                                    //  use /image/dirk/.thumbnails/ 
   {
      thumbfile = zstrdup(imagefile,18);                                   //  /image/dirk/file.xxx
      pp = strrchr(thumbfile,'/');                                         //             |        |
      if (! pp) return 0;                                                  //             pp       +cc
      cc = strlen(pp);
      memmove(pp+12,pp,cc);                                                //  /image/dirk............/file.xxx
      strncpy(pp,"/.thumbnails",12);                                       //  /image/dirk/.thumbnails/file.xxx
      strcpy(pp+12+cc,".jpeg");                                            //  /image/dirk/.thumbnails/file.xxx.jpeg
      return thumbfile;
   }
}


//  Get thumbnail file for the given image file.
//  If missing or stale, add or update thumbnail file on disk.
//  Returned filespec is subject for zfree().
//  Optional arg 'ind' is returned 1 if thumb file created.

char * image_thumbfile(char *imagefile, int *ind)
{
   GdkPixbuf   *thumbpxb;
   GError      *gerror = 0;
   char        *thumbfile, *pp;
   int         err, ftyp;
   STATB       statf, statb;
   static int  Fmkdirerr = 0;
   
   if (ind) *ind = 0;
   
   err = stat(imagefile,&statf);
   if (err) return 0;
   
   ftyp = image_file_type(imagefile);
   
   if (ftyp == 4) {                                                        //  thumbnail file
      thumbfile = zstrdup(imagefile);                                      //  return same file
      return thumbfile;
   }

   if (ftyp != 2 && ftyp != 3) return 0;                                   //  not an image file or RAW file
   
   thumbfile = image2thumbfile(imagefile);                                 //  get thumbnail file for image file
   if (! thumbfile) return 0;

   err = stat(thumbfile,&statb);                                           //  thumbfile exists, up to date ??
   if (! err && statb.st_mtime >= statf.st_mtime) return thumbfile;        //  yes, return it

   pp = strrchr(thumbfile,'/');                                            //  check if thumbnail directory exists
   *pp = 0;
   err = stat(thumbfile,&statb);
   *pp = '/';
   if (err) {                                                              //  no
      pp = thumbfile;
      while (true) {
         pp = strchr(pp+1,'/');                                            //  check each directory level
         if (! pp) break;
         *pp = 0;
         err = stat(thumbfile,&statb);
         if (err) {
            err = mkdir(thumbfile,0750);                                   //  if missing, try to create
            if (err) {
               if (! Fmkdirerr++)                                          //  if unable, print error once only
                  printz("create thumbnail: %s \n",strerror(errno));
               zfree(thumbfile);
               return 0;
            }
         }
         *pp = '/';
      }
   }

   thumbpxb = get_thumbnail_pixbuf(imagefile,thumbfilesize);               //  generate thumbnail pixbuf from image file
   if (! thumbpxb) return 0;

   gdk_pixbuf_save(thumbpxb,thumbfile,"jpeg",&gerror,"quality","80",null); //  save as .jpeg file in thumbnail directory
   g_object_unref(thumbpxb);
   if (ind) *ind = 1;                                                      //  return indicator                   v.14.04.1
   return thumbfile;
}


//  Get thumbnail image (pixbuf) for given image file.
//  Use pixbuf cached in memory if available.
//  Create pixbuf from thumbnail file if size <= thumbnail file size.
//  Create pixbuf from image file if size > thumbnail file size.
//  Add pixbuf to memory cache if not there already.
//  Returned thumbnail belongs to caller: g_object_unref() is necessary.
//  Returns null if thumbnail not found on disk.

GdkPixbuf * image_thumbnail(char *fpath, int size)
{
   GdkPixbuf   *thumbpxb;
   GError      *gerror = 0;
   int         ii, err;
   char        *bpath;
   time_t      mtime;
   STATB       statf;
   const int   cachesize = thumbnail_cachesize;                            //  shorthand

   static int        nextcache, ftf = 1;
   static int        sizecache[cachesize];
   static time_t     mtimecache[cachesize];
   static char       *fpathcache[cachesize];
   static GdkPixbuf  *pixbufcache[cachesize];
   
   zthreadcrash();                                                         //  thread usage not allowed

   if (ftf) {                                                              //  first call
      for (ii = 0; ii < cachesize; ii++) 
         fpathcache[ii] = 0;                                               //  clear cache
      ftf = 0;
   }
   
   err = stat(fpath,&statf);                                               //  fpath status info
   if (err) return 0;

   if (! size) size = thumbfilesize;                                       //  default thumb size

   if (S_ISDIR(statf.st_mode)) {                                           //  if directory, return folder image
      thumbpxb = get_thumbnail_pixbuf(fpath,size);
      return thumbpxb;
   }

   mtime = statf.st_mtime;                                                 //  last modification time

   for (ii = nextcache; ii >= 0; ii--)
      if (fpathcache[ii] && strEqu(fpath,fpathcache[ii]) &&
          sizecache[ii] == size && mtime == mtimecache[ii]) break;         //  check mtime
   if (ii >= 0) {
      thumbpxb = gdk_pixbuf_copy(pixbufcache[ii]);
      return thumbpxb;
   }
   for (ii = cachesize-1; ii > nextcache; ii--)                            //  continue search
      if (fpathcache[ii] && strEqu(fpath,fpathcache[ii]) &&
          sizecache[ii] == size && mtime == mtimecache[ii]) break;
   if (ii > nextcache) {
      thumbpxb = gdk_pixbuf_copy(pixbufcache[ii]);
      return thumbpxb;
   }

   if (size > thumbfilesize)                                               //  get thumbnail from image file
      thumbpxb = get_thumbnail_pixbuf(fpath,size);
   else {
      bpath = image_thumbfile(fpath);                                      //  get thumbnail file path
      if (! bpath) return 0;                                               //    (create thumbnail file if missing)
      thumbpxb = gdk_pixbuf_new_from_file_at_size(bpath,size,size,&gerror);//  get thumbnail from thumbnail file
      zfree(bpath);
   }
   if (! thumbpxb) return 0;

   nextcache++;                                                            //  next cache slot (oldest)
   if (nextcache == cachesize) nextcache = 0;
   ii = nextcache;
   if (fpathcache[ii]) {                                                   //  free prior occupant
      zfree(fpathcache[ii]);
      g_object_unref(pixbufcache[ii]);
   }
   fpathcache[ii] = zstrdup(fpath);                                        //  add new occupant
   pixbufcache[ii] = gdk_pixbuf_copy(thumbpxb);                            //  this memory is not tracked
   sizecache[ii] = size;
   mtimecache[ii] = mtime;

   return thumbpxb;                                                        //  return pixbuf to caller
}


//  Make a thumbnail pixbuf from the image file.
//  File can be a regular supported image file (jpeg etc.)
//    or a supported RAW file type.

GdkPixbuf * get_thumbnail_pixbuf(char *imagefile, int size)
{
   GdkPixbuf      *thumbpxb = 0;
   GError         *gerror = 0;
   PXB            *rawpxb;
   char           *thumbfile, *pp;
   int            err, ftyp;
   static int     ftf = 1;
   static char    folderthumb[300], brokenthumb[300];
   
   if (ftf) {
      ftf = 0;
      strcpy(folderthumb,zfuncs::zicondir);                                //  folder icon
      strcat(folderthumb,"/folder.png");
      strcpy(brokenthumb,zfuncs::zicondir);                                //  broken thumbnail icon
      strcat(brokenthumb,"/broken.png");
   }

   ftyp = image_file_type(imagefile);

   if (ftyp == 1)                                                          //  directory file
      thumbpxb = gdk_pixbuf_new_from_file_at_size(folderthumb,size,size,&gerror);

   else if (ftyp == 2)                                                     //  supported image type (jpeg etc.)
      thumbpxb = gdk_pixbuf_new_from_file_at_size(imagefile,size,size,&gerror);

   else if (ftyp == 3) {                                                   //  supported RAW file type
      err = shell_quiet("dcraw -e \"%s\" ",imagefile);                     //  extract jpeg thumbnail from RAW file
      if (! err) {
         thumbfile = zstrdup(imagefile,12);
         pp = strrchr(thumbfile,'.');
         if (! pp) pp = thumbfile + strlen(thumbfile);
         strcpy(pp,".thumb.jpg");
         thumbpxb = gdk_pixbuf_new_from_file_at_size(thumbfile,size,size,&gerror);
         remove(thumbfile);
         zfree(thumbfile);
      }
      else {                                                               //  RAW has no embedded thumbnail
         rawpxb = RAW_PXB_load(imagefile);                                 //  use full image to make thumbnail   v.14.06
         if (rawpxb) {
            thumbpxb = gdk_pixbuf_scale_simple(rawpxb->pixbuf,size,size,BILINEAR);
            PXB_free(rawpxb);
         }
      }
   }

   else return 0;

   if (! thumbpxb) {
      printz("cannot make thumbnail: %s \n",imagefile);
      if (gerror) printz(" %s \n",gerror->message);
      gerror = 0;
      thumbpxb = gdk_pixbuf_new_from_file_at_size(brokenthumb,size,size,&gerror);
   }

   return thumbpxb;
}


//  popup a new window with a larger image of a clicked thumbnail

void popimage(int Fnewin)                                                  //  v.14.09
{
   static int   ftf = 1, ii;
   static char  *popfiles[20];                                             //  last 20 images
   
   if (ftf) {
      ftf = 0;                                                             //  initz. empty file memory
      for (ii = 0; ii < 20; ii++)
         popfiles[ii] = 0;
      ii = 0;
   }

   if (! clicked_file) return;

   ii++;                                                                   //  use next file memory position
   if (ii == 20) ii = 0;
   if (popfiles[ii]) zfree(popfiles[ii]);
   popfiles[ii] = zstrdup(clicked_file);                                   //  save clicked_file persistently
   clicked_file = 0;                                                       //  reset clicked_file

   popup_image(popfiles[ii],MWIN,Fnewin,512);                              //  popup window with image
   return;
}


//  Monitor a gallery directory for file changes and refresh the file list.
//  Action is "start" or "stop".
//  Called only for gallerytype = 1 = directory.

void gallery_monitor(cchar *action)
{
   using namespace navi;
   
   void gallery_changed(void *, GFile *, void *, GFileMonitorEvent);

   static GFile          *gfile_gallery = 0;                               //  directory being monitored
   static GFileMonitor   *gallerymon = 0;
   GError                *gerror = 0;

   if (gfile_gallery) {
      g_file_monitor_cancel(gallerymon);
      g_object_unref(gallerymon);
      g_object_unref(gfile_gallery);
      gfile_gallery = 0;
   }

   if (strEqu(action,"stop")) return;
   if (gallerytype != 1) return;
   
   gfile_gallery = g_file_new_for_path(galleryname);
   gallerymon = g_file_monitor_directory(gfile_gallery,(GFileMonitorFlags) 0,0,&gerror);
   if (gallerymon)
      G_SIGNAL(gallerymon,"changed",gallery_changed,0);
   else {
      printz("monitor directory failure: %s \n",galleryname);              //  it happens
      printz("%s\n",gerror->message);
      g_object_unref(gfile_gallery);
      gfile_gallery = 0;
   }
   return;
}

void gallery_changed(void *, GFile *, void *, GFileMonitorEvent event)
{
   using namespace navi;
   
   if (gallerytype != 1) {                                                 //  precaution
      gallery_monitor("stop");
      return;
   }
   
   if (event == G_FILE_MONITOR_EVENT_DELETED) {
      gallery(galleryname,"init");                                         //  refresh file list
      if (FGW == 'G') gallery(0,"paint",-1);                               //  repaint from same position
   }

   else if (event == G_FILE_MONITOR_EVENT_CREATED) {
      gallery(galleryname,"init");
      if (FGW == 'G') gallery(0,"paint",-1);
   }
   
   return;
}


/**************************************************************************/

//  Select files from the image gallery window, return list of files selected.
//  The dialog shows the list of files selected and can be edited.
//  The returned file list belongs to caller and is subject for zfree().
//  The file list EOL is marked with null.
//  The gallery() callback function is restored to caller's function.

namespace ggfnames
{
   int  dialog_event(zdialog *zd, cchar *event);
   int  find_file(cchar *imagefile);
   void insert_file(cchar *imagefile);
   void remove_file(cchar *imagefile);
   void Xclick_func(int Nth, char LR);
   int  mouseclick(GtkWidget *, GdkEventButton *event, void *);
   int  showthumb();

   GtkWidget   *drawarea = 0;
   GtkWidget   *Fwin = 0;
   cchar       *font = "Monospace 8";
   int         fontheight = 14;
   int         cursorpos = 0;
   char        **initfiles = 0;
};


char ** gallery_getfiles(char **initfiles0)
{
   using namespace navi;
   using namespace ggfnames;

   PangoLanguage           *plang;
   PangoFontDescription    *pfontdesc;
   PangoContext            *pcontext;
   PangoFont               *pfont;
   PangoFontMetrics        *pmetrics;
   GdkCursor               *cursor;
   GdkWindow               *gdkwin;
   GtkTextBuffer           *textBuff;
   GtkTextIter             iter1, iter2;

   int      fontascent, fontdescent;
   int      line, nlines, ii;
   char     *imagefile = 0, **filelist = 0;
   cchar    *helptopic = 0;

   if (initfiles0)                                                         //  initial files for dialog window
      initfiles = initfiles0;
   else initfiles = 0;
   
   zdialog *zd = zdialog_new(ZTX("Select Files"),Mwin,Bdone,Bcancel,null);
   zd_gallery_getfiles = zd;
   zdialog_add_widget(zd,"hbox","hb1","dialog",0,"expand|space=3");
   zdialog_add_widget(zd,"frame","fr11","hb1",0,"expand");
   zdialog_add_widget(zd,"scrwin","scrwin","fr11",0,"expand");
   zdialog_add_widget(zd,"edit","files","scrwin");
   zdialog_add_widget(zd,"vbox","vb12","hb1");
   zdialog_add_widget(zd,"frame","fr12","vb12");
   zdialog_add_widget(zd,"hbox","hb2","dialog",0,"space=5");
   zdialog_add_widget(zd,"button","delete","hb2",Bdelete,"space=8");
   zdialog_add_widget(zd,"button","insert","hb2",Binsert,"space=8");
   zdialog_add_widget(zd,"button","clear","hb2",Bclear,"space=8");
   zdialog_add_widget(zd,"button","addall","hb2",Baddall,"space=8");

   GtkWidget *textbox = zdialog_widget(zd,"files");                        //  disable text wrap 
   gtk_text_view_set_wrap_mode(GTK_TEXT_VIEW(textbox),GTK_WRAP_NONE);

   GtkWidget *frame = zdialog_widget(zd,"fr12");                           //  drawing area for thumbnail image
   drawarea = gtk_drawing_area_new();
   gtk_widget_set_size_request(drawarea,256,258);                          //  increased
   gdkwin = gtk_widget_get_window(drawarea);                               //  gtk3
   gtk_container_add(GTK_CONTAINER(frame),drawarea);
   
   Fwin = zdialog_widget(zd,"files");                                      //  activate mouse-clicks for
   gtk_widget_add_events(Fwin,GDK_BUTTON_PRESS_MASK);                      //    file list widget
   G_SIGNAL(Fwin,"button-press-event",mouseclick,0);

   pfontdesc = pango_font_description_from_string(font);                   //  set default font for files window
   gtk_widget_override_font(Fwin,pfontdesc);

   plang = pango_language_get_default();                                   //  get font metrics (what a mess)
   pcontext = gtk_widget_get_pango_context(Fwin);
   pfont = pango_context_load_font(pcontext,pfontdesc);
   pmetrics = pango_font_get_metrics(pfont,plang);
   fontascent = pango_font_metrics_get_ascent(pmetrics) / PANGO_SCALE;
   fontdescent = pango_font_metrics_get_descent(pmetrics) / PANGO_SCALE;
   fontheight = fontascent + fontdescent;                                  //  effective line height

   zdialog_resize(zd,600,0);                                               //  start dialog
   zdialog_run(zd,dialog_event,"save");                                    //  keep relative position

   cursor = gdk_cursor_new(GDK_TOP_LEFT_ARROW);                            //  cursor for file list widget
   gdkwin = gtk_text_view_get_window(GTK_TEXT_VIEW(Fwin),TEXTWIN);         //  (do after window realized)
   gdk_window_set_cursor(gdkwin,cursor);
   cursorpos = 0;

   if (! gallerytype) {                                                    //  if no gallery, start with
      gallery(topdirks[0],"init");                                         //    top directory
      gallery(0,"paint",0);
   }

   helptopic = F1_help_topic;                                              //  save F1 help topic 
   m_viewmode(0,"G");                                                      //  open gallery window  (F1_help = "navigation")
   if (initfiles) zdialog_send_event(zd,"initfiles");                      //  stuff initial files
   zdialog_wait(zd);                                                       //  wait for dialog completion
   F1_help_topic = helptopic;                                              //  restore 

   if (zd->zstat != 1) {                                                   //  cancelled
      zdialog_free(zd);                                                    //  kill dialog
      zd_gallery_getfiles = 0;
      return 0;
   }

   textBuff = gtk_text_view_get_buffer(GTK_TEXT_VIEW(Fwin));
   nlines = gtk_text_buffer_get_line_count(textBuff);

   filelist = (char **) zmalloc((nlines+1) * sizeof(char *));

   for (ii = line = 0; line < nlines; line++)                              //  get list of files from dialog
   {
      gtk_text_buffer_get_iter_at_line(textBuff,&iter1,line);              //  iter at line start
      iter2 = iter1;
      gtk_text_iter_forward_to_line_end(&iter2);
      imagefile = gtk_text_buffer_get_text(textBuff,&iter1,&iter2,0);      //  get line of text
      if (imagefile && *imagefile == '/') {
         filelist[ii] = zstrdup(imagefile);                                //  >> next file in list
         zfree(imagefile);
         ii++;
      }
   }
   filelist[ii] = 0;                                                       //  mark EOL
   
   zdialog_free(zd);                                                       //  kill dialog
   zd_gallery_getfiles = 0;

   if (! ii) {
      zfree(filelist);                                                     //  file list is empty
      return 0;
   }

   return filelist;                                                        //  return file list
}


//  gallery getfiles dialog event function

int ggfnames::dialog_event(zdialog *zd, cchar *event)
{
   using namespace ggfnames;

   GtkTextBuffer  *textBuff;
   GtkTextIter    iter1, iter2;
   char           *ftemp, *imagefile;
   static char    *deletedfiles[100];                                      //  last 100 files deleted
   static int     Ndeleted = 0;
   int            ii, line, Nth, ftyp;
   
   if (strEqu(event,"focus")) showthumb();                                 //  GTK bug? thumbnail disappears
   
   if (strEqu(event,"initfiles"))                                          //  insert all files in initial list
   {
      textBuff = gtk_text_view_get_buffer(GTK_TEXT_VIEW(Fwin));
      
      for (ii = 0; initfiles[ii]; ii++)
      {
         imagefile = initfiles[ii];
         ftyp = image_file_type(imagefile);                                //  must be image or RAW file
         if (ftyp != 2 && ftyp != 3) continue;
         gtk_text_buffer_get_iter_at_line(textBuff,&iter1,cursorpos);
         gtk_text_buffer_insert(textBuff,&iter1,"\n",1);                   //  insert new blank line
         gtk_text_buffer_get_iter_at_line(textBuff,&iter1,cursorpos);
         gtk_text_buffer_insert(textBuff,&iter1,imagefile,-1);             //  insert image file
         cursorpos++;                                                      //  advance cursor position
      }
   }

   if (strEqu(event,"delete"))                                             //  delete file at cursor position
   {
      textBuff = gtk_text_view_get_buffer(GTK_TEXT_VIEW(Fwin));
      line = cursorpos;
      gtk_text_buffer_get_iter_at_line(textBuff,&iter1,line);              //  iter at line start
      iter2 = iter1;
      gtk_text_iter_forward_to_line_end(&iter2);                           //  iter at line end

      ftemp = gtk_text_buffer_get_text(textBuff,&iter1,&iter2,0);          //  get selected file
      if (*ftemp != '/') {
         zfree(ftemp);
         return 1;
      }

      if (Ndeleted == 100) {                                               //  capacity reached
         zfree(deletedfiles[0]);                                           //  remove oldest entry
         for (ii = 0; ii < 99; ii++)
            deletedfiles[ii] = deletedfiles[ii+1];
         Ndeleted = 99;
      }
         
      deletedfiles[Ndeleted] = zstrdup(ftemp);                             //  save deleted file for poss. insert
      Ndeleted++;
      zfree(ftemp);
      
      gtk_text_buffer_delete(textBuff,&iter1,&iter2);                      //  delete file text
      gtk_text_buffer_get_iter_at_line(textBuff,&iter2,line+1);
      gtk_text_buffer_delete(textBuff,&iter1,&iter2);                      //  delete empty line (\n)

      showthumb();                                                         //  thumbnail = next file
   }

   if (strEqu(event,"insert"))                                             //  insert first deleted file
   {                                                                       //    at current cursor position
      if (! Ndeleted) return 1;
      insert_file(deletedfiles[0]);
      zfree(deletedfiles[0]);                                              //  remove file from the deleted list
      for (ii = 0; ii < Ndeleted-1; ii++)
         deletedfiles[ii] = deletedfiles[ii+1];
      Ndeleted--;
   }
   
   if (strEqu(event,"clear")) {                                            //  clear all files 
      gtk_text_view_set_buffer(GTK_TEXT_VIEW(Fwin),null);
      cursorpos = 0;
   }

   if (strEqu(event,"addall"))                                             //  insert all files in image gallery
   {
      textBuff = gtk_text_view_get_buffer(GTK_TEXT_VIEW(Fwin));
      Nth = 0;

      while (true)
      {
         imagefile = gallery(0,"find",Nth);                                //  get first or next file
         if (! imagefile) break;
         Nth++;
         ftyp = image_file_type(imagefile);                                //  must be image or RAW file
         if (ftyp != 2 && ftyp != 3) continue;
         gtk_text_buffer_get_iter_at_line(textBuff,&iter1,cursorpos);
         gtk_text_buffer_insert(textBuff,&iter1,"\n",1);                   //  insert new blank line
         gtk_text_buffer_get_iter_at_line(textBuff,&iter1,cursorpos);
         gtk_text_buffer_insert(textBuff,&iter1,imagefile,-1);             //  insert image file
         zfree(imagefile);
         cursorpos++;                                                      //  advance cursor position
      }
   }

   return 1;
}


//  See if image file is in the file list already or not.
//  Return the matching line number or -1 if not found.

int ggfnames::find_file(cchar *imagefile)
{
   using namespace ggfnames;

   GtkTextBuffer  *textBuff;
   GtkTextIter    iter1, iter2;
   char           *ftemp;
   int            line, last = -1, more;

   textBuff = gtk_text_view_get_buffer(GTK_TEXT_VIEW(Fwin));

   for (line = 0; ; line++)
   {   
      gtk_text_buffer_get_iter_at_line(textBuff,&iter1,line);              //  iter at line start
      iter2 = iter1;
      more = gtk_text_iter_forward_to_line_end(&iter2);                    //  iter at line end
      ftemp = gtk_text_buffer_get_text(textBuff,&iter1,&iter2,0);          //  included text
      if (strEqu(ftemp,imagefile)) last = line;                            //  remember last entry found
      zfree(ftemp);
      if (! more) break;
   }
   
   return last;
}


//  add image file to list at current cursor position, set thumbnail = file

void ggfnames::insert_file(cchar *imagefile)
{
   using namespace ggfnames;

   GtkTextIter    iter;
   GtkTextBuffer  *textBuff;
   
   textBuff = gtk_text_view_get_buffer(GTK_TEXT_VIEW(Fwin));
   gtk_text_buffer_get_iter_at_line(textBuff,&iter,cursorpos);
   gtk_text_buffer_insert(textBuff,&iter,imagefile,-1);                    //  insert image file
   gtk_text_buffer_insert(textBuff,&iter,"\n",1);                          //  insert new line
   gtk_text_view_scroll_to_iter(GTK_TEXT_VIEW(Fwin),&iter,0,0,0,0);        //  insure visible                     v.14.09

   showthumb();                                                            //  update thumbnail
   cursorpos++;                                                            //  advance cursor position

   return;
}


//  remove image file at last position found, set thumbnail = next

void ggfnames::remove_file(cchar *imagefile)
{
   using namespace ggfnames;

   GtkTextBuffer  *textBuff;
   GtkTextIter    iter1, iter2;
   int            line;

   line = find_file(imagefile);
   if (line < 0) return;

   textBuff = gtk_text_view_get_buffer(GTK_TEXT_VIEW(Fwin));
   gtk_text_buffer_get_iter_at_line(textBuff,&iter1,line);                 //  iter at line start
   iter2 = iter1;
   gtk_text_iter_forward_to_line_end(&iter2);                              //  iter at line end
   gtk_text_buffer_delete(textBuff,&iter1,&iter2);                         //  delete file text
   gtk_text_buffer_get_iter_at_line(textBuff,&iter2,line+1);               //  next line
   gtk_text_buffer_delete(textBuff,&iter1,&iter2);                         //  delete empty line (\n)
   
   showthumb();                                                            //  thumbnail = curr. position
   if (cursorpos > 0) cursorpos--;                                         //  backup cursor position 
   return;
}


//  called from image gallery window when a thumbnail is clicked
//  add image file to list at current cursor position, set thumbnail = file

void gallery_getfiles_Lclick_func(int Nth)
{
   ggfnames::Xclick_func(Nth,'L');
   return;
}

void gallery_getfiles_Rclick_func(int Nth)
{
   ggfnames::Xclick_func(Nth,'R');
   return;
}

void ggfnames::Xclick_func(int Nth, char LR)
{
   using namespace ggfnames;

   int            ftyp, line;
   static int     pNth = -1;                                               //  previously clicked file
   char           *imagefile;
   int            control, shift;
   int            nn, incr;
   
   if (! zd_gallery_getfiles) return;                                      //  should not happen
   if (Nth < 0) return;                                                    //  gallery gone ?
   
   imagefile = gallery(0,"find",Nth);                                      //  get file at clicked position
   if (! imagefile) {
      pNth = -1;
      return;
   }

   ftyp = image_file_type(imagefile);                                      //  must be image or RAW file
   if (ftyp != 2 && ftyp != 3) {
      zfree(imagefile);
      pNth = -1;
      return;
   }
   
   if (LR == 'R') {                                                        //  right click, unselect
      remove_file(imagefile);
      zfree(imagefile);
      return;
   }
   
   control = shift = 0;                                                    //  left click, select
   if (KBcontrolkey) control = 1;
   if (KBshiftkey) shift = 1;
   
   if (! control && ! shift)                                               //  no control or shift keys
   {
      pNth = Nth;                                                          //  possible start of range
      insert_file(imagefile);                                              //  insert file at current position
      zfree(imagefile);
      return;
   }

   if (control && ! shift)                                                 //  control key
   {
      pNth = -1;                                                           //  add or remove single file
      line = find_file(imagefile);
      if (line < 0) insert_file(imagefile);                                //  not found, add
      else remove_file(imagefile);                                         //  found, remove
      zfree(imagefile);
      return;
   }

   if (! control && shift)                                                 //  shift key, end of range
   {
      if (pNth < 0) return;                                                //  no start of range, ignore

      if (pNth > Nth) incr = -1;                                           //  range is descending
      else incr = +1;                                                      //  ascending

      for (nn = pNth+incr; nn != Nth+incr; nn += incr)                     //  add all files from pNth to Nth
      {                                                                    //    excluding pNth (already added)
         imagefile = gallery(0,"find",nn);
         if (! imagefile) continue;
         ftyp = image_file_type(imagefile);                                //  only image and RAW files
         if (ftyp != 2 && ftyp != 3) {
            zfree(imagefile);
            continue;
         }
         insert_file(imagefile);
         zfree(imagefile);
      }
      pNth = -1;                                                           //  no prior
      return;
   }

   if (control && shift)                                                   //  both control and shift keys
      return;                                                              //  ignore
   
   return;
}


//  process mouse click in files window: 
//  set new cursor position and set thumbnail = clicked file

int ggfnames::mouseclick(GtkWidget *, GdkEventButton *event, void *)
{
   using namespace ggfnames;

   int            mpy;
   GtkWidget      *scrollwin;
   GtkAdjustment  *scrolladj;
   double         scrollpos;

   if (event->type != GDK_BUTTON_PRESS) return 0;
   mpy = int(event->y);
   scrollwin = zdialog_widget(zd_gallery_getfiles,"scrwin");               //  window scroll position
   scrolladj = gtk_scrolled_window_get_vadjustment(GTK_SCROLLED_WINDOW(scrollwin));
   scrollpos = gtk_adjustment_get_value(scrolladj);
   cursorpos = (mpy + scrollpos) / fontheight;                             //  line selected
   showthumb();                                                            //  show thumbnail image
   return 0;
}


//  show thumbnail for file at current cursor position

int ggfnames::showthumb()
{
   using namespace ggfnames;

   int            line;
   char           *imagefile;
   GdkWindow      *gdkwin;
   cairo_t        *cr;
   GtkTextBuffer  *textBuff;
   GtkTextIter    iter1, iter2;
   GdkPixbuf      *thumbnail = 0;
   
   gtk_widget_grab_focus(drawarea);                                        //  GTK bug?
   zmainloop();                                                            //  stop thumbnail disappearing

   gdkwin = gtk_widget_get_window(drawarea);
   cr = gdk_cairo_create(gdkwin);
   
   line = cursorpos;
   textBuff = gtk_text_view_get_buffer(GTK_TEXT_VIEW(Fwin));
   gtk_text_buffer_get_iter_at_line(textBuff,&iter1,line);                 //  iter at line start
   iter2 = iter1;
   gtk_text_iter_forward_to_line_end(&iter2);                              //  iter at line end

   imagefile = gtk_text_buffer_get_text(textBuff,&iter1,&iter2,0);         //  get selected file
   if (*imagefile != '/') {
      zfree(imagefile);
      cairo_set_source_rgb(cr,1,1,1);                                      //  white background 
      cairo_paint(cr); 
      return 0;
   }

   thumbnail = image_thumbnail(imagefile,256);                             //  get thumbnail
   zfree(imagefile);

   if (thumbnail) {
      cairo_set_source_rgb(cr,1,1,1);                                      //  white background
      cairo_paint(cr); 
      gdk_cairo_set_source_pixbuf(cr,thumbnail,0,0);                       //  paint thumbnail
      cairo_paint(cr);
      g_object_unref(thumbnail);
   }

   return 0;
}



