# 46 "cil/src/frontc/clexer.mll"
 
open Cparser
open Pretty
exception Eof
exception InternalError of string
module E = Errormsg
module H = Hashtbl

let matchingParsOpen = ref 0

let currentLoc () = Cabshelper.currentLoc ()

let one_line_ghost = ref false
let is_oneline_ghost () = !one_line_ghost
let enter_oneline_ghost () = one_line_ghost := true
let exit_oneline_ghost () = one_line_ghost := false

let ghost_code = ref false
let is_ghost_code () = !ghost_code
let enter_ghost_code () = ghost_code := true
let exit_ghost_code () = ghost_code := false

(* string -> unit *)
let addComment c =
  let l = currentLoc() in
  let i = GrowArray.max_init_index Cabshelper.commentsGA in
  GrowArray.setg Cabshelper.commentsGA (i+1) (l,c,false)

(* track whitespace for the current token *)
let white = ref ""

let addWhite lexbuf =
  if !Whitetrack.enabled then
    let w = Lexing.lexeme lexbuf in
    white := !white ^ w
let clear_white () = white := ""
let get_white () = !white

let lexeme = ref ""
let addLexeme lexbuf =
    let l = Lexing.lexeme lexbuf in
    lexeme := !lexeme ^ l
let clear_lexeme () = lexeme := ""
let get_extra_lexeme () = !lexeme

let int64_to_char value =
  if (compare value (Int64.of_int 255) > 0) || (compare value Int64.zero < 0) then
    begin
      let msg = Printf.sprintf "clexer:intlist_to_string: character 0x%Lx too big" value in
      E.parse_error msg;
    end
  else
    Char.chr (Int64.to_int value)

(* takes a not-nul-terminated list, and converts it to a string. *)
let rec intlist_to_string (str: int64 list):string =
  match str with
    [] -> ""  (* add nul-termination *)
  | value::rest ->
      let this_char = int64_to_char value in
      (String.make 1 this_char) ^ (intlist_to_string rest)

(* Some debugging support for line numbers *)
let dbgToken (t: token) =
  if false then begin
    ignore (E.log "%a" insert
              (match t with
                IDENT (n, l) -> dprintf "IDENT(%s,%d)\n" n (fst l).Lexing.pos_lnum
              | LBRACE l -> dprintf "LBRACE(%d)\n" (fst l).Lexing.pos_lnum
              | RBRACE l -> dprintf "RBRACE(%d)\n" (fst l).Lexing.pos_lnum
              | IF l -> dprintf "IF(%d)\n" (fst l).Lexing.pos_lnum
              | SWITCH l -> dprintf "SWITCH(%d)\n" (fst l).Lexing.pos_lnum
              | RETURN l -> dprintf "RETURN(%d)\n" (fst l).Lexing.pos_lnum
              | _ -> nil));
    t
  end else
    t

(*
** Keyword hashtable
*)
let lexicon = H.create 211
let init_lexicon _ =
  H.clear lexicon;
  List.iter
    (fun (key, builder) -> H.add lexicon key builder)
    [ ("auto", fun loc -> AUTO loc);
      ("const", fun loc -> CONST loc);
      ("__const", fun loc -> CONST loc);
      ("__const__", fun loc -> CONST loc);
      ("static", fun loc -> STATIC loc);
      ("extern", fun loc -> EXTERN loc);
      ("long", fun loc -> LONG loc);
      ("short", fun loc -> SHORT loc);
      ("register", fun loc -> REGISTER loc);
      ("signed", fun loc -> SIGNED loc);
      ("__signed", fun loc -> SIGNED loc);
      ("unsigned", fun loc -> UNSIGNED loc);
      ("volatile", fun loc -> VOLATILE loc);
      ("__volatile", fun loc -> VOLATILE loc);
      (* WW: see /usr/include/sys/cdefs.h for why __signed and __volatile
       * are accepted GCC-isms *)
      ("char", fun loc -> CHAR loc);
      ("_Bool", fun loc -> BOOL loc);
      ("int", fun loc -> INT loc);
      ("float", fun loc -> FLOAT loc);
      ("double", fun loc -> DOUBLE loc);
      ("void", fun loc -> VOID loc);
      ("enum", fun loc -> ENUM loc);
      ("struct", fun loc -> STRUCT loc);
      ("typedef", fun loc -> TYPEDEF loc);
      ("union", fun loc -> UNION loc);
      ("break", fun loc -> BREAK loc);
      ("continue", fun loc -> CONTINUE loc);
      ("goto", fun loc -> GOTO loc);
      ("return", fun loc -> dbgToken (RETURN loc));
      ("switch", fun loc -> dbgToken (SWITCH loc));
      ("case", fun loc -> CASE loc);
      ("default", fun loc -> DEFAULT loc);
      ("while", fun loc -> WHILE loc);
      ("do", fun loc -> DO loc);
      ("for", fun loc -> FOR loc);
      ("if", fun loc -> dbgToken (IF loc));
      ("else", fun _ -> ELSE);
      (*** Implementation specific keywords ***)
      ("__signed__", fun loc -> SIGNED loc);
      ("__inline__", fun loc -> INLINE loc);
      ("inline", fun loc -> INLINE loc);
      ("__inline", fun loc -> INLINE loc);
      ("_inline", fun loc ->
                      if !Cprint.msvcMode then
                        INLINE loc
                      else
                        IDENT ("_inline", loc));
      ("__attribute__", fun loc -> ATTRIBUTE loc);
      ("__attribute", fun loc -> ATTRIBUTE loc);
(*
      ("__attribute_used__", fun loc -> ATTRIBUTE_USED loc);
*)
      ("__blockattribute__", fun _ -> BLOCKATTRIBUTE);
      ("__blockattribute", fun _ -> BLOCKATTRIBUTE);
      ("__asm__", fun loc -> ASM loc);
      ("asm", fun loc -> ASM loc);
      ("__typeof__", fun loc -> TYPEOF loc);
      ("__typeof", fun loc -> TYPEOF loc);
      ("typeof", fun loc -> TYPEOF loc);
      ("__alignof", fun loc -> ALIGNOF loc);
      ("__alignof__", fun loc -> ALIGNOF loc);
      ("__volatile__", fun loc -> VOLATILE loc);
      ("__volatile", fun loc -> VOLATILE loc);

      ("__FUNCTION__", fun loc -> FUNCTION__ loc);
      ("__func__", fun loc -> FUNCTION__ loc); (* ISO 6.4.2.2 *)
      ("__PRETTY_FUNCTION__", fun loc -> PRETTY_FUNCTION__ loc);
      ("__label__", fun _ -> LABEL__);
      (*** weimer: GCC arcana ***)
      ("__restrict", fun loc -> RESTRICT loc);
      ("restrict", fun loc -> RESTRICT loc);
(*      ("__extension__", EXTENSION); *)
      (**** MS VC ***)
      ("__int64", fun _ -> INT64 (currentLoc ()));
      ("__int32", fun loc -> INT loc);
      ("_cdecl",  fun _ -> MSATTR ("_cdecl", currentLoc ()));
      ("__cdecl", fun _ -> MSATTR ("__cdecl", currentLoc ()));
      ("_stdcall", fun _ -> MSATTR ("_stdcall", currentLoc ()));
      ("__stdcall", fun _ -> MSATTR ("__stdcall", currentLoc ()));
      ("_fastcall", fun _ -> MSATTR ("_fastcall", currentLoc ()));
      ("__fastcall", fun _ -> MSATTR ("__fastcall", currentLoc ()));
      ("__w64", fun _ -> MSATTR("__w64", currentLoc ()));
      ("__declspec", fun loc -> DECLSPEC loc);
      ("__forceinline", fun loc -> INLINE loc); (* !! we turn forceinline
                                                 * into inline *)
      ("__try", fun loc -> TRY loc);
      ("__except", fun loc -> EXCEPT loc);
      ("__finally", fun loc -> FINALLY loc);
      (* weimer: some files produced by 'GCC -E' expect this type to be
       * defined *)
      ("__builtin_va_list",
       fun _ -> NAMED_TYPE ("__builtin_va_list", currentLoc ()));
      ("__builtin_va_arg", fun loc -> BUILTIN_VA_ARG loc);
      ("__builtin_types_compatible_p", fun loc -> BUILTIN_TYPES_COMPAT loc);
      ("__builtin_offsetof", fun loc -> BUILTIN_OFFSETOF loc);
      (* On some versions of GCC __thread is a regular identifier *)
      ("__thread", fun loc ->
                      if Machdep.state.Machdep.__thread_is_keyword then
                         THREAD loc
                       else
                         IDENT ("__thread", loc));
    ]

(* Mark an identifier as a type name. The old mapping is preserved and will
 * be reinstated when we exit this context *)
let add_type name =
   (* ignore (print_string ("adding type name " ^ name ^ "\n"));  *)
  H.add lexicon name (fun loc -> NAMED_TYPE (name, loc));
  H.add Logic_const.typenames name true

let context : string list list ref = ref [ [] ]

let push_context _ = context := []::!context

let pop_context _ =
  match !context with
    [] -> raise (InternalError "Empty context stack")
  | con::sub ->
		(context := sub;
		List.iter (fun name ->
                            (* Format.eprintf
                              "removing lexicon for %s@." name; *)
                             H.remove lexicon name;
                             H.remove Logic_const.typenames name
                          ) con)

(* Mark an identifier as a variable name. The old mapping is preserved and
 * will be reinstated when we exit this context  *)
let add_identifier name =
  match !context with
    [] -> raise (InternalError "Empty context stack")
  | con::sub ->
      (context := (name::con)::sub;
       (*Format.eprintf "adding IDENT for %s@." name;*)
       H.add lexicon name (fun loc ->
         dbgToken (IDENT (name, loc)));
       H.add Logic_const.typenames name false
      )


(*
** Useful primitives
*)
let scan_ident id =
  let here = currentLoc () in
  try (H.find lexicon id) here
  (* default to variable name, as opposed to type *)
  with Not_found -> dbgToken (IDENT (id, here))


(*
** Buffer processor
*)


let init ~(filename: string) : Lexing.lexbuf =
  init_lexicon ();
  (* Inititialize the pointer in Errormsg *)
  Lexerhack.add_type := add_type;
  Lexerhack.push_context := push_context;
  Lexerhack.pop_context := pop_context;
  Lexerhack.add_identifier := add_identifier;
  E.startParsing ~useBasename:false filename


let finish () =
  E.finishParsing ()

(*** Error handling ***)
let error msg =
  E.parse_error msg


(*** escape character management ***)
let scan_escape (char: char) : int64 =
  let result = match char with
    'n' -> '\n'
  | 'r' -> '\r'
  | 't' -> '\t'
  | 'b' -> '\b'
  | 'f' -> '\012'  (* ASCII code 12 *)
  | 'v' -> '\011'  (* ASCII code 11 *)
  | 'a' -> '\007'  (* ASCII code 7 *)
  | 'e' | 'E' -> '\027'  (* ASCII code 27. This is a GCC extension *)
  | '\'' -> '\''
  | '"'-> '"'     (* '"' *)
  | '?' -> '?'
  | '(' when not !Cprint.msvcMode -> '('
  | '{' when not !Cprint.msvcMode -> '{'
  | '[' when not !Cprint.msvcMode -> '['
  | '%' when not !Cprint.msvcMode -> '%'
  | '\\' -> '\\'
  | other -> error ("Unrecognized escape sequence: \\" ^ (String.make 1 other))
  in
  Int64.of_int (Char.code result)

let scan_hex_escape str =
  let radix = Int64.of_int 16 in
  let the_value = ref Int64.zero in
  (* start at character 2 to skip the \x *)
  for i = 2 to (String.length str) - 1 do
    let thisDigit = Cabshelper.valueOfDigit (String.get str i) in
    (* the_value := !the_value * 16 + thisDigit *)
    the_value := Int64.add (Int64.mul !the_value radix) thisDigit
  done;
  !the_value

let scan_oct_escape str =
  let radix = Int64.of_int 8 in
  let the_value = ref Int64.zero in
  (* start at character 1 to skip the \x *)
  for i = 1 to (String.length str) - 1 do
    let thisDigit = Cabshelper.valueOfDigit (String.get str i) in
    (* the_value := !the_value * 8 + thisDigit *)
    the_value := Int64.add (Int64.mul !the_value radix) thisDigit
  done;
  !the_value

let lex_hex_escape remainder lexbuf =
  let prefix = scan_hex_escape (Lexing.lexeme lexbuf) in
  prefix :: remainder lexbuf

let lex_oct_escape remainder lexbuf =
  let prefix = scan_oct_escape (Lexing.lexeme lexbuf) in
  prefix :: remainder lexbuf

let lex_simple_escape remainder lexbuf =
  let lexchar = Lexing.lexeme_char lexbuf 1 in
  let prefix = scan_escape lexchar in
  prefix :: remainder lexbuf

let lex_unescaped remainder lexbuf =
  let prefix = Int64.of_int (Char.code (Lexing.lexeme_char lexbuf 0)) in
  prefix :: remainder lexbuf

let lex_comment remainder lexbuf =
  let ch = Lexing.lexeme_char lexbuf 0 in
  let prefix = Int64.of_int (Char.code ch) in
  if ch = '\n' then E.newline();
  prefix :: remainder lexbuf

let make_char (i:int64):char =
  let min_val = Int64.zero in
  let max_val = Int64.of_int 255 in
  (* if i < 0 || i > 255 then error*)
  if compare i min_val < 0 || compare i max_val > 0 then begin
    let msg = Printf.sprintf "clexer:make_char: character 0x%Lx too big" i in
    error msg
  end;
  Char.chr (Int64.to_int i)


(* ISO standard locale-specific function to convert a wide character
 * into a sequence of normal characters. Here we work on strings.
 * We convert L"Hi" to "H\000i\000"
  matth: this seems unused.
let wbtowc wstr =
  let len = String.length wstr in
  let dest = String.make (len * 2) '\000' in
  for i = 0 to len-1 do
    dest.[i*2] <- wstr.[i] ;
  done ;
  dest
*)

(* This function converst the "Hi" in L"Hi" to { L'H', L'i', L'\0' }
  matth: this seems unused.
let wstr_to_warray wstr =
  let len = String.length wstr in
  let res = ref "{ " in
  for i = 0 to len-1 do
    res := !res ^ (Printf.sprintf "L'%c', " wstr.[i])
  done ;
  res := !res ^ "}" ;
  !res
*)

(* Pragmas get explicit end-of-line tokens.
 * Elsewhere they are silently discarded as whitespace. *)
let pragmaLine = ref false

let annot_char = ref '@'
let annot_start_pos = ref Cabshelper.cabslu
let buf = Buffer.create 1024

let save_current_pos () =
  annot_start_pos := currentLoc ()

let make_annot s =
  let start = snd !annot_start_pos in
  match Logic_lexer.annot (start, s) with
    | Logic_ptree.Adecl d -> DECL d
    | Logic_ptree.Aspec -> SPEC (start,s)
        (* At this point, we only have identified a function spec. Complete
           parsing of the annotation will only occur in the cparser.mly rule.
         *)
    | Logic_ptree.Acode_annot (loc,a) -> CODE_ANNOT (a, loc)
    | Logic_ptree.Aloop_annot (loc,a) -> LOOP_ANNOT (a,loc)
    | Logic_ptree.Aattribute_annot (loc,a) -> ATTRIBUTE_ANNOT (a, loc)

# 390 "cil/src/frontc/clexer.ml"
let __ocaml_lex_tables = {
  Lexing.lex_base = 
   "\000\000\180\255\181\255\091\000\113\000\194\000\193\255\194\255\
    \195\255\196\255\197\255\198\255\199\255\200\255\201\255\202\255\
    \203\255\035\000\067\000\070\000\071\000\177\000\092\000\072\000\
    \073\000\090\000\093\000\121\000\015\001\027\001\065\001\244\255\
    \138\001\246\255\225\001\248\255\001\000\250\255\209\000\178\000\
    \233\255\127\000\145\000\254\255\252\255\249\255\005\000\056\002\
    \143\002\230\002\061\003\148\003\235\003\066\004\153\004\240\004\
    \071\005\158\005\245\005\076\006\163\006\250\006\081\007\168\007\
    \255\007\086\008\173\008\004\009\091\009\178\009\009\010\096\010\
    \183\010\014\011\243\255\245\255\091\011\106\001\125\011\084\000\
    \028\001\137\011\117\011\191\011\095\000\154\011\189\011\183\011\
    \185\000\079\001\100\001\039\001\098\000\186\000\203\011\152\011\
    \103\011\131\000\242\255\006\012\017\012\046\012\056\012\038\001\
    \085\012\214\001\095\012\045\002\156\012\194\012\130\000\034\001\
    \141\000\067\001\124\012\233\012\132\002\241\255\168\000\109\000\
    \111\000\172\000\008\013\041\013\073\013\219\002\181\000\237\255\
    \223\255\116\000\131\000\121\000\003\000\238\255\217\255\236\255\
    \215\255\216\255\235\255\234\255\232\255\207\255\231\255\208\255\
    \230\255\229\255\222\255\179\000\228\255\221\255\191\000\227\255\
    \224\255\117\013\204\013\035\014\122\014\209\014\160\000\168\000\
    \178\000\204\000\001\001\245\000\241\000\255\000\247\000\247\000\
    \253\000\038\001\245\000\011\001\010\001\187\255\025\001\031\001\
    \026\001\035\001\033\001\042\001\051\001\186\255\055\001\054\001\
    \185\255\062\001\077\001\184\255\132\001\255\255\190\001\253\255\
    \134\001\021\002\067\015\141\001\093\003\251\255\072\015\083\001\
    \093\001\082\015\180\003\145\001\126\001\079\011\099\011\199\001\
    \153\001\122\001\220\001\173\001\184\001\189\001\185\001\180\001\
    \220\001\011\004\131\015\207\001\211\001\254\001\002\002\255\001\
    \000\002\004\002\016\002\018\002\052\002\051\002\023\002\033\002\
    \039\002\045\002\091\002\099\002\251\255\096\002\137\002\104\002\
    \096\002\091\002\135\002\119\002\131\002\126\002\129\002\172\002\
    \184\002\179\002\181\002\183\002\192\002\199\002\206\002\221\002\
    \224\002\208\002\221\002\243\002\245\002\000\003\000\003\001\003\
    \255\002\253\002\007\003\025\003\020\003\002\003\007\003\014\003\
    \055\003\044\003\053\003\072\003\062\003\088\003\078\003\082\003\
    \085\003\080\003\080\003\096\003\098\003\099\003\113\003\095\003\
    \109\003\108\003\134\003\125\003\125\003\141\003\130\003\173\003\
    \176\003\158\003\170\003\166\003\168\003\013\012\212\010\098\004\
    \042\011\028\004\029\004\153\010\180\012\142\004\155\015\210\015\
    \060\005\150\010\097\015\147\005\233\015\033\016\234\005\046\011\
    \108\006\157\003\214\011\186\003\199\003\182\003\189\003\095\004\
    \222\003\216\003\213\003\213\003\116\004\028\004\014\005\255\003\
    \251\003\250\003\252\003\118\004";
  Lexing.lex_backtrk = 
   "\255\255\255\255\255\255\073\000\075\000\073\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\035\000\037\000\036\000\051\000\049\000\050\000\045\000\
    \043\000\042\000\041\000\046\000\063\000\016\000\016\000\255\255\
    \073\000\255\255\073\000\255\255\075\000\255\255\004\000\044\000\
    \255\255\002\000\000\000\255\255\255\255\255\255\255\255\073\000\
    \073\000\073\000\073\000\073\000\073\000\008\000\073\000\073\000\
    \073\000\073\000\065\000\073\000\073\000\073\000\073\000\066\000\
    \073\000\073\000\073\000\073\000\073\000\073\000\073\000\073\000\
    \073\000\072\000\255\255\255\255\255\255\255\255\013\000\255\255\
    \016\000\016\000\016\000\015\000\255\255\015\000\015\000\015\000\
    \015\000\015\000\015\000\015\000\255\255\016\000\016\000\016\000\
    \016\000\255\255\255\255\255\255\013\000\255\255\013\000\255\255\
    \013\000\255\255\013\000\255\255\255\255\014\000\255\255\014\000\
    \014\000\014\000\255\255\013\000\255\255\255\255\014\000\014\000\
    \014\000\255\255\255\255\255\255\013\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\029\000\255\255\255\255\030\000\255\255\
    \255\255\073\000\073\000\073\000\073\000\064\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\067\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\001\000\255\255\255\255\255\255\
    \002\000\255\255\255\255\006\000\001\000\255\255\255\255\006\000\
    \006\000\002\000\001\000\255\255\002\000\002\000\002\000\002\000\
    \002\000\002\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\005\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\003\000\001\000\
    \255\255\255\255\255\255\255\255\005\000\002\000\003\000\001\000\
    \002\000\255\255\005\000\002\000\003\000\001\000\002\000\255\255\
    \000\000\255\255\255\255\002\000\255\255\255\255\255\255\003\000\
    \255\255\255\255\255\255\255\255\255\255\003\000\002\000\255\255\
    \255\255\255\255\255\255\255\255";
  Lexing.lex_default = 
   "\001\000\000\000\000\000\255\255\255\255\255\255\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\000\000\
    \255\255\000\000\255\255\000\000\255\255\000\000\255\255\255\255\
    \000\000\044\000\043\000\000\000\000\000\000\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\000\000\000\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\000\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\000\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\000\000\
    \000\000\255\255\255\255\255\255\255\255\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\255\255\000\000\000\000\255\255\000\000\
    \000\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\000\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\000\000\255\255\255\255\
    \000\000\255\255\255\255\000\000\255\255\000\000\191\000\000\000\
    \255\255\043\000\045\000\255\255\255\255\000\000\045\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\000\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\044\000\048\001\255\255\
    \048\001\191\000\043\000\037\000\044\000\255\255\255\255\255\255\
    \255\255\037\000\044\000\255\255\255\255\255\255\255\255\191\000\
    \255\255\043\000\191\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\044\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\043\000";
  Lexing.lex_trans = 
   "\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\038\000\037\000\045\000\038\000\038\000\046\000\045\000\
    \000\000\000\000\046\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\038\000\000\000\000\000\000\000\000\000\000\000\
    \038\000\027\000\031\000\035\000\133\000\023\000\021\000\033\000\
    \009\000\008\000\024\000\026\000\006\000\025\000\028\000\039\000\
    \030\000\029\000\029\000\029\000\029\000\029\000\029\000\029\000\
    \029\000\029\000\015\000\007\000\019\000\017\000\018\000\016\000\
    \004\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\032\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\011\000\036\000\010\000\020\000\034\000\
    \152\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\005\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\013\000\022\000\012\000\014\000\003\000\
    \149\000\150\000\147\000\146\000\145\000\140\000\139\000\137\000\
    \134\000\255\255\097\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\092\000\088\000\138\000\
    \136\000\142\000\135\000\255\255\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\128\000\093\000\
    \121\000\117\000\003\000\255\255\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\159\000\143\000\
    \141\000\118\000\038\000\117\000\042\000\038\000\038\000\158\000\
    \117\000\041\000\117\000\127\000\160\000\161\000\003\000\078\000\
    \078\000\130\000\129\000\038\000\131\000\132\000\144\000\040\000\
    \148\000\038\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\151\000\117\000\077\000\077\000\
    \002\000\185\000\117\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\117\000\077\000\077\000\
    \182\000\003\000\174\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\153\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\126\000\162\000\100\000\
    \100\000\100\000\100\000\100\000\100\000\100\000\100\000\100\000\
    \100\000\078\000\078\000\029\000\029\000\029\000\029\000\029\000\
    \029\000\029\000\029\000\029\000\029\000\078\000\102\000\102\000\
    \102\000\102\000\102\000\102\000\102\000\102\000\102\000\102\000\
    \077\000\077\000\163\000\164\000\165\000\166\000\167\000\081\000\
    \095\000\168\000\169\000\170\000\077\000\171\000\119\000\078\000\
    \080\000\083\000\083\000\083\000\083\000\083\000\083\000\083\000\
    \083\000\029\000\029\000\172\000\173\000\078\000\175\000\255\255\
    \077\000\077\000\176\000\177\000\079\000\079\000\077\000\082\000\
    \096\000\178\000\179\000\110\000\077\000\081\000\120\000\180\000\
    \080\000\255\255\078\000\088\000\077\000\107\000\080\000\107\000\
    \117\000\076\000\106\000\106\000\106\000\106\000\106\000\106\000\
    \106\000\106\000\106\000\106\000\088\000\181\000\077\000\183\000\
    \184\000\077\000\079\000\186\000\074\000\082\000\003\000\118\000\
    \088\000\075\000\187\000\189\000\077\000\189\000\080\000\197\000\
    \117\000\076\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\088\000\213\000\211\000\210\000\
    \043\000\077\000\208\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\191\000\191\000\203\000\
    \192\000\003\000\209\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\104\000\104\000\
    \104\000\104\000\104\000\104\000\104\000\104\000\104\000\104\000\
    \191\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\212\000\191\000\044\000\214\000\189\000\
    \215\000\216\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\048\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\191\000\217\000\039\001\026\001\
    \047\000\019\001\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\106\000\106\000\106\000\
    \106\000\106\000\106\000\106\000\106\000\106\000\106\000\005\001\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\252\000\251\000\247\000\242\000\239\000\238\000\
    \237\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\232\000\231\000\233\000\234\000\003\000\
    \230\000\056\000\003\000\003\000\003\000\054\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \055\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\115\000\115\000\115\000\115\000\
    \115\000\115\000\115\000\115\000\115\000\115\000\255\255\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\235\000\236\000\236\000\236\000\240\000\241\000\236\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\243\000\244\000\245\000\246\000\003\000\236\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\049\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\124\000\124\000\124\000\124\000\124\000\
    \124\000\124\000\124\000\124\000\124\000\189\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \248\000\249\000\250\000\236\000\236\000\253\000\254\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\255\000\000\001\001\001\002\001\003\000\003\001\050\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\004\001\236\000\006\001\007\001\196\000\008\001\
    \009\001\196\000\196\000\010\001\011\001\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\196\000\
    \012\001\013\001\014\001\015\001\016\001\196\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \017\001\018\001\236\000\020\001\003\000\021\001\003\000\003\000\
    \003\000\003\000\003\000\003\000\051\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\022\001\023\001\024\001\025\001\202\000\236\000\027\001\
    \202\000\202\000\028\001\029\001\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\202\000\030\001\
    \031\001\032\001\033\001\034\001\202\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\035\001\
    \036\001\037\001\038\001\003\000\236\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\052\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \040\001\041\001\042\001\043\001\218\000\044\001\236\000\218\000\
    \218\000\068\001\189\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\218\000\189\000\189\000\
    \069\001\070\001\043\000\218\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\073\001\074\001\
    \075\001\189\000\003\000\189\000\053\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\080\001\
    \043\000\191\000\081\001\047\001\043\000\082\001\047\001\047\001\
    \189\000\000\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\047\001\000\000\191\000\043\000\
    \189\000\000\000\047\001\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\255\255\077\001\043\000\
    \000\000\003\000\000\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\056\001\056\001\
    \056\001\056\001\056\001\056\001\056\001\056\001\072\001\000\000\
    \000\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\000\000\000\000\000\000\000\000\
    \003\000\000\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\064\000\003\000\003\000\003\000\000\000\000\000\043\000\
    \000\000\000\000\000\000\043\000\043\000\255\255\000\000\000\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\000\000\000\000\000\000\000\000\043\000\000\000\
    \000\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\000\000\000\000\000\000\043\000\003\000\
    \000\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\059\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\191\000\191\000\191\000\191\000\
    \191\000\191\000\191\000\191\000\043\000\079\001\255\255\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\000\000\000\000\000\000\000\000\003\000\000\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\057\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\062\001\062\001\062\001\062\001\062\001\
    \062\001\062\001\062\001\000\000\000\000\000\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\000\000\000\000\000\000\000\000\003\000\000\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\058\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\191\000\191\000\191\000\191\000\191\000\191\000\
    \191\000\191\000\000\000\000\000\000\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \000\000\000\000\000\000\000\000\003\000\000\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\000\000\000\000\000\000\000\000\064\001\000\000\000\000\
    \064\001\064\001\000\000\000\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\064\001\000\000\
    \000\000\000\000\000\000\000\000\064\001\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\000\000\
    \000\000\000\000\000\000\003\000\000\000\060\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\000\000\000\000\
    \000\000\000\000\003\000\000\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\061\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\000\000\000\000\000\000\
    \000\000\003\000\000\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\062\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\000\000\000\000\000\000\000\000\
    \003\000\000\000\063\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\000\000\000\000\000\000\000\000\003\000\
    \000\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\000\000\000\000\000\000\000\000\003\000\000\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\065\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\000\000\000\000\000\000\000\000\003\000\000\000\003\000\
    \003\000\003\000\003\000\066\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \000\000\000\000\000\000\000\000\003\000\000\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\067\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\000\000\
    \000\000\000\000\000\000\003\000\000\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\068\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\000\000\000\000\
    \000\000\000\000\003\000\000\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\069\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\000\000\000\000\000\000\
    \000\000\003\000\000\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\070\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\000\000\000\000\000\000\000\000\
    \003\000\000\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\071\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\189\000\000\000\189\000\000\000\072\000\
    \000\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\000\000\255\255\000\000\000\000\
    \255\255\000\000\000\000\000\000\000\000\000\000\000\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\000\000\058\001\000\000\000\000\052\001\191\000\000\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\000\000\000\000\000\000\000\000\073\000\000\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\255\255\000\000\000\000\255\255\064\001\
    \000\000\000\000\064\001\064\001\000\000\000\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \064\001\000\000\000\000\000\000\191\000\000\000\064\001\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\000\000\000\000\000\000\000\000\003\000\000\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\108\000\000\000\109\000\109\000\109\000\109\000\109\000\
    \109\000\109\000\109\000\109\000\109\000\078\000\197\000\000\000\
    \000\000\197\000\000\000\207\000\109\000\109\000\109\000\109\000\
    \109\000\109\000\000\000\078\000\191\000\000\000\000\000\000\000\
    \000\000\043\000\000\000\000\000\077\000\100\000\100\000\100\000\
    \100\000\100\000\100\000\100\000\100\000\100\000\100\000\078\000\
    \191\000\000\000\077\000\000\000\109\000\109\000\109\000\109\000\
    \109\000\109\000\099\000\098\000\191\000\000\000\078\000\000\000\
    \078\000\098\000\093\000\000\000\077\000\000\000\077\000\207\000\
    \000\000\000\000\000\000\093\000\255\255\094\000\000\000\000\000\
    \191\000\000\000\077\000\000\000\000\000\077\000\093\000\077\000\
    \189\000\094\000\099\000\098\000\093\000\078\000\090\000\000\000\
    \000\000\098\000\093\000\078\000\000\000\078\000\077\000\083\000\
    \083\000\083\000\083\000\083\000\083\000\083\000\083\000\029\000\
    \029\000\078\000\000\000\000\000\077\000\077\000\093\000\077\000\
    \000\000\000\000\077\000\084\000\077\000\000\000\091\000\000\000\
    \000\000\089\000\000\000\086\000\088\000\000\000\000\000\000\000\
    \077\000\189\000\088\000\000\000\085\000\000\000\047\001\189\000\
    \000\000\047\001\047\001\000\000\077\000\000\000\000\000\000\000\
    \093\000\000\000\077\000\089\000\077\000\000\000\000\000\047\001\
    \084\000\000\000\255\255\087\000\088\000\047\001\255\255\046\001\
    \077\000\105\000\088\000\105\000\085\000\067\001\104\000\104\000\
    \104\000\104\000\104\000\104\000\104\000\104\000\104\000\104\000\
    \093\000\100\000\100\000\100\000\100\000\100\000\100\000\100\000\
    \100\000\100\000\100\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\189\000\000\000\000\000\101\000\098\000\
    \000\000\103\000\000\000\103\000\000\000\098\000\102\000\102\000\
    \102\000\102\000\102\000\102\000\102\000\102\000\102\000\102\000\
    \102\000\102\000\102\000\102\000\102\000\102\000\102\000\102\000\
    \102\000\102\000\000\000\000\000\000\000\000\000\101\000\098\000\
    \000\000\000\000\000\000\000\000\000\000\098\000\098\000\000\000\
    \000\000\000\000\000\000\000\000\098\000\104\000\104\000\104\000\
    \104\000\104\000\104\000\104\000\104\000\104\000\104\000\106\000\
    \106\000\106\000\106\000\106\000\106\000\106\000\106\000\106\000\
    \106\000\000\000\000\000\098\000\000\000\000\000\098\000\000\000\
    \000\000\098\000\000\000\000\000\098\000\098\000\000\000\116\000\
    \000\000\116\000\000\000\098\000\115\000\115\000\115\000\115\000\
    \115\000\115\000\115\000\115\000\115\000\115\000\000\000\000\000\
    \000\000\000\000\000\000\098\000\000\000\000\000\000\000\000\000\
    \000\000\098\000\000\000\000\000\000\000\098\000\000\000\000\000\
    \000\000\000\000\000\000\098\000\122\000\122\000\122\000\122\000\
    \122\000\122\000\122\000\122\000\122\000\122\000\255\255\000\000\
    \000\000\000\000\000\000\000\000\000\000\122\000\122\000\122\000\
    \122\000\122\000\122\000\000\000\053\001\053\001\053\001\053\001\
    \053\001\053\001\053\001\053\001\000\000\000\000\000\000\000\000\
    \108\000\000\000\109\000\109\000\109\000\109\000\109\000\109\000\
    \109\000\109\000\109\000\109\000\000\000\122\000\122\000\122\000\
    \122\000\122\000\122\000\109\000\109\000\109\000\109\000\109\000\
    \109\000\000\000\000\000\000\000\054\001\255\255\112\000\000\000\
    \000\000\000\000\114\000\000\000\000\000\000\000\000\000\111\000\
    \000\000\115\000\115\000\115\000\115\000\115\000\115\000\115\000\
    \115\000\115\000\115\000\109\000\109\000\109\000\109\000\109\000\
    \109\000\000\000\000\000\110\000\054\001\000\000\113\000\098\000\
    \000\000\000\000\114\000\000\000\000\000\098\000\000\000\111\000\
    \122\000\122\000\122\000\122\000\122\000\122\000\122\000\122\000\
    \122\000\122\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\122\000\122\000\122\000\122\000\122\000\122\000\098\000\
    \000\000\000\000\000\000\000\000\125\000\098\000\125\000\000\000\
    \123\000\124\000\124\000\124\000\124\000\124\000\124\000\124\000\
    \124\000\124\000\124\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\122\000\122\000\122\000\122\000\122\000\122\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \123\000\124\000\124\000\124\000\124\000\124\000\124\000\124\000\
    \124\000\124\000\124\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\098\000\
    \000\000\000\000\000\000\000\000\000\000\098\000\000\000\000\000\
    \000\000\003\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\098\000\
    \000\000\000\000\000\000\000\000\255\255\098\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \000\000\000\000\000\000\000\000\003\000\000\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\154\000\
    \003\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\000\000\
    \000\000\000\000\000\000\003\000\000\000\003\000\003\000\003\000\
    \003\000\155\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\000\000\000\000\
    \000\000\000\000\003\000\000\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\156\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\000\000\000\000\000\000\
    \000\000\003\000\000\000\003\000\003\000\003\000\003\000\003\000\
    \157\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\000\000\000\000\000\000\000\000\
    \003\000\000\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\196\000\189\000\000\000\196\000\
    \196\000\202\000\189\000\000\000\202\000\202\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\196\000\000\000\000\000\
    \000\000\000\000\202\000\196\000\000\000\037\000\000\000\000\000\
    \202\000\000\000\000\000\191\000\044\000\000\000\000\000\000\000\
    \000\000\000\000\195\000\000\000\000\000\000\000\000\000\000\000\
    \201\000\201\000\201\000\201\000\201\000\201\000\201\000\201\000\
    \201\000\201\000\201\000\201\000\201\000\201\000\201\000\201\000\
    \201\000\201\000\201\000\201\000\218\000\000\000\000\000\218\000\
    \218\000\059\001\059\001\059\001\059\001\059\001\059\001\059\001\
    \059\001\000\000\000\000\000\000\000\000\218\000\205\000\000\000\
    \000\000\000\000\000\000\218\000\000\000\000\000\000\000\204\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\200\000\000\000\000\000\000\000\
    \199\000\060\001\000\000\203\000\000\000\000\000\206\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\204\000\
    \000\000\000\000\228\000\055\001\055\001\055\001\055\001\055\001\
    \055\001\055\001\055\001\055\001\055\001\000\000\000\000\221\000\
    \000\000\060\001\000\000\000\000\055\001\055\001\055\001\055\001\
    \055\001\055\001\000\000\000\000\224\000\000\000\000\000\000\000\
    \000\000\000\000\220\000\000\000\227\000\000\000\000\000\219\000\
    \000\000\000\000\225\000\000\000\000\000\222\000\226\000\000\000\
    \223\000\000\000\229\000\000\000\055\001\055\001\055\001\055\001\
    \055\001\055\001\055\001\055\001\055\001\055\001\055\001\055\001\
    \055\001\055\001\055\001\055\001\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\055\001\055\001\055\001\055\001\055\001\
    \055\001\061\001\061\001\061\001\061\001\061\001\061\001\061\001\
    \061\001\061\001\061\001\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\061\001\061\001\061\001\061\001\061\001\061\001\
    \000\000\000\000\000\000\055\001\055\001\055\001\055\001\055\001\
    \055\001\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\255\255\000\000\000\000\000\000\000\000\
    \255\255\000\000\061\001\061\001\061\001\061\001\061\001\061\001\
    \000\000\061\001\061\001\061\001\061\001\061\001\061\001\061\001\
    \061\001\061\001\061\001\000\000\000\000\000\000\000\000\000\000\
    \000\000\255\255\061\001\061\001\061\001\061\001\061\001\061\001\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\061\001\061\001\061\001\061\001\061\001\061\001\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000";
  Lexing.lex_check = 
   "\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\000\000\000\000\036\000\000\000\000\000\036\000\046\000\
    \255\255\255\255\046\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\000\000\255\255\255\255\255\255\255\255\255\255\
    \000\000\000\000\000\000\000\000\132\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \017\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\003\000\
    \018\000\018\000\019\000\019\000\020\000\023\000\024\000\025\000\
    \026\000\041\000\079\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\084\000\092\000\025\000\
    \025\000\022\000\026\000\042\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\027\000\097\000\
    \110\000\119\000\003\000\042\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\004\000\021\000\
    \022\000\112\000\038\000\120\000\039\000\038\000\038\000\004\000\
    \121\000\039\000\112\000\126\000\004\000\004\000\005\000\088\000\
    \093\000\129\000\027\000\038\000\130\000\131\000\021\000\039\000\
    \147\000\038\000\005\000\005\000\005\000\005\000\005\000\005\000\
    \005\000\005\000\005\000\005\000\150\000\118\000\088\000\093\000\
    \000\000\158\000\112\000\005\000\005\000\005\000\005\000\005\000\
    \005\000\005\000\005\000\005\000\005\000\005\000\005\000\005\000\
    \005\000\005\000\005\000\005\000\005\000\005\000\005\000\005\000\
    \005\000\005\000\005\000\005\000\005\000\118\000\088\000\093\000\
    \159\000\005\000\160\000\005\000\005\000\005\000\005\000\005\000\
    \005\000\005\000\005\000\005\000\005\000\005\000\005\000\005\000\
    \005\000\005\000\005\000\005\000\005\000\005\000\005\000\005\000\
    \005\000\005\000\005\000\005\000\005\000\028\000\161\000\028\000\
    \028\000\028\000\028\000\028\000\028\000\028\000\028\000\028\000\
    \028\000\029\000\080\000\029\000\029\000\029\000\029\000\029\000\
    \029\000\029\000\029\000\029\000\029\000\091\000\103\000\103\000\
    \103\000\103\000\103\000\103\000\103\000\103\000\103\000\103\000\
    \029\000\080\000\162\000\163\000\164\000\165\000\166\000\029\000\
    \080\000\167\000\168\000\169\000\091\000\170\000\111\000\030\000\
    \029\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\171\000\172\000\089\000\174\000\041\000\
    \029\000\080\000\175\000\176\000\029\000\080\000\030\000\029\000\
    \080\000\177\000\178\000\111\000\091\000\030\000\111\000\179\000\
    \029\000\042\000\090\000\091\000\089\000\077\000\030\000\077\000\
    \113\000\030\000\077\000\077\000\077\000\077\000\077\000\077\000\
    \077\000\077\000\077\000\077\000\089\000\180\000\030\000\182\000\
    \183\000\090\000\030\000\185\000\032\000\030\000\032\000\113\000\
    \090\000\032\000\186\000\188\000\089\000\192\000\030\000\195\000\
    \113\000\030\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\089\000\199\000\200\000\203\000\
    \190\000\090\000\204\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\208\000\209\000\204\000\
    \190\000\032\000\204\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\034\000\105\000\105\000\
    \105\000\105\000\105\000\105\000\105\000\105\000\105\000\105\000\
    \210\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\211\000\207\000\212\000\213\000\193\000\
    \214\000\215\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\207\000\216\000\219\000\220\000\
    \034\000\221\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\047\000\107\000\107\000\107\000\
    \107\000\107\000\107\000\107\000\107\000\107\000\107\000\222\000\
    \047\000\047\000\047\000\047\000\047\000\047\000\047\000\047\000\
    \047\000\047\000\223\000\224\000\225\000\226\000\227\000\228\000\
    \230\000\047\000\047\000\047\000\047\000\047\000\047\000\047\000\
    \047\000\047\000\047\000\047\000\047\000\047\000\047\000\047\000\
    \047\000\047\000\047\000\047\000\047\000\047\000\047\000\047\000\
    \047\000\047\000\047\000\231\000\229\000\232\000\233\000\047\000\
    \229\000\047\000\047\000\047\000\047\000\047\000\047\000\047\000\
    \047\000\047\000\047\000\047\000\047\000\047\000\047\000\047\000\
    \047\000\047\000\047\000\047\000\047\000\047\000\047\000\047\000\
    \047\000\047\000\047\000\048\000\116\000\116\000\116\000\116\000\
    \116\000\116\000\116\000\116\000\116\000\116\000\190\000\048\000\
    \048\000\048\000\048\000\048\000\048\000\048\000\048\000\048\000\
    \048\000\234\000\235\000\237\000\238\000\239\000\240\000\241\000\
    \048\000\048\000\048\000\048\000\048\000\048\000\048\000\048\000\
    \048\000\048\000\048\000\048\000\048\000\048\000\048\000\048\000\
    \048\000\048\000\048\000\048\000\048\000\048\000\048\000\048\000\
    \048\000\048\000\242\000\243\000\244\000\245\000\048\000\246\000\
    \048\000\048\000\048\000\048\000\048\000\048\000\048\000\048\000\
    \048\000\048\000\048\000\048\000\048\000\048\000\048\000\048\000\
    \048\000\048\000\048\000\048\000\048\000\048\000\048\000\048\000\
    \048\000\048\000\049\000\125\000\125\000\125\000\125\000\125\000\
    \125\000\125\000\125\000\125\000\125\000\193\000\049\000\049\000\
    \049\000\049\000\049\000\049\000\049\000\049\000\049\000\049\000\
    \247\000\248\000\249\000\250\000\251\000\252\000\253\000\049\000\
    \049\000\049\000\049\000\049\000\049\000\049\000\049\000\049\000\
    \049\000\049\000\049\000\049\000\049\000\049\000\049\000\049\000\
    \049\000\049\000\049\000\049\000\049\000\049\000\049\000\049\000\
    \049\000\254\000\255\000\000\001\001\001\049\000\002\001\049\000\
    \049\000\049\000\049\000\049\000\049\000\049\000\049\000\049\000\
    \049\000\049\000\049\000\049\000\049\000\049\000\049\000\049\000\
    \049\000\049\000\049\000\049\000\049\000\049\000\049\000\049\000\
    \049\000\050\000\003\001\004\001\005\001\006\001\196\000\007\001\
    \008\001\196\000\196\000\009\001\010\001\050\000\050\000\050\000\
    \050\000\050\000\050\000\050\000\050\000\050\000\050\000\196\000\
    \011\001\012\001\013\001\014\001\015\001\196\000\050\000\050\000\
    \050\000\050\000\050\000\050\000\050\000\050\000\050\000\050\000\
    \050\000\050\000\050\000\050\000\050\000\050\000\050\000\050\000\
    \050\000\050\000\050\000\050\000\050\000\050\000\050\000\050\000\
    \016\001\017\001\018\001\019\001\050\000\020\001\050\000\050\000\
    \050\000\050\000\050\000\050\000\050\000\050\000\050\000\050\000\
    \050\000\050\000\050\000\050\000\050\000\050\000\050\000\050\000\
    \050\000\050\000\050\000\050\000\050\000\050\000\050\000\050\000\
    \051\000\021\001\022\001\023\001\024\001\202\000\025\001\026\001\
    \202\000\202\000\027\001\028\001\051\000\051\000\051\000\051\000\
    \051\000\051\000\051\000\051\000\051\000\051\000\202\000\029\001\
    \030\001\031\001\032\001\033\001\202\000\051\000\051\000\051\000\
    \051\000\051\000\051\000\051\000\051\000\051\000\051\000\051\000\
    \051\000\051\000\051\000\051\000\051\000\051\000\051\000\051\000\
    \051\000\051\000\051\000\051\000\051\000\051\000\051\000\034\001\
    \035\001\036\001\037\001\051\000\038\001\051\000\051\000\051\000\
    \051\000\051\000\051\000\051\000\051\000\051\000\051\000\051\000\
    \051\000\051\000\051\000\051\000\051\000\051\000\051\000\051\000\
    \051\000\051\000\051\000\051\000\051\000\051\000\051\000\052\000\
    \039\001\040\001\041\001\042\001\217\000\043\001\044\001\217\000\
    \217\000\067\001\065\001\052\000\052\000\052\000\052\000\052\000\
    \052\000\052\000\052\000\052\000\052\000\217\000\049\001\050\001\
    \068\001\069\001\070\001\217\000\052\000\052\000\052\000\052\000\
    \052\000\052\000\052\000\052\000\052\000\052\000\052\000\052\000\
    \052\000\052\000\052\000\052\000\052\000\052\000\052\000\052\000\
    \052\000\052\000\052\000\052\000\052\000\052\000\072\001\073\001\
    \074\001\075\001\052\000\077\001\052\000\052\000\052\000\052\000\
    \052\000\052\000\052\000\052\000\052\000\052\000\052\000\052\000\
    \052\000\052\000\052\000\052\000\052\000\052\000\052\000\052\000\
    \052\000\052\000\052\000\052\000\052\000\052\000\053\000\079\001\
    \071\001\071\001\080\001\047\001\071\001\081\001\047\001\047\001\
    \082\001\255\255\053\000\053\000\053\000\053\000\053\000\053\000\
    \053\000\053\000\053\000\053\000\047\001\255\255\076\001\071\001\
    \083\001\255\255\047\001\053\000\053\000\053\000\053\000\053\000\
    \053\000\053\000\053\000\053\000\053\000\053\000\053\000\053\000\
    \053\000\053\000\053\000\053\000\053\000\053\000\053\000\053\000\
    \053\000\053\000\053\000\053\000\053\000\065\001\076\001\071\001\
    \255\255\053\000\255\255\053\000\053\000\053\000\053\000\053\000\
    \053\000\053\000\053\000\053\000\053\000\053\000\053\000\053\000\
    \053\000\053\000\053\000\053\000\053\000\053\000\053\000\053\000\
    \053\000\053\000\053\000\053\000\053\000\054\000\053\001\053\001\
    \053\001\053\001\053\001\053\001\053\001\053\001\071\001\255\255\
    \255\255\054\000\054\000\054\000\054\000\054\000\054\000\054\000\
    \054\000\054\000\054\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\054\000\054\000\054\000\054\000\054\000\054\000\
    \054\000\054\000\054\000\054\000\054\000\054\000\054\000\054\000\
    \054\000\054\000\054\000\054\000\054\000\054\000\054\000\054\000\
    \054\000\054\000\054\000\054\000\255\255\255\255\255\255\255\255\
    \054\000\255\255\054\000\054\000\054\000\054\000\054\000\054\000\
    \054\000\054\000\054\000\054\000\054\000\054\000\054\000\054\000\
    \054\000\054\000\054\000\054\000\054\000\054\000\054\000\054\000\
    \054\000\054\000\054\000\054\000\055\000\255\255\255\255\078\001\
    \255\255\255\255\255\255\078\001\049\001\050\001\255\255\255\255\
    \055\000\055\000\055\000\055\000\055\000\055\000\055\000\055\000\
    \055\000\055\000\255\255\255\255\255\255\255\255\078\001\255\255\
    \255\255\055\000\055\000\055\000\055\000\055\000\055\000\055\000\
    \055\000\055\000\055\000\055\000\055\000\055\000\055\000\055\000\
    \055\000\055\000\055\000\055\000\055\000\055\000\055\000\055\000\
    \055\000\055\000\055\000\255\255\255\255\255\255\078\001\055\000\
    \255\255\055\000\055\000\055\000\055\000\055\000\055\000\055\000\
    \055\000\055\000\055\000\055\000\055\000\055\000\055\000\055\000\
    \055\000\055\000\055\000\055\000\055\000\055\000\055\000\055\000\
    \055\000\055\000\055\000\056\000\056\001\056\001\056\001\056\001\
    \056\001\056\001\056\001\056\001\076\001\078\001\083\001\056\000\
    \056\000\056\000\056\000\056\000\056\000\056\000\056\000\056\000\
    \056\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \056\000\056\000\056\000\056\000\056\000\056\000\056\000\056\000\
    \056\000\056\000\056\000\056\000\056\000\056\000\056\000\056\000\
    \056\000\056\000\056\000\056\000\056\000\056\000\056\000\056\000\
    \056\000\056\000\255\255\255\255\255\255\255\255\056\000\255\255\
    \056\000\056\000\056\000\056\000\056\000\056\000\056\000\056\000\
    \056\000\056\000\056\000\056\000\056\000\056\000\056\000\056\000\
    \056\000\056\000\056\000\056\000\056\000\056\000\056\000\056\000\
    \056\000\056\000\057\000\059\001\059\001\059\001\059\001\059\001\
    \059\001\059\001\059\001\255\255\255\255\255\255\057\000\057\000\
    \057\000\057\000\057\000\057\000\057\000\057\000\057\000\057\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\057\000\
    \057\000\057\000\057\000\057\000\057\000\057\000\057\000\057\000\
    \057\000\057\000\057\000\057\000\057\000\057\000\057\000\057\000\
    \057\000\057\000\057\000\057\000\057\000\057\000\057\000\057\000\
    \057\000\255\255\255\255\255\255\255\255\057\000\255\255\057\000\
    \057\000\057\000\057\000\057\000\057\000\057\000\057\000\057\000\
    \057\000\057\000\057\000\057\000\057\000\057\000\057\000\057\000\
    \057\000\057\000\057\000\057\000\057\000\057\000\057\000\057\000\
    \057\000\058\000\062\001\062\001\062\001\062\001\062\001\062\001\
    \062\001\062\001\255\255\255\255\255\255\058\000\058\000\058\000\
    \058\000\058\000\058\000\058\000\058\000\058\000\058\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\058\000\058\000\
    \058\000\058\000\058\000\058\000\058\000\058\000\058\000\058\000\
    \058\000\058\000\058\000\058\000\058\000\058\000\058\000\058\000\
    \058\000\058\000\058\000\058\000\058\000\058\000\058\000\058\000\
    \255\255\255\255\255\255\255\255\058\000\255\255\058\000\058\000\
    \058\000\058\000\058\000\058\000\058\000\058\000\058\000\058\000\
    \058\000\058\000\058\000\058\000\058\000\058\000\058\000\058\000\
    \058\000\058\000\058\000\058\000\058\000\058\000\058\000\058\000\
    \059\000\255\255\255\255\255\255\255\255\064\001\255\255\255\255\
    \064\001\064\001\255\255\255\255\059\000\059\000\059\000\059\000\
    \059\000\059\000\059\000\059\000\059\000\059\000\064\001\255\255\
    \255\255\255\255\255\255\255\255\064\001\059\000\059\000\059\000\
    \059\000\059\000\059\000\059\000\059\000\059\000\059\000\059\000\
    \059\000\059\000\059\000\059\000\059\000\059\000\059\000\059\000\
    \059\000\059\000\059\000\059\000\059\000\059\000\059\000\255\255\
    \255\255\255\255\255\255\059\000\255\255\059\000\059\000\059\000\
    \059\000\059\000\059\000\059\000\059\000\059\000\059\000\059\000\
    \059\000\059\000\059\000\059\000\059\000\059\000\059\000\059\000\
    \059\000\059\000\059\000\059\000\059\000\059\000\059\000\060\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\060\000\060\000\060\000\060\000\060\000\
    \060\000\060\000\060\000\060\000\060\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\060\000\060\000\060\000\060\000\
    \060\000\060\000\060\000\060\000\060\000\060\000\060\000\060\000\
    \060\000\060\000\060\000\060\000\060\000\060\000\060\000\060\000\
    \060\000\060\000\060\000\060\000\060\000\060\000\255\255\255\255\
    \255\255\255\255\060\000\255\255\060\000\060\000\060\000\060\000\
    \060\000\060\000\060\000\060\000\060\000\060\000\060\000\060\000\
    \060\000\060\000\060\000\060\000\060\000\060\000\060\000\060\000\
    \060\000\060\000\060\000\060\000\060\000\060\000\061\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\061\000\061\000\061\000\061\000\061\000\061\000\
    \061\000\061\000\061\000\061\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\061\000\061\000\061\000\061\000\061\000\
    \061\000\061\000\061\000\061\000\061\000\061\000\061\000\061\000\
    \061\000\061\000\061\000\061\000\061\000\061\000\061\000\061\000\
    \061\000\061\000\061\000\061\000\061\000\255\255\255\255\255\255\
    \255\255\061\000\255\255\061\000\061\000\061\000\061\000\061\000\
    \061\000\061\000\061\000\061\000\061\000\061\000\061\000\061\000\
    \061\000\061\000\061\000\061\000\061\000\061\000\061\000\061\000\
    \061\000\061\000\061\000\061\000\061\000\062\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\062\000\062\000\062\000\062\000\062\000\062\000\062\000\
    \062\000\062\000\062\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\062\000\062\000\062\000\062\000\062\000\062\000\
    \062\000\062\000\062\000\062\000\062\000\062\000\062\000\062\000\
    \062\000\062\000\062\000\062\000\062\000\062\000\062\000\062\000\
    \062\000\062\000\062\000\062\000\255\255\255\255\255\255\255\255\
    \062\000\255\255\062\000\062\000\062\000\062\000\062\000\062\000\
    \062\000\062\000\062\000\062\000\062\000\062\000\062\000\062\000\
    \062\000\062\000\062\000\062\000\062\000\062\000\062\000\062\000\
    \062\000\062\000\062\000\062\000\063\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \063\000\063\000\063\000\063\000\063\000\063\000\063\000\063\000\
    \063\000\063\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\063\000\063\000\063\000\063\000\063\000\063\000\063\000\
    \063\000\063\000\063\000\063\000\063\000\063\000\063\000\063\000\
    \063\000\063\000\063\000\063\000\063\000\063\000\063\000\063\000\
    \063\000\063\000\063\000\255\255\255\255\255\255\255\255\063\000\
    \255\255\063\000\063\000\063\000\063\000\063\000\063\000\063\000\
    \063\000\063\000\063\000\063\000\063\000\063\000\063\000\063\000\
    \063\000\063\000\063\000\063\000\063\000\063\000\063\000\063\000\
    \063\000\063\000\063\000\064\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\064\000\
    \064\000\064\000\064\000\064\000\064\000\064\000\064\000\064\000\
    \064\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \064\000\064\000\064\000\064\000\064\000\064\000\064\000\064\000\
    \064\000\064\000\064\000\064\000\064\000\064\000\064\000\064\000\
    \064\000\064\000\064\000\064\000\064\000\064\000\064\000\064\000\
    \064\000\064\000\255\255\255\255\255\255\255\255\064\000\255\255\
    \064\000\064\000\064\000\064\000\064\000\064\000\064\000\064\000\
    \064\000\064\000\064\000\064\000\064\000\064\000\064\000\064\000\
    \064\000\064\000\064\000\064\000\064\000\064\000\064\000\064\000\
    \064\000\064\000\065\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\065\000\065\000\
    \065\000\065\000\065\000\065\000\065\000\065\000\065\000\065\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\065\000\
    \065\000\065\000\065\000\065\000\065\000\065\000\065\000\065\000\
    \065\000\065\000\065\000\065\000\065\000\065\000\065\000\065\000\
    \065\000\065\000\065\000\065\000\065\000\065\000\065\000\065\000\
    \065\000\255\255\255\255\255\255\255\255\065\000\255\255\065\000\
    \065\000\065\000\065\000\065\000\065\000\065\000\065\000\065\000\
    \065\000\065\000\065\000\065\000\065\000\065\000\065\000\065\000\
    \065\000\065\000\065\000\065\000\065\000\065\000\065\000\065\000\
    \065\000\066\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\066\000\066\000\066\000\
    \066\000\066\000\066\000\066\000\066\000\066\000\066\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\066\000\066\000\
    \066\000\066\000\066\000\066\000\066\000\066\000\066\000\066\000\
    \066\000\066\000\066\000\066\000\066\000\066\000\066\000\066\000\
    \066\000\066\000\066\000\066\000\066\000\066\000\066\000\066\000\
    \255\255\255\255\255\255\255\255\066\000\255\255\066\000\066\000\
    \066\000\066\000\066\000\066\000\066\000\066\000\066\000\066\000\
    \066\000\066\000\066\000\066\000\066\000\066\000\066\000\066\000\
    \066\000\066\000\066\000\066\000\066\000\066\000\066\000\066\000\
    \067\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\067\000\067\000\067\000\067\000\
    \067\000\067\000\067\000\067\000\067\000\067\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\067\000\067\000\067\000\
    \067\000\067\000\067\000\067\000\067\000\067\000\067\000\067\000\
    \067\000\067\000\067\000\067\000\067\000\067\000\067\000\067\000\
    \067\000\067\000\067\000\067\000\067\000\067\000\067\000\255\255\
    \255\255\255\255\255\255\067\000\255\255\067\000\067\000\067\000\
    \067\000\067\000\067\000\067\000\067\000\067\000\067\000\067\000\
    \067\000\067\000\067\000\067\000\067\000\067\000\067\000\067\000\
    \067\000\067\000\067\000\067\000\067\000\067\000\067\000\068\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\068\000\068\000\068\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\068\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\068\000\068\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\068\000\068\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\068\000\068\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\068\000\068\000\255\255\255\255\
    \255\255\255\255\068\000\255\255\068\000\068\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\068\000\068\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\068\000\068\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\068\000\068\000\069\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\069\000\069\000\069\000\069\000\069\000\069\000\
    \069\000\069\000\069\000\069\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\069\000\069\000\069\000\069\000\069\000\
    \069\000\069\000\069\000\069\000\069\000\069\000\069\000\069\000\
    \069\000\069\000\069\000\069\000\069\000\069\000\069\000\069\000\
    \069\000\069\000\069\000\069\000\069\000\255\255\255\255\255\255\
    \255\255\069\000\255\255\069\000\069\000\069\000\069\000\069\000\
    \069\000\069\000\069\000\069\000\069\000\069\000\069\000\069\000\
    \069\000\069\000\069\000\069\000\069\000\069\000\069\000\069\000\
    \069\000\069\000\069\000\069\000\069\000\070\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\070\000\070\000\070\000\070\000\070\000\070\000\070\000\
    \070\000\070\000\070\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\070\000\070\000\070\000\070\000\070\000\070\000\
    \070\000\070\000\070\000\070\000\070\000\070\000\070\000\070\000\
    \070\000\070\000\070\000\070\000\070\000\070\000\070\000\070\000\
    \070\000\070\000\070\000\070\000\255\255\255\255\255\255\255\255\
    \070\000\255\255\070\000\070\000\070\000\070\000\070\000\070\000\
    \070\000\070\000\070\000\070\000\070\000\070\000\070\000\070\000\
    \070\000\070\000\070\000\070\000\070\000\070\000\070\000\070\000\
    \070\000\070\000\070\000\070\000\071\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \071\000\071\000\071\000\071\000\071\000\071\000\071\000\071\000\
    \071\000\071\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\071\000\071\000\071\000\071\000\071\000\071\000\071\000\
    \071\000\071\000\071\000\071\000\071\000\071\000\071\000\071\000\
    \071\000\071\000\071\000\071\000\071\000\071\000\071\000\071\000\
    \071\000\071\000\071\000\051\001\255\255\057\001\255\255\071\000\
    \255\255\071\000\071\000\071\000\071\000\071\000\071\000\071\000\
    \071\000\071\000\071\000\071\000\071\000\071\000\071\000\071\000\
    \071\000\071\000\071\000\071\000\071\000\071\000\071\000\071\000\
    \071\000\071\000\071\000\072\000\255\255\046\001\255\255\255\255\
    \046\001\255\255\255\255\255\255\255\255\255\255\255\255\072\000\
    \072\000\072\000\072\000\072\000\072\000\072\000\072\000\072\000\
    \072\000\255\255\057\001\255\255\255\255\051\001\046\001\255\255\
    \072\000\072\000\072\000\072\000\072\000\072\000\072\000\072\000\
    \072\000\072\000\072\000\072\000\072\000\072\000\072\000\072\000\
    \072\000\072\000\072\000\072\000\072\000\072\000\072\000\072\000\
    \072\000\072\000\255\255\255\255\255\255\255\255\072\000\255\255\
    \072\000\072\000\072\000\072\000\072\000\072\000\072\000\072\000\
    \072\000\072\000\072\000\072\000\072\000\072\000\072\000\072\000\
    \072\000\072\000\072\000\072\000\072\000\072\000\072\000\072\000\
    \072\000\072\000\073\000\048\001\255\255\255\255\048\001\063\001\
    \255\255\255\255\063\001\063\001\255\255\255\255\073\000\073\000\
    \073\000\073\000\073\000\073\000\073\000\073\000\073\000\073\000\
    \063\001\255\255\255\255\255\255\048\001\255\255\063\001\073\000\
    \073\000\073\000\073\000\073\000\073\000\073\000\073\000\073\000\
    \073\000\073\000\073\000\073\000\073\000\073\000\073\000\073\000\
    \073\000\073\000\073\000\073\000\073\000\073\000\073\000\073\000\
    \073\000\255\255\255\255\255\255\255\255\073\000\255\255\073\000\
    \073\000\073\000\073\000\073\000\073\000\073\000\073\000\073\000\
    \073\000\073\000\073\000\073\000\073\000\073\000\073\000\073\000\
    \073\000\073\000\073\000\073\000\073\000\073\000\073\000\073\000\
    \073\000\076\000\255\255\076\000\076\000\076\000\076\000\076\000\
    \076\000\076\000\076\000\076\000\076\000\096\000\057\001\255\255\
    \255\255\051\001\255\255\205\000\076\000\076\000\076\000\076\000\
    \076\000\076\000\255\255\082\000\205\000\255\255\255\255\255\255\
    \255\255\063\001\255\255\255\255\096\000\078\000\078\000\078\000\
    \078\000\078\000\078\000\078\000\078\000\078\000\078\000\081\000\
    \206\000\255\255\082\000\255\255\076\000\076\000\076\000\076\000\
    \076\000\076\000\078\000\078\000\205\000\255\255\095\000\255\255\
    \085\000\078\000\082\000\255\255\096\000\255\255\081\000\206\000\
    \255\255\255\255\255\255\096\000\046\001\081\000\255\255\255\255\
    \206\000\255\255\082\000\255\255\255\255\095\000\081\000\085\000\
    \066\001\082\000\078\000\078\000\095\000\087\000\085\000\255\255\
    \255\255\078\000\082\000\086\000\255\255\083\000\081\000\083\000\
    \083\000\083\000\083\000\083\000\083\000\083\000\083\000\083\000\
    \083\000\094\000\255\255\255\255\087\000\095\000\081\000\085\000\
    \255\255\255\255\086\000\085\000\083\000\255\255\085\000\255\255\
    \255\255\086\000\255\255\083\000\087\000\255\255\255\255\255\255\
    \094\000\066\001\086\000\255\255\083\000\255\255\045\001\045\001\
    \255\255\045\001\045\001\255\255\087\000\255\255\255\255\255\255\
    \094\000\255\255\086\000\087\000\083\000\255\255\255\255\045\001\
    \083\000\255\255\048\001\083\000\087\000\045\001\063\001\045\001\
    \094\000\099\000\086\000\099\000\083\000\066\001\099\000\099\000\
    \099\000\099\000\099\000\099\000\099\000\099\000\099\000\099\000\
    \094\000\100\000\100\000\100\000\100\000\100\000\100\000\100\000\
    \100\000\100\000\100\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\066\001\255\255\255\255\100\000\100\000\
    \255\255\101\000\255\255\101\000\255\255\100\000\101\000\101\000\
    \101\000\101\000\101\000\101\000\101\000\101\000\101\000\101\000\
    \102\000\102\000\102\000\102\000\102\000\102\000\102\000\102\000\
    \102\000\102\000\255\255\255\255\255\255\255\255\100\000\100\000\
    \255\255\255\255\255\255\255\255\255\255\100\000\102\000\255\255\
    \255\255\255\255\255\255\255\255\102\000\104\000\104\000\104\000\
    \104\000\104\000\104\000\104\000\104\000\104\000\104\000\106\000\
    \106\000\106\000\106\000\106\000\106\000\106\000\106\000\106\000\
    \106\000\255\255\255\255\104\000\255\255\255\255\102\000\255\255\
    \255\255\104\000\255\255\255\255\102\000\106\000\255\255\114\000\
    \255\255\114\000\255\255\106\000\114\000\114\000\114\000\114\000\
    \114\000\114\000\114\000\114\000\114\000\114\000\255\255\255\255\
    \255\255\255\255\255\255\104\000\255\255\255\255\255\255\255\255\
    \255\255\104\000\255\255\255\255\255\255\106\000\255\255\255\255\
    \255\255\255\255\255\255\106\000\108\000\108\000\108\000\108\000\
    \108\000\108\000\108\000\108\000\108\000\108\000\066\001\255\255\
    \255\255\255\255\255\255\255\255\255\255\108\000\108\000\108\000\
    \108\000\108\000\108\000\255\255\052\001\052\001\052\001\052\001\
    \052\001\052\001\052\001\052\001\255\255\255\255\255\255\255\255\
    \109\000\255\255\109\000\109\000\109\000\109\000\109\000\109\000\
    \109\000\109\000\109\000\109\000\255\255\108\000\108\000\108\000\
    \108\000\108\000\108\000\109\000\109\000\109\000\109\000\109\000\
    \109\000\255\255\255\255\255\255\052\001\045\001\109\000\255\255\
    \255\255\255\255\109\000\255\255\255\255\255\255\255\255\109\000\
    \255\255\115\000\115\000\115\000\115\000\115\000\115\000\115\000\
    \115\000\115\000\115\000\109\000\109\000\109\000\109\000\109\000\
    \109\000\255\255\255\255\109\000\052\001\255\255\109\000\115\000\
    \255\255\255\255\109\000\255\255\255\255\115\000\255\255\109\000\
    \122\000\122\000\122\000\122\000\122\000\122\000\122\000\122\000\
    \122\000\122\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\122\000\122\000\122\000\122\000\122\000\122\000\115\000\
    \255\255\255\255\255\255\255\255\123\000\115\000\123\000\255\255\
    \122\000\123\000\123\000\123\000\123\000\123\000\123\000\123\000\
    \123\000\123\000\123\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\122\000\122\000\122\000\122\000\122\000\122\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \122\000\124\000\124\000\124\000\124\000\124\000\124\000\124\000\
    \124\000\124\000\124\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\124\000\
    \255\255\255\255\255\255\255\255\255\255\124\000\255\255\255\255\
    \255\255\153\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\153\000\153\000\153\000\
    \153\000\153\000\153\000\153\000\153\000\153\000\153\000\124\000\
    \255\255\255\255\255\255\255\255\052\001\124\000\153\000\153\000\
    \153\000\153\000\153\000\153\000\153\000\153\000\153\000\153\000\
    \153\000\153\000\153\000\153\000\153\000\153\000\153\000\153\000\
    \153\000\153\000\153\000\153\000\153\000\153\000\153\000\153\000\
    \255\255\255\255\255\255\255\255\153\000\255\255\153\000\153\000\
    \153\000\153\000\153\000\153\000\153\000\153\000\153\000\153\000\
    \153\000\153\000\153\000\153\000\153\000\153\000\153\000\153\000\
    \153\000\153\000\153\000\153\000\153\000\153\000\153\000\153\000\
    \154\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\154\000\154\000\154\000\154\000\
    \154\000\154\000\154\000\154\000\154\000\154\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\154\000\154\000\154\000\
    \154\000\154\000\154\000\154\000\154\000\154\000\154\000\154\000\
    \154\000\154\000\154\000\154\000\154\000\154\000\154\000\154\000\
    \154\000\154\000\154\000\154\000\154\000\154\000\154\000\255\255\
    \255\255\255\255\255\255\154\000\255\255\154\000\154\000\154\000\
    \154\000\154\000\154\000\154\000\154\000\154\000\154\000\154\000\
    \154\000\154\000\154\000\154\000\154\000\154\000\154\000\154\000\
    \154\000\154\000\154\000\154\000\154\000\154\000\154\000\155\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\155\000\155\000\155\000\155\000\155\000\
    \155\000\155\000\155\000\155\000\155\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\155\000\155\000\155\000\155\000\
    \155\000\155\000\155\000\155\000\155\000\155\000\155\000\155\000\
    \155\000\155\000\155\000\155\000\155\000\155\000\155\000\155\000\
    \155\000\155\000\155\000\155\000\155\000\155\000\255\255\255\255\
    \255\255\255\255\155\000\255\255\155\000\155\000\155\000\155\000\
    \155\000\155\000\155\000\155\000\155\000\155\000\155\000\155\000\
    \155\000\155\000\155\000\155\000\155\000\155\000\155\000\155\000\
    \155\000\155\000\155\000\155\000\155\000\155\000\156\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\156\000\156\000\156\000\156\000\156\000\156\000\
    \156\000\156\000\156\000\156\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\156\000\156\000\156\000\156\000\156\000\
    \156\000\156\000\156\000\156\000\156\000\156\000\156\000\156\000\
    \156\000\156\000\156\000\156\000\156\000\156\000\156\000\156\000\
    \156\000\156\000\156\000\156\000\156\000\255\255\255\255\255\255\
    \255\255\156\000\255\255\156\000\156\000\156\000\156\000\156\000\
    \156\000\156\000\156\000\156\000\156\000\156\000\156\000\156\000\
    \156\000\156\000\156\000\156\000\156\000\156\000\156\000\156\000\
    \156\000\156\000\156\000\156\000\156\000\157\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\157\000\157\000\157\000\157\000\157\000\157\000\157\000\
    \157\000\157\000\157\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\157\000\157\000\157\000\157\000\157\000\157\000\
    \157\000\157\000\157\000\157\000\157\000\157\000\157\000\157\000\
    \157\000\157\000\157\000\157\000\157\000\157\000\157\000\157\000\
    \157\000\157\000\157\000\157\000\255\255\255\255\255\255\255\255\
    \157\000\255\255\157\000\157\000\157\000\157\000\157\000\157\000\
    \157\000\157\000\157\000\157\000\157\000\157\000\157\000\157\000\
    \157\000\157\000\157\000\157\000\157\000\157\000\157\000\157\000\
    \157\000\157\000\157\000\157\000\194\000\194\000\255\255\194\000\
    \194\000\198\000\198\000\255\255\198\000\198\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\194\000\255\255\255\255\
    \255\255\255\255\198\000\194\000\255\255\194\000\255\255\255\255\
    \198\000\255\255\255\255\194\000\194\000\255\255\255\255\255\255\
    \255\255\255\255\194\000\255\255\255\255\255\255\255\255\255\255\
    \198\000\198\000\198\000\198\000\198\000\198\000\198\000\198\000\
    \198\000\198\000\201\000\201\000\201\000\201\000\201\000\201\000\
    \201\000\201\000\201\000\201\000\218\000\255\255\255\255\218\000\
    \218\000\058\001\058\001\058\001\058\001\058\001\058\001\058\001\
    \058\001\255\255\255\255\255\255\255\255\218\000\201\000\255\255\
    \255\255\255\255\255\255\218\000\255\255\255\255\255\255\201\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\198\000\255\255\255\255\255\255\
    \198\000\058\001\255\255\201\000\255\255\255\255\201\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\201\000\
    \255\255\255\255\218\000\054\001\054\001\054\001\054\001\054\001\
    \054\001\054\001\054\001\054\001\054\001\255\255\255\255\218\000\
    \255\255\058\001\255\255\255\255\054\001\054\001\054\001\054\001\
    \054\001\054\001\255\255\255\255\218\000\255\255\255\255\255\255\
    \255\255\255\255\218\000\255\255\218\000\255\255\255\255\218\000\
    \255\255\255\255\218\000\255\255\255\255\218\000\218\000\255\255\
    \218\000\255\255\218\000\255\255\054\001\054\001\054\001\054\001\
    \054\001\054\001\055\001\055\001\055\001\055\001\055\001\055\001\
    \055\001\055\001\055\001\055\001\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\055\001\055\001\055\001\055\001\055\001\
    \055\001\060\001\060\001\060\001\060\001\060\001\060\001\060\001\
    \060\001\060\001\060\001\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\060\001\060\001\060\001\060\001\060\001\060\001\
    \255\255\255\255\255\255\055\001\055\001\055\001\055\001\055\001\
    \055\001\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\194\000\255\255\255\255\255\255\255\255\
    \198\000\255\255\060\001\060\001\060\001\060\001\060\001\060\001\
    \255\255\061\001\061\001\061\001\061\001\061\001\061\001\061\001\
    \061\001\061\001\061\001\255\255\255\255\255\255\255\255\255\255\
    \255\255\058\001\061\001\061\001\061\001\061\001\061\001\061\001\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\061\001\061\001\061\001\061\001\061\001\061\001\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255";
  Lexing.lex_base_code = 
   "\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\002\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\004\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000";
  Lexing.lex_backtrk_code = 
   "\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000";
  Lexing.lex_default_code = 
   "\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000";
  Lexing.lex_trans_code = 
   "\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\001\000\000\000\001\000\001\000\001\000\001\000\001\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\001\000\000\000\001\000\000\000\000\000\000\000\
    \001\000\000\000\001\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000";
  Lexing.lex_check_code = 
   "\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\217\000\255\255\218\000\217\000\217\000\218\000\218\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\217\000\255\255\218\000\255\255\255\255\255\255\
    \217\000\255\255\218\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255";
  Lexing.lex_code = 
   "\255\001\255\255\000\001\255";
}

let rec initial lexbuf =
    __ocaml_lex_initial_rec lexbuf 0
and __ocaml_lex_initial_rec lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 486 "cil/src/frontc/clexer.mll"
               ( let il = comment lexbuf in
	                                  let sl = intlist_to_string il in
					  addComment sl;
                                          addWhite lexbuf;
                                          initial lexbuf)
# 1839 "cil/src/frontc/clexer.ml"

  | 1 ->
let
# 492 "cil/src/frontc/clexer.mll"
                        c
# 1845 "cil/src/frontc/clexer.ml"
= Lexing.sub_lexeme_char lexbuf (lexbuf.Lexing.lex_start_pos + 2) in
# 492 "cil/src/frontc/clexer.mll"
                                        ( if c = !annot_char then begin
	                                    try
                                              save_current_pos ();
	                                      Buffer.clear buf;
	                                      annot_first_token lexbuf
                                            with Parsing.Parse_error->
                                              Cabshelper.warn_skip_logic ();
                                              initial lexbuf
                                          end else begin
                                          let il = comment lexbuf in
	                                  let sl = intlist_to_string il in
					  addComment sl;
                                          addWhite lexbuf;
                                          initial lexbuf
                                          end )
# 1863 "cil/src/frontc/clexer.ml"

  | 2 ->
# 509 "cil/src/frontc/clexer.mll"
    (
      let il = onelinecomment lexbuf in
      let sl = intlist_to_string il in
      addComment sl; E.newline();
      if is_oneline_ghost () then begin
        exit_oneline_ghost ();
        RGHOST
      end else begin
        addWhite lexbuf;
        initial lexbuf
      end
    )
# 1879 "cil/src/frontc/clexer.ml"

  | 3 ->
let
# 522 "cil/src/frontc/clexer.mll"
                    c
# 1885 "cil/src/frontc/clexer.ml"
= Lexing.sub_lexeme_char lexbuf (lexbuf.Lexing.lex_start_pos + 2) in
# 523 "cil/src/frontc/clexer.mll"
    ( if c = !annot_char then begin
      try
        save_current_pos ();
	Buffer.clear buf;
	annot_one_line lexbuf
      with Parsing.Parse_error->
        Cabshelper.warn_skip_logic ();
        initial lexbuf
    end else begin
      let il = onelinecomment lexbuf in
      let sl = intlist_to_string il in
      addComment sl; E.newline();
      if is_oneline_ghost () then begin
        exit_oneline_ghost ();
        RGHOST
      end else begin
        addWhite lexbuf;
        initial lexbuf
      end
    end
    )
# 1909 "cil/src/frontc/clexer.ml"

  | 4 ->
# 544 "cil/src/frontc/clexer.mll"
           (addWhite lexbuf; initial lexbuf)
# 1914 "cil/src/frontc/clexer.ml"

  | 5 ->
# 545 "cil/src/frontc/clexer.mll"
                                        ( E.newline ();
                                          if !pragmaLine then
                                            begin
                                              pragmaLine := false;
                                              PRAGMA_EOL
                                            end
                                          else if is_oneline_ghost () then
                                            begin
                                              exit_oneline_ghost ();
                                              RGHOST
                                            end
                                          else begin
                                            addWhite lexbuf;
                                            initial lexbuf
                                          end )
# 1933 "cil/src/frontc/clexer.ml"

  | 6 ->
# 560 "cil/src/frontc/clexer.mll"
                                        ( addWhite lexbuf;
                                          E.newline ();
                                          initial lexbuf
                                        )
# 1941 "cil/src/frontc/clexer.ml"

  | 7 ->
# 564 "cil/src/frontc/clexer.mll"
         ( addWhite lexbuf; hash lexbuf)
# 1946 "cil/src/frontc/clexer.ml"

  | 8 ->
# 565 "cil/src/frontc/clexer.mll"
                                   ( PRAGMA (currentLoc ()) )
# 1951 "cil/src/frontc/clexer.ml"

  | 9 ->
# 566 "cil/src/frontc/clexer.mll"
          ( CST_CHAR (chr lexbuf, currentLoc ()))
# 1956 "cil/src/frontc/clexer.ml"

  | 10 ->
# 567 "cil/src/frontc/clexer.mll"
          ( CST_WCHAR (chr lexbuf, currentLoc ()) )
# 1961 "cil/src/frontc/clexer.ml"

  | 11 ->
# 568 "cil/src/frontc/clexer.mll"
         (  addLexeme lexbuf; (* '"' *)
(* matth: BUG:  this could be either a regular string or a wide string.
 *  e.g. if it's the "world" in
 *     L"Hello, " "world"
 *  then it should be treated as wide even though there's no L immediately
 *  preceding it.  See test/small1/wchar5.c for a failure case. *)
                                          try CST_STRING (str lexbuf, currentLoc ())
                                          with e ->
                                             raise (InternalError
                                                     ("str: " ^
                                                      Printexc.to_string e)))
# 1976 "cil/src/frontc/clexer.ml"

  | 12 ->
# 579 "cil/src/frontc/clexer.mll"
           ( (* weimer: wchar_t string literal *)
                                          try CST_WSTRING(str lexbuf, currentLoc ())
                                          with e ->
                                             raise (InternalError
                                                     ("wide string: " ^
                                                      Printexc.to_string e)))
# 1986 "cil/src/frontc/clexer.ml"

  | 13 ->
# 585 "cil/src/frontc/clexer.mll"
             (CST_FLOAT (Lexing.lexeme lexbuf, currentLoc ()))
# 1991 "cil/src/frontc/clexer.ml"

  | 14 ->
# 586 "cil/src/frontc/clexer.mll"
            (CST_INT (Lexing.lexeme lexbuf, currentLoc ()))
# 1996 "cil/src/frontc/clexer.ml"

  | 15 ->
# 587 "cil/src/frontc/clexer.mll"
            (CST_INT (Lexing.lexeme lexbuf, currentLoc ()))
# 2001 "cil/src/frontc/clexer.ml"

  | 16 ->
# 588 "cil/src/frontc/clexer.mll"
            (CST_INT (Lexing.lexeme lexbuf, currentLoc ()))
# 2006 "cil/src/frontc/clexer.ml"

  | 17 ->
# 589 "cil/src/frontc/clexer.mll"
             (EOF)
# 2011 "cil/src/frontc/clexer.ml"

  | 18 ->
# 590 "cil/src/frontc/clexer.mll"
           (ELLIPSIS)
# 2016 "cil/src/frontc/clexer.ml"

  | 19 ->
# 591 "cil/src/frontc/clexer.mll"
          (PLUS_EQ)
# 2021 "cil/src/frontc/clexer.ml"

  | 20 ->
# 592 "cil/src/frontc/clexer.mll"
          (MINUS_EQ)
# 2026 "cil/src/frontc/clexer.ml"

  | 21 ->
# 593 "cil/src/frontc/clexer.mll"
          (STAR_EQ)
# 2031 "cil/src/frontc/clexer.ml"

  | 22 ->
# 594 "cil/src/frontc/clexer.mll"
          (SLASH_EQ)
# 2036 "cil/src/frontc/clexer.ml"

  | 23 ->
# 595 "cil/src/frontc/clexer.mll"
          (PERCENT_EQ)
# 2041 "cil/src/frontc/clexer.ml"

  | 24 ->
# 596 "cil/src/frontc/clexer.mll"
          (PIPE_EQ)
# 2046 "cil/src/frontc/clexer.ml"

  | 25 ->
# 597 "cil/src/frontc/clexer.mll"
          (AND_EQ)
# 2051 "cil/src/frontc/clexer.ml"

  | 26 ->
# 598 "cil/src/frontc/clexer.mll"
          (CIRC_EQ)
# 2056 "cil/src/frontc/clexer.ml"

  | 27 ->
# 599 "cil/src/frontc/clexer.mll"
           (INF_INF_EQ)
# 2061 "cil/src/frontc/clexer.ml"

  | 28 ->
# 600 "cil/src/frontc/clexer.mll"
           (SUP_SUP_EQ)
# 2066 "cil/src/frontc/clexer.ml"

  | 29 ->
# 601 "cil/src/frontc/clexer.mll"
          (INF_INF)
# 2071 "cil/src/frontc/clexer.ml"

  | 30 ->
# 602 "cil/src/frontc/clexer.mll"
          (SUP_SUP)
# 2076 "cil/src/frontc/clexer.ml"

  | 31 ->
# 603 "cil/src/frontc/clexer.mll"
           (EQ_EQ)
# 2081 "cil/src/frontc/clexer.ml"

  | 32 ->
# 604 "cil/src/frontc/clexer.mll"
           (EXCLAM_EQ)
# 2086 "cil/src/frontc/clexer.ml"

  | 33 ->
# 605 "cil/src/frontc/clexer.mll"
          (INF_EQ)
# 2091 "cil/src/frontc/clexer.ml"

  | 34 ->
# 606 "cil/src/frontc/clexer.mll"
          (SUP_EQ)
# 2096 "cil/src/frontc/clexer.ml"

  | 35 ->
# 607 "cil/src/frontc/clexer.mll"
          (EQ)
# 2101 "cil/src/frontc/clexer.ml"

  | 36 ->
# 608 "cil/src/frontc/clexer.mll"
          (INF)
# 2106 "cil/src/frontc/clexer.ml"

  | 37 ->
# 609 "cil/src/frontc/clexer.mll"
          (SUP)
# 2111 "cil/src/frontc/clexer.ml"

  | 38 ->
# 610 "cil/src/frontc/clexer.mll"
          (PLUS_PLUS (currentLoc ()))
# 2116 "cil/src/frontc/clexer.ml"

  | 39 ->
# 611 "cil/src/frontc/clexer.mll"
          (MINUS_MINUS (currentLoc ()))
# 2121 "cil/src/frontc/clexer.ml"

  | 40 ->
# 612 "cil/src/frontc/clexer.mll"
          (ARROW)
# 2126 "cil/src/frontc/clexer.ml"

  | 41 ->
# 613 "cil/src/frontc/clexer.mll"
          (PLUS (currentLoc ()))
# 2131 "cil/src/frontc/clexer.ml"

  | 42 ->
# 614 "cil/src/frontc/clexer.mll"
          (MINUS (currentLoc ()))
# 2136 "cil/src/frontc/clexer.ml"

  | 43 ->
# 616 "cil/src/frontc/clexer.mll"
                    ( if is_ghost_code () then might_end_ghost lexbuf
                      else
                        STAR (currentLoc ()))
# 2143 "cil/src/frontc/clexer.ml"

  | 44 ->
# 619 "cil/src/frontc/clexer.mll"
          (SLASH)
# 2148 "cil/src/frontc/clexer.ml"

  | 45 ->
# 620 "cil/src/frontc/clexer.mll"
          (PERCENT)
# 2153 "cil/src/frontc/clexer.ml"

  | 46 ->
# 621 "cil/src/frontc/clexer.mll"
         (EXCLAM (currentLoc ()))
# 2158 "cil/src/frontc/clexer.ml"

  | 47 ->
# 622 "cil/src/frontc/clexer.mll"
          (AND_AND (currentLoc ()))
# 2163 "cil/src/frontc/clexer.ml"

  | 48 ->
# 623 "cil/src/frontc/clexer.mll"
          (PIPE_PIPE)
# 2168 "cil/src/frontc/clexer.ml"

  | 49 ->
# 624 "cil/src/frontc/clexer.mll"
          (AND (currentLoc ()))
# 2173 "cil/src/frontc/clexer.ml"

  | 50 ->
# 625 "cil/src/frontc/clexer.mll"
          (PIPE)
# 2178 "cil/src/frontc/clexer.ml"

  | 51 ->
# 626 "cil/src/frontc/clexer.mll"
          (CIRC)
# 2183 "cil/src/frontc/clexer.ml"

  | 52 ->
# 627 "cil/src/frontc/clexer.mll"
          (QUEST)
# 2188 "cil/src/frontc/clexer.ml"

  | 53 ->
# 628 "cil/src/frontc/clexer.mll"
          (COLON)
# 2193 "cil/src/frontc/clexer.ml"

  | 54 ->
# 629 "cil/src/frontc/clexer.mll"
               (TILDE (currentLoc ()))
# 2198 "cil/src/frontc/clexer.ml"

  | 55 ->
# 631 "cil/src/frontc/clexer.mll"
               (dbgToken (LBRACE (currentLoc ())))
# 2203 "cil/src/frontc/clexer.ml"

  | 56 ->
# 632 "cil/src/frontc/clexer.mll"
               (dbgToken (RBRACE (currentLoc ())))
# 2208 "cil/src/frontc/clexer.ml"

  | 57 ->
# 633 "cil/src/frontc/clexer.mll"
          (LBRACKET)
# 2213 "cil/src/frontc/clexer.ml"

  | 58 ->
# 634 "cil/src/frontc/clexer.mll"
          (RBRACKET)
# 2218 "cil/src/frontc/clexer.ml"

  | 59 ->
# 635 "cil/src/frontc/clexer.mll"
               (dbgToken (LPAREN (currentLoc ())) )
# 2223 "cil/src/frontc/clexer.ml"

  | 60 ->
# 636 "cil/src/frontc/clexer.mll"
          (RPAREN)
# 2228 "cil/src/frontc/clexer.ml"

  | 61 ->
# 637 "cil/src/frontc/clexer.mll"
               (dbgToken (SEMICOLON (currentLoc ())) )
# 2233 "cil/src/frontc/clexer.ml"

  | 62 ->
# 638 "cil/src/frontc/clexer.mll"
          (COMMA)
# 2238 "cil/src/frontc/clexer.ml"

  | 63 ->
# 639 "cil/src/frontc/clexer.mll"
          (DOT)
# 2243 "cil/src/frontc/clexer.ml"

  | 64 ->
# 640 "cil/src/frontc/clexer.mll"
             (SIZEOF (currentLoc ()))
# 2248 "cil/src/frontc/clexer.ml"

  | 65 ->
# 641 "cil/src/frontc/clexer.mll"
                                        ( if !Cprint.msvcMode then
                                             MSASM (msasm lexbuf, currentLoc ())
                                          else (ASM (currentLoc ())) )
# 2255 "cil/src/frontc/clexer.ml"

  | 66 ->
# 646 "cil/src/frontc/clexer.mll"
                                        ( matchingParsOpen := 0;
                                          let _ = matchingpars lexbuf in
                                          addWhite lexbuf;
                                          initial lexbuf
                                        )
# 2264 "cil/src/frontc/clexer.ml"

  | 67 ->
# 653 "cil/src/frontc/clexer.mll"
                                        (AT_TRANSFORM (currentLoc ()))
# 2269 "cil/src/frontc/clexer.ml"

  | 68 ->
# 654 "cil/src/frontc/clexer.mll"
                                        (AT_TRANSFORMEXPR (currentLoc ()))
# 2274 "cil/src/frontc/clexer.ml"

  | 69 ->
# 655 "cil/src/frontc/clexer.mll"
                                        (AT_SPECIFIER (currentLoc ()))
# 2279 "cil/src/frontc/clexer.ml"

  | 70 ->
# 656 "cil/src/frontc/clexer.mll"
                                        (AT_EXPR (currentLoc ()))
# 2284 "cil/src/frontc/clexer.ml"

  | 71 ->
# 657 "cil/src/frontc/clexer.mll"
                                        (AT_NAME)
# 2289 "cil/src/frontc/clexer.ml"

  | 72 ->
# 661 "cil/src/frontc/clexer.mll"
                                        (addWhite lexbuf; initial lexbuf )
# 2294 "cil/src/frontc/clexer.ml"

  | 73 ->
# 662 "cil/src/frontc/clexer.mll"
           (scan_ident (Lexing.lexeme lexbuf))
# 2299 "cil/src/frontc/clexer.ml"

  | 74 ->
# 664 "cil/src/frontc/clexer.mll"
  ( if is_oneline_ghost() then begin
      exit_oneline_ghost (); RGHOST
    end
    else EOF
  )
# 2308 "cil/src/frontc/clexer.ml"

  | 75 ->
# 669 "cil/src/frontc/clexer.mll"
       (E.parse_error "Invalid symbol")
# 2313 "cil/src/frontc/clexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf; __ocaml_lex_initial_rec lexbuf __ocaml_lex_state

and might_end_ghost lexbuf =
    __ocaml_lex_might_end_ghost_rec lexbuf 188
and __ocaml_lex_might_end_ghost_rec lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 672 "cil/src/frontc/clexer.mll"
        ( exit_ghost_code(); RGHOST )
# 2324 "cil/src/frontc/clexer.ml"

  | 1 ->
# 673 "cil/src/frontc/clexer.mll"
       ( STAR (currentLoc()) )
# 2329 "cil/src/frontc/clexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf; __ocaml_lex_might_end_ghost_rec lexbuf __ocaml_lex_state

and comment lexbuf =
    __ocaml_lex_comment_rec lexbuf 190
and __ocaml_lex_comment_rec lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 676 "cil/src/frontc/clexer.mll"
                                ( addWhite lexbuf; [] )
# 2340 "cil/src/frontc/clexer.ml"

  | 1 ->
# 677 "cil/src/frontc/clexer.mll"
                                ( E.newline (); comment lexbuf )
# 2345 "cil/src/frontc/clexer.ml"

  | 2 ->
# 679 "cil/src/frontc/clexer.mll"
      ( addWhite lexbuf; comment lexbuf )
# 2350 "cil/src/frontc/clexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf; __ocaml_lex_comment_rec lexbuf __ocaml_lex_state

and onelinecomment lexbuf =
    __ocaml_lex_onelinecomment_rec lexbuf 193
and __ocaml_lex_onelinecomment_rec lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 682 "cil/src/frontc/clexer.mll"
                (addWhite lexbuf; [])
# 2361 "cil/src/frontc/clexer.ml"

  | 1 ->
# 683 "cil/src/frontc/clexer.mll"
                (addWhite lexbuf; lex_comment onelinecomment lexbuf )
# 2366 "cil/src/frontc/clexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf; __ocaml_lex_onelinecomment_rec lexbuf __ocaml_lex_state

and matchingpars lexbuf =
    __ocaml_lex_matchingpars_rec lexbuf 194
and __ocaml_lex_matchingpars_rec lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 686 "cil/src/frontc/clexer.mll"
                ( addWhite lexbuf; E.newline (); matchingpars lexbuf )
# 2377 "cil/src/frontc/clexer.ml"

  | 1 ->
# 687 "cil/src/frontc/clexer.mll"
                ( addWhite lexbuf; matchingpars lexbuf )
# 2382 "cil/src/frontc/clexer.ml"

  | 2 ->
# 688 "cil/src/frontc/clexer.mll"
                ( addWhite lexbuf; incr matchingParsOpen; matchingpars lexbuf )
# 2387 "cil/src/frontc/clexer.ml"

  | 3 ->
# 689 "cil/src/frontc/clexer.mll"
                ( addWhite lexbuf; decr matchingParsOpen;
                  if !matchingParsOpen = 0 then
                     ()
                  else
                     matchingpars lexbuf
                )
# 2397 "cil/src/frontc/clexer.ml"

  | 4 ->
# 695 "cil/src/frontc/clexer.mll"
         ( addWhite lexbuf; let il = comment lexbuf in
                  let sl = intlist_to_string il in
		  addComment sl;
                  matchingpars lexbuf)
# 2405 "cil/src/frontc/clexer.ml"

  | 5 ->
# 699 "cil/src/frontc/clexer.mll"
        ( addWhite lexbuf; (* '"' *)
                  let _ = str lexbuf in
                  matchingpars lexbuf
                 )
# 2413 "cil/src/frontc/clexer.ml"

  | 6 ->
# 703 "cil/src/frontc/clexer.mll"
                 ( addWhite lexbuf; matchingpars lexbuf )
# 2418 "cil/src/frontc/clexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf; __ocaml_lex_matchingpars_rec lexbuf __ocaml_lex_state

and hash lexbuf =
  lexbuf.Lexing.lex_mem <- Array.create 2 (-1) ;   __ocaml_lex_hash_rec lexbuf 198
and __ocaml_lex_hash_rec lexbuf __ocaml_lex_state =
  match Lexing.new_engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 707 "cil/src/frontc/clexer.mll"
        ( addWhite lexbuf; E.newline (); initial lexbuf)
# 2429 "cil/src/frontc/clexer.ml"

  | 1 ->
# 708 "cil/src/frontc/clexer.mll"
         ( addWhite lexbuf; hash lexbuf)
# 2434 "cil/src/frontc/clexer.ml"

  | 2 ->
# 709 "cil/src/frontc/clexer.mll"
         ( addWhite lexbuf; (* We are seeing a line number. This is the number for the
                   * next line *)
                 let s = Lexing.lexeme lexbuf in
                 let lineno = try
                   int_of_string s
                 with Failure ("int_of_string") ->
                   (* the int is too big. *)
                   E.warn "Bad line number in preprocessed file: %s" s;
                   (-1)
                 in
                 E.setCurrentLine (lineno - 1);
                  (* A file name may follow *)
		  file lexbuf )
# 2451 "cil/src/frontc/clexer.ml"

  | 3 ->
# 722 "cil/src/frontc/clexer.mll"
                ( addWhite lexbuf; hash lexbuf )
# 2456 "cil/src/frontc/clexer.ml"

  | 4 ->
let
# 725 "cil/src/frontc/clexer.mll"
                                     pragmaName
# 2462 "cil/src/frontc/clexer.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_mem.(0) lexbuf.Lexing.lex_curr_pos in
# 726 "cil/src/frontc/clexer.mll"
                ( let here = currentLoc () in
                  PRAGMA_LINE (pragmaName ^ pragma lexbuf, here)
                )
# 2468 "cil/src/frontc/clexer.ml"

  | 5 ->
# 729 "cil/src/frontc/clexer.mll"
                ( pragmaLine := true; PRAGMA (currentLoc ()) )
# 2473 "cil/src/frontc/clexer.ml"

  | 6 ->
# 730 "cil/src/frontc/clexer.mll"
            ( addWhite lexbuf; endline lexbuf)
# 2478 "cil/src/frontc/clexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf; __ocaml_lex_hash_rec lexbuf __ocaml_lex_state

and file lexbuf =
    __ocaml_lex_file_rec lexbuf 301
and __ocaml_lex_file_rec lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 733 "cil/src/frontc/clexer.mll"
                      (addWhite lexbuf; E.newline (); initial lexbuf)
# 2489 "cil/src/frontc/clexer.ml"

  | 1 ->
# 734 "cil/src/frontc/clexer.mll"
          (addWhite lexbuf; file lexbuf)
# 2494 "cil/src/frontc/clexer.ml"

  | 2 ->
# 735 "cil/src/frontc/clexer.mll"
                                ( addWhite lexbuf;  (* '"' *)
                                   let n = Lexing.lexeme lexbuf in
                                   let n1 = String.sub n 1
                                       ((String.length n) - 2) in
                                   E.setCurrentFile n1;
				 endline lexbuf)
# 2504 "cil/src/frontc/clexer.ml"

  | 3 ->
# 742 "cil/src/frontc/clexer.mll"
      (addWhite lexbuf; endline lexbuf)
# 2509 "cil/src/frontc/clexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf; __ocaml_lex_file_rec lexbuf __ocaml_lex_state

and endline lexbuf =
    __ocaml_lex_endline_rec lexbuf 305
and __ocaml_lex_endline_rec lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 745 "cil/src/frontc/clexer.mll"
                ( addWhite lexbuf; E.newline (); initial lexbuf)
# 2520 "cil/src/frontc/clexer.ml"

  | 1 ->
# 746 "cil/src/frontc/clexer.mll"
                                ( EOF )
# 2525 "cil/src/frontc/clexer.ml"

  | 2 ->
# 747 "cil/src/frontc/clexer.mll"
      ( addWhite lexbuf; endline lexbuf)
# 2530 "cil/src/frontc/clexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf; __ocaml_lex_endline_rec lexbuf __ocaml_lex_state

and pragma lexbuf =
    __ocaml_lex_pragma_rec lexbuf 306
and __ocaml_lex_pragma_rec lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 750 "cil/src/frontc/clexer.mll"
                        ( E.newline (); "" )
# 2541 "cil/src/frontc/clexer.ml"

  | 1 ->
# 751 "cil/src/frontc/clexer.mll"
                        ( let cur = Lexing.lexeme lexbuf in
                          cur ^ (pragma lexbuf) )
# 2547 "cil/src/frontc/clexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf; __ocaml_lex_pragma_rec lexbuf __ocaml_lex_state

and str lexbuf =
    __ocaml_lex_str_rec lexbuf 307
and __ocaml_lex_str_rec lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 755 "cil/src/frontc/clexer.mll"
                        ([])
# 2558 "cil/src/frontc/clexer.ml"

  | 1 ->
# 756 "cil/src/frontc/clexer.mll"
             (addLexeme lexbuf; lex_hex_escape str lexbuf)
# 2563 "cil/src/frontc/clexer.ml"

  | 2 ->
# 757 "cil/src/frontc/clexer.mll"
             (addLexeme lexbuf; lex_oct_escape str lexbuf)
# 2568 "cil/src/frontc/clexer.ml"

  | 3 ->
# 758 "cil/src/frontc/clexer.mll"
          (addLexeme lexbuf; lex_simple_escape str lexbuf)
# 2573 "cil/src/frontc/clexer.ml"

  | 4 ->
# 759 "cil/src/frontc/clexer.mll"
                        (E.parse_error "unterminated string" )
# 2578 "cil/src/frontc/clexer.ml"

  | 5 ->
# 760 "cil/src/frontc/clexer.mll"
     (addLexeme lexbuf; lex_unescaped str lexbuf)
# 2583 "cil/src/frontc/clexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf; __ocaml_lex_str_rec lexbuf __ocaml_lex_state

and chr lexbuf =
    __ocaml_lex_chr_rec lexbuf 313
and __ocaml_lex_chr_rec lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 763 "cil/src/frontc/clexer.mll"
              ([])
# 2594 "cil/src/frontc/clexer.ml"

  | 1 ->
# 764 "cil/src/frontc/clexer.mll"
             (lex_hex_escape chr lexbuf)
# 2599 "cil/src/frontc/clexer.ml"

  | 2 ->
# 765 "cil/src/frontc/clexer.mll"
             (lex_oct_escape chr lexbuf)
# 2604 "cil/src/frontc/clexer.ml"

  | 3 ->
# 766 "cil/src/frontc/clexer.mll"
          (lex_simple_escape chr lexbuf)
# 2609 "cil/src/frontc/clexer.ml"

  | 4 ->
# 767 "cil/src/frontc/clexer.mll"
                        ( E.parse_error "unterminated char" )
# 2614 "cil/src/frontc/clexer.ml"

  | 5 ->
# 768 "cil/src/frontc/clexer.mll"
     (lex_unescaped chr lexbuf)
# 2619 "cil/src/frontc/clexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf; __ocaml_lex_chr_rec lexbuf __ocaml_lex_state

and msasm lexbuf =
    __ocaml_lex_msasm_rec lexbuf 319
and __ocaml_lex_msasm_rec lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 771 "cil/src/frontc/clexer.mll"
                        ( msasm lexbuf )
# 2630 "cil/src/frontc/clexer.ml"

  | 1 ->
# 772 "cil/src/frontc/clexer.mll"
                        ( msasminbrace lexbuf )
# 2635 "cil/src/frontc/clexer.ml"

  | 2 ->
# 773 "cil/src/frontc/clexer.mll"
                        ( let cur = Lexing.lexeme lexbuf in
                          cur ^ (msasmnobrace lexbuf) )
# 2641 "cil/src/frontc/clexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf; __ocaml_lex_msasm_rec lexbuf __ocaml_lex_state

and msasminbrace lexbuf =
    __ocaml_lex_msasminbrace_rec lexbuf 321
and __ocaml_lex_msasminbrace_rec lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 777 "cil/src/frontc/clexer.mll"
                        ( "" )
# 2652 "cil/src/frontc/clexer.ml"

  | 1 ->
# 778 "cil/src/frontc/clexer.mll"
                        ( let cur = Lexing.lexeme lexbuf in
                          cur ^ (msasminbrace lexbuf) )
# 2658 "cil/src/frontc/clexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf; __ocaml_lex_msasminbrace_rec lexbuf __ocaml_lex_state

and msasmnobrace lexbuf =
    __ocaml_lex_msasmnobrace_rec lexbuf 322
and __ocaml_lex_msasmnobrace_rec lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 781 "cil/src/frontc/clexer.mll"
                        ( lexbuf.Lexing.lex_curr_pos <-
                               lexbuf.Lexing.lex_curr_pos - 1;
                          "" )
# 2671 "cil/src/frontc/clexer.ml"

  | 1 ->
# 784 "cil/src/frontc/clexer.mll"
                        ( lexbuf.Lexing.lex_curr_pos <-
                               lexbuf.Lexing.lex_curr_pos - 5;
                          "" )
# 2678 "cil/src/frontc/clexer.ml"

  | 2 ->
# 787 "cil/src/frontc/clexer.mll"
                        ( let cur = Lexing.lexeme lexbuf in

                          cur ^ (msasmnobrace lexbuf) )
# 2685 "cil/src/frontc/clexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf; __ocaml_lex_msasmnobrace_rec lexbuf __ocaml_lex_state

and annot_first_token lexbuf =
    __ocaml_lex_annot_first_token_rec lexbuf 327
and __ocaml_lex_annot_first_token_rec lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 792 "cil/src/frontc/clexer.mll"
            (
      if is_oneline_ghost () then E.parse_error "nested ghost code";
      Buffer.clear buf;
      enter_ghost_code ();
      LGHOST
    )
# 2701 "cil/src/frontc/clexer.ml"

  | 1 ->
let
# 798 "cil/src/frontc/clexer.mll"
                         c
# 2707 "cil/src/frontc/clexer.ml"
= Lexing.sub_lexeme_char lexbuf lexbuf.Lexing.lex_start_pos in
# 798 "cil/src/frontc/clexer.mll"
                           ( Buffer.add_char buf c; annot_first_token lexbuf )
# 2711 "cil/src/frontc/clexer.ml"

  | 2 ->
# 799 "cil/src/frontc/clexer.mll"
         ( E.newline(); Buffer.add_char buf '\n'; annot_first_token lexbuf )
# 2716 "cil/src/frontc/clexer.ml"

  | 3 ->
# 800 "cil/src/frontc/clexer.mll"
       ( annot_token lexbuf )
# 2721 "cil/src/frontc/clexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf; __ocaml_lex_annot_first_token_rec lexbuf __ocaml_lex_state

and annot_token lexbuf =
    __ocaml_lex_annot_token_rec lexbuf 332
and __ocaml_lex_annot_token_rec lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 802 "cil/src/frontc/clexer.mll"
         ( let s = Buffer.contents buf in
           make_annot s )
# 2733 "cil/src/frontc/clexer.ml"

  | 1 ->
# 804 "cil/src/frontc/clexer.mll"
         ( E.parse_error "Unterminated annotation" )
# 2738 "cil/src/frontc/clexer.ml"

  | 2 ->
# 805 "cil/src/frontc/clexer.mll"
         (E.newline(); Buffer.add_char buf '\n'; annot_token lexbuf )
# 2743 "cil/src/frontc/clexer.ml"

  | 3 ->
let
# 806 "cil/src/frontc/clexer.mll"
         c
# 2749 "cil/src/frontc/clexer.ml"
= Lexing.sub_lexeme_char lexbuf lexbuf.Lexing.lex_start_pos in
# 806 "cil/src/frontc/clexer.mll"
           ( Buffer.add_char buf c; annot_token lexbuf )
# 2753 "cil/src/frontc/clexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf; __ocaml_lex_annot_token_rec lexbuf __ocaml_lex_state

and annot_one_line lexbuf =
    __ocaml_lex_annot_one_line_rec lexbuf 334
and __ocaml_lex_annot_one_line_rec lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 809 "cil/src/frontc/clexer.mll"
            (
      if is_oneline_ghost () then E.parse_error "nested ghost code";
      enter_oneline_ghost (); LGHOST
    )
# 2767 "cil/src/frontc/clexer.ml"

  | 1 ->
let
# 813 "cil/src/frontc/clexer.mll"
                         c
# 2773 "cil/src/frontc/clexer.ml"
= Lexing.sub_lexeme_char lexbuf lexbuf.Lexing.lex_start_pos in
# 813 "cil/src/frontc/clexer.mll"
                           ( Buffer.add_char buf c; annot_one_line lexbuf )
# 2777 "cil/src/frontc/clexer.ml"

  | 2 ->
# 814 "cil/src/frontc/clexer.mll"
       ( annot_one_line_logic lexbuf )
# 2782 "cil/src/frontc/clexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf; __ocaml_lex_annot_one_line_rec lexbuf __ocaml_lex_state

and annot_one_line_logic lexbuf =
    __ocaml_lex_annot_one_line_logic_rec lexbuf 339
and __ocaml_lex_annot_one_line_logic_rec lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 816 "cil/src/frontc/clexer.mll"
         ( E.newline (); make_annot (Buffer.contents buf) )
# 2793 "cil/src/frontc/clexer.ml"

  | 1 ->
let
# 817 "cil/src/frontc/clexer.mll"
         c
# 2799 "cil/src/frontc/clexer.ml"
= Lexing.sub_lexeme_char lexbuf lexbuf.Lexing.lex_start_pos in
# 817 "cil/src/frontc/clexer.mll"
           ( Buffer.add_char buf c; annot_one_line_logic lexbuf )
# 2803 "cil/src/frontc/clexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf; __ocaml_lex_annot_one_line_logic_rec lexbuf __ocaml_lex_state

;;

# 819 "cil/src/frontc/clexer.mll"
 


# 2813 "cil/src/frontc/clexer.ml"
